/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.tools.cli.commands.bookie;

import com.beust.jcommander.Parameter;
import com.google.common.util.concurrent.UncheckedExecutionException;
import io.netty.buffer.ByteBuf;
import java.io.File;
import java.io.IOException;
import org.apache.bookkeeper.bookie.ReadOnlyDefaultEntryLogger;
import org.apache.bookkeeper.bookie.storage.EntryLogScanner;
import org.apache.bookkeeper.bookie.storage.EntryLogger;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.tools.cli.commands.bookie.FormatUtil;
import org.apache.bookkeeper.tools.cli.helpers.BookieCommand;
import org.apache.bookkeeper.tools.framework.CliFlags;
import org.apache.bookkeeper.tools.framework.CliSpec;
import org.apache.bookkeeper.util.EntryFormatter;
import org.apache.bookkeeper.util.LedgerIdFormatter;
import org.apache.commons.lang3.mutable.MutableBoolean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class ReadLogCommand
extends BookieCommand<ReadLogFlags> {
    private static final String NAME = "readlog";
    private static final String DESC = "Scan an entry file and format the entries into readable format.";
    private static final Logger LOG = LoggerFactory.getLogger(ReadLogCommand.class);
    private EntryLogger entryLogger;
    private EntryFormatter entryFormatter;
    private LedgerIdFormatter ledgerIdFormatter;

    public ReadLogCommand() {
        this(new ReadLogFlags());
    }

    public ReadLogCommand(LedgerIdFormatter ledgerIdFormatter, EntryFormatter entryFormatter) {
        this(new ReadLogFlags());
        this.ledgerIdFormatter = ledgerIdFormatter;
        this.entryFormatter = entryFormatter;
    }

    private ReadLogCommand(ReadLogFlags flags) {
        super(CliSpec.newBuilder().withName(NAME).withDescription(DESC).withFlags((CliFlags)flags).build());
    }

    @Override
    public boolean apply(ServerConfiguration conf, ReadLogFlags cmdFlags) {
        if (cmdFlags.ledgerIdFormatter != null && this.ledgerIdFormatter == null) {
            this.ledgerIdFormatter = LedgerIdFormatter.newLedgerIdFormatter(cmdFlags.ledgerIdFormatter, conf);
        } else if (this.ledgerIdFormatter == null) {
            this.ledgerIdFormatter = LedgerIdFormatter.newLedgerIdFormatter(conf);
        }
        if (cmdFlags.entryFormatter != null && this.entryFormatter == null) {
            this.entryFormatter = EntryFormatter.newEntryFormatter(cmdFlags.entryFormatter, conf);
        } else if (this.entryFormatter == null) {
            this.entryFormatter = EntryFormatter.newEntryFormatter(conf);
        }
        if (cmdFlags.entryLogId == -1L && cmdFlags.filename == null) {
            LOG.error("Missing entry log id or entry log file name");
            this.usage();
            return false;
        }
        try {
            return this.readLog(conf, cmdFlags);
        }
        catch (Exception e) {
            throw new UncheckedExecutionException(e.getMessage(), (Throwable)e);
        }
    }

    private boolean readLog(ServerConfiguration conf, ReadLogFlags flags) throws Exception {
        long logId = flags.entryLogId;
        if (logId == -1L && flags.filename != null) {
            File f = new File(flags.filename);
            String name = f.getName();
            if (!name.endsWith(".log")) {
                LOG.error("Invalid entry log file name {}", (Object)flags.filename);
                this.usage();
                return false;
            }
            String idString = name.split("\\.")[0];
            logId = Long.parseLong(idString, 16);
        }
        long lId = flags.ledgerId;
        long eId = flags.entryId;
        long startpos = flags.startPos;
        long endpos = flags.endPos;
        if (startpos != -1L) {
            if (endpos != -1L && endpos < startpos) {
                LOG.error("ERROR: StartPosition of the range should be lesser than or equal to EndPosition");
                return false;
            }
            this.scanEntryLogForPositionRange(conf, logId, startpos, endpos, flags.msg);
        } else if (lId != -1L) {
            this.scanEntryLogForSpecificEntry(conf, logId, lId, eId, flags.msg);
        } else {
            this.scanEntryLog(conf, logId, flags.msg);
        }
        return true;
    }

    private void scanEntryLogForPositionRange(ServerConfiguration conf, long logId, final long rangeStartPos, final long rangeEndPos, final boolean printMsg) throws Exception {
        LOG.info("Scan entry log {} ({}.log) for PositionRange: {} - {}", new Object[]{logId, Long.toHexString(logId), rangeStartPos, rangeEndPos});
        final MutableBoolean entryFound = new MutableBoolean(false);
        this.scanEntryLog(conf, logId, new EntryLogScanner(){
            private MutableBoolean stopScanning = new MutableBoolean(false);

            @Override
            public boolean accept(long ledgerId) {
                return !this.stopScanning.booleanValue();
            }

            @Override
            public void process(long ledgerId, long entryStartPos, ByteBuf entry) throws IOException {
                if (!this.stopScanning.booleanValue()) {
                    if (rangeEndPos != -1L && entryStartPos > rangeEndPos) {
                        this.stopScanning.setValue(true);
                    } else {
                        int entrySize = entry.readableBytes();
                        long entryEndPos = entryStartPos + (long)entrySize + 4L - 1L;
                        if ((rangeEndPos == -1L || entryStartPos <= rangeEndPos) && rangeStartPos <= entryEndPos) {
                            FormatUtil.formatEntry(entryStartPos, entry, printMsg, ReadLogCommand.this.ledgerIdFormatter, ReadLogCommand.this.entryFormatter);
                            entryFound.setValue(true);
                        }
                    }
                }
            }
        });
        if (!entryFound.booleanValue()) {
            LOG.info("Entry log {} ({}.log) doesn't has any entry in the range {} - {}. Probably the position range, you have provided is lesser than the LOGFILE_HEADER_SIZE (1024) or greater than the current log filesize.", new Object[]{logId, Long.toHexString(logId), rangeStartPos, rangeEndPos});
        }
    }

    private void scanEntryLog(ServerConfiguration conf, long logId, EntryLogScanner scanner) throws IOException {
        this.initEntryLogger(conf);
        this.entryLogger.scanEntryLog(logId, scanner);
    }

    private synchronized void initEntryLogger(ServerConfiguration conf) throws IOException {
        if (null == this.entryLogger) {
            this.entryLogger = new ReadOnlyDefaultEntryLogger(conf);
        }
    }

    private void scanEntryLogForSpecificEntry(ServerConfiguration conf, long logId, final long ledgerId, final long entryId, final boolean printMsg) throws Exception {
        LOG.info("Scan entry log {} ({}.log) for LedgerId {} {}", new Object[]{logId, Long.toHexString(logId), ledgerId, entryId == -1L ? "" : " for EntryId " + entryId});
        final MutableBoolean entryFound = new MutableBoolean(false);
        this.scanEntryLog(conf, logId, new EntryLogScanner(){

            @Override
            public boolean accept(long candidateLedgerId) {
                return candidateLedgerId == ledgerId && (!entryFound.booleanValue() || entryId == -1L);
            }

            @Override
            public void process(long candidateLedgerId, long startPos, ByteBuf entry) {
                long entrysLedgerId = entry.getLong(entry.readerIndex());
                long entrysEntryId = entry.getLong(entry.readerIndex() + 8);
                if (candidateLedgerId == entrysLedgerId && candidateLedgerId == ledgerId && (entrysEntryId == entryId || entryId == -1L)) {
                    entryFound.setValue(true);
                    FormatUtil.formatEntry(startPos, entry, printMsg, ReadLogCommand.this.ledgerIdFormatter, ReadLogCommand.this.entryFormatter);
                }
            }
        });
        if (!entryFound.booleanValue()) {
            LOG.info("LedgerId {} {} is not available in the entry log {} ({}.log)", new Object[]{ledgerId, entryId == -1L ? "" : " EntryId " + entryId, logId, Long.toHexString(logId)});
        }
    }

    private void scanEntryLog(ServerConfiguration conf, long logId, final boolean printMsg) throws Exception {
        LOG.info("Scan entry log {} ({}.log)", (Object)logId, (Object)Long.toHexString(logId));
        this.scanEntryLog(conf, logId, new EntryLogScanner(){

            @Override
            public boolean accept(long ledgerId) {
                return true;
            }

            @Override
            public void process(long ledgerId, long startPos, ByteBuf entry) {
                FormatUtil.formatEntry(startPos, entry, printMsg, ReadLogCommand.this.ledgerIdFormatter, ReadLogCommand.this.entryFormatter);
            }
        });
    }

    public static class ReadLogFlags
    extends CliFlags {
        @Parameter(names={"-m", "msg"}, description="Print message body")
        private boolean msg;
        @Parameter(names={"-l", "--ledgerid"}, description="Ledger ID")
        private long ledgerId = -1L;
        @Parameter(names={"-e", "--entryid"}, description="Entry ID")
        private long entryId = -1L;
        @Parameter(names={"-sp", "--startpos"}, description="Start Position")
        private long startPos = -1L;
        @Parameter(names={"-ep", "--endpos"}, description="End Position")
        private long endPos = -1L;
        @Parameter(names={"-f", "--filename"}, description="Entry log filename")
        private String filename;
        @Parameter(names={"-li", "--entrylogid"}, description="Entry log id")
        private long entryLogId = -1L;
        @Parameter(names={"-lf", "--ledgerIdFormatter"}, description="Set ledger id formatter")
        private String ledgerIdFormatter;
        @Parameter(names={"-ef", "--entryformatter"}, description="set entry formatter")
        private String entryFormatter;

        public ReadLogFlags msg(boolean msg) {
            this.msg = msg;
            return this;
        }

        public ReadLogFlags ledgerId(long ledgerId) {
            this.ledgerId = ledgerId;
            return this;
        }

        public ReadLogFlags entryId(long entryId) {
            this.entryId = entryId;
            return this;
        }

        public ReadLogFlags startPos(long startPos) {
            this.startPos = startPos;
            return this;
        }

        public ReadLogFlags endPos(long endPos) {
            this.endPos = endPos;
            return this;
        }

        public ReadLogFlags filename(String filename) {
            this.filename = filename;
            return this;
        }

        public ReadLogFlags entryLogId(long entryLogId) {
            this.entryLogId = entryLogId;
            return this;
        }

        public ReadLogFlags ledgerIdFormatter(String ledgerIdFormatter) {
            this.ledgerIdFormatter = ledgerIdFormatter;
            return this;
        }

        public ReadLogFlags entryFormatter(String entryFormatter) {
            this.entryFormatter = entryFormatter;
            return this;
        }
    }
}

