/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.serde2.json;

import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.JsonNodeType;
import com.fasterxml.jackson.databind.node.TextNode;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.time.ZoneId;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.commons.lang3.tuple.ImmutablePair;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.hadoop.hive.common.type.Date;
import org.apache.hadoop.hive.common.type.HiveChar;
import org.apache.hadoop.hive.common.type.HiveDecimal;
import org.apache.hadoop.hive.common.type.HiveVarchar;
import org.apache.hadoop.hive.common.type.Timestamp;
import org.apache.hadoop.hive.common.type.TimestampTZ;
import org.apache.hadoop.hive.serde2.SerDeException;
import org.apache.hadoop.hive.serde2.json.BinaryEncoding;
import org.apache.hadoop.hive.serde2.objectinspector.ListObjectInspector;
import org.apache.hadoop.hive.serde2.objectinspector.MapObjectInspector;
import org.apache.hadoop.hive.serde2.objectinspector.ObjectInspector;
import org.apache.hadoop.hive.serde2.objectinspector.PrimitiveObjectInspector;
import org.apache.hadoop.hive.serde2.objectinspector.StructField;
import org.apache.hadoop.hive.serde2.objectinspector.StructObjectInspector;
import org.apache.hadoop.hive.serde2.objectinspector.primitive.PrimitiveObjectInspectorFactory;
import org.apache.hadoop.hive.serde2.typeinfo.BaseCharTypeInfo;
import org.apache.hadoop.hive.serde2.typeinfo.DecimalTypeInfo;
import org.apache.hadoop.hive.serde2.typeinfo.HiveDecimalUtils;
import org.apache.hadoop.hive.serde2.typeinfo.PrimitiveTypeInfo;
import org.apache.hadoop.hive.serde2.typeinfo.TimestampLocalTZTypeInfo;
import org.apache.hive.common.util.TimestampParser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HiveJsonReader {
    private static final Logger LOG = LoggerFactory.getLogger(HiveJsonReader.class);
    private final Map<Pair<StructObjectInspector, String>, StructField> discoveredFields = new HashMap<Pair<StructObjectInspector, String>, StructField>();
    private final Set<Pair<StructObjectInspector, String>> discoveredUnknownFields = new HashSet<Pair<StructObjectInspector, String>>();
    private final EnumSet<Feature> features = EnumSet.noneOf(Feature.class);
    private final ObjectMapper objectMapper;
    private final TimestampParser tsParser;
    private final ObjectInspector oi;
    private static final Pattern BASE64_PATTERN = Pattern.compile("^(?:[A-Za-z0-9+/]{4})*(?:[A-Za-z0-9+/]{2}==|[A-Za-z0-9+/]{3}=)?$");
    private final Pattern internalPattern = Pattern.compile("^_col([0-9]+)$");

    public HiveJsonReader(ObjectInspector oi) {
        this(oi, new TimestampParser());
    }

    public HiveJsonReader(ObjectInspector oi, TimestampParser tsParser) {
        this.tsParser = tsParser;
        this.oi = oi;
        this.objectMapper = new ObjectMapper().enable(DeserializationFeature.USE_BIG_DECIMAL_FOR_FLOATS);
    }

    public Object parseStruct(String text) throws IOException, SerDeException {
        Preconditions.checkNotNull((Object)text);
        Preconditions.checkState((this.oi != null ? 1 : 0) != 0);
        JsonNode rootNode = this.objectMapper.reader().readTree(text);
        return this.visitNode(rootNode, this.oi);
    }

    public Object parseStruct(InputStream in) throws IOException, SerDeException {
        Preconditions.checkNotNull((Object)in);
        Preconditions.checkState((this.oi != null ? 1 : 0) != 0);
        JsonNode rootNode = this.objectMapper.reader().readTree(in);
        return this.visitNode(rootNode, this.oi);
    }

    private Object visitNode(JsonNode rootNode, ObjectInspector oi) throws SerDeException {
        if (!rootNode.isNull()) {
            switch (oi.getCategory()) {
                case PRIMITIVE: {
                    Object value = this.visitLeafNode(rootNode, oi);
                    return this.optionallyWrapWritable(value, oi);
                }
                case LIST: {
                    return this.visitArrayNode(rootNode, oi);
                }
                case STRUCT: {
                    return this.visitStructNode(rootNode, oi);
                }
                case MAP: {
                    return this.visitMapNode(rootNode, oi);
                }
            }
            throw new SerDeException("Parsing of: " + oi.getCategory() + " is not supported");
        }
        return null;
    }

    private Object optionallyWrapWritable(Object value, ObjectInspector oi) {
        if (!this.isEnabled(Feature.PRIMITIVE_TO_WRITABLE)) {
            return value;
        }
        PrimitiveObjectInspector poi = (PrimitiveObjectInspector)oi;
        PrimitiveTypeInfo typeInfo = poi.getTypeInfo();
        return PrimitiveObjectInspectorFactory.getPrimitiveJavaObjectInspector(typeInfo).getPrimitiveWritableObject(value);
    }

    private Map<Object, Object> visitMapNode(JsonNode rootNode, ObjectInspector oi) throws SerDeException {
        Preconditions.checkArgument((JsonNodeType.OBJECT == rootNode.getNodeType() ? 1 : 0) != 0);
        LinkedHashMap<Object, Object> ret = new LinkedHashMap<Object, Object>();
        ObjectInspector mapKeyInspector = ((MapObjectInspector)oi).getMapKeyObjectInspector();
        ObjectInspector mapValueInspector = ((MapObjectInspector)oi).getMapValueObjectInspector();
        if (!(mapKeyInspector instanceof PrimitiveObjectInspector)) {
            throw new SerDeException("Map key must be a primitive type");
        }
        Iterator it = rootNode.fields();
        while (it.hasNext()) {
            Map.Entry field = (Map.Entry)it.next();
            Object key = this.visitNode((JsonNode)new TextNode((String)field.getKey()), mapKeyInspector);
            Object val = this.visitNode((JsonNode)field.getValue(), mapValueInspector);
            ret.put(key, val);
        }
        return ret;
    }

    private List<Object> visitStructNode(JsonNode rootNode, ObjectInspector oi) throws SerDeException {
        Preconditions.checkArgument((JsonNodeType.OBJECT == rootNode.getNodeType() ? 1 : 0) != 0);
        StructObjectInspector structInspector = (StructObjectInspector)oi;
        int fieldCount = structInspector.getAllStructFieldRefs().size();
        List<Object> ret = Arrays.asList(new Object[fieldCount]);
        Iterator it = rootNode.fields();
        while (it.hasNext()) {
            Map.Entry field = (Map.Entry)it.next();
            String fieldName = (String)field.getKey();
            JsonNode childNode = (JsonNode)field.getValue();
            StructField structField = this.getStructField(structInspector, fieldName);
            if (structField == null) continue;
            Object childValue = this.visitNode(childNode, structField.getFieldObjectInspector());
            ret.set(structField.getFieldID(), childValue);
        }
        return ret;
    }

    private List<Object> visitArrayNode(JsonNode rootNode, ObjectInspector oi) throws SerDeException {
        Preconditions.checkArgument((JsonNodeType.ARRAY == rootNode.getNodeType() ? 1 : 0) != 0);
        ObjectInspector loi = ((ListObjectInspector)oi).getListElementObjectInspector();
        ArrayList<Object> ret = new ArrayList<Object>(rootNode.size());
        Iterator it = rootNode.elements();
        while (it.hasNext()) {
            JsonNode element = (JsonNode)it.next();
            ret.add(this.visitNode(element, loi));
        }
        return ret;
    }

    private Object visitLeafNode(JsonNode leafNode, ObjectInspector oi) throws SerDeException {
        PrimitiveObjectInspector poi = (PrimitiveObjectInspector)oi;
        PrimitiveTypeInfo typeInfo = poi.getTypeInfo();
        if (typeInfo.getPrimitiveCategory() != PrimitiveObjectInspector.PrimitiveCategory.STRING) {
            Preconditions.checkArgument((leafNode.getNodeType() != JsonNodeType.OBJECT ? 1 : 0) != 0);
            Preconditions.checkArgument((leafNode.getNodeType() != JsonNodeType.ARRAY ? 1 : 0) != 0);
        }
        switch (typeInfo.getPrimitiveCategory()) {
            case INT: {
                return leafNode.asInt();
            }
            case BYTE: {
                return (byte)leafNode.asInt();
            }
            case SHORT: {
                return (short)leafNode.asInt();
            }
            case LONG: {
                return leafNode.asLong();
            }
            case BOOLEAN: {
                if ("false".equalsIgnoreCase(leafNode.asText())) {
                    return Boolean.FALSE;
                }
                return leafNode.asBoolean(true);
            }
            case FLOAT: {
                return Float.valueOf((float)leafNode.asDouble());
            }
            case DOUBLE: {
                return leafNode.asDouble();
            }
            case STRING: {
                if (leafNode.isValueNode()) {
                    return leafNode.asText();
                }
                if (this.isEnabled(Feature.STRINGIFY_COMPLEX_FIELDS)) {
                    return leafNode.toString();
                }
                throw new SerDeException("Complex field found in JSON does not match table definition: " + typeInfo.getTypeName() + ", please consider enabling `json.stringify.complex.fields` table property");
            }
            case BINARY: {
                return this.getByteValue(leafNode);
            }
            case DATE: {
                return Date.valueOf((String)leafNode.asText());
            }
            case TIMESTAMP: {
                return this.tsParser.parseTimestamp(leafNode.asText());
            }
            case DECIMAL: {
                HiveDecimal decimal = HiveDecimal.create((String)leafNode.asText());
                return HiveDecimalUtils.enforcePrecisionScale(decimal, (DecimalTypeInfo)typeInfo);
            }
            case TIMESTAMPLOCALTZ: {
                Timestamp ts = this.tsParser.parseTimestamp(leafNode.asText());
                ZoneId zid = ((TimestampLocalTZTypeInfo)typeInfo).timeZone();
                TimestampTZ tstz = new TimestampTZ();
                tstz.set(ts.toEpochSecond(), ts.getNanos(), zid);
                return tstz;
            }
            case VARCHAR: {
                return new HiveVarchar(leafNode.asText(), ((BaseCharTypeInfo)typeInfo).getLength());
            }
            case CHAR: {
                return new HiveChar(leafNode.asText(), ((BaseCharTypeInfo)typeInfo).getLength());
            }
        }
        throw new SerDeException("Could not convert from string to type: " + typeInfo.getTypeName());
    }

    private byte[] getByteValue(JsonNode binaryNode) throws SerDeException {
        try {
            BinaryEncoding binaryEncoding = this.getBinaryEncodingForNode(binaryNode);
            switch (binaryEncoding) {
                case RAWSTRING: {
                    String byteText = binaryNode.asText();
                    if (byteText == null) {
                        return null;
                    }
                    return byteText.getBytes(StandardCharsets.UTF_8);
                }
                case BASE64: {
                    return binaryNode.binaryValue();
                }
            }
            throw new SerDeException("No such binary encoding: " + binaryEncoding);
        }
        catch (IOException e) {
            throw new SerDeException("Error generating JSON binary type from record.", e);
        }
    }

    private BinaryEncoding getBinaryEncodingForNode(JsonNode binaryNode) {
        String jsonValue = binaryNode.textValue();
        if (jsonValue == null || jsonValue.length() % 4 != 0 || !BASE64_PATTERN.matcher(jsonValue).matches()) {
            return BinaryEncoding.RAWSTRING;
        }
        return BinaryEncoding.BASE64;
    }

    private StructField getStructField(StructObjectInspector oi, String fieldName) throws SerDeException {
        int colIndex;
        ImmutablePair pair = ImmutablePair.of((Object)oi, (Object)fieldName);
        if (this.discoveredUnknownFields.contains(pair)) {
            return null;
        }
        StructField structField = this.discoveredFields.get(pair);
        if (structField != null) {
            return structField;
        }
        if (this.isEnabled(Feature.COL_INDEX_PARSING) && (colIndex = this.getColIndex(fieldName)) >= 0) {
            structField = oi.getAllStructFieldRefs().get(colIndex);
        }
        if (structField == null) {
            try {
                structField = oi.getStructFieldRef(fieldName);
            }
            catch (Exception exception) {
                // empty catch block
            }
        }
        if (structField != null) {
            this.discoveredFields.put((Pair<StructObjectInspector, String>)pair, structField);
        } else if (this.isEnabled(Feature.IGNORE_UNKNOWN_FIELDS) && this.discoveredUnknownFields.add((Pair<StructObjectInspector, String>)pair)) {
            LOG.warn("Discovered unknown field: {}. Ignoring.", (Object)fieldName);
        } else {
            throw new SerDeException("Field found in JSON does not match table definition: " + fieldName);
        }
        return structField;
    }

    private int getColIndex(String internalName) {
        Matcher m = this.internalPattern.matcher(internalName);
        return m.matches() ? Integer.parseInt(m.group(1)) : -1;
    }

    public void enable(Feature feature) {
        this.features.add(feature);
    }

    public void disable(Feature feature) {
        this.features.remove((Object)feature);
    }

    public Set<Feature> getFeatures() {
        return Collections.unmodifiableSet(this.features);
    }

    public boolean isEnabled(Feature feature) {
        return this.features.contains((Object)feature);
    }

    public ObjectInspector getObjectInspector() {
        return this.oi;
    }

    public String toString() {
        return "HiveJsonReader [features=" + this.features + ", tsParser=" + this.tsParser;
    }

    public static enum Feature {
        COL_INDEX_PARSING,
        PRIMITIVE_TO_WRITABLE,
        IGNORE_UNKNOWN_FIELDS,
        STRINGIFY_COMPLEX_FIELDS;

    }
}

