/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.server.log.remote.metadata.storage;

import java.io.Closeable;
import java.time.Duration;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.stream.Collectors;
import org.apache.kafka.clients.consumer.Consumer;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.common.TopicIdPartition;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.RetriableException;
import org.apache.kafka.common.errors.WakeupException;
import org.apache.kafka.common.utils.Time;
import org.apache.kafka.server.log.remote.metadata.storage.RemoteLogMetadataTopicPartitioner;
import org.apache.kafka.server.log.remote.metadata.storage.RemotePartitionMetadataEventHandler;
import org.apache.kafka.server.log.remote.metadata.storage.serialization.RemoteLogMetadataSerde;
import org.apache.kafka.server.log.remote.storage.RemoteLogMetadata;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class ConsumerTask
implements Runnable,
Closeable {
    private static final Logger log = LoggerFactory.getLogger(ConsumerTask.class);
    private final RemoteLogMetadataSerde serde = new RemoteLogMetadataSerde();
    private final Consumer<byte[], byte[]> consumer;
    private final RemotePartitionMetadataEventHandler remotePartitionMetadataEventHandler;
    private final RemoteLogMetadataTopicPartitioner topicPartitioner;
    private final long pollTimeoutMs;
    private final Time time;
    private volatile boolean isClosed = false;
    private volatile boolean hasAssignmentChanged = true;
    private final Object assignPartitionsLock = new Object();
    private volatile Set<Integer> assignedMetadataPartitions = Set.of();
    private volatile Map<TopicIdPartition, UserTopicIdPartition> assignedUserTopicIdPartitions = Map.of();
    private volatile Set<TopicIdPartition> processedAssignmentOfUserTopicIdPartitions = Set.of();
    private long uninitializedAt;
    private boolean isAllUserTopicPartitionsInitialized;
    private final Map<Integer, Long> readOffsetsByMetadataPartition = new ConcurrentHashMap<Integer, Long>();
    private final Map<TopicIdPartition, Long> readOffsetsByUserTopicPartition = new HashMap<TopicIdPartition, Long>();
    private Map<TopicPartition, StartAndEndOffsetHolder> offsetHolderByMetadataPartition = new HashMap<TopicPartition, StartAndEndOffsetHolder>();
    private boolean hasLastOffsetsFetchFailed = false;
    private long lastFailedFetchOffsetsTimestamp;
    private final long offsetFetchRetryIntervalMs;

    public ConsumerTask(RemotePartitionMetadataEventHandler remotePartitionMetadataEventHandler, RemoteLogMetadataTopicPartitioner topicPartitioner, Consumer<byte[], byte[]> consumer, long pollTimeoutMs, long offsetFetchRetryIntervalMs, Time time) {
        this.consumer = consumer;
        this.remotePartitionMetadataEventHandler = Objects.requireNonNull(remotePartitionMetadataEventHandler);
        this.topicPartitioner = Objects.requireNonNull(topicPartitioner);
        this.pollTimeoutMs = pollTimeoutMs;
        this.offsetFetchRetryIntervalMs = offsetFetchRetryIntervalMs;
        this.time = Objects.requireNonNull(time);
        this.uninitializedAt = time.milliseconds();
    }

    @Override
    public void run() {
        log.info("Starting consumer task thread.");
        while (!this.isClosed) {
            this.ingestRecords();
        }
        this.closeConsumer();
        log.info("Exited from consumer task thread");
    }

    void ingestRecords() {
        try {
            if (this.hasAssignmentChanged) {
                this.maybeWaitForPartitionAssignments();
            }
            log.trace("Polling consumer to receive remote log metadata topic records");
            ConsumerRecords consumerRecords = this.consumer.poll(Duration.ofMillis(this.pollTimeoutMs));
            for (ConsumerRecord record : consumerRecords) {
                this.processConsumerRecord((ConsumerRecord<byte[], byte[]>)record);
            }
            this.maybeMarkUserPartitionsAsReady();
        }
        catch (WakeupException ex) {
            this.isClosed = true;
        }
        catch (RetriableException ex) {
            log.warn("Retriable error occurred while processing the records. Retrying...", (Throwable)ex);
        }
        catch (Exception ex) {
            this.isClosed = true;
            log.error("Error occurred while processing the records", (Throwable)ex);
        }
    }

    void closeConsumer() {
        try {
            this.consumer.close(Duration.ofSeconds(30L));
        }
        catch (Exception e) {
            log.error("Error encountered while closing the consumer", (Throwable)e);
        }
    }

    private void processConsumerRecord(ConsumerRecord<byte[], byte[]> record) {
        RemoteLogMetadata remoteLogMetadata = this.serde.deserialize((byte[])record.value());
        if (this.shouldProcess(remoteLogMetadata, record.offset())) {
            this.remotePartitionMetadataEventHandler.handleRemoteLogMetadata(remoteLogMetadata);
            this.readOffsetsByUserTopicPartition.put(remoteLogMetadata.topicIdPartition(), record.offset());
        } else {
            log.trace("The event {} is skipped because it is either already processed or not assigned to this consumer", (Object)remoteLogMetadata);
        }
        log.trace("Updating consumed offset: {} for partition {}", (Object)record.offset(), (Object)record.partition());
        this.readOffsetsByMetadataPartition.put(record.partition(), record.offset());
    }

    private boolean shouldProcess(RemoteLogMetadata metadata, long recordOffset) {
        TopicIdPartition tpId = metadata.topicIdPartition();
        Long readOffset = this.readOffsetsByUserTopicPartition.get(tpId);
        return this.processedAssignmentOfUserTopicIdPartitions.contains(tpId) && (readOffset == null || readOffset < recordOffset);
    }

    private void maybeMarkUserPartitionsAsReady() {
        if (this.isAllUserTopicPartitionsInitialized) {
            return;
        }
        this.maybeFetchStartAndEndOffsets();
        boolean isAllInitialized = true;
        for (UserTopicIdPartition utp : this.assignedUserTopicIdPartitions.values()) {
            if (utp.isAssigned && !utp.isInitialized) {
                Integer metadataPartition = utp.metadataPartition;
                StartAndEndOffsetHolder holder = this.offsetHolderByMetadataPartition.get(ConsumerTask.toRemoteLogPartition(metadataPartition));
                if (holder != null) {
                    Long readOffset = this.readOffsetsByMetadataPartition.getOrDefault(metadataPartition, -1L);
                    if (readOffset + 1L >= holder.endOffset || holder.endOffset.equals(holder.startOffset)) {
                        this.markInitialized(utp);
                    } else {
                        log.debug("The user-topic-partition {} could not be marked initialized since the read-offset is {} but the end-offset is {} for the metadata-partition {}", new Object[]{utp, readOffset, holder.endOffset, metadataPartition});
                    }
                } else {
                    log.debug("The offset-holder is null for the metadata-partition {}. The consumer may not have picked up the recent assignment", (Object)metadataPartition);
                }
            }
            isAllInitialized = isAllInitialized && utp.isAssigned && utp.isInitialized;
        }
        if (isAllInitialized) {
            log.info("Initialized for all the {} assigned user-partitions mapped to the {} meta-partitions in {} ms", new Object[]{this.assignedUserTopicIdPartitions.size(), this.assignedMetadataPartitions.size(), this.time.milliseconds() - this.uninitializedAt});
        }
        this.isAllUserTopicPartitionsInitialized = isAllInitialized;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void maybeWaitForPartitionAssignments() throws InterruptedException {
        HashSet<Integer> metadataPartitionSnapshot = new HashSet<Integer>();
        HashSet assignedUserTopicIdPartitionsSnapshot = new HashSet();
        Object object = this.assignPartitionsLock;
        synchronized (object) {
            while (!this.isClosed && this.assignedUserTopicIdPartitions.isEmpty()) {
                log.debug("Waiting for remote log metadata partitions to be assigned");
                this.assignPartitionsLock.wait();
            }
            if (!this.isClosed && this.hasAssignmentChanged) {
                this.assignedUserTopicIdPartitions.values().forEach(utp -> {
                    metadataPartitionSnapshot.add(utp.metadataPartition);
                    assignedUserTopicIdPartitionsSnapshot.add(utp);
                });
                this.hasAssignmentChanged = false;
            }
        }
        if (!metadataPartitionSnapshot.isEmpty()) {
            Set<TopicPartition> remoteLogPartitions = ConsumerTask.toRemoteLogPartitions(metadataPartitionSnapshot);
            this.consumer.assign(remoteLogPartitions);
            this.assignedMetadataPartitions = Collections.unmodifiableSet(metadataPartitionSnapshot);
            Set seekToBeginOffsetPartitions = assignedUserTopicIdPartitionsSnapshot.stream().filter(utp -> !utp.isAssigned).map(utp -> utp.metadataPartition).peek(this.readOffsetsByMetadataPartition::remove).map(ConsumerTask::toRemoteLogPartition).collect(Collectors.toSet());
            this.consumer.seekToBeginning(seekToBeginOffsetPartitions);
            remoteLogPartitions.stream().filter(tp -> !seekToBeginOffsetPartitions.contains(tp) && this.readOffsetsByMetadataPartition.containsKey(tp.partition())).forEach(tp -> this.consumer.seek(tp, this.readOffsetsByMetadataPartition.get(tp.partition()).longValue()));
            HashSet<TopicIdPartition> processedAssignmentPartitions = new HashSet<TopicIdPartition>();
            assignedUserTopicIdPartitionsSnapshot.forEach(utp -> {
                if (!utp.isAssigned) {
                    this.remotePartitionMetadataEventHandler.maybeLoadPartition(utp.topicIdPartition);
                    utp.isAssigned = true;
                }
                processedAssignmentPartitions.add(utp.topicIdPartition);
            });
            this.processedAssignmentOfUserTopicIdPartitions = new HashSet<TopicIdPartition>(processedAssignmentPartitions);
            this.clearResourcesForUnassignedUserTopicPartitions(processedAssignmentPartitions);
            this.isAllUserTopicPartitionsInitialized = false;
            this.uninitializedAt = this.time.milliseconds();
            this.fetchStartAndEndOffsets();
        }
    }

    private void clearResourcesForUnassignedUserTopicPartitions(Set<TopicIdPartition> assignedPartitions) {
        Set<TopicIdPartition> unassignedPartitions = this.readOffsetsByUserTopicPartition.keySet().stream().filter(e -> !assignedPartitions.contains(e)).collect(Collectors.toSet());
        unassignedPartitions.forEach(unassignedPartition -> {
            this.remotePartitionMetadataEventHandler.clearTopicPartition((TopicIdPartition)unassignedPartition);
            this.readOffsetsByUserTopicPartition.remove(unassignedPartition);
        });
        log.info("Unassigned user-topic-partitions: {}", (Object)unassignedPartitions.size());
    }

    void addAssignmentsForPartitions(Set<TopicIdPartition> partitions) {
        this.updateAssignments(Objects.requireNonNull(partitions), Set.of());
    }

    void removeAssignmentsForPartitions(Set<TopicIdPartition> partitions) {
        this.updateAssignments(Set.of(), Objects.requireNonNull(partitions));
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void updateAssignments(Set<TopicIdPartition> addedPartitions, Set<TopicIdPartition> removedPartitions) {
        log.info("Updating assignments for partitions added: {} and removed: {}", addedPartitions, removedPartitions);
        if (!addedPartitions.isEmpty() || !removedPartitions.isEmpty()) {
            Object object = this.assignPartitionsLock;
            synchronized (object) {
                HashMap<TopicIdPartition, UserTopicIdPartition> updatedUserPartitions = new HashMap<TopicIdPartition, UserTopicIdPartition>(this.assignedUserTopicIdPartitions);
                addedPartitions.forEach(tpId -> updatedUserPartitions.putIfAbsent((TopicIdPartition)tpId, this.newUserTopicIdPartition((TopicIdPartition)tpId)));
                removedPartitions.forEach(updatedUserPartitions::remove);
                if (!updatedUserPartitions.equals(this.assignedUserTopicIdPartitions)) {
                    this.assignedUserTopicIdPartitions = Collections.unmodifiableMap(updatedUserPartitions);
                    this.hasAssignmentChanged = true;
                    log.debug("Assigned user-topic-partitions: {}", this.assignedUserTopicIdPartitions);
                    this.assignPartitionsLock.notifyAll();
                }
            }
        }
    }

    Optional<Long> readOffsetForMetadataPartition(int partition) {
        return Optional.ofNullable(this.readOffsetsByMetadataPartition.get(partition));
    }

    boolean isMetadataPartitionAssigned(int partition) {
        return this.assignedMetadataPartitions.contains(partition);
    }

    boolean isUserPartitionAssigned(TopicIdPartition partition) {
        UserTopicIdPartition utp = this.assignedUserTopicIdPartitions.get(partition);
        return utp != null && utp.isAssigned;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        if (!this.isClosed) {
            log.info("Closing the instance");
            Object object = this.assignPartitionsLock;
            synchronized (object) {
                this.isClosed = true;
                this.assignedUserTopicIdPartitions.values().forEach(this::markInitialized);
                this.consumer.wakeup();
                this.assignPartitionsLock.notifyAll();
            }
        }
    }

    Set<Integer> metadataPartitionsAssigned() {
        return Collections.unmodifiableSet(this.assignedMetadataPartitions);
    }

    private void fetchStartAndEndOffsets() {
        try {
            Set<TopicPartition> uninitializedPartitions = this.assignedUserTopicIdPartitions.values().stream().filter(utp -> utp.isAssigned && !utp.isInitialized).map(utp -> ConsumerTask.toRemoteLogPartition(utp.metadataPartition)).collect(Collectors.toSet());
            uninitializedPartitions.forEach(tp -> this.offsetHolderByMetadataPartition.remove(tp));
            if (!uninitializedPartitions.isEmpty()) {
                Map endOffsets = this.consumer.endOffsets(uninitializedPartitions);
                Map startOffsets = this.consumer.beginningOffsets(uninitializedPartitions);
                this.offsetHolderByMetadataPartition = endOffsets.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, e -> new StartAndEndOffsetHolder((Long)startOffsets.get(e.getKey()), (Long)e.getValue())));
            }
            this.hasLastOffsetsFetchFailed = false;
        }
        catch (RetriableException ex) {
            this.hasLastOffsetsFetchFailed = true;
            this.lastFailedFetchOffsetsTimestamp = this.time.milliseconds();
        }
    }

    private void maybeFetchStartAndEndOffsets() {
        if (this.hasLastOffsetsFetchFailed && this.lastFailedFetchOffsetsTimestamp + this.offsetFetchRetryIntervalMs < this.time.milliseconds()) {
            this.fetchStartAndEndOffsets();
        }
    }

    private UserTopicIdPartition newUserTopicIdPartition(TopicIdPartition tpId) {
        return new UserTopicIdPartition(tpId, this.topicPartitioner.metadataPartition(tpId));
    }

    private void markInitialized(UserTopicIdPartition utp) {
        if (!utp.isAssigned) {
            log.warn("Tried to initialize a UTP: {} that was not yet assigned!", (Object)utp);
            return;
        }
        if (!utp.isInitialized) {
            this.remotePartitionMetadataEventHandler.markInitialized(utp.topicIdPartition);
            utp.isInitialized = true;
        }
    }

    static Set<TopicPartition> toRemoteLogPartitions(Set<Integer> partitions) {
        return partitions.stream().map(ConsumerTask::toRemoteLogPartition).collect(Collectors.toSet());
    }

    static TopicPartition toRemoteLogPartition(int partition) {
        return new TopicPartition("__remote_log_metadata", partition);
    }

    static class UserTopicIdPartition {
        private final TopicIdPartition topicIdPartition;
        private final Integer metadataPartition;
        boolean isInitialized;
        boolean isAssigned;

        public UserTopicIdPartition(TopicIdPartition tpId, Integer metadataPartition) {
            this.topicIdPartition = Objects.requireNonNull(tpId);
            this.metadataPartition = Objects.requireNonNull(metadataPartition);
            this.isInitialized = false;
            this.isAssigned = false;
        }

        public String toString() {
            return "UserTopicIdPartition{topicIdPartition=" + String.valueOf(this.topicIdPartition) + ", metadataPartition=" + this.metadataPartition + ", isInitialized=" + this.isInitialized + ", isAssigned=" + this.isAssigned + "}";
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            UserTopicIdPartition that = (UserTopicIdPartition)o;
            return this.topicIdPartition.equals((Object)that.topicIdPartition) && this.metadataPartition.equals(that.metadataPartition);
        }

        public int hashCode() {
            return Objects.hash(this.topicIdPartition, this.metadataPartition);
        }
    }

    static class StartAndEndOffsetHolder {
        Long startOffset;
        Long endOffset;

        public StartAndEndOffsetHolder(Long startOffset, Long endOffset) {
            this.startOffset = startOffset;
            this.endOffset = endOffset;
        }

        public String toString() {
            return "StartAndEndOffsetHolder{startOffset=" + this.startOffset + ", endOffset=" + this.endOffset + "}";
        }
    }
}

