/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.qpid.protonj2.codec.encoders.transport;

import org.apache.qpid.protonj2.buffer.ProtonBuffer;
import org.apache.qpid.protonj2.codec.Encoder;
import org.apache.qpid.protonj2.codec.EncoderState;
import org.apache.qpid.protonj2.codec.EncodingCodes;
import org.apache.qpid.protonj2.codec.encoders.AbstractDescribedListTypeEncoder;
import org.apache.qpid.protonj2.types.Symbol;
import org.apache.qpid.protonj2.types.UnsignedLong;
import org.apache.qpid.protonj2.types.transport.Transfer;

/**
 * Encoder of AMQP Transfer type values to a byte stream.
 */
public final class TransferTypeEncoder extends AbstractDescribedListTypeEncoder<Transfer> {

    @Override
    public UnsignedLong getDescriptorCode() {
        return Transfer.DESCRIPTOR_CODE;
    }

    @Override
    public Symbol getDescriptorSymbol() {
        return Transfer.DESCRIPTOR_SYMBOL;
    }

    @Override
    public Class<Transfer> getTypeClass() {
        return Transfer.class;
    }

    /*
     * This assumes that the value was already check be the setter in Transfer
     */
    private static void writeCheckedUnsignedInteger(final long value, final ProtonBuffer buffer) {
        if (value == 0) {
            buffer.writeByte(EncodingCodes.UINT0);
        } else if (value <= 255) {
            buffer.writeByte(EncodingCodes.SMALLUINT);
            buffer.writeByte((byte) value);
        } else {
            buffer.writeByte(EncodingCodes.UINT);
            buffer.writeInt((int) value);
        }
    }

    @Override
    public void writeElement(Transfer transfer, int index, ProtonBuffer buffer, Encoder encoder, EncoderState state) {
        if (transfer.hasElement(index)) {
            switch (index) {
                case 0:
                    writeCheckedUnsignedInteger(transfer.getHandle(), buffer);
                    break;
                case 1:
                    writeCheckedUnsignedInteger(transfer.getDeliveryId(), buffer);
                    break;
                case 2:
                    encoder.writeDeliveryTag(buffer, state, transfer.getDeliveryTag());
                    break;
                case 3:
                    writeCheckedUnsignedInteger(transfer.getMessageFormat(), buffer);
                    break;
                case 4:
                    buffer.writeByte(transfer.getSettled() ? EncodingCodes.BOOLEAN_TRUE : EncodingCodes.BOOLEAN_FALSE);
                    break;
                case 5:
                    buffer.writeByte(transfer.getMore() ? EncodingCodes.BOOLEAN_TRUE : EncodingCodes.BOOLEAN_FALSE);
                    break;
                case 6:
                    buffer.writeByte(EncodingCodes.UBYTE);
                    buffer.writeByte(transfer.getRcvSettleMode().byteValue());
                    break;
                case 7:
                    encoder.writeObject(buffer, state, transfer.getState());
                    break;
                case 8:
                    buffer.writeByte(transfer.getResume() ? EncodingCodes.BOOLEAN_TRUE : EncodingCodes.BOOLEAN_FALSE);
                    break;
                case 9:
                    buffer.writeByte(transfer.getAborted() ? EncodingCodes.BOOLEAN_TRUE : EncodingCodes.BOOLEAN_FALSE);
                    break;
                case 10:
                    buffer.writeByte(transfer.getBatchable() ? EncodingCodes.BOOLEAN_TRUE : EncodingCodes.BOOLEAN_FALSE);
                    break;
                default:
                    throw new IllegalArgumentException("Unknown Transfer value index: " + index);
            }
        } else {
            buffer.writeByte(EncodingCodes.NULL);
        }
    }

    @Override
    public byte getListEncoding(Transfer value) {
        if (value.getState() != null) {
            return EncodingCodes.LIST32;
        } else if (value.getDeliveryTag() != null && value.getDeliveryTag().tagLength() > 200) {
            return EncodingCodes.LIST32;
        } else {
            return EncodingCodes.LIST8;
        }
    }

    @Override
    public int getElementCount(Transfer transfer) {
        return transfer.getElementCount();
    }

    @Override
    public int getMinElementCount() {
        return 1;
    }
}
