/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.sling.resourceresolver.impl.mapping;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import org.apache.commons.lang3.time.StopWatch;
import org.apache.sling.api.resource.LoginException;
import org.apache.sling.api.resource.QuerySyntaxException;
import org.apache.sling.api.resource.Resource;
import org.apache.sling.api.resource.ResourceResolver;
import org.apache.sling.api.resource.ResourceUtil;
import org.apache.sling.resourceresolver.impl.ResourceResolverImpl;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * All things related to the handling of aliases.
 */
class AliasHandler {

    private static final String JCR_CONTENT = "jcr:content";

    private static final String JCR_CONTENT_PREFIX = JCR_CONTENT + "/";

    private static final String JCR_CONTENT_SUFFIX = "/" + JCR_CONTENT;

    private static final String SERVICE_USER = "mapping";

    private MapConfigurationProvider factory;

    private final ReentrantLock initializing;

    private final Logger log = LoggerFactory.getLogger(AliasHandler.class);

    // keep track of some defunct aliases for diagnostics (thus size-limited)
    private static final int MAX_REPORT_DEFUNCT_ALIASES = 50;

    private final Runnable doUpdateConfiguration;
    private final Runnable sendChangeEvent;
    private final Consumer<String> drain;

    // static value for the case when cache is not (yet) not initialized
    private static final Map<String, Map<String, Collection<String>>> UNITIALIZED_MAP = Collections.emptyMap();

    /**
     * The key of the map is the parent path, while the value is a map with the
     * resource name as key and the actual aliases as values.
     * <p>
     * The only way this map changes away from {@link #UNITIALIZED_MAP} is when
     * alias initialization finished successfully.
     */
    // TODO: check for potential concurrency issues (SLING-12771)
    @NotNull
    Map<String, Map<String, Collection<String>>> aliasMapsMap = UNITIALIZED_MAP;

    final AtomicLong aliasResourcesOnStartup;
    final AtomicLong detectedConflictingAliases;
    final AtomicLong detectedInvalidAliases;

    private final AtomicBoolean aliasesProcessed = new AtomicBoolean(false);

    public AliasHandler(
            @NotNull MapConfigurationProvider factory,
            @NotNull ReentrantLock initializing,
            @NotNull Runnable doUpdateConfiguration,
            @NotNull Runnable sendChangeEvent,
            @NotNull Consumer<String> drain) {
        this.factory = factory;
        this.initializing = initializing;
        this.doUpdateConfiguration = doUpdateConfiguration;
        this.sendChangeEvent = sendChangeEvent;
        this.drain = drain;

        this.aliasResourcesOnStartup = new AtomicLong(0);
        this.detectedConflictingAliases = new AtomicLong(0);
        this.detectedInvalidAliases = new AtomicLong(0);
    }

    public boolean isReady() {
        return this.aliasesProcessed.get();
    }

    public void dispose() {
        this.factory = null;
    }

    /**
     * Actual initializer. Guards itself against concurrent use by using a
     * ReentrantLock. Does nothing if the resource resolver has already been
     * null-ed.
     */
    protected void initializeAliases() {

        this.initializing.lock();

        // as this can be called multiple times, we need to reset
        // the map here
        this.aliasMapsMap = UNITIALIZED_MAP;

        // already disposed?
        if (this.factory == null) {
            log.error("Can't initialize aliases when MapConfigurationProvider is null");
            return;
        }

        log.info(
                "Initializing Aliases ({}={}, {}={}, {}={})",
                "alias_cache_in_background",
                this.factory.isAliasCacheInitInBackground(),
                "optimize_alias_resolution",
                this.factory.isOptimizeAliasResolutionEnabled(),
                "allowed_alias_locations",
                this.factory.getAllowedAliasLocations());

        try {
            aliasesProcessed.set(false);

            if (this.factory.isOptimizeAliasResolutionEnabled()) {
                AliasInitializer ai = new AliasInitializer();
                if (this.factory.isAliasCacheInitInBackground()) {
                    this.log.debug("starting alias initialization in the background");
                    Thread aiinit = new Thread(ai, "AliasInitializer");
                    aiinit.start();
                } else {
                    ai.run();
                }
            }

            doUpdateConfiguration.run();
            sendChangeEvent.run();
        } finally {
            this.initializing.unlock();
        }
    }

    private class AliasInitializer implements Runnable {

        @Override
        public void run() {
            try {
                execute();
            } catch (Exception ex) {
                log.error("alias initializer thread terminated with an exception", ex);
            }
        }

        private void execute() {
            try (ResourceResolver resolver =
                    factory.getServiceResourceResolver(factory.getServiceUserAuthenticationInfo(SERVICE_USER))) {
                List<String> conflictingAliases = new ArrayList<>();
                List<String> invalidAliases = new ArrayList<>();
                StringBuilder diagnostics = new StringBuilder();

                StopWatch sw = StopWatch.createStarted();
                log.debug("alias initialization - start");

                aliasMapsMap = loadAliases(resolver, conflictingAliases, invalidAliases, diagnostics);

                // process pending events
                AliasHandler.this.drain.accept("draining alias event queue (during cache initialization)");

                aliasesProcessed.set(true);

                // drain once more in case more events have arrived
                AliasHandler.this.drain.accept("draining alias event queue (after cache initialization)");

                String message = MapEntries.getTimingMessage(
                        "alias initialization - completed", sw.getDuration(), aliasResourcesOnStartup.get());

                log.info(message);
            } catch (Exception ex) {
                log.error("Alias init failed", ex);
                aliasMapsMap = UNITIALIZED_MAP;
            }
        }

        /**
         * Load aliases - Search for all nodes (except under /jcr:system) below
         * configured alias locations having the sling:alias property
         */
        @NotNull
        private Map<String, Map<String, Collection<String>>> loadAliases(
                @NotNull ResourceResolver resolver,
                @NotNull List<String> conflictingAliases,
                @NotNull List<String> invalidAliases,
                @NotNull StringBuilder diagnostics) {

            Map<String, Map<String, Collection<String>>> map = new ConcurrentHashMap<>();

            String baseQueryString = generateAliasQuery();

            Iterator<Resource> it;
            try {
                String queryStringWithSort =
                        baseQueryString + " AND FIRST([sling:alias]) >= '%s' ORDER BY FIRST([sling:alias])";
                it = new PagedQueryIterator("alias", "sling:alias", resolver, queryStringWithSort, 2000);
            } catch (QuerySyntaxException ex) {
                log.debug("sort with first() not supported, falling back to base query", ex);
                it = queryUnpaged(baseQueryString, resolver);
            } catch (UnsupportedOperationException ex) {
                log.debug("query failed as unsupported, retrying without paging/sorting", ex);
                it = queryUnpaged(baseQueryString, resolver);
            }

            long count = 0;
            while (it.hasNext()) {
                count += 1;
                loadAlias(it.next(), map, conflictingAliases, invalidAliases);
            }

            if (it instanceof PagedQueryIterator) {
                PagedQueryIterator pit = (PagedQueryIterator) it;

                if (!pit.getWarning().isEmpty()) {
                    log.warn(pit.getWarning());
                }

                diagnostics.append(pit.getStatistics());
            }

            // warn if there are more than a few defunct aliases
            if (conflictingAliases.size() >= MAX_REPORT_DEFUNCT_ALIASES) {
                log.warn(
                        "There are {} conflicting aliases; excerpt: {}", conflictingAliases.size(), conflictingAliases);
            } else if (!conflictingAliases.isEmpty()) {
                log.warn("There are {} conflicting aliases: {}", conflictingAliases.size(), conflictingAliases);
            }

            if (invalidAliases.size() >= MAX_REPORT_DEFUNCT_ALIASES) {
                log.warn("There are {} invalid aliases; excerpt: {}", invalidAliases.size(), invalidAliases);
            } else if (!invalidAliases.isEmpty()) {
                log.warn("There are {} invalid aliases: {}", invalidAliases.size(), invalidAliases);
            }

            aliasResourcesOnStartup.set(count);

            return map;
        }

        /*
         * generate alias query based on configured alias locations
         */
        @NotNull
        private String generateAliasQuery() {
            Set<String> allowedLocations = factory.getAllowedAliasLocations();

            StringBuilder baseQuery = new StringBuilder("SELECT [sling:alias] FROM [nt:base] WHERE ");

            if (allowedLocations.isEmpty()) {
                baseQuery.append(QueryBuildHelper.excludeSystemPath());
            } else {
                baseQuery.append(allowedLocations.stream()
                        .map(location -> "isdescendantnode('" + QueryBuildHelper.escapeString(location) + "')")
                        .collect(Collectors.joining(" OR ", "(", ")")));
            }

            baseQuery.append(" AND [sling:alias] IS NOT NULL");
            return baseQuery.toString();
        }

        @NotNull
        private Iterator<Resource> queryUnpaged(@NotNull String query, @NotNull ResourceResolver resolver) {
            log.debug("start alias query: {}", query);
            StopWatch sw = StopWatch.createStarted();
            Iterator<Resource> it = resolver.findResources(query, "JCR-SQL2");
            log.debug(
                    "end alias query; elapsed {} ({}ms)",
                    sw.getDuration(),
                    sw.getDuration().toMillis());
            return it;
        }
    }

    boolean usesCache() {
        return this.aliasMapsMap != UNITIALIZED_MAP;
    }

    boolean doAddAlias(@NotNull Resource resource) {
        if (usesCache()) {
            return loadAlias(resource, this.aliasMapsMap, null, null);
        } else {
            return false;
        }
    }

    /**
     * Remove all aliases for the content path
     *
     * @param contentPath The content path
     * @param path        Optional sub path of the vanity path
     * @return {@code true} if a change happened
     */
    boolean removeAlias(
            @Nullable ResourceResolver resolver,
            @NotNull String contentPath,
            @Nullable String path,
            @NotNull Runnable notifyOfChange) {
        if (usesCache()) {
            return removeAliasInMap(resolver, contentPath, path, notifyOfChange);
        } else {
            return false;
        }
    }

    private boolean removeAliasInMap(
            @Nullable ResourceResolver resolver,
            @NotNull String contentPath,
            @Nullable String path,
            @NotNull Runnable notifyOfChange) {

        String resourcePath = computeResourcePath(contentPath, path);

        if (resourcePath == null) {
            // early exit
            return false;
        }

        this.initializing.lock();

        try {
            Map<String, Collection<String>> aliasMapEntry = aliasMapsMap.get(contentPath);
            if (aliasMapEntry != null) {
                notifyOfChange.run();
                handleAliasRemoval(resolver, contentPath, resourcePath, aliasMapEntry);
            }
            return aliasMapEntry != null;
        } finally {
            this.initializing.unlock();
        }
    }

    // if path is specified we first need to find out if it is
    // a direct child of content path but not jcr:content, or a jcr:content child of a direct child
    // otherwise we can discard the event
    private static @Nullable String computeResourcePath(@NotNull String contentPath, @Nullable String path) {
        String resourcePath = null;

        if (path != null && path.length() > contentPath.length()) {
            // path -> (contentPath + subPath)
            String subPath = path.substring(contentPath.length() + 1);
            int firstSlash = subPath.indexOf('/');

            if (firstSlash == -1) {
                // no slash in subPath
                if (!subPath.equals(JCR_CONTENT)) {
                    resourcePath = path;
                }
            } else if (subPath.lastIndexOf('/') == firstSlash) {
                // exactly one slash in subPath
                if (!subPath.startsWith(JCR_CONTENT_PREFIX) && subPath.endsWith(JCR_CONTENT_SUFFIX)) {
                    resourcePath = ResourceUtil.getParent(path);
                }
            }
        } else {
            resourcePath = contentPath;
        }

        return resourcePath;
    }

    private void handleAliasRemoval(
            @Nullable ResourceResolver resolver,
            @NotNull String contentPath,
            @NotNull String resourcePath,
            @NotNull Map<String, Collection<String>> aliasMapEntry) {
        String prefix = contentPath.endsWith("/") ? contentPath : contentPath + "/";
        if (aliasMapEntry.entrySet().removeIf(e -> (prefix + e.getKey()).startsWith(resourcePath))
                && (aliasMapEntry.isEmpty())) {
            this.aliasMapsMap.remove(contentPath);
        }

        Resource containingResource = resolver != null ? resolver.getResource(resourcePath) : null;

        if (containingResource != null) {
            if (containingResource.getValueMap().containsKey(ResourceResolverImpl.PROP_ALIAS)) {
                doAddAlias(containingResource);
            }
            Resource child = containingResource.getChild(JCR_CONTENT);
            if (child != null && child.getValueMap().containsKey(ResourceResolverImpl.PROP_ALIAS)) {
                doAddAlias(child);
            }
        }
    }

    /**
     * Update alias from a resource
     *
     * @param resource The resource
     * @return {@code true} if any change
     */
    boolean doUpdateAlias(@NotNull Resource resource) {
        if (usesCache()) {
            return doUpdateAliasInMap(resource);
        } else {
            return false;
        }
    }

    private boolean doUpdateAliasInMap(@NotNull Resource resource) {

        // resource to which the alias applies
        Resource containingResource = getResourceToBeAliased(resource);

        if (containingResource != null) {
            String containingResourceName = containingResource.getName();
            String parentPath = ResourceUtil.getParent(containingResource.getPath());

            Map<String, Collection<String>> aliasMapEntry = parentPath == null ? null : aliasMapsMap.get(parentPath);
            if (aliasMapEntry != null) {
                aliasMapEntry.remove(containingResourceName);
                if (aliasMapEntry.isEmpty()) {
                    this.aliasMapsMap.remove(parentPath);
                }
            }

            boolean changed = aliasMapEntry != null;

            if (containingResource.getValueMap().containsKey(ResourceResolverImpl.PROP_ALIAS)) {
                changed |= doAddAlias(containingResource);
            }
            Resource child = containingResource.getChild(JCR_CONTENT);
            if (child != null && child.getValueMap().containsKey(ResourceResolverImpl.PROP_ALIAS)) {
                changed |= doAddAlias(child);
            }

            return changed;
        } else {
            log.warn("containingResource is null for alias on {}, skipping.", resource.getPath());
            return false;
        }
    }

    public @NotNull Map<String, Collection<String>> getAliasMap(@Nullable String parentPath) {
        Map<String, Collection<String>> result =
                usesCache() ? getAliasMapFromCache(parentPath) : getAliasMapFromRepo(parentPath);
        return result != null ? result : Collections.emptyMap();
    }

    public @NotNull Map<String, Collection<String>> getAliasMap(@NotNull Resource parent) {
        Map<String, Collection<String>> result =
                usesCache() ? getAliasMapFromCache(parent.getPath()) : getAliasMapFromRepo(parent);
        return result != null ? result : Collections.emptyMap();
    }

    private @Nullable Map<String, Collection<String>> getAliasMapFromCache(@Nullable String parentPath) {
        return aliasMapsMap.get(parentPath);
    }

    private @Nullable Map<String, Collection<String>> getAliasMapFromRepo(@Nullable String parentPath) {

        if (parentPath == null) {
            return null;
        } else {
            try (ResourceResolver resolver =
                    factory.getServiceResourceResolver(factory.getServiceUserAuthenticationInfo(SERVICE_USER))) {

                return getAliasMapFromRepo(resolver.getResource(parentPath));
            } catch (LoginException ex) {
                log.error("Could not obtain resolver to resolve any aliases from repository", ex);
                return null;
            }
        }
    }

    private @Nullable Map<String, Collection<String>> getAliasMapFromRepo(@Nullable Resource parent) {

        Map<String, Collection<String>> result = null;

        if (parent != null) {
            Map<String, Map<String, Collection<String>>> localMap = new HashMap<>();
            List<String> throwAwayDiagnostics = new ArrayList<>();
            for (Resource child : parent.getChildren()) {
                // ignore jcr:content nodes, they get special treatment
                if (!JCR_CONTENT.equals(child.getName())) {
                    loadAlias(child, localMap, throwAwayDiagnostics, throwAwayDiagnostics);
                    Resource childContentNode = child.getChild(JCR_CONTENT);
                    // check for jcr:content child node...
                    if (childContentNode != null) {
                        // and apply its aliases to the parent node
                        loadAlias(childContentNode, localMap, throwAwayDiagnostics, throwAwayDiagnostics);
                    }
                }
            }
            result = localMap.get(parent.getPath());
        }

        return result;
    }

    /**
     * Load alias given a resource
     */
    private boolean loadAlias(
            @NotNull Resource resource,
            @NotNull Map<String, Map<String, Collection<String>>> map,
            @Nullable List<String> conflictingAliases,
            @Nullable List<String> invalidAliases) {

        // resource containing the alias (i.e., the parent of a jcr:content node, otherwise itself)
        Resource containingResource = getResourceToBeAliased(resource);

        if (containingResource == null) {
            log.warn("containingResource is null for alias on {}, skipping.", resource.getPath());
            return false;
        } else {
            // we read the aliases from the resource given in the method call parameters
            String[] aliasArray = resource.getValueMap().get(ResourceResolverImpl.PROP_ALIAS, String[].class);
            if (aliasArray == null) {
                return false;
            } else {
                // but apply them to the containing resource

                String parentPath = ResourceUtil.getParent(containingResource.getPath());

                if (parentPath == null) {
                    log.debug("the root path cannot have aliases");
                    return false;
                } else {
                    return loadAliasFromArray(
                            aliasArray,
                            map,
                            conflictingAliases,
                            invalidAliases,
                            containingResource.getName(),
                            parentPath);
                }
            }
        }
    }

    /**
     * Load alias given an alias array, return success flag.
     */
    private boolean loadAliasFromArray(
            @Nullable String[] aliasArray,
            @NotNull Map<String, Map<String, Collection<String>>> map,
            @Nullable List<String> conflictingAliases,
            @Nullable List<String> invalidAliases,
            @NotNull String resourceName,
            @NotNull String parentPath) {

        boolean hasAlias = false;

        log.debug("Found alias, total size {}", aliasArray.length);

        // the order matters here, the first alias in the array must come first
        for (String alias : aliasArray) {
            if (isAliasInvalid(alias)) {
                long invalids = detectedInvalidAliases.incrementAndGet();
                log.warn(
                        "Encountered invalid alias '{}' under parent path '{}' (total so far: {}). Refusing to use it.",
                        alias,
                        parentPath,
                        invalids);
                if (invalidAliases != null && invalids < MAX_REPORT_DEFUNCT_ALIASES) {
                    invalidAliases.add((String.format("'%s'/'%s'", parentPath, alias)));
                }
            } else {
                Map<String, Collection<String>> parentMap =
                        map.computeIfAbsent(parentPath, key -> new ConcurrentHashMap<>());
                Optional<String> siblingResourceNameWithDuplicateAlias = parentMap.entrySet().stream()
                        .filter(entry -> !entry.getKey().equals(resourceName)) // ignore entry for the current resource
                        .filter(entry -> entry.getValue().contains(alias))
                        .findFirst()
                        .map(Map.Entry::getKey);
                if (siblingResourceNameWithDuplicateAlias.isPresent()) {
                    long conflicting = detectedConflictingAliases.incrementAndGet();
                    log.warn(
                            "Encountered duplicate alias '{}' under parent path '{}'. Refusing to replace current target '{}' with '{}' (total duplicated aliases so far: {}).",
                            alias,
                            parentPath,
                            siblingResourceNameWithDuplicateAlias.get(),
                            resourceName,
                            conflicting);
                    if (conflictingAliases != null && conflicting < MAX_REPORT_DEFUNCT_ALIASES) {
                        conflictingAliases.add((String.format(
                                "'%s': '%s'/'%s' vs '%s'/'%s'",
                                parentPath, resourceName, alias, siblingResourceNameWithDuplicateAlias.get(), alias)));
                    }
                } else {
                    Collection<String> existingAliases =
                            parentMap.computeIfAbsent(resourceName, name -> new CopyOnWriteArrayList<>());
                    existingAliases.add(alias);
                    hasAlias = true;
                }
            }
        }

        return hasAlias;
    }

    /**
     * Given a resource, check whether the name is "jcr:content", in which case return the parent resource
     *
     * @param resource resource to check
     * @return parent of jcr:content resource (can be null), otherwise the resource itself
     */
    @Nullable
    private Resource getResourceToBeAliased(@NotNull Resource resource) {
        if (JCR_CONTENT.equals(resource.getName())) {
            return resource.getParent();
        } else {
            return resource;
        }
    }

    /**
     * Check alias syntax
     */
    private boolean isAliasInvalid(@Nullable String alias) {
        boolean invalid;
        if (alias == null) {
            invalid = true;
        } else {
            invalid = alias.equals("..") || alias.equals(".") || alias.isEmpty();
            if (!invalid) {
                for (char c : alias.toCharArray()) {
                    // invalid if / or # or a ?
                    if (c == '/' || c == '#' || c == '?') {
                        invalid = true;
                        break;
                    }
                }
            }
        }
        return invalid;
    }
}
