"use strict";
/*
 *  Licensed to the Apache Software Foundation (ASF) under one
 *  or more contributor license agreements.  See the NOTICE file
 *  distributed with this work for additional information
 *  regarding copyright ownership.  The ASF licenses this file
 *  to you under the Apache License, Version 2.0 (the
 *  "License"); you may not use this file except in compliance
 *  with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing,
 *  software distributed under the License is distributed on an
 *  "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 *  KIND, either express or implied.  See the License for the
 *  specific language governing permissions and limitations
 *  under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.Path = exports.Property = exports.VertexProperty = exports.Edge = exports.Vertex = exports.Graph = void 0;
/**
 * @author Jorge Bay Gondra
 */
const graph_traversal_js_1 = require("../process/graph-traversal.cjs");
const traversal_strategy_js_1 = require("../process/traversal-strategy.cjs");
/**
 * An "empty" graph object to server only as a reference.
 */
class Graph {
    /**
     * Returns the graph traversal source.
     * @param TraversalSourceClass The constructor to use for the {@code GraphTraversalSource} instance.
     * @deprecated As of release 3.3.5, replaced by the traversal() anonymous function.
     */
    traversal(TraversalSourceClass = graph_traversal_js_1.GraphTraversalSource) {
        return new TraversalSourceClass(this, new traversal_strategy_js_1.TraversalStrategies());
    }
    toString() {
        return 'graph[]';
    }
}
exports.Graph = Graph;
class Element {
    id;
    label;
    constructor(id, label) {
        this.id = id;
        this.label = label;
    }
    /**
     * Compares this instance to another and determines if they can be considered as equal.
     */
    equals(other) {
        return other instanceof Element && this.id === other.id;
    }
}
class Vertex extends Element {
    properties;
    constructor(id, label, properties) {
        super(id, label);
        this.properties = properties;
    }
    toString() {
        return `v[${this.id}]`;
    }
}
exports.Vertex = Vertex;
class Edge extends Element {
    outV;
    label;
    inV;
    properties;
    constructor(id, outV, label, inV, properties = {}) {
        super(id, label);
        this.outV = outV;
        this.label = label;
        this.inV = inV;
        this.properties = properties;
        if (properties) {
            const keys = Object.keys(properties);
            for (let i = 0; i < keys.length; i++) {
                const k = keys[i];
                this.properties[k] = properties[k].value;
            }
        }
    }
    toString() {
        const outVId = this.outV ? this.outV.id : '?';
        const inVId = this.inV ? this.inV.id : '?';
        return `e[${this.id}][${outVId}-${this.label}->${inVId}]`;
    }
}
exports.Edge = Edge;
class VertexProperty extends Element {
    value;
    properties;
    key;
    constructor(id, label, value, properties) {
        super(id, label);
        this.value = value;
        this.properties = properties;
        this.value = value;
        this.key = this.label;
        this.properties = properties;
    }
    toString() {
        return `vp[${this.label}->${summarize(this.value)}]`;
    }
}
exports.VertexProperty = VertexProperty;
class Property {
    key;
    value;
    constructor(key, value) {
        this.key = key;
        this.value = value;
    }
    toString() {
        return `p[${this.key}->${summarize(this.value)}]`;
    }
    equals(other) {
        return other instanceof Property && this.key === other.key && this.value === other.value;
    }
}
exports.Property = Property;
/**
 * Represents a walk through a graph as defined by a traversal.
 */
class Path {
    labels;
    objects;
    constructor(labels, objects) {
        this.labels = labels;
        this.objects = objects;
    }
    toString() {
        return `path[${(this.objects || []).join(', ')}]`;
    }
    equals(other) {
        if (!(other instanceof Path)) {
            return false;
        }
        if (other === this) {
            return true;
        }
        return areEqual(this.objects, other.objects) && areEqual(this.labels, other.labels);
    }
}
exports.Path = Path;
function areEqual(obj1, obj2) {
    if (obj1 === obj2) {
        return true;
    }
    if (typeof obj1.equals === 'function') {
        return obj1.equals(obj2);
    }
    if (Array.isArray(obj1) && Array.isArray(obj2)) {
        if (obj1.length !== obj2.length) {
            return false;
        }
        for (let i = 0; i < obj1.length; i++) {
            if (!areEqual(obj1[i], obj2[i])) {
                return false;
            }
        }
        return true;
    }
    return false;
}
function summarize(value) {
    if (value === null || value === undefined) {
        return value;
    }
    const strValue = value.toString();
    return strValue.length > 20 ? strValue.substr(0, 20) : strValue;
}
