﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/globalaccelerator/GlobalAccelerator_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace GlobalAccelerator {
namespace Model {

/**
 * <p>A complex type for an endpoint. Specifies information about the endpoint to
 * remove from the endpoint group.</p><p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/globalaccelerator-2018-08-08/EndpointIdentifier">AWS
 * API Reference</a></p>
 */
class EndpointIdentifier {
 public:
  AWS_GLOBALACCELERATOR_API EndpointIdentifier() = default;
  AWS_GLOBALACCELERATOR_API EndpointIdentifier(Aws::Utils::Json::JsonView jsonValue);
  AWS_GLOBALACCELERATOR_API EndpointIdentifier& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_GLOBALACCELERATOR_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>An ID for the endpoint. If the endpoint is a Network Load Balancer or
   * Application Load Balancer, this is the Amazon Resource Name (ARN) of the
   * resource. If the endpoint is an Elastic IP address, this is the Elastic IP
   * address allocation ID. For Amazon EC2 instances, this is the EC2 instance ID.
   * </p> <p>An Application Load Balancer can be either internal or
   * internet-facing.</p>
   */
  inline const Aws::String& GetEndpointId() const { return m_endpointId; }
  inline bool EndpointIdHasBeenSet() const { return m_endpointIdHasBeenSet; }
  template <typename EndpointIdT = Aws::String>
  void SetEndpointId(EndpointIdT&& value) {
    m_endpointIdHasBeenSet = true;
    m_endpointId = std::forward<EndpointIdT>(value);
  }
  template <typename EndpointIdT = Aws::String>
  EndpointIdentifier& WithEndpointId(EndpointIdT&& value) {
    SetEndpointId(std::forward<EndpointIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates whether client IP address preservation is enabled for an endpoint.
   * The value is true or false. </p> <p>If the value is set to true, the client's IP
   * address is preserved in the <code>X-Forwarded-For</code> request header as
   * traffic travels to applications on the endpoint fronted by the accelerator.</p>
   */
  inline bool GetClientIPPreservationEnabled() const { return m_clientIPPreservationEnabled; }
  inline bool ClientIPPreservationEnabledHasBeenSet() const { return m_clientIPPreservationEnabledHasBeenSet; }
  inline void SetClientIPPreservationEnabled(bool value) {
    m_clientIPPreservationEnabledHasBeenSet = true;
    m_clientIPPreservationEnabled = value;
  }
  inline EndpointIdentifier& WithClientIPPreservationEnabled(bool value) {
    SetClientIPPreservationEnabled(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_endpointId;

  bool m_clientIPPreservationEnabled{false};
  bool m_endpointIdHasBeenSet = false;
  bool m_clientIPPreservationEnabledHasBeenSet = false;
};

}  // namespace Model
}  // namespace GlobalAccelerator
}  // namespace Aws
