"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RenderingService = void 0;
var _react = _interopRequireDefault(require("react"));
var _server = require("react-dom/server");
var _operators = require("rxjs/operators");
var _i18n = require("@osd/i18n");
var _https = require("https");
var _axios = _interopRequireDefault(require("axios"));
var _views = require("./views");
var _ssl_config = require("../http/ssl_config");
function _interopRequireDefault(e) { return e && e.__esModule ? e : { default: e }; }
function _defineProperty(e, r, t) { return (r = _toPropertyKey(r)) in e ? Object.defineProperty(e, r, { value: t, enumerable: !0, configurable: !0, writable: !0 }) : e[r] = t, e; }
function _toPropertyKey(t) { var i = _toPrimitive(t, "string"); return "symbol" == typeof i ? i : i + ""; }
function _toPrimitive(t, r) { if ("object" != typeof t || !t) return t; var e = t[Symbol.toPrimitive]; if (void 0 !== e) { var i = e.call(t, r || "default"); if ("object" != typeof i) return i; throw new TypeError("@@toPrimitive must return a primitive value."); } return ("string" === r ? String : Number)(t); } /*
                                                                                                                                                                                                                                                                                                                          * SPDX-License-Identifier: Apache-2.0
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          * The OpenSearch Contributors require contributions made to
                                                                                                                                                                                                                                                                                                                          * this file be licensed under the Apache-2.0 license or a
                                                                                                                                                                                                                                                                                                                          * compatible open source license.
                                                                                                                                                                                                                                                                                                                          *
                                                                                                                                                                                                                                                                                                                          * Any modifications Copyright OpenSearch Contributors. See
                                                                                                                                                                                                                                                                                                                          * GitHub history for details.
                                                                                                                                                                                                                                                                                                                          */ /*
                                                                                                                                                                                                                                                                                                                              * Licensed to Elasticsearch B.V. under one or more contributor
                                                                                                                                                                                                                                                                                                                              * license agreements. See the NOTICE file distributed with
                                                                                                                                                                                                                                                                                                                              * this work for additional information regarding copyright
                                                                                                                                                                                                                                                                                                                              * ownership. Elasticsearch B.V. licenses this file to you under
                                                                                                                                                                                                                                                                                                                              * the Apache License, Version 2.0 (the "License"); you may
                                                                                                                                                                                                                                                                                                                              * not use this file except in compliance with the License.
                                                                                                                                                                                                                                                                                                                              * You may obtain a copy of the License at
                                                                                                                                                                                                                                                                                                                              *
                                                                                                                                                                                                                                                                                                                              *    http://www.apache.org/licenses/LICENSE-2.0
                                                                                                                                                                                                                                                                                                                              *
                                                                                                                                                                                                                                                                                                                              * Unless required by applicable law or agreed to in writing,
                                                                                                                                                                                                                                                                                                                              * software distributed under the License is distributed on an
                                                                                                                                                                                                                                                                                                                              * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
                                                                                                                                                                                                                                                                                                                              * KIND, either express or implied.  See the License for the
                                                                                                                                                                                                                                                                                                                              * specific language governing permissions and limitations
                                                                                                                                                                                                                                                                                                                              * under the License.
                                                                                                                                                                                                                                                                                                                              */
const DEFAULT_TITLE = 'OpenSearch Dashboards';

/** @internal */
class RenderingService {
  constructor(coreContext) {
    this.coreContext = coreContext;
    _defineProperty(this, "logger", void 0);
    _defineProperty(this, "httpsAgent", void 0);
    /**
     * Assign values for branding related configurations based on branding validation
     * by calling checkBrandingValid(). If URL is valid, pass in
     * the actual URL; if not, pass in undefined.
     *
     * @param {Readonly<OpenSearchDashboardsConfigType>} opensearchDashboardsConfig
     * @returns {BrandingAssignment} valid URLs or undefined assigned for each branding configs
     */
    _defineProperty(this, "assignBrandingConfig", async opensearchDashboardsConfig => {
      const brandingValidation = await this.checkBrandingValid(opensearchDashboardsConfig);
      const branding = opensearchDashboardsConfig.branding;

      // assign default mode URL based on the brandingValidation function result
      const logoDefault = brandingValidation.isLogoDefaultValid ? branding.logo.defaultUrl : undefined;
      const markDefault = brandingValidation.isMarkDefaultValid ? branding.mark.defaultUrl : undefined;
      const loadingLogoDefault = brandingValidation.isLoadingLogoDefaultValid ? branding.loadingLogo.defaultUrl : undefined;

      // assign dark mode URLs based on brandingValidation function result
      const logoDarkmode = brandingValidation.isLogoDarkmodeValid ? branding.logo.darkModeUrl : undefined;
      const markDarkmode = brandingValidation.isMarkDarkmodeValid ? branding.mark.darkModeUrl : undefined;
      const loadingLogoDarkmode = brandingValidation.isLoadingLogoDarkmodeValid ? branding.loadingLogo.darkModeUrl : undefined;

      // assign favicon based on brandingValidation function result
      const favicon = brandingValidation.isFaviconValid ? branding.faviconUrl : undefined;

      // assign application title based on brandingValidation function result
      const applicationTitle = brandingValidation.isTitleValid ? branding.applicationTitle : DEFAULT_TITLE;

      // use expanded menu by default unless explicitly set to false
      const {
        useExpandedHeader = true
      } = branding;
      const brandingAssignment = {
        logoDefault,
        logoDarkmode,
        markDefault,
        markDarkmode,
        loadingLogoDefault,
        loadingLogoDarkmode,
        favicon,
        applicationTitle,
        useExpandedHeader
      };
      return brandingAssignment;
    });
    /**
     * Assign boolean values for branding related configurations to indicate if
     * user inputs valid or invalid URLs by calling isUrlValid() function. Also
     * check if title is valid by calling isTitleValid() function.
     *
     * @param {Readonly<OpenSearchDashboardsConfigType>} opensearchDashboardsConfig
     * @returns {BrandingValidation} indicate valid/invalid URL for each branding config
     */
    _defineProperty(this, "checkBrandingValid", async opensearchDashboardsConfig => {
      const branding = opensearchDashboardsConfig.branding;
      const isLogoDefaultValid = await this.isUrlValid(branding.logo.defaultUrl, 'logo default');
      const isLogoDarkmodeValid = await this.isUrlValid(branding.logo.darkModeUrl, 'logo darkMode');
      const isMarkDefaultValid = await this.isUrlValid(branding.mark.defaultUrl, 'mark default');
      const isMarkDarkmodeValid = await this.isUrlValid(branding.mark.darkModeUrl, 'mark darkMode');
      const isLoadingLogoDefaultValid = await this.isUrlValid(branding.loadingLogo.defaultUrl, 'loadingLogo default');
      const isLoadingLogoDarkmodeValid = await this.isUrlValid(branding.loadingLogo.darkModeUrl, 'loadingLogo darkMode');
      const isFaviconValid = await this.isUrlValid(branding.faviconUrl, 'favicon');
      const isTitleValid = this.isTitleValid(branding.applicationTitle, 'applicationTitle');
      const brandingValidation = {
        isLogoDefaultValid,
        isLogoDarkmodeValid,
        isMarkDefaultValid,
        isMarkDarkmodeValid,
        isLoadingLogoDefaultValid,
        isLoadingLogoDarkmodeValid,
        isFaviconValid,
        isTitleValid
      };
      return brandingValidation;
    });
    /**
     * Validation function for URLs. Use Axios to call URL and check validity.
     * Also needs to be ended with png, svg, gif, PNG, SVG and GIF.
     *
     * @param {string} url
     * @param {string} configName
     * @returns {boolean} indicate if the URL is valid/invalid
     */
    _defineProperty(this, "isUrlValid", async (url, configName) => {
      if (url === '/') {
        return false;
      }
      if (url.match(/\.(png|svg|gif|PNG|SVG|GIF)$/) === null) {
        this.logger.get('branding').error(`${configName} config is invalid. Using default branding.`);
        return false;
      }
      if (url.startsWith('/')) {
        return true;
      }
      return await _axios.default.get(url, {
        httpsAgent: this.httpsAgent,
        adapter: 'http',
        maxRedirects: 0
      }).then(() => {
        return true;
      }).catch(() => {
        this.logger.get('branding').error(`${configName} URL was not found or invalid. Using default branding.`);
        return false;
      });
    });
    /**
     * Validation function for applicationTitle config.
     * Title length needs to be between 1 to 36 letters.
     *
     * @param {string} title
     * @param {string} configName
     * @returns {boolean} indicate if user input title is valid/invalid
     */
    _defineProperty(this, "isTitleValid", (title, configName) => {
      if (!title) {
        return false;
      }
      if (title.length > 36) {
        this.logger.get('branding').error(`${configName} config is not found or invalid. Title length should be between 1 to 36 characters. Using default title.`);
        return false;
      }
      return true;
    });
    this.logger = this.coreContext.logger;
  }
  async setup({
    http,
    status,
    uiPlugins,
    dynamicConfig
  }) {
    const [opensearchDashboardsConfig, serverConfig] = await Promise.all([this.coreContext.configService.atPath('opensearchDashboards').pipe((0, _operators.first)()).toPromise(), this.coreContext.configService.atPath('server').pipe((0, _operators.first)()).toPromise()]);
    this.setupHttpAgent(serverConfig);
    return {
      render: async (request, uiSettings, {
        includeUserSettings = true,
        vars,
        nonce
      }) => {
        var _request$query, _request$query$trim;
        const env = {
          mode: this.coreContext.env.mode,
          packageInfo: this.coreContext.env.packageInfo
        };
        const basePath = http.basePath.get(request);
        const uiPublicUrl = `${basePath}/ui`;
        const serverBasePath = http.basePath.serverBasePath;
        const settings = {
          defaults: uiSettings.getRegistered(),
          user: includeUserSettings ? await uiSettings.getUserProvided() : {}
        };
        const brandingAssignment = await this.assignBrandingConfig(opensearchDashboardsConfig);
        let locale = _i18n.i18n.getLocale();
        const localeOverride = (_request$query = request.query) === null || _request$query === void 0 || (_request$query = _request$query.locale) === null || _request$query === void 0 || (_request$query$trim = _request$query.trim) === null || _request$query$trim === void 0 ? void 0 : _request$query$trim.call(_request$query);
        if (localeOverride) {
          const normalizedLocale = _i18n.i18n.normalizeLocale(localeOverride);
          if (_i18n.i18nLoader.isRegisteredLocale(normalizedLocale)) {
            locale = normalizedLocale;
          }
        }
        const dynamicConfigStartServices = await dynamicConfig.getStartService();
        const metadata = {
          strictCsp: http.csp.strict,
          uiPublicUrl,
          bootstrapScriptUrl: `${basePath}/bootstrap.js`,
          startupScriptUrl: `${basePath}/startup.js`,
          i18n: _i18n.i18n.translate,
          locale,
          nonce,
          injectedMetadata: {
            version: env.packageInfo.version,
            buildNumber: env.packageInfo.buildNum,
            branch: env.packageInfo.branch,
            basePath,
            serverBasePath,
            env,
            anonymousStatusPage: status.isStatusPageAnonymous(),
            i18n: {
              translationsUrl: `${basePath}/translations/${locale}.json`
            },
            csp: {
              warnLegacyBrowsers: http.csp.warnLegacyBrowsers
            },
            vars: vars !== null && vars !== void 0 ? vars : {},
            uiPlugins: await Promise.all([...uiPlugins.public].map(async ([id, plugin]) => ({
              id,
              plugin,
              // TODO Scope the client so that only exposedToBrowser configs are exposed
              config: this.coreContext.dynamicConfigService.hasDefaultConfigs({
                name: id
              }) ? await dynamicConfigStartServices.getClient().getConfig({
                name: id
              }, {
                asyncLocalStorageContext: dynamicConfigStartServices.getAsyncLocalStore()
              }) : await this.getUiConfig(uiPlugins, id)
            }))),
            legacyMetadata: {
              uiSettings: settings
            },
            branding: {
              assetFolderUrl: `${uiPublicUrl}/default_branding`,
              logo: {
                defaultUrl: brandingAssignment.logoDefault,
                darkModeUrl: brandingAssignment.logoDarkmode
              },
              mark: {
                defaultUrl: brandingAssignment.markDefault,
                darkModeUrl: brandingAssignment.markDarkmode
              },
              loadingLogo: {
                defaultUrl: brandingAssignment.loadingLogoDefault,
                darkModeUrl: brandingAssignment.loadingLogoDarkmode
              },
              faviconUrl: brandingAssignment.favicon,
              applicationTitle: brandingAssignment.applicationTitle,
              useExpandedHeader: brandingAssignment.useExpandedHeader
            },
            survey: opensearchDashboardsConfig.survey.url,
            keyboardShortcuts: {
              enabled: opensearchDashboardsConfig.keyboardShortcuts.enabled
            }
          }
        };
        return `<!DOCTYPE html>${(0, _server.renderToStaticMarkup)( /*#__PURE__*/_react.default.createElement(_views.Template, {
          metadata: metadata
        }))}`;
      }
    };
  }
  async stop() {}

  /**
   * Setups HTTP Agent if SSL is enabled to pass SSL config
   * values to Axios to make requests in while validating
   * resources.
   *
   * @param {Readonly<HttpConfigType>} httpConfig
   */
  setupHttpAgent(httpConfig) {
    var _httpConfig$ssl;
    if (!((_httpConfig$ssl = httpConfig.ssl) !== null && _httpConfig$ssl !== void 0 && _httpConfig$ssl.enabled)) return;
    try {
      const sslConfig = new _ssl_config.SslConfig(httpConfig.ssl);
      this.httpsAgent = new _https.Agent({
        ca: sslConfig.certificateAuthorities,
        cert: sslConfig.certificate,
        key: sslConfig.key,
        passphrase: sslConfig.keyPassphrase,
        rejectUnauthorized: false
      });
    } catch (e) {
      this.logger.get('branding').error('HTTP agent failed to setup for SSL.');
    }
  }
  async getUiConfig(uiPlugins, pluginId) {
    var _await$browserConfig$;
    const browserConfig = uiPlugins.browserConfigs.get(pluginId);
    return (_await$browserConfig$ = await (browserConfig === null || browserConfig === void 0 ? void 0 : browserConfig.pipe((0, _operators.take)(1)).toPromise())) !== null && _await$browserConfig$ !== void 0 ? _await$browserConfig$ : {};
  }
}
exports.RenderingService = RenderingService;