#include <stdio.h>
#include <stdlib.h>

#include <errno.h>
#include <fcntl.h>
#include <poll.h>
#include <signal.h>
#include <unistd.h>

#define FUSE_USE_VERSION 35
#include <cuse_lowlevel.h>

struct client {
	unsigned long long balance;
	struct client *next;
	unsigned refcount;
	pid_t pid;
};

struct reader {
	struct reader *next;
	fuse_req_t req;
	struct client *client;
};

struct poller {
	struct poller *next;
	struct fuse_pollhandle *ph;
};

static struct {
	unsigned long long tokens;
	struct client *clients;
	struct {
		struct reader *head, **ptail;
	} readers;
	struct {
		struct poller *head, **ptail;
	} pollers;
} gm;

static struct client *client_new(pid_t pid)
{
	struct client *c;
	if (!(c = calloc(1, sizeof *c)))
		return c;
	c->next = gm.clients;
	c->pid = pid;
	return gm.clients = c;
}

static struct client *client_find(pid_t pid)
{
	struct client *c;
	for (c = gm.clients; c && c->pid != pid; c = c->next);
	return c;
}

static void client_del(struct client *c)
{
	struct client **pc;
	for (pc = &gm.clients; *pc != c; pc = &(*pc)->next);
	*pc = c->next;
	free(c);
}

static struct client *client_ref(pid_t pid)
{
	struct client *c;
	if ((c = client_find(pid)) || (c = client_new(pid)))
		c->refcount++;
	return c;
}

static void reader_enqueue(fuse_req_t req, struct client *c)
{
	struct reader *r;
	if (!(r = malloc(sizeof *r)))
		return;
	r->next = 0;
	r->req = req;
	r->client = c;
	*gm.readers.ptail = r;
	gm.readers.ptail = &r->next;
}

static void poller_enqueue(struct fuse_pollhandle *ph)
{
	struct poller *p;
	if (!(p = malloc(sizeof *p)))
		return;
	p->next = 0;
	p->ph = ph;
	*gm.pollers.ptail = p;
	gm.pollers.ptail = &p->next;
}

static void wake_readers()
{
	struct reader *r, *rnext;
	for (r = gm.readers.head; r && gm.tokens; r = rnext) {
		rnext = r->next;
		if (r->req) {
			r->client->balance++;
			gm.tokens--;
			fuse_reply_buf(r->req, ".", 1);
		}
		if (!rnext)
			gm.readers.ptail = &gm.readers.head;
		gm.readers.head = rnext;
		free(r);
	}
	if (!gm.tokens)
		return;
	struct poller *p, *pnext;
	for (p = gm.pollers.head; p; p = pnext) {
		pnext = p->next;
		fuse_lowlevel_notify_poll(p->ph);
		fuse_pollhandle_destroy(p->ph);
		if (!pnext)
			gm.pollers.ptail = &gm.pollers.head;
		gm.pollers.head = pnext;
		free(p);
	}
}

#define gm_log(...) fprintf(stderr, __VA_ARGS__)

static void gm_open(fuse_req_t r, struct fuse_file_info *fi)
{
	const struct fuse_ctx *ctx = fuse_req_ctx(r);
	struct client *c;
	if (!(c = client_ref(ctx->pid))) {
		fuse_reply_err(r, ENOMEM);
		return;
	}
	gm_log("new connection for pid %d\n", ctx->pid);
	fi->fh = (uint64_t)c;
	fuse_reply_open(r, fi);
}

static void gm_release(fuse_req_t r, struct fuse_file_info *fi)
{
	struct client *c = (void*)fi->fh;
	gm_log("closed connection for pid %d\n", c->pid);
	if (c->refcount-- == 1) {
		gm_log("final token balance %llu for pid %d\n",
		       c->balance, c->pid);
		gm.tokens += c->balance;
		wake_readers();
		client_del(c);
	}
	fuse_reply_err(r, 0);
}

static void gm_interrupt(fuse_req_t req, void *v)
{
	struct reader *r;
	for (r = gm.readers.head; r && r->req != req; r = r->next);
	if (!r)
		return;
	r->req = 0;
	fuse_reply_err(req, EINTR);
}

static void gm_read(fuse_req_t r, size_t size, off_t off,
		    struct fuse_file_info *fi)
{
	size = !!size;
	struct client *c = (void*)fi->fh;
	if (gm.tokens >= size) {
		c->balance += size;
		gm.tokens -= size;
		fuse_reply_buf(r, ".", size);
	} else if (fi->flags & O_NONBLOCK) {
		fuse_reply_err(r, EAGAIN);
	} else {
		reader_enqueue(r, c);
		fuse_req_interrupt_func(r, gm_interrupt, 0);
	}
}

static void gm_write(fuse_req_t r, const char *buf, size_t size, off_t off,
		     struct fuse_file_info *fi)
{
	size = !!size;
	struct client *c = (void*)fi->fh;
	if (c->balance >= size) {
		c->balance -= size;
		gm.tokens += size;
		wake_readers();
	} else {
		gm_log("ignoring extra token on pid %d\n", c->pid);
	}
	fuse_reply_write(r, size);
}

static void gm_poll(fuse_req_t r, struct fuse_file_info *fi,
		    struct fuse_pollhandle *ph)
{
	int e = fi->poll_events & (POLLIN | POLLOUT);
	if (!gm.tokens && !(e &= ~POLLIN))
		poller_enqueue(ph);
	fuse_reply_poll(r, e);
}

static const struct cuse_lowlevel_ops fops = {
	.open    = gm_open,
	.release = gm_release,
	.read    = gm_read,
	.write   = gm_write,
	.poll    = gm_poll,
};

int main(int argc, char **argv)
{
	gm.readers.ptail = &gm.readers.head;
	gm.pollers.ptail = &gm.pollers.head;
	long nproc;
	if ((nproc = sysconf(_SC_NPROCESSORS_ONLN)) < 0) {
		perror("sysconf");
		return 1;
	}
	gm.tokens = nproc + 1;

	const char *devpath = "/dev/cuse";
	int cuse_fd;
	if ((cuse_fd = open(devpath, O_RDWR)) < 0) {
		perror(devpath);
		return 1;
	}

	char mountpoint[sizeof "/dev/fd/0123456789"];
	snprintf(mountpoint, sizeof mountpoint, "/dev/fd/%u", cuse_fd);

	const char *devname = "DEVNAME=guild";
	struct cuse_info ci = {
		.dev_info_argc = 1,
		.dev_info_argv = &devname,
	};

	struct fuse_args fa = FUSE_ARGS_INIT(argc, argv);

	struct fuse_session *fs;
	if (!(fs = cuse_lowlevel_new(&fa, &ci, &fops, 0))
	    || fuse_session_mount(fs, mountpoint)
	    || fuse_session_loop(fs))
		return 1;
}
