//////////////////////////////////////////////////////////////////////////////
//
// 		  Copyright (C) 1996,1997  Matthew Doar  doar@pobox.com
// 
// Permission to use, copy, and distribute this software and its documentation 
// for any purpose with or without fee is hereby granted, provided that the 
// above copyright notice appears with all copies and that both that copyright 
// notice and this permission notice appear in supporting documentation. 
// 
// Permission to modify the software is granted, but not the right to 
// distribute the modified code. Modifications are to be distributed as 
// patches to the released version. 
// 
// This software is provided "as is" without express or implied warranty. 
//
//////////////////////////////////////////////////////////////////////////////

// tiers.cc

#include <iostream.h>
#include <limits.h>

#ifndef _TIERS_HH
#include "tiers.hh"
#endif



////////////////////////////////////////////////////////////////////////
// Model::Model
//
// Constructor for top level class
////////////////////////////////////////////////////////////////////////
Model::Model(void) :
 NW(0), NM(0), NL(0),
 SW(0), SM(0), SL(0),
 RW(0), RM(0), RL(0),
 RMW(0), RLM(0),
 WanList(0), ManList(0), LanList(0),
 theCostTable(0), CurrNodeNum(0)
{
  // 9 == 3*3
  theCostTable = new CostTableEntry[9];
  if (!theCostTable)
	{
	  cerr << "Tiers:: unable to create the cost table" << endl;
	  abort();
	}

  // Set all the entries to unreasonable values
  for (unsigned long int i = 0; i < 9; i++)
	{
	  theCostTable[i].bw = ULONG_MAX;	
	  theCostTable[i].d1 = ULONG_MAX;	
	  theCostTable[i].d2 = ULONG_MAX;	
	}
}



////////////////////////////////////////////////////////////////////////
// Model::~Model
//
// Destructor for top level class
////////////////////////////////////////////////////////////////////////
Model::~Model(void)
{
  delete [] WanList; WanList = 0;
  delete [] ManList; ManList = 0;
  delete [] LanList; LanList = 0;

  delete theCostTable; theCostTable = 0;

  delete [] title; title = 0;
  delete [] outputdir; outputdir = 0;
}



////////////////////////////////////////////////////////////////////////
// Model::CreateCostTable
//
// Create the cost table or check it
////////////////////////////////////////////////////////////////////////
bool Model::CreateCostTable(CostTableEntry *aCostTable)
{
  bool ret = true;


  if (aCostTable)
	{
	  // Enter the values from the table which was passed in
	  // There are three network types
	  for (unsigned long int i = 0; i < 3; i++)
		{
		  for (unsigned long int j = 0; j < 3; j++)
			{
			  unsigned long int index = i*3 + j;
			  theCostTable[index] = aCostTable[index];	
			}
		}
	}

  // Check the table has reasonable values, either from the configuration file
  // or from the table which was passed in
  for (unsigned long int i = 0; i < 3; i++)
	{
	  if(theCostTable[i*3].bw == ULONG_MAX)	
		{
		  ret = false;
		}
	  if(theCostTable[i*3].d1 == ULONG_MAX)	
		{
		  ret = false;
		}
	  if(theCostTable[i*3].d2 == ULONG_MAX)	
		{
		  ret = false;
		}
	}

  return ret;
}



////////////////////////////////////////////////////////////////////////
// Model::CreateModel
//
// Create the network graph according the parameters passed
// If the redundancy parameters (aR*) are not defined, they default to 1
////////////////////////////////////////////////////////////////////////
bool Model::CreateModel(const unsigned long int aNW,
						const unsigned long int aNM,
						const unsigned long int aNL,
						const unsigned long int aSW,
						const unsigned long int aSM,
						const unsigned long int aSL,
						const unsigned long int aRW,
						const unsigned long int aRM,
						const unsigned long int aRL,
						const unsigned long int aRMW,
						const unsigned long int aRLM)
{
  // store the model parameters
  NW = aNW; NM = aNM; NL = aNL;
  SW = aSW; SM = aSM; SL = aSL; 
  RW = aRW; RM = aRM; RL = aRL;
  RMW = aRMW; RLM = aRLM;

  //
  // Create the nodes for the networks
  //

  if (!CreateWanNodes()) 
	{
	  cerr << "Tiers:: WAN creation error" << endl;
	  return false;
	}

  if (!CreateManNodes()) 
	{
	  cerr << "Tiers:: MAN creation error" << endl;
	  return false;
	}

  if (!CreateLanNodes()) 
	{
	  cerr << "Tiers:: LAN creation error" << endl;
	  return false;
	}

  //
  // Create the edges with no redundancy. Prepare to add redundant edges
  //

  if (!CreateWanEdges()) 
	{
	  cerr << "Tiers:: WAN edge creation error" << endl;
	  return false;
	}

  if (!CreateManEdges()) 
	{
	  cerr << "Tiers:: MAN edge creation error" << endl;
	  return false;
	}

  if (!CreateLanEdges()) 
	{
	  cerr << "Tiers:: LAN edge creation error" << endl;
	  return false;
	}

  //
  // Create the redundant intranetwork edges 
  //

  if (!CreateWanRedundancy()) 
	{
	  cerr << "Tiers:: WAN redundancy creation error" << endl;
	  return false;
	}

  if (!CreateManRedundancy()) 
	{
	  cerr << "Tiers:: MAN redundancy creation error" << endl;
	  return false;
	}

  if (!CreateLanRedundancy()) 
	{
	  cerr << "Tiers:: LAN redundancy creation error" << endl;
	  return false;
	}

  //
  // Create the essential and redundant internetwork edges
  //

  if (!ConnectMans()) 
	{
	  cerr << "Tiers:: MAN to WAN redundancy creation error" << endl;
	  return false;
	}

  if (!ConnectLans()) 
	{
	  cerr << "Tiers:: LAN to MAN redundancy creation error" << endl;
	  return false;
	}

  return true;
}



////////////////////////////////////////////////////////////////////////
// Model::OutputModel
//
// Print out the nodes and edges of all the networks which make up the
// model. Optionally print out all the parameters associated with the model
////////////////////////////////////////////////////////////////////////
bool Model::OutputModel(OpType optype)
{
  bool ret = true;

  switch (optype)
	{
	case GENERIC:
	  {
		if (!OutputParams()) 
		  {
			cerr << "Tiers:: parameter output error" << endl;
			ret = false;
		  }

		if (!OutputNodes()) 
		  {
			cerr << "Tiers:: node output error" << endl;
			ret = false;
		  }

		if (!OutputEdges()) 
		  {
			cerr << "Tiers:: edge output error" << endl;
			ret = false;
		  }
		break;
	  }
	case GNUPLOT:
	  {
		// Print out the nodes and edges of all the networks which make up the
		// model in a format which can be taken by gnuplot as a script. 
		// The action of outputing the edges causes CreateGlobalNodeList() 
		// to be called
		// which can optionally print out labels for each node
		// Optionally print out all the parameters associated with the model.
		if (!OutputGnuplotHeader()) 
		  {
			cerr << "Tiers:: gnuplot header output error" << endl;
			ret = false;
		  }

		if (!OutputParams()) 
		  {
			cerr << "Tiers:: parameter output error" << endl;
			ret = false;
		  }

		if (!OutputEdgesGnuplot()) 
		  {
			cerr << "Tiers:: edges output error" << endl;
			ret = false;
		  }

		if (!OutputGnuplotTrailer()) 
		  {
			cerr << "Tiers:: gnuplot trailer output error" << endl;
			ret = false;
		  }

	  break;
	  }
	default:
	  {
		cout << "Tiers:: unrecognized output type" << endl;
		ret = false;
		break;
	  }
	}

  return ret;
}



////////////////////////////////////////////////////////////////////////
// NodesAndEdges::NodesAndEdges
//
// Constructor for a network
////////////////////////////////////////////////////////////////////////
NodesAndEdges::NodesAndEdges() : 
  nodes(0), edges(0), newedges(0),
  m_NumNodes(0), m_MaxNumEdges(0), m_NumEdges(0) 
{
}



////////////////////////////////////////////////////////////////////////
// NodesAndEdges::~NodesAndEdges
//
// Destructor for a network
////////////////////////////////////////////////////////////////////////
NodesAndEdges::~NodesAndEdges() 
{ 
  if (nodes) delete [] nodes; nodes = 0;
  if (edges) delete [] edges; edges = 0;		// set to zero when deleted
  if (newedges)	delete [] newedges; newedges = 0; // never gets used in LANs

  if (m_NumEdges) delete [] m_NumEdges; m_NumEdges = 0;
}



////////////////////////////////////////////////////////////////////////
// F
//
// Do
////////////////////////////////////////////////////////////////////////
unsigned long int NodesAndEdges::
NumNodes()
{ 
  return m_NumNodes; 
}



////////////////////////////////////////////////////////////////////////
// F
//
// Do
////////////////////////////////////////////////////////////////////////
void NodesAndEdges::
NumNodes(const unsigned long int N)
{ 
  m_NumNodes = N; 
}



////////////////////////////////////////////////////////////////////////
// F
//
// Do
////////////////////////////////////////////////////////////////////////
unsigned long int NodesAndEdges::
MaxNumEdges()
{ 
  return m_MaxNumEdges; 
}



////////////////////////////////////////////////////////////////////////
// F
//
// Do
////////////////////////////////////////////////////////////////////////
void NodesAndEdges::
MaxNumEdges(const unsigned long int E)
{ 
  m_MaxNumEdges = E; 
}

  

////////////////////////////////////////////////////////////////////////
// F
//
// Do
////////////////////////////////////////////////////////////////////////
unsigned long int NodesAndEdges::
NumEdges(const unsigned long int index)
{ 
  return m_NumEdges[index]; 
}



////////////////////////////////////////////////////////////////////////
// F
//
// Do
////////////////////////////////////////////////////////////////////////
void NodesAndEdges::
NumEdges(const unsigned long int index,
			  const unsigned long int E)
{ 
  m_NumEdges[index] = E; 
}

  

////////////////////////////////////////////////////////////////////////
// F
//
// Do
////////////////////////////////////////////////////////////////////////
void NodesAndEdges::
IncNumEdges(const unsigned long int index)
{ 
  m_NumEdges[index]++; 
}

  

////////////////////////////////////////////////////////////////////////
// F
//
// Do
////////////////////////////////////////////////////////////////////////
unsigned long int NodesAndEdges::
NumNetworks()
{ 
  return m_NumNetworks; 
}



////////////////////////////////////////////////////////////////////////
// F
//
// Do
////////////////////////////////////////////////////////////////////////
void NodesAndEdges::
NumNetworks(const unsigned long int N)
{ 
  m_NumNetworks = N; 
}



////////////////////////////////////////////////////////////////////////
// F
//
// Do
////////////////////////////////////////////////////////////////////////
bool NodesAndEdges::
newEdgeList(unsigned long int size) 
{ 
  m_NumEdges = new unsigned long int[size];
  if (!m_NumEdges)
	return false;
  else
	return true;
}



////////////////////////////////////////////////////////////////////////
// NodesAndEdges::test_dup_edges
//
// Ensure that only one of an edge is printed out
////////////////////////////////////////////////////////////////////////
bool NodesAndEdges::
test_dup_edges(const unsigned long int from,
			   const unsigned long int to)
{
  bool ret = true;

  if (REMOVE_DUP_EDGES)
	{
	  // Only print intranetwork edges (i,j) where i < j
	  // and internetwork edges (i,j) from WAN to MAN, and MAN to LAN
	  // Simple test works because node numbers WAN < MAN < LAN and no
	  // connections between peer networks
	  if (from <= to) 
		{
		  return false;
		}
	}

  return ret;
}
  
// end of file

