package aQute.lib.bundle;

import java.io.*;
import java.net.*;
import java.util.*;
import java.util.jar.*;

public class Jar {

	public final static int	UNKNOWN	= 0;
	public final static int	LIBRARY	= 1;
	public final static int	ACTIVE	= 2;

	Map						packages;
	Map						resources;
	Set						main;
	Set						activator;
	Set						web;
	Map						headers	= new TreeMap();
	int						type;
	String					source;

	Manifest				manifest;

	public Jar() {
	}

	public Jar(String source) throws Exception {
		setSource(source);
		parse();
	}

	public void setSource(String source) {
		this.source = source;
	}

	public void init() {
		packages = new TreeMap();
		resources = new TreeMap();
		main = new TreeSet();
		activator = new TreeSet();
		web = new TreeSet();
	}

	public void parse() throws Exception {
		init();
		parseEntries();

		packages.remove("META-INF");
		Set referenced = new TreeSet();

		for (Iterator i = resources.values().iterator(); i.hasNext();) {
			Resource r = (Resource) i.next();
			Set referencedByClass = r.process();
			if (referencedByClass != null)
				referenced.addAll(referencedByClass);

			if (r.isMain())
				main.add(r);
			if (r.isActivator())
				activator.add(r);

			if (r.isWeb())
				web.add(getPackage(r.getPackageName()));
		}
		for (Iterator i = referenced.iterator(); i.hasNext();)
			addResource((Resource) i.next());
	}

	void parseEntries() throws java.io.IOException, java.lang.Exception {
		Object in = open(source);
		if (in instanceof InputStream)
			parseJar((InputStream) in);
		else
			parseDir("", (File) in);
	}

	public void parseJar(InputStream in) throws Exception {
		JarInputStream jar = new JarInputStream(in);
		manifest = jar.getManifest();
		if (manifest != null)
			parseManifest(manifest);
		else
			System.out.println("No Manifest");

		JarEntry entry = jar.getNextJarEntry();
		while (entry != null) {
			if (!entry.isDirectory()) {
				ByteArrayOutputStream bo = new ByteArrayOutputStream();
				byte[] b = new byte[1024];
				int size = jar.read(b, 0, b.length);
				while (size > 0) {
					bo.write(b, 0, size);
					size = jar.read(b, 0, b.length);
				}

				byte[] bytes = bo.toByteArray();
				String path = entry.getName();
				Resource r;
				if (path.endsWith(".class"))
					r = (Resource) new Clazz(this, path);
				else
					r = new Resource(this, path);

				r.setBytes(bytes);
				addResource(r);
			}
			entry = jar.getNextJarEntry();
		}
		jar.close();
	}

	public void parseDir(String prefix, File dir) throws Exception {
		String files[] = dir.list();
		for (int f = 0; f < files.length; f++) {
			File current = new File(dir, files[f]);
			String path = prefix + files[f];
			if (current.isDirectory())
				parseDir(path + "/", current);
			else {
				ByteArrayOutputStream bo = new ByteArrayOutputStream();
				FileInputStream in = new FileInputStream(
						new File(dir, files[f]));
				byte[] b = new byte[1024];
				int size = in.read(b, 0, b.length);
				while (size > 0) {
					bo.write(b, 0, size);
					size = in.read(b, 0, b.length);
				}
				in.close();
				byte[] bytes = bo.toByteArray();
				Resource r;
				if (files[f].endsWith(".class"))
					r = new Clazz(this, path);
				else
					r = new Resource(this, path);

				r.setBytes(bytes);
				addResource(r);
			}
		}
	}

	void parseManifest(Manifest manifest) {
		try {
			Attributes attrs = manifest.getMainAttributes();
			for (Iterator i = attrs.keySet().iterator(); i.hasNext();) {
				Attributes.Name key = (Attributes.Name) i.next();
				if (!key.toString().equalsIgnoreCase("Manifest-Version")) {
					System.out.println("Header set " + key.toString() + "="
							+ attrs.getValue(key));
					setHeader(key.toString(), attrs.getValue(key));
				}
			}
		} catch (Exception e) {
			e.printStackTrace();
		}
	}

	Object open(String source) throws IOException {
		int n = source.indexOf(':');
		if (n > 3 && n < 8) {
			URL url = new URL(source);
			return url.openStream();
		} else {
			File file = new File(source);
			if (file.exists()) {
				if (file.isDirectory())
					return file;
				else
					return new FileInputStream(file);
			}
		}
		return null;
	}

	boolean addResource(Resource r) {
		Resource existing = (Resource) resources.get(r.getPath());
		if (existing == null) {
			String pname = r.getPackageName();
			Package p = (Package) getPackage(pname);
			p.addResource(r);
			resources.put(r.getPath(), r);
			return true;
		}
		return false;
	}

	Package getPackage(String qn) {
		Package p = (Package) packages.get(qn);
		if (p == null) {
			p = new Package(this, qn);
			packages.put(qn, p);
		}
		return p;
	}

	public Collection getPackages() {
		return packages.values();
	}

	public String toString() {
		return source + "\t";
	}

	public boolean equals(Object o) {
		if (o instanceof Jar) {
			Jar j = (Jar) o;
			return j.source.equals(source);
		}
		return false;
	}

	public int hashCode() {
		return source.hashCode();
	}

	public Set getMain() {
		return main;
	}
	public Set getActivator() {
		return activator;
	}
	public Set getWeb() {
		return web;
	}

	public String getHeader(String hdr) {
		return (String) headers.get(hdr.toLowerCase());
	}
	public String getHeader(String hdr, String dflt) {
		String v = getHeader(hdr);
		if (v == null)
			return dflt;
		else
			return v;
	}
	public void setHeader(String hdr, String value) {
		headers.put(hdr.toLowerCase(), value);
	}

	public void removeHeader(String hdr) {
		headers.remove(hdr.toLowerCase());
	}

	public int getType() {
		if (type == UNKNOWN) {
			if (getMain().isEmpty() && getActivator().isEmpty())
				return LIBRARY;
			else
				return ACTIVE;
		}
		return type;
	}

}

