package aQute.maven.bnd;

import java.io.Closeable;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import java.util.TreeSet;
import java.util.concurrent.Callable;
import java.util.concurrent.Executor;

import org.osgi.util.promise.Failure;
import org.osgi.util.promise.Promise;
import org.osgi.util.promise.Success;

import aQute.bnd.annotation.plugin.BndPlugin;
import aQute.bnd.header.Attrs;
import aQute.bnd.header.Parameters;
import aQute.bnd.http.HttpClient;
import aQute.bnd.osgi.Jar;
import aQute.bnd.osgi.Processor;
import aQute.bnd.osgi.Resource;
import aQute.bnd.service.Plugin;
import aQute.bnd.service.Refreshable;
import aQute.bnd.service.Registry;
import aQute.bnd.service.RegistryPlugin;
import aQute.bnd.service.RepositoryListenerPlugin;
import aQute.bnd.service.RepositoryPlugin;
import aQute.bnd.tool.Tool;
import aQute.bnd.version.MavenVersion;
import aQute.bnd.version.Version;
import aQute.lib.converter.Converter;
import aQute.lib.io.IO;
import aQute.libg.cryptography.SHA1;
import aQute.libg.glob.Glob;
import aQute.maven.bnd.LocalRepoWatcher.ProgramsChanged;
import aQute.maven.bnd.ReleaseDTO.ReleaseType;
import aQute.maven.repo.api.Archive;
import aQute.maven.repo.api.IMavenRepo;
import aQute.maven.repo.api.POM;
import aQute.maven.repo.api.Program;
import aQute.maven.repo.api.Release;
import aQute.maven.repo.api.Revision;
import aQute.maven.repo.provider.MavenStorage;
import aQute.maven.repo.provider.RemoteRepo;
import aQute.service.reporter.Reporter;

/**
 * This is the Bnd repository for Maven.
 */
@BndPlugin(name = "MavenBndRepository")
public class MavenBndRepository implements RepositoryPlugin, RegistryPlugin, Plugin, Closeable, Refreshable {

	public enum Classifiers {
		SOURCE, JAVADOC;
	}

	public interface Configuration {

		/**
		 * The url to the remote release repository. If this is not specified,
		 * the repository is only local.
		 */
		String url();

		/**
		 * The url to the remote snapshot repository. If this is not specified,
		 * it falls back to the release repository or just local if this is also
		 * not specified.
		 */
		String snapshotUrl();

		/**
		 * The path to the local repository
		 */
		// default "~/.m2/repository"
		String local();

		/**
		 * The classifiers to release. These will be generated automatically if
		 * sufficient information is available.
		 */
		// default { Classifiers.BINARY}
		Set<Classifiers> generate();

		// default false
		boolean readOnly();

		String name(String deflt);
	}

	private Configuration		configuration;
	private Registry			registry;
	private File				localRepo;
	private Reporter			reporter;
	private IMavenRepo			storage;
	private boolean				inited;
	private boolean				ok	= true;
	private LocalRepoWatcher	repoWatcher;

	@Override
	public PutResult put(InputStream stream, PutOptions options) throws Exception {

		init();
		File binaryFile = File.createTempFile("put", ".jar");
		File pomFile = File.createTempFile("pom", ".jar");
		try {
			PutResult result = new PutResult();
			IO.copy(stream, binaryFile);

			if (options.digest != null) {
				byte[] digest = SHA1.digest(binaryFile).digest();
				if (!Arrays.equals(options.digest, digest))
					throw new IllegalArgumentException("The given sha-1 does not match the contents sha-1");
			}

			try (Jar binary = new Jar(binaryFile);) {

				Resource pomResource = getPomResource(binary);
				if (pomResource == null)
					throw new IllegalArgumentException(
							"No POM resource in META-INF/maven/... The Maven Bnd Repository requires this pom.");

				IO.copy(pomResource.openInputStream(), pomFile);
				POM pom = new POM(pomFile);

				try (Release releaser = storage.release(pom.getRevision());) {

					ReleaseDTO instructions = getReleaseDTO(options.context);

					if (isLocal(instructions))
						releaser.setLocalOnly();

					Archive binaryArchive = pom.binaryArchive();

					releaser.add(pom.getRevision().pomArchive(), pomFile);
					releaser.add(binaryArchive, binaryFile);

					result.artifact = isLocal(instructions) ? storage.toLocalFile(binaryArchive).toURI()
							: storage.toRemoteURI(binaryArchive);

					if (!isLocal(instructions)) {

						try (Tool tool = new Tool(options.context, binary);) {

							if (instructions.javadoc != null) {
								try (Jar jar = getJavadoc(tool, options.context, instructions.javadoc.path,
										instructions.javadoc.options);) {
									save(releaser, pom.getRevision(), jar, "javadoc");
								}
							}

							if (instructions.sources != null) {
								try (Jar jar = getSource(tool, options.context, instructions.javadoc.path);) {
									save(releaser, pom.getRevision(), jar, "source");
								}
							}
						}
					}
				}
			}
			return result;
		} finally {
			binaryFile.delete();
			pomFile.delete();
		}

	}

	boolean isLocal(ReleaseDTO instructions) {
		return instructions.type == ReleaseType.LOCAL;
	}

	private Jar getSource(Tool tool, Processor context, String path) throws Exception {
		Jar jar = toJar(context, path);
		if (jar != null)
			return jar;

		return tool.doSource();
	}

	private Jar getJavadoc(Tool tool, Processor context, String path, Map<String,String> options) throws Exception {
		Jar jar = toJar(context, path);
		if (jar != null)
			return jar;

		return tool.doJavadoc(options);
	}

	private Jar toJar(Processor context, String path) {
		if (path == null)
			return null;

		File f = context.getFile(path);
		if (f.exists())
			return new Jar(path);

		return null;
	}

	private void save(Release releaser, Revision revision, Jar jar, String classifier) throws Exception {
		String extension = IO.getExtension(jar.getName(), ".jar");
		File tmp = File.createTempFile(classifier, extension);
		try {
			jar.write(tmp);
			releaser.add(revision.archive(extension, classifier), tmp);
		} finally {
			tmp.delete();
		}

	}

	/*
	 * Parse the -maven-release header.
	 */
	private ReleaseDTO getReleaseDTO(Processor context) {
		ReleaseDTO release = new ReleaseDTO();
		Parameters p = new Parameters(context.getProperty("-maven-release"));

		if (p.containsKey("local"))
			release.type = ReleaseType.LOCAL;
		else if (p.containsKey("remote"))
			release.type = ReleaseType.REMOTE;

		p.remove("local");
		p.remove("remote");

		Attrs javadoc = p.remove("javadoc");
		if (javadoc != null) {
			release.javadoc.path = javadoc.get("path");
			release.javadoc.options = javadoc;

		}

		Attrs sources = p.remove("sources");
		if (sources != null) {
			release.sources.path = sources.get("path");
		}

		Attrs pom = p.remove("pom");
		if (pom != null) {
			release.pom.path = pom.get("path");
		}

		if (!p.isEmpty()) {
			reporter.warning("The -maven-release instruction contains unrecognized options: ", p);
		}
		return release;
	}

	private Resource getPomResource(Jar jar) {
		for (Map.Entry<String,Resource> e : jar.getResources().entrySet()) {
			String path = e.getKey();

			if (path.startsWith("META-INF/maven/")) {
				return e.getValue();
			}
		}
		return null;
	}

	@Override
	public File get(String bsn, Version version, Map<String,String> properties, final DownloadListener... listeners)
			throws Exception {
		init();

		Archive archive = getArchive(bsn, version);
		if (archive != null) {
			final File file = storage.toLocalFile(archive);
			Promise<File> promise = storage.get(archive);
			promise = ensurePom(archive, promise);

			if (listeners.length == 0)
				return promise.getValue();

			promise.onResolve(getResolveable(file, promise, listeners));
			return file;
		}
		return null;
	}

	private Promise<File> ensurePom(final Archive archive, Promise<File> promise) throws Exception {
		if (!archive.isPom()) {
			final File pomFile = storage.toLocalFile(archive.getPomArchive());
			if (!pomFile.exists()) {
				return promise.then(new Success<File,File>() {

					@Override
					public Promise<File> call(Promise<File> resolved) throws Exception {
						return storage.get(archive.getPomArchive());
					}
				}, new Failure() {

					@Override
					public void fail(Promise< ? > resolved) throws Exception {
						// TODO Auto-generated method stub
						resolved.getFailure().printStackTrace();
					}
				});
			}
		}
		return promise;
	}

	@Override
	public boolean canWrite() {
		return !configuration.readOnly();
	}

	@Override
	public List<String> list(String pattern) throws Exception {
		init();
		Glob g = pattern == null ? null : new Glob(pattern);

		List<String> bsns = new ArrayList<>();

		for (Program p : repoWatcher.getLocalPrograms()) {
			String ga = p.getCoordinate();
			if (g == null || g.matcher(ga).matches())
				bsns.add(ga);
		}
		return bsns;
	}

	@Override
	public SortedSet<Version> versions(String bsn) throws Exception {
		init();
		TreeSet<Version> versions = new TreeSet<Version>();
		Program program = Program.valueOf(bsn);
		if (program == null)
			return versions;

		for (Revision revision : storage.getRevisions(program)) {
			MavenVersion v = revision.version;
			Version osgi = v.getOSGiVersion();
			versions.add(osgi);
		}

		return versions;
	}

	@Override
	public String getName() {
		init();
		return configuration.name(getLocation());
	}

	private synchronized void init() {
		try {
			if (inited)
				return;
			inited = true;

			localRepo = IO.getFile(configuration.local());

			HttpClient client = registry.getPlugin(HttpClient.class);
			Executor executor = registry.getPlugin(Executor.class);
			RemoteRepo release = configuration.url() != null
					? new RemoteRepo(localRepo, client, clean(configuration.url())) : null;
			RemoteRepo snapshot = configuration.snapshotUrl() != null
					? new RemoteRepo(localRepo, client, clean(configuration.snapshotUrl())) : null;
			storage = new MavenStorage(localRepo, getName(), release, snapshot, executor, reporter,
					getRefreshCallback());

			repoWatcher = new LocalRepoWatcher(localRepo.toPath(), new ProgramsChanged() {

				@Override
				public boolean changed(Set<Program> added, Set<Program> removed) throws Exception {
					return false;
				}
			}, reporter, Processor.getScheduledExecutor(), 5000);

			repoWatcher.open();

		} catch (Exception e) {
			reporter.exception(e, "Init for maven repo failed %s", configuration);
			throw new RuntimeException(e);
		}
	}

	private String clean(String url) {
		if (url.endsWith("/"))
			return url;

		return url + "/";
	}

	@Override
	public String getLocation() {
		return configuration.url() == null ? configuration.local() : configuration.url();
	}

	@Override
	public void setProperties(Map<String,String> map) throws Exception {
		configuration = Converter.cnv(Configuration.class, map);

	}

	@Override
	public void setReporter(Reporter reporter) {
		this.reporter = reporter;
	}

	@Override
	public void setRegistry(Registry registry) {
		this.registry = registry;
	}

	@Override
	public void close() throws IOException {
		storage.close();
		if (repoWatcher != null)
			repoWatcher.close();
	}

	private Runnable getResolveable(final File file, final Promise<File> promise, final DownloadListener... listeners) {
		return new Runnable() {

			@Override
			public void run() {
				try {
					if (promise.getFailure() == null) {
						doSuccess(promise, listeners);
					} else {
						doFailure(file, promise, listeners);
					}
				} catch (Exception e) {
					// can't happen, we checked
					throw new RuntimeException(e);
				}
			}

			void doFailure(final File file, final Promise<File> promise, final DownloadListener... listeners) {
				for (DownloadListener dl : listeners)
					try {
						dl.failure(file, promise.getFailure().getMessage());
					} catch (Exception e) {
						reporter.exception(e, "Download listener failed in failure callback %s", dl);
					}
			}

			void doSuccess(final Promise<File> promise, final DownloadListener... listeners)
					throws InvocationTargetException, InterruptedException {
				File file = promise.getValue();
				for (DownloadListener dl : listeners)
					try {
						dl.success(file);
					} catch (Exception e) {
						reporter.exception(e, "Download listener failed in success callback %s ", dl);
					}
			}

		};
	}

	private Archive getArchive(String bsn, Version version) throws Exception {
		String parts[] = bsn.split(":");
		if (parts.length == 2) {
			Program program = Program.valueOf(bsn);
			MavenVersion mavenVersion = matchVersions(program, version);
			if (mavenVersion == null)
				return null;

			bsn += ":" + mavenVersion;
		}

		return storage.getArchive(bsn);
	}

	private MavenVersion matchVersions(Program program, Version version) throws Exception {
		List<Revision> revisions = storage.getRevisions(program);
		for (Revision r : revisions) {
			if (r.version.getOSGiVersion().equals(version)) {
				return r.version;
			}
		}
		return null;
	}

	private Callable<Boolean> getRefreshCallback() {
		return new Callable<Boolean>() {

			@Override
			public Boolean call() throws Exception {
				for (RepositoryListenerPlugin rp : registry.getPlugins(RepositoryListenerPlugin.class)) {
					try {
						rp.repositoryRefreshed(MavenBndRepository.this);
					} catch (Exception e) {
						reporter.exception(e, "Updating listener plugin %s", rp);
					}
				}
				return ok;
			}
		};
	}

	@Override
	public boolean refresh() throws Exception {
		repoWatcher.refresh();
		return true;
	}

	@Override
	public File getRoot() throws Exception {
		return localRepo;
	}

}
