package aQute.lib.tiny;

import java.util.*;
import java.util.regex.*;

import aQute.lib.reporter.*;

public class Closure extends Context {
	static Pattern		NUMBER		= Pattern.compile("-?\\d+");
	CharSequence		program;
	int					current;
	final static Set	keywords	= new HashSet();
	static {
		keywords.add("for");
		keywords.add("if");
		keywords.add("while");
		keywords.add("do");
		keywords.add("else");
		keywords.add("static");
		keywords.add("volatile");
		keywords.add("return");
		keywords.add("break");
		keywords.add("continue");
		keywords.add("public");
		keywords.add("private");
		keywords.add("protected");
		keywords.add("synchronized");
		keywords.add("class");
		keywords.add("extends");
		keywords.add("interface");
		keywords.add("import");
		keywords.add("package");
		keywords.add("int");
		keywords.add("byte");
		keywords.add("short");
		keywords.add("long");
		keywords.add("double");
		keywords.add("float");
		keywords.add("final");
		keywords.add("new");
		keywords.add("boolean");
		keywords.add("false");
		keywords.add("true");
		keywords.add("goto");
		keywords.add("char");
		keywords.add("transient");
		keywords.add("switch");
		keywords.add("case");
		keywords.add("default");
		keywords.add("instanceof");
		keywords.add("const");
	}

	public Closure(Context parent, CharSequence program) {
		this.parent = parent;
		this.program = program;
	}

	Object eval() throws Exception {
		current = 0;
		Object result = null;
		while (peek() != 0) {
			result = expression();
			if (peek() == ';' || peek() == '\r')
				next();
			else
				break;
		}
		return result;
	}

	public Object expression() throws Exception {
		List l = parse(';');
		if (l == null)
			return null;
		System.out.println(l);
		return execute(l);
	}

	public List parse(char terminator) throws Exception {
		List l = new ArrayList();

		while (true) {
			skipWhiteSpace();
			char c = peek();
			if (c == terminator) {
				return l;
			}
			switch (c) {
			case '$':
				l.add(variable());
				break;

			case '{':
				l.add(closure('}'));
				break;

			case '[':
				l.add(array());
				break;

			case '"':
			case '\'':
				l.add(string());
				break;

			case '(':
				next();
				l.add(expression());
				break;

			case '-':
				unary(l);
				break;

			case '*':
			case '/':
			case '%':
			case '^':
			case '+':
				operator(l, c);
				break;

			case '>':
				if (next('='))
					compare(l, 0, Integer.MAX_VALUE, false);
				else
					compare(l, 1, Integer.MAX_VALUE, false);
				break;

			case '<':
				if (next('='))
					compare(l, Integer.MIN_VALUE, 0, false);
				else
					compare(l, Integer.MIN_VALUE, -1, false);
				break;

			case '!':
				if (next('='))
					compare(l, 0, 0, true);
				else
					error("unknown operator: !");
				break;

			case '=':
				if (next('='))
					compare(l, 0, 0, false);
				else
					error("Did not expect: =");
				break;

			case 0:
				if (l.isEmpty())
					return null;

			case ')':
			case ']':
			case '}':
			case '\r':
			case ';':
				next();
				return l;

			default:
				if (Character.isJavaIdentifierPart(c))
					l.add(identifier());
				else
					error("Unrecognized character: " + c);
				break;
			}
		}
	}

	void unary(List l) throws Exception {
		char c = next();
		if (c == 0)
			error("Unexpected EOF");

		current--;

		if (Character.isWhitespace(c)) {
			operator(l, '-');
		} else
			l.add(simple());
	}

	Object identifier() throws Exception {
		Object name = simple();
		skipWhiteSpace();
		if (isAssignment()) {
			Object value = expression();

			setProperty(name.toString(), value);
			return value;
		} else {
			return name;
		}
	}

	boolean isAssignment() {
		if (peek() == '=') {
			if (next() == '=') {
				current--;
				return false;
			} else
				return true;
		} else
			return false;
	}

	List array() throws Exception {
		next();
		List l = parse(']');
		skipWhiteSpace();
		if (peek() != ']')
			error("no closing ']'");
		next();
		return l;
	}

	private void compare(List l, int minValue, int maxValue, boolean reverse)
			throws Exception {
		next();
		Object right = expression();
		Object left = l.get(l.size() - 1);
		if (right instanceof Comparable) {
			Comparable comp = (Comparable) left;
			Object r = Reference.coerce(comp.getClass(), right);
			int result = comp.compareTo(r);
			boolean answer = (result >= minValue && result <= maxValue);
			if (reverse)
				answer = !answer;
			l.set(l.size() - 1, new Boolean(answer));
		} else
			error("Cannot compare two values because no comparable implemented " +
					left + " " + right);
	}

	private boolean next(char c) {
		if (current < program.length() - 1)
			if (program.charAt(current + 1) == c) {
				current++;
				return true;
			}
		return false;
	}

	private Object execute(List tail) throws Exception {
		if (tail.isEmpty())
			return null;

		Object target = tail.remove(0);

		if (target instanceof CharSequence) {
			String name = target.toString();
			Object cmd = getRoot().getProperty(name);
			if (cmd == null) {
				// Check built-in commands
				Object[] targets = new Object[] { getRoot(), this };
				try {
					if (tail.isEmpty())
						return Reference.get(targets, name);
					else {
						if (keywords.contains(name))
							name = "_" + name;
						return Reference.call(targets, name, tail);
					}
				} catch (NoSuchMethodException e) {
					// Ignore, is string, will be treated as message
				}
			} else if (cmd instanceof Function) {
				Function f = (Function) cmd;
				return f.invoke(this, tail.toArray());
			} else
				target = cmd;
		}

		while (!tail.isEmpty()) {
			Object parameter = tail.remove(0);
			if (parameter instanceof String) {
				String message = (String) parameter;
				try {
					if (message.startsWith("-")) {
						String name = message.substring(1);
						try {
							Reference.call(target, name, tail);
						} catch (NoSuchMethodException nsme) {
							String method = "set" + Character.toUpperCase(name.charAt(0)) +
							name.subSequence(1, name.length());
							Reference.call(target, method, tail);
						}
					} else
						target = Reference.call(target, message, tail);
				} catch (NoSuchMethodException nsme) {
					Reference.call(target, "add", Arrays
							.asList(new Object[] { message }));
				}
			} else {
				Reference.call(target, "add", Arrays
						.asList(new Object[] { parameter }));
			}
			if ( target instanceof Reporter ) {
				report((Reporter) target);				
			}
		}

		try {
			return Reference.call(target, "execute", Collections.EMPTY_LIST);
		} catch (NoSuchMethodException nsme) {
			return target;
		}
	}

	private void report(Reporter target) {
		setProperty("_errors", target.getErrors());
		setProperty("_warnings", target.getWarnings());
	}

	private void operator(List list, char op) throws Exception {
		next();
		Object right = expression();
		Object left = list.get(list.size() - 1);
		if (right instanceof Number && left instanceof Number) {
			nummericOperator(list, op, right, left);
		} else if (left instanceof String) {
			stringOperator(list, op, right, left);
		} else if (left instanceof Collection) {
			collectionOperator(op, right, left);
		} else {
			error("Unknown operator: " + op);
		}
	}

	private void collectionOperator(char op, Object right, Object left) {
		Collection l = (Collection) left;
		switch (op) {
		case '+':
			l.add(right);
			break;
		case '-':
			l.remove(right);
			break;
		case '/':
			while (l.remove(right))
				;
			break;

		default:
			error("unknown operator: " + op);
		}
		// Already in list
	}

	private void stringOperator(List list, char op, Object right, Object left) {
		String l = (String) left.toString();
		String result = "";
		switch (op) {
		case '+':
			result += l + right;
			break;
		case '-':
			result = l.replaceAll(right.toString(), "");
			break;
		case '*':
			StringBuffer sb = new StringBuffer();
			int n = ((Number) right).intValue();
			for (int i = 0; i < n; i++) {
				sb.append(l);
			}
			result = sb.toString();
			break;

		default:
			error("unknown string operator: " + op);
		}
		list.set(list.size() - 1, result);
	}

	private void nummericOperator(List list, char op, Object right, Object left) {
		int r = ((Number) right).intValue();
		int l = ((Number) left).intValue();
		int result = 0;

		switch (op) {
		case '+':
			result = l + r;
			break;
		case '-':
			result = l - r;
			break;
		case '/':
			result = l / r;
			break;
		case '*':
			result = l * r;
			break;
		case '^':
			result = (int) l ^ (int) r;
			break;
		case '%':
			result = l % r;
			break;
		default:
			error("unknown operator: " + op);
		}

		list.set(list.size() - 1, new Integer(result));
	}

	private CharSequence string() throws Exception {
		char end = peek();
		StringBuffer sb = new StringBuffer();
		while (true) {
			char c = next();
			if (c == end) {
				next();
				return sb.toString();
			}
			switch (c) {
			case '$':
				sb.append(variable());
				break;

			case 0:
				error("EOF, \" not found ");

			case '\\':
				c = escape();

			default:
				sb.append(c);
			}
		}
	}

	private char escape() {
		char cc = next();
		switch (cc) {
		case '"':
			return cc;

		case 'u':
			return (char) Integer.parseInt(program.subSequence(current,
					current + 4).toString(), 16);

		case 'r':
			return '\r';
		case 't':
			return '\t';
		case 'b':
			return '\b';
		case 'n':
			return '\n';
		}
		error("Illegal escape code: " + cc);
		return 0;
	}

	private Closure closure(char limit) {
		next();
		int start = current;
		int end = findMatching(limit);
		assert peek() == limit;
		next();
		return new Closure(this, program.subSequence(start, end));
	}

	int findMatching(char limit) {
		char c = 0;
		while ((c = next()) > 0) {
			if (c == limit)
				return current;

			switch (c) {
			case '{':
				findMatching('}');
				break;
			case '[':
				findMatching(']');
				break;
			case '(':
				findMatching(')');
				break;

			case '\\':
				next();
				break;
			}
		}
		error("Cannot match " + limit + " in " + program);
		return 0;
	}

	private char next() {
		if (current < program.length())
			current++;
		return peek();
	}

	char peek() {
		if (current >= program.length())
			return 0;

		return program.charAt(current);
	}

	private Object variable() throws Exception {
		if (!Character.isJavaIdentifierStart(next()))
			error("Not an identifier");

		return getProperty(simple().toString());
	}

	void error(String string) {
		int begin = Math.max(0, current - 20);
		int end = Math.min(program.length(), current + 20);
		throw new IllegalArgumentException(string + " in ..." +
				program.subSequence(begin, end) + "...");
	}

	Object simple() {
		skipWhiteSpace();
		char c = peek();
		int start = current;
		while (c > 0 && Character.isJavaIdentifierPart(c) ||
				".:%#@!~*^-_".indexOf(c) >= 0)
			c = next();

		CharSequence s = program.subSequence(start, current);
		if (NUMBER.matcher(s).matches()) {
			return new Integer(s.toString());
		} else
			return s;
	}

	private void skipWhiteSpace() {
		while (Character.isWhitespace(peek()))
			next();
	}

	public void _for(Closure init, Closure check, Closure incr, Closure body)
			throws Exception {
		check.parent = init;
		body.parent = init;
		incr.parent = init;
		init.expression();
		while (isTrue(check.eval())) {
			try {
				body.eval();
				incr.eval();
			} catch (FlowException be) {
				if (be.what == FlowException.BREAK)
					break;
				if (be.what == FlowException.RETURN)
					throw be;
			}
		}
	}

	public void print(Object o) {
		System.out.println(o);
	}

	public boolean isTrue(Object o) {
		if (o == null)
			return false;
		if (o instanceof Boolean)
			return ((Boolean) o).booleanValue();

		if (o instanceof Number) {
			double value = ((Number) o).doubleValue();
			return value != 0;
		}
		return false;
	}

	public String toString() {
		return "{" + program + "}";
	}

	public Object _new(Object args[]) throws Exception {
		assert args.length > 0;
		assert args[0] instanceof String;
		String name = (String) args[0];
		Class c = Class.forName(name);
		Object[] parms = new Object[args.length - 1];
		System.arraycopy(args, 1, parms, 0, parms.length);
		return Reference.newInstance(c, parms);
	}

	public Object _if(Closure condition, Closure body) throws Exception {
		if (isTrue(condition.eval()))
			return body.eval();
		else
			return null;
	}

	public Object _if(Closure condition, Closure body, Closure alternative)
			throws Exception {
		if (isTrue(condition.eval()))
			return body.eval();
		else
			return alternative.eval();
	}

	public Object _while(Closure condition, Closure body) throws Exception {
		Object result = null;
		while (isTrue(condition.eval()))
			result = body.eval();

		return result;
	}

	public Object function(String name, List args, Closure body) {
		Function f = new Function(name, args, body.program);
		getRoot().setProperty(name, f);
		return f;
	}
}
