//
// jja: swiss army knife for chess file formats
// src/eco.rs: ECO classification
//
// Copyright (c) 2023 Ali Polatel <alip@chesswob.org>
// Based in part upon Scid's scid.eco which is:
//     Copyright (C) 1999-2003  Shane Hudson (sgh@users.sourceforge.net)
//
// SPDX-License-Identifier: GPL-3.0-or-later

/// `ECO_MAX` is a constant representing the maximum number of ECO (Encyclopedia of Chess Openings)
/// codes
pub const ECO_MAX: usize = 10359;

/// `ECO` is an array of tuples containing ECO codes and their descriptions. Each tuple has three elements:
/// 1. The ECO code as a &str (e.g., "A00a")
/// 2. The name of the opening as a &str (e.g., "Start position")
/// 3. The moves of the opening as a &str (e.g., "*")
pub const ECO: [(&str, &str, & str); ECO_MAX] = [
("A00a", "Start position", "*"),
("A00b", "Barnes Opening", "1.f3 *"),
("A00b", "Fried fox", "1.f3 e5 2.Kf2 *"),
("A00c", "Kadas Opening", "1.h4 *"),
("A00d", "Clemenz Opening", "1.h3 *"),
("A00e", "Ware Opening", "1.a4 *"),
("A00f", "Anderssen Opening", "1.a3 *"),
("A00f", "Creepy Crawly Opening (Basman)", "1.a3 e5 2.h3 d5 *"),
("A00g", "Amar/Paris Opening", "1.Nh3 *"),
("A00g", "Amar: Paris Gambit", "1.Nh3 d5 2.g3 e5 3.f4 *"),
("A00h", "Durkin", "1.Na3 *"),
("A00i", "Saragossa", "1.c3 *"),
("A00j", "Mieses", "1.d3 *"),
("A00j", "Mieses: 1...e5", "1.d3 e5 *"),
("A00j", "Mieses: 1...d5", "1.d3 d5 *"),
("A00j", "Spike Deferred", "1.d3 g6 2.g4 *"),
("A00k", "Van Kruijs", "1.e3 *"),
("A00l", "Van Geet (Dunst) Opening", "1.Nc3 *"),
("A00l", "Van Geet: 1...Nf6", "1.Nc3 Nf6 *"),
("A00l", "Van Geet: 1...Nf6 2.Nf3", "1.Nc3 Nf6 2.Nf3 *"),
("A00l", "Van Geet: Tuebingen Gambit", "1.Nc3 Nf6 2.g4 *"),
("A00l", "Van Geet: 1...e5", "1.Nc3 e5 *"),
("A00l", "Van Geet: 1...e5 2.Nf3", "1.Nc3 e5 2.Nf3 *"),
("A00l", "Van Geet: Sicilian Variation", "1.Nc3 c5 *"),
("A00l", "Van Geet: Sicilian Variation, 2.Nf3", "1.Nc3 c5 2.Nf3 *"),
("A00l", "Van Geet: Sicilian Variation, 2.Nf3 Nc6", "1.Nc3 c5 2.Nf3 Nc6 *"),
("A00m", "Van Geet: 1...d5", "1.Nc3 d5 *"),
("A00m", "Van Geet: 1...d5 2.Nf3", "1.Nc3 d5 2.Nf3 *"),
("A00m", "Van Geet: 1...d5 2.Nf3 Nf6", "1.Nc3 d5 2.Nf3 Nf6 *"),
("A00m", "Van Geet: 1...d5 2.e4", "1.Nc3 d5 2.e4 *"),
("A00m", "Van Geet: 1...d5 2.e4 d4", "1.Nc3 d5 2.e4 d4 *"),
("A00m", "Van Geet: 1...d5 2.e4 dxe4", "1.Nc3 d5 2.e4 dxe4 *"),
("A00m", "Van Geet: Hector Gambit", "1.Nc3 d5 2.e4 dxe4 3.Bc4 *"),
("A00n", "Grob", "1.g4 *"),
("A00n", "Grob: Alessi Gambit", "1.g4 f5 *"),
("A00n", "Grob: Double Grob", "1.g4 g5 *"),
("A00n", "Grob: 1...e5", "1.g4 e5 *"),
("A00o", "Grob: 1...d5", "1.g4 d5 *"),
("A00o", "Grob Gambit", "1.g4 d5 2.Bg2 *"),
("A00o", "Grob Gambit: e5", "1.g4 d5 2.Bg2 e5 *"),
("A00o", "Grob Gambit: Hurst Attack", "1.g4 d5 2.Bg2 e5 3.c4 *"),
("A00o", "Grob Gambit: 2...c6", "1.g4 d5 2.Bg2 c6 *"),
("A00o", "Grob Gambit: Spike Attack", "1.g4 d5 2.Bg2 c6 3.g5 *"),
("A00o", "Grob Gambit Accepted", "1.g4 d5 2.Bg2 Bxg4 *"),
("A00o", "Grob Gambit Accepted: Fritz Gambit", "1.g4 d5 2.Bg2 Bxg4 3.c4 *"),
("A00p", "Polish (Sokolsky; Orang-Utan)", "1.b4 *"),
("A00p", "Polish: Birmingham Gambit", "1.b4 c5 *"),
("A00p", "Polish: 1...Nf6", "1.b4 Nf6 *"),
("A00p", "Polish: 1...Nf6 2.Bb2", "1.b4 Nf6 2.Bb2 *"),
("A00p", "Polish: 1...Nf6 2.Bb2 e6", "1.b4 Nf6 2.Bb2 e6 *"),
("A00p", "Polish: 1...c6", "1.b4 c6 *"),
("A00p", "Polish: Schuehler Gambit", "1.b4 c6 2.Bb2 a5 3.b5 *"),
("A00q", "Polish: 1...d5", "1.b4 d5 *"),
("A00q", "Polish: 1...d5 2.Bb2", "1.b4 d5 2.Bb2 *"),
("A00q", "Polish: 1...d5 2.Bb2 Bf5", "1.b4 d5 2.Bb2 Bf5 *"),
("A00q", "Polish: 1...d5 2.Bb2 Nf6", "1.b4 d5 2.Bb2 Nf6 *"),
("A00q", "Polish: 1...d5 2.Bb2 Nf6 3.e3", "1.b4 d5 2.Bb2 Nf6 3.e3 *"),
("A00r", "Polish: 1...e5", "1.b4 e5 *"),
("A00r", "Polish: Bugayev Attack", "1.b4 e5 2.a3 *"),
("A00r", "Polish: 1...e5 2.Bb2", "1.b4 e5 2.Bb2 *"),
("A00r", "Polish: Wolfertz Gambit", "1.b4 e5 2.Bb2 c5 *"),
("A00r", "Polish: 1...e5 2.Bb2 f6", "1.b4 e5 2.Bb2 f6 *"),
("A00r", "Polish: Tartakower Gambit", "1.b4 e5 2.Bb2 f6 3.e4 Bxb4 *"),
("A00r", "Polish: 1...e5 2.Bb2 d6", "1.b4 e5 2.Bb2 d6 *"),
("A00s", "Polish: 2...Bxb4", "1.b4 e5 2.Bb2 Bxb4 *"),
("A00s", "Polish: 2...Bxb4 3.Bxe5", "1.b4 e5 2.Bb2 Bxb4 3.Bxe5 *"),
("A00s", "Polish: 2...Bxb4 3.Bxe5 Nf6", "1.b4 e5 2.Bb2 Bxb4 3.Bxe5 Nf6 *"),
("A00s", "Polish: 2...Bxb4 3.Bxe5 Nf6 4.c4", "1.b4 e5 2.Bb2 Bxb4 3.Bxe5 Nf6 4.c4 *"),
("A00s", "Polish: 2...Bxb4 3.Bxe5 Nf6 4.Nf3", "1.b4 e5 2.Bb2 Bxb4 3.Bxe5 Nf6 4.Nf3 *"),
("A00t", "Benko Opening", "1.g3 *"),
("A00t", "Benko Opening", "1.g3 Nf6 *"),
("A00t", "Benko Opening: Symmetrical", "1.g3 g6 *"),
("A00u", "Benko Opening", "1.g3 e5 *"),
("A00v", "Benko Opening", "1.g3 d5 *"),
("A00v", "Benko Opening", "1.g3 d5 2.Bg2 *"),
("A00v", "Benko Opening", "1.g3 d5 2.Bg2 c6 *"),
("A00v", "Benko Opening", "1.g3 d5 2.Bg2 e5 *"),
("A00v", "Benko Opening", "1.g3 d5 2.Bg2 Nf6 *"),
("A01", "Nimzowitsch-Larsen Attack", "1.b3 *"),
("A01", "Nimzowitsch-Larsen: Polish Variation", "1.b3 b5 *"),
("A01", "Nimzowitsch-Larsen: Symmetrical", "1.b3 b6 *"),
("A01", "Nimzowitsch-Larsen: Dutch Variation", "1.b3 f5 *"),
("A01", "Nimzowitsch-Larsen: Ringelbach Gambit", "1.b3 f5 2.Bb2 e6 3.e4 *"),
("A01", "Nimzowitsch-Larsen: English Variation", "1.b3 c5 *"),
("A01", "Nimzowitsch-Larsen: Indian Variation", "1.b3 Nf6 *"),
("A01", "Nimzowitsch-Larsen: Indian Variation", "1.b3 Nf6 2.Bb2 g6 *"),
("A01", "Nimzowitsch-Larsen: Spike Variation", "1.b3 Nf6 2.Bb2 g6 3.g4 *"),
("A01", "Nimzowitsch-Larsen: 1...d5", "1.b3 d5 *"),
("A01", "Nimzowitsch-Larsen: 1...d5 2.Bb2", "1.b3 d5 2.Bb2 *"),
("A01", "Nimzowitsch-Larsen: 1...e5", "1.b3 e5 *"),
("A01", "Nimzowitsch-Larsen: 1...e5 2.Bb2", "1.b3 e5 2.Bb2 *"),
("A01", "Nimzowitsch-Larsen: 1...e5 2.Bb2 d6", "1.b3 e5 2.Bb2 d6 *"),
("A01", "Nimzowitsch-Larsen: 1...e5 2.Bb2 Nc6", "1.b3 e5 2.Bb2 Nc6 *"),
("A01", "Nimzowitsch-Larsen: Paschmann Gambit", "1.b3 e5 2.Bb2 Nc6 3.f4 *"),
("A01", "Nimzowitsch-Larsen: 1...e5 2.Bb2 Nc6 3.e3", "1.b3 e5 2.Bb2 Nc6 3.e3 *"),
("A02", "Bird", "1.f4 *"),
("A02", "Bird: Hobbs Gambit", "1.f4 g5 *"),
("A02", "Bird: Symmetrical", "1.f4 f5 *"),
("A02", "Bird: Swiss Gambit", "1.f4 f5 2.e4 *"),
("A02", "Bird: Swiss Gambit", "1.f4 f5 2.e4 fxe4 3.Nc3 Nf6 4.g4 *"),
("A02", "Bird: 1..d6", "1.f4 d6 *"),
("A02", "Bird: 1..g6", "1.f4 g6 *"),
("A02", "Bird: 1..g6", "1.f4 g6 2.Nf3 Bg7 3.e3 *"),
("A02", "Bird: 1..g6", "1.f4 g6 2.Nf3 Bg7 3.g3 *"),
("A02", "Bird: 1..c5", "1.f4 c5 *"),
("A02", "Bird: 1..c5 2.Nf3 Nc6", "1.f4 c5 2.Nf3 Nc6 *"),
("A02", "Bird: From Gambit", "1.f4 e5 *"),
("A02", "Bird: From Gambit Accepted", "1.f4 e5 2.fxe5 *"),
("A02", "Bird: From Gambit, Schlecter", "1.f4 e5 2.fxe5 Nc6 *"),
("A02", "Bird: From Gambit, 2...d6", "1.f4 e5 2.fxe5 d6 *"),
("A02", "Bird: From Gambit, 3.exd6", "1.f4 e5 2.fxe5 d6 3.exd6 *"),
("A02", "Bird: From Gambit, Langheld Gambit", "1.f4 e5 2.fxe5 d6 3.exd6 Nf6 *"),
("A02", "Bird: From Gambit, 3...Bxd6", "1.f4 e5 2.fxe5 d6 3.exd6 Bxd6 *"),
("A02", "Bird: From Gambit, Lipke", "1.f4 e5 2.fxe5 d6 3.exd6 Bxd6 4.Nf3 Nh6 5.d4 *"),
("A02", "Bird: From Gambit, Lasker Variation", "1.f4 e5 2.fxe5 d6 3.exd6 Bxd6 4.Nf3 g5 *"),
("A02", "Bird: From Gambit, Lasker, 5.d4", "1.f4 e5 2.fxe5 d6 3.exd6 Bxd6 4.Nf3 g5 5.d4 *"),
("A02", "Bird: From Gambit, Lasker, Queenswap Line", "1.f4 e5 2.fxe5 d6 3.exd6 Bxd6 4.Nf3 g5 5.d4 g4 6.Ne5 Bxe5 7.dxe5 Qxd1+ *"),
("A02", "Bird: From Gambit, Lasker, 5.g3", "1.f4 e5 2.fxe5 d6 3.exd6 Bxd6 4.Nf3 g5 5.g3 *"),
("A02", "Bird: 1..Nf6", "1.f4 Nf6 *"),
("A02", "Bird: 1..Nf6 2.g3", "1.f4 Nf6 2.g3 *"),
("A02", "Bird: 1..Nf6 2.e3", "1.f4 Nf6 2.e3 *"),
("A02", "Bird: 1..Nf6 2.b3", "1.f4 Nf6 2.b3 *"),
("A02", "Bird: 1..Nf6 2.Nf3", "1.f4 Nf6 2.Nf3 *"),
("A02", "Bird: 1..Nf6 2.Nf3 d6", "1.f4 Nf6 2.Nf3 d6 *"),
("A02", "Bird: 1..Nf6 2.Nf3 c5", "1.f4 Nf6 2.Nf3 c5 *"),
("A02", "Bird: 1..Nf6 2.Nf3 g6", "1.f4 Nf6 2.Nf3 g6 *"),
("A02", "Bird: Batavo Polish Attack", "1.f4 Nf6 2.Nf3 g6 3.b4 *"),
("A02", "Bird: 1..Nf6 2.Nf3 g6 3.g3", "1.f4 Nf6 2.Nf3 g6 3.g3 *"),
("A02", "Bird: 1..Nf6 2.Nf3 g6 3.g3", "1.f4 Nf6 2.Nf3 g6 3.g3 Bg7 4.Bg2 *"),
("A02", "Bird: 1..Nf6 2.Nf3 g6 3.g3", "1.f4 Nf6 2.Nf3 g6 3.g3 Bg7 4.Bg2 d6 *"),
("A03", "Bird: 1...d5", "1.f4 d5 *"),
("A03", "Bird: Dudweiler Gambit", "1.f4 d5 2.g4 *"),
("A03", "Bird: Sturm Gambit", "1.f4 d5 2.c4 *"),
("A03", "Bird: Williams Gambit", "1.f4 d5 2.e4 *"),
("A03", "Bird: 1...d5 2.b3", "1.f4 d5 2.b3 *"),
("A03", "Bird: 1...d5 2.b3", "1.f4 d5 2.b3 Nf6 *"),
("A03", "Bird: 1...d5 2.b3", "1.f4 d5 2.b3 Nf6 3.Bb2 *"),
("A03", "Bird: 1...d5 2.g3", "1.f4 d5 2.g3 *"),
("A03", "Bird: 1...d5 2.g3", "1.f4 d5 2.g3 Nf6 *"),
("A03", "Bird: 1...d5 2.g3", "1.f4 d5 2.g3 Nf6 3.Bg2 *"),
("A03", "Bird: Lasker Variation", "1.f4 d5 2.e3 *"),
("A03", "Bird: Lasker Variation", "1.f4 d5 2.e3 Nf6 *"),
("A03", "Bird: 1...d5 2.Nf3", "1.f4 d5 2.Nf3 *"),
("A03", "Bird: 1...d5 2.Nf3 c5", "1.f4 d5 2.Nf3 c5 *"),
("A03", "Bird: Batavo Gambit", "1.f4 d5 2.Nf3 c5 3.e4 *"),
("A03", "Bird: 1...d5 2.Nf3 c5 3.e3", "1.f4 d5 2.Nf3 c5 3.e3 *"),
("A03", "Bird: 1...d5 2.Nf3 g6", "1.f4 d5 2.Nf3 g6 *"),
("A03", "Bird: 1...d5 2.Nf3 g6 3.e3", "1.f4 d5 2.Nf3 g6 3.e3 *"),
("A03", "Bird: 1...d5 2.Nf3 g6 3.g3", "1.f4 d5 2.Nf3 g6 3.g3 *"),
("A03", "Bird: 1...d5 2.Nf3 g6 3.g3", "1.f4 d5 2.Nf3 g6 3.g3 Bg7 4.Bg2 *"),
("A03", "Bird: 1...d5 2.Nf3 Nf6", "1.f4 d5 2.Nf3 Nf6 *"),
("A03", "Bird: 1...d5 2.Nf3 Nf6 3.b3", "1.f4 d5 2.Nf3 Nf6 3.b3 *"),
("A03", "Bird: 1...d5 2.Nf3 Nf6 3.g3", "1.f4 d5 2.Nf3 Nf6 3.g3 *"),
("A03", "Bird: 1...d5 2.Nf3 Nf6 3.g3 g6", "1.f4 d5 2.Nf3 Nf6 3.g3 g6 *"),
("A03", "Bird: 1...d5 2.Nf3 Nf6 3.g3 g6", "1.f4 d5 2.Nf3 Nf6 3.g3 g6 4.Bg2 Bg7 *"),
("A03", "Bird: 1...d5 2.Nf3 Nf6 3.g3 g6: 5.d3", "1.f4 d5 2.Nf3 Nf6 3.g3 g6 4.Bg2 Bg7 5.d3 *"),
("A03", "Bird: 1...d5 2.Nf3 Nf6 3.g3 g6: 5.O-O", "1.f4 d5 2.Nf3 Nf6 3.g3 g6 4.Bg2 Bg7 5.O-O *"),
("A03", "Bird: 1...d5 2.Nf3 Nf6 3.g3 g6: 5.O-O O-O", "1.f4 d5 2.Nf3 Nf6 3.g3 g6 4.Bg2 Bg7 5.O-O O-O *"),
("A03", "Bird: 1...d5 2.Nf3 Nf6 3.g3 g6: 5.O-O O-O 6.d3", "1.f4 d5 2.Nf3 Nf6 3.g3 g6 4.Bg2 Bg7 5.O-O O-O 6.d3 *"),
("A03", "Bird: 1...d5 2.Nf3 Nf6 3.g3 g6: 6.d3 c6", "1.f4 d5 2.Nf3 Nf6 3.g3 g6 4.Bg2 Bg7 5.O-O O-O 6.d3 c6 *"),
("A03", "Bird: 1...d5 2.Nf3 Nf6 3.g3 g6: 6.d3 c5", "1.f4 d5 2.Nf3 Nf6 3.g3 g6 4.Bg2 Bg7 5.O-O O-O 6.d3 c5 *"),
("A03", "Bird: Lasker Variation", "1.f4 d5 2.Nf3 Nf6 3.e3 *"),
("A03", "Bird: Lasker, 3...Bg4", "1.f4 d5 2.Nf3 Nf6 3.e3 Bg4 *"),
("A03", "Bird: Lasker, 3...e6", "1.f4 d5 2.Nf3 Nf6 3.e3 e6 *"),
("A03", "Bird: Lasker, 3...c5", "1.f4 d5 2.Nf3 Nf6 3.e3 c5 *"),
("A03", "Bird: Lasker, 3...c5 4.b3", "1.f4 d5 2.Nf3 Nf6 3.e3 c5 4.b3 *"),
("A03", "Bird: Lasker, 3...g6", "1.f4 d5 2.Nf3 Nf6 3.e3 g6 *"),
("A04", "Reti", "1.Nf3 *"),
("A04", "Reti: Herrstrom Gambit", "1.Nf3 g5 *"),
("A04", "Reti: 1...b6", "1.Nf3 b6 *"),
("A04", "Reti: 1...b5", "1.Nf3 b5 *"),
("A04", "Reti: 1...Nc6", "1.Nf3 Nc6 *"),
("A04", "Reti: 1...e6", "1.Nf3 e6 *"),
("A04", "Reti: 1...e6", "1.Nf3 e6 2.g3 *"),
("A04", "Reti: 1...g6", "1.Nf3 g6 *"),
("A04", "Reti: 1...g6", "1.Nf3 g6 2.g3 *"),
("A04", "Reti: 1...g6", "1.Nf3 g6 2.g3 Bg7 *"),
("A04", "Reti: 1...g6", "1.Nf3 g6 2.g3 Bg7 3.Bg2 *"),
("A04", "Reti: 1...f5", "1.Nf3 f5 *"),
("A04", "Reti: 1...f5 2.d3", "1.Nf3 f5 2.d3 *"),
("A04", "Reti: 1...f5 2.d3 Nf6", "1.Nf3 f5 2.d3 Nf6 *"),
("A04", "Reti: Lisitsin Deferred", "1.Nf3 f5 2.d3 Nf6 3.e4 *"),
("A04", "Reti: 1...f5 2.g3", "1.Nf3 f5 2.g3 *"),
("A04", "Reti: Lisitsin", "1.Nf3 f5 2.e4 *"),
("A04", "Reti: Lisitsin: 3.Ng5 Nf6", "1.Nf3 f5 2.e4 fxe4 3.Ng5 Nf6 *"),
("A04", "Reti: Lisitsin: 3.Ng5 Nf6", "1.Nf3 f5 2.e4 fxe4 3.Ng5 Nf6 4.d3 e5 *"),
("A04", "Reti: Lisitsin: 3.Ng5 Nf6", "1.Nf3 f5 2.e4 fxe4 3.Ng5 Nf6 4.d3 e3 *"),
("A04", "Reti: Lisitsin: 3.Ng5 e5", "1.Nf3 f5 2.e4 fxe4 3.Ng5 e5 *"),
("A04", "Reti: Lisitsin: 3.Ng5 d5", "1.Nf3 f5 2.e4 fxe4 3.Ng5 d5 *"),
("A04", "Reti: 1...d6", "1.Nf3 d6 *"),
("A04", "Reti: 1...c5", "1.Nf3 c5 *"),
("A04", "Reti: 1...c5, Nimzowitsch-Larsen", "1.Nf3 c5 2.b3 *"),
("A04", "Reti: 1...c5", "1.Nf3 c5 2.g3 *"),
("A04", "Reti: 1...c5", "1.Nf3 c5 2.g3 b6 *"),
("A04", "Reti: 1...c5", "1.Nf3 c5 2.g3 b6 3.Bg2 Bb7 *"),
("A04", "Reti: 1...c5", "1.Nf3 c5 2.g3 g6 *"),
("A04", "Reti: 1...c5", "1.Nf3 c5 2.g3 g6 3.Bg2 Bg7 *"),
("A04", "Reti: 1...c5", "1.Nf3 c5 2.g3 g6 3.Bg2 Bg7 4.O-O Nc6 *"),
("A04", "Reti: 1...c5", "1.Nf3 c5 2.g3 g6 3.Bg2 Bg7 4.O-O Nc6 5.d3 *"),
("A04", "Reti: 1...c5", "1.Nf3 c5 2.g3 g6 3.Bg2 Bg7 4.O-O Nc6 5.d3 Nf6 6.e4 *"),
("A04", "Reti: 1...c5", "1.Nf3 c5 2.g3 g6 3.Bg2 Bg7 4.O-O Nc6 5.d3 Nf6 *"),
("A04", "Reti: 1...c5", "1.Nf3 c5 2.g3 g6 3.Bg2 Bg7 4.O-O Nc6 5.d3 d6 6.e4 *"),
("A04", "Reti: 1...c5", "1.Nf3 c5 2.g3 g6 3.Bg2 Bg7 4.O-O Nc6 5.d3 e6 6.e4 *"),
("A04", "Reti: 1...c5", "1.Nf3 c5 2.g3 Nc6 *"),
("A04", "Reti: 1...c5", "1.Nf3 c5 2.g3 Nc6 3.Bg2 *"),
("A05", "Reti: 1...Nf6", "1.Nf3 Nf6 *"),
("A05", "Reti: 1...Nf6 2.b3", "1.Nf3 Nf6 2.b3 *"),
("A05", "Reti: Santasiere's folly", "1.Nf3 Nf6 2.b4 *"),
("A05", "Reti: 1...Nf6 2.e3", "1.Nf3 Nf6 2.e3 *"),
("A05", "Reti: KIA", "1.Nf3 Nf6 2.g3 *"),
("A05", "Reti: KIA", "1.Nf3 Nf6 2.g3 c5 *"),
("A05", "Reti: KIA", "1.Nf3 Nf6 2.g3 c5 3.Bg2 *"),
("A05", "Reti: KIA", "1.Nf3 Nf6 2.g3 c5 3.Bg2 Nc6 *"),
("A05", "Reti: KIA", "1.Nf3 Nf6 2.g3 b6 *"),
("A05", "Reti: KIA", "1.Nf3 Nf6 2.g3 b6 3.Bg2 *"),
("A05", "Reti: KIA", "1.Nf3 Nf6 2.g3 b6 3.Bg2 Bb7 *"),
("A05", "Reti: KIA, Spassky", "1.Nf3 Nf6 2.g3 b5 *"),
("A05", "Reti: KIA, Spassky", "1.Nf3 Nf6 2.g3 b5 3.Bg2 *"),
("A05", "Reti: KIA", "1.Nf3 Nf6 2.g3 g6 *"),
("A05", "Reti: KIA, Reti-Smyslov Variation", "1.Nf3 Nf6 2.g3 g6 3.b4 *"),
("A05", "Reti: KIA, Reti-Smyslov Variation", "1.Nf3 Nf6 2.g3 g6 3.b4 Bg7 5.Bb2 *"),
("A05", "Reti: KIA", "1.Nf3 Nf6 2.g3 g6 3.Bg2 *"),
("A05", "Reti: KIA", "1.Nf3 Nf6 2.g3 g6 3.Bg2 Bg7 *"),
("A05", "Reti: KIA", "1.Nf3 Nf6 2.g3 g6 3.Bg2 Bg7 4.O-O *"),
("A05", "Reti: KIA", "1.Nf3 Nf6 2.g3 g6 3.Bg2 Bg7 4.O-O O-O *"),
("A05", "Reti: KIA", "1.Nf3 Nf6 2.g3 g6 3.Bg2 Bg7 4.O-O O-O 5.d3 *"),
("A05", "Reti: KIA", "1.Nf3 Nf6 2.g3 g6 3.Bg2 Bg7 4.O-O O-O 5.d3 c5 *"),
("A05", "Reti: KIA", "1.Nf3 Nf6 2.g3 g6 3.Bg2 Bg7 4.O-O O-O 5.d3 d6 *"),
("A06", "Reti: 1...d5", "1.Nf3 d5 *"),
("A06", "Reti: Ampel Variation", "1.Nf3 d5 2.Rg1 *"),
("A06", "Reti: Old Indian Attack", "1.Nf3 d5 2.d3 *"),
("A06", "Reti: Old Indian Attack", "1.Nf3 d5 2.d3 Nf6 *"),
("A06", "Reti: 1...d5 2.e3", "1.Nf3 d5 2.e3 *"),
("A06", "Reti: Santasiere's folly", "1.Nf3 d5 2.b4 *"),
("A06", "Reti: Santasiere's folly", "1.Nf3 d5 2.b4 Nf6 *"),
("A06", "Reti: Tennison/Zukertort Gambit", "1.Nf3 d5 2.e4 *"),
("A06", "Reti: Tennison Gambit Accepted", "1.Nf3 d5 2.e4 dxe4 *"),
("A06", "Reti: Nimzowitsch-Larsen", "1.Nf3 d5 2.b3 *"),
("A06", "Reti: Nimzowitsch-Larsen, 2...c5", "1.Nf3 d5 2.b3 c5 *"),
("A06", "Reti: Nimzowitsch-Larsen, 2...Bg4", "1.Nf3 d5 2.b3 Bg4 *"),
("A06", "Reti: Nimzowitsch-Larsen, 2...Bg4", "1.Nf3 d5 2.b3 Bg4 3.Bb2 *"),
("A06", "Reti: Nimzowitsch-Larsen, 2...Bg4", "1.Nf3 d5 2.b3 Bg4 3.Bb2 Nd7 4.e3 *"),
("A06", "Reti: Nimzowitsch-Larsen, 2...Nf6", "1.Nf3 d5 2.b3 Nf6 *"),
("A06", "Reti: Nimzowitsch-Larsen, 2...Nf6", "1.Nf3 d5 2.b3 Nf6 3.Bb2 *"),
("A06", "Reti: Nimzowitsch-Larsen, 2...Nf6", "1.Nf3 d5 2.b3 Nf6 3.Bb2 e6 *"),
("A06", "Reti: Nimzowitsch-Larsen, 2...Nf6", "1.Nf3 d5 2.b3 Nf6 3.Bb2 e6 4.e3 *"),
("A07", "Reti: KIA", "1.Nf3 d5 2.g3 *"),
("A07", "Reti: KIA", "1.Nf3 d5 2.g3 Nc6 *"),
("A07", "Reti: KIA", "1.Nf3 d5 2.g3 Nc6 3.Bg2 e5 *"),
("A07", "Reti: KIA", "1.Nf3 d5 2.g3 Nc6 3.Bg2 e5 4.d3 Nf6 *"),
("A07", "Reti: KIA", "1.Nf3 d5 2.g3 Nc6 3.Bg2 e5 4.d3 Nf6 5.O-O *"),
("A07", "Reti: KIA", "1.Nf3 d5 2.g3 Nc6 3.Bg2 e5 4.d3 Nf6 5.O-O Be7 *"),
("A07", "Reti: KIA", "1.Nf3 d5 2.g3 c6 *"),
("A07", "Reti: KIA, Yugoslav", "1.Nf3 d5 2.g3 c6 3.Bg2 Bg4 *"),
("A07", "Reti: KIA, Yugoslav", "1.Nf3 d5 2.g3 c6 3.Bg2 Bg4 4.O-O *"),
("A07", "Reti: KIA, Yugoslav", "1.Nf3 d5 2.g3 c6 3.Bg2 Bg4 4.O-O Nd7 *"),
("A07", "Reti: KIA, Yugoslav", "1.Nf3 d5 2.g3 c6 3.Bg2 Bg4 4.O-O Nd7 5.d3 *"),
("A07", "Reti: KIA, Yugoslav", "1.Nf3 d5 2.g3 Nf6 3.Bg2 c6 4.O-O Bg4 *"),
("A07", "Reti: KIA, Yugoslav", "1.Nf3 d5 2.g3 Nf6 3.Bg2 c6 4.O-O Bg4 5.d3 *"),
("A07", "Reti: KIA, Yugoslav, Main Line", "1.Nf3 d5 2.g3 Nf6 3.Bg2 c6 4.O-O Bg4 5.d3 Nbd7 *"),
("A07", "Reti: KIA, Yugoslav, Main Line, 6.Nbd2", "1.Nf3 d5 2.g3 Nf6 3.Bg2 c6 4.O-O Bg4 5.d3 Nbd7 6.Nbd2 *"),
("A07", "Reti: KIA, Yugoslav, Main Line, 6.Nbd2 e6", "1.Nf3 d5 2.g3 Nf6 3.Bg2 c6 4.O-O Bg4 5.d3 Nbd7 6.Nbd2 e6 *"),
("A07", "Reti: KIA, Yugoslav, Main Line, 6.Nbd2 e5", "1.Nf3 d5 2.g3 Nf6 3.Bg2 c6 4.O-O Bg4 5.d3 Nbd7 6.Nbd2 e5 *"),
("A07", "Reti: KIA", "1.Nf3 d5 2.g3 Bg4 *"),
("A07", "Reti: KIA, Keres Variation", "1.Nf3 d5 2.g3 Bg4 3.Bg2 Nd7 *"),
("A07", "Reti: KIA", "1.Nf3 d5 2.g3 Nf6 *"),
("A07", "Reti: KIA", "1.Nf3 d5 2.g3 Nf6 3.Bg2 *"),
("A07", "Reti: KIA, Neo-Gruenfeld", "1.Nf3 d5 2.g3 Nf6 3.Bg2 g6 *"),
("A07", "Reti: KIA", "1.Nf3 d5 2.g3 Nf6 3.Bg2 Bf5 *"),
("A07", "Reti: KIA", "1.Nf3 d5 2.g3 Nf6 3.Bg2 e6 *"),
("A07", "Reti: KIA", "1.Nf3 d5 2.g3 Nf6 3.Bg2 c6 *"),
("A07", "Reti: KIA, Petrosian Variation", "1.Nf3 d5 2.g3 Nf6 3.Bg2 c6 4.O-O Bf5 5.d3 e6 6.Nbd2 *"),
("A07", "Reti: KIA", "1.Nf3 d5 2.g3 g6 *"),
("A07", "Reti: KIA", "1.Nf3 d5 2.g3 g6 3.Bg2 *"),
("A07", "Reti: KIA, Pachman", "1.Nf3 d5 2.g3 g6 3.Bg2 Bg7 4.O-O e5 5.d3 Ne7 *"),
("A08", "Reti: KIA, 2...c5", "1.Nf3 d5 2.g3 c5 *"),
("A08", "Reti: KIA, 2...c5", "1.Nf3 d5 2.g3 c5 3.Bg2 *"),
("A08", "Reti: KIA, 2...c5 + 3...g6", "1.Nf3 d5 2.g3 c5 3.Bg2 g6 4.O-O Bg7 *"),
("A08", "Reti: KIA, 2...c5 + 3...g6", "1.Nf3 d5 2.g3 c5 3.Bg2 g6 4.O-O Bg7 5.d3 *"),
("A08", "Reti: KIA, 2...c5 + 3...g6", "1.Nf3 d5 2.g3 c5 3.Bg2 g6 4.O-O Bg7 5.d3 Nf6 *"),
("A08", "Reti: KIA, 2...c5 + 3...g6", "1.Nf3 d5 2.g3 c5 3.Bg2 g6 4.O-O Bg7 5.d3 Nf6 6.Nbd2 O-O *"),
("A08", "Reti: KIA, 2...c5 + 3...g6", "1.Nf3 d5 2.g3 c5 3.Bg2 g6 4.O-O Nc6 5.d3 Nf6 *"),
("A08", "Reti: KIA, 2...c5  3.Bg2 Nc6", "1.Nf3 d5 2.g3 c5 3.Bg2 Nc6 *"),
("A08", "Reti: KIA, 2...c5, 3.Bg2 Nc6 4.O-O", "1.Nf3 d5 2.g3 c5 3.Bg2 Nc6 4.O-O *"),
("A08", "Reti: KIA, 2...c5, 3.Bg2 Nc6 4.d4", "1.Nf3 d5 2.g3 c5 3.Bg2 Nc6 4.d4 *"),
("A08", "Reti: KIA, 2...c5 3.Bg2 Nf6", "1.Nf3 d5 2.g3 c5 3.Bg2 Nf6 *"),
("A08", "Reti: KIA, 2...c5 3.Bg2 Nf6 4.O-O", "1.Nf3 d5 2.g3 c5 3.Bg2 Nf6 4.O-O *"),
("A08", "Reti: KIA, 2...c5 3.Bg2 Nf6 4.O-O Nc6", "1.Nf3 d5 2.g3 c5 3.Bg2 Nf6 4.O-O Nc6 *"),
("A08", "Reti: KIA, French Variation", "1.Nf3 d5 2.g3 c5 3.Bg2 Nc6 4.O-O e6 5.d3 Nge7 6.Nbd2 b6 7.e4 *"),
("A08", "Reti: KIA, French Variation", "1.Nf3 d5 2.g3 c5 3.Bg2 Nc6 4.O-O e6 5.d3 Nf6 6.Nbd2 b6 7.e4 *"),
("A08", "Reti: KIA, French Variation", "1.Nf3 d5 2.g3 c5 3.Bg2 Nc6 4.O-O e6 5.d3 Nge7 6.Nbd2 g6 7.e4 Bg7 *"),
("A08", "Reti: KIA, French Variation", "1.Nf3 d5 2.g3 c5 3.Bg2 Nc6 4.O-O e6 5.d3 Nf6 6.Nbd2 g6 7.e4 Bg7 *"),
("A08", "Reti: KIA, French Variation", "1.Nf3 d5 2.g3 c5 3.Bg2 Nc6 4.O-O e6 5.d3 Nf6 6.Nbd2 Be7 7.e4 *"),
("A08", "Reti: KIA, French Variation", "1.Nf3 d5 2.g3 c5 3.Bg2 Nc6 4.O-O e6 5.d3 Nf6 6.Nbd2 Be7 7.e4 O-O 8.Re1 *"),
("A08", "Reti: KIA, French Variation", "1.Nf3 d5 2.g3 c5 3.Bg2 Nc6 4.O-O e6 5.d3 Bd6 6.Nbd2 Nge7 7.e4 *"),
("A08", "Reti: KIA, French Variation", "1.Nf3 d5 2.g3 c5 3.Bg2 Nc6 4.O-O e6 5.d3 Bd6 6.Nbd2 Nge7 7.e4 O-O 8.Re1 *"),
("A09a", "Reti: 2.c4", "1.Nf3 d5 2.c4 *"),
("A09b", "Reti: Advance Variation", "1.Nf3 d5 2.c4 d4 *"),
("A09c", "Reti: Advance, Anglo-Polish Attack", "1.Nf3 d5 2.c4 d4 3.b4 *"),
("A09c", "Reti: Advance, Anglo-Polish, 3...g6", "1.Nf3 d5 2.c4 d4 3.b4 g6 *"),
("A09d", "Reti: Advance, 3.e3", "1.Nf3 d5 2.c4 d4 3.e3 *"),
("A09e", "Reti: Advance, 3.e3 c5", "1.Nf3 d5 2.c4 d4 3.e3 c5 *"),
("A09f", "Reti: Advance, 3.e3 Nc6", "1.Nf3 d5 2.c4 d4 3.e3 Nc6 *"),
("A09f", "Reti: Advance, 3.e3 Nc6 4.exd4 Nxd4", "1.Nf3 d5 2.c4 d4 3.e3 Nc6 4.exd4 Nxd4 *"),
("A09g", "Reti: Advance, 3.g3", "1.Nf3 d5 2.c4 d4 3.g3 *"),
("A09h", "Reti: Advance, 3.g3 Nc6", "1.Nf3 d5 2.c4 d4 3.g3 Nc6 *"),
("A09h", "Reti: Advance, 3.g3 Nc6 4.Bg2 e5", "1.Nf3 d5 2.c4 d4 3.g3 Nc6 4.Bg2 e5 *"),
("A09i", "Reti: Advance, 3.g3 g6", "1.Nf3 d5 2.c4 d4 3.g3 g6 *"),
("A09i", "Reti: Advance, 3.g3 g6 4.Bg2 Bg7", "1.Nf3 d5 2.c4 d4 3.g3 g6 4.Bg2 Bg7 *"),
("A09j", "Reti: Advance, 3.g3 c5", "1.Nf3 d5 2.c4 d4 3.g3 c5 *"),
("A09j", "Reti: Advance, 3.g3 c5 4.Bg2 Nc6", "1.Nf3 d5 2.c4 d4 3.g3 c5 4.Bg2 Nc6 *"),
("A09j", "Reti: Advance, 3.g3 c5 4.Bg2 Nc6 5.d3 e5", "1.Nf3 d5 2.c4 d4 3.g3 c5 4.Bg2 Nc6 5.d3 e5 *"),
("A09k", "Reti: Accepted", "1.Nf3 d5 2.c4 dxc4 *"),
("A09l", "Reti: Accepted, 3.g3", "1.Nf3 d5 2.c4 dxc4 3.g3 *"),
("A09m", "Reti: Accepted, 3.g3 e6", "1.Nf3 d5 2.c4 dxc4 3.g3 e6 *"),
("A09n", "Reti: Accepted, 3.Qa4+", "1.Nf3 d5 2.c4 dxc4 3.Qa4+ *"),
("A09o", "Reti: Accepted, 3.Na3", "1.Nf3 d5 2.c4 dxc4 3.Na3 *"),
("A09p", "Reti: Accepted, 3.Na3 a6", "1.Nf3 d5 2.c4 dxc4 3.Na3 a6 *"),
("A09q", "Reti: Accepted, 3.Na3 c5", "1.Nf3 d5 2.c4 dxc4 3.Na3 c5 *"),
("A09r", "Reti: Accepted, 3.e3", "1.Nf3 d5 2.c4 dxc4 3.e3 *"),
("A09s", "Reti: Accepted, Keres Variation", "1.Nf3 d5 2.c4 dxc4 3.e3 Be6 *"),
("A09t", "Reti: Accepted, 3.e3 Nf6", "1.Nf3 d5 2.c4 dxc4 3.e3 Nf6 *"),
("A09u", "Reti: Accepted, 3.e3 Nf6 4.Bxc4 e6", "1.Nf3 d5 2.c4 dxc4 3.e3 Nf6 4.Bxc4 e6 *"),
("A10", "English", "1.c4 *"),
("A10", "English: 1...g5", "1.c4 g5 *"),
("A10", "English: 1...g5 2.d4", "1.c4 g5 2.d4 *"),
("A10", "English: Myers Gambit", "1.c4 g5 2.d4 Bg7 *"),
("A10", "English: 1...Nc6", "1.c4 Nc6 *"),
("A10", "English: 1...Nc6 2.Nc3", "1.c4 Nc6 2.Nc3 *"),
("A10", "English: Jaenisch Gambit", "1.c4 b5 *"),
("A10", "English: Vector", "1.c4 d5 *"),
("A10", "English: 1...b6", "1.c4 b6 *"),
("A10", "English: 1...b6 2.Nf3", "1.c4 b6 2.Nf3 *"),
("A10", "English: 1...b6 2.Nf3 Bb7", "1.c4 b6 2.Nf3 Bb7 *"),
("A10", "English: 1...b6 2.Nc3", "1.c4 b6 2.Nc3 *"),
("A10", "English: 1...b6 2.Nc3 e6", "1.c4 b6 2.Nc3 e6 *"),
("A10", "English: 1...b6 2.Nc3 e6 3.e4", "1.c4 b6 2.Nc3 e6 3.e4 *"),
("A10", "English: 1...b6 2.Nc3 Bb7", "1.c4 b6 2.Nc3 Bb7 *"),
("A10", "English: 1...b6 2.Nc3 Bb7 3.e4", "1.c4 b6 2.Nc3 Bb7 3.e4 *"),
("A10", "English: 1...b6 2.Nc3 Bb7 3.e4 e6", "1.c4 b6 2.Nc3 Bb7 3.e4 e6 *"),
("A10", "English: 1...d6", "1.c4 d6 *"),
("A10", "English: 1...d6", "1.c4 d6 2.Nc3 *"),
("A10", "English: 1...d6", "1.c4 d6 2.Nf3 *"),
("A10", "English: 1...g6", "1.c4 g6 *"),
("A10", "English: 1...g6 2.g3", "1.c4 g6 2.g3 *"),
("A10", "English: 1...g6 2.Nc3", "1.c4 g6 2.Nc3 *"),
("A10", "English: 1...g6 2.Nc3 Bg7", "1.c4 g6 2.Nc3 Bg7 *"),
("A10", "English: 1...g6 2.Nc3 Bg7 3.g3", "1.c4 g6 2.Nc3 Bg7 3.g3 *"),
("A10", "English: 1...g6", "1.c4 g6 2.Nf3 *"),
("A10", "English: 1...g6", "1.c4 g6 2.Nf3 Bg7 *"),
("A10", "English: 1...g6 2.e4", "1.c4 g6 2.e4 *"),
("A10", "English: Adorjan Defence", "1.c4 g6 2.e4 e5 *"),
("A10", "English: Anglo-Dutch", "1.c4 f5 *"),
("A10", "English: Wade Gambit", "1.c4 f5 2.g4 *"),
("A10", "English: Anglo-Dutch", "1.c4 f5 2.g3 *"),
("A10", "English: Anglo-Dutch", "1.c4 f5 2.g3 Nf6 *"),
("A10", "English: Anglo-Dutch", "1.c4 f5 2.g3 Nf6 3.Bg2 *"),
("A10", "English: Anglo-Dutch", "1.c4 f5 2.Nc3 *"),
("A10", "English: Anglo-Dutch", "1.c4 f5 2.Nc3 Nf6 *"),
("A10", "English: Anglo-Dutch", "1.c4 f5 2.Nc3 Nf6 3.g3 *"),
("A10", "English: Anglo-Dutch", "1.c4 f5 2.Nc3 Nf6 3.g3 g6 *"),
("A10", "English: Anglo-Dutch", "1.c4 f5 2.Nf3 *"),
("A10", "English: Anglo-Dutch", "1.c4 f5 2.Nf3 e6 *"),
("A10", "English: Anglo-Dutch", "1.c4 f5 2.Nf3 Nf6 *"),
("A10", "English: Anglo-Dutch", "1.c4 f5 2.Nf3 Nf6 3.Nc3 *"),
("A10", "English: Anglo-Dutch", "1.c4 f5 2.Nf3 Nf6 3.g3 *"),
("A10", "English: Anglo-Dutch", "1.c4 f5 2.Nf3 Nf6 3.g3 e6 *"),
("A10", "English: Anglo-Dutch", "1.c4 f5 2.Nf3 Nf6 3.g3 e6 4.Bg2 *"),
("A10", "English: Anglo-Dutch", "1.c4 f5 2.Nf3 Nf6 3.g3 e6 4.Bg2 c6 5.O-O d5 *"),
("A10", "English: Anglo-Dutch", "1.c4 f5 2.Nf3 Nf6 3.g3 e6 4.Bg2 Be7 *"),
("A10", "English: Anglo-Dutch", "1.c4 f5 2.Nf3 Nf6 3.g3 e6 4.Bg2 Be7 5.O-O *"),
("A10", "English: Anglo-Dutch", "1.c4 f5 2.Nf3 Nf6 3.g3 e6 4.Bg2 Be7 5.O-O O-O *"),
("A10", "English: Anglo-Dutch", "1.c4 f5 2.Nf3 Nf6 3.g3 e6 4.Bg2 Be7 5.O-O O-O 6.Nc3 *"),
("A11", "English: Caro-Kann Defence", "1.c4 c6 *"),
("A11", "English: Caro-Kann Defence", "1.c4 c6 2.g3 *"),
("A11", "English: Caro-Kann Defence", "1.c4 c6 2.g3 Nf6 *"),
("A11", "English: Caro-Kann Defence", "1.c4 c6 2.g3 Nf6 3.Bg2 d5 *"),
("A11", "English: Caro-Kann Defence", "1.c4 c6 2.g3 Nf6 3.Bg2 d5 4.Nf3 *"),
("A11", "English: Caro-Kann Defence", "1.c4 c6 2.g3 Nf6 3.Bg2 d5 4.Nf3 Bf5 *"),
("A11", "English: Caro-Kann Defence", "1.c4 c6 2.g3 Nf6 3.Bg2 d5 4.Nf3 Bf5 5.O-O *"),
("A11", "English: Caro-Kann Defence", "1.c4 c6 2.g3 Nf6 3.Bg2 d5 4.Nf3 Bf5 5.O-O e6 6.d3 *"),
("A11", "English: Caro-Kann Defence", "1.c4 c6 2.g3 Nf6 3.Bg2 d5 4.Nf3 Bg4 *"),
("A11", "English: Caro-Kann Defence", "1.c4 c6 2.g3 Nf6 3.Bg2 d5 4.Nf3 Bg4 5.O-O *"),
("A11", "English: Caro-Kann Defence", "1.c4 c6 2.Nc3 *"),
("A11", "English: Caro-Kann Defence", "1.c4 c6 2.Nc3 d5 *"),
("A11", "English: Caro-Kann Defence", "1.c4 c6 2.Nf3 *"),
("A11", "English: Caro-Kann Defence", "1.c4 c6 2.Nf3 Nf6 *"),
("A11", "English: Caro-Kann Defence", "1.c4 c6 2.Nf3 d5 *"),
("A11", "English: Caro-Kann Defence, 3.g3", "1.c4 c6 2.Nf3 d5 3.g3 *"),
("A11", "English: Caro-Kann Defence, 3.g3 Bg4", "1.c4 c6 2.Nf3 d5 3.g3 Bg4 *"),
("A11", "English: Caro-Kann Defence, 3.g3 Bg4", "1.c4 c6 2.Nf3 d5 3.g3 Bg4 4.Bg2 *"),
("A11", "English: Caro-Kann Defence, 3.e3", "1.c4 c6 2.Nf3 d5 3.e3 *"),
("A11", "English: Caro-Kann Defence, 3.e3 Nf6", "1.c4 c6 2.Nf3 d5 3.e3 Nf6 *"),
("A11", "English: Caro-Kann Defence, 3.e3 Nf6 4.Nc3", "1.c4 c6 2.Nf3 d5 3.e3 Nf6 4.Nc3 *"),
("A12", "English: Caro-Kann Defence, 3.b3", "1.c4 c6 2.Nf3 d5 3.b3 *"),
("A12", "English: Torre Defence", "1.c4 c6 2.Nf3 d5 3.b3 Nf6 4.g3 Bg4 *"),
("A12", "English: Torre Defence", "1.c4 c6 2.Nf3 d5 3.b3 Nf6 4.g3 Bg4 5.Bg2 *"),
("A12", "English: Torre Defence", "1.c4 c6 2.Nf3 d5 3.b3 Nf6 4.g3 Bg4 5.Bg2 e6 *"),
("A12", "English: Torre Defence", "1.c4 c6 2.Nf3 d5 3.b3 Nf6 4.g3 Bg4 5.Bg2 e6 6.Bb2 *"),
("A12", "English: London Defence", "1.c4 c6 2.Nf3 d5 3.b3 Nf6 4.g3 Bf5 *"),
("A12", "English: London Defence", "1.c4 c6 2.Nf3 d5 3.b3 Nf6 4.g3 Bf5 5.Bg2 *"),
("A12", "English: London Defence", "1.c4 c6 2.Nf3 d5 3.b3 Nf6 4.g3 Bf5 5.Bg2 e6 *"),
("A12", "English: London Defence", "1.c4 c6 2.Nf3 d5 3.b3 Nf6 4.g3 Bf5 5.Bg2 e6 6.Bb2 *"),
("A12", "English: Caro-Kann Defence, 3.b3", "1.c4 c6 2.Nf3 d5 3.b3 Nf6 4.Bb2 *"),
("A12", "English: Bled Variation", "1.c4 c6 2.Nf3 d5 3.b3 Nf6 4.Bb2 g6 *"),
("A12", "English: Bled Variation", "1.c4 c6 2.Nf3 d5 3.b3 Nf6 4.Bb2 g6 5.e3 Bg7 *"),
("A12", "English: New York/London Defence", "1.c4 c6 2.Nf3 d5 3.b3 Nf6 4.Bb2 Bf5 *"),
("A12", "English: Capablanca", "1.c4 c6 2.Nf3 d5 3.b3 Nf6 4.Bb2 Bg4 *"),
("A12", "English: Bogoljubow Variation", "1.c4 c6 2.Nf3 d5 3.b3 Bg4 *"),
("A13a", "English: 1...e6", "1.c4 e6 *"),
("A13b", "English: 1...e6 2.g3", "1.c4 e6 2.g3 *"),
("A13c", "English: 1...e6 2.g3 d5", "1.c4 e6 2.g3 d5 *"),
("A13d", "English: 1...e6 2.g3 d5", "1.c4 e6 2.g3 d5 3.Bg2 *"),
("A13e", "English: 1...e6 2.Nc3", "1.c4 e6 2.Nc3 *"),
("A13f", "English: 1...e6 2.Nc3 Bb4", "1.c4 e6 2.Nc3 Bb4 *"),
("A13g", "English: 1...e6 2.Nc3 d5", "1.c4 e6 2.Nc3 d5 *"),
("A13h", "English: 1...e6 2.Nf3", "1.c4 e6 2.Nf3 *"),
("A13i", "English: 1...e6 2.Nf3 Nf6", "1.c4 e6 2.Nf3 Nf6 *"),
("A13j", "English: 1...e6 2.Nf3 Nf6 3.g3", "1.c4 e6 2.Nf3 Nf6 3.g3 *"),
("A13k", "English: Romanishin Gambit", "1.c4 e6 2.Nf3 Nf6 3.g3 a6 4.Bg2 b5 *"),
("A13l", "English: 1...e6 2.Nf3 d5", "1.c4 e6 2.Nf3 d5 *"),
("A13m", "English: Agincourt Variation", "1.c4 e6 2.Nf3 d5 3.b3 *"),
("A13n", "English: Wimpey System", "1.c4 e6 2.Nf3 d5 3.b3 Nf6 4.Bb2 c5 5.e3 *"),
("A13n", "English: Wimpey System", "1.c4 e6 2.Nf3 d5 3.b3 Nf6 4.Bb2 c5 5.e3 Nc6 *"),
("A13o", "English: Agincourt Variation", "1.c4 e6 2.Nf3 d5 3.g3 *"),
("A13p", "English: Kurajica Defence", "1.c4 e6 2.Nf3 d5 3.g3 c6 *"),
("A13q", "English: Kurajica Defence", "1.c4 e6 2.Nf3 d5 3.g3 c6 4.Qc2 *"),
("A13r", "English: Neo-Catalan", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 *"),
("A13s", "English: Neo-Catalan", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 *"),
("A13s", "English: Neo-Catalan, 4...c6", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 c6 *"),
("A13s", "English: Neo-Catalan, 4...c6 5.b3", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 c6 5.b3 *"),
("A13t", "English: Neo-Catalan, 4...c5", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 c5 *"),
("A13t", "English: Neo-Catalan, 4...c5 5.O-O", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 c5 5.O-O *"),
("A13u", "English: Neo-Catalan Accepted", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 dxc4 *"),
("A13v", "English: Neo-Catalan Accepted, 5.Qa4+", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 dxc4 5.Qa4+ *"),
("A13v", "English: Neo-Catalan Accepted, 5.Qa4+ Nbd7", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 dxc4 5.Qa4+ Nbd7 *"),
("A13v", "English: Neo-Catalan Accepted, 5.Qa4+ Nbd7 6.O-O", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 dxc4 5.Qa4+ Nbd7 6.O-O *"),
("A13w", "English: Neo-Catalan Accepted, 5.Qa4+ Nbd7 6.Qxc4", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 dxc4 5.Qa4+ Nbd7 6.Qxc4 *"),
("A13w", "English: Neo-Catalan Accepted, 5.Qa4+ Nbd7 6.Qxc4 a6", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 dxc4 5.Qa4+ Nbd7 6.Qxc4 a6 *"),
("A13w", "English: Neo-Catalan Accepted, 5.Qa4+ Nbd7 6.Qxc4 c5", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 dxc4 5.Qa4+ Nbd7 6.Qxc4 c5 *"),
("A14", "English: Neo-Catalan Declined", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 Be7 *"),
("A14", "English: Neo-Catalan Declined", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 Be7 5.O-O *"),
("A14", "English: Neo-Catalan Declined, 5...c6", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 Be7 5.O-O c6 *"),
("A14", "English: Neo-Catalan Declined, 5...c5", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 Be7 5.O-O c5 *"),
("A14", "English: Neo-Catalan Declined, 5...O-O", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 Be7 5.O-O O-O *"),
("A14", "English: Neo-Catalan Declined, 5...O-O 6.b3", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 Be7 5.O-O O-O 6.b3 *"),
("A14", "English: Neo-Catalan Declined, 5...O-O 6.b3 b6", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 Be7 5.O-O O-O 6.b3 b6 *"),
("A14", "English: Neo-Catalan Declined, 5...O-O 6.b3 b6 7.Bb2 Bb7", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 Be7 5.O-O O-O 6.b3 b6 7.Bb2 Bb7 *"),
("A14", "English: Neo-Catalan Declined, 5...O-O 6.b3 b6 7.Bb2 Bb7 8.e3", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 Be7 5.O-O O-O 6.b3 b6 7.Bb2 Bb7 8.e3 *"),
("A14", "English: Neo-Catalan Declined, 5...O-O 6.b3 c5", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 Be7 5.O-O O-O 6.b3 c5 *"),
("A14", "English: Neo-Catalan Declined, 5...O-O 6.b3 c5 7.Bb2", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 Be7 5.O-O O-O 6.b3 c5 7.Bb2 *"),
("A14", "English: Neo-Catalan Declined, 5...O-O 6.b3 c5 7.Bb2 Nc6", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 Be7 5.O-O O-O 6.b3 c5 7.Bb2 Nc6 *"),
("A14", "English: Neo-Catalan Declined, 5...O-O 6.b3 c5 7.Bb2 Nc6 8.e3", "1.c4 e6 2.Nf3 d5 3.g3 Nf6 4.Bg2 Be7 5.O-O O-O 6.b3 c5 7.Bb2 Nc6 8.e3 *"),
("A15", "English: Anglo-Indian", "1.c4 Nf6 *"),
("A15", "English: Anglo-Indian, Polish", "1.c4 Nf6 2.b4 *"),
("A15", "English: Anglo-Indian, 2.g3", "1.c4 Nf6 2.g3 *"),
("A15", "English: Anglo-Indian, 2.g3 e6", "1.c4 Nf6 2.g3 e6 *"),
("A15", "English: Anglo-Indian, 2.g3 e6", "1.c4 Nf6 2.g3 e6 3.Bg2 *"),
("A15", "English: Anglo-Indian, 2.g3 g6", "1.c4 Nf6 2.g3 g6 *"),
("A15", "English: Anglo-Indian, 2.g3 g6", "1.c4 Nf6 2.g3 g6 3.Bg2 Bg7 *"),
("A15", "English: Anglo-Indian, 2.Nf3", "1.c4 Nf6 2.Nf3 *"),
("A15", "English: Anglo-Indian, 2.Nf3 g6", "1.c4 Nf6 2.Nf3 g6 *"),
("A15", "English: Anglo-Indian, 2.Nf3 g6", "1.c4 Nf6 2.Nf3 g6 3.g3 Bg7 4.Bg2 *"),
("A15", "English: Anglo-Indian, 2.Nf3 g6", "1.c4 Nf6 2.Nf3 g6 3.g3 Bg7 4.Bg2 O-O *"),
("A16", "English: Anglo-Indian, 2.Nc3", "1.c4 Nf6 2.Nc3 *"),
("A16", "English: Anglo-Indian, 2.Nc3 Nc6", "1.c4 Nf6 2.Nc3 Nc6 *"),
("A16", "English: Anglo-Indian, 2.Nc3 c6", "1.c4 Nf6 2.Nc3 c6 *"),
("A16", "English: Anglo-Indian, 2.Nc3 c6 3.e4", "1.c4 Nf6 2.Nc3 c6 3.e4 *"),
("A16", "English: Anglo-Indian, 2.Nc3 c6 3.e4 d5", "1.c4 Nf6 2.Nc3 c6 3.e4 d5 *"),
("A16", "English: Anglo-Indian, 2.Nc3 c6 3.e4 e5", "1.c4 Nf6 2.Nc3 c6 3.e4 e5 *"),
("A16", "English: Anglo-Indian, 2.Nc3 d6", "1.c4 Nf6 2.Nc3 d6 *"),
("A16", "English: Anglo-Indian, 2.Nc3 g6", "1.c4 Nf6 2.Nc3 g6 *"),
("A16", "English: Anglo-Indian, 2.Nc3 g6 3.e4", "1.c4 Nf6 2.Nc3 g6 3.e4 *"),
("A16", "English: Anglo-Indian, 2.Nc3 g6 3.g3", "1.c4 Nf6 2.Nc3 g6 3.g3 *"),
("A16", "English: Anglo-Indian, 2.Nc3 g6 3.g3", "1.c4 Nf6 2.Nc3 g6 3.g3 Bg7 *"),
("A16", "English: Anglo-Indian, 2.Nc3 g6 3.g3", "1.c4 Nf6 2.Nc3 g6 3.g3 Bg7 4.Bg2 *"),
("A16", "English: Anglo-Indian, 2.Nc3 g6 3.g3", "1.c4 Nf6 2.Nc3 g6 3.g3 Bg7 4.Bg2 O-O *"),
("A16", "English: Anglo-Indian, 2.Nc3 g6 3.g3", "1.c4 Nf6 2.Nc3 g6 3.g3 Bg7 4.Bg2 O-O 5.e4 *"),
("A16", "English: Anglo-Indian, 2.Nc3 g6 3.g3", "1.c4 Nf6 2.Nc3 g6 3.g3 Bg7 4.Bg2 O-O 5.Nf3 *"),
("A16", "English: Anglo-Indian, 2.Nc3 g6 3.g3", "1.c4 Nf6 2.Nc3 g6 3.g3 Bg7 4.Bg2 O-O 5.Nf3 d6 6.O-O *"),
("A16", "English: Anglo-Gruenfeld", "1.c4 Nf6 2.Nc3 d5 *"),
("A16", "English: Anglo-Gruenfeld", "1.c4 Nf6 2.Nc3 d5 3.Nf3 *"),
("A16", "English: Anglo-Gruenfeld", "1.c4 Nf6 2.Nc3 d5 3.Nf3 g6 *"),
("A16", "English: Anglo-Gruenfeld", "1.c4 Nf6 2.Nc3 d5 3.Nf3 g6 4.Qa4+ *"),
("A16", "English: Anglo-Gruenfeld", "1.c4 Nf6 2.Nc3 d5 3.Nf3 g6 4.g3 *"),
("A16", "English: Anglo-Gruenfeld, 3.cxd5", "1.c4 Nf6 2.Nc3 d5 3.cxd5 *"),
("A16", "English: Anglo-Gruenfeld, Smyslov Variation", "1.c4 Nf6 2.Nc3 d5 3.cxd5 Nxd5 4.g3 g6 5.Bg2 Nxc3 *"),
("A16", "English: Anglo-Gruenfeld, Smyslov Variation", "1.c4 Nf6 2.Nc3 d5 3.cxd5 Nxd5 4.g3 g6 5.Bg2 Nxc3 6.bxc3 *"),
("A16", "English: Anglo-Gruenfeld, Smyslov Variation", "1.c4 Nf6 2.Nc3 d5 3.cxd5 Nxd5 4.g3 g6 5.Bg2 Nxc3 6.bxc3 Bg7 *"),
("A16", "English: Anglo-Gruenfeld, Smyslov, 7.Rb1", "1.c4 Nf6 2.Nc3 d5 3.cxd5 Nxd5 4.g3 g6 5.Bg2 Nxc3 6.bxc3 Bg7 7.Rb1 *"),
("A16", "English: Anglo-Gruenfeld, Czech Variation", "1.c4 Nf6 2.Nc3 d5 3.cxd5 Nxd5 4.g3 g6 5.Bg2 Nb6 *"),
("A16", "English: Anglo-Gruenfeld, 4.Nf3", "1.c4 Nf6 2.Nc3 d5 3.cxd5 Nxd5 4.Nf3 *"),
("A16", "English: Anglo-Gruenfeld, 4.Nf3 g6", "1.c4 Nf6 2.Nc3 d5 3.cxd5 Nxd5 4.Nf3 g6 *"),
("A16", "English: Anglo-Gruenfeld, 4.Nf3 g6 5.g3", "1.c4 Nf6 2.Nc3 d5 3.cxd5 Nxd5 4.Nf3 g6 5.g3 *"),
("A16", "English: Anglo-Gruenfeld, 4.Nf3 g6 5.g3", "1.c4 Nf6 2.Nc3 d5 3.cxd5 Nxd5 4.Nf3 g6 5.g3 Bg7 *"),
("A16", "English: Anglo-Gruenfeld, Korchnoi Variation", "1.c4 Nf6 2.Nc3 d5 3.cxd5 Nxd5 4.Nf3 g6 5.g3 Bg7 6.Bg2 e5 *"),
("A16", "English: Anglo-Gruenfeld, Andersson-Boeoek Variation", "1.c4 Nf6 2.Nc3 d5 3.cxd5 Nxd5 4.Nf3 g6 5.e4 *"),
("A16", "English: Anglo-Gruenfeld, Andersson-Boeoek, Queenswap", "1.c4 Nf6 2.Nc3 d5 3.cxd5 Nxd5 4.Nf3 g6 5.e4 Nxc3 6.dxc3 Qxd1+ 7.Kxd1 *"),
("A16", "English: Anglo-Gruenfeld, 4.Nf3 g6 5.Qa4+", "1.c4 Nf6 2.Nc3 d5 3.cxd5 Nxd5 4.Nf3 g6 5.Qa4+ *"),
("A17", "English: Anglo-Indian, 2.Nc3 e6", "1.c4 Nf6 2.Nc3 e6 *"),
("A17", "English: Anglo-Indian, 2.Nc3 e6 3.g3", "1.c4 Nf6 2.Nc3 e6 3.g3 *"),
("A17", "English: Anglo-Indian, 2.Nc3 e6 3.Nf3", "1.c4 Nf6 2.Nc3 e6 3.Nf3 *"),
("A17", "English: Anglo-Indian, 2.Nc3 e6 3.Nf3 d5", "1.c4 Nf6 2.Nc3 e6 3.Nf3 d5 *"),
("A17", "English: Anglo-Queen's Indian", "1.c4 Nf6 2.Nc3 e6 3.Nf3 b6 *"),
("A17", "English: Anglo-Queen's Indian, 4.e4", "1.c4 Nf6 2.Nc3 e6 3.Nf3 b6 4.e4 *"),
("A17", "English: Anglo-Queen's Indian, Romanishin", "1.c4 Nf6 2.Nc3 e6 3.Nf3 b6 4.e4 Bb7 5.Bd3 *"),
("A17", "English: Anglo-Queen's Indian, 4.g3", "1.c4 Nf6 2.Nc3 e6 3.Nf3 b6 4.g3 *"),
("A17", "English: Anglo-Queen's Indian, 4.g3", "1.c4 Nf6 2.Nc3 e6 3.Nf3 b6 4.g3 Bb7 5.Bg2 *"),
("A17", "English: Anglo-Queen's Indian, 4.g3", "1.c4 Nf6 2.Nc3 e6 3.Nf3 b6 4.g3 Bb7 5.Bg2 Be7 6.O-O O-O *"),
("A17", "English: Anglo-Queen's Indian, 4.g3", "1.c4 Nf6 2.Nc3 e6 3.Nf3 b6 4.g3 Bb7 5.Bg2 Be7 6.O-O O-O 7.Re1 *"),
("A17", "English: Nimzo-English", "1.c4 Nf6 2.Nc3 e6 3.Nf3 Bb4 *"),
("A17", "English: Nimzo-English, 4.g3", "1.c4 Nf6 2.Nc3 e6 3.Nf3 Bb4 4.g3 *"),
("A17", "English: Nimzo-English, 4.Qc2", "1.c4 Nf6 2.Nc3 e6 3.Nf3 Bb4 4.Qc2 *"),
("A17", "English: Nimzo-English, 4.Qc2 O-O", "1.c4 Nf6 2.Nc3 e6 3.Nf3 Bb4 4.Qc2 O-O *"),
("A18", "English: Mikenas", "1.c4 Nf6 2.Nc3 e6 3.e4 *"),
("A18", "English: Mikenas, Kevitz Defence", "1.c4 Nf6 2.Nc3 e6 3.e4 Nc6 *"),
("A18", "English: Mikenas, 3...d6", "1.c4 Nf6 2.Nc3 e6 3.e4 d6 *"),
("A18", "English: Mikenas, 3...d6 4.d4 Be7", "1.c4 Nf6 2.Nc3 e6 3.e4 d6 4.d4 Be7 *"),
("A18", "English: Mikenas, French Variation", "1.c4 Nf6 2.Nc3 e6 3.e4 d5 *"),
("A18", "English: Mikenas, French, 4.cxd5", "1.c4 Nf6 2.Nc3 e6 3.e4 d5 4.cxd5 *"),
("A18", "English: Mikenas, Flohr Variation", "1.c4 Nf6 2.Nc3 e6 3.e4 d5 4.e5 *"),
("A18", "English: Mikenas, Flohr, 4...Ne4", "1.c4 Nf6 2.Nc3 e6 3.e4 d5 4.e5 Ne4 *"),
("A18", "English: Mikenas, Flohr, 4...d4", "1.c4 Nf6 2.Nc3 e6 3.e4 d5 4.e5 d4 *"),
("A18", "English: Mikenas, Flohr, 7.Nf3", "1.c4 Nf6 2.Nc3 e6 3.e4 d5 4.e5 d4 5.exf6 dxc3 6.bxc3 Qxf6 7.Nf3 *"),
("A18", "English: Mikenas, Flohr, 7.d4", "1.c4 Nf6 2.Nc3 e6 3.e4 d5 4.e5 d4 5.exf6 dxc3 6.bxc3 Qxf6 7.d4 *"),
("A18", "English: Mikenas, Flohr, 7.d4 c5", "1.c4 Nf6 2.Nc3 e6 3.e4 d5 4.e5 d4 5.exf6 dxc3 6.bxc3 Qxf6 7.d4 c5 *"),
("A18", "English: Mikenas, Flohr, 7.d4 e5", "1.c4 Nf6 2.Nc3 e6 3.e4 d5 4.e5 d4 5.exf6 dxc3 6.bxc3 Qxf6 7.d4 e5 *"),
("A19", "English: Mikenas, Sicilian Variation", "1.c4 Nf6 2.Nc3 e6 3.e4 c5 *"),
("A19", "English: Mikenas, Sicilian, 4.e5", "1.c4 Nf6 2.Nc3 e6 3.e4 c5 4.e5 *"),
("A19", "English: Mikenas, Sicilian, 4.e5 Ng8", "1.c4 Nf6 2.Nc3 e6 3.e4 c5 4.e5 Ng8 *"),
("A19", "English: Mikenas, Nei Gambit", "1.c4 Nf6 2.Nc3 e6 3.e4 c5 4.e5 Ng8 5.d4 *"),
("A19", "English: Mikenas, 5.Nf3", "1.c4 Nf6 2.Nc3 e6 3.e4 c5 4.e5 Ng8 5.Nf3 *"),
("A19", "English: Mikenas, 5.Nf3 Nc6", "1.c4 Nf6 2.Nc3 e6 3.e4 c5 4.e5 Ng8 5.Nf3 Nc6 *"),
("A19", "English: Mikenas, 5.Nf3 Nc6 6.d4", "1.c4 Nf6 2.Nc3 e6 3.e4 c5 4.e5 Ng8 5.Nf3 Nc6 6.d4 *"),
("A19", "English: Mikenas, 6.d4 cxd4 7.Nxd4 Nxe5", "1.c4 Nf6 2.Nc3 e6 3.e4 c5 4.e5 Ng8 5.Nf3 Nc6 6.d4 cxd4 7.Nxd4 Nxe5 *"),
("A20", "English: King's (1...e5)", "1.c4 e5 *"),
("A20", "English: King's, 2.d3", "1.c4 e5 2.d3 *"),
("A20", "English: King's, 2.e3", "1.c4 e5 2.e3 *"),
("A20", "English: King's, 2.g3", "1.c4 e5 2.g3 *"),
("A20", "English: King's, 2.g3 f5", "1.c4 e5 2.g3 f5 *"),
("A20", "English: King's, 2.g3 g6", "1.c4 e5 2.g3 g6 *"),
("A20", "English: King's, 2.g3 g6", "1.c4 e5 2.g3 g6 3.Bg2 *"),
("A20", "English: King's, 2.g3 g6", "1.c4 e5 2.g3 g6 3.Bg2 Bg7 *"),
("A20", "English: King's, 2.g3 c6", "1.c4 e5 2.g3 c6 *"),
("A20", "English: King's, 2.g3 c6 3.d4", "1.c4 e5 2.g3 c6 3.d4 *"),
("A20", "English: King's, 2.g3 d6", "1.c4 e5 2.g3 d6 *"),
("A20", "English: King's, 2.g3 d6", "1.c4 e5 2.g3 d6 3.Bg2 *"),
("A20", "English: King's, 2.g3 Nc6", "1.c4 e5 2.g3 Nc6 *"),
("A20", "English: King's, 2.g3 Nc6 3.Bg2", "1.c4 e5 2.g3 Nc6 3.Bg2 *"),
("A20", "English: King's, 2.g3 Nf6", "1.c4 e5 2.g3 Nf6 *"),
("A20", "English: King's, 2.g3 Nf6 3.Bg2", "1.c4 e5 2.g3 Nf6 3.Bg2 *"),
("A20", "English: King's, 2.g3 Nf6 3.Bg2 Bc5", "1.c4 e5 2.g3 Nf6 3.Bg2 Bc5 *"),
("A20", "English: King's, 2.g3 Nf6 3.Bg2 Nc6", "1.c4 e5 2.g3 Nf6 3.Bg2 Nc6 *"),
("A20", "English: King's, 2.g3 Nf6 3.Bg2 c6", "1.c4 e5 2.g3 Nf6 3.Bg2 c6 *"),
("A20", "English: King's, 2.g3 Nf6 3.Bg2 d5", "1.c4 e5 2.g3 Nf6 3.Bg2 d5 *"),
("A20", "English: King's, Nimzowitsch", "1.c4 e5 2.Nf3 *"),
("A20", "English: King's, Nimzowitsch, 2...Nc6", "1.c4 e5 2.Nf3 Nc6 *"),
("A20", "English: King's, Nimzowitsch, Flohr Variation", "1.c4 e5 2.Nf3 e4 *"),
("A21", "English: King's, 2.Nc3", "1.c4 e5 2.Nc3 *"),
("A21", "English: King's, 2.Nc3 f5", "1.c4 e5 2.Nc3 f5 *"),
("A21", "English: King's, 2.Nc3 f5", "1.c4 e5 2.Nc3 f5 3.g3 Nf6 *"),
("A21", "English: King's, 2.Nc3 f5", "1.c4 e5 2.Nc3 f5 3.g3 Nf6 4.Bg2 *"),
("A21", "English: King's, 2.Nc3 g6", "1.c4 e5 2.Nc3 g6 *"),
("A21", "English: King's, 2.Nc3 g6", "1.c4 e5 2.Nc3 g6 3.g3 Bg7 4.Bg2 *"),
("A21", "English: King's, 2.Nc3 d6", "1.c4 e5 2.Nc3 d6 *"),
("A21", "English: King's, Keres Variation", "1.c4 e5 2.Nc3 d6 3.g3 c6 *"),
("A21", "English: King's, Keres Variation", "1.c4 e5 2.Nc3 d6 3.g3 c6 4.Bg2 *"),
("A21", "English: King's, 2.Nc3 d6 3.d4", "1.c4 e5 2.Nc3 d6 3.d4 *"),
("A21", "English: King's, 2.Nc3 d6 3.g3", "1.c4 e5 2.Nc3 d6 3.g3 *"),
("A21", "English: King's, 2.Nc3 d6 3.Nf3", "1.c4 e5 2.Nc3 d6 3.Nf3 *"),
("A21", "English: King's, 2.Nc3 d6 3.Nf3 g6", "1.c4 e5 2.Nc3 d6 3.Nf3 g6 *"),
("A21", "English: Lukin Variation", "1.c4 e5 2.Nc3 d6 3.Nf3 f5 *"),
("A21", "English: Lukin, 4.d4 e4", "1.c4 e5 2.Nc3 d6 3.Nf3 f5 4.d4 e4 *"),
("A21", "English: Lukin, 5.Nd2", "1.c4 e5 2.Nc3 d6 3.Nf3 f5 4.d4 e4 5.Nd2 *"),
("A21", "English: Lukin, 5.Nd2 Nf6 6.e3", "1.c4 e5 2.Nc3 d6 3.Nf3 f5 4.d4 e4 5.Nd2 Nf6 6.e3 *"),
("A21", "English: Lukin, 5.Ng5", "1.c4 e5 2.Nc3 d6 3.Nf3 f5 4.d4 e4 5.Ng5 *"),
("A21", "English: Lukin, 5.Ng5 Nf6", "1.c4 e5 2.Nc3 d6 3.Nf3 f5 4.d4 e4 5.Ng5 Nf6 *"),
("A21", "English: Lukin, 5.Ng5 Be7", "1.c4 e5 2.Nc3 d6 3.Nf3 f5 4.d4 e4 5.Ng5 Be7 *"),
("A21", "English: Lukin, 5.Ng5 c6", "1.c4 e5 2.Nc3 d6 3.Nf3 f5 4.d4 e4 5.Ng5 c6 *"),
("A21", "English: Smyslov Defence", "1.c4 e5 2.Nc3 d6 3.Nf3 Bg4 *"),
("A21", "English: Kramnik-Shirov Counterattack", "1.c4 e5 2.Nc3 Bb4 *"),
("A21", "English: Kramnik-Shirov, 3.g3", "1.c4 e5 2.Nc3 Bb4 3.g3 *"),
("A21", "English: Kramnik-Shirov, 3.g3 Bxc3", "1.c4 e5 2.Nc3 Bb4 3.g3 Bxc3 *"),
("A21", "English: Kramnik-Shirov, 3.g3 Bxc3 4.bxc3", "1.c4 e5 2.Nc3 Bb4 3.g3 Bxc3 4.bxc3 *"),
("A21", "English: Kramnik-Shirov, 3.Nd5", "1.c4 e5 2.Nc3 Bb4 3.Nd5 *"),
("A21", "English: Kramnik-Shirov, 3.Nd5 a5", "1.c4 e5 2.Nc3 Bb4 3.Nd5 a5 *"),
("A21", "English: Kramnik-Shirov, 3.Nd5 Ba5", "1.c4 e5 2.Nc3 Bb4 3.Nd5 Ba5 *"),
("A21", "English: Kramnik-Shirov, 3.Nd5 Bc5", "1.c4 e5 2.Nc3 Bb4 3.Nd5 Bc5 *"),
("A21", "English: Kramnik-Shirov, 3.Nd5 Be7", "1.c4 e5 2.Nc3 Bb4 3.Nd5 Be7 *"),
("A21", "English: Kramnik-Shirov, 3.Nd5 Be7 4.d4", "1.c4 e5 2.Nc3 Bb4 3.Nd5 Be7 4.d4 *"),
("A22", "English: King's, 2.Nc3 Nf6", "1.c4 e5 2.Nc3 Nf6 *"),
("A22", "English: King's, 2.Nc3 Nf6 3.e4", "1.c4 e5 2.Nc3 Nf6 3.e4 *"),
("A22", "English: King's, 2.Nc3 Nf6 3.e3", "1.c4 e5 2.Nc3 Nf6 3.e3 *"),
("A22", "English: King's, 2.Nc3 Nf6 3.e3 Bb4", "1.c4 e5 2.Nc3 Nf6 3.e3 Bb4 *"),
("A22", "English: King's, 2.Nc3 Nf6 3.Nf3", "1.c4 e5 2.Nc3 Nf6 3.Nf3 *"),
("A22", "English: King's, 2.Nc3 Nf6 3.Nf3 d6", "1.c4 e5 2.Nc3 Nf6 3.Nf3 d6 *"),
("A22", "English: King's, 2.Nc3 Nf6 3.Nf3 e4", "1.c4 e5 2.Nc3 Nf6 3.Nf3 e4 *"),
("A22", "English: Bellon Gambit", "1.c4 e5 2.Nc3 Nf6 3.Nf3 e4 4.Ng5 b5 *"),
("A22", "English: Bremen System", "1.c4 e5 2.Nc3 Nf6 3.g3 *"),
("A22", "English: Bremen, 3...Bc5", "1.c4 e5 2.Nc3 Nf6 3.g3 Bc5 *"),
("A22", "English: Bremen, Reverse Dragon", "1.c4 e5 2.Nc3 Nf6 3.g3 d5 *"),
("A22", "English: Bremen, Reverse Dragon, 4.cxd5", "1.c4 e5 2.Nc3 Nf6 3.g3 d5 4.cxd5 *"),
("A22", "English: Bremen, Reverse Dragon, 4.cxd5 Nxd5", "1.c4 e5 2.Nc3 Nf6 3.g3 d5 4.cxd5 Nxd5 *"),
("A22", "English: Bremen, Reverse Dragon, 4.cxd5 Nxd5 5.Bg2", "1.c4 e5 2.Nc3 Nf6 3.g3 d5 4.cxd5 Nxd5 5.Bg2 *"),
("A22", "English: Bremen, Reverse Dragon, 4.cxd5 Nxd5 5.Bg2 Nb6", "1.c4 e5 2.Nc3 Nf6 3.g3 d5 4.cxd5 Nxd5 5.Bg2 Nb6 *"),
("A22", "English: Bremen, Smyslov System", "1.c4 e5 2.Nc3 Nf6 3.g3 Bb4 *"),
("A22", "English: Bremen, Smyslov, 4.Bg2", "1.c4 e5 2.Nc3 Nf6 3.g3 Bb4 4.Bg2 *"),
("A22", "English: Bremen, Smyslov, 4.Bg2 Bxc3", "1.c4 e5 2.Nc3 Nf6 3.g3 Bb4 4.Bg2 Bxc3 *"),
("A22", "English: Bremen, Smyslov, 4.Bg2 O-O", "1.c4 e5 2.Nc3 Nf6 3.g3 Bb4 4.Bg2 O-O *"),
("A22", "English: Bremen, Smyslov, 4.Bg2 O-O 5.e4", "1.c4 e5 2.Nc3 Nf6 3.g3 Bb4 4.Bg2 O-O 5.e4 *"),
("A23", "English: Bremen, Keres System", "1.c4 e5 2.Nc3 Nf6 3.g3 c6 *"),
("A23", "English: Bremen, Keres, 4.Nf3", "1.c4 e5 2.Nc3 Nf6 3.g3 c6 4.Nf3 *"),
("A23", "English: Bremen, Keres, 4.Nf3 d6", "1.c4 e5 2.Nc3 Nf6 3.g3 c6 4.Nf3 d6 *"),
("A23", "English: Bremen, Keres, 4.Nf3 d6", "1.c4 e5 2.Nc3 Nf6 3.g3 c6 4.Nf3 d6 5.Bg2 *"),
("A23", "English: Bremen, Keres, 4.Nf3 e4", "1.c4 e5 2.Nc3 Nf6 3.g3 c6 4.Nf3 e4 *"),
("A23", "English: Bremen, Keres, 4.Bg2", "1.c4 e5 2.Nc3 Nf6 3.g3 c6 4.Bg2 *"),
("A23", "English: Bremen, Keres, 4.Bg2 d5", "1.c4 e5 2.Nc3 Nf6 3.g3 c6 4.Bg2 d5 *"),
("A23", "English: Bremen, Keres, 4.Bg2 d5 5.cxd5", "1.c4 e5 2.Nc3 Nf6 3.g3 c6 4.Bg2 d5 5.cxd5 *"),
("A23", "English: Bremen, Keres, 4.Bg2 d5 5.cxd5 cxd5", "1.c4 e5 2.Nc3 Nf6 3.g3 c6 4.Bg2 d5 5.cxd5 cxd5 *"),
("A24", "English: Bremen, 3...g6", "1.c4 e5 2.Nc3 Nf6 3.g3 g6 *"),
("A24", "English: Bremen, 3...g6", "1.c4 e5 2.Nc3 Nf6 3.g3 g6 4.Bg2 *"),
("A24", "English: Bremen, 3...g6", "1.c4 e5 2.Nc3 Nf6 3.g3 g6 4.Bg2 Bg7 *"),
("A24", "English: Bremen, 3...g6", "1.c4 e5 2.Nc3 Nf6 3.g3 g6 4.Bg2 Bg7 5.d3 *"),
("A24", "English: Bremen, 3...g6", "1.c4 e5 2.Nc3 Nf6 3.g3 g6 4.Bg2 Bg7 5.e3 *"),
("A24", "English: Bremen, 3...g6", "1.c4 e5 2.Nc3 Nf6 3.g3 g6 4.Bg2 Bg7 5.e3 d6 6.Nge2 O-O *"),
("A24", "English: Bremen, 3...g6", "1.c4 e5 2.Nc3 Nf6 3.g3 g6 4.Bg2 Bg7 5.e4 *"),
("A24", "English: Bremen, 3...g6", "1.c4 e5 2.Nc3 Nf6 3.g3 g6 4.Bg2 Bg7 5.e4 d6 *"),
("A24", "English: Bremen, 3...g6", "1.c4 e5 2.Nc3 Nf6 3.g3 g6 4.Bg2 Bg7 5.e4 d6 6.Nf3 *"),
("A24", "English: Bremen, 3...g6", "1.c4 e5 2.Nc3 Nf6 3.g3 g6 4.Bg2 Bg7 5.e4 d6 6.Nge2 O-O *"),
("A24", "English: Bremen, 3...g6", "1.c4 e5 2.Nc3 Nf6 3.g3 g6 4.Bg2 Bg7 5.e4 d6 6.Nge2 O-O 7.d3 *"),
("A24", "English: Bremen, 3...g6", "1.c4 e5 2.Nc3 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 *"),
("A24", "English: Bremen, 3...g6", "1.c4 e5 2.Nc3 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 d6 6.O-O O-O *"),
("A24", "English: Bremen, 3...g6", "1.c4 e5 2.Nc3 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 d6 6.d3 O-O 7.O-O *"),
("A24", "English: Bremen, 3...g6", "1.c4 e5 2.Nc3 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 d6 6.d3 O-O 7.O-O c6 *"),
("A25", "English: Closed", "1.c4 e5 2.Nc3 Nc6 *"),
("A25", "English: Closed", "1.c4 e5 2.Nc3 Nc6 3.e3 *"),
("A25", "English: Closed", "1.c4 e5 2.Nc3 Nc6 3.e3 Nf6 *"),
("A25", "English: Closed", "1.c4 e5 2.Nc3 Nc6 3.g3 *"),
("A25", "English: Closed", "1.c4 e5 2.Nc3 Nc6 3.g3 f5 *"),
("A25", "English: Closed", "1.c4 e5 2.Nc3 Nc6 3.g3 f5 4.Bg2 *"),
("A25", "English: Closed", "1.c4 e5 2.Nc3 Nc6 3.g3 f5 4.Bg2 Nf6 *"),
("A25", "English: Closed", "1.c4 e5 2.Nc3 Nc6 3.g3 f5 4.Bg2 Nf6 5.e3 g6 *"),
("A25", "English: Closed", "1.c4 e5 2.Nc3 Nc6 3.g3 f5 4.Bg2 Nf6 5.d3 *"),
("A25", "English: Closed", "1.c4 e5 2.Nc3 Nc6 3.g3 Nf6 *"),
("A25", "English: Closed", "1.c4 e5 2.Nc3 Nc6 3.g3 Nf6 4.Bg2 *"),
("A25", "English: Closed", "1.c4 e5 2.Nc3 Nc6 3.g3 Nf6 4.Bg2 Bc5 *"),
("A25", "English: Closed", "1.c4 e5 2.Nc3 Nc6 3.g3 Nf6 4.Bg2 Bc5 5.e3 *"),
("A25", "English: Closed", "1.c4 e5 2.Nc3 Nc6 3.g3 Nf6 4.Bg2 Bb4 *"),
("A25", "English: Closed", "1.c4 e5 2.Nc3 Nc6 3.g3 Nf6 4.Bg2 Bb4 5.Nd5 *"),
("A25", "English: Closed", "1.c4 e5 2.Nc3 Nc6 3.g3 d6 4.Bg2 *"),
("A25", "English: Closed, Troeger Defence", "1.c4 e5 2.Nc3 Nc6 3.g3 d6 4.Bg2 Be6 *"),
("A25", "English: Closed, Troeger, 5.d3", "1.c4 e5 2.Nc3 Nc6 3.g3 d6 4.Bg2 Be6 5.d3 *"),
("A25", "English: Closed", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 *"),
("A25", "English: Closed", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 *"),
("A25", "English: Closed", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 *"),
("A25", "English: Closed, 5.e3", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e3 *"),
("A25", "English: Closed, Taimanov Variation", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e3 d6 6.Nge2 Nh6 *"),
("A25", "English: Closed, Hort Variation", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e3 d6 6.Nge2 Be6 *"),
("A25", "English: Closed, Hort, 7.d3", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e3 d6 6.Nge2 Be6 7.d3 *"),
("A25", "English: Closed, 5.Rb1", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Rb1 *"),
("A25", "English: Closed, 5.Rb1, Taimanov Variation", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Rb1 Nh6 *"),
("A25", "English: Closed, 5.Rb1", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Rb1 a5 *"),
("A25", "English: Closed, 5.e4", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e4 *"),
("A25", "English: Closed, 5.d3", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 *"),
("A25", "English: Closed, 5.d3 Nge7", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 Nge7 *"),
("A26", "English: Closed, 5.d3 d6", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 *"),
("A26", "English: Closed, 5.d3 d6 6.e3", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.e3 *"),
("A26", "English: Closed, 5.d3 d6 6.Rb1", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Rb1 *"),
("A26", "English: Closed, 5.d3 d6 6.Rb1 f5", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Rb1 f5 *"),
("A26", "English: Closed, 5.d3 d6 6.Rb1 a5", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Rb1 a5 *"),
("A26", "English: Closed, 5.d3 d6 6.Nf3", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Nf3 *"),
("A26", "English: Closed, 5.d3 d6 6.Nf3 Nf6", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Nf3 Nf6 *"),
("A26", "English: Closed, 5.d3 d6 6.Nf3 Nf6 7.O-O", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Nf3 Nf6 7.O-O *"),
("A26", "English: Closed, 5.d3 d6 6.Nf3 Nf6 7.O-O O-O", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Nf3 Nf6 7.O-O O-O *"),
("A26", "English: Closed, 5.d3 d6 6.Nf3 Nf6 7.O-O O-O 8.Rb1", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Nf3 Nf6 7.O-O O-O 8.Rb1 *"),
("A26", "English: Closed, 5.d3 d6 6.Nf3 Nf6 7.O-O O-O 8.Rb1 a5", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Nf3 Nf6 7.O-O O-O 8.Rb1 a5 *"),
("A26", "English: Closed, 5.d3 d6 6.Nf3 Nf6 7.O-O O-O 8.Rb1 a5 9.a3", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Nf3 Nf6 7.O-O O-O 8.Rb1 a5 9.a3 *"),
("A26", "English: Closed, 5.d3 d6 6.Nf3 Nf6 7.O-O O-O 8.Rb1 a5 9.a3 h6", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Nf3 Nf6 7.O-O O-O 8.Rb1 a5 9.a3 h6 *"),
("A26", "English: Botvinnik System", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.e4 *"),
("A26", "English: Botvinnik System, 6...Nf6", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.e4 Nf6 *"),
("A26", "English: Botvinnik System, 6...Nf6", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.e4 Nf6 7.Nf3 *"),
("A26", "English: Botvinnik System, 6...Nf6", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.e4 Nf6 7.Nf3 O-O 8.O-O *"),
("A26", "English: Botvinnik System, 6...Nf6", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.e4 Nf6 7.Nge2 *"),
("A26", "English: Botvinnik System, 6...Nf6", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.e4 Nf6 7.Nge2 O-O 8.O-O *"),
("A26", "English: Botvinnik System, 6...Nge7", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.e4 Nge7 *"),
("A26", "English: Botvinnik System, 6...Nge7 7.Nge2", "1.c4 e5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.e4 Nge7 7.Nge2 *"),
("A27", "English: Three Knights", "1.c4 e5 2.Nc3 Nc6 3.Nf3 *"),
("A27", "English: Three Knights, 3...Bb4", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Bb4 *"),
("A27", "English: Three Knights, 3...d6", "1.c4 e5 2.Nc3 Nc6 3.Nf3 d6 *"),
("A27", "English: Three Knights, 3...f5", "1.c4 e5 2.Nc3 Nc6 3.Nf3 f5 *"),
("A27", "English: Three Knights, 3...f5 4.d4", "1.c4 e5 2.Nc3 Nc6 3.Nf3 f5 4.d4 *"),
("A27", "English: Three Knights, 3...f5 4.d4 e4", "1.c4 e5 2.Nc3 Nc6 3.Nf3 f5 4.d4 e4 *"),
("A27", "English: Three Knights, 3...g6", "1.c4 e5 2.Nc3 Nc6 3.Nf3 g6 *"),
("A27", "English: Three Knights, 3...g6 4.d4", "1.c4 e5 2.Nc3 Nc6 3.Nf3 g6 4.d4 *"),
("A27", "English: Three Knights, 3...g6 4.d4 exd4", "1.c4 e5 2.Nc3 Nc6 3.Nf3 g6 4.d4 exd4 *"),
("A27", "English: Three Knights, 3...g6 4.d4 exd4 5.Nxd4", "1.c4 e5 2.Nc3 Nc6 3.Nf3 g6 4.d4 exd4 5.Nxd4 *"),
("A28", "English: Four Knights", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 *"),
("A28", "English: Four Knights, Nimzowitsch Variation", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.e4 *"),
("A28", "English: Four Knights, Marini Variation", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.a3 *"),
("A28", "English: Four Knights, Capablanca Variation", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.d3 *"),
("A28", "English: Four Knights, 4.d4", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.d4 *"),
("A28", "English: Four Knights, Nenarokov Variation", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.d4 exd4 5.Nxd4 Bb4 6.Bg5 h6 7.Bh4 Bxc3+ 8.bxc3 Ne5 *"),
("A28", "English: Four Knights, Bradley Beach Variation", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.d4 e4 *"),
("A28", "English: Four Knights, 4.e3", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.e3 *"),
("A28", "English: Four Knights, 4.e3 Bb4", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.e3 Bb4 *"),
("A28", "English: Four Knights, 4.e3 Bb4 5.Qc2", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.e3 Bb4 5.Qc2 *"),
("A28", "English: Four Knights, Stean Variation", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.e3 Bb4 5.Qc2 O-O 6.Nd5 Re8 7.Qf5 *"),
("A28", "English: Four Knights, Romanishin Variation", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.e3 Bb4 5.Qc2 Bxc3 *"),
("A29", "English: Four Knights, 4.g3", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 *"),
("A29", "English: Four Knights, 4.g3 g6", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 g6 *"),
("A29", "English: Four Knights, 4.g3 g6 5.d4", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 g6 5.d4 *"),
("A29", "English: Four Knights, 4.g3 d5", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 d5 *"),
("A29", "English: Four Knights, 4.g3 d5 5.cxd5", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 d5 5.cxd5 *"),
("A29", "English: Four Knights, 4.g3 d5 5.cxd5 Nxd5", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 d5 5.cxd5 Nxd5 *"),
("A29", "English: Four Knights, 4.g3 d5 5.cxd5 Nxd5 6.Bg2", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 d5 5.cxd5 Nxd5 6.Bg2 *"),
("A29", "English: Four Knights, 4.g3 d5 5.cxd5 Nxd5 6.Bg2 Be6", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 d5 5.cxd5 Nxd5 6.Bg2 Be6 *"),
("A29", "English: Four Knights, 4.g3 d5 5.cxd5 Nxd5 6.Bg2 Nb6", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 d5 5.cxd5 Nxd5 6.Bg2 Nb6 *"),
("A29", "English: Four Knights, 4.g3 d5 5.cxd5 Nxd5 6.Bg2 Nb6 7.O-O", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 d5 5.cxd5 Nxd5 6.Bg2 Nb6 7.O-O *"),
("A29", "English: Four Knights, 4.g3 d5 5.cxd5 Nxd5 6.Bg2 Nb6 7.O-O Be7", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 d5 5.cxd5 Nxd5 6.Bg2 Nb6 7.O-O Be7 *"),
("A29", "English: Four Knights, 4.g3 d5 5.cxd5 Nxd5 6.Bg2 Nb6 7.O-O Be7 8.Rb1", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 d5 5.cxd5 Nxd5 6.Bg2 Nb6 7.O-O Be7 8.Rb1 *"),
("A29", "English: Four Knights, 4.g3 d5 5.cxd5 Nxd5 6.Bg2 Nb6 7.O-O Be7 8.a3", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 d5 5.cxd5 Nxd5 6.Bg2 Nb6 7.O-O Be7 8.a3 *"),
("A29", "English: Four Knights, 4.g3 d5 5.cxd5 Nxd5 6.Bg2 Nb6 7.O-O Be7 8.d3", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 d5 5.cxd5 Nxd5 6.Bg2 Nb6 7.O-O Be7 8.d3 *"),
("A29", "English: Four Knights, 4.g3 d5 5.cxd5 Nxd5 6.Bg2 Nb6 7.O-O Be7 8.d3", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 d5 5.cxd5 Nxd5 6.Bg2 Nb6 7.O-O Be7 8.d3 O-O *"),
("A29", "English: Four Knights, 4.g3 Bc5", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 Bc5 *"),
("A29", "English: Four Knights, 4.g3 Bc5", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 Bc5 5.Bg2 *"),
("A29", "English: Four Knights, 4.g3 Bc5", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 Bc5 5.Bg2 d6 6.O-O *"),
("A29", "English: Four Knights, 4.g3 Bc5", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 Bc5 5.Bg2 d6 6.O-O O-O *"),
("A29", "English: Four Knights, 4.g3 Bc5", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 Bc5 5.Bg2 d6 6.O-O O-O 7.d3 *"),
("A29", "English: Four Knights, 4.g3 Bb4", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 Bb4 *"),
("A29", "English: Four Knights, 4.g3 Bb4 5.Nd5", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 Bb4 5.Nd5 *"),
("A29", "English: Four Knights, 4.g3 Bb4 5.Bg2", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 Bb4 5.Bg2 *"),
("A29", "English: Four Knights, Main Line 6.O-O", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 Bb4 5.Bg2 O-O 6.O-O *"),
("A29", "English: Four Knights, Main Line 6.O-O e4", "1.c4 e5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 Bb4 5.Bg2 O-O 6.O-O e4 *"),
("A30a", "English: Symmetrical", "1.c4 c5 *"),
("A30a", "English: Symmetrical, 2.b3", "1.c4 c5 2.b3 *"),
("A30a", "English: Symmetrical, 2.g3", "1.c4 c5 2.g3 *"),
("A30a", "English: Symmetrical, 2.g3 g6", "1.c4 c5 2.g3 g6 *"),
("A30b", "English: Symmetrical, 2.Nf3", "1.c4 c5 2.Nf3 *"),
("A30b", "English: Symmetrical, 2.Nf3 b6", "1.c4 c5 2.Nf3 b6 *"),
("A30b", "English: Symmetrical, 2.Nf3 b6 3.g3", "1.c4 c5 2.Nf3 b6 3.g3 *"),
("A30b", "English: Symmetrical, 2.Nf3 b6 3.g3 Bb7 4.Bg2", "1.c4 c5 2.Nf3 b6 3.g3 Bb7 4.Bg2 *"),
("A30b", "English: Symmetrical, 2.Nf3 g6", "1.c4 c5 2.Nf3 g6 *"),
("A30b", "English: Symmetrical, 2.Nf3 Nc6", "1.c4 c5 2.Nf3 Nc6 *"),
("A30c", "English: Symmetrical, 2.Nf3 Nf6", "1.c4 c5 2.Nf3 Nf6 *"),
("A30d", "English: Symmetrical, 2.Nf3 Nf6 3.g3", "1.c4 c5 2.Nf3 Nf6 3.g3 *"),
("A30e", "English: Symmetrical, b6 System", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 *"),
("A30e", "English: Symmetrical, b6 System", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 *"),
("A30e", "English: Symmetrical, b6 System", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.Nc3 *"),
("A30e", "English: Symmetrical, b6 System", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O *"),
("A30f", "English: Symmetrical, Double Fianchetto", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O g6 *"),
("A30f", "English: Symmetrical, Double Fianchetto", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O g6 6.Nc3 Bg7 *"),
("A30g", "English: Symmetrical, Mutual Double Fianchetto", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O g6 6.b3 *"),
("A30g", "English: Symmetrical, Mutual Double Fianchetto", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O g6 6.b3 Bg7 7.Bb2 O-O *"),
("A30g", "English: Symmetrical, Mutual Double Fianchetto", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O g6 6.b3 Bg7 7.Bb2 O-O 8.Nc3 *"),
("A30g", "English: Symmetrical, Mutual Double Fianchetto with ...d5", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O g6 6.b3 Bg7 7.Bb2 O-O 8.Nc3 d5 *"),
("A30h", "English: Symmetrical, Hedgehog System", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 *"),
("A30h", "English: Symmetrical, Hedgehog System", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 *"),
("A30i", "English: Symmetrical, Hedgehog, 6...a6", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 a6 *"),
("A30j", "English: Symmetrical, Hedgehog, 6...d6", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 d6 *"),
("A30k", "English: Symmetrical, Hedgehog, 6...d6 7.b3", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 d6 7.b3 *"),
("A30l", "English: Symmetrical, Hedgehog, 6...d6 7.d4", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 d6 7.d4 *"),
("A30m", "English: Symmetrical, Hedgehog System", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 Be7 *"),
("A30n", "English: Symmetrical, Hedgehog, 7.Re1", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 Be7 7.Re1 *"),
("A30o", "English: Symmetrical, Hedgehog, 7.Re1 d5", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 Be7 7.Re1 d5 *"),
("A30p", "English: Symmetrical, Hedgehog, 7.b3", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 Be7 7.b3 *"),
("A30p", "English: Symmetrical, Hedgehog, 7.b3", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 Be7 7.b3 O-O *"),
("A30p", "English: Symmetrical, Hedgehog, 7.b3", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 Be7 7.b3 O-O 8.Bb2 d6 *"),
("A30q", "English: Symmetrical, Hedgehog, 7.d4", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 Be7 7.d4 *"),
("A30r", "English: Symmetrical, Hedgehog, 8.Qxd4", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 Be7 7.d4 cxd4 8.Qxd4 *"),
("A30s", "English: Symmetrical, Hedgehog, 8.Qxd4 O-O", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 Be7 7.d4 cxd4 8.Qxd4 O-O *"),
("A30s", "English: Symmetrical, Hedgehog, 8.Qxd4 O-O", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 Be7 7.d4 cxd4 8.Qxd4 O-O 9.Rd1 Nc6 *"),
("A30t", "English: Symmetrical, Hedgehog, 8.Qxd4 Nc6", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 Be7 7.d4 cxd4 8.Qxd4 Nc6 *"),
("A30u", "English: Symmetrical, Hedgehog, 8.Qxd4 d6", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 Be7 7.d4 cxd4 8.Qxd4 d6 *"),
("A30v", "English: Symmetrical, Hedgehog, 9.b3", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 Be7 7.d4 cxd4 8.Qxd4 d6 9.b3 *"),
("A30v", "English: Symmetrical, Hedgehog, 9.b3 Nbd7", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 Be7 7.d4 cxd4 8.Qxd4 d6 9.b3 Nbd7 *"),
("A30v", "English: Symmetrical, Hedgehog, 9.b3 Nbd7 10.Nb5", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 Be7 7.d4 cxd4 8.Qxd4 d6 9.b3 Nbd7 10.Nb5 *"),
("A30w", "English: Symmetrical, Hedgehog, 9.Rd1", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 Be7 7.d4 cxd4 8.Qxd4 d6 9.Rd1 *"),
("A30x", "English: Symmetrical, Hedgehog, Flexible Formation", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 Be7 7.d4 cxd4 8.Qxd4 d6 9.Rd1 a6 10.b3 Nbd7 *"),
("A30y", "English: Symmetrical, Hedgehog, Flexible Formation", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 Be7 7.d4 cxd4 8.Qxd4 d6 9.Rd1 a6 10.b3 Nbd7 11.e4 *"),
("A30y", "English: Symmetrical, Hedgehog, Flexible Formation, 11.e4 Qc7", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 Be7 7.d4 cxd4 8.Qxd4 d6 9.Rd1 a6 10.b3 Nbd7 11.e4 Qc7 *"),
("A30y", "English: Symmetrical, Hedgehog, Flexible Formation, 11.e4 O-O", "1.c4 c5 2.Nf3 Nf6 3.g3 b6 4.Bg2 Bb7 5.O-O e6 6.Nc3 Be7 7.d4 cxd4 8.Qxd4 d6 9.Rd1 a6 10.b3 Nbd7 11.e4 O-O *"),
("A31", "English: Symmetrical, Two Knights", "1.c4 c5 2.Nf3 Nf6 3.d4 *"),
("A31", "English: Symmetrical, Two Knights, 3...a6", "1.c4 c5 2.Nf3 Nf6 3.d4 a6 *"),
("A31", "English: Symmetrical, Two Knights, 3...g6", "1.c4 c5 2.Nf3 Nf6 3.d4 g6 *"),
("A31", "English: Symmetrical, Two Knights, 3...g6", "1.c4 c5 2.Nf3 Nf6 3.d4 g6 4.d5 *"),
("A31", "English: Symmetrical, Two Knights, 3...g6", "1.c4 c5 2.Nf3 Nf6 3.d4 g6 4.Nc3 *"),
("A31", "English: Symmetrical, Two Knights", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 *"),
("A31", "English: Symmetrical, Two Knights", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 *"),
("A31", "English: Symmetrical, Two Knights, 4...b6", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 b6 *"),
("A31", "English: Symmetrical, Two Knights, 4...b6", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 b6 5.Nc3 Bb7 *"),
("A31", "English: Symmetrical, Two Knights, 4...g6", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 g6 *"),
("A31", "English: Symmetrical, Two Knights, 4...g6", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 *"),
("A31", "English: Symmetrical, Two Knights, 4...g6", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 *"),
// Should this be A31? It doesn't transpose to a B3 Sicilian until ...Nc6.
("A31", "English: Symmetrical, Two Knights, 4...g6", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.e4 *"),
("A31", "English: Symmetrical, Two Knights, 4...Nc6", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 Nc6 *"),
("A31", "English: Symmetrical, Two Knights, 4...Nc6", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 *"),
("A31", "English: Symmetrical, Two Knights, 4...Nc6", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 g6 *"),
("A31", "English: Symmetrical, Two Knights, 4...e5", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e5 *"),
("A31", "English: Symmetrical, Two Knights, 4...e5 5.Nb5", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e5 5.Nb5 *"),
("A32", "English: Symmetrical, Two Knights, 4...e6", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 *"),
("A32", "English: Symmetrical, Two Knights, 5.e3", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.e3 *"),
("A32", "English: Symmetrical, Two Knights, 5.g3", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.g3 *"),
("A32", "English: Symmetrical, Two Knights, 5.g3 a6", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.g3 a6 *"),
("A32", "English: Symmetrical, Two Knights, 5.g3 a6", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.g3 a6 6.Bg2 *"),
("A32", "English: Symmetrical, Two Knights, 5.g3 Qb6", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.g3 Qb6 *"),
("A32", "English: Symmetrical, Two Knights, 5.g3 Qb6", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.g3 Qb6 6.Bg2 *"),
("A32", "English: Symmetrical, Two Knights, 5.g3 Qc7", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.g3 Qc7 *"),
("A32", "English: Symmetrical, Two Knights, 5.g3 Nc6", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.g3 Nc6 *"),
("A32", "English: Symmetrical, Two Knights, 5.g3 Nc6", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.g3 Nc6 6.Bg2 *"),
("A32", "English: Symmetrical, Two Knights, 5.g3 Bb4+", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.g3 Bb4+ *"),
("A32", "English: Symmetrical, Two Knights, 5.g3 d5", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.g3 d5 *"),
("A32", "English: Symmetrical, Two Knights, 5.g3 d5 6.Bg2", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.g3 d5 6.Bg2 *"),
("A32", "English: Symmetrical, Two Knights, 5.g3 d5 6.Bg2 e5 7.Nf3", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.g3 d5 6.Bg2 e5 7.Nf3 *"),
("A32", "English: Symmetrical, Two Knights, 5.Nc3", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 *"),
("A32", "English: Symmetrical, Two Knights, 5.Nc3 d5", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 d5 *"),
("A32", "English: Symmetrical, Two Knights, 5.Nc3 a6", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 a6 *"),
("A32", "English: Symmetrical, Two Knights, 5.Nc3 a6 6.g3", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 a6 6.g3 *"),
("A32", "English: Symmetrical, Two Knights, 5.Nc3 Bb4", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 Bb4 *"),
("A32", "English: Symmetrical, Two Knights, 5.Nc3 Bb4 6.Qb3", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 Bb4 6.Qb3 *"),
("A32", "English: Symmetrical, Two Knights, 5.Nc3 Bb4 6.Nb5", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 Bb4 6.Nb5 *"),
("A32", "English: Symmetrical, Two Knights, 5.Nc3 Bb4 6.Bd2", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 Bb4 6.Bd2 *"),
("A32", "English: Symmetrical, Two Knights, 5.Nc3 Bb4 6.Bd2 Nc6", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 Bb4 6.Bd2 Nc6 *"),
("A33a", "English: Symmetrical, Two Knights, 5.Nc3 Nc6", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 Nc6 *"),
("A33b", "English: Symmetrical, Two Knights, 5.Nc3 Nc6 6.Bf4", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 Nc6 6.Bf4 *"),
("A33c", "English: Symmetrical, Two Knights, 5.Nc3 Nc6 6.Bg5", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 Nc6 6.Bg5 *"),
("A33d", "English: Symmetrical, Two Knights, 5.Nc3 Nc6 6.e3", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 Nc6 6.e3 *"),
("A33e", "English: Symmetrical, Two Knights, 5.Nc3 Nc6 6.a3", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 Nc6 6.a3 *"),
("A33f", "English: Symmetrical, Two Knights, 5.Nc3 Nc6 6.a3 Bc5", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 Nc6 6.a3 Bc5 *"),
("A33g", "English: Symmetrical, Two Knights, 5.Nc3 Nc6 6.Ndb5", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 Nc6 6.Ndb5 *"),
("A33h", "English: Symmetrical, Two Knights, 5.Nc3 Nc6 6.Ndb5 Bb4", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 Nc6 6.Ndb5 Bb4 *"),
("A33i", "English: Symmetrical, Two Knights, 5.Nc3 Nc6 6.Ndb5 d5", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 Nc6 6.Ndb5 d5 *"),
("A33j", "English: Symmetrical, Two Knights, 5.Nc3 Nc6 6.Ndb5 d5 Queenswap", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 Nc6 6.Ndb5 d5 7.Bf4 e5 8.cxd5 exf4 9.dxc6 bxc6 10.Qxd8+ Kxd8 *"),
("A33k", "English: Symmetrical, Two Knights, 5.Nc3 Nc6 6.g3", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 Nc6 6.g3 *"),
("A33l", "English: Symmetrical, Two Knights, 5.Nc3 Nc6 6.g3 a6", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 Nc6 6.g3 a6 *"),
("A33m", "English: Symmetrical, Two Knights, 5.Nc3 Nc6 6.g3 Bb4", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 Nc6 6.g3 Bb4 *"),
("A33n", "English: Symmetrical, Two Knights, 5.Nc3 Nc6 6.g3 Bc5", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 Nc6 6.g3 Bc5 *"),
("A33o", "English: Symmetrical, Geller Variation", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 Nc6 6.g3 Qb6 *"),
("A33p", "English: Symmetrical, Geller, 7.Nb3", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 Nc6 6.g3 Qb6 7.Nb3 *"),
("A33q", "English: Symmetrical, Geller, 7.Nb3 Ne5", "1.c4 c5 2.Nf3 Nf6 3.d4 cxd4 4.Nxd4 e6 5.Nc3 Nc6 6.g3 Qb6 7.Nb3 Ne5 *"),
("A34", "English: Symmetrical", "1.c4 c5 2.Nc3 *"),
("A34", "English: Symmetrical, 2...b6", "1.c4 c5 2.Nc3 b6 *"),
("A34", "English: Symmetrical, 2...b6", "1.c4 c5 2.Nc3 b6 3.Nf3 *"),
("A34", "English: Symmetrical, 2...b6", "1.c4 c5 2.Nc3 b6 3.Nf3 Bb7 *"),
("A34", "English: Symmetrical, 2...g6", "1.c4 c5 2.Nc3 g6 *"),
("A34", "English: Symmetrical, 2...g6", "1.c4 c5 2.Nc3 g6 3.Nf3 *"),
("A34", "English: Symmetrical, 2...g6", "1.c4 c5 2.Nc3 g6 3.g3 *"),
("A34", "English: Symmetrical, 2...g6", "1.c4 c5 2.Nc3 g6 3.g3 Bg7 4.Bg2 *"),
("A34", "English: Symmetrical, 2...Nf6", "1.c4 c5 2.Nc3 Nf6 *"),
("A34", "English: Symmetrical, 2...Nf6", "1.c4 c5 2.Nc3 Nf6 3.g3 d5 *"),
("A34", "English: Symmetrical, 2...Nf6 3.g3", "1.c4 c5 2.Nc3 Nf6 3.g3 *"),
("A34", "English: Symmetrical, 2...Nf6 3.g3", "1.c4 c5 2.Nc3 Nf6 3.g3 d5 4.cxd5 Nxd5 5.Bg2 *"),
("A34", "English: Symmetrical, Rubinstein System", "1.c4 c5 2.Nc3 Nf6 3.g3 d5 4.cxd5 Nxd5 5.Bg2 Nc7 *"),
("A34", "English: Symmetrical, 3 Knights", "1.c4 c5 2.Nc3 Nf6 3.Nf3 *"),
("A34", "English: Symmetrical, 3 Knights, 3...b6", "1.c4 c5 2.Nc3 Nf6 3.Nf3 b6 *"),
("A34", "English: Symmetrical, 3 Knights, 3...e6", "1.c4 c5 2.Nc3 Nf6 3.Nf3 e6 *"),
("A34", "English: Symmetrical, 3 Knights, 3...e6 4.g3", "1.c4 c5 2.Nc3 Nf6 3.Nf3 e6 4.g3 *"),
("A34", "English: Symmetrical, 3 Knights, 3...g6", "1.c4 c5 2.Nc3 Nf6 3.Nf3 g6 *"),
("A34", "English: Symmetrical, 3 Knights", "1.c4 c5 2.Nc3 Nf6 3.Nf3 d5 *"),
("A34", "English: Symmetrical, 3 Knights, Exchange", "1.c4 c5 2.Nc3 Nf6 3.Nf3 d5 4.cxd5 Nxd5 *"),
("A34", "English: Symmetrical, 3 Knights, 5.g3", "1.c4 c5 2.Nc3 Nf6 3.Nf3 d5 4.cxd5 Nxd5 5.g3 *"),
("A34", "English: Symmetrical, 3 Knights, 5.e4", "1.c4 c5 2.Nc3 Nf6 3.Nf3 d5 4.cxd5 Nxd5 5.e4 *"),
("A34", "English: Symmetrical, 3 Knights, 5.e4 Nxc3", "1.c4 c5 2.Nc3 Nf6 3.Nf3 d5 4.cxd5 Nxd5 5.e4 Nxc3 *"),
("A34", "English: Symmetrical, 3 Knights, Queenswap", "1.c4 c5 2.Nc3 Nf6 3.Nf3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.dxc3 Qxd1+ 7.Kxd1 *"),
("A34", "English: Symmetrical, 3 Knights, 5.e4 Nb4", "1.c4 c5 2.Nc3 Nf6 3.Nf3 d5 4.cxd5 Nxd5 5.e4 Nb4 *"),
("A34", "English: Symmetrical, 3 Knights, 5.e4 Nb4 6.Bb5+", "1.c4 c5 2.Nc3 Nf6 3.Nf3 d5 4.cxd5 Nxd5 5.e4 Nb4 6.Bb5+ *"),
("A34", "English: Symmetrical, 3 Knights, 5.e4 Nb4 6.Bc4", "1.c4 c5 2.Nc3 Nf6 3.Nf3 d5 4.cxd5 Nxd5 5.e4 Nb4 6.Bc4 *"),
("A34", "English: Symmetrical, 3 Knights, 5.d4", "1.c4 c5 2.Nc3 Nf6 3.Nf3 d5 4.cxd5 Nxd5 5.d4 *"),
("A34", "English: Symmetrical, 3 Knights, 5.d4 cxd4", "1.c4 c5 2.Nc3 Nf6 3.Nf3 d5 4.cxd5 Nxd5 5.d4 cxd4 *"),
("A34", "English: Symmetrical, 3 Knights, 5.d4 Nxc3", "1.c4 c5 2.Nc3 Nf6 3.Nf3 d5 4.cxd5 Nxd5 5.d4 Nxc3 *"),
("A35a", "English: Symmetrical", "1.c4 c5 2.Nc3 Nc6 *"),
("A35b", "English: Symmetrical, 2.Nc3 Nc6 3.e3", "1.c4 c5 2.Nc3 Nc6 3.e3 *"),
("A35c", "English: Symmetrical, 2.Nc3 Nc6 3.Nf3", "1.c4 c5 2.Nc3 Nc6 3.Nf3 *"),
("A35d", "English: Symmetrical, 2.Nc3 Nc6 3.Nf3 g6", "1.c4 c5 2.Nc3 Nc6 3.Nf3 g6 *"),
("A35e", "English: Symmetrical, 2.Nc3 Nc6 3.Nf3 g6 4.e3", "1.c4 c5 2.Nc3 Nc6 3.Nf3 g6 4.e3 *"),
("A35f", "English: Symmetrical, 2.Nc3 Nc6 3.Nf3 g6 4.e3 Bg7", "1.c4 c5 2.Nc3 Nc6 3.Nf3 g6 4.e3 Bg7 *"),
("A35g", "English: Symmetrical, Four Knights", "1.c4 c5 2.Nc3 Nc6 3.Nf3 Nf6 *"),
("A35g", "English: Symmetrical, Four Knights, 4.d4", "1.c4 c5 2.Nc3 Nc6 3.Nf3 Nf6 4.d4 *"),
("A35h", "English: Symmetrical, Four Knights, 4.e3", "1.c4 c5 2.Nc3 Nc6 3.Nf3 Nf6 4.e3 *"),
("A35i", "English: Symmetrical, Four Knights, 4.e3 e5", "1.c4 c5 2.Nc3 Nc6 3.Nf3 Nf6 4.e3 e5 *"),
("A35j", "English: Symmetrical, Four Knights, 4.g3", "1.c4 c5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 *"),
("A35k", "English: Symmetrical, Four Knights, 4.g3 d5", "1.c4 c5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 d5 *"),
("A35l", "English: Symmetrical, Four Knights, 4.g3 d5 5.cxd5", "1.c4 c5 2.Nc3 Nc6 3.Nf3 Nf6 4.g3 d5 5.cxd5 *"),
("A36a", "English: Symmetrical, 3.g3", "1.c4 c5 2.Nc3 Nc6 3.g3 *"),
("A36b", "English: Symmetrical, 3.g3 Nf6", "1.c4 c5 2.Nc3 Nc6 3.g3 Nf6 *"),
("A36c", "English: Symmetrical, 3.g3 e6", "1.c4 c5 2.Nc3 Nc6 3.g3 e6 *"),
("A36c", "English: Symmetrical, 3.g3 e6 4.Nf3", "1.c4 c5 2.Nc3 Nc6 3.g3 e6 4.Nf3 *"),
("A36c", "English: Symmetrical, Keres-Parma System", "1.c4 c5 2.Nc3 Nc6 3.g3 e6 4.Nf3 Nf6 *"),
("A36d", "English: Symmetrical, Keres-Parma, Main Line Exchange", "1.c4 c5 2.Nc3 Nc6 3.g3 e6 4.Nf3 Nf6 5.Bg2 Be7 6.O-O d5 7.cxd5 Nxd5 *"),
("A36e", "English: Symmetrical, 3.g3 g6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 *"),
("A36e", "English: Symmetrical, 3.g3 g6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 *"),
("A36e", "English: Symmetrical, 3.g3 g6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 *"),
("A36f", "English: Symmetrical, 5.d3", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 *"),
("A36g", "English: Symmetrical, 5.a3", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.a3 *"),
("A36h", "English: Symmetrical, 5.a3 e6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.a3 e6 *"),
("A36i", "English: Symmetrical, 5.a3 d6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.a3 d6 *"),
("A36j", "English: Symmetrical, 5.b3", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.b3 *"),
("A36k", "English: Symmetrical, 5.e3", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e3 *"),
("A36l", "English: Symmetrical, 5.e3 e5 (Botvinnik Reversed)", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e3 e5 *"),
("A36m", "English: Symmetrical, 5.e3 e6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e3 e6 *"),
("A36m", "English: Symmetrical, 5.e3 e6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e3 e6 6.Nge2 *"),
("A36m", "English: Symmetrical, 5.e3 e6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e3 e6 6.Nge2 Nge7 *"),
("A36n", "English: Symmetrical, Botvinnik System", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e4 *"),
("A36o", "English: Symmetrical, Botvinnik, 5...Nf6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e4 Nf6 *"),
("A36o", "English: Symmetrical, Botvinnik, 5...Nf6 6.Nge2", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e4 Nf6 6.Nge2 *"),
("A36p", "English: Symmetrical, Botvinnik, 5...e6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e4 e6 *"),
("A36p", "English: Symmetrical, Botvinnik, 5...e6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e4 e6 6.Nge2 Nge7 *"),
("A36q", "English: Symmetrical, Botvinnik, 5...e6, 8.d3", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e4 e6 6.Nge2 Nge7 7.O-O O-O 8.d3 *"),
("A36r", "English: Symmetrical, Botvinnik, 5...e6, 8.d3 d6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e4 e6 6.Nge2 Nge7 7.O-O O-O 8.d3 d6 *"),
("A36s", "English: Symmetrical, Botvinnik, 5...d6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e4 d6 *"),
("A36t", "English: Symmetrical, Botvinnik, 5...d6 6.Nge2", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e4 d6 6.Nge2 *"),
("A36u", "English: Symmetrical, Botvinnik, 5...d6 6.Nge2 Nf6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e4 d6 6.Nge2 Nf6 *"),
("A36v", "English: Symmetrical, Botvinnik System, 5...d6, 7.O-O O-O", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e4 d6 6.Nge2 Nf6 7.O-O O-O *"),
("A36w", "English: Symmetrical, Botvinnik System, 5...d6, 8.d3", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e4 d6 6.Nge2 Nf6 7.O-O O-O 8.d3 *"),
("A36x", "English: Symmetrical, Botvinnik System, 5...d6, 8.d3 Rb8", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e4 d6 6.Nge2 Nf6 7.O-O O-O 8.d3 Rb8 *"),
("A36y", "English: Symmetrical, Botvinnik System, 5...d6, 8.d3 Ne8", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e4 d6 6.Nge2 Nf6 7.O-O O-O 8.d3 Ne8 *"),
("A36z", "English: Symmetrical, Botvinnik System, 5...d6, 8.d3 a6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.e4 d6 6.Nge2 Nf6 7.O-O O-O 8.d3 a6 *"),
("A37a", "English: Symmetrical, 5.Nf3", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 *"),
("A37b", "English: Symmetrical, 5.Nf3 a6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 a6 *"),
("A37c", "English: Symmetrical, 5.Nf3 Nh6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nh6 *"),
("A37d", "English: Symmetrical, 5.Nf3 Nh6 6.O-O", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nh6 6.O-O *"),
("A37e", "English: Symmetrical, 5.Nf3 d6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 d6 *"),
("A37f", "English: Symmetrical, 5.Nf3 d6 6.d3", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 d6 6.d3 *"),
("A37g", "English: Symmetrical, 5.Nf3 d6 6.O-O", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 d6 6.O-O *"),
("A37h", "English: Symmetrical, 5.Nf3 d6 6.O-O Nh6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 d6 6.O-O Nh6 *"),
("A37i", "English: Symmetrical, 5.Nf3 e6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 e6 *"),
("A37j", "English: Symmetrical, 5.Nf3 e6 6.d3", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 e6 6.d3 *"),
("A37k", "English: Symmetrical, 5.Nf3 e6 6.e3", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 e6 6.e3 *"),
("A37l", "English: Symmetrical, 5.Nf3 e6 6.O-O", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 e6 6.O-O *"),
("A37l", "English: Symmetrical, 5.Nf3 e6 6.O-O Nge7", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 e6 6.O-O Nge7 *"),
("A37m", "English: Symmetrical, 5.Nf3 e6 6.O-O Nge7 7.e3", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 e6 6.O-O Nge7 7.e3 *"),
("A37n", "English: Symmetrical, 5.Nf3 e6 6.O-O Nge7 7.d3", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 e6 6.O-O Nge7 7.d3 *"),
("A37o", "English: Symmetrical, 5.Nf3 e6 6.O-O Nge7 7.d3 O-O", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 e6 6.O-O Nge7 7.d3 O-O *"),
("A37p", "English: Symmetrical, 5.Nf3 e6 6.O-O Nge7 7.d3 O-O 8.Bd2", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 e6 6.O-O Nge7 7.d3 O-O 8.Bd2 *"),
("A37q", "English: Symmetrical, 5.Nf3 e5", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 e5 *"),
("A37r", "English: Symmetrical, 5.Nf3 e5 6.a3", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 e5 6.a3 *"),
("A37s", "English: Symmetrical, 5.Nf3 e5 6.d3", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 e5 6.d3 *"),
("A37t", "English: Symmetrical, 5.Nf3 e5 6.d3 Nge7", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 e5 6.d3 Nge7 *"),
("A37u", "English: Symmetrical, 5.Nf3 e5 6.O-O", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 e5 6.O-O *"),
("A37u", "English: Symmetrical, 5.Nf3 e5 6.O-O d6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 e5 6.O-O d6 *"),
("A37v", "English: Symmetrical, 5.Nf3 e5 6.O-O d6 7.d3", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 e5 6.O-O d6 7.d3 *"),
("A37v", "English: Symmetrical, 5.Nf3 e5 6.O-O d6 7.d3 Nge7", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 e5 6.O-O d6 7.d3 Nge7 *"),
("A37w", "English: Symmetrical, 5.Nf3 e5 6.O-O d6 7.d3 Nge7 8.a3", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 e5 6.O-O d6 7.d3 Nge7 8.a3 *"),
("A37x", "English: Symmetrical, 5.Nf3 e5 6.O-O Nge7", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 e5 6.O-O Nge7 *"),
("A37y", "English: Symmetrical, 5.Nf3 e5 6.O-O Nge7 7.d3", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 e5 6.O-O Nge7 7.d3 *"),
("A37y", "English: Symmetrical, 5.Nf3 e5 6.O-O Nge7 7.d3 O-O", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 e5 6.O-O Nge7 7.d3 O-O *"),
("A37y", "English: Symmetrical, 5.Nf3 e5 6.O-O Nge7 7.d3 O-O 8.a3", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 e5 6.O-O Nge7 7.d3 O-O 8.a3 *"),
("A38a", "English: Symmetrical, Main Line", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 *"),
("A38b", "English: Symmetrical, Main Line, 6.d4", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.d4 *"),
("A38c", "English: Symmetrical, Main Line, 6.O-O", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O *"),
("A38d", "English: Symmetrical, Main Line, 6.O-O d5", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O d5 *"),
("A38e", "English: Symmetrical, Main Line, 6.O-O d6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O d6 *"),
("A38f", "English: Symmetrical, Main Line, 6.O-O O-O", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O *"),
("A38g", "English: Symmetrical, Main Line, 7.Rb1", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.Rb1 *"),
("A38h", "English: Symmetrical, Main Line, 7.b3", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.b3 *"),
("A38i", "English: Symmetrical, Main Line, 7.a3", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.a3 *"),
("A38j", "English: Symmetrical, Main Line, 7.a3 d6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.a3 d6 *"),
("A38k", "English: Symmetrical, Main Line, 7.d3", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d3 *"),
("A38l", "English: Symmetrical, Main Line, 7.d3 a6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d3 a6 *"),
("A38m", "English: Symmetrical, Main Line, 7.d3 d5", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d3 d5 *"),
("A38n", "English: Symmetrical, Main Line, 7.d3 d6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d3 d6 *"),
("A38o", "English: Symmetrical, Main Line, 7.d3 d6 8.Rb1", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d3 d6 8.Rb1 *"),
("A38p", "English: Symmetrical, Main Line, 7.d3 d6 8.Bd2", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d3 d6 8.Bd2 *"),
("A38q", "English: Symmetrical, Main Line, 7.d3 d6 8.a3", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d3 d6 8.a3 *"),
("A38r", "English: Symmetrical, Main Line, 7.d3 d6 8.a3 a6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d3 d6 8.a3 a6 *"),
("A39a", "English: Symmetrical, Main Line 7.d4", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d4 *"),
("A39b", "English: Symmetrical, Main Line 7.d4", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d4 cxd4 *"),
("A39c", "English: Symmetrical, Main Line 7.d4", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d4 cxd4 8.Nxd4 *"),
("A39d", "English: Symmetrical, Main Line 7.d4, 8...Qa5", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d4 cxd4 8.Nxd4 Qa5 *"),
("A39e", "English: Symmetrical, Main Line 7.d4, 8...Qb6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d4 cxd4 8.Nxd4 Qb6 *"),
("A39e", "English: Symmetrical, Main Line 7.d4, 8...Qb6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d4 cxd4 8.Nxd4 Qb6 9.Nc2 *"),
("A39f", "English: Symmetrical, Main Line 7.d4, 8...a6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d4 cxd4 8.Nxd4 a6 *"),
("A39g", "English: Symmetrical, Main Line 7.d4, 8...d6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d4 cxd4 8.Nxd4 d6 *"),
("A39h", "English: Symmetrical, Main Line 7.d4, 8...Ng4", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d4 cxd4 8.Nxd4 Ng4 *"),
("A39i", "English: Symmetrical, Main Line 7.d4, 8...Ng4 9.e3 d6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d4 cxd4 8.Nxd4 Ng4 9.e3 d6 *"),
("A39j", "English: Symmetrical, Main Line 7.d4, 8...Nxd4", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d4 cxd4 8.Nxd4 Nxd4 *"),
("A39j", "English: Symmetrical, Main Line 7.d4, 9.Qxd4", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d4 cxd4 8.Nxd4 Nxd4 9.Qxd4 *"),
("A39k", "English: Symmetrical, Main Line 7.d4, 9...d6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d4 cxd4 8.Nxd4 Nxd4 9.Qxd4 d6 *"),
("A39l", "English: Symmetrical, Main Line 7.d4, 10.Qd2", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d4 cxd4 8.Nxd4 Nxd4 9.Qxd4 d6 10.Qd2 *"),
("A39m", "English: Symmetrical, Main Line 7.d4, 10.Qd3", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d4 cxd4 8.Nxd4 Nxd4 9.Qxd4 d6 10.Qd3 *"),
("A39n", "English: Symmetrical, Main Line 7.d4, 10.Qd3 Bf5", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d4 cxd4 8.Nxd4 Nxd4 9.Qxd4 d6 10.Qd3 Bf5 *"),
("A39o", "English: Symmetrical, Main Line 7.d4, 10.Qd3 Rb8", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d4 cxd4 8.Nxd4 Nxd4 9.Qxd4 d6 10.Qd3 Rb8 *"),
("A39p", "English: Symmetrical, Main Line 7.d4, 10.Qd3 a6", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d4 cxd4 8.Nxd4 Nxd4 9.Qxd4 d6 10.Qd3 a6 *"),
("A39q", "English: Symmetrical, Main Line 7.d4, 10.Qd3 a6 11.Bd2", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d4 cxd4 8.Nxd4 Nxd4 9.Qxd4 d6 10.Qd3 a6 11.Bd2 *"),
("A39q", "English: Symmetrical, Main Line 7.d4, 10.Qd3 a6 11.Bd2 Rb8", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d4 cxd4 8.Nxd4 Nxd4 9.Qxd4 d6 10.Qd3 a6 11.Bd2 Rb8 *"),
("A39r", "English: Symmetrical, Main Line 7.d4, 10.Qd3 a6 11.Bd2 Rb8 12.Rac1", "1.c4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nf3 Nf6 6.O-O O-O 7.d4 cxd4 8.Nxd4 Nxd4 9.Qxd4 d6 10.Qd3 a6 11.Bd2 Rb8 12.Rac1 *"),
("A40a", "Queen's Pawn Game", "1.d4 *"),
("A40b", "Queen's Pawn: 1...c6", "1.d4 c6 *"),
("A40b", "Queen's Pawn: 1...c6 2.Nf3", "1.d4 c6 2.Nf3 *"),
("A40b", "Queen's Pawn: 1...c6 2.c4", "1.d4 c6 2.c4 *"),
("A40b", "Queen's Pawn: Jadoul", "1.d4 c6 2.c4 b5 *"),
("A40c", "Queen's Pawn: Polish Defence", "1.d4 b5 *"),
("A40c", "Queen's Pawn: Polish Defence", "1.d4 b5 2.e4 *"),
("A40c", "Queen's Pawn: Polish Defence", "1.d4 b5 2.e4 Bb7 *"),
("A40c", "Queen's Pawn: Polish Defence, Spassky Gambit", "1.d4 b5 2.e4 Bb7 3.Bxb5 *"),
("A40d", "Queen's Pawn: English Defence", "1.d4 b6 *"),
("A40d", "Queen's Pawn: English Defence, 2.c4", "1.d4 b6 2.c4 *"),
("A40d", "Queen's Pawn: English Defence, 2.c4 Bb7", "1.d4 b6 2.c4 Bb7 *"),
("A40e", "Queen's Pawn: English Defence, 2.c4 e6", "1.d4 b6 2.c4 e6 *"),
("A40f", "Queen's Pawn: English Defence, 3.a3", "1.d4 b6 2.c4 e6 3.a3 *"),
("A40g", "Queen's Pawn: English Defence, 3.e4", "1.d4 b6 2.c4 e6 3.e4 *"),
("A40g", "Queen's Pawn: English Defence, Polli Gambit", "1.d4 b6 2.c4 e6 3.e4 Bb7 4.f3 f5 *"),
("A40g", "Queen's Pawn: English Defence, Hartlaub Gambit", "1.d4 b6 2.c4 Bb7 3.Nc3 e6 4.e4 f5 *"),
("A40h", "Queen's Pawn: English Defence, 3.Nc3", "1.d4 b6 2.c4 e6 3.Nc3 *"),
("A40i", "Englund Gambit", "1.d4 e5 *"),
("A40i", "Englund Gambit Accepted", "1.d4 e5 2.dxe5 *"),
("A40i", "Englund Gambit: Soller", "1.d4 e5 2.dxe5 f6 *"),
("A40i", "Englund Gambit: Hartlaub", "1.d4 e5 2.dxe5 d6 *"),
("A40i", "Englund Gambit: 2.dxe5 Nc6", "1.d4 e5 2.dxe5 Nc6 *"),
("A40i", "Englund Gambit: 2.dxe5 Nc6 3.Nf3", "1.d4 e5 2.dxe5 Nc6 3.Nf3 *"),
("A40i", "Englund Gambit: Soller Deferred", "1.d4 e5 2.dxe5 Nc6 3.Nf3 f6 *"),
("A40i", "Englund Gambit: Zilbermints", "1.d4 e5 2.dxe5 Nc6 3.Nf3 Nge7 *"),
("A40i", "Englund Gambit: 2.dxe5 Nc6 3.Nf3 Qe7", "1.d4 e5 2.dxe5 Nc6 3.Nf3 Qe7 *"),
("A40j", "Queen's Pawn: Bogoljubow-Miles Defence", "1.d4 Nc6 *"),
("A40j", "Queen's Pawn: Bogoljubow-Miles, 2.Bg5", "1.d4 Nc6 2.Bg5 *"),
("A40j", "Queen's Pawn: Bogoljubow-Miles, 2.d5", "1.d4 Nc6 2.d5 *"),
("A40k", "Queen's Pawn: Bogoljubow-Miles, 2.c4", "1.d4 Nc6 2.c4 *"),
("A40k", "Queen's Pawn: Bogoljubow-Miles, Pozarek Gambit", "1.d4 Nc6 2.c4 e5 3.dxe5 Nxe5 4.Nc3 Nxc4 *"),
("A40l", "Queen's Pawn: Bogoljubow-Miles, 2.Nf3", "1.d4 Nc6 2.Nf3 *"),
("A40m", "Queen's Pawn: 1...e6", "1.d4 e6 *"),
("A40n", "Queen's Pawn: 1...e6 2.Nf3", "1.d4 e6 2.Nf3 *"),
("A40n", "Queen's Pawn: 1...e6 2.Nf3 c5", "1.d4 e6 2.Nf3 c5 *"),
("A40o", "Queen's Pawn: 1...e6 2.c4", "1.d4 e6 2.c4 *"),
("A40p", "Queen's Pawn: Keres Defence", "1.d4 e6 2.c4 Bb4+ *"),
("A40q", "Queen's Pawn: Keres Defence, 3.Bd2", "1.d4 e6 2.c4 Bb4+ 3.Bd2 *"),
("A40q", "Queen's Pawn: Keres Defence, 3.Bd2 Bxd2+", "1.d4 e6 2.c4 Bb4+ 3.Bd2 Bxd2+ *"),
("A40q", "Queen's Pawn: Keres Defence, Kangaroo Variation", "1.d4 e6 2.c4 Bb4+ 3.Bd2 Bxd2+ 4.Qxd2 b6 5.Nc3 Bb7 6.e4 Nh6 *"),
("A40r", "Queen's Pawn: Keres Defence, 3.Bd2 a5", "1.d4 e6 2.c4 Bb4+ 3.Bd2 a5 *"),
("A40s", "Queen's Pawn: Modern", "1.d4 g6 *"),
("A40t", "Queen's Pawn: Modern", "1.d4 g6 2.Nf3 *"),
("A40u", "Queen's Pawn: Modern", "1.d4 g6 2.c4 *"),
("A40u", "Queen's Pawn: Modern", "1.d4 g6 2.c4 Bg7 *"),
("A40v", "Queen's Pawn: Modern", "1.d4 g6 2.c4 Bg7 3.e4 *"),
("A40w", "Queen's Pawn: Modern", "1.d4 g6 2.c4 Bg7 3.Nf3 *"),
("A40w", "Queen's Pawn: Modern", "1.d4 g6 2.c4 Bg7 3.Nf3 c5 *"),
("A40x", "Queen's Pawn: Modern", "1.d4 g6 2.c4 Bg7 3.Nc3 *"),
("A40x", "Queen's Pawn: Modern", "1.d4 g6 2.c4 Bg7 3.Nc3 c5 *"),
("A40x", "Queen's Pawn: Modern", "1.d4 g6 2.c4 Bg7 3.Nc3 c5 4.d5 *"),
("A40y", "Queen's Pawn: Modern, Beefeater Defence", "1.d4 g6 2.c4 Bg7 3.Nc3 c5 4.d5 Bxc3+ 5.bxc3 f5 *"),
("A41a", "Neo-Old Indian", "1.d4 d6 *"),
("A41b", "Neo-Old Indian: 2.Bg5", "1.d4 d6 2.Bg5 *"),
("A41c", "Neo-Old Indian: 2.g3", "1.d4 d6 2.g3 *"),
("A41d", "Neo-Old Indian: 2.Nf3", "1.d4 d6 2.Nf3 *"),
("A41e", "Neo-Old Indian / Modern", "1.d4 d6 2.Nf3 g6 *"),
("A41e", "Neo-Old Indian / Modern: 3.Bf4", "1.d4 d6 2.Nf3 g6 3.Bf4 *"),
("A41e", "Neo-Old Indian / Modern: 3.Bf4", "1.d4 d6 2.Nf3 g6 3.Bf4 Bg7 *"),
("A41e", "Neo-Old Indian / Modern: 3.g3", "1.d4 d6 2.Nf3 g6 3.g3 *"),
("A41e", "Neo-Old Indian / Modern: 3.g3", "1.d4 d6 2.Nf3 g6 3.g3 Bg7 *"),
("A41e", "Neo-Old Indian / Modern: 3.g3", "1.d4 d6 2.Nf3 g6 3.g3 Bg7 4.Bg2 *"),
("A41f", "Neo-Old Indian: Wade Defence", "1.d4 d6 2.Nf3 Bg4 *"),
("A41f", "Neo-Old Indian: Wade Defence, 3.e3", "1.d4 d6 2.Nf3 Bg4 3.e3 *"),
("A41f", "Neo-Old Indian: Wade Defence, 3.e3 Nd7", "1.d4 d6 2.Nf3 Bg4 3.e3 Nd7 *"),
("A41f", "Neo-Old Indian: Wade Defence, 3.e3 Nf6", "1.d4 d6 2.Nf3 Bg4 3.e3 Nf6 *"),
("A41g", "Neo-Old Indian: Wade Defence, 3.c4", "1.d4 d6 2.Nf3 Bg4 3.c4 *"),
("A41g", "Neo-Old Indian: Wade Defence, 3.c4 e5", "1.d4 d6 2.Nf3 Bg4 3.c4 e5 *"),
("A41g", "Neo-Old Indian: Wade Defence, 3.c4 e5 4.Nc3", "1.d4 d6 2.Nf3 Bg4 3.c4 e5 4.Nc3 *"),
("A41g", "Neo-Old Indian: Wade Defence, 3.c4 e5 4.Nc3 Nc6", "1.d4 d6 2.Nf3 Bg4 3.c4 e5 4.Nc3 Nc6 *"),
("A41g", "Neo-Old Indian: Wade Defence, 3.c4 e5 4.dxe5", "1.d4 d6 2.Nf3 Bg4 3.c4 e5 4.dxe5 *"),
("A41g", "Neo-Old Indian: Wade Defence, 3.c4 e5 4.dxe5 Nc6 Gambit", "1.d4 d6 2.Nf3 Bg4 3.c4 e5 4.dxe5 Nc6 *"),
("A41h", "Neo-Old Indian: Wade Defence, 3.c4 Nd7", "1.d4 d6 2.Nf3 Bg4 3.c4 Nd7 *"),
("A41h", "Neo-Old Indian: Wade Defence, 3.c4 Nd7 4.Nc3", "1.d4 d6 2.Nf3 Bg4 3.c4 Nd7 4.Nc3 *"),
("A41h", "Neo-Old Indian: Wade Defence, 3.c4 Nd7 4.Nc3 e5", "1.d4 d6 2.Nf3 Bg4 3.c4 Nd7 4.Nc3 e5 *"),
("A41i", "Neo-Old Indian: Wade Defence, 3.c4 Bxf3", "1.d4 d6 2.Nf3 Bg4 3.c4 Bxf3 *"),
("A41j", "Neo-Old Indian: Wade Defence, 3.e4", "1.d4 d6 2.Nf3 Bg4 3.e4 *"),
("A41k", "Neo-Old Indian: Wade Defence, 3.e4 Nf6", "1.d4 d6 2.Nf3 Bg4 3.e4 Nf6 *"),
("A41l", "Neo-Old Indian: 2.c4", "1.d4 d6 2.c4 *"),
("A41m", "Neo-Old Indian: 2.c4 e5", "1.d4 d6 2.c4 e5 *"),
("A41n", "Neo-Old Indian: 2.c4 e5 3.d5", "1.d4 d6 2.c4 e5 3.d5 *"),
("A41o", "Neo-Old Indian: 2.c4 e5 3.dxe5", "1.d4 d6 2.c4 e5 3.dxe5 *"),
("A41o", "Neo-Old Indian: Queenswap", "1.d4 d6 2.c4 e5 3.dxe5 dxe5 4.Qxd8+ Kxd8 *"),
("A41p", "Neo-Old Indian: 2.c4 e5 3.Nf3", "1.d4 d6 2.c4 e5 3.Nf3 *"),
("A41p", "Neo-Old Indian: 2.c4 e5 3.Nf3 e4", "1.d4 d6 2.c4 e5 3.Nf3 e4 *"),
("A41q", "Neo-Old Indian: Modern", "1.d4 d6 2.c4 g6 *"),
("A41q", "Neo-Old Indian: Modern, 3.e4", "1.d4 d6 2.c4 g6 3.e4 *"),
("A41q", "Neo-Old Indian: Modern, 3.e4 Bg7", "1.d4 d6 2.c4 g6 3.e4 Bg7 *"),
("A41r", "Neo-Old Indian: Modern, 3.Nf3", "1.d4 d6 2.c4 g6 3.Nf3 *"),
("A41r", "Neo-Old Indian: Modern, 3.Nf3 Bg7", "1.d4 d6 2.c4 g6 3.Nf3 Bg7 *"),
("A41r", "Neo-Old Indian: Modern, 3.Nf3 Bg7 4.g3", "1.d4 d6 2.c4 g6 3.Nf3 Bg7 4.g3 *"),
("A41s", "Neo-Old Indian: Modern, 3.Nf3 Bg7 4.e4", "1.d4 d6 2.c4 g6 3.Nf3 Bg7 4.e4 *"),
("A41s", "Neo-Old Indian: Modern, Rossolimo Variation", "1.d4 d6 2.c4 g6 3.Nf3 Bg7 4.e4 Bg4 *"),
("A41t", "Neo-Old Indian: Modern, 3.Nc3", "1.d4 d6 2.c4 g6 3.Nc3 *"),
("A41t", "Neo-Old Indian: Modern, 3.Nc3 Bg7", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 *"),
("A41u", "Neo-Old Indian: Modern, 3.Nc3 Bg7 4.Nf3", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.Nf3 *"),
("A41v", "Neo-Old Indian: Modern, 3.Nc3 Bg7 4.Nf3 Bf4", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.Nf3 Bg4 *"),
("A41w", "Neo-Old Indian: Modern, 3.Nc3 Bg7 4.Nf3 Bf4 5.e3", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.Nf3 Bg4 5.e3 *"),
("A41w", "Neo-Old Indian: Modern, 3.Nc3 Bg7 4.Nf3 Bf4 5.e3 Nc6", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.Nf3 Bg4 5.e3 Nc6 *"),
("A42a", "Modern: Averbakh", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 *"),
("A42b", "Modern: Averbakh, Randspringer Variation", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 f5 *"),
("A42b", "Modern: Averbakh, Randspringer, 5.Nf3", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 f5 5.Nf3 *"),
("A42c", "Modern: c4 Pterodactyl", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 c5 *"),
("A42c", "Modern: c4 Pterodactyl", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 c5 5.Nf3 *"),
("A42c", "Modern: c4 Pterodactyl", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 c5 5.Nf3 Qa5 *"),
("A42d", "Modern: Averbakh, 4...c6", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 c6 *"),
("A42d", "Modern: Averbakh, 4...c6 5.Be3", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 c6 5.Be3 *"),
("A42e", "Modern: Averbakh, 4...c6 5.Nf3", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 c6 5.Nf3 *"),
("A42f", "Modern: Averbakh, 4...Nd7", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 Nd7 *"),
("A42f", "Modern: Averbakh, 4...Nd7 5.Nf3", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 Nd7 5.Nf3 *"),
("A42g", "Modern: Averbakh, Kotov Variation", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 Nc6 *"),
("A42g", "Modern: Averbakh, Kotov, 5.Nf3", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 Nc6 5.Nf3 *"),
("A42g", "Modern: Averbakh, Kotov, 5.Nge2", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 Nc6 5.Nge2 *"),
("A42h", "Modern: Averbakh, Kotov, 5.d5", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 Nc6 5.d5 *"),
("A42i", "Modern: Averbakh, Kotov, 5.Be3", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 Nc6 5.Be3 *"),
("A42j", "Modern: Averbakh, Kotov, 5.Be3 e5", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 Nc6 5.Be3 e5 *"),
("A42j", "Modern: Averbakh, Kotov, 5.Be3 e5 6.Nge2", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 Nc6 5.Be3 e5 6.Nge2 *"),
("A42k", "Modern: Averbakh, Kotov, 5.Be3 e5 6.d5", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 Nc6 5.Be3 e5 6.d5 *"),
("A42l", "Modern: Averbakh, Kotov, 5.Be3 e5 6.d5 Nce7", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 Nc6 5.Be3 e5 6.d5 Nce7 *"),
("A42l", "Modern: Averbakh, Kotov, 5.Be3 e5 6.d5 Nce7 7.c5", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 Nc6 5.Be3 e5 6.d5 Nce7 7.c5 *"),
("A42m", "Modern: Averbakh, Kotov, 5.Be3 e5 6.d5 Nce7 7.g4", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 Nc6 5.Be3 e5 6.d5 Nce7 7.g4 *"),
("A42n", "Modern: Averbakh, 4...e5", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 e5 *"),
("A42o", "Modern: Averbakh, 4...e5 5.Nge2", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 e5 5.Nge2 *"),
("A42p", "Modern: Averbakh, 4...e5 5.Nf3", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 e5 5.Nf3 *"),
("A42q", "Modern: Averbakh, 4...e5 5.Nf3 Nd7", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 e5 5.Nf3 Nd7 *"),
("A42r", "Modern: Averbakh, 4...e5 5.Nf3 Nd7 6.Be2", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 e5 5.Nf3 Nd7 6.Be2 *"),
("A42s", "Modern: Averbakh, 4...e5 5.Nf3 Nd7 6.Be2 Ne7", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 e5 5.Nf3 Nd7 6.Be2 Ne7 *"),
("A42t", "Modern: Averbakh, 4...e5 5.d5", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 e5 5.d5 *"),
("A42u", "Modern: Averbakh, 4...e5 5.d5 Nd7", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 e5 5.d5 Nd7 *"),
("A42v", "Modern: Averbakh, 4...e5 5.dxe5", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 e5 5.dxe5 *"),
("A42v", "Modern: Averbakh, 4...e5 5.dxe5 dxe5", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 e5 5.dxe5 dxe5 *"),
("A42v", "Modern: Averbakh, Queenswap", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 e5 5.dxe5 dxe5 6.Qxd8+ Kxd8 *"),
("A42w", "Modern: Averbakh, Queenswap, 7.f4", "1.d4 d6 2.c4 g6 3.Nc3 Bg7 4.e4 e5 5.dxe5 dxe5 6.Qxd8+ Kxd8 7.f4 *"),
("A43a", "Old Benoni", "1.d4 c5 *"),
("A43a", "Old Benoni: Nakamura Gambit", "1.d4 c5 2.b4 *"),
("A43b", "Old Benoni: 2.dxc5", "1.d4 c5 2.dxc5 *"),
("A43b", "Old Benoni: Cormorant Gambit", "1.d4 c5 2.dxc5 b6 *"),
("A43c", "Old Benoni: 2.c3", "1.d4 c5 2.c3 *"),
("A43d", "Old Benoni: 2.e3", "1.d4 c5 2.e3 *"),
("A43e", "Old Benoni: 2.d5", "1.d4 c5 2.d5 *"),
("A43e", "Old Benoni: 2.d5 b5", "1.d4 c5 2.d5 b5 *"),
("A43f", "Old Benoni: 2.d5 e6", "1.d4 c5 2.d5 e6 *"),
("A43g", "Old Benoni: Franco-Benoni", "1.d4 c5 2.d5 e6 3.e4 *"),
("A43h", "Old Benoni: 2.d5 e6 3.c4", "1.d4 c5 2.d5 e6 3.c4 *"),
("A43h", "Old Benoni: 2.d5 e6 3.c4", "1.d4 c5 2.d5 e6 3.c4 exd5 4.cxd5 *"),
("A43h", "Old Benoni: 2.d5 e6 3.c4", "1.d4 c5 2.d5 e6 3.c4 exd5 4.cxd5 d6 *"),
("A43h", "Old Benoni: 2.d5 e6 3.c4", "1.d4 c5 2.d5 e6 3.c4 exd5 4.cxd5 d6 5.e4 *"),
("A43i", "Old Benoni: 2.d5 e6 3.c4", "1.d4 c5 2.d5 e6 3.c4 exd5 4.cxd5 d6 5.Nc3 *"),
("A43i", "Old Benoni: 2.d5 e6 3.c4", "1.d4 c5 2.d5 e6 3.c4 exd5 4.cxd5 d6 5.Nc3 g6 6.e4 Bg7 *"),
("A43i", "Old Benoni: 2.d5 e6 3.c4", "1.d4 c5 2.d5 e6 3.c4 exd5 4.cxd5 d6 5.Nc3 g6 6.e4 Bg7 7.Nf3 *"),
("A43j", "Old Benoni: Clarendon Court Defence", "1.d4 c5 2.d5 f5 *"),
("A43j", "Old Benoni: Clarendon Court, 3.c4", "1.d4 c5 2.d5 f5 3.c4 *"),
("A43j", "Old Benoni: Clarendon Court, 3.g3", "1.d4 c5 2.d5 f5 3.g3 *"),
("A43j", "Old Benoni: Clarendon Court, 3.Nc3", "1.d4 c5 2.d5 f5 3.Nc3 *"),
("A43k", "Old Benoni: 2.d5 Nf6", "1.d4 c5 2.d5 Nf6 *"),
("A43k", "Old Benoni: 2.d5 Nf6 3.Nc3", "1.d4 c5 2.d5 Nf6 3.Nc3 *"),
("A43k", "Old Benoni: Woozle", "1.d4 c5 2.d5 Nf6 3.Nc3 Qa5 *"),
("A43l", "Old Benoni: 2.d5 Nf6 3.Nf3", "1.d4 c5 2.d5 Nf6 3.Nf3 *"),
("A43l", "Old Benoni: 2.d5 Nf6 3.Nf3 e6", "1.d4 c5 2.d5 Nf6 3.Nf3 e6 *"),
("A43l", "Old Benoni: 2.d5 Nf6 3.Nf3 e6 4.Nc3", "1.d4 c5 2.d5 Nf6 3.Nf3 e6 4.Nc3 *"),
("A43m", "Old Benoni: 2.d5 Nf6 3.Nf3 g6", "1.d4 c5 2.d5 Nf6 3.Nf3 g6 *"),
("A43m", "Old Benoni: 2.d5 Nf6 3.Nf3 g6 4.Nc3", "1.d4 c5 2.d5 Nf6 3.Nf3 g6 4.Nc3 *"),
("A43n", "Old Benoni: Neo-Benko", "1.d4 c5 2.d5 Nf6 3.Nf3 b5 *"),
("A43o", "Old Benoni: Neo-Benko, 4.Bg5", "1.d4 c5 2.d5 Nf6 3.Nf3 b5 4.Bg5 *"),
("A43p", "Old Benoni: Neo-Benko, 4.Bg5 d6", "1.d4 c5 2.d5 Nf6 3.Nf3 b5 4.Bg5 d6 *"),
("A43q", "Old Benoni: Neo-Benko, 4.Bg5 Ne4", "1.d4 c5 2.d5 Nf6 3.Nf3 b5 4.Bg5 Ne4 *"),
("A43r", "Old Benoni: Hawk", "1.d4 c5 2.d5 Nf6 3.Nf3 c4 *"),
("A43r", "Old Benoni: Hawk, 4.e4", "1.d4 c5 2.d5 Nf6 3.Nf3 c4 4.e4 *"),
("A43s", "Old Benoni: Schmidt", "1.d4 c5 2.d5 d6 *"),
("A43s", "Old Benoni: Schmidt, 3.Nf3", "1.d4 c5 2.d5 d6 3.Nf3 *"),
("A43s", "Old Benoni: Schmidt, 3.Nf3 Nf6", "1.d4 c5 2.d5 d6 3.Nf3 Nf6 *"),
("A43s", "Old Benoni: Schmidt, 3.Nc3", "1.d4 c5 2.d5 d6 3.Nc3 *"),
("A43s", "Old Benoni: Schmidt, 3.Nc3 g6", "1.d4 c5 2.d5 d6 3.Nc3 g6 *"),
("A43s", "Old Benoni: Schmidt, 3.Nc3 Nf6", "1.d4 c5 2.d5 d6 3.Nc3 Nf6 *"),
("A43s", "Old Benoni: Schmidt, 3.e4", "1.d4 c5 2.d5 d6 3.e4 *"),
("A43t", "Old Benoni: Schmidt, 3.e4 g6", "1.d4 c5 2.d5 d6 3.e4 g6 *"),
("A43t", "Old Benoni: Schmidt, 3.e4 g6", "1.d4 c5 2.d5 d6 3.e4 g6 4.Nf3 Bg7 *"),
("A43t", "Old Benoni: Schmidt, 3.e4 g6", "1.d4 c5 2.d5 d6 3.e4 g6 4.Nf3 Bg7 5.Be2 *"),
("A43t", "Old Benoni: Schmidt, 3.e4 g6", "1.d4 c5 2.d5 d6 3.e4 g6 4.Nc3 Bg7 *"),
("A43u", "Old Benoni: Schmidt, 3.e4 Nf6", "1.d4 c5 2.d5 d6 3.e4 Nf6 *"),
("A43u", "Old Benoni: Schmidt, 3.e4 Nf6", "1.d4 c5 2.d5 d6 3.e4 Nf6 4.Nc3 *"),
("A43u", "Old Benoni: Schmidt, 3.e4 Nf6", "1.d4 c5 2.d5 d6 3.e4 Nf6 4.Nc3 g6 *"),
("A43u", "Old Benoni: Schmidt, 3.e4 Nf6", "1.d4 c5 2.d5 d6 3.e4 Nf6 4.Nc3 g6 5.Nf3 *"),
("A43u", "Old Benoni: Schmidt, 3.e4 Nf6", "1.d4 c5 2.d5 d6 3.e4 Nf6 4.Nc3 g6 5.Nf3 Bg7 *"),
("A43v", "Old Benoni: Schmidt, 6.h3", "1.d4 c5 2.d5 d6 3.e4 Nf6 4.Nc3 g6 5.Nf3 Bg7 6.h3 *"),
("A43v", "Old Benoni: Schmidt, 6.h3", "1.d4 c5 2.d5 d6 3.e4 Nf6 4.Nc3 g6 5.Nf3 Bg7 6.h3 O-O *"),
("A43w", "Old Benoni: Schmidt, 6.Bb5+", "1.d4 c5 2.d5 d6 3.e4 Nf6 4.Nc3 g6 5.Nf3 Bg7 6.Bb5+ *"),
("A43x", "Old Benoni: Schmidt, 6.Be2", "1.d4 c5 2.d5 d6 3.e4 Nf6 4.Nc3 g6 5.Nf3 Bg7 6.Be2 *"),
("A43x", "Old Benoni: Schmidt, 6.Be2 O-O", "1.d4 c5 2.d5 d6 3.e4 Nf6 4.Nc3 g6 5.Nf3 Bg7 6.Be2 O-O *"),
("A43x", "Old Benoni: Schmidt, 6.Be2 O-O 7.O-O", "1.d4 c5 2.d5 d6 3.e4 Nf6 4.Nc3 g6 5.Nf3 Bg7 6.Be2 O-O 7.O-O *"),
("A43x", "Old Benoni: Schmidt, 6.Be2 O-O 7.O-O e6", "1.d4 c5 2.d5 d6 3.e4 Nf6 4.Nc3 g6 5.Nf3 Bg7 6.Be2 O-O 7.O-O e6 *"),
("A43y", "Old Benoni: Schmidt, 6.Be2 O-O 7.O-O Na6", "1.d4 c5 2.d5 d6 3.e4 Nf6 4.Nc3 g6 5.Nf3 Bg7 6.Be2 O-O 7.O-O Na6 *"),
("A43z", "Old Benoni: Schmidt, 6.Be2 O-O 7.O-O Na6 8.h3", "1.d4 c5 2.d5 d6 3.e4 Nf6 4.Nc3 g6 5.Nf3 Bg7 6.Be2 O-O 7.O-O Na6 8.h3 *"),
("A44a", "Old Benoni: Czech", "1.d4 c5 2.d5 e5 *"),
("A44b", "Old Benoni: Czech, 3.dxe6", "1.d4 c5 2.d5 e5 3.dxe6 *"),
("A44c", "Old Benoni: Czech, 3.c4", "1.d4 c5 2.d5 e5 3.c4 *"),
("A44d", "Old Benoni: Czech, 3.c4 d6", "1.d4 c5 2.d5 e5 3.c4 d6 *"),
("A44e", "Old Benoni: Czech, 3.c4 d6 4.e4", "1.d4 c5 2.d5 e5 3.c4 d6 4.e4 *"),
("A44f", "Old Benoni: Czech, 3.c4 d6 4.e4 Be7", "1.d4 c5 2.d5 e5 3.c4 d6 4.e4 Be7 *"),
("A44g", "Old Benoni: Czech, 3.c4 d6 4.e4 Be7 5.Nc3", "1.d4 c5 2.d5 e5 3.c4 d6 4.e4 Be7 5.Nc3 *"),
("A44h", "Old Benoni: Czech, 3.c4 d6 4.e4 g6", "1.d4 c5 2.d5 e5 3.c4 d6 4.e4 g6 *"),
("A44h", "Old Benoni: Czech, 3.c4 d6 4.e4 g6 5.Nc3", "1.d4 c5 2.d5 e5 3.c4 d6 4.e4 g6 5.Nc3 *"),
("A44i", "Old Benoni: Czech, 3.c4 d6 4.e4 g6 5.Nc3 Bg7", "1.d4 c5 2.d5 e5 3.c4 d6 4.e4 g6 5.Nc3 Bg7 *"),
("A44j", "Old Benoni: Czech, 3.c4 d6 4.e4 g6 5.Nc3 Bg7 6.Nf3", "1.d4 c5 2.d5 e5 3.c4 d6 4.e4 g6 5.Nc3 Bg7 6.Nf3 *"),
("A44k", "Old Benoni: Czech, 3.c4 d6 4.e4 g6 5.Nc3 Bg7 6.Bd3", "1.d4 c5 2.d5 e5 3.c4 d6 4.e4 g6 5.Nc3 Bg7 6.Bd3 *"),
("A44l", "Old Benoni: Czech, 3.e4", "1.d4 c5 2.d5 e5 3.e4 *"),
("A44m", "Old Benoni: Czech, 3.e4 d6", "1.d4 c5 2.d5 e5 3.e4 d6 *"),
("A44n", "Old Benoni: Czech, 3.e4 d6 4.Nf3", "1.d4 c5 2.d5 e5 3.e4 d6 4.Nf3 *"),
("A44o", "Old Benoni: Czech, 3.e4 d6 4.Bb5+", "1.d4 c5 2.d5 e5 3.e4 d6 4.Bb5+ *"),
("A44p", "Old Benoni: Czech, 3.e4 d6 4.Bd3", "1.d4 c5 2.d5 e5 3.e4 d6 4.Bd3 *"),
("A44q", "Old Benoni: Czech, 3.e4 d6 4.Nc3", "1.d4 c5 2.d5 e5 3.e4 d6 4.Nc3 *"),
("A44r", "Old Benoni: Czech, 3.e4 d6 4.Nc3 Nf6", "1.d4 c5 2.d5 e5 3.e4 d6 4.Nc3 Nf6 *"),
("A44s", "Old Benoni: Czech, 3.e4 d6 4.Nc3 g6", "1.d4 c5 2.d5 e5 3.e4 d6 4.Nc3 g6 *"),
("A44t", "Old Benoni: Czech, 3.e4 d6 4.Nc3 a6", "1.d4 c5 2.d5 e5 3.e4 d6 4.Nc3 a6 *"),
("A44u", "Old Benoni: Czech, 3.e4 d6 4.Nc3 a6 5.a4 Be7", "1.d4 c5 2.d5 e5 3.e4 d6 4.Nc3 a6 5.a4 Be7 *"),
("A44v", "Old Benoni: Czech, 3.e4 d6 4.Nc3 Be7", "1.d4 c5 2.d5 e5 3.e4 d6 4.Nc3 Be7 *"),
("A44w", "Old Benoni: Czech, 3.e4 d6 4.Nc3 Be7 5.Bb5+", "1.d4 c5 2.d5 e5 3.e4 d6 4.Nc3 Be7 5.Bb5+ *"),
("A44x", "Old Benoni: Czech, 3.e4 d6 4.Nc3 Be7 5.Nf3", "1.d4 c5 2.d5 e5 3.e4 d6 4.Nc3 Be7 5.Nf3 *"),
("A44y", "Old Benoni: Czech, 3.e4 d6 4.Nc3 Be7 5.Nf3 Bg4", "1.d4 c5 2.d5 e5 3.e4 d6 4.Nc3 Be7 5.Nf3 Bg4 *"),
("A45a", "Queen's Pawn: Indian", "1.d4 Nf6 *"),
("A45a", "Indian: Paleface Attack", "1.d4 Nf6 2.f3 *"),
("A45a", "Indian: Blackmar-Diemer Gambit (without Nc3)", "1.d4 Nf6 2.f3 d5 3.e4 *"),
("A45a", "Indian: Gedult Attack", "1.d4 Nf6 2.f3 d5 3.g4 *"),
("A45a", "Indian: Omega Gambit", "1.d4 Nf6 2.e4 *"),
("A45a", "Indian: Arafat Gambit", "1.d4 Nf6 2.e4 Nxe4 3.Bd3 Nf6 4.Nf3 *"),
("A45a", "Indian: Gibbins Gambit", "1.d4 Nf6 2.g4 *"),
("A45a", "Indian: Gibbins Gambit, Oshima Defence", "1.d4 Nf6 2.g4 e5 *"),
("A45a", "Indian: Gibbins Gambit Accepted", "1.d4 Nf6 2.g4 Nxg4 *"),
("A45b", "Indian: Canard Opening", "1.d4 Nf6 2.f4 *"),
("A45c", "Indian: 2.Nd2", "1.d4 Nf6 2.Nd2 *"),
("A45c", "Indian: Lazard Gambit", "1.d4 Nf6 2.Nd2 e5 *"),
("A45d", "Indian: 2.e3", "1.d4 Nf6 2.e3 *"),
("A45d", "Indian: 2.e3 e6", "1.d4 Nf6 2.e3 e6 *"),
("A45d", "Indian: 2.e3 g6", "1.d4 Nf6 2.e3 g6 *"),
("A45e", "Indian: 2.c3", "1.d4 Nf6 2.c3 *"),
("A45e", "Indian: 2.c3 g6", "1.d4 Nf6 2.c3 g6 *"),
("A45e", "Indian: 2.c3 g6 3.Bg5", "1.d4 Nf6 2.c3 g6 3.Bg5 *"),
("A45f", "Indian: 2.Nc3", "1.d4 Nf6 2.Nc3 *"),
("A45g", "Indian: 2.Bf4", "1.d4 Nf6 2.Bf4 *"),
("A45h", "Indian: 2.g3", "1.d4 Nf6 2.g3 *"),
("A45h", "Indian: 2.g3 g6", "1.d4 Nf6 2.g3 g6 *"),
("A45i", "Indian: 2.g3 c5", "1.d4 Nf6 2.g3 c5 *"),
("A45i", "Indian: 2.g3 c5 3.d5 b5", "1.d4 Nf6 2.g3 c5 3.d5 b5 *"),
("A45j", "Trompowsky Opening", "1.d4 Nf6 2.Bg5 *"),
("A45j", "Trompowsky 2...d6", "1.d4 Nf6 2.Bg5 d6 *"),
("A45j", "Trompowsky 2...d6 3.Nc3", "1.d4 Nf6 2.Bg5 d6 3.Nc3 *"),
("A45j", "Trompowsky 2...d6 3.Bxf6", "1.d4 Nf6 2.Bg5 d6 3.Bxf6 *"),
("A45k", "Trompowsky 2...g6", "1.d4 Nf6 2.Bg5 g6 *"),
("A45k", "Trompowsky 2...g6 3.Nc3", "1.d4 Nf6 2.Bg5 g6 3.Nc3 *"),
("A45k", "Trompowsky 2...g6 3.Bxf6", "1.d4 Nf6 2.Bg5 g6 3.Bxf6 *"),
("A45l", "Trompowsky 2...g6 3.Bxf6 exf6 4.e3", "1.d4 Nf6 2.Bg5 g6 3.Bxf6 exf6 4.e3 *"),
("A45m", "Trompowsky: 2...e6", "1.d4 Nf6 2.Bg5 e6 *"),
("A45m", "Trompowsky: 2...e6 3.Nc3", "1.d4 Nf6 2.Bg5 e6 3.Nc3 *"),
("A45m", "Trompowsky: 2...e6 3.e3", "1.d4 Nf6 2.Bg5 e6 3.e3 *"),
("A45m", "Trompowsky: 2...e6 3.Nd2", "1.d4 Nf6 2.Bg5 e6 3.Nd2 *"),
("A45n", "Trompowsky: 2...e6 3.e4", "1.d4 Nf6 2.Bg5 e6 3.e4 *"),
("A45n", "Trompowsky: 2...e6 3.e4 h6", "1.d4 Nf6 2.Bg5 e6 3.e4 h6 *"),
("A45n", "Trompowsky: 2...e6 3.e4 h6 4.Bxf6", "1.d4 Nf6 2.Bg5 e6 3.e4 h6 4.Bxf6 *"),
("A45n", "Trompowsky: 2...e6 3.e4 h6 4.Bxf6 Qxf6", "1.d4 Nf6 2.Bg5 e6 3.e4 h6 4.Bxf6 Qxf6 *"),
("A45n", "Trompowsky: 2...e6 3.e4 h6 4.Bxf6 Qxf6 5.c3", "1.d4 Nf6 2.Bg5 e6 3.e4 h6 4.Bxf6 Qxf6 5.c3 *"),
("A45o", "Trompowsky: 2...e6 3.e4 h6 4.Bxf6 Qxf6 5.Nc3", "1.d4 Nf6 2.Bg5 e6 3.e4 h6 4.Bxf6 Qxf6 5.Nc3 *"),
("A45o", "Trompowsky: 2...e6 3.e4 h6 4.Bxf6 Qxf6 5.Nc3 Bb4", "1.d4 Nf6 2.Bg5 e6 3.e4 h6 4.Bxf6 Qxf6 5.Nc3 Bb4 *"),
("A45o", "Trompowsky: 2...e6 3.e4 h6 4.Bxf6 Qxf6 5.Nc3 d6", "1.d4 Nf6 2.Bg5 e6 3.e4 h6 4.Bxf6 Qxf6 5.Nc3 d6 *"),
("A45o", "Trompowsky: 2...e6 3.e4 h6 4.Bxf6 Qxf6 5.Nc3 d6 6.Qd2", "1.d4 Nf6 2.Bg5 e6 3.e4 h6 4.Bxf6 Qxf6 5.Nc3 d6 6.Qd2 *"),
("A45o", "Trompowsky: 2...e6 3.e4 h6 4.Bxf6 Qxf6 5.Nc3 d6 6.Qd2 g5", "1.d4 Nf6 2.Bg5 e6 3.e4 h6 4.Bxf6 Qxf6 5.Nc3 d6 6.Qd2 g5 *"),
("A45p", "Trompowsky: 2...c5", "1.d4 Nf6 2.Bg5 c5 *"),
("A45p", "Trompowsky: 2...c5 3.dxc5", "1.d4 Nf6 2.Bg5 c5 3.dxc5 *"),
("A45p", "Trompowsky: 2...c5 3.Nc3", "1.d4 Nf6 2.Bg5 c5 3.Nc3 *"),
("A45p", "Trompowsky: 2...c5 3.d5", "1.d4 Nf6 2.Bg5 c5 3.d5 *"),
("A45p", "Trompowsky: 2...c5 3.d5 Qb6", "1.d4 Nf6 2.Bg5 c5 3.d5 Qb6 *"),
("A45p", "Trompowsky: 2...c5 3.d5 Qb6 4.Nc3", "1.d4 Nf6 2.Bg5 c5 3.d5 Qb6 4.Nc3 *"),
("A45q", "Trompowsky: 2...c5 3.Bxf6", "1.d4 Nf6 2.Bg5 c5 3.Bxf6 *"),
("A45q", "Trompowsky: 2...c5 3.Bxf6 gxf6", "1.d4 Nf6 2.Bg5 c5 3.Bxf6 gxf6 *"),
("A45q", "Trompowsky: 2...c5 3.Bxf6 gxf6 4.d5", "1.d4 Nf6 2.Bg5 c5 3.Bxf6 gxf6 4.d5 *"),
("A45r", "Trompowsky: 2...c5 3.Bxf6 gxf6 4.d5 Qb6", "1.d4 Nf6 2.Bg5 c5 3.Bxf6 gxf6 4.d5 Qb6 *"),
("A45r", "Trompowsky: 2...c5 3.Bxf6 gxf6 4.d5 Qb6 5.Qc1", "1.d4 Nf6 2.Bg5 c5 3.Bxf6 gxf6 4.d5 Qb6 5.Qc1 *"),
("A45r", "Trompowsky: 2...c5 3.Bxf6 gxf6 4.d5 Qb6 5.Qc1 f5", "1.d4 Nf6 2.Bg5 c5 3.Bxf6 gxf6 4.d5 Qb6 5.Qc1 f5 *"),
("A45r", "Trompowsky: 2...c5 3.Bxf6 gxf6 4.d5 Qb6 5.Qc1 f5 6.c4", "1.d4 Nf6 2.Bg5 c5 3.Bxf6 gxf6 4.d5 Qb6 5.Qc1 f5 6.c4 *"),
("A45r", "Trompowsky: 2...c5 3.Bxf6 gxf6 4.d5 Qb6 5.Qc1 f5 6.g3", "1.d4 Nf6 2.Bg5 c5 3.Bxf6 gxf6 4.d5 Qb6 5.Qc1 f5 6.g3 *"),
("A45r", "Trompowsky: 2...c5 3.Bxf6 gxf6 4.d5 Qb6 5.Qc1 f5 6.e3", "1.d4 Nf6 2.Bg5 c5 3.Bxf6 gxf6 4.d5 Qb6 5.Qc1 f5 6.e3 *"),
("A45r", "Trompowsky: 2...c5 3.Bxf6 gxf6 4.d5 Qb6 5.Qc1 f5 6.e3 Bg7", "1.d4 Nf6 2.Bg5 c5 3.Bxf6 gxf6 4.d5 Qb6 5.Qc1 f5 6.e3 Bg7 *"),
("A45s", "Trompowsky: 2...Ne4", "1.d4 Nf6 2.Bg5 Ne4 *"),
("A45s", "Trompowsky: 2...Ne4 3.h4", "1.d4 Nf6 2.Bg5 Ne4 3.h4 *"),
("A45s", "Trompowsky: 2...Ne4 3.h4 d5", "1.d4 Nf6 2.Bg5 Ne4 3.h4 d5 *"),
("A45s", "Trompowsky: 2...Ne4 3.h4 c5", "1.d4 Nf6 2.Bg5 Ne4 3.h4 c5 *"),
("A45s", "Trompowsky: 2...Ne4 3.h4 c5 4.dxc5", "1.d4 Nf6 2.Bg5 Ne4 3.h4 c5 4.dxc5 *"),
("A45s", "Trompowsky: 2...Ne4 3.h4 c5 4.d5", "1.d4 Nf6 2.Bg5 Ne4 3.h4 c5 4.d5 *"),
("A45t", "Trompowsky: 2...Ne4 3.Bh4", "1.d4 Nf6 2.Bg5 Ne4 3.Bh4 *"),
("A45t", "Trompowsky: 2...Ne4 3.Bh4 g5", "1.d4 Nf6 2.Bg5 Ne4 3.Bh4 g5 *"),
("A45t", "Trompowsky: 2...Ne4 3.Bh4 d5", "1.d4 Nf6 2.Bg5 Ne4 3.Bh4 d5 *"),
("A45t", "Trompowsky: 2...Ne4 3.Bh4 c5", "1.d4 Nf6 2.Bg5 Ne4 3.Bh4 c5 *"),
("A45t", "Trompowsky: 2...Ne4 3.Bh4 c5 4.f3", "1.d4 Nf6 2.Bg5 Ne4 3.Bh4 c5 4.f3 *"),
("A45u", "Trompowsky: 2...Ne4 3.Bf4", "1.d4 Nf6 2.Bg5 Ne4 3.Bf4 *"),
("A45u", "Trompowsky: Borg Variation", "1.d4 Nf6 2.Bg5 Ne4 3.Bf4 g5 *"),
("A45u", "Trompowsky: 2...Ne4 3.Bf4 d5", "1.d4 Nf6 2.Bg5 Ne4 3.Bf4 d5 *"),
("A45u", "Trompowsky: 2...Ne4 3.Bf4 d5 4.Nd2", "1.d4 Nf6 2.Bg5 Ne4 3.Bf4 d5 4.Nd2 *"),
("A45v", "Trompowsky: 2...Ne4 3.Bf4 d5 4.f3", "1.d4 Nf6 2.Bg5 Ne4 3.Bf4 d5 4.f3 *"),
("A45v", "Trompowsky: 2...Ne4 3.Bf4 d5 4.f3 Nf6", "1.d4 Nf6 2.Bg5 Ne4 3.Bf4 d5 4.f3 Nf6 *"),
("A45w", "Trompowsky: 2...Ne4 3.Bf4 d5 4.e3", "1.d4 Nf6 2.Bg5 Ne4 3.Bf4 d5 4.e3 *"),
("A45w", "Trompowsky: 2...Ne4 3.Bf4 d5 4.e3 c5", "1.d4 Nf6 2.Bg5 Ne4 3.Bf4 d5 4.e3 c5 *"),
("A45x", "Trompowsky: 2...Ne4 3.Bf4 c5", "1.d4 Nf6 2.Bg5 Ne4 3.Bf4 c5 *"),
("A45x", "Trompowsky: 2...Ne4 3.Bf4 c5 4.d5", "1.d4 Nf6 2.Bg5 Ne4 3.Bf4 c5 4.d5 *"),
("A45x", "Trompowsky: 2...Ne4 3.Bf4 c5 4.d5 Qb6", "1.d4 Nf6 2.Bg5 Ne4 3.Bf4 c5 4.d5 Qb6 *"),
("A45y", "Trompowsky: 2...Ne4 3.Bf4 c5 4.f3", "1.d4 Nf6 2.Bg5 Ne4 3.Bf4 c5 4.f3 *"),
("A45y", "Trompowsky: 2...Ne4 3.Bf4 c5 4.f3 Qa5+", "1.d4 Nf6 2.Bg5 Ne4 3.Bf4 c5 4.f3 Qa5+ *"),
("A45y", "Trompowsky: 2...Ne4 3.Bf4 c5 4.f3 Qa5+ 5.c3 Nf6 6.d5", "1.d4 Nf6 2.Bg5 Ne4 3.Bf4 c5 4.f3 Qa5+ 5.c3 Nf6 6.d5 *"),
("A45z", "Trompowsky: 2...Ne4 3.Bf4 c5 4.f3 Qa5+ 5.c3 Nf6 6.Nd2", "1.d4 Nf6 2.Bg5 Ne4 3.Bf4 c5 4.f3 Qa5+ 5.c3 Nf6 6.Nd2 *"),
("A46a", "Indian: 2.Nf3", "1.d4 Nf6 2.Nf3 *"),
("A46a", "Indian: Doery Defence", "1.d4 Nf6 2.Nf3 Ne4 *"),
("A46b", "Indian: 2.Nf3 b5", "1.d4 Nf6 2.Nf3 b5 *"),
("A46b", "Indian: 2.Nf3 b5 3.g3", "1.d4 Nf6 2.Nf3 b5 3.g3 *"),
("A46c", "Neo-Benoni", "1.d4 Nf6 2.Nf3 c5 *"),
("A46c", "Neo-Benoni 3.dxc5", "1.d4 Nf6 2.Nf3 c5 3.dxc5 *"),
("A46c", "Neo-Benoni: 3.e3", "1.d4 Nf6 2.Nf3 c5 3.e3 *"),
("A46c", "Neo-Benoni: 3.e3 cxd4", "1.d4 Nf6 2.Nf3 c5 3.e3 cxd4 *"),
("A46d", "Neo-Benoni: 3.c3", "1.d4 Nf6 2.Nf3 c5 3.c3 *"),
("A46d", "Neo-Benoni: 3.c3 cxd4", "1.d4 Nf6 2.Nf3 c5 3.c3 cxd4 *"),
("A46d", "Neo-Benoni: 3.c3 b6", "1.d4 Nf6 2.Nf3 c5 3.c3 b6 *"),
("A46d", "Neo-Benoni: 3.c3 g6", "1.d4 Nf6 2.Nf3 c5 3.c3 g6 *"),
("A46e", "Neo-Benoni: 3.c3 e6", "1.d4 Nf6 2.Nf3 c5 3.c3 e6 *"),
("A46f", "Neo-Benoni: 3.g3", "1.d4 Nf6 2.Nf3 c5 3.g3 *"),
("A46f", "Neo-Benoni: 3.g3 cxd4", "1.d4 Nf6 2.Nf3 c5 3.g3 cxd4 *"),
("A46f", "Neo-Benoni: 3.g3 cxd4 4.Nxd4", "1.d4 Nf6 2.Nf3 c5 3.g3 cxd4 4.Nxd4 *"),
("A46g", "Indian: 2.Nf3 d6", "1.d4 Nf6 2.Nf3 d6 *"),
("A46h", "Indian: 2.Nf3 d6 3.g3", "1.d4 Nf6 2.Nf3 d6 3.g3 *"),
("A46i", "Indian: 2.Nf3 d6 3.Bg5", "1.d4 Nf6 2.Nf3 d6 3.Bg5 *"),
("A46i", "Indian: 2.Nf3 d6 3.Bg5 Nbd7", "1.d4 Nf6 2.Nf3 d6 3.Bg5 Nbd7 *"),
("A46j", "Indian: 2.Nf3 e6", "1.d4 Nf6 2.Nf3 e6 *"),
("A46j", "Indian: 2.Nf3 e6 3.c3", "1.d4 Nf6 2.Nf3 e6 3.c3 *"),
("A46j", "Indian: 2.Nf3 e6 3.c3 b6", "1.d4 Nf6 2.Nf3 e6 3.c3 b6 *"),
("A46j", "Indian: 1.d4 Nf6 2.Nf3 e6 3.c3 b6 4.Bg5", "1.d4 Nf6 2.Nf3 e6 3.c3 b6 4.Bg5 *"),
("A46k", "Indian: 1.d4 Nf6 2.Nf3 e6 3.e3", "1.d4 Nf6 2.Nf3 e6 3.e3 *"),
("A46k", "Indian: 1.d4 Nf6 2.Nf3 e6 3.e3 c5", "1.d4 Nf6 2.Nf3 e6 3.e3 c5 *"),
("A46l", "Indian: 1.d4 Nf6 2.Nf3 e6 3.e3 c5 4.Bd3", "1.d4 Nf6 2.Nf3 e6 3.e3 c5 4.Bd3 *"),
("A46m", "Indian: 1.d4 Nf6 2.Nf3 e6 3.g3", "1.d4 Nf6 2.Nf3 e6 3.g3 *"),
("A46m", "Indian: 1.d4 Nf6 2.Nf3 e6 3.g3 b5", "1.d4 Nf6 2.Nf3 e6 3.g3 b5 *"),
("A46m", "Indian: 1.d4 Nf6 2.Nf3 e6 3.g3 b5 4.Bg2 Bb7", "1.d4 Nf6 2.Nf3 e6 3.g3 b5 4.Bg2 Bb7 *"),
("A46n", "Indian: 1.d4 Nf6 2.Nf3 e6 3.g3 b5 4.Bg2 Bb7 5.O-O", "1.d4 Nf6 2.Nf3 e6 3.g3 b5 4.Bg2 Bb7 5.O-O *"),
("A46o", "Indian: 1.d4 Nf6 2.Nf3 e6 3.g3 c5", "1.d4 Nf6 2.Nf3 e6 3.g3 c5 *"),
("A46o", "Indian: 1.d4 Nf6 2.Nf3 e6 3.g3 c5 4.Bg2", "1.d4 Nf6 2.Nf3 e6 3.g3 c5 4.Bg2 *"),
("A46o", "Indian: 1.d4 Nf6 2.Nf3 e6 3.g3 c5 4.Bg2 cxd4", "1.d4 Nf6 2.Nf3 e6 3.g3 c5 4.Bg2 cxd4 *"),
("A46o", "Indian: 1.d4 Nf6 2.Nf3 e6 3.g3 c5 4.Bg2 Nc6", "1.d4 Nf6 2.Nf3 e6 3.g3 c5 4.Bg2 Nc6 *"),
("A46o", "Indian: 1.d4 Nf6 2.Nf3 e6 3.g3 c5 4.Bg2 Qc7", "1.d4 Nf6 2.Nf3 e6 3.g3 c5 4.Bg2 Qc7 *"),
("A46o", "Indian: 1.d4 Nf6 2.Nf3 e6 3.g3 c5 4.Bg2 Qa5+", "1.d4 Nf6 2.Nf3 e6 3.g3 c5 4.Bg2 Qa5+ *"),
("A46p", "Indian: London System", "1.d4 Nf6 2.Nf3 e6 3.Bf4 *"),
("A46p", "Indian: London, 3...c5", "1.d4 Nf6 2.Nf3 e6 3.Bf4 c5 *"),
("A46p", "Indian: London, 3...c5 4.e3", "1.d4 Nf6 2.Nf3 e6 3.Bf4 c5 4.e3 *"),
("A46p", "Indian: London, 3...c5 4.e3 Qb6", "1.d4 Nf6 2.Nf3 e6 3.Bf4 c5 4.e3 Qb6 *"),
("A46q", "Indian: London, 3...c5 4.c3", "1.d4 Nf6 2.Nf3 e6 3.Bf4 c5 4.c3 *"),
("A46r", "Torre Attack", "1.d4 Nf6 2.Nf3 e6 3.Bg5 *"),
("A46r", "Torre Attack: 3...b6", "1.d4 Nf6 2.Nf3 e6 3.Bg5 b6 *"),
("A46r", "Torre Attack: 3...Be7", "1.d4 Nf6 2.Nf3 e6 3.Bg5 Be7 *"),
("A46s", "Torre Attack: 3...c5", "1.d4 Nf6 2.Nf3 e6 3.Bg5 c5 *"),
("A46s", "Torre Attack: Wagner Gambit", "1.d4 Nf6 2.Nf3 e6 3.Bg5 c5 4.e4 *"),
("A46t", "Torre Attack: 3...c5 4.c3", "1.d4 Nf6 2.Nf3 e6 3.Bg5 c5 4.c3 *"),
("A46t", "Torre Attack: 3...c5 4.c3 Qb6", "1.d4 Nf6 2.Nf3 e6 3.Bg5 c5 4.c3 Qb6 *"),
("A46t", "Torre Attack: 3...c5 4.c3 h6", "1.d4 Nf6 2.Nf3 e6 3.Bg5 c5 4.c3 h6 *"),
("A46t", "Torre Attack: 3...c5 4.c3 h6 5.Bh4", "1.d4 Nf6 2.Nf3 e6 3.Bg5 c5 4.c3 h6 5.Bh4 *"),
("A46u", "Torre Attack: 3...c5 4.e3", "1.d4 Nf6 2.Nf3 e6 3.Bg5 c5 4.e3 *"),
("A46u", "Torre Attack: 3...c5 4.e3 cxd4", "1.d4 Nf6 2.Nf3 e6 3.Bg5 c5 4.e3 cxd4 *"),
("A46u", "Torre Attack: 3...c5 4.e3 Qb6", "1.d4 Nf6 2.Nf3 e6 3.Bg5 c5 4.e3 Qb6 *"),
("A46v", "Torre Attack: 3...c5 4.e3 Be7", "1.d4 Nf6 2.Nf3 e6 3.Bg5 c5 4.e3 Be7 *"),
("A46w", "Torre Attack: 3...c5 4.e3 h6", "1.d4 Nf6 2.Nf3 e6 3.Bg5 c5 4.e3 h6 *"),
("A46w", "Torre Attack: 3...c5 4.e3 h6 5.Bh4", "1.d4 Nf6 2.Nf3 e6 3.Bg5 c5 4.e3 h6 5.Bh4 *"),
("A46x", "Torre Attack: 3...h6", "1.d4 Nf6 2.Nf3 e6 3.Bg5 h6 *"),
("A46x", "Torre Attack: 3...h6 4.Bh4", "1.d4 Nf6 2.Nf3 e6 3.Bg5 h6 4.Bh4 *"),
("A46x", "Torre Attack: 3...h6 4.Bh4 g5", "1.d4 Nf6 2.Nf3 e6 3.Bg5 h6 4.Bh4 g5 *"),
("A46y", "Torre Attack: 3...h6 4.Bxf6", "1.d4 Nf6 2.Nf3 e6 3.Bg5 h6 4.Bxf6 *"),
("A46y", "Torre Attack: 3...h6 4.Bxf6 Qxf6 5.e4", "1.d4 Nf6 2.Nf3 e6 3.Bg5 h6 4.Bxf6 Qxf6 5.e4 *"),
("A46y", "Torre Attack: 3...h6 4.Bxf6 Qxf6 5.e4 b6", "1.d4 Nf6 2.Nf3 e6 3.Bg5 h6 4.Bxf6 Qxf6 5.e4 b6 *"),
("A46y", "Torre Attack: 3...h6 4.Bxf6 Qxf6 5.e4 c5", "1.d4 Nf6 2.Nf3 e6 3.Bg5 h6 4.Bxf6 Qxf6 5.e4 c5 *"),
("A46z", "Torre Attack: 3...h6 4.Bxf6 Qxf6 5.e4 d6", "1.d4 Nf6 2.Nf3 e6 3.Bg5 h6 4.Bxf6 Qxf6 5.e4 d6 *"),
("A46z", "Torre Attack: 3...h6 4.Bxf6 Qxf6 5.e4 d6 6.Nc3", "1.d4 Nf6 2.Nf3 e6 3.Bg5 h6 4.Bxf6 Qxf6 5.e4 d6 6.Nc3 *"),
("A46z", "Torre Attack: 3...h6 4.Bxf6 Qxf6 5.e4 d6 6.Nc3 g6", "1.d4 Nf6 2.Nf3 e6 3.Bg5 h6 4.Bxf6 Qxf6 5.e4 d6 6.Nc3 g6 *"),
("A46z", "Torre Attack: 3...h6 4.Bxf6 Qxf6 5.e4 d6 6.Nc3 Nd7", "1.d4 Nf6 2.Nf3 e6 3.Bg5 h6 4.Bxf6 Qxf6 5.e4 d6 6.Nc3 Nd7 *"),
("A46z", "Torre Attack: 3...h6 4.Bxf6 Qxf6 5.e4 d6 6.Nc3 Nd7 7.Qd2", "1.d4 Nf6 2.Nf3 e6 3.Bg5 h6 4.Bxf6 Qxf6 5.e4 d6 6.Nc3 Nd7 7.Qd2 *"),
("A47", "Neo-Queen's Indian", "1.d4 Nf6 2.Nf3 b6 *"),
("A47", "Neo-Queen's Indian", "1.d4 Nf6 2.Nf3 b6 3.Bf4 *"),
("A47", "Neo-Queen's Indian", "1.d4 Nf6 2.Nf3 b6 3.Bf4 e6 *"),
("A47", "Neo-Queen's Indian", "1.d4 Nf6 2.Nf3 b6 3.Bf4 e6 4.e3 *"),
("A47", "Neo-Queen's Indian", "1.d4 Nf6 2.Nf3 b6 3.Bf4 e6 4.e3 c5 *"),
("A47", "Neo-Queen's Indian", "1.d4 Nf6 2.Nf3 b6 3.Bf4 Bb7 *"),
("A47", "Neo-Queen's Indian", "1.d4 Nf6 2.Nf3 b6 3.Bf4 Bb7 4.e3 *"),
("A47", "Neo-Queen's Indian", "1.d4 Nf6 2.Nf3 b6 3.Bf4 Bb7 4.e3 e6 *"),
("A47", "Neo-Queen's Indian", "1.d4 Nf6 2.Nf3 b6 3.Bg5 *"),
("A47", "Neo-Queen's Indian", "1.d4 Nf6 2.Nf3 b6 3.Bg5 Bb7 *"),
("A47", "Neo-Queen's Indian", "1.d4 Nf6 2.Nf3 b6 3.e3 *"),
("A47", "Neo-Queen's Indian", "1.d4 Nf6 2.Nf3 b6 3.e3 e6 *"),
("A47", "Neo-Queen's Indian", "1.d4 Nf6 2.Nf3 b6 3.e3 e6 4.Bd3 *"),
("A47", "Neo-Queen's Indian", "1.d4 Nf6 2.Nf3 b6 3.e3 e6 4.Bd3 c5 *"),
("A47", "Neo-Queen's Indian", "1.d4 Nf6 2.Nf3 b6 3.e3 Bb7 *"),
("A47", "Neo-Queen's Indian", "1.d4 Nf6 2.Nf3 b6 3.e3 Bb7 4.Bd3 *"),
("A47", "Neo-Queen's Indian", "1.d4 Nf6 2.Nf3 b6 3.e3 Bb7 4.Bd3 e6 *"),
("A47", "Neo-Queen's Indian", "1.d4 Nf6 2.Nf3 b6 3.e3 Bb7 4.Bd3 e6 5.O-O *"),
("A47", "Neo-Queen's Indian", "1.d4 Nf6 2.Nf3 b6 3.e3 Bb7 4.Bd3 e6 5.O-O c5 *"),
("A47", "Neo-Queen's Indian", "1.d4 Nf6 2.Nf3 b6 3.g3 *"),
("A47", "Neo-Queen's Indian", "1.d4 Nf6 2.Nf3 b6 3.g3 e6 *"),
("A47", "Neo-Queen's Indian", "1.d4 Nf6 2.Nf3 b6 3.g3 Bb7 4.Bg2 *"),
("A47", "Neo-Queen's Indian", "1.d4 Nf6 2.Nf3 b6 3.g3 Bb7 4.Bg2 e6 *"),
("A47", "Neo-Queen's Indian: Marienbad System", "1.d4 Nf6 2.Nf3 b6 3.g3 Bb7 4.Bg2 c5 *"),
("A47", "Neo-Queen's Indian: Marienbad System, Berg Variation", "1.d4 Nf6 2.Nf3 b6 3.g3 Bb7 4.Bg2 c5 5.c4 cxd4 6.Qxd4 *"),
("A48a", "Neo-King's Indian", "1.d4 Nf6 2.Nf3 g6 *"),
("A48a", "Neo-King's Indian", "1.d4 Nf6 2.Nf3 g6 3.Nbd2 *"),
("A48b", "Neo-King's Indian", "1.d4 Nf6 2.Nf3 g6 3.c3 *"),
("A48b", "Neo-King's Indian", "1.d4 Nf6 2.Nf3 g6 3.c3 Bg7 *"),
("A48c", "Neo-King's Indian", "1.d4 Nf6 2.Nf3 g6 3.e3 *"),
("A48c", "Neo-King's Indian", "1.d4 Nf6 2.Nf3 g6 3.e3 Bg7 *"),
("A48d", "Neo-King's Indian", "1.d4 Nf6 2.Nf3 g6 3.e3 c5 *"),
("A48e", "Neo-King's Indian", "1.d4 Nf6 2.Nf3 g6 3.Nc3 *"),
("A48f", "Neo-King's Indian: London System", "1.d4 Nf6 2.Nf3 g6 3.Bf4 *"),
("A48g", "Neo-King's Indian: London System", "1.d4 Nf6 2.Nf3 g6 3.Bf4 Bg7 *"),
("A48h", "Neo-King's Indian: London System", "1.d4 Nf6 2.Nf3 g6 3.Bf4 Bg7 4.Nbd2 *"),
("A48i", "Neo-King's Indian: London System", "1.d4 Nf6 2.Nf3 g6 3.Bf4 Bg7 4.c3 *"),
("A48j", "Neo-King's Indian: London System", "1.d4 Nf6 2.Nf3 g6 3.Bf4 Bg7 4.e3 *"),
("A48j", "Neo-King's Indian: London System", "1.d4 Nf6 2.Nf3 g6 3.Bf4 Bg7 4.e3 O-O *"),
("A48k", "Neo-King's Indian: London System", "1.d4 Nf6 2.Nf3 g6 3.Bf4 Bg7 4.e3 O-O 5.Be2 *"),
("A48k", "Neo-King's Indian: London System", "1.d4 Nf6 2.Nf3 g6 3.Bf4 Bg7 4.e3 O-O 5.Be2 d6 *"),
("A48l", "Neo-King's Indian: London System", "1.d4 Nf6 2.Nf3 g6 3.Bf4 Bg7 4.e3 O-O 5.Be2 d6 6.h3 *"),
("A48m", "Neo-King's Indian: Torre Attack", "1.d4 Nf6 2.Nf3 g6 3.Bg5 *"),
("A48n", "Neo-King's Indian: Torre Attack", "1.d4 Nf6 2.Nf3 g6 3.Bg5 Ne4 *"),
("A48o", "Neo-King's Indian: Torre Attack", "1.d4 Nf6 2.Nf3 g6 3.Bg5 Bg7 *"),
("A48o", "Neo-King's Indian: Torre Attack", "1.d4 Nf6 2.Nf3 g6 3.Bg5 Bg7 4.c3 *"),
("A48o", "Neo-King's Indian: Torre Attack", "1.d4 Nf6 2.Nf3 g6 3.Bg5 Bg7 4.e3 *"),
("A48p", "Neo-King's Indian: Torre Attack", "1.d4 Nf6 2.Nf3 g6 3.Bg5 Bg7 4.Nc3 *"),
("A48q", "Neo-King's Indian: Torre Attack", "1.d4 Nf6 2.Nf3 g6 3.Bg5 Bg7 4.Nbd2 *"),
("A48r", "Neo-King's Indian: Torre Attack", "1.d4 Nf6 2.Nf3 g6 3.Bg5 Bg7 4.Nbd2 O-O *"),
("A48s", "Neo-King's Indian: Torre Attack", "1.d4 Nf6 2.Nf3 g6 3.Bg5 Bg7 4.Nbd2 O-O 5.e4 *"),
("A48t", "Neo-King's Indian: Torre Attack", "1.d4 Nf6 2.Nf3 g6 3.Bg5 Bg7 4.Nbd2 O-O 5.e4 d6 *"),
("A48u", "Neo-King's Indian: Torre Attack", "1.d4 Nf6 2.Nf3 g6 3.Bg5 Bg7 4.Nbd2 O-O 5.e4 d5 *"),
("A48v", "Neo-King's Indian: Torre Attack", "1.d4 Nf6 2.Nf3 g6 3.Bg5 Bg7 4.Nbd2 O-O 5.c3 *"),
("A48v", "Neo-King's Indian: Torre Attack", "1.d4 Nf6 2.Nf3 g6 3.Bg5 Bg7 4.Nbd2 O-O 5.c3 d6 *"),
("A48w", "Neo-King's Indian: Torre Attack", "1.d4 Nf6 2.Nf3 g6 3.Bg5 Bg7 4.Nbd2 O-O 5.c3 d6 6.e4 *"),
("A48x", "Neo-King's Indian: Torre Attack", "1.d4 Nf6 2.Nf3 g6 3.Bg5 Bg7 4.Nbd2 O-O 5.c3 d6 6.e4 c5 *"),
("A49", "Neo-King's Indian: Fianchetto System", "1.d4 Nf6 2.Nf3 g6 3.g3 *"),
("A49", "Neo-King's Indian: Fianchetto System", "1.d4 Nf6 2.Nf3 g6 3.g3 Bg7 *"),
("A49", "Neo-King's Indian: Fianchetto System", "1.d4 Nf6 2.Nf3 g6 3.g3 Bg7 4.Bg2 *"),
("A49", "Neo-King's Indian: Fianchetto System", "1.d4 Nf6 2.Nf3 g6 3.g3 Bg7 4.Bg2 O-O *"),
("A49", "Neo-King's Indian: Double Fianchetto System", "1.d4 Nf6 2.Nf3 g6 3.g3 Bg7 4.Bg2 O-O 5.b3 d6 6.Bb2 *"),
("A49", "Neo-King's Indian: Fianchetto System", "1.d4 Nf6 2.Nf3 g6 3.g3 Bg7 4.Bg2 O-O 5.O-O *"),
("A49", "Neo-King's Indian: Fianchetto System", "1.d4 Nf6 2.Nf3 g6 3.g3 Bg7 4.Bg2 O-O 5.O-O d6 *"),
("A50a", "Indian: 2.c4", "1.d4 Nf6 2.c4 *"),
("A50b", "Indian: Mexican Defence (Two Knights Tango)", "1.d4 Nf6 2.c4 Nc6 *"),
("A50c", "Indian: Mexican Defence, 3.Nc3", "1.d4 Nf6 2.c4 Nc6 3.Nc3 *"),
("A50d", "Indian: Mexican Defence, 3.Nf3", "1.d4 Nf6 2.c4 Nc6 3.Nf3 *"),
("A50d", "Indian: Mexican Defence, 3.Nf3 d6", "1.d4 Nf6 2.c4 Nc6 3.Nf3 d6 *"),
("A50e", "Indian: Mexican Defence, 3.Nf3 e6", "1.d4 Nf6 2.c4 Nc6 3.Nf3 e6 *"),
("A50f", "Indian: Mexican Defence, 3.Nf3 e6 4.a3", "1.d4 Nf6 2.c4 Nc6 3.Nf3 e6 4.a3 *"),
("A50g", "Indian: Mexican Defence, 3.Nf3 e6 4.Nc3", "1.d4 Nf6 2.c4 Nc6 3.Nf3 e6 4.Nc3 *"),
("A50h", "Indian: Slav-Indian", "1.d4 Nf6 2.c4 c6 *"),
("A50i", "Indian: Slav-Indian, 3.Nf3", "1.d4 Nf6 2.c4 c6 3.Nf3 *"),
("A50j", "Indian: Slav-Indian, 3.Nc3", "1.d4 Nf6 2.c4 c6 3.Nc3 *"),
("A50k", "Indian: 2.c4 a6", "1.d4 Nf6 2.c4 a6 *"),
("A50l", "Indian: Queen's Indian Accelerated", "1.d4 Nf6 2.c4 b6 *"),
("A50m", "Indian: Queen's Indian Accelerated", "1.d4 Nf6 2.c4 b6 3.Nc3 Bb7 *"),
("A50n", "Indian: Queen's Indian Accelerated", "1.d4 Nf6 2.c4 b6 3.Nc3 Bb7 4.Qc2 *"),
("A51a", "Budapest", "1.d4 Nf6 2.c4 e5 *"),
("A51b", "Budapest: 3.e3", "1.d4 Nf6 2.c4 e5 3.e3 *"),
("A51c", "Budapest: 3.d5", "1.d4 Nf6 2.c4 e5 3.d5 *"),
("A51d", "Budapest: 3.dxe5", "1.d4 Nf6 2.c4 e5 3.dxe5 *"),
("A51d", "Budapest: Fajarowicz", "1.d4 Nf6 2.c4 e5 3.dxe5 Ne4 *"),
("A51e", "Budapest: Fajarowicz, Steiner Variation", "1.d4 Nf6 2.c4 e5 3.dxe5 Ne4 4.Qc2 *"),
("A51f", "Budapest: Fajarowicz, 4.Nd2", "1.d4 Nf6 2.c4 e5 3.dxe5 Ne4 4.Nd2 *"),
("A51g", "Budapest: Fajarowicz, 4.a3", "1.d4 Nf6 2.c4 e5 3.dxe5 Ne4 4.a3 *"),
("A51h", "Budapest: Fajarowicz, 4.Nf3", "1.d4 Nf6 2.c4 e5 3.dxe5 Ne4 4.Nf3 *"),
("A51i", "Budapest: Fajarowicz, 4.Nf3 Bb4+", "1.d4 Nf6 2.c4 e5 3.dxe5 Ne4 4.Nf3 Bb4+ *"),
("A51j", "Budapest: Fajarowicz, 4.Nf3 Nc6", "1.d4 Nf6 2.c4 e5 3.dxe5 Ne4 4.Nf3 Nc6 *"),
("A51k", "Budapest: Fajarowicz, 4.Nf3 Nc6 5.a3", "1.d4 Nf6 2.c4 e5 3.dxe5 Ne4 4.Nf3 Nc6 5.a3 *"),
("A52a", "Budapest: 3...Ng4", "1.d4 Nf6 2.c4 e5 3.dxe5 Ng4 *"),
("A52b", "Budapest: 3...Ng4 4.e3", "1.d4 Nf6 2.c4 e5 3.dxe5 Ng4 4.e3 *"),
("A52c", "Budapest: Alekhine Variation", "1.d4 Nf6 2.c4 e5 3.dxe5 Ng4 4.e4 *"),
("A52c", "Budapest: Alekhine, Balogh Gambit", "1.d4 Nf6 2.c4 e5 3.dxe5 Ng4 4.e4 d6 *"),
("A52d", "Budapest: Alekhine, Abonyi Variation", "1.d4 Nf6 2.c4 e5 3.dxe5 Ng4 4.e4 Nxe5 5.f4 Nec6 *"),
("A52e", "Budapest: Adler Variation", "1.d4 Nf6 2.c4 e5 3.dxe5 Ng4 4.Nf3 *"),
("A52f", "Budapest: Adler, 4...Bc5", "1.d4 Nf6 2.c4 e5 3.dxe5 Ng4 4.Nf3 Bc5 *"),
("A52g", "Budapest: Adler, 4...Bc5 5.e3 Nc6 6.Be2", "1.d4 Nf6 2.c4 e5 3.dxe5 Ng4 4.Nf3 Bc5 5.e3 Nc6 6.Be2 *"),
("A52h", "Budapest: Rubinstein Variation", "1.d4 Nf6 2.c4 e5 3.dxe5 Ng4 4.Bf4 *"),
("A52i", "Budapest: Rubinstein, 4...Bb4+", "1.d4 Nf6 2.c4 e5 3.dxe5 Ng4 4.Bf4 Bb4+ *"),
("A52j", "Budapest: Rubinstein, 4...Nc6", "1.d4 Nf6 2.c4 e5 3.dxe5 Ng4 4.Bf4 Nc6 *"),
("A52k", "Budapest: Rubinstein, Main Line", "1.d4 Nf6 2.c4 e5 3.dxe5 Ng4 4.Bf4 Nc6 5.Nf3 Bb4+ *"),
("A52l", "Budapest: Rubinstein, Main Line, 6.Nbd2", "1.d4 Nf6 2.c4 e5 3.dxe5 Ng4 4.Bf4 Nc6 5.Nf3 Bb4+ 6.Nbd2 *"),
("A52m", "Budapest: Rubinstein, Main Line, 6.Nbd2 Qe7 7.e3", "1.d4 Nf6 2.c4 e5 3.dxe5 Ng4 4.Bf4 Nc6 5.Nf3 Bb4+ 6.Nbd2 Qe7 7.e3 *"),
("A53", "Old Indian", "1.d4 Nf6 2.c4 d6 *"),
("A53", "Old Indian: 3.g3", "1.d4 Nf6 2.c4 d6 3.g3 *"),
("A53", "Old Indian: 3.Nf3", "1.d4 Nf6 2.c4 d6 3.Nf3 *"),
("A53", "Old Indian: 3.Nf3 c6", "1.d4 Nf6 2.c4 d6 3.Nf3 c6 *"),
("A53", "Old Indian: 3.Nf3 Nbd7", "1.d4 Nf6 2.c4 d6 3.Nf3 Nbd7 *"),
("A53", "Old Indian: 3.Nf3 Bf5", "1.d4 Nf6 2.c4 d6 3.Nf3 Bf5 *"),
("A53", "Old Indian: 3.Nf3 Bg4", "1.d4 Nf6 2.c4 d6 3.Nf3 Bg4 *"),
("A53", "Old Indian: 3.Nf3 Bg4 4.Qb3", "1.d4 Nf6 2.c4 d6 3.Nf3 Bg4 4.Qb3 *"),
("A53", "Old Indian: 3.Nf3 Bg4 4.Nc3", "1.d4 Nf6 2.c4 d6 3.Nf3 Bg4 4.Nc3 *"),
("A53", "Old Indian: 3.Nc3", "1.d4 Nf6 2.c4 d6 3.Nc3 *"),
("A53", "Old Indian: 3.Nc3 c6", "1.d4 Nf6 2.c4 d6 3.Nc3 c6 *"),
("A53", "Old Indian: 3.Nc3 Nbd7", "1.d4 Nf6 2.c4 d6 3.Nc3 Nbd7 *"),
("A53", "Old Indian: 3.Nc3 Nbd7 4.Nf3", "1.d4 Nf6 2.c4 d6 3.Nc3 Nbd7 4.Nf3 *"),
("A53", "Old Indian: 3.Nc3 Nbd7 4.e4", "1.d4 Nf6 2.c4 d6 3.Nc3 Nbd7 4.e4 *"),
("A53", "Old Indian: 3.Nc3 Nbd7 4.e4 e5", "1.d4 Nf6 2.c4 d6 3.Nc3 Nbd7 4.e4 e5 *"),
("A53", "Old Indian: 3.Nc3 Nbd7 4.e4 e5 5.d5", "1.d4 Nf6 2.c4 d6 3.Nc3 Nbd7 4.e4 e5 5.d5 *"),
("A53", "Old Indian: 3.Nc3 Nbd7 4.e4 e5 5.Nge2", "1.d4 Nf6 2.c4 d6 3.Nc3 Nbd7 4.e4 e5 5.Nge2 *"),
("A53", "Old Indian: Janowski", "1.d4 Nf6 2.c4 d6 3.Nc3 Bf5 *"),
("A53", "Old Indian: Ukrainian", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 *"),
("A53", "Old Indian: Ukranian, 4.e4", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.e4 *"),
("A53", "Old Indian: Ukranian, 4.e3", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.e3 *"),
("A53", "Old Indian: Dus-Khotimirsky", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.e3 Nbd7 5.Bd3 *"),
("A53", "Old Indian: Ukrainian, 4.d5", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.d5 *"),
("A53", "Old Indian: Ukrainian, 4.d5 Be7", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.d5 Be7 *"),
("A53", "Old Indian: Ukrainian, 4.dxe5", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.dxe5 *"),
("A53", "Old Indian: Ukrainian, Queenswap", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.dxe5 dxe5 5.Qxd8+ Kxd8 *"),
("A53", "Old Indian: Ukrainian, Queenswap, 6.Nf3", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.dxe5 dxe5 5.Qxd8+ Kxd8 6.Nf3 *"),
("A53", "Old Indian: Ukrainian, Queenswap, 6.Nf3 Nfd7", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.dxe5 dxe5 5.Qxd8+ Kxd8 6.Nf3 Nfd7 *"),
("A54", "Old Indian: 4.Nf3", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 *"),
("A54", "Old Indian: 4.Nf3 exd4", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 exd4 *"),
("A54", "Old Indian: 4.Nf3 Nc6", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nc6 *"),
("A54", "Old Indian: 4.Nf3 e4", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 e4 *"),
("A54", "Old Indian: 4.Nf3 e4 5.Ng5", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 e4 5.Ng5 *"),
("A54", "Old Indian: 4.Nf3 Nbd7", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 *"),
("A54", "Old Indian: 5.e3", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e3 *"),
("A54", "Old Indian: 5.g3", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.g3 *"),
("A54", "Old Indian: 5.g3 c6", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.g3 c6 *"),
("A54", "Old Indian: 5.g3 c6", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.g3 c6 6.Bg2 *"),
("A54", "Old Indian: 5.g3 g6", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.g3 g6 *"),
("A54", "Old Indian: 5.g3 g6", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.g3 g6 6.Bg2 *"),
("A54", "Old Indian: 5.g3 g6", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.g3 g6 6.Bg2 Bg7 *"),
("A54", "Old Indian: 5.g3 Be7", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.g3 Be7 *"),
("A54", "Old Indian: 5.g3 Be7", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.g3 Be7 6.Bg2 *"),
("A54", "Old Indian: 5.g3 Be7", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.g3 Be7 6.Bg2 O-O *"),
("A54", "Old Indian: 5.g3 Be7", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.g3 Be7 6.Bg2 O-O 7.O-O *"),
("A54", "Old Indian: 5.g3 Be7, Main Line", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.g3 Be7 6.Bg2 O-O 7.O-O c6 *"),
("A54", "Old Indian: 5.g3 Be7, Main Line, 8.Qc2", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.g3 Be7 6.Bg2 O-O 7.O-O c6 8.Qc2 *"),
("A54", "Old Indian: 5.Bg5", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.Bg5 *"),
("A54", "Old Indian: 5.Bg5 c6", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.Bg5 c6 *"),
("A54", "Old Indian: 5.Bg5 Be7", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.Bg5 Be7 *"),
("A54", "Old Indian: 5.Bg5 Be7 6.e3", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.Bg5 Be7 6.e3 *"),
("A54", "Old Indian: 5.Bg5 Be7 6.e3 c6", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.Bg5 Be7 6.e3 c6 *"),
("A54", "Old Indian: 5.Bg5 Be7 6.e3 O-O", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.Bg5 Be7 6.e3 O-O *"),
("A54", "Old Indian: 5.Bg5 Be7 6.e3 O-O 7.Qc2", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.Bg5 Be7 6.e3 O-O 7.Qc2 *"),
("A54", "Old Indian: 5.Bg5 Be7 6.e3 O-O 7.Qc2 c6", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.Bg5 Be7 6.e3 O-O 7.Qc2 c6 *"),
("A55a", "Old Indian: 5.e4", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 *"),
("A55b", "Old Indian: 5.e4 g6", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 g6 *"),
("A55c", "Old Indian: 5.e4 c6", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 c6 *"),
("A55c", "Old Indian: 5.e4 c6 6.Be2", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 c6 6.Be2 *"),
("A55d", "Old Indian: 5.e4 Be7", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 *"),
("A55e", "Old Indian: 5.e4 Be7 6.g3", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.g3 *"),
("A55e", "Old Indian: 5.e4 Be7 6.g3 O-O", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.g3 O-O *"),
("A55e", "Old Indian: 5.e4 Be7 6.g3 c6", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.g3 c6 *"),
("A55e", "Old Indian: 5.e4 Be7 6.g3 c6 7.Bg2", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.g3 c6 7.Bg2 *"),
("A55e", "Old Indian: 5.e4 Be7 6.g3 c6 7.Bg2 O-O", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.g3 c6 7.Bg2 O-O *"),
("A55e", "Old Indian: 5.e4 Be7 6.g3 c6 7.Bg2 O-O 8.O-O", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.g3 c6 7.Bg2 O-O 8.O-O *"),
("A55e", "Old Indian: 5.e4 Be7 6.g3 c6 7.Bg2 O-O 8.O-O Qc7", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.g3 c6 7.Bg2 O-O 8.O-O Qc7 *"),
("A55f", "Old Indian: 5.e4 Be7 6.g3 c6 7.Bg2 O-O 8.O-O Re8", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.g3 c6 7.Bg2 O-O 8.O-O Re8 *"),
("A55g", "Old Indian: 5.e4 Be7 6.g3 c6 7.Bg2 O-O 8.O-O a6", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.g3 c6 7.Bg2 O-O 8.O-O a6 *"),
("A55h", "Old Indian: 5.e4 Be7 6.g3 c6 7.Bg2 O-O 8.O-O a6 9.a4", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.g3 c6 7.Bg2 O-O 8.O-O a6 9.a4 *"),
("A55i", "Old Indian: 5.e4 Be7 6.Be2", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.Be2 *"),
("A55i", "Old Indian: 5.e4 Be7 6.Be2 O-O", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.Be2 O-O *"),
("A55j", "Old Indian: 5.e4 Be7 6.Be2 c6", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.Be2 c6 *"),
("A55j", "Old Indian: 5.e4 Be7 6.Be2 c6 7.O-O", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.Be2 c6 7.O-O *"),
("A55j", "Old Indian: 5.e4 Be7 6.Be2 c6 7.O-O a6", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.Be2 c6 7.O-O a6 *"),
("A55k", "Old Indian: Main Line", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.Be2 c6 7.O-O O-O *"),
("A55l", "Old Indian: Main Line, 8.h3", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.Be2 c6 7.O-O O-O 8.h3 *"),
("A55m", "Old Indian: Main Line, 8.Be3", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.Be2 c6 7.O-O O-O 8.Be3 *"),
("A55n", "Old Indian: Main Line, 8.Be3 a6", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.Be2 c6 7.O-O O-O 8.Be3 a6 *"),
("A55o", "Old Indian: Main Line, 8.Qc2", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.Be2 c6 7.O-O O-O 8.Qc2 *"),
("A55o", "Old Indian: Main Line, 8.Qc2 Re8", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.Be2 c6 7.O-O O-O 8.Qc2 Re8 *"),
("A55o", "Old Indian: Main Line, 8.Qc2 Qc7", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.Be2 c6 7.O-O O-O 8.Qc2 Qc7 *"),
("A55p", "Old Indian: Main Line, 8.Qc2 a6", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.Be2 c6 7.O-O O-O 8.Qc2 a6 *"),
("A55q", "Old Indian: Main Line, 8.Re1", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.Be2 c6 7.O-O O-O 8.Re1 *"),
("A55r", "Old Indian: Main Line, 8.Re1 Re8", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.Be2 c6 7.O-O O-O 8.Re1 Re8 *"),
("A55s", "Old Indian: Main Line, 8.Re1 a6", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.Be2 c6 7.O-O O-O 8.Re1 a6 *"),
("A55t", "Old Indian: Main Line, 8.Re1 a6 9.Bf1", "1.d4 Nf6 2.c4 d6 3.Nc3 e5 4.Nf3 Nbd7 5.e4 Be7 6.Be2 c6 7.O-O O-O 8.Re1 a6 9.Bf1 *"),
("A56a", "Benoni: 2...c5", "1.d4 Nf6 2.c4 c5 *"),
("A56b", "Benoni: 2...c5 3.e3", "1.d4 Nf6 2.c4 c5 3.e3 *"),
("A56b", "Benoni: 2...c5 3.e3 e6", "1.d4 Nf6 2.c4 c5 3.e3 e6 *"),
("A56c", "Benoni: 2...c5 3.e3 g6", "1.d4 Nf6 2.c4 c5 3.e3 g6 *"),
("A56d", "Benoni: 2...c5 3.e3 g6 4.Nc3", "1.d4 Nf6 2.c4 c5 3.e3 g6 4.Nc3 *"),
("A56e", "Benoni: 3.dxc5", "1.d4 Nf6 2.c4 c5 3.dxc5 *"),
("A56f", "Benoni: 3.d5", "1.d4 Nf6 2.c4 c5 3.d5 *"),
("A56g", "Benoni: 3.d5 a6", "1.d4 Nf6 2.c4 c5 3.d5 a6 *"),
("A56h", "Benoni: 3.d5 g6", "1.d4 Nf6 2.c4 c5 3.d5 g6 *"),
("A56h", "Benoni: 3.d5 g6", "1.d4 Nf6 2.c4 c5 3.d5 g6 4.Nc3 *"),
("A56i", "Benoni: 3.d5 d6", "1.d4 Nf6 2.c4 c5 3.d5 d6 *"),
("A56j", "Benoni: 3.d5 d6 4.Nc3 g6", "1.d4 Nf6 2.c4 c5 3.d5 d6 4.Nc3 g6 *"),
("A56j", "Benoni: Bronstein's Gambit", "1.d4 Nf6 2.c4 c5 3.d5 g6 4.Nc3 d6 5.e4 b5 *"),
("A56k", "Benoni: 3.d5 d6 4.Nc3 g6 5.e4 Bg7", "1.d4 Nf6 2.c4 c5 3.d5 d6 4.Nc3 g6 5.e4 Bg7 *"),
("A56l", "Benoni: 3.d5 d6 4.Nc3 g6 5.e4 Bg7 6.f3", "1.d4 Nf6 2.c4 c5 3.d5 d6 4.Nc3 g6 5.e4 Bg7 6.f3 *"),
("A56m", "Benoni: 3.d5 d6 4.Nc3 g6 5.e4 Bg7 6.Bd3", "1.d4 Nf6 2.c4 c5 3.d5 d6 4.Nc3 g6 5.e4 Bg7 6.Bd3 *"),
("A56n", "Benoni: Vulture", "1.d4 Nf6 2.c4 c5 3.d5 Ne4 *"),
("A56o", "Benoni: Czech", "1.d4 Nf6 2.c4 c5 3.d5 e5 *"),
("A56p", "Benoni: Czech, 4.Nc3 d6", "1.d4 Nf6 2.c4 c5 3.d5 e5 4.Nc3 d6 *"),
("A56q", "Benoni: Czech, 5.e4 g6", "1.d4 Nf6 2.c4 c5 3.d5 e5 4.Nc3 d6 5.e4 g6 *"),
("A56r", "Benoni: Czech, 5.e4 Be7", "1.d4 Nf6 2.c4 c5 3.d5 e5 4.Nc3 d6 5.e4 Be7 *"),
("A56s", "Benoni: Czech, 5.e4 Be7 6.g3", "1.d4 Nf6 2.c4 c5 3.d5 e5 4.Nc3 d6 5.e4 Be7 6.g3 *"),
("A56s", "Benoni: Czech, 5.e4 Be7 6.g3 O-O", "1.d4 Nf6 2.c4 c5 3.d5 e5 4.Nc3 d6 5.e4 Be7 6.g3 O-O *"),
("A56s", "Benoni: Czech, 5.e4 Be7 6.g3 O-O 7.Bg2", "1.d4 Nf6 2.c4 c5 3.d5 e5 4.Nc3 d6 5.e4 Be7 6.g3 O-O 7.Bg2 *"),
("A56t", "Benoni: Czech, 5.e4 Be7 6.g3 O-O 7.Bg2 Ne8", "1.d4 Nf6 2.c4 c5 3.d5 e5 4.Nc3 d6 5.e4 Be7 6.g3 O-O 7.Bg2 Ne8 *"),
("A56u", "Benoni: Czech, 5.e4 Be7 6.Bd3", "1.d4 Nf6 2.c4 c5 3.d5 e5 4.Nc3 d6 5.e4 Be7 6.Bd3 *"),
("A56v", "Benoni: Czech, 5.e4 Be7 6.Nf3", "1.d4 Nf6 2.c4 c5 3.d5 e5 4.Nc3 d6 5.e4 Be7 6.Nf3 *"),
("A56w", "Benoni: Czech, 5.e4 Be7 6.Nf3 O-O", "1.d4 Nf6 2.c4 c5 3.d5 e5 4.Nc3 d6 5.e4 Be7 6.Nf3 O-O *"),
("A56w", "Benoni: Czech, 5.e4 Be7 6.Nf3 O-O 7.h3", "1.d4 Nf6 2.c4 c5 3.d5 e5 4.Nc3 d6 5.e4 Be7 6.Nf3 O-O 7.h3 *"),
("A56x", "Benoni: Czech, 5.e4 Be7 6.Nf3 O-O 7.Be2", "1.d4 Nf6 2.c4 c5 3.d5 e5 4.Nc3 d6 5.e4 Be7 6.Nf3 O-O 7.Be2 *"),
("A57a", "Benko Gambit", "1.d4 Nf6 2.c4 c5 3.d5 b5 *"),
("A57b", "Benko Gambit: 4.a4", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.a4 *"),
("A57c", "Benko Gambit: 4.Nd2", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.Nd2 *"),
("A57d", "Benko Gambit: 4.Nf3", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.Nf3 *"),
("A57d", "Benko Gambit: 4.Nf3 bxc4", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.Nf3 bxc4 *"),
("A57e", "Benko Gambit: 4.Nf3 Bb7", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.Nf3 Bb7 *"),
("A57e", "Benko Gambit: 4.Nf3 Bb7 5.a4", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.Nf3 Bb7 5.a4 *"),
("A57f", "Benko Gambit: 4.Nf3 g6", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.Nf3 g6 *"),
("A57g", "Benko Gambit: 4.Nf3 g6 5.cxb5", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.Nf3 g6 5.cxb5 *"),
("A57g", "Benko Gambit: 4.Nf3 g6 5.cxb5 a6", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.Nf3 g6 5.cxb5 a6 *"),
("A57h", "Benko Gambit: 4.cxb5", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 *"),
("A57h", "Benko Gambit: 4.cxb5 a6", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 *"),
("A57i", "Benko Gambit: 4.cxb5 a6 5.b6", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.b6 *"),
("A57j", "Benko Gambit: 4.cxb5 a6 5.b6 Qxb6", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.b6 Qxb6 *"),
("A57k", "Benko Gambit: 4.cxb5 a6 5.b6 d6", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.b6 d6 *"),
("A57l", "Benko Gambit: 4.cxb5 a6 5.b6 e6", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.b6 e6 *"),
("A57m", "Benko Gambit: 4.cxb5 a6 5.e3", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.e3 *"),
("A57n", "Benko Gambit: 4.cxb5 a6 5.e3 g6", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.e3 g6 *"),
("A57n", "Benko Gambit: 4.cxb5 a6 5.e3 g6 6.Nc3 d6", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.e3 g6 6.Nc3 d6 *"),
("A57n", "Benko Gambit: 4.cxb5 a6 5.e3 g6 6.Nc3 Bg7", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.e3 g6 6.Nc3 Bg7 *"),
("A57o", "Benko Gambit: 4.cxb5 a6 5.f3", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.f3 *"),
("A57p", "Benko Gambit: 4.cxb5 a6 5.f3 e6", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.f3 e6 *"),
("A57q", "Benko Gambit: 4.cxb5 a6 5.f3 axb5", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.f3 axb5 *"),
("A57r", "Benko Gambit: Zaitsev Variation", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.Nc3 *"),
("A57r", "Benko Gambit: Zaitsev, 5...Qa5", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.Nc3 Qa5 *"),
("A57s", "Benko Gambit: Zaitsev, 5...axb5", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.Nc3 axb5 *"),
("A57s", "Benko Gambit: Zaitsev, 5...axb5 6.e4 b4", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.Nc3 axb5 6.e4 b4 *"),
("A57t", "Benko Gambit: Zaitsev, 8.Nf3", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.Nc3 axb5 6.e4 b4 7.Nb5 d6 8.Nf3 *"),
("A57u", "Benko Gambit: Zaitsev, Nescafe Frappe Attack", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.Nc3 axb5 6.e4 b4 7.Nb5 d6 8.Bc4 *"),
("A57v", "Benko Gambit: Zaitsev, 8.Bf4", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.Nc3 axb5 6.e4 b4 7.Nb5 d6 8.Bf4 *"),
("A57w", "Benko Gambit: Zaitsev, 8.Bf4 g5", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.Nc3 axb5 6.e4 b4 7.Nb5 d6 8.Bf4 g5 *"),
("A58a", "Benko Gambit: 5.bxa6", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 *"),
("A58b", "Benko Gambit: 5.bxa6 g6", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 g6 *"),
("A58c", "Benko Gambit: 5.bxa6 Bxa6", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 *"),
("A58c", "Benko Gambit: Accepted, 6.g3", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.g3 *"),
("A58d", "Benko Gambit: Accepted, 6.g3 d6 7.Bg2 g6", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.g3 d6 7.Bg2 g6 *"),
("A58d", "Benko Gambit: Accepted, 6.g3 d6 7.Bg2 g6 8.b3", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.g3 d6 7.Bg2 g6 8.b3 *"),
("A58e", "Benko Gambit: Accepted, 6.Nc3", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 *"),
("A58f", "Benko Gambit: Accepted, 6.Nc3 g6", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 g6 *"),
("A58g", "Benko Gambit: Accepted, 6.Nc3 d6", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 *"),
("A58g", "Benko Gambit: Accepted, 7.f4", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.f4 *"),
("A58g", "Benko Gambit: Accepted, 7.f4", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.f4 g6 *"),
("A58h", "Benko Gambit: Accepted, 7.g3", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.g3 *"),
("A58h", "Benko Gambit: Accepted, 7.g3", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.g3 g6 *"),
("A58i", "Benko Gambit: Accepted, 7.Nf3", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.Nf3 *"),
("A58i", "Benko Gambit: Accepted, 7.Nf3 g6", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.Nf3 g6 *"),
("A58j", "Benko Gambit: Accepted, 7.Nf3 g6 8.Nd2", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.Nf3 g6 8.Nd2 *"),
("A58j", "Benko Gambit: Accepted, 7.Nf3 g6 8.Nd2 Qa5", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.Nf3 g6 8.Nd2 Qa5 *"),
("A58j", "Benko Gambit: Accepted, 7.Nf3 g6 8.Nd2 Bg7", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.Nf3 g6 8.Nd2 Bg7 *"),
("A58k", "Benko Gambit: Fianchetto Variation", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.Nf3 g6 8.g3 *"),
("A58k", "Benko Gambit: Fianchetto Variation", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.Nf3 g6 8.g3 Bg7 *"),
("A58l", "Benko Gambit: Fianchetto, 9.Bh3", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.Nf3 g6 8.g3 Bg7 9.Bh3 *"),
("A58m", "Benko Gambit: Fianchetto, 9.Bg2", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.Nf3 g6 8.g3 Bg7 9.Bg2 *"),
("A58m", "Benko Gambit: Fianchetto, 9...Nbd7", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.Nf3 g6 8.g3 Bg7 9.Bg2 Nbd7 *"),
("A58n", "Benko Gambit: Fianchetto, 9...Nbd7 10.O-O Nb6", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.Nf3 g6 8.g3 Bg7 9.Bg2 Nbd7 10.O-O Nb6 *"),
("A58o", "Benko Gambit: Fianchetto, 9...O-O", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.Nf3 g6 8.g3 Bg7 9.Bg2 O-O *"),
("A58p", "Benko Gambit: Fianchetto, Main Line", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.Nf3 g6 8.g3 Bg7 9.Bg2 O-O 10.O-O Nbd7 *"),
("A58q", "Benko Gambit: Fianchetto, Main Line, 11.Rb1", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.Nf3 g6 8.g3 Bg7 9.Bg2 O-O 10.O-O Nbd7 11.Rb1 *"),
("A58r", "Benko Gambit: Fianchetto, Main Line, 11.Re1", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.Nf3 g6 8.g3 Bg7 9.Bg2 O-O 10.O-O Nbd7 11.Re1 *"),
("A58s", "Benko Gambit: Fianchetto, Main Line, 11.Qc2", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.Nf3 g6 8.g3 Bg7 9.Bg2 O-O 10.O-O Nbd7 11.Qc2 *"),
("A58t", "Benko Gambit: Fianchetto, Main Line, 11.Qc2 Qb6", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.Nf3 g6 8.g3 Bg7 9.Bg2 O-O 10.O-O Nbd7 11.Qc2 Qb6 *"),
("A59a", "Benko Gambit: 7.e4", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.e4 *"),
("A59b", "Benko Gambit: 7.e4 Bxf1 8.Kxf1 g6", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.e4 Bxf1 8.Kxf1 g6 *"),
("A59c", "Benko Gambit: 7.e4 Line, 9.Nge2", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.e4 Bxf1 8.Kxf1 g6 9.Nge2 *"),
("A59d", "Benko Gambit: 7.e4 Line, 9.Nf3", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.e4 Bxf1 8.Kxf1 g6 9.Nf3 *"),
("A59e", "Benko Gambit: 7.e4 Line, 9.Nf3 Bg7 10.h3", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.e4 Bxf1 8.Kxf1 g6 9.Nf3 Bg7 10.h3 *"),
("A59f", "Benko Gambit: 7.e4 Line, 9.Nf3 Bg7 10.h3 Nbd7", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.e4 Bxf1 8.Kxf1 g6 9.Nf3 Bg7 10.h3 Nbd7 *"),
("A59g", "Benko Gambit: 7.e4 Line, 9.g4", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.e4 Bxf1 8.Kxf1 g6 9.g4 *"),
("A59h", "Benko Gambit: 7.e4 Line, 9.g3", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.e4 Bxf1 8.Kxf1 g6 9.g3 *"),
("A59i", "Benko Gambit: 7.e4, Main Line", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.e4 Bxf1 8.Kxf1 g6 9.g3 Bg7 10.Kg2 O-O 11.Nf3 *"),
("A59j", "Benko Gambit: 7.e4, Main Line, 11...Nbd7", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.e4 Bxf1 8.Kxf1 g6 9.g3 Bg7 10.Kg2 O-O 11.Nf3 Nbd7 *"),
("A59k", "Benko Gambit: 7.e4, Main Line, 11...Nbd7 12.Re1", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.e4 Bxf1 8.Kxf1 g6 9.g3 Bg7 10.Kg2 O-O 11.Nf3 Nbd7 12.Re1 *"),
("A59l", "Benko Gambit: 7.e4, Main Line, 11...Nbd7 12.Re1 Qa5", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.e4 Bxf1 8.Kxf1 g6 9.g3 Bg7 10.Kg2 O-O 11.Nf3 Nbd7 12.Re1 Qa5 *"),
("A59m", "Benko Gambit: 7.e4, Main Line, 11...Nbd7 12.Re1 Ng4", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.e4 Bxf1 8.Kxf1 g6 9.g3 Bg7 10.Kg2 O-O 11.Nf3 Nbd7 12.Re1 Ng4 *"),
("A59n", "Benko Gambit: 7.e4, Main Line, 11...Nbd7 12.h3", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.e4 Bxf1 8.Kxf1 g6 9.g3 Bg7 10.Kg2 O-O 11.Nf3 Nbd7 12.h3 *"),
("A59o", "Benko Gambit: 7.e4, Main Line, 11...Nbd7 12.h3 Ra6", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.e4 Bxf1 8.Kxf1 g6 9.g3 Bg7 10.Kg2 O-O 11.Nf3 Nbd7 12.h3 Ra6 *"),
("A59o", "Benko Gambit: 7.e4, Main Line, 11...Nbd7 12.h3 Ra6 13.Re1", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.e4 Bxf1 8.Kxf1 g6 9.g3 Bg7 10.Kg2 O-O 11.Nf3 Nbd7 12.h3 Ra6 13.Re1 *"),
("A59p", "Benko Gambit: 7.e4, Main Line, 11...Nbd7 12.h3 Qb6", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.e4 Bxf1 8.Kxf1 g6 9.g3 Bg7 10.Kg2 O-O 11.Nf3 Nbd7 12.h3 Qb6 *"),
("A59p", "Benko Gambit: 7.e4, Main Line, 11...Nbd7 12.h3 Qb6 13.Re1", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.e4 Bxf1 8.Kxf1 g6 9.g3 Bg7 10.Kg2 O-O 11.Nf3 Nbd7 12.h3 Qb6 13.Re1 *"),
("A59q", "Benko Gambit: 7.e4, Main Line, 11...Nbd7 12.h3 Qa5", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.e4 Bxf1 8.Kxf1 g6 9.g3 Bg7 10.Kg2 O-O 11.Nf3 Nbd7 12.h3 Qa5 *"),
("A59q", "Benko Gambit: 7.e4, Main Line, 11...Nbd7 12.h3 Qa5 13.Re1", "1.d4 Nf6 2.c4 c5 3.d5 b5 4.cxb5 a6 5.bxa6 Bxa6 6.Nc3 d6 7.e4 Bxf1 8.Kxf1 g6 9.g3 Bg7 10.Kg2 O-O 11.Nf3 Nbd7 12.h3 Qa5 13.Re1 *"),
("A60a", "Benoni: 3.d5 e6", "1.d4 Nf6 2.c4 c5 3.d5 e6 *"),
("A60b", "Benoni: 4.Nf3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nf3 *"),
("A60c", "Benoni: 4.Nf3 exd5 5.cxd5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nf3 exd5 5.cxd5 *"),
("A60d", "Benoni: Snake Variation vs. 4.Nf3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nf3 exd5 5.cxd5 Bd6 *"),
("A60e", "Benoni: 4.g3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.g3 *"),
("A60f", "Benoni: 4.g3 exd5 5.cxd5 b5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.g3 exd5 5.cxd5 b5 *"),
("A60g", "Benoni: 4.Nc3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 *"),
("A60h", "Benoni: 4.Nc3 exd5 5.Nxd5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.Nxd5 *"),
("A60i", "Benoni: 4.Nc3 exd5 5.cxd5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 *"),
("A60j", "Benoni: Snake Variation", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 Bd6 *"),
("A60k", "Benoni: Snake, 6.Nf3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 Bd6 6.Nf3 *"),
("A60l", "Benoni: Snake, 6.Nf3 Bc7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 Bd6 6.Nf3 Bc7 *"),
("A60m", "Benoni: Snake, 6.e4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 Bd6 6.e4 *"),
("A60n", "Benoni: Snake, 6.e4 O-O 7.Nf3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 Bd6 6.e4 O-O 7.Nf3 *"),
("A60o", "Benoni: 4.Nc3 exd5 5.cxd5 g6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 g6 *"),
("A60p", "Benoni: 4.Nc3 exd5 5.cxd5 d6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 *"),
("A60q", "Benoni: 4.Nc3 exd5 5.cxd5 d6 6.Nf3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 *"),
("A61", "Benoni: 6.Nf3 g6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 *"),
("A61", "Benoni: 6.Nf3 g6 7.h3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.h3 *"),
("A61", "Benoni: 6.Nf3 g6 7.Bf4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.Bf4 *"),
("A61", "Benoni: 6.Nf3 g6 7.Bf4 a6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.Bf4 a6 *"),
("A61", "Benoni: 6.Nf3 g6 7.Bf4 Bg7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.Bf4 Bg7 *"),
("A61", "Benoni: 6.Nf3 g6 7.Bf4 Bg7 8.Nd2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.Bf4 Bg7 8.Nd2 *"),
("A61", "Benoni: 6.Nf3 g6 7.Bf4 Bg7 8.Qa4+", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.Bf4 Bg7 8.Qa4+ *"),
("A61", "Benoni: Uhlmann Variation", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.Bg5 *"),
("A61", "Benoni: Uhlmann, 7...Bg7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.Bg5 Bg7 *"),
("A61", "Benoni: Uhlmann, 7...h6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.Bg5 h6 *"),
("A61", "Benoni: Nimzowitsch Variation", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.Nd2 *"),
("A61", "Benoni: Nimzowitsch, 7...Nbd7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.Nd2 Nbd7 *"),
("A61", "Benoni: Nimzowitsch, 7...Bg7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.Nd2 Bg7 *"),
("A61", "Benoni: Nimzowitsch, 7...Bg7 8.Nc4 O-O 9.Bf4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.Nd2 Bg7 8.Nc4 O-O 9.Bf4 *"),
("A61", "Benoni: Nimzowitsch, 7...Bg7 8.Nc4 O-O 9.Bg5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.Nd2 Bg7 8.Nc4 O-O 9.Bg5 *"),
("A61", "Benoni: Fianchetto Variation", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.g3 *"),
("A62", "Benoni: Fianchetto, 8.Bg2 O-O", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.g3 Bg7 8.Bg2 O-O *"),
("A62", "Benoni: Fianchetto, 9.O-O", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.g3 Bg7 8.Bg2 O-O 9.O-O *"),
("A62", "Benoni: Fianchetto, 9.O-O a6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.g3 Bg7 8.Bg2 O-O 9.O-O a6 *"),
("A62", "Benoni: Fianchetto, 9.O-O Na6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.g3 Bg7 8.Bg2 O-O 9.O-O Na6 *"),
("A62", "Benoni: Fianchetto, 9.O-O Na6 10.Nd2 Nc7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.g3 Bg7 8.Bg2 O-O 9.O-O Na6 10.Nd2 Nc7 *"),
("A62", "Benoni: Fianchetto, 9.O-O Re8", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.g3 Bg7 8.Bg2 O-O 9.O-O Re8 *"),
("A63", "Benoni: Fianchetto, 9...Nbd7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.g3 Bg7 8.Bg2 O-O 9.O-O Nbd7 *"),
("A63", "Benoni: Fianchetto, 9...Nbd7 10.Nd2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.g3 Bg7 8.Bg2 O-O 9.O-O Nbd7 10.Nd2 *"),
("A63", "Benoni: Fianchetto, 9...Nbd7 10.Nd2 Re8", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.g3 Bg7 8.Bg2 O-O 9.O-O Nbd7 10.Nd2 Re8 *"),
("A63", "Benoni: Fianchetto, 9...a6 10.a4 Nbd7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.g3 Bg7 8.Bg2 O-O 9.O-O a6 10.a4 Nbd7 *"),
("A63", "Benoni: Fianchetto, 9...Nbd7 10.Nd2 a6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.g3 Bg7 8.Bg2 O-O 9.O-O Nbd7 10.Nd2 a6 *"),
("A63", "Benoni: Fianchetto, 9...Nbd7 10.Nd2 a6 11.a4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.g3 Bg7 8.Bg2 O-O 9.O-O Nbd7 10.Nd2 a6 11.a4 *"),
("A64", "Benoni: Fianchetto, 11...Re8", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.g3 Bg7 8.Bg2 O-O 9.O-O Nbd7 10.Nd2 a6 11.a4 Re8 *"),
("A64", "Benoni: Fianchetto, 11...Re8 12.a5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.g3 Bg7 8.Bg2 O-O 9.O-O Nbd7 10.Nd2 a6 11.a4 Re8 12.a5 *"),
("A64", "Benoni: Fianchetto, 11...Re8 12.Nc4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.g3 Bg7 8.Bg2 O-O 9.O-O Nbd7 10.Nd2 a6 11.a4 Re8 12.Nc4 *"),
("A64", "Benoni: Fianchetto, 11...Re8 12.Nc4 Ne5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.g3 Bg7 8.Bg2 O-O 9.O-O Nbd7 10.Nd2 a6 11.a4 Re8 12.Nc4 Ne5 *"),
("A64", "Benoni: Fianchetto, 11...Re8 12.h3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.g3 Bg7 8.Bg2 O-O 9.O-O Nbd7 10.Nd2 a6 11.a4 Re8 12.h3 *"),
("A64", "Benoni: Fianchetto, 11...Re8 12.h3 Rb8", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.g3 Bg7 8.Bg2 O-O 9.O-O Nbd7 10.Nd2 a6 11.a4 Re8 12.h3 Rb8 *"),
("A64", "Benoni: Fianchetto, 11...Re8 12.h3 Rb8 13.Nc4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.g3 Bg7 8.Bg2 O-O 9.O-O Nbd7 10.Nd2 a6 11.a4 Re8 12.h3 Rb8 13.Nc4 *"),
("A64", "Benoni: Fianchetto, 11...Re8 12.h3 Rb8 13.Nc4 Ne5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.g3 Bg7 8.Bg2 O-O 9.O-O Nbd7 10.Nd2 a6 11.a4 Re8 12.h3 Rb8 13.Nc4 Ne5 *"),
("A64", "Benoni: Fianchetto, 11...Re8 12.h3 Rb8 13.Nc4 Nb6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.Nf3 g6 7.g3 Bg7 8.Bg2 O-O 9.O-O Nbd7 10.Nd2 a6 11.a4 Re8 12.h3 Rb8 13.Nc4 Nb6 *"),
("A65a", "Benoni: 6.e4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 *"),
("A65a", "Benoni: 6.e4 Be7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 Be7 *"),
("A65b", "Benoni: 6.e4 g6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 *"),
("A65b", "Benoni: 6.e4 g6 7.h3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.h3 *"),
("A65c", "Benoni: 6.e4 g6 7.Bf4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Bf4 *"),
("A65d", "Benoni: 6.e4 g6 7.Bf4 a6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Bf4 a6 *"),
("A65e", "Benoni: 6.e4 g6 7.Bf4 a6 8.Nf3 b5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Bf4 a6 8.Nf3 b5 *"),
("A65e", "Benoni: 6.e4 g6 7.Bf4 a6 8.Nf3 b5 9.Qe2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Bf4 a6 8.Nf3 b5 9.Qe2 *"),
("A65f", "Benoni: 6.e4 g6 7.Bd3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Bd3 *"),
("A65f", "Benoni: 6.e4 g6 7.Bd3 Bg7 8.h3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Bd3 Bg7 8.h3 *"),
("A65g", "Benoni: 6.e4 g6 7.Bd3 Bg7 8.Nge2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Bd3 Bg7 8.Nge2 *"),
("A65g", "Benoni: 6.e4 g6 7.Bd3 Bg7 8.Nge2 O-O", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Bd3 Bg7 8.Nge2 O-O *"),
("A65g", "Benoni: 6.e4 g6 7.Bd3 Bg7 8.Nge2 O-O 9.O-O", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Bd3 Bg7 8.Nge2 O-O 9.O-O *"),
("A65h", "Benoni: 6.e4 g6 7.Bd3 Bg7 8.Nge2 O-O 9.O-O b6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Bd3 Bg7 8.Nge2 O-O 9.O-O b6 *"),
("A65i", "Benoni: 6.e4 g6 7.Bd3 Bg7 8.Nge2 O-O 9.O-O Na6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Bd3 Bg7 8.Nge2 O-O 9.O-O Na6 *"),
("A65j", "Benoni: 6.e4 g6 7.Bd3 Bg7 8.Nge2 O-O 9.O-O a6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Bd3 Bg7 8.Nge2 O-O 9.O-O a6 *"),
("A65k", "Benoni: Saemisch (6.e4 g6 7.f3)", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f3 *"),
("A65k", "Benoni: Saemisch, 7...Bg7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f3 Bg7 *"),
("A65l", "Benoni: Saemisch, 8.Nge2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f3 Bg7 8.Nge2 *"),
("A65m", "Benoni: Saemisch, 8.Be3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f3 Bg7 8.Be3 *"),
("A65n", "Benoni: Saemisch, 8.Bg5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f3 Bg7 8.Bg5 *"),
("A65n", "Benoni: Saemisch, 8.Bg5 O-O", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f3 Bg7 8.Bg5 O-O *"),
("A65o", "Benoni: Saemisch, 8.Bg5 O-O 9.Nge2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f3 Bg7 8.Bg5 O-O 9.Nge2 *"),
("A65p", "Benoni: Saemisch, 8.Bg5 O-O 9.Qd2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f3 Bg7 8.Bg5 O-O 9.Qd2 *"),
("A65q", "Benoni: Saemisch, 8.Bg5 O-O 9.Qd2 a6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f3 Bg7 8.Bg5 O-O 9.Qd2 a6 *"),
("A65r", "Benoni: Saemisch, 8.Bg5 h6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f3 Bg7 8.Bg5 h6 *"),
("A65r", "Benoni: Saemisch, 8.Bg5 h6 9.Be3 O-O", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f3 Bg7 8.Bg5 h6 9.Be3 O-O *"),
("A65s", "Benoni: Saemisch, 8.Bg5 h6 9.Be3 O-O 10.Nge2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f3 Bg7 8.Bg5 h6 9.Be3 O-O 10.Nge2 *"),
("A65t", "Benoni: Saemisch, 8.Bg5 h6 9.Be3 O-O 10.Qd2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f3 Bg7 8.Bg5 h6 9.Be3 O-O 10.Qd2 *"),
("A65u", "Benoni: Saemisch, 8.Bg5 h6 9.Be3 O-O 10.Qd2 Re8", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f3 Bg7 8.Bg5 h6 9.Be3 O-O 10.Qd2 Re8 *"),
("A65v", "Benoni: Saemisch, 8.Bg5 h6 9.Be3 O-O 10.Qd2 a6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f3 Bg7 8.Bg5 h6 9.Be3 O-O 10.Qd2 a6 *"),
("A65v", "Benoni: Saemisch, 8.Bg5 h6 9.Be3 O-O 10.Qd2 a6 11.a4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f3 Bg7 8.Bg5 h6 9.Be3 O-O 10.Qd2 a6 11.a4 *"),
("A65w", "Benoni: Saemisch, 8.Bg5 h6 9.Be3 O-O 10.Qd2 a6 11.a4 h5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f3 Bg7 8.Bg5 h6 9.Be3 O-O 10.Qd2 a6 11.a4 h5 *"),
("A65x", "Benoni: Saemisch, 8.Bg5 h6 9.Be3 O-O 10.Qd2 a6 11.a4 Nbd7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f3 Bg7 8.Bg5 h6 9.Be3 O-O 10.Qd2 a6 11.a4 Nbd7 *"),
("A65y", "Benoni: Saemisch, 8.Bg5 h6 9.Be3 O-O 10.Qd2 a6 11.a4 Re8", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f3 Bg7 8.Bg5 h6 9.Be3 O-O 10.Qd2 a6 11.a4 Re8 *"),
("A65y", "Benoni: Saemisch, 8.Bg5 h6 9.Be3 O-O 10.Qd2 a6 11.a4 Re8 12.Nge2 Nbd7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f3 Bg7 8.Bg5 h6 9.Be3 O-O 10.Qd2 a6 11.a4 Re8 12.Nge2 Nbd7 *"),
("A66", "Benoni: Four Pawns Attack", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 *"),
("A66", "Benoni: Four Pawns Attack", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 *"),
("A66", "Benoni: Four Pawns, Mikenas Attack", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.e5 *"),
("A66", "Benoni: Mikenas Attack, 8...dxe5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.e5 dxe5 *"),
("A66", "Benoni: Mikenas Attack, 8...Nfd7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.e5 Nfd7 *"),
("A66", "Benoni: Mikenas Attack, 8...Nfd7 9.e6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.e5 Nfd7 9.e6 *"),
("A66", "Benoni: Mikenas Attack, 8...Nfd7 9.Nf3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.e5 Nfd7 9.Nf3 *"),
("A66", "Benoni: Mikenas Attack, 8...Nfd7 9.Nb5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.e5 Nfd7 9.Nb5 *"),
("A66", "Benoni: Mikenas Attack, 9.Nb5 dxe5 10.Nd6+", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.e5 Nfd7 9.Nb5 dxe5 10.Nd6+ *"),
("A66", "Benoni: Mikenas Attack, 9.Nb5 dxe5 10.Nd6+ Ke7 11.Nxc8+", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.e5 Nfd7 9.Nb5 dxe5 10.Nd6+ Ke7 11.Nxc8+ *"),
("A66", "Benoni: Mikenas Attack, 9.Nb5 dxe5 10.Nd6+ Ke7 11.Nxc8+ Qxc8 12.Nf3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.e5 Nfd7 9.Nb5 dxe5 10.Nd6+ Ke7 11.Nxc8+ Qxc8 12.Nf3 *"),
("A67a", "Benoni: Four Pawns, Taimanov (Alatortsev) Variation", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Bb5+ *"),
("A67b", "Benoni: Four Pawns, Taimanov, 8...Nbd7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Bb5+ Nbd7 *"),
("A67c", "Benoni: Four Pawns, Taimanov, 8...Nbd7 9.e5 dxe5 10.fxe5 Nh5 11.e6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Bb5+ Nbd7 9.e5 dxe5 10.fxe5 Nh5 11.e6 *"),
("A67d", "Benoni: Four Pawns, Taimanov, 8...Nfd7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Bb5+ Nfd7 *"),
("A67e", "Benoni: Four Pawns, Taimanov, 9.Bd3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Bb5+ Nfd7 9.Bd3 *"),
("A67f", "Benoni: Four Pawns, Taimanov, 9.Bd3 O-O", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Bb5+ Nfd7 9.Bd3 O-O *"),
("A67g", "Benoni: Four Pawns, Taimanov, 9.Bd3 O-O 10.Nf3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Bb5+ Nfd7 9.Bd3 O-O 10.Nf3 *"),
("A67h", "Benoni: Four Pawns, Taimanov, 9.Bd3 O-O 10.Nf3 a6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Bb5+ Nfd7 9.Bd3 O-O 10.Nf3 a6 *"),
("A67i", "Benoni: Four Pawns, Taimanov, 9.Bd3 O-O 10.Nf3 a6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Bb5+ Nfd7 9.Bd3 O-O 10.Nf3 a6 11.a4 *"),
("A67j", "Benoni: Four Pawns, Taimanov, 9.Bd3 O-O 10.Nf3 Na6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Bb5+ Nfd7 9.Bd3 O-O 10.Nf3 Na6 *"),
("A67k", "Benoni: Four Pawns, Taimanov, 9.a4 (Zaitsev)", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Bb5+ Nfd7 9.a4 *"),
("A67l", "Benoni: Four Pawns, Taimanov, 9.a4 Qh4+", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Bb5+ Nfd7 9.a4 Qh4+ *"),
("A67m", "Benoni: Four Pawns, Taimanov, 9.a4 a6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Bb5+ Nfd7 9.a4 a6 *"),
("A67n", "Benoni: Four Pawns, Taimanov, 9.a4 a6 10.Bd3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Bb5+ Nfd7 9.a4 a6 10.Bd3 *"),
("A67o", "Benoni: Four Pawns, Taimanov, 9.a4 a6 10.Be2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Bb5+ Nfd7 9.a4 a6 10.Be2 *"),
("A67p", "Benoni: Four Pawns, Taimanov, 9.a4 O-O", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Bb5+ Nfd7 9.a4 O-O *"),
("A67q", "Benoni: Four Pawns, Taimanov, 9.a4 O-O 10.Nf3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Bb5+ Nfd7 9.a4 O-O 10.Nf3 *"),
("A67r", "Benoni: Four Pawns, Taimanov, 9.a4 O-O 10.Nf3 Na6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Bb5+ Nfd7 9.a4 O-O 10.Nf3 Na6 *"),
("A67s", "Benoni: Four Pawns, Taimanov, 9.a4 O-O 10.Nf3 Na6 11.O-O Nc7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Bb5+ Nfd7 9.a4 O-O 10.Nf3 Na6 11.O-O Nc7 *"),
("A68a", "Benoni: Four Pawns, 8.Nf3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 *"),
("A68a", "Benoni: Four Pawns, 8.Nf3 O-O", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O *"),
("A68b", "Benoni: Four Pawns, 9.Bd3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Bd3 *"),
("A68c", "Benoni: Four Pawns, 9.Be2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 *"),
("A68d", "Benoni: Four Pawns, 9.Be2 b5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 b5 *"),
("A68e", "Benoni: Four Pawns, 9.Be2 b5 10.e5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 b5 10.e5 *"),
("A68f", "Benoni: Four Pawns, 9.Be2 b5 10.e5 dxe5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 b5 10.e5 dxe5 *"),
("A68g", "Benoni: Four Pawns, 9.Be2 b5 10.e5 dxe5 11.fxe5 Ng4 12.Bg5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 b5 10.e5 dxe5 11.fxe5 Ng4 12.Bg5 *"),
("A68h", "Benoni: Four Pawns, 9.Be2 Bg4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 Bg4 *"),
("A68i", "Benoni: Four Pawns, 9.Be2 Bg4 10.e5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 Bg4 10.e5 *"),
("A68j", "Benoni: Four Pawns, 9.Be2 Bg4 10.O-O", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 Bg4 10.O-O *"),
("A68k", "Benoni: Four Pawns, 9.Be2 Bg4 10.O-O Nbd7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 Bg4 10.O-O Nbd7 *"),
("A68l", "Benoni: Four Pawns, 9.Be2 Bg4 10.O-O Nbd7 11.h3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 Bg4 10.O-O Nbd7 11.h3 *"),
("A68m", "Benoni: Four Pawns, 9.Be2 Bg4 10.O-O Nbd7 11.h3 Bxf3 12.Bxf3 Re8", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 Bg4 10.O-O Nbd7 11.h3 Bxf3 12.Bxf3 Re8 *"),
("A68n", "Benoni: Four Pawns, 9.Be2 Bg4 10.O-O Nbd7 11.h3 Bxf3 12.Bxf3 Re8 13.Re1", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 Bg4 10.O-O Nbd7 11.h3 Bxf3 12.Bxf3 Re8 13.Re1 *"),
("A69a", "Benoni: Four Pawns, Main Line", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 Re8 *"),
("A69b", "Benoni: Four Pawns, Main Line, 10.O-O", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 Re8 10.O-O *"),
("A69c", "Benoni: Four Pawns, Main Line, 10.Nd2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 Re8 10.Nd2 *"),
("A69d", "Benoni: Four Pawns, Main Line, 10.Nd2 a6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 Re8 10.Nd2 a6 *"),
("A69e", "Benoni: Four Pawns, Main Line, 10.Nd2 Nbd7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 Re8 10.Nd2 Nbd7 *"),
("A69f", "Benoni: Four Pawns, Main Line, 10.e5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 Re8 10.e5 *"),
("A69g", "Benoni: Four Pawns, Main Line, 10.e5: 12.O-O", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 Re8 10.e5 dxe5 11.fxe5 Ng4 12.O-O *"),
("A69h", "Benoni: Four Pawns, Main Line, 10.e5: 12.e6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 Re8 10.e5 dxe5 11.fxe5 Ng4 12.e6 *"),
("A69i", "Benoni: Four Pawns, Main Line, 10.e5: 12.Bg5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 Re8 10.e5 dxe5 11.fxe5 Ng4 12.Bg5 *"),
("A69j", "Benoni: Four Pawns, Main Line, 10.e5: 12.Bg5 f6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 Re8 10.e5 dxe5 11.fxe5 Ng4 12.Bg5 f6 *"),
("A69k", "Benoni: Four Pawns, Main Line, 10.e5: 12.Bg5 Qb6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 Re8 10.e5 dxe5 11.fxe5 Ng4 12.Bg5 Qb6 *"),
("A69l", "Benoni: Four Pawns, Main Line, 10.e5: 12.Bg5 Qb6 13.O-O Nxe5 14.Nxe5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 Re8 10.e5 dxe5 11.fxe5 Ng4 12.Bg5 Qb6 13.O-O Nxe5 14.Nxe5 *"),
("A69m", "Benoni: Four Pawns, Main Line, 10.e5: 12.Bg5 Qb6 13.O-O Nxe5 14.d6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.f4 Bg7 8.Nf3 O-O 9.Be2 Re8 10.e5 dxe5 11.fxe5 Ng4 12.Bg5 Qb6 13.O-O Nxe5 14.d6 *"),
("A70a", "Benoni: Classical", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 *"),
("A70b", "Benoni: Classical, 7...a6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 a6 *"),
("A70b", "Benoni: Classical, 7...a6 8.a4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 a6 8.a4 *"),
("A70b", "Benoni: Classical, 7...a6 8.a4 Bg4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 a6 8.a4 Bg4 *"),
("A70c", "Benoni: Classical, 7...a6 8.a4 Bg4 9.Be2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 a6 8.a4 Bg4 9.Be2 *"),
("A70d", "Benoni: Classical, 7...Bg7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 *"),
("A70e", "Benoni: Classical, 8.Be2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 *"),
("A70e", "Benoni: Classical, 8.Be2 a6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 a6 *"),
("A70f", "Benoni: Classical, 8.Qa4+", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Qa4+ *"),
("A70f", "Benoni: Classical, 8.Qa4+ Bd7 9.Qb3 Qc7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Qa4+ Bd7 9.Qb3 Qc7 *"),
("A70g", "Benoni: Classical, 8.Bf4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Bf4 *"),
("A70g", "Benoni: Classical, 8.Bf4 O-O 9.Nd2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Bf4 O-O 9.Nd2 *"),
("A70h", "Benoni: Classical, 8.Bd3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Bd3 *"),
("A70i", "Benoni: Classical, 8.Bd3 O-O 9.O-O", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Bd3 O-O 9.O-O *"),
("A70j", "Benoni: Classical, 8.h3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.h3 *"),
("A70k", "Benoni: Classical, 8.h3 O-O 9.Bd3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.h3 O-O 9.Bd3 *"),
("A70l", "Benoni: Classical, 8.h3 O-O 9.Bd3 Bd7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.h3 O-O 9.Bd3 Bd7 *"),
("A70m", "Benoni: Classical, 8.h3 O-O 9.Bd3 Na6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.h3 O-O 9.Bd3 Na6 *"),
("A70n", "Benoni: Classical, 8.h3 O-O 9.Bd3 Re8", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.h3 O-O 9.Bd3 Re8 *"),
("A70o", "Benoni: Classical, 8.h3 O-O 9.Bd3 Re8 10.O-O c4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.h3 O-O 9.Bd3 Re8 10.O-O c4 *"),
("A70p", "Benoni: Classical, 8.h3 O-O 9.Bd3 a6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.h3 O-O 9.Bd3 a6 *"),
("A70q", "Benoni: Classical, 8.h3 O-O 9.Bd3 a6 10.O-O b5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.h3 O-O 9.Bd3 a6 10.O-O b5 *"),
("A70r", "Benoni: Classical, 8.h3 O-O 9.Bd3 a6 10.a4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.h3 O-O 9.Bd3 a6 10.a4 *"),
("A70s", "Benoni: Classical, 8.h3 O-O 9.Bd3 a6 10.a4 Nbd7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.h3 O-O 9.Bd3 a6 10.a4 Nbd7 *"),
("A70t", "Benoni: Classical, 8.h3 O-O 9.Bd3 a6 10.a4 Nbd7 11.O-O Re8", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.h3 O-O 9.Bd3 a6 10.a4 Nbd7 11.O-O Re8 *"),
("A70u", "Benoni: Classical, 8.h3 O-O 9.Bd3 b5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.h3 O-O 9.Bd3 b5 *"),
("A70v", "Benoni: Classical, 8.h3 O-O 9.Bd3 b5 10.Bxb5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.h3 O-O 9.Bd3 b5 10.Bxb5 *"),
("A70w", "Benoni: Classical, 8.h3 O-O 9.Bd3 b5 10.Nxb5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.h3 O-O 9.Bd3 b5 10.Nxb5 *"),
("A70w", "Benoni: Classical, 8.h3 O-O 9.Bd3 b5 10.Nxb5 Nxe4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.h3 O-O 9.Bd3 b5 10.Nxb5 Nxe4 *"),
("A70x", "Benoni: Classical, 8.h3 O-O 9.Bd3 b5 10.Nxb5 Re8", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.h3 O-O 9.Bd3 b5 10.Nxb5 Re8 *"),
("A70y", "Benoni: Classical, 8.h3 O-O 9.Bd3 b5 10.Nxb5 Re8 11.O-O Nxe4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.h3 O-O 9.Bd3 b5 10.Nxb5 Re8 11.O-O Nxe4 *"),
("A71", "Benoni: Classical, 8.Bg5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Bg5 *"),
("A71", "Benoni: Classical, 8.Bg5 h6 9.Bh4 g5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Bg5 h6 9.Bh4 g5 *"),
("A71", "Benoni: Classical, 8.Bg5 h6 9.Bh4 a6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Bg5 h6 9.Bh4 a6 *"),
("A71", "Benoni: Classical, 8.Bg5 h6 9.Bh4 a6 10.Nd2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Bg5 h6 9.Bh4 a6 10.Nd2 *"),
("A71", "Benoni: Classical, 8.Bg5 h6 9.Bh4 a6 10.Nd2 b5 11.Be2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Bg5 h6 9.Bh4 a6 10.Nd2 b5 11.Be2 *"),
("A72", "Benoni: Classical, 8.Be2 O-O", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O *"),
("A72", "Benoni: Classical, 8.Be2 O-O 9.Nd2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.Nd2 *"),
("A72", "Benoni: Classical, 8.Be2 O-O 9.Bg5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.Bg5 *"),
("A72", "Benoni: Classical, 8.Be2 O-O 9.Bg5 h6 10.Bh4 g5 11.Bg3 Nh5 12.Nd2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.Bg5 h6 10.Bh4 g5 11.Bg3 Nh5 12.Nd2 *"),
("A72", "Benoni: Classical, 8.Be2 O-O 9.Bf4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.Bf4 *"),
("A72", "Benoni: Classical, 8.Be2 O-O 9.Bf4 b5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.Bf4 b5 *"),
("A72", "Benoni: Classical, 8.Be2 O-O 9.Bf4 a6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.Bf4 a6 *"),
("A73", "Benoni: Classical, 9.O-O", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O *"),
("A73", "Benoni: Classical, 9.O-O Nbd7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Nbd7 *"),
("A73", "Benoni: Classical, 9.O-O Na6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Na6 *"),
("A73", "Benoni: Classical, 9.O-O Na6 10.Nd2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Na6 10.Nd2 *"),
("A73", "Benoni: Classical, 9.O-O Na6 10.Nd2 Nc7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Na6 10.Nd2 Nc7 *"),
("A73", "Benoni: Classical, 9.O-O Bg4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Bg4 *"),
("A73", "Benoni: Classical, 9.O-O Bg4 10.Bg5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Bg4 10.Bg5 *"),
("A73", "Benoni: Classical, 9.O-O Bg4 10.Bf4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Bg4 10.Bf4 *"),
("A73", "Benoni: Classical, 9.O-O Bg4 10.h3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Bg4 10.h3 *"),
("A73", "Benoni: Classical, 9.O-O a6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O a6 *"),
("A74", "Benoni: Classical, 9.O-O a6 10.a4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O a6 10.a4 *"),
("A74", "Benoni: Classical, 9.O-O a6 10.a4 Nbd7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O a6 10.a4 Nbd7 *"),
("A74", "Benoni: Classical, 9.O-O a6 10.a4 Nbd7 11.Nd2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O a6 10.a4 Nbd7 11.Nd2 *"),
("A74", "Benoni: Classical, 9.O-O a6 10.a4 Nbd7 11.Bg5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O a6 10.a4 Nbd7 11.Bg5 *"),
("A74", "Benoni: Classical, 9.O-O a6 10.a4 Nbd7 11.Bf4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O a6 10.a4 Nbd7 11.Bf4 *"),
("A74", "Benoni: Classical, 9.O-O a6 10.a4 Nbd7 11.Bf4 Qe7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O a6 10.a4 Nbd7 11.Bf4 Qe7 *"),
("A75", "Benoni: Classical, 9.O-O a6 10.a4 Bg4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O a6 10.a4 Bg4 *"),
("A75", "Benoni: Classical, 9.O-O a6 10.a4 Bg4 11.h3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O a6 10.a4 Bg4 11.h3 *"),
("A75", "Benoni: Classical, 9.O-O a6 10.a4 Bg4 11.Nd2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O a6 10.a4 Bg4 11.Nd2 *"),
("A75", "Benoni: Classical, 9.O-O a6 10.a4 Bg4 11.Bg5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O a6 10.a4 Bg4 11.Bg5 *"),
("A75", "Benoni: Classical, 9.O-O a6 10.a4 Bg4 11.Bg5 h6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O a6 10.a4 Bg4 11.Bg5 h6 *"),
("A75", "Benoni: Classical, 9.O-O a6 10.a4 Bg4 11.Bf4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O a6 10.a4 Bg4 11.Bf4 *"),
("A75", "Benoni: Classical, 9.O-O a6 10.a4 Bg4 11.Bf4 Qe7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O a6 10.a4 Bg4 11.Bf4 Qe7 *"),
("A75", "Benoni: Classical, 9.O-O a6 10.a4 Bg4 11.Bf4 Re8", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O a6 10.a4 Bg4 11.Bf4 Re8 *"),
("A75", "Benoni: Classical, 9.O-O a6 10.a4 Bg4 11.Bf4 Re8 12.Nd2 Bxe2 13.Qxe2 Nh5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O a6 10.a4 Bg4 11.Bf4 Re8 12.Nd2 Bxe2 13.Qxe2 Nh5 *"),
("A75", "Benoni: Classical, 9.O-O a6 10.a4 Bg4 11.Bf4 Bxf3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O a6 10.a4 Bg4 11.Bf4 Bxf3 *"),
("A75", "Benoni: Classical, 9.O-O a6 10.a4 Bg4 11.Bf4 Bxf3 12.Bxf3 Qe7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O a6 10.a4 Bg4 11.Bf4 Bxf3 12.Bxf3 Qe7 *"),
("A75", "Benoni: Classical, 9.O-O a6 10.a4 Bg4 11.Bf4 Bxf3 12.Bxf3 Qe7 13.Re1", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O a6 10.a4 Bg4 11.Bf4 Bxf3 12.Bxf3 Qe7 13.Re1 *"),
("A75", "Benoni: Classical, 9.O-O a6 10.a4 Bg4 11.Bf4 Bxf3 12.Bxf3 Qe7 with 14.a5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O a6 10.a4 Bg4 11.Bf4 Bxf3 12.Bxf3 Qe7 13.Re1 Nbd7 14.a5 *"),
("A76", "Benoni: Classical, Main Line", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 *"),
("A76", "Benoni: Classical, Main Line, 10.Qc2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Qc2 *"),
("A76", "Benoni: Classical, Main Line, 10.Qc2 Bg4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Qc2 Bg4 *"),
("A76", "Benoni: Classical, Main Line, 10.Qc2 Na6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Qc2 Na6 *"),
("A76", "Benoni: Classical, Main Line, 10.Qc2 Na6 11.Bf4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Qc2 Na6 11.Bf4 *"),
("A76", "Benoni: Classical, Main Line, 10.Qc2 Na6 11.Re1", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Qc2 Na6 11.Re1 *"),
("A76", "Benoni: Classical, Main Line, 10.Qc2 Na6 11.Re1 Bg4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Qc2 Na6 11.Re1 Bg4 *"),
("A77", "Benoni: Classical, Main Line, 10.Nd2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 *"),
("A77", "Benoni: Classical, Main Line, 10.Nd2 a6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 a6 *"),
("A77", "Benoni: Classical, Main Line, 10.Nd2 Nbd7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 Nbd7 *"),
("A77", "Benoni: Classical, Main Line, 10.Nd2 Nbd7 11.a4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 Nbd7 11.a4 *"),
("A77", "Benoni: Classical, Main Line, 10.Nd2 Nbd7 11.a4 a6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 Nbd7 11.a4 a6 *"),
("A77", "Benoni: Classical, Main Line, 10.Nd2 Nbd7 11.a4 a6 12.Ra3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 Nbd7 11.a4 a6 12.Ra3 *"),
("A77", "Benoni: Classical, Main Line, 10.Nd2 Nbd7 11.a4 a6 12.Qc2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 Nbd7 11.a4 a6 12.Qc2 *"),
("A77", "Benoni: Classical, Main Line, 10.Nd2 Nbd7 11.a4 a6 12.h3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 Nbd7 11.a4 a6 12.h3 *"),
("A77", "Benoni: Classical, Main Line, 10.Nd2 Nbd7 11.a4 Ne5", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 Nbd7 11.a4 Ne5 *"),
("A77", "Benoni: Classical, Main Line, 10.Nd2 Nbd7 11.a4 Ne5 12.Ndb1", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 Nbd7 11.a4 Ne5 12.Ndb1 *"),
("A77", "Benoni: Classical, Main Line, 10.Nd2 Nbd7 11.a4 Ne5 12.Ra3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 Nbd7 11.a4 Ne5 12.Ra3 *"),
("A77", "Benoni: Classical, Main Line, 10.Nd2 Nbd7 11.a4 Ne5 12.Re1", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 Nbd7 11.a4 Ne5 12.Re1 *"),
("A77", "Benoni: Classical, Main Line, 10.Nd2 Nbd7 11.a4 Ne5 12.Qc2", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 Nbd7 11.a4 Ne5 12.Qc2 *"),
("A78", "Benoni: Classical, Main Line, 10.Nd2 Na6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 Na6 *"),
("A78", "Benoni: Classical, Main Line, 10.Nd2 Na6 11.Kh1", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 Na6 11.Kh1 *"),
("A78", "Benoni: Classical, Main Line, 10.Nd2 Na6 11.Rb1", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 Na6 11.Rb1 *"),
("A78", "Benoni: Classical, Main Line, 10.Nd2 Na6 11.Re1", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 Na6 11.Re1 *"),
("A78", "Benoni: Classical, Main Line, 10.Nd2 Na6 11.f4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 Na6 11.f4 *"),
("A78", "Benoni: Classical, Main Line, 10.Nd2 Na6 11.f4 Nc7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 Na6 11.f4 Nc7 *"),
("A79", "Benoni: Classical, Main Line, 10.Nd2 Na6 11.f3", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 Na6 11.f3 *"),
("A79", "Benoni: Classical, Main Line, 10.Nd2 Na6 11.f3 Nc7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 Na6 11.f3 Nc7 *"),
("A79", "Benoni: Classical, Main Line, 10.Nd2 Na6 11.f3 Nc7 12.a4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 Na6 11.f3 Nc7 12.a4 *"),
("A79", "Benoni: Classical, Main Line, 10.Nd2 Na6 11.f3 Nc7 12.a4 Nd7", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 Na6 11.f3 Nc7 12.a4 Nd7 *"),
("A79", "Benoni: Classical, Main Line, 10.Nd2 Na6 11.f3 Nc7 12.a4 b6", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 Na6 11.f3 Nc7 12.a4 b6 *"),
("A79", "Benoni: Classical, Main Line, 10.Nd2 Na6 11.f3 Nc7 12.a4 b6 13.Kh1", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 Na6 11.f3 Nc7 12.a4 b6 13.Kh1 *"),
("A79", "Benoni: Classical, Main Line, 10.Nd2 Na6 11.f3 Nc7 12.a4 b6 13.Nc4", "1.d4 Nf6 2.c4 c5 3.d5 e6 4.Nc3 exd5 5.cxd5 d6 6.e4 g6 7.Nf3 Bg7 8.Be2 O-O 9.O-O Re8 10.Nd2 Na6 11.f3 Nc7 12.a4 b6 13.Nc4 *"),
("A80a", "Dutch", "1.d4 f5 *"),
("A80b", "Dutch: Krejcik Gambit", "1.d4 f5 2.g4 *"),
("A80b", "Dutch: Krejcik Gambit Accepted", "1.d4 f5 2.g4 fxg4 *"),
("A80c", "Dutch: Korchnoi Attack", "1.d4 f5 2.h3 *"),
("A80c", "Dutch: Korchnoi, Janzen Gambit", "1.d4 f5 2.h3 Nf6 3.g4 *"),
("A80d", "Dutch: 2.Bg5", "1.d4 f5 2.Bg5 *"),
("A80d", "Dutch: 2.Bg5 d5", "1.d4 f5 2.Bg5 d5 *"),
("A80d", "Dutch: 2.Bg5 c6", "1.d4 f5 2.Bg5 c6 *"),
("A80e", "Dutch: 2.Bg5 Nf6", "1.d4 f5 2.Bg5 Nf6 *"),
("A80f", "Dutch: 2.Bg5 h6", "1.d4 f5 2.Bg5 h6 *"),
("A80g", "Dutch: 2.Bg5 g6", "1.d4 f5 2.Bg5 g6 *"),
("A80h", "Dutch: 2.Bg5 g6 3.Nc3", "1.d4 f5 2.Bg5 g6 3.Nc3 *"),
("A80i", "Dutch: Alapin", "1.d4 f5 2.Qd3 *"),
("A80i", "Dutch: Alapin, Manhattan Gambit", "1.d4 f5 2.Qd3 d6 3.g4 *"),
("A80i", "Dutch: Von Pretzel Gambit", "1.d4 f5 2.Qd3 e6 3.g4 *"),
("A80j", "Dutch: 2.Nc3", "1.d4 f5 2.Nc3 *"),
("A80k", "Dutch: 2.Nc3 d5", "1.d4 f5 2.Nc3 d5 *"),
("A80k", "Dutch: 2.Nc3 d5, Euwe Gambit", "1.d4 f5 2.Nc3 d5 3.e4 *"),
("A80l", "Dutch: 2.Nc3 d5 3.Bg5", "1.d4 f5 2.Nc3 d5 3.Bg5 *"),
("A80m", "Dutch: 2.Nc3 Nf6", "1.d4 f5 2.Nc3 Nf6 *"),
("A80m", "Dutch: Spielmann Gambit", "1.d4 f5 2.Nc3 Nf6 3.g4 *"),
("A80n", "Dutch: 2.Nc3 Nf6 3.Bg5", "1.d4 f5 2.Nc3 Nf6 3.Bg5 *"),
("A80n", "Dutch: 2.Nc3 Nf6 3.Bg5 e6", "1.d4 f5 2.Nc3 Nf6 3.Bg5 e6 *"),
("A80o", "Dutch: 2.Nc3 Nf6 3.Bg5 d5", "1.d4 f5 2.Nc3 Nf6 3.Bg5 d5 *"),
("A80p", "Dutch: 2.Nc3 Nf6 3.Bg5 d5 4.Bxf6 exf6", "1.d4 f5 2.Nc3 Nf6 3.Bg5 d5 4.Bxf6 exf6 *"),
("A80p", "Dutch: 2.Nc3 Nf6 3.Bg5 d5 4.Bxf6 exf6 5.e3", "1.d4 f5 2.Nc3 Nf6 3.Bg5 d5 4.Bxf6 exf6 5.e3 *"),
("A80p", "Dutch: 2.Nc3 Nf6 3.Bg5 d5 4.Bxf6 exf6 5.e3 c6", "1.d4 f5 2.Nc3 Nf6 3.Bg5 d5 4.Bxf6 exf6 5.e3 c6 *"),
("A80q", "Dutch: 2.Nf3", "1.d4 f5 2.Nf3 *"),
("A80r", "Dutch: 2.Nf3 e6", "1.d4 f5 2.Nf3 e6 *"),
("A80s", "Dutch: 2.Nf3 e6 3.d5", "1.d4 f5 2.Nf3 e6 3.d5 *"),
("A80t", "Dutch: 2.Nf3 Nf6", "1.d4 f5 2.Nf3 Nf6 *"),
("A80u", "Dutch: Barcza System", "1.d4 f5 2.Nf3 Nf6 3.c3 *"),
("A80v", "Dutch: 2.Nf3 Nf6 3.Bg5", "1.d4 f5 2.Nf3 Nf6 3.Bg5 *"),
("A81", "Dutch: 2.g3", "1.d4 f5 2.g3 *"),
("A81", "Dutch: 2.g3 e6", "1.d4 f5 2.g3 e6 *"),
("A81", "Dutch: 2.g3 e6 3.Nf3", "1.d4 f5 2.g3 e6 3.Nf3 *"),
("A81", "Dutch: 2.g3 e6 3.Nf3 Nf6", "1.d4 f5 2.g3 e6 3.Nf3 Nf6 *"),
("A81", "Dutch: 2.g3 Nf6", "1.d4 f5 2.g3 Nf6 *"),
("A81", "Dutch: 2.g3 Nf6 3.Nf3", "1.d4 f5 2.g3 Nf6 3.Nf3 *"),
("A81", "Dutch: 2.g3 Nf6 3.Bg2", "1.d4 f5 2.g3 Nf6 3.Bg2 *"),
("A81", "Dutch: 2.g3 Nf6 3.Bg2 e6", "1.d4 f5 2.g3 Nf6 3.Bg2 e6 *"),
("A81", "Dutch: Blackburne Variation", "1.d4 f5 2.g3 Nf6 3.Bg2 e6 4.Nh3 *"),
("A81", "Dutch: 2.g3 Nf6 3.Bg2 g6", "1.d4 f5 2.g3 Nf6 3.Bg2 g6 *"),
("A81", "Dutch: 2.g3 Nf6 3.Bg2 g6 4.Nf3", "1.d4 f5 2.g3 Nf6 3.Bg2 g6 4.Nf3 *"),
("A81", "Dutch: Leningrad, Basman Variation", "1.d4 f5 2.g3 g6 3.Bg2 Bg7 4.Nf3 c6 5.O-O Nh6 *"),
("A81", "Dutch: Leningrad, Carlsbad Variation", "1.d4 f5 2.g3 g6 3.Bg2 Bg7 4.Nh3 *"),
("A82", "Dutch: Staunton Gambit", "1.d4 f5 2.e4 *"),
("A82", "Dutch: Staunton Gambit, Balogh Defence", "1.d4 f5 2.e4 d6 *"),
("A82", "Dutch: Staunton Gambit Accepted", "1.d4 f5 2.e4 fxe4 *"),
("A82", "Dutch: Staunton Gambit, 3.Nc3", "1.d4 f5 2.e4 fxe4 3.Nc3 *"),
("A82", "Dutch: Staunton Gambit, 3.Nc3 e6", "1.d4 f5 2.e4 fxe4 3.Nc3 e6 *"),
("A82", "Dutch: Staunton Gambit, 3.Nc3 g6", "1.d4 f5 2.e4 fxe4 3.Nc3 g6 *"),
("A82", "Dutch: Staunton Gambit, 3.Nc3 Nf6", "1.d4 f5 2.e4 fxe4 3.Nc3 Nf6 *"),
("A82", "Dutch: Staunton Gambit, Tartakower Variation", "1.d4 f5 2.e4 fxe4 3.Nc3 Nf6 4.g4 *"),
("A82", "Dutch: Staunton Gambit, 4.f3", "1.d4 f5 2.e4 fxe4 3.Nc3 Nf6 4.f3 *"),
("A82", "Dutch: Staunton Gambit, 4.f3 e6", "1.d4 f5 2.e4 fxe4 3.Nc3 Nf6 4.f3 e6 *"),
("A82", "Dutch: Staunton Gambit, 4.f3 Nc6", "1.d4 f5 2.e4 fxe4 3.Nc3 Nf6 4.f3 Nc6 *"),
("A82", "Dutch: Staunton Gambit, 4.f3 exf3", "1.d4 f5 2.e4 fxe4 3.Nc3 Nf6 4.f3 exf3 *"),
("A82", "Dutch: Staunton Gambit, 4.f3 exf3 5.Nxf3", "1.d4 f5 2.e4 fxe4 3.Nc3 Nf6 4.f3 exf3 5.Nxf3 *"),
("A82", "Dutch: Staunton Gambit, 4.f3 d5", "1.d4 f5 2.e4 fxe4 3.Nc3 Nf6 4.f3 d5 *"),
("A82", "Dutch: Staunton Gambit, 4.f3 d5 5.fxe4", "1.d4 f5 2.e4 fxe4 3.Nc3 Nf6 4.f3 d5 5.fxe4 *"),
("A83a", "Dutch: Staunton Gambit, Staunton Variation", "1.d4 f5 2.e4 fxe4 3.Nc3 Nf6 4.Bg5 *"),
("A83b", "Dutch: Staunton Gambit, Nimzowitsch Variation", "1.d4 f5 2.e4 fxe4 3.Nc3 Nf6 4.Bg5 b6 *"),
("A83c", "Dutch: Staunton Gambit, Chigorin Variation", "1.d4 f5 2.e4 fxe4 3.Nc3 Nf6 4.Bg5 c6 *"),
("A83d", "Dutch: Staunton Gambit, 4.Bg5 g6", "1.d4 f5 2.e4 fxe4 3.Nc3 Nf6 4.Bg5 g6 *"),
("A83e", "Dutch: Staunton Gambit, Alekhine Variation", "1.d4 f5 2.e4 fxe4 3.Nc3 Nf6 4.Bg5 g6 5.h4 *"),
("A83f", "Dutch: Staunton Gambit, Lasker Variation", "1.d4 f5 2.e4 fxe4 3.Nc3 Nf6 4.Bg5 g6 5.f3 *"),
("A83g", "Dutch: Staunton Gambit, 4.Bg5 e6", "1.d4 f5 2.e4 fxe4 3.Nc3 Nf6 4.Bg5 e6 *"),
("A83h", "Dutch: Staunton Gambit, 4.Bg5 e6 5.Nxe4", "1.d4 f5 2.e4 fxe4 3.Nc3 Nf6 4.Bg5 e6 5.Nxe4 *"),
("A83i", "Dutch: Staunton Gambit, 4.Bg5 e6 5.Nxe4", "1.d4 f5 2.e4 fxe4 3.Nc3 Nf6 4.Bg5 e6 5.Nxe4 Be7 6.Bxf6 Bxf6 7.Nf3 *"),
("A83j", "Dutch: Staunton Gambit, 4.Bg5 Nc6", "1.d4 f5 2.e4 fxe4 3.Nc3 Nf6 4.Bg5 Nc6 *"),
("A83k", "Dutch: Staunton Gambit, 4.Bg5 Nc6 5.f3", "1.d4 f5 2.e4 fxe4 3.Nc3 Nf6 4.Bg5 Nc6 5.f3 *"),
("A83l", "Dutch: Staunton Gambit, 4.Bg5 Nc6 5.d5", "1.d4 f5 2.e4 fxe4 3.Nc3 Nf6 4.Bg5 Nc6 5.d5 *"),
("A83m", "Dutch: Staunton Gambit, 4.Bg5 Nc6 5.d5 Ne5 6.Qd4", "1.d4 f5 2.e4 fxe4 3.Nc3 Nf6 4.Bg5 Nc6 5.d5 Ne5 6.Qd4 *"),
("A84", "Dutch: 2.c4", "1.d4 f5 2.c4 *"),
("A84", "Dutch: 2.c4 d6", "1.d4 f5 2.c4 d6 *"),
("A84", "Dutch: 2.c4 g6", "1.d4 f5 2.c4 g6 *"),
("A84", "Dutch: Bladel Variation", "1.d4 f5 2.c4 g6 3.Nc3 Nh6 *"),
("A84", "Dutch: 2.c4 e6", "1.d4 f5 2.c4 e6 *"),
("A84", "Dutch: 2.c4 e6 3.Nf3", "1.d4 f5 2.c4 e6 3.Nf3 *"),
("A84", "Dutch: 2.c4 e6 3.Nf3 Nf6", "1.d4 f5 2.c4 e6 3.Nf3 Nf6 *"),
("A84", "Dutch: Rubinstein Variation", "1.d4 f5 2.c4 e6 3.Nc3 *"),
("A84", "Dutch: Rubinstein, 3...d5", "1.d4 f5 2.c4 e6 3.Nc3 d5 *"),
("A84", "Dutch: Rubinstein, 3...d5 4.e3 c6", "1.d4 f5 2.c4 e6 3.Nc3 d5 4.e3 c6 *"),
("A84", "Dutch: Staunton Gambit Deferred", "1.d4 f5 2.c4 e6 3.e4 *"),
("A84", "Dutch: 2.c4 Nf6", "1.d4 f5 2.c4 Nf6 *"),
("A84", "Dutch: 2.c4 Nf6", "1.d4 f5 2.c4 Nf6 3.Nf3 *"),
("A85", "Dutch: 2.c4 Nf6 3.Nc3", "1.d4 f5 2.c4 Nf6 3.Nc3 *"),
("A85", "Dutch: 2.c4 Nf6 3.Nc3 d6", "1.d4 f5 2.c4 Nf6 3.Nc3 d6 *"),
("A85", "Dutch: 2.c4 Nf6 3.Nc3 e6", "1.d4 f5 2.c4 Nf6 3.Nc3 e6 *"),
("A85", "Dutch: 2.c4 Nf6 3.Nc3 e6 4.a3", "1.d4 f5 2.c4 Nf6 3.Nc3 e6 4.a3 *"),
("A85", "Dutch: 2.c4 Nf6 3.Nc3 e6 4.Qc2", "1.d4 f5 2.c4 Nf6 3.Nc3 e6 4.Qc2 *"),
("A85", "Dutch: 2.c4 Nf6 3.Nc3 e6 4.Bg5", "1.d4 f5 2.c4 Nf6 3.Nc3 e6 4.Bg5 *"),
("A85", "Dutch: 2.c4 Nf6 3.Nc3 e6 4.e3", "1.d4 f5 2.c4 Nf6 3.Nc3 e6 4.e3 *"),
("A85", "Dutch: 2.c4 Nf6 3.Nc3 e6 4.e3 d5", "1.d4 f5 2.c4 Nf6 3.Nc3 e6 4.e3 d5 *"),
("A85", "Dutch: 2.c4 Nf6 3.Nc3 e6 4.e3 Bb4", "1.d4 f5 2.c4 Nf6 3.Nc3 e6 4.e3 Bb4 *"),
("A85", "Dutch: 2.c4 Nf6 3.Nc3 e6 4.Nf3", "1.d4 f5 2.c4 Nf6 3.Nc3 e6 4.Nf3 *"),
("A85", "Dutch: 2.c4 Nf6 3.Nc3 g6", "1.d4 f5 2.c4 Nf6 3.Nc3 g6 *"),
("A85", "Dutch: 2.c4 Nf6 3.Nc3 g6 4.f3", "1.d4 f5 2.c4 Nf6 3.Nc3 g6 4.f3 *"),
("A85", "Dutch: 2.c4 Nf6 3.Nc3 g6 4.Bg5", "1.d4 f5 2.c4 Nf6 3.Nc3 g6 4.Bg5 *"),
("A85", "Dutch: 2.c4 Nf6 3.Nc3 g6 4.Nf3", "1.d4 f5 2.c4 Nf6 3.Nc3 g6 4.Nf3 *"),
("A85", "Dutch: 2.c4 Nf6 3.Nc3 g6 4.Nf3 Bg7", "1.d4 f5 2.c4 Nf6 3.Nc3 g6 4.Nf3 Bg7 *"),
("A85", "Dutch: 2.c4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.e3", "1.d4 f5 2.c4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.e3 *"),
("A85", "Dutch: 2.c4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.Bf4", "1.d4 f5 2.c4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.Bf4 *"),
("A85", "Dutch: 2.c4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.Bg5", "1.d4 f5 2.c4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.Bg5 *"),
("A86", "Dutch: 2.c4 Nf6 3.g3", "1.d4 f5 2.c4 Nf6 3.g3 *"),
("A86", "Dutch: 2.c4 Nf6 3.g3 d6", "1.d4 f5 2.c4 Nf6 3.g3 d6 *"),
("A86", "Dutch: Hort-Antoshin System", "1.d4 f5 2.c4 Nf6 3.g3 d6 4.Bg2 c6 5.Nc3 Qc7 *"),
("A86", "Dutch: Leningrad Variation", "1.d4 f5 2.c4 Nf6 3.g3 g6 *"),
("A86", "Dutch: Leningrad, 4.Nc3", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Nc3 *"),
("A86", "Dutch: Leningrad, 4.Nc3 Bg7", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Nc3 Bg7 *"),
("A86", "Dutch: Leningrad, 4.Nf3", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Nf3 *"),
("A86", "Dutch: Leningrad, 4.Nf3 Bg7", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Nf3 Bg7 *"),
("A86", "Dutch: Leningrad, 4.Nf3 Bg7 5.Nc3", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Nf3 Bg7 5.Nc3 *"),
("A86", "Dutch: Leningrad, 4.Bg2", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 *"),
("A86", "Dutch: Leningrad, 4.Bg2 d6", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 d6 *"),
("A86", "Dutch: Leningrad, 4.Bg2 Bg7", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 *"),
("A86", "Dutch: Leningrad, 4.Bg2 Bg7 5.Nh3", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nh3 *"),
("A86", "Dutch: Leningrad, 4.Bg2 Bg7 5.Nc3", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nc3 *"),
("A86", "Dutch: Leningrad, 4.Bg2 Bg7 5.Nc3 d6", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nc3 d6 *"),
("A86", "Dutch: Leningrad, 4.Bg2 Bg7 5.Nc3 d6 6.Nh3", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nc3 d6 6.Nh3 *"),
("A86", "Dutch: Leningrad, 4.Bg2 Bg7 5.Nc3 d6 6.d5", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nc3 d6 6.d5 *"),
("A86", "Dutch: Leningrad, 4.Bg2 Bg7 5.Nc3 O-O", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nc3 O-O *"),
("A86", "Dutch: Leningrad, 4.Bg2 Bg7 5.Nc3 O-O 6.Nh3", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nc3 O-O 6.Nh3 *"),
("A87", "Dutch: Leningrad, Main Line", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 *"),
("A87", "Dutch: Leningrad, Main Line, 5...d6", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 d6 *"),
("A87", "Dutch: Leningrad, Main Line, 5...d6 6.Nc3", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 d6 6.Nc3 *"),
("A87", "Dutch: Leningrad, Main Line, 5...O-O", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O *"),
("A87", "Dutch: Leningrad, Main Line, 6.Nc3", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.Nc3 *"),
("A87", "Dutch: Leningrad, Main Line, 6.Nc3 d6", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.Nc3 d6 *"),
("A87", "Dutch: Leningrad, Main Line, 6.O-O", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O *"),
("A87", "Dutch: Leningrad, Main Line, 6...c6", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 *"),
("A87", "Dutch: Leningrad, Main Line, 6...d6", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 *"),
("A87", "Dutch: Leningrad, Main Line, 7.b3", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.b3 *"),
("A87", "Dutch: Leningrad, Main Line, 7.b3 c6", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.b3 c6 *"),
("A87", "Dutch: Leningrad, Main Line, 7.d5", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.d5 *"),
("A87", "Dutch: Leningrad, Main Line, 7.d5 c6", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.d5 c6 *"),
("A87", "Dutch: Leningrad, Main Line, 7.d5 c5", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.d5 c5 *"),
("A87", "Dutch: Leningrad, Main Line, 7.Nc3", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 *"),
("A87", "Dutch: Leningrad, Main Line, 7.Nc3 e6", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 e6 *"),
("A87", "Dutch: Leningrad, Main Line, 7.Nc3 Na6", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 Na6 *"),
("A87", "Dutch: Leningrad, Main Line, 7.Nc3 Qe8", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 *"),
("A87", "Dutch: Leningrad, Main Line, 7.Nc3 Qe8 8.Re1", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.Re1 *"),
("A87", "Dutch: Leningrad, Main Line, 7.Nc3 Qe8 8.Nd5", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.Nd5 *"),
("A87", "Dutch: Leningrad, Main Line, 7.Nc3 Qe8 8.b3", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.b3 *"),
("A87", "Dutch: Leningrad, Main Line, 7.Nc3 Qe8 8.d5", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.d5 *"),
("A87", "Dutch: Leningrad, Main Line, 7.Nc3 Qe8 8.d5 a5", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.d5 a5 *"),
("A87", "Dutch: Leningrad, Main Line, 7.Nc3 Qe8 8.d5 Na6", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.d5 Na6 *"),
("A87", "Dutch: Leningrad, Main Line, 7.Nc3 Qe8 8.d5 Na6 9.Rb1", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.d5 Na6 9.Rb1 *"),
("A88", "Dutch: Leningrad, Main Line, 7.Nc3 c6", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 c6 *"),
("A88", "Dutch: Leningrad, Main Line, 7.Nc3 c6 8.Re1", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 c6 8.Re1 *"),
("A88", "Dutch: Leningrad, Main Line, 7.Nc3 c6 8.Qc2", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 c6 8.Qc2 *"),
("A88", "Dutch: Leningrad, Main Line, 7.Nc3 c6 8.b3", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 c6 8.b3 *"),
("A88", "Dutch: Leningrad, Main Line, 7.Nc3 c6 8.b3 Na6", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 c6 8.b3 Na6 *"),
("A88", "Dutch: Leningrad, Main Line, 7.Nc3 c6 8.b3 Qe8", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 c6 8.b3 Qe8 *"),
("A88", "Dutch: Leningrad, Main Line, 7.Nc3 c6 8.b3 Qa5", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 c6 8.b3 Qa5 *"),
("A88", "Dutch: Leningrad, Main Line, 7.Nc3 c6 8.d5", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 c6 8.d5 *"),
("A88", "Dutch: Leningrad, Main Line, 7.Nc3 c6 8.d5 cxd5", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 c6 8.d5 cxd5 *"),
("A88", "Dutch: Leningrad, Main Line, 7.Nc3 c6 8.d5 Qe8", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 c6 8.d5 Qe8 *"),
("A88", "Dutch: Leningrad, Main Line, 7.Nc3 c6 8.d5 Qa5", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 c6 8.d5 Qa5 *"),
("A88", "Dutch: Leningrad, Main Line, 7.Nc3 c6 8.d5 Bd7", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 c6 8.d5 Bd7 *"),
("A88", "Dutch: Leningrad, Main Line, 7.Nc3 c6 8.d5 e5", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 c6 8.d5 e5 *"),
("A88", "Dutch: Leningrad, Main Line, 7.Nc3 c6 8.d5 e5 9.dxe6 Bxe6 10.b3", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 c6 8.d5 e5 9.dxe6 Bxe6 10.b3 *"),
("A88", "Dutch: Leningrad, Main Line, 7.Nc3 c6 8.d5 e5 9.dxe6 Bxe6 10.Qd3", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 c6 8.d5 e5 9.dxe6 Bxe6 10.Qd3 *"),
("A89", "Dutch: Leningrad, Main Line, 7.Nc3 Nc6", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 Nc6 *"),
("A89", "Dutch: Leningrad, Main Line, 7.Nc3 Nc6 8.b3", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 Nc6 8.b3 *"),
("A89", "Dutch: Leningrad, Main Line, 7.Nc3 Nc6 8.d5", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 Nc6 8.d5 *"),
("A89", "Dutch: Leningrad, Main Line, 7.Nc3 Nc6 8.d5 Na5", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 Nc6 8.d5 Na5 *"),
("A89", "Dutch: Leningrad, Main Line, 7.Nc3 Nc6 8.d5 Na5 9.Qd3", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 Nc6 8.d5 Na5 9.Qd3 *"),
("A89", "Dutch: Leningrad, Main Line, 7.Nc3 Nc6 8.d5 Na5 9.Nd2", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 Nc6 8.d5 Na5 9.Nd2 *"),
("A89", "Dutch: Leningrad, Main Line, 7.Nc3 Nc6 8.d5 Ne5", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 Nc6 8.d5 Ne5 *"),
("A89", "Dutch: Leningrad, Main Line, 7.Nc3 Nc6 8.d5 Ne5 9.Nxe5", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 Nc6 8.d5 Ne5 9.Nxe5 *"),
("A89", "Dutch: Leningrad, Main Line, 7.Nc3 Nc6 8.d5 Ne5 9.Nxe5 dxe5 10.Qb3", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 Nc6 8.d5 Ne5 9.Nxe5 dxe5 10.Qb3 *"),
("A89", "Dutch: Leningrad, Main Line, 7.Nc3 Nc6 8.d5 Ne5 9.Nxe5 dxe5 10.e4", "1.d4 f5 2.c4 Nf6 3.g3 g6 4.Bg2 Bg7 5.Nf3 O-O 6.O-O d6 7.Nc3 Nc6 8.d5 Ne5 9.Nxe5 dxe5 10.e4 *"),
("A90", "Dutch: 2.c4 Nf6 3.g3 e6", "1.d4 f5 2.c4 Nf6 3.g3 e6 *"),
("A90", "Dutch: 2.c4 Nf6 3.g3 e6 4.Nc3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Nc3 *"),
("A90", "Dutch: 2.c4 Nf6 3.g3 e6 4.Nc3 d5", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Nc3 d5 *"),
("A90", "Dutch: 2.c4 Nf6 3.g3 e6 4.Nf3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Nf3 *"),
("A90", "Dutch: 2.c4 Nf6 3.g3 e6 4.Nf3 d5", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Nf3 d5 *"),
("A90", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 *"),
("A90", "Dutch: Dutch-Indian (Nimzo-Dutch) Variation", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Bb4+ *"),
("A90", "Dutch: Dutch-Indian, 5.Nc3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Bb4+ 5.Nc3 *"),
("A90", "Dutch: Dutch-Indian, 5.Nd2", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Bb4+ 5.Nd2 *"),
("A90", "Dutch: Dutch-Indian, 5.Bd2", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Bb4+ 5.Bd2 *"),
("A90", "Dutch: Dutch-Indian, Alekhine Variation", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Bb4+ 5.Bd2 Be7 *"),
("A90", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 d5", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 d5 *"),
("A90", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 d5 5.Nh3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 d5 5.Nh3 *"),
("A90", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 d5 5.Nf3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 d5 5.Nf3 *"),
("A90", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 c6", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 c6 *"),
("A90", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 c6 5.Nh3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 c6 5.Nh3 *"),
("A90", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 c6 5.Nf3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 c6 5.Nf3 *"),
("A90", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 c6 5.Nf3 d5", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 c6 5.Nf3 d5 *"),
("A90", "Dutch: 2.c4 Nf6 3.g3 e6 5.Nf3 d5 6.Nc3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 c6 5.Nf3 d5 6.Nc3 *"),
("A90", "Dutch: 2.c4 Nf6 3.g3 e6 5.Nf3 d5 6.Qc2", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 c6 5.Nf3 d5 6.Qc2 *"),
("A90", "Dutch: 2.c4 Nf6 3.g3 e6 5.Nf3 d5 6.Qc2 Bd6", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 c6 5.Nf3 d5 6.Qc2 Bd6 *"),
("A90", "Dutch: 2.c4 Nf6 3.g3 e6 5.Nf3 d5 6.O-O", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 c6 5.Nf3 d5 6.O-O *"),
("A90", "Dutch: 2.c4 Nf6 3.g3 e6 5.Nf3 d5 6.O-O Bd6", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 c6 5.Nf3 d5 6.O-O Bd6 *"),
("A90", "Dutch: 2.c4 Nf6 3.g3 e6 5.Nf3 d5 6.O-O Bd6 7.b3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 c6 5.Nf3 d5 6.O-O Bd6 7.b3 *"),
("A91", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 Be7", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 *"),
("A91", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nh3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nh3 *"),
("A91", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nh3 O-O 6.O-O", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nh3 O-O 6.O-O *"),
("A91", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nh3 O-O 6.O-O d6", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nh3 O-O 6.O-O d6 *"),
("A91", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nh3 O-O 6.O-O d6 7.Nc3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nh3 O-O 6.O-O d6 7.Nc3 *"),
("A91", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nh3 O-O 6.O-O d6 7.Nc3 c6", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nh3 O-O 6.O-O d6 7.Nc3 c6 *"),
("A91", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nh3 O-O 6.O-O d6 7.Nc3 Qe8", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nh3 O-O 6.O-O d6 7.Nc3 Qe8 *"),
("A91", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nc3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nc3 *"),
("A91", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nc3 d5", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nc3 d5 *"),
("A91", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nc3 O-O", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nc3 O-O *"),
("A91", "Dutch: Botvinnik-Bronstein Variation", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nc3 O-O 6.e3 *"),
("A91", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 *"),
("A91", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 d6", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 d6 *"),
("A91", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 d6 6.Nc3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 d6 6.Nc3 *"),
("A91", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 d5", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 d5 *"),
("A91", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 d5 6.O-O", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 d5 6.O-O *"),
("A91", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 d5 6.O-O c6", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 d5 6.O-O c6 *"),
("A92", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O *"),
("A92", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.d5", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.d5 *"),
("A92", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.Nc3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.Nc3 *"),
("A92", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.Nc3 d6", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.Nc3 d6 *"),
("A92", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O *"),
("A92", "Dutch: Alekhine Variation", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O Ne4 *"),
("A92", "Dutch: Alekhine, 7.d5", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O Ne4 7.d5 *"),
("A92", "Dutch: Alekhine, 7.Nbd2", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O Ne4 7.Nbd2 *"),
("A92", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O c6", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O c6 *"),
("A92", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O c6 7.b3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O c6 7.b3 *"),
("A92", "Dutch: 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O c6 7.b3 a5", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O c6 7.b3 a5 *"),
("A92", "Dutch: Stonewall Variation", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 *"),
("A92", "Dutch: Stonewall, 7.Nbd2", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.Nbd2 *"),
("A92", "Dutch: Stonewall, 7.Nbd2 c6", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.Nbd2 c6 *"),
("A92", "Dutch: Stonewall, 7.Qc2", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.Qc2 *"),
("A92", "Dutch: Stonewall, 7.Qc2 c6", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.Qc2 c6 *"),
("A92", "Dutch: Stonewall, 7.Qc2 c6 8.Nbd2", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.Qc2 c6 8.Nbd2 *"),
("A93", "Dutch: Stonewall, Botvinnik Variation", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.b3 *"),
("A93", "Dutch: Stonewall, Botvinnik, 7...b6", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.b3 b6 *"),
("A93", "Dutch: Stonewall, Botvinnik, 7...Nc6", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.b3 Nc6 *"),
("A93", "Dutch: Stonewall, Botvinnik, 7...c6", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.b3 c6 *"),
("A93", "Dutch: Stonewall, Botvinnik, 8.Bb2", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.b3 c6 8.Bb2 *"),
("A93", "Dutch: Stonewall, Botvinnik, 8.Qc2", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.b3 c6 8.Qc2 *"),
("A93", "Dutch: Stonewall, Botvinnik, 8.Qc2 Ne4", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.b3 c6 8.Qc2 Ne4 *"),
("A94", "Dutch: Stonewall, Botvinnik, 8.Ba3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.b3 c6 8.Ba3 *"),
("A94", "Dutch: Stonewall, Botvinnik, 8.Ba3 Bd7", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.b3 c6 8.Ba3 Bd7 *"),
("A94", "Dutch: Stonewall, Botvinnik, 8.Ba3 Nbd7", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.b3 c6 8.Ba3 Nbd7 *"),
("A94", "Dutch: Stonewall, Botvinnik, 8.Ba3 Nbd7 9.Bxe7", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.b3 c6 8.Ba3 Nbd7 9.Bxe7 *"),
("A94", "Dutch: Stonewall, Botvinnik, 8.Ba3 Bxa3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.b3 c6 8.Ba3 Bxa3 *"),
("A94", "Dutch: Stonewall, Botvinnik, 8.Ba3 Bxa3 9.Nxa3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.b3 c6 8.Ba3 Bxa3 9.Nxa3 *"),
("A94", "Dutch: Stonewall, Botvinnik, 8.Ba3 Bxa3 9.Nxa3 Nbd7", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.b3 c6 8.Ba3 Bxa3 9.Nxa3 Nbd7 *"),
("A94", "Dutch: Stonewall, Botvinnik, 8.Ba3 Bxa3 9.Nxa3 Qe7", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.b3 c6 8.Ba3 Bxa3 9.Nxa3 Qe7 *"),
("A94", "Dutch: Stonewall, Botvinnik, 8.Ba3 Bxa3 9.Nxa3 Qe7 10.Qc1", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.b3 c6 8.Ba3 Bxa3 9.Nxa3 Qe7 10.Qc1 *"),
("A92", "Dutch: Stonewall, 7.Nc3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.Nc3 *"),
("A95", "Dutch: Stonewall, 7.Nc3 c6", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.Nc3 c6 *"),
("A95", "Dutch: Stonewall, 7.Nc3 c6 8.Ne5", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.Nc3 c6 8.Ne5 *"),
("A95", "Dutch: Stonewall, 7.Nc3 c6 8.Bf4", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.Nc3 c6 8.Bf4 *"),
("A95", "Dutch: Stonewall, 7.Nc3 c6 8.Bg5", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.Nc3 c6 8.Bg5 *"),
("A95", "Dutch: Stonewall, 7.Nc3 c6 8.Qc2", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.Nc3 c6 8.Qc2 *"),
("A95", "Dutch: Stonewall: Chekhover Variation", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.Nc3 c6 8.Qc2 Qe8 9.Bg5 *"),
("A95", "Dutch: Stonewall, 7.Nc3 c6 8.b3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.Nc3 c6 8.b3 *"),
("A95", "Dutch: Stonewall, 7.Nc3 c6 8.b3 Qe8", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d5 7.Nc3 c6 8.b3 Qe8 *"),
("A96", "Dutch: Classical", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 *"),
("A96", "Dutch: Classical, 7.b3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.b3 *"),
("A96", "Dutch: Classical, 7.b3 a5", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.b3 a5 *"),
("A96", "Dutch: Classical, 7.b3 a5 8.Bb2", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.b3 a5 8.Bb2 *"),
("A96", "Dutch: Classical, 7.b3 Qe8", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.b3 Qe8 *"),
("A96", "Dutch: Classical, 7.b3 Qe8 8.Bb2", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.b3 Qe8 8.Bb2 *"),
("A96", "Dutch: Classical, 7.Nc3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 *"),
("A96", "Dutch: Classical, 7.Nc3 a5", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 a5 *"),
("A96", "Dutch: Classical, 7.Nc3 a5 8.Qc2", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 a5 8.Qc2 *"),
("A96", "Dutch: Classical, 7.Nc3 a5 8.Re1", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 a5 8.Re1 *"),
("A96", "Dutch: Classical, 7.Nc3 a5 8.b3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 a5 8.b3 *"),
("A97", "Dutch: Ilyin-Zhenevsky Variation", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 *"),
("A97", "Dutch: Ilyin-Zhenevsky, 8.b4", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.b4 *"),
("A97", "Dutch: Ilyin-Zhenevsky, Winter Variation", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.Re1 *"),
("A97", "Dutch: Ilyin-Zhenevsky, Winter, 8...Ne4", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.Re1 Ne4 *"),
("A97", "Dutch: Ilyin-Zhenevsky, Winter, 8...Qh5", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.Re1 Qh5 *"),
("A97", "Dutch: Ilyin-Zhenevsky, Winter, 8...Qg6", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.Re1 Qg6 *"),
("A97", "Dutch: Ilyin-Zhenevsky, Winter, 8...Qg6 9.e4", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.Re1 Qg6 9.e4 *"),
("A97", "Dutch: Ilyin-Zhenevsky, Winter, 8...Qg6 9.e4 fxe4", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.Re1 Qg6 9.e4 fxe4 *"),
("A98", "Dutch: Ilyin-Zhenevsky, 8.Qc2", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.Qc2 *"),
("A98", "Dutch: Ilyin-Zhenevsky, 8.Qc2 Nc6", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.Qc2 Nc6 *"),
("A98", "Dutch: Ilyin-Zhenevsky, 8.Qc2 Qh5", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.Qc2 Qh5 *"),
("A99", "Dutch: Ilyin-Zhenevsky, 8.b3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.b3 *"),
("A99", "Dutch: Ilyin-Zhenevsky, 8.b3 c6", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.b3 c6 *"),
("A99", "Dutch: Ilyin-Zhenevsky, 8.b3 Nc6", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.b3 Nc6 *"),
("A99", "Dutch: Ilyin-Zhenevsky, 8.b3 Nbd7", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.b3 Nbd7 *"),
("A99", "Dutch: Ilyin-Zhenevsky, 8.b3 Qh5", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.b3 Qh5 *"),
("A99", "Dutch: Ilyin-Zhenevsky, 8.b3 Qh5 9.Bb2", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.b3 Qh5 9.Bb2 *"),
("A99", "Dutch: Ilyin-Zhenevsky, 8.b3 a5", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.b3 a5 *"),
("A99", "Dutch: Ilyin-Zhenevsky, 8.b3 a5 9.Qc2", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.b3 a5 9.Qc2 *"),
("A99", "Dutch: Ilyin-Zhenevsky, 8.b3 a5 9.Ba3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.b3 a5 9.Ba3 *"),
("A99", "Dutch: Ilyin-Zhenevsky, 8.b3 a5 9.Bb2", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.b3 a5 9.Bb2 *"),
("A99", "Dutch: Ilyin-Zhenevsky, 8.b3 a5 9.Bb2 c6", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.b3 a5 9.Bb2 c6 *"),
("A99", "Dutch: Ilyin-Zhenevsky, 8.b3 a5 9.Bb2 Na6", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.b3 a5 9.Bb2 Na6 *"),
("A99", "Dutch: Ilyin-Zhenevsky, 8.b3 a5 9.Bb2 Qh5", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.b3 a5 9.Bb2 Qh5 *"),
("A99", "Dutch: Ilyin-Zhenevsky, 8.b3 a5 9.Bb2 Qh5 10.e3", "1.d4 f5 2.c4 Nf6 3.g3 e6 4.Bg2 Be7 5.Nf3 O-O 6.O-O d6 7.Nc3 Qe8 8.b3 a5 9.Bb2 Qh5 10.e3 *"),
("B00a", "King's Pawn", "1.e4 *"),
("B00a", "King's Pawn: Hippopotamus Defence", "1.e4 Nh6 2.d4 g6 3.c4 f6 *"),
("B00a", "King's Pawn: Fred", "1.e4 f5 *"),
("B00b", "Reversed Grob (Borg/Basman Defence)", "1.e4 g5 *"),
("B00b", "Reversed Grob (Borg/Basman Defence)", "1.e4 g5 2.d4 *"),
("B00c", "St. George Defence", "1.e4 a6 *"),
("B00c", "St. George Defence", "1.e4 a6 2.d4 *"),
("B00c", "Basman's Creepy-Crawly System (as Black)", "1.e4 a6 2.d4 h6 *"),
("B00d", "Owen Defence", "1.e4 b6 *"),
("B00d", "Owen Defence", "1.e4 b6 2.d4 *"),
("B00e", "Owen Defence: French", "1.e4 b6 2.d4 e6 *"),
("B00f", "Owen Defence: 2.d4 Bb7", "1.e4 b6 2.d4 Bb7 *"),
("B00f", "Owen Defence: Naselwaus Gambit", "1.e4 b6 2.d4 Bb7 3.Bg5 *"),
("B00f", "Owen Defence: Smith Gambit", "1.e4 b6 2.d4 Bb7 3.Nf3 *"),
("B00g", "Owen Defence: 3.Bd3", "1.e4 b6 2.d4 Bb7 3.Bd3 *"),
("B00g", "Owen Defence: Matinovsky Gambit", "1.e4 b6 2.d4 Bb7 3.Bd3 f5 *"),
("B00h", "Owen Defence: 3.Bd3 Nf6", "1.e4 b6 2.d4 Bb7 3.Bd3 Nf6 *"),
("B00i", "Owen Defence: 3.Bd3 e6", "1.e4 b6 2.d4 Bb7 3.Bd3 e6 *"),
("B00j", "Owen Defence: 3.Bd3 e6 4.Nf3", "1.e4 b6 2.d4 Bb7 3.Bd3 e6 4.Nf3 *"),
("B00k", "Owen Defence: 3.Bd3 e6 4.Nf3 c5", "1.e4 b6 2.d4 Bb7 3.Bd3 e6 4.Nf3 c5 *"),
("B00k", "Owen Defence: 3.Bd3 e6 4.Nf3 c5 5.c3", "1.e4 b6 2.d4 Bb7 3.Bd3 e6 4.Nf3 c5 5.c3 *"),
("B00l", "Nimzowitsch Defence", "1.e4 Nc6 *"),
("B00l", "Nimzowitsch Defence: Wheeler Gambit", "1.e4 Nc6 2.b4 *"),
("B00m", "Nimzowitsch Defence: 2.Nc3", "1.e4 Nc6 2.Nc3 *"),
("B00m", "Nimzowitsch Defence: 2.Nc3 Nf6", "1.e4 Nc6 2.Nc3 Nf6 *"),
("B00m", "Nimzowitsch Defence: 2.Nc3 e6", "1.e4 Nc6 2.Nc3 e6 *"),
("B00n", "Nimzowitsch Defence: 2.Nf3", "1.e4 Nc6 2.Nf3 *"),
("B00n", "Nimzowitsch Defence: Colorado Counter", "1.e4 Nc6 2.Nf3 f5 *"),
("B00o", "Nimzowitsch Defence: 2.Nf3 d6", "1.e4 Nc6 2.Nf3 d6 *"),
("B00o", "Nimzowitsch Defence: 2.Nf3 d6 3.d4", "1.e4 Nc6 2.Nf3 d6 3.d4 *"),
("B00o", "Nimzowitsch Defence: 2.Nf3 d6 3.d4 Bg4", "1.e4 Nc6 2.Nf3 d6 3.d4 Bg4 *"),
("B00p", "Nimzowitsch Defence: 2.Nf3 d6 3.d4 Nf6", "1.e4 Nc6 2.Nf3 d6 3.d4 Nf6 *"),
("B00q", "Nimzowitsch Defence: 2.Nf3 d6 3.d4 Nf6 4.Nc3", "1.e4 Nc6 2.Nf3 d6 3.d4 Nf6 4.Nc3 *"),
("B00q", "Nimzowitsch Defence: 2.Nf3 d6 3.d4 Nf6 4.Nc3 g6", "1.e4 Nc6 2.Nf3 d6 3.d4 Nf6 4.Nc3 g6 *"),
("B00r", "Nimzowitsch Defence: 2.Nf3 d6 3.d4 Nf6 4.Nc3 Bg4", "1.e4 Nc6 2.Nf3 d6 3.d4 Nf6 4.Nc3 Bg4 *"),
("B00r", "Nimzowitsch Defence: 2.Nf3 d6 3.d4 Nf6 4.Nc3 Bg4 5.d5", "1.e4 Nc6 2.Nf3 d6 3.d4 Nf6 4.Nc3 Bg4 5.d5 *"),
("B00r", "Nimzowitsch Defence: 2.Nf3 d6 3.d4 Nf6 4.Nc3 Bg4 5.Bb5", "1.e4 Nc6 2.Nf3 d6 3.d4 Nf6 4.Nc3 Bg4 5.Bb5 *"),
("B00r", "Nimzowitsch Defence: 2.Nf3 d6 3.d4 Nf6 4.Nc3 Bg4 5.Be2", "1.e4 Nc6 2.Nf3 d6 3.d4 Nf6 4.Nc3 Bg4 5.Be2 *"),
("B00s", "Nimzowitsch Defence: 2.Nf3 d6 3.d4 Nf6 4.Nc3 Bg4 5.Be3", "1.e4 Nc6 2.Nf3 d6 3.d4 Nf6 4.Nc3 Bg4 5.Be3 *"),
("B00t", "Nimzowitsch Defence: 2.d4", "1.e4 Nc6 2.d4 *"),
("B00t", "Nimzowitsch Defence: 2.d4 d6", "1.e4 Nc6 2.d4 d6 *"),
("B00t", "Nimzowitsch Defence: 2.d4 d6 3.Nc3", "1.e4 Nc6 2.d4 d6 3.Nc3 *"),
("B00u", "Nimzowitsch Defence: 2.d4 e5", "1.e4 Nc6 2.d4 e5 *"),
("B00u", "Nimzowitsch Defence: 2.d4 e5 3.dxe5", "1.e4 Nc6 2.d4 e5 3.dxe5 *"),
("B00u", "Nimzowitsch Defence: Bielefelder Gambit", "1.e4 Nc6 2.d4 e5 3.dxe5 Bc5 *"),
("B00u", "Nimzowitsch Defence: 2.d4 e5 3.d5", "1.e4 Nc6 2.d4 e5 3.d5 *"),
("B00v", "Nimzowitsch Defence: 2.d4 d5", "1.e4 Nc6 2.d4 d5 *"),
("B00v", "Nimzowitsch Defence: 2.d4 d5 3.exd5", "1.e4 Nc6 2.d4 d5 3.exd5 *"),
("B00v", "Nimzowitsch Defence: Aachen Gambit", "1.e4 Nc6 2.d4 d5 3.exd5 Nb4 *"),
("B00v", "Nimzowitsch Defence: 2.d4 d5 3.exd5 Qxd5", "1.e4 Nc6 2.d4 d5 3.exd5 Qxd5 *"),
("B00v", "Nimzowitsch Defence: Marshall Gambit", "1.e4 Nc6 2.d4 d5 3.exd5 Qxd5 4.Nc3 *"),
("B00v", "Nimzowitsch Defence: 2.d4 d5 3.exd5 Qxd5 4.Nf3", "1.e4 Nc6 2.d4 d5 3.exd5 Qxd5 4.Nf3 *"),
("B00w", "Nimzowitsch Defence: Bogoljubow Variation", "1.e4 Nc6 2.d4 d5 3.Nc3 *"),
("B00w", "Nimzowitsch Defence: Bogoljubow, 3...e5", "1.e4 Nc6 2.d4 d5 3.Nc3 e5 *"),
("B00w", "Nimzowitsch Defence: Bogoljubow, 3...Nf6", "1.e4 Nc6 2.d4 d5 3.Nc3 Nf6 *"),
("B00w", "Nimzowitsch Defence: Bogoljubow, 3...dxe4", "1.e4 Nc6 2.d4 d5 3.Nc3 dxe4 *"),
("B00x", "Nimzowitsch Defence: 2.d4 d5 3.e5", "1.e4 Nc6 2.d4 d5 3.e5 *"),
("B00x", "Nimzowitsch Defence: 2.d4 d5 3.e5 Bf5", "1.e4 Nc6 2.d4 d5 3.e5 Bf5 *"),
("B00x", "Nimzowitsch Defence: 2.d4 d5 3.e5 Bf5 4.c3", "1.e4 Nc6 2.d4 d5 3.e5 Bf5 4.c3 *"),
("B01a", "Scandinavian (Centre Counter)", "1.e4 d5 *"),
// 1.e4 d5 2.Nc3 is listed under A00: 1.Nc3 d5 2.e4
// 1.e4 d5 2.Nf3 is listed under A06: 1.Nf3 d5 2.e4
("B01a", "Scandinavian: 2.d3", "1.e4 d5 2.d3 *"),
("B01a", "Scandinavian: 2.e5", "1.e4 d5 2.e5 *"),
("B01a", "Scandinavian: 2.exd5", "1.e4 d5 2.exd5 *"),
("B01a", "Scandinavian: Boehnke Gambit", "1.e4 d5 2.exd5 e5 *"),
("B01b", "Scandinavian: 2...Qxd5", "1.e4 d5 2.exd5 Qxd5 *"),
("B01b", "Scandinavian: 2...Qxd5 3.Nf3", "1.e4 d5 2.exd5 Qxd5 3.Nf3 *"),
("B01b", "Scandinavian: 2...Qxd5 3.Nf3 Nf6", "1.e4 d5 2.exd5 Qxd5 3.Nf3 Nf6 *"),
("B01b", "Scandinavian: 2...Qxd5 3.d4", "1.e4 d5 2.exd5 Qxd5 3.d4 *"),
// 2...Qxd5 3.d4 Nc6 is B00: Nimzowitsch, 2.d4 d5 3.exd5 Qxd5
("B01b", "Scandinavian: 2...Qxd5 3.d4 Nf6", "1.e4 d5 2.exd5 Qxd5 3.d4 Nf6 *"),
("B01b", "Scandinavian: 2...Qxd5 3.d4 e5", "1.e4 d5 2.exd5 Qxd5 3.d4 e5 *"),
("B01b", "Scandinavian: 2...Qxd5 3.Nc3", "1.e4 d5 2.exd5 Qxd5 3.Nc3 *"),
("B01c", "Scandinavian: 2...Qxd5 3.Nc3 Qd8", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qd8 *"),
("B01d", "Scandinavian: Schiller Defence", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qd6 *"),
("B01d", "Scandinavian: Schiller, 4.d4 Nf6", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qd6 4.d4 Nf6 *"),
("B01d", "Scandinavian: Schiller, Bronstein Variation", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qd6 4.d4 Nf6 5.Nf3 a6 *"),
("B01e", "Scandinavian: 2...Qxd5 3.Nc3 Qa5", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 *"),
("B01e", "Scandinavian, Mieses Gambit", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.b4 *"),
("B01e", "Scandinavian: 2...Qxd5 3.Nc3 Qa5 4.g3", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.g3 *"),
("B01e", "Scandinavian: 2...Qxd5 3.Nc3 Qa5 4.Bc4", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.Bc4 *"),
("B01e", "Scandinavian: 2...Qxd5 3.Nc3 Qa5 4.Bc4 Nf6", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.Bc4 Nf6 *"),
("B01f", "Scandinavian: 2...Qxd5 3.Nc3 Qa5 4.Nf3", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.Nf3 *"),
("B01f", "Scandinavian: 2...Qxd5 3.Nc3 Qa5 4.Nf3 Nf6", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.Nf3 Nf6 *"),
("B01f", "Scandinavian: 2...Qxd5 3.Nc3 Qa5 4.Nf3 Nf6 5.Be2", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.Nf3 Nf6 5.Be2 *"),
("B01g", "Scandinavian: 2...Qxd5 3.Nc3 Qa5 4.d4", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 *"),
("B01g", "Scandinavian: Anderssen Counterattack", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 e5 *"),
("B01g", "Scandinavian: Anderssen Counterattack, Goteborg Variation", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 e5 5.Nf3 *"),
("B01g", "Scandinavian: Anderssen Counterattack, Collijn Variation", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 e5 5.Nf3 Bg4 *"),
("B01g", "Scandinavian: Anderssen Counterattack, 5.dxe5", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 e5 5.dxe5 *"),
("B01h", "Scandinavian: 2...Qxd5 3.Nc3 Qa5 4.d4 c6", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 c6 *"),
("B01h", "Scandinavian: 2...Qxd5 3.Nc3 Qa5 4.d4 c6 5.Bc4", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 c6 5.Bc4 *"),
("B01h", "Scandinavian: 2...Qxd5 3.Nc3 Qa5 4.d4 c6 5.Nf3", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 c6 5.Nf3 *"),
("B01i", "Scandinavian: 2...Qxd5 3.Nc3 Qa5 4.d4 Nf6", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 Nf6 *"),
("B01i", "Scandinavian: 2...Qxd5, 5.Bd2", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 Nf6 5.Bd2 *"),
("B01i", "Scandinavian: 2...Qxd5, 5.Bd2 c6", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 Nf6 5.Bd2 c6 *"),
("B01i", "Scandinavian: 2...Qxd5, 5.Bc4", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 Nf6 5.Bc4 *"),
("B01i", "Scandinavian: 2...Qxd5, 5.Bc4 c6", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 Nf6 5.Bc4 c6 *"),
("B01j", "Scandinavian: 2...Qxd5, 5.Nf3", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 Nf6 5.Nf3 *"),
("B01j", "Scandinavian: 2...Qxd5, 5.Nf3 Bg4", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 Nf6 5.Nf3 Bg4 *"),
("B01k", "Scandinavian: Lasker Variation", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 Nf6 5.Nf3 Bg4 6.h3 *"),
("B01l", "Scandinavian: 2...Qxd5, 5.Nf3 Bf5", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 Nf6 5.Nf3 Bf5 *"),
("B01l", "Scandinavian: Gruenfeld Variation", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 Nf6 5.Nf3 Bf5 6.Ne5 c6 7.g4 *"),
("B01l", "Scandinavian: 2...Qxd5, 5.Nf3 Bf5 6.Bc4", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 Nf6 5.Nf3 Bf5 6.Bc4 *"),
("B01m", "Scandinavian: 2...Qxd5, Main Line", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 Nf6 5.Nf3 c6 *"),
("B01n", "Scandinavian: 2...Qxd5, Main Line, 6.Bc4", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 Nf6 5.Nf3 c6 6.Bc4 *"),
("B01n", "Scandinavian: 2...Qxd5, Main Line, 6.Bc4 Bg4", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 Nf6 5.Nf3 c6 6.Bc4 Bg4 *"),
("B01o", "Scandinavian: 2...Qxd5, Main Line, 6.Bc4 Bf5", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 Nf6 5.Nf3 c6 6.Bc4 Bf5 *"),
("B01p", "Scandinavian: 2...Qxd5, Main Line, 6.Bc4 Bf5 7.Bd2", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 Nf6 5.Nf3 c6 6.Bc4 Bf5 7.Bd2 *"),
("B01p", "Scandinavian: 2...Qxd5, Main Line, 6.Bc4 Bf5 7.Bd2 e6", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 Nf6 5.Nf3 c6 6.Bc4 Bf5 7.Bd2 e6 *"),
("B01p", "Scandinavian: 2...Qxd5, Main Line, 8.Nd5", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 Nf6 5.Nf3 c6 6.Bc4 Bf5 7.Bd2 e6 8.Nd5 *"),
("B01q", "Scandinavian: 2...Qxd5, Main Line, 8.Qe2", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 Nf6 5.Nf3 c6 6.Bc4 Bf5 7.Bd2 e6 8.Qe2 *"),
("B01q", "Scandinavian: 2...Qxd5, Main Line, 8.Qe2 Bb4", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 Nf6 5.Nf3 c6 6.Bc4 Bf5 7.Bd2 e6 8.Qe2 Bb4 *"),
("B01q", "Scandinavian: 2...Qxd5, Main Line, 8.Qe2 Bb4 9.a3", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 Nf6 5.Nf3 c6 6.Bc4 Bf5 7.Bd2 e6 8.Qe2 Bb4 9.a3 *"),
("B01q", "Scandinavian: 2...Qxd5, Main Line, 8.Qe2 Bb4 9.O-O-O", "1.e4 d5 2.exd5 Qxd5 3.Nc3 Qa5 4.d4 Nf6 5.Nf3 c6 6.Bc4 Bf5 7.Bd2 e6 8.Qe2 Bb4 9.O-O-O *"),
("B01r", "Scandinavian: 2...Nf6", "1.e4 d5 2.exd5 Nf6 *"),
("B01r", "Scandinavian: 2...Nf6 3.Bc4", "1.e4 d5 2.exd5 Nf6 3.Bc4 *"),
("B01r", "Scandinavian: 2...Nf6 3.Nf3", "1.e4 d5 2.exd5 Nf6 3.Nf3 *"),
("B01r", "Scandinavian: 2...Nf6 3.Nf3 Nxd5", "1.e4 d5 2.exd5 Nf6 3.Nf3 Nxd5 *"),
// 3.Nc3 here is Alekhine Defence (B02): 1.e4 Nf6 2.Nc3 d5 3.exd5
("B01s", "Scandinavian: 2...Nf6 3.Bb5+", "1.e4 d5 2.exd5 Nf6 3.Bb5+ *"),
("B01s", "Scandinavian: 2...Nf6 3.Bb5+ Nbd7", "1.e4 d5 2.exd5 Nf6 3.Bb5+ Nbd7 *"),
("B01s", "Scandinavian: 2...Nf6 3.Bb5+ Bd7", "1.e4 d5 2.exd5 Nf6 3.Bb5+ Bd7 *"),
("B01s", "Scandinavian: 2...Nf6 3.Bb5+ Bd7 4.Bc4", "1.e4 d5 2.exd5 Nf6 3.Bb5+ Bd7 4.Bc4 *"),
("B01s", "Scandinavian: 2...Nf6 3.Bb5+ Bd7 4.Bc4 Bg4", "1.e4 d5 2.exd5 Nf6 3.Bb5+ Bd7 4.Bc4 Bg4 *"),
("B01s", "Scandinavian: 2...Nf6 3.Bb5+ Bd7 4.Be2", "1.e4 d5 2.exd5 Nf6 3.Bb5+ Bd7 4.Be2 *"),
("B01s", "Scandinavian: 2...Nf6 3.Bb5+ Bd7 4.Be2 Nxd5 5.d4 Bf5", "1.e4 d5 2.exd5 Nf6 3.Bb5+ Bd7 4.Be2 Nxd5 5.d4 Bf5 *"),
("B01t", "Scandinavian: 2...Nf6 3.c4", "1.e4 d5 2.exd5 Nf6 3.c4 *"),
("B01t", "Scandinavian: Scandinavian Gambit", "1.e4 d5 2.exd5 Nf6 3.c4 c6 *"),
("B01t", "Scandinavian: Scandinavian Gambit Accepted", "1.e4 d5 2.exd5 Nf6 3.c4 c6 4.dxc6 *"),
("B01t", "Scandinavian: Icelandic Gambit", "1.e4 d5 2.exd5 Nf6 3.c4 e6 *"),
("B01t", "Scandinavian: Icelandic Gambit Accepted", "1.e4 d5 2.exd5 Nf6 3.c4 e6 4.dxe6 *"),
("B01t", "Scandinavian: Icelandic Gambit Accepted", "1.e4 d5 2.exd5 Nf6 3.c4 e6 4.dxe6 Bxe6 *"),
("B01t", "Scandinavian: Icelandic Gambit, 5.d4", "1.e4 d5 2.exd5 Nf6 3.c4 e6 4.dxe6 Bxe6 5.d4 *"),
("B01t", "Scandinavian: Icelandic Gambit, 5.Nf3", "1.e4 d5 2.exd5 Nf6 3.c4 e6 4.dxe6 Bxe6 5.Nf3 *"),
("B01u", "Scandinavian: 2...Nf6 3.d4", "1.e4 d5 2.exd5 Nf6 3.d4 *"),
("B01u", "Scandinavian: Richter Variation", "1.e4 d5 2.exd5 Nf6 3.d4 g6 *"),
("B01u", "Scandinavian: Portuguese Variation", "1.e4 d5 2.exd5 Nf6 3.d4 Bg4 *"),
("B01u", "Scandinavian: Portuguese, 4.Bb5+", "1.e4 d5 2.exd5 Nf6 3.d4 Bg4 4.Bb5+ *"),
("B01u", "Scandinavian: Portuguese, 4.Nf3", "1.e4 d5 2.exd5 Nf6 3.d4 Bg4 4.Nf3 *"),
("B01u", "Scandinavian: Portuguese, 4.Nf3 Qxd5", "1.e4 d5 2.exd5 Nf6 3.d4 Bg4 4.Nf3 Qxd5 *"),
("B01u", "Scandinavian: Portuguese, 4.Nf3 Qxd5 5.Be2", "1.e4 d5 2.exd5 Nf6 3.d4 Bg4 4.Nf3 Qxd5 5.Be2 *"),
("B01u", "Scandinavian: Portuguese, 4.Nf3 Qxd5 5.Be2 Nc6", "1.e4 d5 2.exd5 Nf6 3.d4 Bg4 4.Nf3 Qxd5 5.Be2 Nc6 *"),
("B01u", "Scandinavian: Portuguese, 4.Be2", "1.e4 d5 2.exd5 Nf6 3.d4 Bg4 4.Be2 *"),
("B01u", "Scandinavian: Portuguese, 4.Be2 Bxe2", "1.e4 d5 2.exd5 Nf6 3.d4 Bg4 4.Be2 Bxe2 *"),
("B01v", "Scandinavian: Portuguese, 4.f3", "1.e4 d5 2.exd5 Nf6 3.d4 Bg4 4.f3 *"),
("B01v", "Scandinavian: Portuguese, 4.f3 Bf5", "1.e4 d5 2.exd5 Nf6 3.d4 Bg4 4.f3 Bf5 *"),
("B01v", "Scandinavian: Portuguese, 4.f3 Bf5 5.c4", "1.e4 d5 2.exd5 Nf6 3.d4 Bg4 4.f3 Bf5 5.c4 *"),
("B01v", "Scandinavian: Portuguese, 4.f3 Bf5 5.Bb5+", "1.e4 d5 2.exd5 Nf6 3.d4 Bg4 4.f3 Bf5 5.Bb5+ *"),
("B01v", "Scandinavian: Portuguese, 4.f3 Bf5 5.Bb5+ Nbd7", "1.e4 d5 2.exd5 Nf6 3.d4 Bg4 4.f3 Bf5 5.Bb5+ Nbd7 *"),
("B01v", "Scandinavian: Portuguese, 4.f3 Bf5 5.Bb5+ Nbd7 6.c4", "1.e4 d5 2.exd5 Nf6 3.d4 Bg4 4.f3 Bf5 5.Bb5+ Nbd7 6.c4 *"),
("B01w", "Scandinavian: Marshall Variation", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 *"),
("B01w", "Scandinavian: Marshall, 4.Nf3", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 4.Nf3 *"),
("B01w", "Scandinavian: Marshall, 4.Nf3 Bf5", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 4.Nf3 Bf5 *"),
("B01w", "Scandinavian: Marshall, 4.Nf3 Bg4", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 4.Nf3 Bg4 *"),
("B01w", "Scandinavian: Marshall, 4.Nf3 Bg4 5.Be2", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 4.Nf3 Bg4 5.Be2 *"),
("B01w", "Scandinavian: Marshall, 4.Nf3 Bg4 5.Be2 e6", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 4.Nf3 Bg4 5.Be2 e6 *"),
("B01x", "Scandinavian: Marshall, 4.Nf3 g6", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 4.Nf3 g6 *"),
("B01x", "Scandinavian: Marshall, 4.Nf3 g6 5.Be2", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 4.Nf3 g6 5.Be2 *"),
("B01x", "Scandinavian: Marshall, 4.Nf3 g6 5.Be2 Bg7", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 4.Nf3 g6 5.Be2 Bg7 *"),
("B01y", "Scandinavian: Marshall, 4.c4", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 4.c4 *"),
("B01y", "Scandinavian: Marshall, 4.c4 Nf6", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 4.c4 Nf6 *"),
("B01y", "Scandinavian: Marshall, 4.c4 Nb6", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 4.c4 Nb6 *"),
("B01y", "Scandinavian: Marshall, 4.c4 Nb6 5.Nc3", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 4.c4 Nb6 5.Nc3 *"),
("B01y", "Scandinavian: Marshall, 4.c4 Nb6 5.Nc3 e5", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 4.c4 Nb6 5.Nc3 e5 *"),
("B01z", "Scandinavian: Marshall, 4.c4 Nb6 5.Nf3", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 4.c4 Nb6 5.Nf3 *"),
("B01z", "Scandinavian: Marshall, 4.c4 Nb6 5.Nf3 Bg4", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 4.c4 Nb6 5.Nf3 Bg4 *"),
("B01z", "Scandinavian: Marshall, 4.c4 Nb6 5.Nf3 Bg4 6.Be2 e6", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 4.c4 Nb6 5.Nf3 Bg4 6.Be2 e6 *"),
("B01z", "Scandinavian: Marshall, 4.c4 Nb6 5.Nf3 Bg4 6.c5", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 4.c4 Nb6 5.Nf3 Bg4 6.c5 *"),
("B01z", "Scandinavian: Marshall, 4.c4 Nb6 5.Nf3 g6", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 4.c4 Nb6 5.Nf3 g6 *"),
("B01z", "Scandinavian: Marshall, 4.c4 Nb6 5.Nf3 g6 6.Be2", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 4.c4 Nb6 5.Nf3 g6 6.Be2 *"),
("B01z", "Scandinavian: Marshall, 4.c4 Nb6 5.Nf3 g6 6.Be2 Bg7", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 4.c4 Nb6 5.Nf3 g6 6.Be2 Bg7 *"),
("B01z", "Scandinavian: Marshall, 4.c4 Nb6 5.Nf3 g6 6.h3", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 4.c4 Nb6 5.Nf3 g6 6.h3 *"),
("B01z", "Scandinavian: Marshall, 4.c4 Nb6 5.Nf3 g6 6.h3 Bg7", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 4.c4 Nb6 5.Nf3 g6 6.h3 Bg7 *"),
("B01z", "Scandinavian: Marshall, 4.c4 Nb6 5.Nf3 g6 6.Nc3", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 4.c4 Nb6 5.Nf3 g6 6.Nc3 *"),
("B01z", "Scandinavian: Marshall, 4.c4 Nb6 5.Nf3 g6 6.Nc3 Bg7", "1.e4 d5 2.exd5 Nf6 3.d4 Nxd5 4.c4 Nb6 5.Nf3 g6 6.Nc3 Bg7 *"),
("B02a", "Alekhine Defence", "1.e4 Nf6 *"),
("B02a", "Alekhine: Krejcik Variation", "1.e4 Nf6 2.Bc4 *"),
("B02b", "Alekhine: Maroczy Variation", "1.e4 Nf6 2.d3 *"),
("B02c", "Alekhine: Scandinavian Variation", "1.e4 Nf6 2.Nc3 *"),
("B02c", "Alekhine: Scandinavian Variation", "1.e4 Nf6 2.Nc3 d5 *"),
("B02d", "Alekhine: Scandinavian, 3.e5", "1.e4 Nf6 2.Nc3 d5 3.e5 *"),
("B02e", "Alekhine: Scandinavian, 3.e5 Ne4", "1.e4 Nf6 2.Nc3 d5 3.e5 Ne4 *"),
("B02f", "Alekhine: Scandinavian, 3.e5 Nfd7", "1.e4 Nf6 2.Nc3 d5 3.e5 Nfd7 *"),
("B02f", "Alekhine: Spielmann Gambit", "1.e4 Nf6 2.Nc3 d5 3.e5 Nfd7 4.e6 *"),
("B02g", "Alekhine: Scandinavian, Exchange", "1.e4 Nf6 2.Nc3 d5 3.exd5 *"),
("B02g", "Alekhine: Scandinavian, Geschev Gambit", "1.e4 Nf6 2.Nc3 d5 3.exd5 c6 *"),
("B02g", "Alekhine: Scandinavian, Exchange", "1.e4 Nf6 2.Nc3 d5 3.exd5 Nxd5 *"),
("B02g", "Alekhine: Scandinavian, Exchange, 4.Nf3", "1.e4 Nf6 2.Nc3 d5 3.exd5 Nxd5 4.Nf3 *"),
("B02h", "Alekhine: Scandinavian, Exchange, 4.Nxd5", "1.e4 Nf6 2.Nc3 d5 3.exd5 Nxd5 4.Nxd5 *"),
("B02h", "Alekhine: Scandinavian, Exchange, 4.Nxd5", "1.e4 Nf6 2.Nc3 d5 3.exd5 Nxd5 4.Nxd5 Qxd5 *"),
("B02i", "Alekhine: Scandinavian, Exchange, 4.Bc4", "1.e4 Nf6 2.Nc3 d5 3.exd5 Nxd5 4.Bc4 *"),
("B02j", "Alekhine: Scandinavian, Exchange, 4.Bc4 e6", "1.e4 Nf6 2.Nc3 d5 3.exd5 Nxd5 4.Bc4 e6 *"),
("B02k", "Alekhine: Scandinavian, Exchange, 4.Bc4 Nb6", "1.e4 Nf6 2.Nc3 d5 3.exd5 Nxd5 4.Bc4 Nb6 *"),
("B02l", "Alekhine: 2.e5", "1.e4 Nf6 2.e5 *"),
("B02l", "Alekhine: Mokele Mbembe (Buecker) Variation", "1.e4 Nf6 2.e5 Ne4 *"),
("B02l", "Alekhine: Brooklyn Defence (Retreat Variation)", "1.e4 Nf6 2.e5 Ng8 *"),
("B02m", "Alekhine: 2.e5 Nd5", "1.e4 Nf6 2.e5 Nd5 *"),
("B02m", "Alekhine: Welling Variation", "1.e4 Nf6 2.e5 Nd5 3.b3 *"),
("B02n", "Alekhine: 3.Bc4", "1.e4 Nf6 2.e5 Nd5 3.Bc4 *"),
("B02n", "Alekhine: Kmoch Variation", "1.e4 Nf6 2.e5 Nd5 3.Bc4 Nb6 4.Bb3 c5 5.d3 *"),
("B02o", "Alekhine: Saemisch Attack", "1.e4 Nf6 2.e5 Nd5 3.Nc3 *"),
("B02p", "Alekhine: Saemisch Attack, 3...Nxc3", "1.e4 Nf6 2.e5 Nd5 3.Nc3 Nxc3 *"),
("B02q", "Alekhine: Saemisch Attack, 3...Nxc3 4.dxc3", "1.e4 Nf6 2.e5 Nd5 3.Nc3 Nxc3 4.dxc3 *"),
("B02r", "Alekhine: Chase Variation", "1.e4 Nf6 2.e5 Nd5 3.c4 *"),
("B02r", "Alekhine: Chase Variation", "1.e4 Nf6 2.e5 Nd5 3.c4 Nb6 *"),
("B02r", "Alekhine: Chase, Steiner Variation", "1.e4 Nf6 2.e5 Nd5 3.c4 Nb6 4.b3 *"),
("B02s", "Alekhine: Two Pawns (Lasker) Attack", "1.e4 Nf6 2.e5 Nd5 3.c4 Nb6 4.c5 *"),
("B02t", "Alekhine: Two Pawns Attack, 5.Nc3", "1.e4 Nf6 2.e5 Nd5 3.c4 Nb6 4.c5 Nd5 5.Nc3 *"),
("B02u", "Alekhine: Two Pawns Attack, 5.Bc4", "1.e4 Nf6 2.e5 Nd5 3.c4 Nb6 4.c5 Nd5 5.Bc4 *"),
("B02u", "Alekhine: Two Pawns Attack, Mikenas Variation", "1.e4 Nf6 2.e5 Nd5 3.c4 Nb6 4.c5 Nd5 5.Bc4 e6 6.Nc3 d6 *"),
("B03a", "Alekhine: 3.d4", "1.e4 Nf6 2.e5 Nd5 3.d4 *"),
("B03a", "Alekhine: O'Sullivan Gambit", "1.e4 Nf6 2.e5 Nd5 3.d4 b5 *"),
("B03b", "Alekhine: 3.d4 d6", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 *"),
("B03c", "Alekhine: 3.d4 d6 4.exd6", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.exd6 *"),
("B03d", "Alekhine: Balogh Variation", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Bc4 *"),
("B03e", "Alekhine: 4.c4", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.c4 *"),
("B03e", "Alekhine: 4.c4 Nb6", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.c4 Nb6 *"),
("B03f", "Alekhine: 4.c4 Nb6 5.Nf3", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.c4 Nb6 5.Nf3 *"),
("B03g", "Alekhine: Four Pawns Attack", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.c4 Nb6 5.f4 *"),
("B03g", "Alekhine: Four Pawns Attack, Planinc Variation", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.c4 Nb6 5.f4 g5 *"),
("B03h", "Alekhine: Four Pawns Attack, Trifunovic Variation", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.c4 Nb6 5.f4 Bf5 *"),
("B03i", "Alekhine: Four Pawns Attack, Fianchetto Variation", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.c4 Nb6 5.f4 g6 *"),
("B03j", "Alekhine: Four Pawns Attack, 5...dxe5", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.c4 Nb6 5.f4 dxe5 *"),
("B03j", "Alekhine: Four Pawns Attack, 5...dxe5", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.c4 Nb6 5.f4 dxe5 6.fxe5 *"),
("B03k", "Alekhine: Four Pawns Attack, 6...c5", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.c4 Nb6 5.f4 dxe5 6.fxe5 c5 *"),
("B03l", "Alekhine: Four Pawns Attack, 6...Bf5", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.c4 Nb6 5.f4 dxe5 6.fxe5 Bf5 *"),
("B03l", "Alekhine: Four Pawns Attack, Fahrni Variation", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.c4 Nb6 5.f4 dxe5 6.fxe5 Bf5 7.Nc3 e6 8.Nf3 Bb4 9.Bd3 *"),
("B03l", "Alekhine: Four Pawns Attack, Korchnoi Variation", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.c4 Nb6 5.f4 dxe5 6.fxe5 Bf5 7.Nc3 e6 8.Nf3 Be7 9.Be2 O-O 10.O-O f6 *"),
("B03m", "Alekhine: Four Pawns Attack, 6...Nc6", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.c4 Nb6 5.f4 dxe5 6.fxe5 Nc6 *"),
("B03m", "Alekhine: Four Pawns Attack, Ilyin-Zhenevsky Variation", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.c4 Nb6 5.f4 dxe5 6.fxe5 Nc6 7.Nf3 Bg4 8.e6 fxe6 9.c5 *"),
("B03n", "Alekhine: Four Pawns Attack, 7.Be3", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.c4 Nb6 5.f4 dxe5 6.fxe5 Nc6 7.Be3 *"),
("B03o", "Alekhine: Exchange Variation", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.c4 Nb6 5.exd6 *"),
("B03o", "Alekhine: Exchange, 5...exd6", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.c4 Nb6 5.exd6 exd6 *"),
("B03p", "Alekhine: Exchange, 5...exd6 6.Nc3", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.c4 Nb6 5.exd6 exd6 6.Nc3 *"),
("B03q", "Alekhine: Exchange, 5...exd6 6.Nc3 Be7", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.c4 Nb6 5.exd6 exd6 6.Nc3 Be7 *"),
("B03r", "Alekhine: Exchange, 5...cxd6", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.c4 Nb6 5.exd6 cxd6 *"),
("B03s", "Alekhine: Exchange, 5...cxd6 6.Nc3", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.c4 Nb6 5.exd6 cxd6 6.Nc3 *"),
("B03t", "Alekhine: Exchange, 5...cxd6 6.Nc3 g6 7.Be3", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.c4 Nb6 5.exd6 cxd6 6.Nc3 g6 7.Be3 *"),
("B04a", "Alekhine: Modern Variation", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 *"),
("B04b", "Alekhine: Modern, 4...Nc6", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Nc6 *"),
("B04c", "Alekhine: Modern, 4...c6", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 c6 *"),
("B04d", "Alekhine: Modern, 4...c6 5.c4", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 c6 5.c4 *"),
("B04e", "Alekhine: Modern, Schmid Variation", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Nb6 *"),
("B04f", "Alekhine: Modern, Larsen Variation", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 dxe5 *"),
("B04g", "Alekhine: Modern, Larsen, 5.Nxe5 g6", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 dxe5 5.Nxe5 g6 *"),
("B04h", "Alekhine: Modern, Larsen, 5.Nxe5 g6 6.Bc4", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 dxe5 5.Nxe5 g6 6.Bc4 *"),
("B04i", "Alekhine: Modern, Larsen, 5.Nxe5 g6 6.Bc4 c6 7.O-O", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 dxe5 5.Nxe5 g6 6.Bc4 c6 7.O-O *"),
("B04j", "Alekhine: Modern, Fianchetto Variation", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 g6 *"),
("B04k", "Alekhine: Modern, Fianchetto, 5.Bc4", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 g6 5.Bc4 *"),
("B04l", "Alekhine: Modern, Fianchetto, 5.Bc4 Nb6", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 g6 5.Bc4 Nb6 *"),
("B04m", "Alekhine: Modern, Fianchetto, 5.Bc4 Nb6 6.Bb3 Bg7", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 g6 5.Bc4 Nb6 6.Bb3 Bg7 *"),
("B04n", "Alekhine: Modern, Fianchetto, Keres Variation", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 g6 5.Bc4 Nb6 6.Bb3 Bg7 7.a4 *"),
("B04o", "Alekhine: Modern, Fianchetto, 5.Bc4 Nb6 6.Bb3 Bg7 7.Ng5", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 g6 5.Bc4 Nb6 6.Bb3 Bg7 7.Ng5 *"),
("B05a", "Alekhine: Modern, 4...Bg4", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 *"),
("B05b", "Alekhine: Modern, Panov Variation", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.h3 *"),
("B05c", "Alekhine: Modern, Alekhine Variation", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.c4 *"),
("B05c", "Alekhine: Modern, Alekhine Variation", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.c4 Nb6 *"),
("B05d", "Alekhine: Modern, Alekhine Variation, 6.exd6", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.c4 Nb6 6.exd6 *"),
("B05d", "Alekhine: Modern, Alekhine Variation, 6.exd6 cxd6", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.c4 Nb6 6.exd6 cxd6 *"),
("B05e", "Alekhine: Modern, Alekhine Variation, 6.exd6 exd6", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.c4 Nb6 6.exd6 exd6 *"),
("B05f", "Alekhine: Modern, 5.Be2", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.Be2 *"),
("B05g", "Alekhine: Modern, 5.Be2 Nc6", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.Be2 Nc6 *"),
("B05h", "Alekhine: Modern, Flohr Variation", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.Be2 c6 *"),
("B05i", "Alekhine: Modern, Flohr, 6.c4", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.Be2 c6 6.c4 *"),
("B05j", "Alekhine: Modern, Flohr, 6.O-O", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.Be2 c6 6.O-O *"),
("B05k", "Alekhine: Modern, 5.Be2 e6", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.Be2 e6 *"),
("B05l", "Alekhine: Modern, 5.Be2 e6 6.O-O", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.Be2 e6 6.O-O *"),
("B05m", "Alekhine: Modern, Main Line", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.Be2 e6 6.O-O Be7 7.c4 *"),
("B05n", "Alekhine: Modern, Main Line, 8.h3", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.Be2 e6 6.O-O Be7 7.c4 Nb6 8.h3 *"),
("B05n", "Alekhine: Modern, Main Line, 8.h3 Bh5", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.Be2 e6 6.O-O Be7 7.c4 Nb6 8.h3 Bh5 *"),
("B05o", "Alekhine: Modern, Main Line, 8.h3 Bh5 9.Nc3", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.Be2 e6 6.O-O Be7 7.c4 Nb6 8.h3 Bh5 9.Nc3 *"),
("B05o", "Alekhine: Modern, Main Line, 8.h3 Bh5 9.Nc3 O-O", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.Be2 e6 6.O-O Be7 7.c4 Nb6 8.h3 Bh5 9.Nc3 O-O *"),
("B05o", "Alekhine: Modern, Main Line, 8.h3 Bh5 9.Nc3 O-O 10.Be3", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.Be2 e6 6.O-O Be7 7.c4 Nb6 8.h3 Bh5 9.Nc3 O-O 10.Be3 *"),
("B05p", "Alekhine: Modern, Main Line, 8.h3 Bh5 9.Nc3 O-O 10.Be3 d5", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.Be2 e6 6.O-O Be7 7.c4 Nb6 8.h3 Bh5 9.Nc3 O-O 10.Be3 d5 *"),
("B05q", "Alekhine: Modern, Main Line, 8.exd6", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.Be2 e6 6.O-O Be7 7.c4 Nb6 8.exd6 *"),
("B05r", "Alekhine: Modern, Main Line, 8.Nc3", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.Be2 e6 6.O-O Be7 7.c4 Nb6 8.Nc3 *"),
("B05s", "Alekhine: Modern, Main Line, 8.Nc3 O-O 9.Be3", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.Be2 e6 6.O-O Be7 7.c4 Nb6 8.Nc3 O-O 9.Be3 *"),
("B05t", "Alekhine: Modern, Main Line, 8.Nc3 O-O 9.Be3 Nc6", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.Be2 e6 6.O-O Be7 7.c4 Nb6 8.Nc3 O-O 9.Be3 Nc6 *"),
("B05u", "Alekhine: Modern, Main Line, 8.Nc3 O-O 9.Be3 Nc6 10.exd6 cxd6", "1.e4 Nf6 2.e5 Nd5 3.d4 d6 4.Nf3 Bg4 5.Be2 e6 6.O-O Be7 7.c4 Nb6 8.Nc3 O-O 9.Be3 Nc6 10.exd6 cxd6 *"),
("B06a", "Modern", "1.e4 g6 *"),
("B06b", "Modern", "1.e4 g6 2.Nc3 *"),
("B06b", "Modern", "1.e4 g6 2.Nc3 d6 *"),
("B06b", "Modern", "1.e4 g6 2.Nc3 Bg7 *"),
("B06c", "Modern", "1.e4 g6 2.Nc3 Bg7 3.f4 d6 *"),
("B06c", "Modern", "1.e4 g6 2.f4 *"),
("B06c", "Modern", "1.e4 g6 2.f4 d6 *"),
("B06c", "Modern", "1.e4 g6 2.f4 Bg7 3.Nf3 *"),
("B06c", "Modern", "1.e4 g6 2.f4 Bg7 3.Nf3 d6 *"),
("B06d", "Modern", "1.e4 g6 2.d4 *"),
("B06d", "Modern: Norwegian Defence", "1.e4 g6 2.d4 Nf6 *"),
("B06d", "Modern: Norwegian Defence", "1.e4 g6 2.d4 Nf6 3.e5 Nh5 4.g4 Ng7 *"),
("B06d", "Modern", "1.e4 g6 2.d4 c6 *"),
("B06e", "Modern", "1.e4 g6 2.d4 d6 *"),
("B06f", "Modern", "1.e4 g6 2.d4 Bg7 *"),
("B06f", "Modern: 3.Bc4", "1.e4 g6 2.d4 Bg7 3.Bc4 *"),
("B06g", "Modern: 3.c3", "1.e4 g6 2.d4 Bg7 3.c3 *"),
("B06g", "Modern: 3.c3", "1.e4 g6 2.d4 Bg7 3.c3 d6 *"),
("B06g", "Modern: c3 Pterodactyl", "1.e4 g6 2.d4 Bg7 3.c3 d6 4.f4 c5 *"),
("B06h", "Modern: Geller System", "1.e4 g6 2.d4 Bg7 3.c3 d6 4.Nf3 *"),
("B06i", "Modern: 3.Nf3", "1.e4 g6 2.d4 Bg7 3.Nf3 *"),
("B06j", "Modern: 3.Nf3 d6", "1.e4 g6 2.d4 Bg7 3.Nf3 d6 *"),
("B06k", "Modern: 3.Nf3 d6 4.Bc4", "1.e4 g6 2.d4 Bg7 3.Nf3 d6 4.Bc4 *"),
("B06l", "Modern: 3.Nc3", "1.e4 g6 2.d4 Bg7 3.Nc3 *"),
("B06l", "Modern: Mittenberger Gambit", "1.e4 g6 2.d4 Bg7 3.Nc3 d5 *"),
("B06l", "Modern: Nc3 Pterodactyl", "1.e4 g6 2.d4 Bg7 3.Nc3 c5 *"),
("B06m", "Modern: 3.Nc3", "1.e4 g6 2.d4 Bg7 3.Nc3 d6 *"),
("B06n", "Modern: 3.Nc3 d6 4.Bc4", "1.e4 g6 2.d4 Bg7 3.Nc3 d6 4.Bc4 *"),
("B06n", "Modern: 3.Nc3 d6 4.Bc4", "1.e4 g6 2.d4 Bg7 3.Nc3 d6 4.Bc4 c6 *"),
("B06n", "Modern: 3.Nc3 d6 4.Bc4", "1.e4 g6 2.d4 Bg7 3.Nc3 d6 4.Bc4 c6 5.Qf3 *"),
("B06n", "Modern: 3.Nc3 d6 4.Bc4", "1.e4 g6 2.d4 Bg7 3.Nc3 d6 4.Bc4 c6 5.Qf3 Nf6 *"),
("B06n", "Modern: 3.Nc3 d6 4.Bc4", "1.e4 g6 2.d4 Bg7 3.Nc3 d6 4.Bc4 c6 5.Qf3 e6 *"),
("B06o", "Modern: 3.Nc3 d6 4.Be3", "1.e4 g6 2.d4 Bg7 3.Nc3 d6 4.Be3 *"),
("B06p", "Modern: 3.Nc3 d6 4.Be3 a6", "1.e4 g6 2.d4 Bg7 3.Nc3 d6 4.Be3 a6 *"),
("B06q", "Modern: 3.Nc3 d6 4.Be3 c6", "1.e4 g6 2.d4 Bg7 3.Nc3 d6 4.Be3 c6 *"),
("B06r", "Modern: Two Knights Variation", "1.e4 g6 2.d4 Bg7 3.Nc3 d6 4.Nf3 *"),
("B06r", "Modern: Two Knights, Suttles Variation", "1.e4 g6 2.d4 Bg7 3.Nc3 d6 4.Nf3 c6 *"),
("B06r", "Modern: Two Knights, Suttles, Tal Gambit", "1.e4 g6 2.d4 Bg7 3.Nc3 d6 4.Nf3 c6 5.Bg5 Qb6 6.Qd2 *"),
("B06s", "Modern: Pseudo-Austrian Attack", "1.e4 g6 2.d4 Bg7 3.Nc3 d6 4.f4 *"),
("B06s", "Modern: Pseudo-Austrian Attack", "1.e4 g6 2.d4 Bg7 3.Nc3 d6 4.f4 c6 *"),
("B06t", "Modern: Pseudo-Austrian Attack", "1.e4 g6 2.d4 Bg7 3.Nc3 d6 4.f4 c6 5.Nf3 *"),
("B06t", "Modern: Pseudo-Austrian Attack", "1.e4 g6 2.d4 Bg7 3.Nc3 d6 4.f4 Nc6 *"),
("B06u", "Modern: Three Pawns Attack", "1.e4 g6 2.d4 Bg7 3.f4 *"),
("B07a", "Pirc", "1.e4 d6 *"),
("B07a", "Pirc", "1.e4 d6 2.g3 *"),
("B07a", "Pirc", "1.e4 d6 2.c4 *"),
("B07a", "Pirc: 2.Nc3", "1.e4 d6 2.Nc3 *"),
("B07a", "Pirc: 2.Nc3", "1.e4 d6 2.Nc3 Nf6 *"),
("B07b", "Pirc: 2.Nf3", "1.e4 d6 2.Nf3 *"),
("B07b", "Pirc/Reti: Wade Defence", "1.e4 d6 2.Nf3 Bg4 *"),
("B07b", "Pirc: 2.Nf3 Nf6", "1.e4 d6 2.Nf3 Nf6 *"),
("B07b", "Pirc: 2.Nf3 Nf6 3.Nc3", "1.e4 d6 2.Nf3 Nf6 3.Nc3 *"),
("B07c", "Pirc: 2.f4", "1.e4 d6 2.f4 *"),
("B07c", "Pirc: 2.f4 Nf6", "1.e4 d6 2.f4 Nf6 *"),
("B07c", "Pirc: 2.f4 Nf6 3.Nc3", "1.e4 d6 2.f4 Nf6 3.Nc3 *"),
("B07c", "Pirc: 2.f4 Nf6 3.Nc3 g6 4.Nf3 Bg7", "1.e4 d6 2.f4 Nf6 3.Nc3 g6 4.Nf3 Bg7 *"),
("B07d", "Pirc: 2.d4", "1.e4 d6 2.d4 *"),
("B07d", "Pirc: 2.d4 c6", "1.e4 d6 2.d4 c6 *"),
("B07d", "Pirc: Lengfellner System", "1.e4 d6 2.d4 e6 *"),
("B07d", "Pirc: 2.d4 Nd7", "1.e4 d6 2.d4 Nd7 *"),
("B07d", "Pirc: 2.d4 Nf6", "1.e4 d6 2.d4 Nf6 *"),
("B07d", "Pirc: Roscher Gambit", "1.e4 d6 2.d4 Nf6 3.Nf3 *"),
("B07d", "Pirc: 3.Nbd2", "1.e4 d6 2.d4 Nf6 3.Nbd2 *"),
("B07d", "Pirc: 3.Nbd2 g6", "1.e4 d6 2.d4 Nf6 3.Nbd2 g6 *"),
("B07d", "Pirc: 3.Nbd2 g6", "1.e4 d6 2.d4 Nf6 3.Nbd2 g6 4.c3 Bg7 *"),
("B07e", "Pirc: 3.f3", "1.e4 d6 2.d4 Nf6 3.f3 *"),
("B07f", "Pirc: 3.f3 e5", "1.e4 d6 2.d4 Nf6 3.f3 e5 *"),
("B07f", "Pirc: 3.f3 e5 Queenswap", "1.e4 d6 2.d4 Nf6 3.f3 e5 4.dxe5 dxe5 5.Qxd8+ *"),
("B07f", "Pirc: 3.f3 e5 4.d5", "1.e4 d6 2.d4 Nf6 3.f3 e5 4.d5 *"),
("B07g", "Pirc: 3.f3 g6", "1.e4 d6 2.d4 Nf6 3.f3 g6 *"),
("B07h", "Pirc: 3.Bd3", "1.e4 d6 2.d4 Nf6 3.Bd3 *"),
("B07h", "Pirc: 3.Bd3 e5", "1.e4 d6 2.d4 Nf6 3.Bd3 e5 *"),
("B07h", "Pirc: 3.Bd3 e5 4.c3 d5", "1.e4 d6 2.d4 Nf6 3.Bd3 e5 4.c3 d5 *"),
("B07h", "Pirc: 3.Bd3 e5 4.c3 d5", "1.e4 d6 2.d4 Nf6 3.Bd3 e5 4.c3 d5 5.dxe5 dxe4 *"),
("B07h", "Pirc: 3.Bd3 e5 4.c3 d5", "1.e4 d6 2.d4 Nf6 3.Bd3 e5 4.c3 d5 5.dxe5 Nxe4 *"),
("B07i", "Pirc: 3.Bd3 g6", "1.e4 d6 2.d4 Nf6 3.Bd3 g6 *"),
("B07i", "Pirc: 3.Bd3 g6", "1.e4 d6 2.d4 Nf6 3.Bd3 g6 4.Nf3 *"),
("B07i", "Pirc: 3.Bd3 g6", "1.e4 d6 2.d4 Nf6 3.Bd3 g6 4.Nf3 Bg7 *"),
("B07j", "Pirc: 3.Bd3 g6 4.Nf3 Bg7 5.c3", "1.e4 d6 2.d4 Nf6 3.Bd3 g6 4.Nf3 Bg7 5.c3 *"),
("B07k", "Pirc: 3.Nc3", "1.e4 d6 2.d4 Nf6 3.Nc3 *"),
("B07l", "Pirc: 3.Nc3 Nbd7", "1.e4 d6 2.d4 Nf6 3.Nc3 Nbd7 *"),
("B07m", "Pirc: 3.Nc3 e5", "1.e4 d6 2.d4 Nf6 3.Nc3 e5 *"),
("B07m", "Pirc: 3.Nc3 e5 Queenswap", "1.e4 d6 2.d4 Nf6 3.Nc3 e5 4.dxe5 dxe5 5.Qxd8+ *"),
("B07n", "Pirc: Pytel/Czech", "1.e4 d6 2.d4 Nf6 3.Nc3 c6 *"),
("B07o", "Pirc: Czech, 4.Nf3", "1.e4 d6 2.d4 Nf6 3.Nc3 c6 4.Nf3 *"),
("B07p", "Pirc: Czech, 4.f4", "1.e4 d6 2.d4 Nf6 3.Nc3 c6 4.f4 *"),
("B07p", "Pirc: Czech, 4.f4 Qa5", "1.e4 d6 2.d4 Nf6 3.Nc3 c6 4.f4 Qa5 *"),
("B07q", "Pirc: 3...g6", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 *"),
("B07q", "Pirc: 3...g6", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.h4 *"),
("B07q", "Pirc: 4.Nge2", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nge2 *"),
("B07q", "Pirc: 4.Nge2", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nge2 Bg7 *"),
("B07r", "Pirc: Sveshnikov (4.g3)", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.g3 *"),
("B07r", "Pirc: Sveshnikov (4.g3)", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.g3 Bg7 *"),
("B07r", "Pirc: Sveshnikov (4.g3)", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.g3 Bg7 5.Bg2 c6 *"),
("B07s", "Pirc: Holmov (4.Bc4)", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Bc4 *"),
("B07s", "Pirc: Holmov (4.Bc4)", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Bc4 Bg7 *"),
("B07t", "Pirc: 4.Be3", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Be3 *"),
("B07t", "Pirc: 4.Be3 c6", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Be3 c6 *"),
("B07u", "Pirc: 4.Be3 Bg7", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Be3 Bg7 *"),
("B07u", "Pirc: 4.Be3, 150 Attack", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Be3 Bg7 5.Qd2 *"),
("B07v", "Pirc: 4.Be3, 150 Attack", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Be3 Bg7 5.Qd2 c6 *"),
("B07v", "Pirc: 4.Be3, 150 Attack", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Be3 Bg7 5.Qd2 c6 6.f3 b5 *"),
("B07w", "Pirc: Byrne 4.Bg5", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Bg5 *"),
("B07x", "Pirc: 4.Bg5 Bg7", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Bg5 Bg7 *"),
("B07y", "Pirc: 4.Be2", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Be2 *"),
("B07y", "Pirc: 4.Be2", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Be2 Bg7 *"),
("B07y", "Pirc: Chinese Variation", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Be2 Bg7 5.g4 *"),
("B07z", "Pirc: Bayonet/Mariotti Attack", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Be2 Bg7 5.h4 *"),
// Many of the B08 lines with ..c6 are reached by the Modern (B06) move order
// with 5...Nf6 -- hence all the lines ending with ...c6 here to catch
// transpositions.
("B08a", "Pirc: Classical", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 *"),
("B08a", "Pirc: Classical", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 *"),
("B08b", "Pirc: Classical, 5.h3", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.h3 *"),
("B08b", "Pirc: Classical, 5.h3", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.h3 c6 *"),
("B08b", "Pirc: Classical, 5.h3", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.h3 O-O *"),
("B08c", "Pirc: Classical, Spassky System", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.h3 O-O 6.Be3 *"),
("B08c", "Pirc: Classical, Spassky System, 6...d5", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.h3 O-O 6.Be3 d5 *"),
("B08d", "Pirc: Classical, Spassky System, 6...a6", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.h3 O-O 6.Be3 a6 *"),
("B08e", "Pirc: Classical, Spassky System, 6...c6", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.h3 O-O 6.Be3 c6 *"),
("B08f", "Pirc: Classical, 5.a4", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.a4 *"),
("B08f", "Pirc: Classical, 5.a4 c6", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.a4 c6 *"),
("B08g", "Pirc: Classical, 5.Bc4", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.Bc4 *"),
("B08g", "Pirc: Classical, 5.Bc4 c6", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.Bc4 c6 *"),
("B08h", "Pirc: Classical, 5.Be3", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.Be3 *"),
("B08h", "Pirc: Classical, 5.Be3 a6 6.a4", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.Be3 a6 6.a4 *"),
("B08i", "Pirc: Classical, 5.Be3 c6", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.Be3 c6 *"),
("B08j", "Pirc: Classical, 5.Be2", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.Be2 *"),
("B08j", "Pirc: Classical, 5.Be2 a6 6.a4", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.Be2 a6 6.a4 *"),
("B08k", "Pirc: Classical, 5.Be2 c6", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.Be2 c6 *"),
("B08l", "Pirc: Classical, 5.Be2 O-O", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.Be2 O-O *"),
("B08m", "Pirc: Classical, 5.Be2 O-O 6.Be3", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.Be2 O-O 6.Be3 *"),
("B08n", "Pirc: Classical, 5.Be2 O-O 6.O-O", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.Be2 O-O 6.O-O *"),
("B08o", "Pirc: Classical, 5.Be2 O-O 6.O-O Nc6", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.Be2 O-O 6.O-O Nc6 *"),
("B08p", "Pirc: Classical, 5.Be2 O-O 6.O-O c6", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.Be2 O-O 6.O-O c6 *"),
("B08q", "Pirc: Classical, 5.Be2 O-O 6.O-O c6 7.h3", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.Be2 O-O 6.O-O c6 7.h3 *"),
("B08r", "Pirc: Classical, 5.Be2 O-O 6.O-O c6 7.a4", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.Be2 O-O 6.O-O c6 7.a4 *"),
("B08s", "Pirc: Classical, 5.Be2 O-O 6.O-O c6 7.a4 Nbd7", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.Be2 O-O 6.O-O c6 7.a4 Nbd7 *"),
("B08t", "Pirc: Classical, 5.Be2 O-O 6.O-O Bg4", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.Be2 O-O 6.O-O Bg4 *"),
("B08u", "Pirc: Classical, 5.Be2 O-O 6.O-O Bg4 7.Be3 Nc6 8.Qd2", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.Be2 O-O 6.O-O Bg4 7.Be3 Nc6 8.Qd2 *"),
("B08v", "Pirc: Classical, 5.Be2 O-O 6.O-O Bg4 7.Be3 Nc6 8.Qd2 e5", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.Nf3 Bg7 5.Be2 O-O 6.O-O Bg4 7.Be3 Nc6 8.Qd2 e5 *"),
("B09a", "Pirc: Austrian Attack", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 *"),
("B09b", "Pirc: Austrian Attack", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 *"),
("B09c", "Pirc: Austrian, Ljubojevic Variation", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 5.Bc4 *"),
("B09d", "Pirc: Austrian, 5.Nf3", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 5.Nf3 *"),
("B09e", "Pirc: Austrian, 5...O-O", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 5.Nf3 O-O *"),
("B09f", "Pirc: Austrian, 5...O-O 6.e5", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 5.Nf3 O-O 6.e5 *"),
("B09g", "Pirc: Austrian, 5...O-O 6.Be3", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 5.Nf3 O-O 6.Be3 *"),
("B09h", "Pirc: Austrian, 5...O-O 6.Be2", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 5.Nf3 O-O 6.Be2 *"),
("B09i", "Pirc: Austrian, 5...O-O 6.Bd3", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 5.Nf3 O-O 6.Bd3 *"),
("B09j", "Pirc: Austrian, 5...O-O 6.Bd3 Nc6", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 5.Nf3 O-O 6.Bd3 Nc6 *"),
("B09k", "Pirc: Austrian, 5...O-O 6.Bd3 Nc6 7.e5", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 5.Nf3 O-O 6.Bd3 Nc6 7.e5 *"),
("B09l", "Pirc: Austrian, 5...O-O 6.Bd3 Na6", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 5.Nf3 O-O 6.Bd3 Na6 *"),
("B09m", "Pirc: Austrian, 5...O-O 6.Bd3 Na6 7.O-O c5", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 5.Nf3 O-O 6.Bd3 Na6 7.O-O c5 *"),
("B09m", "Pirc: Austrian, 5...O-O 6.Bd3 Na6 7.O-O c5 8.d5", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 5.Nf3 O-O 6.Bd3 Na6 7.O-O c5 8.d5 *"),
("B09n", "Pirc: Austrian, 5...O-O 6.Bd3 Na6 7.O-O c5 8.d5 Bg4", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 5.Nf3 O-O 6.Bd3 Na6 7.O-O c5 8.d5 Bg4 *"),
("B09o", "Pirc: Austrian, 5...c5", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 5.Nf3 c5 *"),
("B09p", "Pirc: Austrian, 5...c5 6.dxc5", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 5.Nf3 c5 6.dxc5 *"),
("B09q", "Pirc: Austrian, 5...c5 6.dxc5 Qa5 7.Bd3", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 5.Nf3 c5 6.dxc5 Qa5 7.Bd3 *"),
("B09r", "Pirc: Austrian, 5...c5 6.Bb5+", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 5.Nf3 c5 6.Bb5+ *"),
("B09s", "Pirc: Austrian, 5...c5 6.Bb5+", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 5.Nf3 c5 6.Bb5+ Bd7 *"),
("B09t", "Pirc: Austrian, 5...c5 6.Bb5+ Bd7 7.e5", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 5.Nf3 c5 6.Bb5+ Bd7 7.e5 *"),
("B09u", "Pirc: Austrian, 5...c5 6.Bb5+ Bd7 7.e5 Ng4", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 5.Nf3 c5 6.Bb5+ Bd7 7.e5 Ng4 *"),
("B09v", "Pirc: Austrian, 5...c5 6.Bb5+ Bd7 7.e5 Ng4 8.Bxd7+", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 5.Nf3 c5 6.Bb5+ Bd7 7.e5 Ng4 8.Bxd7+ *"),
("B09w", "Pirc: Austrian, 5...c5 6.Bb5+ Bd7 7.e5 Ng4 8.e6", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 5.Nf3 c5 6.Bb5+ Bd7 7.e5 Ng4 8.e6 *"),
("B09w", "Pirc: Austrian, 5...c5, Seirawan Variation", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 5.Nf3 c5 6.Bb5+ Bd7 7.e5 Ng4 8.e6 fxe6 *"),
("B09x", "Pirc: Austrian, 5...c5 6.Bb5+ Bd7 7.e5 Ng4 8.e6 Bxb5", "1.e4 d6 2.d4 Nf6 3.Nc3 g6 4.f4 Bg7 5.Nf3 c5 6.Bb5+ Bd7 7.e5 Ng4 8.e6 Bxb5 *"),
("B10a", "Caro-Kann", "1.e4 c6 *"),
("B10a", "Caro-Kann: 2.Ne2", "1.e4 c6 2.Ne2 *"),
("B10a", "Caro-Kann: 2.f4", "1.e4 c6 2.f4 *"),
("B10b", "Caro-Kann: Breyer Variation", "1.e4 c6 2.d3 *"),
("B10b", "Caro-Kann: Breyer Variation", "1.e4 c6 2.d3 d5 *"),
("B10c", "Caro-Kann: Breyer, 3.Nd2", "1.e4 c6 2.d3 d5 3.Nd2 *"),
("B10d", "Caro-Kann: Breyer, 3.Nd2 g6", "1.e4 c6 2.d3 d5 3.Nd2 g6 *"),
("B10e", "Caro-Kann: Breyer, 3.Nd2 e5", "1.e4 c6 2.d3 d5 3.Nd2 e5 *"),
("B10f", "Caro-Kann: Breyer, Main Line", "1.e4 c6 2.d3 d5 3.Nd2 e5 4.Ngf3 Bd6 5.g3 *"),
("B10g", "Caro-Kann: English Variation", "1.e4 c6 2.c4 *"),
("B10h", "Caro-Kann: English Variation", "1.e4 c6 2.c4 e5 *"),
("B10i", "Caro-Kann: English Variation", "1.e4 c6 2.c4 d5 *"),
("B10i", "Caro-Kann: English Variation", "1.e4 c6 2.c4 d5 3.cxd5 *"),
("B10i", "Caro-Kann: English Variation", "1.e4 c6 2.c4 d5 3.cxd5 cxd5 *"),
("B10i", "Caro-Kann: English Variation", "1.e4 c6 2.c4 d5 3.exd5 *"),
("B10i", "Caro-Kann: English Variation", "1.e4 c6 2.c4 d5 3.exd5 cxd5 *"),
("B10j", "Caro-Kann: English, Exchange", "1.e4 c6 2.c4 d5 3.exd5 cxd5 4.cxd5 *"),
("B10k", "Caro-Kann: English, Exchange, 4...Nf6", "1.e4 c6 2.c4 d5 3.exd5 cxd5 4.cxd5 Nf6 *"),
("B10l", "Caro-Kann: English, Exchange, 4...Nf6 5.Bb5+", "1.e4 c6 2.c4 d5 3.exd5 cxd5 4.cxd5 Nf6 5.Bb5+ *"),
("B10m", "Caro-Kann: English, Exchange, 4...Nf6 5.Nc3", "1.e4 c6 2.c4 d5 3.exd5 cxd5 4.cxd5 Nf6 5.Nc3 *"),
("B10n", "Caro-Kann: English, Exchange, 4...Nf6 5.Nc3 Nxd5", "1.e4 c6 2.c4 d5 3.exd5 cxd5 4.cxd5 Nf6 5.Nc3 Nxd5 *"),
("B10o", "Caro-Kann: 2.Nf3", "1.e4 c6 2.Nf3 *"),
("B10p", "Caro-Kann: 2.Nc3", "1.e4 c6 2.Nc3 *"),
("B10q", "Caro-Kann: 2.Nc3 d5", "1.e4 c6 2.Nc3 d5 *"),
("B10r", "Caro-Kann: Goldman/Spielmann", "1.e4 c6 2.Nc3 d5 3.Qf3 *"),
("B10s", "Caro-Kann: Two Knights Variation", "1.e4 c6 2.Nc3 d5 3.Nf3 *"),
("B10t", "Caro-Kann: Two Knights, 3...dxe4", "1.e4 c6 2.Nc3 d5 3.Nf3 dxe4 *"),
("B10t", "Caro-Kann: Hector Gambit", "1.e4 c6 2.Nc3 d5 3.Nf3 dxe4 4.Ng5 *"),
("B10u", "Caro-Kann: Two Knights, 3...dxe4 4.Nxe4", "1.e4 c6 2.Nc3 d5 3.Nf3 dxe4 4.Nxe4 *"),
("B10v", "Caro-Kann: Two Knights, 3...dxe4 4.Nxe4 Nf6", "1.e4 c6 2.Nc3 d5 3.Nf3 dxe4 4.Nxe4 Nf6 *"),
("B11a", "Caro-Kann: Two Knights, 3...Bg4", "1.e4 c6 2.Nc3 d5 3.Nf3 Bg4 *"),
("B11b", "Caro-Kann: Two Knights, 3...Bg4 4.h3", "1.e4 c6 2.Nc3 d5 3.Nf3 Bg4 4.h3 *"),
("B11b", "Caro-Kann: Two Knights, 3...Bg4 4.h3 Bh5", "1.e4 c6 2.Nc3 d5 3.Nf3 Bg4 4.h3 Bh5 *"),
("B11c", "Caro-Kann: Two Knights, 3...Bg4 4.h3 Bh5, 7.g4 Bg6", "1.e4 c6 2.Nc3 d5 3.Nf3 Bg4 4.h3 Bh5 5.exd5 cxd5 6.Bb5+ Nc6 7.g4 Bg6 *"),
("B11d", "Caro-Kann: Two Knights, 3...Bg4 4.h3 Bxf3", "1.e4 c6 2.Nc3 d5 3.Nf3 Bg4 4.h3 Bxf3 *"),
("B11d", "Caro-Kann: Two Knights, 3...Bg4 4.h3 Bxf3 5.Qxf3", "1.e4 c6 2.Nc3 d5 3.Nf3 Bg4 4.h3 Bxf3 5.Qxf3 *"),
("B11e", "Caro-Kann: Two Knights, 3...Bg4 4.h3 Bxf3 5.Qxf3 dxe4", "1.e4 c6 2.Nc3 d5 3.Nf3 Bg4 4.h3 Bxf3 5.Qxf3 dxe4 *"),
("B11e", "Caro-Kann: Two Knights, 3...Bg4 4.h3 Bxf3 5.Qxf3 dxe4", "1.e4 c6 2.Nc3 d5 3.Nf3 Bg4 4.h3 Bxf3 5.Qxf3 dxe4 6.Nxe4 *"),
("B11f", "Caro-Kann: Two Knights, 3...Bg4 4.h3 Bxf3 5.Qxf3 Nf6", "1.e4 c6 2.Nc3 d5 3.Nf3 Bg4 4.h3 Bxf3 5.Qxf3 Nf6 *"),
("B11g", "Caro-Kann: Two Knights, 3...Bg4 4.h3 Bxf3 5.Qxf3 e6", "1.e4 c6 2.Nc3 d5 3.Nf3 Bg4 4.h3 Bxf3 5.Qxf3 e6 *"),
("B11h", "Caro-Kann: Two Knights, 3...Bg4 4.h3 Bxf3 5.Qxf3 e6 6.d4", "1.e4 c6 2.Nc3 d5 3.Nf3 Bg4 4.h3 Bxf3 5.Qxf3 e6 6.d4 *"),
("B11i", "Caro-Kann: Two Knights, 3...Bg4 4.h3 Bxf3 5.Qxf3 e6 6.g3", "1.e4 c6 2.Nc3 d5 3.Nf3 Bg4 4.h3 Bxf3 5.Qxf3 e6 6.g3 *"),
("B11i", "Caro-Kann: Two Knights, 3...Bg4 4.h3 Bxf3 5.Qxf3 e6 6.g3 Nf6", "1.e4 c6 2.Nc3 d5 3.Nf3 Bg4 4.h3 Bxf3 5.Qxf3 e6 6.g3 Nf6 *"),
("B11j", "Caro-Kann: Two Knights, 3...Bg4 4.h3 Bxf3 5.Qxf3 e6 6.d3", "1.e4 c6 2.Nc3 d5 3.Nf3 Bg4 4.h3 Bxf3 5.Qxf3 e6 6.d3 *"),
("B11k", "Caro-Kann: Two Knights, 3...Bg4 4.h3 Bxf3 5.Qxf3 e6 6.d3 Nd7", "1.e4 c6 2.Nc3 d5 3.Nf3 Bg4 4.h3 Bxf3 5.Qxf3 e6 6.d3 Nd7 *"),
("B11l", "Caro-Kann: Two Knights, 3...Bg4 4.h3 Bxf3 5.Qxf3 e6 6.d3 Nf6", "1.e4 c6 2.Nc3 d5 3.Nf3 Bg4 4.h3 Bxf3 5.Qxf3 e6 6.d3 Nf6 *"),
("B11m", "Caro-Kann: Two Knights, 3...Bg4 4.h3 Bxf3 5.Qxf3 e6 6.d3 Nf6 7.a3", "1.e4 c6 2.Nc3 d5 3.Nf3 Bg4 4.h3 Bxf3 5.Qxf3 e6 6.d3 Nf6 7.a3 *"),
("B11n", "Caro-Kann: Two Knights, 3...Bg4 4.h3 Bxf3 5.Qxf3 e6 6.d3 Nf6 7.Bd2", "1.e4 c6 2.Nc3 d5 3.Nf3 Bg4 4.h3 Bxf3 5.Qxf3 e6 6.d3 Nf6 7.Bd2 *"),
("B12a", "Caro-Kann: 2.d4", "1.e4 c6 2.d4 *"),
("B12a", "Caro-Kann: de Bruycker Defence", "1.e4 c6 2.d4 Na6 *"),
("B12a", "Caro-Kann: Masi Defence", "1.e4 c6 2.d4 Nf6 *"),
("B12b", "Caro-Kann: 2.d4 d5", "1.e4 c6 2.d4 d5 *"),
("B12b", "Caro-Kann: Ulysses Gambit", "1.e4 c6 2.d4 d5 3.Nf3 *"),
("B12b", "Caro-Kann: Ulysses Gambit", "1.e4 c6 2.d4 d5 3.Nf3 dxe4 4.Ng5 *"),
("B12c", "Caro-Kann: Mieses Gambit", "1.e4 c6 2.d4 d5 3.Be3 *"),
("B12d", "Caro-Kann: Maroczy (Fantasy) Variation", "1.e4 c6 2.d4 d5 3.f3 *"),
("B12e", "Caro-Kann: Maroczy (Fantasy), 3...e6", "1.e4 c6 2.d4 d5 3.f3 e6 *"),
("B12f", "Caro-Kann: Maroczy (Fantasy), 3...dxe4", "1.e4 c6 2.d4 d5 3.f3 dxe4 *"),
("B12f", "Caro-Kann: Maroczy (Fantasy), Gambit line", "1.e4 c6 2.d4 d5 3.f3 dxe4 4.fxe4 e5 5.Nf3 exd4 6.Bc4 *"),
("B12g", "Caro-Kann: 3.Nd2", "1.e4 c6 2.d4 d5 3.Nd2 *"),
("B12g", "Caro-Kann: 3.Nd2 dxe4", "1.e4 c6 2.d4 d5 3.Nd2 dxe4 *"),
("B12g", "Caro-Kann: Edinburgh Variation", "1.e4 c6 2.d4 d5 3.Nd2 Qb6 *"),
("B12h", "Caro-Kann: Gurgenidze/Modern: 3.Nd2 g6", "1.e4 c6 2.d4 d5 3.Nd2 g6 *"),
("B12h", "Caro-Kann: Gurgenidze/Modern: 3.Nd2 g6 4.Ngf3", "1.e4 c6 2.d4 d5 3.Nd2 g6 4.Ngf3 *"),
("B12h", "Caro-Kann: Gurgenidze/Modern: 3.Nd2 g6 4.Ngf3 Bg7", "1.e4 c6 2.d4 d5 3.Nd2 g6 4.Ngf3 Bg7 *"),
("B12h", "Caro-Kann: Gurgenidze/Modern: 3.Nd2 g6 4.Ngf3 Bg7 5.h3", "1.e4 c6 2.d4 d5 3.Nd2 g6 4.Ngf3 Bg7 5.h3 *"),
("B12h", "Caro-Kann: Gurgenidze/Modern: 3.Nd2 g6 4.Ngf3 Bg7 5.c3", "1.e4 c6 2.d4 d5 3.Nd2 g6 4.Ngf3 Bg7 5.c3 *"),
("B12i", "Caro-Kann: Advance Variation", "1.e4 c6 2.d4 d5 3.e5 *"),
("B12j", "Caro-Kann: Advance, 3...c5", "1.e4 c6 2.d4 d5 3.e5 c5 *"),
("B12j", "Caro-Kann: Advance, 3...c5 4.dxc5", "1.e4 c6 2.d4 d5 3.e5 c5 4.dxc5 *"),
("B12k", "Caro-Kann: Advance, 3...c5 4.dxc5 Nc6", "1.e4 c6 2.d4 d5 3.e5 c5 4.dxc5 Nc6 *"),
("B12l", "Caro-Kann: Advance, 3...Bf5", "1.e4 c6 2.d4 d5 3.e5 Bf5 *"),
("B12l", "Caro-Kann: Advance, Bayonet Variation", "1.e4 c6 2.d4 d5 3.e5 Bf5 4.g4 *"),
("B12m", "Caro-Kann: Advance, 4.h4", "1.e4 c6 2.d4 d5 3.e5 Bf5 4.h4 *"),
("B12n", "Caro-Kann: Advance, 4.c3", "1.e4 c6 2.d4 d5 3.e5 Bf5 4.c3 *"),
("B12n", "Caro-Kann: Advance, Short Variation", "1.e4 c6 2.d4 d5 3.e5 Bf5 4.c3 e6 5.Be2 *"),
("B12o", "Caro-Kann: Advance, 4.Bd3", "1.e4 c6 2.d4 d5 3.e5 Bf5 4.Bd3 *"),
("B12p", "Caro-Kann: Advance, 4.Nf3", "1.e4 c6 2.d4 d5 3.e5 Bf5 4.Nf3 *"),
("B12p", "Caro-Kann: Advance, 4.Nf3 e6", "1.e4 c6 2.d4 d5 3.e5 Bf5 4.Nf3 e6 *"),
("B12q", "Caro-Kann: Advance, 4.Nf3 e6 5.Be2", "1.e4 c6 2.d4 d5 3.e5 Bf5 4.Nf3 e6 5.Be2 *"),
("B12r", "Caro-Kann: Advance, 4.Nf3 e6 5.Be2 Nd7", "1.e4 c6 2.d4 d5 3.e5 Bf5 4.Nf3 e6 5.Be2 Nd7 *"),
("B12s", "Caro-Kann: Advance, 4.Nf3 e6 5.Be2 c5", "1.e4 c6 2.d4 d5 3.e5 Bf5 4.Nf3 e6 5.Be2 c5 *"),
("B12t", "Caro-Kann: Advance, 4.Nf3 e6 5.Be2 c5 6.O-O", "1.e4 c6 2.d4 d5 3.e5 Bf5 4.Nf3 e6 5.Be2 c5 6.O-O *"),
("B12t", "Caro-Kann: Advance, 4.Nf3 e6 5.Be2 c5 6.O-O", "1.e4 c6 2.d4 d5 3.e5 Bf5 4.Nf3 e6 5.Be2 c5 6.O-O Ne7 *"),
("B12t", "Caro-Kann: Advance, 4.Nf3 e6 5.Be2 c5 6.O-O", "1.e4 c6 2.d4 d5 3.e5 Bf5 4.Nf3 e6 5.Be2 c5 6.O-O Nd7 *"),
("B12u", "Caro-Kann: Advance, 4.Nf3 e6 5.Be2 c5 6.Be3", "1.e4 c6 2.d4 d5 3.e5 Bf5 4.Nf3 e6 5.Be2 c5 6.Be3 *"),
("B12u", "Caro-Kann: Advance, 4.Nf3 e6 5.Be2 c5 6.Be3", "1.e4 c6 2.d4 d5 3.e5 Bf5 4.Nf3 e6 5.Be2 c5 6.Be3 Ne7 *"),
("B12u", "Caro-Kann: Advance, 4.Nf3 e6 5.Be2 c5 6.Be3", "1.e4 c6 2.d4 d5 3.e5 Bf5 4.Nf3 e6 5.Be2 c5 6.Be3 Nd7 *"),
("B12u", "Caro-Kann: Advance, 4.Nf3 e6 5.Be2 c5 6.Be3", "1.e4 c6 2.d4 d5 3.e5 Bf5 4.Nf3 e6 5.Be2 c5 6.Be3 Nd7 7.O-O *"),
("B12v", "Caro-Kann: Advance, 4.Nc3", "1.e4 c6 2.d4 d5 3.e5 Bf5 4.Nc3 *"),
("B12v", "Caro-Kann: Advance, 4.Nc3 h5", "1.e4 c6 2.d4 d5 3.e5 Bf5 4.Nc3 h5 *"),
("B12v", "Caro-Kann: Advance, 4.Nc3 Qb6", "1.e4 c6 2.d4 d5 3.e5 Bf5 4.Nc3 Qb6 *"),
("B12v", "Caro-Kann: Advance, 4.Nc3 e6", "1.e4 c6 2.d4 d5 3.e5 Bf5 4.Nc3 e6 *"),
("B12w", "Caro-Kann: Advance, Main Line", "1.e4 c6 2.d4 d5 3.e5 Bf5 4.Nc3 e6 5.g4 *"),
("B12w", "Caro-Kann: Advance, Main Line, 6.Nge2", "1.e4 c6 2.d4 d5 3.e5 Bf5 4.Nc3 e6 5.g4 Bg6 6.Nge2 *"),
("B12x", "Caro-Kann: Advance, Main Line, 6.Nge2 c5", "1.e4 c6 2.d4 d5 3.e5 Bf5 4.Nc3 e6 5.g4 Bg6 6.Nge2 c5 *"),
("B13a", "Caro-Kann: Exchange Variation", "1.e4 c6 2.d4 d5 3.exd5 *"),
("B13a", "Caro-Kann: Exchange, 3...Qxd5", "1.e4 c6 2.d4 d5 3.exd5 Qxd5 *"),
("B13b", "Caro-Kann: Exchange, 3...cxd5", "1.e4 c6 2.d4 d5 3.exd5 cxd5 *"),
("B13c", "Caro-Kann: Exchange, 4.Nf3", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.Nf3 *"),
("B13d", "Caro-Kann: Exchange, 4.Nf3 Nf6", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.Nf3 Nf6 *"),
("B13e", "Caro-Kann: Exchange, 4.Bd3", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.Bd3 *"),
("B13f", "Caro-Kann: Exchange, 4.Bd3 Nc6", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.Bd3 Nc6 *"),
("B13f", "Caro-Kann: Exchange, 4.Bd3 Nc6 5.c3", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.Bd3 Nc6 5.c3 *"),
("B13g", "Caro-Kann: Exchange, 4.Bd3 Nc6 5.c3 g6", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.Bd3 Nc6 5.c3 g6 *"),
("B13h", "Caro-Kann: Exchange, 4.Bd3 Nc6 5.c3 Nf6", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.Bd3 Nc6 5.c3 Nf6 *"),
("B13i", "Caro-Kann: Exchange, 4.Bd3 Nc6 5.c3 Nf6 6.Nf3", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.Bd3 Nc6 5.c3 Nf6 6.Nf3 *"),
("B13j", "Caro-Kann: Exchange, Rubinstein Variation", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.Bd3 Nc6 5.c3 Nf6 6.Bf4 *"),
("B13k", "Caro-Kann: Exchange, Rubinstein, 6...Bg4", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.Bd3 Nc6 5.c3 Nf6 6.Bf4 Bg4 *"),
("B13l", "Caro-Kann: Exchange, Rubinstein, 7.Qb3 Qc8", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.Bd3 Nc6 5.c3 Nf6 6.Bf4 Bg4 7.Qb3 Qc8 *"),
("B13m", "Caro-Kann: Exchange, Rubinstein, 7.Qb3 Qd7", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.Bd3 Nc6 5.c3 Nf6 6.Bf4 Bg4 7.Qb3 Qd7 *"),
("B13n", "Caro-Kann: Panov-Botvinnik Attack", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 *"),
("B13n", "Caro-Kann: Panov-Botvinnik, 4...e6", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 e6 *"),
("B13o", "Caro-Kann: Panov-Botvinnik, 4...Nf6", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 *"),
("B13o", "Caro-Kann: Panov-Botvinnik, Gunderam Variation", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.c5 *"),
("B13p", "Caro-Kann: Panov-Botvinnik, 5.Nc3", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 *"),
("B13p", "Caro-Kann: Panov-Botvinnik, 5...dxc4", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 dxc4 *"),
("B13p", "Caro-Kann: Panov-Botvinnik, 5...dxc4", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 dxc4 6.Bxc4 *"),
("B13q", "Caro-Kann: Panov-Botvinnik, 5...Nc6", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 Nc6 *"),
("B13q", "Caro-Kann: Panov-Botvinnik, 5...Nc6 6.cxd5", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 Nc6 6.cxd5 *"),
("B13q", "Caro-Kann: Panov-Botvinnik, 5...Nc6 6.cxd5 Nxd5", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 Nc6 6.cxd5 Nxd5 *"),
("B13r", "Caro-Kann: Panov-Botvinnik, 5...Nc6 6.Nf3", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 Nc6 6.Nf3 *"),
("B13s", "Caro-Kann: Panov-Botvinnik, 5...Nc6 6.Nf3 Bg4", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 Nc6 6.Nf3 Bg4 *"),
("B13s", "Caro-Kann: Panov-Botvinnik, 5...Nc6 6.Nf3 Bg4", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 Nc6 6.Nf3 Bg4 7.cxd5 *"),
("B13s", "Caro-Kann: Panov-Botvinnik, 5...Nc6 6.Nf3 Bg4", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 Nc6 6.Nf3 Bg4 7.cxd5 Nxd5 *"),
("B13t", "Caro-Kann: Panov-Botvinnik, 5...Nc6 6.Nf3 Bg4: 8.Qb3", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 Nc6 6.Nf3 Bg4 7.cxd5 Nxd5 8.Qb3 *"),
("B13t", "Caro-Kann: Panov-Botvinnik, 5...Nc6 6.Nf3 Bg4: 9.gxf3 Nb6", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 Nc6 6.Nf3 Bg4 7.cxd5 Nxd5 8.Qb3 Bxf3 9.gxf3 Nb6 *"),
("B13u", "Caro-Kann: Panov-Botvinnik, 5...Nc6 6.Nf3 Bg4: Main Line", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 Nc6 6.Nf3 Bg4 7.cxd5 Nxd5 8.Qb3 Bxf3 9.gxf3 e6 *"),
("B13v", "Caro-Kann: Panov-Botvinnik, 5...Nc6 6.Bg5", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 Nc6 6.Bg5 *"),
("B13v", "Caro-Kann: Panov-Botvinnik, 5...Nc6, Spielmann Variation", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 Nc6 6.Bg5 Qb6 *"),
("B13v", "Caro-Kann: Panov-Botvinnik, 5...Nc6 6.Bg5 dxc4", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 Nc6 6.Bg5 dxc4 *"),
("B13v", "Caro-Kann: Panov-Botvinnik, 5...Nc6, Herzog Defence", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 Nc6 6.Bg5 dxc4 7.d5 Na5 *"),
("B13w", "Caro-Kann: Panov-Botvinnik, 5...Nc6, Czerniak Variation", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 Nc6 6.Bg5 Qa5 *"),
("B13x", "Caro-Kann: Panov-Botvinnik, 5...Nc6 6.Bg5 Be6", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 Nc6 6.Bg5 Be6 *"),
("B13y", "Caro-Kann: Panov-Botvinnik, 5...Nc6, Normal Variation", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 Nc6 6.Bg5 e6 *"),
("B13y", "Caro-Kann: Panov-Botvinnik, 5...Nc6, Normal Variation", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Nf3 *"),
("B13y", "Caro-Kann: Panov-Botvinnik, 5...Nc6, Normal Variation", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Nf3 Be7 *"),
("B13z", "Caro-Kann: Panov-Botvinnik, 5...Nc6, Normal Variaiton, 8.c5", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Nf3 Be7 8.c5 *"),
("B13z", "Caro-Kann: Panov-Botvinnik, 5...Nc6, Normal Variation, 8.c5 O-O", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Nf3 Be7 8.c5 O-O *"),
("B13z", "Caro-Kann: Panov-Botvinnik, 5...Nc6, Normal Variation, 8.c5 O-O 9.Bb5", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Nf3 Be7 8.c5 O-O 9.Bb5 *"),
("B14a", "Caro-Kann: Panov-Botvinnik, 5...g6", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 g6 *"),
("B14a", "Caro-Kann: Panov-Botvinnik, 5...g6 6.Nf3", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 g6 6.Nf3 *"),
("B14b", "Caro-Kann: Panov-Botvinnik, 5...g6 6.cxd5", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 g6 6.cxd5 *"),
("B14c", "Caro-Kann: Panov-Botvinnik, 5...g6 6.cxd5 Nxd5", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 g6 6.cxd5 Nxd5 *"),
("B14d", "Caro-Kann: Panov-Botvinnik, 5...g6 6.cxd5 Nxd5 7.Bc4", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 g6 6.cxd5 Nxd5 7.Bc4 *"),
("B14e", "Caro-Kann: Panov-Botvinnik, 5...g6 6.cxd5 Bg7", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 g6 6.cxd5 Bg7 *"),
("B14f", "Caro-Kann: Panov-Botvinnik, 5...g6 6.Qb3", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 g6 6.Qb3 *"),
("B14f", "Caro-Kann: Panov-Botvinnik, 5...g6 6.Qb3 Bg7 7.cxd5", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 g6 6.Qb3 Bg7 7.cxd5 *"),
("B14f", "Caro-Kann: Panov-Botvinnik, 5...g6, Main Line", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 g6 6.Qb3 Bg7 7.cxd5 O-O *"),
("B14g", "Caro-Kann: Panov-Botvinnik, 5...g6, Main Line, 8.Be2", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 g6 6.Qb3 Bg7 7.cxd5 O-O 8.Be2 *"),
("B14h", "Caro-Kann: Panov-Botvinnik, 5...g6, Main Line, 8.Be2 Nbd7", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 g6 6.Qb3 Bg7 7.cxd5 O-O 8.Be2 Nbd7 *"),
("B14i", "Caro-Kann: Panov-Botvinnik, 5...e6", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 e6 *"),
("B14j", "Caro-Kann: Panov-Botvinnik, 5...e6 6.Nf3", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 e6 6.Nf3 *"),
("B14j", "Caro-Kann: Panov-Botvinnik, 5...e6 6.Nf3 Nc6", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 e6 6.Nf3 Nc6 *"),
("B14k", "Caro-Kann: Panov-Botvinnik, 5...e6 6.Nf3 Be7", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 e6 6.Nf3 Be7 *"),
("B14l", "Caro-Kann: Panov-Botvinnik, 5...e6 6.Nf3 Be7 7.Bd3", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 e6 6.Nf3 Be7 7.Bd3 *"),
("B14m", "Caro-Kann: Panov-Botvinnik, 5...e6 6.Nf3 Be7 7.cxd5", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 e6 6.Nf3 Be7 7.cxd5 *"),
("B14n", "Caro-Kann: Panov-Botvinnik, 5...e6 6.Nf3 Be7 7.cxd5 Nxd5", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 e6 6.Nf3 Be7 7.cxd5 Nxd5 *"),
("B14o", "Caro-Kann: Panov-Botvinnik, 5...e6 6.Nf3 Be7 7.cxd5 Nxd5 8.Bd3", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 e6 6.Nf3 Be7 7.cxd5 Nxd5 8.Bd3 *"),
("B14p", "Caro-Kann: Panov-Botvinnik, 5...e6 6.Nf3 Bb4", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 e6 6.Nf3 Bb4 *"),
("B14q", "Caro-Kann: Panov-Botvinnik, 5...e6 6.Nf3 Bb4 7.cxd5 exd5", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 e6 6.Nf3 Bb4 7.cxd5 exd5 *"),
("B14r", "Caro-Kann: Panov-Botvinnik, 5...e6 6.Nf3 Bb4 7.cxd5 Nxd5", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 e6 6.Nf3 Bb4 7.cxd5 Nxd5 *"),
("B14s", "Caro-Kann: Panov-Botvinnik, 5...e6 6.Nf3 Bb4 7.cxd5 Nxd5 8.Qc2", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 e6 6.Nf3 Bb4 7.cxd5 Nxd5 8.Qc2 *"),
("B14t", "Caro-Kann: Panov-Botvinnik, 5...e6 6.Nf3 Bb4 7.cxd5 Nxd5 8.Qc2 Nc6", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 e6 6.Nf3 Bb4 7.cxd5 Nxd5 8.Qc2 Nc6 *"),
("B14u", "Caro-Kann: Panov-Botvinnik, 5...e6 6.Nf3 Bb4 7.cxd5 Nxd5 8.Qc2 Nc6 9.Bd3", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 e6 6.Nf3 Bb4 7.cxd5 Nxd5 8.Qc2 Nc6 9.Bd3 *"),
("B14v", "Caro-Kann: Panov-Botvinnik, 5...e6 6.Nf3 Bb4 7.cxd5 Nxd5 8.Bd2", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 e6 6.Nf3 Bb4 7.cxd5 Nxd5 8.Bd2 *"),
("B14w", "Caro-Kann: Panov-Botvinnik, 5...e6 6.Nf3 Bb4 7.cxd5 Nxd5 8.Bd2 Nc6", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 e6 6.Nf3 Bb4 7.cxd5 Nxd5 8.Bd2 Nc6 *"),
("B14x", "Caro-Kann: Panov-Botvinnik, 5...e6 6.Nf3 Bb4 7.cxd5 Nxd5 8.Bd2 Nc6", "1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 Nf6 5.Nc3 e6 6.Nf3 Bb4 7.cxd5 Nxd5 8.Bd2 Nc6 9.Bd3 O-O *"),
("B15a", "Caro-Kann: 3.Nc3", "1.e4 c6 2.d4 d5 3.Nc3 *"),
("B15b", "Caro-Kann: Gurgenidze Counterattack", "1.e4 c6 2.d4 d5 3.Nc3 b5 *"),
("B15c", "Caro-Kann: Gurgenidze/Modern", "1.e4 c6 2.d4 d5 3.Nc3 g6 *"),
("B15d", "Caro-Kann: Gurgenidze: 4.Be3", "1.e4 c6 2.d4 d5 3.Nc3 g6 4.Be3 *"),
("B15d", "Caro-Kann: Gurgenidze: 4.Be3", "1.e4 c6 2.d4 d5 3.Nc3 g6 4.Be3 Bg7 *"),
("B15e", "Caro-Kann: Gurgenidze: 4.h3", "1.e4 c6 2.d4 d5 3.Nc3 g6 4.h3 *"),
("B15e", "Caro-Kann: Gurgenidze: 4.h3", "1.e4 c6 2.d4 d5 3.Nc3 g6 4.h3 Bg7 *"),
("B15f", "Caro-Kann: Gurgenidze: 4.e5", "1.e4 c6 2.d4 d5 3.Nc3 g6 4.e5 *"),
("B15f", "Caro-Kann: Gurgenidze: 4.e5", "1.e4 c6 2.d4 d5 3.Nc3 g6 4.e5 Bg7 *"),
("B15g", "Caro-Kann: Gurgenidze: 4.e5 Bg7 5.f4", "1.e4 c6 2.d4 d5 3.Nc3 g6 4.e5 Bg7 5.f4 *"),
("B15g", "Caro-Kann: Gurgenidze: 4.e5 Bg7 5.f4 Nh6", "1.e4 c6 2.d4 d5 3.Nc3 g6 4.e5 Bg7 5.f4 Nh6 *"),
("B15h", "Caro-Kann: Gurgenidze: 4.e5 Bg7 5.f4 h5", "1.e4 c6 2.d4 d5 3.Nc3 g6 4.e5 Bg7 5.f4 h5 *"),
("B15i", "Caro-Kann: Gurgenidze: 4.e5 Bg7 5.f4 h5 6.Nf3", "1.e4 c6 2.d4 d5 3.Nc3 g6 4.e5 Bg7 5.f4 h5 6.Nf3 *"),
("B15j", "Caro-Kann: Gurgenidze: 4.Nf3", "1.e4 c6 2.d4 d5 3.Nc3 g6 4.Nf3 *"),
("B15k", "Caro-Kann: Gurgenidze: 4.Nf3 Bg7", "1.e4 c6 2.d4 d5 3.Nc3 g6 4.Nf3 Bg7 *"),
("B15k", "Caro-Kann: Gurgenidze: 4.Nf3 Bg7 5.exd5", "1.e4 c6 2.d4 d5 3.Nc3 g6 4.Nf3 Bg7 5.exd5 *"),
("B15k", "Caro-Kann: Gurgenidze: 4.Nf3 Bg7 5.e5", "1.e4 c6 2.d4 d5 3.Nc3 g6 4.Nf3 Bg7 5.e5 *"),
("B15l", "Caro-Kann: Gurgenidze: 4.Nf3 Bg7 5.h3", "1.e4 c6 2.d4 d5 3.Nc3 g6 4.Nf3 Bg7 5.h3 *"),
("B15m", "Caro-Kann: Gurgenidze: 4.Nf3 Bg7 5.h3 dxe4", "1.e4 c6 2.d4 d5 3.Nc3 g6 4.Nf3 Bg7 5.h3 dxe4 *"),
("B15m", "Caro-Kann: Gurgenidze: 4.Nf3 Bg7 5.h3 dxe4", "1.e4 c6 2.d4 d5 3.Nc3 g6 4.Nf3 Bg7 5.h3 dxe4 6.Nxe4 *"),
("B15n", "Caro-Kann: Gurgenidze: 4.Nf3 Bg7 5.h3 Nf6", "1.e4 c6 2.d4 d5 3.Nc3 g6 4.Nf3 Bg7 5.h3 Nf6 *"),
("B15o", "Caro-Kann: Gurgenidze: 4.Nf3 Bg7 5.h3 Nf6 6.e5", "1.e4 c6 2.d4 d5 3.Nc3 g6 4.Nf3 Bg7 5.h3 Nf6 6.e5 *"),
("B15p", "Caro-Kann: 3.Nc3 dxe4", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 *"),
("B15p", "Caro-Kann: Rasa-Studier Gambit", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.f3 *"),
("B15p", "Caro-Kann: von Hennig Gambit", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Bc4 *"),
("B15q", "Caro-Kann: 4.Nxe4", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 *"),
("B15r", "Caro-Kann: 4.Nxe4 Nf6", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nf6 *"),
("B15r", "Caro-Kann: 4.Nxe4 Nf6 5.f3 Gambit", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nf6 5.f3 *"),
("B15r", "Caro-Kann: Alekhine Gambit", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nf6 5.Bd3 *"),
("B15s", "Caro-Kann: 4.Nxe4 Nf6 5.Ng3", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nf6 5.Ng3 *"),
("B15t", "Caro-Kann: 4.Nxe4 Nf6 5.Nxf6+", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nf6 5.Nxf6+ *"),
("B15t", "Caro-Kann: Tartakower (Nimzowitsch) Variation", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nf6 5.Nxf6+ exf6 *"),
("B15u", "Caro-Kann: Tartakower, Forgacs Variation", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nf6 5.Nxf6+ exf6 6.Bc4 *"),
("B15v", "Caro-Kann: Tartakower, 6.c3", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nf6 5.Nxf6+ exf6 6.c3 *"),
("B16a", "Caro-Kann: Bronstein-Larsen", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nf6 5.Nxf6+ gxf6 *"),
("B16b", "Caro-Kann: Bronstein-Larsen, 6.Qd3", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nf6 5.Nxf6+ gxf6 6.Qd3 *"),
("B16c", "Caro-Kann: Bronstein-Larsen, 6.Ne2", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nf6 5.Nxf6+ gxf6 6.Ne2 *"),
("B16d", "Caro-Kann: Bronstein-Larsen, 6.Bc4", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nf6 5.Nxf6+ gxf6 6.Bc4 *"),
("B16e", "Caro-Kann: Bronstein-Larsen, 6.Nf3", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nf6 5.Nxf6+ gxf6 6.Nf3 *"),
("B16f", "Caro-Kann: Bronstein-Larsen, 6.Nf3 Bf5", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nf6 5.Nxf6+ gxf6 6.Nf3 Bf5 *"),
("B16g", "Caro-Kann: Bronstein-Larsen, 6.Nf3 Bg4", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nf6 5.Nxf6+ gxf6 6.Nf3 Bg4 *"),
("B16h", "Caro-Kann: Bronstein-Larsen, 6.Nf3 Bg4 7.Be2 e6", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nf6 5.Nxf6+ gxf6 6.Nf3 Bg4 7.Be2 e6 *"),
("B16i", "Caro-Kann: Bronstein-Larsen, 6.c3", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nf6 5.Nxf6+ gxf6 6.c3 *"),
("B16j", "Caro-Kann: Bronstein-Larsen, 6.c3 Bf5", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nf6 5.Nxf6+ gxf6 6.c3 Bf5 *"),
("B16k", "Caro-Kann: Bronstein-Larsen, 6.c3 Bf5 7.Ne2", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nf6 5.Nxf6+ gxf6 6.c3 Bf5 7.Ne2 *"),
("B16l", "Caro-Kann: Bronstein-Larsen, 6.c3 Bf5 7.Nf3", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nf6 5.Nxf6+ gxf6 6.c3 Bf5 7.Nf3 *"),
("B16m", "Caro-Kann: Bronstein-Larsen, 6.c3 Bf5 7.Nf3 Qc7", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nf6 5.Nxf6+ gxf6 6.c3 Bf5 7.Nf3 Qc7 *"),
("B16n", "Caro-Kann: Bronstein-Larsen, 6.c3 Bf5 7.Nf3 e6", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nf6 5.Nxf6+ gxf6 6.c3 Bf5 7.Nf3 e6 *"),
("B17a", "Caro-Kann: Steinitz Variation", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 *"),
("B17b", "Caro-Kann: Steinitz, 5.Nf3", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Nf3 *"),
("B17c", "Caro-Kann: Steinitz, 5.Nf3 Ngf6", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Nf3 Ngf6 *"),
("B17d", "Caro-Kann: Steinitz, 5.Nf3 Ngf6 6.Nxf6+", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Nf3 Ngf6 6.Nxf6+ *"),
("B17e", "Caro-Kann: Steinitz, 5.Nf3 Ngf6 6.Nxf6+ Nxf6 7.Bc4", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Nf3 Ngf6 6.Nxf6+ Nxf6 7.Bc4 *"),
("B17f", "Caro-Kann: Steinitz, 5.Nf3 Ngf6 6.Ng3", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Nf3 Ngf6 6.Ng3 *"),
("B17g", "Caro-Kann: Steinitz, 5.Nf3 Ngf6 6.Ng3 e6", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Nf3 Ngf6 6.Ng3 e6 *"),
("B17h", "Caro-Kann: Steinitz, 5.Ng5", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Ng5 *"),
("B17i", "Caro-Kann: Steinitz, 5.Ng5 Ngf6", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Ng5 Ngf6 *"),
("B17j", "Caro-Kann: Steinitz, 5.Ng5 Ngf6 6.Bd3", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Ng5 Ngf6 6.Bd3 *"),
("B17k", "Caro-Kann: Steinitz, 5.Ng5 Ngf6 6.Bd3 e6", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Ng5 Ngf6 6.Bd3 e6 *"),
("B17l", "Caro-Kann: Steinitz, 5.Ng5 Ngf6 6.Bd3 e6 7.N1f3 Bd6 8.Qe2", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Ng5 Ngf6 6.Bd3 e6 7.N1f3 Bd6 8.Qe2 *"),
("B17m", "Caro-Kann: Steinitz, 5.Ng5 Ngf6 6.Bd3, Main Line 10.Qxe4", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Ng5 Ngf6 6.Bd3 e6 7.N1f3 Bd6 8.Qe2 h6 9.Ne4 Nxe4 10.Qxe4 *"),
("B17n", "Caro-Kann: Steinitz, 5.Ng5 Ngf6 6.Bd3, Main Line 10.Qxe4 Qc7", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Ng5 Ngf6 6.Bd3 e6 7.N1f3 Bd6 8.Qe2 h6 9.Ne4 Nxe4 10.Qxe4 Qc7 *"),
("B17o", "Caro-Kann: Steinitz, 5.Ng5 Ngf6 6.Bd3, Main Line 10.Qxe4 Nf6", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Ng5 Ngf6 6.Bd3 e6 7.N1f3 Bd6 8.Qe2 h6 9.Ne4 Nxe4 10.Qxe4 Nf6 *"),
("B17p", "Caro-Kann: Steinitz, 5.Bc4", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Bc4 *"),
("B17q", "Caro-Kann: Steinitz, 5.Bc4 Ngf6", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Bc4 Ngf6 *"),
("B17r", "Caro-Kann: Steinitz, 5.Bc4 Ngf6 6.Ng5", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Bc4 Ngf6 6.Ng5 *"),
("B17s", "Caro-Kann: Steinitz, 5.Bc4 Ngf6 6.Ng5, 8.Bb3", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Bc4 Ngf6 6.Ng5 e6 7.Qe2 Nb6 8.Bb3 *"),
("B17t", "Caro-Kann: Steinitz, 5.Bc4 Ngf6 6.Ng5, 8.Bd3", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Bc4 Ngf6 6.Ng5 e6 7.Qe2 Nb6 8.Bd3 *"),
("B17u", "Caro-Kann: Steinitz, 5.Bc4 Ngf6 6.Ng5, 8.Bd3 h6", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Bc4 Ngf6 6.Ng5 e6 7.Qe2 Nb6 8.Bd3 h6 *"),
("B18a", "Caro-Kann: Classical", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 *"),
("B18b", "Caro-Kann: Classical, 5.Nc5", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Nc5 *"),
("B18c", "Caro-Kann: Classical, 5.Nc5 b6", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Nc5 b6 *"),
("B18d", "Caro-Kann: Classical, 5.Ng3", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 *"),
("B18e", "Caro-Kann: Classical, 5.Ng3 Bg6", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 *"),
("B18f", "Caro-Kann: Classical, Maroczy Attack", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.f4 *"),
("B18g", "Caro-Kann: Classical, Flohr Variation", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.Nh3 *"),
("B18h", "Caro-Kann: Classical, 6.N1e2", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.N1e2 *"),
("B18i", "Caro-Kann: Classical, 6.Bc4", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.Bc4 *"),
("B18j", "Caro-Kann: Classical, 6.Bc4 e6 7.N1e2", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.Bc4 e6 7.N1e2 *"),
("B18k", "Caro-Kann: Classical, 6.Bc4 e6 7.N1e2 Nf6", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.Bc4 e6 7.N1e2 Nf6 *"),
("B18l", "Caro-Kann: Classical, 6.Nf3", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.Nf3 *"),
("B18m", "Caro-Kann: Classical, 6.Nf3 Nd7", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.Nf3 Nd7 *"),
("B18n", "Caro-Kann: Classical, 6.Nf3 Nd7 7.Bd3", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.Nf3 Nd7 7.Bd3 *"),
("B18o", "Caro-Kann: Classical, 6.Nf3 Nd7 7.Bd3 e6", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.Nf3 Nd7 7.Bd3 e6 *"),
("B18p", "Caro-Kann: Classical, 6.h4", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 *"),
("B18p", "Caro-Kann: Classical, 6.h4 h6", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 *"),
("B18q", "Caro-Kann: Classical, 6.h4 h6 7.Nh3", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.Nh3 *"),
("B18r", "Caro-Kann: Classical, 6.h4 h6 7.f4", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.f4 *"),
("B18s", "Caro-Kann: Classical, 6.h4 h6 7.h5", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.h5 *"),
("B19a", "Caro-Kann: Classical, 7.Nf3", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.Nf3 *"),
("B19b", "Caro-Kann: Classical, 7.Nf3 e6", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.Nf3 e6 *"),
("B19c", "Caro-Kann: Classical, 7.Nf3 Nf6", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.Nf3 Nf6 *"),
("B19d", "Caro-Kann: Classical, 7.Nf3 Nf6 8.h5", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.Nf3 Nf6 8.h5 *"),
("B19d", "Caro-Kann: Classical, 7.Nf3 Nf6 8.h5", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.Nf3 Nf6 8.h5 Bh7 *"),
("B19e", "Caro-Kann: Classical, 7.Nf3 Nf6 8.Ne5", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.Nf3 Nf6 8.Ne5 *"),
("B19f", "Caro-Kann: Classical, 7.Nf3 Nf6 8.Ne5 Bh7 9.Bd3", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.Nf3 Nf6 8.Ne5 Bh7 9.Bd3 *"),
("B19g", "Caro-Kann: Classical, 7.Nf3 Nd7", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.Nf3 Nd7 *"),
("B19h", "Caro-Kann: Classical, Spassky Variation", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.Nf3 Nd7 8.h5 *"),
("B19h", "Caro-Kann: Classical, Spassky Variation", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.Nf3 Nd7 8.h5 Bh7 *"),
("B19i", "Caro-Kann: Classical, Spassky, 10.Qxd3", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.Nf3 Nd7 8.h5 Bh7 9.Bd3 Bxd3 10.Qxd3 *"),
("B19j", "Caro-Kann: Classical, Spassky, 10.Qxd3 Ngf6", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.Nf3 Nd7 8.h5 Bh7 9.Bd3 Bxd3 10.Qxd3 Ngf6 *"),
("B19k", "Caro-Kann: Classical, Spassky, 10.Qxd3 e6", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.Nf3 Nd7 8.h5 Bh7 9.Bd3 Bxd3 10.Qxd3 e6 *"),
("B19l", "Caro-Kann: Classical, Spassky, 10.Qxd3 e6 11.Bf4", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.Nf3 Nd7 8.h5 Bh7 9.Bd3 Bxd3 10.Qxd3 e6 11.Bf4 *"),
("B19m", "Caro-Kann: Classical, Spassky, 10.Qxd3 Qc7", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.Nf3 Nd7 8.h5 Bh7 9.Bd3 Bxd3 10.Qxd3 Qc7 *"),
("B19n", "Caro-Kann: Classical, Spassky, 10.Qxd3 Qc7 11.Bd2", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.Nf3 Nd7 8.h5 Bh7 9.Bd3 Bxd3 10.Qxd3 Qc7 11.Bd2 *"),
("B19o", "Caro-Kann: Classical, Spassky, 10.Qxd3 Qc7 11.Bd2 e6", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.Nf3 Nd7 8.h5 Bh7 9.Bd3 Bxd3 10.Qxd3 Qc7 11.Bd2 e6 *"),
("B19p", "Caro-Kann: Classical, Spassky, 10.Qxd3 Qc7 11.Bd2 e6 12.O-O-O", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.Nf3 Nd7 8.h5 Bh7 9.Bd3 Bxd3 10.Qxd3 Qc7 11.Bd2 e6 12.O-O-O *"),
("B19q", "Caro-Kann: Classical, Spassky, Main Line", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.Nf3 Nd7 8.h5 Bh7 9.Bd3 Bxd3 10.Qxd3 Qc7 11.Bd2 e6 12.O-O-O Ngf6 *"),
("B19r", "Caro-Kann: Classical, Spassky, Main Line, 13.Ne4", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.Nf3 Nd7 8.h5 Bh7 9.Bd3 Bxd3 10.Qxd3 Qc7 11.Bd2 e6 12.O-O-O Ngf6 13.Ne4 *"),
("B19s", "Caro-Kann: Classical, Spassky, Main Line, 13.Ne4 O-O-O", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.Nf3 Nd7 8.h5 Bh7 9.Bd3 Bxd3 10.Qxd3 Qc7 11.Bd2 e6 12.O-O-O Ngf6 13.Ne4 O-O-O *"),
("B19t", "Caro-Kann: Classical, Spassky, Main Line, 14.g3 Nxe4", "1.e4 c6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bf5 5.Ng3 Bg6 6.h4 h6 7.Nf3 Nd7 8.h5 Bh7 9.Bd3 Bxd3 10.Qxd3 Qc7 11.Bd2 e6 12.O-O-O Ngf6 13.Ne4 O-O-O 14.g3 Nxe4 *"),
("B20", "Sicilian Defence", "1.e4 c5 *"),
("B20", "Sicilian: Mengarini", "1.e4 c5 2.a3 *"),
("B20", "Sicilian: Keres Variation (2.Ne2)", "1.e4 c5 2.Ne2 *"),
("B20", "Sicilian: 2.Bc4", "1.e4 c5 2.Bc4 *"),
("B20", "Sicilian: Wing Gambit", "1.e4 c5 2.b4 *"),
("B20", "Sicilian: Wing Gambit, Santasiere Variation", "1.e4 c5 2.b4 cxb4 3.c4 *"),
("B20", "Sicilian: Wing Gambit, Marshall Variation", "1.e4 c5 2.b4 cxb4 3.a3 *"),
("B20", "Sicilian: Wing Gambit, Marienbad Variation", "1.e4 c5 2.b4 cxb4 3.a3 d5 4.exd5 Qxd5 5.Bb2 *"),
("B20", "Sicilian: Wing Gambit, Carlsbad Variation", "1.e4 c5 2.b4 cxb4 3.a3 bxa3 *"),
("B20", "Sicilian: Snyder", "1.e4 c5 2.b3 *"),
("B20", "Sicilian: Snyder, 2...e6", "1.e4 c5 2.b3 e6 *"),
("B20", "Sicilian: Snyder, 2...d6", "1.e4 c5 2.b3 d6 *"),
("B20", "Sicilian: Snyder, 2...Nc6", "1.e4 c5 2.b3 Nc6 *"),
("B20", "Sicilian: English (2.c4)", "1.e4 c5 2.c4 *"),
("B20", "Sicilian: English, 2...e6", "1.e4 c5 2.c4 e6 *"),
("B20", "Sicilian: English, 2...d6", "1.e4 c5 2.c4 d6 *"),
("B20", "Sicilian: English, 2...Nc6", "1.e4 c5 2.c4 Nc6 *"),
("B20", "Sicilian: English, 2...Nc6 3.Nf3", "1.e4 c5 2.c4 Nc6 3.Nf3 *"),
("B20", "Sicilian: English, 2...Nc6 3.Ne2", "1.e4 c5 2.c4 Nc6 3.Ne2 *"),
("B20", "Sicilian: English, 2...Nc6 3.Nc3", "1.e4 c5 2.c4 Nc6 3.Nc3 *"),
("B20", "Sicilian: English, 2...Nc6 3.Nc3 g6", "1.e4 c5 2.c4 Nc6 3.Nc3 g6 *"),
("B20", "Sicilian: 2.g3", "1.e4 c5 2.g3 *"),
("B20", "Sicilian: 2.g3", "1.e4 c5 2.g3 g6 *"),
("B20", "Sicilian: 2.g3", "1.e4 c5 2.g3 g6 3.Bg2 Bg7 *"),
("B20", "Sicilian: 2.g3", "1.e4 c5 2.g3 g6 3.Bg2 Bg7 4.f4 *"),
("B20", "Sicilian: 2.g3", "1.e4 c5 2.g3 g6 3.Bg2 Bg7 4.f4 d6 *"),
("B20", "Sicilian: 2.d3", "1.e4 c5 2.d3 *"),
("B20", "Sicilian: 2.d3 e6", "1.e4 c5 2.d3 e6 *"),
("B20", "Sicilian: 2.d3 e6 3.g3", "1.e4 c5 2.d3 e6 3.g3 *"),
("B20", "Sicilian: 2.d3 Nc6", "1.e4 c5 2.d3 Nc6 *"),
("B20", "Sicilian: 2.d3 Nc6 3.g3", "1.e4 c5 2.d3 Nc6 3.g3 *"),
("B21a", "Sicilian: Grand Prix Attack", "1.e4 c5 2.f4 *"),
("B21b", "Sicilian: Grand Prix, 2...g6", "1.e4 c5 2.f4 g6 *"),
("B21c", "Sicilian: Grand Prix, 2...d6", "1.e4 c5 2.f4 d6 *"),
("B21d", "Sicilian: Grand Prix, 2...e6", "1.e4 c5 2.f4 e6 *"),
("B21d", "Sicilian: Grand Prix, 2...e6 3.Nf3", "1.e4 c5 2.f4 e6 3.Nf3 *"),
("B21e", "Sicilian: Grand Prix, 2...Nc6", "1.e4 c5 2.f4 Nc6 *"),
("B21e", "Sicilian: Grand Prix, 2...Nc6 3.d3", "1.e4 c5 2.f4 Nc6 3.d3 *"),
("B21f", "Sicilian: Grand Prix, 2...Nc6 3.Nf3", "1.e4 c5 2.f4 Nc6 3.Nf3 *"),
("B21g", "Sicilian: Grand Prix, 2...Nc6 3.Nf3 e6", "1.e4 c5 2.f4 Nc6 3.Nf3 e6 *"),
("B21h", "Sicilian: Grand Prix, 2...Nc6 3.Nf3 g6", "1.e4 c5 2.f4 Nc6 3.Nf3 g6 *"),
("B21i", "Sicilian: Grand Prix, Tal Defence", "1.e4 c5 2.f4 d5 *"),
("B21i", "Sicilian: Grand Prix, Tal Defence, 3.e5", "1.e4 c5 2.f4 d5 3.e5 *"),
("B21i", "Sicilian: Grand Prix, Toilet Variation", "1.e4 c5 2.f4 d5 3.Nc3 *"),
("B21j", "Sicilian: Grand Prix, Tal Defence, 3.exd5", "1.e4 c5 2.f4 d5 3.exd5 *"),
("B21j", "Sicilian: Grand Prix, Tal Defence, 3.exd5 Qxd5", "1.e4 c5 2.f4 d5 3.exd5 Qxd5 *"),
("B21k", "Sicilian: Grand Prix, Tal Gambit", "1.e4 c5 2.f4 d5 3.exd5 Nf6 *"),
("B21l", "Sicilian: Grand Prix, Tal Gambit, 4.Bb5+", "1.e4 c5 2.f4 d5 3.exd5 Nf6 4.Bb5+ *"),
("B21m", "Sicilian: Smith-Morra Gambit", "1.e4 c5 2.d4 *"),
("B21m", "Sicilian: Smith-Morra, 2...d5", "1.e4 c5 2.d4 d5 *"),
("B21m", "Sicilian: Smith-Morra, 2...cxd4", "1.e4 c5 2.d4 cxd4 *"),
("B21m", "Sicilian: Halasz Gambit", "1.e4 c5 2.d4 cxd4 3.f4 *"),
("B21n", "Sicilian: Smith-Morra, Morphy Gambit", "1.e4 c5 2.d4 cxd4 3.Nf3 *"),
("B21n", "Sicilian: Smith-Morra, Andreaschek Gambit", "1.e4 c5 2.d4 cxd4 3.Nf3 e5 4.c3 *"),
("B21o", "Sicilian: Smith-Morra, 3.c3", "1.e4 c5 2.d4 cxd4 3.c3 *"),
("B21p", "Sicilian: Smith-Morra, 3.c3 Nf6", "1.e4 c5 2.d4 cxd4 3.c3 Nf6 *"),
("B21q", "Sicilian: Smith-Morra, 3.c3 d5", "1.e4 c5 2.d4 cxd4 3.c3 d5 *"),
("B21r", "Sicilian: Smith-Morra, 3.c3 d3", "1.e4 c5 2.d4 cxd4 3.c3 d3 *"),
("B21s", "Sicilian: Smith-Morra Accepted", "1.e4 c5 2.d4 cxd4 3.c3 dxc3 *"),
("B21s", "Sicilian: Smith-Morra Accepted", "1.e4 c5 2.d4 cxd4 3.c3 dxc3 4.Nxc3 *"),
("B21t", "Sicilian: Smith-Morra, 4.Nxc3 d6", "1.e4 c5 2.d4 cxd4 3.c3 dxc3 4.Nxc3 d6 *"),
("B21u", "Sicilian: Smith-Morra, 4.Nxc3 e6", "1.e4 c5 2.d4 cxd4 3.c3 dxc3 4.Nxc3 e6 *"),
("B21v", "Sicilian: Smith-Morra, 4.Nxc3 e6 5.Nf3", "1.e4 c5 2.d4 cxd4 3.c3 dxc3 4.Nxc3 e6 5.Nf3 *"),
("B21w", "Sicilian: Smith-Morra, 4.Nxc3 Nc6", "1.e4 c5 2.d4 cxd4 3.c3 dxc3 4.Nxc3 Nc6 *"),
("B21w", "Sicilian: Smith-Morra, 4.Nxc3 Nc6 5.Nf3", "1.e4 c5 2.d4 cxd4 3.c3 dxc3 4.Nxc3 Nc6 5.Nf3 *"),
("B21x", "Sicilian: Smith-Morra, 4.Nxc3 Nc6 5.Nf3 e6", "1.e4 c5 2.d4 cxd4 3.c3 dxc3 4.Nxc3 Nc6 5.Nf3 e6 *"),
("B21x", "Sicilian: Smith-Morra, 4.Nxc3 Nc6 5.Nf3 e6 6.Bc4", "1.e4 c5 2.d4 cxd4 3.c3 dxc3 4.Nxc3 Nc6 5.Nf3 e6 6.Bc4 *"),
("B21x", "Sicilian: Smith-Morra, 4.Nxc3 Nc6 5.Nf3 e6 6.Bc4 a6", "1.e4 c5 2.d4 cxd4 3.c3 dxc3 4.Nxc3 Nc6 5.Nf3 e6 6.Bc4 a6 *"),
("B21y", "Sicilian: Smith-Morra, 4.Nxc3 Nc6 5.Nf3 d6", "1.e4 c5 2.d4 cxd4 3.c3 dxc3 4.Nxc3 Nc6 5.Nf3 d6 *"),
("B21y", "Sicilian: Smith-Morra, 5.Nf3 d6 6.Bc4", "1.e4 c5 2.d4 cxd4 3.c3 dxc3 4.Nxc3 Nc6 5.Nf3 d6 6.Bc4 *"),
("B21y", "Sicilian: Smith-Morra, 5.Nf3 d6 6.Bc4 a6", "1.e4 c5 2.d4 cxd4 3.c3 dxc3 4.Nxc3 Nc6 5.Nf3 d6 6.Bc4 a6 *"),
("B21y", "Sicilian: Smith-Morra, 5.Nf3 d6 6.Bc4 e6", "1.e4 c5 2.d4 cxd4 3.c3 dxc3 4.Nxc3 Nc6 5.Nf3 d6 6.Bc4 e6 *"),
("B21y", "Sicilian: Smith-Morra, 5.Nf3 d6 6.Bc4 e6 7.O-O", "1.e4 c5 2.d4 cxd4 3.c3 dxc3 4.Nxc3 Nc6 5.Nf3 d6 6.Bc4 e6 7.O-O *"),
("B21y", "Sicilian: Smith-Morra Gambit, Chicago Defence", "1.e4 c5 2.d4 cxd4 3.c3 dxc3 4.Nxc3 Nc6 5.Nf3 d6 6.Bc4 e6 7.O-O a6 8.Qe2 b5 9.Bb3 Ra7 *"),
("B21z", "Sicilian: Smith-Morra, 5.Nf3 d6 6.Bc4 e6 7.O-O Nf6", "1.e4 c5 2.d4 cxd4 3.c3 dxc3 4.Nxc3 Nc6 5.Nf3 d6 6.Bc4 e6 7.O-O Nf6 *"),
("B21z", "Sicilian: Smith-Morra, 5.Nf3 d6 6.Bc4 e6 7.O-O Nf6 8.Qe2 Be7", "1.e4 c5 2.d4 cxd4 3.c3 dxc3 4.Nxc3 Nc6 5.Nf3 d6 6.Bc4 e6 7.O-O Nf6 8.Qe2 Be7 *"),
("B22a", "Sicilian: Alapin", "1.e4 c5 2.c3 *"),
("B22a", "Sicilian: Alapin, 2...e5", "1.e4 c5 2.c3 e5 *"),
("B22a", "Sicilian: Alapin, 2...g6", "1.e4 c5 2.c3 g6 *"),
("B22a", "Sicilian: Alapin, 2...g6 3.d4", "1.e4 c5 2.c3 g6 3.d4 *"),
("B22a", "Sicilian: Alapin, 2...g6 3.d4 cxd4", "1.e4 c5 2.c3 g6 3.d4 cxd4 *"),
("B22b", "Sicilian: Alapin, 2...Nc6", "1.e4 c5 2.c3 Nc6 *"),
("B22b", "Sicilian: Alapin, 2...Nc6 3.Nf3", "1.e4 c5 2.c3 Nc6 3.Nf3 *"),
("B22b", "Sicilian: Alapin, 2...Nc6 3.d4", "1.e4 c5 2.c3 Nc6 3.d4 *"),
("B22c", "Sicilian: Alapin, 2...e6", "1.e4 c5 2.c3 e6 *"),
("B22c", "Sicilian: Alapin, 2...e6 3.Nf3", "1.e4 c5 2.c3 e6 3.Nf3 *"),
("B22c", "Sicilian: Alapin, 2...e6 3.Nf3 d5", "1.e4 c5 2.c3 e6 3.Nf3 d5 *"),
("B22d", "Sicilian: Alapin, 2...e6 3.d4", "1.e4 c5 2.c3 e6 3.d4 *"),
("B22d", "Sicilian: Alapin, 2...e6 3.d4 d5", "1.e4 c5 2.c3 e6 3.d4 d5 *"),
("B22d", "Sicilian: Alapin, 2...e6 3.d4 d5 4.exd5", "1.e4 c5 2.c3 e6 3.d4 d5 4.exd5 *"),
("B22d", "Sicilian: Alapin, 2...e6 3.d4 d5 4.exd5 exd5", "1.e4 c5 2.c3 e6 3.d4 d5 4.exd5 exd5 *"),
("B22d", "Sicilian: Alapin, 2...e6 3.d4 d5 4.exd5 exd5 5.Nf3", "1.e4 c5 2.c3 e6 3.d4 d5 4.exd5 exd5 5.Nf3 *"),
("B22d", "Sicilian: Alapin, 2...e6 3.d4 d5 4.exd5 exd5 5.Nf3 Nc6", "1.e4 c5 2.c3 e6 3.d4 d5 4.exd5 exd5 5.Nf3 Nc6 *"),
("B22e", "Sicilian: Alapin, 2...d6", "1.e4 c5 2.c3 d6 *"),
("B22f", "Sicilian: Alapin, 2...d5", "1.e4 c5 2.c3 d5 *"),
("B22f", "Sicilian: Alapin, 2...d5 3.exd5", "1.e4 c5 2.c3 d5 3.exd5 *"),
("B22f", "Sicilian: Alapin, 2...d5 3.exd5 Qxd5", "1.e4 c5 2.c3 d5 3.exd5 Qxd5 *"),
("B22f", "Sicilian: Alapin, 2...d5 3.exd5 Qxd5 4.d4", "1.e4 c5 2.c3 d5 3.exd5 Qxd5 4.d4 *"),
("B22f", "Sicilian: Alapin, 2...d5 3.exd5 Qxd5 4.d4 cxd4", "1.e4 c5 2.c3 d5 3.exd5 Qxd5 4.d4 cxd4 *"),
("B22f", "Sicilian: Alapin, 2...d5 3.exd5 Qxd5 4.d4 cxd4 5.cxd4", "1.e4 c5 2.c3 d5 3.exd5 Qxd5 4.d4 cxd4 5.cxd4 *"),
("B22g", "Sicilian: Alapin, 2...d5 3.exd5 Qxd5 4.d4 e6", "1.e4 c5 2.c3 d5 3.exd5 Qxd5 4.d4 e6 *"),
("B22g", "Sicilian: Alapin, 2...d5 3.exd5 Qxd5 4.d4 e6 5.Nf3", "1.e4 c5 2.c3 d5 3.exd5 Qxd5 4.d4 e6 5.Nf3 *"),
("B22h", "Sicilian: Alapin, 2...d5 3.exd5 Qxd5 4.d4 Nc6", "1.e4 c5 2.c3 d5 3.exd5 Qxd5 4.d4 Nc6 *"),
("B22h", "Sicilian: Alapin, 2...d5 3.exd5 Qxd5 4.d4 Nc6 5.Nf3", "1.e4 c5 2.c3 d5 3.exd5 Qxd5 4.d4 Nc6 5.Nf3 *"),
("B22h", "Sicilian: Alapin, 2...d5 3.exd5 Qxd5 4.d4 Nc6 5.Nf3 cxd4", "1.e4 c5 2.c3 d5 3.exd5 Qxd5 4.d4 Nc6 5.Nf3 cxd4 *"),
("B22h", "Sicilian: Alapin, 2...d5 3.exd5 Qxd5 4.d4 Nc6 5.Nf3 cxd4 6.cxd4", "1.e4 c5 2.c3 d5 3.exd5 Qxd5 4.d4 Nc6 5.Nf3 cxd4 6.cxd4 *"),
("B22i", "Sicilian: Alapin, 2...d5 3.exd5 Qxd5 4.d4 Nc6 5.Nf3 Bg4", "1.e4 c5 2.c3 d5 3.exd5 Qxd5 4.d4 Nc6 5.Nf3 Bg4 *"),
("B22j", "Sicilian: Alapin, 2...d5 3.exd5 Qxd5 4.d4 Nf6", "1.e4 c5 2.c3 d5 3.exd5 Qxd5 4.d4 Nf6 *"),
("B22j", "Sicilian: Alapin, 2...d5 3.exd5 Qxd5 4.d4 Nf6 5.Nf3", "1.e4 c5 2.c3 d5 3.exd5 Qxd5 4.d4 Nf6 5.Nf3 *"),
("B22k", "Sicilian: Alapin, 2...d5 3.exd5 Qxd5 4.d4 Nf6 5.Nf3 Bg4", "1.e4 c5 2.c3 d5 3.exd5 Qxd5 4.d4 Nf6 5.Nf3 Bg4 *"),
("B22l", "Sicilian: Alapin, 2...d5 3.exd5 Qxd5 4.d4 Nf6 5.Nf3 Bg4 6.Be2 e6", "1.e4 c5 2.c3 d5 3.exd5 Qxd5 4.d4 Nf6 5.Nf3 Bg4 6.Be2 e6 *"),
("B22m", "Sicilian: Alapin, 2...d5 3.exd5 Qxd5 4.d4 Nf6 5.Nf3 e6 6.Be3", "1.e4 c5 2.c3 d5 3.exd5 Qxd5 4.d4 Nf6 5.Nf3 e6 6.Be3 *"),
("B22n", "Sicilian: Alapin, 2...d5 3.exd5 Qxd5 4.d4 Nf6 5.Nf3 e6 6.Be2", "1.e4 c5 2.c3 d5 3.exd5 Qxd5 4.d4 Nf6 5.Nf3 e6 6.Be2 *"),
("B22o", "Sicilian: Alapin, 2...Nf6", "1.e4 c5 2.c3 Nf6 *"),
("B22o", "Sicilian: Alapin, 2...Nf6 3.e5", "1.e4 c5 2.c3 Nf6 3.e5 *"),
("B22o", "Sicilian: Alapin, 2...Nf6 3.e5 Nd5", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 *"),
("B22o", "Sicilian: Alapin, 2...Nf6 3.e5 Nd5 4.g3", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.g3 *"),
("B22p", "Sicilian: Alapin, 2...Nf6 3.e5 Nd5 4.Nf3", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.Nf3 *"),
("B22p", "Sicilian: Alapin, 2...Nf6 3.e5 Nd5 4.Nf3 Nc6", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.Nf3 Nc6 *"),
("B22p", "Sicilian: Alapin, Heidenfeld Variation", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.Nf3 Nc6 5.Na3 *"),
("B22p", "Sicilian: Alapin, 2...Nf6 3.e5 Nd5 4.d4", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.d4 *"),
("B22q", "Sicilian: Alapin, 2...Nf6 3.e5 Nd5 4.d4 cxd4", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.d4 cxd4 *"),
("B22q", "Sicilian: Alapin, 2...Nf6, 5.Bc4", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.d4 cxd4 5.Bc4 *"),
("B22q", "Sicilian: Alapin, 2...Nf6, 5.Qxd4", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.d4 cxd4 5.Qxd4 *"),
("B22r", "Sicilian: Alapin, 2...Nf6 3.e5 Nd5 4.d4 cxd4 5.Nf3", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.d4 cxd4 5.Nf3 *"),
("B22r", "Sicilian: Alapin, 2...Nf6 3.e5 Nd5 4.d4 cxd4 5.Nf3 e6", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.d4 cxd4 5.Nf3 e6 *"),
("B22r", "Sicilian: Alapin, 2...Nf6 3.e5 Nd5 4.d4 cxd4 5.Nf3 Nc6", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.d4 cxd4 5.Nf3 Nc6 *"),
("B22s", "Sicilian: Alapin, 2...Nf6 3.e5 Nd5 4.d4 cxd4 5.Nf3 Nc6 6.Bc4", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.d4 cxd4 5.Nf3 Nc6 6.Bc4 *"),
("B22t", "Sicilian: Alapin, 2...Nf6, 5.cxd4", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.d4 cxd4 5.cxd4 *"),
("B22t", "Sicilian: Alapin, 2...Nf6, 5.cxd4 d6", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.d4 cxd4 5.cxd4 d6 *"),
("B22t", "Sicilian: Alapin, 2...Nf6, 5.cxd4 Nc6", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.d4 cxd4 5.cxd4 Nc6 *"),
("B22t", "Sicilian: Alapin, 2...Nf6, 5.cxd4 Nc6 6.Nf3", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.d4 cxd4 5.cxd4 Nc6 6.Nf3 *"),
("B22t", "Sicilian: Alapin, 2...Nf6, 5.cxd4 d6", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.d4 cxd4 5.cxd4 d6 6.Nf3 *"),
("B22t", "Sicilian: Alapin, 2...Nf6, 5.cxd4 d6 6.Nf3 Nc6", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.d4 cxd4 5.cxd4 d6 6.Nf3 Nc6 *"),
("B22u", "Sicilian: Alapin, 2...Nf6, 5.cxd4 d6 6.Nf3 Nc6 7.Bc4", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.d4 cxd4 5.cxd4 d6 6.Nf3 Nc6 7.Bc4 *"),
("B22u", "Sicilian: Alapin, 2...Nf6, 5.cxd4 d6 6.Nf3 Nc6 7.Bc4 Nb6", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.d4 cxd4 5.cxd4 d6 6.Nf3 Nc6 7.Bc4 Nb6 *"),
("B22u", "Sicilian: Alapin, 2...Nf6, 5.cxd4 d6 6.Nf3 Nc6 7.Bc4 Nb6 8.Bb5", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.d4 cxd4 5.cxd4 d6 6.Nf3 Nc6 7.Bc4 Nb6 8.Bb5 *"),
("B22v", "Sicilian: Alapin, 2...Nf6, 5.cxd4 e6", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.d4 cxd4 5.cxd4 e6 *"),
("B22v", "Sicilian: Alapin, 2...Nf6, 5.cxd4 e6 6.Nf3", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.d4 cxd4 5.cxd4 e6 6.Nf3 *"),
("B22v", "Sicilian: Alapin, 2...Nf6, 5.cxd4 e6 6.Nf3 Nc6", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.d4 cxd4 5.cxd4 e6 6.Nf3 Nc6 *"),
("B22w", "Sicilian: Alapin, 2...Nf6, 5.cxd4 e6 6.Nf3 d6", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.d4 cxd4 5.cxd4 e6 6.Nf3 d6 *"),
("B22x", "Sicilian: Alapin, 2...Nf6, 5.cxd4 e6 6.Nf3 b6", "1.e4 c5 2.c3 Nf6 3.e5 Nd5 4.d4 cxd4 5.cxd4 e6 6.Nf3 b6 *"),
("B23a", "Sicilian: Closed", "1.e4 c5 2.Nc3 *"),
("B23b", "Sicilian: Closed", "1.e4 c5 2.Nc3 g6 *"),
("B23c", "Sicilian: Closed", "1.e4 c5 2.Nc3 d6 *"),
("B23c", "Sicilian: Closed", "1.e4 c5 2.Nc3 d6 3.Nge2 *"),
("B23d", "Sicilian: Closed", "1.e4 c5 2.Nc3 d6 3.g3 *"),
("B23e", "Sicilian: Closed", "1.e4 c5 2.Nc3 d6 3.f4 *"),
("B23f", "Sicilian: Closed", "1.e4 c5 2.Nc3 e6 *"),
("B23g", "Sicilian: Closed", "1.e4 c5 2.Nc3 e6 3.f4 *"),
("B23h", "Sicilian: Closed", "1.e4 c5 2.Nc3 e6 3.Nge2 *"),
("B23i", "Sicilian: Closed", "1.e4 c5 2.Nc3 e6 3.g3 *"),
("B23j", "Sicilian: Closed, Korchnoi Variation", "1.e4 c5 2.Nc3 e6 3.g3 d5 *"),
("B23k", "Sicilian: Closed, 2...Nc6", "1.e4 c5 2.Nc3 Nc6 *"),
("B23l", "Sicilian: Closed, 2...Nc6 3.Bb5", "1.e4 c5 2.Nc3 Nc6 3.Bb5 *"),
("B23m", "Sicilian: Closed, 2...Nc6 3.Bb5 Nd4", "1.e4 c5 2.Nc3 Nc6 3.Bb5 Nd4 *"),
("B23n", "Sicilian: Chameleon Variation", "1.e4 c5 2.Nc3 Nc6 3.Nge2 *"),
("B23n", "Sicilian: Chameleon, 3...e5", "1.e4 c5 2.Nc3 Nc6 3.Nge2 e5 *"),
("B23n", "Sicilian: Chameleon, 3...g6", "1.e4 c5 2.Nc3 Nc6 3.Nge2 g6 *"),
("B23o", "Sicilian: Closed, Grand Prix", "1.e4 c5 2.Nc3 Nc6 3.f4 *"),
("B23o", "Sicilian: Closed, Grand Prix, 3...d6", "1.e4 c5 2.Nc3 Nc6 3.f4 d6 *"),
("B23p", "Sicilian: Closed, Grand Prix, 3...d6 4.Nf3 g6", "1.e4 c5 2.Nc3 Nc6 3.f4 d6 4.Nf3 g6 *"),
("B23q", "Sicilian: Closed, Grand Prix, 3...e6", "1.e4 c5 2.Nc3 Nc6 3.f4 e6 *"),
("B23q", "Sicilian: Closed, Grand Prix, 3...e6 4.Nf3", "1.e4 c5 2.Nc3 Nc6 3.f4 e6 4.Nf3 *"),
("B23r", "Sicilian: Closed, Grand Prix, 3...e6 4.Nf3 d5", "1.e4 c5 2.Nc3 Nc6 3.f4 e6 4.Nf3 d5 *"),
("B23s", "Sicilian: Closed, Grand Prix, 3...g6", "1.e4 c5 2.Nc3 Nc6 3.f4 g6 *"),
("B23t", "Sicilian: Closed, Grand Prix, 3...g6 4.Nf3 Bg7 5.Bc4", "1.e4 c5 2.Nc3 Nc6 3.f4 g6 4.Nf3 Bg7 5.Bc4 *"),
("B23u", "Sicilian: Closed, Grand Prix, 3...g6 4.Nf3 Bg7 5.Bc4 e6", "1.e4 c5 2.Nc3 Nc6 3.f4 g6 4.Nf3 Bg7 5.Bc4 e6 *"),
("B23v", "Sicilian: Closed, Grand Prix, Schofman Variation", "1.e4 c5 2.Nc3 Nc6 3.f4 g6 4.Nf3 Bg7 5.Bc4 e6 6.f5 *"),
("B23w", "Sicilian: Closed, Grand Prix, 3...g6 4.Nf3 Bg7 5.Bb5", "1.e4 c5 2.Nc3 Nc6 3.f4 g6 4.Nf3 Bg7 5.Bb5 *"),
("B23w", "Sicilian: Closed, Grand Prix, 3...g6 4.Nf3 Bg7 5.Bb5 Nd4", "1.e4 c5 2.Nc3 Nc6 3.f4 g6 4.Nf3 Bg7 5.Bb5 Nd4 *"),
("B23x", "Sicilian: Closed, Grand Prix, 3...g6 4.Nf3 Bg7 5.Bb5 Nd4 6.Bd3", "1.e4 c5 2.Nc3 Nc6 3.f4 g6 4.Nf3 Bg7 5.Bb5 Nd4 6.Bd3 *"),
("B23y", "Sicilian: Closed, Grand Prix, 3...g6 4.Nf3 Bg7 5.Bb5 Nd4 6.O-O", "1.e4 c5 2.Nc3 Nc6 3.f4 g6 4.Nf3 Bg7 5.Bb5 Nd4 6.O-O *"),
("B24", "Sicilian: Closed, 3.g3", "1.e4 c5 2.Nc3 Nc6 3.g3 *"),
("B24", "Sicilian: Closed, 3.g3 e6", "1.e4 c5 2.Nc3 Nc6 3.g3 e6 *"),
("B24", "Sicilian: Closed, 3.g3 g6", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 *"),
("B24", "Sicilian: Closed, 3.g3 g6", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.d3 *"),
("B24", "Sicilian: Closed, 3.g3 g6", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 *"),
("B24", "Sicilian: Closed, 3.g3 g6", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 *"),
("B24", "Sicilian: Closed, 3.g3 g6, 5.Nge2", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.Nge2 *"),
("B24", "Sicilian: Closed, 3.g3 g6, 5.f4", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.f4 *"),
("B24", "Sicilian: Closed, 3.g3 g6, 5.d3", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 *"),
("B24", "Sicilian: Closed, 3.g3 g6, 5.d3 e6", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 e6 *"),
("B24", "Sicilian: Closed, Smyslov Variation", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 e6 6.Be3 Nd4 7.Nce2 *"),
("B24", "Sicilian: Closed, 3.g3 g6, 5.d3 e6 6.f4", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 e6 6.f4 *"),
("B25a", "Sicilian: Closed, 3.g3, 5.d3 d6", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 *"),
("B25b", "Sicilian: Closed, 3.g3, 5.d3 d6 6.Nge2", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Nge2 *"),
("B25c", "Sicilian: Closed, Botvinnik", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Nge2 e5 *"),
("B25d", "Sicilian: Closed, 3.g3, 5.d3 d6 6.Nge2 e6", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Nge2 e6 *"),
("B25e", "Sicilian: Closed, 6.f4", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.f4 *"),
("B25f", "Sicilian: Closed, 6.f4 Rb8", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.f4 Rb8 *"),
("B25g", "Sicilian: Closed, 6.f4 Nf6", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.f4 Nf6 *"),
("B25h", "Sicilian: Closed, 6.f4 Nf6 7.Nf3", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.f4 Nf6 7.Nf3 *"),
("B25i", "Sicilian: Closed, 6.f4 e5", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.f4 e5 *"),
("B25j", "Sicilian: Closed, 6.f4 e5 7.Nf3", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.f4 e5 7.Nf3 *"),
("B25j", "Sicilian: Closed, 6.f4 e5 7.Nf3 Nge7", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.f4 e5 7.Nf3 Nge7 *"),
("B25k", "Sicilian: Closed, 6.f4 e5 7.Nf3 Nge7 8.O-O", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.f4 e5 7.Nf3 Nge7 8.O-O *"),
("B25l", "Sicilian: Closed, 6.f4 e6", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.f4 e6 *"),
("B25m", "Sicilian: Closed, 6.f4 e6", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.f4 e6 7.Nf3 *"),
("B25n", "Sicilian: Closed, 6.f4 e6", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.f4 e6 7.Nf3 Nge7 *"),
("B25o", "Sicilian: Closed, 6.f4 e6", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.f4 e6 7.Nf3 Nge7 8.O-O *"),
("B25p", "Sicilian: Closed, 6.f4 e6", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.f4 e6 7.Nf3 Nge7 8.O-O O-O *"),
("B25q", "Sicilian: Closed, 6.f4 e6: 9.Be3", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.f4 e6 7.Nf3 Nge7 8.O-O O-O 9.Be3 *"),
("B25r", "Sicilian: Closed, 6.f4 e6: 9.Be3 Rb8", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.f4 e6 7.Nf3 Nge7 8.O-O O-O 9.Be3 Rb8 *"),
("B25s", "Sicilian: Closed, 6.f4 e6: 9.Be3 b6", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.f4 e6 7.Nf3 Nge7 8.O-O O-O 9.Be3 b6 *"),
("B25t", "Sicilian: Closed, 6.f4 e6: 9.Be3 Nd4", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.f4 e6 7.Nf3 Nge7 8.O-O O-O 9.Be3 Nd4 *"),
("B25u", "Sicilian: Closed, 6.f4 e6: 9.Be3 Nd4 10.Rb1", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.f4 e6 7.Nf3 Nge7 8.O-O O-O 9.Be3 Nd4 10.Rb1 *"),
("B25v", "Sicilian: Closed, 6.f4 e6: 9.Be3 Nd4 10.Qd2", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.f4 e6 7.Nf3 Nge7 8.O-O O-O 9.Be3 Nd4 10.Qd2 *"),
("B25w", "Sicilian: Closed, 6.f4 e6: 9.Be3 Nd4 10.e5", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.f4 e6 7.Nf3 Nge7 8.O-O O-O 9.Be3 Nd4 10.e5 *"),
("B25x", "Sicilian: Closed, 6.f4 e6: 9.Be3 Nd4 10.e5 Nef5", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.f4 e6 7.Nf3 Nge7 8.O-O O-O 9.Be3 Nd4 10.e5 Nef5 *"),
("B26a", "Sicilian: Closed, 6.Be3", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Be3 *"),
("B26b", "Sicilian: Closed, 6.Be3 Nf6", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Be3 Nf6 *"),
("B26c", "Sicilian: Closed, 6.Be3 Nf6 7.h3", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Be3 Nf6 7.h3 *"),
("B26d", "Sicilian: Closed, 6.Be3 e5", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Be3 e5 *"),
("B26e", "Sicilian: Closed, 6.Be3 e5 7.Qd2", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Be3 e5 7.Qd2 *"),
("B26f", "Sicilian: Closed, 6.Be3 e5 7.Qd2 Nge7", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Be3 e5 7.Qd2 Nge7 *"),
("B26g", "Sicilian: Closed, 6.Be3 Rb8", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Be3 Rb8 *"),
("B26h", "Sicilian: Closed, 6.Be3 Rb8 7.Qd2", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Be3 Rb8 7.Qd2 *"),
("B26i", "Sicilian: Closed, 6.Be3 Rb8 7.Qd2 b5", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Be3 Rb8 7.Qd2 b5 *"),
("B26j", "Sicilian: Closed, 6.Be3 Rb8 7.Qd2 b5 8.Nge2", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Be3 Rb8 7.Qd2 b5 8.Nge2 *"),
("B26k", "Sicilian: Closed, 6.Be3 e6", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Be3 e6 *"),
("B26l", "Sicilian: Closed, 6.Be3 e6 7.Qd2", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Be3 e6 7.Qd2 *"),
("B26m", "Sicilian: Closed, 6.Be3 e6 7.Qd2 Qa5", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Be3 e6 7.Qd2 Qa5 *"),
("B26n", "Sicilian: Closed, 6.Be3 e6 7.Qd2 Nd4", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Be3 e6 7.Qd2 Nd4 *"),
("B26o", "Sicilian: Closed, 6.Be3 e6 7.Qd2 Nge7", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Be3 e6 7.Qd2 Nge7 *"),
("B26p", "Sicilian: Closed, 6.Be3 e6 7.Qd2 Rb8", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Be3 e6 7.Qd2 Rb8 *"),
("B26q", "Sicilian: Closed, 6.Be3 e6 7.Qd2 Rb8 8.Nge2", "1.e4 c5 2.Nc3 Nc6 3.g3 g6 4.Bg2 Bg7 5.d3 d6 6.Be3 e6 7.Qd2 Rb8 8.Nge2 *"),
("B27a", "Sicilian: 2.Nf3", "1.e4 c5 2.Nf3 *"),
("B27b", "Sicilian: Mongoose Variation", "1.e4 c5 2.Nf3 Qa5 *"),
("B27c", "Sicilian: Quinteros Variation", "1.e4 c5 2.Nf3 Qc7 *"),
("B27d", "Sicilian: Katalimov Variation", "1.e4 c5 2.Nf3 b6 *"),
("B27e", "Sicilian: Hungarian Variation", "1.e4 c5 2.Nf3 g6 *"),
("B27f", "Sicilian: Hungarian, 3.c4", "1.e4 c5 2.Nf3 g6 3.c4 *"),
("B27g", "Sicilian: Hungarian, 3.c3", "1.e4 c5 2.Nf3 g6 3.c3 *"),
("B27h", "Sicilian: Hungarian, 3.c3 Bg7 4.d4 cxd4 5.cxd4", "1.e4 c5 2.Nf3 g6 3.c3 Bg7 4.d4 cxd4 5.cxd4 *"),
("B27i", "Sicilian: Hungarian, 3.c3 Bg7 4.d4 cxd4 5.cxd4 d5", "1.e4 c5 2.Nf3 g6 3.c3 Bg7 4.d4 cxd4 5.cxd4 d5 *"),
("B27j", "Sicilian: Hungarian, 3.d4", "1.e4 c5 2.Nf3 g6 3.d4 *"),
("B27k", "Sicilian: Hungarian, 3.d4 cxd4", "1.e4 c5 2.Nf3 g6 3.d4 cxd4 *"),
("B27l", "Sicilian: Hungarian, 3.d4 cxd4 4.Nxd4", "1.e4 c5 2.Nf3 g6 3.d4 cxd4 4.Nxd4 *"),
("B27m", "Sicilian: Accelerated Pterodactyl", "1.e4 c5 2.Nf3 g6 3.d4 Bg7 *"),
("B27n", "Sicilian: Accelerated Pterodactyl, 4.dxc5", "1.e4 c5 2.Nf3 g6 3.d4 Bg7 4.dxc5 *"),
("B27o", "Sicilian: Accelerated Pterodactyl, 4.dxc5 Qa5+", "1.e4 c5 2.Nf3 g6 3.d4 Bg7 4.dxc5 Qa5+ *"),
("B27p", "Sicilian: Accelerated Pterodactyl, 4.dxc5 Qa5+ 5.Nc3", "1.e4 c5 2.Nf3 g6 3.d4 Bg7 4.dxc5 Qa5+ 5.Nc3 *"),
("B27q", "Sicilian: Accelerated Pterodactyl, 4.dxc5 Qa5+ 5.c3", "1.e4 c5 2.Nf3 g6 3.d4 Bg7 4.dxc5 Qa5+ 5.c3 *"),
("B27r", "Sicilian: Accelerated Pterodactyl, 4.dxc5 Qa5+ 5.c3", "1.e4 c5 2.Nf3 g6 3.d4 Bg7 4.dxc5 Qa5+ 5.c3 Qxc5 *"),
("B27s", "Sicilian: Accelerated Pterodactyl, 4.c4", "1.e4 c5 2.Nf3 g6 3.d4 Bg7 4.c4 *"),
("B27t", "Sicilian: Accelerated Pterodactyl, 4.c4 Qa5+", "1.e4 c5 2.Nf3 g6 3.d4 Bg7 4.c4 Qa5+ *"),
("B27t", "Sicilian: Accelerated Pterodactyl, 4.Nc3", "1.e4 c5 2.Nf3 g6 3.d4 Bg7 4.Nc3 *"),
("B28a", "Sicilian: O'Kelly Variation", "1.e4 c5 2.Nf3 a6 *"),
("B28b", "Sicilian: O'Kelly, 3.Nc3", "1.e4 c5 2.Nf3 a6 3.Nc3 *"),
("B28c", "Sicilian: O'Kelly, 3.d4", "1.e4 c5 2.Nf3 a6 3.d4 *"),
("B28d", "Sicilian: O'Kelly, 3.d4 cxd4 4.Nxd4", "1.e4 c5 2.Nf3 a6 3.d4 cxd4 4.Nxd4 *"),
("B28e", "Sicilian: O'Kelly, 3.d4 cxd4 4.Nxd4 Nf6", "1.e4 c5 2.Nf3 a6 3.d4 cxd4 4.Nxd4 Nf6 *"),
("B28f", "Sicilian: O'Kelly, 3.c4", "1.e4 c5 2.Nf3 a6 3.c4 *"),
("B28g", "Sicilian: O'Kelly, 3.c4 e6", "1.e4 c5 2.Nf3 a6 3.c4 e6 *"),
("B28h", "Sicilian: O'Kelly, 3.c3", "1.e4 c5 2.Nf3 a6 3.c3 *"),
("B28i", "Sicilian: O'Kelly, 3.c3 e6", "1.e4 c5 2.Nf3 a6 3.c3 e6 *"),
("B28j", "Sicilian: O'Kelly, 3.c3 d5", "1.e4 c5 2.Nf3 a6 3.c3 d5 *"),
("B29a", "Sicilian: Nimzowitsch", "1.e4 c5 2.Nf3 Nf6 *"),
("B29b", "Sicilian: Nimzowitsch, 3.d3", "1.e4 c5 2.Nf3 Nf6 3.d3 *"),
("B29c", "Sicilian: Nimzowitsch, 3.Nc3", "1.e4 c5 2.Nf3 Nf6 3.Nc3 *"),
("B29d", "Sicilian: Nimzowitsch, 3.Nc3 d5", "1.e4 c5 2.Nf3 Nf6 3.Nc3 d5 *"),
("B29e", "Sicilian: Nimzowitsch, 3.Nc3 d5 4.exd5", "1.e4 c5 2.Nf3 Nf6 3.Nc3 d5 4.exd5 *"),
("B29f", "Sicilian: Nimzowitsch, 3.e5", "1.e4 c5 2.Nf3 Nf6 3.e5 *"),
("B29f", "Sicilian: Nimzowitsch, 3.e5 Nd5", "1.e4 c5 2.Nf3 Nf6 3.e5 Nd5 *"),
("B29g", "Sicilian: Nimzowitsch, 4.c4", "1.e4 c5 2.Nf3 Nf6 3.e5 Nd5 4.c4 *"),
("B29h", "Sicilian: Nimzowitsch, 4.d4", "1.e4 c5 2.Nf3 Nf6 3.e5 Nd5 4.d4 *"),
("B29i", "Sicilian: Nimzowitsch, 4.Nc3", "1.e4 c5 2.Nf3 Nf6 3.e5 Nd5 4.Nc3 *"),
("B29j", "Sicilian: Nimzowitsch, 4.Nc3 Nxc3", "1.e4 c5 2.Nf3 Nf6 3.e5 Nd5 4.Nc3 Nxc3 *"),
("B29k", "Sicilian: Nimzowitsch, 4.Nc3 e6", "1.e4 c5 2.Nf3 Nf6 3.e5 Nd5 4.Nc3 e6 *"),
("B29l", "Sicilian: Nimzowitsch, 4.Nc3 e6 5.Nxd5", "1.e4 c5 2.Nf3 Nf6 3.e5 Nd5 4.Nc3 e6 5.Nxd5 *"),
("B29m", "Sicilian: Nimzowitsch, Rubinstein Countergambit", "1.e4 c5 2.Nf3 Nf6 3.e5 Nd5 4.Nc3 e6 5.Nxd5 exd5 6.d4 Nc6 *"),
("B29n", "Sicilian: Nimzowitsch, Rubinstein Countergambit, 7.dxc5", "1.e4 c5 2.Nf3 Nf6 3.e5 Nd5 4.Nc3 e6 5.Nxd5 exd5 6.d4 Nc6 7.dxc5 *"),
("B30a", "Sicilian: 2...Nc6", "1.e4 c5 2.Nf3 Nc6 *"),
("B30b", "Sicilian: 2...Nc6 3.b3", "1.e4 c5 2.Nf3 Nc6 3.b3 *"),
("B30c", "Sicilian: 2...Nc6 3.d3", "1.e4 c5 2.Nf3 Nc6 3.d3 *"),
("B30c", "Sicilian: 2...Nc6 3.d3 Nf6", "1.e4 c5 2.Nf3 Nc6 3.d3 Nf6 *"),
("B30d", "Sicilian: 2...Nc6 3.g3", "1.e4 c5 2.Nf3 Nc6 3.g3 *"),
("B30e", "Sicilian: 2...Nc6 3.g3 e6 4.d3", "1.e4 c5 2.Nf3 Nc6 3.g3 e6 4.d3 *"),
("B30f", "Sicilian: 2...Nc6 3.g3 e6 4.d3 d5", "1.e4 c5 2.Nf3 Nc6 3.g3 e6 4.d3 d5 *"),
("B30g", "Sicilian: 2...Nc6 3.Bc4", "1.e4 c5 2.Nf3 Nc6 3.Bc4 *"),
("B30h", "Sicilian: 2...Nc6 3.Nc3", "1.e4 c5 2.Nf3 Nc6 3.Nc3 *"),
("B30h", "Sicilian: 2...Nc6 3.Nc3 Nf6", "1.e4 c5 2.Nf3 Nc6 3.Nc3 Nf6 *"),
("B30i", "Sicilian: 2...Nc6 3.Nc3 g6", "1.e4 c5 2.Nf3 Nc6 3.Nc3 g6 *"),
("B30j", "Sicilian: 2...Nc6 3.Nc3 e5", "1.e4 c5 2.Nf3 Nc6 3.Nc3 e5 *"),
("B30k", "Sicilian: Rossolimo", "1.e4 c5 2.Nf3 Nc6 3.Bb5 *"),
("B30l", "Sicilian: Rossolimo, 3...Qb6", "1.e4 c5 2.Nf3 Nc6 3.Bb5 Qb6 *"),
("B30m", "Sicilian: Rossolimo, 3...Nf6", "1.e4 c5 2.Nf3 Nc6 3.Bb5 Nf6 *"),
("B30n", "Sicilian: Rossolimo, 3...e6", "1.e4 c5 2.Nf3 Nc6 3.Bb5 e6 *"),
("B30o", "Sicilian: Rossolimo, 3...e6 4.b3", "1.e4 c5 2.Nf3 Nc6 3.Bb5 e6 4.b3 *"),
("B30p", "Sicilian: Rossolimo, 3...e6 4.Nc3", "1.e4 c5 2.Nf3 Nc6 3.Bb5 e6 4.Nc3 *"),
("B30q", "Sicilian: Rossolimo, 3...e6 4.Bxc6", "1.e4 c5 2.Nf3 Nc6 3.Bb5 e6 4.Bxc6 *"),
("B30r", "Sicilian: Rossolimo, 3...e6 4.O-O", "1.e4 c5 2.Nf3 Nc6 3.Bb5 e6 4.O-O *"),
("B30s", "Sicilian: Rossolimo, 3...e6 4.O-O Nge7", "1.e4 c5 2.Nf3 Nc6 3.Bb5 e6 4.O-O Nge7 *"),
("B30s", "Sicilian: Rossolimo, 3...e6 4.O-O Nge7 5.b3", "1.e4 c5 2.Nf3 Nc6 3.Bb5 e6 4.O-O Nge7 5.b3 *"),
("B30t", "Sicilian: Rossolimo, 3...e6 4.O-O Nge7 5.Re1", "1.e4 c5 2.Nf3 Nc6 3.Bb5 e6 4.O-O Nge7 5.Re1 *"),
("B30u", "Sicilian: Rossolimo, 3...e6 4.O-O Nge7 5.Nc3", "1.e4 c5 2.Nf3 Nc6 3.Bb5 e6 4.O-O Nge7 5.Nc3 *"),
("B30v", "Sicilian: Rossolimo, 3...e6 4.O-O Nge7 5.c3", "1.e4 c5 2.Nf3 Nc6 3.Bb5 e6 4.O-O Nge7 5.c3 *"),
("B30w", "Sicilian: Rossolimo, 3...e6 4.O-O Nge7 5.c3 a6", "1.e4 c5 2.Nf3 Nc6 3.Bb5 e6 4.O-O Nge7 5.c3 a6 *"),
("B31a", "Sicilian: Rossolimo, 3...g6", "1.e4 c5 2.Nf3 Nc6 3.Bb5 g6 *"),
("B31b", "Sicilian: Rossolimo, 3...g6 4.Nc3", "1.e4 c5 2.Nf3 Nc6 3.Bb5 g6 4.Nc3 *"),
("B31c", "Sicilian: Rossolimo, 3...g6 4.Bxc6", "1.e4 c5 2.Nf3 Nc6 3.Bb5 g6 4.Bxc6 *"),
("B31d", "Sicilian: Rossolimo, 3...g6 4.Bxc6 dxc6", "1.e4 c5 2.Nf3 Nc6 3.Bb5 g6 4.Bxc6 dxc6 *"),
("B31e", "Sicilian: Rossolimo, 3...g6 4.Bxc6 dxc6 5.d3", "1.e4 c5 2.Nf3 Nc6 3.Bb5 g6 4.Bxc6 dxc6 5.d3 *"),
("B31f", "Sicilian: Rossolimo, 3...g6 4.Bxc6 dxc6 5.d3 Bg7 6.h3", "1.e4 c5 2.Nf3 Nc6 3.Bb5 g6 4.Bxc6 dxc6 5.d3 Bg7 6.h3 *"),
("B31g", "Sicilian: Rossolimo, 3...g6 4.O-O", "1.e4 c5 2.Nf3 Nc6 3.Bb5 g6 4.O-O *"),
("B31h", "Sicilian: Rossolimo, 3...g6 4.O-O Bg7 5.c3", "1.e4 c5 2.Nf3 Nc6 3.Bb5 g6 4.O-O Bg7 5.c3 *"),
("B31i", "Sicilian: Rossolimo, 3...g6 4.O-O Bg7 5.c3 e5", "1.e4 c5 2.Nf3 Nc6 3.Bb5 g6 4.O-O Bg7 5.c3 e5 *"),
("B31j", "Sicilian: Rossolimo, 3...g6 4.O-O Bg7 5.Re1", "1.e4 c5 2.Nf3 Nc6 3.Bb5 g6 4.O-O Bg7 5.Re1 *"),
("B31k", "Sicilian: Rossolimo, 3...g6 4.O-O Bg7 5.Re1 Nf6", "1.e4 c5 2.Nf3 Nc6 3.Bb5 g6 4.O-O Bg7 5.Re1 Nf6 *"),
("B31l", "Sicilian: Rossolimo, 3...g6 4.O-O Bg7 5.Re1 Nf6 6.c3", "1.e4 c5 2.Nf3 Nc6 3.Bb5 g6 4.O-O Bg7 5.Re1 Nf6 6.c3 *"),
("B31m", "Sicilian: Rossolimo, 3...g6 4.O-O Bg7 5.Re1 Nf6 6.c3 O-O 7.h3", "1.e4 c5 2.Nf3 Nc6 3.Bb5 g6 4.O-O Bg7 5.Re1 Nf6 6.c3 O-O 7.h3 *"),
("B31n", "Sicilian: Rossolimo, 3...g6 4.O-O Bg7 5.Re1 Nf6 6.c3 O-O 7.d4", "1.e4 c5 2.Nf3 Nc6 3.Bb5 g6 4.O-O Bg7 5.Re1 Nf6 6.c3 O-O 7.d4 *"),
("B31o", "Sicilian: Rossolimo, 3...g6 4.O-O Bg7 5.Re1 e5", "1.e4 c5 2.Nf3 Nc6 3.Bb5 g6 4.O-O Bg7 5.Re1 e5 *"),
("B31p", "Sicilian: Rossolimo, Gurgenidze Variation", "1.e4 c5 2.Nf3 Nc6 3.Bb5 g6 4.O-O Bg7 5.Re1 e5 6.b4 *"),
("B31q", "Sicilian: Rossolimo, 3...g6 4.O-O Bg7 5.Re1 e5", "1.e4 c5 2.Nf3 Nc6 3.Bb5 g6 4.O-O Bg7 5.Re1 e5 6.c3 *"),
("B32a", "Sicilian: 2...Nc6 3.d4", "1.e4 c5 2.Nf3 Nc6 3.d4 *"),
("B32b", "Sicilian: 2...Nc6 3.d4 cxd4", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 *"),
("B32c", "Sicilian: Open, 2...Nc6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 *"),
("B32d", "Sicilian: Open, 2...Nc6, 4...d6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 d6 *"),
("B32e", "Sicilian: Open, 2...Nc6, 4...Qb6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Qb6 *"),
("B32f", "Sicilian: Flohr Variation", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Qc7 *"),
("B32g", "Sicilian: Flohr, 5.Nb5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Qc7 5.Nb5 *"),
("B32h", "Sicilian: 2..Nc6, Nimzowitsch Variation", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 d5 *"),
("B32i", "Sicilian: Loewenthal", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 e5 *"),
("B32j", "Sicilian: Loewenthal, 5.Nb5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 e5 5.Nb5 *"),
("B32k", "Sicilian: Loewenthal, 5...a6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 e5 5.Nb5 a6 *"),
("B32l", "Sicilian: Loewenthal, 5...a6, Queenswap", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 e5 5.Nb5 a6 6.Nd6+ Bxd6 7.Qxd6 Qf6 8.Qxf6 *"),
("B32m", "Sicilian: Loewenthal, 5...a6, 8.Qd1", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 e5 5.Nb5 a6 6.Nd6+ Bxd6 7.Qxd6 Qf6 8.Qd1 *"),
("B32n", "Sicilian: Loewenthal, Kalashnikov", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 e5 5.Nb5 d6 *"),
("B32o", "Sicilian: Loewenthal, Kalashnikov, 6.a4", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 e5 5.Nb5 d6 6.a4 *"),
("B32p", "Sicilian: Loewenthal, Kalashnikov, 6.N1c3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 e5 5.Nb5 d6 6.N1c3 *"),
("B32q", "Sicilian: Loewenthal, Kalashnikov, 6.N1c3 a6 7.Na3 b5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 e5 5.Nb5 d6 6.N1c3 a6 7.Na3 b5 *"),
("B32r", "Sicilian: Loewenthal, Kalashnikov, 6.N1c3 a6 7.Na3 b5 8.Nd5 Nge7", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 e5 5.Nb5 d6 6.N1c3 a6 7.Na3 b5 8.Nd5 Nge7 *"),
("B32s", "Sicilian: Loewenthal, Kalashnikov, 6.c4", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 e5 5.Nb5 d6 6.c4 *"),
("B32t", "Sicilian: Loewenthal, Kalashnikov, 6.c4 Be7", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 e5 5.Nb5 d6 6.c4 Be7 *"),
("B32u", "Sicilian: Loewenthal, Kalashnikov, 6.c4 Be7 7.Be2", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 e5 5.Nb5 d6 6.c4 Be7 7.Be2 *"),
("B32v", "Sicilian: Loewenthal, Kalashnikov, 6.c4 Be7 7.N1c3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 e5 5.Nb5 d6 6.c4 Be7 7.N1c3 *"),
("B32w", "Sicilian: Loewenthal, Kalashnikov, Main Line", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 e5 5.Nb5 d6 6.c4 Be7 7.N1c3 a6 8.Na3 Be6 *"),
("B32x", "Sicilian: Loewenthal, Kalashnikov, Main Line, 9.Nc2", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 e5 5.Nb5 d6 6.c4 Be7 7.N1c3 a6 8.Na3 Be6 9.Nc2 *"),
("B32y", "Sicilian: Loewenthal, Kalashnikov, Main Line, 9.Be2", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 e5 5.Nb5 d6 6.c4 Be7 7.N1c3 a6 8.Na3 Be6 9.Be2 *"),
("B33a", "Sicilian: Open, 2...Nc6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 *"),
("B33b", "Sicilian: Open, 2...Nc6, 5...Qb6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Qb6 *"),
("B33b", "Sicilian: Open, 2...Nc6, 5...Qb6 6.Nb3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Qb6 6.Nb3 *"),
("B33b", "Sicilian: Open, 2...Nc6, 5...Qb6 6.Nb3 e6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Qb6 6.Nb3 e6 *"),
("B33c", "Sicilian: Open, 2...Nc6, 5...Qb6 6.Nb3 e6 7.Bd3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Qb6 6.Nb3 e6 7.Bd3 *"),
("B33d", "Sicilian: Pelikan/Sveshnikov", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 *"),
("B33d", "Sicilian: Pelikan/Sveshnikov, 6.Nxc6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Nxc6 *"),
("B33d", "Sicilian: Pelikan/Sveshnikov, 6.Nf5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Nf5 *"),
("B33d", "Sicilian: Pelikan/Sveshnikov, 6.Nf3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Nf3 *"),
("B33d", "Sicilian: Pelikan/Sveshnikov, 6.Nb3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Nb3 *"),
("B33e", "Sicilian: Pelikan/Sveshnikov, 6.Nbd5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 *"),
("B33e", "Sicilian: Pelikan, Haberditz Variation", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 h6 *"),
("B33f", "Sicilian: Pelikan/Sveshnikov, Pelikan Variation", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 *"),
("B33f", "Sicilian: Pelikan, 7.Na3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Na3 *"),
("B33g", "Sicilian: Pelikan, 7.a4", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.a4 *"),
("B33g", "Sicilian: Pelikan, 7.a4", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.a4 a6 8.Na3 *"),
("B33h", "Sicilian: Pelikan, 7.Nd5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Nd5 *"),
("B33i", "Sicilian: Pelikan, 7.Bg5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 *"),
("B33i", "Sicilian: Pelikan, 7.Bg5 a6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 *"),
("B33j", "Sicilian: Pelikan, 7.Bg5 a6 8.Bxf6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 8.Bxf6 *"),
("B33k", "Sicilian: Pelikan, 7.Bg5 a6 8.Na3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 8.Na3 *"),
("B33k", "Sicilian: Pelikan, Bird Variation", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 8.Na3 Be6 *"),
("B33l", "Sicilian: Pelikan, Chelyabinsk Variation", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 8.Na3 b5 *"),
("B33l", "Sicilian: Pelikan, Chelyabinsk, 9.Nd5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 8.Na3 b5 9.Nd5 *"),
("B33m", "Sicilian: Pelikan, Chelyabinsk, 9.Nd5 Be7", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 8.Na3 b5 9.Nd5 Be7 *"),
("B33n", "Sicilian: Pelikan, Chelyabinsk, 9.Nd5 Be7, 11.c3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 8.Na3 b5 9.Nd5 Be7 10.Bxf6 Bxf6 11.c3 *"),
("B33o", "Sicilian: Pelikan, Chelyabinsk, 9.Nd5 Be7, 11.c3 O-O", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 8.Na3 b5 9.Nd5 Be7 10.Bxf6 Bxf6 11.c3 O-O *"),
("B33p", "Sicilian: Pelikan, Chelyabinsk, 9.Nd5 Be7, 11.c3 Bg5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 8.Na3 b5 9.Nd5 Be7 10.Bxf6 Bxf6 11.c3 Bg5 *"),
("B33q", "Sicilian: Pelikan, Chelyabinsk, 9.Nd5 Be7, 11.c3 O-O 12.Nc2 Bg5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 8.Na3 b5 9.Nd5 Be7 10.Bxf6 Bxf6 11.c3 O-O 12.Nc2 Bg5 *"),
("B33r", "Sicilian: Pelikan, Chelyabinsk, 9.Nd5 Be7, 11.c3 O-O 12.Nc2 Bg5 13.a4", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 8.Na3 b5 9.Nd5 Be7 10.Bxf6 Bxf6 11.c3 O-O 12.Nc2 Bg5 13.a4 *"),
("B33s", "Sicilian: Pelikan, Chelyabinsk, 9.Bxf6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 8.Na3 b5 9.Bxf6 *"),
("B33s", "Sicilian: Pelikan, Chelyabinsk, 9.Bxf6 gxf6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 8.Na3 b5 9.Bxf6 gxf6 *"),
("B33s", "Sicilian: Pelikan, Chelyabinsk, 9.Bxf6 gxf6 10.Nd5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 8.Na3 b5 9.Bxf6 gxf6 10.Nd5 *"),
("B33s", "Sicilian: Pelikan, Chelyabinsk, 9.Bxf6 gxf6 10.Nd5 Bg7", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 8.Na3 b5 9.Bxf6 gxf6 10.Nd5 Bg7 *"),
("B33t", "Sicilian: Pelikan, Chelyabinsk, 9.Bxf6 gxf6 10.Nd5 Bg7 11.Bd3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 8.Na3 b5 9.Bxf6 gxf6 10.Nd5 Bg7 11.Bd3 *"),
("B33u", "Sicilian: Pelikan, Sveshnikov Variation", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 8.Na3 b5 9.Bxf6 gxf6 10.Nd5 f5 *"),
("B33v", "Sicilian: Pelikan, Sveshnikov, 11.c3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 8.Na3 b5 9.Bxf6 gxf6 10.Nd5 f5 11.c3 *"),
("B33v", "Sicilian: Pelikan, Sveshnikov, 11.c3 Bg7", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 8.Na3 b5 9.Bxf6 gxf6 10.Nd5 f5 11.c3 Bg7 *"),
("B33v", "Sicilian: Pelikan, Sveshnikov, 11.c3 Bg7 12.exf5 Bxf5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 8.Na3 b5 9.Bxf6 gxf6 10.Nd5 f5 11.c3 Bg7 12.exf5 Bxf5 *"),
("B33w", "Sicilian: Pelikan, Sveshnikov, 11.c3 Bg7 12.exf5 Bxf5 13.Nc2 O-O", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 8.Na3 b5 9.Bxf6 gxf6 10.Nd5 f5 11.c3 Bg7 12.exf5 Bxf5 13.Nc2 O-O *"),
("B33x", "Sicilian: Pelikan, Sveshnikov, 11.Bd3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 8.Na3 b5 9.Bxf6 gxf6 10.Nd5 f5 11.Bd3 *"),
("B33x", "Sicilian: Pelikan, Sveshnikov, 11.Bd3 Be6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 8.Na3 b5 9.Bxf6 gxf6 10.Nd5 f5 11.Bd3 Be6 *"),
("B33y", "Sicilian: Pelikan, Sveshnikov, 11.Bd3 Be6 12.O-O", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Ndb5 d6 7.Bg5 a6 8.Na3 b5 9.Bxf6 gxf6 10.Nd5 f5 11.Bd3 Be6 12.O-O *"),
("B34a", "Sicilian: Accelerated Fianchetto", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 *"),
("B34b", "Sicilian: Accelerated Fianchetto, 5.Bc4", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Bc4 *"),
("B34c", "Sicilian: Accelerated Fianchetto, Exchange Variation", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nxc6 *"),
("B34d", "Sicilian: Accelerated Fianchetto, 5.Be2", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Be2 *"),
("B34e", "Sicilian: Accelerated Fianchetto, 5.Be3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Be3 *"),
("B34f", "Sicilian: Accelerated Fianchetto, Modern Variation", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 *"),
("B34g", "Sicilian: Accelerated Fianchetto, Modern, 5...Nf6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Nf6 *"),
("B34h", "Sicilian: Accelerated Fianchetto, Modern, 5...Nf6 6.Nxc6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Nf6 6.Nxc6 *"),
("B34i", "Sicilian: Accelerated Fianchetto, Modern, 5...Nf6 6.Nxc6 bxc6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Nf6 6.Nxc6 bxc6 *"),
("B34j", "Sicilian: Accelerated Fianchetto, Modern, 5...Bg7", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 *"),
("B34k", "Sicilian: Accelerated Fianchetto, Modern, 6.Nde2", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Nde2 *"),
("B34l", "Sicilian: Accelerated Fianchetto, Modern, 6.Nb3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Nb3 *"),
("B34m", "Sicilian: Accelerated Fianchetto, Modern, 6.Be3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 *"),
("B34n", "Sicilian: Accelerated Fianchetto, Modern, 6.Be3 d6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 d6 *"),
("B34o", "Sicilian: Accelerated Fianchetto, Modern, 6.Be3 d6 7.Qd2", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 d6 7.Qd2 *"),
("B34p", "Sicilian: Accelerated Fianchetto, Modern, 6.Be3 Nf6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 *"),
("B34q", "Sicilian: Accelerated Fianchetto, Modern, 6.Be3 Nf6 7.f3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.f3 *"),
("B34r", "Sicilian: Accelerated Fianchetto, Modern, 6.Be3 Nf6 7.Nxc6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Nxc6 *"),
("B34r", "Sicilian: Accelerated Fianchetto, Modern, 6.Be3 Nf6 7.Nxc6 bxc6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Nxc6 bxc6 *"),
("B34r", "Sicilian: Accelerated Fianchetto, Modern, 6.Be3 Nf6 7.Nxc6 bxc6 8.e5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Nxc6 bxc6 8.e5 *"),
("B34s", "Sicilian: Accelerated Fianchetto, Modern, 6.Be3 Nf6 7.Nxc6 bxc6 8.e5 Ng8", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Nxc6 bxc6 8.e5 Ng8 *"),
("B34t", "Sicilian: Accelerated Fianchetto, Modern, 6.Be3 Nf6 7.Be2", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Be2 *"),
("B34u", "Sicilian: Accelerated Fianchetto, Modern, 6.Be3 Nf6 7.Be2 O-O", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Be2 O-O *"),
("B34v", "Sicilian: Accelerated Fianchetto, Modern, 6.Be3 Nf6 7.Be2 O-O 8.O-O", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Be2 O-O 8.O-O *"),
("B34v", "Sicilian: Accelerated Fianchetto, Modern, 6.Be3 Nf6 7.Be2 O-O 8.O-O d5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Be2 O-O 8.O-O d5 *"),
("B34w", "Sicilian: Accelerated Fianchetto, Modern, 6.Be3 Nf6 7.Be2 O-O 8.O-O d5 9.exd5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Be2 O-O 8.O-O d5 9.exd5 *"),
("B35a", "Sicilian: Accelerated Fianchetto, Modern, 7.Bc4", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Bc4 *"),
("B35b", "Sicilian: Accelerated Fianchetto, Modern, 7.Bc4 Qa5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Bc4 Qa5 *"),
("B35c", "Sicilian: Accelerated Fianchetto, Modern, 7.Bc4 Qa5 8.O-O O-O", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Bc4 Qa5 8.O-O O-O *"),
("B35d", "Sicilian: Accelerated Fianchetto, Modern, 7.Bc4 Qa5 8.O-O O-O 9.Bb3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Bc4 Qa5 8.O-O O-O 9.Bb3 *"),
("B35e", "Sicilian: Accelerated Fianchetto, Modern, 7.Bc4 Qa5 8.O-O O-O 9.Bb3 d6 10.h3 Bd7 11.f4", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Bc4 Qa5 8.O-O O-O 9.Bb3 d6 10.h3 Bd7 11.f4 *"),
("B35f", "Sicilian: Accelerated Fianchetto, Modern, 7.Bc4 O-O", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Bc4 O-O *"),
("B35g", "Sicilian: Accelerated Fianchetto, Modern, 7.Bc4 O-O 8.f3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Bc4 O-O 8.f3 *"),
("B35h", "Sicilian: Accelerated Fianchetto, Modern, 7.Bc4 O-O 8.O-O", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Bc4 O-O 8.O-O *"),
("B35i", "Sicilian: Accelerated Fianchetto, Modern, 7.Bc4 O-O 8.Bb3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Bc4 O-O 8.Bb3 *"),
("B35j", "Sicilian: Accelerated Fianchetto, Modern, 7.Bc4 O-O 8.Bb3 Ng4", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Bc4 O-O 8.Bb3 Ng4 *"),
("B35k", "Sicilian: Accelerated Fianchetto, Modern, 7.Bc4 O-O 8.Bb3 Qa5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Bc4 O-O 8.Bb3 Qa5 *"),
("B35l", "Sicilian: Accelerated Fianchetto, Modern, 7.Bc4 O-O 8.Bb3 a5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Bc4 O-O 8.Bb3 a5 *"),
("B35m", "Sicilian: Accelerated Fianchetto, Modern, 7.Bc4 O-O 8.Bb3 a5 9.a4", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Bc4 O-O 8.Bb3 a5 9.a4 *"),
("B35n", "Sicilian: Accelerated Fianchetto, Modern, 7.Bc4 O-O 8.Bb3 a5 9.f3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Bc4 O-O 8.Bb3 a5 9.f3 *"),
("B35o", "Sicilian: Accelerated Fianchetto, Modern, 7.Bc4 O-O 8.Bb3 d6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Bc4 O-O 8.Bb3 d6 *"),
("B35o", "Sicilian: Accelerated Fianchetto, Modern, 7.Bc4 O-O 8.Bb3 d6 9.h3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Bc4 O-O 8.Bb3 d6 9.h3 *"),
("B35p", "Sicilian: Accelerated Fianchetto, Modern, 7.Bc4 O-O 8.Bb3 d6 9.h3 Bd7", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Bc4 O-O 8.Bb3 d6 9.h3 Bd7 *"),
("B35q", "Sicilian: Accelerated Fianchetto, Modern, 7.Bc4 O-O 8.Bb3 d6 9.f3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Bc4 O-O 8.Bb3 d6 9.f3 *"),
("B35r", "Sicilian: Accelerated Fianchetto, Modern, 7.Bc4 O-O 8.Bb3 d6 9.f3 Bd7", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.Nc3 Bg7 6.Be3 Nf6 7.Bc4 O-O 8.Bb3 d6 9.f3 Bd7 *"),
("B36a", "Sicilian: Maroczy Bind", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 *"),
("B36b", "Sicilian: Maroczy Bind, 5...d6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 d6 *"),
("B36b", "Sicilian: Maroczy Bind, 5...d6 6.Nc3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 d6 6.Nc3 *"),
("B36b", "Sicilian: Maroczy Bind, 5...d6 6.Nc3 Bg7", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 d6 6.Nc3 Bg7 *"),
("B36c", "Sicilian: Maroczy Bind, 5...Nf6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Nf6 *"),
("B36d", "Sicilian: Maroczy Bind, 5...Nf6 6.Nc3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Nf6 6.Nc3 *"),
("B36e", "Sicilian: Maroczy Bind, 5...Nf6 6.Nc3 Nxd4", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Nf6 6.Nc3 Nxd4 *"),
("B36f", "Sicilian: Maroczy Bind, Gurgenidze Variation", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Nf6 6.Nc3 Nxd4 7.Qxd4 d6 *"),
("B36g", "Sicilian: Maroczy Bind, 5...Nf6 6.Nc3 d6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Nf6 6.Nc3 d6 *"),
("B36h", "Sicilian: Maroczy Bind, 5...Nf6 6.Nc3 d6 7.Be2", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Nf6 6.Nc3 d6 7.Be2 *"),
("B36i", "Sicilian: Maroczy Bind, 5...Nf6 6.Nc3 d6 7.Be2 Nxd4", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Nf6 6.Nc3 d6 7.Be2 Nxd4 *"),
("B36i", "Sicilian: Maroczy Bind, 5...Nf6 6.Nc3 d6 7.Be2 Nxd4 8.Qxd4 Bg7", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Nf6 6.Nc3 d6 7.Be2 Nxd4 8.Qxd4 Bg7 *"),
("B36j", "Sicilian: Maroczy Bind, 5...Nf6 6.Nc3 d6 7.Be2 Nxd4 8.Qxd4 Bg7 9.O-O", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Nf6 6.Nc3 d6 7.Be2 Nxd4 8.Qxd4 Bg7 9.O-O *"),
("B36k", "Sicilian: Maroczy Bind, 5...Nf6 6.Nc3 d6 7.Be2 Nxd4 8.Qxd4 Bg7 9.Bg5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Nf6 6.Nc3 d6 7.Be2 Nxd4 8.Qxd4 Bg7 9.Bg5 *"),
("B36l", "Sicilian: Maroczy Bind, 5...Nf6 6.Nc3 d6 7.Be2 Nxd4 8.Qxd4 Bg7 9.Bg5 O-O", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Nf6 6.Nc3 d6 7.Be2 Nxd4 8.Qxd4 Bg7 9.Bg5 O-O *"),
("B36m", "Sicilian: Maroczy Bind, 5...Nf6 6.Nc3 d6 7.Be2 Nxd4 8.Qxd4 Bg7 9.Bg5 O-O 10.Qd2", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Nf6 6.Nc3 d6 7.Be2 Nxd4 8.Qxd4 Bg7 9.Bg5 O-O 10.Qd2 *"),
("B36n", "Sicilian: Maroczy Bind, 5...Nf6 6.Nc3 d6 7.Be2 Nxd4 8.Qxd4 Bg7 9.Be3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Nf6 6.Nc3 d6 7.Be2 Nxd4 8.Qxd4 Bg7 9.Be3 *"),
("B36n", "Sicilian: Maroczy Bind, 5...Nf6 6.Nc3 d6 7.Be2 Nxd4 8.Qxd4 Bg7 9.Be3 O-O", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Nf6 6.Nc3 d6 7.Be2 Nxd4 8.Qxd4 Bg7 9.Be3 O-O *"),
("B36o", "Sicilian: Maroczy Bind, 5...Nf6 6.Nc3 d6 7.Be2 Nxd4 8.Qxd4 Bg7 9.Be3 O-O 10.Qd2", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Nf6 6.Nc3 d6 7.Be2 Nxd4 8.Qxd4 Bg7 9.Be3 O-O 10.Qd2 *"),
("B36p", "Sicilian: Maroczy Bind, 5...Nf6 6.Nc3 d6 7.Be2 Nxd4 8.Qxd4 Bg7 9.Be3 O-O 10.Qd2 Be6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Nf6 6.Nc3 d6 7.Be2 Nxd4 8.Qxd4 Bg7 9.Be3 O-O 10.Qd2 Be6 *"),
("B36q", "Sicilian: Maroczy Bind, 5...Nf6 6.Nc3 d6 7.Be2 Nxd4 8.Qxd4 Bg7 9.Be3 O-O 10.Qd2 Be6 11.O-O", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Nf6 6.Nc3 d6 7.Be2 Nxd4 8.Qxd4 Bg7 9.Be3 O-O 10.Qd2 Be6 11.O-O *"),
("B37a", "Sicilian: Maroczy Bind, 5...Bg7", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 *"),
("B37b", "Sicilian: Maroczy Bind, 6.Nb3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Nb3 *"),
("B37c", "Sicilian: Maroczy Bind, 6.Nc2", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Nc2 *"),
("B37d", "Sicilian: Maroczy Bind, 6.Nc2 d6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Nc2 d6 *"),
("B37e", "Sicilian: Maroczy Bind, 6.Nc2 d6 7.Be2", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Nc2 d6 7.Be2 *"),
("B37f", "Sicilian: Maroczy Bind, Simagin Variation", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Nc2 d6 7.Be2 Nh6 *"),
("B37g", "Sicilian: Maroczy Bind, 6.Nc2 Nf6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Nc2 Nf6 *"),
("B37g", "Sicilian: Maroczy Bind, 6.Nc2 Nf6 7.Nc3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Nc2 Nf6 7.Nc3 *"),
("B37h", "Sicilian: Maroczy Bind, 6.Nc2 Nf6 7.Nc3 O-O", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Nc2 Nf6 7.Nc3 O-O *"),
("B37i", "Sicilian: Maroczy Bind, 6.Nc2 Nf6 7.Nc3 d6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Nc2 Nf6 7.Nc3 d6 *"),
("B37i", "Sicilian: Maroczy Bind, 6.Nc2 Nf6 7.Nc3 d6 8.Be2", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Nc2 Nf6 7.Nc3 d6 8.Be2 *"),
("B37j", "Sicilian: Maroczy Bind, 6.Nc2 Nf6 7.Nc3 d6 8.Be2 Nd7", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Nc2 Nf6 7.Nc3 d6 8.Be2 Nd7 *"),
("B37k", "Sicilian: Maroczy Bind, 6.Nc2 Nf6 7.Nc3 d6 8.Be2 O-O", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Nc2 Nf6 7.Nc3 d6 8.Be2 O-O *"),
("B37l", "Sicilian: Maroczy Bind, 6.Nc2 Nf6 7.Nc3 d6 8.Be2 O-O 9.O-O", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Nc2 Nf6 7.Nc3 d6 8.Be2 O-O 9.O-O *"),
("B37m", "Sicilian: Maroczy Bind, 6.Nc2 Nf6 7.Nc3 d6 8.Be2 O-O 9.O-O Be6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Nc2 Nf6 7.Nc3 d6 8.Be2 O-O 9.O-O Be6 *"),
("B37n", "Sicilian: Maroczy Bind, 6.Nc2 Nf6 7.Nc3 d6 8.Be2 O-O 9.O-O Nd7", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Nc2 Nf6 7.Nc3 d6 8.Be2 O-O 9.O-O Nd7 *"),
("B37o", "Sicilian: Maroczy Bind, 6.Nc2 Nf6 7.Nc3 d6 8.Be2 O-O 9.O-O Nd7 10.Bd2", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Nc2 Nf6 7.Nc3 d6 8.Be2 O-O 9.O-O Nd7 10.Bd2 *"),
("B37o", "Sicilian: Maroczy Bind, 6.Nc2 Nf6 7.Nc3 d6 8.Be2 O-O 9.O-O Nd7 10.Bd2 a5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Nc2 Nf6 7.Nc3 d6 8.Be2 O-O 9.O-O Nd7 10.Bd2 a5 *"),
("B37p", "Sicilian: Maroczy Bind, 6.Nc2 Nf6 7.Nc3 d6 8.Be2 O-O 9.O-O Nd7 10.Bd2 Nc5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Nc2 Nf6 7.Nc3 d6 8.Be2 O-O 9.O-O Nd7 10.Bd2 Nc5 *"),
("B37q", "Sicilian: Maroczy Bind, 6.Nc2 Nf6 7.Nc3 d6 8.Be2 O-O 9.O-O Nd7 10.Bd2 Nc5 11.b4", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Nc2 Nf6 7.Nc3 d6 8.Be2 O-O 9.O-O Nd7 10.Bd2 Nc5 11.b4 *"),
("B38a", "Sicilian: Maroczy Bind, 6.Be3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 *"),
("B38b", "Sicilian: Maroczy Bind, 6.Be3 Nh6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nh6 *"),
("B38c", "Sicilian: Maroczy Bind, 6.Be3 d6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 d6 *"),
("B38c", "Sicilian: Maroczy Bind, 6.Be3 d6 7.Nc3 Nh6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 d6 7.Nc3 Nh6 *"),
("B38d", "Sicilian: Maroczy Bind, 6.Be3 Nf6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 *"),
("B38d", "Sicilian: Maroczy Bind, 6.Be3 Nf6 7.Nc3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 *"),
("B38e", "Sicilian: Maroczy Bind, 6.Be3 Nf6 7.Nc3 d6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 d6 *"),
("B38f", "Sicilian: Maroczy Bind, 6.Be3 Nf6 7.Nc3 d6 8.Be2", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 d6 8.Be2 *"),
("B38g", "Sicilian: Maroczy Bind, 7.Nc3 O-O", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 O-O *"),
("B38g", "Sicilian: Maroczy Bind, 7.Nc3 O-O 8.Be2", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 O-O 8.Be2 *"),
("B38h", "Sicilian: Maroczy Bind, 7.Nc3 O-O 8.Be2 b6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 O-O 8.Be2 b6 *"),
("B38i", "Sicilian: Maroczy Bind, 7.Nc3 O-O 8.Be2 b6 9.O-O", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 O-O 8.Be2 b6 9.O-O *"),
("B38i", "Sicilian: Maroczy Bind, 7.Nc3 O-O 8.Be2 b6 9.O-O Bb7", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 O-O 8.Be2 b6 9.O-O Bb7 *"),
("B38j", "Sicilian: Maroczy Bind, 7.Nc3 O-O 8.Be2 b6 9.O-O Bb7 10.f3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 O-O 8.Be2 b6 9.O-O Bb7 10.f3 *"),
("B38k", "Sicilian: Maroczy Bind, 7.Nc3 O-O 8.Be2 b6 9.O-O Bb7 10.f3 Qb8", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 O-O 8.Be2 b6 9.O-O Bb7 10.f3 Qb8 *"),
("B38l", "Sicilian: Maroczy Bind, 7.Nc3 O-O 8.Be2 d6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 O-O 8.Be2 d6 *"),
("B38m", "Sicilian: Maroczy Bind, 7.Nc3 O-O 8.Be2 d6 9.f3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 O-O 8.Be2 d6 9.f3 *"),
("B38n", "Sicilian: Maroczy Bind, 7.Nc3 O-O 8.Be2 d6 9.O-O", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 O-O 8.Be2 d6 9.O-O *"),
("B38o", "Sicilian: Maroczy Bind, 7.Nc3 O-O 8.Be2 d6 9.O-O Nd7", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 O-O 8.Be2 d6 9.O-O Nd7 *"),
("B38p", "Sicilian: Maroczy Bind, 7.Nc3 O-O 8.Be2 d6 9.O-O a6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 O-O 8.Be2 d6 9.O-O a6 *"),
("B38q", "Sicilian: Maroczy Bind, 7.Nc3 O-O 8.Be2 d6 9.O-O Nxd4", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 O-O 8.Be2 d6 9.O-O Nxd4 *"),
("B38r", "Sicilian: Maroczy Bind, 7.Nc3 O-O 8.Be2 d6 9.O-O Bd7", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 O-O 8.Be2 d6 9.O-O Bd7 *"),
("B38s", "Sicilian: Maroczy Bind, 7.Nc3 O-O 8.Be2 d6 9.O-O Bd7 10.Nc2", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 O-O 8.Be2 d6 9.O-O Bd7 10.Nc2 *"),
("B38t", "Sicilian: Maroczy Bind, 7.Nc3 O-O 8.Be2 d6 9.O-O Bd7 10.f3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 O-O 8.Be2 d6 9.O-O Bd7 10.f3 *"),
("B38u", "Sicilian: Maroczy Bind, 7.Nc3 O-O 8.Be2 d6 9.O-O Bd7 10.Rc1", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 O-O 8.Be2 d6 9.O-O Bd7 10.Rc1 *"),
("B38v", "Sicilian: Maroczy Bind, 7.Nc3 O-O 8.Be2 d6 9.O-O Bd7 10.Qd2", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 O-O 8.Be2 d6 9.O-O Bd7 10.Qd2 *"),
("B38w", "Sicilian: Maroczy Bind, 7.Nc3 O-O 8.Be2 d6 9.O-O Bd7 10.Qd2 Nxd4", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 O-O 8.Be2 d6 9.O-O Bd7 10.Qd2 Nxd4 *"),
("B38w", "Sicilian: Maroczy Bind, 7.Nc3 O-O 8.Be2 d6 9.O-O Bd7 10.Qd2 Nxd4 11.Bxd4", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 O-O 8.Be2 d6 9.O-O Bd7 10.Qd2 Nxd4 11.Bxd4 *"),
("B38x", "Sicilian: Maroczy Bind, 7.Nc3 O-O 8.Be2 d6 9.O-O Bd7 10.Qd2 Nxd4, 12.f3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 O-O 8.Be2 d6 9.O-O Bd7 10.Qd2 Nxd4 11.Bxd4 Bc6 12.f3 *"),
("B38y", "Sicilian: Maroczy Bind, 7.Nc3 O-O 8.Be2 d6 9.O-O Bd7 10.Qd2 Nxd4, 12.f3 a5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 O-O 8.Be2 d6 9.O-O Bd7 10.Qd2 Nxd4 11.Bxd4 Bc6 12.f3 a5 *"),
("B38z", "Sicilian: Maroczy Bind, 7.Nc3 O-O 8.Be2 d6 9.O-O Bd7 10.Qd2 Nxd4, 12.f3 a5 13.b3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 O-O 8.Be2 d6 9.O-O Bd7 10.Qd2 Nxd4 11.Bxd4 Bc6 12.f3 a5 13.b3 *"),
("B39a", "Sicilian: Maroczy Bind, Breyer Variation", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 Ng4 *"),
("B39b", "Sicilian: Maroczy Bind, Breyer, 8.Qxg4", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 Ng4 8.Qxg4 *"),
("B39c", "Sicilian: Maroczy Bind, Breyer, 8.Qxg4 Nxd4", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 Ng4 8.Qxg4 Nxd4 *"),
("B39d", "Sicilian: Maroczy Bind, Breyer, 8.Qxg4 Nxd4 9.Qd1", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 Ng4 8.Qxg4 Nxd4 9.Qd1 *"),
("B39e", "Sicilian: Maroczy Bind, Breyer, 8.Qxg4 Nxd4 9.Qd1 e5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 Ng4 8.Qxg4 Nxd4 9.Qd1 e5 *"),
("B39f", "Sicilian: Maroczy Bind, Breyer, 8.Qxg4 Nxd4 9.Qd1 e5 10.Nb5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 Ng4 8.Qxg4 Nxd4 9.Qd1 e5 10.Nb5 *"),
("B39g", "Sicilian: Maroczy Bind, Breyer, 8.Qxg4 Nxd4 9.Qd1 e5 10.Bd3", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 Ng4 8.Qxg4 Nxd4 9.Qd1 e5 10.Bd3 *"),
("B39h", "Sicilian: Maroczy Bind, Breyer, 8.Qxg4 Nxd4 9.Qd1 Ne6", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 Ng4 8.Qxg4 Nxd4 9.Qd1 Ne6 *"),
("B39i", "Sicilian: Maroczy Bind, Breyer, 8.Qxg4 Nxd4 9.Qd1 Ne6 10.Qd2", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 Ng4 8.Qxg4 Nxd4 9.Qd1 Ne6 10.Qd2 *"),
("B39j", "Sicilian: Maroczy Bind, Breyer, 8.Qxg4 Nxd4 9.Qd1 Ne6 10.Rc1", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 Ng4 8.Qxg4 Nxd4 9.Qd1 Ne6 10.Rc1 *"),
("B39k", "Sicilian: Maroczy Bind, Breyer, 8.Qxg4 Nxd4 9.Qd1 Ne6 10.Rc1 Qa5", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 Ng4 8.Qxg4 Nxd4 9.Qd1 Ne6 10.Rc1 Qa5 *"),
("B39l", "Sicilian: Maroczy Bind, Breyer, 8.Qxg4 Nxd4 9.Qd1 Ne6 10.Rc1 Qa5 11.Qd2", "1.e4 c5 2.Nf3 Nc6 3.d4 cxd4 4.Nxd4 g6 5.c4 Bg7 6.Be3 Nf6 7.Nc3 Ng4 8.Qxg4 Nxd4 9.Qd1 Ne6 10.Rc1 Qa5 11.Qd2 *"),
("B40a", "Sicilian: 2...e6", "1.e4 c5 2.Nf3 e6 *"),
("B40b", "Sicilian: 2...e6 3.c4", "1.e4 c5 2.Nf3 e6 3.c4 *"),
("B40c", "Sicilian: 2...e6 3.b3", "1.e4 c5 2.Nf3 e6 3.b3 *"),
("B40d", "Sicilian: 2...e6 3.b3 Nc6", "1.e4 c5 2.Nf3 e6 3.b3 Nc6 *"),
("B40e", "Sicilian: 2...e6 3.g3", "1.e4 c5 2.Nf3 e6 3.g3 *"),
("B40f", "Sicilian: 2...e6 3.d3", "1.e4 c5 2.Nf3 e6 3.d3 *"),
("B40g", "Sicilian: 2...e6 3.Nc3", "1.e4 c5 2.Nf3 e6 3.Nc3 *"),
("B40h", "Sicilian: 2...e6 3.Nc3 Nc6", "1.e4 c5 2.Nf3 e6 3.Nc3 Nc6 *"),
("B40i", "Sicilian: 2...e6 3.d4", "1.e4 c5 2.Nf3 e6 3.d4 *"),
("B40j", "Sicilian: Marshall Variation", "1.e4 c5 2.Nf3 e6 3.d4 d5 *"),
("B40k", "Sicilian: Open, 2...e6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 *"),
("B40l", "Sicilian: Open, 2...e6, 4.Nxd4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 *"),
("B40l", "Sicilian: Open, 2...e6, 4.Nxd4 d6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 d6 *"),
("B40m", "Sicilian: Open, 2...e6, 4.Nxd4 Bc5", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Bc5 *"),
("B40n", "Sicilian: Kveinys Variation", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Qb6 *"),
("B40o", "Sicilian: Open, 2...e6, 4.Nxd4 Nf6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nf6 *"),
("B40p", "Sicilian: Open, 2...e6, 4.Nxd4 Nf6 5.Bd3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nf6 5.Bd3 *"),
("B40p", "Sicilian: Open, 2...e6, 4.Nxd4 Nf6 5.Bd3 Nc6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nf6 5.Bd3 Nc6 *"),
("B40q", "Sicilian: Open, 2...e6, 4.Nxd4 Nf6 5.Bd3 Nc6 6.Nxc6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nf6 5.Bd3 Nc6 6.Nxc6 *"),
("B40r", "Sicilian: Open, 2...e6, 4.Nxd4 Nf6 5.Nc3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 *"),
("B40s", "Sicilian: Pin Variation", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Bb4 *"),
("B40t", "Sicilian: Pin, Jaffe Variation", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Bb4 6.Bd3 e5 *"),
("B40u", "Sicilian: Pin, Koch Variation", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Bb4 6.e5 *"),
("B41a", "Sicilian: Kan (Paulsen)", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 *"),
("B41b", "Sicilian: Kan, 5.g3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.g3 *"),
("B41c", "Sicilian: Kan, 5.Be3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Be3 *"),
("B41d", "Sicilian: Kan, 5.Be2", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Be2 *"),
("B41e", "Sicilian: Kan, 5.Be2 Nf6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Be2 Nf6 *"),
("B41f", "Sicilian: Kan, 5.c4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.c4 *"),
("B41g", "Sicilian: Kan, 5.c4 Qc7", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.c4 Qc7 *"),
("B41h", "Sicilian: Kan, 5.c4 Nf6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.c4 Nf6 *"),
("B41i", "Sicilian: Kan, 5.c4 Nf6 6.Nc3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.c4 Nf6 6.Nc3 *"),
("B41j", "Sicilian: Kan, 5.c4 Nf6 6.Nc3 d6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.c4 Nf6 6.Nc3 d6 *"),
("B41k", "Sicilian: Kan, 5.c4 Nf6 6.Nc3 Bb4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.c4 Nf6 6.Nc3 Bb4 *"),
("B41k", "Sicilian: Kan, 5.c4 Nf6, Bronstein Variation", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.c4 Nf6 6.Nc3 Bb4 7.Bd3 Nc6 8.Bc2 *"),
("B41l", "Sicilian: Kan, 5.c4 Nf6 6.Nc3 Qc7", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.c4 Nf6 6.Nc3 Qc7 *"),
("B41m", "Sicilian: Kan, 5.c4 Nf6 6.Nc3 Qc7 7.Be2", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.c4 Nf6 6.Nc3 Qc7 7.Be2 *"),
("B42a", "Sicilian: Kan, 5.Bd3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Bd3 *"),
("B42b", "Sicilian: Kan, Swiss Cheese Variation", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Bd3 g6 *"),
("B42c", "Sicilian: Kan, 5.Bd3 Qb6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Bd3 Qb6 *"),
("B42d", "Sicilian: Kan, 5.Bd3 Qc7", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Bd3 Qc7 *"),
("B42e", "Sicilian: Kan, 5.Bd3 Nc6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Bd3 Nc6 *"),
("B42f", "Sicilian: Kan, Polugaevsky Variation", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Bd3 Bc5 *"),
("B42g", "Sicilian: Kan, Polugaevsky, 6.Nb3 Ba7", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Bd3 Bc5 6.Nb3 Ba7 *"),
("B42h", "Sicilian: Kan, 5.Bd3 Nf6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Bd3 Nf6 *"),
("B42h", "Sicilian: Kan, 5.Bd3 Nf6 6.c4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Bd3 Nf6 6.c4 *"),
("B42h", "Sicilian: Kan, 5.Bd3 Nf6 6.O-O", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Bd3 Nf6 6.O-O *"),
("B42i", "Sicilian: Kan, 5.Bd3 Nf6 6.O-O d6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Bd3 Nf6 6.O-O d6 *"),
("B42j", "Sicilian: Kan, 5.Bd3 Nf6 6.O-O d6 7.c4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Bd3 Nf6 6.O-O d6 7.c4 *"),
("B42k", "Sicilian: Kan, Gipslis Variation", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Bd3 Nf6 6.O-O d6 7.c4 g6 *"),
("B42l", "Sicilian: Kan, 5.Bd3 Nf6 6.O-O Qc7", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Bd3 Nf6 6.O-O Qc7 *"),
("B42m", "Sicilian: Kan, 5.Bd3 Nf6 6.O-O Qc7 7.c4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Bd3 Nf6 6.O-O Qc7 7.c4 *"),
("B42n", "Sicilian: Kan, 5.Bd3 Nf6 6.O-O Qc7 7.Qe2", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Bd3 Nf6 6.O-O Qc7 7.Qe2 *"),
("B42n", "Sicilian: Kan, 5.Bd3 Nf6 6.O-O Qc7 7.Qe2", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Bd3 Nf6 6.O-O Qc7 7.Qe2 d6 *"),
("B42o", "Sicilian: Kan, 5.Bd3 Nf6 6.O-O Qc7 7.Qe2 d6 8.c4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Bd3 Nf6 6.O-O Qc7 7.Qe2 d6 8.c4 *"),
("B43a", "Sicilian: Kan, 5.Nc3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Nc3 *"),
("B43b", "Sicilian: Kan, 5.Nc3 d6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Nc3 d6 *"),
("B43c", "Sicilian: Kan, 5.Nc3 b5", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Nc3 b5 *"),
("B43d", "Sicilian: Kan, 5.Nc3 b5 6.Bd3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Nc3 b5 6.Bd3 *"),
("B43e", "Sicilian: Kan, 5.Nc3 b5 6.Bd3 Qb6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Nc3 b5 6.Bd3 Qb6 *"),
("B43f", "Sicilian: Kan, 5.Nc3 b5 6.Bd3 Qb6 7.Nb3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Nc3 b5 6.Bd3 Qb6 7.Nb3 *"),
("B43g", "Sicilian: Kan, 5.Nc3 Qc7", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Nc3 Qc7 *"),
("B43h", "Sicilian: Kan, 5.Nc3 Qc7 6.g3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Nc3 Qc7 6.g3 *"),
("B43i", "Sicilian: Kan, 5.Nc3 Qc7 6.g3 Nf6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Nc3 Qc7 6.g3 Nf6 *"),
("B43j", "Sicilian: Kan, 5.Nc3 Qc7 6.Be2", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Nc3 Qc7 6.Be2 *"),
("B43k", "Sicilian: Kan, 5.Nc3 Qc7 6.Be2 Nf6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Nc3 Qc7 6.Be2 Nf6 *"),
("B43l", "Sicilian: Kan, 5.Nc3 Qc7 6.Be2 Nf6 7.O-O", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Nc3 Qc7 6.Be2 Nf6 7.O-O *"),
("B43m", "Sicilian: Kan, 5.Nc3 Qc7 6.Bd3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Nc3 Qc7 6.Bd3 *"),
("B43n", "Sicilian: Kan, 5.Nc3 Qc7 6.Bd3 Nf6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Nc3 Qc7 6.Bd3 Nf6 *"),
("B43o", "Sicilian: Kan, 5.Nc3 Qc7 6.Bd3 Nf6 7.O-O", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 a6 5.Nc3 Qc7 6.Bd3 Nf6 7.O-O *"),
("B44a", "Sicilian: Taimanov", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 *"),
("B44b", "Sicilian: Taimanov, 5.g3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.g3 *"),
("B44c", "Sicilian: Taimanov, 5.Be2", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Be2 *"),
("B44d", "Sicilian: Taimanov, 5.c4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.c4 *"),
("B44d", "Sicilian: Taimanov, 5.c4 Nf6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.c4 Nf6 *"),
("B44e", "Sicilian: Taimanov, 5.c4 Nf6 6.Nc3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.c4 Nf6 6.Nc3 *"),
("B44e", "Sicilian: Taimanov, 5.c4 Nf6 6.Nc3 Bb4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.c4 Nf6 6.Nc3 Bb4 *"),
("B44e", "Sicilian: Taimanov, 5.c4 Nf6 6.Nc3 Bb4 7.Nxc6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.c4 Nf6 6.Nc3 Bb4 7.Nxc6 *"),
("B44f", "Sicilian: Taimanov, 5.c4 Nf6 6.Nc3 Bb4 7.Nxc6 bxc6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.c4 Nf6 6.Nc3 Bb4 7.Nxc6 bxc6 *"),
("B44g", "Sicilian: Taimanov, 5.Be3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Be3 *"),
("B44h", "Sicilian: Taimanov, 5.Be3 Nf6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Be3 Nf6 *"),
("B44i", "Sicilian: Taimanov, 5.Nxc6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nxc6 *"),
("B44j", "Sicilian: Taimanov, 5.Nxc6 bxc6 6.Bd3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nxc6 bxc6 6.Bd3 *"),
("B44k", "Sicilian, Taimanov, Szen Variation", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nb5 *"),
("B44k", "Sicilian, Taimanov, Szen, 5...d6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nb5 d6 *"),
("B44l", "Sicilian, Taimanov, Szen, 6.Bf4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nb5 d6 6.Bf4 *"),
("B44m", "Sicilian, Taimanov, Szen, 6.Bf4 e5 7.Be3 a6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nb5 d6 6.Bf4 e5 7.Be3 a6 *"),
("B44n", "Sicilian, Taimanov, Szen, 6.Bf4 e5 7.Be3 Nf6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nb5 d6 6.Bf4 e5 7.Be3 Nf6 *"),
("B44o", "Sicilian, Taimanov, Szen, 6.c4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nb5 d6 6.c4 *"),
("B44o", "Sicilian, Taimanov, Szen, 6.c4 a6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nb5 d6 6.c4 a6 *"),
("B44o", "Sicilian, Taimanov, Szen, 6.c4 Nf6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nb5 d6 6.c4 Nf6 *"),
("B44p", "Sicilian, Taimanov, Szen, 7.N5c3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nb5 d6 6.c4 Nf6 7.N5c3 *"),
("B44q", "Sicilian, Taimanov, Szen, 7.N1c3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nb5 d6 6.c4 Nf6 7.N1c3 *"),
("B44q", "Sicilian, Taimanov, Szen, 7.N1c3 a6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nb5 d6 6.c4 Nf6 7.N1c3 a6 *"),
("B44r", "Sicilian, Taimanov, Szen, 7.N1c3 a6 8.Na3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nb5 d6 6.c4 Nf6 7.N1c3 a6 8.Na3 *"),
("B44r", "Sicilian: Taimanov, Szen, Kasparov Gambit", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nb5 d6 6.c4 Nf6 7.N1c3 a6 8.Na3 d5 *"),
("B44s", "Sicilian, Taimanov, Szen, 7.N1c3 a6 8.Na3 b6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nb5 d6 6.c4 Nf6 7.N1c3 a6 8.Na3 b6 *"),
("B44t", "Sicilian, Taimanov, Szen, 7.N1c3 a6 8.Na3 Be7", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nb5 d6 6.c4 Nf6 7.N1c3 a6 8.Na3 Be7 *"),
("B44u", "Sicilian, Taimanov, Szen, 7.N1c3 a6 8.Na3 Be7 9.Be2", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nb5 d6 6.c4 Nf6 7.N1c3 a6 8.Na3 Be7 9.Be2 *"),
("B44u", "Sicilian, Taimanov, Szen, 7.N1c3 a6 8.Na3 Be7 9.Be2 b6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nb5 d6 6.c4 Nf6 7.N1c3 a6 8.Na3 Be7 9.Be2 b6 *"),
("B44v", "Sicilian, Taimanov, Szen, 7.N1c3 a6 8.Na3 Be7 9.Be2 O-O", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nb5 d6 6.c4 Nf6 7.N1c3 a6 8.Na3 Be7 9.Be2 O-O *"),
("B44w", "Sicilian: Taimanov, Szen, Hedgehog Variation", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nb5 d6 6.c4 Nf6 7.N1c3 a6 8.Na3 Be7 9.Be2 O-O 10.O-O b6 *"),
("B44x", "Sicilian: Taimanov, Szen, Hedgehog, 11.Be3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nb5 d6 6.c4 Nf6 7.N1c3 a6 8.Na3 Be7 9.Be2 O-O 10.O-O b6 11.Be3 *"),
("B44x", "Sicilian: Taimanov, Szen, Hedgehog, 11.Be3 Bd7", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nb5 d6 6.c4 Nf6 7.N1c3 a6 8.Na3 Be7 9.Be2 O-O 10.O-O b6 11.Be3 Bd7 *"),
("B44x", "Sicilian: Taimanov, Szen, Hedgehog, 11.Be3 Ne5", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nb5 d6 6.c4 Nf6 7.N1c3 a6 8.Na3 Be7 9.Be2 O-O 10.O-O b6 11.Be3 Ne5 *"),
("B44y", "Sicilian: Taimanov, Szen, Hedgehog, 11.Be3 Bb7", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nb5 d6 6.c4 Nf6 7.N1c3 a6 8.Na3 Be7 9.Be2 O-O 10.O-O b6 11.Be3 Bb7 *"),
("B44z", "Sicilian: Taimanov, Szen, Hedgehog, 11.Be3 Bb7 12.Qb3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nb5 d6 6.c4 Nf6 7.N1c3 a6 8.Na3 Be7 9.Be2 O-O 10.O-O b6 11.Be3 Bb7 12.Qb3 *"),
("B45a", "Sicilian: Taimanov, 5.Nc3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 *"),
("B45b", "Sicilian: Taimanov, 5.Nc3 Bb4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Bb4 *"),
("B45c", "Sicilian: Taimanov, 5.Nc3 d6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 d6 *"),
("B45c", "Sicilian: Taimanov, 5.Nc3 d6 6.g4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 d6 6.g4 *"),
("B45d", "Sicilian: Taimanov, Four Knights", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Nf6 *"),
("B45e", "Sicilian: Taimanov, Four Knights, 6.Bg5", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Nf6 6.Bg5 *"),
("B45f", "Sicilian: Taimanov, Four Knights, 6.a3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Nf6 6.a3 *"),
("B45g", "Sicilian: Taimanov, Four Knights, 6.g3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Nf6 6.g3 *"),
("B45h", "Sicilian: Taimanov, Four Knights, 6.Be2", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Nf6 6.Be2 *"),
("B45i", "Sicilian: Taimanov, Four Knights, 6.Be3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Nf6 6.Be3 *"),
("B45i", "Sicilian: Taimanov, Four Knights, 6.Be3 Bb4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Nf6 6.Be3 Bb4 *"),
("B45j", "Sicilian: Taimanov, Four Knights, 6.Be3 Bb4 7.Bd3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Nf6 6.Be3 Bb4 7.Bd3 *"),
("B45k", "Sicilian: Taimanov, Four Knights, 6.Nxc6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Nf6 6.Nxc6 *"),
("B45l", "Sicilian: Taimanov, Four Knights, 6.Nxc6 bxc6 7.e5", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Nf6 6.Nxc6 bxc6 7.e5 *"),
("B45l", "Sicilian: Taimanov, Four Knights, 6.Nxc6 bxc6 7.e5 Nd5", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Nf6 6.Nxc6 bxc6 7.e5 Nd5 *"),
("B45m", "Sicilian: Taimanov, Four Knights, 6.Nxc6 bxc6 7.e5 Nd5 8.Ne4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Nf6 6.Nxc6 bxc6 7.e5 Nd5 8.Ne4 *"),
("B45n", "Sicilian: Taimanov, Four Knights, 6.Nxc6 bxc6 7.e5 Nd5 8.Ne4 Qc7", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Nf6 6.Nxc6 bxc6 7.e5 Nd5 8.Ne4 Qc7 *"),
("B45o", "Sicilian: Taimanov, Four Knights, 6.Ndb5", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Nf6 6.Ndb5 *"),
("B45o", "Sicilian: Taimanov, Four Knights, 6.Ndb5 d6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Nf6 6.Ndb5 d6 *"),
("B45o", "Sicilian: Taimanov, Four Knights, 6.Ndb5 d6 7.Bf4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Nf6 6.Ndb5 d6 7.Bf4 *"),
("B45p", "Sicilian: Taimanov, Four Knights, 6.Ndb5 Bb4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Nf6 6.Ndb5 Bb4 *"),
("B45q", "Sicilian: Taimanov, Four Knights, American Attack", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Nf6 6.Ndb5 Bb4 7.Nd6+ *"),
("B45r", "Sicilian: Taimanov, Four Knights, 6.Ndb5 Bb4 7.a3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Nf6 6.Ndb5 Bb4 7.a3 *"),
("B45s", "Sicilian: Taimanov, Four Knights, 6.Ndb5 Bb4 7.a3 Bxc3+ 8.Nxc3 d5", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Nf6 6.Ndb5 Bb4 7.a3 Bxc3+ 8.Nxc3 d5 *"),
("B45t", "Sicilian: Taimanov, Four Knights, 6.Ndb5 Bb4 7.a3, 9.ed5 ed5", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Nf6 6.Ndb5 Bb4 7.a3 Bxc3+ 8.Nxc3 d5 9.exd5 exd5 *"),
("B45u", "Sicilian: Taimanov, Four Knights, 6.Ndb5 Bb4 7.a3, 9.ed5 ed5 10.Bd3 O-O", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Nf6 6.Ndb5 Bb4 7.a3 Bxc3+ 8.Nxc3 d5 9.exd5 exd5 10.Bd3 O-O *"),
("B45v", "Sicilian: Taimanov, Four Knights, 6.Ndb5 Bb4 7.a3, 10.Bd3 O-O 11.O-O d4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Nf6 6.Ndb5 Bb4 7.a3 Bxc3+ 8.Nxc3 d5 9.exd5 exd5 10.Bd3 O-O 11.O-O d4 *"),
("B45w", "Sicilian: Taimanov, Four Knights, 6.Ndb5 Bb4 7.a3, 10.Bd3 O-O 11.O-O d4 12.Ne2", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Nf6 6.Ndb5 Bb4 7.a3 Bxc3+ 8.Nxc3 d5 9.exd5 exd5 10.Bd3 O-O 11.O-O d4 12.Ne2 *"),
("B46a", "Sicilian: Taimanov, 5...a6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 a6 *"),
("B46b", "Sicilian: Taimanov, 5...a6 6.f4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 a6 6.f4 *"),
("B46c", "Sicilian: Taimanov, 5...a6 6.Nxc6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 a6 6.Nxc6 *"),
("B46d", "Sicilian: Taimanov, 5...a6 6.Nxc6 bxc6 7.Bd3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 a6 6.Nxc6 bxc6 7.Bd3 *"),
("B46e", "Sicilian: Taimanov, 5...a6 6.Nxc6 bxc6 7.Bd3 d5", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 a6 6.Nxc6 bxc6 7.Bd3 d5 *"),
("B46f", "Sicilian: Taimanov, 5...a6 6.g3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 a6 6.g3 *"),
("B46g", "Sicilian: Taimanov, 5...a6 6.g3 Nge7", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 a6 6.g3 Nge7 *"),
("B46h", "Sicilian: Taimanov, 5...a6 6.g3 d6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 a6 6.g3 d6 *"),
("B46i", "Sicilian: Taimanov, 5...a6 6.Be3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 a6 6.Be3 *"),
("B46j", "Sicilian: Taimanov, 5...a6 6.Be3 d6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 a6 6.Be3 d6 *"),
("B46k", "Sicilian: Taimanov, 5...a6 6.Be3 Nge7", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 a6 6.Be3 Nge7 *"),
("B46l", "Sicilian: Taimanov, 5...a6 6.Be3 Nf6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 a6 6.Be3 Nf6 *"),
("B46m", "Sicilian: Taimanov, 5...a6 6.Be2", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 a6 6.Be2 *"),
("B46n", "Sicilian: Taimanov, 5...a6 6.Be2 Nf6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 a6 6.Be2 Nf6 *"),
("B46o", "Sicilian: Taimanov, 5...a6 6.Be2 Nge7", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 a6 6.Be2 Nge7 *"),
("B46p", "Sicilian: Taimanov, 5...a6 6.Be2 Nge7 7.O-O", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 a6 6.Be2 Nge7 7.O-O *"),
("B46q", "Sicilian: Taimanov, 5...a6 6.Be2 Nge7 7.Be3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 a6 6.Be2 Nge7 7.Be3 *"),
("B46r", "Sicilian: Taimanov, 5...a6 6.Be2 d6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 a6 6.Be2 d6 *"),
("B47a", "Sicilian: Taimanov, Bastrikov Variation", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 *"),
("B47b", "Sicilian: Taimanov, 6.Nxc6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Nxc6 *"),
("B47c", "Sicilian: Taimanov, 6.Ndb5", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Ndb5 *"),
("B47d", "Sicilian: Taimanov, 6.f4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.f4 *"),
("B47e", "Sicilian: Taimanov, 6.f4 a6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.f4 a6 *"),
("B47f", "Sicilian: Taimanov, 6.f4 a6 7.Nxc6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.f4 a6 7.Nxc6 *"),
("B47g", "Sicilian: Taimanov, 6.g3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.g3 *"),
("B47h", "Sicilian: Taimanov, 6.g3 a6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.g3 a6 *"),
("B47h", "Sicilian: Taimanov, 6.g3 a6 7.Bg2", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.g3 a6 7.Bg2 *"),
("B47i", "Sicilian: Taimanov, 6.g3 a6 7.Bg2 d6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.g3 a6 7.Bg2 d6 *"),
("B47i", "Sicilian: Taimanov, 6.g3 a6 7.Bg2 d6 8.O-O", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.g3 a6 7.Bg2 d6 8.O-O *"),
("B47j", "Sicilian: Taimanov, 6.g3 a6 7.Bg2 d6 8.O-O Bd7", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.g3 a6 7.Bg2 d6 8.O-O Bd7 *"),
("B47k", "Sicilian: Taimanov, 6.g3 a6 7.Bg2 d6 8.O-O Bd7 9.Re1", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.g3 a6 7.Bg2 d6 8.O-O Bd7 9.Re1 *"),
("B47l", "Sicilian: Taimanov, 6.g3 a6 7.Bg2 Nf6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.g3 a6 7.Bg2 Nf6 *"),
("B47m", "Sicilian: Taimanov, 6.g3 a6 7.Bg2 Nf6 8.O-O", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.g3 a6 7.Bg2 Nf6 8.O-O *"),
("B47n", "Sicilian: Taimanov, 6.g3 a6 7.Bg2 Nf6 8.O-O Bc5", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.g3 a6 7.Bg2 Nf6 8.O-O Bc5 *"),
("B47o", "Sicilian: Taimanov, 6.g3 a6 7.Bg2 Nf6 8.O-O Be7", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.g3 a6 7.Bg2 Nf6 8.O-O Be7 *"),
("B47p", "Sicilian: Taimanov, 6.g3 a6 7.Bg2 Nf6 8.O-O Be7 9.Re1", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.g3 a6 7.Bg2 Nf6 8.O-O Be7 9.Re1 *"),
("B47q", "Sicilian: Taimanov, 6.Be2", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be2 *"),
("B47q", "Sicilian: Taimanov, 6.Be2 Nf6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be2 Nf6 *"),
("B47r", "Sicilian: Taimanov, 6.Be2 a6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be2 a6 *"),
("B47s", "Sicilian: Taimanov, 6.Be2 a6 7.f4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be2 a6 7.f4 *"),
("B47t", "Sicilian: Taimanov, 6.Be2 a6 7.O-O", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be2 a6 7.O-O *"),
("B47u", "Sicilian: Taimanov, 6.Be2 a6 7.O-O b5", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be2 a6 7.O-O b5 *"),
("B47v", "Sicilian: Taimanov, 6.Be2 a6 7.O-O Nf6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be2 a6 7.O-O Nf6 *"),
("B47v", "Sicilian: Taimanov, 6.Be2 a6 7.O-O Nf6 8.a3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be2 a6 7.O-O Nf6 8.a3 *"),
("B47w", "Sicilian: Taimanov, 6.Be2 a6 7.O-O Nf6 8.Kh1", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be2 a6 7.O-O Nf6 8.Kh1 *"),
("B47x", "Sicilian: Taimanov, 6.Be2 a6 7.O-O Nf6 8.Kh1 Be7", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be2 a6 7.O-O Nf6 8.Kh1 Be7 *"),
("B47y", "Sicilian: Taimanov, 6.Be2 a6 7.O-O Nf6 8.Kh1 Nxd4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be2 a6 7.O-O Nf6 8.Kh1 Nxd4 *"),
("B47y", "Sicilian: Taimanov, 6.Be2 a6 7.O-O Nf6 8.Kh1 Nxd4, 10.Qd3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be2 a6 7.O-O Nf6 8.Kh1 Nxd4 9.Qxd4 Bc5 10.Qd3 *"),
("B47y", "Sicilian: Taimanov, 6.Be2 a6 7.O-O Nf6 8.Kh1 Nxd4, 10.Qd3 b5", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be2 a6 7.O-O Nf6 8.Kh1 Nxd4 9.Qxd4 Bc5 10.Qd3 b5 *"),
("B47z", "Sicilian: Taimanov, 6.Be2 a6 7.O-O Nf6 8.Kh1 Nxd4, 10.Qd3 b5 11.f4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be2 a6 7.O-O Nf6 8.Kh1 Nxd4 9.Qxd4 Bc5 10.Qd3 b5 11.f4 *"),
("B48a", "Sicilian: Taimanov, 6.Be3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 *"),
("B48b", "Sicilian: Taimanov, 6.Be3 Nf6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 Nf6 *"),
("B48c", "Sicilian: Taimanov, 6.Be3 a6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 *"),
("B48d", "Sicilian: Taimanov, 6.Be3 a6 7.a3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.a3 *"),
("B48e", "Sicilian: Taimanov, 6.Be3 a6 7.f4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.f4 *"),
("B48f", "Sicilian: Taimanov, 6.Be3 a6 7.f4 b5", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.f4 b5 *"),
("B48g", "Sicilian: Taimanov, 6.Be3 a6 7.Qd2", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Qd2 *"),
("B48h", "Sicilian: Taimanov, 6.Be3 a6 7.Qd2 Nf6 8.f3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Qd2 Nf6 8.f3 *"),
("B48i", "Sicilian: Taimanov, 6.Be3 a6 7.Bd3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Bd3 *"),
("B48j", "Sicilian: Taimanov, 6.Be3 a6 7.Bd3 b5", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Bd3 b5 *"),
("B48k", "Sicilian: Taimanov, 6.Be3 a6 7.Bd3 b5 8.Nxc6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Bd3 b5 8.Nxc6 *"),
("B48l", "Sicilian: Taimanov, 6.Be3 a6 7.Bd3 b5 8.O-O", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Bd3 b5 8.O-O *"),
("B48m", "Sicilian: Taimanov, 6.Be3 a6 7.Bd3 Nf6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Bd3 Nf6 *"),
("B48n", "Sicilian: Taimanov, 6.Be3 a6 7.Bd3 Nf6 8.O-O", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Bd3 Nf6 8.O-O *"),
("B48o", "Sicilian: Taimanov, 6.Be3 a6 7.Bd3 Nf6 8.O-O Nxd4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Bd3 Nf6 8.O-O Nxd4 *"),
("B48p", "Sicilian: Taimanov, 6.Be3 a6 7.Bd3 Nf6 8.O-O h5", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Bd3 Nf6 8.O-O h5 *"),
("B48q", "Sicilian: Taimanov, 6.Be3 a6 7.Bd3 Nf6 8.O-O b5", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Bd3 Nf6 8.O-O b5 *"),
("B48r", "Sicilian: Taimanov, 6.Be3 a6 7.Bd3 Nf6 8.O-O Bd6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Bd3 Nf6 8.O-O Bd6 *"),
("B48s", "Sicilian: Taimanov, 6.Be3 a6 7.Bd3 Nf6 8.O-O Ne5", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Bd3 Nf6 8.O-O Ne5 *"),
("B48t", "Sicilian: Taimanov, 6.Be3 a6 7.Bd3 Nf6 8.O-O Ne5 9.h3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Bd3 Nf6 8.O-O Ne5 9.h3 *"),
("B48u", "Sicilian: Taimanov, 6.Be3 a6 7.Bd3 Nf6 8.O-O Ne5 9.h3 Bc5", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Bd3 Nf6 8.O-O Ne5 9.h3 Bc5 *"),
("B48v", "Sicilian: Taimanov, 6.Be3 a6 7.Bd3 Nf6 8.O-O Ne5 9.h3 Bc5 10.Kh1", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Bd3 Nf6 8.O-O Ne5 9.h3 Bc5 10.Kh1 *"),
("B49a", "Sicilian: Taimanov, 6.Be3 a6 7.Be2", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Be2 *"),
("B49a", "Sicilian: Taimanov, 6.Be3 a6 7.Be2 Nge7", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Be2 Nge7 *"),
("B49b", "Sicilian: Taimanov, 6.Be3 a6 7.Be2 b5", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Be2 b5 *"),
("B49b", "Sicilian: Taimanov, 6.Be3 a6 7.Be2 b5 8.Nxc6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Be2 b5 8.Nxc6 *"),
("B49c", "Sicilian: Taimanov, 6.Be3 a6 7.Be2 Nf6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Be2 Nf6 *"),
("B49d", "Sicilian: Taimanov, 6.Be3 a6 7.Be2 Nf6 8.a3", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Be2 Nf6 8.a3 *"),
("B49d", "Sicilian: Taimanov, 6.Be3 a6 7.Be2 Nf6 8.a3 b5", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Be2 Nf6 8.a3 b5 *"),
("B49e", "Sicilian: Taimanov, 6.Be3 a6 7.Be2 Nf6 8.a3 Be7", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Be2 Nf6 8.a3 Be7 *"),
("B49e", "Sicilian: Taimanov, 6.Be3 a6 7.Be2 Nf6 8.a3 Be7 9.O-O", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Be2 Nf6 8.a3 Be7 9.O-O *"),
("B49f", "Sicilian: Taimanov, 6.Be3 a6 7.Be2 Nf6 8.O-O", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Be2 Nf6 8.O-O *"),
("B49f", "Sicilian: Taimanov, 6.Be3 a6 7.Be2 Nf6 8.O-O b5", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Be2 Nf6 8.O-O b5 *"),
("B49g", "Sicilian: Taimanov, 6.Be3 a6 7.Be2 Nf6 8.O-O Be7", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Be2 Nf6 8.O-O Be7 *"),
("B49h", "Sicilian: Taimanov, 6.Be3 a6 7.Be2 Nf6 8.O-O Bb4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Be2 Nf6 8.O-O Bb4 *"),
("B49i", "Sicilian: Taimanov, 6.Be3 a6 7.Be2 Nf6 8.O-O Bb4 9.Nxc6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Be2 Nf6 8.O-O Bb4 9.Nxc6 *"),
("B49j", "Sicilian: Taimanov, 6.Be3 a6 7.Be2 Nf6 8.O-O Bb4 9.Na4", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Be2 Nf6 8.O-O Bb4 9.Na4 *"),
("B49k", "Sicilian: Taimanov, 6.Be3 a6 7.Be2 Nf6 8.O-O Bb4 9.Na4 O-O", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Be2 Nf6 8.O-O Bb4 9.Na4 O-O *"),
("B49l", "Sicilian: Taimanov, 6.Be3 a6 7.Be2 Nf6 8.O-O Bb4 9.Na4 Be7", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Be2 Nf6 8.O-O Bb4 9.Na4 Be7 *"),
("B49m", "Sicilian: Taimanov, 6.Be3 a6 7.Be2 Nf6 8.O-O Bb4 9.Na4 Be7 10.Nxc6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Be2 Nf6 8.O-O Bb4 9.Na4 Be7 10.Nxc6 *"),
("B49m", "Sicilian: Taimanov, 6.Be3 a6 7.Be2 Nf6 8.O-O Bb4 9.Na4 Be7 10.Nxc6 bxc6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Be2 Nf6 8.O-O Bb4 9.Na4 Be7 10.Nxc6 bxc6 *"),
("B49n", "Sicilian: Taimanov, 6.Be3 a6 7.Be2 Nf6 8.O-O Bb4 9.Na4 Be7 10.Nxc6 bxc6 11.Nb6", "1.e4 c5 2.Nf3 e6 3.d4 cxd4 4.Nxd4 Nc6 5.Nc3 Qc7 6.Be3 a6 7.Be2 Nf6 8.O-O Bb4 9.Na4 Be7 10.Nxc6 bxc6 11.Nb6 *"),
("B50a", "Sicilian: 2.Nf3 d6", "1.e4 c5 2.Nf3 d6 *"),
("B50b", "Sicilian: Wing Gambit deferred", "1.e4 c5 2.Nf3 d6 3.b4 *"),
("B50c", "Sicilian: 2.Nf3 d6 3.b3", "1.e4 c5 2.Nf3 d6 3.b3 *"),
("B50d", "Sicilian: 2.Nf3 d6 3.b3 e6", "1.e4 c5 2.Nf3 d6 3.b3 e6 *"),
("B50e", "Sicilian: 2.Nf3 d6 3.d3", "1.e4 c5 2.Nf3 d6 3.d3 *"),
("B50f", "Sicilian: 2.Nf3 d6 3.g3", "1.e4 c5 2.Nf3 d6 3.g3 *"),
("B50g", "Sicilian: 2.Nf3 d6 3.Bc4", "1.e4 c5 2.Nf3 d6 3.Bc4 *"),
("B50h", "Sicilian: 2.Nf3 d6 3.Bc4 Nf6 4.d3", "1.e4 c5 2.Nf3 d6 3.Bc4 Nf6 4.d3 *"),
("B50i", "Sicilian: 2.Nf3 d6 3.Nc3", "1.e4 c5 2.Nf3 d6 3.Nc3 *"),
("B50j", "Sicilian: 2.Nf3 d6 3.Nc3 e6", "1.e4 c5 2.Nf3 d6 3.Nc3 e6 *"),
("B50k", "Sicilian: 2.Nf3 d6 3.Nc3 Nf6", "1.e4 c5 2.Nf3 d6 3.Nc3 Nf6 *"),
("B50l", "Sicilian: 2.Nf3 d6 3.c3", "1.e4 c5 2.Nf3 d6 3.c3 *"),
("B50m", "Sicilian: 2.Nf3 d6 3.c3 Nf6", "1.e4 c5 2.Nf3 d6 3.c3 Nf6 *"),
("B50n", "Sicilian: 2.Nf3 d6 3.c3 Nf6 4.h3", "1.e4 c5 2.Nf3 d6 3.c3 Nf6 4.h3 *"),
("B50o", "Sicilian: 2.Nf3 d6 3.c3 Nf6 4.h3 Nc6", "1.e4 c5 2.Nf3 d6 3.c3 Nf6 4.h3 Nc6 *"),
("B50p", "Sicilian: 2.Nf3 d6 3.c3 Nf6 4.Bc4", "1.e4 c5 2.Nf3 d6 3.c3 Nf6 4.Bc4 *"),
("B50q", "Sicilian: 2.Nf3 d6 3.c3 Nf6 4.Bd3", "1.e4 c5 2.Nf3 d6 3.c3 Nf6 4.Bd3 *"),
("B50r", "Sicilian: 2.Nf3 d6 3.c3 Nf6 4.Bd3 Nc6", "1.e4 c5 2.Nf3 d6 3.c3 Nf6 4.Bd3 Nc6 *"),
("B50s", "Sicilian: 2.Nf3 d6 3.c3 Nf6 4.Be2", "1.e4 c5 2.Nf3 d6 3.c3 Nf6 4.Be2 *"),
("B50t", "Sicilian: 2.Nf3 d6 3.c3 Nf6 4.Be2 Nc6", "1.e4 c5 2.Nf3 d6 3.c3 Nf6 4.Be2 Nc6 *"),
("B50u", "Sicilian: 2.Nf3 d6 3.c3 Nf6 4.Be2 Nbd7", "1.e4 c5 2.Nf3 d6 3.c3 Nf6 4.Be2 Nbd7 *"),
("B50v", "Sicilian: 2.Nf3 d6 3.c3 Nf6 4.Be2 g6", "1.e4 c5 2.Nf3 d6 3.c3 Nf6 4.Be2 g6 *"),
("B50w", "Sicilian: 2.Nf3 d6 3.c3, Torre Variation", "1.e4 c5 2.Nf3 d6 3.c3 Nf6 4.Be2 g6 5.O-O Bg7 6.Bb5+ *"),
("B51a", "Sicilian: Moscow", "1.e4 c5 2.Nf3 d6 3.Bb5+ *"),
("B51a", "Sicilian: Moscow 3...Nd7", "1.e4 c5 2.Nf3 d6 3.Bb5+ Nd7 *"),
("B51b", "Sicilian: Moscow 3...Nd7 4.c3", "1.e4 c5 2.Nf3 d6 3.Bb5+ Nd7 4.c3 *"),
("B51c", "Sicilian: Moscow 3...Nd7 4.O-O", "1.e4 c5 2.Nf3 d6 3.Bb5+ Nd7 4.O-O *"),
("B51d", "Sicilian: Moscow 3...Nd7 4.O-O Nf6", "1.e4 c5 2.Nf3 d6 3.Bb5+ Nd7 4.O-O Nf6 *"),
("B51e", "Sicilian: Moscow 3...Nd7 4.O-O Nf6 5.Re1 a6 6.Bf1", "1.e4 c5 2.Nf3 d6 3.Bb5+ Nd7 4.O-O Nf6 5.Re1 a6 6.Bf1 *"),
("B51f", "Sicilian: Moscow 3...Nd7 4.d4", "1.e4 c5 2.Nf3 d6 3.Bb5+ Nd7 4.d4 *"),
("B51f", "Sicilian: Moscow 3...Nd7 4.d4 cxd4", "1.e4 c5 2.Nf3 d6 3.Bb5+ Nd7 4.d4 cxd4 *"),
("B51g", "Sicilian: Moscow 3...Nd7 4.d4 Nf6", "1.e4 c5 2.Nf3 d6 3.Bb5+ Nd7 4.d4 Nf6 *"),
("B51h", "Sicilian: Moscow 3...Nd7 4.d4 Nf6 5.Nc3", "1.e4 c5 2.Nf3 d6 3.Bb5+ Nd7 4.d4 Nf6 5.Nc3 *"),
("B51i", "Sicilian: Moscow 3...Nd7 4.d4 Nf6 5.Nc3 cxd4", "1.e4 c5 2.Nf3 d6 3.Bb5+ Nd7 4.d4 Nf6 5.Nc3 cxd4 *"),
("B51i", "Sicilian: Moscow 3...Nd7 4.d4 Nf6 5.Nc3 cxd4 6.Qxd4", "1.e4 c5 2.Nf3 d6 3.Bb5+ Nd7 4.d4 Nf6 5.Nc3 cxd4 6.Qxd4 *"),
("B51j", "Sicilian: Moscow 3...Nd7 4.d4 Nf6 5.Nc3 cxd4 6.Qxd4 e5", "1.e4 c5 2.Nf3 d6 3.Bb5+ Nd7 4.d4 Nf6 5.Nc3 cxd4 6.Qxd4 e5 *"),
("B51k", "Sicilian: Moscow 3...Nc6", "1.e4 c5 2.Nf3 d6 3.Bb5+ Nc6 *"),
("B51l", "Sicilian: Moscow 3...Nc6 4.Bxc6+", "1.e4 c5 2.Nf3 d6 3.Bb5+ Nc6 4.Bxc6+ *"),
("B51m", "Sicilian: Moscow 3...Nc6 4.O-O", "1.e4 c5 2.Nf3 d6 3.Bb5+ Nc6 4.O-O *"),
("B51n", "Sicilian: Moscow 3...Nc6 4.O-O Bd7", "1.e4 c5 2.Nf3 d6 3.Bb5+ Nc6 4.O-O Bd7 *"),
("B51o", "Sicilian: Moscow 3...Nc6 4.O-O Bd7 5.Re1", "1.e4 c5 2.Nf3 d6 3.Bb5+ Nc6 4.O-O Bd7 5.Re1 *"),
("B51p", "Sicilian: Moscow 3...Nc6 4.O-O Bd7 5.Re1 a6", "1.e4 c5 2.Nf3 d6 3.Bb5+ Nc6 4.O-O Bd7 5.Re1 a6 *"),
("B51q", "Sicilian: Moscow 3...Nc6 4.O-O Bd7 5.Re1 Nf6", "1.e4 c5 2.Nf3 d6 3.Bb5+ Nc6 4.O-O Bd7 5.Re1 Nf6 *"),
("B51r", "Sicilian: Moscow 3...Nc6 4.O-O Bd7 5.Re1 Nf6 6.c3", "1.e4 c5 2.Nf3 d6 3.Bb5+ Nc6 4.O-O Bd7 5.Re1 Nf6 6.c3 *"),
("B51s", "Sicilian: Moscow 3...Nc6 4.O-O Bd7 5.Re1 Nf6 6.c3 a6 7.Ba4", "1.e4 c5 2.Nf3 d6 3.Bb5+ Nc6 4.O-O Bd7 5.Re1 Nf6 6.c3 a6 7.Ba4 *"),
("B51t", "Sicilian: Moscow 3...Nc6 4.O-O Bd7 5.Re1 Nf6 6.c3 a6 7.Bf1", "1.e4 c5 2.Nf3 d6 3.Bb5+ Nc6 4.O-O Bd7 5.Re1 Nf6 6.c3 a6 7.Bf1 *"),
("B51u", "Sicilian: Moscow 3...Nc6 4.O-O Bd7 5.Re1 Nf6 6.c3 a6 7.Bf1 Bg4 8.h3", "1.e4 c5 2.Nf3 d6 3.Bb5+ Nc6 4.O-O Bd7 5.Re1 Nf6 6.c3 a6 7.Bf1 Bg4 8.h3 *"),
("B52a", "Sicilian: Moscow 3...Bd7", "1.e4 c5 2.Nf3 d6 3.Bb5+ Bd7 *"),
("B52b", "Sicilian: Moscow 3...Bd7 4.Bxd7+ Nxd7", "1.e4 c5 2.Nf3 d6 3.Bb5+ Bd7 4.Bxd7+ Nxd7 *"),
("B52c", "Sicilian: Moscow 3...Bd7 4.Bxd7+ Nxd7 5.O-O", "1.e4 c5 2.Nf3 d6 3.Bb5+ Bd7 4.Bxd7+ Nxd7 5.O-O *"),
("B52d", "Sicilian: Moscow 3...Bd7 4.Bxd7+ Nxd7 5.O-O Ngf6", "1.e4 c5 2.Nf3 d6 3.Bb5+ Bd7 4.Bxd7+ Nxd7 5.O-O Ngf6 *"),
("B52e", "Sicilian: Moscow 3...Bd7 4.Bxd7+ Nxd7 5.O-O Ngf6 6.Qe2", "1.e4 c5 2.Nf3 d6 3.Bb5+ Bd7 4.Bxd7+ Nxd7 5.O-O Ngf6 6.Qe2 *"),
("B52f", "Sicilian: Moscow 3...Bd7 4.Bxd7+ Qxd7", "1.e4 c5 2.Nf3 d6 3.Bb5+ Bd7 4.Bxd7+ Qxd7 *"),
("B52g", "Sicilian: 3.Bb5+, Sokolsky Variation", "1.e4 c5 2.Nf3 d6 3.Bb5+ Bd7 4.Bxd7+ Qxd7 5.c4 *"),
("B52h", "Sicilian: 3.Bb5+, Sokolsky, 5...Nc6", "1.e4 c5 2.Nf3 d6 3.Bb5+ Bd7 4.Bxd7+ Qxd7 5.c4 Nc6 *"),
("B52i", "Sicilian: 3.Bb5+, Sokolsky, 5...Nc6 6.O-O", "1.e4 c5 2.Nf3 d6 3.Bb5+ Bd7 4.Bxd7+ Qxd7 5.c4 Nc6 6.O-O *"),
("B52j", "Sicilian: 3.Bb5+, Sokolsky, 5...Nc6 6.O-O Nf6", "1.e4 c5 2.Nf3 d6 3.Bb5+ Bd7 4.Bxd7+ Qxd7 5.c4 Nc6 6.O-O Nf6 *"),
("B52k", "Sicilian: 3.Bb5+, Sokolsky, 5...Nc6 6.Nc3", "1.e4 c5 2.Nf3 d6 3.Bb5+ Bd7 4.Bxd7+ Qxd7 5.c4 Nc6 6.Nc3 *"),
("B52l", "Sicilian: 3.Bb5+, Sokolsky, 5...Nc6 6.Nc3 Nf6", "1.e4 c5 2.Nf3 d6 3.Bb5+ Bd7 4.Bxd7+ Qxd7 5.c4 Nc6 6.Nc3 Nf6 *"),
("B52m", "Sicilian: 3.Bb5+, Sokolsky, 5...Nc6 6.Nc3 Nf6 7.O-O", "1.e4 c5 2.Nf3 d6 3.Bb5+ Bd7 4.Bxd7+ Qxd7 5.c4 Nc6 6.Nc3 Nf6 7.O-O *"),
("B52n", "Sicilian: 3.Bb5+ Bd7 4.Bxd7+ Qxd7 5.O-O", "1.e4 c5 2.Nf3 d6 3.Bb5+ Bd7 4.Bxd7+ Qxd7 5.O-O *"),
("B52o", "Sicilian: 3.Bb5+ Bd7 4.Bxd7+ Qxd7 5.O-O Nc6", "1.e4 c5 2.Nf3 d6 3.Bb5+ Bd7 4.Bxd7+ Qxd7 5.O-O Nc6 *"),
("B52p", "Sicilian: 3.Bb5+ Bd7 4.Bxd7+ Qxd7 5.O-O Nc6 6.Re1", "1.e4 c5 2.Nf3 d6 3.Bb5+ Bd7 4.Bxd7+ Qxd7 5.O-O Nc6 6.Re1 *"),
("B52q", "Sicilian: 3.Bb5+ Bd7 4.Bxd7+ Qxd7 5.O-O Nc6 6.Re1 Nf6", "1.e4 c5 2.Nf3 d6 3.Bb5+ Bd7 4.Bxd7+ Qxd7 5.O-O Nc6 6.Re1 Nf6 *"),
("B52r", "Sicilian: 3.Bb5+ Bd7 4.Bxd7+ Qxd7 5.O-O Nc6 6.c3", "1.e4 c5 2.Nf3 d6 3.Bb5+ Bd7 4.Bxd7+ Qxd7 5.O-O Nc6 6.c3 *"),
("B52s", "Sicilian: 3.Bb5+ Bd7 4.Bxd7+ Qxd7 5.O-O Nc6 6.c3 Nf6", "1.e4 c5 2.Nf3 d6 3.Bb5+ Bd7 4.Bxd7+ Qxd7 5.O-O Nc6 6.c3 Nf6 *"),
("B52t", "Sicilian: 3.Bb5+, Bronstein Gambit", "1.e4 c5 2.Nf3 d6 3.Bb5+ Bd7 4.Bxd7+ Qxd7 5.O-O Nc6 6.c3 Nf6 7.d4 *"),
("B53a", "Sicilian: 2...d6 3.d4", "1.e4 c5 2.Nf3 d6 3.d4 *"),
("B53b", "Sicilian: 2...d6 3.d4 Nf6", "1.e4 c5 2.Nf3 d6 3.d4 Nf6 *"),
("B53c", "Sicilian: 2...d6 3.d4 cxd4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 *"),
("B53d", "Sicilian, Chekhover Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Qxd4 *"),
("B53e", "Sicilian, Chekhover, 4...Bd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Qxd4 Bd7 *"),
("B53f", "Sicilian, Chekhover, 4...a6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Qxd4 a6 *"),
("B53g", "Sicilian, Chekhover, 4...a6 5.Be3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Qxd4 a6 5.Be3 *"),
("B53h", "Sicilian, Chekhover, 4...a6 5.c4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Qxd4 a6 5.c4 *"),
("B53i", "Sicilian, Chekhover, 4...a6 5.c4 Nc6 6.Qd2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Qxd4 a6 5.c4 Nc6 6.Qd2 *"),
("B53j", "Sicilian, Chekhover, 4...a6 5.c4 Nc6 6.Qd2 g6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Qxd4 a6 5.c4 Nc6 6.Qd2 g6 *"),
("B53k", "Sicilian, Chekhover, 4...Nf6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Qxd4 Nf6 *"),
("B53k", "Sicilian, Chekhover, 4...Nf6 5.Nc3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Qxd4 Nf6 5.Nc3 *"),
("B53l", "Sicilian, Chekhover, 4...Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Qxd4 Nc6 *"),
("B53m", "Sicilian: Chekhover, Zaitsev Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Qxd4 Nc6 5.Bb5 Qd7 *"),
("B53n", "Sicilian, Chekhover, 4...Nc6 5.Bb5 Bd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Qxd4 Nc6 5.Bb5 Bd7 *"),
("B53o", "Sicilian, Chekhover, 4...Nc6 5.Bb5 Bd7 6.Bxc6 Bxc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Qxd4 Nc6 5.Bb5 Bd7 6.Bxc6 Bxc6 *"),
("B53p", "Sicilian, Chekhover, 7.c4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Qxd4 Nc6 5.Bb5 Bd7 6.Bxc6 Bxc6 7.c4 *"),
("B53q", "Sicilian, Chekhover, 7.c4 Nf6 8.Nc3 g6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Qxd4 Nc6 5.Bb5 Bd7 6.Bxc6 Bxc6 7.c4 Nf6 8.Nc3 g6 *"),
("B53r", "Sicilian, Chekhover, 7.Nc3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Qxd4 Nc6 5.Bb5 Bd7 6.Bxc6 Bxc6 7.Nc3 *"),
("B53s", "Sicilian, Chekhover, 7.Nc3 Nf6 8.Bg5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Qxd4 Nc6 5.Bb5 Bd7 6.Bxc6 Bxc6 7.Nc3 Nf6 8.Bg5 *"),
("B53t", "Sicilian, Chekhover, 7.Nc3 Nf6 8.Bg5 e6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Qxd4 Nc6 5.Bb5 Bd7 6.Bxc6 Bxc6 7.Nc3 Nf6 8.Bg5 e6 *"),
("B53u", "Sicilian, Chekhover, Main Line, 10.Qd3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Qxd4 Nc6 5.Bb5 Bd7 6.Bxc6 Bxc6 7.Nc3 Nf6 8.Bg5 e6 9.O-O-O Be7 10.Qd3 *"),
("B53v", "Sicilian, Chekhover, Main Line, 10.Rhe1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Qxd4 Nc6 5.Bb5 Bd7 6.Bxc6 Bxc6 7.Nc3 Nf6 8.Bg5 e6 9.O-O-O Be7 10.Rhe1 *"),
("B53w", "Sicilian, Chekhover, Main Line, 10.Rhe1 O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Qxd4 Nc6 5.Bb5 Bd7 6.Bxc6 Bxc6 7.Nc3 Nf6 8.Bg5 e6 9.O-O-O Be7 10.Rhe1 O-O *"),
("B53x", "Sicilian, Chekhover, Main Line, 11.Qd2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Qxd4 Nc6 5.Bb5 Bd7 6.Bxc6 Bxc6 7.Nc3 Nf6 8.Bg5 e6 9.O-O-O Be7 10.Rhe1 O-O 11.Qd2 *"),
("B54a", "Sicilian: Open, 2...d6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 *"),
("B54b", "Sicilian: Open, 2...d6, 4...e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 e5 *"),
("B54c", "Sicilian: Open, 2...d6, 4...a6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 a6 *"),
("B54d", "Sicilian: Open, 2...d6, 4...Nf6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 *"),
("B54e", "Sicilian: Open, 2...d6, 4...Nf6 5.Bd3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Bd3 *"),
("B54f", "Sicilian: Prins (Moscow) Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.f3 *"),
("B54g", "Sicilian: Prins (Moscow), 5...e6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.f3 e6 *"),
("B54h", "Sicilian: Prins (Moscow), 5...a6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.f3 a6 *"),
("B54i", "Sicilian: Prins (Moscow), 5...Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.f3 Nc6 *"),
("B54j", "Sicilian: Prins (Moscow), 5...Nc6 6.c4 Nxd4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.f3 Nc6 6.c4 Nxd4 *"),
("B54k", "Sicilian: Prins (Moscow), 5...Nc6 6.c4 Qb6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.f3 Nc6 6.c4 Qb6 *"),
("B54l", "Sicilian: Prins (Moscow), 5...e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.f3 e5 *"),
("B54m", "Sicilian: Prins (Moscow), 5...e5 6.Nb3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.f3 e5 6.Nb3 *"),
("B54n", "Sicilian: Prins (Moscow), 5...e5 6.Nb3 Be6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.f3 e5 6.Nb3 Be6 *"),
("B54o", "Sicilian: Prins (Moscow), 5...e5 6.Nb3 d5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.f3 e5 6.Nb3 d5 *"),
("B55", "Sicilian: Prins, Venice Attack", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.f3 e5 6.Bb5+ *"),
("B55", "Sicilian: Prins, Venice Attack, 6...Bd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.f3 e5 6.Bb5+ Bd7 *"),
("B55", "Sicilian: Prins, Venice Attack, 6...Nbd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.f3 e5 6.Bb5+ Nbd7 *"),
("B55", "Sicilian: Prins, Venice Attack, 6...Nbd7 7.Nf5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.f3 e5 6.Bb5+ Nbd7 7.Nf5 *"),
("B55", "Sicilian: Prins, Venice Attack, 6...Nbd7 7.Nf5 d5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.f3 e5 6.Bb5+ Nbd7 7.Nf5 d5 *"),
("B56a", "Sicilian: Open, 2...d6, 5.Nc3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 *"),
("B56b", "Sicilian: Open, 2...d6, 5.Nc3 e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 *"),
("B56c", "Sicilian: Venice Attack", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e5 6.Bb5+ *"),
("B56d", "Sicilian: Open, 2...d6, 5.Nc3 Nbd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nbd7 *"),
("B56e", "Sicilian: Open, 2...d6, 5.Nc3 Nbd7 6.Bc4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nbd7 6.Bc4 *"),
("B56f", "Sicilian: Open, 2...d6, 5.Nc3 Bd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Bd7 *"),
("B56g", "Sicilian: Open, 2...d6, 5.Nc3 Bd7 6.Bg5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Bd7 6.Bg5 *"),
("B56h", "Sicilian: Open, 2...d6 5.Nc3 Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 *"),
("B56i", "Sicilian: Open, 2...d6, 5.Nc3 Nc6 6.h3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.h3 *"),
("B56j", "Sicilian: Open, 2...d6, 5.Nc3 Nc6 6.g3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.g3 *"),
("B56k", "Sicilian: Open, 2...d6, 5.Nc3 Nc6 6.f4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.f4 *"),
("B56l", "Sicilian: Open, 2...d6, 5.Nc3 Nc6 6.f3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.f3 *"),
("B56m", "Sicilian: Open, 2...d6, 5.Nc3 Nc6 6.f3 e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.f3 e5 *"),
("B56m", "Sicilian: Open, 2...d6, 5.Nc3 Nc6 6.f3 e5 7.Nb3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.f3 e5 7.Nb3 *"),
("B56n", "Sicilian: Open, 2...d6, 5.Nc3 Nc6 6.f3 e5 7.Nb3 Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.f3 e5 7.Nb3 Be7 *"),
("B56o", "Sicilian: Open, 2...d6, 5.Nc3 Nc6 6.Be3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be3 *"),
("B56p", "Sicilian: Open, 2...d6, 5.Nc3 Nc6 6.Be3 e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be3 e5 *"),
("B56q", "Sicilian: Open, 2...d6, 5.Nc3 Nc6 6.Be3 Ng4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be3 Ng4 *"),
("B56r", "Sicilian: Open, 2...d6, 5.Nc3 Nc6 6.Be3 Ng4 7.Bb5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be3 Ng4 7.Bb5 *"),
("B57a", "Sicilian: Sozin", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bc4 *"),
("B57a", "Sicilian: Sozin, Magnus Smith Trap", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bc4 g6 7.Nxc6 bxc6 8.e5 *"),
("B57b", "Sicilian: Sozin, 6...Bd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bc4 Bd7 *"),
("B57c", "Sicilian: Sozin, 6...Bd7 7.O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bc4 Bd7 7.O-O *"),
("B57d", "Sicilian: Sozin, Benko Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bc4 Qb6 *"),
("B57e", "Sicilian: Sozin, Benko, 7.Nde2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bc4 Qb6 7.Nde2 *"),
("B57f", "Sicilian: Sozin, Benko, 7.Ndb5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bc4 Qb6 7.Ndb5 *"),
("B57g", "Sicilian: Sozin, Benko, 7.Nxc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bc4 Qb6 7.Nxc6 *"),
("B57h", "Sicilian: Sozin, Benko, 7.Nxc6 bxc6 8.O-O g6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bc4 Qb6 7.Nxc6 bxc6 8.O-O g6 *"),
("B57i", "Sicilian: Sozin, Benko, 7.Nb3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bc4 Qb6 7.Nb3 *"),
("B57j", "Sicilian: Sozin, Benko, 7.Nb3 e6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bc4 Qb6 7.Nb3 e6 *"),
("B57k", "Sicilian: Sozin, Benko, 7.Nb3 e6 8.Be3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bc4 Qb6 7.Nb3 e6 8.Be3 *"),
("B57l", "Sicilian: Sozin, Benko, 7.Nb3 e6 8.Bf4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bc4 Qb6 7.Nb3 e6 8.Bf4 *"),
("B57m", "Sicilian: Sozin, Benko, 7.Nb3 e6 8.O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bc4 Qb6 7.Nb3 e6 8.O-O *"),
("B57n", "Sicilian: Sozin, Benko, 7.Nb3 e6 8.O-O Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bc4 Qb6 7.Nb3 e6 8.O-O Be7 *"),
("B57o", "Sicilian: Sozin, Benko, 7.Nb3 e6 8.O-O Be7 9.Be3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bc4 Qb6 7.Nb3 e6 8.O-O Be7 9.Be3 *"),
("B58a", "Sicilian: Boleslavsky Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 *"),
("B58b", "Sicilian: Boleslavsky, 6...Bd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 Bd7 *"),
("B58c", "Sicilian: Boleslavsky, 6...a6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 a6 *"),
("B58d", "Sicilian: Boleslavsky, 6...Nxd4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 Nxd4 *"),
("B58e", "Sicilian: Boleslavsky, 6...e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 *"),
("B58f", "Sicilian: Boleslavsky, 7.Ndb5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Ndb5 *"),
("B58g", "Sicilian: Boleslavsky, Louma Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nxc6 *"),
("B58g", "Sicilian: Boleslavsky, Louma Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nxc6 bxc6 *"),
("B58h", "Sicilian: Boleslavsky, 7.Nf3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nf3 *"),
("B58i", "Sicilian: Boleslavsky, 7.Nf3 h6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nf3 h6 *"),
("B58j", "Sicilian: Boleslavsky, 7.Nf3 h6 8.O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nf3 h6 8.O-O *"),
("B58k", "Sicilian: Boleslavsky, 7.Nf3 h6 8.O-O Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nf3 h6 8.O-O Be7 *"),
("B58l", "Sicilian: Boleslavsky, 7.Nf3 h6 8.O-O Be7 9.h3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nf3 h6 8.O-O Be7 9.h3 *"),
("B58m", "Sicilian: Boleslavsky, 7.Nf3 h6 8.O-O Be7 9.Re1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nf3 h6 8.O-O Be7 9.Re1 *"),
("B58n", "Sicilian: Boleslavsky, 7.Nf3, 9.Re1 O-O 10.h3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nf3 h6 8.O-O Be7 9.Re1 O-O 10.h3 *"),
("B58o", "Sicilian: Boleslavsky, 7.Nf3, 9.Re1 O-O 10.h3 a6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nf3 h6 8.O-O Be7 9.Re1 O-O 10.h3 a6 *"),
("B58p", "Sicilian: Boleslavsky, 7.Nf3, 9.Re1 O-O 10.h3 Be6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nf3 h6 8.O-O Be7 9.Re1 O-O 10.h3 Be6 *"),
("B58p", "Sicilian: Boleslavsky, 7.Nf3, 9.Re1 O-O 10.h3 Be6 11.Bf1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nf3 h6 8.O-O Be7 9.Re1 O-O 10.h3 Be6 11.Bf1 *"),
("B58q", "Sicilian: Boleslavsky, 7.Nf3, 9.Re1 O-O 10.h3 Be6 11.Bf1 Qa5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nf3 h6 8.O-O Be7 9.Re1 O-O 10.h3 Be6 11.Bf1 Qa5 *"),
("B58r", "Sicilian: Boleslavsky, 7.Nf3, 9.Re1 O-O 10.h3 Be6 11.Bf1 Nb8", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nf3 h6 8.O-O Be7 9.Re1 O-O 10.h3 Be6 11.Bf1 Nb8 *"),
("B58s", "Sicilian: Boleslavsky, 7.Nf3, 9.Re1 O-O 10.h3 Be6 11.Bf1 Nb8 12.a4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nf3 h6 8.O-O Be7 9.Re1 O-O 10.h3 Be6 11.Bf1 Nb8 12.a4 *"),
("B58t", "Sicilian: Boleslavsky, 7.Nf3, 9.Re1 O-O 10.h3 Be6 11.Bf1 Nb8 12.b3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nf3 h6 8.O-O Be7 9.Re1 O-O 10.h3 Be6 11.Bf1 Nb8 12.b3 *"),
("B58u", "Sicilian: Boleslavsky, 7.Nf3, 9.Re1 O-O 10.h3 Be6 11.Bf1 Nb8 12.b3 a6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nf3 h6 8.O-O Be7 9.Re1 O-O 10.h3 Be6 11.Bf1 Nb8 12.b3 a6 *"),
("B59a", "Sicilian: Boleslavsky, 7.Nb3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nb3 *"),
("B59b", "Sicilian: Boleslavsky, 7.Nb3 Be6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nb3 Be6 *"),
("B59c", "Sicilian: Boleslavsky, 7.Nb3 Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nb3 Be7 *"),
("B59d", "Sicilian: Boleslavsky, 7.Nb3 Be7 8.Bg5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nb3 Be7 8.Bg5 *"),
("B59e", "Sicilian: Boleslavsky, 7.Nb3 Be7 8.Be3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nb3 Be7 8.Be3 *"),
("B59f", "Sicilian: Boleslavsky, 7.Nb3 Be7 8.O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nb3 Be7 8.O-O *"),
("B59g", "Sicilian: Boleslavsky, 7.Nb3 Be7 8.O-O O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nb3 Be7 8.O-O O-O *"),
("B59h", "Sicilian: Boleslavsky, 7.Nb3 Be7 8.O-O O-O 9.Bg5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Bg5 *"),
("B59i", "Sicilian: Boleslavsky, 7.Nb3 Be7 8.O-O O-O 9.Kh1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Kh1 *"),
("B59j", "Sicilian: Boleslavsky, 7.Nb3 Be7 8.O-O O-O 9.f4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.f4 *"),
("B59k", "Sicilian: Boleslavsky, 7.Nb3 Be7 8.O-O O-O 9.Be3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Be3 *"),
("B59l", "Sicilian: Boleslavsky, 7.Nb3 Be7 8.O-O O-O 9.Be3 Be6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Be3 Be6 *"),
("B59m", "Sicilian: Boleslavsky, 7.Nb3 Be7 8.O-O O-O 9.Be3 Be6 10.Bf3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Be3 Be6 10.Bf3 *"),
("B59n", "Sicilian: Boleslavsky, 7.Nb3 Be7 8.O-O O-O 9.Be3 a5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Be3 a5 *"),
("B59o", "Sicilian: Boleslavsky, 7.Nb3 Be7 8.O-O O-O 9.Be3 a5 10.a4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Be3 a5 10.a4 *"),
("B60a", "Sicilian: Richter-Rauzer", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 *"),
("B60b", "Sicilian: Richter-Rauzer, Bondarevsky Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 g6 *"),
("B60c", "Sicilian: Richter-Rauzer, 6...Qa5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 Qa5 *"),
("B60d", "Sicilian: Richter-Rauzer, 6...Qb6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 Qb6 *"),
("B60e", "Sicilian: Richter-Rauzer, 6...a6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 a6 *"),
("B60f", "Sicilian: Richter-Rauzer, Larsen Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 Bd7 *"),
("B60g", "Sicilian: Richter-Rauzer, Larsen, 7.Nb3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 Bd7 7.Nb3 *"),
("B60h", "Sicilian: Richter-Rauzer, Larsen, 7.Bxf6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 Bd7 7.Bxf6 *"),
("B60i", "Sicilian: Richter-Rauzer, Larsen, 7.Be2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 Bd7 7.Be2 *"),
("B60j", "Sicilian: Richter-Rauzer, Larsen, 7.Be2 a6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 Bd7 7.Be2 a6 *"),
("B60k", "Sicilian: Richter-Rauzer, Larsen, 7.Be2 Qa5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 Bd7 7.Be2 Qa5 *"),
("B61a", "Sicilian: Richter-Rauzer, Larsen, 7.Qd2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 Bd7 7.Qd2 *"),
("B61b", "Sicilian: Richter-Rauzer, Larsen, 7.Qd2 a6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 Bd7 7.Qd2 a6 *"),
("B61c", "Sicilian: Richter-Rauzer, Larsen, 7.Qd2 Nxd4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 Bd7 7.Qd2 Nxd4 *"),
("B61d", "Sicilian: Richter-Rauzer, Larsen, 7.Qd2 Rc8", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 Bd7 7.Qd2 Rc8 *"),
("B61e", "Sicilian: Richter-Rauzer, Larsen, 7.Qd2 Rc8 8.f4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 Bd7 7.Qd2 Rc8 8.f4 *"),
("B61f", "Sicilian: Richter-Rauzer, Larsen, 7.Qd2 Rc8 8.O-O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 Bd7 7.Qd2 Rc8 8.O-O-O *"),
("B61g", "Sicilian: Richter-Rauzer, Larsen, Main Line", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 Bd7 7.Qd2 Rc8 8.O-O-O Nxd4 9.Qxd4 Qa5 *"),
("B61h", "Sicilian: Richter-Rauzer, Larsen, Main Line, 10.Bd2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 Bd7 7.Qd2 Rc8 8.O-O-O Nxd4 9.Qxd4 Qa5 10.Bd2 *"),
("B61i", "Sicilian: Richter-Rauzer, Larsen, Main Line, 10.f4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 Bd7 7.Qd2 Rc8 8.O-O-O Nxd4 9.Qxd4 Qa5 10.f4 *"),
("B62a", "Sicilian: Richter-Rauzer, 6...e6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 *"),
("B62b", "Sicilian: Richter-Rauzer, 6...e6 7.Be2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Be2 *"),
("B62c", "Sicilian: Richter-Rauzer, 6...e6 7.Be2 Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Be2 Be7 *"),
("B62d", "Sicilian: Richter-Rauzer, Podebrady Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Nb3 *"),
("B62e", "Sicilian: Richter-Rauzer, Richter Attack", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Nxc6 *"),
("B62f", "Sicilian: Richter-Rauzer, Keres Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd3 *"),
("B62g", "Sicilian: Richter-Rauzer, Margate (Alekhine) Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Bb5 *"),
("B62h", "Sicilian: Richter-Rauzer, Margate, 7...Bd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Bb5 Bd7 *"),
("B62i", "Sicilian: Richter-Rauzer, Margate, 7...Bd7 8.Bxc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Bb5 Bd7 8.Bxc6 *"),
("B63a", "Sicilian: Richter-Rauzer, Rauzer Attack (7.Qd2)", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 *"),
("B63b", "Sicilian: Richter-Rauzer, 7.Qd2 Nxd4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Nxd4 *"),
("B63c", "Sicilian: Richter-Rauzer, 7.Qd2 Qb6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Qb6 *"),
("B63d", "Sicilian: Richter-Rauzer, 7.Qd2 Qb6 8.Nb3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Qb6 8.Nb3 *"),
("B63e", "Sicilian: Richter-Rauzer, 7.Qd2 Qb6 8.Nb3 a6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Qb6 8.Nb3 a6 *"),
("B63f", "Sicilian: Richter-Rauzer, 7.Qd2 Qb6 8.Nb3 a6 9.O-O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Qb6 8.Nb3 a6 9.O-O-O *"),
("B63g", "Sicilian: Richter-Rauzer, 7.Qd2 Qb6 8.Nb3 a6 9.O-O-O Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Qb6 8.Nb3 a6 9.O-O-O Be7 *"),
("B63h", "Sicilian: Richter-Rauzer, 7.Qd2 h6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 h6 *"),
("B63i", "Sicilian: Richter-Rauzer, 7.Qd2 h6 8.Bxf6 gxf6 9.O-O-O a6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 h6 8.Bxf6 gxf6 9.O-O-O a6 *"),
("B63j", "Sicilian: Richter-Rauzer, 7.Qd2 h6 8.Bxf6 gxf6 9.O-O-O a6 10.f4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 h6 8.Bxf6 gxf6 9.O-O-O a6 10.f4 *"),
("B63k", "Sicilian: Richter-Rauzer, 7.Qd2 Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 *"),
("B63l", "Sicilian: Richter-Rauzer, 7.Qd2 Be7 8.O-O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O *"),
("B63m", "Sicilian: Richter-Rauzer, 7.Qd2 Be7 8.O-O-O Nxd4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O Nxd4 *"),
("B63m", "Sicilian: Richter-Rauzer, 7.Qd2 Be7 8.O-O-O Nxd4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O Nxd4 9.Qxd4 *"),
("B63n", "Sicilian: Richter-Rauzer, 7.Qd2 Be7 8.O-O-O Nxd4 9.Qxd4 O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O Nxd4 9.Qxd4 O-O *"),
("B63o", "Sicilian: Richter-Rauzer, 7.Qd2 Be7 8.O-O-O O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O O-O *"),
("B63p", "Sicilian: Richter-Rauzer, 7.Qd2 Be7, 9.f3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O O-O 9.f3 *"),
("B63q", "Sicilian: Richter-Rauzer, 7.Qd2 Be7, 9.Nb3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O O-O 9.Nb3 *"),
("B63r", "Sicilian: Richter-Rauzer, 7.Qd2 Be7, 9.Nb3 a5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O O-O 9.Nb3 a5 *"),
("B63s", "Sicilian: Richter-Rauzer, 7.Qd2 Be7, 9.Nb3 a6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O O-O 9.Nb3 a6 *"),
("B63t", "Sicilian: Richter-Rauzer, Podebrad Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O O-O 9.Nb3 Qb6 *"),
("B63u", "Sicilian: Richter-Rauzer, Podebrad, 10.f3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O O-O 9.Nb3 Qb6 10.f3 *"),
("B63v", "Sicilian: Richter-Rauzer, Podebrad, 10.f3 Rd8", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O O-O 9.Nb3 Qb6 10.f3 Rd8 *"),
("B63w", "Sicilian: Richter-Rauzer, Podebrad, 10.f3 Rd8 11.Kb1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O O-O 9.Nb3 Qb6 10.f3 Rd8 11.Kb1 *"),
("B64a", "Sicilian: Richter-Rauzer, 7.Qd2 Be7, 9.f4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O O-O 9.f4 *"),
("B64b", "Sicilian: Richter-Rauzer, Geller Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O O-O 9.f4 e5 *"),
("B64c", "Sicilian: Richter-Rauzer, 7.Qd2 Be7, 9.f4 h6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O O-O 9.f4 h6 *"),
("B64d", "Sicilian: Richter-Rauzer, 7.Qd2 Be7, 9.f4 h6 10.Bh4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O O-O 9.f4 h6 10.Bh4 *"),
("B64e", "Sicilian: Richter-Rauzer, 7.Qd2 Be7, 9.f4 h6 10.Bh4 Bd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O O-O 9.f4 h6 10.Bh4 Bd7 *"),
("B64f", "Sicilian: Richter-Rauzer, 7.Qd2 Be7, 9.f4 h6 10.Bh4 e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O O-O 9.f4 h6 10.Bh4 e5 *"),
("B65a", "Sicilian: Richter-Rauzer, 7...Be7, 9.f4 Nxd4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O O-O 9.f4 Nxd4 *"),
("B65a", "Sicilian: Richter-Rauzer, 7...Be7, 9.f4 Nxd4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O O-O 9.f4 Nxd4 10.Qxd4 *"),
("B65b", "Sicilian: Richter-Rauzer, 7...Be7, 9.f4 Nxd4 10.Qxd4 h6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O O-O 9.f4 Nxd4 10.Qxd4 h6 *"),
("B65c", "Sicilian: Richter-Rauzer, 7...Be7, 9.f4 Nxd4 10.Qxd4 Qa5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O O-O 9.f4 Nxd4 10.Qxd4 Qa5 *"),
("B65d", "Sicilian: Richter-Rauzer, 7...Be7, 9.f4 Nxd4 10.Qxd4 Qa5 11.Kb1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O O-O 9.f4 Nxd4 10.Qxd4 Qa5 11.Kb1 *"),
("B65e", "Sicilian: Richter-Rauzer, 7...Be7, 9.f4 Nxd4 10.Qxd4 Qa5 11.Bc4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O O-O 9.f4 Nxd4 10.Qxd4 Qa5 11.Bc4 *"),
("B65f", "Sicilian: Richter-Rauzer, 7...Be7, 9.f4 Nxd4 10.Qxd4 Qa5 11.Bc4 Bd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O O-O 9.f4 Nxd4 10.Qxd4 Qa5 11.Bc4 Bd7 *"),
("B65g", "Sicilian: Richter-Rauzer, 7...Be7, 9.f4 Nxd4 10.Qxd4 Qa5 11.Bc4 Bd7 12.e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O O-O 9.f4 Nxd4 10.Qxd4 Qa5 11.Bc4 Bd7 12.e5 *"),
("B66a", "Sicilian: Richter-Rauzer, 7...a6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 *"),
("B66b", "Sicilian: Richter-Rauzer, 7...a6 8.Be2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.Be2 *"),
("B66c", "Sicilian: Richter-Rauzer, 7...a6 8.O-O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O *"),
("B66d", "Sicilian: Richter-Rauzer, 7...a6 8.O-O-O Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Be7 *"),
("B66e", "Sicilian: Richter-Rauzer, 7...a6 8.O-O-O Nxd4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Nxd4 *"),
("B66e", "Sicilian: Richter-Rauzer, 7...a6 8.O-O-O Nxd4 9.Qxd4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Nxd4 9.Qxd4 *"),
("B66e", "Sicilian: Richter-Rauzer, 7...a6 8.O-O-O Nxd4 9.Qxd4 Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Nxd4 9.Qxd4 Be7 *"),
("B66f", "Sicilian: Richter-Rauzer, 7...a6 8.O-O-O h6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O h6 *"),
("B66g", "Sicilian: Richter-Rauzer, 7...a6 8.O-O-O h6 9.Bf4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O h6 9.Bf4 *"),
("B66h", "Sicilian: Richter-Rauzer, 7...a6 8.O-O-O h6 9.Bf4, 11.f3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O h6 9.Bf4 Bd7 10.Nxc6 Bxc6 11.f3 *"),
("B66i", "Sicilian: Richter-Rauzer, 7...a6 8.O-O-O h6 9.Be3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O h6 9.Be3 *"),
("B66j", "Sicilian: Richter-Rauzer, 7...a6 8.O-O-O h6 9.Be3 Qc7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O h6 9.Be3 Qc7 *"),
("B66k", "Sicilian: Richter-Rauzer, 7...a6 8.O-O-O h6 9.Be3 Bd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O h6 9.Be3 Bd7 *"),
("B66l", "Sicilian: Richter-Rauzer, 7...a6 8.O-O-O h6 9.Be3 Bd7 10.f3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O h6 9.Be3 Bd7 10.f3 *"),
("B66m", "Sicilian: Richter-Rauzer, 7...a6 8.O-O-O h6 9.Be3 Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O h6 9.Be3 Be7 *"),
("B66m", "Sicilian: Richter-Rauzer, 7...a6 8.O-O-O h6 9.Be3 Be7 10.f3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O h6 9.Be3 Be7 10.f3 *"),
("B66m", "Sicilian: Richter-Rauzer, 7...a6 8.O-O-O h6 9.Be3 Be7 10.f3 Nxd4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O h6 9.Be3 Be7 10.f3 Nxd4 *"),
("B66n", "Sicilian: Richter-Rauzer, 7...a6 8.O-O-O h6 9.Be3 Be7 10.f3 Nxd4 11.Bxd4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O h6 9.Be3 Be7 10.f3 Nxd4 11.Bxd4 *"),
("B66o", "Sicilian: Richter-Rauzer, 7...a6 8.O-O-O h6 9.Be3 Be7 10.f4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O h6 9.Be3 Be7 10.f4 *"),
("B66p", "Sicilian: Richter-Rauzer, 7...a6 8.O-O-O h6 9.Be3 Be7 10.f4 Nxd4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O h6 9.Be3 Be7 10.f4 Nxd4 *"),
("B67a", "Sicilian: Richter-Rauzer, 7...a6 8.O-O-O Bd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 *"),
("B67b", "Sicilian: Richter-Rauzer, 7...a6, 9.Be2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.Be2 *"),
("B67c", "Sicilian: Richter-Rauzer, 7...a6, 9.f3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f3 *"),
("B67d", "Sicilian: Richter-Rauzer, 7...a6, 9.f3 Rc8", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f3 Rc8 *"),
("B67e", "Sicilian: Richter-Rauzer, 7...a6, 9.f3 Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f3 Be7 *"),
("B67f", "Sicilian: Richter-Rauzer, 7...a6, 9.f3 Be7 10.h4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f3 Be7 10.h4 *"),
("B67g", "Sicilian: Richter-Rauzer, 7...a6, 9.f4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f4 *"),
("B67h", "Sicilian: Richter-Rauzer, 7...a6, 9.f4 h6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f4 h6 *"),
("B67i", "Sicilian: Richter-Rauzer, 7...a6, 9.f4 h6 10.Bh4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f4 h6 10.Bh4 *"),
("B67j", "Sicilian: Richter-Rauzer, 7...a6, 9.f4 h6 10.Bh4 g5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f4 h6 10.Bh4 g5 *"),
("B67k", "Sicilian: Richter-Rauzer, 7...a6, 9.f4 b5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f4 b5 *"),
("B67l", "Sicilian: Richter-Rauzer, 7...a6, 9.f4 b5 10.Nxc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f4 b5 10.Nxc6 *"),
("B67m", "Sicilian: Richter-Rauzer, 7...a6, 9.f4 b5 10.Bxf6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f4 b5 10.Bxf6 *"),
("B67m", "Sicilian: Richter-Rauzer, 7...a6, 9.f4 b5 10.Bxf6 gxf6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f4 b5 10.Bxf6 gxf6 *"),
("B67n", "Sicilian: Richter-Rauzer, 7...a6, 9.f4 b5 10.Bxf6 gxf6 11.f5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f4 b5 10.Bxf6 gxf6 11.f5 *"),
("B67o", "Sicilian: Richter-Rauzer, 7...a6, 9.f4 b5 10.Bxf6 gxf6 11.Nxc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f4 b5 10.Bxf6 gxf6 11.Nxc6 *"),
("B67p", "Sicilian: Richter-Rauzer, 7...a6, 9.f4 b5 10.Bxf6 gxf6 11.Nxc6 Bxc6 12.Qe1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f4 b5 10.Bxf6 gxf6 11.Nxc6 Bxc6 12.Qe1 *"),
("B67q", "Sicilian: Richter-Rauzer, 7...a6, 9.f4 b5 10.Bxf6 gxf6 11.Kb1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f4 b5 10.Bxf6 gxf6 11.Kb1 *"),
("B67r", "Sicilian: Richter-Rauzer, 7...a6, 9.f4 b5 10.Bxf6 gxf6 11.Kb1 Qb6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f4 b5 10.Bxf6 gxf6 11.Kb1 Qb6 *"),
("B67s", "Sicilian: Richter-Rauzer, 7...a6, 9.f4 b5 10.Bxf6 gxf6 11.Kb1 Qb6 12.Nxc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f4 b5 10.Bxf6 gxf6 11.Kb1 Qb6 12.Nxc6 *"),
("B68", "Sicilian: Richter-Rauzer, 7...a6, 9.f4 Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f4 Be7 *"),
("B68", "Sicilian: Richter-Rauzer, 7...a6, 9.f4 Be7 10.Be2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f4 Be7 10.Be2 *"),
("B68", "Sicilian: Richter-Rauzer, 7...a6, 9.f4 Be7 10.Be2 Qc7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f4 Be7 10.Be2 Qc7 *"),
("B68", "Sicilian: Richter-Rauzer, 7...a6, 9.f4 Be7 10.Nf3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f4 Be7 10.Nf3 *"),
("B68", "Sicilian: Richter-Rauzer, 7...a6, 9.f4 Be7 10.Nf3 b5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f4 Be7 10.Nf3 b5 *"),
("B68", "Sicilian: Richter-Rauzer, 7...a6, 9.f4 Be7 10.Nf3 b5 11.e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f4 Be7 10.Nf3 b5 11.e5 *"),
("B69", "Sicilian: Richter-Rauzer, 7...a6, 9.f4 Be7 10.Nf3 b5 11.Bxf6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f4 Be7 10.Nf3 b5 11.Bxf6 *"),
("B69", "Sicilian: Richter-Rauzer, 7...a6, 9.f4 Be7 10.Nf3 b5 11.Bxf6 gxf6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f4 Be7 10.Nf3 b5 11.Bxf6 gxf6 *"),
("B69", "Sicilian: Richter-Rauzer, 7...a6, 9.f4 Be7 10.Nf3 b5 11.Bxf6 gxf6 12.Kb1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f4 Be7 10.Nf3 b5 11.Bxf6 gxf6 12.Kb1 *"),
("B69", "Sicilian: Richter-Rauzer, 7...a6, 9.f4 Be7 10.Nf3 b5 11.Bxf6 gxf6 12.f5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 Nc6 6.Bg5 e6 7.Qd2 a6 8.O-O-O Bd7 9.f4 Be7 10.Nf3 b5 11.Bxf6 gxf6 12.f5 *"),
("B70a", "Sicilian: Dragon", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 *"),
("B70a", "Sicilian: Dragon, 6.f3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.f3 *"),
("B70b", "Sicilian: Dragon, 6.g3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.g3 *"),
("B70c", "Sicilian: Dragon, 6.g3 Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.g3 Nc6 *"),
("B70c", "Sicilian: Dragon, 6.g3 Nc6 7.Bg2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.g3 Nc6 7.Bg2 *"),
("B70d", "Sicilian: Dragon, 6.g3 Nc6 7.Nde2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.g3 Nc6 7.Nde2 *"),
("B70e", "Sicilian: Dragon, 6.Bg5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Bg5 *"),
("B70e", "Sicilian: Dragon, 6.Bg5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Bg5 Bg7 *"),
("B70f", "Sicilian: Dragon, 6.Bc4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Bc4 *"),
("B70f", "Sicilian: Dragon, 6.Bc4 Bg7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Bc4 Bg7 *"),
("B70g", "Sicilian: Dragon, 6.Bc4 Bg7 7.h3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Bc4 Bg7 7.h3 *"),
("B70h", "Sicilian: Dragon, 6.Bc4 Bg7 7.O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Bc4 Bg7 7.O-O *"),
("B70i", "Sicilian: Dragon, 6.Be2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be2 *"),
("B70i", "Sicilian: Dragon, 6.Be2 Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be2 Nc6 *"),
("B70j", "Sicilian: Dragon, 6.Be2 a6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be2 a6 *"),
("B70j", "Sicilian: Dragon, 6.Be2 a6 7.a4 Bg7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be2 a6 7.a4 Bg7 *"),
("B70k", "Sicilian: Dragon, 6.Be2 Bg7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be2 Bg7 *"),
("B70l", "Sicilian: Dragon, 6.Be2 Bg7 7.O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be2 Bg7 7.O-O *"),
("B70l", "Sicilian: Dragon, 6.Be2 Bg7 7.O-O Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be2 Bg7 7.O-O Nc6 *"),
("B70m", "Sicilian: Dragon, 6.Be2 Bg7 7.O-O Nc6 8.Nb3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be2 Bg7 7.O-O Nc6 8.Nb3 *"),
("B70n", "Sicilian: Dragon, 6.Be2 Bg7 7.O-O Nc6 8.Nb3 O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be2 Bg7 7.O-O Nc6 8.Nb3 O-O *"),
("B70o", "Sicilian: Dragon, 6.Be2 Bg7 7.O-O O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be2 Bg7 7.O-O O-O *"),
("B70p", "Sicilian: Dragon, 6.Be2 Bg7 7.O-O O-O 8.Re1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be2 Bg7 7.O-O O-O 8.Re1 *"),
("B70p", "Sicilian: Dragon, 6.Be2 Bg7 7.O-O O-O 8.Re1 Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be2 Bg7 7.O-O O-O 8.Re1 Nc6 *"),
("B70q", "Sicilian: Dragon, 6.Be2 Bg7 7.O-O O-O 8.Re1 Nc6 9.Nb3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be2 Bg7 7.O-O O-O 8.Re1 Nc6 9.Nb3 *"),
("B70r", "Sicilian: Dragon, 6.Be2 Bg7 7.O-O O-O 8.Re1 Nc6 9.Nb3 Be6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be2 Bg7 7.O-O O-O 8.Re1 Nc6 9.Nb3 Be6 *"),
("B70s", "Sicilian: Dragon, 6.Be2 Bg7 7.O-O O-O 8.Bg5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be2 Bg7 7.O-O O-O 8.Bg5 *"),
("B70s", "Sicilian: Dragon, 6.Be2 Bg7 7.O-O O-O 8.Bg5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be2 Bg7 7.O-O O-O 8.Bg5 Nc6 *"),
("B70t", "Sicilian: Dragon, 6.Be2 Bg7 7.O-O O-O 8.Bg5 Nc6 9.Nb3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be2 Bg7 7.O-O O-O 8.Bg5 Nc6 9.Nb3 *"),
("B70u", "Sicilian: Dragon, 6.Be2 Bg7 7.O-O O-O 8.Bg5 Nc6 9.Nb3 a6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be2 Bg7 7.O-O O-O 8.Bg5 Nc6 9.Nb3 a6 *"),
("B70v", "Sicilian: Dragon, 6.Be2 Bg7 7.O-O O-O 8.Bg5 Nc6 9.Nb3 Be6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be2 Bg7 7.O-O O-O 8.Bg5 Nc6 9.Nb3 Be6 *"),
("B70w", "Sicilian: Dragon, 6.Be2 Bg7 7.O-O O-O 8.Bg5 Nc6 9.Nb3 Be6 10.Kh1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be2 Bg7 7.O-O O-O 8.Bg5 Nc6 9.Nb3 Be6 10.Kh1 *"),
("B71", "Sicilian: Dragon, Levenfish Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.f4 *"),
("B71", "Sicilian: Dragon, Levenfish, 6...Bg7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.f4 Bg7 *"),
("B71", "Sicilian: Dragon, Levenfish, 6...Bg7 7.e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.f4 Bg7 7.e5 *"),
("B71", "Sicilian: Dragon, Levenfish, 6...Nbd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.f4 Nbd7 *"),
("B71", "Sicilian: Dragon, Levenfish, 6...Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.f4 Nc6 *"),
("B71", "Sicilian: Dragon, Levenfish, 6...Nc6 7.Bb5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.f4 Nc6 7.Bb5 *"),
("B71", "Sicilian: Dragon, Levenfish, 6...Nc6 7.Nf3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.f4 Nc6 7.Nf3 *"),
("B71", "Sicilian: Dragon, Levenfish, 6...Nc6 7.Nxc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.f4 Nc6 7.Nxc6 *"),
("B72", "Sicilian: Dragon, 6.Be3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 *"),
("B72", "Sicilian: Dragon, 6.Be3 Bg7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 *"),
("B72", "Sicilian: Dragon, 6.Be3 Bg7 7.Qd2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Qd2 *"),
("B72", "Sicilian: Dragon, 6.Be3 Bg7 7.Bc4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Bc4 *"),
("B72", "Sicilian: Dragon, 6.Be3 Bg7 7.Bc4 Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Bc4 Nc6 *"),
("B72", "Sicilian: Dragon, 6.Be3 Bg7 7.Bc4 O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Bc4 O-O *"),
("B72", "Sicilian: Dragon, Classical Attack", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Be2 *"),
("B72", "Sicilian: Dragon, Classical Attack", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Be2 O-O *"),
("B72", "Sicilian: Dragon, Classical Attack", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Be2 O-O 8.O-O *"),
("B72", "Sicilian: Dragon, Classical, Amsterdam Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Be2 Nc6 8.Qd2 *"),
("B72", "Sicilian: Dragon, Classical, Grigoriev Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Be2 Nc6 8.Qd2 O-O 9.O-O-O *"),
("B72", "Sicilian: Dragon, Classical, Nottingham Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Be2 Nc6 8.Nb3 *"),
("B73", "Sicilian: Dragon, Classical, 8.O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Be2 Nc6 8.O-O *"),
("B73", "Sicilian: Dragon, Classical, 8.O-O O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Be2 Nc6 8.O-O O-O *"),
("B73", "Sicilian: Dragon, Classical, 9.Kh1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Be2 Nc6 8.O-O O-O 9.Kh1 *"),
("B73", "Sicilian: Dragon, Classical, Richter Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Be2 Nc6 8.O-O O-O 9.Qd2 *"),
("B73", "Sicilian: Dragon, Classical, 9.f4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Be2 Nc6 8.O-O O-O 9.f4 *"),
("B73", "Sicilian: Dragon, Classical, Zollner Gambit", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Be2 Nc6 8.O-O O-O 9.f4 Qb6 10.e5 *"),
("B74", "Sicilian: Dragon, Classical, 9.Nb3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Be2 Nc6 8.O-O O-O 9.Nb3 *"),
("B74", "Sicilian: Dragon, Classical, 9.Nb3 b6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Be2 Nc6 8.O-O O-O 9.Nb3 b6 *"),
("B74", "Sicilian: Dragon, Classical, Alekhine Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Be2 Nc6 8.O-O O-O 9.Nb3 a5 *"),
("B74", "Sicilian: Dragon, Classical, 9.Nb3 a6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Be2 Nc6 8.O-O O-O 9.Nb3 a6 *"),
("B74", "Sicilian: Dragon, Classical, 9.Nb3 Be6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Be2 Nc6 8.O-O O-O 9.Nb3 Be6 *"),
("B74", "Sicilian: Dragon, Classical, 9.Nb3 Be6 10.f4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Be2 Nc6 8.O-O O-O 9.Nb3 Be6 10.f4 *"),
("B74", "Sicilian: Dragon, Classical, 9.Nb3 Be6 10.f4 Rc8", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Be2 Nc6 8.O-O O-O 9.Nb3 Be6 10.f4 Rc8 *"),
("B74", "Sicilian: Dragon, Classical, Tartakower Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Be2 Nc6 8.O-O O-O 9.Nb3 Be6 10.f4 Qc8 *"),
("B74", "Sicilian: Dragon, Classical, Maroczy Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Be2 Nc6 8.O-O O-O 9.Nb3 Be6 10.f4 Na5 *"),
("B74", "Sicilian: Dragon, Classical, Stockholm Attack", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Be2 Nc6 8.O-O O-O 9.Nb3 Be6 10.f4 Na5 11.f5 Bc4 12.Nxa5 Bxe2 13.Qxe2 Qxa5 14.g4 *"),
("B74", "Sicilian: Dragon, Classical, Spielmann Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Be2 Nc6 8.O-O O-O 9.Nb3 Be6 10.f4 Na5 11.f5 Bc4 12.Bd3 *"),
("B74", "Sicilian: Dragon, Classical, Bernard Defence", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.Be2 Nc6 8.O-O O-O 9.Nb3 Be6 10.f4 Na5 11.f5 Bc4 12.Bd3 Bxd3 13.cxd3 d5 *"),
("B75", "Sicilian: Dragon, Yugoslav Attack", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 *"),
("B75", "Sicilian: Dragon, Yugoslav, 7...a6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 a6 *"),
("B75", "Sicilian: Dragon, Yugoslav, 7...Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 Nc6 *"),
("B75", "Sicilian: Dragon, Yugoslav, 7...Nc6 8.Bc4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 Nc6 8.Bc4 *"),
("B75", "Sicilian: Dragon, Yugoslav, 7...Nc6 8.Qd2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 Nc6 8.Qd2 *"),
("B75", "Sicilian: Dragon, Yugoslav, 7...Nc6 8.Qd2 Bd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 Nc6 8.Qd2 Bd7 *"),
("B75", "Sicilian: Dragon, Yugoslav, 7...Nc6 8.Qd2 Bd7 9.O-O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 Nc6 8.Qd2 Bd7 9.O-O-O *"),
("B75", "Sicilian: Dragon, Yugoslav, 7...Nc6 8.Qd2 Bd7 9.O-O-O Rc8", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 Nc6 8.Qd2 Bd7 9.O-O-O Rc8 *"),
("B76a", "Sicilian: Dragon, Yugoslav, 7.f3 O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O *"),
("B76b", "Sicilian: Dragon, Yugoslav, 7.f3 O-O 8.Qd2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 *"),
("B76c", "Sicilian: Dragon, Yugoslav, 8.Qd2 Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 *"),
("B76d", "Sicilian: Dragon, Yugoslav, 9.g4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.g4 *"),
("B76e", "Sicilian: Dragon, Yugoslav, 9.g4 Be6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.g4 Be6 *"),
("B76f", "Sicilian: Dragon, Yugoslav, 9.g4 Nxd4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.g4 Nxd4 *"),
("B76g", "Sicilian: Dragon, Yugoslav, 9.O-O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.O-O-O *"),
("B76h", "Sicilian: Dragon, Yugoslav, 9.O-O-O Bd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.O-O-O Bd7 *"),
("B76i", "Sicilian: Dragon, Yugoslav, 9.O-O-O Bd7 10.g4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.O-O-O Bd7 10.g4 *"),
("B76j", "Sicilian: Dragon, Yugoslav, 9.O-O-O Nxd4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.O-O-O Nxd4 *"),
("B76k", "Sicilian: Dragon, Yugoslav, 9.O-O-O Nxd4 10.Bxd4 Be6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.O-O-O Nxd4 10.Bxd4 Be6 *"),
("B76l", "Sicilian: Dragon, Yugoslav, 9.O-O-O Nxd4: 11.Kb1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.O-O-O Nxd4 10.Bxd4 Be6 11.Kb1 *"),
("B76m", "Sicilian: Dragon, Yugoslav, 9.O-O-O Nxd4: 11.Kb1 Qc7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.O-O-O Nxd4 10.Bxd4 Be6 11.Kb1 Qc7 *"),
("B76n", "Sicilian: Dragon, Yugoslav, 9.O-O-O Nxd4: 11.Kb1 Qc7 12.g4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.O-O-O Nxd4 10.Bxd4 Be6 11.Kb1 Qc7 12.g4 *"),
("B76o", "Sicilian: Dragon, Yugoslav, 9.O-O-O d5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.O-O-O d5 *"),
("B76p", "Sicilian: Dragon, Yugoslav, 9.O-O-O d5 10.Qe1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.O-O-O d5 10.Qe1 *"),
("B76q", "Sicilian: Dragon, Yugoslav, 9.O-O-O d5 10.exd5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.O-O-O d5 10.exd5 *"),
("B76q", "Sicilian: Dragon, Yugoslav, 9.O-O-O d5 10.exd5 Nxd5 11.Nxc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.O-O-O d5 10.exd5 Nxd5 11.Nxc6 *"),
("B76r", "Sicilian: Dragon, Yugoslav, 9.O-O-O d5: 12.Nxd5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.O-O-O d5 10.exd5 Nxd5 11.Nxc6 bxc6 12.Nxd5 *"),
("B76s", "Sicilian: Dragon, Yugoslav, 9.O-O-O d5: 12.Bd4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.O-O-O d5 10.exd5 Nxd5 11.Nxc6 bxc6 12.Bd4 *"),
("B76t", "Sicilian: Dragon, Yugoslav, 9.O-O-O d5: 12.Bd4 e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.O-O-O d5 10.exd5 Nxd5 11.Nxc6 bxc6 12.Bd4 e5 *"),
("B76u", "Sicilian: Dragon, Yugoslav, 9.O-O-O d5: 12.Bd4 e5 13.Bc5 Be6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.O-O-O d5 10.exd5 Nxd5 11.Nxc6 bxc6 12.Bd4 e5 13.Bc5 Be6 *"),
("B77a", "Sicilian: Dragon, Yugoslav, 9.Bc4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 *"),
("B77b", "Sicilian: Dragon, Yugoslav, Byrne Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 a5 *"),
("B77c", "Sicilian: Dragon, Yugoslav, 9.Bc4 Ne5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Ne5 *"),
("B77d", "Sicilian: Dragon, Yugoslav, 9.Bc4 Nd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Nd7 *"),
("B77e", "Sicilian: Dragon, Yugoslav, 9.Bc4 Nxd4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Nxd4 *"),
("B77f", "Sicilian: Dragon, Yugoslav, 9.Bc4 Bd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 *"),
("B77g", "Sicilian: Dragon, Yugoslav, 9.Bc4 Bd7 10.Bb3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.Bb3 *"),
("B77h", "Sicilian: Dragon, Yugoslav, 9.Bc4 Bd7 10.g4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.g4 *"),
("B77i", "Sicilian: Dragon, Yugoslav, 9.Bc4 Bd7 10.h4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.h4 *"),
("B77j", "Sicilian: Dragon, Yugoslav, 9.Bc4 Bd7 10.h4 Rc8", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.h4 Rc8 *"),
("B78a", "Sicilian: Dragon, Yugoslav, 10.O-O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O *"),
("B78b", "Sicilian: Dragon, Yugoslav, 10.O-O-O Ne5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Ne5 *"),
("B78c", "Sicilian: Dragon, Yugoslav, 10.O-O-O Rc8", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Rc8 *"),
("B78c", "Sicilian: Dragon, Yugoslav, 10.O-O-O Rc8 11.Bb3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Rc8 11.Bb3 *"),
("B78d", "Sicilian: Dragon, Yugoslav, Modern Main Line", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Rc8 11.Bb3 Ne5 *"),
("B78e", "Sicilian: Dragon, Yugoslav, Main Line, 12.Kb1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Rc8 11.Bb3 Ne5 12.Kb1 *"),
("B78e", "Sicilian: Dragon, Yugoslav, Main Line, 12.Kb1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Rc8 11.Bb3 Ne5 12.Kb1 Nc4 13.Bxc4 Rxc4 *"),
("B78f", "Sicilian: Dragon, Yugoslav, Main Line, 12.Kb1: 14.g4 b5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Rc8 11.Bb3 Ne5 12.Kb1 Nc4 13.Bxc4 Rxc4 14.g4 b5 *"),
("B78g", "Sicilian: Dragon, Yugoslav, Main Line, 12.h4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Rc8 11.Bb3 Ne5 12.h4 *"),
("B78g", "Sicilian: Dragon, Yugoslav, Main Line, 12.h4 Nc4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Rc8 11.Bb3 Ne5 12.h4 Nc4 *"),
("B78h", "Sicilian: Dragon, Yugoslav, Main Line, 12.h4 Nc4: 14.g4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Rc8 11.Bb3 Ne5 12.h4 Nc4 13.Bxc4 Rxc4 14.g4 *"),
("B78i", "Sicilian: Dragon, Yugoslav, Main Line, 12.h4 Nc4: 14.h5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Rc8 11.Bb3 Ne5 12.h4 Nc4 13.Bxc4 Rxc4 14.h5 *"),
("B78j", "Sicilian: Dragon, Yugoslav, Main Line, 12.h4 h5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Rc8 11.Bb3 Ne5 12.h4 h5 *"),
("B78k", "Sicilian: Dragon, Yugoslav, Main Line, 12.h4 h5 13.Kb1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Rc8 11.Bb3 Ne5 12.h4 h5 13.Kb1 *"),
("B78l", "Sicilian: Dragon, Yugoslav, Main Line, 12.h4 h5 13.Bh6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Rc8 11.Bb3 Ne5 12.h4 h5 13.Bh6 *"),
("B78m", "Sicilian: Dragon, Yugoslav, Main Line, 12.h4 h5 13.Bh6 Bxh6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Rc8 11.Bb3 Ne5 12.h4 h5 13.Bh6 Bxh6 *"),
("B78n", "Sicilian: Dragon, Yugoslav, Main Line, 12.h4 h5 13.Bg5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Rc8 11.Bb3 Ne5 12.h4 h5 13.Bg5 *"),
("B78o", "Sicilian: Dragon, Yugoslav, Main Line, 12.h4 h5 13.Bg5 Rc5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Rc8 11.Bb3 Ne5 12.h4 h5 13.Bg5 Rc5 *"),
("B78p", "Sicilian: Dragon, Yugoslav, Main Line, 12.h4 h5 13.Bg5 Rc5 14.g4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Rc8 11.Bb3 Ne5 12.h4 h5 13.Bg5 Rc5 14.g4 *"),
("B78q", "Sicilian: Dragon, Yugoslav, Main Line, 12.h4 h5 13.Bg5 Rc5 14.Kb1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Rc8 11.Bb3 Ne5 12.h4 h5 13.Bg5 Rc5 14.Kb1 *"),
("B78r", "Sicilian: Dragon, Yugoslav, Old Main Line", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Qa5 *"),
("B78s", "Sicilian: Dragon, Yugoslav, Old Main Line, 11.Bb3 Rac8", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Qa5 11.Bb3 Rac8 *"),
("B78t", "Sicilian: Dragon, Yugoslav, Old Main Line, 11.Bb3 Rfc8", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Qa5 11.Bb3 Rfc8 *"),
("B79a", "Sicilian: Dragon, Yugoslav, Old Main Line, 12.h4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Qa5 11.Bb3 Rfc8 12.h4 *"),
("B79a", "Sicilian: Dragon, Yugoslav, Old Main Line, 12.h4 h5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Qa5 11.Bb3 Rfc8 12.h4 h5 *"),
("B79b", "Sicilian: Dragon, Yugoslav, Old Main Line, 12.h4 Ne5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Qa5 11.Bb3 Rfc8 12.h4 Ne5 *"),
("B79c", "Sicilian: Dragon, Yugoslav, Old Main Line, 12.h4 Ne5 13.Kb1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Qa5 11.Bb3 Rfc8 12.h4 Ne5 13.Kb1 *"),
("B79d", "Sicilian: Dragon, Yugoslav, Old Main Line, 12.h4 Ne5 13.Kb1 Nc4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Qa5 11.Bb3 Rfc8 12.h4 Ne5 13.Kb1 Nc4 *"),
("B79e", "Sicilian: Dragon, Yugoslav, Old Main Line, 12.h4 Ne5 13.g4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Qa5 11.Bb3 Rfc8 12.h4 Ne5 13.g4 *"),
("B79f", "Sicilian: Dragon, Yugoslav, Old Main Line, 12.h4 Ne5 13.h5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 g6 6.Be3 Bg7 7.f3 O-O 8.Qd2 Nc6 9.Bc4 Bd7 10.O-O-O Qa5 11.Bb3 Rfc8 12.h4 Ne5 13.h5 *"),
("B80a", "Sicilian: Scheveningen", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 *"),
("B80b", "Sicilian: Scheveningen, 6.Be3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be3 *"),
("B80c", "Sicilian: Scheveningen, 6.Be3 Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be3 Nc6 *"),
("B80d", "Sicilian: Scheveningen, 6.Be3 Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be3 Nc6 7.f3 *"),
("B80e", "Sicilian: Scheveningen, 6.Be3 a6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be3 a6 *"),
// Here, 7.g4 (the Perenyi Attack) is B81 (a variation of the Keres Attack)
("B80f", "Sicilian: Scheveningen, 7.Qd2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be3 a6 7.Qd2 *"),
("B80g", "Sicilian: Scheveningen, English Attack", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be3 a6 7.f3 *"),
("B80h", "Sicilian: Scheveningen, English Attack, 7...Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be3 a6 7.f3 Nc6 *"),
("B80h", "Sicilian: Scheveningen, English, 7...Nc6 8.Qd2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be3 a6 7.f3 Nc6 8.Qd2 *"),
("B80i", "Sicilian: Scheveningen, English, 7...Nc6 8.Qd2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be3 a6 7.f3 Nc6 8.Qd2 Be7 *"),
("B80j", "Sicilian: Scheveningen, English Attack, 7...b5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be3 a6 7.f3 b5 *"),
("B80k", "Sicilian: Scheveningen, English, 7...b5 8.Qd2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be3 a6 7.f3 b5 8.Qd2 *"),
("B80l", "Sicilian: Scheveningen, English, 7...b5, 8.Qd2 Bb7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be3 a6 7.f3 b5 8.Qd2 Bb7 *"),
("B80l", "Sicilian: Scheveningen, English, 7...b5 8.Qd2 Bb7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be3 a6 7.f3 b5 8.Qd2 Bb7 9.g4 *"),
("B80m", "Sicilian: Scheveningen, English, 7...b5 8.Qd2 Nbd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be3 a6 7.f3 b5 8.Qd2 Nbd7 *"),
("B80n", "Sicilian: Scheveningen, Vitolins Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bb5+ *"),
("B80n", "Sicilian: Scheveningen, Vitolins, 6...Bd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bb5+ Bd7 *"),
("B80o", "Sicilian: Scheveningen, Fianchetto Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g3 *"),
("B80p", "Sicilian: Scheveningen, Fianchetto Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g3 a6 *"),
("B80q", "Sicilian: Scheveningen, Fianchetto Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g3 Nc6 *"),
("B80r", "Sicilian: Scheveningen, Fianchetto Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g3 Nc6 7.Bg2 a6 *"),
("B80r", "Sicilian: Scheveningen, Fianchetto Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g3 Nc6 7.Bg2 a6 8.O-O *"),
("B80r", "Sicilian: Scheveningen, Fianchetto Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g3 Nc6 7.Bg2 a6 8.O-O Bd7 *"),
("B80s", "Sicilian: Scheveningen, Fianchetto Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g3 Nc6 7.Bg2 Qc7 *"),
("B80t", "Sicilian: Scheveningen, Fianchetto Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g3 Nc6 7.Bg2 Qc7 8.O-O a6 *"),
("B80u", "Sicilian: Scheveningen, Fianchetto Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g3 Nc6 7.Bg2 Qc7 8.O-O a6 9.Re1 Be7 *"),
("B81a", "Sicilian: Scheveningen, Keres Attack", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g4 *"),
("B81b", "Sicilian: Scheveningen, Keres, 6...e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g4 e5 *"),
("B81c", "Sicilian: Scheveningen, Keres, 6...a6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g4 a6 *"),
("B81d", "Sicilian: Scheveningen, Keres, Perenyi Attack", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g4 a6 7.Be3 *"),
("B81e", "Sicilian: Scheveningen, Keres, Perenyi Attack, 7...e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g4 a6 7.Be3 e5 *"),
("B81f", "Sicilian: Scheveningen, Keres, Perenyi Attack, 7...h6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g4 a6 7.Be3 h6 *"),
("B81g", "Sicilian: Scheveningen, Keres, Perenyi Attack, 7...h6 8.f4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g4 a6 7.Be3 h6 8.f4 *"),
("B81h", "Sicilian: Scheveningen, Keres, 6...h6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g4 h6 *"),
("B81i", "Sicilian: Scheveningen, Keres, 6...h6 7.h3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g4 h6 7.h3 *"),
("B81i", "Sicilian: Scheveningen, Keres, 6...h6 7.h3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g4 h6 7.h3 Nc6 8.Bg2 *"),
("B81i", "Sicilian: Scheveningen, Keres, 6...h6 7.h3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g4 h6 7.h3 a6 8.Bg2 *"),
("B81j", "Sicilian: Scheveningen, Keres, 6...h6 7.h4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g4 h6 7.h4 *"),
("B81k", "Sicilian: Scheveningen, Keres, 6...h6 7.h4 Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g4 h6 7.h4 Be7 *"),
("B81l", "Sicilian: Scheveningen, Keres, 6...h6 7.h4 Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g4 h6 7.h4 Nc6 *"),
("B81m", "Sicilian: Scheveningen, Keres, 6...h6 7.h4 Nc6 8.Rg1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g4 h6 7.h4 Nc6 8.Rg1 *"),
("B81n", "Sicilian: Scheveningen, Keres, 6...h6 7.h4 Nc6 8.Rg1 h5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g4 h6 7.h4 Nc6 8.Rg1 h5 *"),
("B81o", "Sicilian: Scheveningen, Keres, 6...h6 7.g5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g4 h6 7.g5 *"),
("B81p", "Sicilian: Scheveningen, Keres, 6...h6 7.Rg1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g4 h6 7.Rg1 *"),
("B81q", "Sicilian: Scheveningen, Keres, 6...Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g4 Nc6 *"),
("B81r", "Sicilian: Scheveningen, Keres, 6...Nc6 7.g5 Nd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g4 Nc6 7.g5 Nd7 *"),
("B81s", "Sicilian: Scheveningen, Keres, 6...Nc6 7.g5 Nd7 8.Be3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g4 Nc6 7.g5 Nd7 8.Be3 *"),
("B81t", "Sicilian: Scheveningen, Keres, 6...Nc6 7.g5 Nd7 8.Be3 Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.g4 Nc6 7.g5 Nd7 8.Be3 Be7 *"),
("B82", "Sicilian: Scheveningen, 6.f4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.f4 *"),
("B82", "Sicilian: Scheveningen, 6.f4 Qb6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.f4 Qb6 *"),
("B82", "Sicilian: Scheveningen, 6.f4 Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.f4 Be7 *"),
("B82", "Sicilian: Scheveningen, 6.f4 a6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.f4 a6 *"),
("B82", "Sicilian: Scheveningen, 6.f4 a6 7.Be3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.f4 a6 7.Be3 *"),
("B82", "Sicilian: Scheveningen, 6.f4 a6 7.Be3 b5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.f4 a6 7.Be3 b5 *"),
("B82", "Sicilian: Scheveningen, 6.f4 Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.f4 Nc6 *"),
("B82", "Sicilian: Scheveningen, 6.f4 Nc6 7.Be3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.f4 Nc6 7.Be3 *"),
("B82", "Sicilian: Scheveningen, 6.f4 Nc6 7.Be3 Qc7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.f4 Nc6 7.Be3 Qc7 *"),
("B82", "Sicilian: Scheveningen, 6.f4 Nc6 7.Be3 Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.f4 Nc6 7.Be3 Be7 *"),
("B82", "Sicilian: Scheveningen, Tal Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.f4 Nc6 7.Be3 Be7 8.Qf3 *"),
("B82", "Sicilian: Scheveningen, Tal, 8...e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.f4 Nc6 7.Be3 Be7 8.Qf3 e5 *"),
("B82", "Sicilian: Scheveningen, Tal, 8...Qc7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.f4 Nc6 7.Be3 Be7 8.Qf3 Qc7 *"),
("B83", "Sicilian: Scheveningen, 6.Be2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 *"),
("B83", "Sicilian: Scheveningen, 6.Be2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 Be7 *"),
("B83", "Sicilian: Scheveningen, Modern", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 Nc6 *"),
("B83", "Sicilian: Scheveningen, Modern", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 Nc6 7.O-O Be7 *"),
("B83", "Sicilian: Scheveningen, Modern", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 Nc6 7.O-O Be7 8.Be3 O-O *"),
("B83", "Sicilian: Scheveningen, Modern, 9.f4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 Nc6 7.O-O Be7 8.Be3 O-O 9.f4 *"),
("B83", "Sicilian: Scheveningen, Modern, 9.f4 e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 Nc6 7.O-O Be7 8.Be3 O-O 9.f4 e5 *"),
("B83", "Sicilian: Scheveningen, Modern, 9.f4 e5 10.Nb3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 Nc6 7.O-O Be7 8.Be3 O-O 9.f4 e5 10.Nb3 *"),
("B83", "Sicilian: Scheveningen, Modern, 9.f4 e5 10.Nb3 exf4 11.Bxf4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 Nc6 7.O-O Be7 8.Be3 O-O 9.f4 e5 10.Nb3 exf4 11.Bxf4 *"),
("B83", "Sicilian: Scheveningen, Modern, 9.f4 Bd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 Nc6 7.O-O Be7 8.Be3 O-O 9.f4 Bd7 *"),
("B83", "Sicilian: Scheveningen, Modern, 9.f4 Bd7 10.Nb3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 Nc6 7.O-O Be7 8.Be3 O-O 9.f4 Bd7 10.Nb3 *"),
("B83", "Sicilian: Scheveningen, Modern, 9.f4 Bd7 10.Nb3 a6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 Nc6 7.O-O Be7 8.Be3 O-O 9.f4 Bd7 10.Nb3 a6 *"),
("B83", "Sicilian: Scheveningen, Modern, 9.f4 Bd7 10.Nb3 a6 11.a4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 Nc6 7.O-O Be7 8.Be3 O-O 9.f4 Bd7 10.Nb3 a6 11.a4 *"),
("B83", "Sicilian: Scheveningen, Modern, 9.f4 Bd7 10.Kh1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 Nc6 7.O-O Be7 8.Be3 O-O 9.f4 Bd7 10.Kh1 *"),
("B83", "Sicilian: Scheveningen, Modern, 9.f4 Bd7 10.Qe1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 Nc6 7.O-O Be7 8.Be3 O-O 9.f4 Bd7 10.Qe1 *"),
("B84", "Sicilian: Scheveningen, Classical", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 *"),
("B84", "Sicilian: Scheveningen, Classical, 7.a4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.a4 *"),
("B84", "Sicilian: Scheveningen, Classical, 7.f4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.f4 *"),
("B84", "Sicilian: Scheveningen, Classical, 7.Be3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.Be3 *"),
("B84", "Sicilian: Scheveningen, Classical, 7.O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O *"),
("B84", "Sicilian: Scheveningen, Classical, 7.O-O Nbd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Nbd7 *"),
("B84", "Sicilian: Scheveningen, Classical, 7.O-O Nbd7 8.a4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Nbd7 8.a4 *"),
("B84", "Sicilian: Scheveningen, Classical, 7.O-O Nbd7 8.f4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Nbd7 8.f4 *"),
("B84", "Sicilian: Scheveningen, Classical, 7.O-O Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Nc6 *"),
("B84", "Sicilian: Scheveningen, Classical, 7.O-O Qc7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Qc7 *"),
("B84", "Sicilian: Scheveningen, Classical, 7.O-O Qc7 8.Kh1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Qc7 8.Kh1 *"),
("B84", "Sicilian: Scheveningen, Classical, 7.O-O Qc7 8.Kh1 Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Qc7 8.Kh1 Nc6 *"),
("B84", "Sicilian: Scheveningen, Classical, 7.O-O Qc7 8.a4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Qc7 8.a4 *"),
("B84", "Sicilian: Scheveningen, Classical, 7.O-O Qc7 8.a4 Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Qc7 8.a4 Nc6 *"),
("B84", "Sicilian: Scheveningen, Classical, 7.O-O Qc7 8.Bf3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Qc7 8.Bf3 *"),
("B84", "Sicilian: Scheveningen, Classical, 7.O-O Qc7 8.Be3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Qc7 8.Be3 *"),
("B84", "Sicilian: Scheveningen, Classical, 7.O-O Qc7 8.Be3 Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Qc7 8.Be3 Nc6 *"),
("B84", "Sicilian: Scheveningen, Classical, 7.O-O Qc7 8.f4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Qc7 8.f4 *"),
("B84", "Sicilian: Scheveningen, Classical, 7.O-O Qc7 8.f4 Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Qc7 8.f4 Nc6 *"),
("B84", "Sicilian: Scheveningen, Classical, 7.O-O Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Be7 *"),
("B84", "Sicilian: Scheveningen, Classical, 7.O-O Be7 8.f4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Be7 8.f4 *"),
("B84", "Sicilian: Scheveningen, Classical, 7.O-O Be7 8.f4 Qc7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Be7 8.f4 Qc7 *"),
("B84", "Sicilian: Scheveningen, Classical, 7.O-O Be7 8.f4 Qc7 9.Be3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Be7 8.f4 Qc7 9.Be3 *"),
("B84", "Sicilian: Scheveningen, Classical, 7.O-O Be7 8.f4 Qc7 9.Be3 Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Be7 8.f4 Qc7 9.Be3 Nc6 *"),
("B84", "Sicilian: Scheveningen, Classical, 7.O-O Be7 8.f4 Qc7 9.Kh1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Be7 8.f4 Qc7 9.Kh1 *"),
("B84", "Sicilian: Scheveningen, Classical, 7.O-O Be7 8.f4 O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Be7 8.f4 O-O *"),
("B85", "Sicilian: Scheveningen, Classical, Main Line", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Be7 8.f4 O-O 9.Be3 *"),
("B85", "Sicilian: Scheveningen, Classical, Main Line, 9...Qc7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Be7 8.f4 O-O 9.Be3 Qc7 *"),
("B85", "Sicilian: Scheveningen, Classical, Main Line, 9...Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Be7 8.f4 O-O 9.Be3 Nc6 *"),
("B85", "Sicilian: Scheveningen, Classical, Main Line, 9...Nc6 10.Qe1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Be7 8.f4 O-O 9.Be3 Nc6 10.Qe1 *"),
("B85", "Sicilian: Scheveningen, Classical, Main Line, 9...Nc6 10.Qe1 Qc7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Be7 8.f4 O-O 9.Be3 Nc6 10.Qe1 Qc7 *"),
("B85", "Sicilian: Scheveningen, Classical, Main Line, 9...Nc6 10.Qe1 Qc7 11.Qg3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Be7 8.f4 O-O 9.Be3 Nc6 10.Qe1 Qc7 11.Qg3 *"),
("B85", "Sicilian: Scheveningen, Classical, Main Line, 9...Nc6 10.Kh1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Be7 8.f4 O-O 9.Be3 Nc6 10.Kh1 *"),
("B85", "Sicilian: Scheveningen, Classical, Main Line, 9...Nc6 10.Kh1 Bd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Be7 8.f4 O-O 9.Be3 Nc6 10.Kh1 Bd7 *"),
("B85", "Sicilian: Scheveningen, Classical, Main Line, 9...Nc6 10.Kh1 Qc7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Be7 8.f4 O-O 9.Be3 Nc6 10.Kh1 Qc7 *"),
("B85", "Sicilian: Scheveningen, Classical, Main Line, 9...Nc6 10.a4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Be7 8.f4 O-O 9.Be3 Nc6 10.a4 *"),
("B85", "Sicilian: Scheveningen, Classical, Main Line, 9...Nc6 10.a4 Bd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Be7 8.f4 O-O 9.Be3 Nc6 10.a4 Bd7 *"),
("B85", "Sicilian: Scheveningen, Classical, Main Line, 9...Nc6 10.a4 Qc7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Be7 8.f4 O-O 9.Be3 Nc6 10.a4 Qc7 *"),
("B85", "Sicilian: Scheveningen, Classical, Main Line, 9...Nc6 10.a4 Qc7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Be2 a6 7.O-O Be7 8.f4 O-O 9.Be3 Nc6 10.a4 Qc7 11.Kh1 *"),
("B86", "Sicilian: Sozin-Scheveningen", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bc4 *"),
("B86", "Sicilian: Sozin-Scheveningen, 6...Qb6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bc4 Qb6 *"),
("B86", "Sicilian: Sozin-Najdorf", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bc4 a6 *"),
("B86", "Sicilian: Sozin-Najdorf, 7.Bb3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bc4 a6 7.Bb3 *"),
("B86", "Sicilian: Sozin-Najdorf, 7.Bb3 Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bc4 a6 7.Bb3 Be7 *"),
("B87", "Sicilian: Sozin-Najdorf, 7.Bb3 b5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bc4 a6 7.Bb3 b5 *"),
("B87", "Sicilian: Sozin-Najdorf, 7.Bb3 b5 8.O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bc4 a6 7.Bb3 b5 8.O-O *"),
("B87", "Sicilian: Sozin-Najdorf, 7.Bb3 b5 8.O-O Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bc4 a6 7.Bb3 b5 8.O-O Be7 *"),
("B87", "Sicilian: Sozin-Najdorf, 7.Bb3 b5 8.O-O Be7 9.Qf3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bc4 a6 7.Bb3 b5 8.O-O Be7 9.Qf3 *"),
("B87", "Sicilian: Sozin-Najdorf, 7.Bb3 b5 8.O-O Be7 9.f4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bc4 a6 7.Bb3 b5 8.O-O Be7 9.f4 *"),
("B88", "Sicilian: Sozin-Scheveningen", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bc4 Nc6 *"),
("B88", "Sicilian: Sozin-Scheveningen, 7.O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bc4 Nc6 7.O-O *"),
("B88", "Sicilian: Sozin-Scheveningen, 7.Bb3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bc4 Nc6 7.Bb3 *"),
("B88", "Sicilian: Sozin-Scheveningen, 7.Bb3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bc4 Nc6 7.Bb3 Be7 8.Be3 *"),
("B88", "Sicilian: Sozin, Fischer Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bc4 Nc6 7.Bb3 Be7 8.Be3 O-O 9.f4 *"),
("B89", "Sicilian: Sozin, 7.Be3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bc4 Nc6 7.Be3 *"),
("B89", "Sicilian: Sozin, 7.Be3 a6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bc4 Nc6 7.Be3 a6 *"),
("B89", "Sicilian: Velimirovic vs. 7...a6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bc4 Nc6 7.Be3 a6 8.Qe2 *"),
("B89", "Sicilian: Sozin, 7.Be3 Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bc4 Nc6 7.Be3 Be7 *"),
("B89", "Sicilian: Velimirovic", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bc4 Nc6 7.Be3 Be7 8.Qe2 *"),
("B89", "Sicilian: Velimirovic, 9.O-O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bc4 Nc6 7.Be3 Be7 8.Qe2 O-O 9.O-O-O *"),
("B89", "Sicilian: Velimirovic, 9.O-O-O Qc7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bc4 Nc6 7.Be3 Be7 8.Qe2 O-O 9.O-O-O Qc7 *"),
("B89", "Sicilian: Velimirovic, 9.O-O-O Qc7 10.Bb3 a6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 e6 6.Bc4 Nc6 7.Be3 Be7 8.Qe2 O-O 9.O-O-O Qc7 10.Bb3 a6 *"),
("B90a", "Sicilian: Najdorf", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 *"),
("B90b", "Sicilian: Najdorf, 6.a4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.a4 *"),
("B90c", "Sicilian: Najdorf, 6.a4 e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.a4 e5 *"),
("B90d", "Sicilian: Najdorf, 6.f3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.f3 *"),
("B90e", "Sicilian: Najdorf, Adams Attack", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.h3 *"),
("B90f", "Sicilian: Najdorf, Fischer-Sozin Attack", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bc4 *"),
("B90g", "Sicilian: Najdorf, 6.Be3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be3 *"),
("B90h", "Sicilian: Najdorf, 6.Be3 Ng4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be3 Ng4 *"),
("B90h", "Sicilian: Najdorf, 6.Be3 Ng4 7.Bg5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be3 Ng4 7.Bg5 *"),
("B90i", "Sicilian: Najdorf, 6.Be3 Ng4 7.Bg5 h6 8.Bh4 g5 9.Bg3 Bg7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be3 Ng4 7.Bg5 h6 8.Bh4 g5 9.Bg3 Bg7 *"),
("B90j", "Sicilian: Najdorf, 6.Be3 Ng4 7.Bg5 h6 8.Bh4 g5 9.Bg3 Bg7 10.Be2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be3 Ng4 7.Bg5 h6 8.Bh4 g5 9.Bg3 Bg7 10.Be2 *"),
("B90j", "Sicilian: Najdorf, 6.Be3 Ng4 7.Bg5 h6 8.Bh4 g5 9.Bg3 Bg7 10.Be2 h5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be3 Ng4 7.Bg5 h6 8.Bh4 g5 9.Bg3 Bg7 10.Be2 h5 *"),
("B90k", "Sicilian: Najdorf, 6.Be3 e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be3 e5 *"),
("B90l", "Sicilian: Najdorf, 6.Be3 e5 7.Nb3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be3 e5 7.Nb3 *"),
("B90m", "Sicilian: Najdorf, 6.Be3 e5 7.Nb3 Be6 8.Qd2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be3 e5 7.Nb3 Be6 8.Qd2 *"),
("B90n", "Sicilian: Najdorf, 6.Be3 e5 7.Nb3 Be6 8.f3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be3 e5 7.Nb3 Be6 8.f3 *"),
("B90o", "Sicilian: Najdorf, 6.Be3 e5 7.Nb3 Be6 8.f3 Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be3 e5 7.Nb3 Be6 8.f3 Be7 *"),
("B90o", "Sicilian: Najdorf, 6.Be3 e5 7.Nb3 Be6 8.f3 Be7 9.Qd2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be3 e5 7.Nb3 Be6 8.f3 Be7 9.Qd2 *"),
("B90p", "Sicilian: Najdorf, 6.Be3 e5 7.Nb3 Be6 8.f3 Be7 9.Qd2 Nbd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be3 e5 7.Nb3 Be6 8.f3 Be7 9.Qd2 Nbd7 *"),
("B90q", "Sicilian: Najdorf, 6.Be3 e5 7.Nb3 Be6 8.f3 Be7 9.Qd2 O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be3 e5 7.Nb3 Be6 8.f3 Be7 9.Qd2 O-O *"),
("B90q", "Sicilian: Najdorf, 6.Be3 e5 7.Nb3 Be6 8.f3 Be7 9.Qd2 O-O 10.O-O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be3 e5 7.Nb3 Be6 8.f3 Be7 9.Qd2 O-O 10.O-O-O *"),
("B90r", "Sicilian: Najdorf, 6.Be3 e5 7.Nb3 Be6 8.f3 Be7 9.Qd2 O-O 10.O-O-O Nbd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be3 e5 7.Nb3 Be6 8.f3 Be7 9.Qd2 O-O 10.O-O-O Nbd7 *"),
("B90s", "Sicilian: Najdorf, 6.Be3 e5 7.Nb3 Be6 8.f3 Nbd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be3 e5 7.Nb3 Be6 8.f3 Nbd7 *"),
("B90t", "Sicilian: Najdorf, 6.Be3 e5 7.Nb3 Be6 8.f3 Be7 9.Qd2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be3 e5 7.Nb3 Be6 8.f3 Nbd7 9.Qd2 *"),
("B90u", "Sicilian: Najdorf, 6.Be3 e5 7.Nb3 Be6 8.f3 Be7 9.Qd2 b5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be3 e5 7.Nb3 Be6 8.f3 Nbd7 9.Qd2 b5 *"),
("B91a", "Sicilian: Najdorf, 6.g3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.g3 *"),
("B91b", "Sicilian: Najdorf, 6.g3 b5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.g3 b5 *"),
("B91c", "Sicilian: Najdorf, 6.g3 g6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.g3 g6 *"),
("B91d", "Sicilian: Najdorf, 6.g3 Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.g3 Nc6 *"),
("B91e", "Sicilian: Najdorf, 6.g3 Bg4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.g3 Bg4 *"),
("B91f", "Sicilian: Najdorf, 6.g3 e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.g3 e5 *"),
("B91g", "Sicilian: Najdorf, 6.g3 e5 7.Nb3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.g3 e5 7.Nb3 *"),
("B91h", "Sicilian: Najdorf, 6.g3 e5 7.Nb3 Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.g3 e5 7.Nb3 Be7 *"),
("B91i", "Sicilian: Najdorf, 6.g3 e5 7.Nde2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.g3 e5 7.Nde2 *"),
("B91j", "Sicilian: Najdorf, 6.g3 e5 7.Nde2 b5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.g3 e5 7.Nde2 b5 *"),
("B91k", "Sicilian: Najdorf, 6.g3 e5 7.Nde2 Nbd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.g3 e5 7.Nde2 Nbd7 *"),
("B91l", "Sicilian: Najdorf, 6.g3 e5 7.Nde2 Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.g3 e5 7.Nde2 Be7 *"),
("B91m", "Sicilian: Najdorf, 6.g3 e5 7.Nde2 Be7 8.Bg2 Nbd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.g3 e5 7.Nde2 Be7 8.Bg2 Nbd7 *"),
("B91n", "Sicilian: Najdorf, 6.g3 e5 7.Nde2 Be7 8.Bg2 O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.g3 e5 7.Nde2 Be7 8.Bg2 O-O *"),
("B92a", "Sicilian: Najdorf, 6.Be2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be2 *"),
("B92b", "Sicilian: Najdorf, 6.Be2 Nbd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be2 Nbd7 *"),
("B92c", "Sicilian: Najdorf, 6.Be2 e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be2 e5 *"),
("B92d", "Sicilian: Najdorf, 6.Be2 e5 7.Nf3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be2 e5 7.Nf3 *"),
("B92e", "Sicilian: Najdorf, 6.Be2 e5 7.Nb3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be2 e5 7.Nb3 *"),
("B92f", "Sicilian: Najdorf, 6.Be2 e5 7.Nb3 Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be2 e5 7.Nb3 Be7 *"),
("B92g", "Sicilian: Najdorf, 6.Be2 e5 7.Nb3 Be7 8.Be3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be2 e5 7.Nb3 Be7 8.Be3 *"),
("B92g", "Sicilian: Najdorf, 6.Be2 e5 7.Nb3 Be7 8.Be3 Be6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be2 e5 7.Nb3 Be7 8.Be3 Be6 *"),
("B92h", "Sicilian: Najdorf, 6.Be2 e5 7.Nb3 Be7 8.O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be2 e5 7.Nb3 Be7 8.O-O *"),
("B92h", "Sicilian: Najdorf, 6.Be2 e5 7.Nb3 Be7 8.O-O Be6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be2 e5 7.Nb3 Be7 8.O-O Be6 *"),
("B92i", "Sicilian: Najdorf, 6.Be2 e5 7.Nb3 Be7 8.O-O Be6 9.f4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be2 e5 7.Nb3 Be7 8.O-O Be6 9.f4 *"),
("B92j", "Sicilian: Najdorf, 6.Be2 e5 7.Nb3 Be7 8.O-O O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be2 e5 7.Nb3 Be7 8.O-O O-O *"),
("B92k", "Sicilian: Najdorf, 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Be3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Be3 *"),
("B92k", "Sicilian: Najdorf, 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Be3 Be6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Be3 Be6 *"),
("B92l", "Sicilian: Najdorf, 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Be3 Be6 10.a4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Be3 Be6 10.a4 *"),
("B92m", "Sicilian: Najdorf, 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Be3 Be6 10.Nd5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Be3 Be6 10.Nd5 *"),
("B92n", "Sicilian: Najdorf, 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Be3 Be6 10.Qd2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Be3 Be6 10.Qd2 *"),
("B92o", "Sicilian: Najdorf, 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Kh1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Kh1 *"),
("B92p", "Sicilian: Najdorf, 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Kh1 Be6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Kh1 Be6 *"),
("B92q", "Sicilian: Najdorf, 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Kh1 Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Kh1 Nc6 *"),
("B92r", "Sicilian: Najdorf, 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Kh1 Qc7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Be2 e5 7.Nb3 Be7 8.O-O O-O 9.Kh1 Qc7 *"),
("B93a", "Sicilian: Najdorf, 6.f4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.f4 *"),
("B93b", "Sicilian: Najdorf, 6.f4 Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.f4 Nc6 *"),
("B93c", "Sicilian: Najdorf, 6.f4 Nbd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.f4 Nbd7 *"),
("B93d", "Sicilian: Najdorf, 6.f4 Nbd7 7.Be2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.f4 Nbd7 7.Be2 *"),
("B93e", "Sicilian: Najdorf, 6.f4 Qc7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.f4 Qc7 *"),
("B93f", "Sicilian: Najdorf, 6.f4 Qc7 7.Bd3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.f4 Qc7 7.Bd3 *"),
("B93g", "Sicilian: Najdorf, 6.f4 Qc7 7.Bd3 g6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.f4 Qc7 7.Bd3 g6 *"),
("B93h", "Sicilian: Najdorf, 6.f4 Qc7 7.Nf3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.f4 Qc7 7.Nf3 *"),
("B93i", "Sicilian: Najdorf, 6.f4 Qc7 7.Nf3 Nbd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.f4 Qc7 7.Nf3 Nbd7 *"),
("B93j", "Sicilian: Najdorf, 6.f4 Qc7 7.Nf3 Nbd7 8.Bd3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.f4 Qc7 7.Nf3 Nbd7 8.Bd3 *"),
("B93k", "Sicilian: Najdorf, 6.f4 e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.f4 e5 *"),
("B93k", "Sicilian: Najdorf, 6.f4 e5 7.Nf3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.f4 e5 7.Nf3 *"),
("B93l", "Sicilian: Najdorf, 6.f4 e5 7.Nf3 Qc7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.f4 e5 7.Nf3 Qc7 *"),
("B93m", "Sicilian: Najdorf, 6.f4 e5 7.Nf3 Qc7 8.a4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.f4 e5 7.Nf3 Qc7 8.a4 *"),
("B93n", "Sicilian: Najdorf, 6.f4 e5 7.Nf3 Nbd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.f4 e5 7.Nf3 Nbd7 *"),
("B93o", "Sicilian: Najdorf, 6.f4 e5 7.Nf3 Nbd7 8.Bd3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.f4 e5 7.Nf3 Nbd7 8.Bd3 *"),
("B93p", "Sicilian: Najdorf, 6.f4 e5 7.Nf3 Nbd7 8.a4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.f4 e5 7.Nf3 Nbd7 8.a4 *"),
("B93q", "Sicilian: Najdorf, 6.f4 e5 7.Nf3 Nbd7 8.a4 Qc7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.f4 e5 7.Nf3 Nbd7 8.a4 Qc7 *"),
("B93r", "Sicilian: Najdorf, 6.f4 e5 7.Nf3 Nbd7 8.a4 Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.f4 e5 7.Nf3 Nbd7 8.a4 Be7 *"),
("B93r", "Sicilian: Najdorf, 6.f4 e5 7.Nf3 Nbd7 8.a4 Be7 9.Bc4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.f4 e5 7.Nf3 Nbd7 8.a4 Be7 9.Bc4 *"),
("B93s", "Sicilian: Najdorf, 6.f4 e5 7.Nf3 Nbd7 8.a4 Be7 9.Bd3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.f4 e5 7.Nf3 Nbd7 8.a4 Be7 9.Bd3 *"),
("B93t", "Sicilian: Najdorf, 6.f4 e5 7.Nf3 Nbd7 8.a4 Be7 9.Bd3 O-O 10.O-O Nc5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.f4 e5 7.Nf3 Nbd7 8.a4 Be7 9.Bd3 O-O 10.O-O Nc5 *"),
("B93u", "Sicilian: Najdorf, 6.f4 e5 7.Nf3 Nbd7 8.a4 Be7 9.Bd3 O-O 10.O-O exf4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.f4 e5 7.Nf3 Nbd7 8.a4 Be7 9.Bd3 O-O 10.O-O exf4 *"),
("B94a", "Sicilian: Najdorf, 6.Bg5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 *"),
// After 6.Bg5, 6...Nc6 is not B94, but a Richter-Rauzer (B60).
("B94b", "Sicilian: Najdorf, 6.Bg5 b5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 b5 *"),
("B94c", "Sicilian: Najdorf, 6.Bg5 e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e5 *"),
("B94d", "Sicilian: Najdorf, 6.Bg5 Nbd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 Nbd7 *"),
("B94e", "Sicilian: Najdorf, 6.Bg5 Nbd7 7.f4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 Nbd7 7.f4 *"),
("B94f", "Sicilian: Najdorf, 6.Bg5 Nbd7 7.f4 b5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 Nbd7 7.f4 b5 *"),
("B94g", "Sicilian: Najdorf, 6.Bg5 Nbd7 7.Bc4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 Nbd7 7.Bc4 *"),
("B94h", "Sicilian: Najdorf, 6.Bg5 Nbd7 7.Bc4 e6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 Nbd7 7.Bc4 e6 *"),
("B94i", "Sicilian: Najdorf, 6.Bg5 Nbd7 7.Bc4 e6 8.O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 Nbd7 7.Bc4 e6 8.O-O *"),
("B94j", "Sicilian: Najdorf, 6.Bg5 Nbd7 7.Bc4 Qa5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 Nbd7 7.Bc4 Qa5 *"),
("B94k", "Sicilian: Najdorf, 6.Bg5 Nbd7 7.Bc4 Qa5 8.Qd2 e6 9.O-O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 Nbd7 7.Bc4 Qa5 8.Qd2 e6 9.O-O-O *"),
("B95a", "Sicilian: Najdorf, 6...e6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 *"),
("B95b", "Sicilian: Najdorf, 6...e6 7.Bd3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.Bd3 *"),
("B95c", "Sicilian: Najdorf, 6...e6 7.Be2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.Be2 *"),
("B95d", "Sicilian: Najdorf, 6...e6 7.Qe2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.Qe2 *"),
("B95e", "Sicilian: Najdorf, 6...e6 7.Qd2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.Qd2 *"),
("B95f", "Sicilian: Najdorf, 6...e6 7.Qd2 Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.Qd2 Be7 *"),
("B95g", "Sicilian: Najdorf, 6...e6 7.Qd2 Be7 8.O-O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.Qd2 Be7 8.O-O-O *"),
("B95h", "Sicilian: Najdorf, 6...e6 7.Qd3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.Qd3 *"),
("B95i", "Sicilian: Najdorf, 6...e6 7.Qd3 Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.Qd3 Nc6 *"),
("B95j", "Sicilian: Najdorf, 6...e6 7.Qf3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.Qf3 *"),
("B95k", "Sicilian: Najdorf, 6...e6 7.Qf3 Nbd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.Qf3 Nbd7 *"),
("B95l", "Sicilian: Najdorf, 6...e6 7.Qf3 Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.Qf3 Be7 *"),
("B95m", "Sicilian: Najdorf, 6...e6 7.Qf3 Be7 8.O-O-O Nbd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.Qf3 Be7 8.O-O-O Nbd7 *"),
("B95n", "Sicilian: Najdorf, 6...e6 7.Qf3 h6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.Qf3 h6 *"),
("B96a", "Sicilian: Najdorf, 7.f4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 *"),
("B96b", "Sicilian: Najdorf, 7.f4 h6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 h6 *"),
("B96b", "Sicilian: Najdorf, Delayed Poisoned Pawn", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 h6 8.Bh4 Qb6 *"),
("B96c", "Sicilian: Najdorf, 7.f4 Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Nc6 *"),
("B96c", "Sicilian: Najdorf, 7.f4 Nc6 8.e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Nc6 8.e5 *"),
("B96d", "Sicilian: Najdorf, 7.f4 Nc6 8.Nxc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Nc6 8.Nxc6 *"),
("B96e", "Sicilian: Najdorf, 7.f4 Bd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Bd7 *"),
("B96f", "Sicilian: Najdorf, 7.f4 Qc7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qc7 *"),
("B96f", "Sicilian: Najdorf, 7.f4 Qc7 8.Bxf6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qc7 8.Bxf6 *"),
("B96g", "Sicilian: Najdorf, 7.f4 Qc7 8.Qf3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qc7 8.Qf3 *"),
("B96h", "Sicilian: Najdorf, 7.f4 Qc7 8.Qf3 b5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qc7 8.Qf3 b5 *"),
("B96h", "Sicilian: Najdorf, 7.f4 Qc7 8.Qf3 b5 9.f5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qc7 8.Qf3 b5 9.f5 *"),
("B96h", "Sicilian: Najdorf, 7.f4 Qc7 8.Qf3 b5 9.O-O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qc7 8.Qf3 b5 9.O-O-O *"),
("B96h", "Sicilian: Najdorf, 7.f4 Qc7 8.Qf3 b5 9.Bxf6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qc7 8.Qf3 b5 9.Bxf6 *"),
("B96i", "Sicilian: Najdorf, 7.f4 Nbd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Nbd7 *"),
("B96i", "Sicilian: Najdorf, 7.f4 Nbd7 8.Bc4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Nbd7 8.Bc4 *"),
("B96i", "Sicilian: Najdorf, 7.f4 Nbd7 8.Qe2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Nbd7 8.Qe2 *"),
("B96i", "Sicilian: Najdorf, 7.f4 Nbd7 8.Qe2 Qc7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Nbd7 8.Qe2 Qc7 *"),
("B96j", "Sicilian: Najdorf, 7.f4 Nbd7 8.Qf3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Nbd7 8.Qf3 *"),
("B96j", "Sicilian: Najdorf, 7.f4 Nbd7 8.Qf3 Qc7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Nbd7 8.Qf3 Qc7 *"),
("B96k", "Sicilian: Najdorf, 7.f4 Nbd7 8.Qf3 Qc7 9.O-O-O b5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Nbd7 8.Qf3 Qc7 9.O-O-O b5 *"),
("B96l", "Sicilian: Najdorf, 7.f4 Nbd7 8.Qf3 Qc7 9.O-O-O b5 10.Bd3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Nbd7 8.Qf3 Qc7 9.O-O-O b5 10.Bd3 *"),
("B96m", "Sicilian: Najdorf, 7.f4 Nbd7 8.Qf3 Qc7 9.O-O-O b5 10.e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Nbd7 8.Qf3 Qc7 9.O-O-O b5 10.e5 *"),
("B96n", "Sicilian: Najdorf, Polugaevsky Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 b5 *"),
("B96n", "Sicilian: Najdorf, Polugaevsky, 8.e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 b5 8.e5 *"),
("B96n", "Sicilian: Najdorf, Polugaevsky, 9.fxe5 Qc7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 b5 8.e5 dxe5 9.fxe5 Qc7 *"),
("B96o", "Sicilian: Najdorf, Polugaevsky, 10.Nf3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 b5 8.e5 dxe5 9.fxe5 Qc7 10.Nf3 *"),
("B96p", "Sicilian: Najdorf, Polugaevsky, 10.Bxb5+", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 b5 8.e5 dxe5 9.fxe5 Qc7 10.Bxb5+ *"),
("B96q", "Sicilian: Najdorf, Polugaevsky, 10.exf6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 b5 8.e5 dxe5 9.fxe5 Qc7 10.exf6 *"),
("B96r", "Sicilian: Najdorf, Polugaevsky, 10.exf6 Qe5+ 11.Be2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 b5 8.e5 dxe5 9.fxe5 Qc7 10.exf6 Qe5+ 11.Be2 *"),
("B96s", "Sicilian: Najdorf, Polugaevsky, 10.exf6 Qe5+ 11.Be2 Qxg5 12.Qd3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 b5 8.e5 dxe5 9.fxe5 Qc7 10.exf6 Qe5+ 11.Be2 Qxg5 12.Qd3 *"),
("B96t", "Sicilian: Najdorf, Polugaevsky, 10.Qe2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 b5 8.e5 dxe5 9.fxe5 Qc7 10.Qe2 *"),
("B96u", "Sicilian: Najdorf, Polugaevsky, 10.Qe2 Nfd7 11.O-O-O Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 b5 8.e5 dxe5 9.fxe5 Qc7 10.Qe2 Nfd7 11.O-O-O Nc6 *"),
("B96v", "Sicilian: Najdorf, Polugaevsky, 10.Qe2 Nfd7 11.O-O-O Bb7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 b5 8.e5 dxe5 9.fxe5 Qc7 10.Qe2 Nfd7 11.O-O-O Bb7 *"),
("B96w", "Sicilian: Najdorf, Polugaevsky, 10.Qe2 Nfd7 11.O-O-O Bb7 12.Qg4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 b5 8.e5 dxe5 9.fxe5 Qc7 10.Qe2 Nfd7 11.O-O-O Bb7 12.Qg4 *"),
("B96w", "Sicilian: Najdorf, Polugaevsky, 10.Qe2 Nfd7 11.O-O-O Bb7 12.Qg4 Qxe5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 b5 8.e5 dxe5 9.fxe5 Qc7 10.Qe2 Nfd7 11.O-O-O Bb7 12.Qg4 Qxe5 *"),
("B97a", "Sicilian: Najdorf, Poisoned Pawn", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 *"),
("B97b", "Sicilian: Najdorf, Poisoned Pawn, 8.Bxf6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Bxf6 *"),
("B97c", "Sicilian: Najdorf, Poisoned Pawn, 8.a3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.a3 *"),
("B97d", "Sicilian: Najdorf, Poisoned Pawn, 8.Qd3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd3 *"),
("B97e", "Sicilian: Najdorf, Poisoned Pawn, 8.Nb3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Nb3 *"),
("B97f", "Sicilian: Najdorf, Poisoned Pawn, 8.Nb3 Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Nb3 Be7 *"),
("B97g", "Sicilian: Najdorf, Poisoned Pawn, 8.Nb3 Nbd7 9.Qf3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Nb3 Nbd7 9.Qf3 *"),
("B97h", "Sicilian: Najdorf, Poisoned Pawn, 8.Nb3 Nbd7 9.Qf3 Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Nb3 Nbd7 9.Qf3 Be7 *"),
("B97i", "Sicilian: Najdorf, Poisoned Pawn, 8.Qd2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 *"),
("B97i", "Sicilian: Najdorf, Poisoned Pawn Accepted", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 *"),
("B97j", "Sicilian: Najdorf, Poisoned Pawn, 9.Nb3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Nb3 *"),
("B97j", "Sicilian: Najdorf, Poisoned Pawn, 9.Nb3 Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Nb3 Nc6 *"),
("B97k", "Sicilian: Najdorf, Poisoned Pawn, 9.Nb3 Qa3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Nb3 Qa3 *"),
("B97l", "Sicilian: Najdorf, Poisoned Pawn, 9.Nb3 Qa3 10.Bxf6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Nb3 Qa3 10.Bxf6 *"),
("B97m", "Sicilian: Najdorf, Poisoned Pawn, 9.Rb1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Rb1 *"),
("B97m", "Sicilian: Najdorf, Poisoned Pawn, 9.Rb1 Qa3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Rb1 Qa3 *"),
("B97n", "Sicilian: Najdorf, Poisoned Pawn, 10.Be2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Rb1 Qa3 10.Be2 *"),
("B97n", "Sicilian: Najdorf, Poisoned Pawn, 10.Be2 Be7 11.O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Rb1 Qa3 10.Be2 Be7 11.O-O *"),
("B97n", "Sicilian: Najdorf, Poisoned Pawn, 10.Be2 Be7 11.O-O Nbd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Rb1 Qa3 10.Be2 Be7 11.O-O Nbd7 *"),
("B97o", "Sicilian: Najdorf, Poisoned Pawn, 10.Bxf6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Rb1 Qa3 10.Bxf6 *"),
("B97o", "Sicilian: Najdorf, Poisoned Pawn, 10.Bxf6 gxf6 11.Be2 Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Rb1 Qa3 10.Bxf6 gxf6 11.Be2 Nc6 *"),
("B97o", "Sicilian: Najdorf, Poisoned Pawn, 10.Bxf6 gxf6 11.Be2 Bg7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Rb1 Qa3 10.Bxf6 gxf6 11.Be2 Bg7 *"),
("B97p", "Sicilian: Najdorf, Poisoned Pawn, 10.e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Rb1 Qa3 10.e5 *"),
("B97p", "Sicilian: Najdorf, Poisoned Pawn, 10.e5 dxe5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Rb1 Qa3 10.e5 dxe5 *"),
("B97p", "Sicilian: Najdorf, Poisoned Pawn, 10.e5 dxe5 11.fxe5 Nfd7 12.Bc4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Rb1 Qa3 10.e5 dxe5 11.fxe5 Nfd7 12.Bc4 *"),
("B97p", "Sicilian: Najdorf, Poisoned Pawn, 10.e5 dxe5 11.fxe5 Nfd7 12.Bc4 Bb4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Rb1 Qa3 10.e5 dxe5 11.fxe5 Nfd7 12.Bc4 Bb4 *"),
("B97q", "Sicilian: Najdorf, Poisoned Pawn, 10.f5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Rb1 Qa3 10.f5 *"),
("B97q", "Sicilian: Najdorf, Poisoned Pawn, 10.f5 Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Rb1 Qa3 10.f5 Nc6 *"),
("B97r", "Sicilian: Najdorf, Poisoned Pawn, Main Line", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Rb1 Qa3 10.f5 Nc6 11.fxe6 fxe6 12.Nxc6 bxc6 *"),
("B97s", "Sicilian: Najdorf, Poisoned Pawn, Main Line, Timman's 13.Be2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Rb1 Qa3 10.f5 Nc6 11.fxe6 fxe6 12.Nxc6 bxc6 13.Be2 *"),
("B97t", "Sicilian: Najdorf, Poisoned Pawn, Main Line, 13.e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Rb1 Qa3 10.f5 Nc6 11.fxe6 fxe6 12.Nxc6 bxc6 13.e5 *"),
("B97t", "Sicilian: Najdorf, Poisoned Pawn, Main Line, 13.e5 Nd5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Rb1 Qa3 10.f5 Nc6 11.fxe6 fxe6 12.Nxc6 bxc6 13.e5 Nd5 *"),
("B97u", "Sicilian: Najdorf, Poisoned Pawn, Main Line, 13.e5 dxe5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Rb1 Qa3 10.f5 Nc6 11.fxe6 fxe6 12.Nxc6 bxc6 13.e5 dxe5 *"),
("B97u", "Sicilian: Najdorf, Poisoned Pawn, Main Line, 13.e5 dxe5: 15.Ne4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Rb1 Qa3 10.f5 Nc6 11.fxe6 fxe6 12.Nxc6 bxc6 13.e5 dxe5 14.Bxf6 gxf6 15.Ne4 *"),
("B97v", "Sicilian: Najdorf, Poisoned Pawn, Main Line, 13.e5 dxe5: 15.Ne4 Qxa2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Rb1 Qa3 10.f5 Nc6 11.fxe6 fxe6 12.Nxc6 bxc6 13.e5 dxe5 14.Bxf6 gxf6 15.Ne4 Qxa2 *"),
("B97w", "Sicilian: Najdorf, Poisoned Pawn, Main Line, 13.e5 dxe5: 15.Ne4 Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Rb1 Qa3 10.f5 Nc6 11.fxe6 fxe6 12.Nxc6 bxc6 13.e5 dxe5 14.Bxf6 gxf6 15.Ne4 Be7 *"),
("B97x", "Sicilian: Najdorf, Poisoned Pawn, Main Line, 13.e5 dxe5 with 18.c4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Rb1 Qa3 10.f5 Nc6 11.fxe6 fxe6 12.Nxc6 bxc6 13.e5 dxe5 14.Bxf6 gxf6 15.Ne4 Be7 16.Be2 h5 17.Rb3 Qa4 18.c4 *"),
("B97y", "Sicilian: Najdorf, Poisoned Pawn, Main Line, 13.e5 dxe5 with 18.Nxf6+", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Qb6 8.Qd2 Qxb2 9.Rb1 Qa3 10.f5 Nc6 11.fxe6 fxe6 12.Nxc6 bxc6 13.e5 dxe5 14.Bxf6 gxf6 15.Ne4 Be7 16.Be2 h5 17.Rb3 Qa4 18.Nxf6+ *"),
("B98a", "Sicilian: Najdorf, 7...Be7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 *"),
("B98b", "Sicilian: Najdorf, 7...Be7 8.Qf3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 *"),
("B98c", "Sicilian: Najdorf, Danner Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qa5 *"),
("B98d", "Sicilian: Najdorf, Danner, 9.O-O-O Bd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qa5 9.O-O-O Bd7 *"),
("B98e", "Sicilian: Najdorf, Danner, 9.O-O-O Bd7 10.e5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qa5 9.O-O-O Bd7 10.e5 *"),
("B98f", "Sicilian: Najdorf, 7...Be7 8.Qf3 h6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 h6 *"),
("B98g", "Sicilian: Najdorf, Browne Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 h6 9.Bh4 Qc7 *"),
("B98h", "Sicilian: Najdorf, Browne, 10.O-O-O Nbd7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 h6 9.Bh4 Qc7 10.O-O-O Nbd7 *"),
("B98i", "Sicilian: Najdorf, Browne, 10.O-O-O Nbd7 11.Qg3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 h6 9.Bh4 Qc7 10.O-O-O Nbd7 11.Qg3 *"),
("B98j", "Sicilian: Najdorf, Browne, 10.O-O-O Nbd7 11.Be2", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 h6 9.Bh4 Qc7 10.O-O-O Nbd7 11.Be2 *"),
("B98k", "Sicilian: Najdorf, Gothenburg (Argentine) Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 h6 9.Bh4 g5 *"),
("B98l", "Sicilian: Najdorf, Gothenburg, 11.Nxe6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 h6 9.Bh4 g5 10.fxg5 Nfd7 11.Nxe6 *"),
("B98m", "Sicilian: Najdorf, Gothenburg, 11.Qh5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 h6 9.Bh4 g5 10.fxg5 Nfd7 11.Qh5 *"),
("B98n", "Sicilian: Najdorf, 8...Qc7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 *"),
("B98n", "Sicilian: Najdorf, 8...Qc7 9.O-O-O", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O *"),
("B98o", "Sicilian: Najdorf, 8...Qc7 9.O-O-O b5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O b5 *"),
("B98p", "Sicilian: Najdorf, 8...Qc7 9.O-O-O Nc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nc6 *"),
("B98q", "Sicilian: Najdorf, 8...Qc7 9.O-O-O Nc6 10.Nxc6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nc6 10.Nxc6 *"),
("B99a", "Sicilian: Najdorf, Main Line", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 *"),
("B99b", "Sicilian: Najdorf, Main Line, 10.Qg3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.Qg3 *"),
("B99c", "Sicilian: Najdorf, Main Line, Keres Variation", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.Be2 *"),
("B99c", "Sicilian: Najdorf, Main Line, Keres 10...b5 11.Bxf6 Nxf6 12.e5 Bb7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.Be2 b5 11.Bxf6 Nxf6 12.e5 Bb7 *"),
("B99d", "Sicilian: Najdorf, Main Line, Keres, Queen Sacrifice Line", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.Be2 b5 11.Bxf6 Nxf6 12.e5 Bb7 13.exf6 *"),
("B99e", "Sicilian: Najdorf, Main Line, Keres, 13.Qg3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.Be2 b5 11.Bxf6 Nxf6 12.e5 Bb7 13.Qg3 *"),
("B99e", "Sicilian: Najdorf, Main Line, Keres, 13.Qg3 dxe5 14.fxe5 Nd7 15.Bxb5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.Be2 b5 11.Bxf6 Nxf6 12.e5 Bb7 13.Qg3 dxe5 14.fxe5 Nd7 15.Bxb5 *"),
("B99f", "Sicilian: Najdorf, Main Line, 10.Bd3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.Bd3 *"),
("B99f", "Sicilian: Najdorf, Main Line, 10.Bd3 h6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.Bd3 h6 *"),
("B99g", "Sicilian: Najdorf, Main Line, 10.Bd3 h6 11.Qh3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.Bd3 h6 11.Qh3 *"),
("B99h", "Sicilian: Najdorf, Main Line, 10.Bd3 h6 11.Bh4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.Bd3 h6 11.Bh4 *"),
("B99h", "Sicilian: Najdorf, Main Line, 10.Bd3 h6 11.Bh4 g5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.Bd3 h6 11.Bh4 g5 *"),
("B99i", "Sicilian: Najdorf, Main Line, 10.Bd3 h6 11.Bh4 g5 12.fxg5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.Bd3 h6 11.Bh4 g5 12.fxg5 *"),
("B99j", "Sicilian: Najdorf, Main Line, 10.Bd3 b5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.Bd3 b5 *"),
("B99j", "Sicilian: Najdorf, Main Line, 10.Bd3 b5 11.Rhe1 Bb7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.Bd3 b5 11.Rhe1 Bb7 *"),
("B99k", "Sicilian: Najdorf, Main Line, 10.Bd3 b5 11.Rhe1 Bb7 12.Nd5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.Bd3 b5 11.Rhe1 Bb7 12.Nd5 *"),
("B99l", "Sicilian: Najdorf, Main Line, 10.Bd3 b5 11.Rhe1 Bb7 12.Qg3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.Bd3 b5 11.Rhe1 Bb7 12.Qg3 *"),
("B99m", "Sicilian: Najdorf, Main Line, 10.g4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.g4 *"),
("B99m", "Sicilian: Najdorf, Main Line, 10.g4 h6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.g4 h6 *"),
("B99n", "Sicilian: Najdorf, Main Line, 10.g4 b5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.g4 b5 *"),
("B99n", "Sicilian: Najdorf, Main Line, 10.g4 b5 11.Bd3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.g4 b5 11.Bd3 *"),
("B99n", "Sicilian: Najdorf, Main Line, 10.g4 b5 11.Bd3 Bb7", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.g4 b5 11.Bd3 Bb7 *"),
("B99o", "Sicilian: Najdorf, Main Line, 10.g4 b5 11.a3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.g4 b5 11.a3 *"),
("B99p", "Sicilian: Najdorf, Main Line, 10.g4 b5 11.Bxf6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.g4 b5 11.Bxf6 *"),
("B99q", "Sicilian: Najdorf, Main Line, 10.g4 b5 11.Bxf6 gxf6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.g4 b5 11.Bxf6 gxf6 *"),
("B99r", "Sicilian: Najdorf, Main Line, 10.g4 b5 11.Bxf6 Nxf6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.g4 b5 11.Bxf6 Nxf6 *"),
("B99r", "Sicilian: Najdorf, Main Line, 10.g4 b5 11.Bxf6 Nxf6 12.g5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.g4 b5 11.Bxf6 Nxf6 12.g5 *"),
("B99r", "Sicilian: Najdorf, Main Line, 10.g4 b5 11.Bxf6 Nxf6 12.g5 Nd7 13.a3", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.g4 b5 11.Bxf6 Nxf6 12.g5 Nd7 13.a3 *"),
("B99s", "Sicilian: Najdorf, Modern Main Line 13.f5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.g4 b5 11.Bxf6 Nxf6 12.g5 Nd7 13.f5 *"),
("B99t", "Sicilian: Najdorf, Modern Main Line 13.f5 Bxg5+", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.g4 b5 11.Bxf6 Nxf6 12.g5 Nd7 13.f5 Bxg5+ *"),
("B99u", "Sicilian: Najdorf, Modern Main Line 13.f5 Bxg5+ 14.Kb1 Ne5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.g4 b5 11.Bxf6 Nxf6 12.g5 Nd7 13.f5 Bxg5+ 14.Kb1 Ne5 *"),
("B99v", "Sicilian: Najdorf, Modern Main Line 13.f5 Nc5", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.g4 b5 11.Bxf6 Nxf6 12.g5 Nd7 13.f5 Nc5 *"),
("B99w", "Sicilian: Najdorf, Modern Main Line 13.f5 Nc5 14.h4", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.g4 b5 11.Bxf6 Nxf6 12.g5 Nd7 13.f5 Nc5 14.h4 *"),
("B99x", "Sicilian: Najdorf, Modern Main Line 13.f5 Nc5 14.f6", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.g4 b5 11.Bxf6 Nxf6 12.g5 Nd7 13.f5 Nc5 14.f6 *"),
("B99y", "Sicilian: Najdorf, Modern Main Line 13.f5 Nc5 14.f6: Perenyi's 16.Rg1", "1.e4 c5 2.Nf3 d6 3.d4 cxd4 4.Nxd4 Nf6 5.Nc3 a6 6.Bg5 e6 7.f4 Be7 8.Qf3 Qc7 9.O-O-O Nbd7 10.g4 b5 11.Bxf6 Nxf6 12.g5 Nd7 13.f5 Nc5 14.f6 gxf6 15.gxf6 Bf8 16.Rg1 *"),
("C00a", "French", "1.e4 e6 *"),
("C00a", "French: Bird Variation", "1.e4 e6 2.Bb5 *"),
("C00b", "French: Reti (Spielmann) Variation", "1.e4 e6 2.b3 *"),
("C00c", "French: La Bourdonnais Variation", "1.e4 e6 2.f4 *"),
("C00d", "French: Steinitz Attack", "1.e4 e6 2.e5 *"),
("C00e", "French: Steiner (Anglo-French) Variation", "1.e4 e6 2.c4 *"),
("C00e", "French: Steiner (Anglo-French) Variation", "1.e4 e6 2.c4 d5 *"),
("C00e", "French: Orthoschnapp Gambit", "1.e4 e6 2.c4 d5 3.cxd5 exd5 4.Qb3 *"),
("C00f", "French: 2.Nc3", "1.e4 e6 2.Nc3 *"),
("C00f", "French: 2.Nc3 d5", "1.e4 e6 2.Nc3 d5 *"),
("C00f", "French: Pelikan Variation", "1.e4 e6 2.Nc3 d5 3.f4 *"),
("C00g", "French: Chigorin Variation", "1.e4 e6 2.Qe2 *"),
("C00h", "French: Chigorin, 2...c5", "1.e4 e6 2.Qe2 c5 *"),
("C00i", "French: Chigorin, 2...c5 3.Nf3", "1.e4 e6 2.Qe2 c5 3.Nf3 *"),
("C00j", "French: 2.Nf3", "1.e4 e6 2.Nf3 *"),
("C00j", "French: 2.Nf3 d5", "1.e4 e6 2.Nf3 d5 *"),
("C00j", "French: 2.Nf3 d5 3.e5", "1.e4 e6 2.Nf3 d5 3.e5 *"),
("C00j", "French: 2.Nf3 d5 3.e5 c5", "1.e4 e6 2.Nf3 d5 3.e5 c5 *"),
("C00k", "French: Wing Gambit", "1.e4 e6 2.Nf3 d5 3.e5 c5 4.b4 *"),
("C00l", "French: Two Knights Variation", "1.e4 e6 2.Nf3 d5 3.Nc3 *"),
("C00m", "French: KIA 2.d3", "1.e4 e6 2.d3 *"),
("C00m", "French: KIA 2.d3 d5", "1.e4 e6 2.d3 d5 *"),
("C00n", "French: KIA 2.d3 d5 3.Qe2", "1.e4 e6 2.d3 d5 3.Qe2 *"),
("C00o", "French: KIA 2.d3 d5 3.Qe2 Nf6", "1.e4 e6 2.d3 d5 3.Qe2 Nf6 *"),
("C00p", "French: KIA 2.d3 d5 3.Nd2", "1.e4 e6 2.d3 d5 3.Nd2 *"),
("C00q", "French: KIA 2.d3 d5 3.Nd2 c5", "1.e4 e6 2.d3 d5 3.Nd2 c5 *"),
("C00r", "French: KIA 2.d3 d5 3.Nd2 Nf6", "1.e4 e6 2.d3 d5 3.Nd2 Nf6 *"),
("C00s", "French: KIA, Reversed Philidor", "1.e4 e6 2.d3 d5 3.Nd2 Nf6 4.Ngf3 *"),
("C00t", "French: KIA, Reversed Philidor, 4...b6", "1.e4 e6 2.d3 d5 3.Nd2 Nf6 4.Ngf3 b6 *"),
("C00u", "French: KIA, Reversed Philidor, 4...Nc6", "1.e4 e6 2.d3 d5 3.Nd2 Nf6 4.Ngf3 Nc6 *"),
("C00v", "French: KIA, Reversed Philidor, 4...Nc6 5.c3", "1.e4 e6 2.d3 d5 3.Nd2 Nf6 4.Ngf3 Nc6 5.c3 *"),
("C00w", "French: 2.d4", "1.e4 e6 2.d4 *"),
("C00w", "French: St. George", "1.e4 e6 2.d4 a6 *"),
("C00w", "French: Franco-Benoni", "1.e4 e6 2.d4 c5 *"),
("C00x", "French: 2.d4 d5", "1.e4 e6 2.d4 d5 *"),
("C00x", "French: Alapin-Diemer Gambit (ADG)", "1.e4 e6 2.d4 d5 3.Be3 *"),
("C00y", "French: Schlechter", "1.e4 e6 2.d4 d5 3.Bd3 *"),
("C01a", "French: Exchange", "1.e4 e6 2.d4 d5 3.exd5 *"),
("C01a", "French: Exchange, 3...Qxd5", "1.e4 e6 2.d4 d5 3.exd5 Qxd5 *"),
("C01b", "French: Exchange", "1.e4 e6 2.d4 d5 3.exd5 exd5 *"),
("C01c", "French: Exchange, 4.c4", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.c4 *"),
("C01c", "French: Exchange, 4.c4 c6", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.c4 c6 *"),
("C01d", "French: Exchange, 4.c4 Nf6", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.c4 Nf6 *"),
("C01d", "French: Exchange, 4.c4 Nf6", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.c4 Nf6 5.Nf3 *"),
("C01e", "French: Exchange, 4.c4 Nf6 5.Nc3", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.c4 Nf6 5.Nc3 *"),
("C01e", "French: Exchange, 4.c4 Nf6 5.Nc3 c6", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.c4 Nf6 5.Nc3 c6 *"),
("C01e", "French: Exchange, 4.c4 Nf6 5.Nc3 Bb4", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.c4 Nf6 5.Nc3 Bb4 *"),
("C01e", "French: Exchange, 4.c4 Nf6 5.Nc3 Bb4", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.c4 Nf6 5.Nc3 Bb4 6.Nf3 *"),
("C01f", "French: Exchange, 4.c4 Nf6 5.Nc3 Be7", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.c4 Nf6 5.Nc3 Be7 *"),
("C01f", "French: Exchange, 4.c4 Nf6 5.Nc3 Be7", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.c4 Nf6 5.Nc3 Be7 6.Nf3 *"),
("C01g", "French: Exchange, 4.Bf4", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.Bf4 *"),
("C01h", "French: Exchange, 4.Bd3", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.Bd3 *"),
("C01i", "French: Exchange, 4.Bd3 Nc6", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.Bd3 Nc6 *"),
("C01j", "French: Exchange, 4.Bd3 Bd6", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.Bd3 Bd6 *"),
("C01k", "French: Exchange, 4.Nf3", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.Nf3 *"),
("C01l", "French: Exchange, 4.Nf3 Nf6", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.Nf3 Nf6 *"),
("C01m", "French: Exchange, 4.Nf3 Nf6 5.Bd3", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.Nf3 Nf6 5.Bd3 *"),
("C01n", "French: Exchange, 4.Nf3 Bd6", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.Nf3 Bd6 *"),
("C01o", "French: Exchange, 4.Nf3 Bd6 5.Bd3", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.Bd3 Bd6 5.Nf3 *"),
("C01p", "French: Exchange, 4.Nf3 Bd6 5.Bd3 Nf6", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.Bd3 Bd6 5.Nf3 Nf6 *"),
("C01q", "French: Exchange, 4.Nf3 Bd6 5.c4", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.Nf3 Bd6 5.c4 *"),
("C01r", "French: Exchange, 4.Nc3", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.Nc3 *"),
("C01s", "French: Exchange, 4.Nc3 Nf6", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.Nc3 Nf6 *"),
("C01s", "French: Exchange, 4.Nc3 Nf6 5.Nf3", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.Nc3 Nf6 5.Nf3 *"),
("C01t", "French: Exchange, Svenonius Variation", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.Nc3 Nf6 5.Bg5 *"),
("C01t", "French: Exchange, Bogoljubow Variation", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.Nc3 Nf6 5.Bg5 Nc6 *"),
("C01u", "French: Exchange Winawer", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.Nc3 Bb4 *"),
("C01v", "French: Exchange Winawer, 5.Bd3", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.Nc3 Bb4 5.Bd3 *"),
("C01v", "French: Exchange, Canal Attack", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.Nc3 Bb4 5.Bd3 Ne7 6.Qh5 *"),
("C01w", "French: Exchange Winawer, 5.Bd3 Nc6", "1.e4 e6 2.d4 d5 3.exd5 exd5 4.Nc3 Bb4 5.Bd3 Nc6 *"),
("C02a", "French: Advance", "1.e4 e6 2.d4 d5 3.e5 *"),
("C02a", "French: Advance, 3...Ne7", "1.e4 e6 2.d4 d5 3.e5 Ne7 *"),
("C02b", "French: Advance, 3...b6", "1.e4 e6 2.d4 d5 3.e5 b6 *"),
("C02b", "French: Advance, 3...b6", "1.e4 e6 2.d4 d5 3.e5 b6 4.c3 Qd7 *"),
("C02c", "French: Advance, 3...c5", "1.e4 e6 2.d4 d5 3.e5 c5 *"),
("C02c", "French: Advance, Wing Gambit", "1.e4 e6 2.d4 d5 3.e5 c5 4.b4 *"),
("C02d", "French: Advance, Steinitz", "1.e4 e6 2.d4 d5 3.e5 c5 4.dxc5 *"),
("C02e", "French: Advance, Nimzowitsch Attack", "1.e4 e6 2.d4 d5 3.e5 c5 4.Qg4 *"),
("C02e", "French: Advance, Nimzowitsch Gambit", "1.e4 e6 2.d4 d5 3.e5 c5 4.Qg4 cxd4 5.Nf3 *"),
("C02f", "French: Advance, 4.Nf3", "1.e4 e6 2.d4 d5 3.e5 c5 4.Nf3 *"),
("C02f", "French: Advance, Ruisdonk Gambit", "1.e4 e6 2.d4 d5 3.e5 c5 4.Nf3 cxd4 5.Bd3 *"),
("C02g", "French: Advance, 4.c3", "1.e4 e6 2.d4 d5 3.e5 c5 4.c3 *"),
("C02h", "French: Advance, 4.c3 Qb6", "1.e4 e6 2.d4 d5 3.e5 c5 4.c3 Qb6 *"),
("C02i", "French: Advance, Wade", "1.e4 e6 2.d4 d5 3.e5 c5 4.c3 Qb6 5.Nf3 Bd7 *"),
("C02j", "French: Advance, Wade, 6.Be2", "1.e4 e6 2.d4 d5 3.e5 c5 4.c3 Qb6 5.Nf3 Bd7 6.Be2 *"),
("C02k", "French: Advance, 4...Nc6", "1.e4 e6 2.d4 d5 3.e5 c5 4.c3 Nc6 *"),
("C02l", "French: Advance, Paulsen", "1.e4 e6 2.d4 d5 3.e5 c5 4.c3 Nc6 5.Nf3 *"),
("C02m", "French: Advance, 5.Nf3 Nge7", "1.e4 e6 2.d4 d5 3.e5 c5 4.c3 Nc6 5.Nf3 Nge7 *"),
("C02n", "French: Advance, Euwe", "1.e4 e6 2.d4 d5 3.e5 c5 4.c3 Nc6 5.Nf3 Bd7 *"),
("C02o", "French: Advance, Euwe, 6.a3", "1.e4 e6 2.d4 d5 3.e5 c5 4.c3 Nc6 5.Nf3 Bd7 6.a3 *"),
("C02p", "French: Advance, Euwe, 6.Be2", "1.e4 e6 2.d4 d5 3.e5 c5 4.c3 Nc6 5.Nf3 Bd7 6.Be2 *"),
("C02q", "French: Advance, Euwe, 6.Be2 Nge7", "1.e4 e6 2.d4 d5 3.e5 c5 4.c3 Nc6 5.Nf3 Bd7 6.Be2 Nge7 *"),
("C02r", "French: Advance, 5.Nf3 Qb6", "1.e4 e6 2.d4 d5 3.e5 c5 4.c3 Nc6 5.Nf3 Qb6 *"),
("C02s", "French: Advance, Milner-Barry Gambit", "1.e4 e6 2.d4 d5 3.e5 c5 4.c3 Nc6 5.Nf3 Qb6 6.Bd3 *"),
("C02t", "French: Advance, 5.Nf3 Qb6 6.Be2", "1.e4 e6 2.d4 d5 3.e5 c5 4.c3 Nc6 5.Nf3 Qb6 6.Be2 *"),
("C02u", "French: Advance, 5.Nf3 Qb6 6.Be2 cxd4 7.cxd4", "1.e4 e6 2.d4 d5 3.e5 c5 4.c3 Nc6 5.Nf3 Qb6 6.Be2 cxd4 7.cxd4 *"),
("C02v", "French: Advance, 5.Nf3 Qb6 6.a3", "1.e4 e6 2.d4 d5 3.e5 c5 4.c3 Nc6 5.Nf3 Qb6 6.a3 *"),
("C02w", "French: Advance, 5.Nf3 Qb6 6.a3 Bd7", "1.e4 e6 2.d4 d5 3.e5 c5 4.c3 Nc6 5.Nf3 Qb6 6.a3 Bd7 *"),
("C02x", "French: Advance, 5.Nf3 Qb6 6.a3 Nh6", "1.e4 e6 2.d4 d5 3.e5 c5 4.c3 Nc6 5.Nf3 Qb6 6.a3 Nh6 *"),
("C02y", "French: Advance, 5.Nf3 Qb6 6.a3 c4", "1.e4 e6 2.d4 d5 3.e5 c5 4.c3 Nc6 5.Nf3 Qb6 6.a3 c4 *"),
("C02z", "French: Advance, 5.Nf3 Qb6 6.a3 c4 7.Nbd2", "1.e4 e6 2.d4 d5 3.e5 c5 4.c3 Nc6 5.Nf3 Qb6 6.a3 c4 7.Nbd2 *"),
("C03a", "French: Tarrasch", "1.e4 e6 2.d4 d5 3.Nd2 *"),
("C03a", "French: Tarrasch, 3...dxe4", "1.e4 e6 2.d4 d5 3.Nd2 dxe4 *"),
("C03b", "French: Tarrasch, Haberditz Variation", "1.e4 e6 2.d4 d5 3.Nd2 f5 *"),
("C03c", "French: Tarrasch, 3...b6", "1.e4 e6 2.d4 d5 3.Nd2 b6 *"),
("C03d", "French: Tarrasch, 3...a6", "1.e4 e6 2.d4 d5 3.Nd2 a6 *"),
("C03e", "French: Tarrasch, 3...a6 4.Ngf3", "1.e4 e6 2.d4 d5 3.Nd2 a6 4.Ngf3 *"),
("C03f", "French: Tarrasch, 3...Be7", "1.e4 e6 2.d4 d5 3.Nd2 Be7 *"),
("C03g", "French: Tarrasch, 3...Be7 4.Bd3", "1.e4 e6 2.d4 d5 3.Nd2 Be7 4.Bd3 *"),
("C03h", "French: Tarrasch, 3...Be7 4.Bd3 c5", "1.e4 e6 2.d4 d5 3.Nd2 Be7 4.Bd3 c5 *"),
("C03i", "French: Tarrasch, 3...Be7 4.Bd3 c5 5.dxc5 Nf6", "1.e4 e6 2.d4 d5 3.Nd2 Be7 4.Bd3 c5 5.dxc5 Nf6 *"),
("C03j", "French: Tarrasch, 3...Be7 4.Ngf3", "1.e4 e6 2.d4 d5 3.Nd2 Be7 4.Ngf3 *"),
("C03j", "French: Tarrasch, 3...Be7 4.Ngf3 Nf6", "1.e4 e6 2.d4 d5 3.Nd2 Be7 4.Ngf3 Nf6 *"),
("C03k", "French: Tarrasch, 3...Be7 4.Ngf3 Nf6 5.Bd3", "1.e4 e6 2.d4 d5 3.Nd2 Be7 4.Ngf3 Nf6 5.Bd3 *"),
("C03l", "French: Tarrasch, Guimard Variation", "1.e4 e6 2.d4 d5 3.Nd2 Nc6 *"),
("C03m", "French: Tarrasch, Guimard, 4.c3", "1.e4 e6 2.d4 d5 3.Nd2 Nc6 4.c3 *"),
("C03n", "French: Tarrasch, Guimard, 4.c3 e5", "1.e4 e6 2.d4 d5 3.Nd2 Nc6 4.c3 e5 *"),
("C03o", "French: Tarrasch, Guimard, 4.Ngf3", "1.e4 e6 2.d4 d5 3.Nd2 Nc6 4.Ngf3 *"),
("C04a", "French: Tarrasch, Guimard, 4.Ngf3 Nf6", "1.e4 e6 2.d4 d5 3.Nd2 Nc6 4.Ngf3 Nf6 *"),
("C04b", "French: Tarrasch, Guimard, 4.Ngf3 Nf6 5.e5", "1.e4 e6 2.d4 d5 3.Nd2 Nc6 4.Ngf3 Nf6 5.e5 *"),
("C04c", "French: Tarrasch, Guimard, Main Line", "1.e4 e6 2.d4 d5 3.Nd2 Nc6 4.Ngf3 Nf6 5.e5 Nd7 *"),
("C04d", "French: Tarrasch, Guimard, Main Line, 6.c3", "1.e4 e6 2.d4 d5 3.Nd2 Nc6 4.Ngf3 Nf6 5.e5 Nd7 6.c3 *"),
("C04e", "French: Tarrasch, Guimard, Main Line, 6.Bb5", "1.e4 e6 2.d4 d5 3.Nd2 Nc6 4.Ngf3 Nf6 5.e5 Nd7 6.Bb5 *"),
("C04f", "French: Tarrasch, Guimard, Main Line, 6.Be2", "1.e4 e6 2.d4 d5 3.Nd2 Nc6 4.Ngf3 Nf6 5.e5 Nd7 6.Be2 *"),
("C04g", "French: Tarrasch, Guimard, Main Line, 6.Be2 f6", "1.e4 e6 2.d4 d5 3.Nd2 Nc6 4.Ngf3 Nf6 5.e5 Nd7 6.Be2 f6 *"),
("C04h", "French: Tarrasch, Guimard, Main Line, 6.Nb3", "1.e4 e6 2.d4 d5 3.Nd2 Nc6 4.Ngf3 Nf6 5.e5 Nd7 6.Nb3 *"),
("C04i", "French: Tarrasch, Guimard, Main Line, 6.Nb3 a5", "1.e4 e6 2.d4 d5 3.Nd2 Nc6 4.Ngf3 Nf6 5.e5 Nd7 6.Nb3 a5 *"),
("C04j", "French: Tarrasch, Guimard, Main Line, 6.Nb3 Be7", "1.e4 e6 2.d4 d5 3.Nd2 Nc6 4.Ngf3 Nf6 5.e5 Nd7 6.Nb3 Be7 *"),
("C05a", "French: Tarrasch, Closed", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 *"),
("C05b", "French: Tarrasch, Closed, 4.e5", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 *"),
("C05c", "French: Tarrasch, Closed, 4...Ne4", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Ne4 *"),
("C05c", "French: Tarrasch, Closed, 4...Ne4 5.Nxe4", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Ne4 5.Nxe4 *"),
("C05d", "French: Tarrasch, Closed, 4...Nfd7", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 *"),
("C05e", "French: Tarrasch, Closed, 5.f4", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.f4 *"),
("C05f", "French: Tarrasch, Closed, 5.f4 c5 6.c3 Nc6", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.f4 c5 6.c3 Nc6 *"),
("C05g", "French: Tarrasch, Closed, 5.f4 c5 6.c3 Nc6 7.Ndf3", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.f4 c5 6.c3 Nc6 7.Ndf3 *"),
("C05h", "French: Tarrasch, Closed, 5.f4 c5 6.c3 Nc6 7.Ndf3 cxd4", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.f4 c5 6.c3 Nc6 7.Ndf3 cxd4 *"),
("C05i", "French: Tarrasch, Closed, 5.f4 c5 6.c3 Nc6 7.Ndf3 Qb6", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.f4 c5 6.c3 Nc6 7.Ndf3 Qb6 *"),
("C05j", "French: Tarrasch, Closed, 5.f4 c5 6.c3 Nc6 7.Ndf3 Qb6 8.h4", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.f4 c5 6.c3 Nc6 7.Ndf3 Qb6 8.h4 *"),
("C05j", "French: Tarrasch, Closed, 5.f4 c5 6.c3 Nc6 7.Ndf3 Qb6 8.h4", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.f4 c5 6.c3 Nc6 7.Ndf3 Qb6 8.h4 cxd4 9.cxd4 *"),
("C05k", "French: Tarrasch, Closed, 5.f4 c5 6.c3 Nc6 7.Ndf3 Qb6 8.Ne2", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.f4 c5 6.c3 Nc6 7.Ndf3 Qb6 8.Ne2 *"),
("C05k", "French: Tarrasch, Closed, 5.f4 c5 6.c3 Nc6 7.Ndf3 Qb6 8.Ne2", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.f4 c5 6.c3 Nc6 7.Ndf3 Qb6 8.Ne2 cxd4 9.cxd4 *"),
("C05l", "French: Tarrasch, Closed, 5.f4 c5 6.c3 Nc6 7.Ndf3 Qb6 8.g3", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.f4 c5 6.c3 Nc6 7.Ndf3 Qb6 8.g3 *"),
("C05l", "French: Tarrasch, Closed, 5.f4 c5 6.c3 Nc6 7.Ndf3 Qb6 8.g3", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.f4 c5 6.c3 Nc6 7.Ndf3 Qb6 8.g3 cxd4 9.cxd4 *"),
("C05m", "French: Tarrasch, Closed, 5.c3", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.c3 *"),
("C05n", "French: Tarrasch, 5.Bd3", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 *"),
("C05o", "French: Tarrasch, Botvinnik", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 b6 *"),
("C05p", "French: Tarrasch, Closed, 6...Nc6", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 *"),
("C05p", "French: Tarrasch, Closed, 7.Ndf3", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ndf3 *"),
("C05q", "French: Tarrasch, Closed, 7.Ndf3 Qa5", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ndf3 Qa5 *"),
("C05r", "French: Tarrasch, Closed, 7.Ngf3", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ngf3 *"),
("C05s", "French: Tarrasch, Closed, 7.Ngf3 Be7", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ngf3 Be7 *"),
("C05t", "French: Tarrasch, Closed, 7.Ngf3 Be7 8.O-O g5", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ngf3 Be7 8.O-O g5 *"),
("C05u", "French: Tarrasch, Closed, Nunn-Korchnoi Gambit", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ngf3 Qb6 8.O-O *"),
("C05v", "French: Tarrasch, Closed, 7.Ne2", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ne2 *"),
("C05w", "French: Tarrasch, Closed, 7.Ne2 a5", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ne2 a5 *"),
("C06a", "French: Tarrasch, Closed, Main Line", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ne2 cxd4 8.cxd4 *"),
("C06b", "French: Tarrasch, Closed, Leningrad", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ne2 cxd4 8.cxd4 Nb6 *"),
("C06c", "French: Tarrasch, Closed, Leningrad, 9.Nf3", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ne2 cxd4 8.cxd4 Nb6 9.Nf3 *"),
("C06d", "French: Tarrasch, Closed, Leningrad, 9.O-O", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ne2 cxd4 8.cxd4 Nb6 9.O-O *"),
("C06e", "French: Tarrasch, Closed, Leningrad, 9.O-O a5", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ne2 cxd4 8.cxd4 Nb6 9.O-O a5 *"),
("C06f", "French: Tarrasch, Closed, Leningrad, 9.O-O Bd7", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ne2 cxd4 8.cxd4 Nb6 9.O-O Bd7 *"),
("C06g", "French: Tarrasch, Closed, 8...Qb6", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ne2 cxd4 8.cxd4 Qb6 *"),
("C06h", "French: Tarrasch, Closed, 8...Qb6 9.O-O", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ne2 cxd4 8.cxd4 Qb6 9.O-O *"),
("C06i", "French: Tarrasch, Closed, 8...Qb6 9.Nf3", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ne2 cxd4 8.cxd4 Qb6 9.Nf3 *"),
("C06j", "French: Tarrasch, Closed, 8...Qb6 9.Nf3 f6", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ne2 cxd4 8.cxd4 Qb6 9.Nf3 f6 *"),
("C06k", "French: Tarrasch, Closed, 8...f6", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ne2 cxd4 8.cxd4 f6 *"),
("C06l", "French: Tarrasch, Closed, 8...f6 9.Nf4", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ne2 cxd4 8.cxd4 f6 9.Nf4 *"),
("C06m", "French: Tarrasch, Closed, 8...f6 9.exf6", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ne2 cxd4 8.cxd4 f6 9.exf6 *"),
("C06n", "French: Tarrasch, Closed, 8...f6 9.exf6 Nxf6", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ne2 cxd4 8.cxd4 f6 9.exf6 Nxf6 *"),
("C06o", "French: Tarrasch, Closed, 8...f6 9.exf6 Nxf6 10.O-O Bd6 11.Nf3", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ne2 cxd4 8.cxd4 f6 9.exf6 Nxf6 10.O-O Bd6 11.Nf3 *"),
("C06o", "French: Tarrasch, Closed, 8...f6 9.exf6, 11...Qb6", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ne2 cxd4 8.cxd4 f6 9.exf6 Nxf6 10.O-O Bd6 11.Nf3 Qb6 *"),
("C06p", "French: Tarrasch, Closed, 8...f6 9.exf6, 11...Qb6 12.Nc3", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ne2 cxd4 8.cxd4 f6 9.exf6 Nxf6 10.O-O Bd6 11.Nf3 Qb6 12.Nc3 *"),
("C06q", "French: Tarrasch, Closed, 8...f6 9.exf6, 11...Qc7", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ne2 cxd4 8.cxd4 f6 9.exf6 Nxf6 10.O-O Bd6 11.Nf3 Qc7 *"),
("C06r", "French: Tarrasch, Closed, 8...f6 9.exf6, 11...Qc7 12.Bg5", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ne2 cxd4 8.cxd4 f6 9.exf6 Nxf6 10.O-O Bd6 11.Nf3 Qc7 12.Bg5 *"),
("C06s", "French: Tarrasch, Closed, 8...f6 9.exf6, 11...O-O", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ne2 cxd4 8.cxd4 f6 9.exf6 Nxf6 10.O-O Bd6 11.Nf3 O-O *"),
("C06t", "French: Tarrasch, Closed, 8...f6 9.exf6, 11...O-O 12.Bf4", "1.e4 e6 2.d4 d5 3.Nd2 Nf6 4.e5 Nfd7 5.Bd3 c5 6.c3 Nc6 7.Ne2 cxd4 8.cxd4 f6 9.exf6 Nxf6 10.O-O Bd6 11.Nf3 O-O 12.Bf4 *"),
("C07a", "French: Tarrasch, Open", "1.e4 e6 2.d4 d5 3.Nd2 c5 *"),
("C07b", "French: Tarrasch, Open, 4.c3", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.c3 *"),
("C07c", "French: Tarrasch, Open, 4.dxc5", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.dxc5 *"),
("C07d", "French: Tarrasch, Open, 4.Ngf3", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.Ngf3 *"),
("C07e", "French: Tarrasch, Open, 4.Ngf3 a6", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.Ngf3 a6 *"),
("C07f", "French: Tarrasch, Open, 4.Ngf3 Nf6", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.Ngf3 Nf6 *"),
("C07g", "French: Tarrasch, Open, 4.Ngf3 Nc6", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.Ngf3 Nc6 *"),
("C07h", "French: Tarrasch, Open, 4.Ngf3 cxd4", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.Ngf3 cxd4 *"),
("C07i", "French: Tarrasch, Open, 4.exd5", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 *"),
("C07i", "French: Tarrasch, Shaposhnikov Gambit", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 Nf6 *"),
("C07j", "French: Tarrasch, Open, 4.exd5 Qxd5", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 Qxd5 *"),
("C07k", "French: Tarrasch, Open, 4.exd5 Qxd5 5.Ngf3", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 Qxd5 5.Ngf3 *"),
("C07l", "French: Tarrasch, Open, 4.exd5 Qxd5 5.Ngf3 cxd4", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 Qxd5 5.Ngf3 cxd4 *"),
("C07m", "French: Tarrasch, Open, 4.exd5 Qxd5 5.Ngf3 cxd4 6.Bc4", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 Qxd5 5.Ngf3 cxd4 6.Bc4 *"),
("C07n", "French: Tarrasch, Open, Eliskases Variation", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 Qxd5 5.Ngf3 cxd4 6.Bc4 Qd8 *"),
("C07o", "French: Tarrasch, Open, 4.exd5 Qxd5 5.Ngf3 cxd4 6.Bc4 Qd6", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 Qxd5 5.Ngf3 cxd4 6.Bc4 Qd6 *"),
("C07p", "French: Tarrasch, Open, 4.exd5 Qxd5, Main Line, 10.Qxd4", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 Qxd5 5.Ngf3 cxd4 6.Bc4 Qd6 7.O-O Nf6 8.Nb3 Nc6 9.Nbxd4 Nxd4 10.Qxd4 *"),
("C07p", "French: Tarrasch, Open, 4.exd5 Qxd5, Main Line, 10.Qxd4 Qxd4", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 Qxd5 5.Ngf3 cxd4 6.Bc4 Qd6 7.O-O Nf6 8.Nb3 Nc6 9.Nbxd4 Nxd4 10.Qxd4 Qxd4 *"),
("C07q", "French: Tarrasch, Open, 4.exd5 Qxd5, Main Line, 10.Nxd4", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 Qxd5 5.Ngf3 cxd4 6.Bc4 Qd6 7.O-O Nf6 8.Nb3 Nc6 9.Nbxd4 Nxd4 10.Nxd4 *"),
("C07r", "French: Tarrasch, Open, 4.exd5 Qxd5, Main Line, 10.Nxd4 a6", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 Qxd5 5.Ngf3 cxd4 6.Bc4 Qd6 7.O-O Nf6 8.Nb3 Nc6 9.Nbxd4 Nxd4 10.Nxd4 a6 *"),
("C07s", "French: Tarrasch, Open, 4.exd5 Qxd5, Main Line, 10.Nxd4 a6 11.c3", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 Qxd5 5.Ngf3 cxd4 6.Bc4 Qd6 7.O-O Nf6 8.Nb3 Nc6 9.Nbxd4 Nxd4 10.Nxd4 a6 11.c3 *"),
("C07t", "French: Tarrasch, Open, 4.exd5 Qxd5, Main Line, 10.Nxd4 a6 11.Re1", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 Qxd5 5.Ngf3 cxd4 6.Bc4 Qd6 7.O-O Nf6 8.Nb3 Nc6 9.Nbxd4 Nxd4 10.Nxd4 a6 11.Re1 *"),
("C08a", "French: Tarrasch, Open, 4.exd5 exd5", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 *"),
("C08b", "French: Tarrasch, Open, 4.exd5 exd5 5.Bb5+ Nc6", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Bb5+ Nc6 *"),
("C08c", "French: Tarrasch, Open, 4.exd5 exd5 5.Bb5+ Nc6 6.Qe2+", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Bb5+ Nc6 6.Qe2+ *"),
("C08d", "French: Tarrasch, Open, 4.exd5 exd5 5.Bb5+ Bd7", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Bb5+ Bd7 *"),
("C08e", "French: Tarrasch, Open, 4.exd5 exd5 5.Bb5+ Bd7 6.Qe2+", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Bb5+ Bd7 6.Qe2+ *"),
("C08f", "French: Tarrasch, Open, 4.exd5 exd5 5.Bb5+ Bd7 6.Qe2+ Be7", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Bb5+ Bd7 6.Qe2+ Be7 *"),
("C08g", "French: Tarrasch, Open, 4.exd5 exd5 5.Ngf3", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 *"),
("C08h", "French: Tarrasch, Open, 4.exd5 exd5 5.Ngf3 a6", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 a6 *"),
("C08i", "French: Tarrasch, Open, 4.exd5 exd5 5.Ngf3 a6 6.Be2", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 a6 6.Be2 *"),
("C08j", "French: Tarrasch, Open, 4.exd5 exd5 5.Ngf3 Nf6", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 Nf6 *"),
("C08k", "French: Tarrasch, Open, 4.exd5 exd5 5.Ngf3 Nf6, 7.Bxd7", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 Nf6 6.Bb5+ Bd7 7.Bxd7+ *"),
("C08l", "French: Tarrasch, Open, 4.exd5 exd5 5.Ngf3 Nf6, Main Line", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 Nf6 6.Bb5+ Bd7 7.Bxd7+ Nbd7 8.O-O Be7 9.dxc5 *"),
("C08m", "French: Tarrasch, Open, 4.exd5 exd5 5.Ngf3 Nf6, Main Line, 10.Nb3", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 Nf6 6.Bb5+ Bd7 7.Bxd7+ Nbd7 8.O-O Be7 9.dxc5 Nxc5 10.Nb3 *"),
("C09a", "French: Tarrasch, Open, 5.Ngf3 Nc6", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 Nc6 *"),
("C09b", "French: Tarrasch, Open, 5.Ngf3 Nc6 6.Bb5", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 Nc6 6.Bb5 *"),
("C09c", "French: Tarrasch, Open, 5.Ngf3 Nc6 6.Bb5 cxd4", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 Nc6 6.Bb5 cxd4 *"),
("C09d", "French: Tarrasch, Open, 5.Ngf3 Nc6 6.Bb5 Bd6", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 Nc6 6.Bb5 Bd6 *"),
("C09d", "French: Tarrasch, Open, 7.O-O", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 Nc6 6.Bb5 Bd6 7.O-O *"),
("C09e", "French: Tarrasch, Open, 7.dxc5", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 Nc6 6.Bb5 Bd6 7.dxc5 *"),
("C09f", "French: Tarrasch, Open, Main Line", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 Nc6 6.Bb5 Bd6 7.dxc5 Bxc5 8.O-O Ne7 *"),
("C09g", "French: Tarrasch, Open, Main Line, 9.c3", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 Nc6 6.Bb5 Bd6 7.O-O Ne7 8.dxc5 Bxc5 9.c3 *"),
("C09h", "French: Tarrasch, Open, Main Line, 9.c3 O-O 10.Nb3 Bd6", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 Nc6 6.Bb5 Bd6 7.O-O Ne7 8.dxc5 Bxc5 9.c3 O-O 10.Nb3 Bd6 *"),
("C09i", "French: Tarrasch, Open, Main Line, 9.c3 O-O 10.Nb3 Bb6", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 Nc6 6.Bb5 Bd6 7.O-O Ne7 8.dxc5 Bxc5 9.c3 O-O 10.Nb3 Bb6 *"),
("C09j", "French: Tarrasch, Open, Main Line, 9.c3 O-O 10.Nb3 Bb6 11.Re1", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 Nc6 6.Bb5 Bd6 7.O-O Ne7 8.dxc5 Bxc5 9.c3 O-O 10.Nb3 Bb6 11.Re1 *"),
("C09k", "French: Tarrasch, Open, Main Line, 9.Nb3", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 Nc6 6.Bb5 Bd6 7.O-O Ne7 8.dxc5 Bxc5 9.Nb3 *"),
("C09k", "French: Tarrasch, Open, Main Line, 9.Nb3 Bb6", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 Nc6 6.Bb5 Bd6 7.O-O Ne7 8.dxc5 Bxc5 9.Nb3 Bb6 *"),
("C09l", "French: Tarrasch, Open, Main Line, 9.Nb3 Bd6", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 Nc6 6.Bb5 Bd6 7.O-O Ne7 8.dxc5 Bxc5 9.Nb3 Bd6 *"),
("C09m", "French: Tarrasch, Open, Main Line, 9.Nb3 Bd6 10.Nbd4", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 Nc6 6.Bb5 Bd6 7.O-O Ne7 8.dxc5 Bxc5 9.Nb3 Bd6 10.Nbd4 *"),
("C09n", "French: Tarrasch, Open, Main Line, 9.Nb3 Bd6 10.Bg5", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 Nc6 6.Bb5 Bd6 7.O-O Ne7 8.dxc5 Bxc5 9.Nb3 Bd6 10.Bg5 *"),
("C09o", "French: Tarrasch, Open, Main Line, 9.Nb3 Bd6 10.Re1", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 Nc6 6.Bb5 Bd6 7.O-O Ne7 8.dxc5 Bxc5 9.Nb3 Bd6 10.Re1 *"),
("C09p", "French: Tarrasch, Open, Main Line, 9.Nb3 Bd6 10.Re1 O-O 11.Bg5", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 Nc6 6.Bb5 Bd6 7.O-O Ne7 8.dxc5 Bxc5 9.Nb3 Bd6 10.Re1 O-O 11.Bg5 *"),
("C09q", "French: Tarrasch, Open, Main Line, 9.Nb3 Bd6 10.Re1 O-O 11.Bg5 Bg4", "1.e4 e6 2.d4 d5 3.Nd2 c5 4.exd5 exd5 5.Ngf3 Nc6 6.Bb5 Bd6 7.O-O Ne7 8.dxc5 Bxc5 9.Nb3 Bd6 10.Re1 O-O 11.Bg5 Bg4 *"),
("C10a", "French: 3.Nc3", "1.e4 e6 2.d4 d5 3.Nc3 *"),
("C10b", "French: Marshall Variation", "1.e4 e6 2.d4 d5 3.Nc3 c5 *"),
("C10c", "French: 3.Nc3 Nc6", "1.e4 e6 2.d4 d5 3.Nc3 Nc6 *"),
("C10d", "French: 3.Nc3 Nc6 4.Nf3 Nf6", "1.e4 e6 2.d4 d5 3.Nc3 Nc6 4.Nf3 Nf6 *"),
("C10e", "French: 3.Nc3 Be7", "1.e4 e6 2.d4 d5 3.Nc3 Be7 *"),
("C10f", "French: Rubinstein", "1.e4 e6 2.d4 d5 3.Nc3 dxe4 *"),
("C10f", "French: Rubinstein", "1.e4 e6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 *"),
("C10f", "French: Rubinstein, Ellis Gambit", "1.e4 e6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 e5 *"),
("C10f", "French: Rubinstein, Frere (Becker) Variation", "1.e4 e6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Qd5 *"),
("C10g", "French: Rubinstein, 4...Nf6", "1.e4 e6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nf6 *"),
("C10h", "French: Rubinstein, 4...Nd7", "1.e4 e6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 *"),
("C10i", "French: Rubinstein, 5.Nf3 Be7", "1.e4 e6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Nf3 Be7 *"),
("C10j", "French: Rubinstein, 5.Nf3 Ngf6", "1.e4 e6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Nf3 Ngf6 *"),
("C10k", "French: Rubinstein, 5.Nf3 Ngf6 6.Bd3", "1.e4 e6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Nf3 Ngf6 6.Bd3 *"),
("C10l", "French: Rubinstein, 5.Nf3 Ngf6 6.Nxf6+", "1.e4 e6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Nf3 Ngf6 6.Nxf6+ *"),
("C10m", "French: Rubinstein, Capablanca", "1.e4 e6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Nf3 Ngf6 6.Nxf6+ Nxf6 7.Ne5 *"),
("C10n", "French: Rubinstein, 7.Bd3", "1.e4 e6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Nf3 Ngf6 6.Nxf6+ Nxf6 7.Bd3 *"),
("C10o", "French: Rubinstein, 7.Bd3 c5", "1.e4 e6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Nf3 Ngf6 6.Nxf6+ Nxf6 7.Bd3 c5 *"),
("C10p", "French: Rubinstein, 7.Bd3 c5 8.dxc5", "1.e4 e6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Nf3 Ngf6 6.Nxf6+ Nxf6 7.Bd3 c5 8.dxc5 *"),
("C10q", "French: Rubinstein, 7.Bg5", "1.e4 e6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Nd7 5.Nf3 Ngf6 6.Nxf6+ Nxf6 7.Bg5 *"),
("C10r", "French: Rubinstein, 4...Bd7", "1.e4 e6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bd7 *"),
("C10s", "French: Rubinstein, 4...Bd7 5.Nf3", "1.e4 e6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bd7 5.Nf3 *"),
("C10t", "French: Rubinstein, 4...Bd7 5.Nf3 Bc6 6.Bd3", "1.e4 e6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bd7 5.Nf3 Bc6 6.Bd3 *"),
("C10u", "French: Rubinstein, 4...Bd7 5.Nf3 Bc6 6.Bd3 Nd7", "1.e4 e6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bd7 5.Nf3 Bc6 6.Bd3 Nd7 *"),
("C10v", "French: Rubinstein, 4...Bd7 5.Nf3 Bc6 6.Bd3 Nd7 7.O-O", "1.e4 e6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bd7 5.Nf3 Bc6 6.Bd3 Nd7 7.O-O *"),
("C10w", "French: Rubinstein, 4...Bd7 5.Nf3 Bc6 6.Bd3 Nd7 7.O-O Ngf6", "1.e4 e6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bd7 5.Nf3 Bc6 6.Bd3 Nd7 7.O-O Ngf6 *"),
("C10x", "French: Rubinstein, 4...Bd7 5.Nf3 Bc6 6.Bd3 Nd7 7.O-O Ngf6 8.Ng3", "1.e4 e6 2.d4 d5 3.Nc3 dxe4 4.Nxe4 Bd7 5.Nf3 Bc6 6.Bd3 Nd7 7.O-O Ngf6 8.Ng3 *"),
("C11a", "French: 3.Nc3 Nf6", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 *"),
("C11a", "French: Henneberger Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Be3 *"),
("C11a", "French: 3.Nc3 Nf6 4.exd5", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.exd5 *"),
("C11b", "French: Swiss Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bd3 *"),
("C11c", "French: Steinitz", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.e5 *"),
("C11c", "French: Steinitz, Gledhill Attack", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.e5 Nfd7 5.Qg4 *"),
("C11d", "French: Steinitz, 5.Nf3", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.e5 Nfd7 5.Nf3 *"),
("C11d", "French: Steinitz, 5.Nf3 c5", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.e5 Nfd7 5.Nf3 c5 *"),
("C11e", "French: Steinitz, 5.Nf3 c5 6.dxc5", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.e5 Nfd7 5.Nf3 c5 6.dxc5 *"),
("C11f", "French: Steinitz, 5.f4", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.e5 Nfd7 5.f4 *"),
("C11f", "French: Steinitz, 5.f4 c5", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.e5 Nfd7 5.f4 c5 *"),
("C11g", "French: Steinitz, 6.dxc5", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.e5 Nfd7 5.f4 c5 6.dxc5 *"),
("C11g", "French: Steinitz, Bradford Attack", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.e5 Nfd7 5.f4 c5 6.dxc5 Bxc5 7.Qg4 *"),
("C11g", "French: Steinitz, 6.dxc5 Nc6", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.e5 Nfd7 5.f4 c5 6.dxc5 Nc6 *"),
("C11g", "French: Steinitz, Brodsky-Jones Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.e5 Nfd7 5.f4 c5 6.dxc5 Nc6 7.a3 Bxc5 8.Qg4 O-O 9.Nf3 f6 *"),
("C11h", "French: Steinitz, 6.Nf3", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.e5 Nfd7 5.f4 c5 6.Nf3 *"),
("C11i", "French: Steinitz, Boleslavsky Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.e5 Nfd7 5.f4 c5 6.Nf3 Nc6 7.Be3 *"),
("C11j", "French: Steinitz, Boleslavsky, 7...a6", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.e5 Nfd7 5.f4 c5 6.Nf3 Nc6 7.Be3 a6 *"),
("C11k", "French: Steinitz, Boleslavsky, 7...a6 8.Qd2 b5", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.e5 Nfd7 5.f4 c5 6.Nf3 Nc6 7.Be3 a6 8.Qd2 b5 *"),
("C11l", "French: Steinitz, Boleslavsky, 7...cxd4", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.e5 Nfd7 5.f4 c5 6.Nf3 Nc6 7.Be3 cxd4 *"),
("C11m", "French: Steinitz, Boleslavsky, 7...cxd4 8.Nxd4 Qb6", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.e5 Nfd7 5.f4 c5 6.Nf3 Nc6 7.Be3 cxd4 8.Nxd4 Qb6 *"),
("C11n", "French: Steinitz, Boleslavsky, 7...cxd4 8.Nxd4 Bc5", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.e5 Nfd7 5.f4 c5 6.Nf3 Nc6 7.Be3 cxd4 8.Nxd4 Bc5 *"),
("C11o", "French: 3.Nc3 Nf6 4.Bg5", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 *"),
("C11o", "French: Burn Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 dxe4 *"),
("C11p", "French: Burn, 5.Nxe4", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 dxe4 5.Nxe4 *"),
("C11p", "French: Burn, 5.Nxe4 Nbd7", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 dxe4 5.Nxe4 Nbd7 *"),
("C11q", "French: Burn, 5.Nxe4 Nbd7 6.Nf3", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 dxe4 5.Nxe4 Nbd7 6.Nf3 *"),
("C11r", "French: Burn, 5.Nxe4 Nbd7 6.Nf3 Be7", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 dxe4 5.Nxe4 Nbd7 6.Nf3 Be7 *"),
("C11r", "French: Burn, 5.Nxe4 Nbd7 6.Nf3 Be7", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 dxe4 5.Nxe4 Nbd7 6.Nf3 Be7 7.Nxf6+ Nxf6 *"),
("C11s", "French: Burn, 5.Nxe4 Be7", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 dxe4 5.Nxe4 Be7 *"),
("C11t", "French: Burn, 6.Bxf6 Bxf6", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 dxe4 5.Nxe4 Be7 6.Bxf6 Bxf6 *"),
("C11u", "French: Burn, 6.Bxf6 gxf6", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 dxe4 5.Nxe4 Be7 6.Bxf6 gxf6 *"),
("C11v", "French: Burn, 6.Bxf6 gxf6 7.Nf3", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 dxe4 5.Nxe4 Be7 6.Bxf6 gxf6 7.Nf3 *"),
("C11w", "French: Burn, 6.Bxf6 gxf6 7.Nf3 b6", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 dxe4 5.Nxe4 Be7 6.Bxf6 gxf6 7.Nf3 b6 *"),
("C11w", "French: Burn, 6.Bxf6 gxf6 7.Nf3 b6 8.Bd3", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 dxe4 5.Nxe4 Be7 6.Bxf6 gxf6 7.Nf3 b6 8.Bd3 *"),
("C11w", "French: Burn, 6.Bxf6 gxf6 7.Nf3 b6 8.Bc4", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 dxe4 5.Nxe4 Be7 6.Bxf6 gxf6 7.Nf3 b6 8.Bc4 *"),
("C11x", "French: Burn, 6.Bxf6 gxf6 7.Nf3 f5", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 dxe4 5.Nxe4 Be7 6.Bxf6 gxf6 7.Nf3 f5 *"),
("C12a", "French: MacCutcheon", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Bb4 *"),
("C12b", "French: MacCutcheon, 5.exd5", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Bb4 5.exd5 *"),
("C12c", "French: MacCutcheon, 5.exd5 Qxd5", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Bb4 5.exd5 Qxd5 *"),
("C12c", "French: MacCutcheon, Bogoljubow Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Bb4 5.exd5 Qxd5 6.Bxf6 gxf6 7.Qd2 Qa5 *"),
("C12d", "French: MacCutcheon, Advance Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Bb4 5.e5 *"),
("C12d", "French: MacCutcheon, Advance Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Bb4 5.e5 h6 *"),
("C12e", "French: MacCutcheon, Chigorin Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Bb4 5.e5 h6 6.exf6 *"),
("C12e", "French: MacCutcheon, Grigoriev Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Bb4 5.e5 h6 6.exf6 hxg5 7.fxg7 Rg8 8.h4 gxh4 9.Qg4 *"),
("C12f", "French: MacCutcheon, Olland (Dutch) Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Bb4 5.e5 h6 6.Bc1 *"),
("C12g", "French: MacCutcheon, Bernstein Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Bb4 5.e5 h6 6.Bh4 *"),
("C12h", "French: MacCutcheon, Janowski Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Bb4 5.e5 h6 6.Be3 *"),
("C12i", "French: MacCutcheon, 6.Bd2", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Bb4 5.e5 h6 6.Bd2 *"),
("C12i", "French: MacCutcheon, Tartakower Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Bb4 5.e5 h6 6.Bd2 Nfd7 *"),
("C12j", "French: MacCutcheon, Lasker Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Bb4 5.e5 h6 6.Bd2 Bxc3 *"),
("C12k", "French: MacCutcheon, Lasker, 7.bxc3", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Bb4 5.e5 h6 6.Bd2 Bxc3 7.bxc3 *"),
("C12l", "French: MacCutcheon, 8.Qg4", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Bb4 5.e5 h6 6.Bd2 Bxc3 7.bxc3 Ne4 8.Qg4 *"),
("C12l", "French: MacCutcheon, 8.Qg4 Kf8", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Bb4 5.e5 h6 6.Bd2 Bxc3 7.bxc3 Ne4 8.Qg4 Kf8 *"),
("C12l", "French: MacCutcheon, Duras Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Bb4 5.e5 h6 6.Bd2 Bxc3 7.bxc3 Ne4 8.Qg4 Kf8 9.Bc1 *"),
("C12m", "French: MacCutcheon, 8.Qg4 Kf8 9.Bd3", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Bb4 5.e5 h6 6.Bd2 Bxc3 7.bxc3 Ne4 8.Qg4 Kf8 9.Bd3 *"),
("C12n", "French: MacCutcheon, 8.Qg4 g6", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Bb4 5.e5 h6 6.Bd2 Bxc3 7.bxc3 Ne4 8.Qg4 g6 *"),
("C12o", "French: MacCutcheon, 8.Qg4 g6 9.Bd3", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Bb4 5.e5 h6 6.Bd2 Bxc3 7.bxc3 Ne4 8.Qg4 g6 9.Bd3 *"),
("C12p", "French: MacCutcheon, Main Line", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Bb4 5.e5 h6 6.Bd2 Bxc3 7.bxc3 Ne4 8.Qg4 g6 9.Bd3 Nxd2 10.Kxd2 c5 *"),
("C12q", "French: MacCutcheon, Main Line, 11.Nf3", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Bb4 5.e5 h6 6.Bd2 Bxc3 7.bxc3 Ne4 8.Qg4 g6 9.Bd3 Nxd2 10.Kxd2 c5 11.Nf3 *"),
("C12r", "French: MacCutcheon, Main Line, 11.Nf3 Nc6", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Bb4 5.e5 h6 6.Bd2 Bxc3 7.bxc3 Ne4 8.Qg4 g6 9.Bd3 Nxd2 10.Kxd2 c5 11.Nf3 Nc6 *"),
("C13a", "French: Classical", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 *"),
("C13b", "French: Classical, Anderssen Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.Bxf6 *"),
("C13c", "French: Classical, Anderssen-Richter Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.Bxf6 Bxf6 6.e5 Be7 7.Qg4 *"),
("C13d", "French: Classical, 5.e5", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 *"),
("C13d", "French: Classical, Nimzowitsch Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Ng8 *"),
("C13e", "French: Classical, Frankfurt Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Ng8 6.Be3 b6 *"),
("C13f", "French: Classical, Tartakower Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Ne4 *"),
("C13g", "French: Classical, Tartakower, 6.Bxe7", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Ne4 6.Bxe7 *"),
("C13h", "French: Classical, 5.e5 Nfd7", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 *"),
("C13i", "French: Chatard-Alekhine Attack", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.h4 *"),
("C13j", "French: Chatard-Alekhine, Teichmann Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.h4 f6 *"),
("C13k", "French: Chatard-Alekhine, Spielmann Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.h4 O-O *"),
("C13l", "French: Chatard-Alekhine, 6...Bxg5", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.h4 Bxg5 *"),
("C13l", "French: Chatard-Alekhine, 6...Bxg5 7.hxg5 Qxg5", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.h4 Bxg5 7.hxg5 Qxg5 *"),
("C13m", "French: Chatard-Alekhine, 6...Bxg5 7.hxg5 Qxg5 8.Nh3 Qe7 9.Nf4", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.h4 Bxg5 7.hxg5 Qxg5 8.Nh3 Qe7 9.Nf4 *"),
("C13n", "French: Chatard-Alekhine, Maroczy Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.h4 a6 *"),
("C13o", "French: Chatard-Alekhine, Maroczy, 7.Qg4 Bxg5", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.h4 a6 7.Qg4 Bxg5 *"),
("C13p", "French: Chatard-Alekhine, Breyer Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.h4 c5 *"),
("C13q", "French: Chatard-Alekhine, Breyer, 7.Bxe7", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.h4 c5 7.Bxe7 *"),
("C13r", "French: Chatard-Alekhine, Breyer, 7.Bxe7 Kxe7", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.h4 c5 7.Bxe7 Kxe7 *"),
("C14a", "French: Classical, 6.Bxe7 Qxe7", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.Bxe7 Qxe7 *"),
("C14b", "French: Classical, Pollock Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.Bxe7 Qxe7 7.Qg4 *"),
("C14c", "French: Classical, Tarrasch Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.Bxe7 Qxe7 7.Bd3 *"),
("C14d", "French: Classical, Alapin Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.Bxe7 Qxe7 7.Nb5 *"),
("C14e", "French: Classical, Rubinstein Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.Bxe7 Qxe7 7.Qd2 *"),
("C14f", "French: Classical, Steinitz Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.Bxe7 Qxe7 7.f4 *"),
("C14g", "French: Classical, Steinitz, 7...a6", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.Bxe7 Qxe7 7.f4 a6 *"),
("C14h", "French: Classical, Steinitz, 7...a6 8.Nf3 c5", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.Bxe7 Qxe7 7.f4 a6 8.Nf3 c5 *"),
("C14i", "French: Classical, Steinitz, 7...a6 8.Nf3 c5", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.Bxe7 Qxe7 7.f4 a6 8.Nf3 c5 9.dxc5 *"),
("C14j", "French: Classical, Steinitz, 7...O-O", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.Bxe7 Qxe7 7.f4 O-O *"),
("C14k", "French: Classical, Steinitz, 7...O-O 8.Nf3 c5", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.Bxe7 Qxe7 7.f4 O-O 8.Nf3 c5 *"),
("C14l", "French: Classical, Steinitz, 8.Nf3 c5 9.dxc5", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.Bxe7 Qxe7 7.f4 O-O 8.Nf3 c5 9.dxc5 *"),
("C14m", "French: Classical, Steinitz, 9.Qd2", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.Bxe7 Qxe7 7.f4 O-O 8.Nf3 c5 9.Qd2 *"),
("C14m", "French: Classical, Stahlberg Variation", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.Bxe7 Qxe7 7.f4 O-O 8.Nf3 c5 9.Qd2 Nc6 10.O-O-O c4 *"),
("C14n", "French: Classical, Steinitz, 9.Qd2 Nc6 10.dxc5", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.Bxe7 Qxe7 7.f4 O-O 8.Nf3 c5 9.Qd2 Nc6 10.dxc5 *"),
("C14o", "French: Classical, Steinitz, 9.Qd2 Nc6 10.dxc5 Qxc5", "1.e4 e6 2.d4 d5 3.Nc3 Nf6 4.Bg5 Be7 5.e5 Nfd7 6.Bxe7 Qxe7 7.f4 O-O 8.Nf3 c5 9.Qd2 Nc6 10.dxc5 Qxc5 *"),
("C15a", "French: Winawer", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 *"),
("C15b", "French: Winawer, 4.exd5", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.exd5 *"),
("C15b", "French: Winawer, 4.exd5 Qxd5", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.exd5 Qxd5 *"),
("C15c", "French: Winawer, 4.Qg4", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Qg4 *"),
("C15d", "French: Winawer, 4.Qd3", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Qd3 *"),
("C15d", "French: Winawer, 4.Qd3 dxe4", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Qd3 dxe4 *"),
("C15e", "French: Winawer, 4.Bd3", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Bd3 *"),
("C15f", "French: Winawer, 4.Bd3 c5", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Bd3 c5 *"),
("C15f", "French: Winawer, Kondratiyev Variation", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Bd3 c5 5.exd5 Qxd5 6.Bd2 *"),
("C15g", "French: Winawer, 4.Bd3 dxe4", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Bd3 dxe4 *"),
("C15g", "French: Winawer, 4.Bd3 dxe4 5.Bxe4", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Bd3 dxe4 5.Bxe4 *"),
("C15g", "French: Winawer, 4.Bd3 dxe4 5.Bxe4 Nf6", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Bd3 dxe4 5.Bxe4 Nf6 *"),
("C15h", "French: Winawer, Mueller-Zhuravlev Gambit", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Bd2 *"),
("C15h", "French: Winawer, Mueller-Zhuravlev Gambit", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Bd2 dxe4 *"),
("C15i", "French: Winawer, Mueller-Zhuravlev Gambit, 5.Qg4", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Bd2 dxe4 5.Qg4 *"),
("C15i", "French: Winawer, Mueller-Zhuravlev Gambit, Kunin Variation", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Bd2 dxe4 5.Qg4 Qxd4 *"),
("C15j", "French: Winawer, Mueller-Zhuravlev Gambit, 5.Qg4 Nf6", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Bd2 dxe4 5.Qg4 Nf6 *"),
("C15k", "French: Winawer, 4.a3", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.a3 *"),
("C15k", "French: Winawer, 4.a3 Bxc3+", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.a3 Bxc3+ *"),
("C15k", "French: Winawer, 4.a3 Bxc3+", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.a3 Bxc3+ 5.bxc3 *"),
("C15k", "French: Winawer, 4.a3 Bxc3+ 5.bxc3 dxe4", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.a3 Bxc3+ 5.bxc3 dxe4 *"),
("C15l", "French: Winawer, Winkelmann-Reimer Gambit (WRG)", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.a3 Bxc3+ 5.bxc3 dxe4 6.f3 *"),
("C15l", "French: Winawer, Winkelmann-Reimer Gambit Accepted", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.a3 Bxc3+ 5.bxc3 dxe4 6.f3 exf3 *"),
("C15l", "French: Winawer, Winkelmann-Reimer, Huebner Defence", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.a3 Bxc3+ 5.bxc3 dxe4 6.f3 e5 *"),
("C15l", "French: Winawer, Winkelmann-Reimer, 6...c5", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.a3 Bxc3+ 5.bxc3 dxe4 6.f3 c5 *"),
("C15m", "French: Winawer, 4.a3 Bxc3+ 5.bxc3 dxe4 6.Qg4", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.a3 Bxc3+ 5.bxc3 dxe4 6.Qg4 *"),
("C15n", "French: Winawer, Alekhine Gambit", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Ne2 *"),
("C15o", "French: Winawer, Alekhine Gambit, 4...Nf6", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Ne2 Nf6 *"),
("C15p", "French: Winawer, Alekhine Gambit, 4...Nc6", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Ne2 Nc6 *"),
("C15q", "French: Winawer, Alekhine Gambit Accepted", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Ne2 dxe4 *"),
("C15q", "French: Winawer, Alekhine Gambit, 5.a3 Bxc3+", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Ne2 dxe4 5.a3 Bxc3+ *"),
("C15r", "French: Winawer, Alekhine Gambit, Kan Variation", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Ne2 dxe4 5.a3 Bxc3+ 6.Nxc3 Nc6 *"),
("C15s", "French: Winawer, Alekhine Gambit, Kan, 7.Bb5", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Ne2 dxe4 5.a3 Bxc3+ 6.Nxc3 Nc6 7.Bb5 *"),
("C15t", "French: Winawer, Alekhine Gambit, 5.a3 Be7", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Ne2 dxe4 5.a3 Be7 *"),
("C15u", "French: Winawer, Alekhine Gambit, 5.a3 Be7 6.Nxe4 Nf6", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Ne2 dxe4 5.a3 Be7 6.Nxe4 Nf6 *"),
("C15u", "French: Winawer, Alekhine Gambit, Alatortsev Variation", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Ne2 dxe4 5.a3 Be7 6.Nxe4 Nf6 7.N2g3 O-O 8.Be2 Nc6 *"),
("C15v", "French: Winawer, Alekhine Gambit, 5.a3 Be7 6.Nxe4 Nf6 7.Qd3", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.Ne2 dxe4 5.a3 Be7 6.Nxe4 Nf6 7.Qd3 *"),
("C16a", "French: Winawer, Advance Variation", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 *"),
("C16b", "French: Winawer, 4.e5 b6", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 b6 *"),
("C16c", "French: Winawer, 4.e5 b6 5.Qg4", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 b6 5.Qg4 *"),
("C16d", "French: Winawer, 4.e5 b6 5.Qg4 Bf8 6.Bg5", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 b6 5.Qg4 Bf8 6.Bg5 *"),
("C16e", "French: Winawer, 4.e5 b6 5.a3", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 b6 5.a3 *"),
("C16e", "French: Winawer, 4.e5 b6 5.a3 Bf8", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 b6 5.a3 Bf8 *"),
("C16f", "French: Winawer, Petrosian Variation", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 Qd7 *"),
("C16g", "French: Winawer, Petrosian, 5.Bd2", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 Qd7 5.Bd2 *"),
("C16h", "French: Winawer, Petrosian, 5.a3", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 Qd7 5.a3 *"),
("C16i", "French: Winawer, Petrosian, 5.a3 Bxc3+ 6.bxc3 b6", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 Qd7 5.a3 Bxc3+ 6.bxc3 b6 *"),
("C16i", "French: Winawer, Petrosian, 5.a3 Bxc3+ 6.bxc3 b6 7.Qg4", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 Qd7 5.a3 Bxc3+ 6.bxc3 b6 7.Qg4 *"),
("C16j", "French: Winawer, Advance, 4...Ne7", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 Ne7 *"),
("C16k", "French: Winawer, Advance, 4...Ne7 5.Bd2", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 Ne7 5.Bd2 *"),
("C16l", "French: Winawer, Advance, 4...Ne7 5.Bd2 b6", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 Ne7 5.Bd2 b6 *"),
("C16m", "French: Winawer, Advance, 4...Ne7 5.a3", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 Ne7 5.a3 *"),
("C16n", "French: Winawer, Advance, 4...Ne7 5.a3 Bxc3+ 6.bxc3 b6", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 Ne7 5.a3 Bxc3+ 6.bxc3 b6 *"),
("C16o", "French: Winawer, Advance, 4...Ne7 5.a3 Bxc3+ 6.bxc3 b6 7.Qg4", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 Ne7 5.a3 Bxc3+ 6.bxc3 b6 7.Qg4 *"),
("C17a", "French: Winawer, Advance, 4...c5", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 *"),
("C17b", "French: Winawer, Advance, 5.dxc5", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.dxc5 *"),
("C17c", "French: Winawer, Russian Variation", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.Qg4 *"),
("C17d", "French: Winawer, Russian, 6.dxc5", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.Qg4 Ne7 6.dxc5 *"),
("C17e", "French: Winawer, Bogoljubow Variation", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.Bd2 *"),
("C17f", "French: Winawer, Bogoljubow, 5...cxd4", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.Bd2 cxd4 *"),
("C17g", "French: Winawer, Bogoljubow, 5...Ne7", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.Bd2 Ne7 *"),
("C17h", "French: Winawer, Bogoljubow, 5...Ne7 6.a3", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.Bd2 Ne7 6.a3 *"),
("C17i", "French: Winawer, Bogoljubow, 5...Ne7 6.dxc5", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.Bd2 Ne7 6.dxc5 *"),
("C17j", "French: Winawer, Bogoljubow, 5...Ne7 6.Nb5", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.Bd2 Ne7 6.Nb5 *"),
("C17k", "French: Winawer, 5.a3", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 *"),
("C17l", "French: Winawer, 5.a3 cxd4", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 cxd4 *"),
("C17l", "French: Winawer, Rauzer Variation", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 cxd4 6.axb4 dxc3 7.Nf3 *"),
("C17m", "French: Winawer, Swiss Variation", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Ba5 *"),
("C17n", "French: Winawer, Swiss, 6.b4", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Ba5 6.b4 *"),
("C17o", "French: Winawer, Swiss, 6.b4 cxd4 7.Qg4", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Ba5 6.b4 cxd4 7.Qg4 *"),
("C17p", "French: Winawer, Swiss, 6.b4 cxd4 7.Qg4 Ne7 8.bxa5", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Ba5 6.b4 cxd4 7.Qg4 Ne7 8.bxa5 *"),
("C17q", "French: Winawer, Swiss, 6.b4 cxd4 7.Nb5", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Ba5 6.b4 cxd4 7.Nb5 *"),
("C17r", "French: Winawer, Swiss, 6.b4 cxd4 7.Nb5 Bc7 8.f4", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Ba5 6.b4 cxd4 7.Nb5 Bc7 8.f4 *"),
("C17s", "French: Winawer, Swiss, 6.b4 cxd4 7.Nb5 Bc7 8.f4 Bd7", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Ba5 6.b4 cxd4 7.Nb5 Bc7 8.f4 Bd7 *"),
("C18a", "French: Winawer, 5...Bxc3+", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ *"),
("C18a", "French: Winawer, 5...Bxc3+ 6.bxc3", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 *"),
("C18b", "French: Winawer, 6...Qa5", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Qa5 *"),
("C18c", "French: Winawer, 6...Qc7", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Qc7 *"),
("C18d", "French: Winawer, 6...Qc7 7.Qg4", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Qc7 7.Qg4 *"),
("C18d", "French: Winawer, 6...Qc7 7.Qg4 f6", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Qc7 7.Qg4 f6 *"),
("C18e", "French: Winawer, 6...Qc7 7.Qg4 f5", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Qc7 7.Qg4 f5 *"),
("C18f", "French: Winawer, 6...Ne7", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 *"),
("C18g", "French: Winawer, 6...Ne7 7.h4", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.h4 *"),
("C18h", "French: Winawer, 6...Ne7 7.h4 Qc7", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.h4 Qc7 *"),
("C18h", "French: Winawer, 6...Ne7 7.h4 Qc7", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.h4 Qc7 8.Nf3 *"),
("C18i", "French: Winawer, 6...Ne7 7.h4 Nbc6", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.h4 Nbc6 *"),
("C18i", "French: Winawer, 6...Ne7 7.h4 Nbc6", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.h4 Nbc6 8.Nf3 *"),
("C18j", "French: Winawer, 6...Ne7 7.Qg4", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Qg4 *"),
("C18k", "French: Winawer, 6...Ne7 7.Qg4 Nbc6", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Qg4 Nbc6 *"),
("C18l", "French: Winawer, 6...Ne7 7.Qg4 Kf8", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Qg4 Kf8 *"),
("C18m", "French: Winawer, 6...Ne7 7.Qg4 cxd4", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Qg4 cxd4 *"),
("C18n", "French: Winawer, 6...Ne7 7.Qg4 O-O", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Qg4 O-O *"),
("C18o", "French: Winawer, 6...Ne7 7.Qg4 O-O 8.Bd3", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Qg4 O-O 8.Bd3 *"),
("C18p", "French: Winawer, 6...Ne7 7.Qg4 O-O 8.Bd3 Nbc6", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Qg4 O-O 8.Bd3 Nbc6 *"),
("C18q", "French: Winawer, 6...Ne7 7.Qg4 O-O 8.Nf3", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Qg4 O-O 8.Nf3 *"),
("C18q", "French: Winawer, 6...Ne7 7.Qg4 O-O 8.Nf3 Nbc6", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Qg4 O-O 8.Nf3 Nbc6 *"),
("C18r", "French: Winawer, Poisoned Pawn", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Qg4 Qc7 *"),
("C18r", "French: Winawer, Poisoned Pawn, 8.Qxg7", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Qg4 Qc7 8.Qxg7 *"),
("C18r", "French: Winawer, Poisoned Pawn, 10.Qd3", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Qg4 Qc7 8.Qxg7 Rg8 9.Qxh7 cxd4 10.Qd3 *"),
("C18s", "French: Winawer, Poisoned Pawn, Euwe-Gligoric Variation", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Qg4 Qc7 8.Qxg7 Rg8 9.Qxh7 cxd4 10.Kd1 *"),
("C18t", "French: Winawer, Poisoned Pawn, Konstantinopolsky Variation", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Qg4 Qc7 8.Qxg7 Rg8 9.Qxh7 cxd4 10.Ne2 *"),
("C18u", "French: Winawer, Poisoned Pawn, Main Line", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Qg4 Qc7 8.Qxg7 Rg8 9.Qxh7 cxd4 10.Ne2 Nbc6 11.f4 Bd7 12.Qd3 dxc3 *"),
("C18v", "French: Winawer, Poisoned Pawn, Main Line, 13.Nxc3", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Qg4 Qc7 8.Qxg7 Rg8 9.Qxh7 cxd4 10.Ne2 Nbc6 11.f4 Bd7 12.Qd3 dxc3 13.Nxc3 *"),
("C19a", "French: Winawer, Smyslov Variation", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.a4 *"),
("C19b", "French: Winawer, Smyslov, 7...Qa5", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.a4 Qa5 *"),
("C19c", "French: Winawer, 6...Ne7 7.Nf3", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Nf3 *"),
("C19d", "French: Winawer, 6...Ne7 7.Nf3 Qc7", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Nf3 Qc7 *"),
("C19e", "French: Winawer, 6...Ne7 7.Nf3 Qc7 8.a4", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Nf3 Qc7 8.a4 *"),
("C19f", "French: Winawer, 6...Ne7 7.Nf3 Qc7 8.a4 b6", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Nf3 Qc7 8.a4 b6 *"),
("C19g", "French: Winawer, 6...Ne7 7.Nf3 Qa5", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Nf3 Qa5 *"),
("C19h", "French: Winawer, 6...Ne7 7.Nf3 b6", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Nf3 b6 *"),
("C19h", "French: Winawer, 6...Ne7 7.Nf3 b6 8.a4", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Nf3 b6 8.a4 *"),
("C19i", "French: Winawer, 6...Ne7 7.Nf3 b6 8.Bb5+", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Nf3 b6 8.Bb5+ *"),
("C19j", "French: Winawer, 6...Ne7 7.Nf3 Bd7", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Nf3 Bd7 *"),
("C19k", "French: Winawer, 6...Ne7 7.Nf3 Bd7 8.a4", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Nf3 Bd7 8.a4 *"),
("C19l", "French: Winawer, 6...Ne7 7.Nf3 Nbc6", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Nf3 Nbc6 *"),
("C19m", "French: Winawer, 6...Ne7 7.Nf3 Nbc6 8.a4", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Nf3 Nbc6 8.a4 *"),
("C19m", "French: Winawer, 6...Ne7 7.Nf3 Nbc6 8.a4 Bd7", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Nf3 Nbc6 8.a4 Bd7 *"),
("C19n", "French: Winawer, 6...Ne7 7.Nf3 Nbc6 8.a4 Qa5", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Nf3 Nbc6 8.a4 Qa5 *"),
("C19n", "French: Winawer, 6...Ne7 7.Nf3 Nbc6 8.a4 Qa5 9.Qd2", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Nf3 Nbc6 8.a4 Qa5 9.Qd2 *"),
("C19o", "French: Winawer, 6...Ne7 7.Nf3 Nbc6 8.a4 Qa5 9.Qd2 Bd7", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Nf3 Nbc6 8.a4 Qa5 9.Qd2 Bd7 *"),
("C19p", "French: Winawer, 6...Ne7 7.Nf3 Nbc6 8.a4 Qa5+ 9.Bd2", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Nf3 Nbc6 8.a4 Qa5 9.Bd2 *"),
("C19q", "French: Winawer, 6...Ne7 7.Nf3 Nbc6 8.a4 Qa5+ 9.Bd2 Bd7", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Nf3 Nbc6 8.a4 Qa5 9.Bd2 Bd7 *"),
("C19r", "French: Winawer, 6...Ne7 7.Nf3 Nbc6 8.a4 Qa5+ 9.Bd2 Bd7 10.Bb5", "1.e4 e6 2.d4 d5 3.Nc3 Bb4 4.e5 c5 5.a3 Bxc3+ 6.bxc3 Ne7 7.Nf3 Nbc6 8.a4 Qa5 9.Bd2 Bd7 10.Bb5 *"),
("C20", "Open Game", "1.e4 e5 *"),
("C20", "Open Game: Mengarini Opening", "1.e4 e5 2.a3 *"),
("C20", "Open Game: Patzer/Parnham Opening", "1.e4 e5 2.Qh5 *"),
("C20", "Open Game: Napoleon's Opening", "1.e4 e5 2.Qf3 *"),
("C20", "Open Game: 2.d3", "1.e4 e5 2.d3 *"),
("C20", "Open Game: 2.d3 d5", "1.e4 e5 2.d3 d5 *"),
("C20", "Open Game: 2.c4", "1.e4 e5 2.c4 *"),
("C20", "Open Game: Lopez/Mcleod Opening", "1.e4 e5 2.c3 *"),
("C20", "Open Game: Lopez/Mcleod, Lasa Gambit", "1.e4 e5 2.c3 f5 *"),
("C20", "Open Game: Alapin Opening", "1.e4 e5 2.Ne2 *"),
("C20", "Open Game: Alapin Opening", "1.e4 e5 2.Ne2 Nf6 *"),
("C20", "Open Game: Portuguese Opening", "1.e4 e5 2.Bb5 *"),
("C20", "Open Game: Portuguese, 2...Nf6", "1.e4 e5 2.Bb5 Nf6 *"),
("C20", "Open Game: Portuguese Gambit", "1.e4 e5 2.Bb5 Nf6 3.d4 *"),
("C20", "Open Game: Portuguese, 2...Nc6", "1.e4 e5 2.Bb5 Nc6 *"),
("C20", "Open Game: Portuguese, 2...c6", "1.e4 e5 2.Bb5 c6 *"),
("C21", "Centre Game", "1.e4 e5 2.d4 *"),
("C21", "Centre Game: Maroczy Defence", "1.e4 e5 2.d4 d6 *"),
("C21", "Centre Game: Maroczy Defence, 3.dxe5", "1.e4 e5 2.d4 d6 3.dxe5 *"),
("C21", "Centre Game: Maroczy Defence, Philidor Gambit", "1.e4 e5 2.d4 d6 3.dxe5 Bd7 *"),
("C21", "Centre Game: Queenswap line", "1.e4 e5 2.d4 d6 3.dxe5 dxe5 4.Qxd8+ *"),
("C21", "Centre Game", "1.e4 e5 2.d4 exd4 *"),
("C21", "Centre Game: 3.Nf3", "1.e4 e5 2.d4 exd4 3.Nf3 *"),
("C21", "Centre Game: Kieseritsky Variation", "1.e4 e5 2.d4 exd4 3.Nf3 c5 4.Bc4 b5 *"),
("C21", "Centre Game: Halasz Gambit", "1.e4 e5 2.d4 exd4 3.f4 *"),
("C21", "Danish Gambit", "1.e4 e5 2.d4 exd4 3.c3 *"),
("C21", "Danish Gambit: Svenonius Defence", "1.e4 e5 2.d4 exd4 3.c3 Ne7 *"),
("C21", "Danish Gambit: Sorensen Defence", "1.e4 e5 2.d4 exd4 3.c3 d5 *"),
("C21", "Danish Gambit: Accepted", "1.e4 e5 2.d4 exd4 3.c3 dxc3 *"),
("C21", "Danish Gambit: Accepted, 4.Bc4", "1.e4 e5 2.d4 exd4 3.c3 dxc3 4.Bc4 *"),
("C21", "Danish Gambit: Accepted, 4.Bc4 cxb2 5.Bxb2", "1.e4 e5 2.d4 exd4 3.c3 dxc3 4.Bc4 cxb2 5.Bxb2 *"),
("C21", "Danish Gambit: Copenhagen Defence", "1.e4 e5 2.d4 exd4 3.c3 dxc3 4.Bc4 cxb2 5.Bxb2 Bb4+ *"),
("C21", "Danish Gambit: Chigorin Defence", "1.e4 e5 2.d4 exd4 3.c3 dxc3 4.Bc4 cxb2 5.Bxb2 Qe7 *"),
("C21", "Danish Gambit: Classical Defence", "1.e4 e5 2.d4 exd4 3.c3 dxc3 4.Bc4 cxb2 5.Bxb2 Nf6 *"),
("C21", "Danish Gambit: Schlechter Defence", "1.e4 e5 2.d4 exd4 3.c3 dxc3 4.Bc4 cxb2 5.Bxb2 d5 *"),
("C22", "Centre Game", "1.e4 e5 2.d4 exd4 3.Qxd4 *"),
("C22", "Centre Game", "1.e4 e5 2.d4 exd4 3.Qxd4 Nc6 *"),
("C22", "Centre Game: Hall Variation", "1.e4 e5 2.d4 exd4 3.Qxd4 Nc6 4.Qc4 *"),
("C22", "Centre Game: Paulsen Attack", "1.e4 e5 2.d4 exd4 3.Qxd4 Nc6 4.Qe3 *"),
("C22", "Centre Game: Charousek Variation", "1.e4 e5 2.d4 exd4 3.Qxd4 Nc6 4.Qe3 Bb4+ 5.c3 Be7 *"),
("C22", "Centre Game: Berger Variation", "1.e4 e5 2.d4 exd4 3.Qxd4 Nc6 4.Qe3 Nf6 *"),
("C22", "Centre Game: Kupreichik Variation", "1.e4 e5 2.d4 exd4 3.Qxd4 Nc6 4.Qe3 Nf6 5.Nc3 Bb4 6.Bd2 O-O 7.O-O-O Re8 8.Bc4 d6 9.Nh3 *"),
("C23", "Bishop's Opening", "1.e4 e5 2.Bc4 *"),
("C23", "Bishop's Opening: Anderssen Gambit", "1.e4 e5 2.Bc4 b5 *"),
("C23", "Bishop's Opening: Philidor Counterattack", "1.e4 e5 2.Bc4 c6 *"),
("C23", "Bishop's Opening: Lisitsin Variation", "1.e4 e5 2.Bc4 c6 3.d4 d5 4.exd5 cxd5 5.Bb5+ Bd7 6.Bxd7+ Nxd7 7.dxe5 Nxe5 8.Ne2 *"),
("C23", "Bishop's Opening: Calabrese Countergambit", "1.e4 e5 2.Bc4 f5 *"),
("C23", "Bishop's Opening: Calabrese Countergambit, Jaenisch Variation", "1.e4 e5 2.Bc4 f5 3.d3 *"),
("C23", "Bishop's Opening: Classical Variation", "1.e4 e5 2.Bc4 Bc5 *"),
("C23", "Bishop's Opening: 2...Bc5 3.Qe2", "1.e4 e5 2.Bc4 Bc5 3.Qe2 *"),
("C23", "Bishop's Opening: Lopez Gambit", "1.e4 e5 2.Bc4 Bc5 3.Qe2 Nc6 4.c3 Nf6 5.f4 *"),
("C23", "Bishop's Opening: Lewis Gambit", "1.e4 e5 2.Bc4 Bc5 3.d4 *"),
("C23", "Bishop's Opening: MacDonnell Gambit", "1.e4 e5 2.Bc4 Bc5 3.b4 *"),
("C23", "Bishop's Opening: MacDonnell Double Gambit", "1.e4 e5 2.Bc4 Bc5 3.b4 Bxb4 4.f4 *"),
("C23", "Bishop's Opening: MacDonnell, Four Pawns Gambit", "1.e4 e5 2.Bc4 Bc5 3.b4 Bxb4 4.f4 exf4 5.Nf3 Be7 6.d4 Bh4+ 7.g3 fxg3 8.O-O gxh2+ 9.Kh1 *"),
("C23", "Bishop's Opening: Philidor Variation", "1.e4 e5 2.Bc4 Bc5 3.c3 *"),
("C23", "Bishop's Opening: del Rio Variation", "1.e4 e5 2.Bc4 Bc5 3.c3 Qg5 *"),
("C23", "Bishop's Opening: Lewis Countergambit", "1.e4 e5 2.Bc4 Bc5 3.c3 d5 *"),
("C23", "Bishop's Opening: Lewis Countergambit, Jaenisch", "1.e4 e5 2.Bc4 Bc5 3.c3 d5 4.Bxd5 Nf6 5.Qf3 *"),
("C23", "Bishop's Opening: 2...Bc5 3.c3 Nf6", "1.e4 e5 2.Bc4 Bc5 3.c3 Nf6 *"),
("C23", "Bishop's Opening: Pratt Variation", "1.e4 e5 2.Bc4 Bc5 3.c3 Nf6 4.d4 exd4 5.e5 d5 6.exf6 dxc4 7.Qh5 O-O *"),
("C24", "Bishop's Opening: Berlin Defence", "1.e4 e5 2.Bc4 Nf6 *"),
("C24", "Bishop's Opening: Greco Gambit", "1.e4 e5 2.Bc4 Nf6 3.f4 *"),
("C24", "Bishop's Opening: 3.d3", "1.e4 e5 2.Bc4 Nf6 3.d3 *"),
("C24", "Bishop's Opening: 3.d3 d5", "1.e4 e5 2.Bc4 Nf6 3.d3 d5 *"),
("C24", "Bishop's Opening: Paulsen Defence", "1.e4 e5 2.Bc4 Nf6 3.d3 c6 *"),
("C24", "Bishop's Opening: Paulsen, 4.Nf3", "1.e4 e5 2.Bc4 Nf6 3.d3 c6 4.Nf3 *"),
("C24", "Bishop's Opening: 3.d4", "1.e4 e5 2.Bc4 Nf6 3.d4 *"),
("C24", "Bishop's Opening: Urusov Gambit", "1.e4 e5 2.Bc4 Nf6 3.d4 exd4 4.Nf3 *"),
("C24", "Bishop's Opening: Urusov Gambit, Panov Variation", "1.e4 e5 2.Bc4 Nf6 3.d4 exd4 4.Nf3 d5 5.exd5 Bb4+ 6.c3 Qe7+ *"),
("C24", "Bishop's Opening: Urusov Gambit, Marshall Variation", "1.e4 e5 2.Bc4 Nf6 3.d4 exd4 4.Nf3 Nxe4 5.Qxd4 Nf6 6.Nc3 Nc6 7.Qh4 *"),
("C24", "Bishop's Opening: Urusov Gambit, 6.Bg5 Nc6 Defence", "1.e4 e5 2.Bc4 Nf6 3.d4 exd4 4.Nf3 Nxe4 5.Qxd4 Nf6 6.Bg5 Nc6 *"),
("C24", "Bishop's Opening: Urusov Gambit, 7...c6 Defence", "1.e4 e5 2.Bc4 Nf6 3.d4 exd4 4.Nf3 Nxe4 5.Qxd4 Nf6 6.Bg5 Be7 7.Nc3 c6 *"),
("C24", "Bishop's Opening: Urusov Gambit, Larsen Variation", "1.e4 e5 2.Bc4 Nf6 3.d4 exd4 4.Nf3 Nxe4 5.Qxd4 Nf6 6.Bg5 Be7 7.Nc3 c6 8.O-O-O d5 9.Rhe1 O-O 10.Qh4 *"),
("C24", "Bishop's Opening: Urusov Gambit, Karpov Variation", "1.e4 e5 2.Bc4 Nf6 3.d4 exd4 4.Nf3 Nxe4 5.Qxd4 Nf6 6.Bg5 Be7 7.Nc3 c6 8.O-O-O d5 9.Rhe1 Be6 10.Bd3 Nbd7 11.Qh4 c5 *"),
("C24", "Bishop's Opening: Urusov Gambit, Forintos/Haag Variation", "1.e4 e5 2.Bc4 Nf6 3.d4 exd4 4.Nf3 Nxe4 5.Qxd4 Nf6 6.Bg5 Be7 7.Nc3 c6 8.O-O-O d5 9.Rhe1 Be6 10.Bd3 Nbd7 11.Qh4 Nc5 12. Nd4 Ng8 *"),
("C24", "Bishop's Opening: Urusov Gambit, 7...Nc6 Defence", "1.e4 e5 2.Bc4 Nf6 3.d4 exd4 4.Nf3 Nxe4 5.Qxd4 Nf6 6.Bg5 Be7 7.Nc3 Nc6 *"),
("C24", "Bishop's Opening: Urusov Gambit, Keres Variation", "1.e4 e5 2.Bc4 Nf6 3.d4 exd4 4.Nf3 Nxe4 5.Qxd4 Nf6 6.Bg5 Be7 7.Nc3 Nc6 8.Qh4 d6 *"),
("C24", "Bishop's Opening: Urusov Gambit, Estrin Variation", "1.e4 e5 2.Bc4 Nf6 3.d4 exd4 4.Nf3 Nxe4 5.Qxd4 Nf6 6.Bg5 Be7 7.Nc3 Nc6 8.Qh4 d5 *"),
("C25a", "Vienna Game", "1.e4 e5 2.Nc3 *"),
("C25b", "Vienna: 2...Bb4", "1.e4 e5 2.Nc3 Bb4 *"),
("C25b", "Vienna: Zhuravlev", "1.e4 e5 2.Nc3 Bb4 3.Qg4 *"),
("C25c", "Vienna: 2...d6", "1.e4 e5 2.Nc3 d6 *"),
("C25c", "Vienna: 2...d6 3.Bc4", "1.e4 e5 2.Nc3 d6 3.Bc4 *"),
("C25d", "Vienna: 2...Bc5", "1.e4 e5 2.Nc3 Bc5 *"),
("C25d", "Vienna: Hammpe-Meitner", "1.e4 e5 2.Nc3 Bc5 3.Na4 *"),
("C25e", "Vienna: 2...Bc5 3.Bc4", "1.e4 e5 2.Nc3 Bc5 3.Bc4 *"),
("C25f", "Vienna: 2...Bc5 3.Nf3", "1.e4 e5 2.Nc3 Bc5 3.Nf3 *"),
("C25g", "Vienna: 2...Nc6", "1.e4 e5 2.Nc3 Nc6 *"),
("C25g", "Vienna: Fyfe Gambit", "1.e4 e5 2.Nc3 Nc6 3.d4 *"),
("C25h", "Vienna: Paulsen Variation", "1.e4 e5 2.Nc3 Nc6 3.g3 *"),
("C25i", "Vienna: Paulsen, 3...Bc5", "1.e4 e5 2.Nc3 Nc6 3.g3 Bc5 *"),
("C25j", "Vienna: 2...Nc6 3.Bc4", "1.e4 e5 2.Nc3 Nc6 3.Bc4 *"),
("C25j", "Vienna: 2...Nc6 3.Bc4 Bc5", "1.e4 e5 2.Nc3 Nc6 3.Bc4 Bc5 *"),
("C25k", "Vienna: 2...Nc6 3.Bc4 Bc5 4.d3", "1.e4 e5 2.Nc3 Nc6 3.Bc4 Bc5 4.d3 *"),
("C25l", "Vienna: 2...Nc6 3.Bc4 Bc5 4.Qg4", "1.e4 e5 2.Nc3 Nc6 3.Bc4 Bc5 4.Qg4 *"),
("C25m", "Vienna: 2...Nc6 3.f4", "1.e4 e5 2.Nc3 Nc6 3.f4 *"),
("C25m", "Vienna: 2...Nc6 3.f4", "1.e4 e5 2.Nc3 Nc6 3.f4 exf4 *"),
("C25n", "Vienna: Steinitz Gambit", "1.e4 e5 2.Nc3 Nc6 3.f4 exf4 4.d4 *"),
("C25n", "Vienna: Steinitz Gambit, Fraser-Minckwitz Variation", "1.e4 e5 2.Nc3 Nc6 3.f4 exf4 4.d4 Qh4+ 5.Ke2 b6 *"),
("C25o", "Vienna: Steinitz Gambit, Zukertort Defence", "1.e4 e5 2.Nc3 Nc6 3.f4 exf4 4.d4 Qh4+ 5.Ke2 d5 *"),
("C25p", "Vienna: 2...Nc6 3.f4 exf4 4.Nf3", "1.e4 e5 2.Nc3 Nc6 3.f4 exf4 4.Nf3 *"),
("C25p", "Vienna: 2...Nc6 3.f4 exf4 4.Nf3 g5", "1.e4 e5 2.Nc3 Nc6 3.f4 exf4 4.Nf3 g5 *"),
("C25p", "Vienna: Hamppe-Muzio Gambit", "1.e4 e5 2.Nc3 Nc6 3.f4 exf4 4.Nf3 g5 5.Bc4 g4 6.O-O *"),
("C25p", "Vienna: Hamppe-Muzio, Dubois Variation", "1.e4 e5 2.Nc3 Nc6 3.f4 exf4 4.Nf3 g5 5.Bc4 g4 6.O-O gxf3 7.Qxf3 Ne5 8.Qxf4 Qf6 *"),
("C25q", "Vienna: Hamppe-Allgaier Gambit", "1.e4 e5 2.Nc3 Nc6 3.f4 exf4 4.Nf3 g5 5.h4 *"),
("C25q", "Vienna: Hamppe-Allgaier Gambit, Alapin Variation", "1.e4 e5 2.Nc3 Nc6 3.f4 exf4 4.Nf3 g5 5.h4 g4 6.Ng5 d6 *"),
("C25r", "Vienna: Pierce Gambit", "1.e4 e5 2.Nc3 Nc6 3.f4 exf4 4.Nf3 g5 5.d4 *"),
("C25r", "Vienna: Pierce Gambit, Rushmere Attack", "1.e4 e5 2.Nc3 Nc6 3.f4 exf4 4.Nf3 g5 5.d4 g4 6.Bc4 gxf3 7.O-O d5 8.exd5 Bg4 9.dxc6 *"),
("C26a", "Vienna: 2...Nf6", "1.e4 e5 2.Nc3 Nf6 *"),
("C26a", "Vienna: Mengarini Variation", "1.e4 e5 2.Nc3 Nf6 3.a3 *"),
("C26b", "Vienna: 2...Nf6 3.d3", "1.e4 e5 2.Nc3 Nf6 3.d3 *"),
("C26c", "Vienna: Smyslov Variation", "1.e4 e5 2.Nc3 Nf6 3.g3 *"),
("C26d", "Vienna: Smyslov, 3...Nc6", "1.e4 e5 2.Nc3 Nf6 3.g3 Nc6 *"),
("C26d", "Vienna: Smyslov, 3...Nc6", "1.e4 e5 2.Nc3 Nf6 3.g3 Nc6 4.Bg2 *"),
("C26e", "Vienna: Smyslov, 3...Bc5", "1.e4 e5 2.Nc3 Nf6 3.g3 Bc5 *"),
("C26e", "Vienna: Smyslov, 3...Bc5", "1.e4 e5 2.Nc3 Nf6 3.g3 Bc5 4.Bg2 *"),
("C26f", "Vienna: Smyslov, 3...Bc5 4.Bg2 d6", "1.e4 e5 2.Nc3 Nf6 3.g3 Bc5 4.Bg2 d6 *"),
("C26g", "Vienna: Smyslov, 3...Bc5 4.Bg2 O-O", "1.e4 e5 2.Nc3 Nf6 3.g3 Bc5 4.Bg2 O-O *"),
("C26h", "Vienna: Smyslov, 3...Bc5 4.Bg2 Nc6", "1.e4 e5 2.Nc3 Nf6 3.g3 Bc5 4.Bg2 Nc6 *"),
("C26h", "Vienna: Smyslov, 3...Bc5 4.Bg2 Nc6 5.Nge2", "1.e4 e5 2.Nc3 Nf6 3.g3 Bc5 4.Bg2 Nc6 5.Nge2 *"),
("C26i", "Vienna: Smyslov, 3...d5", "1.e4 e5 2.Nc3 Nf6 3.g3 d5 *"),
("C26i", "Vienna: Smyslov, 3...d5", "1.e4 e5 2.Nc3 Nf6 3.g3 d5 4.exd5 Nxd5 5.Bg2 *"),
("C26j", "Vienna: Smyslov, 3...d5, 5...Nxc3", "1.e4 e5 2.Nc3 Nf6 3.g3 d5 4.exd5 Nxd5 5.Bg2 Nxc3 *"),
("C26k", "Vienna: Smyslov, 3...d5, 5...Nxc3 6.bxc3 Bd6", "1.e4 e5 2.Nc3 Nf6 3.g3 d5 4.exd5 Nxd5 5.Bg2 Nxc3 6.bxc3 Bd6 *"),
("C26l", "Vienna: 3.Bc4", "1.e4 e5 2.Nc3 Nf6 3.Bc4 *"),
("C26l", "Vienna: Horwitz Gambit", "1.e4 e5 2.Nc3 Nf6 3.Bc4 b5 *"),
("C26m", "Vienna: 3.Bc4 Bb4", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Bb4 *"),
("C26n", "Vienna: 3.Bc4 Bb4 4.Nf3", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Bb4 4.Nf3 *"),
("C26o", "Vienna: 3.Bc4 Bb4 4.Nf3 O-O", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Bb4 4.Nf3 O-O *"),
("C26p", "Vienna: 3.Bc4 Bc5", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Bc5 *"),
("C26p", "Vienna: 3.Bc4 Bc5 4.f4", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Bc5 4.f4 *"),
("C26q", "Vienna: 3.Bc4 Bc5 4.Nf3", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Bc5 4.Nf3 *"),
("C26r", "Vienna: 3.Bc4 Bc5 4.d3", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Bc5 4.d3 *"),
("C26s", "Vienna: 3.Bc4 Bc5 4.d3 d6", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Bc5 4.d3 d6 *"),
("C26t", "Vienna: 3.Bc4 Bc5 4.d3 d6 5.Na4", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Bc5 4.d3 d6 5.Na4 *"),
("C26u", "Vienna: 3.Bc4 Bc5 4.d3 d6 5.f4", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Bc5 4.d3 d6 5.f4 *"),
("C27a", "Vienna: 3.Bc4 Nxe4", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nxe4 *"),
("C27b", "Vienna: 3.Bc4 Nxe4 4.Nxe4", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nxe4 4.Nxe4 *"),
("C27c", "Vienna: 3.Bc4 Nxe4 4.Qh5", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nxe4 4.Qh5 *"),
("C27d", "Vienna: 3.Bc4 Nxe4 4.Qh5 Nd6 5.Qxe5+", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nxe4 4.Qh5 Nd6 5.Qxe5+ *"),
("C27e", "Vienna: 3.Bc4 Nxe4 4.Qh5 Nd6 5.Bb3", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nxe4 4.Qh5 Nd6 5.Bb3 *"),
("C27f", "Vienna: 3.Bc4 Nxe4 4.Qh5 Nd6 5.Bb3 Be7", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nxe4 4.Qh5 Nd6 5.Bb3 Be7 *"),
("C27g", "Vienna: Alekhine Variation", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nxe4 4.Qh5 Nd6 5.Bb3 Be7 6.Nf3 Nc6 7.Nxe5 *"),
("C27h", "Vienna: 3.Bc4 Nxe4 4.Qh5 Nd6 5.Bb3 Nc6", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nxe4 4.Qh5 Nd6 5.Bb3 Nc6 *"),
("C27h", "Vienna: Adams' Gambit", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nxe4 4.Qh5 Nd6 5.Bb3 Nc6 6.d4 *"),
("C27h", "Vienna: 3.Bc4 Nxe4 4.Qh5 Nd6 5.Bb3 Nc6", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nxe4 4.Qh5 Nd6 5.Bb3 Nc6 6.Nb5 *"),
("C27i", "Vienna: Frankenstein-Dracula Variation", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nxe4 4.Qh5 Nd6 5.Bb3 Nc6 6.Nb5 g6 7.Qf3 f5 8.Qd5 Qe7 9.Nxc7+ Kd8 10.Nxa8 *"),
("C27j", "Vienna: Frankenstein-Dracula, 11.d3", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nxe4 4.Qh5 Nd6 5.Bb3 Nc6 6.Nb5 g6 7.Qf3 f5 8.Qd5 Qe7 9.Nxc7+ Kd8 10.Nxa8 b6 11.d3 *"),
("C28a", "Vienna: 3.Bc4 Nc6", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nc6 *"),
("C28b", "Vienna: 3.Bc4 Nc6 4.f4", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nc6 4.f4 *"),
("C28b", "Vienna: Bronstein Gambit", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nc6 4.f4 Nxe4 5.Nf3 *"),
("C28c", "Vienna: 3.Bc4 Nc6 4.d3", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nc6 4.d3 *"),
("C28d", "Vienna: 3.Bc4 Nc6 4.d3 Be7", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nc6 4.d3 Be7 *"),
("C28e", "Vienna: 3.Bc4 Nc6 4.d3 Na5", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nc6 4.d3 Na5 *"),
("C28f", "Vienna: 3.Bc4 Nc6 4.d3 Na5 5.Nge2", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nc6 4.d3 Na5 5.Nge2 *"),
("C28g", "Vienna: 3.Bc4 Nc6 4.d3 Bc5", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nc6 4.d3 Bc5 *"),
("C28h", "Vienna: 3.Bc4 Nc6 4.d3 Bc5 5.Bg5", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nc6 4.d3 Bc5 5.Bg5 *"),
("C28i", "Vienna: 3.Bc4 Nc6 4.d3 Bc5 5.f4", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nc6 4.d3 Bc5 5.f4 *"),
("C28j", "Vienna: 3.Bc4 Nc6 4.d3 Bc5 5.f4 d6 6.Nf3", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nc6 4.d3 Bc5 5.f4 d6 6.Nf3 *"),
("C28k", "Vienna: 3.Bc4 Nc6 4.d3 Bc5 5.f4 d6 6.Nf3 Bg4", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nc6 4.d3 Bc5 5.f4 d6 6.Nf3 Bg4 *"),
("C28l", "Vienna: 3.Bc4 Nc6 4.d3 Bc5 5.f4 d6 6.Nf3 a6", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nc6 4.d3 Bc5 5.f4 d6 6.Nf3 a6 *"),
("C28m", "Vienna: 3.Bc4 Nc6 4.d3 Bb4", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nc6 4.d3 Bb4 *"),
("C28n", "Vienna: 3.Bc4 Nc6 4.d3 Bb4 5.Nf3", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nc6 4.d3 Bb4 5.Nf3 *"),
("C28o", "Vienna: 3.Bc4 Nc6 4.d3 Bb4 5.Bg5", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nc6 4.d3 Bb4 5.Bg5 *"),
("C28p", "Vienna: 3.Bc4 Nc6 4.d3 Bb4 5.Ne2", "1.e4 e5 2.Nc3 Nf6 3.Bc4 Nc6 4.d3 Bb4 5.Ne2 *"),
("C29a", "Vienna Gambit", "1.e4 e5 2.Nc3 Nf6 3.f4 *"),
("C29a", "Vienna Gambit: 3...exf4", "1.e4 e5 2.Nc3 Nf6 3.f4 exf4 *"),
("C29b", "Vienna Gambit: 3...d6", "1.e4 e5 2.Nc3 Nf6 3.f4 d6 *"),
("C29b", "Vienna Gambit: 3...d6 4.Nf3", "1.e4 e5 2.Nc3 Nf6 3.f4 d6 4.Nf3 *"),
("C29c", "Vienna Gambit: 3...d5", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 *"),
("C29d", "Vienna Gambit: 3...d5 4.exd5", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.exd5 *"),
("C29d", "Vienna Gambit: 3...d5 4.exd5 exf4", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.exd5 exf4 *"),
("C29e", "Vienna Gambit: Steinitz Variation", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.d3 *"),
("C29f", "Vienna Gambit: 4.fxe5", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.fxe5 *"),
("C29f", "Vienna Gambit: 4.fxe5", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.fxe5 Nxe4 *"),
("C29g", "Vienna Gambit: Oxford Variation", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.fxe5 Nxe4 5.d3 *"),
("C29g", "Vienna Gambit: Wurzburger Trap", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.fxe5 Nxe4 5.d3 Qh4+ 6.g3 Nxg3 7.Nf3 Qh5 8.Nxd5 *"),
("C29h", "Vienna Gambit: Oxford, 5...Bb4", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.fxe5 Nxe4 5.d3 Bb4 *"),
("C29h", "Vienna Gambit: Oxford, 5...Bb4 6.dxe4 Qh4+ 7.Ke2", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.fxe5 Nxe4 5.d3 Bb4 6.dxe4 Qh4+ 7.Ke2 *"),
("C29i", "Vienna Gambit: Oxford, 5...Nxc3", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.fxe5 Nxe4 5.d3 Nxc3 *"),
("C29j", "Vienna Gambit: Oxford, 5...Nxc3 6.bxc3 d4", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.fxe5 Nxe4 5.d3 Nxc3 6.bxc3 d4 *"),
("C29k", "Vienna Gambit: Paulsen Attack", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.fxe5 Nxe4 5.Qf3 *"),
("C29k", "Vienna Gambit: Paulsen Attack, 5...Nc6", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.fxe5 Nxe4 5.Qf3 Nc6 *"),
("C29l", "Vienna Gambit: Paulsen Attack, Bardeleben Variation", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.fxe5 Nxe4 5.Qf3 f5 *"),
("C29l", "Vienna Gambit: Paulsen Attack, Heyde Variation", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.fxe5 Nxe4 5.Qf3 f5 6.d4 *"),
("C29m", "Vienna Gambit: Paulsen Attack, 5...Nxc3", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.fxe5 Nxe4 5.Qf3 Nxc3 *"),
("C29n", "Vienna Gambit: 5.Nf3", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.fxe5 Nxe4 5.Nf3 *"),
("C29o", "Vienna Gambit: 5.Nf3 Bg4", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.fxe5 Nxe4 5.Nf3 Bg4 *"),
("C29o", "Vienna Gambit: Kaufmann Variation", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.fxe5 Nxe4 5.Nf3 Bg4 6.Qe2 *"),
("C29p", "Vienna Gambit: 5.Nf3 Bb4", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.fxe5 Nxe4 5.Nf3 Bb4 *"),
("C29q", "Vienna Gambit: 5.Nf3 Nc6", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.fxe5 Nxe4 5.Nf3 Nc6 *"),
("C29r", "Vienna Gambit: Breyer Variation", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.fxe5 Nxe4 5.Nf3 Be7 *"),
("C29s", "Vienna Gambit: Breyer, 6.Qe2", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.fxe5 Nxe4 5.Nf3 Be7 6.Qe2 *"),
("C29t", "Vienna Gambit: Breyer, 6.Qe2 Nxc3", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.fxe5 Nxe4 5.Nf3 Be7 6.Qe2 Nxc3 *"),
("C29t", "Vienna Gambit: Breyer, 6.Qe2 Nxc3 7.dxc3", "1.e4 e5 2.Nc3 Nf6 3.f4 d5 4.fxe5 Nxe4 5.Nf3 Be7 6.Qe2 Nxc3 7.dxc3 *"),
("C30a", "King's Gambit", "1.e4 e5 2.f4 *"),
("C30b", "King's Gambit: Mafia Defence", "1.e4 e5 2.f4 c5 *"),
("C30c", "King's Gambit: 2...d6", "1.e4 e5 2.f4 d6 *"),
("C30c", "King's Gambit: 2...d6 3.Nf3", "1.e4 e5 2.f4 d6 3.Nf3 *"),
("C30c", "King's Gambit: 2...d6 3.Nf3 Nc6", "1.e4 e5 2.f4 d6 3.Nf3 Nc6 *"),
("C30d", "King's Gambit: Wade Variation", "1.e4 e5 2.f4 Nf6 *"),
("C30e", "King's Gambit: Norwald Variation", "1.e4 e5 2.f4 Qf6 *"),
("C30e", "King's Gambit: Norwald Variation, Schubert line", "1.e4 e5 2.f4 Qf6 3.Nc3 Qxf4 4.d4 *"),
("C30e", "King's Gambit: Norwald Variation, Buecker Gambit", "1.e4 e5 2.f4 Qf6 3.Nf3 Qxf4 4.Nc3 Bb4 5.Bc4 *"),
("C30f", "King's Gambit: Keene Defence", "1.e4 e5 2.f4 Qh4+ *"),
("C30f", "King's Gambit: Keene Defence, 3.g3 Qe7", "1.e4 e5 2.f4 Qh4+ 3.g3 Qe7 *"),
("C30g", "King's Gambit: 2...Nc6", "1.e4 e5 2.f4 Nc6 *"),
("C30h", "King's Gambit: Adelaide-Wahls Variation", "1.e4 e5 2.f4 Nc6 3.Nf3 f5 *"),
("C30i", "King's Gambit: Adelaide-Wahls, 4.exf5", "1.e4 e5 2.f4 Nc6 3.Nf3 f5 4.exf5 *"),
("C30i", "King's Gambit: Adelaide-Wahls, 4.exf5 e4", "1.e4 e5 2.f4 Nc6 3.Nf3 f5 4.exf5 e4 *"),
("C30i", "King's Gambit: Adelaide-Wahls, 4.exf5 e4 5.Ne5 Nf6", "1.e4 e5 2.f4 Nc6 3.Nf3 f5 4.exf5 e4 5.Ne5 Nf6 *"),
("C30j", "King's Gambit: Classical KGD", "1.e4 e5 2.f4 Bc5 *"),
("C30j", "KGD: Classical, 3.Bc4", "1.e4 e5 2.f4 Bc5 3.Bc4 *"),
("C30j", "KGD: Classical, 3.Nf3", "1.e4 e5 2.f4 Bc5 3.Nf3 *"),
("C30j", "KGD: Classical, Senechaud Countergambit", "1.e4 e5 2.f4 Bc5 3.Nf3 g5 *"),
("C30k", "KGD: Classical, 3.Nf3 d6", "1.e4 e5 2.f4 Bc5 3.Nf3 d6 *"),
("C30k", "KGD: Classical, Heath Variation", "1.e4 e5 2.f4 Bc5 3.Nf3 d6 4.b4 *"),
("C30k", "KGD: Classical, Soldatenkov Variation", "1.e4 e5 2.f4 Bc5 3.Nf3 d6 4.fxe5 *"),
("C30l", "KGD: Classical, 3.Nf3 d6 4.Nc3", "1.e4 e5 2.f4 Bc5 3.Nf3 d6 4.Nc3 *"),
("C30l", "KGD: Classical, Hanham Variation", "1.e4 e5 2.f4 Bc5 3.Nf3 d6 4.Nc3 Nd7 *"),
("C30m", "KGD: Classical, 3.Nf3 d6 4.Nc3 Nf6 5.Bc4", "1.e4 e5 2.f4 Bc5 3.Nf3 d6 4.Nc3 Nf6 5.Bc4 *"),
("C30m", "KGD: Classical, Svenonius Variation", "1.e4 e5 2.f4 Bc5 3.Nf3 d6 4.Nc3 Nf6 5.Bc4 Nc6 6.d3 Bg4 7.h3 Bxf3 8.Qxf3 exf4 *"),
("C30n", "KGD: Classical, 4.c3", "1.e4 e5 2.f4 Bc5 3.Nf3 d6 4.c3 *"),
("C30n", "KGD: Classical, Marshall Attack", "1.e4 e5 2.f4 Bc5 3.Nf3 d6 4.c3 Bg4 5.fxe5 dxe5 6.Qa4+ *"),
("C30o", "KGD: Classical, 4.c3 Bb6", "1.e4 e5 2.f4 Bc5 3.Nf3 d6 4.c3 Bb6 *"),
("C30p", "KGD: Classical, 4.c3 Nf6", "1.e4 e5 2.f4 Bc5 3.Nf3 d6 4.c3 Nf6 *"),
("C30q", "KGD: Classical, 4.c3 Nf6 5.d4", "1.e4 e5 2.f4 Bc5 3.Nf3 d6 4.c3 Nf6 5.d4 *"),
("C30q", "KGD: Classical, 4.c3 Nf6 5.d4", "1.e4 e5 2.f4 Bc5 3.Nf3 d6 4.c3 Nf6 5.d4 Bb6 *"),
("C30q", "KGD: Classical, 4.c3 Nf6 5.d4", "1.e4 e5 2.f4 Bc5 3.Nf3 d6 4.c3 Nf6 5.d4 exd4 6.cxd4 Bb6 *"),
("C30r", "KGD: Classical Countergambit", "1.e4 e5 2.f4 Bc5 3.Nf3 d6 4.c3 f5 *"),
("C30r", "KGD: Classical, Reti Variation", "1.e4 e5 2.f4 Bc5 3.Nf3 d6 4.c3 f5 5.fxe5 dxe5 6.d4 exd4 7.Bc4 *"),
("C31", "KGD: Falkbeer Countergambit", "1.e4 e5 2.f4 d5 *"),
("C31", "KGD: Falkbeer, Tartakower Variation", "1.e4 e5 2.f4 d5 3.Nf3 *"),
("C31", "KGD: Falkbeer, Milner-Barry Variation", "1.e4 e5 2.f4 d5 3.Nc3 *"),
("C31", "KGD: Falkbeer, 3.exd5", "1.e4 e5 2.f4 d5 3.exd5 *"),
("C31", "KGD: Falkbeer, 3.exd5 exf4", "1.e4 e5 2.f4 d5 3.exd5 exf4 *"),
("C31", "KGD: Falkbeer, Marshall/Nimzowitsch Countergambit", "1.e4 e5 2.f4 d5 3.exd5 c6 *"),
("C31", "KGD: Falkbeer, Marshall/Nimzowitsch, 4.dxc6", "1.e4 e5 2.f4 d5 3.exd5 c6 4.dxc6 *"),
("C31", "KGD: Falkbeer, Marshall/Nimzowitsch, 4.Qe2", "1.e4 e5 2.f4 d5 3.exd5 c6 4.Qe2 *"),
("C31", "KGD: Falkbeer, Marshall/Nimzowitsch, 4.Nc3", "1.e4 e5 2.f4 d5 3.exd5 c6 4.Nc3 *"),
("C31", "KGD: Falkbeer, 3.exd5 e4", "1.e4 e5 2.f4 d5 3.exd5 e4 *"),
("C31", "KGD: Falkbeer, Rubinstein Variation", "1.e4 e5 2.f4 d5 3.exd5 e4 4.Nc3 Nf6 5.Qe2 *"),
("C31", "KGD: Falkbeer, Nimzowitsch Variation", "1.e4 e5 2.f4 d5 3.exd5 e4 4.Bb5+ *"),
("C31", "KGD: Falkbeer, 4.d3", "1.e4 e5 2.f4 d5 3.exd5 e4 4.d3 *"),
("C32", "KGD: Falkbeer, 4.d3 Nf6", "1.e4 e5 2.f4 d5 3.exd5 e4 4.d3 Nf6 *"),
("C32", "KGD: Falkbeer, Keres Variation", "1.e4 e5 2.f4 d5 3.exd5 e4 4.d3 Nf6 5.Nd2 *"),
("C32", "KGD: Falkbeer, Keres, 5...exd3", "1.e4 e5 2.f4 d5 3.exd5 e4 4.d3 Nf6 5.Nd2 exd3 *"),
("C32", "KGD: Falkbeer, Reti Variation", "1.e4 e5 2.f4 d5 3.exd5 e4 4.d3 Nf6 5.Qe2 *"),
("C32", "KGD: Falkbeer, 4.d3 Nf6 5.Nc3", "1.e4 e5 2.f4 d5 3.exd5 e4 4.d3 Nf6 5.Nc3 *"),
("C32", "KGD: Falkbeer, 4.d3 Nf6 5.Nc3 Bb4", "1.e4 e5 2.f4 d5 3.exd5 e4 4.d3 Nf6 5.Nc3 Bb4 *"),
("C32", "KGD: Falkbeer, Morphy Gambit", "1.e4 e5 2.f4 d5 3.exd5 e4 4.d3 Nf6 5.Nc3 Bb4 6.Bd2 e3 *"),
("C32", "KGD: Falkbeer, 5.dxe4", "1.e4 e5 2.f4 d5 3.exd5 e4 4.d3 Nf6 5.dxe4 *"),
("C32", "KGD: Falkbeer, Charousek Variation", "1.e4 e5 2.f4 d5 3.exd5 e4 4.d3 Nf6 5.dxe4 Nxe4 6.Qe2 *"),
("C32", "KGD: Falkbeer, 5.dxe4 Nxe4 6.Be3", "1.e4 e5 2.f4 d5 3.exd5 e4 4.d3 Nf6 5.dxe4 Nxe4 6.Be3 *"),
("C32", "KGD: Falkbeer, 5.dxe4 Nxe4 6.Nf3", "1.e4 e5 2.f4 d5 3.exd5 e4 4.d3 Nf6 5.dxe4 Nxe4 6.Nf3 *"),
("C32", "KGD: Falkbeer, 5.dxe4 Nxe4 6.Nf3 Bc5 7.Qe2", "1.e4 e5 2.f4 d5 3.exd5 e4 4.d3 Nf6 5.dxe4 Nxe4 6.Nf3 Bc5 7.Qe2 *"),
("C32", "KGD: Falkbeer, Alapin Variation", "1.e4 e5 2.f4 d5 3.exd5 e4 4.d3 Nf6 5.dxe4 Nxe4 6.Nf3 Bc5 7.Qe2 Bf2+ 8.Kd1 Qxd5+ 9.Nfd2 *"),
("C32", "KGD: Falkbeer, Main Line, 7...Bf5", "1.e4 e5 2.f4 d5 3.exd5 e4 4.d3 Nf6 5.dxe4 Nxe4 6.Nf3 Bc5 7.Qe2 Bf5 *"),
("C32", "KGD: Falkbeer, Tarrasch Variation", "1.e4 e5 2.f4 d5 3.exd5 e4 4.d3 Nf6 5.dxe4 Nxe4 6.Nf3 Bc5 7.Qe2 Bf5 8.g4 O-O *"),
("C32", "KGD: Falkbeer, Main Line, 7...Bf5 8.Nc3", "1.e4 e5 2.f4 d5 3.exd5 e4 4.d3 Nf6 5.dxe4 Nxe4 6.Nf3 Bc5 7.Qe2 Bf5 8.Nc3 *"),
("C33", "King's Gambit Accepted (KGA)", "1.e4 e5 2.f4 exf4 *"),
("C33", "KGA: Tumbleweed/Drunken King", "1.e4 e5 2.f4 exf4 3.Kf2 *"),
("C33", "KGA: Orsini Gambit", "1.e4 e5 2.f4 exf4 3.b3 *"),
("C33", "KGA: Stamma (Leonardo) Gambit", "1.e4 e5 2.f4 exf4 3.h4 *"),
("C33", "KGA: Schurig Gambit", "1.e4 e5 2.f4 exf4 3.Bd3 *"),
("C33", "KGA: Basman Gambit", "1.e4 e5 2.f4 exf4 3.Qe2 *"),
("C33", "KGA: Carrera Gambit", "1.e4 e5 2.f4 exf4 3.Qh5 *"),
("C33", "KGA: Eisenberg Gambit", "1.e4 e5 2.f4 exf4 3.Nh3 *"),
("C33", "KGA: Eisenberg Gambit", "1.e4 e5 2.f4 exf4 3.Ne2 *"),
("C33", "KGA: Villemson Gambit", "1.e4 e5 2.f4 exf4 3.d4 *"),
("C33", "KGA: Keres Gambit", "1.e4 e5 2.f4 exf4 3.Nc3 *"),
("C33", "KGA: Breyer Gambit", "1.e4 e5 2.f4 exf4 3.Qf3 *"),
("C33", "KGA: Lesser Bishop's (Tartakower) Gambit", "1.e4 e5 2.f4 exf4 3.Be2 *"),
("C33", "KGA: Bishop's Gambit", "1.e4 e5 2.f4 exf4 3.Bc4 *"),
("C33", "KGA: Bishop's Gambit, Chigorin Attack", "1.e4 e5 2.f4 exf4 3.Bc4 Qh4+ 4.Kf1 d5 5.Bxd5 g5 6.g3 *"),
("C33", "KGA: Bishop's Gambit, Greco Variation", "1.e4 e5 2.f4 exf4 3.Bc4 Qh4+ 4.Kf1 Bc5 *"),
("C33", "KGA: Bishop's Gambit, Classical Defence", "1.e4 e5 2.f4 exf4 3.Bc4 Qh4+ 4.Kf1 g5 *"),
("C33", "KGA: Bishop's Gambit, Grimm Attack", "1.e4 e5 2.f4 exf4 3.Bc4 Qh4+ 4.Kf1 g5 5.Nc3 Bg7 6.d4 d6 7.e5 *"),
("C33", "KGA: Bishop's Gambit, Classical Defence", "1.e4 e5 2.f4 exf4 3.Bc4 Qh4+ 4.Kf1 g5 5.Nc3 Bg7 6.d4 Ne7 *"),
("C33", "KGA: Bishop's Gambit, McDonnell Attack", "1.e4 e5 2.f4 exf4 3.Bc4 Qh4+ 4.Kf1 g5 5.Nc3 Bg7 6.d4 Ne7 7.g3 *"),
("C33", "KGA: bishop's Gambit, McDonnell Attack", "1.e4 e5 2.f4 exf4 3.Bc4 Qh4+ 4.Kf1 g5 5.Nc3 Bg7 6.g3 *"),
("C33", "KGA: Bishop's Gambit, Fraser Variation", "1.e4 e5 2.f4 exf4 3.Bc4 Qh4+ 4.Kf1 g5 5.Nc3 Bg7 6.g3 fxg3 7.Qf3 *"),
("C33", "KGA: Bishop's Gambit, Classical Defence, Cozio Attack", "1.e4 e5 2.f4 exf4 3.Bc4 Qh4+ 4.Kf1 g5 5.Qf3 *"),
("C33", "KGA: Bishop's Gambit, Boden Defence", "1.e4 e5 2.f4 exf4 3.Bc4 Qh4+ 4.Kf1 Nc6 *"),
("C33", "KGA: Bishop's Gambit, Bryan Countergambit", "1.e4 e5 2.f4 exf4 3.Bc4 Qh4+ 4.Kf1 b5 *"),
("C33", "KGA: Bishop's Gambit, Bryan Countergambit", "1.e4 e5 2.f4 exf4 3.Bc4 b5 *"),
("C33", "KGA: Bishop's Gambit, Steinitz Defence", "1.e4 e5 2.f4 exf4 3.Bc4 Ne7 *"),
("C33", "KGA: Bishop's Gambit, Maurian Defence", "1.e4 e5 2.f4 exf4 3.Bc4 Nc6 *"),
("C33", "KGA: Bishop's Gambit, Ruy Lopez Defence", "1.e4 e5 2.f4 exf4 3.Bc4 c6 *"),
("C33", "KGA: Bishop's Gambit, Lopez-Gianutio Countergambit", "1.e4 e5 2.f4 exf4 3.Bc4 f5 *"),
("C33", "KGA: Bishop's Gambit, Lopez-Gianutio Countergambit, Hein Variation", "1.e4 e5 2.f4 exf4 3.Bc4 f5 4.Qe2 Qh4+ 5.Kd1 fxe4 6.Nc3 Kd8 *"),
("C33", "KGA: Bishop's Gambit, Bledow Variation", "1.e4 e5 2.f4 exf4 3.Bc4 d5 *"),
("C33", "KGA: Bishop's Gambit, Bledow, 4.exd5", "1.e4 e5 2.f4 exf4 3.Bc4 d5 4.exd5 *"),
("C33", "KGA: Bishop's Gambit, Bledow, 4.Bxd5", "1.e4 e5 2.f4 exf4 3.Bc4 d5 4.Bxd5 *"),
("C33", "KGA: Bishop's Gambit, Boren-Svenonius Variation", "1.e4 e5 2.f4 exf4 3.Bc4 d5 4.Bxd5 Qh4+ 5.Kf1 Bd6 *"),
("C33", "KGA: Bishop's Gambit, Anderssen Variation", "1.e4 e5 2.f4 exf4 3.Bc4 d5 4.Bxd5 c6 *"),
("C33", "KGA: Bishop's Gambit, Morphy Variation", "1.e4 e5 2.f4 exf4 3.Bc4 d5 4.Bxd5 Nf6 *"),
("C33", "KGA: Bishop's Gambit, Cozio Defence", "1.e4 e5 2.f4 exf4 3.Bc4 Nf6 *"),
("C33", "KGA: Bishop's Gambit, Bogoljubow Variation", "1.e4 e5 2.f4 exf4 3.Bc4 Nf6 4.Nc3 *"),
("C33", "KGA: Bishop's Gambit, Paulsen Attack", "1.e4 e5 2.f4 exf4 3.Bc4 Nf6 4.Nc3 Bb4 5.e5 *"),
("C33", "KGA: Bishop's Gambit, Jaenisch Variation", "1.e4 e5 2.f4 exf4 3.Bc4 Nf6 4.Nc3 c6 *"),
("C34", "KGA: King's Knight Gambit", "1.e4 e5 2.f4 exf4 3.Nf3 *"),
("C34", "KGA: Bonsch-Osmolovsky Variation", "1.e4 e5 2.f4 exf4 3.Nf3 Ne7 *"),
("C34", "KGA: Gianutio Countergambit", "1.e4 e5 2.f4 exf4 3.Nf3 f5 *"),
("C34", "KGA: Becker Defence (Anti-Kieseritzky)", "1.e4 e5 2.f4 exf4 3.Nf3 h6 *"),
("C34", "KGA: Schallop Defence", "1.e4 e5 2.f4 exf4 3.Nf3 Nf6 *"),
("C34", "KGA: Fischer Defence", "1.e4 e5 2.f4 exf4 3.Nf3 d6 *"),
("C34", "KGA: Fischer, 4.Bc4", "1.e4 e5 2.f4 exf4 3.Nf3 d6 4.Bc4 *"),
("C34", "KGA: Fischer, 4.d4", "1.e4 e5 2.f4 exf4 3.Nf3 d6 4.d4 *"),
("C34", "KGA: Fischer, Main Line", "1.e4 e5 2.f4 exf4 3.Nf3 d6 4.d4 g5 5.h4 g4 6.Ng1 *"),
("C35", "KGA: Cunningham Defence", "1.e4 e5 2.f4 exf4 3.Nf3 Be7 *"),
("C35", "KGA: Cunningham, Bertin Gambit", "1.e4 e5 2.f4 exf4 3.Nf3 Be7 4.Bc4 Bh4+ 5.g3 *"),
("C35", "KGA: Cunningham, Three Pawns Gambit", "1.e4 e5 2.f4 exf4 3.Nf3 Be7 4.Bc4 Bh4+ 5.g3 fxg3 6.O-O gxh2+ 7.Kh1 *"),
("C35", "KGA: Cunningham, Euwe Defence", "1.e4 e5 2.f4 exf4 3.Nf3 Be7 4.Bc4 Nf6 *"),
("C36", "KGA: Scandinavian (Abbazia) Variation", "1.e4 e5 2.f4 exf4 3.Nf3 d5 *"),
("C36", "KGA: Scandinavian, 4.exd5", "1.e4 e5 2.f4 exf4 3.Nf3 d5 4.exd5 *"),
("C36", "KGA: Scandinavian, 4.exd5 Bd6", "1.e4 e5 2.f4 exf4 3.Nf3 d5 4.exd5 Bd6 *"),
("C36", "KGA: Scandinavian, Modern Variation", "1.e4 e5 2.f4 exf4 3.Nf3 d5 4.exd5 Nf6 *"),
("C36", "KGA: Scandinavian, Modern, 5.Bc4", "1.e4 e5 2.f4 exf4 3.Nf3 d5 4.exd5 Nf6 5.Bc4 *"),
("C36", "KGA: Scandinavian, Modern, 5.Nc3", "1.e4 e5 2.f4 exf4 3.Nf3 d5 4.exd5 Nf6 5.Nc3 *"),
("C36", "KGA: Scandinavian, Modern, 5.Bb5+ ", "1.e4 e5 2.f4 exf4 3.Nf3 d5 4.exd5 Nf6 5.Bb5+ *"),
("C36", "KGA: Scandinavian, Modern, 5.Bb5+ c6", "1.e4 e5 2.f4 exf4 3.Nf3 d5 4.exd5 Nf6 5.Bb5+ c6 *"),
("C36", "KGA: Scandinavian, Modern, 5.Bb5+ c6 6.dxc6 Nxc6", "1.e4 e5 2.f4 exf4 3.Nf3 d5 4.exd5 Nf6 5.Bb5+ c6 6.dxc6 Nxc6 *"),
("C36", "KGA: Scandinavian, Modern, 5.Bb5+ c6 6.dxc6 bxc6", "1.e4 e5 2.f4 exf4 3.Nf3 d5 4.exd5 Nf6 5.Bb5+ c6 6.dxc6 bxc6 *"),
("C36", "KGA: Scandinavian, Botvinnik Variation", "1.e4 e5 2.f4 exf4 3.Nf3 d5 4.exd5 Nf6 5.Bb5+ c6 6.dxc6 bxc6 7.Bc4 Nd5 *"),
("C37", "KGA: 3.Nf3 g5", "1.e4 e5 2.f4 exf4 3.Nf3 g5 *"),
("C37", "KGA: Quaade Gambit", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Nc3 *"),
("C37", "KGA: Rosentreter Gambit", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.d4 *"),
("C37", "KGA: Sorensen Gambit", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.d4 g4 5.Ne5 *"),
("C37", "KGA: 3.Nf3 g5 4.Bc4", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 *"),
("C37", "KGA: Blachly Gambit", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 Nc6 *"),
("C37", "KGA: 3.Nf3 g5 4.Bc4 g4", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 g4 *"),
("C37", "KGA: Lolli Gambit (Wild Muzio)", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 g4 5.Bxf7+ *"),
("C37", "KGA: Lolli Gambit, Young Variation", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 g4 5.Bxf7+ Kxf7 6.O-O gxf3 7.Qxf3 Qf6 8.d4 Qxd4+ 9.Be3 Qf6 10.Nc3 *"),
("C37", "KGA: Ghulam-Kassim Gambit", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 g4 5.d4 *"),
("C37", "KGA: MacDonnell Gambit", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 g4 5.Nc3 *"),
("C37", "KGA: Salvio Gambit", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 g4 5.Ne5 *"),
("C37", "KGA: Salvio, Silberschmidt Gambit", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 g4 5.Ne5 Qh4+ 6.Kf1 Nh6 7.d4 f3 *"),
("C37", "KGA: Salvio, Anderssen Counterattack", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 g4 5.Ne5 Qh4+ 6.Kf1 Nh6 7.d4 d6 *"),
("C37", "KGA: Salvio, Cochrane Gambit", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 g4 5.Ne5 Qh4+ 6.Kf1 f3 *"),
("C37", "KGA: Salvio, Herzfeld Gambit", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 g4 5.Ne5 Qh4+ 6.Kf1 Nc6 *"),
("C37", "KGA: Muzio Gambit", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 g4 5.O-O *"),
("C37", "KGA: Muzio Gambit, Paulsen Variation", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 g4 5.O-O gxf3 6.Qxf3 Qf6 7.e5 Qxe5 8.d3 Bh6 9.Nc3 Ne7 10.Bd2 Nbc6 11.Rae1 *"),
("C37", "KGA: Double Muzio Gambit", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 g4 5.O-O gxf3 6.Qxf3 Qf6 7.e5 Qxe5 8.Bxf7+ *"),
("C37", "KGA: Muzio Gambit, From Defence", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 g4 5.O-O gxf3 6.Qxf3 Qe7 *"),
("C37", "KGA: Muzio Gambit, Holloway Defence", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 g4 5.O-O gxf3 6.Qxf3 Nc6 *"),
("C37", "KGA: Muzio Gambit, Kling and Horwitz Counterattack", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 g4 5.O-O Qe7 *"),
("C37", "KGA: Muzio Gambit, Brentano Defence", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 g4 5.O-O d5 *"),
("C38", "KGA: 3.Nf3 g5 4.Bc4 Bg7", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 Bg7 *"),
("C38", "KGA: Hanstein Gambit", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 Bg7 5.d4 *"),
("C38", "KGA: Hanstein Gambit", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 Bg7 5.O-O h6 6.d4 d6 *"),
("C38", "KGA: Philidor Gambit", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 Bg7 5.h4 *"),
("C38", "KGA: Greco Gambit", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 Bg7 5.h4 h6 6.d4 d6 7.Nc3 c6 8.hxg5 hxg5 9.Rxh8 Bxh8 10.Ne5 *"),
("C38", "KGA: Philidor Gambit, Schultz Variation", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.Bc4 Bg7 5.h4 h6 6.d4 d6 7.Qd3 *"),
("C39", "KGA: 3.Nf3 g5 4.h4", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 *"),
("C39", "KGA: Allgaier Gambit", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ng5 *"),
("C39", "KGA: Allgaier, Horny Defence", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ng5 h6 6.Nxf7 Kxf7 7.Qxg4 Nf6 8.Qxf4 Bd6 *"),
("C39", "KGA: Allgaier, Thorold Variation", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ng5 h6 6.Nxf7 Kxf7 7.d4 *"),
("C39", "KGA: Allgaier, Cook Variation", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ng5 h6 6.Nxf7 Kxf7 7.d4 d5 8.Bxf4 dxe4 9.Bc4+ Kg7 10.Be5+ *"),
("C39", "KGA: Allgaier, Blackburne Gambit", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ng5 h6 6.Nxf7 Kxf7 7.Nc3 *"),
("C39", "KGA: Allgaier, Walker Attack", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ng5 h6 6.Nxf7 Kxf7 7.Bc4+ *"),
("C39", "KGA: Allgaier, Urusov Attack", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ng5 h6 6.Nxf7 Kxf7 7.Bc4+ d5 8.Bxd5+ Kg7 9.d4 *"),
("C39", "KGA: Allgaier, Schlechter Defence", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ng5 Nf6 *"),
("C39", "KGA: Kieseritsky", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ne5 *"),
("C39", "KGA: Kieseritsky, Green Variation", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ne5 d6 *"),
("C39", "KGA: Kieseritsky, Paulsen Defence", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ne5 Bg7 *"),
("C39", "KGA: Kieseritsky, Long Whip Defence", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ne5 h5 *"),
("C39", "KGA: Kieseritsky, Long Whip Defence, Jaenisch Variation", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ne5 h5 6.Bc4 Rh7 7.d4 Bh6 8.Nc3 *"),
("C39", "KGA: Kieseritsky, Brentano (Campbell) Defence", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ne5 d5 *"),
("C39", "KGA: Kieseritsky, Brentano Defence, Kaplanek Variation", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ne5 d5 6.d4 Nf6 7.exd5 Qxd5 8.Nc3 Bb4 9.Kf2 *"),
("C39", "KGA: Kieseritsky, Brentano Defence", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ne5 d5 6.d4 Nf6 7.Bxf4 *"),
("C39", "KGA: Kieseritsky, Brentano Defence, Caro Variation", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ne5 d5 6.d4 Nf6 7.Bxf4 Nxe4 8.Nd2 *"),
("C39", "KGA: Kieseritsky, Salvio (Rosenthal) Defence", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ne5 Qe7 *"),
("C39", "KGA: Kieseritsky, Salvio Defence, Cozio Variation", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ne5 Qe7 6.d4 f5 7.Bc4 *"),
("C39", "KGA: Kieseritsky, Polerio Defence", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ne5 Be7 *"),
("C39", "KGA: Kieseritsky, Neumann Defence", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ne5 Nc6 *"),
("C39", "KGA: Kieseritsky, Berlin Defence", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ne5 Nf6 *"),
("C39", "KGA: Kieseritsky, Berlin Defence, Riviere Variation", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ne5 Nf6 6.Nxg4 d5 *"),
("C39", "KGA: Kieseritsky, Berlin Defence, 6.Bc4", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ne5 Nf6 6.Bc4 *"),
("C39", "KGA: Kieseritsky, Rice Gambit", "1.e4 e5 2.f4 exf4 3.Nf3 g5 4.h4 g4 5.Ne5 Nf6 6.Bc4 d5 7.exd5 Bd6 8.O-O *"),
("C40a", "Open Game", "1.e4 e5 2.Nf3 *"),
("C40a", "Open Game: Damiano Defence", "1.e4 e5 2.Nf3 f6 *"),
("C40b", "Open Game: Greco Defence", "1.e4 e5 2.Nf3 Qf6 *"),
("C40c", "Open Game: Gunderam Defence", "1.e4 e5 2.Nf3 Qe7 *"),
("C40d", "Open Game: Gunderam Defence, 3.Nc3", "1.e4 e5 2.Nf3 Qe7 3.Nc3 *"),
("C40e", "Elephant Gambit", "1.e4 e5 2.Nf3 d5 *"),
("C40f", "Elephant Gambit: 3.Nxe5", "1.e4 e5 2.Nf3 d5 3.Nxe5 *"),
("C40f", "Elephant Gambit: 3.Nxe5 dxe4 4.Bc4", "1.e4 e5 2.Nf3 d5 3.Nxe5 dxe4 4.Bc4 *"),
("C40g", "Elephant Gambit: 3.exd5", "1.e4 e5 2.Nf3 d5 3.exd5 *"),
("C40h", "Elephant Gambit: Maroczy", "1.e4 e5 2.Nf3 d5 3.exd5 Bd6 *"),
("C40i", "Elephant Gambit: Paulsen", "1.e4 e5 2.Nf3 d5 3.exd5 e4 *"),
("C40j", "Latvian Gambit", "1.e4 e5 2.Nf3 f5 *"),
("C40k", "Latvian Gambit: 3.d3", "1.e4 e5 2.Nf3 f5 3.d3 *"),
("C40l", "Latvian Gambit: 3.Nc3", "1.e4 e5 2.Nf3 f5 3.Nc3 *"),
("C40m", "Latvian Gambit: 3.d4", "1.e4 e5 2.Nf3 f5 3.d4 *"),
("C40m", "Latvian Gambit: 3.d4 fxe4 5.Nxe5 Nf6", "1.e4 e5 2.Nf3 f5 3.d4 fxe4 5.Nxe5 Nf6 *"),
("C40n", "Latvian Gambit: 3.exf5", "1.e4 e5 2.Nf3 f5 3.exf5 *"),
("C40n", "Latvian Gambit: 3.exf5 e4", "1.e4 e5 2.Nf3 f5 3.exf5 e4 *"),
("C40o", "Latvian Gambit: 3.Bc4", "1.e4 e5 2.Nf3 f5 3.Bc4 *"),
// 3.Bc4 Nc6 is classified under Italian Game: C50 (2.Nf3 Nc6 3.Bc4 f5).
("C40o", "Latvian Gambit: Strautins Variation", "1.e4 e5 2.Nf3 f5 3.Bc4 b5 *"),
("C40o", "Latvian Gambit: Morgado Variation", "1.e4 e5 2.Nf3 f5 3.Bc4 Nf6 *"),
("C40p", "Latvian Gambit: 3.Bc4 fxe4", "1.e4 e5 2.Nf3 f5 3.Bc4 fxe4 *"),
("C40p", "Latvian: Blackburne Variation (Corkscrew Countergambit)", "1.e4 e5 2.Nf3 f5 3.Bc4 fxe4 4.Nxe5 Nf6 *"),
("C40p", "Latvian Gambit: Svedenborg Variation", "1.e4 e5 2.Nf3 f5 3.Bc4 fxe4 4.Nxe5 d5 *"),
("C40p", "Latvian Gambit: Svedenborg, 6.Nxg6 Nf6", "1.e4 e5 2.Nf3 f5 3.Bc4 fxe4 4.Nxe5 d5 5.Qh5+ g6 6.Nxg6 Nf6 *"),
("C40p", "Latvian Gambit: Svedenborg, 6.Nxg6 hxg6", "1.e4 e5 2.Nf3 f5 3.Bc4 fxe4 4.Nxe5 d5 5.Qh5+ g6 6.Nxg6 hxg6 *"),
("C40q", "Latvian Gambit: Poisoned Pawn Variation", "1.e4 e5 2.Nf3 f5 3.Bc4 fxe4 4.Nxe5 Qg5 *"),
("C40q", "Latvian Gambit: Poisoned Pawn, Main Line", "1.e4 e5 2.Nf3 f5 3.Bc4 fxe4 4.Nxe5 Qg5 5.d4 Qxg2 6.Qh5+ g6 7.Bf7+ Kd8 *"),
("C40r", "Latvian Gambit: 3.Nxe5", "1.e4 e5 2.Nf3 f5 3.Nxe5 *"),
("C40s", "Latvian Gambit: 3.Nxe5 Nc6", "1.e4 e5 2.Nf3 f5 3.Nxe5 Nc6 *"),
("C40t", "Latvian Gambit: 3.Nxe5 Qf6", "1.e4 e5 2.Nf3 f5 3.Nxe5 Qf6 *"),
("C40t", "Latvian Gambit: 3.Nxe5 Qf6 4.Nc4", "1.e4 e5 2.Nf3 f5 3.Nxe5 Qf6 4.Nc4 *"),
("C40u", "Latvian Gambit: 3.Nxe5 Qf6 4.d4", "1.e4 e5 2.Nf3 f5 3.Nxe5 Qf6 4.d4 *"),
("C40u", "Latvian Gambit: 3.Nxe5 Qf6 4.d4 d6", "1.e4 e5 2.Nf3 f5 3.Nxe5 Qf6 4.d4 d6 *"),
("C40u", "Latvian Gambit: 3.Nxe5 Qf6 4.d4 d6 5.Nc4", "1.e4 e5 2.Nf3 f5 3.Nxe5 Qf6 4.d4 d6 5.Nc4 *"),
("C40u", "Latvian Gambit: 3.Nxe5 Qf6 4.d4 d6 5.Nc4 fxe4", "1.e4 e5 2.Nf3 f5 3.Nxe5 Qf6 4.d4 d6 5.Nc4 fxe4 *"),
("C40v", "Latvian Gambit: 3.Nxe5 Qf6 4.d4 d6 5.Nc4 fxe4 6.Be2", "1.e4 e5 2.Nf3 f5 3.Nxe5 Qf6 4.d4 d6 5.Nc4 fxe4 6.Be2 *"),
("C40w", "Latvian Gambit: Nimzowitsch Variation", "1.e4 e5 2.Nf3 f5 3.Nxe5 Qf6 4.d4 d6 5.Nc4 fxe4 6.Ne3 *"),
("C40x", "Latvian Gambit: 3.Nxe5 Qf6 4.d4 d6 5.Nc4 fxe4 6.Nc3", "1.e4 e5 2.Nf3 f5 3.Nxe5 Qf6 4.d4 d6 5.Nc4 fxe4 6.Nc3 *"),
("C41a", "Philidor Defence", "1.e4 e5 2.Nf3 d6 *"),
("C41b", "Philidor: 3.Bc4", "1.e4 e5 2.Nf3 d6 3.Bc4 *"),
("C41b", "Philidor: Steinitz Variation", "1.e4 e5 2.Nf3 d6 3.Bc4 Be7 4.c3 *"),
("C41b", "Philidor: Lopez Countergambit", "1.e4 e5 2.Nf3 d6 3.Bc4 f5 *"),
("C41b", "Philidor: Lopez Countergambit, Jaenisch Variation", "1.e4 e5 2.Nf3 d6 3.Bc4 f5 4.d4 exd4 5.Ng5 Nh6 6.Nxh7 *"),
("C41c", "Philidor: 3.d4", "1.e4 e5 2.Nf3 d6 3.d4 *"),
("C41d", "Philidor: Philidor Countergambit", "1.e4 e5 2.Nf3 d6 3.d4 f5 *"),
("C41d", "Philidor: Philidor Countergambit, Zukertort Variation", "1.e4 e5 2.Nf3 d6 3.d4 f5 4.Nc3 *"),
("C41e", "Philidor: Philidor Countergambit, 4.dxe5", "1.e4 e5 2.Nf3 d6 3.d4 f5 4.dxe5 *"),
("C41e", "Philidor: Philidor Countergambit, Steinitz Variation", "1.e4 e5 2.Nf3 d6 3.d4 f5 4.dxe5 fxe4 5.Ng5 d5 6.Nc3 *"),
("C41e", "Philidor: Philidor Countergambit, del Rio Attack", "1.e4 e5 2.Nf3 d6 3.d4 f5 4.dxe5 fxe4 5.Ng5 d5 6.e6 *"),
("C41e", "Philidor: Philidor Countergambit, Berger Variation", "1.e4 e5 2.Nf3 d6 3.d4 f5 4.dxe5 fxe4 5.Ng5 d5 6.e6 Bc5 7.Nc3 *"),
("C41f", "Philidor: Hanham Variation", "1.e4 e5 2.Nf3 d6 3.d4 Nd7 *"),
("C41f", "Philidor: Hanham, 4.Bc4", "1.e4 e5 2.Nf3 d6 3.d4 Nd7 4.Bc4 *"),
("C41g", "Philidor: Hanham, 4.Bc4 c6", "1.e4 e5 2.Nf3 d6 3.d4 Nd7 4.Bc4 c6 *"),
("C41g", "Philidor: Hanham, Krause Variation", "1.e4 e5 2.Nf3 d6 3.d4 Nd7 4.Bc4 c6 5.O-O *"),
("C41g", "Philidor: Hanham, Steiner Variation", "1.e4 e5 2.Nf3 d6 3.d4 Nd7 4.Bc4 c6 5.O-O Be7 6.dxe5 *"),
("C41g", "Philidor: Hanham, Kmoch Variation", "1.e4 e5 2.Nf3 d6 3.d4 Nd7 4.Bc4 c6 5.Ng5 *"),
("C41g", "Philidor: Hanham, Berger Variation", "1.e4 e5 2.Nf3 d6 3.d4 Nd7 4.Bc4 c6 5.Ng5 Nh6 6.f4 Be7 7.O-O O-O 8.c3 d5 *"),
("C41g", "Philidor: Hanham, Schlechter Variation", "1.e4 e5 2.Nf3 d6 3.d4 Nd7 4.Bc4 c6 5.Nc3 *"),
("C41g", "Philidor: Hanham, Delmar Variation", "1.e4 e5 2.Nf3 d6 3.d4 Nd7 4.Bc4 c6 5.c3 *"),
("C41h", "Philidor: 3...exd4", "1.e4 e5 2.Nf3 d6 3.d4 exd4 *"),
("C41h", "Philidor: Bird Gambit", "1.e4 e5 2.Nf3 d6 3.d4 exd4 4.c3 *"),
("C41i", "Philidor: Morphy Variation", "1.e4 e5 2.Nf3 d6 3.d4 exd4 4.Qxd4 *"),
("C41i", "Philidor: Morphy, 4...Nc6", "1.e4 e5 2.Nf3 d6 3.d4 exd4 4.Qxd4 Nc6 *"),
("C41i", "Philidor: Morphy, 4...Nf6", "1.e4 e5 2.Nf3 d6 3.d4 exd4 4.Qxd4 Nf6 *"),
("C41i", "Philidor: Morphy, 4...Nf6 5.Nc3", "1.e4 e5 2.Nf3 d6 3.d4 exd4 4.Qxd4 Nf6 5.Nc3 *"),
("C41j", "Philidor: 3...exd4 4.Nxd4", "1.e4 e5 2.Nf3 d6 3.d4 exd4 4.Nxd4 *"),
("C41j", "Philidor: Paulsen Attack", "1.e4 e5 2.Nf3 d6 3.d4 exd4 4.Nxd4 d5 5.exd5 *"),
("C41k", "Philidor: 3...exd4 4.Nxd4 Nf6", "1.e4 e5 2.Nf3 d6 3.d4 exd4 4.Nxd4 Nf6 *"),
("C41k", "Philidor: 3...exd4 4.Nxd4 Nf6 5.Nc3", "1.e4 e5 2.Nf3 d6 3.d4 exd4 4.Nxd4 Nf6 5.Nc3 *"),
("C41k", "Philidor: 3...exd4 4.Nxd4 Nf6 5.Nc3 Be7", "1.e4 e5 2.Nf3 d6 3.d4 exd4 4.Nxd4 Nf6 5.Nc3 Be7 *"),
("C41l", "Philidor: 3...exd4 4.Nxd4 Nf6 5.Nc3 Be7 6.Bc4", "1.e4 e5 2.Nf3 d6 3.d4 exd4 4.Nxd4 Nf6 5.Nc3 Be7 6.Bc4 *"),
("C41m", "Philidor: Antoshin Variation", "1.e4 e5 2.Nf3 d6 3.d4 exd4 4.Nxd4 Nf6 5.Nc3 Be7 6.Be2 *"),
("C41m", "Philidor: Berger Variation", "1.e4 e5 2.Nf3 d6 3.d4 exd4 4.Nxd4 Nf6 5.Nc3 Be7 6.Be2 O-O 7.O-O c5 8.Nf3 Nc6 9.Bg5 Be6 10.Re1 *"),
("C41n", "Philidor: Larsen Variation", "1.e4 e5 2.Nf3 d6 3.d4 exd4 4.Nxd4 g6 *"),
("C41o", "Philidor: Nimzowitsch Variation", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 *"),
("C41o", "Philidor: Nimzowitsch, Klein Variation", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.Bc4 *"),
("C41o", "Philidor: Nimzowitsch, Locock Variation", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.Ng5 *"),
("C41p", "Philidor: Exchange Variation", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.dxe5 *"),
("C41p", "Philidor: Exchange, Sokolsky Variation", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.dxe5 Nxe4 5.Nbd2 *"),
("C41p", "Philidor: Exchange, Rellstab Variation", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.dxe5 Nxe4 5.Qd5 *"),
("C41q", "Philidor: Nimzowitsch Variation", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.Nc3 *"),
("C41r", "Philidor: Improved Hanham", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.Nc3 Nbd7 *"),
("C41s", "Philidor: Improved Hanham, 5.Bc4", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.Nc3 Nbd7 5.Bc4 *"),
("C41s", "Philidor: Improved Hanham, 5.Bc4 Be7", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.Nc3 Nbd7 5.Bc4 Be7 *"),
("C41s", "Philidor: Improved Hanham, 6.Bxf7+", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.Nc3 Nbd7 5.Bc4 Be7 6.Bxf7+ *"),
("C41s", "Philidor: Improved Hanham, 6.Ng5", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.Nc3 Nbd7 5.Bc4 Be7 6.Ng5 *"),
("C41s", "Philidor: Improved Hanham, Larobok Variation", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.Nc3 Nbd7 5.Bc4 Be7 6.Ng5 O-O 7.Bxf7+ *"),
("C41s", "Philidor: Improved Hanham, 6.dxe5", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.Nc3 Nbd7 5.Bc4 Be7 6.dxe5 *"),
("C41s", "Philidor: Improved Hanham, 6.dxe5 Nxe5", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.Nc3 Nbd7 5.Bc4 Be7 6.dxe5 Nxe5 *"),
("C41s", "Philidor: Improved Hanham, 6.dxe5 dxe5", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.Nc3 Nbd7 5.Bc4 Be7 6.dxe5 dxe5 *"),
("C41t", "Philidor: Improved Hanham 6.O-O", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.Nc3 Nbd7 5.Bc4 Be7 6.O-O *"),
("C41u", "Philidor: Improved Hanham, Main Line", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.Nc3 Nbd7 5.Bc4 Be7 6.O-O O-O *"),
("C41v", "Philidor: Improved Hanham, 7.a4", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.Nc3 Nbd7 5.Bc4 Be7 6.O-O O-O 7.a4 *"),
("C41v", "Philidor: Improved Hanham, 7.a4 c6", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.Nc3 Nbd7 5.Bc4 Be7 6.O-O O-O 7.a4 c6 *"),
("C41w", "Philidor: Improved Hanham, 7.Qe2", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.Nc3 Nbd7 5.Bc4 Be7 6.O-O O-O 7.Qe2 *"),
("C41w", "Philidor: Improved Hanham, 7.Qe2 c6 8.a4", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.Nc3 Nbd7 5.Bc4 Be7 6.O-O O-O 7.a4 c6 8.Qe2 *"),
("C41w", "Philidor: Improved Hanham, Sozin Variation", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.Nc3 Nbd7 5.Bc4 Be7 6.O-O O-O 7.Qe2 c6 8.a4 exd4 *"),
("C41x", "Philidor: Improved Hanham, 7.Re1", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.Nc3 Nbd7 5.Bc4 Be7 6.O-O O-O 7.Re1 *"),
("C41x", "Philidor: Improved Hanham, 7.Re1 c6 8.a4", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.Nc3 Nbd7 5.Bc4 Be7 6.O-O O-O 7.Re1 c6 8.a4 *"),
("C41y", "Philidor: Improved Hanham, 7.Re1 c6 8.a4 a5", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.Nc3 Nbd7 5.Bc4 Be7 6.O-O O-O 7.Re1 c6 8.a4 a5 *"),
("C41z", "Philidor: Improved Hanham, 7.Re1 c6 8.a4 b6", "1.e4 e5 2.Nf3 d6 3.d4 Nf6 4.Nc3 Nbd7 5.Bc4 Be7 6.O-O O-O 7.Re1 c6 8.a4 b6 *"),
("C42a", "Russian Game (Petroff Defence)", "1.e4 e5 2.Nf3 Nf6 *"),
("C42b", "Russian Game: 3.d3", "1.e4 e5 2.Nf3 Nf6 3.d3 *"),
("C42c", "Russian-Three Knights Game", "1.e4 e5 2.Nf3 Nf6 3.Nc3 *"),
("C42c", "Russian-Three Knights Game", "1.e4 e5 2.Nf3 Nf6 3.Nc3 Bb4 *"),
("C42d", "Russian-Three Knights Game, 4.Nxe5", "1.e4 e5 2.Nf3 Nf6 3.Nc3 Bb4 4.Nxe5 *"),
("C42e", "Russian Game: Italian Variation", "1.e4 e5 2.Nf3 Nf6 3.Bc4 *"),
("C42f", "Russian Game: Boden-Kieseritsky Gambit", "1.e4 e5 2.Nf3 Nf6 3.Bc4 Nxe4 4.Nc3 *"),
("C42f", "Russian Game: Boden-Kieseritsky Gambit Accepted", "1.e4 e5 2.Nf3 Nf6 3.Bc4 Nxe4 4.Nc3 Nxc3 *"),
("C42g", "Russian Game: 3.Nxe5", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 *"),
("C42g", "Russian Game: Damiano Variation", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 Nxe4 *"),
("C42g", "Russian Game: 3.Nxe5 d6", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 *"),
("C42h", "Russian Game: Cochrane Gambit", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nxf7 *"),
("C42h", "Russian Game: Cochrane Gambit, 5.d4", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nxf7 Kxf7 5.d4 *"),
("C42i", "Russian Game: Paulsen Attack", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nc4 *"),
("C42j", "Russian Game: 3.Nxe5 d6 4.Nf3", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 *"),
("C42k", "Russian Game: French Attack", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.d3 *"),
("C42l", "Russian Game: Vienna/Kaufmann", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.c4 *"),
("C42m", "Russian Game: Nimzowitsch Attack", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.Nc3 *"),
("C42n", "Russian Game: Cozio/Lasker", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.Qe2 *"),
("C42n", "Russian Game: Milner-Barry Variation", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.Qe2 Qe7 6.d3 Nf6 7.Bg5 Nbd7 *"),
("C42o", "Russian Game: 5.Qe2, Queenswap", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.Qe2 Qe7 6.d3 Nf6 7.Bg5 Qxe2+ *"),
("C42p", "Russian Game: Classical", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.d4 *"),
("C42p", "Russian Game: Classical, Close Variation", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.d4 Nf6 *"),
("C42p", "Russian Game: Classical, 5...d5", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.d4 d5 *"),
("C42q", "Russian Game: Classical, 6.Bd3", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.d4 d5 6.Bd3 *"),
("C42q", "Russian Game: Classical, Marshall Variation", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.d4 d5 6.Bd3 Bd6 *"),
("C42q", "Russian Game: Classical, Tarrasch Variation", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.d4 d5 6.Bd3 Bd6 7.O-O O-O 8.c4 Bg4 *"),
("C42q", "Russian Game: Classical, Marshall Trap", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.d4 d5 6.Bd3 Bd6 7.O-O O-O 8.c4 Bg4 9.cxd5 f5 10.Re1 Bxh2+ *"),
("C42r", "Russian Game: Classical, Marshall, 8.c4 c6", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.d4 d5 6.Bd3 Bd6 7.O-O O-O 8.c4 c6 *"),
("C42s", "Russian Game: Classical, 6.Bd3 Be7", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.d4 d5 6.Bd3 Be7 *"),
("C42s", "Russian Game: Classical, Mason Variation", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.d4 d5 6.Bd3 Be7 7.O-O O-O *"),
("C42s", "Russian Game: Classical, 6.Bd3 Be7 7.O-O Nc6", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.d4 d5 6.Bd3 Be7 7.O-O Nc6 *"),
("C42t", "Russian Game: Classical, Jaenisch Variation", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.d4 d5 6.Bd3 Be7 7.O-O Nc6 8.c4 *"),
("C42u", "Russian Game: Classical, Jaenisch, Main Line", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.d4 d5 6.Bd3 Be7 7.O-O Nc6 8.c4 Nb4 9.Be2 O-O *"),
("C42v", "Russian Game: Classical, Chigorin Variation", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.d4 d5 6.Bd3 Be7 7.O-O Nc6 8.Re1 *"),
("C42v", "Russian Game: Classical, Chigorin, 8...Bf5", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.d4 d5 6.Bd3 Be7 7.O-O Nc6 8.Re1 Bf5 *"),
("C42w", "Russian Game: Classical, Chigorin, 8...Bg4", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.d4 d5 6.Bd3 Be7 7.O-O Nc6 8.Re1 Bg4 *"),
("C42w", "Russian Game: Classical, Chigorin, 8...Bg4 9.c4", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.d4 d5 6.Bd3 Be7 7.O-O Nc6 8.Re1 Bg4 9.c4 *"),
("C42x", "Russian Game: Classical, Chigorin, 8...Bg4 9.c3", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.d4 d5 6.Bd3 Be7 7.O-O Nc6 8.Re1 Bg4 9.c3 *"),
("C42x", "Russian Game: Classical, Berger Variation", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.d4 d5 6.Bd3 Be7 7.O-O Nc6 8.Re1 Bg4 9.c3 f5 10.Nbd2 *"),
("C42x", "Russian Game: Classical, Krause Variation", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.d4 d5 6.Bd3 Be7 7.O-O Nc6 8.Re1 Bg4 9.c3 f5 10.c4 *"),
("C42x", "Russian Game: Classical, Maroczy Variation", "1.e4 e5 2.Nf3 Nf6 3.Nxe5 d6 4.Nf3 Nxe4 5.d4 d5 6.Bd3 Be7 7.O-O Nc6 8.Re1 Bg4 9.c3 f5 10.c4 Bh4 *"),
("C43a", "Russian Game: Modern (Steinitz) Attack", "1.e4 e5 2.Nf3 Nf6 3.d4 *"),
("C43b", "Russian Game: Modern Attack", "1.e4 e5 2.Nf3 Nf6 3.d4 exd4 *"),
// 1.e4 e5 2.Nf3 Nf6 3.d4 exd4 4.Bc4 is the Urusov Gambit, code C24.
("C43c", "Russian Game: Modern Attack", "1.e4 e5 2.Nf3 Nf6 3.d4 exd4 4.e5 Ne4 *"),
("C43c", "Russian Game: Modern Attack, Tal Gambit", "1.e4 e5 2.Nf3 Nf6 3.d4 exd4 4.e5 Ne4 5.Bb5 *"),
("C43c", "Russian Game: Modern Attack, Steinitz Variation", "1.e4 e5 2.Nf3 Nf6 3.d4 exd4 4.e5 Ne4 5.Qe2 *"),
("C43c", "Russian Game: Modern Attack, Bardeleben Variation", "1.e4 e5 2.Nf3 Nf6 3.d4 exd4 4.e5 Ne4 5.Qe2 Nc5 6.Nxd4 Nc6 *"),
("C43d", "Russian Game: Modern Attack, 3...exd4, Main Line", "1.e4 e5 2.Nf3 Nf6 3.d4 exd4 4.e5 Ne4 5.Qxd4 *"),
("C43e", "Russian Game: Modern Attack, 3...exd4, Main Line, 7.Nc3", "1.e4 e5 2.Nf3 Nf6 3.d4 exd4 4.e5 Ne4 5.Qxd4 d5 6.exd6 Nxd6 7.Nc3 *"),
("C43e", "Russian Game: Modern Attack, Pillsbury Variation", "1.e4 e5 2.Nf3 Nf6 3.d4 exd4 4.e5 Ne4 5.Qxd4 d5 6.exd6 Nxd6 7.Nc3 Nc6 8.Qf4 Bf5 *"),
("C43f", "Russian Game: Modern Attack, 3...Nxe4", "1.e4 e5 2.Nf3 Nf6 3.d4 Nxe4 *"),
("C43g", "Russian Game: Modern Attack, 3...Nxe4 4.Bd3", "1.e4 e5 2.Nf3 Nf6 3.d4 Nxe4 4.Bd3 *"),
("C43g", "Russian Game: Modern Attack, 3...Nxe4 4.Bd3 Nc6", "1.e4 e5 2.Nf3 Nf6 3.d4 Nxe4 4.Bd3 Nc6 *"),
("C43g", "Russian Game: Modern Attack, 3...Nxe4 4.Bd3 Nc6 5.d5", "1.e4 e5 2.Nf3 Nf6 3.d4 Nxe4 4.Bd3 Nc6 5.d5 *"),
("C43h", "Russian Game: Modern Attack, 3...Nxe4 4.Bd3 Nc6 5.Bxe4", "1.e4 e5 2.Nf3 Nf6 3.d4 Nxe4 4.Bd3 Nc6 5.Bxe4 *"),
("C43i", "Russian Game: Modern Attack, 3...Nxe4 4.Bd3 d5", "1.e4 e5 2.Nf3 Nf6 3.d4 Nxe4 4.Bd3 d5 *"),
("C43i", "Russian Game: Modern Attack, 3...Nxe4 4.Bd3 d5 5.dxe5", "1.e4 e5 2.Nf3 Nf6 3.d4 Nxe4 4.Bd3 d5 5.dxe5 *"),
("C43j", "Russian Game: Modern Attack, 3...Nxe4 4.Bd3 d5 5.Nxe5", "1.e4 e5 2.Nf3 Nf6 3.d4 Nxe4 4.Bd3 d5 5.Nxe5 *"),
("C43k", "Russian Game: Modern Attack, 3...Nxe4 4.Bd3 d5 5.Nxe5 Nc6", "1.e4 e5 2.Nf3 Nf6 3.d4 Nxe4 4.Bd3 d5 5.Nxe5 Nc6 *"),
("C43l", "Russian Game: Modern Attack, 3...Nxe4 4.Bd3 d5 5.Nxe5 Be7", "1.e4 e5 2.Nf3 Nf6 3.d4 Nxe4 4.Bd3 d5 5.Nxe5 Be7 *"),
("C43m", "Russian Game: Modern Attack, 3...Nxe4 4.Bd3 d5 5.Nxe5 Bd6", "1.e4 e5 2.Nf3 Nf6 3.d4 Nxe4 4.Bd3 d5 5.Nxe5 Bd6 *"),
("C43n", "Russian Game: Modern Attack, Trifunovic Variation", "1.e4 e5 2.Nf3 Nf6 3.d4 Nxe4 4.Bd3 d5 5.Nxe5 Bd6 6.O-O O-O 7.c4 Bxe5 *"),
("C43o", "Russian Game: Modern Attack, 3...Nxe4 4.Bd3 d5 5.Nxe5 Nd7", "1.e4 e5 2.Nf3 Nf6 3.d4 Nxe4 4.Bd3 d5 5.Nxe5 Nd7 *"),
("C43p", "Russian Game: Modern Attack, 3...Nxe4 4.Bd3 d5 5.Nxe5 Nd7 6.Qe2", "1.e4 e5 2.Nf3 Nf6 3.d4 Nxe4 4.Bd3 d5 5.Nxe5 Nd7 6.Qe2 *"),
("C43q", "Russian Game: Modern Attack, 3...Nxe4 4.Bd3 d5 5.Nxe5 Nd7 6.Nxd7", "1.e4 e5 2.Nf3 Nf6 3.d4 Nxe4 4.Bd3 d5 5.Nxe5 Nd7 6.Nxd7 *"),
("C43r", "Russian Game: Modern Attack, 3...Nxe4, Main Line", "1.e4 e5 2.Nf3 Nf6 3.d4 Nxe4 4.Bd3 d5 5.Nxe5 Nd7 6.Nxd7 Bxd7 7.O-O *"),
("C43s", "Russian Game: Modern Attack, 3...Nxe4, Main Line, 7...Qh4", "1.e4 e5 2.Nf3 Nf6 3.d4 Nxe4 4.Bd3 d5 5.Nxe5 Nd7 6.Nxd7 Bxd7 7.O-O Qh4 *"),
("C43t", "Russian Game: Modern Attack, 3...Nxe4, Main Line, 7...Bd6", "1.e4 e5 2.Nf3 Nf6 3.d4 Nxe4 4.Bd3 d5 5.Nxe5 Nd7 6.Nxd7 Bxd7 7.O-O Bd6 *"),
("C43u", "Russian Game: Modern Attack, 3...Nxe4, Main Line, 7...Bd6 8.c4", "1.e4 e5 2.Nf3 Nf6 3.d4 Nxe4 4.Bd3 d5 5.Nxe5 Nd7 6.Nxd7 Bxd7 7.O-O Bd6 8.c4 *"),
("C44a", "Open Game", "1.e4 e5 2.Nf3 Nc6 *"),
("C44a", "Open Game: Irish (Chicago) Gambit", "1.e4 e5 2.Nf3 Nc6 3.Nxe5 *"),
("C44a", "Open Game: Paschman Wing Gambit", "1.e4 e5 2.Nf3 Nc6 3.b4 *"),
("C44a", "Open Game: Dresden Opening", "1.e4 e5 2.Nf3 Nc6 3.c4 *"),
("C44b", "Open Game: Konstantinopolsky", "1.e4 e5 2.Nf3 Nc6 3.g3 *"),
("C44c", "Open Game: Inverted Hungarian", "1.e4 e5 2.Nf3 Nc6 3.Be2 *"),
("C44c", "Open Game: Tayler Opening", "1.e4 e5 2.Nf3 Nc6 3.Be2 Nf6 4.d4 *"),
("C44d", "Open Game: Inverted Philidor", "1.e4 e5 2.Nf3 Nc6 3.d3 *"),
("C44d", "Open Game: Inverted Philidor", "1.e4 e5 2.Nf3 Nc6 3.d3 d5 *"),
("C44e", "Open Game: Inverted Philidor", "1.e4 e5 2.Nf3 Nc6 3.d3 Nf6 *"),
("C44f", "Open Game: Inverted Philidor, 4.g3", "1.e4 e5 2.Nf3 Nc6 3.d3 Nf6 4.g3 *"),
("C44g", "Open Game: Inverted Philidor, 4.Be2", "1.e4 e5 2.Nf3 Nc6 3.d3 Nf6 4.Be2 *"),
("C44h", "Open Game: Inverted Hanham", "1.e4 e5 2.Nf3 Nc6 3.Be2 Nf6 4.d3 d5 5.Nbd2 *"),
("C44i", "Ponziani Opening", "1.e4 e5 2.Nf3 Nc6 3.c3 *"),
("C44i", "Ponziani: Reti Variation", "1.e4 e5 2.Nf3 Nc6 3.c3 Nge7 *"),
("C44i", "Ponziani: Romanishin Variation", "1.e4 e5 2.Nf3 Nc6 3.c3 Be7 *"),
("C44j", "Ponziani Countergambit", "1.e4 e5 2.Nf3 Nc6 3.c3 f5 *"),
("C44j", "Ponziani Countergambit, Schmidt Attack", "1.e4 e5 2.Nf3 Nc6 3.c3 f5 4.d4 d6 5.d5 *"),
("C44j", "Ponziani Countergambit, Cordel Variation", "1.e4 e5 2.Nf3 Nc6 3.c3 f5 4.d4 d6 5.d5 fxe4 6.Ng5 Nb8 7.Nxe4 Nf6 8.Bd3 Be7 *"),
("C44k", "Ponziani: 3...d5", "1.e4 e5 2.Nf3 Nc6 3.c3 d5 *"),
("C44l", "Ponziani: 3...d5 4.Qa4", "1.e4 e5 2.Nf3 Nc6 3.c3 d5 4.Qa4 *"),
("C44l", "Ponziani: Caro Variation", "1.e4 e5 2.Nf3 Nc6 3.c3 d5 4.Qa4 Bd7 *"),
("C44l", "Ponziani: Leonhardt Variation", "1.e4 e5 2.Nf3 Nc6 3.c3 d5 4.Qa4 Nf6 *"),
("C44m", "Ponziani: Steinitz Variation", "1.e4 e5 2.Nf3 Nc6 3.c3 d5 4.Qa4 f6 *"),
("C44n", "Ponziani: 3...Nf6", "1.e4 e5 2.Nf3 Nc6 3.c3 Nf6 *"),
("C44n", "Ponziani: 3...Nf6", "1.e4 e5 2.Nf3 Nc6 3.c3 Nf6 4.d4 *"),
("C44o", "Ponziani: 3...Nf6 4.d4 exd4", "1.e4 e5 2.Nf3 Nc6 3.c3 Nf6 4.d4 exd4 *"),
("C44p", "Ponziani: 3...Nf6 4.d4 Nxe4", "1.e4 e5 2.Nf3 Nc6 3.c3 Nf6 4.d4 Nxe4 *"),
("C44p", "Ponziani: Fraser Defence", "1.e4 e5 2.Nf3 Nc6 3.c3 Nf6 4.d4 Nxe4 5.d5 Bc5 *"),
("C44q", "Ponziani: 3...Nf6 4.d4 Nxe4 5.d5 Ne7", "1.e4 e5 2.Nf3 Nc6 3.c3 Nf6 4.d4 Nxe4 5.d5 Ne7 *"),
("C44r", "Scotch Opening", "1.e4 e5 2.Nf3 Nc6 3.d4 *"),
("C44r", "Scotch: Lolli Variation", "1.e4 e5 2.Nf3 Nc6 3.d4 Nxd4 *"),
("C44r", "Scotch: Cochrane Variation", "1.e4 e5 2.Nf3 Nc6 3.d4 Nxd4 4.Nxe5 Ne6 5.Bc4 c6 6.O-O Nf6 7.Nxf7 *"),
("C44s", "Scotch: 3...d6", "1.e4 e5 2.Nf3 Nc6 3.d4 d6 *"),
("C44t", "Scotch: 3...exd4", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 *"),
("C44t", "Scotch: Relfsson Gambit", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Bb5 *"),
("C44t", "Scotch: Goring Gambit", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.c3 *"),
("C44u", "Scotch: Goring Gambit, 4...d5", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.c3 d5 *"),
("C44v", "Scotch: Goring Gambit, 4...d3", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.c3 d3 *"),
("C44w", "Scotch: Goring Gambit, 4...dxc3", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.c3 dxc3 *"),
("C44w", "Scotch: Goring Gambit, 4...dxc3 5.Bc4", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.c3 dxc3 5.Bc4 *"),
("C44w", "Scotch: Goring Gambit, 4...dxc3 5.Bc4 cxb2 6.Bxb2", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.c3 dxc3 5.Bc4 cxb2 6.Bxb2 *"),
("C44w", "Scotch: Goring Gambit, 4...dxc3 5.Nxc3", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.c3 dxc3 5.Nxc3 *"),
("C44w", "Scotch: Goring Gambit, 4...dxc3 5.Nxc3 Bb4", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.c3 dxc3 5.Nxc3 Bb4 *"),
("C44w", "Scotch: Goring Gambit, Bardeleben Variation", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.c3 dxc3 5.Nxc3 Bb4 6.Bc4 Nf6 *"),
("C44w", "Scotch: Goring Gambit, 4...dxc3 5.Nxc3 Bb4 6.Bc4 d6", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.c3 dxc3 5.Nxc3 Bb4 6.Bc4 d6 *"),
("C44w", "Scotch: Goring Gambit, 4...dxc3 5.Nxc3 Bb4 6.Bc4 d6 7.O-O", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.c3 dxc3 5.Nxc3 Bb4 6.Bc4 d6 7.O-O *"),
("C44x", "Scotch Gambit", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Bc4 *"),
("C44x", "Scotch Gambit: London Defence", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Bc4 Bb4+ *"),
("C44y", "Scotch Gambit: 4...Bc5", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Bc4 Bc5 *"),
("C44y", "Scotch Gambit: 5.Ng5", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Bc4 Bc5 5.Ng5 *"),
("C44y", "Scotch Gambit: Vitzhum Attack", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Bc4 Bc5 5.Ng5 Nh6 6.Qh5 *"),
("C44y", "Scotch Gambit: 5.O-O", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Bc4 Bc5 5.O-O *"),
("C44y", "Scotch Gambit: Anderssen (Paulsen) Counterattack", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Bc4 Bc5 5.O-O d6 6.c3 Bg4 *"),
("C44z", "Scotch Gambit: 5.c3", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Bc4 Bc5 5.c3 *"),
("C44z", "Scotch Gambit: 5.c3 d3", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Bc4 Bc5 5.c3 d3 *"),
("C45a", "Scotch: 4.Nxd4", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 *"),
("C45b", "Scotch: 4.Nxd4 Nxd4", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Nxd4 *"),
("C45b", "Scotch: Ghulam Kassim Variation", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Nxd4 5.Qxd4 d6 6.Bd3 *"),
("C45c", "Scotch: 4.Nxd4 Bb4+", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Bb4+ *"),
("C45d", "Scotch: 4.Nxd4 Qf6", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Qf6 *"),
("C45e", "Scotch: Steinitz Variation", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Qh4 *"),
("C45e", "Scotch: Steinitz, 5.Qd3", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Qh4 5.Qd3 *"),
("C45e", "Scotch: Steinitz, Fraser Attack", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Qh4 5.Nf3 *"),
("C45f", "Scotch: Steinitz, 5.Nc3", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Qh4 5.Nc3 *"),
("C45g", "Scotch: Steinitz, 5.Nb5", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Qh4 5.Nb5 *"),
("C45g", "Scotch: Steinitz, 5.Nb5 Qxe4+", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Qh4 5.Nb5 Qxe4+ *"),
("C45g", "Scotch: Steinitz, 5.Nb5 Bc5", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Qh4 5.Nb5 Bc5 *"),
("C45g", "Scotch: Steinitz, 5.Nb5 Bc5 6.Qf3", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Qh4 5.Nb5 Bc5 6.Qf3 *"),
("C45g", "Scotch: Steinitz, 5.Nb5 Bb4+", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Qh4 5.Nb5 Bb4+ *"),
("C45g", "Scotch: Steinitz, Berger Variation", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Qh4 5.Nb5 Bb4+ 6.Nd2 Qxe4+ 7.Be2 Qxg2 8.Bf3 Qh3 9.Nxc7+ Kd8 10.Nxa8 Nf6 11.a3 *"),
("C45g", "Scotch: Steinitz, 5.Nb5 Bb4+ 6.Bd2", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Qh4 5.Nb5 Bb4+ 6.Bd2 *"),
("C45g", "Scotch: Steinitz, Rosenthal Variation", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Qh4 5.Nb5 Bb4+ 6.Bd2 Qxe4+ 7.Be2 Kd8 8.O-O Bxd2 9.Nxd2 Qg6 *"),
("C45h", "Scotch: 4.Nxd4 Nf6", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Nf6 *"),
("C45h", "Scotch: 4.Nxd4 Nf6 5.Nxc6", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Nf6 5.Nxc6 *"),
("C45h", "Scotch: Tartakower Variation", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Nf6 5.Nxc6 bxc6 6.Nd2 *"),
("C45i", "Scotch: 4.Nxd4 Nf6 5.Nxc6 bxc6 6.Bd3", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Nf6 5.Nxc6 bxc6 6.Bd3 *"),
("C45j", "Scotch: Mieses Variation", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Nf6 5.Nxc6 bxc6 6.e5 *"),
("C45j", "Scotch: Mieses, 6...Qe7", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Nf6 5.Nxc6 bxc6 6.e5 Qe7 *"),
("C45j", "Scotch: Mieses, 6...Qe7 7.Qe2", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Nf6 5.Nxc6 bxc6 6.e5 Qe7 7.Qe2 *"),
("C45k", "Scotch: Mieses, 8.c4", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Nf6 5.Nxc6 bxc6 6.e5 Qe7 7.Qe2 Nd5 8.c4 *"),
("C45l", "Scotch: Mieses, 8.c4 Ba6", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Nf6 5.Nxc6 bxc6 6.e5 Qe7 7.Qe2 Nd5 8.c4 Ba6 *"),
("C45m", "Scotch: Mieses, 8.c4 Ba6 9.b3", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Nf6 5.Nxc6 bxc6 6.e5 Qe7 7.Qe2 Nd5 8.c4 Ba6 9.b3 *"),
("C45n", "Scotch: 4.Nxd4 Bc5", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Bc5 *"),
("C45o", "Scotch: 4.Nxd4 Bc5 5.Nb3", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Bc5 5.Nb3 *"),
("C45o", "Scotch: Romanishin Variation", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Bc5 5.Nb3 Bb4+ *"),
("C45p", "Scotch: 4.Nxd4 Bc5 5.Nb3 Bb6", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Bc5 5.Nb3 Bb6 *"),
("C45p", "Scotch: 4.Nxd4 Bc5 5.Nb3 Bb6 6.a4", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Bc5 5.Nb3 Bb6 6.a4 *"),
("C45q", "Scotch: 4.Nxd4 Bc5 5.Nb3 Bb6 6.a4 a6", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Bc5 5.Nb3 Bb6 6.a4 a6 *"),
("C45q", "Scotch: 4.Nxd4 Bc5 5.Nb3 Bb6 6.a4 a6 7.Nc3", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Bc5 5.Nb3 Bb6 6.a4 a6 7.Nc3 *"),
("C45q", "Scotch: Geller Variation", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Bc5 5.Nb3 Bb6 6.a4 a6 7.Nc3 Nf6 *"),
("C45q", "Scotch: 4.Nxd4 Bc5 5.Nb3 Bb6 6.a4 a6 7.Nc3 Qf6", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Bc5 5.Nb3 Bb6 6.a4 a6 7.Nc3 Qf6 *"),
("C45r", "Scotch: 4.Nxd4 Bc5 5.Nxc6", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Bc5 5.Nxc6 *"),
("C45s", "Scotch: 4.Nxd4 Bc5 5.Nxc6 Qf6 6.Qd2", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Bc5 5.Nxc6 Qf6 6.Qd2 *"),
("C45t", "Scotch: 4.Nxd4 Bc5 5.Nxc6 Qf6 6.Qd2 dxc6 7.Nc3", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Bc5 5.Nxc6 Qf6 6.Qd2 dxc6 7.Nc3 *"),
("C45u", "Scotch: 4.Nxd4 Bc5 5.Be3", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Bc5 5.Be3 *"),
("C45v", "Scotch: Blumenfeld Attack", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Bc5 5.Be3 Qf6 6.Nb5 *"),
("C45v", "Scotch: 4.Nxd4 Bc5 5.Be3 Qf6 6.c3", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Bc5 5.Be3 Qf6 6.c3 *"),
("C45w", "Scotch: 4.Nxd4 Bc5 5.Be3 Qf6 6.c3 Nge7", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Bc5 5.Be3 Qf6 6.c3 Nge7 *"),
("C45w", "Scotch: Blackburne Attack", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Bc5 5.Be3 Qf6 6.c3 Nge7 7.Qd2 *"),
("C45w", "Scotch: Meitner Variation", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Bc5 5.Be3 Qf6 6.c3 Nge7 7.Nc2 *"),
("C45w", "Scotch: Paulsen Attack", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Bc5 5.Be3 Qf6 6.c3 Nge7 7.Bb5 *"),
("C45w", "Scotch: Paulsen, Gunsberg Defence", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Bc5 5.Be3 Qf6 6.c3 Nge7 7.Bb5 Nd8 *"),
("C45x", "Scotch: 4.Nxd4 Bc5 5.Be3 Qf6 6.c3 Nge7 7.Bc4", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Bc5 5.Be3 Qf6 6.c3 Nge7 7.Bc4 *"),
("C45y", "Scotch: 4.Nxd4 Bc5 5.Be3 Qf6 6.c3 Nge7 7.Bc4 Ne5", "1.e4 e5 2.Nf3 Nc6 3.d4 exd4 4.Nxd4 Bc5 5.Be3 Qf6 6.c3 Nge7 7.Bc4 Ne5 *"),
("C46a", "Three Knights Game", "1.e4 e5 2.Nf3 Nc6 3.Nc3 *"),
("C46b", "Three Knights: Winawer Defence", "1.e4 e5 2.Nf3 Nc6 3.Nc3 f5 *"),
("C46c", "Three Knights: 3...d6", "1.e4 e5 2.Nf3 Nc6 3.Nc3 d6 *"),
("C46d", "Three Knights: 3...d6 4.d4", "1.e4 e5 2.Nf3 Nc6 3.Nc3 d6 4.d4 *"),
("C46e", "Three Knights: 3...Bc5", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Bc5 *"),
("C46e", "Three Knights: 3...Bc5 4.Bb5", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Bc5 4.Bb5 *"),
("C46f", "Three Knights: 3...Bc5 4.Nxe5", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Bc5 4.Nxe5 *"),
("C46g", "Three Knights: 3...Bb4", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Bb4 *"),
("C46h", "Three Knights: 3...Bb4 4.Nd5", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Bb4 4.Nd5 *"),
("C46h", "Three Knights: Schlechter Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Bb4 4.Nd5 Nf6 *"),
("C46i", "Three Knights: Steinitz Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 g6 *"),
("C46j", "Three Knights: Steinitz, 4.d4", "1.e4 e5 2.Nf3 Nc6 3.Nc3 g6 4.d4 *"),
("C46k", "Three Knights: Steinitz, Rosenthal Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 g6 4.d4 exd4 5.Nd5 *"),
("C46l", "Three Knights: Steinitz, 4.d4 exd4 5.Nxd4", "1.e4 e5 2.Nf3 Nc6 3.Nc3 g6 4.d4 exd4 5.Nxd4 *"),
("C46l", "Three Knights: Steinitz, 4.d4 exd4 5.Nxd4 Bg7", "1.e4 e5 2.Nf3 Nc6 3.Nc3 g6 4.d4 exd4 5.Nxd4 Bg7 *"),
("C46l", "Three Knights: Steinitz, 4.d4 exd4 5.Nxd4 Bg7 6.Be3", "1.e4 e5 2.Nf3 Nc6 3.Nc3 g6 4.d4 exd4 5.Nxd4 Bg7 6.Be3 *"),
("C46m", "Three Knights: Steinitz, 4.d4 exd4 5.Nxd4 Bg7 6.Be3 Nf6", "1.e4 e5 2.Nf3 Nc6 3.Nc3 g6 4.d4 exd4 5.Nxd4 Bg7 6.Be3 Nf6 *"),
("C47a", "Four Knights Game", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 *"),
("C47a", "Four Knights: Halloween Gambit", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Nxe5 *"),
("C47b", "Four Knights: Gunsberg Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.a3 *"),
("C47c", "Four Knights: Van der Wiel Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Be2 *"),
("C47d", "Four Knights: Italian Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bc4 *"),
("C47e", "Four Knights: Glek Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.g3 *"),
("C47f", "Four Knights: Glek, 4...d5", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.g3 d5 *"),
("C47g", "Four Knights: Glek, 4...Bc5", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.g3 Bc5 *"),
("C47g", "Four Knights: Glek, 4...Bc5", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.g3 Bc5 5.Bg2 *"),
("C47h", "Four Knights: Glek, Main Line", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.g3 Bc5 5.Bg2 d6 6.d3 a6 *"),
("C47i", "Four Knights: Scotch Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 *"),
("C47i", "Four Knights: Scotch, 4...d6", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 d6 *"),
("C47j", "Four Knights: Scotch, Bogoljubow", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 Bb4 *"),
("C47j", "Four Knights: Scotch, Bogoljubow, 5.d5", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 Bb4 5.d5 *"),
("C47k", "Four Knights: Scotch, Krause Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 Bb4 5.Nxe5 *"),
("C47l", "Four Knights: Scotch, 4...exd4", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 exd4 *"),
("C47l", "Four Knights: Belgrade Gambit", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 exd4 5.Nd5 *"),
("C47l", "Four Knights: Belgrade Gambit, 5...Nb4", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 exd4 5.Nd5 Nb4 *"),
("C47l", "Four Knights: Belgrade Gambit, 5...Nxd5", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 exd4 5.Nd5 Nxd5 *"),
("C47m", "Four Knights: Belgrade Gambit, 5...Nxe4", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 exd4 5.Nd5 Nxe4 *"),
("C47m", "Four Knights: Belgrade Gambit, 5...Nxe4 6.Qe2 (Old Line)", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 exd4 5.Nd5 Nxe4 6.Qe2 *"),
("C47m", "Four Knights: Belgrade Gambit, 5...Nxe4 6.Bc4 (Modern Line)", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 exd4 5.Nd5 Nxe4 6.Bc4 *"),
("C47n", "Four Knights: Belgrade Gambit, 5...Be7", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 exd4 5.Nd5 Be7 *"),
("C47o", "Four Knights: Scotch, 4...exd4 5.Nxd4", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 exd4 5.Nxd4 *"),
("C47p", "Four Knights: Scotch, 4...exd4 5.Nxd4 d6", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 exd4 5.Nxd4 d6 *"),
("C47q", "Four Knights: Scotch, 4...exd4 5.Nxd4 Bc5", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 exd4 5.Nxd4 Bc5 *"),
("C47r", "Four Knights: Scotch, 4...exd4 5.Nxd4 Bb4", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 exd4 5.Nxd4 Bb4 *"),
("C47s", "Four Knights: Scotch, 4...exd4 5.Nxd4 Bb4 6.Nxc6 bxc6", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 exd4 5.Nxd4 Bb4 6.Nxc6 bxc6 *"),
("C47s", "Four Knights: Scotch, Main Line", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 exd4 5.Nxd4 Bb4 6.Nxc6 bxc6 7.Bd3 *"),
("C47t", "Four Knights: Scotch, Main Line, 7...d5", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 exd4 5.Nxd4 Bb4 6.Nxc6 bxc6 7.Bd3 d5 *"),
("C47t", "Four Knights: Scotch, Main Line, 7...d5 8.exd5 cxd5", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 exd4 5.Nxd4 Bb4 6.Nxc6 bxc6 7.Bd3 d5 8.exd5 cxd5 *"),
("C47u", "Four Knights: Scotch, Main Line, 8.exd5 cxd5 9.O-O O-O", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 exd4 5.Nxd4 Bb4 6.Nxc6 bxc6 7.Bd3 d5 8.exd5 cxd5 9.O-O O-O *"),
("C47v", "Four Knights: Scotch, Main Line, 9.O-O O-O 10.Bg5 c6", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 exd4 5.Nxd4 Bb4 6.Nxc6 bxc6 7.Bd3 d5 8.exd5 cxd5 9.O-O O-O 10.Bg5 c6 *"),
("C47w", "Four Knights: Scotch, Main Line, 10.Bg5 c6 11.Na4", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 exd4 5.Nxd4 Bb4 6.Nxc6 bxc6 7.Bd3 d5 8.exd5 cxd5 9.O-O O-O 10.Bg5 c6 11.Na4 *"),
("C47x", "Four Knights: Scotch, Main Line, 10.Bg5 c6 11.Qf3", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 exd4 5.Nxd4 Bb4 6.Nxc6 bxc6 7.Bd3 d5 8.exd5 cxd5 9.O-O O-O 10.Bg5 c6 11.Qf3 *"),
("C47y", "Four Knights: Scotch, Main Line, 10.Bg5 c6 11.Qf3 Be7", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.d4 exd4 5.Nxd4 Bb4 6.Nxc6 bxc6 7.Bd3 d5 8.exd5 cxd5 9.O-O O-O 10.Bg5 c6 11.Qf3 Be7 *"),
("C48a", "Four Knights: Spanish Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 *"),
("C48a", "Four Knights: Spanish, Ranken Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 a6 5.Bxc6 *"),
("C48a", "Four Knights: Spanish, Spielmann Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 a6 5.Bxc6 dxc6 6.Nxe5 Nxe4 7.Nxe4 Qd4 8.O-O Qxe5 9.Re1 Be6 10.d4 Qd5 *"),
("C48b", "Four Knights: Spanish, 4...d6", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 d6 *"),
("C48c", "Four Knights: Spanish, 4...d6 5.d4", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 d6 5.d4 *"),
("C48d", "Four Knights: Spanish, Classical Defence", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bc5 *"),
("C48e", "Four Knights: Spanish, Classical Defence, 5.Nxe5", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bc5 5.Nxe5 *"),
("C48f", "Four Knights: Spanish, Classical Defence, 5.O-O", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bc5 5.O-O *"),
("C48f", "Four Knights: Spanish, Classical, Bardeleben Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bc5 5.O-O O-O 6.Nxe5 Nxe5 7.d4 Bd6 8.f4 Nc6 9.e5 Bb4 *"),
("C48f", "Four Knights: Spanish, Classical, Marshall Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bc5 5.O-O O-O 6.Nxe5 Nd4 *"),
("C48g", "Four Knights: Rubinstein Countergambit", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Nd4 *"),
("C48g", "Four Knights: Rubinstein, 5.Be2", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Nd4 5.Be2 *"),
("C48h", "Four Knights: Rubinstein, 5.Nxe5", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Nd4 5.Nxe5 *"),
("C48h", "Four Knights: Rubinstein, Bogoljubow Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Nd4 5.Nxe5 Qe7 6.f4 *"),
("C48i", "Four Knights: Rubinstein, Henneberger Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Nd4 5.O-O *"),
("C48j", "Four Knights: Rubinstein, 5.Bc4", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Nd4 5.Bc4 *"),
("C48k", "Four Knights: Rubinstein, Exchange Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Nd4 5.Nxd4 *"),
("C48k", "Four Knights: Rubinstein, Exchange Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Nd4 5.Nxd4 exd4 6.e5 dxc3 7.exf6 *"),
("C48l", "Four Knights: Rubinstein, Exchange Variation, 8...Bc5", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Nd4 5.Nxd4 exd4 6.e5 dxc3 7.exf6 Qxf6 8.dxc3 Bc5 *"),
("C48m", "Four Knights: Rubinstein, Exchange Variation, 8...Qe5+", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Nd4 5.Nxd4 exd4 6.e5 dxc3 7.exf6 Qxf6 8.dxc3 Qe5+ *"),
("C48n", "Four Knights: Rubinstein, 5.Ba4", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Nd4 5.Ba4 *"),
("C48n", "Four Knights: Rubinstein, 5.Ba4 Nxf3+", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Nd4 5.Ba4 Nxf3+ *"),
("C48o", "Four Knights: Rubinstein, 5.Ba4 c6", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Nd4 5.Ba4 c6 *"),
("C48p", "Four Knights: Rubinstein, 5.Ba4 c6 6.Nxe5", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Nd4 5.Ba4 c6 6.Nxe5 *"),
("C48q", "Four Knights: Rubinstein, 5.Ba4 Bc5", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Nd4 5.Ba4 Bc5 *"),
("C48r", "Four Knights: Rubinstein, 5.Ba4 Bc5 6.Nxe5", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Nd4 5.Ba4 Bc5 6.Nxe5 *"),
("C49a", "Four Knights: 4.Bb5 Bb4", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bb4 *"),
("C49b", "Four Knights: 4.Bb5 Bb4 5.O-O", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bb4 5.O-O *"),
("C49c", "Four Knights: 4.Bb5 Bb4 5.O-O O-O", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bb4 5.O-O O-O *"),
("C49c", "Four Knights: Gunsberg Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bb4 5.O-O O-O 6.Nd5 Nxd5 7.exd5 e4 *"),
("C49d", "Four Knights: Nimzowitsch (Paulsen) Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bb4 5.O-O O-O 6.Bxc6 *"),
("C49e", "Four Knights: Double Bishops, 6.d3", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bb4 5.O-O O-O 6.d3 *"),
("C49f", "Four Knights: Double Bishops, 6.d3 Bxc3", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bb4 5.O-O O-O 6.d3 Bxc3 *"),
("C49f", "Four Knights: Janowski Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bb4 5.O-O O-O 6.d3 Bxc3 7.bxc3 d6 8.Re1 *"),
("C49f", "Four Knights: Svenonius Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bb4 5.O-O O-O 6.d3 Bxc3 7.bxc3 d5 *"),
("C49g", "Four Knights: Symmetrical Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bb4 5.O-O O-O 6.d3 d6 *"),
("C49h", "Four Knights: Symmetrical, Maroczy System", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bb4 5.O-O O-O 6.d3 d6 7.Ne2 *"),
("C49i", "Four Knights: Symmetrical, 7.Bg5", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bb4 5.O-O O-O 6.d3 d6 7.Bg5 *"),
("C49j", "Four Knights: Symmetrical, Tarrasch Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bb4 5.O-O O-O 6.d3 d6 7.Bg5 Be6 *"),
("C49k", "Four Knights: Symmetrical, Pillsbury Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bb4 5.O-O O-O 6.d3 d6 7.Bg5 Ne7 *"),
("C49k", "Four Knights: Symmetrical, Blake Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bb4 5.O-O O-O 6.d3 d6 7.Bg5 Ne7 8.Nh4 c6 9.Bc4 d5 10.Bb3 Qd6 *"),
("C49l", "Four Knights: Symmetrical, 7.Bg5 Bxc3", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bb4 5.O-O O-O 6.d3 d6 7.Bg5 Bxc3 *"),
("C49l", "Four Knights: Symmetrical, 7.Bg5 Bxc3", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bb4 5.O-O O-O 6.d3 d6 7.Bg5 Bxc3 8.bxc3 *"),
("C49m", "Four Knights: Symmetrical, 7.Bg5 Bxc3 8.bxc3 h6", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bb4 5.O-O O-O 6.d3 d6 7.Bg5 Bxc3 8.bxc3 h6 *"),
("C49n", "Four Knights: Symmetrical, Metger unpin", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bb4 5.O-O O-O 6.d3 d6 7.Bg5 Bxc3 8.bxc3 Qe7 *"),
("C49n", "Four Knights: Symmetrical, Metger, 10.d4", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bb4 5.O-O O-O 6.d3 d6 7.Bg5 Bxc3 8.bxc3 Qe7 9.Re1 Nd8 10.d4 *"),
("C49o", "Four Knights: Symmetrical, Metger, Capablanca Variation", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bb4 5.O-O O-O 6.d3 d6 7.Bg5 Bxc3 8.bxc3 Qe7 9.Re1 Nd8 10.d4 Bg4 *"),
("C49p", "Four Knights: Symmetrical, Metger, 10.d4 Ne6", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bb4 5.O-O O-O 6.d3 d6 7.Bg5 Bxc3 8.bxc3 Qe7 9.Re1 Nd8 10.d4 Ne6 *"),
("C49q", "Four Knights: Symmetrical, Metger, 10.d4 Ne6 11.Bc1 Rd8", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bb4 5.O-O O-O 6.d3 d6 7.Bg5 Bxc3 8.bxc3 Qe7 9.Re1 Nd8 10.d4 Ne6 11.Bc1 Rd8 *"),
("C49r", "Four Knights: Symmetrical, Metger, 10.d4 Ne6 11.Bc1 c5", "1.e4 e5 2.Nf3 Nc6 3.Nc3 Nf6 4.Bb5 Bb4 5.O-O O-O 6.d3 d6 7.Bg5 Bxc3 8.bxc3 Qe7 9.Re1 Nd8 10.d4 Ne6 11.Bc1 c5 *"),
("C50a", "Italian Game", "1.e4 e5 2.Nf3 Nc6 3.Bc4 *"),
("C50b", "Italian: Rousseau Gambit", "1.e4 e5 2.Nf3 Nc6 3.Bc4 f5 *"),
("C50c", "Italian: Blackburne-Shilling Gambit", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nd4 *"),
("C50d", "Italian: 3...d6", "1.e4 e5 2.Nf3 Nc6 3.Bc4 d6 *"),
("C50e", "Italian: Hungarian Defence", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Be7 *"),
("C50f", "Italian: Hungarian Defence, 4.d4 exd4", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Be7 4.d4 exd4 *"),
("C50f", "Italian: Hungarian Defence, Tartakower Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Be7 4.d4 exd4 5.c3 Nf6 6.e5 Ne4 *"),
("C50g", "Italian: Hungarian Defence, 4.d4 d6", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Be7 4.d4 d6 *"),
("C50h", "Giuoco Piano", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 *"),
("C50h", "Giuoco Piano: Jerome Gambit", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.Bxf7+ *"),
("C50h", "Giuoco Piano: Rosentreter Gambit", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.d4 *"),
("C50i", "Giuoco Piano: Three Knights Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.Nc3 *"),
("C50j", "Giuoco Piano: Four Knights Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.Nc3 Nf6 *"),
("C50k", "Giuoco Piano: 4.O-O", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.O-O *"),
("C50l", "Giuoco Piano: 4.O-O Nf6", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.O-O Nf6 *"),
("C50m", "Giuoco Piano: 4.O-O Nf6 5.Nc3", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.O-O Nf6 5.Nc3 *"),
("C50n", "Giuoco Pianissimo: 4.d3", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.d3 *"),
("C50n", "Giuoco Pianissimo: Lucchini Gambit", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.d3 f5 *"),
("C50n", "Giuoco Pianissimo: Dubois Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.d3 f5 5.Ng5 f4 *"),
("C50o", "Giuoco Pianissimo: 4.d3 Nf6", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.d3 Nf6 *"),
("C50p", "Giuoco Pianissimo: 4.d3 Nf6 5.O-O", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.d3 Nf6 5.O-O *"),
("C50q", "Giuoco Pianissimo: 4.d3 Nf6 5.O-O d6", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.d3 Nf6 5.O-O d6 *"),
("C50r", "Giuoco Pianissimo: Italian Four Knights", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.d3 Nf6 5.Nc3 *"),
("C50s", "Giuoco Pianissimo: Italian Four Knights, 5...d6", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.d3 Nf6 5.Nc3 d6 *"),
("C50t", "Giuoco Pianissimo: Italian Four Knights, 5...d6 6.Be3", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.d3 Nf6 5.Nc3 d6 6.Be3 *"),
("C50u", "Giuoco Pianissimo: Canal Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.d3 Nf6 5.Nc3 d6 6.Bg5 *"),
("C50v", "Giuoco Pianissimo: Canal, 6...h6", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.d3 Nf6 5.Nc3 d6 6.Bg5 h6 *"),
("C51", "Evans Gambit", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 *"),
("C51", "Evans Gambit Declined: 4...Be7", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Be7 *"),
("C51", "Evans Gambit: Hein Countergambit", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 d5 *"),
("C51", "Evans Gambit Declined", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bb6 *"),
("C51", "Evans Gambit Declined, Cordel Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bb6 5.Bb2 *"),
("C51", "Evans Gambit Declined, 5.b5", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bb6 5.b5 *"),
("C51", "Evans Gambit Declined, Lange Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bb6 5.b5 Na5 6.Nxe5 Nh6 *"),
("C51", "Evans Gambit Declined, Pavlov Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bb6 5.b5 Na5 6.Nxe5 Nh6 7.d4 d6 8.Bxh6 dxe5 9.Bxg7 Rg8 10.Bxf7+ Kxf7 11.Bxe5 Qg5 12.Nd2 *"),
("C51", "Evans Gambit Declined, Hirschbach Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bb6 5.b5 Na5 6.Nxe5 Qg5 *"),
("C51", "Evans Gambit Declined, Vasquez Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bb6 5.b5 Na5 6.Nxe5 Qg5 7.Bxf7+ Ke7 8.Qh5 *"),
("C51", "Evans Gambit Declined, Hicken Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bb6 5.b5 Na5 6.Nxe5 Qg5 7.Qf3 Qxe5 8.Qxf7+ Kd8 9.Bb2 *"),
("C51", "Evans Gambit Declined, 5.a4", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bb6 5.a4 *"),
("C51", "Evans Gambit Declined, Showalter Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bb6 5.a4 a6 6.Nc3 *"),
("C51", "Evans Gambit Accepted", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 *"),
("C51", "Evans Gambit Accepted, 5.c3", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 *"),
("C51", "Evans Gambit: Mayet Defence", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Bf8 *"),
("C51", "Evans Gambit: Stone-Ware Defence", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Bd6 *"),
("C51", "Evans Gambit: Anderssen Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Be7 *"),
("C51", "Evans Gambit: Anderssen, 6.Qb3", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Be7 6.Qb3 *"),
("C51", "Evans Gambit: Anderssen, 6.d4", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Be7 6.d4 *"),
("C51", "Evans Gambit: Cordel Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Be7 6.d4 Na5 *"),
("C51", "Evans Gambit: Normal Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Bc5 6.d4 exd4 7.O-O d6 8.cxd4 Bb6 *"),
("C51", "Evans Gambit: Anderssen Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Bc5 6.d4 exd4 7.O-O d6 8.cxd4 Bb6 9.d5 *"),
("C51", "Evans Gambit: Ulvestad Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Bc5 6.d4 exd4 7.O-O d6 8.cxd4 Bb6 9.d5 Na5 10.Bb2 *"),
("C51", "Evans Gambit: Paulsen Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Bc5 6.d4 exd4 7.O-O d6 8.cxd4 Bb6 9.d5 Na5 10.Bb2 Ne7 *"),
("C51", "Evans Gambit: Morphy Attack", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Bc5 6.d4 exd4 7.O-O d6 8.cxd4 Bb6 9.Nc3 *"),
("C51", "Evans Gambit: Goering Attack", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Bc5 6.d4 exd4 7.O-O d6 8.cxd4 Bb6 9.Nc3 Na5 10.Bg5 *"),
("C51", "Evans Gambit: Steinitz Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Bc5 6.d4 exd4 7.O-O d6 8.cxd4 Bb6 9.Nc3 Na5 10.Bg5 f6 11.Be3 *"),
("C51", "Evans Gambit: Normal Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Bc5 6.d4 exd4 7.O-O d6 8.cxd4 Bb6 9.Nc3 Bg4 *"),
("C51", "Evans Gambit: Fraser Attack", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Bc5 6.d4 exd4 7.O-O d6 8.cxd4 Bb6 9.Nc3 Bg4 10.Qa4 *"),
("C51", "Evans Gambit: Fraser-Mortimer Attack", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Bc5 6.d4 exd4 7.O-O d6 8.cxd4 Bb6 9.Nc3 Bg4 10.Qa4 Bd7 11.Qb3 Na5 12.Bxf7+ Kf8 13.Qc2 *"),
("C52", "Evans Gambit: 5...Ba5", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Ba5 *"),
("C52", "Evans Gambit: 5...Ba5 6.Qb3", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Ba5 6.Qb3 *"),
("C52", "Evans Gambit: 5...Ba5 6.O-O", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Ba5 6.O-O *"),
("C52", "Evans Gambit: 5...Ba5 6.O-O Nf6", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Ba5 6.O-O Nf6 *"),
("C52", "Evans Gambit: Richardson Attack", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Ba5 6.O-O Nf6 7.d4 O-O 8.Nxe5 *"),
("C52", "Evans Gambit: 5...Ba5 6.O-O d6", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Ba5 6.O-O d6 *"),
("C52", "Evans Gambit: 5...Ba5 6.O-O d6 7.d4", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Ba5 6.O-O d6 7.d4 *"),
("C52", "Evans Gambit: Waller Attack", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Ba5 6.O-O d6 7.d4 exd4 8.Qb3 *"),
("C52", "Evans Gambit: Sanders-Alapin Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Ba5 6.O-O d6 7.d4 Bd7 *"),
("C52", "Evans Gambit: Alapin-Steinitz Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Ba5 6.O-O d6 7.d4 Bg4 *"),
("C52", "Evans Gambit: Lasker's Defence", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Ba5 6.O-O d6 7.d4 Bb6 *"),
("C52", "Evans Gambit: 5...Ba5 6.d4", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Ba5 6.d4 *"),
("C52", "Evans Gambit: Leonhardt Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Ba5 6.d4 b5 *"),
("C52", "Evans Gambit: 5...Ba5 6.d4 exd4", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Ba5 6.d4 exd4 *"),
("C52", "Evans Gambit: Compromised Defence", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Ba5 6.d4 exd4 7.O-O dxc3 *"),
("C52", "Evans Gambit: Compromised Defence, Paulsen Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Ba5 6.d4 exd4 7.O-O dxc3 8.Qb3 Qf6 9.e5 Qg6 10.Nxc3 Nge7 11.Ba3 *"),
("C52", "Evans Gambit: Compromised Defence, Potter Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Ba5 6.d4 exd4 7.O-O dxc3 8.Qb3 Qf6 9.e5 Qg6 10.Nxc3 Nge7 11.Rd1 *"),
("C52", "Evans Gambit: 5...Ba5 6.d4 d6", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Ba5 6.d4 d6 *"),
("C52", "Evans Gambit: Sokolsky Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Ba5 6.d4 d6 7.Bg5 *"),
("C52", "Evans Gambit: Tartakower Attack", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Ba5 6.d4 d6 7.Qb3 *"),
("C52", "Evans Gambit: Tartakower Attack, 7...Qd7", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Ba5 6.d4 d6 7.Qb3 Qd7 *"),
("C52", "Evans Gambit: Tartakower Attack, 8.dxe5 Bb6", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Ba5 6.d4 d6 7.Qb3 Qd7 8.dxe5 Bb6 *"),
("C52", "Evans Gambit: Tartakower Attack, 8.dxe5 dxe5", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Ba5 6.d4 d6 7.Qb3 Qd7 8.dxe5 dxe5 *"),
("C52", "Evans Gambit: Tartakower, Levenfish Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.b4 Bxb4 5.c3 Ba5 6.d4 d6 7.Qb3 Qd7 8.dxe5 dxe5 9.O-O Bb6 10.Ba3 Na5 11.Nxe5 *"),
("C53", "Giuoco Piano: 4.c3", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 *"),
("C53", "Giuoco Piano: LaBourdonnais Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 d6 5.d4 exd4 6.cxd4 Bb6 *"),
("C53", "Giuoco Piano: Close Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Qe7 *"),
("C53", "Giuoco Piano: Centre-Holding Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Qe7 5.d4 Bb6 *"),
("C53", "Giuoco Piano: Tarrasch Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Qe7 5.d4 Bb6 6.O-O Nf6 7.a4 a6 8.Re1 d6 9.h3 *"),
("C53", "Giuoco Piano: Mestel Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Qe7 5.d4 Bb6 6.Bg5 *"),
("C53", "Giuoco Piano: Eisinger Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Qe7 5.d4 Bb6 6.d5 Nb8 7.d6 *"),
("C54a", "Giuoco Piano: 4.c3 Nf6", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 *"),
("C54b", "Giuoco Piano: Albin Gambit", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.O-O *"),
("C54c", "Giuoco Piano: Bird's Attack", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.b4 *"),
("C54c", "Giuoco Piano: Bird's Attack", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.b4 Bb6 6.a4 *"),
("C54d", "Giuoco Pianissimo: 5.d3", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d3 *"),
("C54e", "Giuoco Pianissimo: 5.d3 a6", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d3 a6 *"),
("C54f", "Giuoco Pianissimo: 5.d3 d6", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d3 d6 *"),
("C54g", "Giuoco Pianissimo: 5.d3 d6 6.Nbd2", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d3 d6 6.Nbd2 *"),
("C54h", "Giuoco Pianissimo: 5.d3 d6 6.O-O", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d3 d6 6.O-O *"),
("C54i", "Giuoco Pianissimo: 5.d3 d6 6.O-O O-O", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d3 d6 6.O-O O-O *"),
("C54j", "Giuoco Piano: 5.d4", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d4 *"),
("C54k", "Giuoco Piano: 6.e5", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d4 exd4 6.e5 *"),
("C54k", "Giuoco Piano: Ghulam Kassim Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d4 exd4 6.e5 Ne4 7.Bd5 Nxf2 8.Kxf2 dxc3+ 9.Kg3 *"),
("C54k", "Giuoco Piano: 6.e5 d5", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d4 exd4 6.e5 d5 *"),
("C54k", "Giuoco Piano: Anderssen Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d4 exd4 6.e5 d5 7.Bb5 Ne4 8.cxd4 Bb4+ *"),
("C54l", "Giuoco Piano: 6.cxd4", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d4 exd4 6.cxd4 *"),
("C54l", "Giuoco Piano: 6.cxd4 Bb4+", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d4 exd4 6.cxd4 Bb4+ *"),
("C54l", "Giuoco Piano: Krakow Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d4 exd4 6.cxd4 Bb4+ 7.Kf1 *"),
("C54m", "Giuoco Piano: 6.cxd4 Bb4+ 7.Bd2", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d4 exd4 6.cxd4 Bb4+ 7.Bd2 *"),
("C54m", "Giuoco Piano: Krause Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d4 exd4 6.cxd4 Bb4+ 7.Bd2 Nxe4 8.Bxb4 Nxb4 9.Bxf7+ Kxf7 10.Qb3+ d5 11.Ne5+ Kf6 12.f3 *"),
("C54n", "Giuoco Piano: Greco Attack", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d4 exd4 6.cxd4 Bb4+ 7.Nc3 *"),
("C54n", "Giuoco Piano: Greco Attack", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d4 exd4 6.cxd4 Bb4+ 7.Nc3 Nxe4 8.O-O Nxc3 *"),
("C54n", "Giuoco Piano: Bernstein Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d4 exd4 6.cxd4 Bb4+ 7.Nc3 Nxe4 8.O-O Nxc3 9.bxc3 Bxc3 10.Qb3 d5 *"),
("C54n", "Giuoco Piano: Aitken Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d4 exd4 6.cxd4 Bb4+ 7.Nc3 Nxe4 8.O-O Nxc3 9.bxc3 Bxc3 10.Ba3 *"),
("C54o", "Giuoco Piano: Greco Attack", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d4 exd4 6.cxd4 Bb4+ 7.Nc3 Nxe4 8.O-O Bxc3 *"),
("C54o", "Giuoco Piano: Steinitz Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d4 exd4 6.cxd4 Bb4+ 7.Nc3 Nxe4 8.O-O Bxc3 9.bxc3 d5 10.Ba3 *"),
("C54o", "Giuoco Piano: Moeller (Therkatz) Attack", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d4 exd4 6.cxd4 Bb4+ 7.Nc3 Nxe4 8.O-O Bxc3 9.d5 *"),
("C54o", "Giuoco Piano: Moeller-Herzog Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d4 exd4 6.cxd4 Bb4+ 7.Nc3 Nxe4 8.O-O Bxc3 9.d5 Bf6 10.Re1 Ne7 11.Rxe4 d6 12.Bg5 Bxg5 13.Nxg5 O-O 14.Nxh7 *"),
("C54o", "Giuoco Piano: Moeller, Bayonet Attack", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Bc5 4.c3 Nf6 5.d4 exd4 6.cxd4 Bb4+ 7.Nc3 Nxe4 8.O-O Bxc3 9.d5 Bf6 10.Re1 Ne7 11.Rxe4 d6 12.g4 *"),
("C55a", "Two Knights Defence", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 *"),
("C55b", "Two Knights: Deutz Gambit", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.O-O Bc5 5.d4 *"),
("C55b", "Two Knights: 4.O-O Gambit, Rosentreter Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.O-O Bc5 5.d4 Bxd4 6.Nxd4 Nxd4 7.Bg5 h6 8.Bh4 g5 9.f4 *"),
("C55b", "Two Knights: 4.O-O Gambit, Holzhausen Attack", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.O-O Bc5 5.d4 Bxd4 6.Nxd4 Nxd4 7.Bg5 d6 8.f4 Qe7 9.fxe5 dxe5 10.Nc3 *"),
("C55c", "Two Knights: 4.d3", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d3 *"),
("C55d", "Two Knights: 4.d3 h6", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d3 h6 *"),
("C55e", "Two Knights: 4.d3 Be7", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d3 Be7 *"),
("C55e", "Two Knights: 4.d3 Be7 5.Bb3 O-O", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d3 Be7 5.Bb3 O-O *"),
("C55f", "Two Knights: 4.d3 Be7 5.c3", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d3 Be7 5.c3 *"),
("C55g", "Two Knights: 4.d3 Be7 5.O-O", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d3 Be7 5.O-O *"),
("C55h", "Two Knights: 4.d3 Be7 5.O-O O-O", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d3 Be7 5.O-O O-O *"),
("C55i", "Two Knights: 4.d3 Be7 5.O-O O-O 6.Bb3", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d3 Be7 5.O-O O-O 6.Bb3 *"),
("C55j", "Two Knights: 4.d3 Be7 5.O-O O-O 6.Bb3 d6 7.c3", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d3 Be7 5.O-O O-O 6.Bb3 d6 7.c3 *"),
("C55k", "Two Knights: 4.d3 Be7 5.O-O O-O 6.Re1", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d3 Be7 5.O-O O-O 6.Re1 *"),
("C55l", "Two Knights: 4.d4", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 *"),
("C55l", "Two Knights: 4.d4 exd4", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 *"),
("C55m", "Two Knights: 4.d4 exd4 5.Nxd4", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.Nxd4 *"),
("C55n", "Two Knights: Perreux Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.Ng5 *"),
("C55o", "Two Knights: Modern Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.e5 *"),
("C55p", "Two Knights: Modern, 5.e5 d5", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.e5 d5 *"),
("C55q", "Two Knights: Modern, Main Line", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.e5 d5 6.Bb5 Ne4 7.Nxd4 Bd7 *"),
("C55r", "Two Knights: Modern, Main Line, 8.Bxc6 bxc6 9.O-O Bc5", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.e5 d5 6.Bb5 Ne4 7.Nxd4 Bd7 8.Bxc6 bxc6 9.O-O Bc5 *"),
("C55s", "Two Knights: 5.O-O", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O *"),
("C55s", "Two Knights: 5.O-O Be7", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Be7 *"),
("C55t", "Two Knights: 5.O-O d6", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O d6 *"),
("C55u", "Two Knights: Max Lange Attack", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Bc5 *"),
("C55v", "Two Knights: Max Lange, Steinitz Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Bc5 6.e5 Ng4 *"),
("C55v", "Two Knights: Max Lange, Krause Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Bc5 6.e5 Ng4 7.c3 *"),
("C55w", "Two Knights: Max Lange, 6.e5 d5", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Bc5 6.e5 d5 *"),
("C55w", "Two Knights: Max Lange, Schlechter Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Bc5 6.e5 d5 7.exf6 dxc4 8.Re1+ Be6 9.fxg7 *"),
("C55x", "Two Knights: Max Lange, Berger Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Bc5 6.e5 d5 7.exf6 dxc4 8.Re1+ Be6 9.Ng5 Qd5 10.Nc3 Qf5 11.g4 Qg6 12.Nce4 Bb6 13.f4 O-O-O *"),
("C55x", "Two Knights: Max Lange, Marshall Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Bc5 6.e5 d5 7.exf6 dxc4 8.Re1+ Be6 9.Ng5 Qd5 10.Nc3 Qf5 11.Nce4 *"),
("C55x", "Two Knights: Max Lange, Rubinstein Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Bc5 6.e5 d5 7.exf6 dxc4 8.Re1+ Be6 9.Ng5 Qd5 10.Nc3 Qf5 11.Nce4 Bf8 *"),
("C55x", "Two Knights: Max Lange, Loman Defence", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Bc5 6.e5 d5 7.exf6 dxc4 8.Re1+ Be6 9.Ng5 g6 *"),
("C56a", "Two Knights: Classical", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Nxe4 *"),
("C56a", "Two Knights: Classical, Nakhmanson Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Nxe4 6.Nc3 *"),
("C56b", "Two Knights: Classical, 6.Re1", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Nxe4 6.Re1 *"),
("C56b", "Two Knights: Classical, 6.Re1 d5", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Nxe4 6.Re1 d5 *"),
("C56c", "Two Knights: Classical, Canal Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Nxe4 6.Re1 d5 7.Nc3 *"),
("C56d", "Two Knights: Classical, 7.Bxd5", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Nxe4 6.Re1 d5 7.Bxd5 *"),
("C56d", "Two Knights: Classical, 7.Bxd5 Qxd5", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Nxe4 6.Re1 d5 7.Bxd5 Qxd5 *"),
("C56d", "Two Knights: Classical, 8.Nc3", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Nxe4 6.Re1 d5 7.Bxd5 Qxd5 8.Nc3 *"),
("C56e", "Two Knights: Classical, 8.Nc3 Qd8", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Nxe4 6.Re1 d5 7.Bxd5 Qxd5 8.Nc3 Qd8 *"),
("C56f", "Two Knights: Classical, 8.Nc3 Qh5", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Nxe4 6.Re1 d5 7.Bxd5 Qxd5 8.Nc3 Qh5 *"),
("C56g", "Two Knights: Classical, 8.Nc3 Qh5 9.Nxe4 Be6 10.Bg5 Bd6", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Nxe4 6.Re1 d5 7.Bxd5 Qxd5 8.Nc3 Qh5 9.Nxe4 Be6 10.Bg5 Bd6 *"),
("C56h", "Two Knights: Classical, Moller Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Nxe4 6.Re1 d5 7.Bxd5 Qxd5 8.Nc3 Qa5 *"),
("C56h", "Two Knights: Classical, Moller, 9.Nxe4", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Nxe4 6.Re1 d5 7.Bxd5 Qxd5 8.Nc3 Qa5 9.Nxe4 *"),
("C56h", "Two Knights: Classical, Moller, 9.Nxe4 Be6", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Nxe4 6.Re1 d5 7.Bxd5 Qxd5 8.Nc3 Qa5 9.Nxe4 Be6 *"),
("C56i", "Two Knights: Classical, Moller, 10.Bg5", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Nxe4 6.Re1 d5 7.Bxd5 Qxd5 8.Nc3 Qa5 9.Nxe4 Be6 10.Bg5 *"),
("C56i", "Two Knights: Classical, Yurdansky Attack", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Nxe4 6.Re1 d5 7.Bxd5 Qxd5 8.Nc3 Qa5 9.Nxe4 Be6 10.Bg5 h6 11.Bh4 g5 12.Nf6+ Ke7 13.b4 *"),
("C56j", "Two Knights: Classical, Moller, 10.Neg5", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Nxe4 6.Re1 d5 7.Bxd5 Qxd5 8.Nc3 Qa5 9.Nxe4 Be6 10.Neg5 *"),
("C56k", "Two Knights: Classical, Moller, 10.Neg5 O-O-O 11.Nxe6 fxe6 12.Rxe6 Bd6", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Nxe4 6.Re1 d5 7.Bxd5 Qxd5 8.Nc3 Qa5 9.Nxe4 Be6 10.Neg5 O-O-O 11.Nxe6 fxe6 12.Rxe6 Bd6 *"),
("C56l", "Two Knights: Classical, Moller, 10.Bd2", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Nxe4 6.Re1 d5 7.Bxd5 Qxd5 8.Nc3 Qa5 9.Nxe4 Be6 10.Bd2 *"),
("C56m", "Two Knights: Classical, Moller, 10.Bd2 Qf5", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Nxe4 6.Re1 d5 7.Bxd5 Qxd5 8.Nc3 Qa5 9.Nxe4 Be6 10.Bd2 Qf5 *"),
("C56n", "Two Knights: Classical, Moller, 10.Bd2 Qd5", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Nxe4 6.Re1 d5 7.Bxd5 Qxd5 8.Nc3 Qa5 9.Nxe4 Be6 10.Bd2 Qd5 *"),
("C56o", "Two Knights: Classical, Moller, 10.Bd2 Bb4", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.d4 exd4 5.O-O Nxe4 6.Re1 d5 7.Bxd5 Qxd5 8.Nc3 Qa5 9.Nxe4 Be6 10.Bd2 Bb4 *"),
("C57a", "Two Knights: 4.Ng5", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 *"),
("C57a", "Two Knights: Traxler (Wilkes-Barre) Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 Bc5 *"),
("C57a", "Two Knights: Traxler, 5.d4", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 Bc5 5.d4 *"),
("C57b", "Two Knights: Traxler, 5.Nxf7", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 Bc5 5.Nxf7 *"),
("C57b", "Two Knights: Traxler, 5.Nxf7 & 6.Kxf2", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 Bc5 5.Nxf7 Bxf2+ 6.Kxf2 *"),
("C57c", "Two Knights: Traxler, 5.Nxf7 & 6.Kf1", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 Bc5 5.Nxf7 Bxf2+ 6.Kf1 *"),
("C57c", "Two Knights: Traxler, 5.Nxf7 & 6.Kf1, Beyer's 8...Bg4", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 Bc5 5.Nxf7 Bxf2+ 6.Kf1 Qe7 7.Nxh8 d5 8.exd5 Bg4 *"),
("C57c", "Two Knights: Traxler, 5.Nxf7 & 6.Kf1, 8...Nd4", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 Bc5 5.Nxf7 Bxf2+ 6.Kf1 Qe7 7.Nxh8 d5 8.exd5 Nd4 *"),
("C57c", "Two Knights: Traxler, 5.Nxf7 & 6.Kf1, Palkin Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 Bc5 5.Nxf7 Bxf2+ 6.Kf1 Qe7 7.Nxh8 d5 8.exd5 Nd4 9.h3 *"),
("C57c", "Two Knights: Traxler, 5.Nxf7 & 6.Kf1, Menovsky Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 Bc5 5.Nxf7 Bxf2+ 6.Kf1 Qe7 7.Nxh8 d5 8.exd5 Nd4 9.d6 *"),
("C57d", "Two Knights: Traxler, 5.Bxf7+", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 Bc5 5.Bxf7+ *"),
("C57d", "Two Knights: Traxler, 5.Bxf7+ Ke7 6.Bb3", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 Bc5 5.Bxf7+ Ke7 6.Bb3 *"),
("C57d", "Two Knights: Traxler, Chigorin/Pithart Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 Bc5 5.Bxf7+ Ke7 6.Bd5 *"),
("C57e", "Two Knights: 4.Ng5 d5", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 *"),
("C57e", "Two Knights: Kloss Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Nb4 *"),
("C57e", "Two Knights: 4.Ng5 d5 5.exd5 Nxd5", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Nxd5 *"),
("C57f", "Two Knights: Lolli Attack", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Nxd5 6.d4 *"),
("C57f", "Two Knights: Lolli Attack, Pinkus Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Nxd5 6.d4 Bb4+ *"),
("C57g", "Two Knights: Fegatello (Fried Liver) Attack", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Nxd5 6.Nxf7 *"),
("C57g", "Two Knights: Fegatello, Leonhardt Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Nxd5 6.Nxf7 Kxf7 7.Qf3+ Ke6 8.Nc3 Nb4 9.Qe4 c6 10.a3 Na6 11.d4 Nc7 *"),
("C57g", "Two Knights: Fegatello, Polerio Defence", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Nxd5 6.Nxf7 Kxf7 7.Qf3+ Ke6 8.Nc3 Ne7 *"),
("C57h", "Two Knights: Ulvestad Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 b5 *"),
("C57i", "Two Knights: Fritz Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Nd4 *"),
("C57i", "Two Knights: Fritz Variation, Main Line", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Nd4 6.c3 b5 7.Bf1 Nxd5 *"),
("C57j", "Two Knights: Fritz, 8.Nxf7", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Nd4 6.c3 b5 7.Bf1 Nxd5 8.Nxf7 *"),
("C57j", "Two Knights: Fritz, 8.cxd5", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Nd4 6.c3 b5 7.Bf1 Nxd5 8.cxd4 *"),
("C57j", "Two Knights: Fritz, Paoli's 8.h4", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Nd4 6.c3 b5 7.Bf1 Nxd5 8.h4 *"),
("C57k", "Two Knights: Fritz, Gruber Variation (8.Ne4)", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Nd4 6.c3 b5 7.Bf1 Nxd5 8.Ne4 *"),
("C57l", "Two Knights: Fritz, Radchenko Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Nd4 6.c3 b5 7.Bf1 Nxd5 8.Ne4 Qh4 9.Ng3 Bb7 *"),
("C57m", "Two Knights: Fritz, Berliner Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Nd4 6.c3 b5 7.Bf1 Nxd5 8.Ne4 Qh4 9.Ng3 Bg4 10.f3 e4 *"),
("C58a", "Two Knights: Morphy Variation (5...Na5)", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 *"),
("C58b", "Two Knights: Morphy, Kieseritsky Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.d3 *"),
("C58c", "Two Knights: Morphy, Kieseritsky Variation, 6...h6 7.Nf3 e4 8.Qe2", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.d3 h6 7.Nf3 e4 8.Qe2 *"),
("C58c", "Two Knights: Morphy, Yankovich Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.d3 h6 7.Nf3 e4 8.Qe2 Nxc4 9.dxc4 Bc5 10.Nfd2 *"),
("C58c", "Two Knights: Morphy, Maroczy Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.d3 h6 7.Nf3 e4 8.Qe2 Nxc4 9.dxc4 Be7 *"),
("C58d", "Two Knights: Morphy, Polerio (6.Bb5+)", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ *"),
("C58d", "Two Knights: Morphy, Polerio, 6...Bd7", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ Bd7 *"),
("C58e", "Two Knights: Morphy, Polerio, 6...c6", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ c6 *"),
("C58f", "Two Knights: Morphy, Bogoljubow Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ c6 7.dxc6 bxc6 8.Qf3 *"),
("C58f", "Two Knights: Morphy, Blackburne Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ c6 7.dxc6 bxc6 8.Qf3 cxb5 *"),
("C58f", "Two Knights: Morphy, Paoli Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ c6 7.dxc6 bxc6 8.Qf3 Qc7 9.Bd3 *"),
("C58g", "Two Knights: Morphy, Colman Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ c6 7.dxc6 bxc6 8.Qf3 Rb8 *"),
("C58h", "Two Knights: Morphy, 8.Be2", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ c6 7.dxc6 bxc6 8.Be2 *"),
("C59a", "Two Knights: Morphy, 8.Be2 h6", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ c6 7.dxc6 bxc6 8.Be2 h6 *"),
("C59b", "Two Knights: Morphy, Steinitz Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ c6 7.dxc6 bxc6 8.Be2 h6 9.Nh3 *"),
("C59c", "Two Knights: Morphy, Main Line 9.Nf3", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ c6 7.dxc6 bxc6 8.Be2 h6 9.Nf3 *"),
("C59c", "Two Knights: Morphy, Main Line 9.Nf3 e4", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ c6 7.dxc6 bxc6 8.Be2 h6 9.Nf3 e4 *"),
("C59c", "Two Knights: Morphy, Main Line 9.Nf3 e4 10.Ne5", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ c6 7.dxc6 bxc6 8.Be2 h6 9.Nf3 e4 10.Ne5 *"),
("C59d", "Two Knights: Morphy, Steiner Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ c6 7.dxc6 bxc6 8.Be2 h6 9.Nf3 e4 10.Ne5 Bc5 *"),
("C59e", "Two Knights: Morphy, Goering Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ c6 7.dxc6 bxc6 8.Be2 h6 9.Nf3 e4 10.Ne5 Qc7 *"),
("C59f", "Two Knights: Morphy, Geller Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ c6 7.dxc6 bxc6 8.Be2 h6 9.Nf3 e4 10.Ne5 Qd4 *"),
("C59g", "Two Knights: Main Line 10...Bd6", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ c6 7.dxc6 bxc6 8.Be2 h6 9.Nf3 e4 10.Ne5 Bd6 *"),
("C59h", "Two Knights: Main Line, 11.f4", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ c6 7.dxc6 bxc6 8.Be2 h6 9.Nf3 e4 10.Ne5 Bd6 11.f4 *"),
("C59i", "Two Knights: Main Line, 11.f4 exf3", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ c6 7.dxc6 bxc6 8.Be2 h6 9.Nf3 e4 10.Ne5 Bd6 11.f4 exf3 *"),
("C59j", "Two Knights: Main Line, 11.d4", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ c6 7.dxc6 bxc6 8.Be2 h6 9.Nf3 e4 10.Ne5 Bd6 11.d4 *"),
("C59j", "Two Knights: Main Line, 11.d4 Qc7", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ c6 7.dxc6 bxc6 8.Be2 h6 9.Nf3 e4 10.Ne5 Bd6 11.d4 Qc7 *"),
("C59j", "Two Knights: Main Line, Knorre Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ c6 7.dxc6 bxc6 8.Be2 h6 9.Nf3 e4 10.Ne5 Bd6 11.d4 Qc7 12.Bd2 *"),
("C59k", "Two Knights: Main Line, 11.d4 exd3", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ c6 7.dxc6 bxc6 8.Be2 h6 9.Nf3 e4 10.Ne5 Bd6 11.d4 exd3 *"),
("C59k", "Two Knights: Main Line, 11.d4 exd3 12.Nxd3 Qc7", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ c6 7.dxc6 bxc6 8.Be2 h6 9.Nf3 e4 10.Ne5 Bd6 11.d4 exd3 12.Nxd3 Qc7 *"),
("C59l", "Two Knights: Main Line, 11.d4, Honfi Variation", "1.e4 e5 2.Nf3 Nc6 3.Bc4 Nf6 4.Ng5 d5 5.exd5 Na5 6.Bb5+ c6 7.dxc6 bxc6 8.Be2 h6 9.Nf3 e4 10.Ne5 Bd6 11.d4 exd3 12.Nxd3 Qc7 13.b3 *"),
("C60a", "Spanish (Ruy Lopez)", "1.e4 e5 2.Nf3 Nc6 3.Bb5 *"),
("C60a", "Spanish: Spanish Countergambit", "1.e4 e5 2.Nf3 Nc6 3.Bb5 d5 *"),
("C60b", "Spanish: 3...a5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a5 *"),
("C60c", "Spanish: Nuernberg Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 f6 *"),
("C60d", "Spanish: Pollock Defence", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Na5 *"),
("C60e", "Spanish: Lucena Defence", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Be7 *"),
("C60f", "Spanish: Vinogradov Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Qe7 *"),
("C60g", "Spanish: Brentano Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 g5 *"),
("C60h", "Spanish: Alapin Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bb4 *"),
("C60i", "Spanish: Alapin, 4.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bb4 4.c3 *"),
("C60j", "Spanish: Fianchetto (Smyslov) Defence", "1.e4 e5 2.Nf3 Nc6 3.Bb5 g6 *"),
("C60k", "Spanish: Fianchetto, 4.O-O", "1.e4 e5 2.Nf3 Nc6 3.Bb5 g6 4.O-O *"),
("C60l", "Spanish: Fianchetto, 4.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 g6 4.d4 *"),
("C60m", "Spanish: Fianchetto, 4.d4 exd4 5.Bg5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 g6 4.d4 exd4 5.Bg5 *"),
("C60n", "Spanish: Fianchetto, 4.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 g6 4.c3 *"),
("C60o", "Spanish: Cozio Defence", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nge7 *"),
("C60p", "Spanish: Cozio, 4.Nc3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nge7 4.Nc3 *"),
("C60q", "Spanish: Cozio, Paulsen Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nge7 4.Nc3 g6 *"),
("C60r", "Spanish: Cozio, 4.O-O", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nge7 4.c3 *"),
("C60s", "Spanish: Cozio, 4.O-O", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nge7 4.O-O *"),
("C60t", "Spanish: Cozio, 4.O-O g6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nge7 4.O-O g6 *"),
("C60u", "Spanish: Cozio, 4.O-O g6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nge7 4.O-O g6 5.c3 *"),
("C61a", "Spanish: Bird's Defence", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nd4 *"),
("C61b", "Spanish: Bird's, 4.Bc4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nd4 4.Bc4 *"),
("C61c", "Spanish: Bird's, 4.Nxd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nd4 4.Nxd4 *"),
("C61c", "Spanish: Bird's, 4.Nxd4 exd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nd4 4.Nxd4 exd4 *"),
("C61d", "Spanish: Bird's, 5.d3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nd4 4.Nxd4 exd4 5.d3 *"),
("C61e", "Spanish: Bird's, 5.Bc4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nd4 4.Nxd4 exd4 5.Bc4 *"),
("C61f", "Spanish: Bird's, 5.O-O", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nd4 4.Nxd4 exd4 5.O-O *"),
("C61f", "Spanish: Bird's, Paulsen Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nd4 4.Nxd4 exd4 5.O-O Ne7 *"),
("C61g", "Spanish: Bird's, 5.O-O c6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nd4 4.Nxd4 exd4 5.O-O c6 *"),
("C61h", "Spanish: Bird's, 5.O-O c6 6.Bc4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nd4 4.Nxd4 exd4 5.O-O c6 6.Bc4 *"),
("C61i", "Spanish: Bird's, 5.O-O Bc5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nd4 4.Nxd4 exd4 5.O-O Bc5 *"),
("C61j", "Spanish: Bird's, 5.O-O Bc5 6.d3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nd4 4.Nxd4 exd4 5.O-O Bc5 6.d3 *"),
("C61k", "Spanish: Bird's, 5.O-O Bc5 6.d3 c6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nd4 4.Nxd4 exd4 5.O-O Bc5 6.d3 c6 *"),
("C61l", "Spanish: Bird's, 5.O-O Bc5 6.d3 c6 7.Bc4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nd4 4.Nxd4 exd4 5.O-O Bc5 6.d3 c6 7.Bc4 *"),
("C61m", "Spanish: Bird's, 5.O-O Bc5 6.d3 c6 7.Bc4 d5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nd4 4.Nxd4 exd4 5.O-O Bc5 6.d3 c6 7.Bc4 d5 *"),
("C62", "Spanish: Old Steinitz", "1.e4 e5 2.Nf3 Nc6 3.Bb5 d6 *"),
("C62", "Spanish: Old Steinitz, 4.Bxc6+", "1.e4 e5 2.Nf3 Nc6 3.Bb5 d6 4.Bxc6+ *"),
("C62", "Spanish: Old Steinitz, 4.O-O", "1.e4 e5 2.Nf3 Nc6 3.Bb5 d6 4.O-O *"),
("C62", "Spanish: Old Steinitz, 4.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 d6 4.c3 *"),
("C62", "Spanish: Old Steinitz, 4.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 d6 4.d4 *"),
("C62", "Spanish: Old Steinitz, 4.d4 exd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 d6 4.d4 exd4 *"),
("C62", "Spanish: Old Steinitz, 4.d4 exd4 5.Qxd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 d6 4.d4 exd4 5.Qxd4 *"),
("C62", "Spanish: Old Steinitz, 4.d4 exd4 5.Nxd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 d6 4.d4 exd4 5.Nxd4 *"),
("C62", "Spanish: Old Steinitz, 4.d4 Bd7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 d6 4.d4 Bd7 *"),
("C62", "Spanish: Old Steinitz, 4.d4 Bd7 5.Nc3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 d6 4.d4 Bd7 5.Nc3 *"),
("C62", "Spanish: Old Steinitz, Nimzowitsch Attack", "1.e4 e5 2.Nf3 Nc6 3.Bb5 d6 4.d4 Bd7 5.Nc3 Nf6 6.Bxc6 *"),
("C62", "Spanish: Old Steinitz, Semi-Duras Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 d6 4.d4 Bd7 5.c4 *"),
("C63a", "Spanish: Schliemann (Jaenisch)", "1.e4 e5 2.Nf3 Nc6 3.Bb5 f5 *"),
("C63a", "Spanish: Schliemann, 4.Qe2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 f5 4.Qe2 *"),
("C63b", "Spanish: Schliemann, 4.exf5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 f5 4.exf5 *"),
("C63c", "Spanish: Schliemann, 4.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 f5 4.d4 *"),
("C63d", "Spanish: Schliemann, 4.d3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 f5 4.d3 *"),
("C63e", "Spanish: Schliemann, 4.d3 fxe4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 f5 4.d3 fxe4 *"),
("C63f", "Spanish: Schliemann, 4.d3 fxe4 5.dxe4 Nf6 6.O-O", "1.e4 e5 2.Nf3 Nc6 3.Bb5 f5 4.d3 fxe4 5.dxe4 Nf6 6.O-O *"),
("C63g", "Spanish: Schliemann, 4.Bxc6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 f5 4.Bxc6 *"),
("C63h", "Spanish: Schliemann, 4.Bxc6 dxc6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 f5 4.Bxc6 dxc6 *"),
("C63i", "Spanish: Schliemann, 4.Nc3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 f5 4.Nc3 *"),
("C63j", "Spanish: Schliemann, 4.Nc3 Nd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 f5 4.Nc3 Nd4 *"),
("C63k", "Spanish: Schliemann, 4.Nc3 Nd4 5.Ba4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 f5 4.Nc3 Nd4 5.Ba4 *"),
("C63l", "Spanish: Schliemann, 4.Nc3 Nf6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 f5 4.Nc3 Nf6 *"),
("C63m", "Spanish: Schliemann, 4.Nc3 Nf6 5.exf5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 f5 4.Nc3 Nf6 5.exf5 *"),
("C63n", "Spanish: Schliemann, 4.Nc3 fxe4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 f5 4.Nc3 fxe4 *"),
("C63o", "Spanish: Schliemann, 4.Nc3 fxe4 5.Nxe4 Nf6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 f5 4.Nc3 fxe4 5.Nxe4 Nf6 *"),
("C63p", "Spanish: Schliemann, 4.Nc3 fxe4 5.Nxe4 Nf6 6.Qe2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 f5 4.Nc3 fxe4 5.Nxe4 Nf6 6.Qe2 *"),
("C63q", "Spanish: Schliemann, 4.Nc3 fxe4 5.Nxe4 Nf6 6.Nxf6+", "1.e4 e5 2.Nf3 Nc6 3.Bb5 f5 4.Nc3 fxe4 5.Nxe4 Nf6 6.Nxf6+ *"),
("C63r", "Spanish: Schliemann, 4.Nc3 fxe4 5.Nxe4 d5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 f5 4.Nc3 fxe4 5.Nxe4 d5 *"),
("C63s", "Spanish: Schliemann, 4.Nc3 fxe4 5.Nxe4 d5 6.Nxe5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 f5 4.Nc3 fxe4 5.Nxe4 d5 6.Nxe5 *"),
("C63t", "Spanish: Schliemann, 4.Nc3 fxe4 5.Nxe4 d5 6.Nxe5 dxe4 7.Nxc6 Qd5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 f5 4.Nc3 fxe4 5.Nxe4 d5 6.Nxe5 dxe4 7.Nxc6 Qd5 *"),
("C63u", "Spanish: Schliemann, 4.Nc3 fxe4 5.Nxe4 d5 6.Nxe5 dxe4 7.Nxc6 Qg5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 f5 4.Nc3 fxe4 5.Nxe4 d5 6.Nxe5 dxe4 7.Nxc6 Qg5 *"),
("C64a", "Spanish: Classical Defence", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 *"),
("C64a", "Spanish: Classical, Exchange", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.Bxc6 *"),
("C64b", "Spanish: Classical, 4.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.c3 *"),
("C64b", "Spanish: Classical, Boden Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.c3 Qe7 *"),
("C64c", "Spanish: Classical, Charousek Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.c3 Bb6 *"),
("C64d", "Spanish: Classical, 4.c3 Nge7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.c3 Nge7 *"),
("C64e", "Spanish: Classical, 4.c3 Qf6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.c3 Qf6 *"),
("C64f", "Spanish: Classical, Cordel Gambit", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.c3 f5 *"),
("C64g", "Spanish: Classical, Cordel Gambit, 5.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.c3 f5 5.d4 *"),
("C64h", "Spanish: Classical, 4.c3 Nf6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.c3 Nf6 *"),
("C64i", "Spanish: Classical, 4.c3 Nf6 5.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.c3 Nf6 5.d4 *"),
("C64j", "Spanish: Classical, 4.c3 Nf6 5.d4 Bb6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.c3 Nf6 5.d4 Bb6 *"),
("C64k", "Spanish: Classical, 4.O-O", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.O-O *"),
("C64l", "Spanish: Classical, 4.O-O Nge7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.O-O Nge7 *"),
("C64m", "Spanish: Classical, 4.O-O Qf6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.O-O Qf6 *"),
("C64n", "Spanish: Classical, 4.O-O d6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.O-O d6 *"),
("C64o", "Spanish: Classical, 4.O-O d6 5.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.O-O d6 5.c3 *"),
("C64p", "Spanish: Classical, 4.O-O Nd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.O-O Nd4 *"),
("C64q", "Spanish: Classical, Zaitsev Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.O-O Nd4 5.b4 *"),
("C64r", "Spanish: Classical, 4.O-O Nd4 5.Bc4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.O-O Nd4 5.Bc4 *"),
("C64s", "Spanish: Classical, 4.O-O Nd4 5.Nxd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.O-O Nd4 5.Nxd4 *"),
("C64s", "Spanish: Classical, 4.O-O Nd4 5.Nxd4 Bxd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.O-O Nd4 5.Nxd4 Bxd4 *"),
("C64s", "Spanish: Classical, 4.O-O Nd4 5.Nxd4 Bxd4 6.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.O-O Nd4 5.Nxd4 Bxd4 6.c3 *"),
("C64s", "Spanish: Classical, 4.O-O Nd4 5.Nxd4 Bxd4 6.c3 Bb6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.O-O Nd4 5.Nxd4 Bxd4 6.c3 Bb6 *"),
("C64t", "Spanish: Classical, 4.O-O Nd4 5.Nxd4 Bxd4 6.c3 Bb6 7.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.O-O Nd4 5.Nxd4 Bxd4 6.c3 Bb6 7.d4 *"),
("C64u", "Spanish: Classical, 6.c3 Bb6 7.d4 c6 8.Ba4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.O-O Nd4 5.Nxd4 Bxd4 6.c3 Bb6 7.d4 c6 8.Ba4 *"),
("C64v", "Spanish: Classical, 6.c3 Bb6 7.d4 c6 8.Ba4 d6 9.Na3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Bc5 4.O-O Nd4 5.Nxd4 Bxd4 6.c3 Bb6 7.d4 c6 8.Ba4 d6 9.Na3 *"),
("C65a", "Spanish: Berlin Defence", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 *"),
("C65b", "Spanish: Berlin, 4.Qe2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.Qe2 *"),
("C65c", "Spanish: Berlin, 4.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.d4 *"),
("C65d", "Spanish: Berlin, 4.d4 exd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.d4 exd4 *"),
("C65e", "Spanish: Berlin, Nyholm Attack", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.d4 exd4 5.O-O *"),
("C65f", "Spanish: Berlin, 4.d3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.d3 *"),
("C65g", "Spanish: Berlin, Mortimer Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.d3 Ne7 *"),
("C65g", "Spanish: Berlin, Mortimer Trap", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.d3 Ne7 5.Nxe5 c6 *"),
("C65h", "Spanish: Berlin, 4.d3 d6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.d3 d6 *"),
("C65h", "Spanish: Berlin, Anderssen Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.d3 d6 5.Bxc6+ *"),
("C65h", "Spanish: Berlin, Duras Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.d3 d6 5.c4 *"),
("C65i", "Spanish: Berlin, 4.d3 d6 5.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.d3 d6 5.c3 *"),
("C65j", "Spanish: Berlin, 4.d3 Bc5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.d3 Bc5 *"),
("C65j", "Spanish: Berlin, Kaufmann Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.d3 Bc5 5.Be3 *"),
("C65k", "Spanish: Berlin, 4.O-O", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O *"),
("C65l", "Spanish: Berlin, 4.O-O Be7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Be7 *"),
("C65m", "Spanish: Berlin, Beverwijk Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Bc5 *"),
("C65n", "Spanish: Berlin, Beverwijk, 5.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Bc5 5.c3 *"),
("C65o", "Spanish: Berlin, Benelux Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Bc5 5.c3 O-O 6.d4 Bb6 *"),
("C65p", "Spanish: Berlin, Benelux Variation, 7.Bg5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Bc5 5.c3 O-O 6.d4 Bb6 7.Bg5 *"),
("C65q", "Spanish: Berlin, Beverwijk, 5.Nxe5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Bc5 5.Nxe5 *"),
("C66", "Spanish: Closed Berlin", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O d6 *"),
("C66", "Spanish: Closed Berlin, 5.Re1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O d6 5.Re1 *"),
("C66", "Spanish: Closed Berlin, 5.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O d6 5.d4 *"),
("C66", "Spanish: Closed Berlin, Chigorin Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O d6 5.d4 Nd7 *"),
("C66", "Spanish: Closed Berlin, 5.d4 Bd7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O d6 5.d4 Bd7 *"),
("C66", "Spanish: Closed Berlin, Wolf Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O d6 5.d4 Bd7 6.Nc3 exd4 *"),
("C66", "Spanish: Closed Berlin, Hedgehog Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O d6 5.d4 Bd7 6.Nc3 Be7 *"),
("C66", "Spanish: Closed Berlin, Tarrasch Trap", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O d6 5.d4 Bd7 6.Nc3 Be7 7.Re1 O-O *"),
("C66", "Spanish: Closed Berlin, Bernstein Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O d6 5.d4 Bd7 6.Nc3 Be7 7.Bg5 *"),
("C66", "Spanish: Closed Berlin, Showalter Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O d6 5.d4 Bd7 6.Nc3 Be7 7.Bxc6 *"),
("C67a", "Spanish: Open Berlin", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 *"),
("C67b", "Spanish: Open Berlin, 5.Qe2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.Qe2 *"),
("C67c", "Spanish: Open Berlin, 5.Re1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.Re1 *"),
("C67d", "Spanish: Open Berlin, 5.Re1 Nd6 6.Nxe5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.Re1 Nd6 6.Nxe5 *"),
("C67e", "Spanish: Open Berlin, 5.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 *"),
("C67e", "Spanish: Open Berlin, Rosenthal Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 a6 *"),
("C67f", "Spanish: Open Berlin, 5...Be7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 Be7 *"),
("C67g", "Spanish: Berlin, Minckwitz Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 Be7 6.dxe5 *"),
("C67h", "Spanish: Open Berlin, 5...Be7 6.Qe2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 Be7 6.Qe2 *"),
("C67h", "Spanish: Open Berlin, Trifunovic Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 Be7 6.Qe2 d5 *"),
("C67i", "Spanish: Open Berlin, 5...Be7 6.Qd2 Nd6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 Be7 6.Qe2 Nd6 *"),
("C67i", "Spanish: Open Berlin, Cordel Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 Be7 6.Qe2 Nd6 7.Bxc6 bxc6 8.dxe5 Nf5 *"),
("C67j", "Spanish: Open Berlin, 5...Be7 6.Qd2 Nd6 7.Bxc6 bxc6 8.dxe5 Nb7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 Be7 6.Qe2 Nd6 7.Bxc6 bxc6 8.dxe5 Nb7 *"),
("C67j", "Spanish: Open Berlin, Pillsbury Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 Be7 6.Qe2 Nd6 7.Bxc6 bxc6 8.dxe5 Nb7 9.b3 *"),
("C67j", "Spanish: Open Berlin, Zukertort Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 Be7 6.Qe2 Nd6 7.Bxc6 bxc6 8.dxe5 Nb7 9.c4 *"),
("C67k", "Spanish: Open Berlin, Winawer Attack", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 Be7 6.Qe2 Nd6 7.Bxc6 bxc6 8.dxe5 Nb7 9.Nd4 *"),
("C67l", "Spanish: Open Berlin, Main Line 9.Nc3 (Rio de Janerio)", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 Be7 6.Qe2 Nd6 7.Bxc6 bxc6 8.dxe5 Nb7 9.Nc3 *"),
("C67m", "Spanish: Open Berlin, 5.d4 Nd6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 Nd6 *"),
("C67m", "Spanish: Open Berlin, Showalter Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 Nd6 6.Ba4 *"),
("C67n", "Spanish: Open Berlin, 5.d4 Nd6 6.dxe5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 Nd6 6.dxe5 *"),
("C67o", "Spanish: Open Berlin, 5.d4 Nd6 6.Bxc6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 Nd6 6.Bxc6 *"),
("C67p", "Spanish: Open Berlin, 5.d4 Nd6 6.Bxc6 dxc6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 Nd6 6.Bxc6 dxc6 *"),
("C67q", "Spanish: Open Berlin, 5.d4 Nd6 6.Bxc6 dxc6 7.dxe5 Nf5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 Nd6 6.Bxc6 dxc6 7.dxe5 Nf5 *"),
("C67r", "Spanish: Open Berlin, 5.d4 Nd6 Queenswap", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 Nd6 6.Bxc6 dxc6 7.dxe5 Nf5 8.Qxd8+ *"),
("C67s", "Spanish: Open Berlin, 5.d4 Nd6 Queenswap, 9.Nc3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 Nd6 6.Bxc6 dxc6 7.dxe5 Nf5 8.Qxd8+ Kxd8 9.Nc3 *"),
("C67t", "Spanish: Open Berlin, 5.d4 Nd6 Queenswap, 9.Nc3 h6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 Nd6 6.Bxc6 dxc6 7.dxe5 Nf5 8.Qxd8+ Kxd8 9.Nc3 h6 *"),
("C67u", "Spanish: Open Berlin, 5.d4 Nd6 Queenswap, 9.Nc3 Ke8", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 Nd6 6.Bxc6 dxc6 7.dxe5 Nf5 8.Qxd8+ Kxd8 9.Nc3 Ke8 *"),
("C67v", "Spanish: Open Berlin, 5.d4 Nd6 Queenswap, 9.Nc3 Ke8 10.Rd1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 Nd6 6.Bxc6 dxc6 7.dxe5 Nf5 8.Qxd8+ Kxd8 9.Nc3 Ke8 10.Rd1 *"),
("C67w", "Spanish: Open Berlin, 5.d4 Nd6 Queenswap, 9.Nc3 Ke8 10.h3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 Nd6 6.Bxc6 dxc6 7.dxe5 Nf5 8.Qxd8+ Kxd8 9.Nc3 Ke8 10.h3 *"),
("C67x", "Spanish: Open Berlin, 5.d4 Nd6 Queenswap, 9.Nc3 Ke8 10.h3 a5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 Nf6 4.O-O Nxe4 5.d4 Nd6 6.Bxc6 dxc6 7.dxe5 Nf5 8.Qxd8+ Kxd8 9.Nc3 Ke8 10.h3 a5 *"),
("C68a", "Spanish: 3...a6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 *"),
("C68b", "Spanish: 3...a6 4.Bc4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bc4 *"),
("C68c", "Spanish: Exchange Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 *"),
("C68c", "Spanish: Exchange, 4...bxc6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 bxc6 *"),
("C68d", "Spanish: Exchange, 4...dxc6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 *"),
("C68e", "Spanish: Exchange, Lasker Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.d4 *"),
("C68f", "Spanish: Exchange, Alekhine Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.d4 exd4 6.Qxd4 Qxd4 7.Nxd4 Bd7 *"),
("C68g", "Spanish: Exchange, Keres Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.Nc3 *"),
("C68h", "Spanish: Exchange, Keres, 5...f6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.Nc3 f6 *"),
("C68h", "Spanish: Exchange, Romanovsky Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.Nc3 f6 6.d3 *"),
("C68i", "Spanish: Exchange, 5.Nc3 f6 6.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.Nc3 f6 6.d4 *"),
("C68j", "Spanish: Exchange, 5.O-O", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O *"),
("C68k", "Spanish: Exchange, 5.O-O Ne7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O Ne7 *"),
("C68l", "Spanish: Exchange, 5.O-O Bd6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O Bd6 *"),
("C68m", "Spanish: Exchange, 5.O-O Bd6 6.d4 exd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O Bd6 6.d4 exd4 *"),
("C68n", "Spanish: Exchange, 5.O-O Bg4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O Bg4 *"),
("C68n", "Spanish: Exchange, 5.O-O Bg4 6.h3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O Bg4 6.h3 *"),
("C68o", "Spanish: Exchange, Alapin Gambit", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O Bg4 6.h3 h5 *"),
("C68p", "Spanish: Exchange, Alapin Gambit, 7.d3 Qf6 8.Nbd2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O Bg4 6.h3 h5 7.d3 Qf6 8.Nbd2 *"),
("C68q", "Spanish: Exchange, Bronstein Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O Qd6 *"),
("C68r", "Spanish: Exchange, Bronstein, 6.d3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O Qd6 6.d3 *"),
("C68s", "Spanish: Exchange, Bronstein, 6.Na3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O Qd6 6.Na3 *"),
("C68t", "Spanish: Exchange, Bronstein, 6.Na3 b5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O Qd6 6.Na3 b5 *"),
("C68u", "Spanish: Exchange, Bronstein, 6.Na3 Be6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O Qd6 6.Na3 Be6 *"),
("C69a", "Spanish: Exchange, Gligoric Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O f6 *"),
("C69b", "Spanish: Exchange, Gligoric, 6.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O f6 6.d4 *"),
("C69b", "Spanish: Exchange, Gligoric, 6.d4 Bg4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O f6 6.d4 Bg4 *"),
("C69c", "Spanish: Exchange, Gligoric, 6.d4 Bg4 7.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O f6 6.d4 Bg4 7.c3 *"),
("C69d", "Spanish: Exchange, Gligoric, 6.d4 Bg4 7.c3 Bd6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O f6 6.d4 Bg4 7.c3 Bd6 *"),
("C69e", "Spanish: Exchange, Gligoric, 6.d4 Bg4 7.dxe5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O f6 6.d4 Bg4 7.dxe5 *"),
("C69f", "Spanish: Exchange, Gligoric, 6.d4 Bg4 Queenswap, 9.Rd3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O f6 6.d4 Bg4 7.dxe5 Qxd1 8.Rxd1 fxe5 9.Rd3 *"),
("C69g", "Spanish: Exchange, Gligoric, 6.d4 Bg4 Queenswap, 9.Rd3 Bd6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O f6 6.d4 Bg4 7.dxe5 Qxd1 8.Rxd1 fxe5 9.Rd3 Bd6 *"),
("C69h", "Spanish: Exchange, Gligoric, 6.d4 exd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O f6 6.d4 exd4 *"),
("C69h", "Spanish: Exchange, Gligoric, 6.d4 exd4 7.Qxd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O f6 6.d4 exd4 7.Nxd4 *"),
("C69i", "Spanish: Exchange, Gligoric, 6.d4 exd4 7.Nxd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O f6 6.d4 exd4 7.Qxd4 *"),
("C69i", "Spanish: Exchange, Gligoric, 6.d4 exd4 7.Nxd4 Ne7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O f6 6.d4 exd4 7.Nxd4 Ne7 *"),
("C69j", "Spanish: Exchange, Gligoric, 6.d4 exd4 7.Nxd4 c5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O f6 6.d4 exd4 7.Nxd4 c5 *"),
("C69j", "Spanish: Exchange, Gligoric, 8.Ne2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O f6 6.d4 exd4 7.Nxd4 c5 8.Ne2 *"),
("C69k", "Spanish: Exchange, Gligoric, 8.Nb3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O f6 6.d4 exd4 7.Nxd4 c5 8.Nb3 *"),
("C69k", "Spanish: Exchange, Gligoric, 8.Nb3 Queenswap", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O f6 6.d4 exd4 7.Nxd4 c5 8.Nb3 Qxd1 9.Rxd1 *"),
("C69l", "Spanish: Exchange, Gligoric, 8.Nb3 Queenswap, 9...Bd7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O f6 6.d4 exd4 7.Nxd4 c5 8.Nb3 Qxd1 9.Rxd1 Bd7 *"),
("C69m", "Spanish: Exchange, Gligoric, 8.Nb3 Queenswap, 9...Bg4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O f6 6.d4 exd4 7.Nxd4 c5 8.Nb3 Qxd1 9.Rxd1 Bg4 *"),
("C69n", "Spanish: Exchange, Gligoric, 8.Nb3 Queenswap, 9...Bg4 10.f3 Be6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Bxc6 dxc6 5.O-O f6 6.d4 exd4 7.Nxd4 c5 8.Nb3 Qxd1 9.Rxd1 Bg4 10.f3 Be6 *"),
("C70", "Spanish: 4.Ba4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 *"),
("C70", "Spanish: Brentano Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 g5 *"),
("C70", "Spanish: 4.Ba4 Be7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Be7 *"),
("C70", "Spanish: Fianchetto Deferred", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 g6 *"),
("C70", "Spanish: Alapin Deferred", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Bb4 *"),
("C70", "Spanish: Cozio Deferred", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nge7 *"),
("C70", "Spanish: Bird's Deferred", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nd4 *"),
("C70", "Spanish: Classical Deferred", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Bc5 *"),
("C70", "Spanish: Caro Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 b5 *"),
("C70", "Spanish: Graz Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 b5 5.Bb3 Bc5 *"),
("C70", "Spanish: Taimanov (Wing) Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 b5 5.Bb3 Na5 *"),
("C70", "Spanish: Schliemann Deferred", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 f5 *"),
("C70", "Spanish: Schliemann Deferred, 5.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 f5 5.d4 *"),
("C70", "Spanish: Schliemann Deferred, 5.d4 exd4 6.e5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 f5 5.d4 exd4 6.e5 *"),
("C71", "Spanish: Modern Steinitz Defence", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 *"),
("C71", "Spanish: Modern Steinitz, Three Knights Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.Nc3 *"),
("C71", "Spanish: Modern Steinitz, 5.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.d4 *"),
("C71", "Spanish: Modern Steinitz, Noah's Ark Trap", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.d4 b5 6.Bb3 Nxd4 7.Nxd4 exd4 8.Qxd4 c5 *"),
("C71", "Spanish: Modern Steinitz, Keres Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.c4 *"),
("C72", "Spanish: Modern Steinitz, 5.O-O", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.O-O *"),
("C72", "Spanish: Modern Steinitz, 5.O-O Ne7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.O-O Ne7 *"),
("C72", "Spanish: Modern Steinitz, 5.O-O Bg4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.O-O Bg4 *"),
("C72", "Spanish: Modern Steinitz, 5.O-O Bg4 6.h3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.O-O Bg4 6.h3 *"),
("C72", "Spanish: Modern Steinitz, 5.O-O Bd7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.O-O Bd7 *"),
("C72", "Spanish: Modern Steinitz, 5.O-O Bd7 6.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.O-O Bd7 6.d4 *"),
("C73", "Spanish: Modern Steinitz, Richter Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.Bxc6+ *"),
("C73", "Spanish: Modern Steinitz, Richter Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.Bxc6+ bxc6 *"),
("C73", "Spanish: Modern Steinitz, Richter Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.Bxc6+ bxc6 6.d4 *"),
("C73", "Spanish: Modern Steinitz, Alapin Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.Bxc6+ bxc6 6.d4 f6 *"),
("C74", "Spanish: Modern Steinitz, 5.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.c3 *"),
("C74", "Spanish: Modern Steinitz, 5.c3 g6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.c3 g6 *"),
("C74", "Spanish: Modern Steinitz, Siesta Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.c3 f5 *"),
("C74", "Spanish: Modern Steinitz, Siesta, 6.exf5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.c3 f5 6.exf5 *"),
("C74", "Spanish: Modern Steinitz, Siesta, Kopayev Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.c3 f5 6.exf5 Bxf5 7.O-O *"),
("C74", "Spanish: Modern Steinitz, Siesta, Kopayev, Main Line", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.c3 f5 6.exf5 Bxf5 7.O-O Bd3 8.Re1 Be7 9.Bc2 Bxc2 10.Qxc2 *"),
("C75", "Spanish: Modern Steinitz, 5.c3 Bd7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.c3 Bd7 *"),
("C75", "Spanish: Modern Steinitz, 5.c3 Bd7 6.O-O", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.c3 Bd7 6.O-O *"),
("C75", "Spanish: Modern Steinitz, 5.c3 Bd7 6.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.c3 Bd7 6.d4 *"),
("C75", "Spanish: Modern Steinitz, 5.c3 Bd7 6.d4 Nf6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.c3 Bd7 6.d4 Nf6 *"),
("C75", "Spanish: Modern Steinitz, Rubinstein Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.c3 Bd7 6.d4 Nge7 *"),
("C75", "Spanish: Modern Steinitz, Rubinstein, 7.Bb3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.c3 Bd7 6.d4 Nge7 7.Bb3 *"),
("C75", "Spanish: Modern Steinitz, Rubinstein, 7.Be3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.c3 Bd7 6.d4 Nge7 7.Be3 *"),
("C76", "Spanish: Modern Steinitz, Bronstein Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.c3 Bd7 6.d4 g6 *"),
("C76", "Spanish: Modern Steinitz, Bronstein, 7.O-O", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.c3 Bd7 6.d4 g6 7.O-O *"),
("C76", "Spanish: Modern Steinitz, Bronstein, 7.O-O Bg7 8.Re1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.c3 Bd7 6.d4 g6 7.O-O Bg7 8.Re1 *"),
("C76", "Spanish: Modern Steinitz, Bronstein, 7.O-O Bg7 8.dxe5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.c3 Bd7 6.d4 g6 7.O-O Bg7 8.dxe5 *"),
("C76", "Spanish: Modern Steinitz, Bronstein, 7.O-O Bg7 8.dxe5 dxe5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.c3 Bd7 6.d4 g6 7.O-O Bg7 8.dxe5 dxe5 *"),
("C76", "Spanish: Modern Steinitz, Bronstein, 7.O-O Bg7 8.dxe5 Nxe5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.c3 Bd7 6.d4 g6 7.O-O Bg7 8.dxe5 Nxe5 *"),
("C76", "Spanish: Modern Steinitz, Bronstein, 7.O-O Bg7 8.d5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 d6 5.c3 Bd7 6.d4 g6 7.O-O Bg7 8.d5 *"),
("C77a", "Spanish: Morphy Defence", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 *"),
("C77b", "Spanish: Anderssen Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.d3 *"),
("C77c", "Spanish: Anderssen, 5...b5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.d3 b5 *"),
("C77d", "Spanish: Anderssen, 5...d6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.d3 d6 *"),
("C77e", "Spanish: Duras Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.d3 d6 6.c4 *"),
("C77f", "Spanish: Anderssen, 5...d6 6.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.d3 d6 6.c3 *"),
("C77g", "Spanish: Anderssen, 5...d6 6.c3 g6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.d3 d6 6.c3 g6 *"),
("C77h", "Spanish: Four Knights (Tarrasch) Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.Nc3 *"),
("C77i", "Spanish: Exchange Deferred", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.Bxc6 *"),
("C77j", "Spanish: Centre Attack", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.d4 *"),
("C77k", "Spanish: Centre Attack, 5...b5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.d4 b5 *"),
("C77l", "Spanish: Centre Attack, 5...Nxd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.d4 Nxd4 *"),
("C77m", "Spanish: Centre Attack, 5...Nxe4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.d4 Nxe4 *"),
("C77n", "Spanish: Centre Attack, 5...Be7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.d4 Be7 *"),
("C77o", "Spanish: Centre Attack, 5...exd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.d4 exd4 *"),
("C77p", "Spanish: Wormald Attack", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.Qe2 *"),
("C77q", "Spanish: Wormald Attack, 5...Be7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.Qe2 Be7 *"),
("C77r", "Spanish: Wormald Attack, 5...b5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.Qe2 b5 *"),
("C77s", "Spanish: Wormald Attack, 5...b5 6.Bb3 Bc5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.Qe2 b5 6.Bb3 Bc5 *"),
("C77t", "Spanish: Wormald Attack, 5...b5 6.Bb3 Bc5 7.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.Qe2 b5 6.Bb3 Bc5 7.c3 *"),
("C77u", "Spanish: Wormald Attack, 5...b5 6.Bb3 Be7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.Qe2 b5 6.Bb3 Be7 *"),
("C77u", "Spanish: Wormald Attack, Gruenfeld Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.Qe2 b5 6.Bb3 Be7 7.d4 d6 8.c3 Bg4 *"),
("C78a", "Spanish: 5.O-O", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O *"),
("C78b", "Spanish: Moeller Defence", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Bc5 *"),
("C78c", "Spanish: Moeller Defence, 6.Nxe5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Bc5 6.Nxe5 *"),
("C78d", "Spanish: Moeller Defence, 6.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Bc5 6.c3 *"),
("C78e", "Spanish: 5.O-O b5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 *"),
("C78e", "Spanish: 5.O-O b5 6.Bb3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 *"),
("C78f", "Spanish: 5.O-O b5 6.Bb3 d6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 d6 *"),
("C78f", "Spanish: Rabinovich Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 d6 7.Ng5 d5 8.exd5 Nd4 9.Re1 Bc5 10.Rxe5+ Kf8 *"),
("C78g", "Spanish: 5.O-O b5 6.Bb3 Be7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 Be7 *"),
("C78g", "Spanish: 5.O-O b5 6.Bb3 Be7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 Be7 7.d4 *"),
("C78h", "Spanish: 5.O-O b5 6.Bb3 Bc5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 Bc5 *"),
("C78i", "Spanish: 5.O-O b5 6.Bb3 Bc5 7.Nxe5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 Bc5 7.Nxe5 *"),
("C78i", "Spanish: 5.O-O b5 6.Bb3 Bc5 7.Nxe5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 Bc5 7.Nxe5 Nxe5 8.d4 *"),
("C78j", "Spanish: 5.O-O b5 6.Bb3 Bc5 7.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 Bc5 7.c3 *"),
("C78j", "Spanish: 5.O-O b5 6.Bb3 Bc5 7.c3 d6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 Bc5 7.c3 d6 *"),
("C78k", "Spanish: 5.O-O b5 6.Bb3 Bc5 7.c3 d6 8.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 Bc5 7.c3 d6 8.d4 *"),
("C78l", "Spanish: 5.O-O b5 6.Bb3 Bc5 7.a4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 Bc5 7.a4 *"),
("C78m", "Spanish: 5.O-O b5 6.Bb3 Bc5 7.a4 Rb8", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 Bc5 7.a4 Rb8 *"),
("C78n", "Spanish: Archangelsk Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 Bb7 *"),
("C78o", "Spanish: Archangelsk, 7.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 Bb7 7.d4 *"),
("C78p", "Spanish: Archangelsk, 7.d3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 Bb7 7.d3 *"),
("C78q", "Spanish: Archangelsk, 7.d3 Be7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 Bb7 7.d3 Be7 *"),
("C78r", "Spanish: Archangelsk, 7.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 Bb7 7.c3 *"),
("C78s", "Spanish: Archangelsk, 7.c3 Nxe4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 Bb7 7.c3 Nxe4 *"),
("C78t", "Spanish: Archangelsk, 7.Re1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 Bb7 7.Re1 *"),
("C78t", "Spanish: Archangelsk, 7.Re1 Bc5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 Bb7 7.Re1 Bc5 *"),
("C78u", "Spanish: Archangelsk, 7.Re1 Bc5 8.c3 d6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 Bb7 7.Re1 Bc5 8.c3 d6 *"),
("C78u", "Spanish: Archangelsk, Main Line", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 Bb7 7.Re1 Bc5 8.c3 d6 9.d4 Bb6 *"),
("C78v", "Spanish: Archangelsk, Main Line, 10.a4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 Bb7 7.Re1 Bc5 8.c3 d6 9.d4 Bb6 10.a4 *"),
("C78w", "Spanish: Archangelsk, Main Line, 10.Bg5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 Bb7 7.Re1 Bc5 8.c3 d6 9.d4 Bb6 10.Bg5 *"),
("C78x", "Spanish: Archangelsk, Main Line, 10.Be3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O b5 6.Bb3 Bb7 7.Re1 Bc5 8.c3 d6 9.d4 Bb6 10.Be3 *"),
("C79", "Spanish: Steinitz Deferred", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O d6 *"),
("C79", "Spanish: Steinitz Deferred, 6.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O d6 6.c3 *"),
("C79", "Spanish: Steinitz Deferred, 6.Re1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O d6 6.Re1 *"),
("C79", "Spanish: Steinitz Deferred, Exchange", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O d6 6.Bxc6+ *"),
("C79", "Spanish: Steinitz Deferred, Lipnitsky Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O d6 6.Bxc6+ bxc6 7.d4 Bg4 *"),
("C79", "Spanish: Steinitz Deferred, Rubinstein Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O d6 6.Bxc6+ bxc6 7.d4 Nxe4 *"),
("C79", "Spanish: Steinitz Deferred, Boleslavsky Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O d6 6.Bxc6+ bxc6 7.d4 Nxe4 8.Re1 f5 9.dxe5 d5 10.Nc3 *"),
("C80a", "Spanish: Open", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 *"),
("C80b", "Spanish: Open, Knorre Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.Nc3 *"),
("C80c", "Spanish: Open, Tartakower Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.Qe2 *"),
("C80d", "Spanish: Open, 6.Re1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.Re1 *"),
("C80e", "Spanish: Open, 6.Re1 Nc5 7.Bxc6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.Re1 Nc5 7.Bxc6 *"),
("C80f", "Spanish: Open, 6.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 *"),
("C80g", "Spanish: Open, Riga Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 exd4 *"),
("C80h", "Spanish: Open, 6.d4 Be7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 Be7 *"),
("C80i", "Spanish: Open, 6.d4 b5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 *"),
("C80i", "Spanish: Open, Friess Attack", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Nxe5 *"),
("C80i", "Spanish: Open, Richter Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.d5 *"),
("C80j", "Spanish: Open, 7.Bb3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 *"),
("C80k", "Spanish: Open, 7.Bb3 d5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 *"),
("C80k", "Spanish: Open, 7.Bb3 d5 8.a4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.a4 *"),
("C80k", "Spanish: Open, Schlechter Defence", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.a4 Nxd4 *"),
("C80k", "Spanish: Open, Schlecter, Berger Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.a4 Nxd4 9.Nxd4 exd4 10.Nc3 *"),
("C80k", "Spanish: Open, Harksen Gambit", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.c4 *"),
("C80l", "Spanish: Open, 8.Nxe5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.Nxe5 *"),
("C80m", "Spanish: Open, 8.dxe5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 *"),
("C80m", "Spanish: Open, Zukertort Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Ne7 *"),
("C80n", "Spanish: Open, 8...Be6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 *"),
("C80n", "Spanish: Open, 8...Be6 9.a4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.a4 *"),
("C80o", "Spanish: Open, 8...Be6 9.Be3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.Be3 *"),
("C80p", "Spanish: Open, Bernstein Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.Nbd2 *"),
("C80q", "Spanish: Open, Bernstein, 9...Bc5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.Nbd2 Bc5 *"),
("C80r", "Spanish: Open, Bernstein, 9...Nc5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.Nbd2 Nc5 *"),
("C80s", "Spanish: Open, Bernstein, 9...Nc5 10.c3 d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.Nbd2 Nc5 10.c3 d4 *"),
("C80s", "Spanish: Open, Bernstein, 11.cxd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.Nbd2 Nc5 10.c3 d4 11.cxd4 *"),
("C80t", "Spanish: Open, Bernstein, Karpov Gambit", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.Nbd2 Nc5 10.c3 d4 11.Ng5 *"),
("C80u", "Spanish: Open, Bernstein, 11.Bxe6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.Nbd2 Nc5 10.c3 d4 11.Bxe6 *"),
("C81", "Spanish: Open, Keres Attack", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.Qe2 *"),
("C81", "Spanish: Open, Keres Attack, 9...Bc5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.Qe2 Bc5 *"),
("C81", "Spanish: Open, Keres Attack, 9...Be7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.Qe2 Be7 *"),
("C81", "Spanish: Open, Keres, Adam Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.Qe2 Be7 10.c4 *"),
("C81", "Spanish: Open, Keres, 10.Rd1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.Qe2 Be7 10.Rd1 *"),
("C81", "Spanish: Open, Keres, 10.Rd1 O-O", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.Qe2 Be7 10.Rd1 O-O *"),
("C81", "Spanish: Open, Keres, 10.Rd1 O-O 11.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.Qe2 Be7 10.Rd1 O-O 11.c3 *"),
("C81", "Spanish: Open, Keres, 10.Rd1 O-O 11.c4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.Qe2 Be7 10.Rd1 O-O 11.c4 *"),
("C81", "Spanish: Open, Keres, 10.Rd1 O-O 11.c4 bxc4 12.Bxc4 Bc5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.Qe2 Be7 10.Rd1 O-O 11.c4 bxc4 12.Bxc4 Bc5 *"),
("C81", "Spanish: Open, Keres, Ekstrom Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.Qe2 Be7 10.Rd1 O-O 11.c4 bxc4 12.Bxc4 Qd7 *"),
("C82a", "Spanish: Open, 9.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 *"),
("C82b", "Spanish: Open, Berlin Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Nc5 *"),
("C82b", "Spanish: Open, Berlin, 10.Bc2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Nc5 10.Bc2 *"),
("C82c", "Spanish: Open, Berlin, 10.Bc2 Bg4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Nc5 10.Bc2 Bg4 *"),
("C82d", "Spanish: Open, Berlin, 10.Bc2 Bg4 11.Re1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Nc5 10.Bc2 Bg4 11.Re1 *"),
("C82e", "Spanish: Open, Berlin, 10.Bc2 Bg4 11.Nbd2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Nc5 10.Bc2 Bg4 11.Nbd2 *"),
("C82e", "Spanish: Open, Berlin, 10.Bc2 Bg4 11.Nbd2 Be7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Nc5 10.Bc2 Bg4 11.Nbd2 Be7 *"),
("C82f", "Spanish: Open, Berlin, 10.Bc2 Bg4 11.Nbd2 Be7 12.Re1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Nc5 10.Bc2 Bg4 11.Nbd2 Be7 12.Re1 *"),
("C82g", "Spanish: Open, Berlin, 10.Bc2 Bg4 11.Nbd2 Be7 12.Re1 Qd7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Nc5 10.Bc2 Bg4 11.Nbd2 Be7 12.Re1 Qd7 *"),
("C82h", "Spanish: Open, Berlin, 10.Bc2 Bg4 11.Nbd2 Be7 12.Re1 O-O", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Nc5 10.Bc2 Bg4 11.Nbd2 Be7 12.Re1 O-O *"),
("C82i", "Spanish: Open, Italian Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Bc5 *"),
("C82j", "Spanish: Open, Motzko Attack", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Bc5 10.Qd3 *"),
("C82j", "Spanish: Open, Motzko Attack, Nenarokov Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Bc5 10.Qd3 Ne7 *"),
("C82k", "Spanish: Open, St. Petersburg Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Bc5 10.Nbd2 *"),
("C82k", "Spanish: Open, St. Petersburg Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Bc5 10.Nbd2 O-O *"),
("C82l", "Spanish: Open, St. Petersburg, 11.Bc2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Bc5 10.Nbd2 O-O 11.Bc2 *"),
("C82m", "Spanish: Open, St. Petersburg, 11.Bc2 f5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Bc5 10.Nbd2 O-O 11.Bc2 f5 *"),
("C82n", "Spanish: Open, Baguio Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Bc5 10.Nbd2 O-O 11.Bc2 Bf5 *"),
("C82n", "Spanish: Open, Baguio, 12.Nb3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Bc5 10.Nbd2 O-O 11.Bc2 Bf5 12.Nb3 *"),
("C82o", "Spanish: Open, Dilworth Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Bc5 10.Nbd2 O-O 11.Bc2 Nxf2 *"),
("C82o", "Spanish: Open, Dilworth, 12.Rxf2 f6 13.exf6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Bc5 10.Nbd2 O-O 11.Bc2 Nxf2 12.Rxf2 f6 13.exf6 *"),
("C82p", "Spanish: Open, Dilworth, 14.Kxf2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Bc5 10.Nbd2 O-O 11.Bc2 Nxf2 12.Rxf2 f6 13.exf6 Bxf2+ 14.Kxf2 *"),
("C82q", "Spanish: Open, Dilworth, 14.Kxf2 Qxf6 15.Nf1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Bc5 10.Nbd2 O-O 11.Bc2 Nxf2 12.Rxf2 f6 13.exf6 Bxf2+ 14.Kxf2 Qxf6 15.Nf1 *"),
("C82r", "Spanish: Open, Dilworth, 14.Kxf2 Qxf6 15.Kg1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Bc5 10.Nbd2 O-O 11.Bc2 Nxf2 12.Rxf2 f6 13.exf6 Bxf2+ 14.Kxf2 Qxf6 15.Kg1 *"),
("C82s", "Spanish: Open, Dilworth, 14.Kxf2 Qxf6 15.Kg1 g5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Bc5 10.Nbd2 O-O 11.Bc2 Nxf2 12.Rxf2 f6 13.exf6 Bxf2+ 14.Kxf2 Qxf6 15.Kg1 g5 *"),
("C83", "Spanish: Open, Classical Defence", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Be7 *"),
("C83", "Spanish: Open, Classical, 10.Re1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Be7 10.Re1 *"),
("C83", "Spanish: Open, Classical, Tarrasch Trap", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Be7 10.Re1 O-O 11.Nd4 Qd7 12.Nxe6 fxe6 13.Rxe4 *"),
("C83", "Spanish: Open, Classical, Breslau Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Be7 10.Re1 O-O 11.Nd4 Nxe5 *"),
("C83", "Spanish: Open, Classical, 10.Be3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Be7 10.Be3 *"),
("C83", "Spanish: Open, Classical, 10.Nbd2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Be7 10.Nbd2 *"),
("C83", "Spanish: Open, Classical, 10.Nbd2 Nc5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Be7 10.Nbd2 Nc5 *"),
("C83", "Spanish: Open, Classical, 10.Nbd2 O-O", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Be7 10.Nbd2 O-O *"),
("C83", "Spanish: Open, Classical, Malkin Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Be7 10.Nbd2 O-O 11.Qe2 *"),
("C83", "Spanish: Open, Classical, 10.Nbd2 O-O 11.Bc2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Be7 10.Nbd2 O-O 11.Bc2 *"),
("C83", "Spanish: Open, Classical, 10.Nbd2 O-O 11.Bc2 f5 12.exf6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Be7 10.Nbd2 O-O 11.Bc2 f5 12.exf6 *"),
("C83", "Spanish: Open, Classical, 10.Nbd2 O-O 11.Bc2 f5 12.Nb3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Nxe4 6.d4 b5 7.Bb3 d5 8.dxe5 Be6 9.c3 Be7 10.Nbd2 O-O 11.Bc2 f5 12.Nb3 *"),
("C84a", "Spanish: Closed System", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 *"),
("C84b", "Spanish: Closed, 6.Nc3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Nc3 *"),
("C84c", "Spanish: Closed, 6.Nc3 b5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Nc3 b5 *"),
("C84c", "Spanish: Closed, 6.Nc3 b5 7.Bb3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Nc3 b5 7.Bb3 *"),
("C84c", "Spanish: Closed, 6.Nc3 b5 7.Bb3 d6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Nc3 b5 7.Bb3 d6 *"),
("C84d", "Spanish: Closed, 6.Nc3 b5 7.Bb3 d6 8.Nd5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Nc3 b5 7.Bb3 d6 8.Nd5 *"),
("C84e", "Spanish: Closed, 6.d3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.d3 *"),
("C84f", "Spanish: Closed, 6.d3 b5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.d3 b5 *"),
("C84f", "Spanish: Closed, 6.d3 b5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.d3 b5 7.Bb3 *"),
("C84g", "Spanish: Closed, Centre Attack", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.d4 *"),
("C84g", "Spanish: Closed, Centre Attack", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.d4 exd4 *"),
("C84h", "Spanish: Closed, Centre Attack, 7.e5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.d4 exd4 7.e5 *"),
("C84h", "Spanish: Closed, Centre Attack, 7.e5 Ne4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.d4 exd4 7.e5 Ne4 *"),
("C84h", "Spanish: Closed, Centre Attack, Basque Gambit", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.d4 exd4 7.e5 Ne4 8.c3 *"),
("C84h", "Spanish: Closed, Centre Attack, Basque Gambit Accepted", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.d4 exd4 7.e5 Ne4 8.c3 dxc3 *"),
("C84h", "Spanish: Closed, Centre Attack, 7.e5 Ne4 8.b4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.d4 exd4 7.e5 Ne4 8.b4 *"),
("C84i", "Spanish: Closed, Centre Attack, 7.e5 Ne4 8.Nxd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.d4 exd4 7.e5 Ne4 8.Nxd4 *"),
("C84j", "Spanish: Closed, Centre Attack, 7.e5 Ne4 8.Nxd4 Nxd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.d4 exd4 7.e5 Ne4 8.Nxd4 Nxd4 *"),
("C84k", "Spanish: Closed, Centre Attack, 7.e5 Ne4 8.Nxd4 O-O", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.d4 exd4 7.e5 Ne4 8.Nxd4 O-O *"),
("C84l", "Spanish: Closed, Centre Attack, 7.Re1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.d4 exd4 7.Re1 *"),
("C84m", "Spanish: Closed, Centre Attack, 7.Re1 b5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.d4 exd4 7.Re1 b5 *"),
("C84n", "Spanish: Closed, Centre Attack, 7.Re1 b5 8.e5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.d4 exd4 7.Re1 b5 8.e5 *"),
("C84n", "Spanish: Closed, Centre Attack, 7.Re1 b5 8.e5 Nxe5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.d4 exd4 7.Re1 b5 8.e5 Nxe5 *"),
("C84n", "Spanish: Closed, Centre Attack, 7.Re1 b5 8.e5 Nxe5 9.Rxe5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.d4 exd4 7.Re1 b5 8.e5 Nxe5 9.Rxe5 *"),
("C84o", "Spanish: Closed, Centre Attack, 7.Re1 O-O", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.d4 exd4 7.Re1 O-O *"),
("C84p", "Spanish: Closed, Centre Attack, 7.Re1 O-O 8.e5 Ne8", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.d4 exd4 7.Re1 O-O 8.e5 Ne8 *"),
("C84p", "Spanish: Closed, Centre Attack, 7.Re1 O-O 8.e5 Ne8 9.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.d4 exd4 7.Re1 O-O 8.e5 Ne8 9.c3 *"),
("C84q", "Spanish: Closed, Centre Attack, 7.Re1 O-O 8.e5 Ne8 9.Bf4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.d4 exd4 7.Re1 O-O 8.e5 Ne8 9.Bf4 *"),
("C85a", "Spanish: Closed, Exchange", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Bxc6 *"),
("C85b", "Spanish: Closed, Exchange", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Bxc6 dxc6 *"),
("C85c", "Spanish: Closed, Exchange, 7.Qe2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Bxc6 dxc6 7.Qe2 *"),
("C85d", "Spanish: Closed, Exchange, 7.Nc3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Bxc6 dxc6 7.Nc3 *"),
("C85e", "Spanish: Closed, Exchange, 7.Nc3 Bg4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Bxc6 dxc6 7.Nc3 Bg4 *"),
("C85f", "Spanish: Closed, Exchange, 7.Qe1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Bxc6 dxc6 7.Qe1 *"),
("C85g", "Spanish: Closed, Exchange, 7.Qe1 c5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Bxc6 dxc6 7.Qe1 c5 *"),
("C85h", "Spanish: Closed, Exchange, 7.Qe1 Nd7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Bxc6 dxc6 7.Qe1 Nd7 *"),
("C85i", "Spanish: Closed, Exchange, 7.Qe1 Nd7 8.b3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Bxc6 dxc6 7.Qe1 Nd7 8.b3 *"),
("C85j", "Spanish: Closed, Exchange, 7.d3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Bxc6 dxc6 7.d3 *"),
("C85k", "Spanish: Closed, Exchange, 7.d3 Bg4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Bxc6 dxc6 7.d3 Bg4 *"),
("C85l", "Spanish: Closed, Exchange, 7.d3 Nd7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Bxc6 dxc6 7.d3 Nd7 *"),
("C85m", "Spanish: Closed, Exchange, 7.d3 Nd7 8.Nbd2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Bxc6 dxc6 7.d3 Nd7 8.Nbd2 *"),
("C85n", "Spanish: Closed, Exchange, 7.d3 Nd7 8.Nbd2 O-O 9.Nc4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Bxc6 dxc6 7.d3 Nd7 8.Nbd2 O-O 9.Nc4 *"),
("C85o", "Spanish: Closed, Exchange, 7.d3 Nd7 8.Nbd2 O-O 9.Nc4 f6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Bxc6 dxc6 7.d3 Nd7 8.Nbd2 O-O 9.Nc4 f6 *"),
("C85p", "Spanish: Closed, Exchange, 7.d3 Nd7 8.Nbd2 O-O 9.Nc4 f6 10.Nh4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Bxc6 dxc6 7.d3 Nd7 8.Nbd2 O-O 9.Nc4 f6 10.Nh4 *"),
("C85q", "Spanish: Closed, Exchange, 7.d3 Nd7 8.Nbd2 O-O 9.Nc4 f6 10.Nh4 Nc5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Bxc6 dxc6 7.d3 Nd7 8.Nbd2 O-O 9.Nc4 f6 10.Nh4 Nc5 *"),
("C85r", "Spanish: Closed, Exchange, 7.d3 Nd7 8.Nbd2 O-O 9.Nc4 f6 10.Nh4 Nc5 11.Nf5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Bxc6 dxc6 7.d3 Nd7 8.Nbd2 O-O 9.Nc4 f6 10.Nh4 Nc5 11.Nf5 *"),
("C86a", "Spanish: Worrall Attack", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Qe2 *"),
("C86a", "Spanish: Worrall Attack, 6...d6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Qe2 d6 *"),
("C86b", "Spanish: Worrall Attack, 6...b5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Qe2 b5 *"),
("C86b", "Spanish: Worrall Attack, 6...b5 7.Bb3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Qe2 b5 7.Bb3 *"),
("C86c", "Spanish: Worrall Attack, 7...d6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Qe2 b5 7.Bb3 d6 *"),
("C86d", "Spanish: Worrall Attack, 7...d6 8.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Qe2 b5 7.Bb3 d6 8.c3 *"),
("C86e", "Spanish: Worrall Attack, 7...O-O", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Qe2 b5 7.Bb3 O-O *"),
("C86e", "Spanish: Worrall Attack, 7...O-O 8.a4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Qe2 b5 7.Bb3 O-O 8.a4 *"),
("C86f", "Spanish: Worrall Attack, 7...O-O 8.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Qe2 b5 7.Bb3 O-O 8.c3 *"),
("C86g", "Spanish: Worrall Attack, 7...O-O 8.c3 d6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Qe2 b5 7.Bb3 O-O 8.c3 d6 *"),
("C86h", "Spanish: Worrall Attack, 7...O-O 8.c3 d6 9.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Qe2 b5 7.Bb3 O-O 8.c3 d6 9.d4 *"),
("C86i", "Spanish: Worrall Attack, 7...O-O 8.c3 d6 9.Rd1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Qe2 b5 7.Bb3 O-O 8.c3 d6 9.Rd1 *"),
("C86j", "Spanish: Worrall Attack, 7...O-O 8.c3 d5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Qe2 b5 7.Bb3 O-O 8.c3 d5 *"),
("C86j", "Spanish: Worrall Attack, 7...O-O 8.c3 d5 9.exd5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Qe2 b5 7.Bb3 O-O 8.c3 d5 9.exd5 *"),
("C86k", "Spanish: Worrall Attack, 7...O-O 8.c3 d5 9.d3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Qe2 b5 7.Bb3 O-O 8.c3 d5 9.d3 *"),
("C86l", "Spanish: Worrall Attack, 7...O-O 8.c3 d5 9.d3 Bb7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Qe2 b5 7.Bb3 O-O 8.c3 d5 9.d3 Bb7 *"),
("C87a", "Spanish: Closed, 6.Re1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 *"),
("C87b", "Spanish: Closed, Averbakh (Russian) Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 d6 *"),
("C87c", "Spanish: Closed, Averbakh, 7.Bxc6+", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 d6 7.Bxc6+ *"),
("C87c", "Spanish: Closed, Averbakh, 7.Bxc6+ bxc6 8.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 d6 7.Bxc6+ bxc6 8.d4 *"),
("C87d", "Spanish: Closed, Averbakh, 7.Bxc6+ bxc6 8.d4 exd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 d6 7.Bxc6+ bxc6 8.d4 exd4 *"),
("C87d", "Spanish: Closed, Averbakh, 7.Bxc6+ bxc6 8.d4 exd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 d6 7.Bxc6+ bxc6 8.d4 exd4 9.Nxd4 *"),
("C87e", "Spanish: Closed, Averbakh, 7.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 d6 7.c3 *"),
("C87f", "Spanish: Closed, Averbakh, 7.c3 Bg4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 d6 7.c3 Bg4 *"),
("C87g", "Spanish: Closed, Averbakh, 7.c3 Bg4 8.h3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 d6 7.c3 Bg4 8.h3 *"),
("C87h", "Spanish: Closed, Averbakh, 7.c3 Bg4 8.d3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 d6 7.c3 Bg4 8.d3 *"),
("C87i", "Spanish: Closed, Averbakh, 7.c3 O-O", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 d6 7.c3 O-O *"),
("C87j", "Spanish: Closed, Averbakh, 7.c3 O-O 8.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 d6 7.c3 O-O 8.d4 *"),
("C87k", "Spanish: Closed, Averbakh, 7.c3 O-O 8.d4 Bd7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 d6 7.c3 O-O 8.d4 Bd7 *"),
("C87l", "Spanish: Closed, Averbakh, 7.c3 O-O 8.d4 Bd7 9.Nbd2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 d6 7.c3 O-O 8.d4 Bd7 9.Nbd2 *"),
("C87m", "Spanish: Closed, Averbakh, 7.c3 O-O 8.d4 Bd7 9.Nbd2 Re8", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 d6 7.c3 O-O 8.d4 Bd7 9.Nbd2 Re8 *"),
("C87n", "Spanish: Closed, Averbakh, 7.c3 O-O 8.h3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 d6 7.c3 O-O 8.h3 *"),
("C87o", "Spanish: Closed, Averbakh, 7.c3 O-O 8.h3 Bd7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 d6 7.c3 O-O 8.h3 Bd7 *"),
("C88a", "Spanish: Closed, 6...b5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 *"),
("C88a", "Spanish: Closed, 6...b5 7.Bb3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 *"),
("C88b", "Spanish: Closed, Trajkovic Counterattack", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 Bb7 *"),
("C88c", "Spanish: Closed 7...d6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 d6 *"),
("C88c", "Spanish: Closed, 7...d6 8.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 d6 8.d4 *"),
("C88c", "Spanish: Closed, Noah's Ark Trap", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 d6 8.d4 Nxd4 9.Nxd4 exd4 10.Qxd4 c5 *"),
("C88d", "Spanish: Closed 7...d6 8.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 d6 8.c3 *"),
("C88d", "Spanish: Closed 7...d6 8.c3 Bg4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 d6 8.c3 Bg4 *"),
("C88e", "Spanish: Closed 7...d6 8.c3 Na5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 d6 8.c3 Na5 *"),
("C88e", "Spanish: Closed, Leonhardt Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 d6 8.c3 Na5 9.Bc2 c5 10.d4 Qc7 11.h3 Nc6 12.d5 Nb8 13.Nbd2 g5 *"),
("C88e", "Spanish: Closed, Balla Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 d6 8.c3 Na5 9.Bc2 c5 10.d4 Qc7 11.a4 *"),
("C88f", "Spanish: Closed, 7...O-O", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O *"),
("C88g", "Spanish: Closed, 8.d3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.d3 *"),
("C88g", "Spanish: Closed, 8.d3 d6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.d3 d6 *"),
("C88h", "Spanish: Closed, 8.h3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.h3 *"),
("C88i", "Spanish: Closed, 8.h3 Bb7 9.d3 d6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.h3 Bb7 9.d3 d6 *"),
("C88j", "Spanish: Closed, Anti-Marshall 8.a4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.a4 *"),
("C88k", "Spanish: Closed, Anti-Marshall 8.a4 b4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.a4 b4 *"),
("C88l", "Spanish: Closed, Anti-Marshall 8.a4 Bb7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.a4 Bb7 *"),
("C88m", "Spanish: Closed, Anti-Marshall 8.a4 Bb7 9.d3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.a4 Bb7 9.d3 *"),
("C88n", "Spanish: Closed, Anti-Marshall 8.a4 Bb7 9.d3 d6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.a4 Bb7 9.d3 d6 *"),
("C88o", "Spanish: Closed, Anti-Marshall 8.a4 Bb7 9.d3 d6 10.Nc3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.a4 Bb7 9.d3 d6 10.Nc3 *"),
("C88p", "Spanish: Closed, Anti-Marshall 8.a4 Bb7 9.d3 d6 10.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.a4 Bb7 9.d3 d6 10.c3 *"),
("C88q", "Spanish: Closed, 8.c3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 *"),
("C89a", "Spanish: Marshall Counterattack", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 *"),
("C89b", "Spanish: Marshall, 9.exd5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 *"),
("C89b", "Spanish: Marshall, Herman Steiner Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 e4 *"),
("C89c", "Spanish: Marshall, 9.exd5 Nxd5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 *"),
("C89d", "Spanish: Marshall, 9.exd5 Nxd5 10.Nxe5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 *"),
("C89d", "Spanish: Marshall, 9.exd5 Nxd5 10.Nxe5 Nxe5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 *"),
("C89d", "Spanish: Marshall, 11.Rxe5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 *"),
("C89e", "Spanish: Marshall, 11.Rxe5 Nf6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 Nf6 *"),
("C89f", "Spanish: Marshall, 11.Rxe5 c6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 *"),
("C89g", "Spanish: Marshall, 12.Bxd5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.Bxd5 *"),
("C89h", "Spanish: Marshall, Kevitz Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.Bxd5 cxd5 13.d4 Bd6 14.Re3 *"),
("C89i", "Spanish: Marshall, 12.d3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.d3 *"),
("C89i", "Spanish: Marshall, 12.d3 Bd6 13.Re1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.d3 Bd6 13.Re1 *"),
("C89j", "Spanish: Marshall, 12.d3 Bd6 13.Re1 Qh4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.d3 Bd6 13.Re1 Qh4 *"),
("C89k", "Spanish: Marshall, Main Line (12.d4)", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.d4 *"),
("C89k", "Spanish: Marshall, Main Line (12.d4 Bd6)", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.d4 Bd6 *"),
("C89l", "Spanish: Marshall, Main Line, 13.Re2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.d4 Bd6 13.Re2 *"),
("C89m", "Spanish: Marshall, Main Line, 13.Re1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.d4 Bd6 13.Re1 *"),
("C89m", "Spanish: Marshall, Main Line, 13.Re1 Qh4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.d4 Bd6 13.Re1 Qh4 *"),
("C89m", "Spanish: Marshall, Main Line, 13.Re1 Qh4 14.g3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.d4 Bd6 13.Re1 Qh4 14.g3 *"),
("C89n", "Spanish: Marshall, Main Line, 14.g3 Qh3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.d4 Bd6 13.Re1 Qh4 14.g3 Qh3 *"),
("C89o", "Spanish: Marshall, Main Line, 15.Re4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.d4 Bd6 13.Re1 Qh4 14.g3 Qh3 15.Re4 *"),
("C89p", "Spanish: Marshall, Main Line, 15.Be3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.d4 Bd6 13.Re1 Qh4 14.g3 Qh3 15.Be3 *"),
("C89p", "Spanish: Marshall, Main Line, 15.Be3 Bg4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.d4 Bd6 13.Re1 Qh4 14.g3 Qh3 15.Be3 Bg4 *"),
("C89p", "Spanish: Marshall, Main Line, 15.Be3 Bg4 16.Qd3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.d4 Bd6 13.Re1 Qh4 14.g3 Qh3 15.Be3 Bg4 16.Qd3 *"),
("C89q", "Spanish: Marshall, Main Line, 15.Be3 Bg4 16.Qd3 Rae8", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.d4 Bd6 13.Re1 Qh4 14.g3 Qh3 15.Be3 Bg4 16.Qd3 Rae8 *"),
("C89q", "Spanish: Marshall, Main Line, 16.Qd3 Rae8 17.Nd2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.d4 Bd6 13.Re1 Qh4 14.g3 Qh3 15.Be3 Bg4 16.Qd3 Rae8 17.Nd2 *"),
("C89r", "Spanish: Marshall, Main Line, Pawn Push Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.d4 Bd6 13.Re1 Qh4 14.g3 Qh3 15.Be3 Bg4 16.Qd3 Rae8 17.Nd2 f5 *"),
("C89s", "Spanish: Marshall, Main Line, Classical Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.d4 Bd6 13.Re1 Qh4 14.g3 Qh3 15.Be3 Bg4 16.Qd3 Rae8 17.Nd2 Re6 *"),
("C89t", "Spanish: Marshall, Main Line, Classical, 18.c4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.d4 Bd6 13.Re1 Qh4 14.g3 Qh3 15.Be3 Bg4 16.Qd3 Rae8 17.Nd2 Re6 18.c4 *"),
("C89u", "Spanish: Marshall, Main Line, Classical, 18.Bxd5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.d4 Bd6 13.Re1 Qh4 14.g3 Qh3 15.Be3 Bg4 16.Qd3 Rae8 17.Nd2 Re6 18.Bxd5 *"),
("C89v", "Spanish: Marshall, Main Line, Classical, 18.Qf1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.d4 Bd6 13.Re1 Qh4 14.g3 Qh3 15.Be3 Bg4 16.Qd3 Rae8 17.Nd2 Re6 18.Qf1 *"),
("C89w", "Spanish: Marshall, Main Line, Classical, 18.a4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.d4 Bd6 13.Re1 Qh4 14.g3 Qh3 15.Be3 Bg4 16.Qd3 Rae8 17.Nd2 Re6 18.a4 *"),
("C89x", "Spanish: Marshall, Main Line, Classical, Spassky Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.d4 Bd6 13.Re1 Qh4 14.g3 Qh3 15.Be3 Bg4 16.Qd3 Rae8 17.Nd2 Re6 18.a4 Qh5 *"),
("C89y", "Spanish: Marshall, Main Line, Classical, 18.a4 f5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d5 9.exd5 Nxd5 10.Nxe5 Nxe5 11.Rxe5 c6 12.d4 Bd6 13.Re1 Qh4 14.g3 Qh3 15.Be3 Bg4 16.Qd3 Rae8 17.Nd2 Re6 18.a4 f5 *"),
("C90a", "Spanish: Closed, 8...d6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 *"),
("C90b", "Spanish: Closed, Lutikov Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.Bc2 *"),
("C90c", "Spanish: Closed, Suetin Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.a3 *"),
("C90d", "Spanish: Closed, 8...d6 9.a4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.a4 *"),
("C90d", "Spanish: Closed, 8...d6 9.a4 Bg4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.a4 Bg4 *"),
("C90e", "Spanish: Closed, Pilnik Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.d3 *"),
("C90f", "Spanish: Closed, Pilnik, 9...h6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.d3 h6 *"),
("C90g", "Spanish: Closed, Pilnik, 9...Na5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.d3 Na5 *"),
("C90g", "Spanish: Closed, Pilnik, 9...Na5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.d3 Na5 10.Bc2 *"),
("C90g", "Spanish: Closed, Pilnik, 9...Na5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.d3 Na5 10.Bc2 c5 *"),
("C90h", "Spanish: Closed, Pilnik, 11.Nbd2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.d3 Na5 10.Bc2 c5 11.Nbd2 *"),
("C90i", "Spanish: Closed, Pilnik, 11.Nbd2 Qc7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.d3 Na5 10.Bc2 c5 11.Nbd2 Qc7 *"),
("C90j", "Spanish: Closed, Pilnik, 11.Nbd2 Nc6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.d3 Na5 10.Bc2 c5 11.Nbd2 Nc6 *"),
("C90k", "Spanish: Closed, Pilnik, 11.Nbd2 Re8", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.d3 Na5 10.Bc2 c5 11.Nbd2 Re8 *"),
("C90k", "Spanish: Closed, Pilnik, 12.Nf1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.d3 Na5 10.Bc2 c5 11.Nbd2 Re8 12.Nf1 *"),
("C90l", "Spanish: Closed, Pilnik, 12.Nf1 Nc6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.d3 Na5 10.Bc2 c5 11.Nbd2 Re8 12.Nf1 Nc6 *"),
("C90m", "Spanish: Closed, Pilnik, 12.Nf1 Bf8", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.d3 Na5 10.Bc2 c5 11.Nbd2 Re8 12.Nf1 Bf8 *"),
("C90n", "Spanish: Closed, Pilnik, 12.Nf1 h6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.d3 Na5 10.Bc2 c5 11.Nbd2 Re8 12.Nf1 h6 *"),
("C91a", "Spanish: Closed, 9.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.d4 *"),
("C91b", "Spanish: Closed, Bogoljubow Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.d4 Bg4 *"),
("C91c", "Spanish: Closed, Bogoljubow, 10.Be3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.d4 Bg4 10.Be3 *"),
("C91d", "Spanish: Closed, Bogoljubow, 10.Be3 exd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.d4 Bg4 10.Be3 exd4 *"),
("C91e", "Spanish: Closed, Bogoljubow, 10.d5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.d4 Bg4 10.d5 *"),
("C91f", "Spanish: Closed, Bogoljubow, 10.d5 Na5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.d4 Bg4 10.d5 Na5 *"),
("C91f", "Spanish: Closed, Bogoljubow, 10.d5 Na5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.d4 Bg4 10.d5 Na5 11.Bc2 *"),
("C91g", "Spanish: Closed, Bogoljubow, 10.d5 Na5 11.Bc2 Qc8", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.d4 Bg4 10.d5 Na5 11.Bc2 Qc8 *"),
("C91h", "Spanish: Closed, Bogoljubow, 10.d5 Na5 11.Bc2 c6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.d4 Bg4 10.d5 Na5 11.Bc2 c6 *"),
("C91i", "Spanish: Closed, Bogoljubow, 10.d5 Na5 11.Bc2 c6 12.h3 Bc8", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.d4 Bg4 10.d5 Na5 11.Bc2 c6 12.h3 Bc8 *"),
("C92a", "Spanish: Closed, 9.h3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 *"),
("C92b", "Spanish: Closed, Keres Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 a5 *"),
("C92c", "Spanish: Closed, Kholmov Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Be6 *"),
("C92d", "Spanish: Closed, Kholmov, 11.Qxb3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Be6 10.d4 Bxb3 11.Qxb3 *"),
("C92e", "Spanish: Closed, Karpov Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nd7 *"),
("C92f", "Spanish: Closed, Karpov Variation, 10.d4 Bf6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nd7 10.d4 Bf6 *"),
("C92g", "Spanish: Closed, Karpov Variation, 10.d4 Bf6 11.a4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nd7 10.d4 Bf6 11.a4 *"),
("C92h", "Spanish: Closed, Karpov Variation, 10.d4 Bf6 11.a4 Bb7 12.Na3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nd7 10.d4 Bf6 11.a4 Bb7 12.Na3 *"),
("C92i", "Spanish: Closed, Zaitsev (Flohr) Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Bb7 *"),
("C92j", "Spanish: Closed, Zaitsev, 10.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Bb7 10.d4 *"),
("C92k", "Spanish: Closed, Zaitsev, 10.d4 Re8", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Bb7 10.d4 Re8 *"),
("C92l", "Spanish: Closed, Zaitsev, 11.Ng5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Bb7 10.d4 Re8 11.Ng5 *"),
("C92m", "Spanish: Closed, Zaitsev, 11.Nbd2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Bb7 10.d4 Re8 11.Nbd2 *"),
("C92n", "Spanish: Closed, Zaitsev, 11.Nbd2 Bf8 12.d5 Nb8", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Bb7 10.d4 Re8 11.Nbd2 Bf8 12.d5 Nb8 *"),
("C92o", "Spanish: Closed, Zaitsev, 12.Bc2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Bb7 10.d4 Re8 11.Nbd2 Bf8 12.Bc2 *"),
("C92p", "Spanish: Closed, Zaitsev, 12.a3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Bb7 10.d4 Re8 11.Nbd2 Bf8 12.a3 *"),
("C92q", "Spanish: Closed, Zaitsev, 12.a4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Bb7 10.d4 Re8 11.Nbd2 Bf8 12.a4 *"),
("C92r", "Spanish: Closed, Zaitsev, 12.a4 h6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Bb7 10.d4 Re8 11.Nbd2 Bf8 12.a4 h6 *"),
("C92s", "Spanish: Closed, Zaitsev, 12.a4 h6 13.Bc2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Bb7 10.d4 Re8 11.Nbd2 Bf8 12.a4 h6 13.Bc2 *"),
("C92t", "Spanish: Closed, Zaitsev, 12.a4 h6 13.Bc2 exd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Bb7 10.d4 Re8 11.Nbd2 Bf8 12.a4 h6 13.Bc2 exd4 *"),
("C92u", "Spanish: Closed, Zaitsev, 12.a4 h6 13.Bc2 exd4 14.cxd4 Nb4 15.Bb1 c5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Bb7 10.d4 Re8 11.Nbd2 Bf8 12.a4 h6 13.Bc2 exd4 14.cxd4 Nb4 15.Bb1 c5 *"),
("C93", "Spanish: Closed, Smyslov Defence", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 h6 *"),
("C93", "Spanish: Closed, Smyslov, 10.d3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 h6 10.d3 *"),
("C93", "Spanish: Closed, Smyslov, 10.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 h6 10.d4 *"),
("C93", "Spanish: Closed, Smyslov, 10.d4 Re8", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 h6 10.d4 Re8 *"),
("C93", "Spanish: Closed, Smyslov, 10.d4 Re8 11.a4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 h6 10.d4 Re8 11.a4 *"),
("C93", "Spanish: Closed, Smyslov, 10.d4 Re8 11.Be3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 h6 10.d4 Re8 11.Be3 *"),
("C93", "Spanish: Closed, Smyslov, 10.d4 Re8 11.Nbd2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 h6 10.d4 Re8 11.Nbd2 *"),
("C93", "Spanish: Closed, Smyslov, 10.d4 Re8 11.Nbd2 Bf8", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 h6 10.d4 Re8 11.Nbd2 Bf8 *"),
("C93", "Spanish: Closed, Smyslov, 12.a3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 h6 10.d4 Re8 11.Nbd2 Bf8 12.a3 *"),
("C93", "Spanish: Closed, Smyslov, 12.Bc2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 h6 10.d4 Re8 11.Nbd2 Bf8 12.Bc2 *"),
("C93", "Spanish: Closed, Smyslov, 12.Nf1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 h6 10.d4 Re8 11.Nbd2 Bf8 12.Nf1 *"),
("C93", "Spanish: Closed, Smyslov, 12.Nf1 Bb7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 h6 10.d4 Re8 11.Nbd2 Bf8 12.Nf1 Bb7 *"),
("C93", "Spanish: Closed, Smyslov, 12.Nf1 Bb7 13.Ng3 Na5 14.Bc2 Nc4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 h6 10.d4 Re8 11.Nbd2 Bf8 12.Nf1 Bb7 13.Ng3 Na5 14.Bc2 Nc4 *"),
("C93", "Spanish: Closed, Smyslov, 12.Nf1 Bd7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 h6 10.d4 Re8 11.Nbd2 Bf8 12.Nf1 Bd7 *"),
("C93", "Spanish: Closed, Smyslov, 12.Nf1 Bd7 13.Ng3 Na5 14.Bc2 c5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 h6 10.d4 Re8 11.Nbd2 Bf8 12.Nf1 Bd7 13.Ng3 Na5 14.Bc2 c5 *"),
("C94a", "Spanish: Closed, Breyer Defence", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 *"),
("C94b", "Spanish: Closed, Breyer, Matulovic Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.a4 *"),
("C94c", "Spanish: Closed, Breyer, 10.d3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d3 *"),
("C94d", "Spanish: Closed, Breyer, 10.d3 Nbd7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d3 Nbd7 *"),
("C94e", "Spanish: Closed, Breyer, 10.d3 Nbd7 11.Nbd2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d3 Nbd7 11.Nbd2 *"),
("C94f", "Spanish: Closed, Breyer, 10.d3 Nbd7 11.Nbd2 Bb7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d3 Nbd7 11.Nbd2 Bb7 *"),
("C95a", "Spanish: Closed, Breyer, 10.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d4 *"),
("C95a", "Spanish: Closed, Breyer, 10.d4 Bb7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d4 Bb7 *"),
("C95b", "Spanish: Closed, Breyer, 10.d4 Nbd7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d4 Nbd7 *"),
("C95c", "Spanish: Closed, Breyer, Simagin Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d4 Nbd7 11.Nh4 *"),
("C95d", "Spanish: Closed, Breyer, 10.d4 Nbd7 11.Bg5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d4 Nbd7 11.Bg5 *"),
("C95e", "Spanish: Closed, Breyer, Arseniev Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d4 Nbd7 11.c4 *"),
("C95e", "Spanish: Closed, Breyer, Arseniev, 11...c6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d4 Nbd7 11.c4 c6 *"),
("C95f", "Spanish: Closed, Breyer, 10.d4 Nbd7 11.Nbd2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d4 Nbd7 11.Nbd2 *"),
("C95f", "Spanish: Closed, Breyer, 10.d4 Nbd7 11.Nbd2 Bb7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d4 Nbd7 11.Nbd2 Bb7 *"),
("C95f", "Spanish: Closed, Breyer, 10.d4 Nbd7 11.Nbd2 Bb7 12.a4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d4 Nbd7 11.Nbd2 Bb7 12.a4 *"),
("C95g", "Spanish: Closed, Breyer, 10.d4 Nbd7 11.Nbd2 Bb7 12.Bc2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d4 Nbd7 11.Nbd2 Bb7 12.Bc2 *"),
("C95g", "Spanish: Closed, Breyer, Gligoric Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d4 Nbd7 11.Nbd2 Bb7 12.Bc2 c5 *"),
("C95h", "Spanish: Closed, Breyer, Main Line", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d4 Nbd7 11.Nbd2 Bb7 12.Bc2 Re8 *"),
("C95i", "Spanish: Closed, Breyer, Main Line, 13.b4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d4 Nbd7 11.Nbd2 Bb7 12.Bc2 Re8 13.b4 *"),
("C95j", "Spanish: Closed, Breyer, Main Line, 13.a4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d4 Nbd7 11.Nbd2 Bb7 12.Bc2 Re8 13.a4 *"),
("C95k", "Spanish: Closed, Breyer, Main Line, 13.Nf1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d4 Nbd7 11.Nbd2 Bb7 12.Bc2 Re8 13.Nf1 *"),
("C95k", "Spanish: Closed, Breyer, Main Line, 13.Nf1 Bf8", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d4 Nbd7 11.Nbd2 Bb7 12.Bc2 Re8 13.Nf1 Bf8 *"),
("C95k", "Spanish: Closed, Breyer, Main Line, 13.Nf1 Bf8 14.Ng3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d4 Nbd7 11.Nbd2 Bb7 12.Bc2 Re8 13.Nf1 Bf8 14.Ng3 *"),
("C95k", "Spanish: Closed, Breyer, Main Line, 14.Ng3 c5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d4 Nbd7 11.Nbd2 Bb7 12.Bc2 Re8 13.Nf1 Bf8 14.Ng3 c5 *"),
("C95l", "Spanish: Closed, Breyer, Main Line, 14.Ng3 g6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d4 Nbd7 11.Nbd2 Bb7 12.Bc2 Re8 13.Nf1 Bf8 14.Ng3 g6 *"),
("C95l", "Spanish: Closed, Breyer, Main Line, 14.Ng3 g6 15.b3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d4 Nbd7 11.Nbd2 Bb7 12.Bc2 Re8 13.Nf1 Bf8 14.Ng3 g6 15.b3 *"),
("C95m", "Spanish: Closed, Breyer, Main Line, 14.Ng3 g6 15.a4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d4 Nbd7 11.Nbd2 Bb7 12.Bc2 Re8 13.Nf1 Bf8 14.Ng3 g6 15.a4 *"),
("C95n", "Spanish: Closed, Breyer, Main Line, 15.a4 c5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d4 Nbd7 11.Nbd2 Bb7 12.Bc2 Re8 13.Nf1 Bf8 14.Ng3 g6 15.a4 c5 *"),
("C95o", "Spanish: Closed, Breyer, Main Line, 15.a4 c5 16.d5 c4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Nb8 10.d4 Nbd7 11.Nbd2 Bb7 12.Bc2 Re8 13.Nf1 Bf8 14.Ng3 g6 15.a4 c5 16.d5 c4 *"),
("C96a", "Spanish: Closed, Chigorin", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 *"),
("C96b", "Spanish: Closed, Chigorin, 10.Bc2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 *"),
("C96c", "Spanish: Closed, Chigorin, 10...c6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c6 *"),
("C96d", "Spanish: Closed, Chigorin, Rossolimo Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c6 11.d4 Qc7 *"),
("C96e", "Spanish: Closed, Chigorin, 10...Bb7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 Bb7 *"),
("C96f", "Spanish: Closed, Chigorin, 10...c5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 *"),
("C96g", "Spanish: Closed, Chigorin, 11.d3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d3 *"),
("C96h", "Spanish: Closed, Chigorin, 11.d3 Nc6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d3 Nc6 *"),
("C96i", "Spanish: Closed, Chigorin, 11.d4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 *"),
("C96j", "Spanish: Closed, Chigorin, 11.d4 cxd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 cxd4 *"),
("C96k", "Spanish: Closed, Chigorin, 11.d4 Bb7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Bb7 *"),
("C96l", "Spanish: Closed, Chigorin, 11.d4 Bb7 12.Nbd2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Bb7 12.Nbd2 *"),
("C96m", "Spanish: Closed, Chigorin, Borisenko Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Nc6 *"),
("C96n", "Spanish: Closed, Chigorin, Keres Variation", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Nd7 *"),
("C96o", "Spanish: Closed, Chigorin, Keres, 11.Nbd2 cxd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Nd7 12.Nbd2 cxd4 *"),
("C97a", "Spanish: Closed, Chigorin, 11.d4 Qc7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 *"),
("C97b", "Spanish: Closed, Chigorin, 11.d4 Qc7 12.d5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.d5 *"),
("C97c", "Spanish: Closed, Chigorin, 11.d4 Qc7 12.d5 c4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.d5 c4 *"),
("C97d", "Spanish: Closed, Chigorin, 11.d4 Qc7 12.Nbd2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 *"),
("C97e", "Spanish: Closed, Chigorin, 12...Bb7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 Bb7 *"),
("C97f", "Spanish: Closed, Chigorin, 12...Re8", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 Re8 *"),
("C97g", "Spanish: Closed, Chigorin, 12...Rd8", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 Rd8 *"),
("C97h", "Spanish: Closed, Chigorin, 12...Bd7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 Bd7 *"),
("C97i", "Spanish: Closed, Chigorin, 12...Bd7 13.Nf1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 Bd7 13.Nf1 *"),
("C97j", "Spanish: Closed, Chigorin, 12...Bd7 13.Nf1 Nc4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 Bd7 13.Nf1 Nc4 *"),
("C97k", "Spanish: Closed, Chigorin, 12...Bd7 13.Nf1 Rfe8", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 Bd7 13.Nf1 Rfe8 *"),
("C97l", "Spanish: Closed, Chigorin, Yugoslav System", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 Bd7 13.Nf1 Rfe8 14.Ne3 g6 *"),
("C98a", "Spanish: Closed, Chigorin, 12...Nc6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 Nc6 *"),
("C98b", "Spanish: Closed, Chigorin, Rauzer Attack", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 Nc6 13.dxc5 *"),
("C98b", "Spanish: Closed, Chigorin, Rauzer Attack", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 Nc6 13.dxc5 dxc5 *"),
("C98c", "Spanish: Closed, Chigorin, Rauzer, 14.Nf1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 Nc6 13.dxc5 dxc5 14.Nf1 *"),
("C98d", "Spanish: Closed, Chigorin, Rauzer, 14.Nf1 Be6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 Nc6 13.dxc5 dxc5 14.Nf1 Be6 *"),
("C98e", "Spanish: Closed, Chigorin, 12...Nc6 13.d5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 Nc6 13.d5 *"),
("C98f", "Spanish: Closed, Chigorin, 12...Nc6 13.d5 Na5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 Nc6 13.d5 Na5 *"),
("C98g", "Spanish: Closed, Chigorin, 12...Nc6 13.d5 Nd8", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 Nc6 13.d5 Nd8 *"),
("C98h", "Spanish: Closed, Chigorin, 12...Nc6 13.d5 Nd8 14.Nf1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 Nc6 13.d5 Nd8 14.Nf1 *"),
("C98i", "Spanish: Closed, Chigorin, 12...Nc6 13.d5 Nd8 14.a4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 Nc6 13.d5 Nd8 14.a4 *"),
("C99a", "Spanish: Closed, Chigorin, 12...cxd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 cxd4 *"),
("C99a", "Spanish: Closed, Chigorin, 12...cxd4 13.cxd4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 cxd4 13.cxd4 *"),
("C99b", "Spanish: Closed, Chigorin, 13...Rd8", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 cxd4 13.cxd4 Rd8 *"),
("C99c", "Spanish: Closed, Chigorin, 13...Bd7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 cxd4 13.cxd4 Bd7 *"),
("C99c", "Spanish: Closed, Chigorin, 13...Bd7 14.Nf1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 cxd4 13.cxd4 Bd7 14.Nf1 *"),
("C99d", "Spanish: Closed, Chigorin, 13...Bd7 14.Nf1 Rac8 15.Ne3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 cxd4 13.cxd4 Bd7 14.Nf1 Rac8 15.Ne3 *"),
("C99e", "Spanish: Closed, Chigorin, 13...Bb7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 cxd4 13.cxd4 Bb7 *"),
("C99e", "Spanish: Closed, Chigorin, 13...Bb7 14.Nf1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 cxd4 13.cxd4 Bb7 14.Nf1 *"),
("C99e", "Spanish: Closed, Chigorin, 13...Bb7 14.Nf1 Rac8", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 cxd4 13.cxd4 Bb7 14.Nf1 Rac8 *"),
("C99f", "Spanish: Closed, Chigorin, 13...Bb7 14.Nf1 Rac8 15.Re2", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 cxd4 13.cxd4 Bb7 14.Nf1 Rac8 15.Re2 *"),
("C99g", "Spanish: Closed, Chigorin, 13...Bb7 14.d5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 cxd4 13.cxd4 Bb7 14.d5 *"),
("C99h", "Spanish: Closed, Chigorin, 13...Bb7 14.d5 Rac8", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 cxd4 13.cxd4 Bb7 14.d5 Rac8 *"),
("C99i", "Spanish: Closed, Chigorin, 13...Nc6", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 cxd4 13.cxd4 Nc6 *"),
("C99i", "Spanish: Closed, Chigorin, 13...Nc6 14.a3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 cxd4 13.cxd4 Nc6 14.a3 *"),
("C99j", "Spanish: Closed, Chigorin, 13...Nc6 14.Nf1", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 cxd4 13.cxd4 Nc6 14.Nf1 *"),
("C99k", "Spanish: Closed, Chigorin, 13...Nc6 14.d5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 cxd4 13.cxd4 Nc6 14.d5 *"),
("C99l", "Spanish: Closed, Chigorin, 13...Nc6 14.Nb3", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 cxd4 13.cxd4 Nc6 14.Nb3 *"),
("C99l", "Spanish: Closed, Chigorin, 13...Nc6 14.Nb3 a5", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 cxd4 13.cxd4 Nc6 14.Nb3 a5 *"),
("C99m", "Spanish: Closed, Chigorin, 13...Nc6 14.Nb3 a5 15.Be3 a4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 cxd4 13.cxd4 Nc6 14.Nb3 a5 15.Be3 a4 *"),
("C99n", "Spanish: Closed, Chigorin, 13...Nc6 14.Nb3 a5 15.Be3 a4 16.Nbd2 Nb4", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 cxd4 13.cxd4 Nc6 14.Nb3 a5 15.Be3 a4 16.Nbd2 Nb4 *"),
("C99o", "Spanish: Closed, Chigorin, 13...Nc6 14.Nb3 a5 15.Be3 a4 16.Nbd2 Bd7", "1.e4 e5 2.Nf3 Nc6 3.Bb5 a6 4.Ba4 Nf6 5.O-O Be7 6.Re1 b5 7.Bb3 O-O 8.c3 d6 9.h3 Na5 10.Bc2 c5 11.d4 Qc7 12.Nbd2 cxd4 13.cxd4 Nc6 14.Nb3 a5 15.Be3 a4 16.Nbd2 Bd7 *"),
("D00a", "Queen's Pawn Game", "1.d4 d5 *"),
("D00a", "Queen's Pawn: 2.f4", "1.d4 d5 2.f4 *"),
("D00a", "Queen's Pawn: 2.g3", "1.d4 d5 2.g3 *"),
("D00b", "Queen's Pawn: 2.c3", "1.d4 d5 2.c3 *"),
("D00b", "Queen's Pawn: 2.c3 Nf6", "1.d4 d5 2.c3 Nf6 *"),
("D00b", "Queen's Pawn: 2.c3 Nf6 3.Bf4", "1.d4 d5 2.c3 Nf6 3.Bf4 *"),
("D00b", "Queen's Pawn: 2.c3 Nf6 3.Bg5", "1.d4 d5 2.c3 Nf6 3.Bg5 *"),
("D00c", "Queen's Pawn: Mason Variation", "1.d4 d5 2.Bf4 *"),
("D00c", "Queen's Pawn, Mason, Steinitz Countergambit", "1.d4 d5 2.Bf4 c5 *"),
("D00d", "Queen's Pawn: 2.e3", "1.d4 d5 2.e3 *"),
("D00d", "Queen's Pawn: 2.e3 Nf6", "1.d4 d5 2.e3 Nf6 *"),
("D00e", "Queen's Pawn: Stonewall Attack", "1.d4 d5 2.e3 Nf6 3.Bd3 *"),
("D00f", "Hodgson Attack (Trompowsky vs. 1...d5)", "1.d4 d5 2.Bg5 *"),
("D00f", "Hodgson Attack: Welling Variation", "1.d4 d5 2.Bg5 Bg4 *"),
("D00f", "Hodgson Attack: 2...f6", "1.d4 d5 2.Bg5 f6 *"),
("D00f", "Hodgson Attack: 2...g6", "1.d4 d5 2.Bg5 g6 *"),
("D00g", "Hodgson Attack: 2...c6", "1.d4 d5 2.Bg5 c6 *"),
("D00h", "Hodgson Attack, 2...h6", "1.d4 d5 2.Bg5 h6 *"),
("D00h", "Hodgson Attack: 2...h6 3.Bh4 c6", "1.d4 d5 2.Bg5 h6 3.Bh4 c6 *"),
("D00h", "Hodgson Attack: 2...h6 3.Bh4 c6 4.e3", "1.d4 d5 2.Bg5 h6 3.Bh4 c6 4.e3 *"),
("D00h", "Hodgson Attack: 2...h6 3.Bh4 c6 4.e3 Qb6", "1.d4 d5 2.Bg5 h6 3.Bh4 c6 4.e3 Qb6 *"),
("D00i", "Trompowsky: 2...d5", "1.d4 Nf6 2.Bg5 d5 *"),
("D00i", "Trompowsky: 2...d5 3.Nd2", "1.d4 Nf6 2.Bg5 d5 3.Nd2 *"),
("D00i", "Trompowsky: 2...d5 3.e3", "1.d4 Nf6 2.Bg5 d5 3.e3 *"),
("D00i", "Trompowsky: 2...d5 3.e3 e6", "1.d4 Nf6 2.Bg5 d5 3.e3 e6 *"),
("D00j", "Trompowsky: 2...d5 3.Bxf6", "1.d4 Nf6 2.Bg5 d5 3.Bxf6 *"),
("D00j", "Trompowsky: 2...d5 3.Bxf6 gxf6", "1.d4 Nf6 2.Bg5 d5 3.Bxf6 gxf6 *"),
("D00j", "Trompowsky: 2...d5 3.Bxf6 gxf6 4.e3", "1.d4 Nf6 2.Bg5 d5 3.Bxf6 gxf6 4.e3 *"),
("D00j", "Trompowsky: 2...d5 3.Bxf6 gxf6 4.e3 c5", "1.d4 Nf6 2.Bg5 d5 3.Bxf6 gxf6 4.e3 c5 *"),
("D00k", "Trompowsky: 2...d5 3.Bxf6 exf6", "1.d4 Nf6 2.Bg5 d5 3.Bxf6 exf6 *"),
("D00k", "Trompowsky: 2...d5 3.Bxf6 exf6 4.e3", "1.d4 Nf6 2.Bg5 d5 3.Bxf6 exf6 4.e3 *"),
("D00k", "Trompowsky: 2...d5 3.Bxf6 exf6 4.e3 Bf5", "1.d4 Nf6 2.Bg5 d5 3.Bxf6 exf6 4.e3 Bf5 *"),
("D00k", "Trompowsky: 2...d5 3.Bxf6 exf6 4.e3 c6", "1.d4 Nf6 2.Bg5 d5 3.Bxf6 exf6 4.e3 c6 *"),
("D00k", "Trompowsky: 2...d5 3.Bxf6 exf6 4.e3 Be6", "1.d4 Nf6 2.Bg5 d5 3.Bxf6 exf6 4.e3 Be6 *"),
("D00k", "Trompowsky: 2...d5 3.Bxf6 exf6 4.e3 Bd6", "1.d4 Nf6 2.Bg5 d5 3.Bxf6 exf6 4.e3 Bd6 *"),
("D00l", "Blackmar-Diemer Gambit (BDG): 2.e4", "1.d4 d5 2.e4 *"),
("D00l", "Blackmar-Diemer: Beyer Countergambit", "1.d4 d5 2.e4 e5 *"),
("D00l", "Blackmar-Diemer Gambit (BDG): 2.e4 dxe4", "1.d4 d5 2.e4 dxe4 *"),
("D00l", "Blackmar-Diemer: Gedult Gambit", "1.d4 d5 2.e4 dxe4 3.f3 *"),
("D00l", "Blackmar-Diemer: Fritz Attack", "1.d4 d5 2.e4 dxe4 3.Bc4 *"),
("D00l", "Blackmar-Diemer: 2.e4 dxe4 3.Nc3", "1.d4 d5 2.e4 dxe4 3.Nc3 *"),
("D00l", "Blackmar-Diemer: Grosshans Defence", "1.d4 d5 2.e4 dxe4 3.Nc3 Bd7 *"),
("D00l", "Blackmar-Diemer: Zeller Defence", "1.d4 d5 2.e4 dxe4 3.Nc3 Bf5 *"),
("D00l", "Blackmar-Diemer: Pohmlann Defence", "1.d4 d5 2.e4 dxe4 3.Nc3 f5 *"),
("D00m", "Blackmar-Diemer: Lemberger Countergambit", "1.d4 d5 2.e4 dxe4 3.Nc3 e5 *"),
("D00m", "Blackmar-Diemer: Lemberger Countergambit, Rassmussen Attack", "1.d4 d5 2.e4 dxe4 3.Nc3 e5 4.Nge2 *"),
("D00m", "Blackmar-Diemer: Lemberger Countergambit, Sneider Attack", "1.d4 d5 2.e4 dxe4 3.Nc3 e5 4.Qh5 *"),
("D00n", "Queen's Pawn: Veresov Attack", "1.d4 d5 2.Nc3 *"),
("D00n", "Queen's Pawn: Veresov Attack", "1.d4 d5 2.Nc3 Bf5 *"),
("D00n", "Queen's Pawn: Veresov Attack", "1.d4 d5 2.Nc3 Nf6 *"),
("D00o", "Queen's Pawn: Veresov, 3.Bf4", "1.d4 d5 2.Nc3 Nf6 3.Bf4 *"),
("D00p", "Queen's Pawn: Veresov, 3.Nf3", "1.d4 d5 2.Nc3 Nf6 3.Nf3 *"),
("D00p", "Queen's Pawn: Veresov, 3.Nf3 g6", "1.d4 d5 2.Nc3 Nf6 3.Nf3 g6 *"),
("D00q", "Queen's Pawn: Anti-King's Indian", "1.d4 d5 2.Nc3 Nf6 3.Nf3 g6 4.Bf4 *"),
("D00r", "Queen's Pawn: Anti-King's Indian, Main Line", "1.d4 d5 2.Nc3 Nf6 3.Nf3 g6 4.Bf4 Bg7 5.e3 O-O 6.Be2 *"),
("D00s", "Blackmar-Diemer Gambit (BDG)", "1.d4 d5 2.Nc3 Nf6 3.e4 *"),
("D00s", "Blackmar-Diemer: Hubsch Gambit", "1.d4 d5 2.Nc3 Nf6 3.e4 Nxe4 *"),
("D00t", "Blackmar-Diemer: 3...dxe4", "1.d4 d5 2.Nc3 Nf6 3.e4 dxe4 *"),
("D00t", "Blackmar-Diemer: von Popiel Attack", "1.d4 d5 2.Nc3 Nf6 3.e4 dxe4 4.Bg5 *"),
("D00u", "Blackmar-Diemer: 4.f3", "1.d4 d5 2.Nc3 Nf6 3.e4 dxe4 4.f3 *"),
("D00u", "Blackmar-Diemer: O'Kelly Defence", "1.d4 d5 2.Nc3 Nf6 3.e4 dxe4 4.f3 c6 *"),
("D00u", "Blackmar-Diemer: Langeheinecke Defence", "1.d4 d5 2.Nc3 Nf6 3.e4 dxe4 4.f3 e3 *"),
("D00u", "Blackmar-Diemer: Elbert Countergambit", "1.d4 d5 2.Nc3 Nf6 3.e4 dxe4 4.f3 e5 *"),
("D00u", "Blackmar-Diemer: Weinspach Defence", "1.d4 d5 2.Nc3 Nf6 3.e4 dxe4 4.f3 e6 *"),
("D00u", "Blackmar-Diemer: Lamb Defence", "1.d4 d5 2.Nc3 Nf6 3.e4 dxe4 4.f3 Nc6 *"),
("D00u", "Blackmar-Diemer: Vienna Defence", "1.d4 d5 2.Nc3 Nf6 3.e4 dxe4 4.f3 Bf5 *"),
("D00v", "Blackmar-Diemer: Accepted", "1.d4 d5 2.Nc3 Nf6 3.e4 dxe4 4.f3 exf3 *"),
("D00v", "Blackmar-Diemer: Ryder Gambit", "1.d4 d5 2.Nc3 Nf6 3.e4 dxe4 4.f3 exf3 5.Qxf3 *"),
("D00w", "Blackmar-Diemer: 4.f3 exf3 5.Nxf3", "1.d4 d5 2.Nc3 Nf6 3.e4 dxe4 4.f3 exf3 5.Nxf3 *"),
("D00w", "Blackmar-Diemer: Gunderam Defence", "1.d4 d5 2.Nc3 Nf6 3.e4 dxe4 4.f3 exf3 5.Nxf3 Bf5 *"),
("D00w", "Blackmar-Diemer: Tartakower Defence", "1.d4 d5 2.Nc3 Nf6 3.e4 dxe4 4.f3 exf3 5.Nxf3 Bg4 *"),
("D00w", "Blackmar-Diemer: Ziegler Defence", "1.d4 d5 2.Nc3 Nf6 3.e4 dxe4 4.f3 exf3 5.Nxf3 c6 *"),
("D00x", "Blackmar-Diemer: Euwe Defence", "1.d4 d5 2.Nc3 Nf6 3.e4 dxe4 4.f3 exf3 5.Nxf3 e6 *"),
("D00y", "Blackmar-Diemer: Bogoljubow Defence", "1.d4 d5 2.Nc3 Nf6 3.e4 dxe4 4.f3 exf3 5.Nxf3 g6 *"),
("D01a", "Richter-Veresov Attack", "1.d4 d5 2.Nc3 Nf6 3.Bg5 *"),
("D01b", "Richter-Veresov: 3...Ne4", "1.d4 d5 2.Nc3 Nf6 3.Bg5 Ne4 *"),
("D01c", "Richter-Veresov: 3...e6", "1.d4 d5 2.Nc3 Nf6 3.Bg5 e6 *"),
("D01d", "Richter-Veresov: 3...h6", "1.d4 d5 2.Nc3 Nf6 3.Bg5 h6 *"),
("D01e", "Richter-Veresov: 3...g6", "1.d4 d5 2.Nc3 Nf6 3.Bg5 g6 *"),
("D01f", "Richter-Veresov: 3...c6", "1.d4 d5 2.Nc3 Nf6 3.Bg5 c6 *"),
("D01g", "Richter-Veresov: 3...c5", "1.d4 d5 2.Nc3 Nf6 3.Bg5 c5 *"),
("D01h", "Richter-Veresov: 3...Bf5", "1.d4 d5 2.Nc3 Nf6 3.Bg5 Bf5 *"),
("D01i", "Richter-Veresov: 3...Bf5 4.f3", "1.d4 d5 2.Nc3 Nf6 3.Bg5 Bf5 4.f3 *"),
("D01j", "Richter-Veresov: 3...Bf5 4.Nf3", "1.d4 d5 2.Nc3 Nf6 3.Bg5 Bf5 4.Nf3 *"),
("D01k", "Richter-Veresov: 3...Bf5 4.Bxf6", "1.d4 d5 2.Nc3 Nf6 3.Bg5 Bf5 4.Bxf6 *"),
("D01l", "Richter-Veresov: 3...Nbd7", "1.d4 d5 2.Nc3 Nf6 3.Bg5 Nbd7 *"),
("D01m", "Richter-Veresov: 3...Nbd7 4.f3", "1.d4 d5 2.Nc3 Nf6 3.Bg5 Nbd7 4.f3 *"),
("D01n", "Richter-Veresov: 3...Nbd7 4.Nf3", "1.d4 d5 2.Nc3 Nf6 3.Bg5 Nbd7 4.Nf3 *"),
("D01o", "Richter-Veresov: 3...Nbd7 4.Nf3 h6", "1.d4 d5 2.Nc3 Nf6 3.Bg5 Nbd7 4.Nf3 h6 *"),
("D01p", "Richter-Veresov: 3...Nbd7 4.Nf3 g6", "1.d4 d5 2.Nc3 Nf6 3.Bg5 Nbd7 4.Nf3 g6 *"),
("D02a", "Queen's Pawn: 2.Nf3", "1.d4 d5 2.Nf3 *"),
("D02a", "Queen's Pawn: 2.Nf3 g6", "1.d4 d5 2.Nf3 g6 *"),
("D02a", "Queen's Pawn: 2.Nf3 Bg4", "1.d4 d5 2.Nf3 Bg4 *"),
("D02b", "Queen's Pawn: 2.Nf3 c6", "1.d4 d5 2.Nf3 c6 *"),
("D02c", "Queen's Pawn: London", "1.d4 d5 2.Nf3 c6 3.Bf4 *"),
("D02c", "Queen's Pawn: London, Alapin Variation", "1.d4 d5 2.Nf3 c6 3.Bf4 Qb6 *"),
("D02d", "Queen's Pawn: 2.Nf3 Bf5", "1.d4 d5 2.Nf3 Bf5 *"),
("D02d", "Queen's Pawn: 2.Nf3 Bf5 3.e3", "1.d4 d5 2.Nf3 Bf5 3.e3 *"),
("D02d", "Queen's Pawn: 2.Nf3 Bf5 3.e3 c6", "1.d4 d5 2.Nf3 Bf5 3.e3 c6 *"),
("D02d", "Queen's Pawn: 2.Nf3 Bf5 3.Bf4", "1.d4 d5 2.Nf3 Bf5 3.Bf4 *"),
("D02d", "Queen's Pawn: 2.Nf3 Bf5 3.Bf4 c6", "1.d4 d5 2.Nf3 Bf5 3.Bf4 c6 *"),
("D02d", "Queen's Pawn: 2.Nf3 Bf5 3.Bf4 e6", "1.d4 d5 2.Nf3 Bf5 3.Bf4 e6 *"),
("D02e", "Queen's Pawn: 2.Nf3 Nc6", "1.d4 d5 2.Nf3 Nc6 *"),
("D02f", "Queen's Pawn: 2.Nf3 Nc6 3.Bf4", "1.d4 d5 2.Nf3 Nc6 3.Bf4 *"),
("D02g", "Queen's Pawn: 2.Nf3 Nc6 3.g3", "1.d4 d5 2.Nf3 Nc6 3.g3 *"),
("D02g", "Queen's Pawn: 2.Nf3 Nc6 3.g3 Bg4", "1.d4 d5 2.Nf3 Nc6 3.g3 Bg4 *"),
("D02h", "Queen's Pawn: 2.Nf3 e6", "1.d4 d5 2.Nf3 e6 *"),
("D02i", "Queen's Pawn: 2.Nf3 e6 3.g3", "1.d4 d5 2.Nf3 e6 3.g3 *"),
("D02i", "Queen's Pawn: 2.Nf3 e6 3.g3 c5", "1.d4 d5 2.Nf3 e6 3.g3 c5 *"),
("D02i", "Queen's Pawn: 2.Nf3 e6 3.g3 c5", "1.d4 d5 2.Nf3 e6 3.g3 c5 4.Bg2 *"),
("D02i", "Queen's Pawn: 2.Nf3 e6 3.g3 c5", "1.d4 d5 2.Nf3 e6 3.g3 c5 4.Bg2 cxd4 5.Nxd4 *"),
("D02i", "Queen's Pawn: 2.Nf3 e6 3.g3 c5", "1.d4 d5 2.Nf3 e6 3.g3 c5 4.Bg2 cxd4 5.Nxd4 e5 *"),
("D02j", "Queen's Pawn: Krause Variation", "1.d4 d5 2.Nf3 c5 *"),
("D02j", "Queen's Pawn: Krause, 3.c4", "1.d4 d5 2.Nf3 c5 3.c4 *"),
("D02j", "Queen's Pawn: Krause, Reversed Slav", "1.d4 d5 2.Nf3 c5 3.c3 *"),
("D02j", "Queen's Pawn: Krause, Reversed QGD", "1.d4 d5 2.Nf3 c5 3.e3 *"),
("D02j", "Queen's Pawn: Krause, Reversed QGA", "1.d4 d5 2.Nf3 c5 3.dxc5 *"),
("D02k", "Queen's Pawn: 2.Nf3 Nf6", "1.d4 d5 2.Nf3 Nf6 *"),
("D02k", "Queen's Pawn: 3.c3", "1.d4 d5 2.Nf3 Nf6 3.c3 *"),
("D02l", "Queen's Pawn: London", "1.d4 d5 2.Nf3 Nf6 3.Bf4 *"),
("D02m", "Queen's Pawn: London", "1.d4 d5 2.Nf3 Nf6 3.Bf4 Bf5 *"),
("D02n", "Queen's Pawn: London", "1.d4 d5 2.Nf3 Nf6 3.Bf4 e6 *"),
("D02o", "Queen's Pawn: London", "1.d4 d5 2.Nf3 Nf6 3.Bf4 c5 *"),
("D02p", "Queen's Pawn: London", "1.d4 d5 2.Nf3 Nf6 3.Bf4 c5 4.e3 e6 *"),
("D02p", "Queen's Pawn: London", "1.d4 d5 2.Nf3 Nf6 3.Bf4 c5 4.e3 e6 5.c3 Nc6 *"),
("D02q", "Queen's Pawn: London", "1.d4 d5 2.Nf3 Nf6 3.Bf4 e6 4.e3 c5 5.c3 Nc6 6.Bd3 Bd6 *"),
("D02r", "Queen's Pawn: 3.g3", "1.d4 d5 2.Nf3 Nf6 3.g3 *"),
("D02s", "Queen's Pawn: 3.g3 c6", "1.d4 d5 2.Nf3 Nf6 3.g3 c6 *"),
("D02s", "Queen's Pawn: 3.g3 c6", "1.d4 d5 2.Nf3 Nf6 3.g3 c6 4.Bg2 *"),
("D02t", "Queen's Pawn: 3.g3 c6 4.Bg2 Bg4", "1.d4 d5 2.Nf3 Nf6 3.g3 c6 4.Bg2 Bg4 *"),
("D02t", "Queen's Pawn: 3.g3 c6 4.Bg2 Bg4", "1.d4 d5 2.Nf3 Nf6 3.g3 c6 4.Bg2 Bg4 5.O-O *"),
("D02t", "Queen's Pawn: 3.g3 c6 4.Bg2 Bg4", "1.d4 d5 2.Nf3 Nf6 3.g3 c6 4.Bg2 Bg4 5.O-O Nbd7 *"),
("D02t", "Queen's Pawn: 3.g3 c6 4.Bg2 Bg4", "1.d4 d5 2.Nf3 Nf6 3.g3 c6 4.Bg2 Bg4 5.O-O Nbd7 6.Nbd2 e6 *"),
("D02u", "Queen's Pawn: 3.g3 g6", "1.d4 d5 2.Nf3 Nf6 3.g3 g6 *"),
("D02u", "Queen's Pawn: 3.g3 g6", "1.d4 d5 2.Nf3 Nf6 3.g3 g6 4.Bg2 Bg7 *"),
("D02u", "Queen's Pawn: 3.g3 g6", "1.d4 d5 2.Nf3 Nf6 3.g3 g6 4.Bg2 Bg7 5.O-O O-O *"),
("D03a", "Torre Attack (Tartakower)", "1.d4 d5 2.Nf3 Nf6 3.Bg5 *"),
("D03b", "Torre Attack: 3...Ne4", "1.d4 d5 2.Nf3 Nf6 3.Bg5 Ne4 *"),
("D03c", "Torre Attack: 3...Ne4 4.Bf4", "1.d4 d5 2.Nf3 Nf6 3.Bg5 Ne4 4.Bf4 *"),
("D03d", "Torre Attack: 3...e6", "1.d4 d5 2.Nf3 Nf6 3.Bg5 e6 *"),
("D03e", "Torre Attack: 3...e6 4.e3", "1.d4 d5 2.Nf3 Nf6 3.Bg5 e6 4.e3 *"),
("D03f", "Torre Attack: 3...e6 4.e3 Nbd7", "1.d4 d5 2.Nf3 Nf6 3.Bg5 e6 4.e3 Nbd7 *"),
("D03f", "Torre Attack: 3...e6 4.e3 Nbd7", "1.d4 d5 2.Nf3 Nf6 3.Bg5 e6 4.e3 Nbd7 5.Nbd2 Be7 *"),
("D03g", "Torre Attack: 3...e6 4.e3 c5", "1.d4 d5 2.Nf3 Nf6 3.Bg5 e6 4.e3 c5 *"),
("D03h", "Torre Attack: 3...e6 4.e3 c5", "1.d4 d5 2.Nf3 Nf6 3.Bg5 e6 4.e3 c5 5.Nbd2 Nbd7 *"),
("D03i", "Torre Attack: 3...e6 4.e3 c5", "1.d4 d5 2.Nf3 Nf6 3.Bg5 e6 4.e3 c5 5.Nbd2 Be7 *"),
("D03j", "Torre Attack: 3...e6 4.e3 c5", "1.d4 d5 2.Nf3 Nf6 3.Bg5 e6 4.e3 c5 5.Nbd2 Be7 6.c3 Nbd7 *"),
("D03k", "Torre Attack: 3...g6", "1.d4 d5 2.Nf3 Nf6 3.Bg5 g6 *"),
("D03k", "Torre Attack: 3...g6", "1.d4 d5 2.Nf3 Nf6 3.Bg5 g6 4.c3 Bg7 *"),
("D03l", "Torre Attack: 3...g6", "1.d4 d5 2.Nf3 Nf6 3.Bg5 g6 4.Nbd2 *"),
("D03l", "Torre Attack: 3...g6", "1.d4 d5 2.Nf3 Nf6 3.Bg5 g6 4.Nbd2 Bg7 *"),
("D03m", "Torre Attack: 3...g6 4.e3", "1.d4 d5 2.Nf3 Nf6 3.Bg5 g6 4.e3 *"),
("D03m", "Torre Attack: 3...g6 4.e3", "1.d4 d5 2.Nf3 Nf6 3.Bg5 g6 4.e3 Bg7 *"),
("D03n", "Torre Attack: 3...g6 4.e3 Bg7 5.Nbd2", "1.d4 d5 2.Nf3 Nf6 3.Bg5 g6 4.e3 Bg7 5.Nbd2 *"),
("D03o", "Torre Attack: 3...g6 4.e3 Bg7 5.Nbd2 O-O", "1.d4 d5 2.Nf3 Nf6 3.Bg5 g6 4.e3 Bg7 5.Nbd2 O-O *"),
("D03p", "Torre Attack: 3...g6 4.e3 Bg7 5.Nbd2 O-O 6.c3", "1.d4 d5 2.Nf3 Nf6 3.Bg5 g6 4.Nbd2 Bg7 5.e3 O-O 6.c3 *"),
("D03q", "Torre Attack: 3...g6 4.e3 Bg7 5.Nbd2 O-O 6.c3 Nbd7", "1.d4 d5 2.Nf3 Nf6 3.Bg5 g6 4.Nbd2 Bg7 5.e3 O-O 6.c3 Nbd7 *"),
("D03r", "Torre Attack: 3...g6 4.e3 Bg7 5.Nbd2 O-O 6.c3 Nbd7 7.Be2", "1.d4 d5 2.Nf3 Nf6 3.Bg5 g6 4.Nbd2 Bg7 5.e3 O-O 6.c3 Nbd7 7.Be2 *"),
("D03s", "Torre Attack: 3...g6 4.e3 Bg7 5.Nbd2 O-O 6.Bd3", "1.d4 d5 2.Nf3 Nf6 3.Bg5 g6 4.e3 Bg7 5.Nbd2 O-O 6.Bd3 *"),
("D03t", "Torre Attack: 3...g6 4.e3 Bg7 5.Nbd2 O-O 6.Bd3 c5", "1.d4 d5 2.Nf3 Nf6 3.Bg5 g6 4.Nbd2 Bg7 5.e3 O-O 6.Bd3 c5 *"),
("D03t", "Torre Attack: 3...g6 4.e3 Bg7 5.Nbd2 O-O 6.Bd3 c5", "1.d4 d5 2.Nf3 Nf6 3.Bg5 g6 4.Nbd2 Bg7 5.e3 O-O 6.Bd3 c5 7.c3 *"),
("D03u", "Torre Attack: 3...g6 4.e3 Bg7 5.Nbd2 O-O 6.Bd3 c5 7.c3 Nbd7", "1.d4 d5 2.Nf3 Nf6 3.Bg5 g6 4.Nbd2 Bg7 5.e3 O-O 6.Bd3 c5 7.c3 Nbd7 *"),
("D04a", "Queen's Pawn: Colle", "1.d4 d5 2.Nf3 Nf6 3.e3 *"),
("D04b", "Colle: 3...Bg4", "1.d4 d5 2.Nf3 Nf6 3.e3 Bg4 *"),
("D04c", "Colle: 3...Bf5", "1.d4 d5 2.Nf3 Nf6 3.e3 Bf5 *"),
("D04d", "Colle: 3...Bf5, Alekhine Variation", "1.d4 d5 2.Nf3 Nf6 3.e3 Bf5 4.Bd3 e6 *"),
("D04e", "Colle: 3...g6", "1.d4 d5 2.Nf3 Nf6 3.e3 g6 *"),
("D04f", "Colle: 3...c6", "1.d4 d5 2.Nf3 Nf6 3.e3 c6 *"),
("D04g", "Colle: 3...c5", "1.d4 d5 2.Nf3 Nf6 3.e3 c5 *"),
("D04h", "Colle: 3...c5 4.c3", "1.d4 d5 2.Nf3 Nf6 3.e3 c5 4.c3 *"),
("D04i", "Colle: 3...c5 4.c3 Nc6", "1.d4 d5 2.Nf3 Nf6 3.e3 c5 4.c3 Nc6 *"),
("D04j", "Colle: 3...c5 4.c3 Nbd7", "1.d4 d5 2.Nf3 Nf6 3.e3 c5 4.c3 Nbd7 *"),
("D05a", "Colle: 3...e6", "1.d4 d5 2.Nf3 Nf6 3.e3 e6 *"),
("D05b", "Colle: 3...e6 4.Nbd2", "1.d4 d5 2.Nf3 Nf6 3.e3 e6 4.Nbd2 *"),
("D05c", "Colle: Zukertort Variation", "1.d4 d5 2.Nf3 Nf6 3.e3 e6 4.Nbd2 c5 5.b3 *"),
("D05d", "Colle: 3...e6 4.Bd3", "1.d4 d5 2.Nf3 Nf6 3.e3 e6 4.Bd3 *"),
("D05e", "Colle: 3...e6 4.Bd3 c5", "1.d4 d5 2.Nf3 Nf6 3.e3 e6 4.Bd3 c5 *"),
("D05f", "Colle: Rubinstein's Attack", "1.d4 d5 2.Nf3 Nf6 3.e3 e6 4.Bd3 c5 5.b3 *"),
("D05g", "Colle: Rubinstein's Attack, 5...Nc6", "1.d4 d5 2.Nf3 Nf6 3.e3 e6 4.Bd3 c5 5.b3 Nc6 *"),
("D05h", "Colle: 5.c3", "1.d4 d5 2.Nf3 Nf6 3.e3 e6 4.Bd3 c5 5.c3 *"),
("D05i", "Colle: 5.c3 Nbd7", "1.d4 d5 2.Nf3 Nf6 3.e3 e6 4.Bd3 c5 5.c3 Nbd7 *"),
("D05j", "Colle: 5.c3 Nc6", "1.d4 d5 2.Nf3 Nf6 3.e3 e6 4.Bd3 c5 5.c3 Nc6 *"),
("D05k", "Colle: 5.c3 Nc6 6.Nbd2", "1.d4 d5 2.Nf3 Nf6 3.e3 e6 4.Bd3 c5 5.c3 Nc6 6.Nbd2 *"),
("D05l", "Colle: 5.c3 Nc6 6.Nbd2 Bd6", "1.d4 d5 2.Nf3 Nf6 3.e3 e6 4.Bd3 c5 5.c3 Nc6 6.Nbd2 Bd6 *"),
("D05m", "Colle: 5.c3 Nc6 6.Nbd2 Bd6 7.O-O O-O", "1.d4 d5 2.Nf3 Nf6 3.e3 e6 4.Bd3 c5 5.c3 Nc6 6.Nbd2 Bd6 7.O-O O-O *"),
("D05n", "Colle: 5.c3 Nc6 6.Nbd2 Bd6 7.O-O O-O 8.dxc5 Bxc5", "1.d4 d5 2.Nf3 Nf6 3.e3 e6 4.Bd3 c5 5.c3 Nc6 6.Nbd2 Bd6 7.O-O O-O 8.dxc5 Bxc5 *"),
("D06a", "Queen's Gambit", "1.d4 d5 2.c4 *"),
("D06b", "QGD: Austrian Defence", "1.d4 d5 2.c4 c5 *"),
("D06b", "QGD: Austrian, Rubinstein Variation", "1.d4 d5 2.c4 c5 3.cxd5 Nf6 *"),
("D06b", "QGD: Austrian, Rubinstein, 4.dxc5", "1.d4 d5 2.c4 c5 3.cxd5 Nf6 4.dxc5 *"),
("D06c", "QGD: Austrian, Rubinstein, 4.Nf3", "1.d4 d5 2.c4 c5 3.cxd5 Nf6 4.Nf3 *"),
("D06d", "QGD: Austrian, Rubinstein, 4.Nf3 cxd4", "1.d4 d5 2.c4 c5 3.cxd5 Nf6 4.Nf3 cxd4 *"),
("D06e", "QGD: Marshall Defence", "1.d4 d5 2.c4 Nf6 *"),
("D06f", "QGD: Marshall Defence, 3.Nc3", "1.d4 d5 2.c4 Nf6 3.Nc3 *"),
("D06g", "QGD: Marshall Defence, 3.Nf3", "1.d4 d5 2.c4 Nf6 3.Nf3 *"),
("D06h", "QGD: Marshall Defence, 3.cxd5", "1.d4 d5 2.c4 Nf6 3.cxd5 *"),
("D06i", "QGD: Marshall Defence, 3.cxd5 Nxd5", "1.d4 d5 2.c4 Nf6 3.cxd5 Nxd5 *"),
("D06j", "QGD: Marshall Defence, 3.cxd5 Nxd5 4.Nf3", "1.d4 d5 2.c4 Nf6 3.cxd5 Nxd5 4.Nf3 *"),
("D06k", "QGD: Marshall Defence, 3.cxd5 Nxd5 4.e4", "1.d4 d5 2.c4 Nf6 3.cxd5 Nxd5 4.e4 *"),
("D06l", "QGD: 2...Bf5", "1.d4 d5 2.c4 Bf5 *"),
("D06l", "QGD: 2...Bf5 3.Qb3", "1.d4 d5 2.c4 Bf5 3.Qb3 *"),
("D06m", "QGD: 2...Bf5 3.Nc3", "1.d4 d5 2.c4 Bf5 3.Nc3 *"),
("D06m", "QGD: 2...Bf5 3.Nc3 e6", "1.d4 d5 2.c4 Bf5 3.Nc3 e6 *"),
("D06n", "QGD: 2...Bf5 3.Nf3", "1.d4 d5 2.c4 Bf5 3.Nf3 *"),
("D06n", "QGD: 2...Bf5 3.Nf3 e6", "1.d4 d5 2.c4 Bf5 3.Nf3 e6 *"),
("D06o", "QGD: 2...Bf5 3.Nf3 e6 4.Qb3", "1.d4 d5 2.c4 Bf5 3.Nf3 e6 4.Qb3 *"),
("D06p", "QGD: 2...Bf5 3.Nf3 e6 4.Nc3", "1.d4 d5 2.c4 Bf5 3.Nf3 e6 4.Nc3 *"),
("D06q", "QGD: 2...Bf5 3.Nf3 e6 4.Nc3 Nf6", "1.d4 d5 2.c4 Bf5 3.Nf3 e6 4.Nc3 Nf6 *"),
("D06r", "QGD: 2...Bf5 3.cxd5", "1.d4 d5 2.c4 Bf5 3.cxd5 *"),
("D06s", "QGD: 2...Bf5 3.cxd5 Bxb1 4.Rxb1", "1.d4 d5 2.c4 Bf5 3.cxd5 Bxb1 4.Rxb1 *"),
("D06t", "QGD: 2...Bf5 3.cxd5 Bxb1 4.Qa4+", "1.d4 d5 2.c4 Bf5 3.cxd5 Bxb1 4.Qa4+ *"),
("D06u", "QGD: 2...Bf5 3.cxd5 Bxb1 4.Qa4+ c6 5.Rxb1", "1.d4 d5 2.c4 Bf5 3.cxd5 Bxb1 4.Qa4+ c6 5.Rxb1 *"),
("D07a", "QGD: Chigorin Defence", "1.d4 d5 2.c4 Nc6 *"),
("D07b", "QGD: Chigorin, 3.e3", "1.d4 d5 2.c4 Nc6 3.e3 *"),
("D07c", "QGD: Chigorin, 3.cxd5", "1.d4 d5 2.c4 Nc6 3.cxd5 *"),
("D07c", "QGD: Chigorin, 3.cxd5, Main Line", "1.d4 d5 2.c4 Nc6 3.cxd5 Qxd5 4.e3 e5 5.Nc3 Bb4 6.Bd2 Bxc3 *"),
("D07d", "QGD: Chigorin, 3.cxd5, Main Line, 7.bxc3", "1.d4 d5 2.c4 Nc6 3.cxd5 Qxd5 4.e3 e5 5.Nc3 Bb4 6.Bd2 Bxc3 7.bxc3 *"),
("D07e", "QGD: Chigorin, 3.cxd5 Main Line, 7.Bxc3", "1.d4 d5 2.c4 Nc6 3.cxd5 Qxd5 4.e3 e5 5.Nc3 Bb4 6.Bd2 Bxc3 7.Bxc3 *"),
("D07f", "QGD: Chigorin, 3.Nf3", "1.d4 d5 2.c4 Nc6 3.Nf3 *"),
("D07g", "QGD: Chigorin, Lazard Gambit", "1.d4 d5 2.c4 Nc6 3.Nf3 e5 *"),
("D07h", "QGD: Chigorin, 3.Nf3 Bg4", "1.d4 d5 2.c4 Nc6 3.Nf3 Bg4 *"),
("D07i", "QGD: Chigorin, 3.Nf3 Bg4 4.e3", "1.d4 d5 2.c4 Nc6 3.Nf3 Bg4 4.e3 *"),
("D07j", "QGD: Chigorin, 3.Nf3 Bg4 4.Nc3", "1.d4 d5 2.c4 Nc6 3.Nf3 Bg4 4.Nc3 *"),
("D07k", "QGD: Chigorin, 3.Nf3 Bg4 4.cxd5", "1.d4 d5 2.c4 Nc6 3.Nf3 Bg4 4.cxd5 *"),
("D07l", "QGD: Chigorin, 3.Nf3 Bg4 4.cxd5 Bxf3 5.dxc6", "1.d4 d5 2.c4 Nc6 3.Nf3 Bg4 4.cxd5 Bxf3 5.dxc6 *"),
("D07m", "QGD: Chigorin, 3.Nf3 Bg4 4.cxd5 Bxf3 5.gxf3", "1.d4 d5 2.c4 Nc6 3.Nf3 Bg4 4.cxd5 Bxf3 5.gxf3 *"),
("D07n", "QGD: Chigorin, 3.Nc3", "1.d4 d5 2.c4 Nc6 3.Nc3 *"),
("D07o", "QGD: Chigorin, Tartakower Gambit", "1.d4 d5 2.c4 Nc6 3.Nc3 e5 *"),
("D07p", "QGD: Chigorin, 3.Nc3 Nf6", "1.d4 d5 2.c4 Nc6 3.Nc3 Nf6 *"),
("D07q", "QGD: Chigorin, 3.Nc3 Nf6 4.Nf3", "1.d4 d5 2.c4 Nc6 3.Nc3 Nf6 4.Nf3 *"),
("D07r", "QGD: Chigorin, 3.Nc3 Nf6 4.Nf3 Bg4", "1.d4 d5 2.c4 Nc6 3.Nc3 Nf6 4.Nf3 Bg4 *"),
("D07s", "QGD: Chigorin, 3.Nc3 dxc4", "1.d4 d5 2.c4 Nc6 3.Nc3 dxc4 *"),
("D07t", "QGD: Chigorin, 3.Nc3 dxc4 4.d5", "1.d4 d5 2.c4 Nc6 3.Nc3 dxc4 4.d5 *"),
("D07u", "QGD: Chigorin, 3.Nc3 dxc4 4.Nf3", "1.d4 d5 2.c4 Nc6 3.Nc3 dxc4 4.Nf3 *"),
("D07v", "QGD: Chigorin, 3.Nc3 dxc4 4.Nf3 Nf6", "1.d4 d5 2.c4 Nc6 3.Nc3 dxc4 4.Nf3 Nf6 *"),
("D07w", "QGD: Chigorin, 3.Nc3 dxc4 4.Nf3 Nf6 5.e4", "1.d4 d5 2.c4 Nc6 3.Nc3 dxc4 4.Nf3 Nf6 5.e4 *"),
("D08", "QGD: Albin Countergambit", "1.d4 d5 2.c4 e5 *"),
("D08", "QGD: Albin, 3.e3", "1.d4 d5 2.c4 e5 3.e3 *"),
("D08", "QGD: Albin, 3.dxe5", "1.d4 d5 2.c4 e5 3.dxe5 *"),
("D08", "QGD: Albin, 3.dxe5 d4", "1.d4 d5 2.c4 e5 3.dxe5 d4 *"),
("D08", "QGD: Albin, Lasker Trap", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.e3 Bb4+ 5.Bd2 dxe3 *"),
("D08", "QGD: Albin, 4.e4", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.e4 *"),
("D08", "QGD: Albin, 4.a3", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.a3 *"),
("D08", "QGD: Albin, 4.Nf3", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 *"),
("D08", "QGD: Albin, 4.Nf3 Nc6", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 *"),
("D08", "QGD: Albin, Alapin Variation", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 5.Nbd2 *"),
("D08", "QGD: Albin, Janowski Variation", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 5.Nbd2 f6 *"),
("D08", "QGD: Albin, Balogh Variation", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 5.Nbd2 Qe7 *"),
("D08", "QGD: Albin, Alapin, 5...Bg4", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 5.Nbd2 Bg4 *"),
("D08", "QGD: Albin, Krenosz Variation", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 5.Nbd2 Bg4 6.h3 Bxf3 7.Nxf3 Bb4+ 8.Bd2 Qe7 *"),
("D08", "QGD: Albin, 4.Nf3 Nc6 5.a3", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 5.a3 *"),
("D08", "QGD: Albin, 4.Nf3 Nc6 5.a3 a5", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 5.a3 a5 *"),
("D08", "QGD: Albin, 4.Nf3 Nc6 5.a3 Be6", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 5.a3 Be6 *"),
("D08", "QGD: Albin, 4.Nf3 Nc6 5.a3 Bg4", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 5.a3 Bg4 *"),
("D08", "QGD: Albin, 4.Nf3 Nc6 5.a3 Bg4 6.Nbd2", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 5.a3 Bg4 6.Nbd2 *"),
("D09", "QGD: Albin, 5.g3", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 5.g3 *"),
("D09", "QGD: Albin, 5.g3 Nge7", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 5.g3 Nge7 *"),
("D09", "QGD: Albin, 5.g3 Bf5", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 5.g3 Bf5 *"),
("D09", "QGD: Albin, 5.g3 Bg4", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 5.g3 Bg4 *"),
("D09", "QGD: Albin, 5.g3 Bg4 6.Bg2 Qd7", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 5.g3 Bg4 6.Bg2 Qd7 *"),
("D09", "QGD: Albin, 5.g3 Bg4 6.Bg2 Qd7 7.O-O O-O-O", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 5.g3 Bg4 6.Bg2 Qd7 7.O-O O-O-O *"),
("D09", "QGD: Albin, 5.g3 Be6", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 5.g3 Be6 *"),
("D09", "QGD: Albin, 5.g3 Be6 6.b3", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 5.g3 Be6 6.b3 *"),
("D09", "QGD: Albin, 5.g3 Be6 6.Bg2", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 5.g3 Be6 6.Bg2 *"),
("D09", "QGD: Albin, 5.g3 Be6 6.Nbd2", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 5.g3 Be6 6.Nbd2 *"),
("D09", "QGD: Albin, 5.g3 Be6 6.Nbd2 Qd7", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 5.g3 Be6 6.Nbd2 Qd7 *"),
("D09", "QGD: Albin, 5.g3 Be6 6.Nbd2 Qd7 7.Bg2", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 5.g3 Be6 6.Nbd2 Qd7 7.Bg2 *"),
("D09", "QGD: Albin, 5.g3 Be6 6.Nbd2 Qd7 7.Bg2 O-O-O", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 5.g3 Be6 6.Nbd2 Qd7 7.Bg2 O-O-O *"),
("D09", "QGD: Albin, 5.g3 Be6 6.Nbd2 Qd7 7.Bg2 Nge7", "1.d4 d5 2.c4 e5 3.dxe5 d4 4.Nf3 Nc6 5.g3 Be6 6.Nbd2 Qd7 7.Bg2 Nge7 *"),
("D10a", "Slav Defence", "1.d4 d5 2.c4 c6 *"),
("D10a", "Diemer-Duhm Gambit (DDG) vs. Slav/Caro-Kann", "1.d4 d5 2.c4 c6 3.e4 *"),
("D10a", "Slav: 3.g3", "1.d4 d5 2.c4 c6 3.g3 *"),
("D10a", "Slav: 3.Bf4", "1.d4 d5 2.c4 c6 3.Bf4 *"),
("D10b", "Slav: Exchange", "1.d4 d5 2.c4 c6 3.cxd5 *"),
("D10b", "Slav: Exchange", "1.d4 d5 2.c4 c6 3.cxd5 cxd5 *"),
("D10c", "Slav: Exchange, 4.Bf4", "1.d4 d5 2.c4 c6 3.cxd5 cxd5 4.Bf4 *"),
("D10d", "Slav: Exchange, 4.Nf3", "1.d4 d5 2.c4 c6 3.cxd5 cxd5 4.Nf3 *"),
("D10e", "Slav: Exchange, 4.Nc3", "1.d4 d5 2.c4 c6 3.cxd5 cxd5 4.Nc3 *"),
("D10e", "Slav: Exchange, 4.Nc3 Nf6", "1.d4 d5 2.c4 c6 3.cxd5 cxd5 4.Nc3 Nf6 *"),
("D10e", "Slav: Exchange, 4.Nc3 Nf6 5.Bf4", "1.d4 d5 2.c4 c6 3.cxd5 cxd5 4.Nc3 Nf6 5.Bf4 *"),
("D10f", "Slav: Exchange, 4.Nc3 Nf6 5.Bf4 Nc6", "1.d4 d5 2.c4 c6 3.cxd5 cxd5 4.Nc3 Nf6 5.Bf4 Nc6 *"),
("D10f", "Slav: Exchange, 4.Nc3 Nf6 5.Bf4 Nc6 6.e3", "1.d4 d5 2.c4 c6 3.cxd5 cxd5 4.Nc3 Nf6 5.Bf4 Nc6 6.e3 *"),
("D10g", "Slav: Exchange, 4.Nc3 Nf6 5.Bf4 Nc6 6.e3 a6", "1.d4 d5 2.c4 c6 3.cxd5 cxd5 4.Nc3 Nf6 5.Bf4 Nc6 6.e3 a6 *"),
("D10h", "Slav: 3.e3", "1.d4 d5 2.c4 c6 3.e3 *"),
("D10h", "Slav: 3.e3 Bf5", "1.d4 d5 2.c4 c6 3.e3 Bf5 *"),
("D10h", "Slav: 3.e3 Nf6", "1.d4 d5 2.c4 c6 3.e3 Nf6 *"),
("D10i", "Slav: 3.Nc3", "1.d4 d5 2.c4 c6 3.Nc3 *"),
("D10j", "Slav: Winawer Countergambit", "1.d4 d5 2.c4 c6 3.Nc3 e5 *"),
("D10k", "Slav: Winawer Countergambit, 4.cxd5", "1.d4 d5 2.c4 c6 3.Nc3 e5 4.cxd5 *"),
("D10k", "Slav: Winawer Countergambit, 4.cxd5 cxd5 5.dxe5", "1.d4 d5 2.c4 c6 3.Nc3 e5 4.cxd5 cxd5 5.dxe5 *"),
("D10l", "Slav: Winawer Countergambit, 4.cxd5 cxd5 5.Nf3", "1.d4 d5 2.c4 c6 3.Nc3 e5 4.cxd5 cxd5 5.Nf3 *"),
("D10m", "Slav: Winawer Countergambit Accepted", "1.d4 d5 2.c4 c6 3.Nc3 e5 4.dxe5 *"),
("D10m", "Slav: Winawer Countergambit Accepted, 6.Nd2", "1.d4 d5 2.c4 c6 3.Nc3 e5 4.dxe5 d4 5.Ne4 Qa5+ 6.Nd2 *"),
("D10n", "Slav: Winawer Countergambit Accepted, 6.Bd2", "1.d4 d5 2.c4 c6 3.Nc3 e5 4.dxe5 d4 5.Ne4 Qa5+ 6.Bd2 *"),
("D10o", "Slav: 3.Nc3 dxc4", "1.d4 d5 2.c4 c6 3.Nc3 dxc4 *"),
("D10o", "Slav: 3.Nc3 dxc4 4.a4", "1.d4 d5 2.c4 c6 3.Nc3 dxc4 4.a4 *"),
("D10o", "Slav: 3.Nc3 dxc4 4.e3", "1.d4 d5 2.c4 c6 3.Nc3 dxc4 4.e3 *"),
("D10p", "Slav: 3.Nc3 dxc4 4.e4", "1.d4 d5 2.c4 c6 3.Nc3 dxc4 4.e4 *"),
("D10q", "Slav: 3.Nc3 dxc4 4.e4 b5", "1.d4 d5 2.c4 c6 3.Nc3 dxc4 4.e4 b5 *"),
("D10r", "Slav: 3.Nc3 dxc4 4.e4 b5 5.a4", "1.d4 d5 2.c4 c6 3.Nc3 dxc4 4.e4 b5 5.a4 *"),
("D10r", "Slav: 3.Nc3 Nf6", "1.d4 d5 2.c4 c6 3.Nc3 Nf6 *"),
("D10s", "Slav: 3.Nc3 Nf6 4.Bg5", "1.d4 d5 2.c4 c6 3.Nc3 Nf6 4.Bg5 *"),
("D10t", "Slav: 3.Nc3 Nf6 4.Bg5 dxc4", "1.d4 d5 2.c4 c6 3.Nc3 Nf6 4.Bg5 dxc4 *"),
("D10u", "Slav: 3.Nc3 Nf6 4.e3", "1.d4 d5 2.c4 c6 3.Nc3 Nf6 4.e3 *"),
("D10v", "Slav: 3.Nc3 Nf6 4.e3 Bf5", "1.d4 d5 2.c4 c6 3.Nc3 Nf6 4.e3 Bf5 *"),
("D10w", "Slav: 3.Nc3 Nf6 4.e3 a6", "1.d4 d5 2.c4 c6 3.Nc3 Nf6 4.e3 a6 *"),
("D10w", "Slav: 3.Nc3 Nf6 4.e3 a6 5.Bd3", "1.d4 d5 2.c4 c6 3.Nc3 Nf6 4.e3 a6 5.Bd3 *"),
("D10x", "Slav: 3.Nc3 Nf6 4.e3 a6 5.Qc2", "1.d4 d5 2.c4 c6 3.Nc3 Nf6 4.e3 a6 5.Qc2 *"),
("D10y", "Slav: 3.Nc3 Nf6 4.e3 a6 5.Qc2 b5", "1.d4 d5 2.c4 c6 3.Nc3 Nf6 4.e3 a6 5.Qc2 b5 *"),
("D11a", "Slav: 3.Nf3", "1.d4 d5 2.c4 c6 3.Nf3 *"),
("D11b", "Slav: 3.Nf3 Bg4", "1.d4 d5 2.c4 c6 3.Nf3 Bg4 *"),
("D11c", "Slav: 3.Nf3 Bf5", "1.d4 d5 2.c4 c6 3.Nf3 Bf5 *"),
("D11c", "Slav: 3.Nf3 Bf5 4.Nc3", "1.d4 d5 2.c4 c6 3.Nf3 Bf5 4.Nc3 *"),
("D11c", "Slav: 3.Nf3 Bf5 4.Nc3 e6", "1.d4 d5 2.c4 c6 3.Nf3 Bf5 4.Nc3 e6 *"),
("D11d", "Slav: 3.Nf3 Bf5 4.Nc3 e6 5.Qb3", "1.d4 d5 2.c4 c6 3.Nf3 Bf5 4.Nc3 e6 5.Qb3 *"),
("D11e", "Slav: 3.Nf3 dxc4", "1.d4 d5 2.c4 c6 3.Nf3 dxc4 *"),
("D11e", "Slav: 3.Nf3 dxc4 4.Nc3", "1.d4 d5 2.c4 c6 3.Nf3 dxc4 4.Nc3 *"),
("D11e", "Slav: 3.Nf3 dxc4 4.e3", "1.d4 d5 2.c4 c6 3.Nf3 dxc4 4.e3 *"),
("D11e", "Slav: 3.Nf3 dxc4 4.e3 Be6", "1.d4 d5 2.c4 c6 3.Nf3 dxc4 4.e3 Be6 *"),
("D11f", "Slav: 3.Nf3 dxc4 4.e3 b5", "1.d4 d5 2.c4 c6 3.Nf3 dxc4 4.e3 b5 *"),
("D11f", "Slav: 3.Nf3 dxc4 4.e3 b5 5.a4", "1.d4 d5 2.c4 c6 3.Nf3 dxc4 4.e3 b5 5.a4 *"),
("D11f", "Slav: 3.Nf3 dxc4 4.e3 b5 5.a4 e6", "1.d4 d5 2.c4 c6 3.Nf3 dxc4 4.e3 b5 5.a4 e6 *"),
("D11g", "Slav: 3.Nf3 Nf6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 *"),
("D11h", "Slav: 4.Qb3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Qb3 *"),
("D11i", "Slav: 4.Qc2", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Qc2 *"),
("D11j", "Slav: 4.Qc2 g6 5.Bf4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Qc2 g6 5.Bf4 *"),
("D11k", "Slav: Breyer Variation", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nbd2 *"),
//
// There are a lot of Slav-Reti system lines here, to try to catch
// transpositions with a delayed d4/c4 around move 6 or 7:
//
("D11l", "Slav: Slav-Reti System", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 *"),
("D11m", "Slav: Slav-Reti System", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bg4 *"),
("D11m", "Slav: Slav-Reti System", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bg4 5.Bg2 *"),
("D11m", "Slav: Slav-Reti System", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bg4 5.Bg2 e6 *"),
("D11m", "Slav: Slav-Reti System", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bg4 5.Bg2 e6 6.O-O *"),
("D11m", "Slav: Slav-Reti System", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bg4 5.Bg2 e6 6.O-O Nbd7 *"),
("D11m", "Slav: Slav-Reti System", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bg4 5.Bg2 e6 6.O-O Nbd7 7.Nbd2 *"),
("D11m", "Slav: Slav-Reti System", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bg4 5.Bg2 e6 6.O-O Nbd7 7.Nbd2 Bd6 *"),
("D11m", "Slav: Slav-Reti System", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bg4 5.Bg2 e6 6.O-O Nbd7 7.Nbd2 Be7 *"),
("D11n", "Slav: Slav-Reti with b3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bg4 5.Bg2 e6 6.O-O Nbd7 7.b3 *"),
("D11n", "Slav: Slav-Reti with b3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bg4 5.Bg2 e6 6.O-O Nbd7 7.b3 Bd6 *"),
("D11n", "Slav: Slav-Reti with b3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bg4 5.Bg2 e6 6.O-O Nbd7 7.b3 Bd6 8.Bb2 *"),
("D11n", "Slav: Slav-Reti with b3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bg4 5.Bg2 e6 6.O-O Nbd7 7.b3 Bd6 8.Bb2 O-O 9.Nbd2 *"),
("D11n", "Slav: Slav-Reti with b3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bg4 5.Bg2 e6 6.O-O Nbd7 7.b3 Be7 *"),
("D11n", "Slav: Slav-Reti with b3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bg4 5.Bg2 e6 6.O-O Nbd7 7.b3 Be7 8.Bb2 *"),
("D11n", "Slav: Slav-Reti with b3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bg4 5.Bg2 e6 6.O-O Nbd7 7.b3 Be7 8.Bb2 O-O 9.Nbd2 *"),
("D11o", "Slav: Slav-Reti System", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bf5 *"),
("D11o", "Slav: Slav-Reti System", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bf5 5.Bg2 *"),
("D11o", "Slav: Slav-Reti System", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bf5 5.Bg2 e6 *"),
("D11o", "Slav: Slav-Reti System", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bf5 5.Bg2 e6 6.O-O *"),
("D11o", "Slav: Slav-Reti System", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bf5 5.Bg2 e6 6.O-O Nbd7 *"),
("D11o", "Slav: Slav-Reti System", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bf5 5.Bg2 e6 6.O-O Be7 *"),
("D11p", "Slav: Slav-Reti System", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bf5 5.Bg2 e6 6.O-O h6 *"),
("D11p", "Slav: Slav-Reti System", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bf5 5.Bg2 e6 6.O-O h6 7.Nbd2 *"),
("D11p", "Slav: Slav-Reti System", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bf5 5.Bg2 e6 6.O-O h6 7.Nc3 *"),
("D11p", "Slav: Slav-Reti System", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.g3 Bf5 5.Bg2 e6 6.O-O h6 7.b3 *"),
("D11q", "Slav: 4.e3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 *"),
("D11r", "Slav: 4.e3 g6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 g6 *"),
("D11s", "Slav: 4.e3 a6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 a6 *"),
("D11t", "Slav: 4.e3 Bg4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 Bg4 *"),
("D11t", "Slav: 4.e3 Bg4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 Bg4 5.Nc3 *"),
("D12a", "Slav: 4.e3 Bf5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 Bf5 *"),
("D12b", "Slav: 4.e3 Bf5 5.Qb3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 Bf5 5.Qb3 *"),
("D12c", "Slav: 4.e3 Bf5 5.cxd5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 Bf5 5.cxd5 *"),
("D12c", "Slav: 4.e3 Bf5 5.cxd5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 Bf5 5.cxd5 cxd5 *"),
("D12d", "Slav: 4.e3 Bf5 5.cxd5 cxd5 6.Qb3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 Bf5 5.cxd5 cxd5 6.Qb3 *"),
("D12e", "Slav: 4.e3 Bf5 5.cxd5 cxd5 6.Qb3 Qc7", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 Bf5 5.cxd5 cxd5 6.Qb3 Qc7 *"),
("D12f", "Slav: 4.e3 Bf5 5.cxd5 cxd5 6.Nc3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 Bf5 5.cxd5 cxd5 6.Nc3 *"),
("D12f", "Slav: 4.e3 Bf5 5.cxd5 cxd5 6.Nc3, Amsterdam Variation", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 Bf5 5.cxd5 cxd5 6.Nc3 e6 7.Ne5 Nfd7 *"),
("D12g", "Slav: 4.e3 Bf5 5.Bd3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 Bf5 5.Bd3 *"),
("D12h", "Slav: 4.e3 Bf5 5.Bd3 Bxd3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 Bf5 5.Bd3 Bxd3 *"),
("D12i", "Slav: 4.e3 Bf5 5.Nc3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 Bf5 5.Nc3 *"),
("D12i", "Slav: 4.e3 Bf5 5.Nc3 e6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 Bf5 5.Nc3 e6 *"),
("D12j", "Slav: 4.e3 Bf5 5.Nc3 e6 6.Be2", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 Bf5 5.Nc3 e6 6.Be2 *"),
("D12k", "Slav: 4.e3 Bf5 5.Nc3 e6 6.Bd3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 Bf5 5.Nc3 e6 6.Bd3 *"),
("D12l", "Slav: 4.e3 Bf5 5.Nc3 e6 6.Bd3 Bxd3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 Bf5 5.Nc3 e6 6.Bd3 Bxd3 *"),
("D12l", "Slav: 4.e3 Bf5 5.Nc3 e6 6.Bd3 Bxd3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 Bf5 5.Nc3 e6 6.Bd3 Bxd3 7.Qxd3 *"),
("D12m", "Slav: 4.e3 Bf5 5.Nc3 e6 6.Bd3 Bxd3 7.Qxd3 Nbd7", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 Bf5 5.Nc3 e6 6.Bd3 Bxd3 7.Qxd3 Nbd7 *"),
("D12n", "Slav: 4.e3 Bf5 5.Nc3 e6 6.Nh4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 Bf5 5.Nc3 e6 6.Nh4 *"),
("D12o", "Slav: 4.e3 Bf5 5.Nc3 e6 6.Nh4 Bg4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 Bf5 5.Nc3 e6 6.Nh4 Bg4 *"),
("D12p", "Slav: 4.e3 Bf5 5.Nc3 e6 6.Nh4 Be4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.e3 Bf5 5.Nc3 e6 6.Nh4 Be4 *"),
("D13a", "Slav: Exchange", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 *"),
("D13b", "Slav: Exchange", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 *"),
("D13c", "Slav: Exchange, 5.Nc3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 *"),
("D13d", "Slav: Exchange, 5.Nc3 Bf5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Bf5 *"),
("D13e", "Slav: Exchange, 5.Nc3 a6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 a6 *"),
("D13f", "Slav: Exchange, 5.Nc3 a6 6.Ne5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 a6 6.Ne5 *"),
("D13g", "Slav: Exchange, 5.Nc3 a6 6.Bf4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 a6 6.Bf4 *"),
("D13h", "Slav: Exchange, 5.Nc3 e6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 e6 *"),
("D13i", "Slav: Exchange, 5.Nc3 e6 6.Bf4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 e6 6.Bf4 *"),
("D13j", "Slav: Exchange, 5.Nc3 Nc6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 *"),
("D13k", "Slav: Exchange, 6.Bf4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 *"),
("D13l", "Slav: Exchange, 6.Bf4 a6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 a6 *"),
("D13m", "Slav: Exchange, 6.Bf4 a6 7.e3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 a6 7.e3 *"),
("D13n", "Slav: Exchange, 6.Bf4 a6 7.e3 Bg4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 a6 7.e3 Bg4 *"),
("D13o", "Slav: Exchange, 6.Bf4 a6 7.e3 Bg4 8.Be2", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 a6 7.e3 Bg4 8.Be2 *"),
("D13p", "Slav: Exchange, 6.Bf4 e6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 e6 *"),
("D13q", "Slav: Exchange, 6.Bf4 e6 7.e3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 e6 7.e3 *"),
("D13r", "Slav: Exchange, 6.Bf4 e6 7.e3 Be7", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 e6 7.e3 Be7 *"),
("D13s", "Slav: Exchange, 6.Bf4 e6 7.e3 Bd6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 e6 7.e3 Bd6 *"),
("D13t", "Slav: Exchange, 6.Bf4 e6 7.e3 Bd6 8.Bxd6 Qxd6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 e6 7.e3 Bd6 8.Bxd6 Qxd6 *"),
("D14a", "Slav: Exchange, 6.Bf4 Bf5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 Bf5 *"),
("D14b", "Slav: Exchange, 6.Bf4 Bf5 7.e3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 Bf5 7.e3 *"),
("D14c", "Slav: Exchange, 6.Bf4 Bf5 7.e3 a6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 Bf5 7.e3 a6 *"),
("D14d", "Slav: Exchange, 6.Bf4 Bf5 7.e3 e6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 Bf5 7.e3 e6 *"),
("D14e", "Slav: Exchange, 8.Ne5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 Bf5 7.e3 e6 8.Ne5 *"),
("D14f", "Slav: Exchange, 8.Qb3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 Bf5 7.e3 e6 8.Qb3 *"),
("D14g", "Slav: Exchange, Trifunovic Variation", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 Bf5 7.e3 e6 8.Qb3 Bb4 *"),
("D14h", "Slav: Exchange, 8.Bd3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 Bf5 7.e3 e6 8.Bd3 *"),
("D14i", "Slav: Exchange, 8.Bd3 Bxd3 9.Qxd3 Bd6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 Bf5 7.e3 e6 8.Bd3 Bxd3 9.Qxd3 Bd6 *"),
("D14j", "Slav: Exchange, 8.Bd3 Bxd3 9.Qxd3 Bd6 10.Bxd6 Qxd6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 Bf5 7.e3 e6 8.Bd3 Bxd3 9.Qxd3 Bd6 10.Bxd6 Qxd6 *"),
("D14k", "Slav: Exchange, 8.Bd3 Main Line", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 Bf5 7.e3 e6 8.Bd3 Bxd3 9.Qxd3 Bd6 10.Bxd6 Qxd6 11.O-O O-O *"),
("D14l", "Slav: Exchange, 8.Bd3 Main Line, 12.Rac1", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 Bf5 7.e3 e6 8.Bd3 Bxd3 9.Qxd3 Bd6 10.Bxd6 Qxd6 11.O-O O-O 12.Rac1 *"),
("D14m", "Slav: Exchange, 8.Bd3 Main Line, 12.Rfc1", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 Bf5 7.e3 e6 8.Bd3 Bxd3 9.Qxd3 Bd6 10.Bxd6 Qxd6 11.O-O O-O 12.Rfc1 *"),
("D14n", "Slav: Exchange, 8.Bd3 Main Line, 12.Rfc1 Rfc8", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 Bf5 7.e3 e6 8.Bd3 Bxd3 9.Qxd3 Bd6 10.Bxd6 Qxd6 11.O-O O-O 12.Rfc1 Rfc8 *"),
("D14o", "Slav: Exchange, 8.Bb5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 Bf5 7.e3 e6 8.Bb5 *"),
("D14p", "Slav: Exchange, 8.Bb5 Nd7", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 Bf5 7.e3 e6 8.Bb5 Nd7 *"),
("D14q", "Slav: Exchange, 8.Bb5 Nd7 9.Qa4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.cxd5 cxd5 5.Nc3 Nc6 6.Bf4 Bf5 7.e3 e6 8.Bb5 Nd7 9.Qa4 *"),
("D15a", "Slav: 4.Nc3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 *"),
("D15b", "Slav: 4.Nc3 Bf5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 Bf5 *"),
("D15b", "Slav: 4.Nc3 Bf5 5.Qb3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 Bf5 5.Qb3 *"),
("D15c", "Slav: Suechting Variation", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 Qb6 *"),
("D15d", "Slav: Chameleon Variation", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 a6 *"),
("D15e", "Slav: Chameleon, 5.Bg5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 a6 5.Bg5 *"),
("D15f", "Slav: Chameleon, 5.Ne5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 a6 5.Ne5 *"),
("D15g", "Slav: Chameleon, 5.e3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 a6 5.e3 *"),
("D15h", "Slav: Chameleon, 5.e3 b5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 a6 5.e3 b5 *"),
("D15i", "Slav: Chameleon, 5.e3 b5 6.b3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 a6 5.e3 b5 6.b3 *"),
("D15j", "Slav: Chameleon, 5.c5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 a6 5.c5 *"),
("D15k", "Slav: Chameleon, 5.c5 Nbd7", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 a6 5.c5 Nbd7 *"),
("D15l", "Slav: Chameleon, 5.a4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 a6 5.a4 *"),
("D15m", "Slav: Chameleon, 5.a4 e6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 a6 5.a4 e6 *"),
("D15n", "Slav: Chameleon, 5.a4 e6 6.Bg5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 a6 5.a4 e6 6.Bg5 *"),
("D15o", "Slav: Accepted", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 *"),
("D15p", "Slav: Accepted, 5.Ne5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.Ne5 *"),
("D15q", "Slav: Accepted, Alekhine", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.e3 *"),
("D15r", "Slav: Accepted, Alekhine: 5...b5 6.a4 b4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.e3 b5 6.a4 b4 *"),
("D15s", "Slav: Accepted, Alekhine: 5...b5 6.a4 b4 7.Nb1", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.e3 b5 6.a4 b4 7.Nb1 *"),
("D15t", "Slav: Geller (Tolush) Gambit", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.e4 *"),
("D15t", "Slav: Geller Gambit", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.e4 b5 *"),
("D15u", "Slav: Geller Gambit, Spassky Variation", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.e4 b5 6.Qc2 *"),
("D15v", "Slav: Geller Gambit, 6.e5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.e4 b5 6.e5 *"),
("D15w", "Slav: Geller Gambit, 6.e5 Nd5 7.Ng5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.e4 b5 6.e5 Nd5 7.Ng5 *"),
("D15x", "Slav: Geller Gambit, 6.e5 Nd5 7.a4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.e4 b5 6.e5 Nd5 7.a4 *"),
("D15y", "Slav: Geller Gambit, 6.e5 Nd5 7.a4 e6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.e4 b5 6.e5 Nd5 7.a4 e6 *"),
("D16a", "Slav: Alapin", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 *"),
("D16b", "Slav: 5.a4 Nd5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Nd5 *"),
("D16c", "Slav: Soultanbeieff Variation", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 e6 *"),
("D16d", "Slav: Murey Variation", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 a5 *"),
("D16e", "Slav: Smyslov Variation", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Na6 *"),
("D16f", "Slav: Smyslov, 6.Ne5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Na6 6.Ne5 *"),
("D16g", "Slav: Smyslov, 6.e3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Na6 6.e3 *"),
("D16h", "Slav: Smyslov, 6.e3 Bg4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Na6 6.e3 Bg4 *"),
("D16i", "Slav: Smyslov, 6.e4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Na6 6.e4 *"),
("D16j", "Slav: Smyslov, 6.e4 Bg4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Na6 6.e4 Bg4 *"),
("D16k", "Slav: Smyslov, 6.e4 Bg4 7.Bxc4 e6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Na6 6.e4 Bg4 7.Bxc4 e6 *"),
("D16l", "Slav: Bronstein (Steiner) Variation", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bg4 *"),
("D16m", "Slav: Bronstein, 6.Ne5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bg4 6.Ne5 *"),
("D16n", "Slav: Bronstein, 6.Ne5 Bh5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bg4 6.Ne5 Bh5 *"),
("D16o", "Slav: Bronstein, 6.Ne5 Bh5 7.h3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bg4 6.Ne5 Bh5 7.h3 *"),
("D16p", "Slav: Bronstein, 6.Ne5 Bh5 7.g3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bg4 6.Ne5 Bh5 7.g3 *"),
("D16q", "Slav: Bronstein, 6.Ne5 Bh5 7.f3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bg4 6.Ne5 Bh5 7.f3 *"),
("D16r", "Slav: Bronstein, 6.Ne5 Bh5 7.f3 Nfd7", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bg4 6.Ne5 Bh5 7.f3 Nfd7 *"),
("D16s", "Slav: Bronstein, 6.Ne5 Bh5 7.f3 Nfd7 8.Nxc4 e5 9.Ne4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bg4 6.Ne5 Bh5 7.f3 Nfd7 8.Nxc4 e5 9.Ne4 *"),
("D17a", "Slav: Czech Defence", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 *"),
("D17b", "Slav: Czech, 6.Nh4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.Nh4 *"),
("D17c", "Slav: Czech, 6.Nh4 e6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.Nh4 e6 *"),
("D17d", "Slav: Czech, 6.Nh4 e6 7.Nxf5 exf5 8.e3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.Nh4 e6 7.Nxf5 exf5 8.e3 *"),
("D17e", "Slav: Czech, 6.Nh4 Bc8", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.Nh4 Bc8 *"),
("D17e", "Slav: Czech, 6.Nh4 Bc8 7.e3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.Nh4 Bc8 7.e3 *"),
("D17f", "Slav: Central Variation", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.Ne5 *"),
("D17g", "Slav: Central, 6.Ne5 Na6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.Ne5 Na6 *"),
("D17g", "Slav: Central, 6.Ne5 Na6 7.f3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.Ne5 Na6 7.f3 *"),
("D17h", "Slav: Central, 6.Ne5 Nbd7", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.Ne5 Nbd7 *"),
("D17i", "Slav: Central, Carlsbad Variation", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.Ne5 Nbd7 7.Nxc4 Qc7 8.g3 e5 *"),
("D17j", "Slav: Central, Carlsbad, Main Line", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.Ne5 Nbd7 7.Nxc4 Qc7 8.g3 e5 9.dxe5 Nxe5 10.Bf4 Nfd7 11.Bg2 f6 *"),
("D17k", "Slav: Central, 6.Ne5 e6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.Ne5 e6 *"),
("D17k", "Slav: Central, 6.Ne5 e6 7.f3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.Ne5 e6 7.f3 *"),
("D17l", "Slav: Central, Huebner (7...c5)", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.Ne5 e6 7.f3 c5 *"),
("D17m", "Slav: Central, 7.f3 Bb4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.Ne5 e6 7.f3 Bb4 *"),
("D17m", "Slav: Central, 7.f3 Bb4 8.Bg5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.Ne5 e6 7.f3 Bb4 8.Bg5 *"),
("D17n", "Slav: Central, 7.f3 Bb4 8.Nxc4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.Ne5 e6 7.f3 Bb4 8.Nxc4 *"),
("D17o", "Slav: Central, 7.f3 Bb4 8.e4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.Ne5 e6 7.f3 Bb4 8.e4 *"),
("D17p", "Slav: Central, Piece-Sac Line", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.Ne5 e6 7.f3 Bb4 8.e4 Bxe4 *"),
("D17q", "Slav: Central, Main Line", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.Ne5 e6 7.f3 Bb4 8.e4 Bxe4 9.fxe4 Nxe4 10.Bd2 Qxd4 11.Nxe4 Qxe4+ 12.Qe2 Bxd2+ 13.Kxd2 *"),
("D17r", "Slav: Central, Main Line, 15.Nxc4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.Ne5 e6 7.f3 Bb4 8.e4 Bxe4 9.fxe4 Nxe4 10.Bd2 Qxd4 11.Nxe4 Qxe4+ 12.Qe2 Bxd2+ 13.Kxd2 Qd5+ 14.Kc2 Na6 15.Nxc4 *"),
("D17s", "Slav: Central, Main Line, 15.Nxc4 O-O", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.Ne5 e6 7.f3 Bb4 8.e4 Bxe4 9.fxe4 Nxe4 10.Bd2 Qxd4 11.Nxe4 Qxe4+ 12.Qe2 Bxd2+ 13.Kxd2 Qd5+ 14.Kc2 Na6 15.Nxc4 O-O *"),
("D17t", "Slav: Central, Main Line, 15.Nxc4 O-O-O", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.Ne5 e6 7.f3 Bb4 8.e4 Bxe4 9.fxe4 Nxe4 10.Bd2 Qxd4 11.Nxe4 Qxe4+ 12.Qe2 Bxd2+ 13.Kxd2 Qd5+ 14.Kc2 Na6 15.Nxc4 O-O-O *"),
("D18a", "Slav: Dutch Variation", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 *"),
("D18b", "Slav: Dutch, Lasker Variation", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 Na6 *"),
("D18c", "Slav: Dutch, 6...e6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 *"),
("D18d", "Slav: Dutch, 6...e6 7.Bxc4 Bb4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 *"),
("D18e", "Slav: Dutch, 8.O-O", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O *"),
("D18f", "Slav: Dutch, 8...Nbd7", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O Nbd7 *"),
("D18g", "Slav: Dutch, 8...Nbd7 9.Nh4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O Nbd7 9.Nh4 *"),
("D18h", "Slav: Dutch, 8...Nbd7 9.Nh4 Bg6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O Nbd7 9.Nh4 Bg6 *"),
("D18i", "Slav: Dutch, 8...Nbd7 9.Qb3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O Nbd7 9.Qb3 *"),
("D18j", "Slav: Dutch, 8...Nbd7 9.Qb3 a5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O Nbd7 9.Qb3 a5 *"),
("D18k", "Slav: Dutch, 8...Nbd7 9.Qe2", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O Nbd7 9.Qe2 *"),
("D18l", "Slav: Dutch, 8...Nbd7 9.Qe2 Bg6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O Nbd7 9.Qe2 Bg6 *"),
("D18m", "Slav: Dutch, 8...Nbd7 9.Qe2 Bg6 10.e4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O Nbd7 9.Qe2 Bg6 10.e4 *"),
("D18n", "Slav: Dutch, 8...O-O", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O *"),
("D18o", "Slav: Dutch, 8...O-O 9.Qb3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Qb3 *"),
("D18p", "Slav: Dutch, 8...O-O 9.Nh4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Nh4 *"),
("D18p", "Slav: Dutch, 8...O-O 9.Nh4 Bg6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Nh4 Bg6 *"),
("D18q", "Slav: Dutch, 8...O-O 9.Nh4 Bg4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Nh4 Bg4 *"),
("D18r", "Slav: Dutch, 8...O-O 9.Nh4 Nbd7", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Nh4 Nbd7 *"),
("D18s", "Slav: Dutch, 8...O-O 9.Nh4 Nbd7 10.Nxf5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Nh4 Nbd7 10.Nxf5 *"),
("D18s", "Slav: Dutch, 8...O-O 9.Nh4 Nbd7 10.Nxf5 exf5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Nh4 Nbd7 10.Nxf5 exf5 *"),
("D18t", "Slav: Dutch, 8...O-O 9.Nh4 Nbd7 10.Nxf5 exf5 11.Qc2", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Nh4 Nbd7 10.Nxf5 exf5 11.Qc2 *"),
("D19a", "Slav: Dutch, 8...O-O 9.Qe2", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Qe2 *"),
("D19b", "Slav: Dutch, 8...O-O 9.Qe2 Bg4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Qe2 Bg4 *"),
("D19c", "Slav: Dutch, 8...O-O 9.Qe2 Bg6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Qe2 Bg6 *"),
("D19d", "Slav: Dutch, 8...O-O 9.Qe2 Bg6 10.Rd1", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Qe2 Bg6 10.Rd1 *"),
("D19d", "Slav: Dutch, 8...O-O 9.Qe2 Bg6 10.Rd1 Nbd7", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Qe2 Bg6 10.Rd1 Nbd7 *"),
("D19e", "Slav: Dutch, 8...O-O 9.Qe2 Bg6 10.Ne5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Qe2 Bg6 10.Ne5 *"),
("D19e", "Slav: Dutch, 8...O-O 9.Qe2 Bg6 10.Ne5 Nbd7, 12.Rd1", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Qe2 Bg6 10.Ne5 Nbd7 *"),
("D19f", "Slav: Dutch, 8...O-O 9.Qe2 Bg6 10.Ne5 Nbd7, 12.Rd1", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Qe2 Bg6 10.Ne5 Nbd7 11.Nxg6 hxg6 12.Rd1 *"),
("D19g", "Slav: Dutch, 8...O-O 9.Qe2 Ne4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Qe2 Ne4 *"),
("D19g", "Slav: Dutch, Saemisch Variation", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Qe2 Ne4 10.g4 *"),
("D19h", "Slav: Dutch, 8...O-O 9.Qe2 Nbd7", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Qe2 Nbd7 *"),
("D19i", "Slav: Dutch, 8...O-O 9.Qe2 Nbd7 10.e4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Qe2 Nbd7 10.e4 *"),
("D19i", "Slav: Dutch, 8...O-O 9.Qe2 Nbd7 10.e4 Bg4", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Qe2 Nbd7 10.e4 Bg4 *"),
("D19j", "Slav: Dutch, 8...O-O 9.Qe2 Nbd7 10.e4 Bg6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Qe2 Nbd7 10.e4 Bg6 *"),
("D19k", "Slav: Dutch, 8...O-O 9.Qe2 Nbd7 10.e4 Bg6 11.Bd3", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Qe2 Nbd7 10.e4 Bg6 11.Bd3 *"),
("D19l", "Slav: Dutch, 8...O-O 9.Qe2 Nbd7 10.e4 Bg6 11.Bd3 h6", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Qe2 Nbd7 10.e4 Bg6 11.Bd3 h6 *"),
("D19m", "Slav: Dutch, 8...O-O 9.Qe2 Nbd7 10.e4 Bg6 11.Bd3 Re8", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Qe2 Nbd7 10.e4 Bg6 11.Bd3 Re8 *"),
("D19n", "Slav: Dutch, 8...O-O 9.Qe2 Nbd7 10.e4 Bg6 11.Bd3 Bh5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Qe2 Nbd7 10.e4 Bg6 11.Bd3 Bh5 *"),
("D19o", "Slav: Dutch, 8...O-O 9.Qe2 Nbd7 10.e4 Bg6 11.Bd3 Bh5 12.e5", "1.d4 d5 2.c4 c6 3.Nf3 Nf6 4.Nc3 dxc4 5.a4 Bf5 6.e3 e6 7.Bxc4 Bb4 8.O-O O-O 9.Qe2 Nbd7 10.e4 Bg6 11.Bd3 Bh5 12.e5 *"),
("D20a", "Queen's Gambit Accepted (QGA)", "1.d4 d5 2.c4 dxc4 *"),
("D20a", "QGA: 3.Qa4+", "1.d4 d5 2.c4 dxc4 3.Qa4+ *"),
("D20b", "QGA: 3.e3", "1.d4 d5 2.c4 dxc4 3.e3 *"),
("D20c", "QGA: 3.e3 c5", "1.d4 d5 2.c4 dxc4 3.e3 c5 *"),
("D20c", "QGA: 3.e3 c5 4.Bxc4", "1.d4 d5 2.c4 dxc4 3.e3 c5 4.Bxc4 *"),
// The following line can arise from a Caro-Kann, Panov-Botvinnik:
// 1.e4 c6 2.d4 d5 3.exd5 cxd5 4.c4 dxc4 5.Bxc4
("D20c", "QGA: 3.e3 c5 4.Bxc4", "1.d4 d5 2.c4 dxc4 3.e3 c5 4.Bxc4 cxd4 5.exd4 *"),
("D20d", "QGA: 3.e3 e6", "1.d4 d5 2.c4 dxc4 3.e3 e6 *"),
("D20d", "QGA: 3.e3 e6", "1.d4 d5 2.c4 dxc4 3.e3 e6 4.Bxc4 Nf6 *"),
("D20e", "QGA: 3.e3 e5", "1.d4 d5 2.c4 dxc4 3.e3 e5 *"),
("D20e", "QGA: 3.e3 e5", "1.d4 d5 2.c4 dxc4 3.e3 e5 4.Bxc4 exd4 5.exd4 *"),
("D20f", "QGA: 3.e3 e5", "1.d4 d5 2.c4 dxc4 3.e3 e5 4.Bxc4 exd4 5.exd4 Bb4+ *"),
("D20f", "QGA: 3.e3 e5", "1.d4 d5 2.c4 dxc4 3.e3 e5 4.Bxc4 exd4 5.exd4 Bb4+ 5.Nc3 Nf6 *"),
("D20g", "QGA: 3.e3 e5", "1.d4 d5 2.c4 dxc4 3.e3 e5 4.Bxc4 exd4 5.exd4 Nf6 *"),
("D20h", "QGA: 3.e3 Nf6", "1.d4 d5 2.c4 dxc4 3.e3 Nf6 *"),
("D20i", "QGA: 3.Nc3", "1.d4 d5 2.c4 dxc4 3.Nc3 *"),
("D20i", "QGA: 3.Nc3 c5", "1.d4 d5 2.c4 dxc4 3.Nc3 c5 *"),
("D20i", "QGA: 3.Nc3 e5", "1.d4 d5 2.c4 dxc4 3.Nc3 e5 *"),
("D20i", "QGA: 3.Nc3 Nf6", "1.d4 d5 2.c4 dxc4 3.Nc3 Nf6 *"),
("D20j", "QGA: 3.Nc3 e6", "1.d4 d5 2.c4 dxc4 3.Nc3 e6 *"),
("D20j", "QGA: 3.Nc3 e6 4.e4", "1.d4 d5 2.c4 dxc4 3.Nc3 e6 4.e4 *"),
("D20k", "QGA: 3.Nc3 a6", "1.d4 d5 2.c4 dxc4 3.Nc3 a6 *"),
("D20k", "QGA: 3.Nc3 a6 4.a4", "1.d4 d5 2.c4 dxc4 3.Nc3 a6 4.a4 *"),
("D20l", "QGA: 3.e4", "1.d4 d5 2.c4 dxc4 3.e4 *"),
("D20l", "QGA: 3.e4, Schwartz Defence", "1.d4 d5 2.c4 dxc4 3.e4 f5 *"),
("D20m", "QGA: 3.e4 Nc6", "1.d4 d5 2.c4 dxc4 3.e4 Nc6 *"),
("D20m", "QGA: 3.e4 Nc6 4.Nf3", "1.d4 d5 2.c4 dxc4 3.e4 Nc6 4.Nf3 *"),
("D20n", "QGA: 3.e4 Nc6 4.Be3", "1.d4 d5 2.c4 dxc4 3.e4 Nc6 4.Be3 *"),
("D20o", "QGA: 3.e4 Nf6", "1.d4 d5 2.c4 dxc4 3.e4 Nf6 *"),
("D20o", "QGA: 3.e4 Nf6 4.Nc3", "1.d4 d5 2.c4 dxc4 3.e4 Nf6 4.Nc3 *"),
("D20o", "QGA: 3.e4 Nf6 4.e5", "1.d4 d5 2.c4 dxc4 3.e4 Nf6 4.e5 *"),
("D20o", "QGA: 3.e4 Nf6 4.e5 Nd5", "1.d4 d5 2.c4 dxc4 3.e4 Nf6 4.e5 Nd5 *"),
("D20p", "QGA: 3.e4 Nf6 4.e5 Nd5 5.Bxc4 Nb6", "1.d4 d5 2.c4 dxc4 3.e4 Nf6 4.e5 Nd5 5.Bxc4 Nb6 *"),
("D20p", "QGA: 3.e4 Nf6 4.e5 Nd5 5.Bxc4 Nb6 6.Bd3", "1.d4 d5 2.c4 dxc4 3.e4 Nf6 4.e5 Nd5 5.Bxc4 Nb6 6.Bd3 *"),
("D20p", "QGA: 3.e4 Nf6 4.e5 Nd5 5.Bxc4 Nb6 6.Bd3", "1.d4 d5 2.c4 dxc4 3.e4 Nf6 4.e5 Nd5 5.Bxc4 Nb6 6.Bd3 Nc6 *"),
("D20q", "QGA: 3.e4 Nf6 4.e5 Nd5 5.Bxc4 Nb6 6.Bb3", "1.d4 d5 2.c4 dxc4 3.e4 Nf6 4.e5 Nd5 5.Bxc4 Nb6 6.Bb3 *"),
("D20q", "QGA: 3.e4 Nf6 4.e5 Nd5 5.Bxc4 Nb6 6.Bb3", "1.d4 d5 2.c4 dxc4 3.e4 Nf6 4.e5 Nd5 5.Bxc4 Nb6 6.Bb3 Nc6 *"),
("D20r", "QGA: 3.e4 Nf6 4.e5 Nd5 5.Bxc4 Nb6 6.Bb3 Nc6 7.Ne2", "1.d4 d5 2.c4 dxc4 3.e4 Nf6 4.e5 Nd5 5.Bxc4 Nb6 6.Bb3 Nc6 7.Ne2 *"),
("D20s", "QGA: 3.e4 c5", "1.d4 d5 2.c4 dxc4 3.e4 c5 *"),
("D20s", "QGA: 3.e4 c5", "1.d4 d5 2.c4 dxc4 3.e4 c5 4.Nf3 *"),
("D20t", "QGA: 3.e4 c5 4.d5", "1.d4 d5 2.c4 dxc4 3.e4 c5 4.d5 *"),
("D20u", "QGA: Linares Variation", "1.d4 d5 2.c4 dxc4 3.e4 c5 4.d5 Nf6 5.Nc3 b5 *"),
("D20v", "QGA: 3.e4 e5", "1.d4 d5 2.c4 dxc4 3.e4 e5 *"),
("D20w", "QGA: 3.e4 e5 4.Nf3 Bb4+", "1.d4 d5 2.c4 dxc4 3.e4 e5 4.Nf3 Bb4+ *"),
("D20x", "QGA: 3.e4 e5 4.Nf3 exd4", "1.d4 d5 2.c4 dxc4 3.e4 e5 4.Nf3 exd4 *"),
("D20y", "QGA: 3.e4 e5 4.Nf3 exd4 5.Bxc4 Nc6", "1.d4 d5 2.c4 dxc4 3.e4 e5 4.Nf3 exd4 5.Bxc4 Nc6 *"),
("D21a", "QGA: 3.Nf3", "1.d4 d5 2.c4 dxc4 3.Nf3 *"),
("D21a", "QGA: Ericson Variation", "1.d4 d5 2.c4 dxc4 3.Nf3 b5 *"),
("D21b", "QGA: 3.Nf3 Nd7", "1.d4 d5 2.c4 dxc4 3.Nf3 Nd7 *"),
("D21c", "QGA: 3.Nf3 Bg4", "1.d4 d5 2.c4 dxc4 3.Nf3 Bg4 *"),
("D21d", "QGA: 3.Nf3 e6", "1.d4 d5 2.c4 dxc4 3.Nf3 e6 *"),
("D21e", "QGA: 3.Nf3 e6 4.Qa4+", "1.d4 d5 2.c4 dxc4 3.Nf3 e6 4.Qa4+ *"),
("D21f", "QGA: 3.Nf3 e6 4.e4", "1.d4 d5 2.c4 dxc4 3.Nf3 e6 4.e4 *"),
("D21g", "QGA: 3.Nf3 e6 4.e3", "1.d4 d5 2.c4 dxc4 3.Nf3 e6 4.e3 *"),
("D21g", "QGA: 3.Nf3 e6 4.e3 c5", "1.d4 d5 2.c4 dxc4 3.Nf3 e6 4.e3 c5 *"),
("D21h", "QGA: 3.Nf3 c5", "1.d4 d5 2.c4 dxc4 3.Nf3 c5 *"),
("D21i", "QGA: 3.Nf3 c5 4.Nc3", "1.d4 d5 2.c4 dxc4 3.Nf3 c5 4.Nc3 *"),
("D21j", "QGA: 3.Nf3 c5 4.e3", "1.d4 d5 2.c4 dxc4 3.Nf3 c5 4.e3 *"),
("D21j", "QGA: 3.Nf3 c5 4.e3 cxd4", "1.d4 d5 2.c4 dxc4 3.Nf3 c5 4.e3 cxd4 *"),
("D21k", "QGA: 3.Nf3 c5 4.e3 cxd4 5.Bxc4", "1.d4 d5 2.c4 dxc4 3.Nf3 c5 4.e3 cxd4 5.Bxc4 *"),
("D21l", "QGA: 3.Nf3 c5 4.d5", "1.d4 d5 2.c4 dxc4 3.Nf3 c5 4.d5 *"),
("D21m", "QGA: 3.Nf3 c5 4.d5 e6", "1.d4 d5 2.c4 dxc4 3.Nf3 c5 4.d5 e6 *"),
("D21m", "QGA: 3.Nf3 c5 4.d5 e6 5.e4", "1.d4 d5 2.c4 dxc4 3.Nf3 c5 4.d5 e6 5.e4 *"),
("D21n", "QGA: 3.Nf3 c5 4.d5 e6 5.Nc3", "1.d4 d5 2.c4 dxc4 3.Nf3 c5 4.d5 e6 5.Nc3 *"),
("D21o", "QGA: 3.Nf3 c5 4.d5 e6 5.Nc3 exd5", "1.d4 d5 2.c4 dxc4 3.Nf3 c5 4.d5 e6 5.Nc3 exd5 *"),
("D21p", "QGA: 3.Nf3 c5 4.d5 e6 5.Nc3 exd5 6.Qxd5 Qxd5 7.Nxd5", "1.d4 d5 2.c4 dxc4 3.Nf3 c5 4.d5 e6 5.Nc3 exd5 6.Qxd5 Qxd5 7.Nxd5 *"),
("D21q", "QGA: 3.Nf3 c5 4.d5 e6 5.Nc3 exd5 6.Qxd5 Qxd5 7.Nxd5 Bd6", "1.d4 d5 2.c4 dxc4 3.Nf3 c5 4.d5 e6 5.Nc3 exd5 6.Qxd5 Qxd5 7.Nxd5 Bd6 *"),
("D21r", "QGA: 3.Nf3 c5 4.d5 e6 5.Nc3 exd5 6.Qxd5 Qxd5 7.Nxd5 Bd6 8.Nd2", "1.d4 d5 2.c4 dxc4 3.Nf3 c5 4.d5 e6 5.Nc3 exd5 6.Qxd5 Qxd5 7.Nxd5 Bd6 8.Nd2 *"),
("D21s", "QGA: Seirawan Variation", "1.d4 d5 2.c4 dxc4 3.Nf3 c5 4.d5 e6 5.Nc3 exd5 6.Qxd5 Qxd5 7.Nxd5 Bd6 8.Nd2 Ne7 *"),
("D21t", "QGA: Alekhine Defence", "1.d4 d5 2.c4 dxc4 3.Nf3 a6 *"),
("D21u", "QGA: Alekhine, Borisenko-Furman Variation", "1.d4 d5 2.c4 dxc4 3.Nf3 a6 4.e4 *"),
("D21v", "QGA: Alekhine, 4.a4", "1.d4 d5 2.c4 dxc4 3.Nf3 a6 4.a4 *"),
("D22a", "QGA: Alekhine, 4.e3", "1.d4 d5 2.c4 dxc4 3.Nf3 a6 4.e3 *"),
("D22b", "QGA: Alekhine, Haberditz Variation", "1.d4 d5 2.c4 dxc4 3.Nf3 a6 4.e3 b5 *"),
("D22c", "QGA: Alekhine, 4.e3 e6", "1.d4 d5 2.c4 dxc4 3.Nf3 a6 4.e3 e6 *"),
("D22d", "QGA: Alekhine, 4.e3 e6 5.Bxc4", "1.d4 d5 2.c4 dxc4 3.Nf3 a6 4.e3 e6 5.Bxc4 *"),
("D22e", "QGA: Alekhine, 4.e3 e6 5.Bxc4 c5", "1.d4 d5 2.c4 dxc4 3.Nf3 a6 4.e3 e6 5.Bxc4 c5 *"),
("D22f", "QGA: Alekhine, 4.e3 e6 5.Bxc4 c5 6.Qe2", "1.d4 d5 2.c4 dxc4 3.Nf3 a6 4.e3 e6 5.Bxc4 c5 6.Qe2 *"),
("D22g", "QGA: Alekhine, 4.e3 Bg4", "1.d4 d5 2.c4 dxc4 3.Nf3 a6 4.e3 Bg4 *"),
("D22h", "QGA: Alekhine, 4.e3 Bg4 5.Bxc4 e6", "1.d4 d5 2.c4 dxc4 3.Nf3 a6 4.e3 Bg4 5.Bxc4 e6 *"),
("D22i", "QGA: Alekhine, Alatortsev Variation", "1.d4 d5 2.c4 dxc4 3.Nf3 a6 4.e3 Bg4 5.Bxc4 e6 6.d5 *"),
("D22j", "QGA: Alekhine, 4.e3 Bg4 5.Bxc4 e6 6.Qb3", "1.d4 d5 2.c4 dxc4 3.Nf3 a6 4.e3 Bg4 5.Bxc4 e6 6.Qb3 *"),
("D22k", "QGA: Alekhine, 4.e3 Bg4 5.Bxc4 e6 6.Nc3", "1.d4 d5 2.c4 dxc4 3.Nf3 a6 4.e3 Bg4 5.Bxc4 e6 6.Nc3 *"),
("D22l", "QGA: Alekhine, 4.e3 Bg4 5.Bxc4 e6 6.h3", "1.d4 d5 2.c4 dxc4 3.Nf3 a6 4.e3 Bg4 5.Bxc4 e6 6.h3 *"),
("D23a", "QGA: 3.Nf3 Nf6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 *"),
("D23b", "QGA: 3.Nf3 Nf6 4.g3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.g3 *"),
("D23c", "QGA: Mannheim Variation", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Qa4+ *"),
("D23d", "QGA: Mannheim, 4...Nc6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Qa4+ Nc6 *"),
("D23e", "QGA: Mannheim, 4...Nc6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Qa4+ Nc6 5.Nc3 *"),
("D23f", "QGA: Mannheim, 4...Nbd7", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Qa4+ Nbd7 *"),
("D23g", "QGA: Mannheim, 4...Nbd7 5.Nc3 e6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Qa4+ Nbd7 5.Nc3 e6 *"),
("D23h", "QGA: Mannheim, 4...Nbd7 5.Nc3 e6 6.e4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Qa4+ Nbd7 5.Nc3 e6 6.e4 *"),
("D23i", "QGA: Mannheim, 4...c6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Qa4+ c6 *"),
("D23i", "QGA: Mannheim, 4...c6 5.Qxc4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Qa4+ c6 5.Qxc4 *"),
("D23j", "QGA: Mannheim, 4...c6, 5.Qxc4 Bf5", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Qa4+ c6 5.Qxc4 Bf5 *"),
("D23k", "QGA: Mannheim, 4...c6, 5.Qxc4 Bf5 6.Nc3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Qa4+ c6 5.Qxc4 Bf5 6.Nc3 *"),
("D23l", "QGA: Mannheim, 4...c6, 5.Qxc4 Bf5 6.g3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Qa4+ c6 5.Qxc4 Bf5 6.g3 *"),
("D23m", "QGA: Mannheim, 4...c6, 5.Qxc4 Bf5 6.g3 e6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Qa4+ c6 5.Qxc4 Bf5 6.g3 e6 *"),
("D23m", "QGA: Mannheim, 4...c6, 5.Qxc4 Bf5 6.g3 e6 7.Bg2", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Qa4+ c6 5.Qxc4 Bf5 6.g3 e6 7.Bg2 *"),
("D23n", "QGA: Mannheim, 4...c6, 5.Qxc4 Bf5 6.g3 e6 7.Bg2 Nbd7", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Qa4+ c6 5.Qxc4 Bf5 6.g3 e6 7.Bg2 Nbd7 *"),
("D23o", "QGA: Mannheim, Main Line", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Qa4+ c6 5.Qxc4 Bf5 6.g3 e6 7.Bg2 Nbd7 8.O-O Be7 9.Nc3 *"),
("D23p", "QGA: Mannheim, Main Line, 9...O-O", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Qa4+ c6 5.Qxc4 Bf5 6.g3 e6 7.Bg2 Nbd7 8.O-O Be7 9.Nc3 O-O *"),
("D23q", "QGA: Mannheim, Main Line, 10.Bg5", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Qa4+ c6 5.Qxc4 Bf5 6.g3 e6 7.Bg2 Nbd7 8.O-O Be7 9.Nc3 O-O 10.Bg5 *"),
("D23r", "QGA: Mannheim, Main Line, 10.e3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Qa4+ c6 5.Qxc4 Bf5 6.g3 e6 7.Bg2 Nbd7 8.O-O Be7 9.Nc3 O-O 10.e3 *"),
("D23s", "QGA: Mannheim, Main Line, 10.e3 Ne4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Qa4+ c6 5.Qxc4 Bf5 6.g3 e6 7.Bg2 Nbd7 8.O-O Be7 9.Nc3 O-O 10.e3 Ne4 *"),
("D23t", "QGA: Mannheim, Main Line, 10.e3 Ne4 11.Qe2", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Qa4+ c6 5.Qxc4 Bf5 6.g3 e6 7.Bg2 Nbd7 8.O-O Be7 9.Nc3 O-O 10.e3 Ne4 11.Qe2 *"),
("D24a", "QGA: 4.Nc3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 *"),
("D24a", "QGA: 4.Nc3 Nd5", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 Nd5 *"),
("D24b", "QGA: 4.Nc3 e6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 e6 *"),
("D24b", "QGA: 4.Nc3 e6 5.Bg5", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 e6 5.Bg5 *"),
("D24b", "QGA: 4.Nc3 e6 5.e4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 e6 5.e4 *"),
("D24c", "QGA: 4.Nc3 e6 5.e3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 e6 5.e3 *"),
("D24d", "QGA: 4.Nc3 c5", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 c5 *"),
("D24e", "QGA: 4.Nc3 c5 5.d5", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 c5 5.d5 *"),
("D24f", "QGA: 4.Nc3 c5 5.d5 e6 6.e4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 c5 5.d5 e6 6.e4 *"),
("D24g", "QGA: 4.Nc3 c5 5.d5 e6 6.e4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 c5 5.d5 e6 6.e4 exd5 7.e5 Nfd7 *"),
("D24g", "QGA: 4.Nc3 c5 5.d5 e6 6.e4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 c5 5.d5 e6 6.e4 exd5 7.e5 Nfd7 8.Bg5 *"),
("D24g", "QGA: 4.Nc3 c5 5.d5 e6 6.e4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 c5 5.d5 e6 6.e4 exd5 7.e5 Nfd7 8.Bg5 Be7 *"),
("D24h", "QGA: 4.Nc3 a6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 a6 *"),
("D24i", "QGA: 4.Nc3 a6 5.a4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 a6 5.a4 *"),
("D24j", "QGA: 4.Nc3 a6 5.a4 Nc6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 a6 5.a4 Nc6 *"),
("D24j", "QGA: 4.Nc3 a6 5.a4 Nc6 5.e4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 a6 5.a4 Nc6 6.e4 *"),
("D24k", "QGA: Bogoljubow", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 a6 5.e4 *"),
("D24l", "QGA: Bogoljubow, 7.a4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 a6 5.e4 b5 6.e5 Nd5 7.a4 *"),
("D24m", "QGA: Bogoljubow, 7.a4 e6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 a6 5.e4 b5 6.e5 Nd5 7.a4 e6 *"),
("D24n", "QGA: Bogoljubow, 7.a4 c6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 a6 5.e4 b5 6.e5 Nd5 7.a4 c6 *"),
("D24o", "QGA: Bogoljubow, 7.a4 Bb7", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 a6 5.e4 b5 6.e5 Nd5 7.a4 Bb7 *"),
("D24p", "QGA: Bogoljubow, 7.a4 Nb4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 a6 5.e4 b5 6.e5 Nd5 7.a4 Nb4 *"),
("D24q", "QGA: Bogoljubow, 7.a4 Nxc3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 a6 5.e4 b5 6.e5 Nd5 7.a4 Nxc3 *"),
("D24r", "QGA: Bogoljubow, 7.a4 Nxc3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 a6 5.e4 b5 6.e5 Nd5 7.a4 Nxc3 8.bxc3 Bb7 *"),
("D24s", "QGA: Bogoljubow, 7.a4 Nxc3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 a6 5.e4 b5 6.e5 Nd5 7.a4 Nxc3 8.bxc3 Qd5 *"),
("D24t", "QGA: Bogoljubow, 7.a4 Nxc3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.Nc3 a6 5.e4 b5 6.e5 Nd5 7.a4 Nxc3 8.bxc3 Qd5 9.g3 Bb7 *"),
("D25a", "QGA: 4.e3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 *"),
("D25b", "QGA: 4.e3 c5", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 c5 *"),
("D25c", "QGA: 4.e3 a6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 a6 *"),
("D25d", "QGA: Smyslov Variation", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 g6 *"),
("D25d", "QGA: Smyslov, 5.Bxc4 Bg7", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 g6 5.Bxc4 Bg7 *"),
("D25d", "QGA: Smyslov, 5.Bxc4 Bg7 6.Nc3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 g6 5.Bxc4 Bg7 6.Nc3 *"),
("D25e", "QGA: Smyslov, 5.Bxc4 Bg7 6.O-O", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 g6 5.Bxc4 Bg7 6.O-O *"),
("D25e", "QGA: Smyslov, 5.Bxc4 Bg7 6.O-O O-O", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 g6 5.Bxc4 Bg7 6.O-O O-O *"),
("D25e", "QGA: Smyslov, 5.Bxc4 Bg7 6.O-O O-O 7.Nc3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 g6 5.Bxc4 Bg7 6.O-O O-O 7.Nc3 *"),
("D25f", "QGA: Flohr Variation", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 Be6 *"),
("D25f", "QGA: Flohr, 5.Nc3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 Be6 5.Nc3 *"),
("D25f", "QGA: Flohr, 5.Nc3 c6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 Be6 5.Nc3 c6 *"),
("D25g", "QGA: Janowski-Larsen Variation", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 Bg4 *"),
("D25g", "QGA: Janowski-Larsen, 5.h3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 Bg4 5.h3 *"),
("D25h", "QGA: Janowski-Larsen, 5.Bxc4 e6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 Bg4 5.Bxc4 e6 *"),
("D25i", "QGA: Janowski-Larsen, 6.Qb3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 Bg4 5.Bxc4 e6 6.Qb3 *"),
("D25j", "QGA: Janowski-Larsen, 6.O-O", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 Bg4 5.Bxc4 e6 6.O-O *"),
("D25k", "QGA: Janowski-Larsen, 6.Nc3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 Bg4 5.Bxc4 e6 6.Nc3 *"),
("D25l", "QGA: Janowski-Larsen, 6.h3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 Bg4 5.Bxc4 e6 6.h3 *"),
("D25m", "QGA: Janowski-Larsen, 6.h3 Bh5 7.Nc3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 Bg4 5.Bxc4 e6 6.h3 Bh5 7.Nc3 *"),
("D25n", "QGA: Janowski-Larsen, 6.h3 Bh5 7.Nc3 a6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 Bg4 5.Bxc4 e6 6.h3 Bh5 7.Nc3 a6 *"),
("D25o", "QGA: Janowski-Larsen, 6.h3 Bh5 7.Nc3 a6 8.g4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 Bg4 5.Bxc4 e6 6.h3 Bh5 7.Nc3 a6 8.g4 *"),
("D25p", "QGA: Janowski-Larsen, 6.h3 Bh5 7.Nc3 Nbd7", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 Bg4 5.Bxc4 e6 6.h3 Bh5 7.Nc3 Nbd7 *"),
("D25q", "QGA: Janowski-Larsen, 6.h3 Bh5 7.Nc3 Nbd7 8.O-O", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 Bg4 5.Bxc4 e6 6.h3 Bh5 7.Nc3 Nbd7 8.O-O *"),
("D25r", "QGA: Janowski-Larsen, 6.h3 Bh5 7.Nc3 Nbd7 8.O-O Bd6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 Bg4 5.Bxc4 e6 6.h3 Bh5 7.Nc3 Nbd7 8.O-O Bd6 *"),
("D25r", "QGA: Janowski-Larsen, 6.h3 Bh5 7.Nc3 Nbd7 8.O-O Bd6 9.Be2", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 Bg4 5.Bxc4 e6 6.h3 Bh5 7.Nc3 Nbd7 8.O-O Bd6 9.Be2 *"),
("D25s", "QGA: Janowski-Larsen, 6.h3 Bh5 7.Nc3 Nbd7 8.O-O Bd6 9.e4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 Bg4 5.Bxc4 e6 6.h3 Bh5 7.Nc3 Nbd7 8.O-O Bd6 9.e4 *"),
("D26a", "QGA: 4.e3 e6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 *"),
("D26b", "QGA: 4.e3 e6 5.Bxc4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 *"),
("D26c", "QGA: 4.e3 e6 5.Bxc4 a6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 a6 *"),
("D26d", "QGA: 4.e3 e6 5.Bxc4 a6 6.O-O", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 a6 6.O-O *"),
("D26e", "QGA: 4.e3 e6 5.Bxc4 a6 6.O-O b5", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 a6 6.O-O b5 *"),
("D26f", "QGA: Classical Variation", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 *"),
("D26g", "QGA: Classical, Early Queenswap", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.dxc5 *"),
("D26h", "QGA: Classical, 6.Nc3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.Nc3 *"),
("D26i", "QGA: Classical, 6.Nc3 a6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.Nc3 a6 *"),
("D26j", "QGA: Classical, Furman Variation", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.Qe2 *"),
("D26k", "QGA: Classical, Furman, 6...cxd4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.Qe2 cxd4 *"),
("D26l", "QGA: Classical, Furman, 6...a6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.Qe2 a6 *"),
("D26m", "QGA: Classical, Furman, 6...a6 7.dxc5 Bxc5 8.O-O", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.Qe2 a6 7.dxc5 Bxc5 8.O-O *"),
("D26n", "QGA: Classical, Furman, 6...a6 7.dxc5 Bxc5 8.O-O Nc6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.Qe2 a6 7.dxc5 Bxc5 8.O-O Nc6 *"),
("D26o", "QGA: Classical, Furman, 6...a6 7.dxc5 Bxc5 8.O-O b5", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.Qe2 a6 7.dxc5 Bxc5 8.O-O b5 *"),
("D26p", "QGA: Classical, 6.O-O", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O *"),
("D26q", "QGA: Classical, Steinitz Variation", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O cxd4 *"),
("D26r", "QGA: Classical, 6.O-O Nc6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O Nc6 *"),
("D26s", "QGA: Classical, 6.O-O Nc6 7.Nc3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O Nc6 7.Nc3 *"),
("D26t", "QGA: Classical, 6.O-O Nc6 7.Qe2", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O Nc6 7.Qe2 *"),
("D27a", "QGA: Classical, 6...a6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 *"),
("D27a", "QGA: Classical, 6...a6 7.a3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.a3 *"),
("D27b", "QGA: Classical, 6...a6 7.b3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.b3 *"),
("D27c", "QGA: Classical, 6...a6 7.Nc3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Nc3 *"),
("D27d", "QGA: Classical, 6...a6 7.Bd3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Bd3 *"),
("D27e", "QGA: Classical, 6...a6 7.Bd3 Nbd7", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Bd3 Nbd7 *"),
("D27f", "QGA: Classical, 6...a6 7.Bb3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Bb3 *"),
("D27g", "QGA: Classical, 6...a6 7.Bb3 b5", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Bb3 b5 *"),
("D27h", "QGA: Classical, 6...a6 7.Bb3 cxd4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Bb3 cxd4 *"),
("D27i", "QGA: Classical, 6...a6 7.Bb3 Nc6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Bb3 Nc6 *"),
("D27j", "QGA: Classical, 6...a6 7.Bb3 Nc6 8.Nc3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Bb3 Nc6 8.Nc3 *"),
("D27k", "QGA: Classical, Geller Variation", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.e4 *"),
("D27l", "QGA: Classical, Spassky Variation", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.dxc5 *"),
("D27m", "QGA: Classical, Spassky, 7...Qxd1", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.dxc5 Qxd1 *"),
("D27n", "QGA: Classical, Spassky, 9.Nbd2", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.dxc5 Qxd1 8.Rxd1 Bxc5 9.Nbd2 *"),
("D27o", "QGA: Classical, Spassky, 9.b3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.dxc5 Qxd1 8.Rxd1 Bxc5 9.b3 *"),
("D27p", "QGA: Classical, Rubinstein Variation", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.a4 *"),
("D27p", "QGA: Classical, Rubinstein, 7...Nc6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.a4 Nc6 *"),
("D27q", "QGA: Classical, Rubinstein, 8.Nc3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.a4 Nc6 8.Nc3 *"),
("D27r", "QGA: Classical, Rubinstein, 8.Nc3 Be7", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.a4 Nc6 8.Nc3 Be7 *"),
("D27s", "QGA: Classical, Rubinstein, 8.Nc3 Be7 9.Qe2", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.a4 Nc6 8.Nc3 Be7 9.Qe2 *"),
("D27t", "QGA: Classical, Rubinstein, 8.Qe2", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.a4 Nc6 8.Qe2 *"),
("D27u", "QGA: Classical, Rubinstein, 8.Qe2 Qc7", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.a4 Nc6 8.Qe2 Qc7 *"),
("D27v", "QGA: Classical, Rubinstein, 8.Qe2 cxd4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.a4 Nc6 8.Qe2 cxd4 *"),
("D27v", "QGA: Classical, Rubinstein, 8.Qe2 cxd4, 11.Nc3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.a4 Nc6 8.Qe2 cxd4 9.Rd1 Be7 10.exd4 O-O 11.Nc3 *"),
("D27w", "QGA: Classical, Rubinstein, 8.Qe2 cxd4, 11.Nc3 Nd5", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.a4 Nc6 8.Qe2 cxd4 9.Rd1 Be7 10.exd4 O-O 11.Nc3 Nd5 *"),
("D28a", "QGA: Classical, 7.Qe2", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 *"),
("D28b", "QGA: Classical, 7.Qe2 cxd4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 cxd4 *"),
("D28c", "QGA: Classical, 7.Qe2 Nc6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 Nc6 *"),
("D28d", "QGA: Classical, 7.Qe2 b5", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 *"),
("D28e", "QGA: Classical, 7.Qe2 b5 8.Bd3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bd3 *"),
("D28f", "QGA: Classical, 7.Qe2 b5 8.Bd3 cxd4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bd3 cxd4 *"),
("D28g", "QGA: Classical, 7.Qe2 b5 8.Bd3 cxd4 9.exd4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bd3 cxd4 9.exd4 *"),
("D28h", "QGA: Classical, 7.Qe2 b5 8.Bb3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bb3 *"),
("D28i", "QGA: Classical, 7.Qe2 b5 8.Bb3 Nc6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bb3 Nc6 *"),
("D28j", "QGA: Classical, 7.Qe2 b5 8.Bb3 Nc6 9.Rd1", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bb3 Nc6 9.Rd1 *"),
("D28k", "QGA: Classical, 7.Qe2 b5 8.Bb3 Nc6 9.Rd1 c4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bb3 Nc6 9.Rd1 c4 *"),
("D28l", "QGA: Classical, 7.Qe2 b5 8.Bb3 Nc6 9.Nc3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bb3 Nc6 9.Nc3 *"),
("D28m", "QGA: Classical, 7.Qe2 b5 8.Bb3 Nc6 9.Nc3 Bb7", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bb3 Nc6 9.Nc3 Bb7 *"),
("D28n", "QGA: Classical, 7.Qe2 b5 8.Bb3 Nc6 9.Nc3 Be7", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bb3 Nc6 9.Nc3 Be7 *"),
("D29a", "QGA: Classical, 8...Bb7", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bb3 Bb7 *"),
("D29b", "QGA: Classical, 8...Bb7 9.Nc3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bb3 Bb7 9.Nc3 *"),
("D29c", "QGA: Classical, 8...Bb7 9.a4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bb3 Bb7 9.a4 *"),
("D29d", "QGA: Classical, 8...Bb7 9.a4 b4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bb3 Bb7 9.a4 b4 *"),
("D29e", "QGA: Classical, 8...Bb7 9.a4 Nbd7", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bb3 Bb7 9.a4 Nbd7 *"),
("D29f", "QGA: Classical, 8...Bb7 9.Rd1", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bb3 Bb7 9.Rd1 *"),
("D29g", "QGA: Classical, 8...Bb7 9.Rd1 Nbd7", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bb3 Bb7 9.Rd1 Nbd7 *"),
("D29g", "QGA: Classical, 8...Bb7 9.Rd1 Nbd7 10.e4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bb3 Bb7 9.Rd1 Nbd7 10.e4 *"),
("D29h", "QGA: Classical, 8...Bb7 9.Rd1 Nbd7 10.a4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bb3 Bb7 9.Rd1 Nbd7 10.a4 *"),
("D29i", "QGA: Classical, 8...Bb7 9.Rd1 Nbd7 10.a4 b4", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bb3 Bb7 9.Rd1 Nbd7 10.a4 b4 *"),
("D29j", "QGA: Classical, 8...Bb7 9.Rd1 Nbd7 10.Nc3", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bb3 Bb7 9.Rd1 Nbd7 10.Nc3 *"),
("D29k", "QGA: Classical, Smyslov Variation", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bb3 Bb7 9.Rd1 Nbd7 10.Nc3 Bd6 *"),
("D29l", "QGA: Classical, 8...Bb7 9.Rd1 Nbd7 10.Nc3 Qc7", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bb3 Bb7 9.Rd1 Nbd7 10.Nc3 Qc7 *"),
("D29m", "QGA: Classical, 8...Bb7 9.Rd1 Nbd7 10.Nc3 Qb8", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bb3 Bb7 9.Rd1 Nbd7 10.Nc3 Qb8 *"),
("D29n", "QGA: Classical, 8...Bb7 9.Rd1 Nbd7 10.Nc3 Qb6", "1.d4 d5 2.c4 dxc4 3.Nf3 Nf6 4.e3 e6 5.Bxc4 c5 6.O-O a6 7.Qe2 b5 8.Bb3 Bb7 9.Rd1 Nbd7 10.Nc3 Qb6 *"),
("D30a", "Queen's Gambit Declined (QGD)", "1.d4 d5 2.c4 e6 *"),
("D30a", "QGD: 3.e3", "1.d4 d5 2.c4 e6 3.e3 *"),
("D30b", "Diemer-Duhm Gambit (DDG)", "1.d4 d5 2.c4 e6 3.e4 *"),
("D30b", "Diemer-Duhm Gambit (DDG) Accepted", "1.d4 d5 2.c4 e6 3.e4 dxe4 *"),
("D30b", "Diemer-Duhm Gambit (DDG): 4...f5", "1.e4 e6 2.d4 d5 3.c4 dxe4 4.Nc3 f5 *"),
("D30b", "Diemer-Duhm Gambit (DDG): Alapin Variation", "1.e4 e6 2.d4 d5 3.c4 dxe4 4.Nc3 Nf6 5.Be3 *"),
("D30b", "Diemer-Duhm Gambit (DDG): Duhm Variation", "1.e4 e6 2.d4 d5 3.c4 dxe4 4.Nc3 Nf6 5.a3 *"),
("D30b", "Diemer-Duhm Gambit (DDG): 5.f3", "1.e4 e6 2.d4 d5 3.c4 dxe4 4.Nc3 Nf6 5.f3 *"),
("D30b", "Diemer-Duhm Gambit (DDG): Exchange Variation", "1.e4 e6 2.d4 d5 3.c4 dxe4 4.Nc3 Nf6 5.f3 exf3 *"),
("D30b", "Diemer-Duhm Gambit (DDG): Keres Defence", "1.e4 e6 2.d4 d5 3.c4 dxe4 4.Nc3 Nf6 5.f3 c5 *"),
("D30b", "Diemer-Duhm Gambit (DDG): Main Line", "1.e4 e6 2.d4 d5 3.c4 dxe4 4.Nc3 Nf6 5.f3 Bb4 *"),
("D30c", "QGD: 3.g3", "1.d4 d5 2.c4 e6 3.g3 *"),
("D30d", "QGD: 3.cxd5", "1.d4 d5 2.c4 e6 3.cxd5 *"),
("D30d", "QGD: 3.cxd5", "1.d4 d5 2.c4 e6 3.cxd5 exd5 *"),
("D30d", "QGD: 3.cxd5", "1.d4 d5 2.c4 e6 3.cxd5 exd5 4.Nc3 *"),
("D30e", "QGD: 3.Bf4", "1.d4 d5 2.c4 e6 3.Bf4 *"),
("D30f", "QGD: 3.Nf3", "1.d4 d5 2.c4 e6 3.Nf3 *"),
("D30g", "QGD: 3.Nf3 Nbd7 (Westphalia)", "1.d4 d5 2.c4 e6 3.Nf3 Nbd7 *"),
("D30h", "QGD: Tarrasch without Nc3", "1.d4 d5 2.c4 e6 3.Nf3 c5 *"),
("D30h", "QGD: Tarrasch without Nc3: 4.e3", "1.d4 d5 2.c4 e6 3.Nf3 c5 4.e3 *"),
("D30h", "QGD: Tarrasch without Nc3: 4.e3 Nf6", "1.d4 d5 2.c4 e6 3.Nf3 c5 4.e3 Nf6 *"),
("D30h", "QGD: Tarrasch without Nc3", "1.d4 d5 2.c4 e6 3.Nf3 c5 4.cxd5 *"),
("D30h", "QGD: Tarrasch without Nc3", "1.d4 d5 2.c4 e6 3.Nf3 c5 4.cxd5 exd5 *"),
("D30i", "QGD: Tarrasch without Nc3: 5.g3", "1.d4 d5 2.c4 e6 3.Nf3 c5 4.cxd5 exd5 5.g3 *"),
("D30i", "QGD: Tarrasch without Nc3: 5.g3", "1.d4 d5 2.c4 e6 3.Nf3 c5 4.cxd5 exd5 5.g3 Nf6 *"),
("D30i", "QGD: Tarrasch without Nc3: 5.g3", "1.d4 d5 2.c4 e6 3.Nf3 c5 4.cxd5 exd5 5.g3 Nf6 6.Bg2 *"),
("D30i", "QGD: Tarrasch without Nc3: 5.g3 Nc6", "1.d4 d5 2.c4 e6 3.Nf3 c5 4.cxd5 exd5 5.g3 Nc6 *"),
("D30i", "QGD: Tarrasch without Nc3: 5.g3 Nc6 6.Bg2", "1.d4 d5 2.c4 e6 3.Nf3 c5 4.cxd5 exd5 5.g3 Nc6 6.Bg2 *"),
("D30i", "QGD: Tarrasch without Nc3: 5.g3 Nc6 6.Bg2 Nf6", "1.d4 d5 2.c4 e6 3.Nf3 c5 4.cxd5 exd5 5.g3 Nc6 6.Bg2 Nf6 *"),
("D30i", "QGD: Tarrasch without Nc3: 5.g3 Nc6 6.Bg2 Nf6 7.O-O", "1.d4 d5 2.c4 e6 3.Nf3 c5 4.cxd5 exd5 5.g3 Nc6 6.Bg2 Nf6 7.O-O *"),
("D30i", "QGD: Tarrasch without Nc3: Main Line", "1.d4 d5 2.c4 e6 3.Nf3 c5 4.cxd5 exd5 5.g3 Nc6 6.Bg2 Nf6 7.O-O Be7 *"),
("D30j", "QGD: 3.Nf3 c6", "1.d4 d5 2.c4 e6 3.Nf3 c6 *"),
("D30k", "QGD: 3.Nf3 c6 4.e3", "1.d4 d5 2.c4 e6 3.Nf3 c6 4.e3 *"),
("D30l", "QGD: 3.Nf3 c6 4.Nbd2", "1.d4 d5 2.c4 e6 3.Nf3 c6 4.Nbd2 *"),
("D30m", "QGD: 3.Nf3 c6 4.Qc2", "1.d4 d5 2.c4 e6 3.Nf3 c6 4.Qc2 *"),
("D30n", "QGD: 3.Nf3 c6 4.Qc2 Nf6", "1.d4 d5 2.c4 e6 3.Nf3 c6 4.Qc2 Nf6 *"),
("D30o", "QGD: 3.Nf3 c6 4.Qc2 Nf6 5.Bg5", "1.d4 d5 2.c4 e6 3.Nf3 c6 4.Qc2 Nf6 5.Bg5 *"),
("D30p", "QGD: 3.Nf3 c6 4.Qc2 Nf6 5.g3", "1.d4 d5 2.c4 e6 3.Nf3 c6 4.Qc2 Nf6 5.g3 *"),
("D30q", "QGD: 3.Nf3 Nf6", "1.d4 d5 2.c4 e6 3.Nf3 Nf6 *"),
("D30r", "QGD: 3.Nf3 Nf6 4.e3", "1.d4 d5 2.c4 e6 3.Nf3 Nf6 4.e3 *"),
("D30s", "QGD: 3.Nf3 Nf6 4.e3 c6", "1.d4 d5 2.c4 e6 3.Nf3 Nf6 4.e3 c6 *"),
("D30t", "QGD: 3.Nf3 Nf6 4.e3 c6 5.Nbd2", "1.d4 d5 2.c4 e6 3.Nf3 Nf6 4.e3 c6 5.Nbd2 *"),
("D30t", "QGD: Spielmann Variation", "1.d4 d5 2.c4 e6 3.Nf3 Nf6 4.e3 c6 5.Nbd2 g6 *"),
("D30t", "QGD: Stonewall Formation", "1.d4 d5 2.c4 e6 3.Nf3 Nf6 4.e3 c6 5.Nbd2 Ne4 6.Bd3 f5 *"),
("D30t", "QGD: 3.Nf3 Nf6 4.e3 c6 5.Nbd2 Nbd7", "1.d4 d5 2.c4 e6 3.Nf3 Nf6 4.e3 c6 5.Nbd2 Nbd7 *"),
("D30u", "QGD: 3.Nf3 Nf6 4.e3 c6 5.Nbd2 Nbd7 6.Bd3", "1.d4 d5 2.c4 e6 3.Nf3 Nf6 4.e3 c6 5.Nbd2 Nbd7 6.Bd3 *"),
("D30u", "QGD: Semmering Variation", "1.d4 d5 2.c4 e6 3.Nf3 Nf6 4.e3 c6 5.Nbd2 Nbd7 6.Bd3 c5 *"),
("D30v", "QGD: 3.Nf3 Nf6 4.Bg5", "1.d4 d5 2.c4 e6 3.Nf3 Nf6 4.Bg5 *"),
("D30v", "QGD: 3.Nf3 Nf6 4.Bg5 dxc4", "1.d4 d5 2.c4 e6 3.Nf3 Nf6 4.Bg5 dxc4 *"),
("D30v", "QGD: Vienna Variation", "1.d4 d5 2.c4 e6 3.Nf3 Nf6 4.Bg5 Bb4+ *"),
("D30v", "QGD: 3.Nf3 Nf6 4.Bg5 Nbd7", "1.d4 d5 2.c4 e6 3.Nf3 Nf6 4.Bg5 Nbd7 *"),
("D30v", "QGD: 3.Nf3 Nf6 4.Bg5, Capablanca Variation", "1.d4 d5 2.c4 e6 3.Nf3 Nf6 4.Bg5 Nbd7 5.e3 c6 6.Nbd2 *"),
("D30w", "QGD: Capablanca-Duras Variation", "1.d4 d5 2.c4 e6 3.Nf3 Nf6 4.Bg5 h6 *"),
("D30x", "QGD: 3.Nf3 Nf6 4.Bg5 Be7", "1.d4 d5 2.c4 e6 3.Nf3 Nf6 4.Bg5 Be7 *"),
("D30x", "QGD: 3.Nf3 Nf6 4.Bg5 Be7 5.e3", "1.d4 d5 2.c4 e6 3.Nf3 Nf6 4.Bg5 Be7 5.e3 *"),
("D31a", "QGD: 3.Nc3", "1.d4 d5 2.c4 e6 3.Nc3 *"),
("D31a", "QGD: 3.Nc3 Nc6", "1.d4 d5 2.c4 e6 3.Nc3 Nc6 *"),
("D31a", "QGD: Alapin Variation", "1.d4 d5 2.c4 e6 3.Nc3 b6 *"),
("D31b", "QGD: Janowski Variation", "1.d4 d5 2.c4 e6 3.Nc3 a6 *"),
("D31c", "QGD: 3.Nc3 Bb4", "1.d4 d5 2.c4 e6 3.Nc3 Bb4 *"),
("D31c", "QGD: 3.Nc3 Bb4 4.a3", "1.d4 d5 2.c4 e6 3.Nc3 Bb4 4.a3 *"),
("D31d", "QGD: Alatortsev Variation", "1.d4 d5 2.c4 e6 3.Nc3 Be7 *"),
("D31d", "QGD: Alatortsev, 4.Nf3", "1.d4 d5 2.c4 e6 3.Nc3 Be7 4.Nf3 *"),
("D31d", "QGD: Alatortsev, 4.Bf4", "1.d4 d5 2.c4 e6 3.Nc3 Be7 4.Bf4 *"),
("D31e", "QGD: Alatortsev, Exchange", "1.d4 d5 2.c4 e6 3.Nc3 Be7 4.cxd5 *"),
("D31e", "QGD: Alatortsev, Exchange", "1.d4 d5 2.c4 e6 3.Nc3 Be7 4.cxd5 exd5 *"),
("D31e", "QGD: Alatortsev, 5.Bf4", "1.d4 d5 2.c4 e6 3.Nc3 Be7 4.cxd5 exd5 5.Bf4 *"),
("D31f", "QGD: Alatortsev, 5.Bf4 c6", "1.d4 d5 2.c4 e6 3.Nc3 Be7 4.cxd5 exd5 5.Bf4 c6 *"),
("D31g", "QGD: Alatortsev, 5.Bf4 c6 6.e3 Bf5", "1.d4 d5 2.c4 e6 3.Nc3 Be7 4.cxd5 exd5 5.Bf4 c6 6.e3 Bf5 *"),
("D31h", "QGD: Alatortsev, 5.Bf4 c6 6.e3 Bf5 7.g4", "1.d4 d5 2.c4 e6 3.Nc3 Be7 4.cxd5 exd5 5.Bf4 c6 6.e3 Bf5 7.g4 *"),
("D31i", "Semi-Slav", "1.d4 d5 2.c4 e6 3.Nc3 c6 *"),
("D31j", "Semi-Slav: 4.cxd5", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.cxd5 *"),
("D31j", "Semi-Slav: 4.cxd5", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.cxd5 exd5 *"),
("D31k", "Semi-Slav: 4.cxd5 exd5 5.Nf3", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.cxd5 exd5 5.Nf3 *"),
("D31l", "Semi-Slav: 4.cxd5 exd5 5.Nf3 Bf5", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.cxd5 exd5 5.Nf3 Bf5 *"),
("D31m", "Semi-Slav: 4.cxd5 exd5 5.Bf4", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.cxd5 exd5 5.Bf4 *"),
("D31n", "Semi-Slav: 4.e3", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.e3 *"),
("D31o", "Semi-Slav: 4.e3 Nf6", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.e3 Nf6 *"),
("D31p", "Semi-Slav: 4.Nf3", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.Nf3 *"),
("D31q", "Semi-Slav: Noteboom Variation", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.Nf3 dxc4 *"),
("D31r", "Semi-Slav: Noteboom, 5.a4", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.Nf3 dxc4 5.a4 *"),
("D31s", "Semi-Slav: Noteboom, 5.a4 Bb4 6.e3 b5", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.Nf3 dxc4 5.a4 Bb4 6.e3 b5 *"),
("D31s", "Semi-Slav: Noteboom, Koomen Variation", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.Nf3 dxc4 5.a4 Bb4 6.e3 b5 7.Bd2 Qe7 *"),
("D31s", "Semi-Slav: Noteboom, Junge Variation", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.Nf3 dxc4 5.a4 Bb4 6.e3 b5 7.Bd2 Qb6 *"),
("D31t", "Semi-Slav: Noteboom, Abrahams Variation", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.Nf3 dxc4 5.a4 Bb4 6.e3 b5 7.Bd2 a5 *"),
("D31u", "Semi-Slav: Noteboom, Abrahams, Main Line", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.Nf3 dxc4 5.a4 Bb4 6.e3 b5 7.Bd2 a5 8.axb5 Bxc3 9.Bxc3 cxb5 10.b3 Bb7 *"),
("D31v", "Semi-Slav: Noteboom, Abrahams, Main Line, 14.O-O O-O", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.Nf3 dxc4 5.a4 Bb4 6.e3 b5 7.Bd2 a5 8.axb5 Bxc3 9.Bxc3 cxb5 10.b3 Bb7 11.bxc4 b4 12.Bb2 Nf6 13.Bd3 Nbd7 14.O-O O-O *"),
("D31w", "Semi-Slav: Marshall Gambit", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.e4 *"),
("D31w", "Semi-Slav: Marshall Gambit, 4...Bb4", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.e4 Bb4 *"),
("D31w", "Semi-Slav: Marshall Gambit, 4...dxe4", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.e4 dxe4 *"),
("D31w", "Semi-Slav: Marshall Gambit, Gunderam Gambit", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.e4 dxe4 5.f3 *"),
("D31w", "Semi-Slav: Marshall Gambit, 5.Nxe4", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.e4 dxe4 5.Nxe4 *"),
("D31w", "Semi-Slav: Marshall Gambit, 5.Nxe4 Nf6", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.e4 dxe4 5.Nxe4 Nf6 *"),
("D31x", "Semi-Slav: Marshall Gambit, 5.Nxe4 Bb4+", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.e4 dxe4 5.Nxe4 Bb4+ *"),
("D31x", "Semi-Slav: Marshall Gambit, 5.Nxe4 Bb4+ 6.Nc3", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.e4 dxe4 5.Nxe4 Bb4+ 6.Nc3 *"),
("D31x", "Semi-Slav: Marshall Gambit, 5.Nxe4 Bb4+ 6.Bd2", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.e4 dxe4 5.Nxe4 Bb4+ 6.Bd2 *"),
("D31x", "Semi-Slav: Marshall Gambit, 8.Ne2", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.e4 dxe4 5.Nxe4 Bb4+ 6.Bd2 Qxd4 7.Bxb4 Qxe4+ 8.Ne2 *"),
("D31y", "Semi-Slav: Marshall Gambit, 8.Be2", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.e4 dxe4 5.Nxe4 Bb4+ 6.Bd2 Qxd4 7.Bxb4 Qxe4+ 8.Be2 *"),
("D31z", "Semi-Slav: Marshall Gambit, 8.Be2 Na6", "1.d4 d5 2.c4 e6 3.Nc3 c6 4.e4 dxe4 5.Nxe4 Bb4+ 6.Bd2 Qxd4 7.Bxb4 Qxe4+ 8.Be2 Na6 *"),
("D32a", "QGD Tarrasch", "1.d4 d5 2.c4 e6 3.Nc3 c5 *"),
("D32a", "QGD Tarrasch: 4.e3", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.e3 *"),
("D32b", "QGD Tarrasch: 4.Nf3", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.Nf3 *"),
("D32c", "QGD Tarrasch: 4.Nf3 cxd4 5.Nxd4 e5", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.Nf3 cxd4 5.Nxd4 e5 *"),
("D32d", "QGD Tarrasch: 4.cxd5", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 *"),
("D32d", "QGD Tarrasch: von Hennig-Schara Gambit", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 cxd4 *"),
("D32d", "QGD Tarrasch: von Hennig-Schara Gambit, 5.Qxd4", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 cxd4 5.Qxd4 *"),
("D32e", "QGD Tarrasch: von Hennig-Schara Gambit, 5.Qxd4 Nc6 6.Qd1 exd5 7.e3", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 cxd4 5.Qxd4 Nc6 6.Qd1 exd5 7.e3 *"),
("D32f", "QGD Tarrasch: von Hennig-Schara Gambit, 5.Qa4+", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 cxd4 5.Qa4+ *"),
("D32f", "QGD Tarrasch: von Hennig-Schara Gambit, 7.Qxd5 Nc6", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 cxd4 5.Qa4+ Bd7 6.Qxd4 exd5 7.Qxd5 Nc6 *"),
("D32g", "QGD Tarrasch: von Hennig-Schara Gambit, 7.Qxd5 Nc6 8.Bg5", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 cxd4 5.Qa4+ Bd7 6.Qxd4 exd5 7.Qxd5 Nc6 8.Bg5 *"),
("D32h", "QGD Tarrasch: von Hennig-Schara Gambit, 7.Qxd5 Nc6 8.Nf3", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 cxd4 5.Qa4+ Bd7 6.Qxd4 exd5 7.Qxd5 Nc6 8.Nf3 *"),
("D32i", "QGD Tarrasch: von Hennig-Schara Gambit, Main Line (11.Be2 O-O-O)", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 cxd4 5.Qa4+ Bd7 6.Qxd4 exd5 7.Qxd5 Nc6 8.Nf3 Nf6 9.Qd1 Bc5 10.e3 Qe7 11.Be2 O-O-O *"),
("D32j", "QGD Tarrasch: 4.cxd5 exd5", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 *"),
("D32j", "QGD Tarrasch: 5.g3", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.g3 *"),
("D32k", "QGD Tarrasch: 5.dxc5", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.dxc5 *"),
("D32k", "QGD Tarrasch: 5.dxc5, Tarrasch Gambit", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.dxc5 d4 6.Na4 b5 *"),
("D32l", "QGD Tarrasch: Marshall Gambit", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.e4 *"),
("D32l", "QGD Tarrasch: Marshall Gambit", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.e4 dxe4 *"),
("D32l", "QGD Tarrasch: Marshall Gambit, Modern 6.Bc4", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.e4 dxe4 6.Bc4 *"),
("D32l", "QGD Tarrasch: Marshall Gambit, 6.d5", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.e4 dxe4 6.d5 *"),
("D32m", "QGD Tarrasch: 5.Nf3", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 *"),
("D32n", "QGD Tarrasch: 5.Nf3 Nc6", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 *"),
("D32o", "QGD Tarrasch: 5.Nf3 Nc6 6.Bf4", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.Bf4 *"),
("D32o", "QGD Tarrasch: 5.Nf3 Nc6 6.Bf4 Nf6", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.Bf4 Nf6 *"),
("D32p", "QGD Tarrasch: 5.Nf3 Nc6 6.Bg5", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.Bg5 *"),
("D32q", "QGD Tarrasch: 5.Nf3 Nc6 6.Bg5 Be7", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.Bg5 Be7 *"),
("D32r", "QGD Tarrasch: 5.Nf3 Nc6 6.e3", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.e3 *"),
("D32r", "QGD Tarrasch: 5.Nf3 Nc6 6.e3 Nf6", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.e3 Nf6 *"),
("D32s", "QGD Tarrasch: 5.Nf3 Nc6 6.e3 Nf6 7.Bb5", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.e3 Nf6 7.Bb5 *"),
("D32t", "QGD Tarrasch: 5.Nf3 Nc6 6.e3 Nf6 7.Be2", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.e3 Nf6 7.Be2 *"),
("D32u", "QGD Tarrasch: 5.Nf3 Nc6 6.e3 Nf6 7.Be2 Be7", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.e3 Nf6 7.Be2 Be7 *"),
("D32u", "QGD Tarrasch: 5.Nf3 Nc6 6.e3 Nf6 7.Be2 Be7 8.dxc5 Bxc5", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.e3 Nf6 7.Be2 Be7 8.dxc5 Bxc5 *"),
("D32v", "QGD Tarrasch: 5.Nf3 Nc6 6.e3 Nf6 7.Be2 cxd4", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.e3 Nf6 7.Be2 cxd4 *"),
("D32v", "QGD Tarrasch: 5.Nf3 Nc6 6.e3 Nf6 7.Be2 cxd4 8.Nxd4 Bd6", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.e3 Nf6 7.Be2 cxd4 8.Nxd4 Bd6 *"),
("D32v", "QGD Tarrasch: 5.Nf3 Nc6 6.e3 Nf6 7.Be2 cxd4 8.Nxd4 Bd6 9.O-O", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.e3 Nf6 7.Be2 cxd4 8.Nxd4 Bd6 9.O-O *"),
("D32w", "QGD Tarrasch: 5.Nf3 Nc6 6.e3 Nf6 7.Be2 cxd4 8.Nxd4 Bd6 9.O-O O-O", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.e3 Nf6 7.Be2 cxd4 8.Nxd4 Bd6 9.O-O O-O *"),
("D32x", "QGD Tarrasch: 5.Nf3 Nc6 6.e3 Nf6 7.Be2 cxd4 8.Nxd4 Bd6 9.O-O O-O 10.b3", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.e3 Nf6 7.Be2 cxd4 8.Nxd4 Bd6 9.O-O O-O 10.b3 *"),
("D33a", "QGD Tarrasch: 6.g3 (Schlecter/Rubinstein)", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 *"),
("D33b", "QGD Tarrasch: 6.g3 cxd4", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 cxd4 *"),
("D33c", "QGD Tarrasch: Swedish Variation", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 c4 *"),
("D33d", "QGD Tarrasch: Swedish, Rey Ardid Variation", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 c4 7.e4 *"),
("D33e", "QGD Tarrasch: Swedish, 7.Bg2", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 c4 7.Bg2 *"),
("D33f", "QGD Tarrasch: Swedish, 7.Bg2 Bb4 8.O-O Nge7", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 c4 7.Bg2 Bb4 8.O-O Nge7 *"),
("D33g", "QGD Tarrasch: Swedish, 9.a3", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 c4 7.Bg2 Bb4 8.O-O Nge7 9.a3 *"),
("D33h", "QGD Tarrasch: Swedish, 9.Bd2", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 c4 7.Bg2 Bb4 8.O-O Nge7 9.Bd2 *"),
("D33i", "QGD Tarrasch: Swedish, 9.Ne5", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 c4 7.Bg2 Bb4 8.O-O Nge7 9.Ne5 *"),
("D33j", "QGD Tarrasch: Swedish, 9.e4", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 c4 7.Bg2 Bb4 8.O-O Nge7 9.e4 *"),
("D33k", "QGD Tarrasch: Swedish, 9.e4 dxe4", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 c4 7.Bg2 Bb4 8.O-O Nge7 9.e4 dxe4 *"),
("D33l", "QGD Tarrasch: 6.g3 Nf6 (Prague)", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 *"),
("D33l", "QGD Tarrasch: 6.g3 Nf6 (Prague)", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 *"),
("D33m", "QGD Tarrasch: 7.Bg2 Be6", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be6 *"),
("D33n", "QGD Tarrasch: Wagner Variation", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Bg4 *"),
("D33o", "QGD Tarrasch: 7.Bg2 cxd4", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 cxd4 *"),
("D33o", "QGD Tarrasch: 7.Bg2 cxd4 8.Nxd4", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 cxd4 8.Nxd4 *"),
("D33p", "QGD Tarrasch: 7.Bg2 cxd4 8.Nxd4 Be7", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 cxd4 8.Nxd4 Be7 *"),
("D33p", "QGD Tarrasch: 7.Bg2 cxd4 8.Nxd4 Be7", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 cxd4 8.Nxd4 Be7 9.O-O *"),
("D33p", "QGD Tarrasch: 7.Bg2 cxd4 8.Nxd4 Be7", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 cxd4 8.Nxd4 Be7 9.O-O O-O *"),
("D33q", "QGD Tarrasch: 7.Bg2 cxd4 8.Nxd4 Be7 9.O-O O-O 10.Be3", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 cxd4 8.Nxd4 Be7 9.O-O O-O 10.Be3 *"),
("D34a", "QGD Tarrasch: 7.Bg2 Be7", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 *"),
("D34a", "QGD Tarrasch: 7.Bg2 Be7 8.O-O", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O *"),
("D34b", "QGD Tarrasch: 7.Bg2 Be7 8.O-O Be6", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O Be6 *"),
("D34c", "QGD Tarrasch: 8.O-O O-O (Main Line)", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O *"),
("D34d", "QGD Tarrasch: 9.b3", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.b3 *"),
("D34e", "QGD Tarrasch: 9.b3 Ne4", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.b3 Ne4 *"),
("D34f", "QGD Tarrasch: 9.Bf4", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.Bf4 *"),
("D34g", "QGD Tarrasch: 9.Be3", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.Be3 *"),
("D34h", "QGD Tarrasch: 9.dxc5", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.dxc5 *"),
("D34h", "QGD Tarrasch: Tarrasch's Gambit", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.dxc5 d4 *"),
("D34i", "QGD Tarrasch: 9.dxc5 Bxc5", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.dxc5 Bxc5 *"),
("D34j", "QGD Tarrasch: Reti Variation", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.dxc5 Bxc5 10.Na4 *"),
("D34k", "QGD Tarrasch: 9.dxc5 Bxc5 10.Bg5", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.dxc5 Bxc5 10.Bg5 *"),
("D34k", "QGD Tarrasch: 9.dxc5 Bxc5 10.Bg5 Be6", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.dxc5 Bxc5 10.Bg5 Be6 *"),
("D34k", "QGD Tarrasch: 9.dxc5 Bxc5 10.Bg5 Be6 11.Bxf6", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.dxc5 Bxc5 10.Bg5 Be6 11.Bxf6 *"),
("D34l", "QGD Tarrasch: 9.dxc5 Bxc5 10.Bg5 d4", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.dxc5 Bxc5 10.Bg5 d4 *"),
("D34m", "QGD Tarrasch: 9.dxc5 Bxc5 10.Bg5 d4 11.Bxf6 Qxf6 12.Nd5", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.dxc5 Bxc5 10.Bg5 d4 11.Bxf6 Qxf6 12.Nd5 *"),
("D34n", "QGD Tarrasch: 9.dxc5 Bxc5 10.Bg5 d4 11.Bxf6: Main Line", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.dxc5 Bxc5 10.Bg5 d4 11.Bxf6 Qxf6 12.Nd5 Qd8 13.Nd2 Re8 *"),
("D34o", "QGD Tarrasch: 9.Bg5", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.Bg5 *"),
("D34o", "QGD Tarrasch: 9.Bg5 Be6", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.Bg5 Be6 *"),
("D34o", "QGD Tarrasch: Stoltz Variation", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.Bg5 Be6 10.Rc1 b6 *"),
("D34o", "QGD Tarrasch: Bogoljubow Variation", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.Bg5 Be6 10.Rc1 c4 *"),
("D34p", "QGD Tarrasch: 9.Bg5 c4", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.Bg5 c4 *"),
("D34q", "QGD Tarrasch: 9.Bg5 c4 10.Ne5 Be6 11.Nxc6", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.Bg5 c4 10.Ne5 Be6 11.Nxc6 *"),
("D34r", "QGD Tarrasch: 9.Bg5 cxd4", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.Bg5 cxd4 *"),
("D34r", "QGD Tarrasch: 9.Bg5 cxd4", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.Bg5 cxd4 10.Nxd4 *"),
("D34s", "QGD Tarrasch: 9.Bg5 cxd4 10.Nxd4 h6", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.Bg5 cxd4 10.Nxd4 h6 *"),
("D34t", "QGD Tarrasch: 9.Bg5 cxd4 10.Nxd4 h6 11.Be3", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.Bg5 cxd4 10.Nxd4 h6 11.Be3 *"),
("D34u", "QGD Tarrasch: 9.Bg5 cxd4 10.Nxd4 h6 11.Be3 Re8", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.Bg5 cxd4 10.Nxd4 h6 11.Be3 Re8 *"),
("D34v", "QGD Tarrasch: 9.Bg5 cxd4 10.Nxd4 h6 11.Be3 Re8 12.Qb3", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.Bg5 cxd4 10.Nxd4 h6 11.Be3 Re8 12.Qb3 *"),
("D34w", "QGD Tarrasch: 9.Bg5 cxd4 10.Nxd4 h6 11.Be3 Re8 12.Rc1", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.Bg5 cxd4 10.Nxd4 h6 11.Be3 Re8 12.Rc1 *"),
("D34x", "QGD Tarrasch: 9.Bg5 cxd4 10.Nxd4 h6 11.Be3 Re8 12.Rc1 Bf8", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.Bg5 cxd4 10.Nxd4 h6 11.Be3 Re8 12.Rc1 Bf8 *"),
("D34y", "QGD Tarrasch: 9.Bg5 cxd4 10.Nxd4 h6 11.Be3 Re8 12.Rc1 Bf8 13.Nxc6", "1.d4 d5 2.c4 e6 3.Nc3 c5 4.cxd5 exd5 5.Nf3 Nc6 6.g3 Nf6 7.Bg2 Be7 8.O-O O-O 9.Bg5 cxd4 10.Nxd4 h6 11.Be3 Re8 12.Rc1 Bf8 13.Nxc6 *"),
("D35a", "QGD: 3.Nc3 Nf6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 *"),
("D35a", "QGD: 3.Nc3 Nf6 4.e3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.e3 *"),
("D35a", "QGD: Harrwitz Attack", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bf4 *"),
("D35a", "QGD: Catalan without Nf3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.g3 *"),
("D35b", "QGD: Exchange", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 *"),
("D35b", "QGD: Exchange, 4...Nxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 Nxd5 *"),
("D35b", "QGD: Exchange", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 *"),
("D35c", "QGD: Exchange, Saemisch Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bf4 *"),
("D35d", "QGD: Exchange, 5.Nf3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Nf3 *"),
("D35d", "QGD: Exchange, 5.Nf3 c6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Nf3 c6 *"),
("D35d", "QGD: Exchange, 5.Nf3 c6 6.e3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Nf3 c6 6.e3 *"),
("D35e", "QGD: Exchange, 5.Nf3 Nbd7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Nf3 Nbd7 *"),
("D35e", "QGD: Exchange, 5.Nf3 Nbd7 6.e3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Nf3 Nbd7 6.e3 *"),
("D35e", "QGD: Exchange, 5.Nf3 Nbd7 6.e3 c6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Nf3 Nbd7 6.e3 c6 *"),
("D35f", "QGD: Exchange, 5.Nf3 Nbd7 6.Bf4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Nf3 Nbd7 6.Bf4 *"),
("D35g", "QGD: Exchange, 5.Nf3 Be7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Nf3 Be7 *"),
("D35g", "QGD: Exchange, 5.Nf3 Be7 6.Bf4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Nf3 Be7 6.Bf4 *"),
("D35h", "QGD: Exchange, 5.Nf3 Be7 6.Bf4 c6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Nf3 Be7 6.Bf4 c6 *"),
("D35h", "QGD: Exchange, 5.Nf3 Be7 6.Bf4 c6 7.Qc2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Nf3 Be7 6.Bf4 c6 7.Qc2 *"),
("D35h", "QGD: Exchange, 5.Nf3 Be7 6.Bf4 c6 7.Qc2 Nbd7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Nf3 Be7 6.Bf4 c6 7.Qc2 Nbd7 *"),
("D35i", "QGD: Exchange, 5.Bg5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 *"),
("D35i", "QGD: Exchange, 5.Bg5 Nbd7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 Nbd7 *"),
("D35i", "QGD: Exchange, 5.Bg5 Nbd7 6.Nf3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 Nbd7 6.Nf3 *"),
("D35i", "QGD: Exchange, 5.Bg5 Nbd7 6.e3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 Nbd7 6.e3 *"),
("D35i", "QGD: Exchange, 5.Bg5 Be7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 Be7 *"),
("D35i", "QGD: Exchange, 5.Bg5 Be7 6.Nf3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 Be7 6.Nf3 *"),
("D35j", "QGD: Exchange, 5.Bg5 Be7 6.e3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 Be7 6.e3 *"),
("D35j", "QGD: Exchange, 5.Bg5 Be7 6.e3 h6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 Be7 6.e3 h6 *"),
("D35j", "QGD: Exchange, 5.Bg5 Be7 6.e3 Nbd7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 Be7 6.e3 Nbd7 *"),
("D35k", "QGD: Exchange, 5.Bg5 Be7 6.e3 O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 Be7 6.e3 O-O *"),
("D35l", "QGD: Exchange, 5.Bg5 c6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 *"),
("D35l", "QGD: Exchange, 5.Bg5 c6 6.Nf3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Nf3 *"),
("D35m", "QGD: Exchange, 5.Bg5 c6 6.e3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.e3 *"),
("D35n", "QGD: Exchange, 5.Bg5 c6 6.e3 Nbd7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.e3 Nbd7 *"),
("D35o", "QGD: Exchange, 5.Bg5 c6 6.e3 Be7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.e3 Be7 *"),
("D35o", "QGD: Exchange, 5.Bg5 c6 6.e3 Be7 7.Nf3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.e3 Be7 7.Nf3 *"),
("D35p", "QGD: Exchange, 5.Bg5 c6 6.e3 Be7 7.Nf3 Bf5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.e3 Be7 7.Nf3 Bf5 *"),
("D35q", "QGD: Exchange, 5.Bg5 c6 6.e3 Be7 7.Nf3 Bf5 8.Bd3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.e3 Be7 7.Nf3 Bf5 8.Bd3 *"),
("D35r", "QGD: Exchange, 5.Bg5 c6 6.e3 Be7 7.Bd3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.e3 Be7 7.Bd3 *"),
("D35s", "QGD: Exchange, 5.Bg5 c6 6.e3 Be7 7.Bd3 Ne4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.e3 Be7 7.Bd3 Ne4 *"),
("D35t", "QGD: Exchange, 5.Bg5 c6 6.e3 Be7 7.Bd3 Nbd7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.e3 Be7 7.Bd3 Nbd7 *"),
("D35t", "QGD: Exchange, 5.Bg5 c6 6.e3 Be7 7.Bd3 Nbd7 8.Nge2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.e3 Be7 7.Bd3 Nbd7 8.Nge2 *"),
("D35u", "QGD: Exchange, 5.Bg5 c6 6.e3 Be7 7.Bd3 Nbd7 8.Nf3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.e3 Be7 7.Bd3 Nbd7 8.Nf3 *"),
("D35v", "QGD: Exchange, 5.Bg5 c6 6.e3 Be7 7.Bd3 O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.e3 Be7 7.Bd3 O-O *"),
("D35v", "QGD: Exchange, 5.Bg5 c6 6.e3 Be7 7.Bd3 O-O 8.Nf3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.e3 Be7 7.Bd3 O-O 8.Nf3 *"),
("D35v", "QGD: Exchange, 5.Bg5 c6 6.e3 Be7 7.Bd3 O-O 8.Nge2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.e3 Be7 7.Bd3 O-O 8.Nge2 *"),
("D36a", "QGD: Exchange, 5.Bg5 c6 6.Qc2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 *"),
("D36a", "QGD: Exchange, 5.Bg5 c6 6.Qc2 g6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 g6 *"),
("D36b", "QGD: Exchange, 5.Bg5 c6 6.Qc2 Na6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Na6 *"),
("D36c", "QGD: Exchange, 6.Qc2 Be7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 *"),
("D36c", "QGD: Exchange, 6.Qc2 Be7 7.Nf3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.Nf3 *"),
("D36c", "QGD: Exchange, 6.Qc2 Be7 7.Nf3 Nbd7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.Nf3 Nbd7 *"),
("D36c", "QGD: Exchange, 6.Qc2 Be7 7.Nf3 O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.Nf3 O-O *"),
("D36d", "QGD: Exchange, 6.Qc2 Be7 7.Nf3 g6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.Nf3 g6 *"),
("D36e", "QGD: Exchange, 6.Qc2 Be7 7.Nf3 g6 8.e3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.Nf3 g6 8.e3 *"),
("D36f", "QGD: Exchange, 6.Qc2 Be7 7.e3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 *"),
("D36f", "QGD: Exchange, 6.Qc2 Be7 7.e3 O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 O-O *"),
("D36f", "QGD: Exchange, 6.Qc2 Be7 7.e3 O-O 8.Bd3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 O-O 8.Bd3 *"),
("D36g", "QGD: Exchange, 6.Qc2 Be7 7.e3 h6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 h6 *"),
("D36g", "QGD: Exchange, 6.Qc2 Be7 7.e3 h6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 h6 8.Bh4 O-O *"),
("D36g", "QGD: Exchange, 6.Qc2 Be7 7.e3 h6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 h6 8.Bh4 O-O 9.Bd3 Re8 *"),
("D36h", "QGD: Exchange, 6.Qc2 Be7 7.e3 Nbd7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 *"),
("D36h", "QGD: Exchange, 6.Qc2 Be7 7.e3 Nbd7 8.Nf3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Nf3 *"),
("D36i", "QGD: Exchange, 6.Qc2 Be7 7.e3 Nbd7 8.Bd3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 *"),
("D36j", "QGD: Exchange, 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 Nf8", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 Nf8 *"),
("D36k", "QGD: Exchange, Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O *"),
("D36k", "QGD: Exchange, Main Line, 9.Nge2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nge2 *"),
("D36k", "QGD: Exchange, Main Line, 9.Nge2 h6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nge2 h6 *"),
("D36k", "QGD: Exchange, Main Line, 9.Nge2 h6 10.Bh4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nge2 h6 10.Bh4 *"),
("D36l", "QGD: Exchange, Main Line, 9.Nge2 Re8", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nge2 Re8 *"),
("D36m", "QGD: Exchange, Main Line, 9.Nge2 Re8 10.O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nge2 Re8 10.O-O *"),
("D36m", "QGD: Exchange, Main Line, 9.Nge2 Re8 10.O-O Nf8", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nge2 Re8 10.O-O Nf8 *"),
("D36n", "QGD: Exchange, Main Line, 9.Nge2 Re8 10.O-O Nf8 11.f3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nge2 Re8 10.O-O Nf8 11.f3 *"),
("D36o", "QGD: Exchange, Main Line, 9.Nf3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nf3 *"),
("D36o", "QGD: Exchange, Main Line, 9.Nf3 h6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nf3 h6 *"),
("D36o", "QGD: Exchange, Main Line, 9.Nf3 h6 10.Bh4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nf3 h6 10.Bh4 *"),
("D36p", "QGD: Exchange, Main Line, 9.Nf3 Re8", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nf3 Re8 *"),
("D36p", "QGD: Exchange, Main Line, 9.Nf3 Re8 10.h3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nf3 Re8 10.h3 *"),
("D36p", "QGD: Exchange, Main Line, 9.Nf3 Re8 10.h3 Nf8", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nf3 Re8 10.h3 Nf8 *"),
("D36q", "QGD: Exchange, Main Line, 9.Nf3 Re8 10.h3 Nf8 11.Bf4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nf3 Re8 10.h3 Nf8 11.Bf4 *"),
("D36r", "QGD: Exchange, Main Line, 9.Nf3 Re8 10.O-O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nf3 Re8 10.O-O-O *"),
("D36r", "QGD: Exchange, Main Line, 9.Nf3 Re8 10.O-O-O h6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nf3 Re8 10.O-O-O h6 11.Bh4 *"),
("D36r", "QGD: Exchange, Main Line, 9.Nf3 Re8 10.O-O-O Nf8", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nf3 Re8 10.O-O-O Nf8 *"),
("D36r", "QGD: Exchange, Main Line, 9.Nf3 Re8 10.O-O-O Nf8 11.h3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nf3 Re8 10.O-O-O Nf8 11.h3 *"),
("D36s", "QGD: Exchange, Main Line, 9.Nf3 Re8 10.O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nf3 Re8 10.O-O *"),
("D36t", "QGD: Exchange, Main Line, 9.Nf3 Re8 10.O-O h6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nf3 Re8 10.O-O h6 *"),
("D36t", "QGD: Exchange, Main Line, 9.Nf3 Re8 10.O-O h6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nf3 Re8 10.O-O h6 11.Bf4 *"),
("D36t", "QGD: Exchange, Main Line, 9.Nf3 Re8 10.O-O h6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nf3 Re8 10.O-O h6 11.Bh4 *"),
("D36u", "QGD: Exchange, Main Line, 9.Nf3 Re8 10.O-O Nf8", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nf3 Re8 10.O-O Nf8 *"),
("D36v", "QGD: Exchange, Main Line, 9.Nf3 Re8 10.O-O Nf8 11.Rae1", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nf3 Re8 10.O-O Nf8 11.Rae1 *"),
("D36w", "QGD: Exchange, Main Line, 9.Nf3 Re8 10.O-O Nf8 11.h3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nf3 Re8 10.O-O Nf8 11.h3 *"),
("D36x", "QGD: Exchange, Main Line, 9.Nf3 Re8 10.O-O Nf8 11.h3 g6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nf3 Re8 10.O-O Nf8 11.h3 g6 *"),
("D36y", "QGD: Exchange, Main Line, 9.Nf3 Re8 10.O-O Nf8 11.Rab1", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nf3 Re8 10.O-O Nf8 11.Rab1 *"),
("D36z", "QGD: Exchange, Main Line, 9.Nf3 Re8 10.O-O Nf8 11.Rab1 a5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.cxd5 exd5 5.Bg5 c6 6.Qc2 Be7 7.e3 Nbd7 8.Bd3 O-O 9.Nf3 Re8 10.O-O Nf8 11.Rab1 a5 *"),
("D37a", "QGD: 4.Nf3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 *"),
("D37b", "QGD: Westphalia without Bg5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Nbd7 *"),
("D37c", "QGD: Westphalia: 4.Nf3 Nbd7 5.Bf4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Nbd7 5.Bf4 *"),
("D37d", "QGD: 4.Nf3 Be7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Be7 *"),
("D37d", "QGD: 4.Nf3 Be7 5.cxd5 Nxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Be7 5.cxd5 Nxd5 *"),
("D37e", "QGD: 4.Nf3 Be7 5.e3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Be7 5.e3 *"),
("D37f", "QGD: 4.Nf3 Be7 5.e3 O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Be7 5.e3 O-O *"),
("D37f", "QGD: 4.Nf3 Be7 5.e3 O-O 6.b3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Be7 5.e3 O-O 6.b3 *"),
("D37g", "QGD: 4.Nf3 Be7 5.e3 O-O 6.Bd3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Be7 5.e3 O-O 6.Bd3 *"),
("D37h", "QGD: Classical Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Be7 5.Bf4 *"),
("D37i", "QGD: Classical, 5...O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Be7 5.Bf4 O-O *"),
("D37j", "QGD: Classical, 5...O-O 6.e3 Nbd7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Be7 5.Bf4 O-O 6.e3 Nbd7 *"),
("D37k", "QGD: Classical, 5...O-O 6.e3 b6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Be7 5.Bf4 O-O 6.e3 b6 *"),
("D37l", "QGD: Classical, 5...O-O 6.e3 c6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Be7 5.Bf4 O-O 6.e3 c6 *"),
("D37m", "QGD: Classical, 5...O-O 6.e3 c5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Be7 5.Bf4 O-O 6.e3 c5 *"),
("D37n", "QGD: Classical, 5...O-O 6.e3 c5 7.dxc5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Be7 5.Bf4 O-O 6.e3 c5 7.dxc5 *"),
("D37o", "QGD: Classical, 5...O-O 6.e3 c5 7.dxc5 Bxc5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Be7 5.Bf4 O-O 6.e3 c5 7.dxc5 Bxc5 *"),
("D37p", "QGD: Classical, 5...O-O 6.e3 c5 7.dxc5 Bxc5 8.a3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Be7 5.Bf4 O-O 6.e3 c5 7.dxc5 Bxc5 8.a3 *"),
("D37q", "QGD: Classical, 5...O-O 6.e3 c5 7.dxc5 Bxc5 8.cxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Be7 5.Bf4 O-O 6.e3 c5 7.dxc5 Bxc5 8.cxd5 *"),
("D37r", "QGD: Classical, 5...O-O 6.e3 c5 7.dxc5 Bxc5 8.Qc2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Be7 5.Bf4 O-O 6.e3 c5 7.dxc5 Bxc5 8.Qc2 *"),
("D37s", "QGD: Classical, 5...O-O 6.e3 c5 7.dxc5 Bxc5 8.Qc2 Nc6 9.a3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Be7 5.Bf4 O-O 6.e3 c5 7.dxc5 Bxc5 8.Qc2 Nc6 9.a3 *"),
("D37t", "QGD: Classical, 5...O-O 6.e3 c5 7.dxc5 Bxc5 8.Qc2 Nc6 9.a3 Qa5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Be7 5.Bf4 O-O 6.e3 c5 7.dxc5 Bxc5 8.Qc2 Nc6 9.a3 Qa5 *"),
("D37u", "QGD: Classical, 5...O-O 6.e3 c5 7.dxc5 Bxc5 8.Qc2 Nc6 9.a3 Qa5 10.Rd1", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Be7 5.Bf4 O-O 6.e3 c5 7.dxc5 Bxc5 8.Qc2 Nc6 9.a3 Qa5 10.Rd1 *"),
("D37v", "QGD: Classical, 5...O-O 6.e3 c5 7.dxc5 Bxc5 8.Qc2 Nc6 9.a3 Qa5 10.O-O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Be7 5.Bf4 O-O 6.e3 c5 7.dxc5 Bxc5 8.Qc2 Nc6 9.a3 Qa5 10.O-O-O *"),
("D38a", "QGD: Ragozin", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 *"),
("D38b", "QGD: Ragozin, 5.Qa4+", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Qa4+ *"),
("D38c", "QGD: Ragozin, 5.Qa4+ Nc6 6.cxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Qa4+ Nc6 6.cxd5 *"),
("D38d", "QGD: Ragozin, 5.Qa4+ Nc6 6.cxd5 exd5 7.Bg5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Qa4+ Nc6 6.cxd5 exd5 7.Bg5 *"),
("D38e", "QGD: Ragozin, 5.cxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.cxd5 *"),
("D38f", "QGD: Ragozin, 5.cxd5 exd5 6.Bg5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.cxd5 exd5 6.Bg5 *"),
("D38g", "QGD: Ragozin, 5.cxd5 exd5 6.Bg5 h6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.cxd5 exd5 6.Bg5 h6 *"),
("D38g", "QGD: Ragozin, 5.cxd5 exd5 6.Bg5 h6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.cxd5 exd5 6.Bg5 h6 7.Bh4 *"),
("D38h", "QGD: Ragozin, 5.Bg5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Bg5 *"),
("D38i", "QGD: Ragozin, Westphalia Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Bg5 Nbd7 *"),
("D38j", "QGD: Ragozin, Westphalia, 7.Qc2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Bg5 Nbd7 6.cxd5 exd5 7.Qc2 *"),
("D38k", "QGD: Ragozin, Westphalia, 7.e3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Bg5 Nbd7 6.cxd5 exd5 7.e3 *"),
("D38l", "QGD: Ragozin, 5.Bg5 h6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Bg5 h6 *"),
("D38m", "QGD: Ragozin, 5.Bg5 h6 6.Bxf6 Qxf6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Bg5 h6 6.Bxf6 Qxf6 *"),
("D38n", "QGD: Ragozin, 5.Bg5 h6 6.Bxf6 Qxf6 7.cxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Bg5 h6 6.Bxf6 Qxf6 7.cxd5 *"),
("D38n", "QGD: Ragozin, 5.Bg5 h6 6.Bxf6 Qxf6 7.cxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Bg5 h6 6.Bxf6 Qxf6 7.cxd5 exd5 *"),
("D38o", "QGD: Ragozin, 5.Bg5 h6 6.Bxf6 Qxf6 7.e3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Bg5 h6 6.Bxf6 Qxf6 7.e3 *"),
("D38p", "QGD: Ragozin, 5.Bg5 h6 6.Bxf6 Qxf6 7.e3 O-O 8.Rc1", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Bg5 h6 6.Bxf6 Qxf6 7.e3 O-O 8.Rc1 *"),
("D39a", "QGD: Ragozin, Vienna Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Bg5 dxc4 *"),
("D39b", "QGD: Ragozin, Vienna, 6.Qa4+", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Bg5 dxc4 6.Qa4+ *"),
("D39c", "QGD: Ragozin, Vienna, 6.e4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Bg5 dxc4 6.e4 *"),
("D39d", "QGD: Ragozin, Vienna, 6.e4 c5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Bg5 dxc4 6.e4 c5 *"),
("D39e", "QGD: Ragozin, Vienna, 6.e4 c5 7.e5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Bg5 dxc4 6.e4 c5 7.e5 *"),
("D39f", "QGD: Ragozin, Vienna, 6.e4 c5 7.e5 cxd4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Bg5 dxc4 6.e4 c5 7.e5 cxd4 *"),
("D39g", "QGD: Ragozin, Vienna, 6.e4 c5 7.e5 cxd4 8.Qa4+ Nc6 9.O-O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Bg5 dxc4 6.e4 c5 7.e5 cxd4 8.Qa4+ Nc6 9.O-O-O *"),
("D39h", "QGD: Ragozin, Vienna, 6.e4 c5 7.Bxc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Bg5 dxc4 6.e4 c5 7.Bxc4 *"),
("D39h", "QGD: Ragozin, Vienna, 6.e4 c5 7.Bxc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Bg5 dxc4 6.e4 c5 7.Bxc4 cxd4 *"),
("D39h", "QGD: Ragozin, Vienna, 6.e4 c5 7.Bxc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Bg5 dxc4 6.e4 c5 7.Bxc4 cxd4 8.Nxd4 *"),
("D39i", "QGD: Ragozin, Vienna, 6.e4 c5 7.Bxc4, 8...Qa5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Bg5 dxc4 6.e4 c5 7.Bxc4 cxd4 8.Nxd4 Qa5 *"),
("D39j", "QGD: Ragozin, Vienna, 6.e4 c5 7.Bxc4, 8...Bxc3+", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Bg5 dxc4 6.e4 c5 7.Bxc4 cxd4 8.Nxd4 Bxc3+ *"),
("D39j", "QGD: Ragozin, Vienna, 6.e4 c5 7.Bxc4, 8...Bxc3+", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Bg5 dxc4 6.e4 c5 7.Bxc4 cxd4 8.Nxd4 Bxc3+ 9.bxc3 *"),
("D39k", "QGD: Ragozin, Vienna, 6.e4 c5 7.Bxc4, 8...Bxc3+, 10.Bb5+", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 Bb4 5.Bg5 dxc4 6.e4 c5 7.Bxc4 cxd4 8.Nxd4 Bxc3+ 9.bxc3 Qa5 10.Bb5+ *"),
("D40a", "QGD: Semi-Tarrasch", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 *"),
("D40a", "QGD: Semi-Tarrasch, 5.e3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.e3 *"),
("D40b", "QGD: Semi-Tarrasch, 5.e3 Be7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.e3 Be7 *"),
("D40c", "QGD: Semi-Tarrasch, 5.e3 cxd4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.e3 cxd4 *"),
// Here, 4.e3 cxd4 5.cxd4 is a Panov-Botvinnik with 5...e6 6.Nf3
("D40d", "QGD: Semi-Tarrasch, 5.e3 a6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.e3 a6 *"),
("D40e", "QGD: Semi-Tarrasch, 5.e3 a6 6.cxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.e3 a6 6.cxd5 *"),
("D40e", "QGD: Semi-Tarrasch, 5.e3 a6 6.cxd5 exd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.e3 a6 6.cxd5 exd5 *"),
("D40f", "QGD: Semi-Tarrasch, 5.e3 Nc6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.e3 Nc6 *"),
("D40g", "QGD: Semi-Tarrasch, 5.e3 Nc6 6.Bd3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.e3 Nc6 6.Bd3 *"),
("D40h", "QGD: Semi-Tarrasch, Symmetrical Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.e3 Nc6 6.Bd3 Bd6 7.O-O O-O *"),
("D40i", "QGD: Semi-Tarrasch, 5.e3 Nc6 6.a3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.e3 Nc6 6.a3 *"),
("D40j", "QGD: Semi-Tarrasch, 5.e3 Nc6 6.a3 Ne4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.e3 Nc6 6.a3 Ne4 *"),
("D40k", "QGD: Semi-Tarrasch, 5.e3 Nc6 6.a3 dxc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.e3 Nc6 6.a3 dxc4 *"),
("D40l", "QGD: Semi-Tarrasch, 5.e3 Nc6 6.a3 cxd4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.e3 Nc6 6.a3 cxd4 *"),
("D40l", "QGD: Semi-Tarrasch, 5.e3 Nc6 6.a3 cxd4 7.exd4 Be7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.e3 Nc6 6.a3 cxd4 7.exd4 Be7 *"),
("D40m", "QGD: Semi-Tarrasch, 5.e3 Nc6 6.a3 cxd4 7.exd4 Be7 8.c5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.e3 Nc6 6.a3 cxd4 7.exd4 Be7 8.c5 *"),
("D40n", "QGD: Semi-Tarrasch, 5.e3 Nc6 6.a3 cxd4 7.exd4 Be7 8.Bd3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.e3 Nc6 6.a3 cxd4 7.exd4 Be7 8.Bd3 *"),
("D40o", "QGD: Semi-Tarrasch, 5.e3 Nc6 6.a3 a6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.e3 Nc6 6.a3 a6 *"),
("D40p", "QGD: Semi-Tarrasch, 5.e3 Nc6 6.a3 a6 7.b3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.e3 Nc6 6.a3 a6 7.b3 *"),
("D40q", "QGD: Semi-Tarrasch, 5.e3 Nc6 6.a3 a6 7.dxc5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.e3 Nc6 6.a3 a6 7.dxc5 *"),
("D40q", "QGD: Semi-Tarrasch, 5.e3 Nc6 6.a3 a6 7.dxc5 Bxc5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.e3 Nc6 6.a3 a6 7.dxc5 Bxc5 *"),
("D40r", "QGD: Semi-Tarrasch, 5.e3 Nc6 6.a3 a6 7.dxc5 Bxc5 8.b4 Ba7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.e3 Nc6 6.a3 a6 7.dxc5 Bxc5 8.b4 Ba7 *"),
("D41a", "QGD: Semi-Tarrasch, 5.cxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 *"),
("D41b", "QGD: Semi-Tarrasch, Keres Counterattack", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 cxd4 *"),
("D41c", "QGD: Semi-Tarrasch, 5.cxd5 Nxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 *"),
("D41c", "QGD: Semi-Tarrasch, 6.g3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.g3 *"),
("D41c", "QGD: Semi-Tarrasch, 6.g3 Nc6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.g3 Nc6 *"),
("D41c", "QGD: Semi-Tarrasch, 6.g3 Nc6 7.Bg2 Be7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.g3 Nc6 7.Bg2 Be7 *"),
("D41d", "QGD: Semi-Tarrasch, 6.g3, Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.g3 Nc6 7.Bg2 Be7 8.O-O O-O *"),
("D41e", "QGD: Semi-Tarrasch, 6.g3, Main Line, 9.e4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.g3 Nc6 7.Bg2 Be7 8.O-O O-O 9.e4 *"),
("D41f", "QGD: Semi-Tarrasch, 6.g3, Main Line, 9.e4 Nb6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.g3 Nc6 7.Bg2 Be7 8.O-O O-O 9.e4 Nb6 *"),
("D41g", "QGD: Semi-Tarrasch, 6.g3, Main Line, 9.e4 Ndb4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.g3 Nc6 7.Bg2 Be7 8.O-O O-O 9.e4 Ndb4 *"),
("D41h", "QGD: Semi-Tarrasch, 6.g3, Main Line, 9.Nxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.g3 Nc6 7.Bg2 Be7 8.O-O O-O 9.Nxd5 *"),
("D41i", "QGD: Semi-Tarrasch, 6.g3, Main Line, 9.Nxd5 exd5 10.dxc5 Bxc5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.g3 Nc6 7.Bg2 Be7 8.O-O O-O 9.Nxd5 exd5 10.dxc5 Bxc5 *"),
("D41j", "QGD: Semi-Tarrasch, 6.g3, Main Line, 9.Nxd5 exd5 10.dxc5 Bxc5 11.Bg5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.g3 Nc6 7.Bg2 Be7 8.O-O O-O 9.Nxd5 exd5 10.dxc5 Bxc5 11.Bg5 *"),
("D41k", "QGD: Semi-Tarrasch, 6.e4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e4 *"),
("D41k", "QGD: Semi-Tarrasch, 6.e4 Nxc3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e4 Nxc3 *"),
("D41k", "QGD: Semi-Tarrasch, 6.e4, 8.cxd4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e4 Nxc3 7.bxc3 cxd4 8.cxd4 *"),
("D41l", "QGD: Semi-Tarrasch, 6.e4, 8.cxd4 Nc6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e4 Nxc3 7.bxc3 cxd4 8.cxd4 Nc6 *"),
("D41m", "QGD: Semi-Tarrasch, 6.e4, 8.cxd4 Bb4+", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e4 Nxc3 7.bxc3 cxd4 8.cxd4 Bb4+ *"),
("D41m", "QGD: Semi-Tarrasch, 6.e4, San Sebastian Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e4 Nxc3 7.bxc3 cxd4 8.cxd4 Bb4+ 9.Bd2 Qa5 *"),
("D41m", "QGD: Semi-Tarrasch, 6.e4, Kmoch Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e4 Nxc3 7.bxc3 cxd4 8.cxd4 Bb4+ 9.Bd2 Bxd2+ 10.Qxd2 O-O 11.Bb5 *"),
("D41n", "QGD: Semi-Tarrasch, 6.e4, Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e4 Nxc3 7.bxc3 cxd4 8.cxd4 Bb4+ 9.Bd2 Bxd2+ 10.Qxd2 O-O 11.Bc4 *"),
("D41n", "QGD: Semi-Tarrasch, 6.e4, Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e4 Nxc3 7.bxc3 cxd4 8.cxd4 Bb4+ 9.Bd2 Bxd2+ 10.Qxd2 O-O 11.Bc4 Nc6 12.O-O *"),
("D41o", "QGD: Semi-Tarrasch, 6.e4, Main Line, 12...b6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e4 Nxc3 7.bxc3 cxd4 8.cxd4 Bb4+ 9.Bd2 Bxd2+ 10.Qxd2 O-O 11.Bc4 Nc6 12.O-O b6 *"),
("D41o", "QGD: Semi-Tarrasch, 6.e4, Main Line, 12...b6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e4 Nxc3 7.bxc3 cxd4 8.cxd4 Bb4+ 9.Bd2 Bxd2+ 10.Qxd2 O-O 11.Bc4 Nc6 12.O-O b6 13.Rad1 *"),
("D41p", "QGD: Semi-Tarrasch, 6.e3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 *"),
("D41p", "QGD: Semi-Tarrasch, 6.e3 cxd4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 cxd4 *"),
("D41p", "QGD: Semi-Tarrasch, 6.e3 cxd4 7.exd4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 cxd4 7.exd4 *"),
("D41q", "QGD: Semi-Tarrasch, 6.e3 Nc6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 *"),
("D41r", "QGD: Semi-Tarrasch, 6.e3 Nc6 7.Bc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bc4 *"),
("D41r", "QGD: Semi-Tarrasch, 6.e3 Nc6 7.Bc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bc4 cxd4 8.exd4 *"),
("D41r", "QGD: Semi-Tarrasch, 6.e3 Nc6 7.Bc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bc4 cxd4 8.exd4 Be7 *"),
("D41s", "QGD: Semi-Tarrasch, 6.e3 Nc6 7.Bc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bc4 cxd4 8.exd4 Be7 9.O-O O-O *"),
("D41t", "QGD: Semi-Tarrasch, 7.Bc4 Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bc4 cxd4 8.exd4 Be7 9.O-O O-O 10.Re1 Nxc3 11.bxc3 *"),
("D41t", "QGD: Semi-Tarrasch, 7.Bc4 Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bc4 cxd4 8.exd4 Be7 9.O-O O-O 10.Re1 Nxc3 11.bxc3 b6 12.Bd3 Bb7 *"),
("D41u", "QGD: Semi-Tarrasch, 7.Bc4 Main Line, 13.Qc2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bc4 cxd4 8.exd4 Be7 9.O-O O-O 10.Re1 Nxc3 11.bxc3 b6 12.Bd3 Bb7 13.Qc2 *"),
("D41v", "QGD: Semi-Tarrasch, 7.Bc4 Main Line, 13.h4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bc4 cxd4 8.exd4 Be7 9.O-O O-O 10.Re1 Nxc3 11.bxc3 b6 12.Bd3 Bb7 13.h4 *"),
("D42a", "QGD: Semi-Tarrasch, 6.e3 Nc6 7.Bd3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bd3 *"),
("D42b", "QGD: Semi-Tarrasch, 6.e3 Nc6 7.Bd3 Be7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bd3 Be7 *"),
("D42c", "QGD: Semi-Tarrasch, 6.e3 Nc6 7.Bd3 Be7 8.O-O O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bd3 Be7 8.O-O O-O *"),
("D42d", "QGD: Semi-Tarrasch, 6.e3 Nc6 7.Bd3 cxd4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bd3 cxd4 *"),
("D42d", "QGD: Semi-Tarrasch, 6.e3 Nc6 7.Bd3 cxd4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bd3 cxd4 8.exd4 *"),
("D42e", "QGD: Semi-Tarrasch, 7.Bd3 Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bd3 cxd4 8.exd4 Be7 *"),
("D42f", "QGD: Semi-Tarrasch, 7.Bd3 Main Line, 9.O-O O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bd3 cxd4 8.exd4 Be7 9.O-O O-O *"),
("D42g", "QGD: Semi-Tarrasch, 7.Bd3 Main Line, 10.a3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bd3 cxd4 8.exd4 Be7 9.O-O O-O 10.a3 *"),
("D42h", "QGD: Semi-Tarrasch, 7.Bd3 Main Line, 10.a3 Nf6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bd3 cxd4 8.exd4 Be7 9.O-O O-O 10.a3 Nf6 *"),
("D42i", "QGD: Semi-Tarrasch, 7.Bd3 Main Line, 10.a3 Bf6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bd3 cxd4 8.exd4 Be7 9.O-O O-O 10.a3 Bf6 *"),
("D42j", "QGD: Semi-Tarrasch, 7.Bd3 Main Line, 10.Re1", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bd3 cxd4 8.exd4 Be7 9.O-O O-O 10.Re1 *"),
("D42k", "QGD: Semi-Tarrasch, 7.Bd3 Main Line, 10.Re1 Ncb4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bd3 cxd4 8.exd4 Be7 9.O-O O-O 10.Re1 Ncb4 *"),
("D42l", "QGD: Semi-Tarrasch, 7.Bd3 Main Line, 10.Re1 Nf6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bd3 cxd4 8.exd4 Be7 9.O-O O-O 10.Re1 Nf6 *"),
("D42l", "QGD: Semi-Tarrasch, 7.Bd3 Main Line, 10.Re1 Nf6 11.Bg5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bd3 cxd4 8.exd4 Be7 9.O-O O-O 10.Re1 Nf6 11.Bg5 *"),
("D42m", "QGD: Semi-Tarrasch, 7.Bd3 Main Line, 10.Re1 Nf6 11.a3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bd3 cxd4 8.exd4 Be7 9.O-O O-O 10.Re1 Nf6 11.a3 *"),
("D42m", "QGD: Semi-Tarrasch, 7.Bd3 Main Line, 10.Re1 Nf6 11.a3 b6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bd3 cxd4 8.exd4 Be7 9.O-O O-O 10.Re1 Nf6 11.a3 b6 *"),
("D42m", "QGD: Semi-Tarrasch, 7.Bd3 Main Line, 10.Re1 Nf6 11.a3 b6 12.Be3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bd3 cxd4 8.exd4 Be7 9.O-O O-O 10.Re1 Nf6 11.a3 b6 12.Be3 *"),
("D42n", "QGD: Semi-Tarrasch, 7.Bd3 Main Line, 10.Re1 Nf6 11.a3 b6 12.Bg5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bd3 cxd4 8.exd4 Be7 9.O-O O-O 10.Re1 Nf6 11.a3 b6 12.Bg5 *"),
("D42o", "QGD: Semi-Tarrasch, 7.Bd3 Main Line, 10.Re1 Nf6 11.a3 b6 12.Bc2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bd3 cxd4 8.exd4 Be7 9.O-O O-O 10.Re1 Nf6 11.a3 b6 12.Bc2 *"),
("D42p", "QGD: Semi-Tarrasch, 7.Bd3 Main Line, 10.Re1 Bf6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bd3 cxd4 8.exd4 Be7 9.O-O O-O 10.Re1 Bf6 *"),
("D42p", "QGD: Semi-Tarrasch, 7.Bd3 Main Line, 10.Re1 Bf6 11.Be4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bd3 cxd4 8.exd4 Be7 9.O-O O-O 10.Re1 Bf6 11.Be4 *"),
("D42q", "QGD: Semi-Tarrasch, 7.Bd3 Main Line, 10.Re1 Bf6 11.Be4 Nce7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bd3 cxd4 8.exd4 Be7 9.O-O O-O 10.Re1 Bf6 11.Be4 Nce7 *"),
("D42r", "QGD: Semi-Tarrasch, 7.Bd3 Main Line, 10.Re1 Bf6 11.Be4 Nce7 12.Ne5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c5 5.cxd5 Nxd5 6.e3 Nc6 7.Bd3 cxd4 8.exd4 Be7 9.O-O O-O 10.Re1 Bf6 11.Be4 Nce7 12.Ne5 *"),
("D43a", "Semi-Slav", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 *"),
("D43b", "Semi-Slav: 5.g3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.g3 *"),
("D43c", "Semi-Slav: 5.Qd3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Qd3 *"),
("D43d", "Semi-Slav: 5.Qb3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Qb3 *"),
("D43e", "Semi-Slav: 5.Qb3 dxc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Qb3 dxc4 6.Qxc4 *"),
("D43f", "Semi-Slav: Botvinnik (Anti-Meran)", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 *"),
("D43f", "Semi-Slav: Botvinnik, 5...Be7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 Be7 *"),
("D43g", "Semi-Slav: Botvinnik, 5...Be7 6.e3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 Be7 6.e3 *"),
("D43h", "Semi-Slav: Botvinnik, 5...Be7 6.e3 O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 Be7 6.e3 O-O *"),
("D43i", "Semi-Slav: Botvinnik, 5...Be7 6.e3 Nbd7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 Be7 6.e3 Nbd7 *"),
("D43j", "Semi-Slav: Botvinnik, 5...Be7 6.e3 Nbd7 7.Bd3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 Be7 6.e3 Nbd7 7.Bd3 *"),
("D43k", "Semi-Slav: Moscow Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 h6 *"),
("D43k", "Semi-Slav: Moscow, 6.Bh4 Gambit", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 h6 6.Bh4 *"),
("D43k", "Semi-Slav: Moscow, 6.Bh4 Be7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 h6 6.Bh4 Be7 *"),
("D43k", "Semi-Slav: Moscow, 6.Bh4 Be7 7.e3 O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 h6 6.Bh4 Be7 7.e3 O-O *"),
("D43k", "Semi-Slav: Moscow, 6.Bh4 Gambit Accepted", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 h6 6.Bh4 dxc4 *"),
("D43l", "Semi-Slav: Moscow, 6.Bh4 Gambit, 7.e4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 h6 6.Bh4 dxc4 7.e4 *"),
("D43m", "Semi-Slav: Moscow, 6.Bh4 Gambit, Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 h6 6.Bh4 dxc4 7.e4 g5 8.Bg3 b5 9.Be2 *"),
("D43n", "Semi-Slav: Moscow, 6.Bh4 Gambit, Main Line, 9...Bb7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 h6 6.Bh4 dxc4 7.e4 g5 8.Bg3 b5 9.Be2 Bb7 *"),
("D43o", "Semi-Slav: Moscow, 6.Bxf6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 h6 6.Bxf6 *"),
("D43p", "Semi-Slav: Moscow, 7.g3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 h6 6.Bxf6 Qxf6 7.g3 *"),
("D43q", "Semi-Slav: Moscow, Hastings Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 h6 6.Bxf6 Qxf6 7.Qb3 *"),
("D43r", "Semi-Slav: Moscow, 7.Qc2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 h6 6.Bxf6 Qxf6 7.Qc2 *"),
("D43s", "Semi-Slav: Moscow, 7.e3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 h6 6.Bxf6 Qxf6 7.e3 *"),
("D43t", "Semi-Slav: Moscow, 7.e3 Nd7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 h6 6.Bxf6 Qxf6 7.e3 Nd7 *"),
("D43u", "Semi-Slav: Moscow, 7.e3 Nd7 8.Bd3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 h6 6.Bxf6 Qxf6 7.e3 Nd7 8.Bd3 *"),
("D43v", "Semi-Slav: Moscow, 7.e3 Nd7 8.Bd3 dxc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 h6 6.Bxf6 Qxf6 7.e3 Nd7 8.Bd3 dxc4 *"),
("D43v", "Semi-Slav: Moscow, Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 h6 6.Bxf6 Qxf6 7.e3 Nd7 8.Bd3 dxc4 9.Bxc4 g6 *"),
("D43v", "Semi-Slav: Moscow, Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 h6 6.Bxf6 Qxf6 7.e3 Nd7 8.Bd3 dxc4 9.Bxc4 g6 10.O-O *"),
("D43v", "Semi-Slav: Moscow, Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 h6 6.Bxf6 Qxf6 7.e3 Nd7 8.Bd3 dxc4 9.Bxc4 g6 10.O-O Bg7 *"),
("D43w", "Semi-Slav: Moscow, Main Line, 11.b4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 h6 6.Bxf6 Qxf6 7.e3 Nd7 8.Bd3 dxc4 9.Bxc4 g6 10.O-O Bg7 11.b4 *"),
("D43x", "Semi-Slav: Moscow, Main Line, 11.Rc1", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 h6 6.Bxf6 Qxf6 7.e3 Nd7 8.Bd3 dxc4 9.Bxc4 g6 10.O-O Bg7 11.Rc1 *"),
("D44a", "Semi-Slav: Botvinnik Accepted", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 dxc4 *"),
("D44b", "Semi-Slav: Botvinnik, 6.a4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 dxc4 6.a4 *"),
("D44c", "Semi-Slav: Botvinnik, 6.e4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 dxc4 6.e4 *"),
("D44c", "Semi-Slav: Botvinnik, 6.e4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 dxc4 6.e4 b5 *"),
("D44d", "Semi-Slav: Botvinnik, 7.a4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 dxc4 6.e4 b5 7.a4 *"),
("D44e", "Semi-Slav: Botvinnik, 7.e5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 dxc4 6.e4 b5 7.e5 *"),
("D44f", "Semi-Slav: Botvinnik, 9.exf6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 dxc4 6.e4 b5 7.e5 h6 8.Bh4 g5 9.exf6 *"),
("D44g", "Semi-Slav: Botvinnik, Ekstrom Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 dxc4 6.e4 b5 7.e5 h6 8.Bh4 g5 9.exf6 gxh4 10.Ne5 *"),
("D44h", "Semi-Slav: Botvinnik, 9.Nxg5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 dxc4 6.e4 b5 7.e5 h6 8.Bh4 g5 9.Nxg5 *"),
("D44i", "Semi-Slav: Botvinnik, Alatortsev Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 dxc4 6.e4 b5 7.e5 h6 8.Bh4 g5 9.Nxg5 Nd5 *"),
("D44j", "Semi-Slav: Botvinnik, 9.Nxg5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 dxc4 6.e4 b5 7.e5 h6 8.Bh4 g5 9.Nxg5 hxg5 *"),
("D44k", "Semi-Slav: Botvinnik, Lilienthal Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 dxc4 6.e4 b5 7.e5 h6 8.Bh4 g5 9.Nxg5 hxg5 10.Bxg5 Nbd7 11.g3 *"),
("D44l", "Semi-Slav: Botvinnik, Lilienthal, 11...Qa5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 dxc4 6.e4 b5 7.e5 h6 8.Bh4 g5 9.Nxg5 hxg5 10.Bxg5 Nbd7 11.g3 Qa5 *"),
("D44m", "Semi-Slav: Botvinnik, Lilienthal, 11...Bb7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 dxc4 6.e4 b5 7.e5 h6 8.Bh4 g5 9.Nxg5 hxg5 10.Bxg5 Nbd7 11.g3 Bb7 *"),
("D44n", "Semi-Slav: Botvinnik, Szabo Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 dxc4 6.e4 b5 7.e5 h6 8.Bh4 g5 9.Nxg5 hxg5 10.Bxg5 Nbd7 11.Qf3 *"),
("D44o", "Semi-Slav: Botvinnik, Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 dxc4 6.e4 b5 7.e5 h6 8.Bh4 g5 9.Nxg5 hxg5 10.Bxg5 Nbd7 11.exf6 *"),
("D44p", "Semi-Slav: Botvinnik, Main Line, 12.g3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 dxc4 6.e4 b5 7.e5 h6 8.Bh4 g5 9.Nxg5 hxg5 10.Bxg5 Nbd7 11.exf6 Bb7 12.g3 *"),
("D44q", "Semi-Slav: Botvinnik, Main Line, 12.g3 c5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 dxc4 6.e4 b5 7.e5 h6 8.Bh4 g5 9.Nxg5 hxg5 10.Bxg5 Nbd7 11.exf6 Bb7 12.g3 c5 *"),
("D44r", "Semi-Slav: Botvinnik, Main Line, 12.g3 c5 13.d5 Qb6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.Bg5 dxc4 6.e4 b5 7.e5 h6 8.Bh4 g5 9.Nxg5 hxg5 10.Bxg5 Nbd7 11.exf6 Bb7 12.g3 c5 13.d5 Qb6 *"),
("D45a", "Semi-Slav: 5.e3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 *"),
("D45b", "Semi-Slav: 5.e3 Bd6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Bd6 *"),
("D45c", "Semi-Slav: 5.e3 Be7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Be7 *"),
("D45d", "Semi-Slav: 5.e3 Ne4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Ne4 *"),
("D45d", "Semi-Slav: Stonewall Defence", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Ne4 6.Bd3 f5 *"),
("D45e", "Semi-Slav: Accelerated Meran (Alekhine Variation)", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 a6 *"),
("D45f", "Semi-Slav: 5.e3 Nbd7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 *"),
("D45f", "Semi-Slav: Rubinstein System", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Ne5 *"),
("D45g", "Semi-Slav: 6.a3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.a3 *"),
("D45h", "Semi-Slav: 6.Qc2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Qc2 *"),
("D45i", "Semi-Slav: 6.Qc2 b6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Qc2 b6 *"),
("D45j", "Semi-Slav: 6.Qc2 Be7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Qc2 Be7 *"),
("D45k", "Semi-Slav: 6.Qc2 Bd6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Qc2 Bd6 *"),
("D45k", "Semi-Slav: 6.Qc2 Bd6 7.e4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Qc2 Bd6 7.e4 *"),
("D45l", "Semi-Slav: 6.Qc2 Bd6 7.g4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Qc2 Bd6 7.g4 *"),
("D45l", "Semi-Slav: 6.Qc2 Bd6 7.g4 h6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Qc2 Bd6 7.g4 h6 *"),
("D45m", "Semi-Slav: 6.Qc2 Bd6 7.g4 dxc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Qc2 Bd6 7.g4 dxc4 *"),
("D45n", "Semi-Slav: 6.Qc2 Bd6 7.g4 Bb4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Qc2 Bd6 7.g4 Bb4 *"),
("D45o", "Semi-Slav: 6.Qc2 Bd6 7.g4 Nxg4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Qc2 Bd6 7.g4 Nxg4 *"),
("D45p", "Semi-Slav: 6.Qc2 Bd6 7.b3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Qc2 Bd6 7.b3 *"),
("D45p", "Semi-Slav: 6.Qc2 Bd6 7.b3 O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Qc2 Bd6 7.b3 O-O *"),
("D45q", "Semi-Slav: 6.Qc2 Bd6 7.b3 O-O 8.Bb2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Qc2 Bd6 7.b3 O-O 8.Bb2 *"),
("D45r", "Semi-Slav: 6.Qc2 Bd6 7.b3 O-O 8.Be2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Qc2 Bd6 7.b3 O-O 8.Be2 *"),
("D45s", "Semi-Slav: 6.Qc2 Bd6 7.b3 O-O 8.Be2 b6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Qc2 Bd6 7.b3 O-O 8.Be2 b6 *"),
("D45t", "Semi-Slav: 6.Qc2 Bd6 7.b3 O-O 8.Be2 Re8", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Qc2 Bd6 7.b3 O-O 8.Be2 Re8 *"),
("D45u", "Semi-Slav: 6.Qc2 Bd6 7.Bd2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Qc2 Bd6 7.Bd2 *"),
("D45v", "Semi-Slav: 6.Qc2 Bd6 7.Be2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Qc2 Bd6 7.Be2 *"),
("D45v", "Semi-Slav: 6.Qc2 Bd6 7.Be2 O-O 8.O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Qc2 Bd6 7.Be2 O-O 8.O-O *"),
("D45w", "Semi-Slav: 6.Qc2 Bd6 7.Be2 O-O 8.O-O e5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Qc2 Bd6 7.Be2 O-O 8.O-O e5 *"),
("D45x", "Semi-Slav: 6.Qc2 Bd6 7.Be2 O-O 8.O-O Qe7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Qc2 Bd6 7.Be2 O-O 8.O-O Qe7 *"),
("D45y", "Semi-Slav: 6.Qc2 Bd6 7.Be2 O-O 8.O-O Re8", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Qc2 Bd6 7.Be2 O-O 8.O-O Re8 *"),
("D46a", "Semi-Slav: 6.Bd3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 *"),
("D46a", "Semi-Slav: 6.Bd3 a6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 a6 *"),
("D46b", "Semi-Slav: Romih Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bb4 *"),
("D46b", "Semi-Slav: Romih, 7.O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bb4 7.O-O *"),
("D46b", "Semi-Slav: Romih, 7.O-O O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bb4 7.O-O O-O *"),
("D46c", "Semi-Slav: Bogoljubow Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Be7 *"),
("D46c", "Semi-Slav: Bogoljubow, 7.O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Be7 7.O-O *"),
("D46c", "Semi-Slav: Bogoljubow, 7.O-O O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Be7 7.O-O O-O *"),
("D46d", "Semi-Slav: Bogoljubow, 7.O-O O-O 8.b3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Be7 7.O-O O-O 8.b3 *"),
("D46e", "Semi-Slav: Chigorin Defence", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 *"),
("D46e", "Semi-Slav: Chigorin, 7.e4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.e4 *"),
("D46e", "Semi-Slav: Chigorin, 7.e4 dxe4 8.Nxe4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.e4 dxe4 8.Nxe4 *"),
("D46f", "Semi-Slav: Chigorin, 7.Qc2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.Qc2 *"),
("D46g", "Semi-Slav: Chigorin, 7.Qc2 dxc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.Qc2 dxc4 *"),
("D46g", "Semi-Slav: Chigorin, 7.Qc2 dxc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.Qc2 dxc4 8.Bxc4 *"),
("D46h", "Semi-Slav: Chigorin, 7.Qc2 O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.Qc2 O-O *"),
("D46i", "Semi-Slav: Chigorin, 7.Qc2 Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.Qc2 O-O 8.O-O dxc4 9.Bxc4 *"),
("D46i", "Semi-Slav: Chigorin, 7.Qc2 Main Line, 9...e5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.Qc2 O-O 8.O-O dxc4 9.Bxc4 e5 *"),
("D46j", "Semi-Slav: Chigorin, 7.Qc2 Main Line, 9...Qe7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.Qc2 O-O 8.O-O dxc4 9.Bxc4 Qe7 *"),
("D46j", "Semi-Slav: Chigorin, 7.Qc2 Main Line, 9...Qe7 10.Bd2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.Qc2 O-O 8.O-O dxc4 9.Bxc4 Qe7 10.Bd2 *"),
("D46j", "Semi-Slav: Chigorin, 7.Qc2 Main Line, 9...Qe7 10.a3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.Qc2 O-O 8.O-O dxc4 9.Bxc4 Qe7 10.a3 *"),
("D46j", "Semi-Slav: Chigorin, 7.Qc2 Main Line, 9...Qe7 10.h3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.Qc2 O-O 8.O-O dxc4 9.Bxc4 Qe7 10.h3 *"),
("D46k", "Semi-Slav: Chigorin, 7.Qc2 Main Line, 9...a6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.Qc2 O-O 8.O-O dxc4 9.Bxc4 a6 *"),
("D46k", "Semi-Slav: Chigorin, 7.Qc2 Main Line, 9...a6 10.Rd1", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.Qc2 O-O 8.O-O dxc4 9.Bxc4 a6 10.Rd1 *"),
("D46k", "Semi-Slav: Chigorin, 7.Qc2 Main Line, 9...a6 10.Rd1 b5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.Qc2 O-O 8.O-O dxc4 9.Bxc4 a6 10.Rd1 b5 *"),
("D46l", "Semi-Slav: Chigorin, 7.Qc2 Main Line, 9...b5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.Qc2 O-O 8.O-O dxc4 9.Bxc4 b5 *"),
("D46m", "Semi-Slav: Chigorin, 7.Qc2 Main Line, 9...b5 10.Be2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.Qc2 O-O 8.O-O dxc4 9.Bxc4 b5 10.Be2 *"),
("D46m", "Semi-Slav: Chigorin, 7.Qc2 Main Line, 9...b5 10.Be2 Bb7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.Qc2 O-O 8.O-O dxc4 9.Bxc4 b5 10.Be2 Bb7 *"),
("D46m", "Semi-Slav: Chigorin, 7.Qc2 Main Line, 9...b5 10.Be2 Bb7 11.Rd1", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.Qc2 O-O 8.O-O dxc4 9.Bxc4 b5 10.Be2 Bb7 11.Rd1 *"),
("D46n", "Semi-Slav: Chigorin, 7.O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.O-O *"),
("D46o", "Semi-Slav: Chigorin, 7.O-O O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.O-O O-O *"),
("D46p", "Semi-Slav: Chigorin, 7.O-O O-O 8.e4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.O-O O-O 8.e4 *"),
("D46p", "Semi-Slav: Chigorin, 7.O-O O-O 8.e4 dxc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.O-O O-O 8.e4 dxc4 *"),
("D46p", "Semi-Slav: Chigorin, 7.O-O O-O 8.e4 dxc4 9.Bxc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.O-O O-O 8.e4 dxc4 9.Bxc4 *"),
("D46p", "Semi-Slav: Chigorin, 7.O-O O-O 8.e4 dxc4 9.Bxc4 e5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.O-O O-O 8.e4 dxc4 9.Bxc4 e5 *"),
("D46q", "Semi-Slav: Chigorin, Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.O-O O-O 8.e4 dxe4 *"),
("D46q", "Semi-Slav: Chigorin, Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.O-O O-O 8.e4 dxe4 9.Nxe4 *"),
("D46q", "Semi-Slav: Chigorin, Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.O-O O-O 8.e4 dxe4 9.Nxe4 Nxe4 10.Bxe4 *"),
("D46r", "Semi-Slav: Chigorin, Main Line, 10...h6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.O-O O-O 8.e4 dxe4 9.Nxe4 Nxe4 10.Bxe4 h6 *"),
("D46r", "Semi-Slav: Chigorin, Main Line, 10...h6 11.Bc2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.O-O O-O 8.e4 dxe4 9.Nxe4 Nxe4 10.Bxe4 h6 11.Bc2 *"),
("D46r", "Semi-Slav: Chigorin, Main Line, 10...h6 11.Bc2 e5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 Bd6 7.O-O O-O 8.e4 dxe4 9.Nxe4 Nxe4 10.Bxe4 h6 11.Bc2 e5 *"),
("D47a", "Semi-Slav: Meran Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 *"),
("D47a", "Semi-Slav: Meran Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 *"),
("D47a", "Semi-Slav: Meran, 7...b5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 *"),
("D47b", "Semi-Slav: Meran, Freymann Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bb3 *"),
("D47c", "Semi-Slav: Meran, Freymann, 8...b4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bb3 b4 *"),
("D47d", "Semi-Slav: Meran, Capablanca Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Be2 *"),
("D47d", "Semi-Slav: Meran, Capablanca, 8...b4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Be2 b4 *"),
("D47e", "Semi-Slav: Meran, Capablanca, 8...a6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Be2 a6 *"),
("D47f", "Semi-Slav: Meran, Capablanca, 8...Bb7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Be2 Bb7 *"),
("D47f", "Semi-Slav: Meran, Capablanca, 8...Bb7 9.O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Be2 Bb7 9.O-O *"),
("D47g", "Semi-Slav: Meran, Capablanca, 8...Bb7 9.O-O a6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Be2 Bb7 9.O-O a6 *"),
("D47h", "Semi-Slav: Meran, 8.Bd3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 *"),
("D47h", "Semi-Slav: Meran, 8.Bd3 Bd6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 Bd6 *"),
("D47i", "Semi-Slav: Meran, Lundin Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 b4 *"),
("D47i", "Semi-Slav: Meran, Lundin, 9.Na4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 b4 9.Na4 *"),
("D47j", "Semi-Slav: Meran, Lundin, 9.Ne4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 b4 9.Ne4 *"),
("D47k", "Semi-Slav: Meran, Lundin, 9.Ne4 Nxe4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 b4 9.Ne4 Nxe4 *"),
("D47l", "Semi-Slav: Meran, Lundin, 9.Ne4 Be7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 b4 9.Ne4 Be7 *"),
("D47l", "Semi-Slav: Meran, Lundin, 9.Ne4 Be7 10.O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 b4 9.Ne4 Be7 10.O-O *"),
("D47m", "Semi-Slav: Meran, Wade Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 Bb7 *"),
("D47m", "Semi-Slav: Meran, Wade, 9.a3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 Bb7 9.a3 *"),
("D47n", "Semi-Slav: Meran, Wade, 9.O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 Bb7 9.O-O *"),
("D47n", "Semi-Slav: Meran, Wade, 9.O-O b4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 Bb7 9.O-O b4 *"),
("D47o", "Semi-Slav: Meran, Wade, 9.O-O b4 10.Ne4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 Bb7 9.O-O b4 10.Ne4 *"),
("D47o", "Semi-Slav: Meran, Wade, 9.O-O b4 10.Ne4 Be7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 Bb7 9.O-O b4 10.Ne4 Be7 *"),
("D47p", "Semi-Slav: Meran, Wade, 9.O-O b4 10.Ne4 Be7 11.Nxf6+", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 Bb7 9.O-O b4 10.Ne4 Be7 11.Nxf6+ *"),
("D47p", "Semi-Slav: Meran, Wade, 9.O-O b4 10.Ne4 Be7 11.Nxf6+ Nxf6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 Bb7 9.O-O b4 10.Ne4 Be7 11.Nxf6+ Nxf6 *"),
("D47p", "Semi-Slav: Meran, Wade, 9.O-O b4 10.Ne4 Be7 11.Nxf6+ Nxf6 12.e4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 Bb7 9.O-O b4 10.Ne4 Be7 11.Nxf6+ Nxf6 12.e4 *"),
("D47q", "Semi-Slav: Meran, Wade, 9.e4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 Bb7 9.e4 *"),
("D47r", "Semi-Slav: Meran, Wade, Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 Bb7 9.e4 b4 10.Na4 c5 11.e5 Nd5 *"),
("D47s", "Semi-Slav: Meran, Wade, Main Line, 12.Nxc5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 Bb7 9.e4 b4 10.Na4 c5 11.e5 Nd5 12.Nxc5 *"),
("D47t", "Semi-Slav: Meran, Wade, Main Line, 12.O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 Bb7 9.e4 b4 10.Na4 c5 11.e5 Nd5 12.O-O *"),
("D48a", "Semi-Slav: Meran, 8...a6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 *"),
("D48b", "Semi-Slav: Meran, 8...a6 9.a4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.a4 *"),
("D48c", "Semi-Slav: Meran, 8...a6 9.O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.O-O *"),
("D48c", "Semi-Slav: Meran, 8...a6 9.O-O c5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.O-O c5 *"),
("D48d", "Semi-Slav: Meran, 8...a6 9.O-O c5 10.Qe2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.O-O c5 10.Qe2 *"),
("D48e", "Semi-Slav: Meran, 8...a6 9.O-O Bb7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.O-O Bb7 *"),
("D48f", "Semi-Slav: Meran, 8...a6 9.O-O Bb7 10.e4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.O-O Bb7 10.e4 *"),
("D48f", "Semi-Slav: Meran, 8...a6 9.O-O Bb7 10.e4 c5 11.d5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.O-O Bb7 10.e4 c5 11.d5 *"),
("D48g", "Semi-Slav: Meran, 8...a6 9.O-O Bb7 10.e4 c5 11.d5 Qc7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.O-O Bb7 10.e4 c5 11.d5 Qc7 *"),
("D48g", "Semi-Slav: Meran, 8...a6 9.O-O Bb7 10.e4 c5 11.d5 Qc7 12.dxe6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.O-O Bb7 10.e4 c5 11.d5 Qc7 12.dxe6 *"),
("D48g", "Semi-Slav: Meran, 8...a6 9.O-O Bb7 10.e4 c5 11.d5 Qc7 12.dxe6 fxe6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.O-O Bb7 10.e4 c5 11.d5 Qc7 12.dxe6 fxe6 *"),
("D48h", "Semi-Slav: Meran, 8...a6 9.e4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 *"),
("D48i", "Semi-Slav: Meran, 8...a6 9.e4 Bb7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 Bb7 *"),
("D48j", "Semi-Slav: Meran, Pirc Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 b4 *"),
("D48k", "Semi-Slav: Meran, 8...a6 9.e4 c5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 c5 *"),
("D48l", "Semi-Slav: Meran, Reynolds Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 c5 10.d5 *"),
("D48m", "Semi-Slav: Meran, Reynolds, 10...e5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 c5 10.d5 e5 *"),
("D48n", "Semi-Slav: Meran, Reynolds, 10...Qc7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 c5 10.d5 Qc7 *"),
("D48o", "Semi-Slav: Meran, Reynolds, 10...c4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 c5 10.d5 c4 *"),
("D48p", "Semi-Slav: Meran, Reynolds, 10...c4 11.dxe6 fxe6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 c5 10.d5 c4 11.dxe6 fxe6 *"),
("D48p", "Semi-Slav: Meran, Reynolds, 10...c4 11.dxe6 fxe6 12.Bc2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 c5 10.d5 c4 11.dxe6 fxe6 12.Bc2 *"),
("D48p", "Semi-Slav: Meran, Reynolds, 10...c4 11.dxe6 fxe6 12.Bc2 Qc7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 c5 10.d5 c4 11.dxe6 fxe6 12.Bc2 Qc7 *"),
("D48p", "Semi-Slav: Meran, Reynolds, 10...c4 11.dxe6 fxe6 12.Bc2 Qc7 13.O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 c5 10.d5 c4 11.dxe6 fxe6 12.Bc2 Qc7 13.O-O *"),
("D48q", "Semi-Slav: Meran, Old Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 c5 10.e5 *"),
("D48r", "Semi-Slav: Meran, Rabinovich Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 c5 10.e5 Ng4 *"),
("D48s", "Semi-Slav: Meran, Old Main Line, 10...cxd4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 c5 10.e5 cxd4 *"),
("D49a", "Semi-Slav: Meran, Blumenfeld Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 c5 10.e5 cxd4 11.Nxb5 *"),
("D49b", "Semi-Slav: Meran, Old Main Line, Gligoric Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 c5 10.e5 cxd4 11.Nxb5 Ng4 *"),
("D49c", "Semi-Slav: Meran, Old Main Line, Gligoric, 12.Qa4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 c5 10.e5 cxd4 11.Nxb5 Ng4 12.Qa4 *"),
("D49d", "Semi-Slav: Meran, Trifunovic Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 c5 10.e5 cxd4 11.Nxb5 Ng4 12.Qa4 Ngxe5 *"),
("D49e", "Semi-Slav: Meran, Sozin Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 c5 10.e5 cxd4 11.Nxb5 Nxe5 *"),
("D49f", "Semi-Slav: Meran, Sozin, Rellstab Attack", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 c5 10.e5 cxd4 11.Nxb5 Nxe5 12.Nxe5 axb5 13.O-O *"),
("D49g", "Semi-Slav: Meran, Sozin, Stahlberg Attack", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 c5 10.e5 cxd4 11.Nxb5 Nxe5 12.Nxe5 axb5 13.Qf3 *"),
("D49g", "Semi-Slav: Meran, Sozin, Stahlberg Attack, 13...Bb4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 c5 10.e5 cxd4 11.Nxb5 Nxe5 12.Nxe5 axb5 13.Qf3 Bb4 *"),
("D49h", "Semi-Slav: Meran, Sozin, 13.Bxb5+", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 c5 10.e5 cxd4 11.Nxb5 Nxe5 12.Nxe5 axb5 13.Bxb5+ *"),
("D49i", "Semi-Slav: Meran, Old Main Line, 11.Nxb5 axb5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 c5 10.e5 cxd4 11.Nxb5 axb5 *"),
("D49j", "Semi-Slav: Meran, Old Main Line, Botvinnik Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 c5 10.e5 cxd4 11.Nxb5 axb5 12.exf6 Qb6 *"),
("D49k", "Semi-Slav: Meran, Old Main Line, 11.Nxb5 axb5 12.exf6 gxf6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Nf3 c6 5.e3 Nbd7 6.Bd3 dxc4 7.Bxc4 b5 8.Bd3 a6 9.e4 c5 10.e5 cxd4 11.Nxb5 axb5 12.exf6 gxf6 *"),
("D50", "QGD: 4.Bg5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 *"),
("D50", "QGD: 4.Bg5 c6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 c6 *"),
("D50", "QGD: 4.Bg5 dxc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 dxc4 *"),
("D50", "QGD: 4.Bg5 Bb4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Bb4 *"),
("D50", "QGD: Dutch-Peruvian Gambit", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 c5 *"),
("D50", "QGD: Dutch-Peruvian, 5.e3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 c5 5.e3 *"),
("D50", "QGD: Semi-Tarrasch, Pillsbury Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 c5 5.Nf3 *"),
("D50", "QGD: Semi-Tarrasch, Old Pillsbury Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 c5 5.Nf3 cxd4 6.Qxd4 *"),
("D50", "QGD: Semi-Tarrasch, Pillsbury, 6.Nxd4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 c5 5.Nf3 cxd4 6.Nxd4 *"),
("D50", "QGD: Semi-Tarrasch, Pillsbury, 6.Nxd4 e5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 c5 5.Nf3 cxd4 6.Nxd4 e5 *"),
("D50", "QGD: Semi-Tarrasch, Pillsbury, Krause Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 c5 5.Nf3 cxd4 6.Nxd4 e5 7.Ndb5 a6 8.Qa4 *"),
("D50", "QGD: Dutch-Peruvian, 5.cxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 c5 5.cxd5 *"),
("D50", "QGD: Dutch-Peruvian, Canal (Venice) Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 c5 5.cxd5 Qb6 *"),
("D50", "QGD: Dutch-Peruvian, Prins Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 c5 5.cxd5 cxd4 *"),
("D50", "QGD: Dutch-Peruvian, Prins, 6.Qxd4 Be7 7.e4 Nc6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 c5 5.cxd5 cxd4 6.Qxd4 Be7 7.e4 Nc6 *"),
("D50", "QGD: Dutch-Peruvian, Prins, 6.Qxd4 Be7 7.e4 Nc6 8.Qd2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 c5 5.cxd5 cxd4 6.Qxd4 Be7 7.e4 Nc6 8.Qd2 *"),
("D51", "QGD: 4.Bg5 Nbd7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 *"),
("D51", "QGD: 4.Bg5 Nbd7 5.Nf3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.Nf3 *"),
("D51", "QGD: 4.Bg5 Nbd7 5.Nf3 c6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.Nf3 c6 *"),
("D51", "QGD: Rochlin Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.Nf3 c6 6.Rc1 Qa5 7.Bd2 *"),
("D51", "QGD: Alekhine 4.Bg5 Nbd7 5.Nf3 c6 6.e4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.Nf3 c6 6.e4 *"),
("D51", "QGD: 4.Bg5 Nbd7 5.e3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.e3 *"),
("D51", "QGD: Manhattan Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.e3 Bb4 *"),
("D51", "QGD: 4.Bg5 Nbd7 5.e3 c6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.e3 c6 *"),
("D51", "QGD: Capablanca's Anti-Cambridge Springs", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.e3 c6 6.a3 *"),
("D51", "QGD: 4.Bg5 Nbd7 5.e3 c6 6.cxd5 cxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.e3 c6 6.cxd5 cxd5 *"),
("D51", "QGD: 4.Bg5 Nbd7 5.e3 c6 6.Qc2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.e3 c6 6.Qc2 *"),
("D52a", "QGD: 4.Bg5 Nbd7 5.e3 c6 6.Nf3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.e3 c6 6.Nf3 *"),
("D52b", "QGD: Cambridge Springs Defence", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.e3 c6 6.Nf3 Qa5 *"),
("D52c", "QGD: Cambridge Springs, Capablanca Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.e3 c6 6.Nf3 Qa5 7.Bxf6 *"),
("D52d", "QGD: Cambridge Springs, 7.cxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.e3 c6 6.Nf3 Qa5 7.cxd5 *"),
("D52e", "QGD: Cambridge Springs, Yugoslav Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.e3 c6 6.Nf3 Qa5 7.cxd5 Nxd5 *"),
("D52f", "QGD: Cambridge Springs, Yugoslav, 8.Qd2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.e3 c6 6.Nf3 Qa5 7.cxd5 Nxd5 8.Qd2 *"),
("D52g", "QGD: Cambridge Springs, 7.Nd2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.e3 c6 6.Nf3 Qa5 7.Nd2 *"),
("D52h", "QGD: Cambridge Springs, Rubinstein Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.e3 c6 6.Nf3 Qa5 7.Nd2 dxc4 *"),
("D52i", "QGD: Cambridge Springs, Bogoljubow Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.e3 c6 6.Nf3 Qa5 7.Nd2 Bb4 *"),
("D52j", "QGD: Cambridge Springs, Bogoljubow, 8.Qc2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.e3 c6 6.Nf3 Qa5 7.Nd2 Bb4 8.Qc2 *"),
("D52k", "QGD: Cambridge Springs, 8.Qc2 dxc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.e3 c6 6.Nf3 Qa5 7.Nd2 Bb4 8.Qc2 dxc4 *"),
("D52l", "QGD: Cambridge Springs, 8.Qc2 O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.e3 c6 6.Nf3 Qa5 7.Nd2 Bb4 8.Qc2 O-O *"),
("D52m", "QGD: Cambridge Springs, Argentine Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.e3 c6 6.Nf3 Qa5 7.Nd2 Bb4 8.Qc2 O-O 9.Bh4 *"),
("D52n", "QGD: Cambridge Springs, 8.Qc2 O-O 9.Be2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.e3 c6 6.Nf3 Qa5 7.Nd2 Bb4 8.Qc2 O-O 9.Be2 *"),
("D52o", "QGD: Cambridge Springs, 8.Qc2 O-O 9.Be2 e5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Nbd7 5.e3 c6 6.Nf3 Qa5 7.Nd2 Bb4 8.Qc2 O-O 9.Be2 e5 *"),
("D53a", "QGD: 4.Bg5 Be7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 *"),
("D53b", "QGD: 4.Bg5 Be7 5.cxd5 Nxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.cxd5 Nxd5 *"),
("D53c", "QGD: 4.Bg5 Be7 5.Nf3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.Nf3 *"),
("D53d", "QGD: 4.Bg5 Be7 5.Nf3 h6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.Nf3 h6 *"),
("D53e", "QGD: 4.Bg5 Be7 5.Nf3 h6 6.Bxf6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.Nf3 h6 6.Bxf6 *"),
("D53f", "QGD: 4.Bg5 Be7 5.Nf3 h6 6.Bxf6 Bxf6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.Nf3 h6 6.Bxf6 Bxf6 *"),
("D53g", "QGD: 4.Bg5 Be7 5.Nf3 h6 6.Bxf6 Bxf6 7.e3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.Nf3 h6 6.Bxf6 Bxf6 7.e3 *"),
("D53h", "QGD: 4.Bg5 Be7 5.Nf3 h6 6.Bh4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.Nf3 h6 6.Bh4 *"),
("D53i", "QGD: 4.Bg5 Be7 5.Nf3 h6 6.Bh4 O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.Nf3 h6 6.Bh4 O-O *"),
("D53j", "QGD: 4.Bg5 Be7 5.Nf3 h6 6.Bh4 O-O 7.Qc2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.Nf3 h6 6.Bh4 O-O 7.Qc2 *"),
("D53k", "QGD: 4.Bg5 Be7 5.Nf3 h6 6.Bh4 O-O 7.Rc1", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.Nf3 h6 6.Bh4 O-O 7.Rc1 *"),
("D53l", "QGD: 4.Bg5 Be7 5.Nf3 O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.Nf3 O-O *"),
("D53m", "QGD: 4.Bg5 Be7 5.Nf3 O-O 6.Qc2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.Nf3 O-O 6.Qc2 *"),
("D53n", "QGD: 4.Bg5 Be7 5.Nf3 O-O 6.Qc2 Nbd7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.Nf3 O-O 6.Qc2 Nbd7 *"),
("D53o", "QGD: 4.Bg5 Be7 5.Nf3 O-O 6.Qc2 c5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.Nf3 O-O 6.Qc2 c5 *"),
("D53p", "QGD: 4.Bg5 Be7 5.e3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 *"),
("D53q", "QGD: Early Lasker Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 Ne4 *"),
("D53r", "QGD: 4.Bg5 Be7 5.e3 Nbd7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 Nbd7 *"),
("D53s", "QGD: 4.Bg5 Be7 5.e3 Nbd7 6.Nf3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 Nbd7 6.Nf3 *"),
("D53t", "QGD: 4.Bg5 Be7 5.e3 h6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 h6 *"),
("D53u", "QGD: 4.Bg5 Be7 5.e3 O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O *"),
("D53v", "QGD: 4.Bg5 Be7 5.e3 O-O 6.Qc2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Qc2 *"),
("D53v", "QGD: 4.Bg5 Be7 5.e3 O-O 6.Qc2 c5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Qc2 c5 *"),
("D54a", "QGD: Anti-Neo-Orthodox Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Rc1 *"),
("D54b", "QGD: Anti-Neo-Orthodox Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Rc1 h6 7.Bh4 *"),
("D54c", "QGD: Anti-Neo-Orthodox, 7.Bh4 b6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Rc1 h6 7.Bh4 b6 *"),
("D54c", "QGD: Anti-Neo-Orthodox, 8.cxd5 Nxd5 9.Nxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Rc1 h6 7.Bh4 b6 8.cxd5 Nxd5 9.Nxd5 *"),
("D54d", "QGD: Anti-Neo-Orthodox, Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Rc1 h6 7.Bh4 b6 8.cxd5 Nxd5 9.Nxd5 exd5 10.Bxe7 Qxe7 *"),
("D54e", "QGD: Anti-Neo-Orthodox, Main Line, 11.Be2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Rc1 h6 7.Bh4 b6 8.cxd5 Nxd5 9.Nxd5 exd5 10.Bxe7 Qxe7 11.Be2 *"),
("D55a", "QGD: 6.Nf3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 *"),
("D55b", "QGD: Lasker without ...h6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Ne4 *"),
("D55c", "QGD: 6.Nf3 b6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 b6 *"),
("D55d", "QGD: 6.Nf3 b6 7.Be2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 b6 7.Be2 *"),
("D55e", "QGD: 6.Nf3 b6 7.cxd5 exd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 b6 7.cxd5 exd5 *"),
("D55f", "QGD: 6.Nf3 b6 7.cxd5 Nxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 b6 7.cxd5 Nxd5 *"),
("D55g", "QGD: Neo-Orthodox Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 *"),
("D55h", "QGD: Neo-Orthodox, 7.Bxf6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bxf6 *"),
("D55i", "QGD: Neo-Orthodox, 7.Bxf6 Bxf6 8.Qb3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bxf6 Bxf6 8.Qb3 *"),
("D55j", "QGD: Neo-Orthodox, Gligoric Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bxf6 Bxf6 8.Rc1 *"),
("D55k", "QGD: Neo-Orthodox, Petrosian Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bxf6 Bxf6 8.Rc1 c6 9.Bd3 Nd7 10.O-O dxc4 11.Bxc4 *"),
("D56", "QGD: Neo-Orthodox, 7.Bh4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 *"),
("D56", "QGD: Lasker Defence", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 Ne4 *"),
("D56", "QGD: Lasker Defence, 9.Nxe4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 Ne4 8.Bxe7 Qxe7 9.Nxe4 *"),
("D56", "QGD: Lasker Defence, Teichmann Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 Ne4 8.Bxe7 Qxe7 9.Qc2 *"),
("D56", "QGD: Lasker Defence, Teichmann, 9...c6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 Ne4 8.Bxe7 Qxe7 9.Qc2 c6 *"),
("D56", "QGD: Lasker Defence, Teichmann, 9...Nxc3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 Ne4 8.Bxe7 Qxe7 9.Qc2 Nxc3 *"),
("D56", "QGD: Lasker Defence, 9.Rc1", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 Ne4 8.Bxe7 Qxe7 9.Rc1 *"),
("D56", "QGD: Lasker Defence, 9.Rc1 c6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 Ne4 8.Bxe7 Qxe7 9.Rc1 c6 *"),
("D56", "QGD: Lasker Defence, 9.Rc1 c6 10.Bd3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 Ne4 8.Bxe7 Qxe7 9.Rc1 c6 10.Bd3 *"),
("D57", "QGD: Lasker Defence, Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 Ne4 8.Bxe7 Qxe7 9.cxd5 *"),
("D57", "QGD: Lasker Defence, Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 Ne4 8.Bxe7 Qxe7 9.cxd5 Nxc3 *"),
("D57", "QGD: Lasker Defence, Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 Ne4 8.Bxe7 Qxe7 9.cxd5 Nxc3 10.bxc3 *"),
("D57", "QGD: Lasker Defence, Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 Ne4 8.Bxe7 Qxe7 9.cxd5 Nxc3 10.bxc3 exd5 *"),
("D57", "QGD: Lasker Defence, Main Line, 11.Qb3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 Ne4 8.Bxe7 Qxe7 9.cxd5 Nxc3 10.bxc3 exd5 11.Qb3 *"),
("D57", "QGD: Lasker Defence, Main Line, Bernstein Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 Ne4 8.Bxe7 Qxe7 9.cxd5 Nxc3 10.bxc3 exd5 11.Qb3 Qd6 *"),
("D57", "QGD: Lasker Defence, Main Line, Bernstein, 12.c4 dxc4 13.Bxc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 Ne4 8.Bxe7 Qxe7 9.cxd5 Nxc3 10.bxc3 exd5 11.Qb3 Qd6 12.c4 dxc4 13.Bxc4 *"),
("D57", "QGD: Lasker Defence, Main Line, 11.Qb3 Rd8", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 Ne4 8.Bxe7 Qxe7 9.cxd5 Nxc3 10.bxc3 exd5 11.Qb3 Rd8 *"),
("D58a", "QGD: Tartakower System", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 *"),
("D58b", "QGD: Tartakower, 8.Qb3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Qb3 *"),
("D58c", "QGD: Tartakower, 8.Qc2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Qc2 *"),
("D58d", "QGD: Tartakower, 8.Bxf6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Bxf6 *"),
("D58e", "QGD: Tartakower, 8.Bd3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Bd3 *"),
("D58f", "QGD: Tartakower, 8.Bd3 Bb7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Bd3 Bb7 *"),
("D58g", "QGD: Tartakower, 8.Bd3 Bb7 9.O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Bd3 Bb7 9.O-O *"),
("D58h", "QGD: Tartakower, 8.Bd3 Bb7 9.O-O Nbd7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Bd3 Bb7 9.O-O Nbd7 *"),
("D58h", "QGD: Tartakower, 8.Bd3 Bb7 9.O-O Nbd7 10.Bg3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Bd3 Bb7 9.O-O Nbd7 10.Bg3 *"),
("D58h", "QGD: Tartakower, 8.Bd3 Bb7 9.O-O Nbd7 10.Rc1", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Bd3 Bb7 9.O-O Nbd7 10.Rc1 *"),
("D58h", "QGD: Tartakower, 8.Bd3 Bb7 9.O-O Nbd7 10.Rc1 c5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Bd3 Bb7 9.O-O Nbd7 10.Rc1 c5 *"),
("D58i", "QGD: Tartakower, 8.Bd3 Bb7 9.O-O Nbd7 10.Qe2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Bd3 Bb7 9.O-O Nbd7 10.Qe2 *"),
("D58j", "QGD: Tartakower, 8.Bd3 Bb7 9.O-O Nbd7 10.Qe2 c5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Bd3 Bb7 9.O-O Nbd7 10.Qe2 c5 *"),
("D58j", "QGD: Tartakower, 8.Bd3 Bb7 9.O-O Nbd7 10.Qe2 c5 11.Bg3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Bd3 Bb7 9.O-O Nbd7 10.Qe2 c5 11.Bg3 *"),
("D58j", "QGD: Tartakower, 8.Bd3 Bb7 9.O-O Nbd7 10.Qe2 c5 11.Rfd1", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Bd3 Bb7 9.O-O Nbd7 10.Qe2 c5 11.Rfd1 *"),
("D58k", "QGD: Tartakower, 8.Be2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Be2 *"),
("D58k", "QGD: Tartakower, 8.Be2 Bb7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Be2 Bb7 *"),
("D58l", "QGD: Tartakower, 8.Be2 Bb7 9.Bxf6 Bxf6 10.cxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Be2 Bb7 9.Bxf6 Bxf6 10.cxd5 *"),
("D58l", "QGD: Tartakower, 8.Be2 Bb7 9.Bxf6 Bxf6 10.cxd5 exd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Be2 Bb7 9.Bxf6 Bxf6 10.cxd5 exd5 *"),
("D58m", "QGD: Tartakower, 8.Be2 Bb7 9.Bxf6 Bxf6 10.cxd5 exd5 11.O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Be2 Bb7 9.Bxf6 Bxf6 10.cxd5 exd5 11.O-O *"),
("D58n", "QGD: Tartakower, 8.Be2 Bb7 9.Bxf6 Bxf6 10.cxd5 exd5 11.b4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Be2 Bb7 9.Bxf6 Bxf6 10.cxd5 exd5 11.b4 *"),
("D58n", "QGD: Tartakower, 8.Be2 Bb7 9.Bxf6 Bxf6 10.cxd5 exd5 11.b4 c5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Be2 Bb7 9.Bxf6 Bxf6 10.cxd5 exd5 11.b4 c5 *"),
("D58o", "QGD: Tartakower, 8.Be2 Bb7 9.Bxf6 Bxf6 10.cxd5 exd5 11.b4 c6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Be2 Bb7 9.Bxf6 Bxf6 10.cxd5 exd5 11.b4 c6 *"),
("D58o", "QGD: Tartakower, 8.Be2 Bb7 9.Bxf6 Bxf6 10.cxd5 exd5 11.b4 c6 12.O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Be2 Bb7 9.Bxf6 Bxf6 10.cxd5 exd5 11.b4 c6 12.O-O *"),
("D58p", "QGD: Tartakower, 8.Rc1", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Rc1 *"),
("D58p", "QGD: Tartakower, 8.Rc1 Bb7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Rc1 Bb7 *"),
("D58q", "QGD: Tartakower, 8.Rc1 Bb7 9.cxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Rc1 Bb7 9.cxd5 *"),
("D58q", "QGD: Tartakower, 8.Rc1 Bb7 9.cxd5 exd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Rc1 Bb7 9.cxd5 exd5 *"),
("D58r", "QGD: Tartakower, 8.Rc1 Bb7 9.Bxf6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Rc1 Bb7 9.Bxf6 *"),
("D58r", "QGD: Tartakower, 8.Rc1 Bb7 9.Bxf6 Bxf6 10.cxd5 exd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.Rc1 Bb7 9.Bxf6 Bxf6 10.cxd5 exd5 *"),
("D58s", "QGD: Tartakower, 8.cxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.cxd5 *"),
("D58s", "QGD: Tartakower, 8.cxd5 exd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.cxd5 exd5 *"),
("D58t", "QGD: Tartakower, 8.cxd5 exd5 9.Bd3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.cxd5 exd5 9.Bd3 *"),
("D59a", "QGD: Tartakower, 8.cxd5 Nxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.cxd5 Nxd5 *"),
("D59b", "QGD: Tartakower, 8.cxd5 Nxd5 9.Nxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.cxd5 Nxd5 9.Nxd5 *"),
("D59c", "QGD: Tartakower, 8.cxd5 Nxd5 9.Bxe7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.cxd5 Nxd5 9.Bxe7 *"),
("D59d", "QGD: Tartakower, 8.cxd5 Nxd5 9.Bxe7 Qxe7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.cxd5 Nxd5 9.Bxe7 Qxe7 *"),
("D59e", "QGD: Tartakower, 8.cxd5 Nxd5 9.Bxe7 Qxe7 10.Rc1", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.cxd5 Nxd5 9.Bxe7 Qxe7 10.Rc1 *"),
("D59e", "QGD: Tartakower, 8.cxd5 Nxd5 9.Bxe7 Qxe7 10.Rc1 Bb7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.cxd5 Nxd5 9.Bxe7 Qxe7 10.Rc1 Bb7 *"),
("D59f", "QGD: Tartakower, 8.cxd5 Nxd5 9.Bxe7 Qxe7 10.Rc1 Bb7 11.Nxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.cxd5 Nxd5 9.Bxe7 Qxe7 10.Rc1 Bb7 11.Nxd5 *"),
("D59f", "QGD: Tartakower, 8.cxd5 Nxd5 9.Bxe7 Qxe7 10.Rc1 Bb7 11.Nxd5 Bxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.cxd5 Nxd5 9.Bxe7 Qxe7 10.Rc1 Bb7 11.Nxd5 Bxd5 *"),
("D59g", "QGD: Tartakower, 8.cxd5 Nxd5 9.Bxe7 Qxe7 10.Rc1 Bb7 11.Bd3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.cxd5 Nxd5 9.Bxe7 Qxe7 10.Rc1 Bb7 11.Bd3 *"),
("D59h", "QGD: Tartakower, 8.cxd5 Nxd5 9.Bxe7 Qxe7 10.Rc1 Bb7 11.Be2", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.cxd5 Nxd5 9.Bxe7 Qxe7 10.Rc1 Bb7 11.Be2 *"),
("D59i", "QGD: Tartakower, 8.cxd5 Nxd5 9.Bxe7 Qxe7 10.Nxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.cxd5 Nxd5 9.Bxe7 Qxe7 10.Nxd5 *"),
("D59j", "QGD: Tartakower, Main Line, 11.Rc1", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.cxd5 Nxd5 9.Bxe7 Qxe7 10.Nxd5 exd5 11.Be2 *"),
("D59k", "QGD: Tartakower, Main Line, 11.Rc1", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.cxd5 Nxd5 9.Bxe7 Qxe7 10.Nxd5 exd5 11.Rc1 *"),
("D59l", "QGD: Tartakower, Main Line, 11.Rc1 Be6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.cxd5 Nxd5 9.Bxe7 Qxe7 10.Nxd5 exd5 11.Rc1 Be6 *"),
("D59m", "QGD: Tartakower, Main Line, 12.Bd3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.cxd5 Nxd5 9.Bxe7 Qxe7 10.Nxd5 exd5 11.Rc1 Be6 12.Bd3 *"),
("D59n", "QGD: Tartakower, Main Line, 12.Qa4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.cxd5 Nxd5 9.Bxe7 Qxe7 10.Nxd5 exd5 11.Rc1 Be6 12.Qa4 *"),
("D59n", "QGD: Tartakower, Main Line, 12.Qa4 c5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 h6 7.Bh4 b6 8.cxd5 Nxd5 9.Bxe7 Qxe7 10.Nxd5 exd5 11.Rc1 Be6 12.Qa4 c5 *"),
("D60", "QGD: Orthodox Defence", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 *"),
("D60", "QGD: Orthodox, 7.cxd5 Nxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.cxd5 Nxd5 *"),
("D60", "QGD: Orthodox, Rauzer Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Qb3 *"),
("D60", "QGD: Orthodox, Rauzer Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Qb3 c6 *"),
("D60", "QGD: Orthodox, Botvinnik Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Bd3 *"),
("D60", "QGD: Orthodox, Botvinnik, 7...b6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Bd3 b6 *"),
("D60", "QGD: Orthodox, Botvinnik, 7...c6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Bd3 c6 *"),
("D60", "QGD: Orthodox, Botvinnik, 7...dxc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Bd3 dxc4 *"),
("D60", "QGD: Orthodox, Botvinnik, 7...dxc4 8.Bxc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Bd3 dxc4 8.Bxc4 *"),
("D60", "QGD: Orthodox, Botvinnik, 8...a6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Bd3 dxc4 8.Bxc4 a6 *"),
("D60", "QGD: Orthodox, Botvinnik, 8...c5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Bd3 dxc4 8.Bxc4 c5 *"),
("D60", "QGD: Orthodox, Botvinnik, 8...c5 9.O-O a6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Bd3 dxc4 8.Bxc4 c5 9.O-O a6 *"),
("D61", "QGD: Orthodox, Rubinstein Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Qc2 *"),
("D61", "QGD: Orthodox, Rubinstein, 7.Qc2 h6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Qc2 h6 *"),
("D61", "QGD: Orthodox, Rubinstein, 7.Qc2 h6 8.Bh4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Qc2 h6 8.Bh4 *"),
("D61", "QGD: Orthodox, Rubinstein, 7.Qc2 h6 8.Bh4 c5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Qc2 h6 8.Bh4 c5 *"),
("D61", "QGD: Orthodox, Rubinstein, 7.Qc2 a6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Qc2 a6 *"),
("D61", "QGD: Orthodox, Rubinstein, 7.Qc2 c6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Qc2 c6 *"),
("D61", "QGD: Orthodox, Rubinstein, 7.Qc2 c6 8.a3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Qc2 c6 8.a3 *"),
("D61", "QGD: Orthodox, Rubinstein, 7.Qc2 c6 8.Bd3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Qc2 c6 8.Bd3 *"),
("D61", "QGD: Orthodox, Rubinstein, 7.Qc2 c6 8.Bd3 dxc4 9.Bxc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Qc2 c6 8.Bd3 dxc4 9.Bxc4 *"),
("D61", "QGD: Orthodox, Rubinstein, 7.Qc2 c6 8.Rd1", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Qc2 c6 8.Rd1 *"),
("D61", "QGD: Orthodox, Rubinstein, 7.Qc2 c5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Qc2 c5 *"),
("D61", "QGD: Orthodox, Rubinstein, 7.Qc2 c5 8.Rd1", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Qc2 c5 8.Rd1 *"),
("D61", "QGD: Orthodox, Rubinstein, 7.Qc2 c5 8.O-O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Qc2 c5 8.O-O-O *"),
("D62", "QGD: Orthodox, Rubinstein, 7.Qc2 c5 8.cxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Qc2 c5 8.cxd5 *"),
("D62", "QGD: Orthodox, Rubinstein, 7.Qc2 c5 8.cxd5 cxd4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Qc2 c5 8.cxd5 cxd4 *"),
("D62", "QGD: Orthodox, Rubinstein, 7.Qc2 c5 8.cxd5 exd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Qc2 c5 8.cxd5 exd5 *"),
("D62", "QGD: Orthodox, Rubinstein, 7.Qc2 c5 8.cxd5 exd5 9.Bd3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Qc2 c5 8.cxd5 exd5 9.Bd3 *"),
("D62", "QGD: Orthodox, Rubinstein, 7.Qc2 c5 8.cxd5 Nxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Qc2 c5 8.cxd5 Nxd5 *"),
("D62", "QGD: Orthodox, Rubinstein, 7.Qc2 c5 8.cxd5 Nxd5 9.Bxe7 Qxe7", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Qc2 c5 8.cxd5 Nxd5 9.Bxe7 Qxe7 *"),
("D62", "QGD: Orthodox, Rubinstein, 7.Qc2 c5 8.cxd5, Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Qc2 c5 8.cxd5 Nxd5 9.Bxe7 Qxe7 10.Nxd5 exd5 *"),
("D62", "QGD: Orthodox, Rubinstein, 7.Qc2 c5 8.cxd5, Main Line, 11.Bd3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Qc2 c5 8.cxd5 Nxd5 9.Bxe7 Qxe7 10.Nxd5 exd5 11.Bd3 *"),
("D62", "QGD: Orthodox, Rubinstein, 7.Qc2 c5 8.cxd5, Main Line, 11.Bd3 g6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Qc2 c5 8.cxd5 Nxd5 9.Bxe7 Qxe7 10.Nxd5 exd5 11.Bd3 g6 *"),
("D63", "QGD: Orthodox, 7.Rc1", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 *"),
("D63", "QGD: Orthodox, 7.Rc1 Re8", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 Re8 *"),
("D63", "QGD: Orthodox, 7.Rc1 h6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 h6 *"),
("D63", "QGD: Orthodox, 7.Rc1 h6 8.Bh4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 h6 8.Bh4 *"),
("D63", "QGD: Orthodox, 7.Rc1 dxc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 dxc4 *"),
("D63", "QGD: Orthodox, 7.Rc1 dxc4 8.Bxc4 a6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 dxc4 8.Bxc4 a6 *"),
("D63", "QGD: Orthodox, 7.Rc1 b6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 b6 *"),
("D63", "QGD: Orthodox, 7.Rc1 b6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 b6 8.cxd5 exd5 *"),
("D63", "QGD: Orthodox, 7.Rc1 b6, Duras Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 b6 8.cxd5 exd5 9.Qa4 *"),
("D63", "QGD: Orthodox, 7.Rc1 b6, Pillsbury Attack", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 b6 8.cxd5 exd5 9.Bd3 *"),
("D63", "QGD: Orthodox, 7.Rc1 b6, Capablanca Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 b6 8.cxd5 exd5 9.Bb5 *"),
("D63", "QGD: Orthodox, Swiss Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 a6 *"),
("D63", "QGD: Orthodox, Swiss, 8.a3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 a6 8.a3 *"),
("D63", "QGD: Orthodox, Swiss, 8.c5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 a6 8.c5 *"),
("D63", "QGD: Orthodox, Swiss, 8.c5 c6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 a6 8.c5 c6 *"),
("D63", "QGD: Orthodox, Swiss, Karlsbad Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 a6 8.cxd5 *"),
("D63", "QGD: Orthodox, Swiss, Karlsbad Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 a6 8.cxd5 exd5 *"),
("D63", "QGD: Orthodox, Swiss, Karlsbad, 9.Bd3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 a6 8.cxd5 exd5 9.Bd3 *"),
("D63", "QGD: Orthodox, Swiss, Karlsbad, 9.Bd3 Re8", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 a6 8.cxd5 exd5 9.Bd3 Re8 *"),
("D63", "QGD: Orthodox, Swiss, Karlsbad, 9.Bd3 c6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 a6 8.cxd5 exd5 9.Bd3 c6 *"),
("D63", "QGD: Orthodox, 7.Rc1 c6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 *"),
("D63", "QGD: Orthodox, 7.Rc1 c6 8.a3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.a3 *"),
("D64", "QGD: Orthodox, Rubinstein Attack", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Qc2 *"),
("D64", "QGD: Orthodox, Rubinstein Attack, 8...dxc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Qc2 dxc4 *"),
("D64", "QGD: Orthodox, Rubinstein Attack, Wolf Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Qc2 Ne4 *"),
("D64", "QGD: Orthodox, Rubinstein Attack, 8...Re8", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Qc2 Re8 *"),
("D64", "QGD: Orthodox, Rubinstein Attack, 8...Re8 9.a3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Qc2 Re8 9.a3 *"),
("D64", "QGD: Orthodox, Rubinstein Attack, 8...Re8 9.a3 a6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Qc2 Re8 9.a3 a6 *"),
("D64", "QGD: Orthodox, Rubinstein Attack, 8...h6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Qc2 h6 *"),
("D64", "QGD: Orthodox, Rubinstein Attack, Carlsbad Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Qc2 a6 *"),
("D64", "QGD: Orthodox, Rubinstein Attack, Gruenfeld Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Qc2 a6 9.a3 *"),
("D64", "QGD: Orthodox, Rubinstein Attack, Gruenfeld Variation, 9...h6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Qc2 a6 9.a3 h6 *"),
("D64", "QGD: Orthodox, Rubinstein Attack, Gruenfeld Variation, 9...h6 10.Bh4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Qc2 a6 9.a3 h6 10.Bh4 *"),
("D65", "QGD: Orthodox, Rubinstein Attack, 9.cxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Qc2 a6 9.cxd5 *"),
("D65", "QGD: Orthodox, Rubinstein Attack, 9.cxd5 Nxd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Qc2 a6 9.cxd5 Nxd5 *"),
("D65", "QGD: Orthodox, Rubinstein Attack, 9.cxd5 exd5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Qc2 a6 9.cxd5 exd5 *"),
("D65", "QGD: Orthodox, Rubinstein, 9.cxd5 exd5 10.Bd3 h6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Qc2 a6 9.cxd5 exd5 10.Bd3 h6 *"),
("D65", "QGD: Orthodox, Rubinstein, 9.cxd5 exd5 10.Bd3 h6 11.Bh4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Qc2 a6 9.cxd5 exd5 10.Bd3 h6 11.Bh4 *"),
("D65", "QGD: Orthodox, Rubinstein, 9.cxd5 exd5 10.Bd3 Re8", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Qc2 a6 9.cxd5 exd5 10.Bd3 Re8 *"),
("D65", "QGD: Orthodox, Rubinstein, 9.cxd5 exd5 10.Bd3 Re8 11.O-O Nf8", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Qc2 a6 9.cxd5 exd5 10.Bd3 Re8 11.O-O Nf8 *"),
("D65", "QGD: Orthodox, Rubinstein, 9.cxd5 exd5 10.Bd3 Re8 11.O-O Nf8 12.Ne5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Qc2 a6 9.cxd5 exd5 10.Bd3 Re8 11.O-O Nf8 12.Ne5 *"),
("D65", "QGD: Orthodox, Rubinstein, 9.cxd5 exd5 10.Bd3 Re8 11.O-O Nf8 12.h3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Qc2 a6 9.cxd5 exd5 10.Bd3 Re8 11.O-O Nf8 12.h3 *"),
("D66", "QGD: Orthodox, Main Line", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 *"),
("D66", "QGD: Orthodox, Main Line, 8...a6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 a6 *"),
("D66", "QGD: Orthodox, Main Line, 8...h6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 h6 *"),
("D66", "QGD: Orthodox, Main Line, 8...h6 9.Bh4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 h6 9.Bh4 *"),
("D66", "QGD: Orthodox, Main Line, 8...h6 9.Bh4 dxc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 h6 9.Bh4 dxc4 *"),
("D66", "QGD: Orthodox, Main Line, 8...h6 9.Bh4 dxc4 10.Bxc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 h6 9.Bh4 dxc4 10.Bxc4 *"),
("D66", "QGD: Orthodox, Main Line, 8...h6 9.Bh4 dxc4 10.Bxc4 b5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 h6 9.Bh4 dxc4 10.Bxc4 b5 *"),
("D66", "QGD: Orthodox, Main Line, 8...Re8", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 Re8 *"),
("D66", "QGD: Orthodox, Main Line, 8...dxc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 *"),
("D66", "QGD: Orthodox, Main Line, 8...dxc4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 *"),
("D66", "QGD: Orthodox, Main Line, Fianchetto Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 b5 *"),
("D66", "QGD: Orthodox, Main Line, Fianchetto Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 b5 10.Bd3 a6 *"),
("D66", "QGD: Orthodox, Main Line, Fianchetto, 11.O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 b5 10.Bd3 a6 11.O-O *"),
("D66", "QGD: Orthodox, Main Line, Fianchetto, 11.e4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 b5 10.Bd3 a6 11.e4 *"),
("D67", "QGD: Orthodox, Main Line, Capablanca Freeing Manoevure", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 Nd5 *"),
("D67", "QGD: Orthodox, Main Line, Janowski Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 Nd5 10.h4 *"),
("D67", "QGD: Orthodox, Main Line, Capablanca Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 Nd5 10.Bxe7 Qxe7 *"),
("D67", "QGD: Orthodox, Main Line, Alekhine Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 Nd5 10.Bxe7 Qxe7 11.Ne4 *"),
("D67", "QGD: Orthodox, Main Line, Alekhine, 11...e5 (Lasker)", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 Nd5 10.Bxe7 Qxe7 11.Ne4 e5 *"),
("D67", "QGD: Orthodox, Main Line, Alekhine, 11...N5f6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 Nd5 10.Bxe7 Qxe7 11.Ne4 N5f6 *"),
("D67", "QGD: Orthodox, Main Line, Capablanca, 11.O-O", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 Nd5 10.Bxe7 Qxe7 11.O-O *"),
("D67", "QGD: Orthodox, Main Line, Capablanca, 11.O-O Nxc3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 Nd5 10.Bxe7 Qxe7 11.O-O Nxc3 *"),
("D68", "QGD: Orthodox, Classical Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 Nd5 10.Bxe7 Qxe7 11.O-O Nxc3 12.Rxc3 e5 *"),
("D68", "QGD: Orthodox, Classical, 13.Bb3", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 Nd5 10.Bxe7 Qxe7 11.O-O Nxc3 12.Rxc3 e5 13.Bb3 *"),
("D68", "QGD: Orthodox, Classical, Maroczy Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 Nd5 10.Bxe7 Qxe7 11.O-O Nxc3 12.Rxc3 e5 13.Qb1 *"),
("D68", "QGD: Orthodox, Classical, Maroczy, 13...exd4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 Nd5 10.Bxe7 Qxe7 11.O-O Nxc3 12.Rxc3 e5 13.Qb1 exd4 *"),
("D68", "QGD: Orthodox, Classical, Vidmar Variation", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 Nd5 10.Bxe7 Qxe7 11.O-O Nxc3 12.Rxc3 e5 13.Qc2 *"),
("D68", "QGD: Orthodox, Classical, Vidmar, 13...e4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 Nd5 10.Bxe7 Qxe7 11.O-O Nxc3 12.Rxc3 e5 13.Qc2 e4 *"),
("D68", "QGD: Orthodox, Classical, Vidmar, 13...exd4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 Nd5 10.Bxe7 Qxe7 11.O-O Nxc3 12.Rxc3 e5 13.Qc2 exd4 *"),
("D68", "QGD: Orthodox, Classical, Vidmar, 13...exd4 14.exd4 Nb6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 Nd5 10.Bxe7 Qxe7 11.O-O Nxc3 12.Rxc3 e5 13.Qc2 exd4 14.exd4 Nb6 *"),
("D69", "QGD: Orthodox, Classical, 13.dxe5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 Nd5 10.Bxe7 Qxe7 11.O-O Nxc3 12.Rxc3 e5 13.dxe5 *"),
("D69", "QGD: Orthodox, Classical, 13.dxe5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 Nd5 10.Bxe7 Qxe7 11.O-O Nxc3 12.Rxc3 e5 13.dxe5 Nxe5 14.Nxe5 Qxe5 *"),
("D69", "QGD: Orthodox, Classical, 13.dxe5: 15.f4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 Nd5 10.Bxe7 Qxe7 11.O-O Nxc3 12.Rxc3 e5 13.dxe5 Nxe5 14.Nxe5 Qxe5 15.f4 *"),
("D69", "QGD: Orthodox, Classical, 13.dxe5: 15.f4 Qe4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 Nd5 10.Bxe7 Qxe7 11.O-O Nxc3 12.Rxc3 e5 13.dxe5 Nxe5 14.Nxe5 Qxe5 15.f4 Qe4 *"),
("D69", "QGD: Orthodox, Classical, 13.dxe5: 15.f4 Qf6", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 Nd5 10.Bxe7 Qxe7 11.O-O Nxc3 12.Rxc3 e5 13.dxe5 Nxe5 14.Nxe5 Qxe5 15.f4 Qf6 *"),
("D69", "QGD: Orthodox, Classical, 13.dxe5: 15.f4 Qf6 16.f5", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 Nd5 10.Bxe7 Qxe7 11.O-O Nxc3 12.Rxc3 e5 13.dxe5 Nxe5 14.Nxe5 Qxe5 15.f4 Qf6 16.f5 *"),
("D69", "QGD: Orthodox, Classical, 13.dxe5: 15.f4 Qf6 16.e4", "1.d4 d5 2.c4 e6 3.Nc3 Nf6 4.Bg5 Be7 5.e3 O-O 6.Nf3 Nbd7 7.Rc1 c6 8.Bd3 dxc4 9.Bxc4 Nd5 10.Bxe7 Qxe7 11.O-O Nxc3 12.Rxc3 e5 13.dxe5 Nxe5 14.Nxe5 Qxe5 15.f4 Qf6 16.e4 *"),
("D70", "Neo-Gruenfeld: 3.Nf3 d5", "1.d4 Nf6 2.c4 g6 3.Nf3 d5 *"),
("D70", "Neo-Gruenfeld: Alekhine's Anti-Gruenfeld", "1.d4 Nf6 2.c4 g6 3.f3 d5 *"),
("D70", "Neo-Gruenfeld: Alekhine's, 5.e4 Nb6", "1.d4 Nf6 2.c4 g6 3.f3 d5 4.cxd5 Nxd5 5.e4 Nb6 *"),
("D70", "Neo-Gruenfeld: Alekhine's, 7.Be3 O-O", "1.d4 Nf6 2.c4 g6 3.f3 d5 4.cxd5 Nxd5 5.e4 Nb6 6.Nc3 Bg7 7.Be3 O-O *"),
("D70", "Neo-Gruenfeld: 3.g3 d5", "1.d4 Nf6 2.c4 g6 3.g3 d5 *"),
("D70", "Neo-Gruenfeld: 4.cxd5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.cxd5 *"),
("D70", "Neo-Gruenfeld: 4.Bg2", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 *"),
("D70", "Neo-Gruenfeld: 4.Bg2 c6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 c6 *"),
("D70", "Neo-Gruenfeld: 4.Bg2 Bg7", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 *"),
("D71", "Neo-Gruenfeld, 5.cxd5 Nxd5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.cxd5 Nxd5 *"),
("D71", "Neo-Gruenfeld, 5.cxd5 Nxd5 6.Nc3", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.cxd5 Nxd5 6.Nc3 *"),
("D71", "Neo-Gruenfeld, 5.cxd5 Nxd5 6.Nc3 Nxc3", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.cxd5 Nxd5 6.Nc3 Nxc3 *"),
("D71", "Neo-Gruenfeld, 5.cxd5 Nxd5 6.Nc3 Nxc3 7.bxc3 c5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.cxd5 Nxd5 6.Nc3 Nxc3 7.bxc3 c5 *"),
("D71", "Neo-Gruenfeld, 5.cxd5 Nxd5 6.Nc3 Nb6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.cxd5 Nxd5 6.Nc3 Nb6 *"),
("D71", "Neo-Gruenfeld, 5.cxd5 Nxd5 6.e4", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.cxd5 Nxd5 6.e4 *"),
("D71", "Neo-Gruenfeld, 5.cxd5 Nxd5 6.e4 Nb6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.cxd5 Nxd5 6.e4 Nb6 *"),
("D71", "Neo-Gruenfeld, 5.cxd5 Nxd5 6.e4 Nb4", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.cxd5 Nxd5 6.e4 Nb4 *"),
("D71", "Neo-Gruenfeld, 5.cxd5 Nxd5 6.e4 Nb4 7.d5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.cxd5 Nxd5 6.e4 Nb4 7.d5 *"),
("D72", "Neo-Gruenfeld, 5.cxd5 Nxd5 6.e4 Nb6 7.Ne2", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.cxd5 Nxd5 6.e4 Nb6 7.Ne2 *"),
("D72", "Neo-Gruenfeld, 5.cxd5 Nxd5 6.e4 Nb6 7.Ne2 Nc6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.cxd5 Nxd5 6.e4 Nb6 7.Ne2 Nc6 *"),
("D72", "Neo-Gruenfeld, 5.cxd5 Nxd5 6.e4 Nb6 7.Ne2 e5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.cxd5 Nxd5 6.e4 Nb6 7.Ne2 e5 *"),
("D72", "Neo-Gruenfeld, 5.cxd5 Nxd5 6.e4 Nb6 7.Ne2 c5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.cxd5 Nxd5 6.e4 Nb6 7.Ne2 c5 *"),
("D72", "Neo-Gruenfeld, 5.cxd5 Nxd5 6.e4 Nb6 7.Ne2 O-O", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.cxd5 Nxd5 6.e4 Nb6 7.Ne2 O-O *"),
("D72", "Neo-Gruenfeld, 5.cxd5 Nxd5 6.e4 Nb6 7.Ne2 O-O 8.O-O", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.cxd5 Nxd5 6.e4 Nb6 7.Ne2 O-O 8.O-O *"),
("D73", "Neo-Gruenfeld, 5.Nf3", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 *"),
("D73", "Neo-Gruenfeld, 5.Nf3 c6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 c6 *"),
("D73", "Neo-Gruenfeld, 5.Nf3 c6 6.cxd5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 c6 6.cxd5 *"),
("D73", "Neo-Gruenfeld, 5.Nf3 c6 6.cxd5 cxd5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 c6 6.cxd5 cxd5 *"),
("D73", "Neo-Gruenfeld, 5.Nf3 c6 6.cxd5 cxd5 7.Nc3", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 c6 6.cxd5 cxd5 7.Nc3 *"),
("D73", "Neo-Gruenfeld, 5.Nf3 c5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 c5 *"),
("D73", "Neo-Gruenfeld, 5.Nf3 dxc4", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 dxc4 *"),
("D73", "Neo-Gruenfeld, 5.Nf3 dxc4 6.Na3", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 dxc4 6.Na3 *"),
("D73", "Neo-Gruenfeld, 5.Nf3 O-O", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O *"),
("D73", "Neo-Gruenfeld, 6.Qb3", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.Qb3 *"),
("D73", "Neo-Gruenfeld, 6.Nc3", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.Nc3 *"),
("D73", "Neo-Gruenfeld, 6.cxd5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 *"),
("D73", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.e4", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.e4 *"),
("D74", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O *"),
("D74", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O Na6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O Na6 *"),
("D74", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O Nc6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O Nc6 *"),
("D74", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O c6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O c6 *"),
("D74", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O c5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O c5 *"),
("D74", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O c5 8.e4", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O c5 8.e4 *"),
("D74", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O c5 8.e4 Nf6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O c5 8.e4 Nf6 *"),
("D74", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O c5 8.e4 Nf6 9.e5 Nd5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O c5 8.e4 Nf6 9.e5 Nd5 *"),
("D75", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O c5 8.Nc3", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O c5 8.Nc3 *"),
("D75", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O c5 8.Nc3 Nxc3", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O c5 8.Nc3 Nxc3 *"),
("D75", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O c5 8.Nc3 Nxc3 9.bxc3 cxd4", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O c5 8.Nc3 Nxc3 9.bxc3 cxd4 *"),
("D75", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O c5 8.dxc5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O c5 8.dxc5 *"),
("D75", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O c5 8.dxc5 Na6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O c5 8.dxc5 Na6 *"),
("D76a", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O Nb6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O Nb6 *"),
("D76b", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 *"),
("D76c", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 Nc6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 Nc6 *"),
("D76d", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 Nc6 9.e3", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 Nc6 9.e3 *"),
("D76e", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 Nc6 9.e3 e5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 Nc6 9.e3 e5 *"),
("D76e", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 Nc6 9.e3 e5 10.d5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 Nc6 9.e3 e5 10.d5 *"),
("D76f", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 Nc6 9.e3 e5 10.d5 Na5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 Nc6 9.e3 e5 10.d5 Na5 *"),
("D76g", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 Nc6 9.e3 Re8", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 Nc6 9.e3 Re8 *"),
("D76h", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 Nc6 9.e3 Re8 10.d5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 Nc6 9.e3 Re8 10.d5 *"),
("D76i", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 Nc6 9.d5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 Nc6 9.d5 *"),
("D76j", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 Nc6 9.d5 Na5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 Nc6 9.d5 Na5 *"),
("D76k", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 Nc6 9.d5 Na5 10.e4", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 Nc6 9.d5 Na5 10.e4 *"),
("D76k", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 Nc6 9.d5 Na5 10.e4 c6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 Nc6 9.d5 Na5 10.e4 c6 *"),
("D76l", "Neo-Gruenfeld, 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 Nc6 9.d5 Na5 10.e4 c6 11.Bg5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.cxd5 Nxd5 7.O-O Nb6 8.Nc3 Nc6 9.d5 Na5 10.e4 c6 11.Bg5 *"),
("D77a", "Neo-Gruenfeld, 6.O-O", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O *"),
("D77b", "Neo-Gruenfeld, 6.O-O e6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O e6 *"),
("D77c", "Neo-Gruenfeld, 6.O-O c5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c5 *"),
("D77c", "Neo-Gruenfeld, 6.O-O c5 7.dxc5 dxc4", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c5 7.dxc5 dxc4 *"),
("D77d", "Neo-Gruenfeld, 6.O-O Nc6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O Nc6 *"),
("D77e", "Neo-Gruenfeld, 6.O-O dxc4", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O dxc4 *"),
("D77f", "Neo-Gruenfeld, 6.O-O dxc4 7.Na3", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O dxc4 7.Na3 *"),
("D77g", "Neo-Gruenfeld, 6.O-O dxc4 7.Na3 Na6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O dxc4 7.Na3 Na6 *"),
("D77h", "Neo-Gruenfeld, 6.O-O dxc4 7.Na3 c3", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O dxc4 7.Na3 c3 *"),
("D77h", "Neo-Gruenfeld, 6.O-O dxc4 7.Na3 c3 8.bxc3 c5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O dxc4 7.Na3 c3 8.bxc3 c5 *"),
("D77i", "Neo-Gruenfeld, 6.O-O dxc4 7.Na3 c3 8.bxc3 c5 9.e3", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O dxc4 7.Na3 c3 8.bxc3 c5 9.e3 *"),
("D77j", "Neo-Gruenfeld, 6.O-O dxc4 7.Na3 Nc6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O dxc4 7.Na3 Nc6 *"),
("D77j", "Neo-Gruenfeld, 6.O-O dxc4 7.Na3 Nc6 8.Nxc4", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O dxc4 7.Na3 Nc6 8.Nxc4 *"),
("D77j", "Neo-Gruenfeld, 6.O-O dxc4 7.Na3 Nc6 8.Nxc4 Be6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O dxc4 7.Na3 Nc6 8.Nxc4 Be6 *"),
("D77k", "Neo-Gruenfeld, 6.O-O dxc4 7.Na3 Nc6 8.Nxc4 Be6 9.b3", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O dxc4 7.Na3 Nc6 8.Nxc4 Be6 9.b3 *"),
("D77l", "Neo-Gruenfeld, 6.O-O dxc4 7.Na3 Nc6 8.Nxc4 Be6 9.b3 Bd5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O dxc4 7.Na3 Nc6 8.Nxc4 Be6 9.b3 Bd5 *"),
("D78a", "Neo-Gruenfeld, 6.O-O c6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 *"),
("D78b", "Neo-Gruenfeld, 6.O-O c6 7.Na3", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.Na3 *"),
("D78c", "Neo-Gruenfeld, 6.O-O c6 7.Ne5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.Ne5 *"),
("D78d", "Neo-Gruenfeld, 6.O-O c6 7.Qa4", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.Qa4 *"),
("D78e", "Neo-Gruenfeld, 6.O-O c6 7.Nc3", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.Nc3 *"),
("D78f", "Neo-Gruenfeld, 6.O-O c6 7.Nc3 dxc4", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.Nc3 dxc4 *"),
("D78g", "Neo-Gruenfeld, 6.O-O c6 7.Qb3", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.Qb3 *"),
("D78h", "Neo-Gruenfeld, 6.O-O c6 7.Qb3 Qb6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.Qb3 Qb6 *"),
("D78i", "Neo-Gruenfeld, 6.O-O c6 7.Qb3 dxc4", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.Qb3 dxc4 *"),
("D78j", "Neo-Gruenfeld, 6.O-O c6 7.Qb3 dxc4", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.Qb3 dxc4 8.Qxc4 *"),
("D78k", "Neo-Gruenfeld, 6.O-O c6 7.Nbd2", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.Nbd2 *"),
("D78l", "Neo-Gruenfeld, 6.O-O c6 7.Nbd2 Nbd7", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.Nbd2 Nbd7 *"),
("D78m", "Neo-Gruenfeld, 6.O-O c6 7.Nbd2 Ne4", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.Nbd2 Ne4 *"),
("D78n", "Neo-Gruenfeld, 6.O-O c6 7.Nbd2 Bf5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.Nbd2 Bf5 *"),
("D78o", "Neo-Gruenfeld, 6.O-O c6 7.b3", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.b3 *"),
("D78p", "Neo-Gruenfeld, 6.O-O c6 7.b3 Bf5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.b3 Bf5 *"),
("D78q", "Neo-Gruenfeld, 6.O-O c6 7.b3 Ne4", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.b3 Ne4 *"),
("D79a", "Neo-Gruenfeld, 6.O-O c6 7.cxd5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.cxd5 *"),
("D79a", "Neo-Gruenfeld, 6.O-O c6 7.cxd5 cxd5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.cxd5 cxd5 *"),
("D79b", "Neo-Gruenfeld, 6.O-O c6 7.cxd5 cxd5 8.Nc3", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.cxd5 cxd5 8.Nc3 *"),
("D79c", "Neo-Gruenfeld, 6.O-O c6 7.cxd5 cxd5 8.Nc3 e6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.cxd5 cxd5 8.Nc3 e6 *"),
("D79d", "Neo-Gruenfeld, 6.O-O c6 7.cxd5 cxd5 8.Nc3 Ne4", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.cxd5 cxd5 8.Nc3 Ne4 *"),
("D79d", "Neo-Gruenfeld, 6.O-O c6 7.cxd5 cxd5 8.Nc3 Ne4 9.Ne5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.cxd5 cxd5 8.Nc3 Ne4 9.Ne5 *"),
("D79e", "Neo-Gruenfeld, 6.O-O c6 7.cxd5 cxd5 8.Nc3 Ne4 9.Nxe4", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.cxd5 cxd5 8.Nc3 Ne4 9.Nxe4 *"),
("D79f", "Neo-Gruenfeld, 6.O-O c6 7.cxd5 cxd5 8.Nc3 Nc6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.cxd5 cxd5 8.Nc3 Nc6 *"),
("D79g", "Neo-Gruenfeld, 6.O-O c6 7.cxd5 cxd5 8.Nc3 Nc6 9.Ne5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.cxd5 cxd5 8.Nc3 Nc6 9.Ne5 *"),
("D79h", "Neo-Gruenfeld, 6.O-O c6 7.cxd5 cxd5 8.Nc3 Nc6 9.Ne5 e6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.cxd5 cxd5 8.Nc3 Nc6 9.Ne5 e6 *"),
("D79i", "Neo-Gruenfeld, 6.O-O c6 7.cxd5 cxd5 8.Ne5", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.cxd5 cxd5 8.Ne5 *"),
("D79i", "Neo-Gruenfeld, 6.O-O c6 7.cxd5 cxd5 8.Ne5 e6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.cxd5 cxd5 8.Ne5 e6 *"),
("D79j", "Neo-Gruenfeld, Main Line", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.cxd5 cxd5 8.Ne5 e6 9.Nc3 *"),
("D79k", "Neo-Gruenfeld, Main Line, 10.f4", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.cxd5 cxd5 8.Ne5 e6 9.Nc3 Nfd7 10.f4 *"),
("D79l", "Neo-Gruenfeld, Main Line, 10.f4 Nc6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.cxd5 cxd5 8.Ne5 e6 9.Nc3 Nfd7 10.f4 Nc6 *"),
("D79m", "Neo-Gruenfeld, Main Line, 10.f4 Nc6 11.Be3", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.cxd5 cxd5 8.Ne5 e6 9.Nc3 Nfd7 10.f4 Nc6 11.Be3 *"),
("D79m", "Neo-Gruenfeld, Main Line, 10.f4 Nc6 11.Be3 f6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.cxd5 cxd5 8.Ne5 e6 9.Nc3 Nfd7 10.f4 Nc6 11.Be3 f6 *"),
("D79n", "Neo-Gruenfeld, Main Line, 10.f4 Nc6 11.Be3 Nb6", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.cxd5 cxd5 8.Ne5 e6 9.Nc3 Nfd7 10.f4 Nc6 11.Be3 Nb6 *"),
("D79n", "Neo-Gruenfeld, Main Line, 10.f4 Nc6 11.Be3 Nb6 12.Bf2", "1.d4 Nf6 2.c4 g6 3.g3 d5 4.Bg2 Bg7 5.Nf3 O-O 6.O-O c6 7.cxd5 cxd5 8.Ne5 e6 9.Nc3 Nfd7 10.f4 Nc6 11.Be3 Nb6 12.Bf2 *"),
("D80", "Gruenfeld Defence", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 *"),
("D80", "Gruenfeld: Spike/Gibbon Gambit", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.g4 *"),
("D80", "Gruenfeld: 4.h4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.h4 *"),
("D80", "Gruenfeld: 4.g3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.g3 *"),
("D80", "Gruenfeld: 4.f3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.f3 *"),
("D80", "Gruenfeld: 4.e3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.e3 *"),
("D80", "Gruenfeld: 4.e3 Bg7", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.e3 Bg7 *"),
("D80", "Gruenfeld: 4.e3 Bg7 5.Qb3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.e3 Bg7 5.Qb3 *"),
("D80", "Gruenfeld: Stockholm Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bg5 *"),
("D80", "Gruenfeld: Stockholm, 4...Ne4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bg5 Ne4 *"),
("D80", "Gruenfeld: Stockholm, 4...Ne4 5.Nxe4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bg5 Ne4 5.Nxe4 *"),
("D80", "Gruenfeld: Stockholm, Lundin Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bg5 Ne4 5.Nxe4 dxe4 6.Qd2 c5 *"),
("D80", "Gruenfeld: Stockholm, 4...Ne4 5.Bf4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bg5 Ne4 5.Bf4 *"),
("D80", "Gruenfeld: Stockholm, Taimanov Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bg5 Ne4 5.Bh4 *"),
("D80", "Gruenfeld: Stockholm, Taimanov, 5...Nxc3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bg5 Ne4 5.Bh4 Nxc3 *"),
("D80", "Gruenfeld: Stockholm, Taimanov, 5...Nxc3 6.bxc3 Bg7", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bg5 Ne4 5.Bh4 Nxc3 6.bxc3 Bg7 *"),
("D81", "Gruenfeld: Early Russian Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Qb3 *"),
("D81", "Gruenfeld: Early Russian Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Qb3 dxc4 5.Qxc4 *"),
("D81", "Gruenfeld: Early Russian, 5...Be6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Qb3 dxc4 5.Qxc4 Be6 *"),
("D81", "Gruenfeld: Early Russian, 5...Be6 6.Qb5+", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Qb3 dxc4 5.Qxc4 Be6 6.Qb5+ *"),
("D81", "Gruenfeld: Early Russian, Adorjan Gambit", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Qb3 dxc4 5.Qxc4 Be6 6.Qb5+ Bd7 *"),
("D81", "Gruenfeld: Early Russian, 5...Bg7", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Qb3 dxc4 5.Qxc4 Bg7 *"),
("D81", "Gruenfeld: Early Russian, 5...Bg7 6.e4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Qb3 dxc4 5.Qxc4 Bg7 6.e4 *"),
("D81", "Gruenfeld: Early Russian, 5...Bg7 6.e4 O-O", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Qb3 dxc4 5.Qxc4 Bg7 6.e4 O-O *"),
("D82", "Gruenfeld: 4.Bf4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bf4 *"),
("D82", "Gruenfeld: 4.Bf4 Bg7", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bf4 Bg7 *"),
("D82", "Gruenfeld: 4.Bf4 Bg7 5.e3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bf4 Bg7 5.e3 *"),
("D82", "Gruenfeld: 4.Bf4 Bg7 5.e3 c5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bf4 Bg7 5.e3 c5 *"),
("D82", "Gruenfeld: 4.Bf4 Bg7 5.e3 c5 6.dxc5 Qa5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bf4 Bg7 5.e3 c5 6.dxc5 Qa5 *"),
("D82", "Gruenfeld: 4.Bf4 Bg7 5.e3 c5 6.dxc5 Qa5 7.Qa4+", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bf4 Bg7 5.e3 c5 6.dxc5 Qa5 7.Qa4+ *"),
("D82", "Gruenfeld: 4.Bf4 Bg7 5.e3 c5 6.dxc5 Qa5 7.Rc1", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bf4 Bg7 5.e3 c5 6.dxc5 Qa5 7.Rc1 *"),
("D83", "Gruenfeld: Gruenfeld Gambit", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bf4 Bg7 5.e3 O-O *"),
("D83", "Gruenfeld: Gruenfeld Gambit, 6.Qb3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bf4 Bg7 5.e3 O-O 6.Qb3 *"),
("D83", "Gruenfeld: Gruenfeld Gambit, Capablanca Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bf4 Bg7 5.e3 O-O 6.Rc1 *"),
("D83", "Gruenfeld: Gruenfeld Gambit, Capablanca Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bf4 Bg7 5.e3 O-O 6.Rc1 c5 *"),
("D83", "Gruenfeld: Gruenfeld Gambit, Botvinnik Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bf4 Bg7 5.e3 O-O 6.Rc1 c5 7.dxc5 Be6 *"),
("D84", "Gruenfeld: Gruenfeld Gambit Accepted", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bf4 Bg7 5.e3 O-O 6.cxd5 *"),
("D84", "Gruenfeld: Gruenfeld Gambit Accepted", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bf4 Bg7 5.e3 O-O 6.cxd5 Nxd5 7.Nxd5 Qxd5 8.Bxc7 *"),
("D84", "Gruenfeld: Gruenfeld Gambit Accepted, 8...b6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bf4 Bg7 5.e3 O-O 6.cxd5 Nxd5 7.Nxd5 Qxd5 8.Bxc7 b6 *"),
("D84", "Gruenfeld: Gruenfeld Gambit Accepted, 8...Bf5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bf4 Bg7 5.e3 O-O 6.cxd5 Nxd5 7.Nxd5 Qxd5 8.Bxc7 Bf5 *"),
("D84", "Gruenfeld: Gruenfeld Gambit Accepted, 8...Na6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bf4 Bg7 5.e3 O-O 6.cxd5 Nxd5 7.Nxd5 Qxd5 8.Bxc7 Na6 *"),
("D84", "Gruenfeld: Gruenfeld Gambit Accepted, 8...Na6 9.Bxa6 Qxg2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bf4 Bg7 5.e3 O-O 6.cxd5 Nxd5 7.Nxd5 Qxd5 8.Bxc7 Na6 9.Bxa6 Qxg2 *"),
("D84", "Gruenfeld: Gruenfeld Gambit Accepted, 8...Nc6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bf4 Bg7 5.e3 O-O 6.cxd5 Nxd5 7.Nxd5 Qxd5 8.Bxc7 Nc6 *"),
("D84", "Gruenfeld: Gruenfeld Gambit Accepted, 8...Nc6 9.Ne2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bf4 Bg7 5.e3 O-O 6.cxd5 Nxd5 7.Nxd5 Qxd5 8.Bxc7 Nc6 9.Ne2 *"),
("D84", "Gruenfeld: Gruenfeld Gambit Accepted, 8...Nc6 9.Ne2 Bg4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Bf4 Bg7 5.e3 O-O 6.cxd5 Nxd5 7.Nxd5 Qxd5 8.Bxc7 Nc6 9.Ne2 Bg4 *"),
("D85a", "Gruenfeld: Exchange Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 *"),
("D85a", "Gruenfeld: Exchange Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 *"),
("D85a", "Gruenfeld: Exchange, 5.g3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.g3 *"),
("D85a", "Gruenfeld: Exchange, 5.Na4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.Na4 *"),
("D85b", "Gruenfeld: Exchange, 5.Bd2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.Bd2 *"),
("D85c", "Gruenfeld: Exchange, 5.Bd2 Bg7 6.e4 Nb6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.Bd2 Bg7 6.e4 Nb6 *"),
("D85c", "Gruenfeld: Exchange, 5.Bd2 Bg7 6.e4 Nb6 7.Be3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.Bd2 Bg7 6.e4 Nb6 7.Be3 *"),
("D85d", "Gruenfeld: Exchange, 5.e4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 *"),
("D85d", "Gruenfeld: Exchange, 5.e4 Nb6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nb6 *"),
("D85d", "Gruenfeld: Exchange, 5.e4 Nxc3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 *"),
("D85d", "Gruenfeld: Exchange, 5.e4 Nxc3 6.bxc3 c5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 c5 *"),
("D85d", "Gruenfeld: Exchange, 5.e4 Nxc3 6.bxc3 Bg7", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 *"),
("D85d", "Gruenfeld: Exchange, 7.Ba3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Ba3 *"),
("D85e", "Gruenfeld: Exchange, 7.Bb5+", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bb5+ *"),
("D85e", "Gruenfeld: Exchange, 7.Bb5+ c6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bb5+ c6 *"),
("D85e", "Gruenfeld: Exchange, 7.Bb5+ c6 8.Ba4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bb5+ c6 8.Ba4 *"),
("D85f", "Gruenfeld: Exchange, 7.Bb5+ c6 8.Ba4 O-O", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bb5+ c6 8.Ba4 O-O *"),
("D85g", "Gruenfeld: Exchange, 7.Be3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Be3 *"),
("D85g", "Gruenfeld: Exchange, 7.Be3 c5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Be3 c5 *"),
("D85g", "Gruenfeld: Exchange, 7.Be3 c5 8.Qd2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Be3 c5 8.Qd2 *"),
("D85g", "Gruenfeld: Exchange, 7.Be3 c5 8.Qd2 O-O", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Be3 c5 8.Qd2 O-O *"),
("D85g", "Gruenfeld: Exchange, 7.Be3 c5 8.Qd2 O-O 9.Rc1", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Be3 c5 8.Qd2 O-O 9.Rc1 *"),
("D85h", "Gruenfeld: Exchange, 7.Be3 c5 8.Qd2 Qa5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Be3 c5 8.Qd2 Qa5 *"),
("D85h", "Gruenfeld: Exchange, 7.Be3 c5 8.Qd2 Qa5 9.Rc1", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Be3 c5 8.Qd2 Qa5 9.Rc1 *"),
("D85h", "Gruenfeld: Exchange, 7.Be3 c5 8.Qd2 Qa5 9.Rb1", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Be3 c5 8.Qd2 Qa5 9.Rb1 *"),
("D85h", "Gruenfeld: Exchange, 7.Be3 c5 8.Qd2 Qa5 9.Rb1 b6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Be3 c5 8.Qd2 Qa5 9.Rb1 b6 *"),
("D85i", "Gruenfeld: Modern Exchange", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 *"),
("D85i", "Gruenfeld: Modern Exchange, 7...O-O", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 O-O *"),
("D85j", "Gruenfeld: Modern Exchange, 7...c5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 *"),
("D85k", "Gruenfeld: Modern Exchange, 8.Bb5+", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Bb5+ *"),
("D85l", "Gruenfeld: Modern Exchange, 8.Be2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Be2 *"),
("D85l", "Gruenfeld: Modern Exchange, 8.Be2 O-O", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Be2 O-O *"),
("D85l", "Gruenfeld: Modern Exchange, 8.Be2 O-O 9.O-O", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Be2 O-O 9.O-O *"),
("D85l", "Gruenfeld: Modern Exchange, 8.Be2 O-O 9.O-O b6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Be2 O-O 9.O-O b6 *"),
("D85l", "Gruenfeld: Modern Exchange, 8.Be2 O-O 9.O-O Nc6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Be2 O-O 9.O-O Nc6 *"),
("D85m", "Gruenfeld: Modern Exchange, 8.Rb1", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Rb1 *"),
("D85m", "Gruenfeld: Modern Exchange, 8.Rb1 O-O", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Rb1 O-O *"),
("D85m", "Gruenfeld: Modern Exchange, 8.Rb1 O-O 9.Be2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Rb1 O-O 9.Be2 *"),
("D85m", "Gruenfeld: Modern Exchange, 8.Rb1 O-O 9.Be2 Qa5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Rb1 O-O 9.Be2 Qa5 *"),
("D85n", "Gruenfeld: Modern Exchange, 8.Rb1 O-O 9.Be2 b6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Rb1 O-O 9.Be2 b6 *"),
("D85o", "Gruenfeld: Modern Exchange, 8.Rb1 O-O 9.Be2 Nc6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Rb1 O-O 9.Be2 Nc6 *"),
("D85p", "Gruenfeld: Modern Exchange, 8.Rb1 O-O 9.Be2 Nc6, Main Line", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Rb1 O-O 9.Be2 Nc6 10.d5 Ne5 11.Nxe5 Bxe5 12.Qd2 *"),
("D85p", "Gruenfeld: Modern Exchange, 8.Rb1 O-O 9.Be2 Nc6, Main Line, 12...e6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Rb1 O-O 9.Be2 Nc6 10.d5 Ne5 11.Nxe5 Bxe5 12.Qd2 e6 *"),
("D85q", "Gruenfeld: Modern Exchange, 8.Rb1 O-O 9.Be2 cd", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Rb1 O-O 9.Be2 cxd4 *"),
("D85q", "Gruenfeld: Modern Exchange, 8.Rb1 O-O 9.Be2 cd 10.cd", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Rb1 O-O 9.Be2 cxd4 10.cxd4 *"),
("D85q", "Gruenfeld: Modern Exchange, 8.Rb1 O-O 9.Be2 cd 10.cd Qa5+", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Rb1 O-O 9.Be2 cxd4 10.cxd4 Qa5+ *"),
("D85q", "Gruenfeld: Modern Exchange, 8.Rb1 O-O 9.Be2 cd 10.cd Qa5+ 11.Qd2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Rb1 O-O 9.Be2 cxd4 10.cxd4 Qa5+ 11.Qd2 *"),
("D85r", "Gruenfeld: Modern Exchange, 8.Rb1 O-O 9.Be2 cd 10.cd Qa5+ 11.Bd2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Rb1 O-O 9.Be2 cxd4 10.cxd4 Qa5+ 11.Bd2 *"),
("D85s", "Gruenfeld: Modern Exchange, 8.Rb1, 10...Qa5+ 11.Bd2 Qxa2 12.O-O Bg4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Rb1 O-O 9.Be2 cxd4 10.cxd4 Qa5+ 11.Bd2 Qxa2 12.O-O Bg4 *"),
("D85s", "Gruenfeld: Modern Exchange, 8.Rb1, 10...Qa5+ 11.Bd2 Qxa2 12.O-O Bg4 13.Bg5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Rb1 O-O 9.Be2 cxd4 10.cxd4 Qa5+ 11.Bd2 Qxa2 12.O-O Bg4 13.Bg5 *"),
("D85t", "Gruenfeld: Modern Exchange, 8.Be3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Be3 *"),
("D85t", "Gruenfeld: Modern Exchange, 8.Be3 Bg4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Be3 Bg4 *"),
("D85t", "Gruenfeld: Modern Exchange, 8.Be3 Nc6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Be3 Nc6 *"),
("D85u", "Gruenfeld: Modern Exchange, 8.Be3 Qa5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Be3 Qa5 *"),
("D85u", "Gruenfeld: Modern Exchange, 8.Be3 Qa5 9.Qd2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Be3 Qa5 9.Qd2 *"),
("D85v", "Gruenfeld: Modern Exchange, 8.Be3 Qa5 9.Qd2 Nc6 10.Rc1", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Be3 Qa5 9.Qd2 Nc6 10.Rc1 *"),
("D85w", "Gruenfeld: Modern Exchange, 8.Be3 O-O", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Be3 O-O *"),
("D85x", "Gruenfeld: Modern Exchange, 8.Be3 O-O 9.Rc1 Qa5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Be3 O-O 9.Rc1 Qa5 *"),
("D85x", "Gruenfeld: Modern Exchange, 8.Be3 O-O 9.Rc1 Qa5 10.Qd2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Be3 O-O 9.Rc1 Qa5 10.Qd2 *"),
("D85x", "Gruenfeld: Modern Exchange, 8.Be3 O-O 9.Rc1 Qa5 Queenswap, 12.Kxd2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Be3 O-O 9.Rc1 Qa5 10.Qd2 cxd4 11.cxd4 Qxd2+ 12.Kxd2 *"),
("D85y", "Gruenfeld: Modern Exchange, 8.Be3 O-O 9.Rc1 Qa5 Queenswap, 12.Nxd2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Nf3 c5 8.Be3 O-O 9.Rc1 Qa5 10.Qd2 cxd4 11.cxd4 Qxd2+ 12.Nxd2 *"),
("D86", "Gruenfeld: Classical Exchange", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 *"),
("D86", "Gruenfeld: Classical Exchange, 7...b6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 b6 *"),
("D86", "Gruenfeld: Classical Exchange, 7...c5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 c5 *"),
("D86", "Gruenfeld: Classical Exchange, 7...O-O", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O *"),
("D86", "Gruenfeld: Classical Exchange, 8.Ne2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 *"),
("D86", "Gruenfeld: Classical Exchange, Larsen Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 Qd7 9.O-O b6 *"),
("D86", "Gruenfeld: Classical Exchange, Simagin", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 b6 *"),
("D86", "Gruenfeld: Classical Exchange, Simagin, 9.h4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 b6 9.h4 *"),
("D86", "Gruenfeld: Classical Exchange, Simagin, 9.O-O", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 b6 9.O-O *"),
("D86", "Gruenfeld: Classical Exchange, Simagin Improved", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 Nc6 *"),
("D86", "Gruenfeld: Classical Exchange, Simagin Improved, 9.Be3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 Nc6 9.Be3 *"),
("D86", "Gruenfeld: Classical Exchange, Simagin Improved, 9.O-O", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 Nc6 9.O-O *"),
("D86", "Gruenfeld: Classical Exchange, Simagin Improved, 9.O-O e5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 Nc6 9.O-O e5 *"),
("D86", "Gruenfeld: Classical Exchange, Simagin Improved, 9.O-O b6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 Nc6 9.O-O b6 *"),
("D87a", "Gruenfeld: Classical Exchange, 8...c5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 *"),
("D87b", "Gruenfeld: Classical Exchange, 8...c5 9.Be3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.Be3 *"),
("D87c", "Gruenfeld: Classical Exchange, 8...c5 9.Be3 Qa5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.Be3 Qa5 *"),
("D87c", "Gruenfeld: Classical Exchange, 8...c5 9.Be3 Qa5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.Be3 Qa5 10.O-O Nd7 *"),
("D87d", "Gruenfeld: Classical Exchange, 8...c5 9.Be3 Nc6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.Be3 Nc6 *"),
("D87e", "Gruenfeld: Classical Exchange, 9.O-O", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O *"),
("D87f", "Gruenfeld: Classical Exchange, 9.O-O Nc6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 *"),
("D87g", "Gruenfeld: Classical Exchange, 9.O-O Nc6 10.Be3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 *"),
("D87h", "Gruenfeld: Classical Exchange, 10.Be3 Qa5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 Qa5 *"),
("D87i", "Gruenfeld: Classical Exchange, 10.Be3 Na5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 Na5 *"),
("D87j", "Gruenfeld: Classical Exchange, 10.Be3 Qc7", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 Qc7 *"),
("D87k", "Gruenfeld: Classical Exchange, 10.Be3 Qc7 11.Rc1 Rd8", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 Qc7 11.Rc1 Rd8 *"),
("D87l", "Gruenfeld: Classical Exchange, 10.Be3 Qc7 11.Rc1 Rd8 12.Qd2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 Qc7 11.Rc1 Rd8 12.Qd2 *"),
("D87m", "Gruenfeld: Classical Exchange, 10.Be3 Qc7 11.Rc1 Rd8 12.Bf4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 Qc7 11.Rc1 Rd8 12.Bf4 *"),
("D87n", "Gruenfeld: Classical Exchange, 10.Be3 Bg4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 Bg4 *"),
("D87o", "Gruenfeld: Classical Exchange, 10.Be3 Bg4 11.f3 Na5 12.Bd5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 Bg4 11.f3 Na5 12.Bd5 *"),
("D87p", "Gruenfeld: Classical Exchange, Seville Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 Bg4 11.f3 Na5 12.Bxf7+ *"),
("D88", "Gruenfeld: Classical Exchange, Main Line", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 cxd4 *"),
("D88", "Gruenfeld: Classical Exchange, Main Line, 11.cxd4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 cxd4 11.cxd4 *"),
("D88", "Gruenfeld: Classical Exchange, Main Line, 11.cxd4 Na5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 cxd4 11.cxd4 Na5 *"),
("D88", "Gruenfeld: Classical Exchange, Main Line, 11.cxd4 Na5 12.Bd3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 cxd4 11.cxd4 Na5 12.Bd3 *"),
("D88", "Gruenfeld: Classical Exchange, Main Line, 11.cxd4 Bg4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 cxd4 11.cxd4 Bg4 *"),
("D88", "Gruenfeld: Classical Exchange, Main Line, 11.cxd4 Bg4 12.f3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 cxd4 11.cxd4 Bg4 12.f3 *"),
("D88", "Gruenfeld: Classical Exchange, Main Line, 11.cxd4 Bg4 12.f3 Na5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 cxd4 11.cxd4 Bg4 12.f3 Na5 *"),
("D88", "Gruenfeld: Classical Exchange, Main Line, Neo-Seville", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 cxd4 11.cxd4 Bg4 12.f3 Na5 13.Bxf7+ *"),
("D88", "Gruenfeld: Classical Exchange, Main Line, 11.cxd4 Bg4 12.f3 Na5 13.Bd5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 cxd4 11.cxd4 Bg4 12.f3 Na5 13.Bd5 *"),
("D89a", "Gruenfeld: Classical Exchange, Main Line, 13.Bd3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 cxd4 11.cxd4 Bg4 12.f3 Na5 13.Bd3 *"),
("D89b", "Gruenfeld: Classical Exchange, Main Line, 13.Bd3 Be6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 cxd4 11.cxd4 Bg4 12.f3 Na5 13.Bd3 Be6 *"),
("D89c", "Gruenfeld: Classical Exchange, Main Line, 13.Bd3 Be6 14.Rc1", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 cxd4 11.cxd4 Bg4 12.f3 Na5 13.Bd3 Be6 14.Rc1 *"),
("D89d", "Gruenfeld: Classical Exchange, 14.Rc1 Bxa2 15.Qa4 Be6 16.d5 Be7 17.Qb4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 cxd4 11.cxd4 Bg4 12.f3 Na5 13.Bd3 Be6 14.Rc1 Bxa2 15.Qa4 Be6 16.d5 Bd7 17.Qb4 *"),
("D89e", "Gruenfeld: Classical Exchange, 14.Rc1 Bxa2 15.Qa4 with 17.Qb4 b6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 cxd4 11.cxd4 Bg4 12.f3 Na5 13.Bd3 Be6 14.Rc1 Bxa2 15.Qa4 Be6 16.d5 Bd7 17.Qb4 b6 *"),
("D89f", "Gruenfeld: Classical Exchange, 14.Rc1 Bxa2 15.Qa4 with 17.Qb4 e6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 cxd4 11.cxd4 Bg4 12.f3 Na5 13.Bd3 Be6 14.Rc1 Bxa2 15.Qa4 Be6 16.d5 Bd7 17.Qb4 e6 *"),
("D89g", "Gruenfeld: Classical Exchange, Main Line, Sokolsky Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 cxd4 11.cxd4 Bg4 12.f3 Na5 13.Bd3 Be6 14.d5 *"),
("D89h", "Gruenfeld: Classical Exchange, Main Line, Sokolsky, 16.Rb1", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 cxd4 11.cxd4 Bg4 12.f3 Na5 13.Bd3 Be6 14.d5 Bxa1 15.Qxa1 f6 16.Rb1 *"),
("D89i", "Gruenfeld: Classical Exchange, Main Line, Sokolsky, 16.Bh6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.cxd5 Nxd5 5.e4 Nxc3 6.bxc3 Bg7 7.Bc4 O-O 8.Ne2 c5 9.O-O Nc6 10.Be3 cxd4 11.cxd4 Bg4 12.f3 Na5 13.Bd3 Be6 14.d5 Bxa1 15.Qxa1 f6 16.Bh6 *"),
("D90", "Gruenfeld: Three Knights Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 *"),
("D90", "Gruenfeld: Schlechter Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 c6 *"),
("D90", "Gruenfeld: Schlechter, 5.Qb3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 c6 5.Qb3 *"),
("D90", "Gruenfeld: Schlechter, Exchange", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 c6 5.cxd5 *"),
("D90", "Gruenfeld: Schlechter, Exchange", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 c6 5.cxd5 cxd5 *"),
("D90", "Gruenfeld: Three Knights Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 *"),
("D90", "Gruenfeld: Three Knights, 5.g3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.g3 *"),
("D90", "Gruenfeld: Flohr Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qa4+ *"),
("D90", "Gruenfeld: Three Knights Exchange", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.cxd5 *"),
("D90", "Gruenfeld: Three Knights Exchange", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.cxd5 Nxd5 *"),
("D90", "Gruenfeld: Three Knights Exchange, Romanishin Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.cxd5 Nxd5 6.Qb3 *"),
("D90", "Gruenfeld: Three Knights Exchange, 6.Bd2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.cxd5 Nxd5 6.Bd2 *"),
("D90", "Gruenfeld: Three Knights Exchange, 6.Bd2 O-O", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.cxd5 Nxd5 6.Bd2 O-O *"),
("D90", "Gruenfeld: Three Knights Exchange, 6.Bd2 O-O 7.Rc1", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.cxd5 Nxd5 6.Bd2 O-O 7.Rc1 *"),
("D91a", "Gruenfeld: 5.Bg5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 *"),
("D91a", "Gruenfeld: 5.Bg5 c6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 c6 *"),
("D91b", "Gruenfeld: 5.Bg5 dxc4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 dxc4 *"),
("D91c", "Gruenfeld: 5.Bg5 dxc4 6.e4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 dxc4 6.e4 *"),
("D91d", "Gruenfeld: 5.Bg5 Ne4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 Ne4 *"),
("D91d", "Gruenfeld: 5.Bg5 Ne4 6.Bf4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 Ne4 6.Bf4 *"),
("D91e", "Gruenfeld: 5.Bg5 Ne4 6.Bh4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 Ne4 6.Bh4 *"),
("D91f", "Gruenfeld: 5.Bg5 Ne4 6.Bh4 Nxc3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 Ne4 6.Bh4 Nxc3 *"),
("D91g", "Gruenfeld: 5.Bg5 Ne4 6.Bh4 Nxc3 7.bxc3 dxc4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 Ne4 6.Bh4 Nxc3 7.bxc3 dxc4 *"),
("D91h", "Gruenfeld: 5.Bg5 Ne4 6.cxd5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 Ne4 6.cxd5 *"),
("D91i", "Gruenfeld: 5.Bg5 Ne4 6.cxd5 Nxg5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 Ne4 6.cxd5 Nxg5 *"),
("D91j", "Gruenfeld: 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 c6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 c6 *"),
("D91k", "Gruenfeld: 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 *"),
("D91l", "Gruenfeld: 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 8.Qd2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 8.Qd2 *"),
("D91m", "Gruenfeld: 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 8.Qd2 exd5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 8.Qd2 exd5 *"),
("D91n", "Gruenfeld: 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 8.Qd2 exd5: 11.h4 h6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 8.Qd2 exd5 9.Qe3+ Kf8 10.Qf4 Bf6 11.h4 h6 *"),
("D91o", "Gruenfeld: 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 8.Nf3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 8.Nf3 *"),
("D91p", "Gruenfeld: 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 8.Nf3 exd5 9.e3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 8.Nf3 exd5 9.e3 *"),
("D91q", "Gruenfeld: 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 8.Nf3 exd5 9.e3 O-O", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 8.Nf3 exd5 9.e3 O-O *"),
("D91r", "Gruenfeld: 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 8.Nf3 exd5 9.e3 O-O 10.Bd3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 8.Nf3 exd5 9.e3 O-O 10.Bd3 *"),
("D91r", "Gruenfeld: 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 8.Nf3 exd5 9.e3 O-O 10.Bd3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 8.Nf3 exd5 9.e3 O-O 10.Bd3 c6 *"),
("D91s", "Gruenfeld: 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 8.Nf3 exd5 9.e3 O-O 10.Be2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 8.Nf3 exd5 9.e3 O-O 10.Be2 *"),
("D91s", "Gruenfeld: 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 8.Nf3 exd5 9.e3 O-O 10.Be2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 8.Nf3 exd5 9.e3 O-O 10.Be2 c6 *"),
("D91t", "Gruenfeld: 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 8.Nf3 exd5 9.e3 O-O 10.b4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 8.Nf3 exd5 9.e3 O-O 10.b4 *"),
("D91t", "Gruenfeld: 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 8.Nf3 exd5 9.e3 O-O 10.b4 c6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bg5 Ne4 6.cxd5 Nxg5 7.Nxg5 e6 8.Nf3 exd5 9.e3 O-O 10.b4 c6 *"),
("D92", "Gruenfeld: 5.Bf4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bf4 *"),
("D92", "Gruenfeld: 5.Bf4 c6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bf4 c6 *"),
("D92", "Gruenfeld: 5.Bf4 O-O", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bf4 O-O *"),
("D92", "Gruenfeld: 5.Bf4 O-O 6.Rc1", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bf4 O-O 6.Rc1 *"),
("D92", "Gruenfeld: 5.Bf4 O-O 6.Rc1 dxc4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bf4 O-O 6.Rc1 dxc4 *"),
("D92", "Gruenfeld: 5.Bf4 O-O 6.Rc1 c5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bf4 O-O 6.Rc1 c5 *"),
("D93", "Gruenfeld: 5.Bf4 O-O 6.e3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bf4 O-O 6.e3 *"),
("D93", "Gruenfeld: 5.Bf4 O-O 6.e3 c5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bf4 O-O 6.e3 c5 *"),
("D93", "Gruenfeld: 5.Bf4 O-O 6.e3 c5 7.dxc5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bf4 O-O 6.e3 c5 7.dxc5 *"),
("D93", "Gruenfeld: 5.Bf4 O-O 6.e3 c5 7.dxc5 Qa5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bf4 O-O 6.e3 c5 7.dxc5 Qa5 *"),
("D93", "Gruenfeld: 5.Bf4 O-O 6.e3 c5 7.dxc5 Qa5 8.Rc1", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bf4 O-O 6.e3 c5 7.dxc5 Qa5 8.Rc1 *"),
("D93", "Gruenfeld: 5.Bf4 O-O 6.e3 c5 7.dxc5 Qa5 8.Rc1 dxc4 9.Bxc4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bf4 O-O 6.e3 c5 7.dxc5 Qa5 8.Rc1 dxc4 9.Bxc4 *"),
("D93", "Gruenfeld: 5.Bf4 O-O 6.e3 c6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bf4 O-O 6.e3 c6 *"),
("D93", "Gruenfeld: 5.Bf4 O-O 6.e3 c6 7.Qb3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bf4 O-O 6.e3 c6 7.Qb3 *"),
("D93", "Gruenfeld: 5.Bf4 O-O 6.e3 c6 7.Rc1", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Bf4 O-O 6.e3 c6 7.Rc1 *"),
("D94", "Gruenfeld: 5.e3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 *"),
("D94", "Gruenfeld: Slav/Schlecter", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 c6 *"),
("D94", "Gruenfeld: Slav/Schlecter, 6.Bd3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 c6 6.Bd3 *"),
("D94", "Gruenfeld: Slav/Schlecter, 6.Be2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 c6 6.Be2 *"),
("D94", "Gruenfeld: Slav/Schlecter, 6.Be2 O-O", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 c6 6.Be2 O-O *"),
("D94", "Gruenfeld: Slav/Schlecter, 6.Be2 O-O 7.O-O", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 c6 6.Be2 O-O 7.O-O *"),
("D94", "Gruenfeld: 5.e3 O-O", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 O-O *"),
("D94", "Gruenfeld: Makogonov Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 O-O 6.b4 *"),
("D94", "Gruenfeld: Opovcensky Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 O-O 6.Bd2 *"),
("D94", "Gruenfeld: 5.e3 O-O 6.Be2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 O-O 6.Be2 *"),
("D94", "Gruenfeld: 5.e3 O-O 6.Be2 c5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 O-O 6.Be2 c5 *"),
("D94", "Gruenfeld: 5.e3 O-O 6.Bd3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 O-O 6.Bd3 *"),
("D94", "Gruenfeld: 5.e3 O-O 6.Bd3 c6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 O-O 6.Bd3 c6 *"),
("D94", "Gruenfeld: Smyslov Defence", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 O-O 6.Bd3 c6 7.O-O Bg4 *"),
("D94", "Gruenfeld: Flohr Defence", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 O-O 6.Bd3 c6 7.O-O Bf5 *"),
("D94", "Gruenfeld: 5.e3 O-O 6.cxd5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 O-O 6.cxd5 *"),
("D94", "Gruenfeld: 5.e3 O-O 6.cxd5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 O-O 6.cxd5 Nxd5 *"),
("D94", "Gruenfeld: 5.e3 O-O 6.cxd5 Nxd5 7.Bc4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 O-O 6.cxd5 Nxd5 7.Bc4 *"),
("D95a", "Gruenfeld: 5.e3 O-O 6.Qb3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 O-O 6.Qb3 *"),
("D95b", "Gruenfeld: 5.e3 O-O 6.Qb3 b6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 O-O 6.Qb3 b6 *"),
("D95c", "Gruenfeld: 5.e3 O-O 6.Qb3 c6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 O-O 6.Qb3 c6 *"),
("D95d", "Gruenfeld: 5.e3 O-O 6.Qb3 c6 7.Bd2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 O-O 6.Qb3 c6 7.Bd2 *"),
("D95e", "Gruenfeld: 5.e3 O-O 6.Qb3 c6 7.Bd2 e6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 O-O 6.Qb3 c6 7.Bd2 e6 *"),
("D95f", "Gruenfeld: 5.e3 O-O 6.Qb3 c6 7.Bd2 e6 8.Bd3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 O-O 6.Qb3 c6 7.Bd2 e6 8.Bd3 *"),
("D95g", "Gruenfeld: Botvinnik Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 O-O 6.Qb3 e6 *"),
("D95h", "Gruenfeld: Botvinnik, 7.Bd2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 O-O 6.Qb3 e6 7.Bd2 *"),
("D95i", "Gruenfeld: 5.e3 O-O 6.Qb3 dxc4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 O-O 6.Qb3 dxc4 *"),
("D95i", "Gruenfeld: 5.e3 O-O 6.Qb3 dxc4 7.Bxc4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 O-O 6.Qb3 dxc4 7.Bxc4 *"),
("D95i", "Gruenfeld: Pachman Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 O-O 6.Qb3 dxc4 7.Bxc4 Nbd7 8.Ng5 *"),
("D95j", "Gruenfeld: 5.e3 O-O 6.Qb3 dxc4 7.Bxc4 Nc6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.e3 O-O 6.Qb3 dxc4 7.Bxc4 Nc6 *"),
("D96", "Gruenfeld: Russian Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 *"),
("D96", "Gruenfeld: Russian, 5...c6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 c6 *"),
("D96", "Gruenfeld: Russian, 5...c6 6.Bf4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 c6 6.Bf4 *"),
("D96", "Gruenfeld: Russian, 5...c6 6.cxd5 cxd5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 c6 6.cxd5 cxd5 *"),
("D96", "Gruenfeld: Russian, 5...c6 6.cxd5 Nxd5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 c6 6.cxd5 Nxd5 *"),
("D96", "Gruenfeld: Russian, 5...dxc4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 *"),
("D96", "Gruenfeld: Russian, 5...dxc4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 *"),
("D96", "Gruenfeld: Russian, 6...O-O", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 a6 *"),
("D96", "Gruenfeld: Russian, 6...a6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O *"),
("D96", "Gruenfeld: Russian, 7.Bf4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.Bf4 *"),
("D96", "Gruenfeld: Russian, 7.Bf4 c6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.Bf4 c6 *"),
("D97a", "Gruenfeld: Russian, 7.e4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 *"),
("D97a", "Gruenfeld: Russian, 7.e4 Nfd7", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Nfd7 *"),
("D97b", "Gruenfeld: Russian, Levenfish Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 b6 *"),
("D97c", "Gruenfeld: Russian, Szabo Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 c6 *"),
("D97c", "Gruenfeld: Russian, Szabo, 8.Be2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 c6 8.Be2 *"),
("D97d", "Gruenfeld: Russian, Alekhine (Hungarian) Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 a6 *"),
("D97e", "Gruenfeld: Russian, Alekhine, 8.Be2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 a6 8.Be2 *"),
("D97f", "Gruenfeld: Russian, Alekhine, 8.Be2 b5 9.Qb3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 a6 8.Be2 b5 9.Qb3 *"),
("D97g", "Gruenfeld: Russian, Alekhine, 8.Qb3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 a6 8.Qb3 *"),
("D97h", "Gruenfeld: Russian, Alekhine, 8.e5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 a6 8.e5 *"),
("D97i", "Gruenfeld: Russian, Alekhine, 8.e5 b5 9.Qb3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 a6 8.e5 b5 9.Qb3 *"),
("D97j", "Gruenfeld: Russian, Alekhine, 8.e5 b5 9.Qb3 Nfd7", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 a6 8.e5 b5 9.Qb3 Nfd7 *"),
("D97k", "Gruenfeld: Russian, Alekhine, 8.e5 b5 9.Qb3 Nfd7 10.Be3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 a6 8.e5 b5 9.Qb3 Nfd7 10.Be3 *"),
("D97l", "Gruenfeld: Russian, Simagin Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Nc6 *"),
("D97l", "Gruenfeld: Russian, Simagin, 8.Be2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Nc6 8.Be2 *"),
("D97m", "Gruenfeld: Russian, Prins Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Na6 *"),
("D97n", "Gruenfeld: Russian, Prins, 8.Bf4", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Na6 8.Bf4 *"),
("D97o", "Gruenfeld: Russian, Prins, 8.Be2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Na6 8.Be2 *"),
("D97p", "Gruenfeld: Russian, Prins, 8.Be2, Main Line", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Na6 8.Be2 c5 9.d5 e6 10.O-O exd5 11.exd5 Bf5 *"),
("D98a", "Gruenfeld: Russian, Smyslov Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Bg4 *"),
("D98b", "Gruenfeld: Russian, Smyslov, 8.Be2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Bg4 8.Be2 *"),
("D98c", "Gruenfeld: Russian, Smyslov, 8.Be2 Nc6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Bg4 8.Be2 Nc6 *"),
("D98d", "Gruenfeld: Russian, Smyslov, 8.Be2 Nc6 9.d5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Bg4 8.Be2 Nc6 9.d5 *"),
("D98e", "Gruenfeld: Russian, Smyslov, 8.Be3", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Bg4 8.Be3 *"),
("D98f", "Gruenfeld: Russian, Smyslov, 8.Be3 Nfd7", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Bg4 8.Be3 Nfd7 *"),
("D98g", "Gruenfeld: Russian, Smyslov, 8.Be3 Nfd7 9.Be2", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Bg4 8.Be3 Nfd7 9.Be2 *"),
("D98h", "Gruenfeld: Russian, Smyslov, Keres Variation", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Bg4 8.Be3 Nfd7 9.Be2 Nb6 10.Qd3 Nc6 11.O-O-O *"),
("D98i", "Gruenfeld: Russian, Smyslov, 8.Be3 Nfd7 9.O-O-O", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Bg4 8.Be3 Nfd7 9.O-O-O *"),
("D98j", "Gruenfeld: Russian, Smyslov, 8.Be3 Nfd7 9.Rd1", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Bg4 8.Be3 Nfd7 9.Rd1 *"),
("D98k", "Gruenfeld: Russian, Smyslov, 8.Be3 Nfd7 9.Rd1 Nc6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Bg4 8.Be3 Nfd7 9.Rd1 Nc6 *"),
("D99a", "Gruenfeld: Russian, Smyslov, Main Line", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Bg4 8.Be3 Nfd7 9.Qb3 *"),
("D99b", "Gruenfeld: Russian, Smyslov, Yugoslav", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Bg4 8.Be3 Nfd7 9.Qb3 c5 *"),
("D99c", "Gruenfeld: Russian, Smyslov, Main Line, 9...Nb6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Bg4 8.Be3 Nfd7 9.Qb3 Nb6 *"),
("D99d", "Gruenfeld: Russian, Smyslov, Main Line, 9...Nb6 10.Rd1", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Bg4 8.Be3 Nfd7 9.Qb3 Nb6 10.Rd1 *"),
("D99e", "Gruenfeld: Russian, Smyslov, Main Line, 9...Nb6 10.Rd1 e6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Bg4 8.Be3 Nfd7 9.Qb3 Nb6 10.Rd1 e6 *"),
("D99f", "Gruenfeld: Russian, Smyslov, Main Line, 9...Nb6 10.Rd1 Nc6", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Bg4 8.Be3 Nfd7 9.Qb3 Nb6 10.Rd1 Nc6 *"),
("D99g", "Gruenfeld: Russian, Smyslov, Main Line, 9...Nb6 10.Rd1 Nc6 11.d5 Ne5", "1.d4 Nf6 2.c4 g6 3.Nc3 d5 4.Nf3 Bg7 5.Qb3 dxc4 6.Qxc4 O-O 7.e4 Bg4 8.Be3 Nfd7 9.Qb3 Nb6 10.Rd1 Nc6 11.d5 Ne5 *"),
("E00a", "Queen's Pawn: Neo-Indian", "1.d4 Nf6 2.c4 e6 *"),
("E00a", "Queen's Pawn: Neo-Indian, Devin Gambit", "1.d4 Nf6 2.c4 e6 3.g4 *"),
("E00b", "Queen's Pawn: Anti-Nimzo-Indian", "1.d4 Nf6 2.c4 e6 3.a3 *"),
("E00b", "Queen's Pawn: Anti-Nimzo-Indian, 3...d5", "1.d4 Nf6 2.c4 e6 3.a3 d5 *"),
("E00c", "Neo-Indian (Seirawan) Attack", "1.d4 Nf6 2.c4 e6 3.Bg5 *"),
("E00d", "Queen's Pawn: Neo-Indian", "1.d4 Nf6 2.c4 e6 3.Nc3 *"),
("E00e", "Queen's Pawn: Neo-Indian, 3...b6", "1.d4 Nf6 2.c4 e6 3.Nc3 b6 *"),
("E00e", "Queen's Pawn: Neo-Indian, 3...b6", "1.d4 Nf6 2.c4 e6 3.Nc3 b6 4.e4 *"),
("E00f", "Queen's Pawn: Neo-Indian, 3...c5", "1.d4 Nf6 2.c4 e6 3.Nc3 c5 *"),
("E00g", "Catalan", "1.d4 Nf6 2.c4 e6 3.g3 *"),
("E00g", "Catalan: Hungarian Gambit", "1.d4 Nf6 2.c4 e6 3.g3 e5 *"),
("E00g", "Catalan: 3...c6", "1.d4 Nf6 2.c4 e6 3.g3 c6 *"),
("E00h", "Catalan: 3...c5", "1.d4 Nf6 2.c4 e6 3.g3 c5 *"),
("E00h", "Catalan: 3...c5 4.Nf3", "1.d4 Nf6 2.c4 e6 3.g3 c5 4.Nf3 *"),
("E00i", "Catalan: 3...Bb4+", "1.d4 Nf6 2.c4 e6 3.g3 Bb4+ *"),
("E00i", "Catalan: 3...Bb4+ 4.Nd2", "1.d4 Nf6 2.c4 e6 3.g3 Bb4+ 4.Nd2 *"),
("E00j", "Catalan: 3...Bb4+ 4.Bd2", "1.d4 Nf6 2.c4 e6 3.g3 Bb4+ 4.Bd2 *"),
("E00k", "Catalan: 3...Bb4+ 4.Bd2 Bxd2+", "1.d4 Nf6 2.c4 e6 3.g3 Bb4+ 4.Bd2 Bxd2+ *"),
("E00l", "Catalan: 3...Bb4+ 4.Bd2 Be7", "1.d4 Nf6 2.c4 e6 3.g3 Bb4+ 4.Bd2 Be7 *"),
("E00m", "Catalan: 3...Bb4+ 4.Bd2 Qe7", "1.d4 Nf6 2.c4 e6 3.g3 Bb4+ 4.Bd2 Qe7 *"),
("E00m", "Catalan: 3...Bb4+ 4.Bd2 Qe7", "1.d4 Nf6 2.c4 e6 3.g3 Bb4+ 4.Bd2 Qe7 5.Bg2 *"),
("E00n", "Catalan: 3...d5", "1.d4 Nf6 2.c4 e6 3.g3 d5 *"),
("E00o", "Catalan: 4.Nf3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Nf3 *"),
("E00p", "Catalan: 4.Nf3 c6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Nf3 c6 *"),
("E00q", "Catalan: 4.Nf3 c5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Nf3 c5 *"),
("E00r", "Catalan: 4.Nf3 Bb4+", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Nf3 Bb4+ *"),
("E00r", "Catalan: 4.Nf3 Bb4+ 5.Bd2", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Nf3 Bb4+ 5.Bd2 *"),
("E00s", "Catalan: 4.Nf3 Be7", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Nf3 Be7 *"),
("E00t", "Catalan: 4.Nf3 dxc4", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Nf3 dxc4 *"),
("E00u", "Catalan: 4.Nf3 dxc4 5.Qa4+", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Nf3 dxc4 5.Qa4+ *"),
("E00v", "Catalan: 4.Nf3 dxc4 5.Qa4+ Nbd7", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Nf3 dxc4 5.Qa4+ Nbd7 *"),
("E01", "Catalan: 4.Bg2", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 *"),
("E01", "Catalan: 4...Bb4+", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Bb4+ *"),
("E01", "Catalan: 4...Bb4+ 5.Bd2", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Bb4+ 5.Bd2 *"),
("E01", "Catalan: 4...Bb4+ 5.Nd2", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Bb4+ 5.Nd2 *"),
("E01", "Catalan: 4...Bb4+ 5.Nd2 O-O", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Bb4+ 5.Nd2 O-O *"),
("E01", "Catalan: 4...c6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 c6 *"),
("E01", "Catalan: 4...c6 5.Qc2", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 c6 5.Qc2 *"),
("E01", "Catalan: 4...c6 5.Nf3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 c6 5.Nf3 *"),
("E01", "Catalan: 4...c6 5.Nf3 Nbd7", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 c6 5.Nf3 Nbd7 *"),
("E01", "Catalan: 4...c5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 c5 *"),
("E01", "Catalan: 4...c5 5.Nf3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 c5 5.Nf3 *"),
("E01", "Catalan: Open", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 *"),
("E02", "Catalan: Open, 5.Qa4+", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Qa4+ *"),
("E02", "Catalan: Open, 5.Qa4+ Bd7", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Qa4+ Bd7 *"),
("E02", "Catalan: Open, 5.Qa4+ Bd7 6.Qxc4", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Qa4+ Bd7 6.Qxc4 *"),
("E02", "Catalan: Open, 5.Qa4+ Nbd7", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Qa4+ Nbd7 *"),
("E02", "Catalan: Open, 5.Qa4+ Nbd7 6.Nf3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Qa4+ Nbd7 6.Nf3 *"),
("E02", "Catalan: Open, 5.Qa4+ Nbd7 6.Nf3 a6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Qa4+ Nbd7 6.Nf3 a6 *"),
("E02", "Catalan: Open, 5.Qa4+ Nbd7 6.Nf3 a6 7.Nc3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Qa4+ Nbd7 6.Nf3 a6 7.Nc3 *"),
("E03", "Catalan: Open, 5.Qa4+ Nbd7 6.Qxc4", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Qa4+ Nbd7 6.Qxc4 *"),
("E03", "Catalan: Open, 5.Qa4+ Nbd7 6.Qxc4 a6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Qa4+ Nbd7 6.Qxc4 a6 *"),
("E03", "Catalan: Open, Alekhine Variation", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Qa4+ Nbd7 6.Qxc4 a6 7.Qc2 *"),
("E03", "Catalan: Open, 5.Qa4+ Nbd7 6.Qxc4 a6 7.Nf3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Qa4+ Nbd7 6.Qxc4 a6 7.Nf3 *"),
("E03", "Catalan: Open, 5.Qa4+ Nbd7 6.Qxc4 a6 7.Nf3 b5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Qa4+ Nbd7 6.Qxc4 a6 7.Nf3 b5 *"),
("E03", "Catalan: Open, 5.Qa4+ Nbd7 6.Qxc4 a6 7.Nf3 b5 8.Qc2", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Qa4+ Nbd7 6.Qxc4 a6 7.Nf3 b5 8.Qc2 *"),
("E03", "Catalan: Open, 5.Qa4+ Nbd7 6.Qxc4 a6 7.Nf3 b5 8.Qc6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Qa4+ Nbd7 6.Qxc4 a6 7.Nf3 b5 8.Qc6 *"),
("E03", "Catalan: Open, 5.Qa4+ Nbd7 6.Qxc4 c5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Qa4+ Nbd7 6.Qxc4 c5 *"),
("E03", "Catalan: Open, 5.Qa4+ Nbd7 6.Qxc4 c5 7.Nf3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Qa4+ Nbd7 6.Qxc4 c5 7.Nf3 *"),
("E03", "Catalan: Open, 5.Qa4+ Nbd7 6.Qxc4 c5 7.Nf3 a6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Qa4+ Nbd7 6.Qxc4 c5 7.Nf3 a6 *"),
("E03", "Catalan: Open, 5.Qa4+ Nbd7 6.Qxc4 c5 7.Nf3 a6 8.O-O", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Qa4+ Nbd7 6.Qxc4 c5 7.Nf3 a6 8.O-O *"),
("E03", "Catalan: Open, 5.Qa4+ Nbd7 6.Qxc4 c5 7.Nf3 a6 8.Qc2", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Qa4+ Nbd7 6.Qxc4 c5 7.Nf3 a6 8.Qc2 *"),
("E03", "Catalan: Open, 5.Qa4+ Nbd7 6.Qxc4 c5 7.Nf3 a6 8.Qc2 b6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Qa4+ Nbd7 6.Qxc4 c5 7.Nf3 a6 8.Qc2 b6 *"),
("E04a", "Catalan: Open, 5.Nf3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 *"),
("E04a", "Catalan: Open, 5.Nf3 Bd7", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Bd7 *"),
("E04b", "Catalan: Open, 5.Nf3 Bb4+", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Bb4+ *"),
("E04c", "Catalan: Open, 5.Nf3 Nbd7", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Nbd7 *"),
("E04d", "Catalan: Open, 5.Nf3 b5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 b5 *"),
("E04e", "Catalan: Open, 5.Nf3 a6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 a6 *"),
("E04f", "Catalan: Open, 5.Nf3 a6 6.O-O", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 a6 6.O-O *"),
("E04g", "Catalan: Open, 5.Nf3 a6 6.O-O b5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 a6 6.O-O b5 *"),
("E04g", "Catalan: Open, 5.Nf3 a6 6.O-O b5 7.Ne5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 a6 6.O-O b5 7.Ne5 *"),
("E04h", "Catalan: Open, 5.Nf3 a6 6.O-O b5 7.Ne5 Nd5 8.a4", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 a6 6.O-O b5 7.Ne5 Nd5 8.a4 *"),
("E04i", "Catalan: Open, 5.Nf3 a6 6.O-O Nc6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 a6 6.O-O Nc6 *"),
("E04j", "Catalan: Open, 5.Nf3 a6 6.O-O Nc6 7.e3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 a6 6.O-O Nc6 7.e3 *"),
("E04k", "Catalan: Open, 5.Nf3 c5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 c5 *"),
("E04k", "Catalan: Open, 5.Nf3 c5 6.O-O", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 c5 6.O-O *"),
("E04l", "Catalan: Open, 5.Nf3 c5 6.O-O Nc6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 c5 6.O-O Nc6 *"),
("E04m", "Catalan: Open, 5.Nf3 c5 6.O-O Nc6 7.Ne5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 c5 6.O-O Nc6 7.Ne5 *"),
("E04n", "Catalan: Open, 5.Nf3 c5 6.O-O Nc6 7.Ne5 Bd7 8.Na3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 c5 6.O-O Nc6 7.Ne5 Bd7 8.Na3 *"),
("E04n", "Catalan: Open, 5.Nf3 c5 6.O-O Nc6 7.Ne5 Bd7 8.Na3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 c5 6.O-O Nc6 7.Ne5 Bd7 8.Na3 cxd4 9.Naxc4 *"),
("E04o", "Catalan: Open, 5.Nf3 c5 6.O-O Nc6 7.Qa4", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 c5 6.O-O Nc6 7.Qa4 *"),
("E04o", "Catalan: Open, 5.Nf3 c5 6.O-O Nc6 7.Qa4 cxd4", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 c5 6.O-O Nc6 7.Qa4 cxd4 *"),
("E04p", "Catalan: Open, 5.Nf3 c5 6.O-O Nc6 7.Qa4 Bd7", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 c5 6.O-O Nc6 7.Qa4 Bd7 *"),
("E04q", "Catalan: Open, 5.Nf3 c5 6.O-O Nc6 7.Qa4 Bd7 8.Qxc4", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 c5 6.O-O Nc6 7.Qa4 Bd7 8.Qxc4 *"),
("E04r", "Catalan: Open, 5.Nf3 c5 6.O-O Nc6 7.Qa4 Bd7 8.Qxc4 b5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 c5 6.O-O Nc6 7.Qa4 Bd7 8.Qxc4 b5 *"),
("E04s", "Catalan: Open, 5.Nf3 Nc6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Nc6 *"),
("E04s", "Catalan: Open, 5.Nf3 Nc6 6.O-O", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Nc6 6.O-O *"),
("E04t", "Catalan: Open, 5.Nf3 Nc6 6.O-O Rb8", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Nc6 6.O-O Rb8 *"),
("E04u", "Catalan: Open, 5.Nf3 Nc6 6.O-O Rb8 7.Nc3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Nc6 6.O-O Rb8 7.Nc3 *"),
("E04v", "Catalan: Open, 5.Nf3 Nc6 6.Qa4", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Nc6 6.Qa4 *"),
("E04w", "Catalan: Open, 5.Nf3 Nc6 6.Qa4 Bb4+", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Nc6 6.Qa4 Bb4+ *"),
("E04x", "Catalan: Open, 5.Nf3 Nc6 6.Qa4 Bb4+ 7.Bd2 Nd5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Nc6 6.Qa4 Bb4+ 7.Bd2 Nd5 *"),
("E05a", "Catalan: Open, Classical", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Be7 *"),
("E05b", "Catalan: Open, Classical, 6.Nc3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Be7 6.Nc3 *"),
("E05c", "Catalan: Open, Classical, 6.Nc3 O-O", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Be7 6.Nc3 O-O *"),
("E05d", "Catalan: Open, Classical, 6.Nc3 O-O 7.Ne5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Be7 6.Nc3 O-O 7.Ne5 *"),
("E05e", "Catalan: Open, Classical, 6.O-O", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Be7 6.O-O *"),
("E05f", "Catalan: Open, Classical, 6.O-O O-O", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Be7 6.O-O O-O *"),
("E05g", "Catalan: Open, Classical, 7.Ne5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Be7 6.O-O O-O 7.Ne5 *"),
("E05h", "Catalan: Open, Classical, 7.Ne5 Nc6 8.Nxc6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Be7 6.O-O O-O 7.Ne5 Nc6 8.Nxc6 *"),
("E05i", "Catalan: Open, Classical, 7.Qc2", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Be7 6.O-O O-O 7.Qc2 *"),
("E05j", "Catalan: Open, Classical Main Line", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Be7 6.O-O O-O 7.Qc2 a6 *"),
("E05k", "Catalan: Open, Classical, 8.a4", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Be7 6.O-O O-O 7.Qc2 a6 8.a4 *"),
("E05l", "Catalan: Open, Classical, 8.a4 Bd7", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Be7 6.O-O O-O 7.Qc2 a6 8.a4 Bd7 *"),
("E05m", "Catalan: Open, Classical, 8.a4 Bd7 9.Qxc4", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Be7 6.O-O O-O 7.Qc2 a6 8.a4 Bd7 9.Qxc4 *"),
("E05m", "Catalan: Open, Classical, 8.a4 Bd7 9.Qxc4", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Be7 6.O-O O-O 7.Qc2 a6 8.a4 Bd7 9.Qxc4 Bc6 *"),
("E05n", "Catalan: Open, Classical, 8.a4, 10.Bg5 Bd5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Be7 6.O-O O-O 7.Qc2 a6 8.a4 Bd7 9.Qxc4 Bc6 10.Bg5 Bd5 *"),
("E05o", "Catalan: Open, Classical, 8.a4, 10.Bg5 a5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Be7 6.O-O O-O 7.Qc2 a6 8.a4 Bd7 9.Qxc4 Bc6 10.Bg5 a5 *"),
("E05p", "Catalan: Open, Classical, 8.Qxc4", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Be7 6.O-O O-O 7.Qc2 a6 8.Qxc4 *"),
("E05q", "Catalan: Open, Classical, 8.Qxc4 b5 9.Qc2 Bb7", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Be7 6.O-O O-O 7.Qc2 a6 8.Qxc4 b5 9.Qc2 Bb7 *"),
("E05r", "Catalan: Open, Classical, 8.Qxc4, 10.Bg5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Be7 6.O-O O-O 7.Qc2 a6 8.Qxc4 b5 9.Qc2 Bb7 10.Bg5 *"),
("E05s", "Catalan: Open, Classical, 8.Qxc4, 10.Bd2", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Be7 6.O-O O-O 7.Qc2 a6 8.Qxc4 b5 9.Qc2 Bb7 10.Bd2 *"),
("E05t", "Catalan: Open, Classical, 8.Qxc4, 10.Bd2 Be4", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Be7 6.O-O O-O 7.Qc2 a6 8.Qxc4 b5 9.Qc2 Bb7 10.Bd2 Be4 *"),
("E05u", "Catalan: Open, Classical, 8.Qxc4, 10.Bf4", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Be7 6.O-O O-O 7.Qc2 a6 8.Qxc4 b5 9.Qc2 Bb7 10.Bf4 *"),
("E05v", "Catalan: Open, Classical, 8.Qxc4, 10.Bf4 Nc6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Be7 6.O-O O-O 7.Qc2 a6 8.Qxc4 b5 9.Qc2 Bb7 10.Bf4 Nc6 *"),
("E05v", "Catalan: Open, Classical, 8.Qxc4, 10.Bf4 Nc6 11.Rd1", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 dxc4 5.Nf3 Be7 6.O-O O-O 7.Qc2 a6 8.Qxc4 b5 9.Qc2 Bb7 10.Bf4 Nc6 11.Rd1 *"),
("E06", "Catalan: Closed", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 *"),
("E06", "Catalan: Closed, 5.Nf3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 *"),
("E06", "Catalan: Closed, 5.Nf3 O-O", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O *"),
("E06", "Catalan: Closed, 6.Qc2", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.Qc2 *"),
("E06", "Catalan: Closed, 6.O-O", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O *"),
("E06", "Catalan: Closed, 6.O-O c5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O c5 *"),
("E06", "Catalan: Closed, 6.O-O c5 7.cxd5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O c5 7.cxd5 *"),
("E06", "Catalan: Closed, 6.O-O c6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O c6 *"),
("E06", "Catalan: Closed, 6.O-O c6 7.b3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O c6 7.b3 *"),
("E06", "Catalan: Closed, 6.O-O c6 7.Nbd2", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O c6 7.Nbd2 *"),
("E06", "Catalan: Closed, 6.O-O c6 7.Nc3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O c6 7.Nc3 *"),
("E06", "Catalan: Closed, 6.O-O c6 7.Nc3 b6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O c6 7.Nc3 b6 *"),
("E06", "Catalan: Closed, 6.O-O c6 7.Nc3 b6 8.Ne5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O c6 7.Nc3 b6 8.Ne5 *"),
("E06", "Catalan: Closed, 6.O-O c6 7.Qc2", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O c6 7.Qc2 *"),
("E06", "Catalan: Closed, 6.O-O c6 7.Qc2 b6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O c6 7.Qc2 b6 *"),
("E07", "Catalan: Closed, 6...Nbd7", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 *"),
("E07", "Catalan: Closed, 6...Nbd7 7.Nbd2", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Nbd2 *"),
("E07", "Catalan: Closed, 6...Nbd7 7.Qd3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qd3 *"),
("E07", "Catalan: Closed, 6...Nbd7 7.b3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.b3 *"),
("E07", "Catalan: Closed, 6...Nbd7 7.b3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.b3 c6 *"),
("E07", "Catalan: Closed, 6...Nbd7 7.b3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.b3 c6 8.Ba3 *"),
("E07", "Catalan: Closed, 6...Nbd7 7.b3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.b3 c6 8.Bb2 *"),
("E07", "Catalan: Closed, 6...Nbd7 7.Nc3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Nc3 *"),
// I am not sure if this delayed 7...dxc4 is E07:
("E07", "Catalan: Closed, 6...Nbd7 7.Nc3 dxc4", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Nc3 dxc4 *"),
("E07", "Catalan: Closed, 6...Nbd7 7.Nc3 dxc4", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Nc3 dxc4 8.e4 *"),
("E07", "Catalan: Closed, 6...Nbd7 7.Nc3 c6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Nc3 c6 *"),
("E07", "Catalan: Closed, 6...Nbd7 7.Nc3 c6 8.Qb3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Nc3 c6 8.Qb3 *"),
("E07", "Catalan: Closed, Botvinnik Variation", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Nc3 c6 8.Qd3 *"),
("E07", "Catalan: Closed, 6...Nbd7 7.Nc3 c6 8.b3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Nc3 c6 8.b3 *"),
("E08a", "Catalan: Closed, 7.Qc2", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 *"),
("E08b", "Catalan: Closed, 7.Qc2 c5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c5 *"),
("E08c", "Catalan: Closed, 7.Qc2 b6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 b6 *"),
("E08d", "Catalan: Closed, 7.Qc2 c6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 *"),
("E08e", "Catalan: Closed, 7.Qc2 c6 8.Rd1", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Rd1 *"),
("E08f", "Catalan: Closed, 7.Qc2 c6 8.Rd1 b6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Rd1 b6 *"),
("E08f", "Catalan: Closed, Zagoryansky Variation", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Rd1 b6 9.a4 *"),
("E08g", "Catalan: Closed, 7.Qc2 c6 8.Bf4", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Bf4 *"),
("E08h", "Catalan: Closed, 7.Qc2 c6 8.Bf4 Nh5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Bf4 Nh5 *"),
("E08i", "Catalan: Closed, 7.Qc2 c6 8.Bf4 Ne4", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Bf4 Ne4 *"),
("E08j", "Catalan: Closed, 7.Qc2 c6 8.Bf4 b6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Bf4 b6 *"),
("E08k", "Catalan: Closed, 7.Qc2 c6 8.b3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.b3 *"),
("E08l", "Catalan: Closed, 7.Qc2 c6 8.b3 b5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.b3 b5 *"),
("E08m", "Catalan: Closed, 7.Qc2 c6 8.b3 b6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.b3 b6 *"),
("E08n", "Catalan: Closed, 7.Qc2 c6 8.b3 b6 9.Rd1", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.b3 b6 9.Rd1 *"),
("E08n", "Catalan: Closed, 7.Qc2 c6 8.b3 b6 9.Rd1 Ba6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.b3 b6 9.Rd1 Ba6 *"),
("E08o", "Catalan: Closed, 7.Qc2 c6 8.b3 b6 9.Rd1 Ba6 10.Nbd2", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.b3 b6 9.Rd1 Ba6 10.Nbd2 *"),
("E08p", "Catalan: Closed, 7.Qc2 c6 8.b3 b6 9.Rd1 Bb7", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.b3 b6 9.Rd1 Bb7 *"),
("E08p", "Catalan: Closed, 7.Qc2 c6 8.b3 b6 9.Rd1 Bb7 10.Nc3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.b3 b6 9.Rd1 Bb7 10.Nc3 *"),
("E08p", "Catalan: Closed, Spassky Gambit", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.b3 b6 9.Rd1 Bb7 10.Nc3 b5 *"),
("E08q", "Catalan: Closed, 7.Qc2 c6 8.b3 b6 9.Rd1 Bb7 10.Nc3 Rc8", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.b3 b6 9.Rd1 Bb7 10.Nc3 Rc8 *"),
("E08q", "Catalan: Closed, 7.Qc2 c6 8.b3 b6 9.Rd1 Bb7 10.Nc3 Rc8 11.e4", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.b3 b6 9.Rd1 Bb7 10.Nc3 Rc8 11.e4 *"),
("E08r", "Catalan: Closed, 7.Qc2 c6 8.b3 b6 9.Rd1 Bb7 10.Nc3 Rc8 11.e4 dxe4", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.b3 b6 9.Rd1 Bb7 10.Nc3 Rc8 11.e4 dxe4 *"),
("E09a", "Catalan: Closed, Main Line", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Nbd2 *"),
("E09b", "Catalan: Closed, Main Line, 8...b5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Nbd2 b5 *"),
("E09c", "Catalan: Closed, Main Line, 8...Re8", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Nbd2 Re8 *"),
("E09d", "Catalan: Closed, Main Line, 8...b6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Nbd2 b6 *"),
("E09e", "Catalan: Closed, Main Line, 9.b3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Nbd2 b6 9.b3 *"),
("E09e", "Catalan: Closed, Main Line, Sokolsky Variation", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Nbd2 b6 9.b3 a5 10.Bb2 Ba6 *"),
("E09e", "Catalan: Closed, Main Line, 9.b3 Bb7", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Nbd2 b6 9.b3 Bb7 *"),
("E09f", "Catalan: Closed, Main Line, 9.b3 Bb7 10.Bb2", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Nbd2 b6 9.b3 Bb7 10.Bb2 *"),
("E09g", "Catalan: Closed, Main Line, 9.b3 Bb7 10.Bb2 Rc8", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Nbd2 b6 9.b3 Bb7 10.Bb2 Rc8 *"),
("E09h", "Catalan: Closed, Main Line, 9.e4", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Nbd2 b6 9.e4 *"),
("E09i", "Catalan: Closed, Main Line, 9.e4 Ba6", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Nbd2 b6 9.e4 Ba6 *"),
("E09i", "Catalan: Closed, Main Line, 9.e4 Ba6 10.b3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Nbd2 b6 9.e4 Ba6 10.b3 *"),
("E09j", "Catalan: Closed, Main Line, 9.e4 dxe4", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Nbd2 b6 9.e4 dxe4 *"),
("E09k", "Catalan: Closed, Main Line, 9.e4 dxe4 10.Nxe4 Nxe4 11.Qxe4", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Nbd2 b6 9.e4 dxe4 10.Nxe4 Nxe4 11.Qxe4 *"),
("E09l", "Catalan: Closed, Main Line, 9.e4 Bb7", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Nbd2 b6 9.e4 Bb7 *"),
("E09m", "Catalan: Closed, Main Line, 9.e4 Bb7 10.e5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Nbd2 b6 9.e4 Bb7 10.e5 *"),
("E09n", "Catalan: Closed, Main Line, 9.e4 Bb7 10.b3", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Nbd2 b6 9.e4 Bb7 10.b3 *"),
("E09o", "Catalan: Closed, Main Line, 9.e4 Bb7 10.b3 Rc8 11.Bb2", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Nbd2 b6 9.e4 Bb7 10.b3 Rc8 11.Bb2 *"),
("E09p", "Catalan: Closed, Main Line, 9.e4 Bb7 10.b3 Rc8 11.Bb2 Qc7", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Nbd2 b6 9.e4 Bb7 10.b3 Rc8 11.Bb2 Qc7 *"),
("E09q", "Catalan: Closed, Main Line, 9.e4 Bb7 10.b3 Rc8 11.Bb2 c5", "1.d4 Nf6 2.c4 e6 3.g3 d5 4.Bg2 Be7 5.Nf3 O-O 6.O-O Nbd7 7.Qc2 c6 8.Nbd2 b6 9.e4 Bb7 10.b3 Rc8 11.Bb2 c5 *"),
("E10a", "Neo-Indian: 3.Nf3", "1.d4 Nf6 2.c4 e6 3.Nf3 *"),
("E10b", "Neo-Indian: Doery Defence", "1.d4 Nf6 2.c4 e6 3.Nf3 Ne4 *"),
("E10c", "Neo-Indian: 3.Nf3 Be7", "1.d4 Nf6 2.c4 e6 3.Nf3 Be7 *"),
("E10d", "Neo-Indian: 3.Nf3 a6", "1.d4 Nf6 2.c4 e6 3.Nf3 a6 *"),
("E10e", "Neo-Indian: 3.Nf3 a6 4.Nc3", "1.d4 Nf6 2.c4 e6 3.Nf3 a6 4.Nc3 *"),
("E10f", "Neo-Indian: 3.Nf3 a6 4.Nc3 c5", "1.d4 Nf6 2.c4 e6 3.Nf3 a6 4.Nc3 c5 *"),
("E10g", "Neo-Indian: Blumenfeld/Benoni", "1.d4 Nf6 2.c4 e6 3.Nf3 c5 *"),
("E10h", "Neo-Indian: Blumenfeld/Benoni, 4.e3", "1.d4 Nf6 2.c4 e6 3.Nf3 c5 4.e3 *"),
("E10i", "Neo-Indian: Blumenfeld/Benoni, 4.e3 b6", "1.d4 Nf6 2.c4 e6 3.Nf3 c5 4.e3 b6 *"),
("E10j", "Neo-Indian: Blumenfeld/Benoni, 4.e3 cxd4", "1.d4 Nf6 2.c4 e6 3.Nf3 c5 4.e3 cxd4 *"),
// E10k "Neo-Indian: Blumenfeld/Benoni, 4.d5"  1.d4 Nf6 2.c4 e6 3.Nf3 c5 4.d5 *
("E10k", "Blumenfeld Countergambit", "1.d4 Nf6 2.c4 e6 3.Nf3 c5 4.d5 b5 *"),
("E10k", "Blumenfeld: 5.dxe6", "1.d4 Nf6 2.c4 e6 3.Nf3 c5 4.d5 b5 5.dxe6 *"),
("E10k", "Blumenfeld: 5.dxe6 fxe6 6.cxb5", "1.d4 Nf6 2.c4 e6 3.Nf3 c5 4.d5 b5 5.dxe6 fxe6 6.cxb5 *"),
("E10l", "Blumenfeld: 5.dxe6 fxe6 6.cxb5 d5", "1.d4 Nf6 2.c4 e6 3.Nf3 c5 4.d5 b5 5.dxe6 fxe6 6.cxb5 d5 *"),
("E10m", "Blumenfeld: 5.Bg5", "1.d4 Nf6 2.c4 e6 3.Nf3 c5 4.d5 b5 5.Bg5 *"),
("E10n", "Blumenfeld: 5.Bg5 h6", "1.d4 Nf6 2.c4 e6 3.Nf3 c5 4.d5 b5 5.Bg5 h6 *"),
("E10o", "Blumenfeld: 5.Bg5 Qa5+", "1.d4 Nf6 2.c4 e6 3.Nf3 c5 4.d5 b5 5.Bg5 Qa5+ *"),
("E10p", "Blumenfeld: 5.Bg5 exd5", "1.d4 Nf6 2.c4 e6 3.Nf3 c5 4.d5 b5 5.Bg5 exd5 *"),
("E10q", "Blumenfeld: Spielmann Variation", "1.d4 Nf6 2.c4 e6 3.Nf3 c5 4.d5 b5 5.Bg5 exd5 6.cxd5 h6 *"),
("E11a", "Bogo-Indian", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ *"),
("E11a", "Bogo-Indian: 4.Nbd2", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Nbd2 *"),
("E11b", "Bogo-Indian: 4.Nbd2 d5", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Nbd2 d5 *"),
("E11c", "Bogo-Indian: 4.Nbd2 b6", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Nbd2 b6 *"),
("E11c", "Bogo-Indian: 4.Nbd2 b6 5.a3", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Nbd2 b6 5.a3 *"),
("E11c", "Bogo-Indian: 4.Nbd2 b6 5.a3 Bxd2+", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Nbd2 b6 5.a3 Bxd2+ *"),
("E11d", "Bogo-Indian: 4.Nbd2 b6 5.a3 Bxd2+ 6.Bxd2", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Nbd2 b6 5.a3 Bxd2+ 6.Bxd2 *"),
("E11d", "Bogo-Indian: 4.Nbd2 b6 5.a3 Bxd2+ 6.Bxd2 Bb7", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Nbd2 b6 5.a3 Bxd2+ 6.Bxd2 Bb7 *"),
("E11e", "Bogo-Indian: 4.Nbd2 O-O", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Nbd2 O-O *"),
("E11f", "Bogo-Indian: 4.Nbd2 O-O 5.a3", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Nbd2 O-O 5.a3 *"),
("E11f", "Bogo-Indian: 4.Nbd2 O-O 5.a3 Bxd2+", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Nbd2 O-O 5.a3 Bxd2+ *"),
("E11f", "Bogo-Indian: 4.Nbd2 O-O 5.a3 Be7", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Nbd2 O-O 5.a3 Be7 *"),
("E11f", "Bogo-Indian: Gruenfeld Variation", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Nbd2 O-O 5.a3 Be7 6.e4 d5 *"),
("E11g", "Bogo-Indian: 4.Bd2", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 *"),
("E11g", "Bogo-Indian: 4.Bd2 Be7", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Be7 *"),
("E11g", "Bogo-Indian: 4.Bd2 Be7 5.g3", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Be7 5.g3 *"),
("E11g", "Bogo-Indian: 4.Bd2 Be7 5.g3 d5", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Be7 5.g3 d5 *"),
("E11g", "Bogo-Indian: 4.Bd2 Be7 5.g3 d5 6.Bg2", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Be7 5.g3 d5 6.Bg2 *"),
("E11h", "Bogo-Indian: 4.Bd2 Be7 5.g3 d5 6.Bg2 O-O", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Be7 5.g3 d5 6.Bg2 O-O *"),
("E11h", "Bogo-Indian: 4.Bd2 Be7 5.g3 d5 6.Bg2 O-O 7.O-O c6", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Be7 5.g3 d5 6.Bg2 O-O 7.O-O c6 *"),
("E11i", "Bogo-Indian: 4.Bd2 Be7 5.g3 d5 6.Bg2 O-O 7.O-O c6 8.Qc2", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Be7 5.g3 d5 6.Bg2 O-O 7.O-O c6 8.Qc2 *"),
("E11i", "Bogo-Indian: 4.Bd2 Be7 5.g3 d5 6.Bg2 O-O 7.O-O c6 8.Qc2 b6", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Be7 5.g3 d5 6.Bg2 O-O 7.O-O c6 8.Qc2 b6 *"),
("E11i", "Bogo-Indian: 4.Bd2 Be7 5.g3 d5 6.Bg2 O-O 7.O-O c6 8.Qc2 b6 9.Bf4", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Be7 5.g3 d5 6.Bg2 O-O 7.O-O c6 8.Qc2 b6 9.Bf4 *"),
("E11j", "Bogo-Indian: Vitolins Variation", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 c5 *"),
("E11k", "Bogo-Indian: Vitolins, 6.g3", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 c5 5.Bxb4 cxb4 6.g3 *"),
("E11l", "Bogo-Indian: 4.Bd2 a5", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 a5 *"),
("E11l", "Bogo-Indian: 4.Bd2 a5 5.Nc3", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 a5 5.Nc3 *"),
("E11m", "Bogo-Indian: 4.Bd2 a5 5.g3", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 a5 5.g3 *"),
("E11n", "Bogo-Indian: Nimzowitsch Variation", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Qe7 *"),
("E11o", "Bogo-Indian: Nimzowitsch, 5.Nc3", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Qe7 5.Nc3 *"),
("E11p", "Bogo-Indian: Nimzowitsch, 5.g3", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Qe7 5.g3 *"),
("E11q", "Bogo-Indian: Nimzowitsch, 5.g3 O-O", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Qe7 5.g3 O-O *"),
("E11q", "Bogo-Indian: Nimzowitsch, 5.g3 O-O", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Qe7 5.g3 O-O 6.Bg2 *"),
("E11r", "Bogo-Indian: Nimzowitsch, 5.g3 Nc6", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Qe7 5.g3 Nc6 *"),
("E11r", "Bogo-Indian: Nimzowitsch, 5.g3 Nc6 6.Nc3", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Qe7 5.g3 Nc6 6.Nc3 *"),
("E11s", "Bogo-Indian: Nimzowitsch, 5.g3 Nc6 6.Nc3 Bxc3", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Qe7 5.g3 Nc6 6.Nc3 Bxc3 *"),
("E11s", "Bogo-Indian: Nimzowitsch, 5.g3 Nc6 6.Nc3 Bxc3 7.Bxc3", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Qe7 5.g3 Nc6 6.Nc3 Bxc3 7.Bxc3 *"),
("E11s", "Bogo-Indian: Nimzowitsch, 5.g3 Nc6 6.Nc3 Bxc3 7.Bxc3 Ne4", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Qe7 5.g3 Nc6 6.Nc3 Bxc3 7.Bxc3 Ne4 *"),
("E11s", "Bogo-Indian: Nimzowitsch, 5.g3 Nc6 6.Nc3 Bxc3 7.Bxc3 Ne4 8.Rc1", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Qe7 5.g3 Nc6 6.Nc3 Bxc3 7.Bxc3 Ne4 8.Rc1 *"),
("E11t", "Bogo-Indian: Nimzowitsch, 5.g3 Nc6 6.Bg2", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Qe7 5.g3 Nc6 6.Bg2 *"),
("E11t", "Bogo-Indian: Nimzowitsch, 5.g3 Nc6 6.Bg2 Bxd2+", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Qe7 5.g3 Nc6 6.Bg2 Bxd2+ *"),
("E11u", "Bogo-Indian: Nimzowitsch, 5.g3 Nc6 6.Bg2 Bxd2+ 7.Nbxd2", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Qe7 5.g3 Nc6 6.Bg2 Bxd2+ 7.Nbxd2 *"),
("E11v", "Bogo-Indian: Nimzowitsch, 5.g3 Nc6 Main Line", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Qe7 5.g3 Nc6 6.Bg2 Bxd2+ 7.Nbxd2 d6 8.O-O *"),
("E11v", "Bogo-Indian: Nimzowitsch, 5.g3 Nc6 Main Line, 8...O-O", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Qe7 5.g3 Nc6 6.Bg2 Bxd2+ 7.Nbxd2 d6 8.O-O O-O *"),
("E11w", "Bogo-Indian: Nimzowitsch, 5.g3 Nc6 Main Line, 8...a5", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Qe7 5.g3 Nc6 6.Bg2 Bxd2+ 7.Nbxd2 d6 8.O-O a5 *"),
("E11w", "Bogo-Indian: Nimzowitsch, 5.g3 Nc6 Main Line, 8...a5 9.e4", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Qe7 5.g3 Nc6 6.Bg2 Bxd2+ 7.Nbxd2 d6 8.O-O a5 9.e4 *"),
("E11w", "Bogo-Indian: Nimzowitsch, 5.g3 Nc6 Main Line, 8...a5 9.e4 e5 10.d5 Nb8", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Qe7 5.g3 Nc6 6.Bg2 Bxd2+ 7.Nbxd2 d6 8.O-O a5 9.e4 e5 10.d5 Nb8 *"),
("E11x", "Bogo-Indian: 4.Bd2 Bxd2+", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Bxd2+ *"),
("E11x", "Bogo-Indian: 4.Bd2 Bxd2+ 5.Nbxd2", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Bxd2+ 5.Nbxd2 *"),
("E11y", "Bogo-Indian: 4.Bd2 Bxd2+ 5.Qxd2", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Bxd2+ 5.Qxd2 *"),
("E11y", "Bogo-Indian: 4.Bd2 Bxd2+ 5.Qxd2 O-O", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Bxd2+ 5.Qxd2 O-O *"),
("E11y", "Bogo-Indian: 4.Bd2 Bxd2+ 5.Qxd2 O-O 6.Nc3", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Bxd2+ 5.Qxd2 O-O 6.Nc3 *"),
("E11y", "Bogo-Indian: 4.Bd2 Bxd2+ 5.Qxd2 O-O 6.Nc3 d5", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Bxd2+ 5.Qxd2 O-O 6.Nc3 d5 *"),
("E11z", "Bogo-Indian: 4.Bd2 Bxd2+ 5.Qxd2 O-O 6.Nc3 d5 7.e3", "1.d4 Nf6 2.c4 e6 3.Nf3 Bb4+ 4.Bd2 Bxd2+ 5.Qxd2 O-O 6.Nc3 d5 7.e3 *"),
("E12a", "Queen's Indian", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 *"),
("E12b", "Queen's Indian: Miles Variation", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Bf4 *"),
("E12b", "Queen's Indian: Miles, 5.e3 Be7", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Bf4 Bb7 5.e3 Be7 *"),
("E12b", "Queen's Indian: Miles, 5.e3 Be7 6.h3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Bf4 Bb7 5.e3 Be7 6.h3 *"),
("E12c", "Queen's Indian: Miles, 5.e3 Bb4+", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Bf4 Bb7 5.e3 Bb4+ *"),
("E12c", "Queen's Indian: Miles, 5.e3 Bb4+ 6.Nbd2", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Bf4 Bb7 5.e3 Bb4+ 6.Nbd2 *"),
("E12c", "Queen's Indian: Miles, 5.e3 Bb4+ 6.Nfd2", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Bf4 Bb7 5.e3 Bb4+ 6.Nfd2 *"),
("E12d", "Queen's Indian: Petrosian", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 *"),
("E12e", "Queen's Indian: Petrosian, 4...c5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 c5 *"),
("E12e", "Queen's Indian: Petrosian, 4...c5 5.d5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 c5 5.d5 *"),
("E12e", "Queen's Indian: Petrosian, 4...c5 5.d5 Ba6", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 c5 5.d5 Ba6 *"),
("E12f", "Queen's Indian: Petrosian, 4...c5 5.d5 Ba6 6.Qc2", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 c5 5.d5 Ba6 6.Qc2 *"),
("E12f", "Queen's Indian: Petrosian, 4...c5 5.d5 Ba6 6.Qc2 exd5 6.cxd5 g6", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 c5 5.d5 Ba6 6.Qc2 exd5 6.cxd5 g6 *"),
("E12g", "Queen's Indian: Petrosian, 4...Ba6", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Ba6 *"),
("E12g", "Queen's Indian: Petrosian, 4...Ba6 5.e3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Ba6 5.e3 *"),
("E12g", "Queen's Indian: Petrosian, 4...Ba6 5.Qc2", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Ba6 5.Qc2 *"),
("E12g", "Queen's Indian: Petrosian, 4...Ba6 5.Qc2 c5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Ba6 5.Qc2 c5 *"),
("E12h", "Queen's Indian: Petrosian, 4...Ba6 5.Qc2 Bb7", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Ba6 5.Qc2 Bb7 *"),
("E12h", "Queen's Indian: Petrosian, 4...Ba6 5.Qc2 Bb7 6.Nc3 c5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Ba6 5.Qc2 Bb7 6.Nc3 c5 *"),
("E12i", "Queen's Indian: Petrosian, 4...Ba6 5.Qc2 Bb7 6.Nc3 c5 7.e4", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Ba6 5.Qc2 Bb7 6.Nc3 c5 7.e4 *"),
("E12i", "Queen's Indian: Petrosian, 4...Ba6 5.Qc2 Bb7 6.Nc3 c5 7.e4 cd 8.Nxd4 Nc6", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Ba6 5.Qc2 Bb7 6.Nc3 c5 7.e4 cxd4 8.Nxd4 Nc6 *"),
("E12j", "Queen's Indian: Petrosian, 4...Bb7", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 *"),
("E12j", "Queen's Indian: Petrosian, 5.Nc3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 *"),
("E12j", "Queen's Indian: Petrosian, 5.Nc3 Bxf3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 Bxf3 *"),
("E12j", "Queen's Indian: Petrosian, 5.Nc3 Be7", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 Be7 *"),
("E12k", "Queen's Indian: Petrosian, 5.Nc3 Ne4", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 Ne4 *"),
("E12l", "Queen's Indian: Petrosian, 5.Nc3 g6", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 g6 *"),
("E12m", "Queen's Indian: Petrosian, 5.Nc3 d5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 d5 *"),
("E12n", "Queen's Indian: Petrosian, 5.Nc3 d5 6.Bg5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 d5 6.Bg5 *"),
("E12n", "Queen's Indian: Petrosian, 5.Nc3 d5 6.Bg5 Be7", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 d5 6.Bg5 Be7 *"),
("E12o", "Queen's Indian: Petrosian, 5.Nc3 d5 6.Bg5 Be7 7.Qa4+", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 d5 6.Bg5 Be7 7.Qa4+ *"),
("E12o", "Queen's Indian: Petrosian, 5.Nc3 d5 6.Bg5 Be7 7.Qa4+ c6", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 d5 6.Bg5 Be7 7.Qa4+ c6 *"),
("E12o", "Queen's Indian: Petrosian, 5.Nc3 d5 6.Bg5 Be7 7.Qa4+ c6 8.Bxf6 Bxf6 9.cxd5 exd5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 d5 6.Bg5 Be7 7.Qa4+ c6 8.Bxf6 Bxf6 9.cxd5 exd5 *"),
("E12p", "Queen's Indian: Petrosian, 6.cxd5 exd5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 d5 6.cxd5 exd5 *"),
("E12p", "Queen's Indian: Petrosian, 6.cxd5 exd5 7.g3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 d5 6.cxd5 exd5 7.g3 *"),
("E12q", "Queen's Indian: Petrosian, 6.cxd5 exd5 7.Bg5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 d5 6.cxd5 exd5 7.Bg5 *"),
("E12r", "Queen's Indian: Petrosian, 6.cxd5 Nxd5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 d5 6.cxd5 Nxd5 *"),
("E12r", "Queen's Indian: Petrosian, 6.cxd5 Nxd5 7.e4", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 d5 6.cxd5 Nxd5 7.e4 *"),
("E12s", "Queen's Indian: Petrosian, 6.cxd5 Nxd5 7.e3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 d5 6.cxd5 Nxd5 7.e3 *"),
("E12s", "Queen's Indian: Petrosian, 6.cxd5 Nxd5 7.e3 g6", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 d5 6.cxd5 Nxd5 7.e3 g6 *"),
("E12s", "Queen's Indian: Petrosian, 6.cxd5 Nxd5 7.e3 Be7", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 d5 6.cxd5 Nxd5 7.e3 Be7 *"),
("E12t", "Queen's Indian: Petrosian, Kasparov Variation", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 d5 6.cxd5 Nxd5 7.Qc2 *"),
("E12t", "Queen's Indian: Petrosian, Kasparov, 7...Be7", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 d5 6.cxd5 Nxd5 7.Qc2 Be7 *"),
("E12t", "Queen's Indian: Petrosian, Kasparov, 7...c5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 d5 6.cxd5 Nxd5 7.Qc2 c5 *"),
("E12t", "Queen's Indian: Petrosian, Kasparov, 7...Nxc3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 d5 6.cxd5 Nxd5 7.Qc2 Nxc3 *"),
("E12u", "Queen's Indian: Petrosian, Kasparov, 7...Nxc3 8.bxc3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 d5 6.cxd5 Nxd5 7.Qc2 Nxc3 8.bxc3 *"),
("E12u", "Queen's Indian: Petrosian, Kasparov, 7...Nxc3 8.bxc3 c5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 d5 6.cxd5 Nxd5 7.Qc2 Nxc3 8.bxc3 c5 *"),
("E12u", "Queen's Indian: Petrosian, Kasparov, 7...Nxc3 8.bxc3 c5 9.e4", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 d5 6.cxd5 Nxd5 7.Qc2 Nxc3 8.bxc3 c5 8.e4 *"),
("E12u", "Queen's Indian: Petrosian, Kasparov, 7...Nxc3 8.bxc3 Be7", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 d5 6.cxd5 Nxd5 7.Qc2 Nxc3 8.bxc3 Be7 *"),
("E12u", "Queen's Indian: Petrosian, Kasparov, 7...Nxc3 8.bxc3 Be7 9.e4", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.a3 Bb7 5.Nc3 d5 6.cxd5 Nxd5 7.Qc2 Nxc3 8.bxc3 Be7 8.e4 *"),
("E12v", "Queen's Indian: 4.Bg5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Bg5 *"),
("E12w", "Queen's Indian: 4.Nc3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 *"),
("E12w", "Queen's Indian: 4.Nc3 Bb7", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 *"),
("E12w", "Queen's Indian: 4.Nc3 Bb7 5.Bg5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 *"),
("E12x", "Queen's Indian: 5.Bg5 h6 6.Bh4 Be7", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Be7 *"),
("E12x", "Queen's Indian: 5.Bg5 h6 6.Bh4 Be7 7.e3 Ne4", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Be7 7.e3 Ne4 *"),
("E12x", "Queen's Indian: 5.Bg5 h6 6.Bh4 Be7 7.e3 c5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Be7 7.e3 c5 *"),
("E12y", "Queen's Indian: Botvinnik Variation", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 g5 *"),
("E12y", "Queen's Indian: Botvinnik, 8.Qc2", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 g5 7.Bg3 Nh5 8.Qc2 *"),
("E12y", "Queen's Indian: Botvinnik, 8.e3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 g5 7.Bg3 Nh5 8.e3 *"),
("E13a", "Queen's Indian: 5.Bg5 h6 6.Bh4 Bb4", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Bb4 *"),
("E13b", "Queen's Indian: 5.Bg5 h6 6.Bh4 Bb4 7.Qc2", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Bb4 7.Qc2 *"),
("E13c", "Queen's Indian: 5.Bg5 h6 6.Bh4 Bb4 7.Qc2 g5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Bb4 7.Qc2 g5 *"),
("E13d", "Queen's Indian: 5.Bg5 h6 6.Bh4 Bb4 7.Nd2", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Bb4 7.Nd2 *"),
("E13e", "Queen's Indian: 5.Bg5 h6 6.Bh4 Bb4 7.Nd2 Bxc3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Bb4 7.Nd2 Bxc3 *"),
("E13e", "Queen's Indian: 5.Bg5 h6 6.Bh4 Bb4 7.Nd2 Bxc3 8.bxc3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Bb4 7.Nd2 Bxc3 8.bxc3 *"),
("E13f", "Queen's Indian: 5.Bg5 h6 6.Bh4 Bb4 7.e3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Bb4 7.e3 *"),
("E13g", "Queen's Indian: 5.Bg5 h6 6.Bh4 Bb4 7.e3 c5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Bb4 7.e3 c5 *"),
("E13h", "Queen's Indian: 5.Bg5 h6 6.Bh4 Bb4 7.e3 Bxc3+", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Bb4 7.e3 Bxc3+ *"),
("E13h", "Queen's Indian: 5.Bg5 h6 6.Bh4 Bb4 7.e3 Bxc3+ 8.bxc3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Bb4 7.e3 Bxc3+ 8.bxc3 *"),
("E13i", "Queen's Indian: 5.Bg5 h6 6.Bh4 Bb4 7.e3 Bxc3+ 8.bxc3 d6", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Bb4 7.e3 Bxc3+ 8.bxc3 d6 *"),
("E13j", "Queen's Indian: 5.Bg5 h6 6.Bh4 Bb4 7.e3 Bxc3+ 8.bxc3 d6 9.Nd2", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Bb4 7.e3 Bxc3+ 8.bxc3 d6 9.Nd2 *"),
("E13k", "Queen's Indian: 5.Bg5 h6 6.Bh4 Bb4 7.e3 Bxc3+ 8.bxc3 d6 9.Nd2 Nbd7", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Bb4 7.e3 Bxc3+ 8.bxc3 d6 9.Nd2 Nbd7 *"),
("E13k", "Queen's Indian: 5.Bg5 h6 6.Bh4 Bb4 7.e3 Bxc3+ 8.bxc3 d6 9.Nd2 Nbd7 10.f3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Bb4 7.e3 Bxc3+ 8.bxc3 d6 9.Nd2 Nbd7 10.f3 *"),
("E13l", "Queen's Indian: 5.Bg5 h6 6.Bh4 Bb4 7.e3 Bxc3+ 8.bxc3 d6 9.Nd2 Nbd7 10.f3 Qe7", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Bb4 7.e3 Bxc3+ 8.bxc3 d6 9.Nd2 Nbd7 10.f3 Qe7 *"),
("E13m", "Queen's Indian: 5.Bg5 h6 6.Bh4 Bb4 7.e3 g5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Bb4 7.e3 g5 *"),
("E13m", "Queen's Indian: 5.Bg5 h6 6.Bh4 Bb4 7.e3 g5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Bb4 7.e3 g5 8.Bg3 *"),
("E13m", "Queen's Indian: 5.Bg5 h6 6.Bh4 Bb4 7.e3 g5 8.Bg3 Ne4", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Bb4 7.e3 g5 8.Bg3 Ne4 *"),
("E13n", "Queen's Indian: 5.Bg5 h6 6.Bh4 Bb4 7.e3 g5 8.Bg3 Ne4 9.Qc2", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Bb4 7.e3 g5 8.Bg3 Ne4 9.Qc2 *"),
("E13o", "Queen's Indian: 5.Bg5 h6 6.Bh4 Bb4 7.e3 g5 8.Bg3 Ne4 9.Qc2 Bxc3+", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Bb4 7.e3 g5 8.Bg3 Ne4 9.Qc2 Bxc3+ *"),
("E13o", "Queen's Indian: 5.Bg5 h6 6.Bh4 Bb4 7.e3 g5 8.Bg3 Ne4 9.Qc2 Bxc3+", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Bb4 7.e3 g5 8.Bg3 Ne4 9.Qc2 Bxc3+ 10.bxc3 *"),
("E13p", "Queen's Indian: 5.Bg5 h6 6.Bh4 Bb4 7.e3 g5 8.Bg3 Ne4 9.Qc2 Bxc3+ 10.bxc3 d6", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Bb4 7.e3 g5 8.Bg3 Ne4 9.Qc2 Bxc3+ 10.bxc3 d6 *"),
("E13p", "Queen's Indian: 5.Bg5 h6 6.Bh4 Bb4 7.e3 g5 8.Bg3 Ne4 9.Qc2 Bxc3+ 10.bxc3 d6 11.Bd3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Bb4 7.e3 g5 8.Bg3 Ne4 9.Qc2 Bxc3+ 10.bxc3 d6 11.Bd3 *"),
("E13q", "Queen's Indian: 5.Bg5 h6 6.Bh4 Bb4 7.e3 g5 8.Bg3 Ne4 9.Qc2 Bxc3+ 10.bxc3 d6 11.Bd3 f5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.Nc3 Bb7 5.Bg5 h6 6.Bh4 Bb4 7.e3 g5 8.Bg3 Ne4 9.Qc2 Bxc3+ 10.bxc3 d6 11.Bd3 f5 *"),
("E14a", "Queen's Indian: 4.e3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 *"),
("E14a", "Queen's Indian: 4.e3 Bb4+", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb4+ *"),
("E14b", "Queen's Indian: 4.e3 Bb7", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 *"),
("E14c", "Queen's Indian: 4.e3 Bb7 5.Nc3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Nc3 *"),
("E14d", "Queen's Indian: 4.e3 Bb7 5.Nc3 d5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Nc3 d5 *"),
("E14e", "Queen's Indian: 4.e3 Bb7 5.Bd3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 *"),
("E14f", "Queen's Indian: 4.e3 Bb7 5.Bd3 Bb4+", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 Bb4+ *"),
("E14g", "Queen's Indian: 4.e3 Bb7 5.Bd3 Bb4+ 6.Nbd2", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 Bb4+ 6.Nbd2 *"),
("E14g", "Queen's Indian: 4.e3 Bb7 5.Bd3 Bb4+ 6.Nbd2 c5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 Bb4+ 6.Nbd2 c5 *"),
("E14h", "Queen's Indian: 4.e3 Bb7 5.Bd3 Bb4+ 6.Nbd2 O-O", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 Bb4+ 6.Nbd2 O-O *"),
("E14i", "Queen's Indian: 4.e3 Bb7 5.Bd3 Bb4+ 6.Nbd2 O-O 7.O-O d5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 Bb4+ 6.Nbd2 O-O 7.O-O d5 *"),
("E14i", "Queen's Indian: 4.e3 Bb7 5.Bd3 Bb4+ 6.Nbd2 O-O 7.O-O d5 8.a3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 Bb4+ 6.Nbd2 O-O 7.O-O d5 8.a3 *"),
("E14i", "Queen's Indian: 4.e3 Bb7 5.Bd3 Bb4+ 6.Nbd2 O-O 7.O-O d5 8.a3 Be7", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 Bb4+ 6.Nbd2 O-O 7.O-O d5 8.a3 Be7 *"),
("E14i", "Queen's Indian: Dreev Variation", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 Bb4+ 6.Nbd2 O-O 7.O-O d5 8.a3 Be7 9.b4 *"),
("E14j", "Queen's Indian: 4.e3 Bb7 5.Bd3 Be7", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 Be7 *"),
("E14k", "Queen's Indian: 4.e3 Bb7 5.Bd3 Be7 6.O-O", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 Be7 6.O-O *"),
("E14k", "Queen's Indian: 4.e3 Bb7 5.Bd3 Be7 6.O-O O-O", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 Be7 6.O-O O-O *"),
("E14l", "Queen's Indian: 4.e3 Bb7 5.Bd3 Be7 6.O-O O-O 7.Nc3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 Be7 6.O-O O-O 7.Nc3 *"),
("E14m", "Queen's Indian: 4.e3 Bb7 5.Bd3 d5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 d5 *"),
("E14n", "Queen's Indian: 4.e3 Bb7 5.Bd3 d5 6.O-O", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 d5 6.O-O *"),
("E14n", "Queen's Indian: 4.e3 Bb7 5.Bd3 d5 6.O-O Bd6", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 d5 6.O-O Bd6 *"),
("E14o", "Queen's Indian: 4.e3 Bb7 5.Bd3 d5 6.O-O Bd6 7.b3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 d5 6.O-O Bd6 7.b3 *"),
("E14p", "Queen's Indian: 4.e3 Bb7 5.Bd3 d5 6.O-O Bd6 7.Nc3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 d5 6.O-O Bd6 7.Nc3 *"),
("E14q", "Queen's Indian: 4.e3 Bb7 5.Bd3 c5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 c5 *"),
("E14q", "Queen's Indian: 4.e3 Bb7 5.Bd3 c5 6.Nc3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 c5 6.Nc3 *"),
("E14r", "Queen's Indian: 4.e3 Bb7 5.Bd3 c5 6.O-O", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 c5 6.O-O *"),
("E14s", "Queen's Indian: 4.e3 Bb7 5.Bd3 c5 6.O-O cxd4", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 c5 6.O-O cxd4 *"),
("E14s", "Queen's Indian: 4.e3 Bb7 5.Bd3 c5 6.O-O cxd4 7.exd4", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 c5 6.O-O cxd4 7.exd4 *"),
("E14t", "Queen's Indian: 4.e3 Bb7 5.Bd3 c5 6.O-O Be7", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 *"),
("E14t", "Queen's Indian: 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.Nbd2", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.Nbd2 *"),
("E14u", "Queen's Indian: 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.b3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.b3 *"),
("E14u", "Queen's Indian: 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.b3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.b3 O-O 8.Bb2 *"),
("E14v", "Queen's Indian: 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.b3 O-O 8.Bb2 d5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.b3 O-O 8.Bb2 d5 *"),
("E14w", "Queen's Indian: 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.b3 O-O 8.Bb2 cxd4", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.b3 O-O 8.Bb2 cxd4 *"),
("E14w", "Queen's Indian: Averbakh Variation", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.b3 O-O 8.Bb2 cxd4 9.Nxd4 *"),
("E14w", "Queen's Indian: 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.b3 O-O 8.Bb2 cxd4 9.exd4", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.b3 O-O 8.Bb2 cxd4 9.exd4 *"),
("E14w", "Queen's Indian: 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.b3 O-O 8.Bb2 cxd4 9.exd4 d5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.b3 O-O 8.Bb2 cxd4 9.exd4 d5 *"),
("E14x", "Queen's Indian: 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.Nc3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.Nc3 *"),
("E14x", "Queen's Indian: 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.Nc3 O-O", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.Nc3 O-O *"),
("E14x", "Queen's Indian: 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.Nc3 cxd4", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.Nc3 cxd4 *"),
("E14y", "Queen's Indian: 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.Nc3 cxd4 8.exd4", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.Nc3 cxd4 8.exd4 *"),
("E14z", "Queen's Indian: 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.Nc3 cxd4 8.exd4 d5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.e3 Bb7 5.Bd3 c5 6.O-O Be7 7.Nc3 cxd4 8.exd4 d5 *"),
("E15a", "Queen's Indian: 4.g3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 *"),
("E15b", "Queen's Indian: 4.g3 Bb4+", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Bb4+ *"),
("E15b", "Queen's Indian: 4.g3 Bb4+", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Bb4+ 5.Bd2 *"),
("E15c", "Queen's Indian: 4.g3 Bb4+ 5.Bd2 Bxd2+", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Bb4+ 5.Bd2 Bxd2+ *"),
("E15c", "Queen's Indian: 4.g3 Bb4+ 5.Bd2 Bxd2+ 6.Qxd2 Ba6", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Bb4+ 5.Bd2 Bxd2+ 6.Qxd2 Ba6 *"),
("E15d", "Queen's Indian: Nimzowitsch Variation (4.g3 Ba6)", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 *"),
("E15d", "Queen's Indian: Nimzowitsch, 5.Qb3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.Qb3 *"),
("E15d", "Queen's Indian: Nimzowitsch, 5.Qc2", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.Qc2 *"),
("E15e", "Queen's Indian: Nimzowitsch, 5.Nbd2", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.Nbd2 *"),
("E15e", "Queen's Indian: Nimzowitsch, 5.Nbd2 c5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.Nbd2 c5 *"),
("E15f", "Queen's Indian: Nimzowitsch, 5.Nbd2 Bb7", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.Nbd2 Bb7 *"),
("E15f", "Queen's Indian: Nimzowitsch, 5.Nbd2 Bb7 6.Bg2", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.Nbd2 Bb7 6.Bg2 *"),
("E15f", "Queen's Indian: Nimzowitsch, 5.Nbd2 Bb7 6.Bg2 c5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.Nbd2 Bb7 6.Bg2 c5 *"),
("E15g", "Queen's Indian: Nimzowitsch, 5.Nbd2 Bb4", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.Nbd2 Bb4 *"),
("E15h", "Queen's Indian: Nimzowitsch, 5.Qa4", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.Qa4 *"),
("E15h", "Queen's Indian: Nimzowitsch, 5.Qa4 Be7", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.Qa4 Be7 *"),
("E15i", "Queen's Indian: Nimzowitsch, 5.Qa4 c6", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.Qa4 c6 *"),
("E15i", "Queen's Indian: Nimzowitsch, 5.Qa4 c6 6.Nc3 b5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.Qa4 c6 6.Nc3 b5 *"),
("E15j", "Queen's Indian: Nimzowitsch, 5.Qa4 c5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.Qa4 c5 *"),
("E15j", "Queen's Indian: Nimzowitsch, 5.Qa4 c5 6.Bg2 Bb7", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.Qa4 c5 6.Bg2 Bb7 *"),
("E15j", "Queen's Indian: Nimzowitsch, 5.Qa4 c5 6.Bg2 Bb7 7.dxc5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.Qa4 c5 6.Bg2 Bb7 7.dxc5 *"),
("E15k", "Queen's Indian: Nimzowitsch, 5.Qa4 c5 6.Bg2 Bb7 7.O-O", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.Qa4 c5 6.Bg2 Bb7 7.O-O *"),
("E15l", "Queen's Indian: Nimzowitsch, 5.b3 ", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.b3 *"),
("E15l", "Queen's Indian: Nimzowitsch, 5.b3 b5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.b3 b5 *"),
("E15l", "Queen's Indian: Nimzowitsch, 5.b3 b5 6.cxb5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.b3 b5 6.cxb5 *"),
("E15m", "Queen's Indian: Nimzowitsch, 5.b3 d5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.b3 d5 *"),
("E15m", "Queen's Indian: Nimzowitsch, 5.b3 d5 6.cxd5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.b3 d5 6.cxd5 *"),
("E15m", "Queen's Indian: Nimzowitsch, 5.b3 d5 6.Bg2", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.b3 d5 6.Bg2 *"),
("E15n", "Queen's Indian: Nimzowitsch, 5.b3 Bb7", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.b3 Bb7 *"),
("E15n", "Queen's Indian: Nimzowitsch, 5.b3 Bb7 6.Bg2", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.b3 Bb7 6.Bg2 *"),
("E15n", "Queen's Indian: Nimzowitsch, 5.b3 Bb7, 7...a5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.b3 Bb7 6.Bg2 Bb4+ 7.Bd2 a5 *"),
("E15n", "Queen's Indian: Nimzowitsch, 5.b3 Bb7, 7...a5 8.O-O O-O", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.b3 Bb7 6.Bg2 Bb4+ 7.Bd2 a5 8.O-O O-O *"),
("E15o", "Queen's Indian: Nimzowitsch, 5.b3 Bb4+", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.b3 Bb4+ *"),
("E15o", "Queen's Indian: Nimzowitsch, 5.b3 Bb4+ 6.Bd2 Be7", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.b3 Bb4+ 6.Bd2 Be7 *"),
("E15o", "Queen's Indian: Nimzowitsch, 5.b3 Bb4+ 6.Bd2 Be7 7.Nc3", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.b3 Bb4+ 6.Bd2 Be7 7.Nc3 *"),
("E15p", "Queen's Indian: Nimzowitsch, 5.b3 Bb4+ 6.Bd2 Be7 7.Bg2", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.b3 Bb4+ 6.Bd2 Be7 7.Bg2 *"),
("E15p", "Queen's Indian: Nimzowitsch, 5.b3 Bb4+ 6.Bd2 Be7 7.Bg2 d5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.b3 Bb4+ 6.Bd2 Be7 7.Bg2 d5 *"),
("E15p", "Queen's Indian: Nimzowitsch, 5.b3 Bb4+ 6.Bd2 Be7 7.Bg2 d5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.b3 Bb4+ 6.Bd2 Be7 7.Bg2 d5 8.cxd5 exd5 9.O-O O-O *"),
("E15q", "Queen's Indian: Nimzowitsch, 5.b3 Bb4+ 6.Bd2 Be7 7.Bg2 c6", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.b3 Bb4+ 6.Bd2 Be7 7.Bg2 c6 *"),
("E15q", "Queen's Indian: Nimzowitsch, 5.b3 Bb4+ 6.Bd2 Be7 7.Bg2 c6 8.O-O", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.b3 Bb4+ 6.Bd2 Be7 7.Bg2 c6 8.O-O *"),
("E15q", "Queen's Indian: Nimzowitsch, 5.b3 Bb4+ 6.Bd2 Be7 7.Bg2 c6 8.O-O d5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.b3 Bb4+ 6.Bd2 Be7 7.Bg2 c6 8.O-O d5 *"),
("E15r", "Queen's Indian: Nimzowitsch, 5.b3 Bb4+, Main Line", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.b3 Bb4+ 6.Bd2 Be7 7.Bg2 c6 8.Bc3 *"),
("E15s", "Queen's Indian: Nimzowitsch, 5.b3 Bb4+, Main Line, 9.Ne5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.b3 Bb4+ 6.Bd2 Be7 7.Bg2 c6 8.Bc3 d5 9.Ne5 *"),
("E15s", "Queen's Indian: Nimzowitsch, 5.b3 Bb4+, Main Line, 9.Ne5 Nfd7", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Ba6 5.b3 Bb4+ 6.Bd2 Be7 7.Bg2 c6 8.Bc3 d5 9.Ne5 Nfd7 *"),
("E15t", "Queen's Indian: 4.g3 Bb7", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Bb7 *"),
("E15t", "Queen's Indian: 4.g3 Bb7", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Bb7 5.Bg2 *"),
("E15t", "Queen's Indian: 4.g3 Bb7 5.Bg2 Qc8", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Bb7 5.Bg2 Qc8 *"),
("E15t", "Queen's Indian: 4.g3 Bb7 5.Bg2 Qc8 6.O-O c5 7.d5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Bb7 5.Bg2 Qc8 6.O-O c5 7.d5 *"),
("E15u", "Queen's Indian: 4.g3 Bb7 5.Bg2 c5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Bb7 5.Bg2 c5 *"),
("E15u", "Queen's Indian: Buerger Variation", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Bb7 5.Bg2 c5 6.d5 exd5 7.Ng5 *"),
("E15v", "Queen's Indian: Rubinstein Variation", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Bb7 5.Bg2 c5 6.d5 exd5 7.Nh4 *"),
("E16a", "Queen's Indian: Capablanca Variation", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Bb7 5.Bg2 Bb4+ *"),
("E16b", "Queen's Indian: Capablanca, 6.Nbd2", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Bb7 5.Bg2 Bb4+ 6.Nbd2 *"),
("E16c", "Queen's Indian: Capablanca, 6.Nbd2 O-O", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Bb7 5.Bg2 Bb4+ 6.Nbd2 O-O *"),
("E16d", "Queen's Indian: Capablanca, 6.Nbd2 O-O 7.O-O d5", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Bb7 5.Bg2 Bb4+ 6.Nbd2 O-O 7.O-O d5 *"),
("E16e", "Queen's Indian: Capablanca, 6.Bd2", "1.d4 Nf6 2.c4 e6 3.Nf3 b6 4.g3 Bb7 5.Bg2 Bb4+ 6.Bd2 *"),
("E16e", "Queen's Indian: Capab