// Copyright 2025 Google LLC.
// Use of this source code is governed by a BSD-style
// license that can be found in the LICENSE file.

// Code generated file. DO NOT EDIT.

// Package accessapproval provides access to the Access Approval API.
//
// For product documentation, see: https://cloud.google.com/assured-workloads/access-approval/docs
//
// # Library status
//
// These client libraries are officially supported by Google. However, this
// library is considered complete and is in maintenance mode. This means
// that we will address critical bugs and security issues but will not add
// any new features.
//
// When possible, we recommend using our newer
// [Cloud Client Libraries for Go](https://pkg.go.dev/cloud.google.com/go)
// that are still actively being worked and iterated on.
//
// # Creating a client
//
// Usage example:
//
//	import "google.golang.org/api/accessapproval/v1"
//	...
//	ctx := context.Background()
//	accessapprovalService, err := accessapproval.NewService(ctx)
//
// In this example, Google Application Default Credentials are used for
// authentication. For information on how to create and obtain Application
// Default Credentials, see https://developers.google.com/identity/protocols/application-default-credentials.
//
// # Other authentication options
//
// To use an API key for authentication (note: some APIs do not support API
// keys), use [google.golang.org/api/option.WithAPIKey]:
//
//	accessapprovalService, err := accessapproval.NewService(ctx, option.WithAPIKey("AIza..."))
//
// To use an OAuth token (e.g., a user token obtained via a three-legged OAuth
// flow, use [google.golang.org/api/option.WithTokenSource]:
//
//	config := &oauth2.Config{...}
//	// ...
//	token, err := config.Exchange(ctx, ...)
//	accessapprovalService, err := accessapproval.NewService(ctx, option.WithTokenSource(config.TokenSource(ctx, token)))
//
// See [google.golang.org/api/option.ClientOption] for details on options.
package accessapproval // import "google.golang.org/api/accessapproval/v1"

import (
	"bytes"
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"io"
	"log/slog"
	"net/http"
	"net/url"
	"strconv"
	"strings"

	"github.com/googleapis/gax-go/v2/internallog"
	googleapi "google.golang.org/api/googleapi"
	internal "google.golang.org/api/internal"
	gensupport "google.golang.org/api/internal/gensupport"
	option "google.golang.org/api/option"
	internaloption "google.golang.org/api/option/internaloption"
	htransport "google.golang.org/api/transport/http"
)

// Always reference these packages, just in case the auto-generated code
// below doesn't.
var _ = bytes.NewBuffer
var _ = strconv.Itoa
var _ = fmt.Sprintf
var _ = json.NewDecoder
var _ = io.Copy
var _ = url.Parse
var _ = gensupport.MarshalJSON
var _ = googleapi.Version
var _ = errors.New
var _ = strings.Replace
var _ = context.Canceled
var _ = internaloption.WithDefaultEndpoint
var _ = internal.Version
var _ = internallog.New

const apiId = "accessapproval:v1"
const apiName = "accessapproval"
const apiVersion = "v1"
const basePath = "https://accessapproval.googleapis.com/"
const basePathTemplate = "https://accessapproval.UNIVERSE_DOMAIN/"
const mtlsBasePath = "https://accessapproval.mtls.googleapis.com/"

// OAuth2 scopes used by this API.
const (
	// See, edit, configure, and delete your Google Cloud data and see the email
	// address for your Google Account.
	CloudPlatformScope = "https://www.googleapis.com/auth/cloud-platform"
)

// NewService creates a new Service.
func NewService(ctx context.Context, opts ...option.ClientOption) (*Service, error) {
	scopesOption := internaloption.WithDefaultScopes(
		"https://www.googleapis.com/auth/cloud-platform",
	)
	// NOTE: prepend, so we don't override user-specified scopes.
	opts = append([]option.ClientOption{scopesOption}, opts...)
	opts = append(opts, internaloption.WithDefaultEndpoint(basePath))
	opts = append(opts, internaloption.WithDefaultEndpointTemplate(basePathTemplate))
	opts = append(opts, internaloption.WithDefaultMTLSEndpoint(mtlsBasePath))
	opts = append(opts, internaloption.EnableNewAuthLibrary())
	client, endpoint, err := htransport.NewClient(ctx, opts...)
	if err != nil {
		return nil, err
	}
	s := &Service{client: client, BasePath: basePath, logger: internaloption.GetLogger(opts)}
	s.Folders = NewFoldersService(s)
	s.Organizations = NewOrganizationsService(s)
	s.Projects = NewProjectsService(s)
	if err != nil {
		return nil, err
	}
	if endpoint != "" {
		s.BasePath = endpoint
	}
	return s, nil
}

// New creates a new Service. It uses the provided http.Client for requests.
//
// Deprecated: please use NewService instead.
// To provide a custom HTTP client, use option.WithHTTPClient.
// If you are using google.golang.org/api/googleapis/transport.APIKey, use option.WithAPIKey with NewService instead.
func New(client *http.Client) (*Service, error) {
	if client == nil {
		return nil, errors.New("client is nil")
	}
	return NewService(context.TODO(), option.WithHTTPClient(client))
}

type Service struct {
	client    *http.Client
	logger    *slog.Logger
	BasePath  string // API endpoint base URL
	UserAgent string // optional additional User-Agent fragment

	Folders *FoldersService

	Organizations *OrganizationsService

	Projects *ProjectsService
}

func (s *Service) userAgent() string {
	if s.UserAgent == "" {
		return googleapi.UserAgent
	}
	return googleapi.UserAgent + " " + s.UserAgent
}

func NewFoldersService(s *Service) *FoldersService {
	rs := &FoldersService{s: s}
	rs.ApprovalRequests = NewFoldersApprovalRequestsService(s)
	return rs
}

type FoldersService struct {
	s *Service

	ApprovalRequests *FoldersApprovalRequestsService
}

func NewFoldersApprovalRequestsService(s *Service) *FoldersApprovalRequestsService {
	rs := &FoldersApprovalRequestsService{s: s}
	return rs
}

type FoldersApprovalRequestsService struct {
	s *Service
}

func NewOrganizationsService(s *Service) *OrganizationsService {
	rs := &OrganizationsService{s: s}
	rs.ApprovalRequests = NewOrganizationsApprovalRequestsService(s)
	return rs
}

type OrganizationsService struct {
	s *Service

	ApprovalRequests *OrganizationsApprovalRequestsService
}

func NewOrganizationsApprovalRequestsService(s *Service) *OrganizationsApprovalRequestsService {
	rs := &OrganizationsApprovalRequestsService{s: s}
	return rs
}

type OrganizationsApprovalRequestsService struct {
	s *Service
}

func NewProjectsService(s *Service) *ProjectsService {
	rs := &ProjectsService{s: s}
	rs.ApprovalRequests = NewProjectsApprovalRequestsService(s)
	return rs
}

type ProjectsService struct {
	s *Service

	ApprovalRequests *ProjectsApprovalRequestsService
}

func NewProjectsApprovalRequestsService(s *Service) *ProjectsApprovalRequestsService {
	rs := &ProjectsApprovalRequestsService{s: s}
	return rs
}

type ProjectsApprovalRequestsService struct {
	s *Service
}

// AccessApprovalServiceAccount: Access Approval service account related to a
// project/folder/organization.
type AccessApprovalServiceAccount struct {
	// AccountEmail: Email address of the service account.
	AccountEmail string `json:"accountEmail,omitempty"`
	// Name: The resource name of the Access Approval service account. Format is
	// one of: * "projects/{project}/serviceAccount" *
	// "folders/{folder}/serviceAccount" *
	// "organizations/{organization}/serviceAccount"
	Name string `json:"name,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "AccountEmail") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "AccountEmail") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AccessApprovalServiceAccount) MarshalJSON() ([]byte, error) {
	type NoMethod AccessApprovalServiceAccount
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AccessApprovalSettings: Settings on a Project/Folder/Organization related to
// Access Approval.
type AccessApprovalSettings struct {
	// ActiveKeyVersion: The asymmetric crypto key version to use for signing
	// approval requests. Empty active_key_version indicates that a Google-managed
	// key should be used for signing. This property will be ignored if set by an
	// ancestor of this resource, and new non-empty values may not be set.
	ActiveKeyVersion string `json:"activeKeyVersion,omitempty"`
	// AncestorHasActiveKeyVersion: Output only. This field is read only (not
	// settable via UpdateAccessApprovalSettings method). If the field is true,
	// that indicates that an ancestor of this Project or Folder has set
	// active_key_version (this field will always be unset for the organization
	// since organizations do not have ancestors).
	AncestorHasActiveKeyVersion bool `json:"ancestorHasActiveKeyVersion,omitempty"`
	// EnrolledAncestor: Output only. This field is read only (not settable via
	// UpdateAccessApprovalSettings method). If the field is true, that indicates
	// that at least one service is enrolled for Access Approval in one or more
	// ancestors of the Project or Folder (this field will always be unset for the
	// organization since organizations do not have ancestors).
	EnrolledAncestor bool `json:"enrolledAncestor,omitempty"`
	// EnrolledServices: A list of Google Cloud Services for which the given
	// resource has Access Approval enrolled. Access requests for the resource
	// given by name against any of these services contained here will be required
	// to have explicit approval. If name refers to an organization, enrollment can
	// be done for individual services. If name refers to a folder or project,
	// enrollment can only be done on an all or nothing basis. If a cloud_product
	// is repeated in this list, the first entry will be honored and all following
	// entries will be discarded. A maximum of 10 enrolled services will be
	// enforced, to be expanded as the set of supported services is expanded.
	EnrolledServices []*EnrolledService `json:"enrolledServices,omitempty"`
	// InvalidKeyVersion: Output only. This field is read only (not settable via
	// UpdateAccessApprovalSettings method). If the field is true, that indicates
	// that there is some configuration issue with the active_key_version
	// configured at this level in the resource hierarchy (e.g. it doesn't exist or
	// the Access Approval service account doesn't have the correct permissions on
	// it, etc.) This key version is not necessarily the effective key version at
	// this level, as key versions are inherited top-down.
	InvalidKeyVersion bool `json:"invalidKeyVersion,omitempty"`
	// Name: The resource name of the settings. Format is one of: *
	// "projects/{project}/accessApprovalSettings" *
	// "folders/{folder}/accessApprovalSettings" *
	// "organizations/{organization}/accessApprovalSettings"
	Name string `json:"name,omitempty"`
	// NotificationEmails: A list of email addresses to which notifications
	// relating to approval requests should be sent. Notifications relating to a
	// resource will be sent to all emails in the settings of ancestor resources of
	// that resource. A maximum of 50 email addresses are allowed.
	NotificationEmails []string `json:"notificationEmails,omitempty"`
	// NotificationPubsubTopic: Optional. A pubsub topic to which notifications
	// relating to approval requests should be sent.
	NotificationPubsubTopic string `json:"notificationPubsubTopic,omitempty"`
	// PreferNoBroadApprovalRequests: This preference is communicated to Google
	// personnel when sending an approval request but can be overridden if
	// necessary.
	PreferNoBroadApprovalRequests bool `json:"preferNoBroadApprovalRequests,omitempty"`
	// PreferredRequestExpirationDays: This preference is shared with Google
	// personnel, but can be overridden if said personnel deems necessary. The
	// approver ultimately can set the expiration at approval time.
	PreferredRequestExpirationDays int64 `json:"preferredRequestExpirationDays,omitempty"`
	// RequestScopeMaxWidthPreference: Optional. A setting to indicate the maximum
	// width of an Access Approval request.
	//
	// Possible values:
	//   "REQUEST_SCOPE_MAX_WIDTH_PREFERENCE_UNSPECIFIED" - Default value for
	// proto, shouldn't be used.
	//   "ORGANIZATION" - This is the widest scope possible. It means the customer
	// has no scope restriction when it comes to Access Approval requests.
	//   "FOLDER" - Customer allows the scope of Access Approval requests as broad
	// as the Folder level.
	//   "PROJECT" - Customer allows the scope of Access Approval requests as broad
	// as the Project level.
	RequestScopeMaxWidthPreference string `json:"requestScopeMaxWidthPreference,omitempty"`
	// RequireCustomerVisibleJustification: Optional. A setting to require approval
	// request justifications to be customer visible.
	RequireCustomerVisibleJustification bool `json:"requireCustomerVisibleJustification,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ActiveKeyVersion") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ActiveKeyVersion") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AccessApprovalSettings) MarshalJSON() ([]byte, error) {
	type NoMethod AccessApprovalSettings
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AccessLocations: Home office and physical location of the principal.
type AccessLocations struct {
	// PrincipalOfficeCountry: The "home office" location of the principal. A
	// two-letter country code (ISO 3166-1 alpha-2), such as "US", "DE" or "GB" or
	// a region code. In some limited situations Google systems may refer refer to
	// a region code instead of a country code. Possible Region Codes: * ASI: Asia
	// * EUR: Europe * OCE: Oceania * AFR: Africa * NAM: North America * SAM: South
	// America * ANT: Antarctica * ANY: Any location
	PrincipalOfficeCountry string `json:"principalOfficeCountry,omitempty"`
	// PrincipalPhysicalLocationCountry: Physical location of the principal at the
	// time of the access. A two-letter country code (ISO 3166-1 alpha-2), such as
	// "US", "DE" or "GB" or a region code. In some limited situations Google
	// systems may refer refer to a region code instead of a country code. Possible
	// Region Codes: * ASI: Asia * EUR: Europe * OCE: Oceania * AFR: Africa * NAM:
	// North America * SAM: South America * ANT: Antarctica * ANY: Any location
	PrincipalPhysicalLocationCountry string `json:"principalPhysicalLocationCountry,omitempty"`
	// ForceSendFields is a list of field names (e.g. "PrincipalOfficeCountry") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "PrincipalOfficeCountry") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AccessLocations) MarshalJSON() ([]byte, error) {
	type NoMethod AccessLocations
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type AccessReason struct {
	// Detail: More detail about certain reason types. See comments for each type
	// above.
	Detail string `json:"detail,omitempty"`
	// Type: Type of access justification.
	//
	// Possible values:
	//   "TYPE_UNSPECIFIED" - Default value for proto, shouldn't be used.
	//   "CUSTOMER_INITIATED_SUPPORT" - Customer made a request or raised an issue
	// that required the principal to access customer data. `detail` is of the form
	// ("#####" is the issue ID): * "Feedback Report: #####" * "Case Number: #####"
	// * "Case ID: #####" * "E-PIN Reference: #####" * "Google-#####" * "T-#####"
	//   "GOOGLE_INITIATED_SERVICE" - The principal accessed customer data in order
	// to diagnose or resolve a suspected issue in services. Often this access is
	// used to confirm that customers are not affected by a suspected service issue
	// or to remediate a reversible system issue.
	//   "GOOGLE_INITIATED_REVIEW" - Google initiated service for security, fraud,
	// abuse, or compliance purposes.
	//   "THIRD_PARTY_DATA_REQUEST" - The principal was compelled to access
	// customer data in order to respond to a legal third party data request or
	// process, including legal processes from customers themselves.
	//   "GOOGLE_RESPONSE_TO_PRODUCTION_ALERT" - The principal accessed customer
	// data in order to diagnose or resolve a suspected issue in services or a
	// known outage.
	//   "CLOUD_INITIATED_ACCESS" - Similar to 'GOOGLE_INITIATED_SERVICE' or
	// 'GOOGLE_INITIATED_REVIEW', but with universe agnostic naming. The principal
	// accessed customer data in order to diagnose or resolve a suspected issue in
	// services or a known outage, or for security, fraud, abuse, or compliance
	// review purposes.
	Type string `json:"type,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Detail") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Detail") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AccessReason) MarshalJSON() ([]byte, error) {
	type NoMethod AccessReason
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ApprovalRequest: A request for the customer to approve access to a resource.
type ApprovalRequest struct {
	// Approve: Access was approved.
	Approve *ApproveDecision `json:"approve,omitempty"`
	// Dismiss: The request was dismissed.
	Dismiss *DismissDecision `json:"dismiss,omitempty"`
	// Name: The resource name of the request. Format is
	// "{projects|folders|organizations}/{id}/approvalRequests/{approval_request}".
	Name string `json:"name,omitempty"`
	// RequestTime: The time at which approval was requested.
	RequestTime string `json:"requestTime,omitempty"`
	// RequestedAugmentedInfo: This field contains the augmented information of the
	// request.
	RequestedAugmentedInfo *AugmentedInfo `json:"requestedAugmentedInfo,omitempty"`
	// RequestedDuration: The requested access duration.
	RequestedDuration string `json:"requestedDuration,omitempty"`
	// RequestedExpiration: The original requested expiration for the approval.
	// Calculated by adding the requested_duration to the request_time.
	RequestedExpiration string `json:"requestedExpiration,omitempty"`
	// RequestedLocations: The locations for which approval is being requested.
	RequestedLocations *AccessLocations `json:"requestedLocations,omitempty"`
	// RequestedReason: The justification for which approval is being requested.
	RequestedReason *AccessReason `json:"requestedReason,omitempty"`
	// RequestedResourceName: The resource for which approval is being requested.
	// The format of the resource name is defined at
	// https://cloud.google.com/apis/design/resource_names. The resource name here
	// may either be a "full" resource name (e.g.
	// "//library.googleapis.com/shelves/shelf1/books/book2") or a "relative"
	// resource name (e.g. "shelves/shelf1/books/book2") as described in the
	// resource name specification.
	RequestedResourceName string `json:"requestedResourceName,omitempty"`
	// RequestedResourceProperties: Properties related to the resource represented
	// by requested_resource_name.
	RequestedResourceProperties *ResourceProperties `json:"requestedResourceProperties,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "Approve") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Approve") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ApprovalRequest) MarshalJSON() ([]byte, error) {
	type NoMethod ApprovalRequest
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ApproveApprovalRequestMessage: Request to approve an ApprovalRequest.
type ApproveApprovalRequestMessage struct {
	// ExpireTime: The expiration time of this approval.
	ExpireTime string `json:"expireTime,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ExpireTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ExpireTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ApproveApprovalRequestMessage) MarshalJSON() ([]byte, error) {
	type NoMethod ApproveApprovalRequestMessage
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ApproveDecision: A decision that has been made to approve access to a
// resource.
type ApproveDecision struct {
	// ApproveTime: The time at which approval was granted.
	ApproveTime string `json:"approveTime,omitempty"`
	// AutoApproved: True when the request has been auto-approved.
	AutoApproved bool `json:"autoApproved,omitempty"`
	// ExpireTime: The time at which the approval expires.
	ExpireTime string `json:"expireTime,omitempty"`
	// InvalidateTime: If set, denotes the timestamp at which the approval is
	// invalidated.
	InvalidateTime string `json:"invalidateTime,omitempty"`
	// SignatureInfo: The signature for the ApprovalRequest and details on how it
	// was signed.
	SignatureInfo *SignatureInfo `json:"signatureInfo,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ApproveTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ApproveTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ApproveDecision) MarshalJSON() ([]byte, error) {
	type NoMethod ApproveDecision
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// AugmentedInfo: This field contains the augmented information of the request.
type AugmentedInfo struct {
	// Command: For command-line tools, the full command-line exactly as entered by
	// the actor without adding any additional characters (such as quotation
	// marks).
	Command string `json:"command,omitempty"`
	// ForceSendFields is a list of field names (e.g. "Command") to unconditionally
	// include in API requests. By default, fields with empty or default values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "Command") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s AugmentedInfo) MarshalJSON() ([]byte, error) {
	type NoMethod AugmentedInfo
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// DismissApprovalRequestMessage: Request to dismiss an approval request.
type DismissApprovalRequestMessage struct {
}

// DismissDecision: A decision that has been made to dismiss an approval
// request.
type DismissDecision struct {
	// DismissTime: The time at which the approval request was dismissed.
	DismissTime string `json:"dismissTime,omitempty"`
	// Implicit: This field will be true if the ApprovalRequest was implicitly
	// dismissed due to inaction by the access approval approvers (the request is
	// not acted on by the approvers before the exiration time).
	Implicit bool `json:"implicit,omitempty"`
	// ForceSendFields is a list of field names (e.g. "DismissTime") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "DismissTime") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s DismissDecision) MarshalJSON() ([]byte, error) {
	type NoMethod DismissDecision
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// Empty: A generic empty message that you can re-use to avoid defining
// duplicated empty messages in your APIs. A typical example is to use it as
// the request or the response type of an API method. For instance: service Foo
// { rpc Bar(google.protobuf.Empty) returns (google.protobuf.Empty); }
type Empty struct {
	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
}

// EnrolledService: Represents the enrollment of a cloud resource into a
// specific service.
type EnrolledService struct {
	// CloudProduct: The product for which Access Approval will be enrolled.
	// Allowed values are listed below (case-sensitive): * all * GA * App Engine *
	// Artifact Registry * BigQuery * Certificate Authority Service * Cloud
	// Bigtable * Cloud Key Management Service * Compute Engine * Cloud Composer *
	// Cloud Dataflow * Cloud Dataproc * Cloud DLP * Cloud EKM * Cloud Firestore *
	// Cloud HSM * Cloud Identity and Access Management * Cloud Logging * Cloud NAT
	// * Cloud Pub/Sub * Cloud Spanner * Cloud SQL * Cloud Storage * Eventarc *
	// Google Kubernetes Engine * Organization Policy Serivice * Persistent Disk *
	// Resource Manager * Secret Manager * Speaker ID Note: These values are
	// supported as input for legacy purposes, but will not be returned from the
	// API. * all * ga-only * appengine.googleapis.com *
	// artifactregistry.googleapis.com * bigquery.googleapis.com *
	// bigtable.googleapis.com * container.googleapis.com * cloudkms.googleapis.com
	// * cloudresourcemanager.googleapis.com * cloudsql.googleapis.com *
	// compute.googleapis.com * dataflow.googleapis.com * dataproc.googleapis.com *
	// dlp.googleapis.com * iam.googleapis.com * logging.googleapis.com *
	// orgpolicy.googleapis.com * pubsub.googleapis.com * spanner.googleapis.com *
	// secretmanager.googleapis.com * speakerid.googleapis.com *
	// storage.googleapis.com Calls to UpdateAccessApprovalSettings using 'all' or
	// any of the XXX.googleapis.com will be translated to the associated product
	// name ('all', 'App Engine', etc.). Note: 'all' will enroll the resource in
	// all products supported at both 'GA' and 'Preview' levels. More information
	// about levels of support is available at
	// https://cloud.google.com/access-approval/docs/supported-services
	CloudProduct string `json:"cloudProduct,omitempty"`
	// EnrollmentLevel: The enrollment level of the service.
	//
	// Possible values:
	//   "ENROLLMENT_LEVEL_UNSPECIFIED" - Default value for proto, shouldn't be
	// used.
	//   "BLOCK_ALL" - Service is enrolled in Access Approval for all requests
	EnrollmentLevel string `json:"enrollmentLevel,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CloudProduct") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CloudProduct") to include in API
	// requests with the JSON null value. By default, fields with empty values are
	// omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s EnrolledService) MarshalJSON() ([]byte, error) {
	type NoMethod EnrolledService
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// InvalidateApprovalRequestMessage: Request to invalidate an existing
// approval.
type InvalidateApprovalRequestMessage struct {
}

// ListApprovalRequestsResponse: Response to listing of ApprovalRequest
// objects.
type ListApprovalRequestsResponse struct {
	// ApprovalRequests: Approval request details.
	ApprovalRequests []*ApprovalRequest `json:"approvalRequests,omitempty"`
	// NextPageToken: Token to retrieve the next page of results, or empty if there
	// are no more.
	NextPageToken string `json:"nextPageToken,omitempty"`

	// ServerResponse contains the HTTP response code and headers from the server.
	googleapi.ServerResponse `json:"-"`
	// ForceSendFields is a list of field names (e.g. "ApprovalRequests") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ApprovalRequests") to include in
	// API requests with the JSON null value. By default, fields with empty values
	// are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ListApprovalRequestsResponse) MarshalJSON() ([]byte, error) {
	type NoMethod ListApprovalRequestsResponse
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// ResourceProperties: The properties associated with the resource of the
// request.
type ResourceProperties struct {
	// ExcludesDescendants: Whether an approval will exclude the descendants of the
	// resource being requested.
	ExcludesDescendants bool `json:"excludesDescendants,omitempty"`
	// ForceSendFields is a list of field names (e.g. "ExcludesDescendants") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "ExcludesDescendants") to include
	// in API requests with the JSON null value. By default, fields with empty
	// values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s ResourceProperties) MarshalJSON() ([]byte, error) {
	type NoMethod ResourceProperties
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

// SignatureInfo: Information about the digital signature of the resource.
type SignatureInfo struct {
	// CustomerKmsKeyVersion: The resource name of the customer CryptoKeyVersion
	// used for signing.
	CustomerKmsKeyVersion string `json:"customerKmsKeyVersion,omitempty"`
	// GoogleKeyAlgorithm: The hashing algorithm used for signature verification.
	// It will only be present in the case of Google managed keys.
	//
	// Possible values:
	//   "CRYPTO_KEY_VERSION_ALGORITHM_UNSPECIFIED" - Not specified.
	//   "GOOGLE_SYMMETRIC_ENCRYPTION" - Creates symmetric encryption keys.
	//   "AES_128_GCM" - AES-GCM (Galois Counter Mode) using 128-bit keys.
	//   "AES_256_GCM" - AES-GCM (Galois Counter Mode) using 256-bit keys.
	//   "AES_128_CBC" - AES-CBC (Cipher Block Chaining Mode) using 128-bit keys.
	//   "AES_256_CBC" - AES-CBC (Cipher Block Chaining Mode) using 256-bit keys.
	//   "AES_128_CTR" - AES-CTR (Counter Mode) using 128-bit keys.
	//   "AES_256_CTR" - AES-CTR (Counter Mode) using 256-bit keys.
	//   "RSA_SIGN_PSS_2048_SHA256" - RSASSA-PSS 2048 bit key with a SHA256 digest.
	//   "RSA_SIGN_PSS_3072_SHA256" - RSASSA-PSS 3072 bit key with a SHA256 digest.
	//   "RSA_SIGN_PSS_4096_SHA256" - RSASSA-PSS 4096 bit key with a SHA256 digest.
	//   "RSA_SIGN_PSS_4096_SHA512" - RSASSA-PSS 4096 bit key with a SHA512 digest.
	//   "RSA_SIGN_PKCS1_2048_SHA256" - RSASSA-PKCS1-v1_5 with a 2048 bit key and a
	// SHA256 digest.
	//   "RSA_SIGN_PKCS1_3072_SHA256" - RSASSA-PKCS1-v1_5 with a 3072 bit key and a
	// SHA256 digest.
	//   "RSA_SIGN_PKCS1_4096_SHA256" - RSASSA-PKCS1-v1_5 with a 4096 bit key and a
	// SHA256 digest.
	//   "RSA_SIGN_PKCS1_4096_SHA512" - RSASSA-PKCS1-v1_5 with a 4096 bit key and a
	// SHA512 digest.
	//   "RSA_SIGN_RAW_PKCS1_2048" - RSASSA-PKCS1-v1_5 signing without encoding,
	// with a 2048 bit key.
	//   "RSA_SIGN_RAW_PKCS1_3072" - RSASSA-PKCS1-v1_5 signing without encoding,
	// with a 3072 bit key.
	//   "RSA_SIGN_RAW_PKCS1_4096" - RSASSA-PKCS1-v1_5 signing without encoding,
	// with a 4096 bit key.
	//   "RSA_DECRYPT_OAEP_2048_SHA256" - RSAES-OAEP 2048 bit key with a SHA256
	// digest.
	//   "RSA_DECRYPT_OAEP_3072_SHA256" - RSAES-OAEP 3072 bit key with a SHA256
	// digest.
	//   "RSA_DECRYPT_OAEP_4096_SHA256" - RSAES-OAEP 4096 bit key with a SHA256
	// digest.
	//   "RSA_DECRYPT_OAEP_4096_SHA512" - RSAES-OAEP 4096 bit key with a SHA512
	// digest.
	//   "RSA_DECRYPT_OAEP_2048_SHA1" - RSAES-OAEP 2048 bit key with a SHA1 digest.
	//   "RSA_DECRYPT_OAEP_3072_SHA1" - RSAES-OAEP 3072 bit key with a SHA1 digest.
	//   "RSA_DECRYPT_OAEP_4096_SHA1" - RSAES-OAEP 4096 bit key with a SHA1 digest.
	//   "EC_SIGN_P256_SHA256" - ECDSA on the NIST P-256 curve with a SHA256
	// digest. Other hash functions can also be used:
	// https://cloud.google.com/kms/docs/create-validate-signatures#ecdsa_support_for_other_hash_algorithms
	//   "EC_SIGN_P384_SHA384" - ECDSA on the NIST P-384 curve with a SHA384
	// digest. Other hash functions can also be used:
	// https://cloud.google.com/kms/docs/create-validate-signatures#ecdsa_support_for_other_hash_algorithms
	//   "EC_SIGN_SECP256K1_SHA256" - ECDSA on the non-NIST secp256k1 curve. This
	// curve is only supported for HSM protection level. Other hash functions can
	// also be used:
	// https://cloud.google.com/kms/docs/create-validate-signatures#ecdsa_support_for_other_hash_algorithms
	//   "EC_SIGN_ED25519" - EdDSA on the Curve25519 in pure mode (taking data as
	// input).
	//   "HMAC_SHA256" - HMAC-SHA256 signing with a 256 bit key.
	//   "HMAC_SHA1" - HMAC-SHA1 signing with a 160 bit key.
	//   "HMAC_SHA384" - HMAC-SHA384 signing with a 384 bit key.
	//   "HMAC_SHA512" - HMAC-SHA512 signing with a 512 bit key.
	//   "HMAC_SHA224" - HMAC-SHA224 signing with a 224 bit key.
	//   "EXTERNAL_SYMMETRIC_ENCRYPTION" - Algorithm representing symmetric
	// encryption by an external key manager.
	//   "PQ_SIGN_ML_DSA_65" - The post-quantum Module-Lattice-Based Digital
	// Signature Algorithm, at security level 3. Randomized version.
	//   "PQ_SIGN_SLH_DSA_SHA2_128S" - The post-quantum stateless hash-based
	// digital signature algorithm, at security level 1. Randomized version.
	GoogleKeyAlgorithm string `json:"googleKeyAlgorithm,omitempty"`
	// GooglePublicKeyPem: The public key for the Google default signing, encoded
	// in PEM format. The signature was created using a private key which may be
	// verified using this public key.
	GooglePublicKeyPem string `json:"googlePublicKeyPem,omitempty"`
	// SerializedApprovalRequest: The ApprovalRequest that is serialized without
	// the SignatureInfo message field. This data is used with the hashing
	// algorithm to generate the digital signature, and it can be used for
	// signature verification.
	SerializedApprovalRequest string `json:"serializedApprovalRequest,omitempty"`
	// Signature: The digital signature.
	Signature string `json:"signature,omitempty"`
	// ForceSendFields is a list of field names (e.g. "CustomerKmsKeyVersion") to
	// unconditionally include in API requests. By default, fields with empty or
	// default values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-ForceSendFields for more
	// details.
	ForceSendFields []string `json:"-"`
	// NullFields is a list of field names (e.g. "CustomerKmsKeyVersion") to
	// include in API requests with the JSON null value. By default, fields with
	// empty values are omitted from API requests. See
	// https://pkg.go.dev/google.golang.org/api#hdr-NullFields for more details.
	NullFields []string `json:"-"`
}

func (s SignatureInfo) MarshalJSON() ([]byte, error) {
	type NoMethod SignatureInfo
	return gensupport.MarshalJSON(NoMethod(s), s.ForceSendFields, s.NullFields)
}

type FoldersDeleteAccessApprovalSettingsCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// DeleteAccessApprovalSettings: Deletes the settings associated with a
// project, folder, or organization. This will have the effect of disabling
// Access Approval for the project, folder, or organization, but only if all
// ancestors also have Access Approval disabled. If Access Approval is enabled
// at a higher level of the hierarchy, then Access Approval will still be
// enabled at this level as the settings are inherited.
//
// - name: Name of the AccessApprovalSettings to delete.
func (r *FoldersService) DeleteAccessApprovalSettings(name string) *FoldersDeleteAccessApprovalSettingsCall {
	c := &FoldersDeleteAccessApprovalSettingsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FoldersDeleteAccessApprovalSettingsCall) Fields(s ...googleapi.Field) *FoldersDeleteAccessApprovalSettingsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FoldersDeleteAccessApprovalSettingsCall) Context(ctx context.Context) *FoldersDeleteAccessApprovalSettingsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FoldersDeleteAccessApprovalSettingsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FoldersDeleteAccessApprovalSettingsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.folders.deleteAccessApprovalSettings", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.folders.deleteAccessApprovalSettings" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *FoldersDeleteAccessApprovalSettingsCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.folders.deleteAccessApprovalSettings", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FoldersGetAccessApprovalSettingsCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GetAccessApprovalSettings: Gets the settings associated with a project,
// folder, or organization.
//
//   - name: The name of the AccessApprovalSettings to retrieve. Format:
//     "{projects|folders|organizations}/{id}/accessApprovalSettings".
func (r *FoldersService) GetAccessApprovalSettings(name string) *FoldersGetAccessApprovalSettingsCall {
	c := &FoldersGetAccessApprovalSettingsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FoldersGetAccessApprovalSettingsCall) Fields(s ...googleapi.Field) *FoldersGetAccessApprovalSettingsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *FoldersGetAccessApprovalSettingsCall) IfNoneMatch(entityTag string) *FoldersGetAccessApprovalSettingsCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FoldersGetAccessApprovalSettingsCall) Context(ctx context.Context) *FoldersGetAccessApprovalSettingsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FoldersGetAccessApprovalSettingsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FoldersGetAccessApprovalSettingsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.folders.getAccessApprovalSettings", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.folders.getAccessApprovalSettings" call.
// Any non-2xx status code is an error. Response headers are in either
// *AccessApprovalSettings.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *FoldersGetAccessApprovalSettingsCall) Do(opts ...googleapi.CallOption) (*AccessApprovalSettings, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AccessApprovalSettings{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.folders.getAccessApprovalSettings", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FoldersGetServiceAccountCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GetServiceAccount: Retrieves the service account that is used by Access
// Approval to access KMS keys for signing approved approval requests.
//
// - name: Name of the AccessApprovalServiceAccount to retrieve.
func (r *FoldersService) GetServiceAccount(name string) *FoldersGetServiceAccountCall {
	c := &FoldersGetServiceAccountCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FoldersGetServiceAccountCall) Fields(s ...googleapi.Field) *FoldersGetServiceAccountCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *FoldersGetServiceAccountCall) IfNoneMatch(entityTag string) *FoldersGetServiceAccountCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FoldersGetServiceAccountCall) Context(ctx context.Context) *FoldersGetServiceAccountCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FoldersGetServiceAccountCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FoldersGetServiceAccountCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.folders.getServiceAccount", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.folders.getServiceAccount" call.
// Any non-2xx status code is an error. Response headers are in either
// *AccessApprovalServiceAccount.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *FoldersGetServiceAccountCall) Do(opts ...googleapi.CallOption) (*AccessApprovalServiceAccount, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AccessApprovalServiceAccount{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.folders.getServiceAccount", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FoldersUpdateAccessApprovalSettingsCall struct {
	s                      *Service
	name                   string
	accessapprovalsettings *AccessApprovalSettings
	urlParams_             gensupport.URLParams
	ctx_                   context.Context
	header_                http.Header
}

// UpdateAccessApprovalSettings: Updates the settings associated with a
// project, folder, or organization. Settings to update are determined by the
// value of field_mask.
//
//   - name: The resource name of the settings. Format is one of: *
//     "projects/{project}/accessApprovalSettings" *
//     "folders/{folder}/accessApprovalSettings" *
//     "organizations/{organization}/accessApprovalSettings".
func (r *FoldersService) UpdateAccessApprovalSettings(name string, accessapprovalsettings *AccessApprovalSettings) *FoldersUpdateAccessApprovalSettingsCall {
	c := &FoldersUpdateAccessApprovalSettingsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.accessapprovalsettings = accessapprovalsettings
	return c
}

// UpdateMask sets the optional parameter "updateMask": The update mask applies
// to the settings. Only the top level fields of AccessApprovalSettings
// (notification_emails & enrolled_services) are supported. For each field, if
// it is included, the currently stored value will be entirely overwritten with
// the value of the field passed in this request. For the `FieldMask`
// definition, see
// https://developers.google.com/protocol-buffers/docs/reference/google.protobuf#fieldmask
// If this field is left unset, only the notification_emails field will be
// updated.
func (c *FoldersUpdateAccessApprovalSettingsCall) UpdateMask(updateMask string) *FoldersUpdateAccessApprovalSettingsCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FoldersUpdateAccessApprovalSettingsCall) Fields(s ...googleapi.Field) *FoldersUpdateAccessApprovalSettingsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FoldersUpdateAccessApprovalSettingsCall) Context(ctx context.Context) *FoldersUpdateAccessApprovalSettingsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FoldersUpdateAccessApprovalSettingsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FoldersUpdateAccessApprovalSettingsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.accessapprovalsettings)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.folders.updateAccessApprovalSettings", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.folders.updateAccessApprovalSettings" call.
// Any non-2xx status code is an error. Response headers are in either
// *AccessApprovalSettings.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *FoldersUpdateAccessApprovalSettingsCall) Do(opts ...googleapi.CallOption) (*AccessApprovalSettings, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AccessApprovalSettings{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.folders.updateAccessApprovalSettings", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FoldersApprovalRequestsApproveCall struct {
	s                             *Service
	name                          string
	approveapprovalrequestmessage *ApproveApprovalRequestMessage
	urlParams_                    gensupport.URLParams
	ctx_                          context.Context
	header_                       http.Header
}

// Approve: Approves a request and returns the updated ApprovalRequest. Returns
// NOT_FOUND if the request does not exist. Returns FAILED_PRECONDITION if the
// request exists but is not in a pending state.
//
// - name: Name of the approval request to approve.
func (r *FoldersApprovalRequestsService) Approve(name string, approveapprovalrequestmessage *ApproveApprovalRequestMessage) *FoldersApprovalRequestsApproveCall {
	c := &FoldersApprovalRequestsApproveCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.approveapprovalrequestmessage = approveapprovalrequestmessage
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FoldersApprovalRequestsApproveCall) Fields(s ...googleapi.Field) *FoldersApprovalRequestsApproveCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FoldersApprovalRequestsApproveCall) Context(ctx context.Context) *FoldersApprovalRequestsApproveCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FoldersApprovalRequestsApproveCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FoldersApprovalRequestsApproveCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.approveapprovalrequestmessage)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}:approve")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.folders.approvalRequests.approve", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.folders.approvalRequests.approve" call.
// Any non-2xx status code is an error. Response headers are in either
// *ApprovalRequest.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *FoldersApprovalRequestsApproveCall) Do(opts ...googleapi.CallOption) (*ApprovalRequest, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ApprovalRequest{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.folders.approvalRequests.approve", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FoldersApprovalRequestsDismissCall struct {
	s                             *Service
	name                          string
	dismissapprovalrequestmessage *DismissApprovalRequestMessage
	urlParams_                    gensupport.URLParams
	ctx_                          context.Context
	header_                       http.Header
}

// Dismiss: Dismisses a request. Returns the updated ApprovalRequest. NOTE:
// This does not deny access to the resource if another request has been made
// and approved. It is equivalent in effect to ignoring the request altogether.
// Returns NOT_FOUND if the request does not exist. Returns FAILED_PRECONDITION
// if the request exists but is not in a pending state.
//
// - name: Name of the ApprovalRequest to dismiss.
func (r *FoldersApprovalRequestsService) Dismiss(name string, dismissapprovalrequestmessage *DismissApprovalRequestMessage) *FoldersApprovalRequestsDismissCall {
	c := &FoldersApprovalRequestsDismissCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.dismissapprovalrequestmessage = dismissapprovalrequestmessage
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FoldersApprovalRequestsDismissCall) Fields(s ...googleapi.Field) *FoldersApprovalRequestsDismissCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FoldersApprovalRequestsDismissCall) Context(ctx context.Context) *FoldersApprovalRequestsDismissCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FoldersApprovalRequestsDismissCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FoldersApprovalRequestsDismissCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.dismissapprovalrequestmessage)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}:dismiss")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.folders.approvalRequests.dismiss", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.folders.approvalRequests.dismiss" call.
// Any non-2xx status code is an error. Response headers are in either
// *ApprovalRequest.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *FoldersApprovalRequestsDismissCall) Do(opts ...googleapi.CallOption) (*ApprovalRequest, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ApprovalRequest{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.folders.approvalRequests.dismiss", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FoldersApprovalRequestsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets an approval request. Returns NOT_FOUND if the request does not
// exist.
//
//   - name: The name of the approval request to retrieve. Format:
//     "{projects|folders|organizations}/{id}/approvalRequests/{approval_request}".
func (r *FoldersApprovalRequestsService) Get(name string) *FoldersApprovalRequestsGetCall {
	c := &FoldersApprovalRequestsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FoldersApprovalRequestsGetCall) Fields(s ...googleapi.Field) *FoldersApprovalRequestsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *FoldersApprovalRequestsGetCall) IfNoneMatch(entityTag string) *FoldersApprovalRequestsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FoldersApprovalRequestsGetCall) Context(ctx context.Context) *FoldersApprovalRequestsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FoldersApprovalRequestsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FoldersApprovalRequestsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.folders.approvalRequests.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.folders.approvalRequests.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *ApprovalRequest.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *FoldersApprovalRequestsGetCall) Do(opts ...googleapi.CallOption) (*ApprovalRequest, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ApprovalRequest{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.folders.approvalRequests.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FoldersApprovalRequestsInvalidateCall struct {
	s                                *Service
	name                             string
	invalidateapprovalrequestmessage *InvalidateApprovalRequestMessage
	urlParams_                       gensupport.URLParams
	ctx_                             context.Context
	header_                          http.Header
}

// Invalidate: Invalidates an existing ApprovalRequest. Returns the updated
// ApprovalRequest. NOTE: This does not deny access to the resource if another
// request has been made and approved. It only invalidates a single approval.
// Returns FAILED_PRECONDITION if the request exists but is not in an approved
// state.
//
// - name: Name of the ApprovalRequest to invalidate.
func (r *FoldersApprovalRequestsService) Invalidate(name string, invalidateapprovalrequestmessage *InvalidateApprovalRequestMessage) *FoldersApprovalRequestsInvalidateCall {
	c := &FoldersApprovalRequestsInvalidateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.invalidateapprovalrequestmessage = invalidateapprovalrequestmessage
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FoldersApprovalRequestsInvalidateCall) Fields(s ...googleapi.Field) *FoldersApprovalRequestsInvalidateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FoldersApprovalRequestsInvalidateCall) Context(ctx context.Context) *FoldersApprovalRequestsInvalidateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FoldersApprovalRequestsInvalidateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FoldersApprovalRequestsInvalidateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.invalidateapprovalrequestmessage)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}:invalidate")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.folders.approvalRequests.invalidate", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.folders.approvalRequests.invalidate" call.
// Any non-2xx status code is an error. Response headers are in either
// *ApprovalRequest.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *FoldersApprovalRequestsInvalidateCall) Do(opts ...googleapi.CallOption) (*ApprovalRequest, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ApprovalRequest{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.folders.approvalRequests.invalidate", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type FoldersApprovalRequestsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists approval requests associated with a project, folder, or
// organization. Approval requests can be filtered by state (pending, active,
// dismissed). The order is reverse chronological.
//
//   - parent: The parent resource. This may be "projects/{project}",
//     "folders/{folder}", or "organizations/{organization}".
func (r *FoldersApprovalRequestsService) List(parent string) *FoldersApprovalRequestsListCall {
	c := &FoldersApprovalRequestsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// Filter sets the optional parameter "filter": A filter on the type of
// approval requests to retrieve. Must be one of the following values: * [not
// set]: Requests that are pending or have active approvals. * ALL: All
// requests. * PENDING: Only pending requests. * ACTIVE: Only active (i.e.
// currently approved) requests. * DISMISSED: Only requests that have been
// dismissed, or requests that are not approved and past expiration. * EXPIRED:
// Only requests that have been approved, and the approval has expired. *
// HISTORY: Active, dismissed and expired requests.
func (c *FoldersApprovalRequestsListCall) Filter(filter string) *FoldersApprovalRequestsListCall {
	c.urlParams_.Set("filter", filter)
	return c
}

// PageSize sets the optional parameter "pageSize": Requested page size.
func (c *FoldersApprovalRequestsListCall) PageSize(pageSize int64) *FoldersApprovalRequestsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A token identifying the
// page of results to return.
func (c *FoldersApprovalRequestsListCall) PageToken(pageToken string) *FoldersApprovalRequestsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *FoldersApprovalRequestsListCall) Fields(s ...googleapi.Field) *FoldersApprovalRequestsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *FoldersApprovalRequestsListCall) IfNoneMatch(entityTag string) *FoldersApprovalRequestsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *FoldersApprovalRequestsListCall) Context(ctx context.Context) *FoldersApprovalRequestsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *FoldersApprovalRequestsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *FoldersApprovalRequestsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/approvalRequests")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.folders.approvalRequests.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.folders.approvalRequests.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListApprovalRequestsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *FoldersApprovalRequestsListCall) Do(opts ...googleapi.CallOption) (*ListApprovalRequestsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListApprovalRequestsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.folders.approvalRequests.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *FoldersApprovalRequestsListCall) Pages(ctx context.Context, f func(*ListApprovalRequestsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type OrganizationsDeleteAccessApprovalSettingsCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// DeleteAccessApprovalSettings: Deletes the settings associated with a
// project, folder, or organization. This will have the effect of disabling
// Access Approval for the project, folder, or organization, but only if all
// ancestors also have Access Approval disabled. If Access Approval is enabled
// at a higher level of the hierarchy, then Access Approval will still be
// enabled at this level as the settings are inherited.
//
// - name: Name of the AccessApprovalSettings to delete.
func (r *OrganizationsService) DeleteAccessApprovalSettings(name string) *OrganizationsDeleteAccessApprovalSettingsCall {
	c := &OrganizationsDeleteAccessApprovalSettingsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *OrganizationsDeleteAccessApprovalSettingsCall) Fields(s ...googleapi.Field) *OrganizationsDeleteAccessApprovalSettingsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *OrganizationsDeleteAccessApprovalSettingsCall) Context(ctx context.Context) *OrganizationsDeleteAccessApprovalSettingsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *OrganizationsDeleteAccessApprovalSettingsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *OrganizationsDeleteAccessApprovalSettingsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.organizations.deleteAccessApprovalSettings", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.organizations.deleteAccessApprovalSettings" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *OrganizationsDeleteAccessApprovalSettingsCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.organizations.deleteAccessApprovalSettings", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type OrganizationsGetAccessApprovalSettingsCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GetAccessApprovalSettings: Gets the settings associated with a project,
// folder, or organization.
//
//   - name: The name of the AccessApprovalSettings to retrieve. Format:
//     "{projects|folders|organizations}/{id}/accessApprovalSettings".
func (r *OrganizationsService) GetAccessApprovalSettings(name string) *OrganizationsGetAccessApprovalSettingsCall {
	c := &OrganizationsGetAccessApprovalSettingsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *OrganizationsGetAccessApprovalSettingsCall) Fields(s ...googleapi.Field) *OrganizationsGetAccessApprovalSettingsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *OrganizationsGetAccessApprovalSettingsCall) IfNoneMatch(entityTag string) *OrganizationsGetAccessApprovalSettingsCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *OrganizationsGetAccessApprovalSettingsCall) Context(ctx context.Context) *OrganizationsGetAccessApprovalSettingsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *OrganizationsGetAccessApprovalSettingsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *OrganizationsGetAccessApprovalSettingsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.organizations.getAccessApprovalSettings", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.organizations.getAccessApprovalSettings" call.
// Any non-2xx status code is an error. Response headers are in either
// *AccessApprovalSettings.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *OrganizationsGetAccessApprovalSettingsCall) Do(opts ...googleapi.CallOption) (*AccessApprovalSettings, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AccessApprovalSettings{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.organizations.getAccessApprovalSettings", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type OrganizationsGetServiceAccountCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GetServiceAccount: Retrieves the service account that is used by Access
// Approval to access KMS keys for signing approved approval requests.
//
// - name: Name of the AccessApprovalServiceAccount to retrieve.
func (r *OrganizationsService) GetServiceAccount(name string) *OrganizationsGetServiceAccountCall {
	c := &OrganizationsGetServiceAccountCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *OrganizationsGetServiceAccountCall) Fields(s ...googleapi.Field) *OrganizationsGetServiceAccountCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *OrganizationsGetServiceAccountCall) IfNoneMatch(entityTag string) *OrganizationsGetServiceAccountCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *OrganizationsGetServiceAccountCall) Context(ctx context.Context) *OrganizationsGetServiceAccountCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *OrganizationsGetServiceAccountCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *OrganizationsGetServiceAccountCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.organizations.getServiceAccount", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.organizations.getServiceAccount" call.
// Any non-2xx status code is an error. Response headers are in either
// *AccessApprovalServiceAccount.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *OrganizationsGetServiceAccountCall) Do(opts ...googleapi.CallOption) (*AccessApprovalServiceAccount, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AccessApprovalServiceAccount{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.organizations.getServiceAccount", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type OrganizationsUpdateAccessApprovalSettingsCall struct {
	s                      *Service
	name                   string
	accessapprovalsettings *AccessApprovalSettings
	urlParams_             gensupport.URLParams
	ctx_                   context.Context
	header_                http.Header
}

// UpdateAccessApprovalSettings: Updates the settings associated with a
// project, folder, or organization. Settings to update are determined by the
// value of field_mask.
//
//   - name: The resource name of the settings. Format is one of: *
//     "projects/{project}/accessApprovalSettings" *
//     "folders/{folder}/accessApprovalSettings" *
//     "organizations/{organization}/accessApprovalSettings".
func (r *OrganizationsService) UpdateAccessApprovalSettings(name string, accessapprovalsettings *AccessApprovalSettings) *OrganizationsUpdateAccessApprovalSettingsCall {
	c := &OrganizationsUpdateAccessApprovalSettingsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.accessapprovalsettings = accessapprovalsettings
	return c
}

// UpdateMask sets the optional parameter "updateMask": The update mask applies
// to the settings. Only the top level fields of AccessApprovalSettings
// (notification_emails & enrolled_services) are supported. For each field, if
// it is included, the currently stored value will be entirely overwritten with
// the value of the field passed in this request. For the `FieldMask`
// definition, see
// https://developers.google.com/protocol-buffers/docs/reference/google.protobuf#fieldmask
// If this field is left unset, only the notification_emails field will be
// updated.
func (c *OrganizationsUpdateAccessApprovalSettingsCall) UpdateMask(updateMask string) *OrganizationsUpdateAccessApprovalSettingsCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *OrganizationsUpdateAccessApprovalSettingsCall) Fields(s ...googleapi.Field) *OrganizationsUpdateAccessApprovalSettingsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *OrganizationsUpdateAccessApprovalSettingsCall) Context(ctx context.Context) *OrganizationsUpdateAccessApprovalSettingsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *OrganizationsUpdateAccessApprovalSettingsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *OrganizationsUpdateAccessApprovalSettingsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.accessapprovalsettings)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.organizations.updateAccessApprovalSettings", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.organizations.updateAccessApprovalSettings" call.
// Any non-2xx status code is an error. Response headers are in either
// *AccessApprovalSettings.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *OrganizationsUpdateAccessApprovalSettingsCall) Do(opts ...googleapi.CallOption) (*AccessApprovalSettings, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AccessApprovalSettings{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.organizations.updateAccessApprovalSettings", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type OrganizationsApprovalRequestsApproveCall struct {
	s                             *Service
	name                          string
	approveapprovalrequestmessage *ApproveApprovalRequestMessage
	urlParams_                    gensupport.URLParams
	ctx_                          context.Context
	header_                       http.Header
}

// Approve: Approves a request and returns the updated ApprovalRequest. Returns
// NOT_FOUND if the request does not exist. Returns FAILED_PRECONDITION if the
// request exists but is not in a pending state.
//
// - name: Name of the approval request to approve.
func (r *OrganizationsApprovalRequestsService) Approve(name string, approveapprovalrequestmessage *ApproveApprovalRequestMessage) *OrganizationsApprovalRequestsApproveCall {
	c := &OrganizationsApprovalRequestsApproveCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.approveapprovalrequestmessage = approveapprovalrequestmessage
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *OrganizationsApprovalRequestsApproveCall) Fields(s ...googleapi.Field) *OrganizationsApprovalRequestsApproveCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *OrganizationsApprovalRequestsApproveCall) Context(ctx context.Context) *OrganizationsApprovalRequestsApproveCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *OrganizationsApprovalRequestsApproveCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *OrganizationsApprovalRequestsApproveCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.approveapprovalrequestmessage)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}:approve")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.organizations.approvalRequests.approve", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.organizations.approvalRequests.approve" call.
// Any non-2xx status code is an error. Response headers are in either
// *ApprovalRequest.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *OrganizationsApprovalRequestsApproveCall) Do(opts ...googleapi.CallOption) (*ApprovalRequest, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ApprovalRequest{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.organizations.approvalRequests.approve", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type OrganizationsApprovalRequestsDismissCall struct {
	s                             *Service
	name                          string
	dismissapprovalrequestmessage *DismissApprovalRequestMessage
	urlParams_                    gensupport.URLParams
	ctx_                          context.Context
	header_                       http.Header
}

// Dismiss: Dismisses a request. Returns the updated ApprovalRequest. NOTE:
// This does not deny access to the resource if another request has been made
// and approved. It is equivalent in effect to ignoring the request altogether.
// Returns NOT_FOUND if the request does not exist. Returns FAILED_PRECONDITION
// if the request exists but is not in a pending state.
//
// - name: Name of the ApprovalRequest to dismiss.
func (r *OrganizationsApprovalRequestsService) Dismiss(name string, dismissapprovalrequestmessage *DismissApprovalRequestMessage) *OrganizationsApprovalRequestsDismissCall {
	c := &OrganizationsApprovalRequestsDismissCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.dismissapprovalrequestmessage = dismissapprovalrequestmessage
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *OrganizationsApprovalRequestsDismissCall) Fields(s ...googleapi.Field) *OrganizationsApprovalRequestsDismissCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *OrganizationsApprovalRequestsDismissCall) Context(ctx context.Context) *OrganizationsApprovalRequestsDismissCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *OrganizationsApprovalRequestsDismissCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *OrganizationsApprovalRequestsDismissCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.dismissapprovalrequestmessage)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}:dismiss")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.organizations.approvalRequests.dismiss", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.organizations.approvalRequests.dismiss" call.
// Any non-2xx status code is an error. Response headers are in either
// *ApprovalRequest.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *OrganizationsApprovalRequestsDismissCall) Do(opts ...googleapi.CallOption) (*ApprovalRequest, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ApprovalRequest{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.organizations.approvalRequests.dismiss", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type OrganizationsApprovalRequestsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets an approval request. Returns NOT_FOUND if the request does not
// exist.
//
//   - name: The name of the approval request to retrieve. Format:
//     "{projects|folders|organizations}/{id}/approvalRequests/{approval_request}".
func (r *OrganizationsApprovalRequestsService) Get(name string) *OrganizationsApprovalRequestsGetCall {
	c := &OrganizationsApprovalRequestsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *OrganizationsApprovalRequestsGetCall) Fields(s ...googleapi.Field) *OrganizationsApprovalRequestsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *OrganizationsApprovalRequestsGetCall) IfNoneMatch(entityTag string) *OrganizationsApprovalRequestsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *OrganizationsApprovalRequestsGetCall) Context(ctx context.Context) *OrganizationsApprovalRequestsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *OrganizationsApprovalRequestsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *OrganizationsApprovalRequestsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.organizations.approvalRequests.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.organizations.approvalRequests.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *ApprovalRequest.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *OrganizationsApprovalRequestsGetCall) Do(opts ...googleapi.CallOption) (*ApprovalRequest, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ApprovalRequest{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.organizations.approvalRequests.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type OrganizationsApprovalRequestsInvalidateCall struct {
	s                                *Service
	name                             string
	invalidateapprovalrequestmessage *InvalidateApprovalRequestMessage
	urlParams_                       gensupport.URLParams
	ctx_                             context.Context
	header_                          http.Header
}

// Invalidate: Invalidates an existing ApprovalRequest. Returns the updated
// ApprovalRequest. NOTE: This does not deny access to the resource if another
// request has been made and approved. It only invalidates a single approval.
// Returns FAILED_PRECONDITION if the request exists but is not in an approved
// state.
//
// - name: Name of the ApprovalRequest to invalidate.
func (r *OrganizationsApprovalRequestsService) Invalidate(name string, invalidateapprovalrequestmessage *InvalidateApprovalRequestMessage) *OrganizationsApprovalRequestsInvalidateCall {
	c := &OrganizationsApprovalRequestsInvalidateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.invalidateapprovalrequestmessage = invalidateapprovalrequestmessage
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *OrganizationsApprovalRequestsInvalidateCall) Fields(s ...googleapi.Field) *OrganizationsApprovalRequestsInvalidateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *OrganizationsApprovalRequestsInvalidateCall) Context(ctx context.Context) *OrganizationsApprovalRequestsInvalidateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *OrganizationsApprovalRequestsInvalidateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *OrganizationsApprovalRequestsInvalidateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.invalidateapprovalrequestmessage)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}:invalidate")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.organizations.approvalRequests.invalidate", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.organizations.approvalRequests.invalidate" call.
// Any non-2xx status code is an error. Response headers are in either
// *ApprovalRequest.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *OrganizationsApprovalRequestsInvalidateCall) Do(opts ...googleapi.CallOption) (*ApprovalRequest, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ApprovalRequest{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.organizations.approvalRequests.invalidate", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type OrganizationsApprovalRequestsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists approval requests associated with a project, folder, or
// organization. Approval requests can be filtered by state (pending, active,
// dismissed). The order is reverse chronological.
//
//   - parent: The parent resource. This may be "projects/{project}",
//     "folders/{folder}", or "organizations/{organization}".
func (r *OrganizationsApprovalRequestsService) List(parent string) *OrganizationsApprovalRequestsListCall {
	c := &OrganizationsApprovalRequestsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// Filter sets the optional parameter "filter": A filter on the type of
// approval requests to retrieve. Must be one of the following values: * [not
// set]: Requests that are pending or have active approvals. * ALL: All
// requests. * PENDING: Only pending requests. * ACTIVE: Only active (i.e.
// currently approved) requests. * DISMISSED: Only requests that have been
// dismissed, or requests that are not approved and past expiration. * EXPIRED:
// Only requests that have been approved, and the approval has expired. *
// HISTORY: Active, dismissed and expired requests.
func (c *OrganizationsApprovalRequestsListCall) Filter(filter string) *OrganizationsApprovalRequestsListCall {
	c.urlParams_.Set("filter", filter)
	return c
}

// PageSize sets the optional parameter "pageSize": Requested page size.
func (c *OrganizationsApprovalRequestsListCall) PageSize(pageSize int64) *OrganizationsApprovalRequestsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A token identifying the
// page of results to return.
func (c *OrganizationsApprovalRequestsListCall) PageToken(pageToken string) *OrganizationsApprovalRequestsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *OrganizationsApprovalRequestsListCall) Fields(s ...googleapi.Field) *OrganizationsApprovalRequestsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *OrganizationsApprovalRequestsListCall) IfNoneMatch(entityTag string) *OrganizationsApprovalRequestsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *OrganizationsApprovalRequestsListCall) Context(ctx context.Context) *OrganizationsApprovalRequestsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *OrganizationsApprovalRequestsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *OrganizationsApprovalRequestsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/approvalRequests")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.organizations.approvalRequests.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.organizations.approvalRequests.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListApprovalRequestsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *OrganizationsApprovalRequestsListCall) Do(opts ...googleapi.CallOption) (*ListApprovalRequestsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListApprovalRequestsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.organizations.approvalRequests.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *OrganizationsApprovalRequestsListCall) Pages(ctx context.Context, f func(*ListApprovalRequestsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}

type ProjectsDeleteAccessApprovalSettingsCall struct {
	s          *Service
	name       string
	urlParams_ gensupport.URLParams
	ctx_       context.Context
	header_    http.Header
}

// DeleteAccessApprovalSettings: Deletes the settings associated with a
// project, folder, or organization. This will have the effect of disabling
// Access Approval for the project, folder, or organization, but only if all
// ancestors also have Access Approval disabled. If Access Approval is enabled
// at a higher level of the hierarchy, then Access Approval will still be
// enabled at this level as the settings are inherited.
//
// - name: Name of the AccessApprovalSettings to delete.
func (r *ProjectsService) DeleteAccessApprovalSettings(name string) *ProjectsDeleteAccessApprovalSettingsCall {
	c := &ProjectsDeleteAccessApprovalSettingsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsDeleteAccessApprovalSettingsCall) Fields(s ...googleapi.Field) *ProjectsDeleteAccessApprovalSettingsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsDeleteAccessApprovalSettingsCall) Context(ctx context.Context) *ProjectsDeleteAccessApprovalSettingsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsDeleteAccessApprovalSettingsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsDeleteAccessApprovalSettingsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("DELETE", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.projects.deleteAccessApprovalSettings", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.projects.deleteAccessApprovalSettings" call.
// Any non-2xx status code is an error. Response headers are in either
// *Empty.ServerResponse.Header or (if a response was returned at all) in
// error.(*googleapi.Error).Header. Use googleapi.IsNotModified to check
// whether the returned error was because http.StatusNotModified was returned.
func (c *ProjectsDeleteAccessApprovalSettingsCall) Do(opts ...googleapi.CallOption) (*Empty, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &Empty{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.projects.deleteAccessApprovalSettings", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsGetAccessApprovalSettingsCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GetAccessApprovalSettings: Gets the settings associated with a project,
// folder, or organization.
//
//   - name: The name of the AccessApprovalSettings to retrieve. Format:
//     "{projects|folders|organizations}/{id}/accessApprovalSettings".
func (r *ProjectsService) GetAccessApprovalSettings(name string) *ProjectsGetAccessApprovalSettingsCall {
	c := &ProjectsGetAccessApprovalSettingsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsGetAccessApprovalSettingsCall) Fields(s ...googleapi.Field) *ProjectsGetAccessApprovalSettingsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ProjectsGetAccessApprovalSettingsCall) IfNoneMatch(entityTag string) *ProjectsGetAccessApprovalSettingsCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsGetAccessApprovalSettingsCall) Context(ctx context.Context) *ProjectsGetAccessApprovalSettingsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsGetAccessApprovalSettingsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsGetAccessApprovalSettingsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.projects.getAccessApprovalSettings", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.projects.getAccessApprovalSettings" call.
// Any non-2xx status code is an error. Response headers are in either
// *AccessApprovalSettings.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *ProjectsGetAccessApprovalSettingsCall) Do(opts ...googleapi.CallOption) (*AccessApprovalSettings, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AccessApprovalSettings{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.projects.getAccessApprovalSettings", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsGetServiceAccountCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// GetServiceAccount: Retrieves the service account that is used by Access
// Approval to access KMS keys for signing approved approval requests.
//
// - name: Name of the AccessApprovalServiceAccount to retrieve.
func (r *ProjectsService) GetServiceAccount(name string) *ProjectsGetServiceAccountCall {
	c := &ProjectsGetServiceAccountCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsGetServiceAccountCall) Fields(s ...googleapi.Field) *ProjectsGetServiceAccountCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ProjectsGetServiceAccountCall) IfNoneMatch(entityTag string) *ProjectsGetServiceAccountCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsGetServiceAccountCall) Context(ctx context.Context) *ProjectsGetServiceAccountCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsGetServiceAccountCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsGetServiceAccountCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.projects.getServiceAccount", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.projects.getServiceAccount" call.
// Any non-2xx status code is an error. Response headers are in either
// *AccessApprovalServiceAccount.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *ProjectsGetServiceAccountCall) Do(opts ...googleapi.CallOption) (*AccessApprovalServiceAccount, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AccessApprovalServiceAccount{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.projects.getServiceAccount", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsUpdateAccessApprovalSettingsCall struct {
	s                      *Service
	name                   string
	accessapprovalsettings *AccessApprovalSettings
	urlParams_             gensupport.URLParams
	ctx_                   context.Context
	header_                http.Header
}

// UpdateAccessApprovalSettings: Updates the settings associated with a
// project, folder, or organization. Settings to update are determined by the
// value of field_mask.
//
//   - name: The resource name of the settings. Format is one of: *
//     "projects/{project}/accessApprovalSettings" *
//     "folders/{folder}/accessApprovalSettings" *
//     "organizations/{organization}/accessApprovalSettings".
func (r *ProjectsService) UpdateAccessApprovalSettings(name string, accessapprovalsettings *AccessApprovalSettings) *ProjectsUpdateAccessApprovalSettingsCall {
	c := &ProjectsUpdateAccessApprovalSettingsCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.accessapprovalsettings = accessapprovalsettings
	return c
}

// UpdateMask sets the optional parameter "updateMask": The update mask applies
// to the settings. Only the top level fields of AccessApprovalSettings
// (notification_emails & enrolled_services) are supported. For each field, if
// it is included, the currently stored value will be entirely overwritten with
// the value of the field passed in this request. For the `FieldMask`
// definition, see
// https://developers.google.com/protocol-buffers/docs/reference/google.protobuf#fieldmask
// If this field is left unset, only the notification_emails field will be
// updated.
func (c *ProjectsUpdateAccessApprovalSettingsCall) UpdateMask(updateMask string) *ProjectsUpdateAccessApprovalSettingsCall {
	c.urlParams_.Set("updateMask", updateMask)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsUpdateAccessApprovalSettingsCall) Fields(s ...googleapi.Field) *ProjectsUpdateAccessApprovalSettingsCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsUpdateAccessApprovalSettingsCall) Context(ctx context.Context) *ProjectsUpdateAccessApprovalSettingsCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsUpdateAccessApprovalSettingsCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsUpdateAccessApprovalSettingsCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.accessapprovalsettings)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("PATCH", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.projects.updateAccessApprovalSettings", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.projects.updateAccessApprovalSettings" call.
// Any non-2xx status code is an error. Response headers are in either
// *AccessApprovalSettings.ServerResponse.Header or (if a response was returned
// at all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *ProjectsUpdateAccessApprovalSettingsCall) Do(opts ...googleapi.CallOption) (*AccessApprovalSettings, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &AccessApprovalSettings{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.projects.updateAccessApprovalSettings", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsApprovalRequestsApproveCall struct {
	s                             *Service
	name                          string
	approveapprovalrequestmessage *ApproveApprovalRequestMessage
	urlParams_                    gensupport.URLParams
	ctx_                          context.Context
	header_                       http.Header
}

// Approve: Approves a request and returns the updated ApprovalRequest. Returns
// NOT_FOUND if the request does not exist. Returns FAILED_PRECONDITION if the
// request exists but is not in a pending state.
//
// - name: Name of the approval request to approve.
func (r *ProjectsApprovalRequestsService) Approve(name string, approveapprovalrequestmessage *ApproveApprovalRequestMessage) *ProjectsApprovalRequestsApproveCall {
	c := &ProjectsApprovalRequestsApproveCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.approveapprovalrequestmessage = approveapprovalrequestmessage
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsApprovalRequestsApproveCall) Fields(s ...googleapi.Field) *ProjectsApprovalRequestsApproveCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsApprovalRequestsApproveCall) Context(ctx context.Context) *ProjectsApprovalRequestsApproveCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsApprovalRequestsApproveCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsApprovalRequestsApproveCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.approveapprovalrequestmessage)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}:approve")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.projects.approvalRequests.approve", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.projects.approvalRequests.approve" call.
// Any non-2xx status code is an error. Response headers are in either
// *ApprovalRequest.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *ProjectsApprovalRequestsApproveCall) Do(opts ...googleapi.CallOption) (*ApprovalRequest, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ApprovalRequest{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.projects.approvalRequests.approve", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsApprovalRequestsDismissCall struct {
	s                             *Service
	name                          string
	dismissapprovalrequestmessage *DismissApprovalRequestMessage
	urlParams_                    gensupport.URLParams
	ctx_                          context.Context
	header_                       http.Header
}

// Dismiss: Dismisses a request. Returns the updated ApprovalRequest. NOTE:
// This does not deny access to the resource if another request has been made
// and approved. It is equivalent in effect to ignoring the request altogether.
// Returns NOT_FOUND if the request does not exist. Returns FAILED_PRECONDITION
// if the request exists but is not in a pending state.
//
// - name: Name of the ApprovalRequest to dismiss.
func (r *ProjectsApprovalRequestsService) Dismiss(name string, dismissapprovalrequestmessage *DismissApprovalRequestMessage) *ProjectsApprovalRequestsDismissCall {
	c := &ProjectsApprovalRequestsDismissCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.dismissapprovalrequestmessage = dismissapprovalrequestmessage
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsApprovalRequestsDismissCall) Fields(s ...googleapi.Field) *ProjectsApprovalRequestsDismissCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsApprovalRequestsDismissCall) Context(ctx context.Context) *ProjectsApprovalRequestsDismissCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsApprovalRequestsDismissCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsApprovalRequestsDismissCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.dismissapprovalrequestmessage)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}:dismiss")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.projects.approvalRequests.dismiss", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.projects.approvalRequests.dismiss" call.
// Any non-2xx status code is an error. Response headers are in either
// *ApprovalRequest.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *ProjectsApprovalRequestsDismissCall) Do(opts ...googleapi.CallOption) (*ApprovalRequest, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ApprovalRequest{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.projects.approvalRequests.dismiss", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsApprovalRequestsGetCall struct {
	s            *Service
	name         string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// Get: Gets an approval request. Returns NOT_FOUND if the request does not
// exist.
//
//   - name: The name of the approval request to retrieve. Format:
//     "{projects|folders|organizations}/{id}/approvalRequests/{approval_request}".
func (r *ProjectsApprovalRequestsService) Get(name string) *ProjectsApprovalRequestsGetCall {
	c := &ProjectsApprovalRequestsGetCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsApprovalRequestsGetCall) Fields(s ...googleapi.Field) *ProjectsApprovalRequestsGetCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ProjectsApprovalRequestsGetCall) IfNoneMatch(entityTag string) *ProjectsApprovalRequestsGetCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsApprovalRequestsGetCall) Context(ctx context.Context) *ProjectsApprovalRequestsGetCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsApprovalRequestsGetCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsApprovalRequestsGetCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.projects.approvalRequests.get", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.projects.approvalRequests.get" call.
// Any non-2xx status code is an error. Response headers are in either
// *ApprovalRequest.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *ProjectsApprovalRequestsGetCall) Do(opts ...googleapi.CallOption) (*ApprovalRequest, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ApprovalRequest{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.projects.approvalRequests.get", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsApprovalRequestsInvalidateCall struct {
	s                                *Service
	name                             string
	invalidateapprovalrequestmessage *InvalidateApprovalRequestMessage
	urlParams_                       gensupport.URLParams
	ctx_                             context.Context
	header_                          http.Header
}

// Invalidate: Invalidates an existing ApprovalRequest. Returns the updated
// ApprovalRequest. NOTE: This does not deny access to the resource if another
// request has been made and approved. It only invalidates a single approval.
// Returns FAILED_PRECONDITION if the request exists but is not in an approved
// state.
//
// - name: Name of the ApprovalRequest to invalidate.
func (r *ProjectsApprovalRequestsService) Invalidate(name string, invalidateapprovalrequestmessage *InvalidateApprovalRequestMessage) *ProjectsApprovalRequestsInvalidateCall {
	c := &ProjectsApprovalRequestsInvalidateCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.name = name
	c.invalidateapprovalrequestmessage = invalidateapprovalrequestmessage
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsApprovalRequestsInvalidateCall) Fields(s ...googleapi.Field) *ProjectsApprovalRequestsInvalidateCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsApprovalRequestsInvalidateCall) Context(ctx context.Context) *ProjectsApprovalRequestsInvalidateCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsApprovalRequestsInvalidateCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsApprovalRequestsInvalidateCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "application/json", c.header_)
	body, err := googleapi.WithoutDataWrapper.JSONBuffer(c.invalidateapprovalrequestmessage)
	if err != nil {
		return nil, err
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+name}:invalidate")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("POST", urls, body)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"name": c.name,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.projects.approvalRequests.invalidate", "request", internallog.HTTPRequest(req, body.Bytes()))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.projects.approvalRequests.invalidate" call.
// Any non-2xx status code is an error. Response headers are in either
// *ApprovalRequest.ServerResponse.Header or (if a response was returned at
// all) in error.(*googleapi.Error).Header. Use googleapi.IsNotModified to
// check whether the returned error was because http.StatusNotModified was
// returned.
func (c *ProjectsApprovalRequestsInvalidateCall) Do(opts ...googleapi.CallOption) (*ApprovalRequest, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ApprovalRequest{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.projects.approvalRequests.invalidate", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

type ProjectsApprovalRequestsListCall struct {
	s            *Service
	parent       string
	urlParams_   gensupport.URLParams
	ifNoneMatch_ string
	ctx_         context.Context
	header_      http.Header
}

// List: Lists approval requests associated with a project, folder, or
// organization. Approval requests can be filtered by state (pending, active,
// dismissed). The order is reverse chronological.
//
//   - parent: The parent resource. This may be "projects/{project}",
//     "folders/{folder}", or "organizations/{organization}".
func (r *ProjectsApprovalRequestsService) List(parent string) *ProjectsApprovalRequestsListCall {
	c := &ProjectsApprovalRequestsListCall{s: r.s, urlParams_: make(gensupport.URLParams)}
	c.parent = parent
	return c
}

// Filter sets the optional parameter "filter": A filter on the type of
// approval requests to retrieve. Must be one of the following values: * [not
// set]: Requests that are pending or have active approvals. * ALL: All
// requests. * PENDING: Only pending requests. * ACTIVE: Only active (i.e.
// currently approved) requests. * DISMISSED: Only requests that have been
// dismissed, or requests that are not approved and past expiration. * EXPIRED:
// Only requests that have been approved, and the approval has expired. *
// HISTORY: Active, dismissed and expired requests.
func (c *ProjectsApprovalRequestsListCall) Filter(filter string) *ProjectsApprovalRequestsListCall {
	c.urlParams_.Set("filter", filter)
	return c
}

// PageSize sets the optional parameter "pageSize": Requested page size.
func (c *ProjectsApprovalRequestsListCall) PageSize(pageSize int64) *ProjectsApprovalRequestsListCall {
	c.urlParams_.Set("pageSize", fmt.Sprint(pageSize))
	return c
}

// PageToken sets the optional parameter "pageToken": A token identifying the
// page of results to return.
func (c *ProjectsApprovalRequestsListCall) PageToken(pageToken string) *ProjectsApprovalRequestsListCall {
	c.urlParams_.Set("pageToken", pageToken)
	return c
}

// Fields allows partial responses to be retrieved. See
// https://developers.google.com/gdata/docs/2.0/basics#PartialResponse for more
// details.
func (c *ProjectsApprovalRequestsListCall) Fields(s ...googleapi.Field) *ProjectsApprovalRequestsListCall {
	c.urlParams_.Set("fields", googleapi.CombineFields(s))
	return c
}

// IfNoneMatch sets an optional parameter which makes the operation fail if the
// object's ETag matches the given value. This is useful for getting updates
// only after the object has changed since the last request.
func (c *ProjectsApprovalRequestsListCall) IfNoneMatch(entityTag string) *ProjectsApprovalRequestsListCall {
	c.ifNoneMatch_ = entityTag
	return c
}

// Context sets the context to be used in this call's Do method.
func (c *ProjectsApprovalRequestsListCall) Context(ctx context.Context) *ProjectsApprovalRequestsListCall {
	c.ctx_ = ctx
	return c
}

// Header returns a http.Header that can be modified by the caller to add
// headers to the request.
func (c *ProjectsApprovalRequestsListCall) Header() http.Header {
	if c.header_ == nil {
		c.header_ = make(http.Header)
	}
	return c.header_
}

func (c *ProjectsApprovalRequestsListCall) doRequest(alt string) (*http.Response, error) {
	reqHeaders := gensupport.SetHeaders(c.s.userAgent(), "", c.header_)
	if c.ifNoneMatch_ != "" {
		reqHeaders.Set("If-None-Match", c.ifNoneMatch_)
	}
	c.urlParams_.Set("alt", alt)
	c.urlParams_.Set("prettyPrint", "false")
	urls := googleapi.ResolveRelative(c.s.BasePath, "v1/{+parent}/approvalRequests")
	urls += "?" + c.urlParams_.Encode()
	req, err := http.NewRequest("GET", urls, nil)
	if err != nil {
		return nil, err
	}
	req.Header = reqHeaders
	googleapi.Expand(req.URL, map[string]string{
		"parent": c.parent,
	})
	c.s.logger.DebugContext(c.ctx_, "api request", "serviceName", apiName, "rpcName", "accessapproval.projects.approvalRequests.list", "request", internallog.HTTPRequest(req, nil))
	return gensupport.SendRequest(c.ctx_, c.s.client, req)
}

// Do executes the "accessapproval.projects.approvalRequests.list" call.
// Any non-2xx status code is an error. Response headers are in either
// *ListApprovalRequestsResponse.ServerResponse.Header or (if a response was
// returned at all) in error.(*googleapi.Error).Header. Use
// googleapi.IsNotModified to check whether the returned error was because
// http.StatusNotModified was returned.
func (c *ProjectsApprovalRequestsListCall) Do(opts ...googleapi.CallOption) (*ListApprovalRequestsResponse, error) {
	gensupport.SetOptions(c.urlParams_, opts...)
	res, err := c.doRequest("json")
	if res != nil && res.StatusCode == http.StatusNotModified {
		if res.Body != nil {
			res.Body.Close()
		}
		return nil, gensupport.WrapError(&googleapi.Error{
			Code:   res.StatusCode,
			Header: res.Header,
		})
	}
	if err != nil {
		return nil, err
	}
	defer googleapi.CloseBody(res)
	if err := googleapi.CheckResponse(res); err != nil {
		return nil, gensupport.WrapError(err)
	}
	ret := &ListApprovalRequestsResponse{
		ServerResponse: googleapi.ServerResponse{
			Header:         res.Header,
			HTTPStatusCode: res.StatusCode,
		},
	}
	target := &ret
	b, err := gensupport.DecodeResponseBytes(target, res)
	if err != nil {
		return nil, err
	}
	c.s.logger.DebugContext(c.ctx_, "api response", "serviceName", apiName, "rpcName", "accessapproval.projects.approvalRequests.list", "response", internallog.HTTPResponse(res, b))
	return ret, nil
}

// Pages invokes f for each page of results.
// A non-nil error returned from f will halt the iteration.
// The provided context supersedes any context provided to the Context method.
func (c *ProjectsApprovalRequestsListCall) Pages(ctx context.Context, f func(*ListApprovalRequestsResponse) error) error {
	c.ctx_ = ctx
	defer c.PageToken(c.urlParams_.Get("pageToken"))
	for {
		x, err := c.Do()
		if err != nil {
			return err
		}
		if err := f(x); err != nil {
			return err
		}
		if x.NextPageToken == "" {
			return nil
		}
		c.PageToken(x.NextPageToken)
	}
}
