package mobile

import (
	"fmt"

	"0xacab.org/leap/bitmask-core/models"
	"0xacab.org/leap/bitmask-core/pkg/bootstrap"
	"0xacab.org/leap/bitmask-core/pkg/introducer"
	"0xacab.org/leap/bitmask-core/pkg/mobile/mobilemodels"
	"0xacab.org/leap/bitmask-core/pkg/storage"
)

type BitmaskMobile struct {
	config  *bootstrap.Config
	storage *storage.Storage
}

func NewBitmaskMobile(url string, countryCode string, store mobilemodels.Store) (*BitmaskMobile, error) {
	config, err := bootstrap.NewConfigFromURL(url)
	if err != nil {
		return nil, fmt.Errorf("failed to create BitmaskMobile struct: %v", err)
	}

	bm := &BitmaskMobile{
		config:  config,
		storage: storage.NewStorageWithStore(store),
	}
	return bm, nil
}

func (bm *BitmaskMobile) SetCountryCode(cc string) {
	bm.config.CountryCode = cc
}

func (bm *BitmaskMobile) SetUseTls(useTls bool) {
	bm.config.UseTLS = useTls
}

func (bm *BitmaskMobile) SetResolveWithDoH(resolveWithDoH bool) {
	bm.config.ResolveWithDoH = resolveWithDoH
}

func (bm *BitmaskMobile) SetIntroducer(introducerUrl string) error {
	introducer, err := introducer.NewIntroducerFromURL(introducerUrl)
	if err != nil {
		return err
	}
	err = introducer.Validate()
	if err != nil {
		return fmt.Errorf("invalid introducer: %v", err)
	}
	_ = bm.storage.AddIntroducer(introducerUrl)
	bm.config.Introducer = introducerUrl
	return nil
}

// GetProvider returns the essential provider infos, including urls to the API and supported API version
func (bm *BitmaskMobile) GetProvider() (*models.ModelsProvider, error) {
	api, err := bootstrap.NewAPI(bm.config)
	if err != nil {
		return nil, err
	}
	provider, err := api.GetProvider()
	if err != nil {
		return nil, err
	}

	return provider, nil
}

// GetService returns service information, including available locations and common tunnel config
func (bm *BitmaskMobile) GetService() (*models.ModelsEIPService, error) {
	api, err := bootstrap.NewAPI(bm.config)
	if err != nil {
		return nil, err
	}
	service, err := api.GetService()
	if err != nil {
		return nil, err
	}

	return service, nil
}

func (bm *BitmaskMobile) GetBestGateway() (string, error) {
	return "", fmt.Errorf("Not implemented yet!")
}

func (bm *BitmaskMobile) GetBestBridge() (string, error) {
	return "", fmt.Errorf("Not implemented yet!")
}

func (bm *BitmaskMobile) GetAllBridges(location, port, transport, transportType string) (*mobilemodels.Bridges, error) {
	api, err := bootstrap.NewAPI(bm.config)
	if err != nil {
		return nil, err
	}

	bridgeParams := &bootstrap.BridgeParams{
		Location:  location,
		Port:      port,
		Transport: transport,
		Type:      transportType,
		CC:        bm.config.CountryCode,
	}
	models, err := api.GetAllBridges(bridgeParams)
	if err != nil {
		return nil, err
	}
	return mobilemodels.NewBridges(models), nil

}

func (bm *BitmaskMobile) GetAllGateways(location, port, transport string) (*mobilemodels.Gateways, error) {
	api, err := bootstrap.NewAPI(bm.config)
	if err != nil {
		return nil, err
	}

	gatewayParams := &bootstrap.GatewayParams{
		Location:  location,
		Port:      port,
		Transport: transport,
		CC:        bm.config.CountryCode,
	}
	models, err := api.GetGateways(gatewayParams)
	if err != nil {
		return nil, err
	}

	return mobilemodels.NewGateways(models), nil
}

func (bm *BitmaskMobile) GetOpenVPNCert() (string, error) {
	api, err := bootstrap.NewAPI(bm.config)
	if err != nil {
		return "", err
	}

	cert, err := api.GetOpenVPNCert()
	if err != nil {
		return "", err
	}

	return cert, nil
}
