#include <unistd.h>
#include <stdlib.h>
#include <pthread.h>
#include <string.h>
#include <errno.h>
#include <lufs/proto.h>
#include <lufs/fs.h>

#include <sys/types.h>
#include <sys/stat.h>

#include "list.h"
#include "gnet.h"
#include "gnetfs.h"
#include "vtree.h"
#include "search.h"
#include "xfer.h"

static struct global_ctx*
init_global(struct list_head *cfg){
    struct global_ctx *glob;
    struct gnet_config gcfg;
    int i, known_peers = 5;
    char buf[32], *sep;
    const char *c;
    struct lufs_fattr root_attr;

    TRACE("creating global context...");

    gnet_set_defaults(&gcfg);

    root_attr.f_nlink = 1;
    root_attr.f_uid = 1;
    root_attr.f_gid = 1;
    root_attr.f_blksize = 512;
    root_attr.f_mtime = root_attr.f_atime = root_attr.f_ctime = time(NULL);
    root_attr.f_mode = S_IFDIR | S_IRUSR | S_IXUSR | S_IRGRP | S_IXGRP | S_IROTH | S_IXOTH;
    root_attr.f_size = 512;
    root_attr.f_blocks = 1;
    

    WARN("FIXME: parse gnet options");

    if((c = lu_opt_getchar(cfg, "GNETFS", "KnownPeers")) && atoi(c))
	known_peers = atoi(c);

    if((c = lu_opt_getchar(cfg, "MOUNT", "KnownPeers")) && atoi(c))
	known_peers = atoi(c);

    TRACE("known_peers: %d", known_peers);



    if(!(glob = malloc(sizeof(struct global_ctx))))
	return NULL;

    memset(glob, 0, sizeof(struct global_ctx));

    if(!(glob->vtree = lu_vtree_create(&root_attr)))
	goto fail_glob;

    
    root_attr.f_mode |= S_IWUSR;
    lu_vtree_add(glob->vtree, "/", SEARCH_DIR + 1, NULL, &root_attr, NULL);

    INIT_LIST_HEAD(&glob->searches);
    pthread_mutex_init(&glob->lock, NULL);

    if(!(glob->gnet = gnet_init(&gcfg))){
	ERROR("could not initialize gnutella engine!");
	goto fail_vtree;
    }

    for(i = 0; i < known_peers; i++){
	sprintf(buf, "Host%d", i + 1);

	if(!(c = lu_opt_getchar(cfg, "MOUNT", buf)))
	    c = lu_opt_getchar(cfg, "GNETFS", buf);

	if(c && (sep = strchr(c, ':')) && (atoi(sep + 1))){
	    TRACE("gnutella host found: %s", c);
	    *sep = 0;
	    gnet_add_peer(glob->gnet, (char*)c, atoi(sep + 1));
	    *sep = ':';
	}
    }

    return glob;

  fail_vtree:
    lu_vtree_destroy(glob->vtree);
  fail_glob:
    free(glob);
    return NULL;
}

static void
destroy_global(struct global_ctx *glob){
    TRACE("destroying global context...");

    gnet_shutdown(glob->gnet);
    lu_vtree_destroy(glob->vtree);

    free(glob);
}

void*
gnetfs_init(struct list_head *cfg, struct dir_cache *cache, struct credentials *cred, void **global_ctx){
    struct local_ctx *ctx;

    TRACE("initializing");

    if(!(ctx = malloc(sizeof(struct local_ctx)))){
	ERROR("could not allocate local context: %s", strerror(errno));
	return NULL;
    }

    memset(ctx, 0, sizeof(struct local_ctx));

    ctx->global = (struct global_ctx**)global_ctx;
    ctx->cfg = cfg;
    INIT_LIST_HEAD(&ctx->opened_files);

    if(*ctx->global){
	TRACE("we already have a global context");

	pthread_mutex_lock(&(*ctx->global)->lock);
	(*ctx->global)->count++;
	pthread_mutex_unlock(&(*ctx->global)->lock);
    }

    return ctx;
}

void
gnetfs_free(void *c){
    struct local_ctx *ctx = c;
    struct global_ctx *glob = *ctx->global;

    TRACE("freeing filesystem");

    if(glob){
	pthread_mutex_lock(&glob->lock);

	if(--glob->count <= 0)
	    destroy_global(glob);
	else
	    pthread_mutex_unlock(&glob->lock);
    }

    free(ctx);    
}

int
gnetfs_mount(void *ctx){
    TRACE("mounting");

    return 1;
}

void
gnetfs_umount(void *ctx){
    TRACE("unmounting");
}

int
gnetfs_stat(void *c, char *name, struct lufs_fattr *fattr){
    struct local_ctx *ctx = c;
    struct global_ctx *glob;
    int res;

    TRACE("stating %s", name);

    if(!(*ctx->global)){
	if(!(*ctx->global = init_global(ctx->cfg))){
	    WARN("could not create global context!");
	    return -1;
	}else{
	    TRACE("global context created");
	    (*ctx->global)->count = 1;
	}
    }

    glob = (*ctx->global);

    pthread_mutex_lock(&glob->lock);

    if((res = lu_vtree_lookup(glob->vtree, name, fattr, NULL, 0, NULL)) < 0){
	WARN("lookup failed!");	
    }

    pthread_mutex_unlock(&glob->lock);

    return res;
}

int
gnetfs_readdir(void *c, char *dir_name, struct directory *dir){
    struct local_ctx *ctx = c;
    struct global_ctx *glob = *ctx->global;
    struct ventry *ve, *e;
    struct list_head *p;
    int res = -1;

    TRACE("reading dir %s", dir_name);

    pthread_mutex_lock(&glob->lock);

    if(*dir_name != '/'){
	TRACE("we need an absolute path here...");
	goto out;
    }

    if(!strcmp(dir_name, "/"))
	ve = &glob->vtree->root;
    else 
	if(!(ve = lu_vtree_search(&glob->vtree->root, dir_name + 1)))
	    goto out;

    TRACE("directory found.");

    list_for_each(p, &ve->children){
	e = list_entry(p, struct ventry, list);

	lu_cache_add2dir(dir, e->name, e->link, &e->fattr);
    }
    
    res = 0;

  out:
    pthread_mutex_unlock(&glob->lock);

    return res;
}


int
gnetfs_mkdir(void *c, char *dir, int mode){
    struct local_ctx *ctx = c;
    struct global_ctx *glob = *ctx->global;
    struct lufs_fattr dattr;
    int res;

    TRACE("mkdir %s", dir);

    if(strncasecmp(dir, SEARCH_DIR, strlen(SEARCH_DIR))){
	WARN("can only create dirs in %s", SEARCH_DIR);
	return -EPERM;
    }

    if(start_search(ctx, dir + strlen(SEARCH_DIR) + 1) < 0){
	WARN("could not start search");
	return -EIO;
    }

    memset(&dattr, 0, sizeof(struct lufs_fattr));
    dattr.f_nlink = 1;
    dattr.f_uid = dattr.f_gid = 1;
    dattr.f_mode = S_IRUSR | S_IRGRP | S_IROTH | S_IXUSR | S_IXGRP | S_IXOTH | S_IFDIR | S_IWUSR;
    dattr.f_mtime = dattr.f_ctime = dattr.f_atime = time(NULL);
    dattr.f_size = 512;
    
    pthread_mutex_lock(&glob->lock);
    
    res = lu_vtree_add(glob->vtree, SEARCH_DIR, dir + strlen(SEARCH_DIR) + 1, NULL, &dattr, NULL);

    pthread_mutex_unlock(&glob->lock);

    return res;
}

int
gnetfs_rmdir(void *c, char *dir){
    struct local_ctx *ctx = c;
    struct global_ctx *glob = *ctx->global;
    struct search *srch;
    struct ventry *ve;

    TRACE("rmdir %s", dir);

    if(strncasecmp(dir, SEARCH_DIR, strlen(SEARCH_DIR))){
	WARN("can only erase dirs in %s", SEARCH_DIR);
	return -EPERM;
    }

    pthread_mutex_lock(&glob->lock);

    if((srch = find_search_by_txt(glob, dir + strlen(SEARCH_DIR) + 1))){
	gnet_stop_search(glob->gnet, srch->id);
	delete_search(srch);
    }

    if((ve = lu_vtree_find(glob->vtree, dir)))
	lu_vtree_delete(ve);
    
    pthread_mutex_unlock(&glob->lock);

    return 0;
}

int
gnetfs_unlink(void *c, char *file){
    struct local_ctx *ctx = c;
    struct global_ctx *glob = *ctx->global;
    struct ventry *ve;
    struct search *srch;
    struct result *res;
    char *s, *sep;

    TRACE("unlink %s", file);

    if(strncasecmp(file, SEARCH_DIR, strlen(SEARCH_DIR))){
	WARN("can only delete files in %s", SEARCH_DIR);
	return -EPERM;
    }

    pthread_mutex_lock(&glob->lock);

    if((ve = lu_vtree_find(glob->vtree, file)))
	lu_vtree_delete(ve);

    s = file + strlen(SEARCH_DIR) + 1;

    if((sep = strchr(s, '/'))){
	*sep++ = 0;

	TRACE("search: %s, result: %s", s, sep);

	if((srch = find_search_by_txt(glob, s)) && (res = find_result_by_name(srch, sep)))
	    delete_result(res);
    }

    pthread_mutex_unlock(&glob->lock);

    return 0;
}

int
gnetfs_open(void *c, char *file, unsigned mode){
    struct local_ctx *ctx = c;
    struct global_ctx *glob = *ctx->global;
    struct ventry *ve;

    TRACE("open %s", file);

    if(!glob)
	return -1;

    pthread_mutex_lock(&glob->lock);
    ve = lu_vtree_find(glob->vtree, file);
    pthread_mutex_unlock(&glob->lock);

    if(!ve)
	return -1;

    return 0;
}

int
gnetfs_release(void *c, char *file){
    TRACE("close %s", file);

    return 0;
}

int
gnetfs_read(void *c, char *file, unsigned long offset, unsigned long count, char *buf){
    TRACE("read %s@%lu, %lubytes", file, offset, count);

    if(strncasecmp(file, SEARCH_DIR, strlen(SEARCH_DIR))){
	WARN("can only read files in %s", SEARCH_DIR);
	return -EPERM;
    }

    return xfer_read((struct local_ctx*)c, file + strlen(SEARCH_DIR) + 1, offset, count, buf);    
}
