// Copyright (c) 2019-2021 Advanced Micro Devices, Inc. All rights reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

#include <iostream>
#include <fstream>
#include <cstdint>
#include <string>
#include <iomanip>

using namespace std;

constexpr const char* preamble
    = R"(// Copyright (c) 2022 Advanced Micro Devices, Inc. All rights reserved.
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL THE
// AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
// OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
// THE SOFTWARE.

// Auto-generated file. Do not edit!
// Generated by tools/sobol32_direction_vector_generator

#include "rocrand/rocrandapi.h"

)";

struct sobol_set
{
    uint32_t d;
    uint32_t s;
    uint32_t a;
    uint32_t m[18];
};

bool read_sobol_set(struct sobol_set * inputs, int N, const std::string name)
{
    std::ifstream infile(name ,std::ios::in);
    if (!infile) {
        std::cout << "Input file containing direction numbers cannot be found!\n";
        return false;
    }
    char buffer[1000];
    infile.getline(buffer,1000,'\n');

    for (int32_t i = 0; i < N; i++) {
        infile >> inputs[i].d >> inputs[i].s >> inputs[i].a;
        for (uint32_t j = 0; j < inputs[i].s; j++)
            infile >> inputs[i].m[j];
    }

    return true;
}

template<typename DirectionVectorType>
void init_direction_vectors(struct sobol_set * inputs, DirectionVectorType* directions, int n_directions, int n)
{
    constexpr uint32_t shift_adjust = (sizeof(DirectionVectorType) * 8) - 1;
    for (int j = 0 ; j < n_directions ; j++)
    {
        directions[j] = 1lu << (shift_adjust - j);
    }
    directions += n_directions;
    for (int i = 1; i < n ; i++) {
        int ix = i - 1;
        int s = inputs[ix].s;
        for (int j = 0; j < s ; j++)
        {
            directions[j] = (DirectionVectorType)inputs[ix].m[j] << (shift_adjust - j);
        }
        for (int j = s; j < n_directions ; j++)
        {
            directions[j] = directions[j - s] ^ (directions[j - s] >> s);
            for (int k = 1; k < s ; k++)
            {
                directions[j] ^= ((((DirectionVectorType)inputs[ix].a >> (s - 1 - k)) & 1) * directions[j - k]);
            }
        }
        directions += n_directions;
    }
}

template<typename DirectionVectorType>
void write_matrices(std::ofstream&       fout,
                    const std::string    name,
                    DirectionVectorType* directions,
                    int32_t              n,
                    int32_t              bits)
{
    const char* dim_sym = (sizeof(DirectionVectorType) == 4) ? "SOBOL32_N" : "SOBOL64_N";
    fout << preamble << "// clang-format off\n"
         << "#define " << dim_sym << " " << n << "\nextern ROCRANDAPI const "
         << ((sizeof(DirectionVectorType) == 4) ? "unsigned int " : "unsigned long long int ")
         << name << "[" << dim_sym << "] =\n{";
    for (int k = 0; k < n; k++)
    {
        if(k % bits == 0)
            fout << std::endl << "    ";
        fout << "0x";
        fout << std::hex << std::setw(sizeof(DirectionVectorType) * 2) << std::setfill('0')
             << directions[k] << ", ";
    }
    fout << "\n};\n";
}

int main(int argc, char const *argv[])
{
    if(argc != 4 || std::string(argv[1]) == "--help")
    {
        std::cout << "Usage:" << std::endl;
        std::cout << "  ./sobol32_direction_vector_generator "
                     "<sets of direction numbers file> "
                     "../../library/include/rocrand/rocrand_sobol32_precomputed.h "
                     "../../library/src/rocrand_sobol32_precomputed.cpp"
                  << std::endl;
        return -1;
    }
    const std::string vector_file(argv[1]);
    uint32_t SOBOL_DIM = 20000;
    uint32_t SOBOL32_N = SOBOL_DIM * 32;
    struct sobol_set * inputs = new struct sobol_set[SOBOL_DIM];
    uint32_t * directions_32 = new uint32_t[SOBOL32_N]();
    bool read = read_sobol_set(inputs, SOBOL_DIM, vector_file);

    if (read)
    {
        init_direction_vectors<uint32_t>(inputs, directions_32, 32, SOBOL_DIM);

        const std::string header_path(argv[2]);
        const std::string source_path(argv[3]);

        std::ofstream header_out(header_path, std::ios_base::out | std::ios_base::trunc);
        std::ofstream source_out(source_path, std::ios_base::out | std::ios_base::trunc);

        constexpr const char* symbol = "rocrand_h_sobol32_direction_vectors";

        header_out << preamble << "#ifndef ROCRAND_SOBOL32_PRECOMPUTED_H_\n"
                   << "#define ROCRAND_SOBOL32_PRECOMPUTED_H_\n"
                   << "\n"
                   << "#ifndef SOBOL_DIM\n"
                   << "    #define SOBOL_DIM " << SOBOL_DIM << "\n"
                   << "#endif\n"
                   << "#define SOBOL32_N " << SOBOL32_N << "\n"
                   << "\n"
                   << "extern ROCRANDAPI const unsigned int " << symbol << "[SOBOL32_N];\n"
                   << "\n"
                   << "#endif // ROCRAND_SOBOL32_PRECOMPUTED_H_\n";

        write_matrices(source_out, symbol, directions_32, SOBOL32_N, 32);
    }

    delete[] inputs;
    delete[] directions_32;

    return 0;
}
