---
myst:
  html_meta:
    keywords: LaTeX, programmation, environnements, environnement, définir des environnements, nouvel environnement, begin...end
---
# Que font `\newenvironment` et `\renewenvironment` ?

La commande `\newenvironment` sert à définir un [environnement](que_sont_les_environnements.md) tandis que `\renewenvironment` sert à redéfinir un environnement préexistant. Ces commandes sont assez semblables aux commandes `\newcommand` et `\renewcommand` de la question "[](/2_programmation/macros/commande_newcommand)" mais présentent un argument obligatoire supplémentaire, le code inséré à la fin de l'environnement.

## Syntaxe des commandes

La syntaxe de `\newenvironment` et de `\renewenvironment` présente trois formes dépendant de la présence d'arguments et de la présence d'arguments optionnels. Ne sont données ici que celles de `\newenvironment`, celles de `\renewenvironment` étant identiques :

```{noedit}
% Forme sans argument
\newenvironment{⟨nom⟩}{⟨déf_début⟩}{⟨déf_fin⟩}

% Forme avec arguments non optionnels
\newenvironment{⟨nom⟩}[⟨nb_args⟩]{⟨déf_début⟩}{⟨déf_fin⟩}

% Forme avec arguments, dont un, le premier, est optionnel
\newenvironment{⟨nom⟩}[⟨nb_args⟩][⟨défaut⟩]{⟨déf_début⟩}{⟨déf_fin⟩}
```

Les paramètres suivent les règles suivantes :
- le `⟨nom⟩` de l'environnement ne commence pas par une contre-oblique, contrairement à ce qui se fait avec `\newcommand` ;
- dans la forme avec arguments, le nombre `⟨nb_args⟩` est compris entre 1 et 9 et donne le nombre d'arguments. Ces éventuels arguments de l'environnement devront être fournis au moment du `\begin{⟨nom⟩}` dans des groupes d'accolades ;
- `⟨défaut⟩` est à valeur par défaut du premier argument ;
- `⟨déf_début⟩` est le code qui sera placé en début d'environnement. Ce code pourra contenir les arguments de l'environnement avec la notation `#` suivi du numéro de l'argument souhaité ;
- `⟨déf_fin⟩` est le code placé en fin d'environnement. Ce code ne peut contenir les arguments d'environnement. 


En voici un exemple d'utilisation (largement perfectible) :

```
\documentclass{article}
  \usepackage{lmodern} % Caractères plus lisibles
  \pagestyle{empty}    % N'affiche pas de numéro de page

  \newenvironment{tirade}[1]{%
    \\ \textsc{#1} : }%
    {(...)}

\begin{document}
Et voici un extrait d'une tirade célèbre de Molière :
\begin{tirade}{Harpagon}
Au voleur, au voleur, à l'assassin, au meurtrier. Justice
\end{tirade}
\end{document}
```

### La gestion des paramètres en fin d'environnement

Les paramètres d'environnement *ne sont pas transmis* au code en fin d'environnement `⟨déf_fin⟩`. Dès lors, le code suivant pose problème :

```{noedit}
\newenvironment{tirade}[2][(...)]%
  {\\ \textsc{#2} : }%
  {#1}
```

Il se produit ici l'erreur suivante (détaillée à la question « [](/2_programmation/erreurs/i/illegal_parameter_number_in_definition_of_command) » :

```text
! Illegal parameter number in definition of \endtirade.
```

 Une solution à ce problème passe par l'utilisation d'une définition de commande placée dans l'argument `⟨déf_début⟩`. Il faut noter que la commande définie à l'intérieur de l'environnement n'a de sens que dans celui-ci et est inconnue à l'extérieur de celui-ci. L'exemple suivant illustre cette méthode ainsi que le cas d'un argument facultatif.

```
\documentclass{article}
  \usepackage{lmodern} % Caractères plus lisibles
  \pagestyle{empty}    % N'affiche pas de numéro de page

  \newenvironment{tirade}[2][(...)]{%
    \newcommand{\fintirade}{#1}% Commande provisoire pour passer l'argument #1
    \\ \textsc{#2} : }%
    {\fintirade}

\begin{document}
Et voici un extrait d'une tirade célèbre de Molière :
\begin{tirade}{Harpagon}
Au voleur, au voleur, à l'assassin, au meurtrier. Justice
\end{tirade}

Et une seconde, tout aussi renommée :
\begin{tirade}[]{Géronte}
Mais que diable allait-il faire à cette galère ?
\end{tirade}
\end{document}
```

:::{sources}
[What are LaTeX environments](faquk:FAQ-whatenv)
:::
