/**
 * Tests that change streams operates correctly when it encounters the oplog events generated by
 * dbCheck, which include an unusual entry whose 'ns' is the target collection namespace despite
 * being an {op: 'c'} event. Exercises the case described in SERVER-85156.
 * @tags: [assumes_against_mongod_not_mongos]
 */
(function() {
"use strict";

load("jstests/libs/collection_drop_recreate.js");  // For assertDropAndRecreateCollection.

const testDb = db.getSiblingDB(jsTestName());
const testColl = assertDropAndRecreateCollection(testDb, jsTestName());
const oplogColl = db.getSiblingDB("local").oplog.rs;

// Create a change stream cursor.
const csCursor = testColl.watch();

// Insert one document before the dbCheck.
assert.commandWorked(testColl.insert({_id: 1}));
assert.soon(() => csCursor.hasNext());
const beforeDbCheckInsert = csCursor.next();
assert.eq(beforeDbCheckInsert.operationType, "insert");

// Now perform the dbCheck.
assert.commandWorked(testDb.runCommand({"dbCheck": testColl.getName()}));

// Confirm that this produces an {op: 'c'} oplog entry on the test collection namespace, and that
// its timestamp is later than the _id:1 insert change event. We must wrap this in an assert.soon
// because the dbCheck command returns success immediately but the relevant oplog entries will be
// written asynchronously as the dbCheck progresses.
assert.soon(() => oplogColl.findOne({
    op: "c",
    ns: testColl.getFullName(),
    "o.dbCheck": testColl.getFullName(),
    ts: {$gt: beforeDbCheckInsert.clusterTime}
}));

// Insert a second document.
assert.commandWorked(testColl.insert({_id: 2}));
assert.soon(() => csCursor.hasNext());

// We should see the second insert in the change stream, having filtered out the dbCheck entry.
const postDbCheckInsert = csCursor.next();
assert.eq(postDbCheckInsert.operationType, "insert", postDbCheckInsert);
})();
