Google Test integration
=======================
RapidCheck comes with support for integrating with Google Test and allows you to easily create properties that are also Google Test test cases.

## Usage ##
This support is available through the `extras/gtest` module. You can either directly add the `extras/gtest/include` directory to your include path or link against the `rapidcheck_gtest` target in your `CMakeLists.txt`. You can then simply `#include <rapidcheck/gtest.h>`. Note that `rapidcheck/gtest.h` needs to be included after `gtest/gtest.h`.

## Reference ##
### `RC_GTEST_PROP(TestCase, TestName, (args...))` ###
Defines a RapidCheck property as a Google Test test. This macro essentially works the same as the Google Test `TEST` macro but it includes an extra parenthesized list of arguments that will be generated by RapidCheck, just like when using `rc::check`:

```C++
RC_GTEST_PROP(MyTestCase,
              copyOfStringIsIdenticalToOriginal,
              (const std::string &str)) {
  const auto strCopy = str;
  RC_ASSERT(strCopy == str);
}
```

The parenthesis around the argument list are required because of how the preprocessor works and can not be omitted. This also means that if you don't want any arguments generated, you need to include an empty set of paranthesis:

```C++
// If you don't have any arguments, you have to have an empty paren
RC_GTEST_PROP(MyTestCase, inRangeValueIsInRange, ()) {
  const auto range = *rc::gen::arbitrary<std::pair<int, int>>();
  const auto x = *rc::gen::inRange(range.first, range.second);
  RC_ASSERT(x >= range.first);
  RC_ASSERT(x < range.second);
}
```

### `RC_GTEST_FIXTURE_PROP(Fixture, Name, (args...))` ###
Analogous to the Google Test `TEST_F` macro, defines a RapidCheck property as a Google Test fixture based test. `Fixture` names the test fixture class. The fixture will be reinstantiated for every test case that is run. Otherwise, this macro works the same as `RC_GTEST_PROP`.

Since this macro is implemented in terms of Google Test's `TEST` macro and Google Test does not allow mixing of `TEST` and `TEST_F` for the same test case, test cases, a property tied to a fixture named `Fixture` will be registered under a test case named `Fixed_RapidCheck`. This is usually not a big issue but is something to be aware of, in particular when filtering Google Test case names from the command line.

## Assertions ##
RapidCheck will treat any exception as a property failure so you should be able to use any assertion mechanism that signals failures as exceptions. However, Google Test assertions are not implemented using exceptions which means that you should avoid them and use RapidCheck assertions such as `RC_ASSERT` in your RapidCheck properties instead.
