#include "common.h"

/* Computes the start of a pixel given the nexessary info. */
#define PX_START(base, row, rowwidth, col, colwidth) ((base) + ((row)*(rowwidth)) + ((col)*(colwidth)))

#define GREY(start) start

#define RGB24_R(start) start+0
#define RGB24_G(start) start+1
#define RGB24_B(start) start+2

#define BGR24_B(start) start+0
#define BGR24_G(start) start+1
#define BGR24_R(start) start+2

/* Global vars */
gint force_palette;

void
compose_yuv_to_rgb(guint8 y, gint32 ug, gint32 ub, gint32 vg, gint32 vr,
                   gint32 *r, gint32 *b, gint32 *g);

/* Functions for converting all FWKS_PALETTE_* to rgb24 */
void
grey_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride);
void
yuv420_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride);
void
yuv420p_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride);
void
yuv411_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride);
void
yuv411p_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride);
void
yuv410p_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride);
void
yuv422_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride);
void
yuyv_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride);
void
uyvy_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride);
void
yuv422p_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride);
void
rgb555_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride);
void
rgb565_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride);
void
rgb24_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride);
void
bgr24_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride);
void
rgb32_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride);
void
bgr32_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride);

/* Convert palettes to the frameworks name space */
FwksPalette
colorspace_palette(FwksPaletteNameSpace ns, gint palette)
{
    FwksPalette fp;

    switch (ns) {
    case V4L:
        switch (palette) {
        case VIDEO_PALETTE_GREY:
            fp = FWKS_PALETTE_GREY;
            break;
        case VIDEO_PALETTE_RGB565:
            fp = FWKS_PALETTE_RGB565;
            break;
        case VIDEO_PALETTE_RGB24:
            fp = FWKS_PALETTE_RGB24;
            break;
        case VIDEO_PALETTE_RGB32:
            fp = FWKS_PALETTE_RGB32;
            break;
        case VIDEO_PALETTE_RGB555:
            fp = FWKS_PALETTE_RGB555;
            break;
        case VIDEO_PALETTE_YUV422:
            fp = FWKS_PALETTE_YUV422;
            break;
        case VIDEO_PALETTE_YUYV:
            fp = FWKS_PALETTE_YUYV;
            break;
        case VIDEO_PALETTE_UYVY:
            fp = FWKS_PALETTE_UYVY;
            break;
        case VIDEO_PALETTE_YUV420:
            fp = FWKS_PALETTE_YUV420;
            break;
        case VIDEO_PALETTE_YUV411:
            fp = FWKS_PALETTE_YUV411;
            break;
        case VIDEO_PALETTE_YUV422P:
            fp = FWKS_PALETTE_YUV422P;
            break;
        case VIDEO_PALETTE_YUV411P:
            fp = FWKS_PALETTE_YUV411P;
            break;
        case VIDEO_PALETTE_YUV420P:
            fp = FWKS_PALETTE_YUV420P;
            break;
        case VIDEO_PALETTE_YUV410P:
            fp = FWKS_PALETTE_YUV410P;
            break;
        default:
            fp = FWKS_PALETTE_GREY;
            break;
        }
        break;
    default:
        fp = FWKS_PALETTE_GREY;
        break;
    }
    return fp;
}

/* Return bytes per pixel in a given palette. */
gfloat
colorspace_bytes_per_pixel(FwksPalette palette)
{
    switch (palette) {
    case FWKS_PALETTE_GREY:
        return 1.0;
    case FWKS_PALETTE_YUV420:
    case FWKS_PALETTE_YUV420P:
    case FWKS_PALETTE_YUV411:
    case FWKS_PALETTE_YUV411P:
        return 1.5;
    case FWKS_PALETTE_YUV410P:
        return 1.125;
    case FWKS_PALETTE_YUV422:
    case FWKS_PALETTE_YUYV:
    case FWKS_PALETTE_UYVY:
    case FWKS_PALETTE_YUV422P:
    case FWKS_PALETTE_RGB555:
    case FWKS_PALETTE_RGB565:
        return 2.0;
    case FWKS_PALETTE_RGB24:
    case FWKS_PALETTE_BGR24:
        return 3.0;
    case FWKS_PALETTE_RGB32:
    case FWKS_PALETTE_BGR32:
        return 4.0;
    default:
        return 1.0;
    }
}

/* Copies the data in rawbuf into rawrgb24, converting rawbuf's format
 * to RGB24.  Rowstride is the length (in bytes) of one line in the
 * image.  It may be equal to the length of the image or
 * not. (rowstride is for compatibility with data in a gdk pixbuf).
 * rawrgb24 must be of sufficient size. */
void
colorspace_rawbuf_to_rgb24(struct RawBuf *rawbuf,
                           guint8 *rawrgb24,
                           gint rowstride)
{
    gint format;
    format = rawbuf->format;
    if (force_palette > -1)
        format = force_palette;

    switch (format) {
    case FWKS_PALETTE_GREY:
        grey_to_rgb24(rawbuf, rawrgb24, rowstride);
        break;

    case FWKS_PALETTE_YUV420:
        yuv420_to_rgb24(rawbuf, rawrgb24, rowstride);
        break;
    case FWKS_PALETTE_YUV420P:
        yuv420p_to_rgb24(rawbuf, rawrgb24, rowstride);
        break;
    case FWKS_PALETTE_YUV411:
        yuv411_to_rgb24(rawbuf, rawrgb24, rowstride);
        break;
    case FWKS_PALETTE_YUV411P:
        yuv411p_to_rgb24(rawbuf, rawrgb24, rowstride);
        break;

    case FWKS_PALETTE_YUV410P:
        yuv410p_to_rgb24(rawbuf, rawrgb24, rowstride);
        break;

    case FWKS_PALETTE_YUV422:
        yuv422_to_rgb24(rawbuf, rawrgb24, rowstride);
        break;
    case FWKS_PALETTE_YUYV:
        yuyv_to_rgb24(rawbuf, rawrgb24, rowstride);
        break;
    case FWKS_PALETTE_UYVY:
        uyvy_to_rgb24(rawbuf, rawrgb24, rowstride);
        break;
    case FWKS_PALETTE_YUV422P:
        yuv422p_to_rgb24(rawbuf, rawrgb24, rowstride);
        break;

    case FWKS_PALETTE_RGB555:
        rgb555_to_rgb24(rawbuf, rawrgb24, rowstride);
        break;
    case FWKS_PALETTE_RGB565:
        rgb565_to_rgb24(rawbuf, rawrgb24, rowstride);
        break;

    case FWKS_PALETTE_RGB24:
        rgb24_to_rgb24(rawbuf, rawrgb24, rowstride);
        break;
    case FWKS_PALETTE_BGR24:
        bgr24_to_rgb24(rawbuf, rawrgb24, rowstride);
        break;

    case FWKS_PALETTE_RGB32:
        rgb32_to_rgb24(rawbuf, rawrgb24, rowstride);
        break;
    case FWKS_PALETTE_BGR32:
        bgr32_to_rgb24(rawbuf, rawrgb24, rowstride);
        break;
    default:
        /* do nothing */
        break;
    }
}



/* ------------------------------------------------------------ */

void
compose_yuv_to_rgb(const guint8 y,
                   const gint32 ug,
                   const gint32 ub,
                   const gint32 vg,
                   const gint32 vr,
                   gint32 *r,
                   gint32 *g,
                   gint32 *b)
{

/* Takes args as specified below, return r g b between 0 and 255.

   The following YUV -> RGB scheme is taken and used under the GPL
   from the file "ccvt_c1.c" from the library "Colour ConVerT:
   conversion routines from YUV to RGB colour space and back" Version:
   0.3 (24-10-2004 [sic?]) (C) 2001-2002 Nemosoft Unv.
   nemosoft@smcc.demon.nl */

    /* These are the values that should get passed in.  We don't
     * calculate ourselves because in most YUV, u and v (and hence ug,
     * ub, vg, and vr) are shared among multiple pixels.  The author
     * of libccvt asserts that these numbers are from Philips
     * documentation.  Since that's the only cam I know of that does
     * YUVP, we'll use them.

                yy = y << 8;
                ug =  88 * (u - 128);
                ub = 454 * (u - 128);
                vg = 183 * (v - 128);
                vr = 359 * (v - 128);

      These are calculated from v4l2 documentation

                yy = 298 * (y -  16);
                ug = 100 * (u - 128);
                ub = 516 * (u - 128);
                vg = 208 * (v - 128);
                vr = 409 * (v - 128);
*/

    gint32 yy;

    yy = y << 8;
    *r = (yy      + vr) >> 8;
    *g = (yy - ug - vg) >> 8;
    *b = (yy + ub     ) >> 8;

    if (*r > 255)     *r = 255;
    else if (*r < 0)  *r = 0;
    if (*g > 255)     *g = 255;
    else if (*g < 0)  *g = 0;
    if (*b > 255)     *b = 255;
    else if (*b < 0)  *b = 0;

    /* Screw it, just do greyscale */
    /* *r = y; *g = y; *b = y; */
}


/* Functions for converting all FWKS_PALETTE_* to rgb24 */

gboolean warned(void)
{
    static gboolean warned = FALSE;
    gboolean state;
    state = warned;
    if (warned == FALSE)
        warned = TRUE;
    return state;
}

#define NOT_IMPLEMENTED(p) if (!warned()) fprintf(stderr, "Unsupported palette: %s\nPlease run \"frameworks --dump > dump\" and mail \"dump\" to pat@polycrystal.org\n", p);
#define EXPERIMENTAL(p) if (!warned()) fprintf(stderr, "Experimental palette: %s\nPlease run \"frameworks --dump > dump\" and mail \"dump\" to pat@polycrystal.org\n", p);

void
grey_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride)
{
    gint width, height;
    gint row, col;
    guint8 *src;

    width = buf->width;
    height = buf->height;
    src = buf->data;

    EXPERIMENTAL("grey");

#define SRC PX_START(src, row, width * 1, col, 1)
#define DEST PX_START(dest, row, rowstride, col, 3)
    for (row = 0; row < height; row++) {  /* for each row */
        for (col = 0; col < width; col++) {  /* for each pixel in the row */
            *(RGB24_R(DEST)) = *(GREY(SRC));
            *(RGB24_G(DEST)) = *(GREY(SRC));
            *(RGB24_B(DEST)) = *(GREY(SRC));
        }
    }
#undef SRC
#undef DEST
}

/* YUV420 is has two planes (as opposed to three planes Y, U, and V). The first
 * plane is Y, the second is UV packed. The UV plane has the same width but
 * half the height of the Y plane. Each U and V correspond to four Y pixels
 * (see YUV420P diagram). */

void
yuv420_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride)
{
    gint height, width;
    gint row, col;
    guint8 *src;

    guint8 *chunk;
    guint8 *uv_start;

    guint8 y0, y1, y2, y3;
    guint8 u, v;
    gint32 r, g, b;
    gint32 ug, ub, vg, vr;

    EXPERIMENTAL("yuv420");
  
    height = buf->height;
    width = buf->width;
    src = buf->data;

    /* Note: disregards rowstride. */

    /* UV plane starts after Y plane which is (height * width *
     * 1byte_per_pixel) large */
    uv_start = src + height * width;

    /* iterate through each 2x2 chunk of Y that all use the same U and V */
    for (row = 0; row < height; row += 2) {
        for (col = 0; col < width; col += 2) {
            chunk = PX_START(src, row, width, col, 1);
            y0 = *(chunk + 0);
            y1 = *(chunk + 1);
            y2 = *(chunk + width + 0);
            y3 = *(chunk + width + 1);

            u = *(PX_START(uv_start, row/2, width, col, 1));
            v = u + 1;

            ug =  88 * (u - 128);
            ub = 454 * (u - 128);
            vg = 183 * (v - 128);
            vr = 359 * (v - 128);

#define PX_00 PX_START(dest, row, rowstride, col, 3)
#define PX_10 PX_START(dest, row, rowstride, col + 1, 3)
#define PX_01 PX_START(dest, row + 1, rowstride, col, 3)
#define PX_11 PX_START(dest, row + 1, rowstride, col + 1, 3)

            compose_yuv_to_rgb(y0, ug, ub, vg, vr, &r, &g, &b);
            *(RGB24_R(PX_00)) = r;
            *(RGB24_G(PX_00)) = g;
            *(RGB24_B(PX_00)) = b;

            compose_yuv_to_rgb(y1, ug, ub, vg, vr, &r, &g, &b);
            *(RGB24_R(PX_10)) = r;
            *(RGB24_G(PX_10)) = g;
            *(RGB24_B(PX_10)) = b;

            compose_yuv_to_rgb(y2, ug, ub, vg, vr, &r, &g, &b);
            *(RGB24_R(PX_01)) = r;
            *(RGB24_G(PX_01)) = g;
            *(RGB24_B(PX_01)) = b;

            compose_yuv_to_rgb(y3, ug, ub, vg, vr, &r, &g, &b);
            *(RGB24_R(PX_11)) = r;
            *(RGB24_G(PX_11)) = g;
            *(RGB24_B(PX_11)) = b;
        }
    }

#undef PX_00
#undef PX_10
#undef PX_01
#undef PX_11
}

/* YUV420P is a YUV planar format.  The Y plane comes first, and is
 * the luminance (~grey) at one byte per pixel.  Next is Cb, then Cr.
 * Each Cb byte goes with four Y bytes like so:

    Y0   Y1
     U0/V0
    Y2   Y3

  The Y data is transmitted simply in order, like a greyscale image,
  not in chunks of 2x2 pixel squares.  Width and height must be even
  integers.
*/
void
yuv420p_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride)
{
    gint height, width;
    gint row, col;
    guint8 *src;

    guint8 *chunk;
    guint8 *u_start;
    guint8 *v_start;

    guint8 y0, y1, y2, y3;
    guint8 u, v;
    gint32 r, g, b;
    gint32 ug, ub, vg, vr;

    height = buf->height;
    width = buf->width;
    src = buf->data;

    u_start = src + height * width;
    v_start = u_start + (height * width) / 4;

    /* iterate through each 2x2 chunk */
    for (row = 0; row < height; row += 2) {
        for (col = 0; col < width; col += 2) {
            chunk = PX_START(src, row, width, col, 1);
            y0 = *(0 + chunk);
            y1 = *(1 + chunk);
            y2 = *(0 + chunk + width);
            y3 = *(1 + chunk + width);

            u = *(PX_START(u_start, row/2, width/2, col/2, 1));
            v = *(PX_START(v_start, row/2, width/2, col/2, 1));

            ug =  88 * (u - 128);
            ub = 454 * (u - 128);
            vg = 183 * (v - 128);
            vr = 359 * (v - 128);

#define PX_00 PX_START(dest, row, rowstride, col, 3)
#define PX_10 PX_START(dest, row, rowstride, col + 1, 3)
#define PX_01 PX_START(dest, row + 1, rowstride, col, 3)
#define PX_11 PX_START(dest, row + 1, rowstride, col + 1, 3)

            compose_yuv_to_rgb(y0, ug, ub, vg, vr, &r, &g, &b);
            *(RGB24_R(PX_00)) = r;
            *(RGB24_G(PX_00)) = g;
            *(RGB24_B(PX_00)) = b;

            compose_yuv_to_rgb(y1, ug, ub, vg, vr, &r, &g, &b);
            *(RGB24_R(PX_10)) = r;
            *(RGB24_G(PX_10)) = g;
            *(RGB24_B(PX_10)) = b;

            compose_yuv_to_rgb(y2, ug, ub, vg, vr, &r, &g, &b);
            *(RGB24_R(PX_01)) = r;
            *(RGB24_G(PX_01)) = g;
            *(RGB24_B(PX_01)) = b;

            compose_yuv_to_rgb(y3, ug, ub, vg, vr, &r, &g, &b);
            *(RGB24_R(PX_11)) = r;
            *(RGB24_G(PX_11)) = g;
            *(RGB24_B(PX_11)) = b;
        }
    }

#undef PX_00
#undef PX_10
#undef PX_01
#undef PX_11
}

void
yuv411_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride)
{
    NOT_IMPLEMENTED("yuv411");
}

void
yuv411p_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride)
{
    NOT_IMPLEMENTED("yuv411p");
}


void
yuv410p_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride)
{
    EXPERIMENTAL("yuv410p");

	gint height, width;
    guint8 *src;

    height = buf->height;
    width = buf->width;
    src = buf->data;

    gint row, col;
    gint chunk, chunk_row, chunk_col;

    const guint8 *y, *u, *v;
    guint8 *r, *g, *b;
	gint32 rr, gg, bb;
    gint32 ug, ub, vg, vr;

    y = src;
    u = src +  height * width;
    v = u   + (height * width) / 16;

    r = dest;
    g = dest + 1;
    b = dest + 2;

    /* Iterate over each Cb/Cr sample */
    for (u = src + height*width, v = u + (height*width)/16, chunk = 0;
            chunk < (height*width)/16; /*u < src + height*width + (height*width)/16; */
            u++, v++, chunk++) {
        /* printf("chunk %d: ", chunk); */
        ug = 88 * (*u - 128);
        ub = 454 * (*u - 128);
        vg = 183 * (*v - 128);
        vr = 359 * (*v - 128);

        /* iterate over each chunk in the Y plane */
        for (chunk_row = 0; chunk_row < 4; chunk_row++) {
            for (chunk_col = 0; chunk_col < 4; chunk_col++) {
                /* printf("%d ", y - src); */
                compose_yuv_to_rgb( *y, ug, ub, vg, vr, &rr, &gg, &bb );
				*r = rr;
				*g = gg;
				*b = bb;

                y++;
                r += 3;
                g += 3;
                b += 3;
            }

            if (chunk_row < 3) {
                y += width - 4; /* wrap around width minus chunk width */
                r += (width - 4) * 3;
                g = r+1;
                b = r+2;
            }
        }
        /* printf("\n"); */

        /* if we are not at the rightmost edge, move y and r to point to the
         * start of the next chunk three rows back */
        if (((chunk+1) * 4) % width != 0) {
            y -= width * 3;

            r -= width * 3 * 3;
            g = r+1;
            b = r+2;
        }
    }
    /* printf("\n"); */
}


void
yuv422_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride)
{
    gint height, width;
    gint row, col;
    guint8 *src;
    guint8 *chunk; /* Start of a chunk of YUYV 4 byte data for two pixels. */

    guint8 y0, y1;
    guint8 u, v;
    gint32 r, g, b;
    gint32 ug, ub, vg, vr;

    height = buf->height;
    width = buf->width;
    src = buf->data;

    /* Iterate through each pixel in the buffer, doing 2 columns at a time. */
    for (row = 0; row < height; row++) {
        for (col = 0; col < width; col += 2) {
            chunk = PX_START(src, row, width*2, col, 2);
            y0 = *(0 + chunk);
            y1 = *(2 + chunk);
            u = *(1 + chunk);
            v = *(3 + chunk);

            ug =  88 * (u - 128);
            ub = 454 * (u - 128);
            vg = 183 * (v - 128);
            vr = 359 * (v - 128);

#define PX_0 PX_START(dest, row, rowstride, col, 3)
#define PX_1 PX_START(dest, row, rowstride, col + 1, 3)

            compose_yuv_to_rgb(y0, ug, ub, vg, vr, &r, &g, &b);
            *(RGB24_R(PX_0)) = r;
            *(RGB24_G(PX_0)) = g;
            *(RGB24_B(PX_0)) = b;

            compose_yuv_to_rgb(y1, ug, ub, vg, vr, &r, &g, &b);
            *(RGB24_R(PX_1)) = r;
            *(RGB24_G(PX_1)) = g;
            *(RGB24_B(PX_1)) = b;
        }
    }
#undef PX_0
#undef PX_1
}


void
yuyv_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride)
{
    NOT_IMPLEMENTED("yuyv");
}


void
uyvy_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride)
{
    NOT_IMPLEMENTED("uyvy");
}


void
yuv422p_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride)
{
    NOT_IMPLEMENTED("yuv422p");
}


void
rgb555_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride)
{
    NOT_IMPLEMENTED("rgb555");
}


void
rgb565_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride)
{
    NOT_IMPLEMENTED("rgb565");
}


void
rgb24_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride)
{
    gint width, height;
    gint row, col;
    guint8 *src;

    width = buf->width;
    height = buf->height;
    src = buf->data;

#define SRC PX_START(src, row, width * 3, col, 3)
#define DEST PX_START(dest, row, rowstride, col, 3)
    for (row = 0; row < height; row++) {  /* for each row */
        for (col = 0; col < width; col++) {  /* for each pixel in the row */
            *(RGB24_R(DEST)) = *(RGB24_R(SRC));
            *(RGB24_G(DEST)) = *(RGB24_G(SRC));
            *(RGB24_B(DEST)) = *(RGB24_B(SRC));
        }
    }
#undef SRC
#undef DEST
}

void
bgr24_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride)
{
    gint width, height;
    gint row, col;
    guint8 *src;

    width = buf->width;
    height = buf->height;
    src = buf->data;

#define SRC PX_START(src, row, width * 3, col, 3)
#define DEST PX_START(dest, row, rowstride, col, 3)
    for (row = 0; row < height; row++) {  /* for each row */
        for (col = 0; col < width; col++) {  /* for each pixel in the row */
            *(RGB24_R(DEST)) = *(BGR24_R(SRC));
            *(RGB24_G(DEST)) = *(BGR24_G(SRC));
            *(RGB24_B(DEST)) = *(BGR24_B(SRC));
        }
    }
#undef SRC
#undef DEST
}

void
rgb32_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride)
{
    NOT_IMPLEMENTED("rgb32");
}


void
bgr32_to_rgb24(const struct RawBuf *buf, guint8 *dest, gint rowstride)
{
    NOT_IMPLEMENTED("bgr32");
}
