//
// Syd: rock-solid application kernel
// src/kernel/truncate.rs: truncate and allocate handlers
//
// Copyright (c) 2023, 2024, 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

use std::os::fd::AsRawFd;

use libseccomp::{ScmpArch, ScmpNotifResp};
use nix::{errno::Errno, fcntl::FallocateFlags};

use crate::{
    cookie::{safe_ftruncate, safe_ftruncate64, safe_truncate, safe_truncate64},
    fs::is_valid_fd,
    hook::{SysArg, UNotifyEventRequest},
    kernel::syscall_path_handler,
    path::XPathBuf,
    scmp_arch_bits, scmp_big_endian,
};

pub(crate) fn sys_truncate(request: UNotifyEventRequest) -> ScmpNotifResp {
    // SAFETY:
    // 1. If first argument is NULL, return EFAULT.
    // 2. If second argument is negative, return EINVAL.
    let req = request.scmpreq;
    let is32 = scmp_arch_bits(req.data.arch) == 32;

    #[allow(clippy::cast_lossless)]
    #[allow(clippy::cast_possible_truncation)]
    #[allow(clippy::cast_possible_wrap)]
    let len = if is32 {
        libc::off_t::from(req.data.args[1] as i32)
    } else {
        req.data.args[1] as libc::off_t
    };

    if len < 0 {
        return request.fail_syscall(Errno::EINVAL);
    } else if req.data.args[0] == 0 {
        return request.fail_syscall(Errno::EFAULT);
    }

    let argv = &[SysArg {
        path: Some(0),
        ..Default::default()
    }];

    syscall_path_handler(request, "truncate", argv, |path_args, request, sandbox| {
        drop(sandbox); // release the read-lock.

        // SAFETY:
        // 1. SysArg has one element.
        // 2. `/` is not a regular file -> EINVAL.
        #[allow(clippy::disallowed_methods)]
        let fd = path_args
            .0
            .as_ref()
            .map(|path| path.dir.as_ref().unwrap().as_raw_fd())
            .ok_or(Errno::EINVAL)?;

        // SAFETY: ftruncate(2) requires write fds.
        // We fallback to `/proc` indirection to avoid opening the file as write.
        //
        // path to fd is open already!
        let mut pfd = XPathBuf::from("/proc/self/fd");
        pfd.push_fd(fd);

        // SAFETY: Record blocking call so it can get invalidated.
        request.cache.add_sys_block(req, false)?;

        let result = safe_truncate(&pfd, len).map(|_| request.return_syscall(0));

        if !matches!(result, Err(Errno::EINTR)) {
            request.cache.del_sys_block(req.id);
        }

        result
    })
}

pub(crate) fn sys_truncate64(request: UNotifyEventRequest) -> ScmpNotifResp {
    // SAFETY:
    // 1. If first argument is NULL, return EFAULT.
    // 2. If second argument is negative, return EINVAL.
    let req = request.scmpreq;
    let is32 = scmp_arch_bits(req.data.arch) == 32;

    // truncate64 needs argument translation on some architectures.
    #[allow(clippy::arithmetic_side_effects)]
    #[allow(clippy::cast_possible_truncation)]
    #[allow(clippy::cast_possible_wrap)]
    let len = if is32 {
        // Align arg_no to the next even number for specific 32-bit architectures.
        let arg_no = match req.data.arch {
            ScmpArch::Arm | ScmpArch::Ppc => 2,
            _ => 1,
        };

        let len_low = req.data.args[arg_no] as u32;
        let len_high = req.data.args[arg_no + 1] as u32;

        if scmp_big_endian(req.data.arch) {
            (libc::off64_t::from(len_low) << 32) | libc::off64_t::from(len_high)
        } else {
            (libc::off64_t::from(len_high) << 32) | libc::off64_t::from(len_low)
        }
    } else {
        // Align arg_no to the next even number for specific 64-bit architectures.
        let arg_no = match req.data.arch {
            ScmpArch::Aarch64 | ScmpArch::Ppc64 | ScmpArch::Ppc64Le => 2,
            _ => 1,
        };

        req.data.args[arg_no] as libc::off64_t
    };

    if len < 0 {
        return request.fail_syscall(Errno::EINVAL);
    } else if req.data.args[0] == 0 {
        return request.fail_syscall(Errno::EFAULT);
    }

    let argv = &[SysArg {
        path: Some(0),
        ..Default::default()
    }];

    syscall_path_handler(
        request,
        "truncate64",
        argv,
        |path_args, request, sandbox| {
            drop(sandbox); // release the read-lock.

            // SAFETY:
            // 1. SysArg has one element.
            // 2. `/` is not a regular file -> EINVAL.
            #[allow(clippy::disallowed_methods)]
            let fd = path_args
                .0
                .as_ref()
                .unwrap()
                .dir
                .as_ref()
                .ok_or(Errno::EINVAL)?;

            // SAFETY: ftruncate64(2) requires write fds.
            // We fallback to `/proc` indirection to avoid opening the file as write.
            //
            // path to fd is open already!
            let mut pfd = XPathBuf::from("/proc/self/fd");
            pfd.push_fd(fd.as_raw_fd());

            // SAFETY: Record blocking call so it can get invalidated.
            request.cache.add_sys_block(req, false)?;

            let result = safe_truncate64(&pfd, len).map(|_| request.return_syscall(0));

            if !matches!(result, Err(Errno::EINTR)) {
                request.cache.del_sys_block(req.id);
            }

            result
        },
    )
}

pub(crate) fn sys_ftruncate(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    // SAFETY: FD-only call:
    // 1. Assert valid fd before other arguments.
    // 2. AT_FDCWD is an invalid fd argument.
    if !is_valid_fd(req.data.args[0]) {
        return request.fail_syscall(Errno::EBADF);
    }

    // SAFETY: If second argument is negative, return EINVAL.
    let is32 = scmp_arch_bits(req.data.arch) == 32;

    #[allow(clippy::cast_lossless)]
    #[allow(clippy::cast_possible_truncation)]
    #[allow(clippy::cast_possible_wrap)]
    let len = if is32 {
        libc::off_t::from(req.data.args[1] as i32)
    } else {
        req.data.args[1] as libc::off_t
    };

    if len < 0 {
        return request.fail_syscall(Errno::EINVAL);
    }

    let argv = &[SysArg {
        dirfd: Some(0),
        ..Default::default()
    }];

    syscall_path_handler(request, "ftruncate", argv, |path_args, request, sandbox| {
        drop(sandbox); // release the read-lock.

        // SAFETY:
        // 1. SysArg has one element.
        // 2. SysArg.path is None asserting dir is Some.
        #[allow(clippy::disallowed_methods)]
        let fd = path_args
            .0
            .as_ref()
            .map(|path| path.dir.as_ref().unwrap())
            .ok_or(Errno::EINVAL)?;

        // SAFETY: Record blocking call so it can get invalidated.
        request.cache.add_sys_block(req, false)?;

        let result = safe_ftruncate(fd, len).map(|_| request.return_syscall(0));

        if !matches!(result, Err(Errno::EINTR)) {
            request.cache.del_sys_block(req.id);
        }

        result
    })
}

pub(crate) fn sys_ftruncate64(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    // SAFETY: FD-only call:
    // 1. Assert valid fd before other arguments.
    // 2. AT_FDCWD is an invalid fd argument.
    if !is_valid_fd(req.data.args[0]) {
        return request.fail_syscall(Errno::EBADF);
    }

    // SAFETY: If second argument is negative, return EINVAL.
    let is32 = scmp_arch_bits(req.data.arch) == 32;

    // truncate64 needs argument translation on some architectures.
    #[allow(clippy::arithmetic_side_effects)]
    #[allow(clippy::cast_possible_truncation)]
    #[allow(clippy::cast_possible_wrap)]
    let len = if is32 {
        // Align arg_no to the next even number for specific 32-bit architectures.
        let arg_no = match req.data.arch {
            ScmpArch::Arm | ScmpArch::Ppc => 2,
            _ => 1,
        };

        let len_low = req.data.args[arg_no] as u32;
        let len_high = req.data.args[arg_no + 1] as u32;

        if scmp_big_endian(req.data.arch) {
            (libc::off64_t::from(len_low) << 32) | libc::off64_t::from(len_high)
        } else {
            (libc::off64_t::from(len_high) << 32) | libc::off64_t::from(len_low)
        }
    } else {
        req.data.args[1] as libc::off64_t
    };

    if len < 0 {
        return request.fail_syscall(Errno::EINVAL);
    }

    let argv = &[SysArg {
        dirfd: Some(0),
        ..Default::default()
    }];

    syscall_path_handler(
        request,
        "ftruncate64",
        argv,
        |path_args, request, sandbox| {
            drop(sandbox); // release the read-lock.

            // SAFETY:
            // 1. SysArg has one element.
            // 2. SysArg.path is None asserting dir is Some.
            #[allow(clippy::disallowed_methods)]
            let fd = path_args.0.as_ref().unwrap().dir.as_ref().unwrap();

            // SAFETY: Record blocking call so it can get invalidated.
            request.cache.add_sys_block(req, false)?;

            let result = safe_ftruncate64(fd, len).map(|_| request.return_syscall(0));

            if !matches!(result, Err(Errno::EINTR)) {
                request.cache.del_sys_block(req.id);
            }

            result
        },
    )
}

pub(crate) fn sys_fallocate(request: UNotifyEventRequest) -> ScmpNotifResp {
    let req = request.scmpreq;

    // SAFETY: FD-only call:
    // 1. Assert valid fd before other arguments.
    // 2. AT_FDCWD is an invalid fd argument.
    if !is_valid_fd(req.data.args[0]) {
        return request.fail_syscall(Errno::EBADF);
    }

    // SAFETY: Reject undefined mode.
    let mode = match req.data.args[1].try_into() {
        Ok(mode) => mode,
        Err(_) => return request.fail_syscall(Errno::EINVAL),
    };

    // SAFETY: Reject invalid mode.
    let mode = match FallocateFlags::from_bits(mode) {
        Some(mode) => mode,
        None => return request.fail_syscall(Errno::EINVAL),
    };

    // SAFETY: EINVAL offset was less than 0, or len was less than or equal to 0.
    let is32 = scmp_arch_bits(req.data.arch) == 32;

    // fallocate needs argument translation for offset argument on some architectures.
    #[allow(clippy::arithmetic_side_effects)]
    #[allow(clippy::cast_possible_truncation)]
    #[allow(clippy::cast_possible_wrap)]
    let off = if is32 {
        let len_low = req.data.args[2] as u32;
        let len_high = req.data.args[3] as u32;

        if scmp_big_endian(req.data.arch) {
            (libc::off64_t::from(len_low) << 32) | libc::off64_t::from(len_high)
        } else {
            (libc::off64_t::from(len_high) << 32) | libc::off64_t::from(len_low)
        }
    } else {
        req.data.args[2] as libc::off64_t
    };

    if off < 0 {
        return request.fail_syscall(Errno::EINVAL);
    }

    // fallocate needs argument translation for length argument on some architectures.
    #[allow(clippy::arithmetic_side_effects)]
    #[allow(clippy::cast_possible_truncation)]
    #[allow(clippy::cast_possible_wrap)]
    let len = if is32 {
        let len_low = req.data.args[4] as u32;
        let len_high = req.data.args[5] as u32;

        if scmp_big_endian(req.data.arch) {
            (libc::off64_t::from(len_low) << 32) | libc::off64_t::from(len_high)
        } else {
            (libc::off64_t::from(len_high) << 32) | libc::off64_t::from(len_low)
        }
    } else {
        req.data.args[3] as libc::off64_t
    };

    if len <= 0 {
        return request.fail_syscall(Errno::EINVAL);
    }

    let argv = &[SysArg {
        dirfd: Some(0),
        ..Default::default()
    }];
    syscall_path_handler(request, "fallocate", argv, |path_args, request, sandbox| {
        drop(sandbox); // release the read-lock.

        // SAFETY:
        // 1. SysArg has one element.
        // 2. SysArg.path is None asserting dir is Some.
        #[allow(clippy::disallowed_methods)]
        let fd = path_args.0.as_ref().unwrap().dir.as_ref().unwrap();

        // SAFETY: Record blocking call so it can get invalidated.
        request.cache.add_sys_block(req, false)?;

        let result =
                // SAFETY: nix does not have an interface for fallocate64.
                Errno::result(unsafe { libc::fallocate64(fd.as_raw_fd(), mode.bits(), off, len) })
                    .map(|_| request.return_syscall(0));

        if !matches!(result, Err(Errno::EINTR)) {
            request.cache.del_sys_block(req.id);
        }

        result
    })
}
