//
// Syd: rock-solid application kernel
// src/landlock_policy.rs: Landlock policy helper library for Syd
//
// Copyright (c) 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

// SAFETY: This module has been liberated from unsafe code!
#![forbid(unsafe_code)]

use std::{
    collections::{HashMap, HashSet},
    ops::RangeInclusive,
};

use nix::{
    errno::Errno,
    fcntl::{open, OFlag},
    sys::stat::Mode,
};

use crate::{
    hash::SydRandomState,
    landlock::{
        Access, AccessFs, AccessNet, CompatLevel, Compatible, CreateRulesetError, NetPort,
        PathBeneath, PathFd, RestrictionStatus, Ruleset, RulesetAttr, RulesetCreatedAttr,
        RulesetError, Scope, ABI,
    },
    path::{XPath, XPathBuf},
};

/// Data structure to store the landlock security policy.
#[derive(Clone, Debug, Default)]
pub struct LandlockPolicy {
    /// Set compatibility level to handle unsupported features
    ///
    /// Defaults to `CompatLevel::BestEffort`.
    pub compat_level: Option<CompatLevel>,
    /// Landlock read pathset
    pub read_pathset: Option<HashSet<XPathBuf, SydRandomState>>,
    /// Landlock write pathset
    pub write_pathset: Option<HashSet<XPathBuf, SydRandomState>>,
    /// Landlock execute pathset
    pub exec_pathset: Option<HashSet<XPathBuf, SydRandomState>>,
    /// Landlock ioctl(2) pathset
    pub ioctl_pathset: Option<HashSet<XPathBuf, SydRandomState>>,
    /// Landlock create pathset
    pub create_pathset: Option<HashSet<XPathBuf, SydRandomState>>,
    /// Landlock delete pathset
    pub delete_pathset: Option<HashSet<XPathBuf, SydRandomState>>,
    /// Landlock rename pathset
    pub rename_pathset: Option<HashSet<XPathBuf, SydRandomState>>,
    /// Landlock symlink pathset
    pub symlink_pathset: Option<HashSet<XPathBuf, SydRandomState>>,
    /// Landlock truncate pathset
    pub truncate_pathset: Option<HashSet<XPathBuf, SydRandomState>>,
    /// Landlock readdir pathset
    pub readdir_pathset: Option<HashSet<XPathBuf, SydRandomState>>,
    /// Landlock mkdir pathset
    pub mkdir_pathset: Option<HashSet<XPathBuf, SydRandomState>>,
    /// Landlock rmdir pathset
    pub rmdir_pathset: Option<HashSet<XPathBuf, SydRandomState>>,
    /// Landlock mkdev pathset
    pub mkdev_pathset: Option<HashSet<XPathBuf, SydRandomState>>,
    /// Landlock mkfifo pathset
    pub mkfifo_pathset: Option<HashSet<XPathBuf, SydRandomState>>,
    /// Landlock make socket pathset
    pub bind_pathset: Option<HashSet<XPathBuf, SydRandomState>>,
    /// Landlock bind portset
    pub bind_portset: Option<HashSet<RangeInclusive<u16>, SydRandomState>>,
    /// Landlock connect portset
    pub conn_portset: Option<HashSet<RangeInclusive<u16>, SydRandomState>>,
    /// Scoped abstract UNIX sockets
    pub scoped_abs: bool,
    /// Scoped UNIX signals
    pub scoped_sig: bool,
}

impl LandlockPolicy {
    /// A helper function to wrap the operations and reduce duplication.
    #[allow(clippy::arithmetic_side_effects)]
    #[allow(clippy::cognitive_complexity)]
    #[allow(clippy::disallowed_methods)]
    pub fn restrict_self(&self, abi: ABI) -> Result<RestrictionStatus, RulesetError> {
        // from_all includes IoctlDev of ABI >= 5 as necessary.
        let mut ruleset = Ruleset::default().handle_access(AccessFs::from_all(abi))?;
        let ruleset_ref = &mut ruleset;

        // Set compatibility level as necessary.
        // For `None` case, use landlock crate default
        // which is `CompatLevel::BestEffort`.
        let level = if let Some(compat_level) = self.compat_level {
            ruleset_ref.set_compatibility(compat_level);
            compat_level
        } else {
            CompatLevel::BestEffort
        };

        // Network is ABI >= 4.
        let mut network_rules_bind: HashSet<u16, SydRandomState> = HashSet::default();
        if let Some(ref port_set) = self.bind_portset {
            for port_range in port_set {
                for port in port_range.clone() {
                    network_rules_bind.insert(port);
                }
            }
        }
        if network_rules_bind.len() <= usize::from(u16::MAX) + 1 {
            ruleset_ref.handle_access(AccessNet::BindTcp)?;
        } else {
            // SAFETY: All ports are allowed, do not handle the access right,
            // rather than allowing each and every port.
            network_rules_bind.clear();
        }

        let mut network_rules_conn: HashSet<u16, SydRandomState> = HashSet::default();
        if let Some(ref port_set) = self.conn_portset {
            for port_range in port_set {
                for port in port_range.clone() {
                    network_rules_conn.insert(port);
                }
            }
        }
        if network_rules_conn.len() <= usize::from(u16::MAX) + 1 {
            ruleset_ref.handle_access(AccessNet::ConnectTcp)?;
        } else {
            // SAFETY: All ports are allowed, do not handle the access right,
            // rather than allowing each and every port.
            network_rules_conn.clear();
        }

        // Scopes are ABI >= 6.
        if self.scoped_abs {
            ruleset_ref.scope(Scope::AbstractUnixSocket)?;
        }
        if self.scoped_sig {
            ruleset_ref.scope(Scope::Signal)?;
        }

        // Merge path rules based on access rights.
        //
        // Step 1: Accumulate all paths in a single set.
        let mut all_pathset: HashSet<XPathBuf, SydRandomState> = HashSet::default();
        if let Some(ref pathset) = self.read_pathset {
            all_pathset.extend(pathset.iter().cloned());
        }
        if let Some(ref pathset) = self.write_pathset {
            all_pathset.extend(pathset.iter().cloned());
        }
        if let Some(ref pathset) = self.exec_pathset {
            all_pathset.extend(pathset.iter().cloned());
        }
        if let Some(ref pathset) = self.ioctl_pathset {
            all_pathset.extend(pathset.iter().cloned());
        }
        if let Some(ref pathset) = self.create_pathset {
            all_pathset.extend(pathset.iter().cloned());
        }
        if let Some(ref pathset) = self.delete_pathset {
            all_pathset.extend(pathset.iter().cloned());
        }
        if let Some(ref pathset) = self.rename_pathset {
            all_pathset.extend(pathset.iter().cloned());
        }
        if let Some(ref pathset) = self.symlink_pathset {
            all_pathset.extend(pathset.iter().cloned());
        }
        if let Some(ref pathset) = self.truncate_pathset {
            all_pathset.extend(pathset.iter().cloned());
        }
        if let Some(ref pathset) = self.readdir_pathset {
            all_pathset.extend(pathset.iter().cloned());
        }
        if let Some(ref pathset) = self.mkdir_pathset {
            all_pathset.extend(pathset.iter().cloned());
        }
        if let Some(ref pathset) = self.rmdir_pathset {
            all_pathset.extend(pathset.iter().cloned());
        }
        if let Some(ref pathset) = self.mkdev_pathset {
            all_pathset.extend(pathset.iter().cloned());
        }
        if let Some(ref pathset) = self.mkfifo_pathset {
            all_pathset.extend(pathset.iter().cloned());
        }
        if let Some(ref pathset) = self.bind_pathset {
            all_pathset.extend(pathset.iter().cloned());
        }

        // Step 2: Accumulate access rights using the `all_pathset`.
        let mut acl: HashMap<AccessFs, Vec<XPathBuf>, SydRandomState> = HashMap::default();
        for path in all_pathset {
            let mut access = AccessFs::EMPTY;

            if self
                .read_pathset
                .as_ref()
                .map(|set| set.contains(&path))
                .unwrap_or(false)
            {
                access |= AccessFs::ReadFile;
            }
            if self
                .write_pathset
                .as_ref()
                .map(|set| set.contains(&path))
                .unwrap_or(false)
            {
                access |= AccessFs::WriteFile;
            }
            if self
                .exec_pathset
                .as_ref()
                .map(|set| set.contains(&path))
                .unwrap_or(false)
            {
                access |= AccessFs::Execute;
            }
            if self
                .ioctl_pathset
                .as_ref()
                .map(|set| set.contains(&path))
                .unwrap_or(false)
            {
                access |= AccessFs::IoctlDev;
            }
            if self
                .create_pathset
                .as_ref()
                .map(|set| set.contains(&path))
                .unwrap_or(false)
            {
                access |= AccessFs::MakeReg;
            }
            if self
                .delete_pathset
                .as_ref()
                .map(|set| set.contains(&path))
                .unwrap_or(false)
            {
                access |= AccessFs::RemoveFile;
            }
            if self
                .rename_pathset
                .as_ref()
                .map(|set| set.contains(&path))
                .unwrap_or(false)
            {
                access |= AccessFs::Refer;
            }
            if self
                .symlink_pathset
                .as_ref()
                .map(|set| set.contains(&path))
                .unwrap_or(false)
            {
                access |= AccessFs::MakeSym;
            }
            if self
                .truncate_pathset
                .as_ref()
                .map(|set| set.contains(&path))
                .unwrap_or(false)
            {
                access |= AccessFs::Truncate;
            }
            if self
                .readdir_pathset
                .as_ref()
                .map(|set| set.contains(&path))
                .unwrap_or(false)
            {
                access |= AccessFs::ReadDir;
            }
            if self
                .mkdir_pathset
                .as_ref()
                .map(|set| set.contains(&path))
                .unwrap_or(false)
            {
                access |= AccessFs::MakeDir;
            }
            if self
                .rmdir_pathset
                .as_ref()
                .map(|set| set.contains(&path))
                .unwrap_or(false)
            {
                access |= AccessFs::RemoveDir;
            }
            if self
                .mkdev_pathset
                .as_ref()
                .map(|set| set.contains(&path))
                .unwrap_or(false)
            {
                access |= AccessFs::MakeChar;
            }
            if self
                .mkfifo_pathset
                .as_ref()
                .map(|set| set.contains(&path))
                .unwrap_or(false)
            {
                access |= AccessFs::MakeFifo;
            }
            if self
                .bind_pathset
                .as_ref()
                .map(|set| set.contains(&path))
                .unwrap_or(false)
            {
                access |= AccessFs::MakeSock;
            }

            if access.is_empty() {
                continue;
            }

            acl.entry(access).or_default().push(path);
        }

        // Step 3: Create ruleset and enter (access, path-set) pairs.
        let mut ruleset = ruleset.create()?;
        for (access, paths) in &acl {
            ruleset = ruleset.add_rules(landlock_path_beneath_rules(level, paths, *access))?;
        }

        ruleset
            .add_rules(
                network_rules_bind.into_iter().map(|port| {
                    Ok::<NetPort, RulesetError>(NetPort::new(port, AccessNet::BindTcp))
                }),
            )?
            .add_rules(network_rules_conn.into_iter().map(|port| {
                Ok::<NetPort, RulesetError>(NetPort::new(port, AccessNet::ConnectTcp))
            }))?
            .restrict_self()
    }
}

// syd::landlock::path_beneath_rules tailored for Syd use-case.
#[allow(clippy::cognitive_complexity)]
#[allow(clippy::disallowed_methods)]
fn landlock_path_beneath_rules<I, P>(
    level: CompatLevel,
    paths: I,
    access: AccessFs,
) -> impl Iterator<Item = Result<PathBeneath<PathFd>, RulesetError>>
where
    I: IntoIterator<Item = P>,
    P: AsRef<XPath>,
{
    let compat_level = match level {
        CompatLevel::HardRequirement => "hard-requirement",
        CompatLevel::SoftRequirement => "soft-requirement",
        CompatLevel::BestEffort => "best-effort",
    };

    paths.into_iter().filter_map(move |p| {
        let p = p.as_ref();
        #[allow(clippy::cast_possible_truncation)]
        match open(p, OFlag::O_PATH | OFlag::O_CLOEXEC, Mode::empty()) {
            Ok(fd) => Some(Ok(PathBeneath::new(PathFd { fd }, access))),
            Err(errno @ Errno::ENOENT) if level == CompatLevel::BestEffort => {
                crate::info!("ctx": "init", "op": "landlock_create_ruleset",
                    "path": p, "access": access,
                    "cmp": compat_level, "err": errno as i32,
                    "msg": format!("open path `{p}' for Landlock failed: {errno}"));
                None
            }
            Err(errno) => {
                crate::error!("ctx": "init", "op": "landlock_create_ruleset",
                    "path": p, "access": access,
                    "cmp": compat_level, "err": errno as i32,
                    "msg": format!("open path `{p}' for Landlock failed: {errno}"),
                    "tip": "set `default/lock:warn' to ignore file-not-found errors for Landlock");
                Some(Err(RulesetError::CreateRuleset(
                    CreateRulesetError::CreateRulesetCall {
                        source: errno.into(),
                    },
                )))
            }
        }
    })
}
