/* ScummVM - Graphic Adventure Engine
 *
 * ScummVM is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 */

#ifndef NANCY11DATA_H
#define NANCY11DATA_H

#include "types.h"

const GameConstants _nancy11Constants ={
	70,												// numItems
	1251,											// numEventFlags
	{	0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10,			// genericEventFlags
		11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
		21, 22, 23, 24, 25, 26, 27, 28, 29, 30 },
	20,												// numCursorTypes
	4000,											// logoEndAfter
	32												// wonGameFlagID
};

const Common::Array<Common::Array<ConditionalDialogue>> _nancy11ConditionalDialogue = {
{	// Mrs. Leticia Drake, 18 responses
	{	0, 1050, "NMD50",
		{ { kEv, 944, true }, { kEv, 538, false } } },
	{	0, 1055, "NMD55",
		{ { kEv, 968, true }, { kEv, 838, true }, { kEv, 535, false } } },
	{	0, 1058, "NMD58",
		{ { kEv, 576, true }, { kEv, 544, false } } },
	{	0, 1060, "NMD60",
		{ { kEv, 395, true }, { kEv, 528, false } } },
	{	0, 1061, "NDC08d",
		{ { kEv, 519, false } } },
	{	0, 1062, "NMD62",
		{ { kEv, 529, false } } },
	{	0, 1063, "NMD63",
		{ { kEv, 528, true }, { kEv, 515, false } } },
	{	0, 1065, "NMD65",
		{ { kEv, 954, true }, { kEv, 545, false } } },
	{	0, 1067, "NMD67",
		{ { kEv, 569, true }, { kEv, 525, false } } },
	{	0, 1068, "NMD68",
		{ { kEv, 467, true }, { kEv, 525, true }, { kEv, 526, false } } },
	{	0, 1072, "NMD72",
		{ { kEv, 262, true }, { kEv, 523, false } } },
	{	0, 1075, "NMD75",
		{ { kEv, 865, true }, { kEv, 349, false }, { kEv, 522, false } } },
	{	0, 1076, "NMD76",
		{ { kEv, 941, true }, { kEv, 536, false } } },
	{	0, 1077, "NMD77",
		{ { kEv, 857, true }, { kEv, 518, false } } },
	{	0, 1078, "NMD78",
		{ { kEv, 1069, true }, { kEv, 985, false }, { kEv, 385, false }, { kEv, 524, false } } },
	{	0, 1082, "NMD82",
		{ { kEv, 349, true }, { kEv, 543, false } } },
	{	0, 1086, "NMD86",
		{ { kEv, 535, true }, { kEv, 1080, true }, { kEv, 1002, false }, { kEv, 530, false } } },
	{	0, 1088, "NMD88",
		{ { kEv, 1002, true }, { kEv, 541, false } } },
},
{	// Jane Penvellyn, 22 responses
	{	0, 1245, "NJP45",
		{ { kEv, 366, false }, { kIn, 21, true } } },
	{	0, 1250, "NJP50",
		{ { kEv, 494, true }, { kEv, 404, false } } },
	{	0, 1257, "NJP57",
		{ { kEv, 435, true }, { kEv, 405, false } } },
	{	0, 1262, "NJP62",
		{ { kEv, 968, true }, { kEv, 838, true }, { kEv, 535, false }, { kEv, 333, false } } },
	{	0, 1263, "NJP63",
		{ { kEv, 405, true }, { kEv, 1058, true }, { kIn, 7, false } } },
	{	0, 1265, "NJP65",
		{ { kEv, 332, false } } },
	{	0, 1268, "NJP67",
		{ { kEv, 567, true } } },
	{	0, 1269, "NJP69",
		{ { kEv, 405, true }, { kEv, 971, true }, { kEv, 997, false }, { kEv, 329, false } } },
	{	0, 1270, "NJP70",
		{ { kEv, 704, true }, { kEv, 430, false } } },
	{	0, 1271, "NJP71",
		{ { kEv, 905, true }, { kEv, 424, false } } },
	{	0, 1274, "NJP74",
		{ { kEv, 424, true }, { kEv, 980, false } } },
	{	0, 1275, "NJP75",
		{ { kEv, 1069, true }, { kEv, 985, false }, { kEv, 385, false } } },
	{	0, 1297, "NJP97",
		{ { kEv, 419, false }, { kEv, 1069, true }, { kEv, 385, true }, { kEv, 985, false } } },
	{	0, 1277, "NJP77",
		{ { kEv, 823, true }, { kEv, 958, false }, { kEv, 316, false } } },
	{	0, 1278, "NJP78",
		{ { kEv, 467, true }, { kEv, 774, false }, { kEv, 387, false } } },
	{	0, 1279, "NJP79",
		{ { kEv, 383, false } } },
	{	0, 1280, "NJP80",
		{ { kEv, 865, true }, { kEv, 349, false } } },
	{	0, 1282, "NJP82",
		{ { kEv, 865, true }, { kEv, 349, true }, { kIn, 5, true }, { kIn, 13, false } } },
	{	0, 1283, "NJP83",
		{ { kEv, 944, true }, { kEv, 414, false } } },
	{	0, 1288, "NJP88",
		{ { kEv, 863, true }, { kEv, 492, true }, { kEv, 400, false } } },
	{	0, 5055, "NJJ55",
		{ { kEv, 242, true }, { kEv, 399, false } } },
	{	0, 5056, "NJJ56",
		{ { kEv, 922, true }, { kEv, 694, false }, { kEv, 403, false } } },
},
{	// Nigel Mookerjee, 22 responses
	{	0, 1306, "NNM06",
		{ { kEv, 573, true }, { kEv, 701, false } } },
	{	0, 1313, "NNM13",
		{ { kEv, 874, true }, { kEv, 699, false } } },
	{	0, 1315, "NNM15",
		{ { kEv, 573, true }, { kEv, 704, false } } },
	{	0, 1317, "NNM17",
		{ { kEv, 701, true }, { kEv, 677, false } } },
	{	0, 1350, "NNM50",
		{ { kEv, 573, true }, { kEv, 677, false } } },
	{	0, 1352, "NNM52",
		{ { kEv, 935, true }, { kEv, 698, false } } },
	{	0, 1353, "NNM53",
		{ { kEv, 863, true }, { kEv, 1062, true }, { kEv, 679, false } } },
	{	0, 1365, "NNM65",
		{ { kEv, 687, true }, { kEv, 1014, false } } },
	{	0, 1367, "NNM67",
		{ { kEv, 687, true }, { kEv, 937, true }, { kEv, 685, false } } },
	{	0, 1370, "NNM70",
		{ { kEv, 945, true }, { kEv, 1003, false }, { kEv, 700, false } } },
	{	0, 1371, "NNM71",
		{ { kEv, 922, true }, { kEv, 694, false } } },
	{	0, 1372, "NNM72",
		{ { kEv, 901, true }, { kEv, 690, false } } },
	{	0, 1373, "NNM73",
		{ { kEv, 563, true }, { kEv, 683, false } } },
},
{	// Ethel Bossiny, 4 responses
	{	0, 1450, "NEB50",
		{ { kEv, 895, true }, { kEv, 131, false } } },
	{	0, 1451, "NEB51",
		{ { kEv, 913, true }, { kEv, 132, false } } },
	{	0, 1454, "NEB54",
		{ { kEv, 130, false } } },
	{	0, 1455, "NEB55",
		{ { kEv, 133, false } } },
},
{	// Linda Penvellyn, 6 responses
	{	0, 1150, "NLP50",
		{ { kEv, 494, true }, { kEv, 493, false }, { kEv, 498, false } } },
	{	0, 1151, "NLP51",
		{ { kEv, 786, true }, { kEv, 493, false }, { kEv, 501, false }, { kEv, 499, false } } },
	{	0, 1152, "NLP52",
		{ { kEv, 567, true }, { kEv, 493, false }, { kEv, 497, false } } },
	{	0, 1157, "NLP57",
		{ { kEv, 944, true }, { kEv, 493, false }, { kEv, 495, false } } },
	{	0, 1159, "NLP59",
		{ { kEv, 880, true }, { kEv, 493, false } } },
	{	0, 1166, "NLP66",
		{ { kEv, 493, false }, { kEv, 500, false }, { kIn, 14, true } } },
},
{ // Loulou the parrot, 44 responses
	{	0, 1552, "NLL52",
		{ { kEv, 836, true }, { kEv, 467, false } } },
	{	0, 1556, "NLL56",
		{ { kEv, 526, true }, { kEv, 774, false } } },
	{	0, 1559, "NLL59",
		{ { kEv, 932, true }, { kEv, 967, false }, { kEv, 447, false }, { kEv, 487, false } } },
	{	0, 1569, "NLL69",
		{ { kEv, 487, true }, { kIn, 0, true }, { kIn, 1, true } } },
	{	0, 1560, "NLL60",
		{ { kEv, 835, true }, { kEv, 966, false }, { kEv, 446, false }, { kEv, 487, false } } },
	{	0, 1561, "NLL61",
		{ { kEv, 881, true }, { kEv, 977, false }, { kEv, 458, false }, { kEv, 487, false } } },
	{	0, 1562, "NLL62",
		{ { kEv, 832, true }, { kEv, 979, false }, { kEv, 461, false }, { kEv, 487, false } } },
	{	0, 5060, "NHIN60",
		{ { kDi, 0, true }, { kEv, 667, true }, { kEv, 462, false }, { kEv, 455, false }, { kEv, 459, false } } },
	{	0, 5061, "NHIN61",
		{ { kEv, 455, true }, { kEv, 462, false }, { kEv, 667, true } } },
	{	0, 5003, "NHIN03",
		{ { kEv, 279, true }, { kEv, 470, false } } },
	{	0, 5004, "NHIN04",
		{ { kEv, 274, true }, { kEv, 471, false } } },
	{	0, 5005, "NHIN05",
		{ { kEv, 283, true }, { kEv, 474, false } } },
	{	0, 5006, "NHIN06",
		{ { kEv, 296, true }, { kEv, 489, false } } },
	{	0, 5007, "NHIN07",
		{ { kEv, 292, true }, { kEv, 460, false } } },
	{	0, 5008, "NHIN08",
		{ { kEv, 268, true }, { kEv, 450, false } } },
	{	0, 5009, "NHIN09",
		{ { kEv, 282, true }, { kEv, 473, false } } },
	{	0, 5010, "NHIN10",
		{ { kEv, 273, true }, { kEv, 457, false } } },
	{	0, 5011, "NHIN11",
		{ { kEv, 289, true }, { kEv, 463, false } } },
	{	0, 5012, "NHIN12",
		{ { kEv, 280, true }, { kEv, 463, false } } },
	{	0, 5013, "NHIN13",
		{ { kEv, 286, true }, { kEv, 463, false } } },
	{	0, 5014, "NHIN14",
		{ { kEv, 293, true }, { kEv, 463, false } } },
	{	0, 5015, "NHIN15",
		{ { kEv, 278, true }, { kEv, 463, false } } },
	{	0, 5016, "NHIN16",
		{ { kEv, 276, true }, { kEv, 463, false } } },
	{	0, 5017, "NHIN17",
		{ { kEv, 266, true }, { kEv, 445, false } } },
	{	0, 5018, "NHIN18",
		{ { kEv, 294, true }, { kEv, 486, false } } },
	{	0, 5019, "NHIN19",
		{ { kEv, 272, true }, { kEv, 456, false } } },
	{	0, 5020, "NHIN20",
		{ { kEv, 270, true }, { kEv, 453, false } } },
	{	0, 5021, "NHIN21",
		{ { kEv, 281, true }, { kEv, 472, false } } },
	{	0, 5022, "NHIN22",
		{ { kEv, 269, true }, { kEv, 451, false } } },
	{	0, 5023, "NHIN23",
		{ { kEv, 288, true }, { kEv, 482, false } } },
	{	0, 5024, "NHIN24",
		{ { kEv, 291, true }, { kEv, 485, false } } },
	{	0, 5025, "NHIN25",
		{ { kEv, 264, true }, { kEv, 441, false } } },
	{	0, 5026, "NHIN26",
		{ { kEv, 265, true }, { kEv, 444, false } } },
	{	0, 5028, "NHIN28",
		{ { kEv, 271, true }, { kEv, 454, false } } },
	{	0, 5029, "NHIN29",
		{ { kEv, 277, true }, { kEv, 468, false } } },
	{	0, 5030, "NHIN30",
		{ { kEv, 284, true }, { kEv, 477, false } } },
	{	0, 5031, "NHIN31",
		{ { kEv, 285, true }, { kEv, 477, false } } },
	{	0, 5032, "NHIN32",
		{ { kEv, 295, true }, { kEv, 488, false } } },
	{	0, 5033, "NHIN33",
		{ { kEv, 267, true }, { kEv, 449, false } } },
	{	0, 5034, "NHIN34",
		{ { kEv, 275, true }, { kEv, 466, false } } },
	{	0, 5035, "NHIN35",
		{ { kEv, 263, true }, { kEv, 440, false } } },
	{	0, 5039, "NHIN36",
		{ { kEv, 290, true }, { kEv, 483, false } } },
	{	0, 5037, "NLX37",
		{ { kEv, 894, true }, { kEv, 843, true }, { kEv, 953, true }, { kEv, 949, true }, { kEv, 834, true }, { kEv, 464, false } } },
	{	0, 5039, "NHIN39",
		{ { kEv, 287, true }, { kEv, 481, false } } },
},
{ // Mrs. Petrov, 6 responses
	{	0, 1650, "NMP50",
		{ { kEv, 416, false }, { kEv, 610, false } } },
	{	0, 1653, "NMP53",
		{ { kEv, 498, true }, { kEv, 878, false }, { kEv, 609, false } } },
	{	0, 1655, "NMP55",
		{ { kEv, 601, false } } },
	{	0, 1656, "NMP56",
		{ { kEv, 242, true }, { kEv, 606, false } } },
	{	0, 1658, "NMP58",
		{ { kEv, 493, true }, { kEv, 611, false } } },
	{	0, 1662, "NMP62",
		{ { kEv, 1014, true }, { kEv, 605, false } } },
},
{	// Paliki Vadas, 5 responses
	{	0, 1974, "NXP74",
		{ { kEv, 145, false } } },
	{	0, 1981, "NXP81",
		{ { kEv, 145, true }, { kEv, 144, false } } },
	{	0, 1984, "NXP84",
		{ { kEv, 144, true }, { kEv, 494, true }, { kEv, 141, false } } },
	{	0, 1986, "NXP86",
		{ { kEv, 145, true }, { kEv, 566, true }, { kEv, 610, true }, { kEv, 140, false } } },
	{	0, 1988, "NXP88",
		{ { kEv, 145, true }, { kEv, 880, true }, { kEv, 142, false } } },
},
{	// Ned Nickerson, 10 responses
	{	0, 1708, "NDP08",
		{ { kEv, 570, true }, { kEv, 880, false }, { kEv, 864, false }, { kEv, 662, false } } },
	{	0, 1710, "NDP10",
		{ { kEv, 494, true }, { kEv, 662, true }, { kEv, 864, false }, { kEv, 880, false }, { kEv, 660, false } } },
	{	0, 1712, "NNP12",
		{ { kEv, 913, true }, { kEv, 1004, false }, { kEv, 663, false } } },
	{	0, 1715, "NDP15",
		{ { kEv, 944, true }, { kEv, 531, true }, { kEv, 999, false }, { kEv, 665, false } } },
	{	0, 1718, "NDP18",
		{ { kEv, 828, true }, { kEv, 1004, false }, { kEv, 659, false } } },
	{	0, 1721, "NDP21",
		{ { kEv, 145, true }, { kEv, 494, true }, { kEv, 677, true }, { kEv, 880, false }, { kEv, 658, false } } },
	{	0, 1725, "NDP25",
		{ { kEv, 941, true }, { kEv, 664, false } } },
	{	0, 1726, "NDP26",
		{ { kEv, 131, true }, { kEv, 259, true }, { kEv, 665, true }, { kEv, 666, false } } },
	{	0, 1730, "NDP30",
		{ { kEv, 880, true }, { kEv, 658, true }, { kEv, 661, false } } },
	{	0, 1731, "NDP21",
		{ { kEv, 880, true }, { kEv, 658, false }, { kEv, 661, false } } },
},
{	// Hugh Penvellyn, 11 responses
	{	0, 1838, "NHP38",
		{ { kEv, 571, true }, { kEv, 601, true }, { kEv, 299, false } } },
	{	0, 1843, "NHP43",
		{ { kEv, 567, true }, { kEv, 302, false } } },
	{	0, 1846, "NHP46",
		{ { kEv, 492, true }, { kEv, 303, false } } },
	{	0, 1850, "NHP50",
		{ { kEv, 1069, true }, { kEv, 301, false } } },
	{	0, 1853, "NHP53",
		{ { kEv, 563, true }, { kEv, 300, false } } },
	{	0, 1857, "NHP57",
		{ { kEv, 307, false }, { kIn, 14, true } } },
	{	0, 1860, "NHP60",
		{ { kEv, 494, true }, { kEv, 677, true }, { kEv, 297, false } } },
	{	0, 1865, "NHP65",
		{ { kEv, 892, true }, { kEv, 999, false }, { kEv, 298, false } } },
	{	0, 1867, "NHP67",
		{ { kEv, 307, true }, { kEv, 410, true }, { kEv, 304, false }, { kIn, 14, true } } },
	{	0, 1868, "NHP68",
		{ { kEv, 307, true }, { kEv, 410, false }, { kEv, 304, false }, { kIn, 14, true } } },
	{	0, 1871, "NHP71",
		{ { kEv, 941, true }, { kEv, 306, false } } },
}
};

const Common::Array<Goodbye> _nancy11Goodbyes = {
	{ "NDE90", { { { 1090, 1091, 1092 		}, {}, NOFLAG } } },	// Mrs. Leticia Drake
	{ "NJP90", { { { 1290, 1291, 1292, 1293 }, {}, NOFLAG } } },	// Jane Penvellyn
	{ "NNM90", { { { 1390, 1391, 1392, 1393 }, {}, NOFLAG } } },	// Nigel Mookerjee
	{ "NEB90", { { { 1490 					}, {}, NOFLAG } } },	// Ethel Bossiny
	{ "NLP90", { { { 1190, 1191, 1192 		}, {}, NOFLAG } } },	// Linda Penvellyn
	{ "NLL90", { { { 1590, 1591, 1592, 1593, 1594, 1595, 1596 }, {}, NOFLAG } } },	// Loulou the parrot
	{ "NMP90", { { { 1690, 1691, 1692 		}, {}, NOFLAG } } },	// Mrs. Petrov
	{ "NXP95", { { { 1995, 1996, 1997 		}, {}, NOFLAG } } },	// Paliki Vadas
	{ "NDP90", { { { 1790, 1791, 1792, 1793 }, {}, NOFLAG } } },	// Ned Nickerson
	{ "NHP90", { { { 1890, 1891, 1892, 1893 }, {}, NOFLAG } } },	// Hugh Penvellyn
};

const Common::Array<const char *> _nancy11EventFlagNames = {
	"Generic0",
	"Generic1",
	"Generic2",
	"Generic3",
	"Generic4",
	"Generic5",
	"Generic6",
	"Generic7",
	"Generic8",
	"Generic9",
	"Generic10",
	"Generic11",
	"Generic12",
	"Generic13",
	"Generic14",
	"Generic15",
	"Generic16",
	"Generic17",
	"Generic18",
	"Generic19",
	"Generic20",
	"Generic21",
	"Generic22",
	"Generic23",
	"Generic24",
	"Generic25",
	"Generic26",
	"Generic27",
	"Generic28",
	"Generic29",
	"Generic30",
	"TimeForEndgame",
	"PlayerWonGame",
	"StopPlayerScrolling",
	"Easter_Eggs",
	"EV_Aeolus_A4",
	"EV_Aeolus_A5",
	"EV_Aeolus_B4",
	"EV_Aeolus_B5",
	"EV_Aeolus_C4",
	"EV_Aeolus_C5",
	"EV_Aeolus_D1",
	"EV_Aeolus_D2",
	"EV_Aeolus_D3",
	"EV_Aeolus_D4",
	"EV_Aeolus_D5",
	"EV_Aeolus_D6",
	"EV_Aeolus_D7",
	"EV_Aeolus_D8",
	"EV_Aeolus_E1",
	"EV_Aeolus_E2",
	"EV_Aeolus_E3",
	"EV_Aeolus_E4",
	"EV_Aeolus_E5",
	"EV_Aeolus_E6",
	"EV_Aeolus_E7",
	"EV_Aeolus_E8",
	"EV_Aeolus_F1",
	"EV_Aeolus_F2",
	"EV_Aeolus_F3",
	"EV_Aeolus_F4",
	"EV_Aeolus_F5",
	"EV_Aeolus_F6",
	"EV_Aeolus_F7",
	"EV_Aeolus_F8",
	"EV_Aeolus_G1",
	"EV_Aeolus_G2",
	"EV_Aeolus_G3",
	"EV_Aeolus_G4",
	"EV_Aeolus_G5",
	"EV_Aeolus_G6",
	"EV_Aeolus_G7",
	"EV_Aeolus_G8",
	"EV_Aeolus_H1",
	"EV_Aeolus_H2",
	"EV_Aeolus_H3",
	"EV_Aeolus_H4",
	"EV_Aeolus_H5",
	"EV_Aeolus_H6",
	"EV_Aeolus_H7",
	"EV_Aeolus_H8",
	"EV_Bell_DoorA_Stopper_Out",
	"EV_Bell_DoorB_Stopper_Out",
	"EV_Bell_DoorC_Stopper_Out",
	"EV_Bell_DoorD_Stopper_Out",
	"EV_Bell_DoorE_Stopper_Out",
	"EV_Bell_DoorF_Stopper_Out",
	"EV_Bell_DoorG_Stopper_Out",
	"EV_Bell_DoorH_Stopper_Out",
	"EV_Bell_FingerA",
	"EV_Bell_FingerB",
	"EV_Bell_FingerC",
	"EV_Bell_FingerD",
	"EV_Bell_FingerE",
	"EV_Bell_FingerF",
	"EV_Bell_FingerG",
	"EV_Bell_FingerH",
	"EV_Bell_Main_Stopper_Out",
	"EV_Captures_Boreas",
	"EV_Captures_Eurus",
	"EV_Captures_Favonius",
	"EV_Captures_Zephyros",
	"EV_Caught_One_Wind",
	"EV_Chant_Ready",
	"EV_Difficulty_Level",
	"EV_Draw_Skulls",
	"EV_Dream_Ready",
	"EV_DT_Difficulty_Level",
	"EV_E_Blue_Fst_Quar",
	"EV_E_Blue_Full",
	"EV_E_Blue_Last_Quar",
	"EV_E_Blue_Wane_Cres",
	"EV_E_Blue_Wane_Gib",
	"EV_E_Blue_Wax_Cres",
	"EV_E_Blue_Wax_Gib",
	"EV_E_Green_Fst_Quar",
	"EV_E_Green_Full",
	"EV_E_Green_Last_Quar",
	"EV_E_Green_Wane_Cres",
	"EV_E_Green_Wane_Gib",
	"EV_E_Green_Wax_Cres",
	"EV_E_Green_Wax_Gib",
	"EV_E_New_Moon",
	"EV_E_Red_Fst_Quar",
	"EV_E_Red_Full",
	"EV_E_Red_Lst_Quar",
	"EV_E_Red_Wane_Cres",
	"EV_E_Red_Wane_Gib",
	"EV_E_Red_Wax_Cres",
	"EV_E_Red_Wax_Gib",
	"EV_EB_Said_Beast",
	"EV_EB_Said_Curriculum",
	"EV_EB_Said_Mutus",
	"EV_EB_Said_Penvellyn",
	"EV_Entered_Well",
	"EV_EWind_H3",
	"EV_EWind_H4",
	"EV_EWind_H5",
	"EV_EWind_H6",
	"EV_EWind_H7",
	"EV_Expert_Said_Anger",
	"EV_Expert_Said_Curse",
	"EV_Expert_Said_Hand",
	"EV_Expert_Said_Listening",
	"EV_Expert_Said_Suggestion",
	"EV_Expert_Said_Symptoms",
	"EV_Expert_Stall1",
	"EV_Expert_Stall2",
	"EV_Fire_Lock_L1",
	"EV_Fire_Lock_L2",
	"EV_Fire_Lock_L3",
	"EV_Fire_Lock_L4",
	"EV_Fire_Lock_L5",
	"EV_Fire_Lock_L6",
	"EV_Fire_Lock_R1",
	"EV_Fire_Lock_R2",
	"EV_Fire_Lock_R3",
	"EV_Fire_Lock_R4",
	"EV_Fire_Lock_R5",
	"EV_Fire_Lock_R6",
	"EV_Fixed_Betty",
	"EV_Food_Available",
	"EV_Found_Ghost01",
	"EV_Found_Ghost02",
	"EV_Found_Ghost03",
	"EV_Found_Ghost04",
	"EV_Found_Ghost05",
	"EV_Found_Ghost06",
	"EV_Found_Ghost07",
	"EV_Found_Ghost08",
	"EV_Found_Ghost09",
	"EV_Found_Ghost10",
	"EV_Found_Ghost11",
	"EV_Found_Ghost12",
	"EV_Found_Ghost13",
	"EV_Found_Moon_Box_Pack",
	"EV_Gar_Pos1",
	"EV_Gar_Pos2",
	"EV_Gar_Pos3",
	"EV_Gar_Pos4",
	"EV_Gargoyle_Timer",
	"EV_Gargoyle01_Lit",
	"EV_Gargoyle02_Lit",
	"EV_Gargoyle03_Lit",
	"EV_Gargoyle04_Lit",
	"EV_Gargoyle05_Lit",
	"EV_Gargoyle06_Lit",
	"EV_Gargoyle07_Lit",
	"EV_Gargoyle08_Lit",
	"EV_Gargoyle09_Lit",
	"EV_Gargoyle10_Lit",
	"EV_Ghost_Check_In",
	"EV_Ghost_Timer_On",
	"EV_Goggles_On",
	"EV_Golem_In_Pit",
	"EV_Got_Key",
	"EV_Got_Many_Keys",
	"EV_Grenny_Boat_Sink",
	"EV_Grenny_Frog_A1",
	"EV_Grenny_Frog_A2",
	"EV_Grenny_Frog_A3",
	"EV_Grenny_Frog_B1",
	"EV_Grenny_Frog_B2",
	"EV_Grenny_Frog_B3",
	"EV_Grenny_Frog_C1",
	"EV_Grenny_Frog_C2",
	"EV_Grenny_Frog_C3",
	"EV_Grenny_Frog_D1",
	"EV_Grenny_Frog_D2",
	"EV_Grenny_Frog_D3",
	"EV_Grenny_Frog_E1",
	"EV_Grenny_Frog_E2",
	"EV_Grenny_Frog_E3",
	"EV_Grenny_Frog_F1",
	"EV_Grenny_Frog_F2",
	"EV_Grenny_Frog_F3",
	"EV_Grenny_Frog_G1",
	"EV_Grenny_Frog_G2",
	"EV_Grenny_Frog_G3",
	"EV_Grenny_Pad_A1",
	"EV_Grenny_Pad_A2",
	"EV_Grenny_Pad_A3",
	"EV_Grenny_Pad_B1",
	"EV_Grenny_Pad_B2",
	"EV_Grenny_Pad_B3",
	"EV_Grenny_Pad_C1",
	"EV_Grenny_Pad_C2",
	"EV_Grenny_Pad_C3",
	"EV_Grenny_Pad_D1",
	"EV_Grenny_Pad_D2",
	"EV_Grenny_Pad_D3",
	"EV_Grenny_Pad_E1",
	"EV_Grenny_Pad_E2",
	"EV_Grenny_Pad_E3",
	"EV_Grenny_Pad_F1",
	"EV_Grenny_Pad_F2",
	"EV_Grenny_Pad_F3",
	"EV_Grenny_Pad_G1",
	"EV_Grenny_Pad_G2",
	"EV_Grenny_Pad_G3",
	"EV_Hear_French_Repeat",
	"EV_Heard_Ballad",
	"EV_Heard_Chanting",
	"EV_Heard_Error_Msg",
	"EV_Heard_Filius",
	"EV_Heard_French_Lesson01",
	"EV_Heard_French_Lesson02",
	"EV_Heard_French_Lesson03",
	"EV_Heard_French_Lesson04",
	"EV_Heard_French_Lesson05",
	"EV_Heard_French_Repeat",
	"EV_Heard_Ghost_Time",
	"EV_Heard_Howl",
	"EV_Heard_Latin_Lesson01",
	"EV_Heard_Latin_Lesson02",
	"EV_Heard_Latin_Lesson03",
	"EV_Heard_Latin_Lesson04",
	"EV_Heard_Latin_Lesson05",
	"EV_Heard_LP_Sob",
	"EV_Heard_Milo_Skip",
	"EV_Heard_Nonsense",
	"EV_Heard_Randulf_Lesson",
	"EV_Heard_Scratching",
	"EV_Hint_Air_Flow",
	"EV_Hint_Alchemy",
	"EV_Hint_Bells",
	"EV_Hint_Betty",
	"EV_Hint_Bul",
	"EV_Hint_Bulls_Eye",
	"EV_Hint_Butter",
	"EV_Hint_Corbin",
	"EV_Hint_Fire_Lock",
	"EV_Hint_Flow",
	"EV_Hint_Ghosts",
	"EV_Hint_Lens",
	"EV_Hint_Login",
	"EV_Hint_Mars_Key",
	"EV_Hint_Maze",
	"EV_Hint_Mercury_Key",
	"EV_Hint_Moon_Box",
	"EV_Hint_Moon_Key",
	"EV_Hint_Multiarm",
	"EV_Hint_Muse_Code",
	"EV_Hint_Panel",
	"EV_Hint_Rune",
	"EV_Hint_Rune_Word",
	"EV_Hint_Saturn_Key",
	"EV_Hint_Slide",
	"EV_Hint_Slider",
	"EV_Hint_Slider_Key",
	"EV_Hint_Smithing",
	"EV_Hint_Squeaky",
	"EV_Hint_Squirt",
	"EV_Hint_Venus_Key",
	"EV_Hint_Wand",
	"EV_Hint_Well",
	"EV_Hint_Zodiac",
	"EV_HP_Said_Beast",
	"EV_HP_Said_Columns",
	"EV_HP_Said_Drake",
	"EV_HP_Said_Ethel",
	"EV_HP_Said_Father",
	"EV_HP_Said_Jane",
	"EV_HP_Said_Passageways",
	"EV_HP_Said_Pig",
	"EV_HP_Said_Pressed",
	"EV_HP_Said_Renee",
	"EV_HP_Said_Roger",
	"EV_HP_Said_Very_Busy",
	"EV_Inserted_Moon_Lens",
	"EV_Jigsaw01_Done",
	"EV_Jigsaw02_Done",
	"EV_Jigsaw03_Done",
	"EV_JP_Available",
	"EV_JP_Gets_Threat",
	"EV_JP_Said_10sec",
	"EV_JP_Said_13_Ghosts",
	"EV_JP_Said_1min",
	"EV_JP_Said_2min",
	"EV_JP_Said_30sec",
	"EV_JP_Said_3min",
	"EV_JP_Said_4min",
	"EV_JP_Said_5min",
	"EV_JP_Said_Adam",
	"EV_JP_Said_Agatha",
	"EV_JP_Said_Alan",
	"EV_JP_Said_Albert",
	"EV_JP_Said_Anicia",
	"EV_JP_Said_Annor",
	"EV_JP_Said_Arms",
	"EV_JP_Said_Arthur",
	"EV_JP_Said_Aw",
	"EV_JP_Said_Beast",
	"EV_JP_Said_Betty_Broke",
	"EV_JP_Said_Boy",
	"EV_JP_Said_Brigitte",
	"EV_JP_Said_Brigitte2",
	"EV_JP_Said_Bul_Directions",
	"EV_JP_Said_Cake_Maker",
	"EV_JP_Said_Caroline",
	"EV_JP_Said_Casandra",
	"EV_JP_Said_Catherine",
	"EV_JP_Said_Cecilia",
	"EV_JP_Said_Charles",
	"EV_JP_Said_Colin",
	"EV_JP_Said_Congrats",
	"EV_JP_Said_Corbin",
	"EV_JP_Said_Curriculum",
	"EV_JP_Said_Cynthia",
	"EV_JP_Said_Dark",
	"EV_JP_Said_Edmund",
	"EV_JP_Said_Edward",
	"EV_JP_Said_Edward2",
	"EV_JP_Said_Elinor",
	"EV_JP_Said_Elizabeth",
	"EV_JP_Said_Esther",
	"EV_JP_Said_Family_Tree",
	"EV_JP_Said_First_Glow",
	"EV_JP_Said_Francis",
	"EV_JP_Said_Francois",
	"EV_JP_Said_Frederic",
	"EV_JP_Said_Games",
	"EV_JP_Said_Garrett",
	"EV_JP_Said_George",
	"EV_JP_Said_George2",
	"EV_JP_Said_Gillian",
	"EV_JP_Said_Goggles",
	"EV_JP_Said_Guydo",
	"EV_JP_Said_Hands",
	"EV_JP_Said_Helene",
	"EV_JP_Said_Hottie",
	"EV_JP_Said_Hugo",
	"EV_JP_Said_Isabelle",
	"EV_JP_Said_Jacobus",
	"EV_JP_Said_Jacques",
	"EV_JP_Said_James",
	"EV_JP_Said_Jean",
	"EV_JP_Said_Jeffrey",
	"EV_JP_Said_Jenet",
	"EV_JP_Said_John1",
	"EV_JP_Said_John2",
	"EV_JP_Said_Jone",
	"EV_JP_Said_Josephus",
	"EV_JP_Said_Lady",
	"EV_JP_Said_Leticia",
	"EV_JP_Said_Login",
	"EV_JP_Said_Lucia",
	"EV_JP_Said_Magic",
	"EV_JP_Said_Make_Games",
	"EV_JP_Said_Malachi",
	"EV_JP_Said_Marge",
	"EV_JP_Said_Margery",
	"EV_JP_Said_Marianne",
	"EV_JP_Said_Martha",
	"EV_JP_Said_Milo",
	"EV_JP_Said_Mutus",
	"EV_JP_Said_Nahum",
	"EV_JP_Said_Nicholina",
	"EV_JP_Said_Nightmare",
	"EV_JP_Said_No_Chanting",
	"EV_JP_Said_No_Corbin",
	"EV_JP_Said_Obadiah",
	"EV_JP_Said_Odo",
	"EV_JP_Said_Ouroboros",
	"EV_JP_Said_Passage",
	"EV_JP_Said_Passage_Location",
	"EV_JP_Said_Penelope",
	"EV_JP_Said_Peter",
	"EV_JP_Said_Petro",
	"EV_JP_Said_Philippe",
	"EV_JP_Said_Pig",
	"EV_JP_Said_Rachel",
	"EV_JP_Said_Randulf",
	"EV_JP_Said_Richard",
	"EV_JP_Said_Ritual",
	"EV_JP_Said_Robertus",
	"EV_JP_Said_Rome",
	"EV_JP_Said_Rose",
	"EV_JP_Said_Rules",
	"EV_JP_Said_See_COA",
	"EV_JP_Said_Simon",
	"EV_JP_Said_Snickerdoodle",
	"EV_JP_Said_Sophia",
	"EV_JP_Said_Tapestry",
	"EV_JP_Said_Telescope",
	"EV_JP_Said_Telescope_Time",
	"EV_JP_Said_Theophile",
	"EV_JP_Said_Thomas",
	"EV_JP_Said_Threat",
	"EV_JP_Said_Time_Up",
	"EV_JP_Said_Treasure",
	"EV_JP_Said_Virginie",
	"EV_JP_Said_Walter",
	"EV_JP_Said_William",
	"EV_JP_Said_William2",
	"EV_JP_Said_Win_Passage",
	"EV_Jupiter_KnightA",
	"EV_Jupiter_KnightB",
	"EV_Jupiter_KnightC",
	"EV_Jupiter_KnightD",
	"EV_LL_Said_Air_Flow",
	"EV_LL_Said_Alchemy",
	"EV_LL_Said_Apple",
	"EV_LL_Said_Bee",
	"EV_LL_Said_Bells",
	"EV_LL_Said_Betty",
	"EV_LL_Said_Bonnet",
	"EV_LL_Said_Book",
	"EV_LL_Said_Bottle",
	"EV_LL_Said_Bul",
	"EV_LL_Said_Bulls_Eye",
	"EV_LL_Said_Butter",
	"EV_LL_Said_Cake_First",
	"EV_LL_Said_Corbin",
	"EV_LL_Said_Fire",
	"EV_LL_Said_First_Hint",
	"EV_LL_Said_Flow",
	"EV_LL_Said_Ghosts",
	"EV_LL_Said_Glass",
	"EV_LL_Said_Got_Hints",
	"EV_LL_Said_Grenny",
	"EV_LL_Said_Hill",
	"EV_LL_Said_Hint",
	"EV_LL_Said_Key",
	"EV_LL_Said_Latin",
	"EV_LL_Said_Leech",
	"EV_LL_Said_Login",
	"EV_LL_Said_Magic",
	"EV_LL_Said_Maze",
	"EV_LL_Said_Mitten",
	"EV_LL_Said_Moon_Box",
	"EV_LL_Said_Moon_Lens",
	"EV_LL_Said_MultiArm",
	"EV_LL_Said_Muse_Code",
	"EV_LL_Said_Panel",
	"EV_LL_Said_Rock",
	"EV_LL_Said_Rules",
	"EV_LL_Said_Rune",
	"EV_LL_Said_Scared",
	"EV_LL_Said_Set_Alarm",
	"EV_LL_Said_Silly",
	"EV_LL_Said_Slide",
	"EV_LL_Said_Slider",
	"EV_LL_Said_Smithing",
	"EV_LL_Said_Sonnet",
	"EV_LL_Said_Squeaky",
	"EV_LL_Said_Wand",
	"EV_LL_Said_Want_Cake",
	"EV_LL_Said_Well",
	"EV_LL_Said_Zodiac",
	"EV_LP_Available",
	"EV_LP_Said_Bells",
	"EV_LP_Said_Gargoyle",
	"EV_LP_Said_Go",
	"EV_LP_Said_Help",
	"EV_LP_Said_Hungry",
	"EV_LP_Said_Lady",
	"EV_LP_Said_Lycanthropy",
	"EV_LP_Said_No_Passage",
	"EV_LP_Said_No_Solution",
	"EV_LP_Said_Pig",
	"EV_LP_Stall1",
	"EV_Made_Cake_Bad",
	"EV_Made_Cake_Good",
	"EV_Made_Key_Right",
	"EV_Made_Key_Wrong",
	"EV_Mars_KnightA",
	"EV_Mars_KnightB",
	"EV_Mars_KnightC",
	"EV_Mars_KnightD",
	"EV_Maze_Position1",
	"EV_Maze_Position2",
	"EV_Maze_Position3",
	"EV_Maze_Position4",
	"EV_MD_Available",
	"EV_MD_Said_Alan",
	"EV_MD_Said_Allergies",
	"EV_MD_Said_Aralia",
	"EV_MD_Said_Charm",
	"EV_MD_Said_Chased",
	"EV_MD_Said_Fish",
	"EV_MD_Said_Fixed",
	"EV_MD_Said_Flashlight",
	"EV_MD_Said_Haunted",
	"EV_MD_Said_Login",
	"EV_MD_Said_Loulou",
	"EV_MD_Said_Magic",
	"EV_MD_Said_Mites",
	"EV_MD_Said_Mutus",
	"EV_MD_Said_Nigel",
	"EV_MD_Said_No_Crank",
	"EV_MD_Said_Oil",
	"EV_MD_Said_Peaked",
	"EV_MD_Said_Philadendron",
	"EV_MD_Said_Pinch",
	"EV_MD_Said_Plant_Task",
	"EV_MD_Said_Renee",
	"EV_MD_Said_Right_Rain",
	"EV_MD_Said_Ritual",
	"EV_MD_Said_Spider",
	"EV_MD_Said_Tad_Dry",
	"EV_MD_Said_Take_Crank",
	"EV_MD_Said_Tiger",
	"EV_MD_Said_Torch",
	"EV_MD_Said_TT",
	"EV_MD_Said_Well",
	"EV_MD_Talks_To_Plants",
	"EV_Meal_Done",
	"EV_Meat_Stall",
	"EV_Mercury_Faces_East",
	"EV_Mercury_Faces_North",
	"EV_Mercury_Faces_South",
	"EV_Mercury_Faces_West",
	"EV_Mercury_KnightA",
	"EV_Mercury_KnightB",
	"EV_Mercury_KnightC",
	"EV_Mercury_KnightD",
	"EV_Mercury_Looks_Down",
	"EV_Mercury_Looks_Left",
	"EV_Mercury_Looks_Right",
	"EV_Mercury_Looks_Up",
	"EV_Met_Aeolus",
	"EV_Met_EB1",
	"EV_Met_EB2",
	"EV_Met_EB3",
	"EV_Met_Expert",
	"EV_Met_Hugh",
	"EV_Met_JP",
	"EV_Met_Lab",
	"EV_Met_LL",
	"EV_Met_LP",
	"EV_Met_MD",
	"EV_Met_Ned",
	"EV_Met_NM",
	"EV_Met_Petrov",
	"EV_Met_Slide",
	"EV_Met_TT",
	"EV_Meta_Award01",
	"EV_Meta_Award02",
	"EV_Meta_Award03",
	"EV_Meta_Award04",
	"EV_Meta_Award05",
	"EV_Meta_Award06",
	"EV_Meta_Award07",
	"EV_Meta_Award08",
	"EV_Meta_Award09",
	"EV_Meta_Award10",
	"EV_Meta_Award11",
	"EV_Meta_Award12",
	"EV_Meta_Award13",
	"EV_Meta_Award14",
	"EV_Meta_Award15",
	"EV_Meta_Award16",
	"EV_Meta_Award17",
	"EV_Meta_Award18",
	"EV_Meta_Award19",
	"EV_Meta_Award20",
	"EV_Moon_KnightA",
	"EV_Moon_KnightB",
	"EV_Moon_KnightC",
	"EV_Moon_KnightD",
	"EV_MP_Said_Drake",
	"EV_MP_Said_Drake_Lose",
	"EV_MP_Said_Hello1",
	"EV_MP_Said_Niece",
	"EV_MP_Said_Nigel",
	"EV_MP_Said_Odd",
	"EV_MP_Said_Pale",
	"EV_MP_Said_Parrot_Lose",
	"EV_MP_Said_Passage",
	"EV_MP_Said_Rome",
	"EV_MP_Said_Stay",
	"EV_MultiA_Pos1",
	"EV_MultiA_Pos2",
	"EV_MultiA_Pos3",
	"EV_MultiA_Pos4",
	"EV_MultiB_Pos1",
	"EV_MultiB_Pos2",
	"EV_MultiB_Pos3",
	"EV_MultiB_Pos4",
	"EV_MultiC_Pos1",
	"EV_MultiC_Pos2",
	"EV_MultiC_Pos3",
	"EV_MultiC_Pos4",
	"EV_MultiD_Pos1",
	"EV_MultiD_Pos2",
	"EV_MultiD_Pos3",
	"EV_MultiD_Pos4",
	"EV_MultiE_Pos1",
	"EV_MultiE_Pos2",
	"EV_MultiE_Pos3",
	"EV_MultiE_Pos4",
	"EV_MultiF_Pos1",
	"EV_MultiF_Pos2",
	"EV_MultiF_Pos3",
	"EV_MultiF_Pos4",
	"EV_N_Blue_Fst_Quar",
	"EV_N_Blue_Full",
	"EV_N_Blue_Last_Quar",
	"EV_N_Blue_Wane_Cres",
	"EV_N_Blue_Wane_Gib",
	"EV_N_Blue_Wax_Cres",
	"EV_N_Blue_Wax_Gib",
	"EV_N_Green_Fst_Quar",
	"EV_N_Green_Full",
	"EV_N_Green_Last_Quar",
	"EV_N_Green_Wane_Cres",
	"EV_N_Green_Wane_Gib",
	"EV_N_Green_Wax_Cres",
	"EV_N_Green_Wax_Gib",
	"EV_N_New_Moon",
	"EV_N_Red_Fst_Quar",
	"EV_N_Red_Full",
	"EV_N_Red_Lst_Quar",
	"EV_N_Red_Wane_Cres",
	"EV_N_Red_Wane_Gib",
	"EV_N_Red_Wax_Cres",
	"EV_N_Red_Wax_Gib",
	"EV_Ned_Said_Beast",
	"EV_Ned_Said_Coats",
	"EV_Ned_Said_Curse",
	"EV_Ned_Said_Hairy",
	"EV_Ned_Said_LP_Strange",
	"EV_Ned_Said_Mutus",
	"EV_Ned_Said_Renee",
	"EV_Ned_Said_Ritual",
	"EV_Ned_Said_Teaching",
	"EV_Needs_Hint",
	"EV_Night_Passed",
	"EV_Nightmare_Ready",
	"EV_NM_Afraid",
	"EV_NM_Afraid_Stall",
	"EV_NM_Available",
	"EV_NM_Left",
	"EV_NM_Runs_Out",
	"EV_NM_Said_Afraid",
	"EV_NM_Said_Alan",
	"EV_NM_Said_Beast",
	"EV_NM_Said_Chas",
	"EV_NM_Said_Corbin",
	"EV_NM_Said_Detective",
	"EV_NM_Said_Dexter",
	"EV_NM_Said_Elinor",
	"EV_NM_Said_Ethel",
	"EV_NM_Said_Facts",
	"EV_NM_Said_Famous",
	"EV_NM_Said_Heraldry",
	"EV_NM_Said_Improve",
	"EV_NM_Said_James",
	"EV_NM_Said_JP_Computer",
	"EV_NM_Said_Mercury",
	"EV_NM_Said_No_Password",
	"EV_NM_Said_No_Touch",
	"EV_NM_Said_Odo",
	"EV_NM_Said_Ouroboros",
	"EV_NM_Said_Penelope",
	"EV_NM_Said_Philippe",
	"EV_NM_Said_Pirate",
	"EV_NM_Said_Portrait",
	"EV_NM_Said_Randulf",
	"EV_NM_Said_Rune",
	"EV_NM_Said_Scandal",
	"EV_NM_Said_Shooting",
	"EV_NM_Said_Ten",
	"EV_NM_Said_Treasure",
	"EV_NM_Said_Typing",
	"EV_NM_Sees_Statue",
	"EV_NM_Sneeze_French",
	"EV_NM_Sneeze_German",
	"EV_NM_Sneeze_Russian",
	"EV_NN_Said_Forced_Call",
	"EV_NWind_D2",
	"EV_NWind_E2",
	"EV_NWind_F2",
	"EV_NWind_G2",
	"EV_NWind_H2",
	"EV_O_Blue_Fst_Quar",
	"EV_O_Blue_Full",
	"EV_O_Blue_Last_Quar",
	"EV_O_Blue_Wane_Cres",
	"EV_O_Blue_Wane_Gib",
	"EV_O_Blue_Wax_Cres",
	"EV_O_Blue_Wax_Gib",
	"EV_O_Green_Fst_Quar",
	"EV_O_Green_Full",
	"EV_O_Green_Last_Quar",
	"EV_O_Green_Wane_Cres",
	"EV_O_Green_Wane_Gib",
	"EV_O_Green_Wax_Cres",
	"EV_O_Green_Wax_Gib",
	"EV_O_New_Moon",
	"EV_O_Red_Fst_Quar",
	"EV_O_Red_Full",
	"EV_O_Red_Lst_Quar",
	"EV_O_Red_Wane_Cres",
	"EV_O_Red_Wane_Gib",
	"EV_O_Red_Wax_Cres",
	"EV_O_Red_Wax_Gib",
	"EV_Ordered_Butter",
	"EV_Ordered_Food",
	"EV_Ordered_Meat_Pie",
	"EV_Ordered_Sausages",
	"EV_Ordered_Soup",
	"EV_Ordered_Turkey",
	"EV_Played_Bells",
	"EV_Prints_Cleaned",
	"EV_Removed_Moon_Lens",
	"EV_RotateMaze",
	"EV_S_Blue_Fst_Quar",
	"EV_S_Blue_Full",
	"EV_S_Blue_Last_Quar",
	"EV_S_Blue_Wane_Cres",
	"EV_S_Blue_Wane_Gib",
	"EV_S_Blue_Wax_Cres",
	"EV_S_Blue_Wax_Gib",
	"EV_S_Green_Fst_Quar",
	"EV_S_Green_Full",
	"EV_S_Green_Last_Quar",
	"EV_S_Green_Wane_Cres",
	"EV_S_Green_Wane_Gib",
	"EV_S_Green_Wax_Cres",
	"EV_S_Green_Wax_Gib",
	"EV_S_New_Moon",
	"EV_S_Red_Fst_Quar",
	"EV_S_Red_Full",
	"EV_S_Red_Lst_Quar",
	"EV_S_Red_Wane_Cres",
	"EV_S_Red_Wane_Gib",
	"EV_S_Red_Wax_Cres",
	"EV_S_Red_Wax_Gib",
	"EV_Said_Aeolus_Moving",
	"EV_Said_Alan_Password",
	"EV_Said_Almost",
	"EV_Said_Bad_Idea",
	"EV_Said_Beautiful",
	"EV_Said_Better_Ask_Jane",
	"EV_Said_Better_Not",
	"EV_Said_Big_Key",
	"EV_Said_Butter",
	"EV_Said_Cake_Okay1",
	"EV_Said_Cake_Okay2",
	"EV_Said_Can_Win_Key",
	"EV_Said_Familiar_Symbols",
	"EV_Said_Figure_Out",
	"EV_Said_Food_Here",
	"EV_Said_Forge_Lit",
	"EV_Said_Gargoyle",
	"EV_Said_Get_Cake",
	"EV_Said_Goggles",
	"EV_Said_HP_Busy",
	"EV_Said_Indentation",
	"EV_Said_Interesting",
	"EV_Said_Leave_Blackmoor",
	"EV_Said_Lights_Out",
	"EV_Said_Loop",
	"EV_Said_Make_Merc_Move",
	"EV_Said_Me_Again",
	"EV_Said_Meal_Worms",
	"EV_Said_Merc_Moved",
	"EV_Said_Mutus_Is_Guide",
	"EV_Said_Need_Grease",
	"EV_Said_Niece",
	"EV_Said_NM_Laptop",
	"EV_Said_No_Exit",
	"EV_Said_No_Need_Sticks",
	"EV_Said_No_Phone",
	"EV_Said_Penny",
	"EV_Said_Pie",
	"EV_Said_Rest_Of_Grid",
	"EV_Said_Running_Out",
	"EV_Said_Sausages",
	"EV_Said_Security_System",
	"EV_Said_See_Library",
	"EV_Said_Still_No_Phone",
	"EV_Said_Tripod",
	"EV_Said_Turkey",
	"EV_Said_Well_Full",
	"EV_Said_Winds_Not_Happy",
	"EV_Said_Word_Lock",
	"EV_Saturn_KnightA",
	"EV_Saturn_KnightB",
	"EV_Saturn_KnightC",
	"EV_Saturn_KnightD",
	"EV_Saw_13_Ghosts",
	"EV_Saw_Alan_COA",
	"EV_Saw_Alan_Portrait",
	"EV_Saw_Albert_COA",
	"EV_Saw_Alchemy_URL",
	"EV_Saw_All_Coats",
	"EV_Saw_All_JAN",
	"EV_Saw_All_LAB",
	"EV_Saw_All_Portraits",
	"EV_Saw_Ant",
	"EV_Saw_Astro_URL",
	"EV_Saw_Autumnalis",
	"EV_Saw_Baby",
	"EV_Saw_Barber",
	"EV_Saw_Beast_URL",
	"EV_Saw_Betty_Broke",
	"EV_Saw_Betty_Broken",
	"EV_Saw_Both_Sides",
	"EV_Saw_Brig_COA",
	"EV_Saw_Brig_Portrait",
	"EV_Saw_Bruma",
	"EV_Saw_Bulls_Eye",
	"EV_Saw_Busy",
	"EV_Saw_Carnivore",
	"EV_Saw_Chapter1",
	"EV_Saw_Chapter10",
	"EV_Saw_Chapter2",
	"EV_Saw_Chapter3",
	"EV_Saw_Chapter4",
	"EV_Saw_Chapter5",
	"EV_Saw_Chapter6",
	"EV_Saw_Chapter7",
	"EV_Saw_Chapter8",
	"EV_Saw_Chapter9",
	"EV_Saw_Charm",
	"EV_Saw_Charm_Cinematic",
	"EV_Saw_Chas_COA",
	"EV_Saw_Chas_Portrait",
	"EV_Saw_Cockney_URL",
	"EV_Saw_Corbin_Notes",
	"EV_Saw_Corbin_Portrait",
	"EV_Saw_Curse",
	"EV_Saw_Dark",
	"EV_Saw_Dream",
	"EV_Saw_Edmund_COA",
	"EV_Saw_Edward_COA",
	"EV_Saw_Edward_Portrait",
	"EV_Saw_Element_Symbols",
	"EV_Saw_Elinor_COA",
	"EV_Saw_Elinor_Portrait",
	"EV_Saw_Expert_Number",
	"EV_Saw_Family_Tree",
	"EV_Saw_Fire_Lock_Passage",
	"EV_Saw_Forge",
	"EV_Saw_Gargoyle_Entrance",
	"EV_Saw_Gargoyle_Passage",
	"EV_Saw_Grenny_Book",
	"EV_Saw_Hairy_Hand",
	"EV_Saw_Hour",
	"EV_Saw_Hugh_Portrait",
	"EV_Saw_Hugo_COA",
	"EV_Saw_James_COA",
	"EV_Saw_James_Portrait",
	"EV_Saw_John_COA",
	"EV_Saw_John_Portrait",
	"EV_Saw_Key_Mold",
	"EV_Saw_Keyhole",
	"EV_Saw_Kitchen",
	"EV_Saw_Kitten",
	"EV_Saw_Knight",
	"EV_Saw_Lady",
	"EV_Saw_Latin",
	"EV_Saw_Lesson_Schedule",
	"EV_Saw_Lycan_Chapt",
	"EV_Saw_Mal_COA",
	"EV_Saw_Mal_Portrait",
	"EV_Saw_Martha_COA",
	"EV_Saw_Martha_Portrait",
	"EV_Saw_Mercury_Covered",
	"EV_Saw_Milk",
	"EV_Saw_Milo_COA",
	"EV_Saw_Monster_Book",
	"EV_Saw_Moon_Box_Diagram",
	"EV_Saw_Mural_Symbols",
	"EV_Saw_Mutus_Air",
	"EV_Saw_Mutus_Comet",
	"EV_Saw_Mutus_Escape",
	"EV_Saw_Mutus_Fire",
	"EV_Saw_Mutus_Forge",
	"EV_Saw_Mutus_Key",
	"EV_Saw_Mutus_Liber",
	"EV_Saw_Mutus_Meteor_Touch",
	"EV_Saw_Mutus_Monsters",
	"EV_Saw_Mutus_Oil",
	"EV_Saw_Mutus_Water",
	"EV_Saw_Newton",
	"EV_Saw_Nightmare",
	"EV_Saw_Ode",
	"EV_Saw_Odo_COA",
	"EV_Saw_Ouroboros",
	"EV_Saw_Parrot",
	"EV_Saw_Parrot_Diet_URL",
	"EV_Saw_Passage_Edward",
	"EV_Saw_Penelope_Letter",
	"EV_Saw_Penny_COA",
	"EV_Saw_Penny_Portrait",
	"EV_Saw_Phil_COA",
	"EV_Saw_Phil_Portrait",
	"EV_Saw_Phone_Co_Msg",
	"EV_Saw_Picture",
	"EV_Saw_Plant_Close",
	"EV_Saw_Plaque",
	"EV_Saw_Portrait",
	"EV_Saw_Prints",
	"EV_Saw_Publisher_Email",
	"EV_Saw_Randulf_COA",
	"EV_Saw_Raw_Meat",
	"EV_Saw_Renee_Picture",
	"EV_Saw_Renee_Postcard",
	"EV_Saw_Rich_COA",
	"EV_Saw_Rich_Portrait",
	"EV_Saw_Ritual",
	"EV_Saw_Rune_Tutorial",
	"EV_Saw_Scratch_Marks",
	"EV_Saw_Slide_Instructions",
	"EV_Saw_Solicitor_Letter",
	"EV_Saw_Solstitium",
	"EV_Saw_Thomas_COA",
	"EV_Saw_Thomas_Portrait",
	"EV_Saw_Tooth_Curse",
	"EV_Saw_Vernum",
	"EV_Saw_Well",
	"EV_Saw_Will_COA",
	"EV_Saw_Will_Portrait",
	"EV_Smelled_Oil",
	"EV_Solved_13_Ghosts",
	"EV_Solved_Air_Flow",
	"EV_Solved_Alchemy_Test",
	"EV_Solved_All_Columns",
	"EV_Solved_Apple",
	"EV_Solved_Automaton",
	"EV_Solved_Bee",
	"EV_Solved_Bells",
	"EV_Solved_Bonnet",
	"EV_Solved_Book",
	"EV_Solved_Brigitte",
	"EV_Solved_Bul",
	"EV_Solved_Bulls_Eye",
	"EV_Solved_East",
	"EV_Solved_Fire_Lock",
	"EV_Solved_Gargoyle_Light",
	"EV_Solved_Gargoyle_Turn",
	"EV_Solved_Ghosts",
	"EV_Solved_Ghosts_Again",
	"EV_Solved_Glass",
	"EV_Solved_Grenny",
	"EV_Solved_Hill",
	"EV_Solved_Jigsaw",
	"EV_Solved_Jupiter_Column",
	"EV_Solved_Jupiter_Knight",
	"EV_Solved_Keys",
	"EV_Solved_Leech",
	"EV_Solved_Login",
	"EV_Solved_Mars_Column",
	"EV_Solved_Mars_Knight",
	"EV_Solved_Match_Constellation",
	"EV_Solved_Maze",
	"EV_Solved_Mercury_Column",
	"EV_Solved_Mercury_Knight",
	"EV_Solved_Mitten",
	"EV_Solved_Moon_Box",
	"EV_Solved_Moon_Column",
	"EV_Solved_Moon_Knight",
	"EV_Solved_Moon_Panel",
	"EV_Solved_MultiArm",
	"EV_Solved_Muse_Code",
	"EV_Solved_Oil_Flow",
	"EV_Solved_One_Column",
	"EV_Solved_Petroglyph",
	"EV_Solved_Plant_Task",
	"EV_Solved_Rune_Lock",
	"EV_Solved_Sanctum_Key",
	"EV_Solved_Saturn_Column",
	"EV_Solved_Saturn_Knight",
	"EV_Solved_Silly",
	"EV_Solved_Skulls",
	"EV_Solved_Slider_Lock",
	"EV_Solved_Slider_Puzzle",
	"EV_Solved_Sonnet",
	"EV_Solved_Squeaky_Key",
	"EV_Solved_Squeaky_Lock",
	"EV_Solved_Type",
	"EV_Solved_Venus_Column",
	"EV_Solved_Venus_Knight",
	"EV_Solved_Well_Lock",
	"EV_Solved_Word_Lock",
	"EV_Solved_Zodiac",
	"EV_Spooky01",
	"EV_Spooky02",
	"EV_Spooky03",
	"EV_Spooky04",
	"EV_Spooky05",
	"EV_Spooky06",
	"EV_Spooky07",
	"EV_Spooky08",
	"EV_Spooky09",
	"EV_Spooky10",
	"EV_Statue_Covered",
	"EV_SWind_D8",
	"EV_SWind_E8",
	"EV_SWind_F8",
	"EV_SWind_G8",
	"EV_SWind_H8",
	"EV_Telescope_Available",
	"EV_Telescope_Stall",
	"EV_Time_To_Flicker",
	"EV_Timer01",
	"EV_Timer02",
	"EV_Timer03",
	"EV_Timer04",
	"EV_Timer05",
	"EV_Timer06",
	"EV_Timer07",
	"EV_Timer08",
	"EV_Timer09",
	"EV_Timer10",
	"EV_Too_Slow",
	"EV_Took_Ball",
	"EV_Took_Wand",
	"EV_Tried_Air_Flow",
	"EV_Tried_Alchemy_Test",
	"EV_Tried_Automaton",
	"EV_Tried_Bells",
	"EV_Tried_Bul",
	"EV_Tried_Bulls_Eye",
	"EV_Tried_East",
	"EV_Tried_Fire_Lock_Left",
	"EV_Tried_Fire_Lock_Right",
	"EV_Tried_Gargoyle_Light",
	"EV_Tried_Gargoyle_Turn",
	"EV_Tried_Ghosts",
	"EV_Tried_Grenny",
	"EV_Tried_Jigsaw",
	"EV_Tried_Jupiter_Knight",
	"EV_Tried_Key_Mold",
	"EV_Tried_Key_Smithing",
	"EV_Tried_Login",
	"EV_Tried_Many_Knights",
	"EV_Tried_Mars_Knight",
	"EV_Tried_Match_Constellation",
	"EV_Tried_Maze",
	"EV_Tried_Mercury_Knight",
	"EV_Tried_Moon_Box",
	"EV_Tried_Moon_Knight",
	"EV_Tried_Moon_Panel",
	"EV_Tried_MultiArm",
	"EV_Tried_Muse_Code",
	"EV_Tried_Plant_Task",
	"EV_Tried_Rune_Lock",
	"EV_Tried_Sanctum_Key",
	"EV_Tried_Saturn_Knight",
	"EV_Tried_Skulls",
	"EV_Tried_Slider_Puzzle",
	"EV_Tried_Squeaky_Lock",
	"EV_Tried_Squirt_Gun_Empty",
	"EV_Tried_Typing_Test",
	"EV_Tried_Venus_Knight",
	"EV_Tried_Well_Lock",
	"EV_Tried_Word_Lock",
	"EV_Tried_Zodiac",
	"EV_TT_Said_Wallace",
	"EV_Used_Butter",
	"EV_Venus_KnightA",
	"EV_Venus_KnightB",
	"EV_Venus_KnightC",
	"EV_Venus_KnightD",
	"EV_W_Blue_Fst_Quar",
	"EV_W_Blue_Full",
	"EV_W_Blue_Last_Quar",
	"EV_W_Blue_Wane_Cres",
	"EV_W_Blue_Wane_Gib",
	"EV_W_Blue_Wax_Cres",
	"EV_W_Blue_Wax_Gib",
	"EV_W_Green_Fst_Quar",
	"EV_W_Green_Full",
	"EV_W_Green_Last_Quar",
	"EV_W_Green_Wane_Cres",
	"EV_W_Green_Wane_Gib",
	"EV_W_Green_Wax_Cres",
	"EV_W_Green_Wax_Gib",
	"EV_W_New_Moon",
	"EV_W_Red_Fst_Quar",
	"EV_W_Red_Full",
	"EV_W_Red_Lst_Quar",
	"EV_W_Red_Wane_Cres",
	"EV_W_Red_Wane_Gib",
	"EV_W_Red_Wax_Cres",
	"EV_W_Red_Wax_Gib",
	"EV_Won_Glow_Stick",
	"EV_WWind_D3",
	"EV_WWind_D4",
	"EV_WWind_D5",
	"EV_WWind_D6",
	"EV_WWind_D7",
	"EV_Empty1",
	"EV_Empty2",
	"EV_Empty3",
	"EV_Empty4",
	"EV_Empty5",
	"EV_Empty6",
	"EV_Empty7",
	"EV_Empty8",
	"EV_Empty9",
	"EV_Empty10",
	"EV_Empty11",
	"EV_Empty12",
	"EV_Empty13",
	"EV_Empty14",
	"EV_Empty15",
	"EV_Empty16",
	"EV_Empty17",
	"EV_Empty18",
	"EV_Empty19",
	"EV_Empty20",
	"EV_Empty21",
	"EV_Empty22",
	"EV_Empty23",
	"EV_Empty24",
	"EV_Empty25",
	"EV_Empty26",
	"EV_Empty27",
	"EV_Empty28",
	"EV_Empty29",
	"EV_Empty30",
	"EV_Empty31",
	"EV_Empty32",
	"EV_Empty33",
	"EV_Empty34",
	"EV_Empty35",
	"EV_Empty36",
	"EV_Empty37",
	"EV_Empty38",
	"EV_Empty39",
	"EV_Empty40",
	"EV_Empty41",
	"EV_Empty42",
	"EV_Empty43",
	"EV_Empty44",
	"EV_Empty45",
	"EV_Empty46",
	"EV_Empty47",
	"EV_Empty48",
	"EV_Empty49",
	"EV_Empty50",
	"EV_Empty51",
	"EV_Empty52",
	"EV_Empty53",
	"EV_Empty54",
	"EV_Empty55",
	"EV_Empty56",
	"EV_Empty57",
	"EV_Empty58",
	"EV_Empty59",
	"EV_Empty60",
	"EV_Empty61",
	"EV_Empty62",
	"EV_Empty63",
	"EV_Empty64",
	"EV_Empty65",
	"EV_Empty66",
	"EV_Empty67",
	"EV_Empty68",
	"EV_Empty69",
	"EV_Empty70",
	"EV_Empty71",
	"EV_Empty72",
	"EV_Empty73",
	"EV_Empty74",
	"EV_Empty75",
	"EV_Empty76",
	"EV_Empty77",
	"EV_Empty78",
	"EV_Empty79",
	"EV_Empty80",
	"EV_Empty81",
	"EV_Empty82",
	"EV_Empty83",
	"EV_Empty84",
	"EV_Empty85",
	"EV_Empty86",
	"EV_Empty87",
	"EV_Empty88",
	"EV_Empty89",
	"EV_Empty90",
	"EV_Empty91",
	"EV_Empty92",
	"EV_Empty93",
	"EV_Empty94",
	"EV_Empty95",
	"EV_Empty96",
	"EV_Empty97",
	"EV_Empty98",
	"EV_Empty99",
	"EV_Said_Comment01",
	"EV_Said_Comment02",
	"EV_Said_Comment03",
	"EV_Said_Comment04",
	"EV_Said_Comment05",
	"EV_Said_Comment06",
	"EV_Said_Comment07",
	"EV_Said_Comment08",
	"EV_Said_Comment09",
	"EV_Said_Comment10",
	"EV_Said_Comment11",
	"EV_Said_Comment12",
	"EV_Said_Comment13",
	"EV_Said_Comment14",
	"EV_Said_Comment15",
	"EV_Said_Comment16",
	"EV_Said_Comment17",
	"EV_Said_Comment18",
	"EV_Said_Comment19",
	"EV_Said_Comment20",
	"EV_Said_Comment21",
	"EV_Said_Comment22",
	"EV_Said_Comment23",
	"EV_Said_Comment24",
	"EV_Said_Comment25"
};

#endif // NANCY11DATA_H
