// Copyright 2023-2024 Gentoo Authors
// Distributed under the terms of the GNU General Public License v2

using System.IO.Path;
using System.Linq;

var target = Argument("target", "pack");
var configuration = Argument("configuration", "Release");

var pwshRunner = $"tool run pwsh -NoLogo -NoProfile -NonInteractive";

var admin = GetFullPath("./admin");
var codegen = GetFullPath("./codegen/src/Codegen");
var dotnetWrapper = Combine(admin, "dotnet_wrapper.sh");

IEnumerable<string> GetToolPackageNames() {
    GlobberSettings globberSettings = new() { };

    return GetDirectories("./gdmt*/src/Gdmt.*", globberSettings)
        .Select(cakePath => GetFileName($"{cakePath}"))
        .Where(s => s != "Gdmt.Shared")
        .OrderBy(s => s);
}

Task("create-tools-list")
    .Does(() =>
    {
        StartProcess(dotnetWrapper, new ProcessSettings {
                Arguments = $"{pwshRunner} {codegen}/create_tools_list.ps1"
            });
    });

Task("create-source")
    .Does(() =>
    {
        StartProcess(dotnetWrapper, new ProcessSettings {
                Arguments = $"fsi {codegen}/create_license_mapping_fs.fsx"
            });

        StartProcess(dotnetWrapper, new ProcessSettings {
                Arguments = $"fsi {codegen}/create_subcommands_fs.fsx"
            });
    });

Task("format")
    .Does(() =>
    {
        StartProcess(dotnetWrapper, new ProcessSettings {
                Arguments = $"{pwshRunner} {admin}/format_fs.ps1"
            });
    });

Task("regen-source")
    .IsDependentOn("create-tools-list")
    .IsDependentOn("create-source")
    .IsDependentOn("format");

Task("clean")
    .Does(() =>
    {
        StartProcess(dotnetWrapper, new ProcessSettings {
                Arguments = "clean --verbosity quiet"
            });

        CleanDirectory(".cache/nupkg");
    });

Task("restore")
    .Does(() =>
    {
        StartProcess(dotnetWrapper, new ProcessSettings {
                Arguments = $"msbuild /t:PrintEnvironmentVariable_DOTNET_PKG_COMPAT"
            });

        StartProcess(dotnetWrapper, new ProcessSettings {
                Arguments = $"restore --force-evaluate --verbosity quiet"
            });
    });

Task("build")
    .IsDependentOn("restore")
    .Does(() =>
    {
        StartProcess(dotnetWrapper, new ProcessSettings {
                Arguments = $"build --configuration {configuration} --no-restore"
            });
    });

Task("debug")
    .Does(() =>
    {
        StartProcess(dotnetWrapper, new ProcessSettings {
                Arguments = $"{pwshRunner} {admin}/build_debug.ps1"
            });
    });

Task("test")
    .IsDependentOn("build")
    .Does(() =>
    {
        StartProcess(dotnetWrapper, new ProcessSettings {
                Arguments = $"test --configuration {configuration}"
            });
    });

Task("pack")
    .IsDependentOn("build")
    .Does(() =>
    {
        StartProcess(dotnetWrapper, new ProcessSettings {
                Arguments = $"pack --configuration {configuration}"
            });
    });

Task("uninstall")
    .Does(() =>
    {
        foreach (var tool in GetToolPackageNames())
        {
            Information($"Uninstalling tool: {tool}");

            StartProcess(dotnetWrapper, new ProcessSettings {
                    Arguments = $"{pwshRunner} {admin}/uninstall_tool.ps1 {tool}"
                });
        }
    });

Task("install")
    .IsDependentOn("pack")
    .Does(() =>
    {
        foreach (var tool in GetToolPackageNames())
        {
            Information($"Reinstalling tool: {tool}");

            StartProcess(dotnetWrapper, new ProcessSettings {
                    Arguments = $"{pwshRunner} {admin}/reinstall_tool.ps1 {tool}"
                });
        }
    });

Task("before-commit")
    .IsDependentOn("clean")
    .IsDependentOn("regen-source")
    .IsDependentOn("debug")
    .IsDependentOn("test")
    .IsDependentOn("pack");

RunTarget(target);
