#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <ctype.h>
#include <string.h>
#define DEBUG 1
#ifdef DEBUG
   FILE *fplog=NULL;
#endif
#ifndef MYPP
   #define MYPP "xextract"
#endif
#ifndef MYFKO
   #define MYFKO "fko"
#endif
#ifndef MYASM
   #define MYASM "gcc"
#endif
/*
 * This is a helper wrapper for fko (floating point kernel optimizer) to make
 * it act more like a compiler, with an option to apply a preprocessor.
 * fko takes only one source file, and output assembly, whereas most
 * build processes assume that multiple files can be passed, with the
 * output being an object file.  This wrapper does *not* support linking.
 * This wrapper does:
 * (1) For files ending in .B, run MYPP (extract) as preprocessor to create
 *     an FKO HIL file (ending in .b) for compiling by fko
 * (2) Run MYFKO fko on all .b files to produce assembly (.s)
 * (3) Run MYASM on all .s files to produce an object file (.o)
 * The script itself takes flags of (ones w/o -- are standard unix flags):
 * --<flag> : as shown in PrintUsage
 * -Dhand=def -D hand=def -> only used for .B files, passed as -def to extract
 * -o file.[o,obj] -> output object file name (must not contain ., or end
 *    in .o or .obj)
 * -@[extract flag] : eg. -@type=sreal -@-langC -@'-def type double'
 *                    embedded spaces muse be surrounded by '' (not "")
 * -s : produce assembly rather than object file
 * NOTE: any arg not beginning with - and ending in ".b" or ".B" is assumed to
 *       be a file to be compiled.
 */
char *CatStrs(char *s1, char *s2)
{
   int i1, i2, i;
   char *sp;
   i1 = strlen(s1);
   i2 = strlen(s2);
   sp = malloc(i1 + i2 + 1);
   assert(sp);
   for (i=0; i < i1; i++)
      sp[i] = s1[i];
   for (i=0; i < i2; i++)
      sp[i1+i] = s2[i];
   sp[i1+i2] = '\0';
   return(sp);
}

typedef struct ARGNODE ATL_arg_t;
struct ARGNODE
{
   ATL_arg_t *next;
   char *arg;
   int len, nespc;    /* len of arg, # of embedded spaces in arg */
};

ATL_arg_t *NewArg(char *arg)
{
   ATL_arg_t *ap;
   char *sp;
   int ib, ie, i, nes, len;

   ap = malloc(sizeof(ATL_arg_t));
   assert(ap);
   ap->next = NULL;

   if (!arg)
   {
      ap->arg = NULL;
      ap->len = ap->nespc = 0;
      return(ap);
   }
   for (ib=0; arg[ib] != '\0' && isspace(arg[ib]); ib++);
   for (ie=ib; arg[ie] != '\0'; ie++);
   for (ie--; isspace(arg[ie]); ie--);
   len = ie - ib + 1;
   ap->arg = sp = malloc(len+1);
   assert(sp);
   for (len=nes=0,i=ib; i <= ie; i++)
   {
      char ch;
      sp[len++] = ch = arg[i];
      if (isspace(ch))
         nes++;
   }
   sp[len] = '\0';
   ap->len = len;
   ap->nespc = nes;
   return(ap);
}

ATL_arg_t *DupArg(ATL_arg_t *ap)
{
   ATL_arg_t *np;
   np = malloc(sizeof(ATL_arg_t));
   assert(np);
   np->len = ap->len;
   np->nespc = ap->nespc;
   np->next = NULL;
   np->arg = malloc(ap->len+1);
   assert(np->arg);
   strcpy(np->arg, ap->arg);
   return(np);
}

ATL_arg_t *KillArg(ATL_arg_t *ap)
{
   ATL_arg_t *ret=NULL;
   if (ap)
   {
      ret = ap->next;
      if (ap->arg)
         free(ap->arg);
      free(ap);
   }
   return(ret);
}

void KillAllArgs(ATL_arg_t *ap)
{
   while(ap)
      ap = KillArg(ap);
}

ATL_arg_t *CombineArgs(ATL_arg_t *a1, ATL_arg_t *a2)
/*
 * RETURNS: one argument that cats the two args together, separated by a space
 */
{
   ATL_arg_t *np;
   int l1, l2, len;
   assert(a1 && a2);
   assert(a1->arg && a2->arg);
   l1 = a1->len;
   l2 = a2->len;
   len = l1 + l2 + 1;
   np = malloc(sizeof(ATL_arg_t));
   assert(np);
   np->arg = malloc(len);
   assert(np->arg);
   np->len = len;
   strcpy(np->arg, a1->arg);
   np->arg[l1] = ' ';
   strcpy(np->arg+l1+1, a2->arg);
   np->nespc = a1->nespc + a2->nespc + 1;
   np->next = NULL;
   return(np);
}

ATL_arg_t *CombAndKill(ATL_arg_t *ap, char *str)
/*
 * Cats string to argument in ap.
 */
{
   ATL_arg_t *np, *tp;

   tp = NewArg(str);
   np = CombineArgs(ap, tp);
   np->next = NULL;
   KillArg(ap);
   return(np);
}

int FindTotLen(ATL_arg_t *ap)
/*
 * Finds total length of all arguments, including enclosing "" for args with
 * embedded spaces and space added to end to seperate with next flag
 */
{
   int len = 0;
   while (ap)
   {
      len += ap->len + 1;
      if (ap->nespc > 0)
         len += 2;
      ap = ap->next;
   }
   return(len);
}

int FindMaxLen(ATL_arg_t *ap)
/*
 * Finds max length of any arg, including enclosing "" for args with
 * embedded spaces
 */
{
   int maxlen = 0;
   while (ap)
   {
      int len;
      len = ap->len;
      if (ap->nespc > 0)
         len += 2;
      maxlen = (maxlen >= len) ? maxlen : len;
      ap = ap->next;
   }
   return(maxlen);
}

int PrintArgToStr(ATL_arg_t *ap, char *str)
/*
 * Prints ap->arg to str, RETURNS: # of characters printed
 */
{
   int i;

   if (ap->nespc > 0)
      i = sprintf(str, "\"%s\" ", ap->arg);
   else
      i = sprintf(str, "%s ", ap->arg);
   return(i);
}

int PrintAllArgsToStr(ATL_arg_t *ab, char *str)
/*
 * Prints all arguments in ab to str, RETURNS: # of characters printed
 * Args with embedded spaces are placed in quotes (kept as 1 arg)
 */
{
   int i=0;

   while (ab)
   {
      if (ab->nespc > 0)
         i += sprintf(str+i, "\"%s\" ", ab->arg);
      else
         i += sprintf(str+i, "%s ", ab->arg);
      ab = ab->next;
   }
   return(i);
}

int PrintAllArgsToStrNoQuote(ATL_arg_t *ab, char *str)
/*
 * Prints all arguments in ab to str, RETURNS: # of characters printed
 * Args with embedded spaces not quoted (allowed to be mult commandline args)
 */
{
   int i=0;

   while (ab)
   {
      i += sprintf(str+i, "%s ", ab->arg);
      ab = ab->next;
   }
   return(i);
}

char *ArgsToStrNQ(ATL_arg_t *ab, ATL_arg_t *ab1)
{
   char *sp;
   int i0, i1, i;

   i0 = FindTotLen(ab);
   i1 = FindTotLen(ab1);
   sp = malloc(i0+i1+1);
   assert(sp);
   i = PrintAllArgsToStrNoQuote(ab, sp);
   assert(i <= i0);
   i += PrintAllArgsToStrNoQuote(ab1, sp+i);
   assert(i <= i0+i1);
   return(sp);
}
#define FKO_DOCMP 0
#define FKO_DOASM 1
#define FKO_SVTMP 2
char *ParseCompFlags(int nargs, char **args, int *FLAG, ATL_arg_t **files,
                     ATL_arg_t **PP, ATL_arg_t **CMP, ATL_arg_t **ASM)
{
   int i, nfiles=0, DOCMP=1, DOASM=1, SVTMP=0, flg, k;
   ATL_arg_t *ppb, *pp, *cmpb, *cp, *asmb, *ap, *filb;
   char *sp, *ks;
   char *outnm=NULL;
/*
 * Go through compiler flags and separate them into flags affecting
 * preprocessing, compilation, and assembling
 */
   ppb = pp = NewArg(MYPP);
   cmpb = cp = NewArg(MYFKO);
   asmb = ap = NewArg(MYASM);
   filb = NULL;
   for (i=1; i < nargs; i++)
   {
      ATL_arg_t *at;
      at = NewArg(args[i]);
      if (at->arg[0] != '-')  /* file name or arg to flag */
      {
         flg = 1;  /* by default assume flag */
         if (at->len > 2 && at->arg[at->len-2] == '.') /* poss filename */
         {
            char ch = at->arg[at->len-1];
            if (ch == 's' || ch == 'B' || ch == 'b')
            {
               flg = 0;
               at->next = filb;
               filb = at;
            }
         }
         if (flg)
         {
            cp->next = at;
            cp = cp->next;
         }
      }
      else if (at->len == 4 && at->arg[0] == '-' && at->arg[1] == 'm' &&
               ((at->arg[2] == '6' && at->arg[3] == '4') ||
                (at->arg[2] == '3' && at->arg[3] == '2')))
      {  /* -m32/-m64 is passed to assembler, but not fko */
         ap->next = at;
         ap = ap->next;
      }
      else /* if (at->arg[0] == '-')  */
      {
         switch (at->arg[1])
         {
/*
 *       FKO has -I that means start from intermediate files, but fkoc can't
 *       use this option.  Instead, this is assumed to be cpp include
 *       directories, and thus ignored by fkoc
 */
         case 'I' :
            if (at->len == 2) /* path in next arg */
               assert(++i < nargs);
            KillArg(at);
            break;
         case 'o' :  /* -o : found name of output file */
            if (outnm)
               free(outnm);
/*
 *          If this is just -o, then actual name is given in the next arg
 */
            if (at->len == 2)
            {
               assert(++i < nargs);
               sp = args[i];
            }
            else
            {
               sp = at->arg + 2;
               while (*sp && isspace(*sp))
                  sp++;
            }
            flg = strlen(sp);
            outnm = malloc(flg+1);
            assert(outnm);
            strcpy(outnm, sp);
            KillArg(at);
            break;
         case 's' : /* possible -save-temps */
            if (at->len == 11 && !strcmp(at->arg, "-save-temps"))
            {
               SVTMP = 1;
               KillArg(at);
            }
            else /* fko flag */
            {
               cp->next = at;
               cp = cp->next;
            }
            break;
         case 'D' :  /* -Dhan=def or -D han=def are extract arg */
            if (at->len == 2)  /* next arg has definition */
            {
               assert(++i < nargs);
               sp = args[i];
            }
            else  /* is -Dhan=def */
               sp = at->arg + 2;
            for (k=0; sp[k] && sp[k] != '='; k++);  /* find = */
            pp->next = NewArg("-def");
            pp = pp->next;
            pp->next = NewArg(sp);
            pp = pp->next;
            if (sp[k] == '\0') /* if no = in def, define it to 1 */
            {
               pp->next = NewArg("1");
               pp = pp->next;
            }
            else
               pp->arg[k] = ' ';  /* change = to space */
            KillArg(at);
            break;
         case '@' : /* -@ args go only to preprocessor (extract)  */
            if (at->len == 2) /* next arg has extract command */
            {
               assert(++i < nargs);
               sp = args[i];
            }
            else  /* rest of line has extract command */
               sp = at->arg+2;
/*
 *          If extract command surrounded by '' or "", remove quotes
 */
            if (*sp == '\"' || *sp == '\'')
            {
               char q=*sp;
               int k;
               k = strlen(sp+1);
               assert(q == sp[k]);
               ks = malloc(k+1);
               assert(ks);
               strcpy(ks, sp+1);
               ks[k-1] = '\0';
               sp = ks;
            }
            else
               ks = NULL;
            pp->next = NewArg(sp);
            pp = pp->next;
            KillArg(at);
            if (ks)
               free(ks);
            break;
         case 'S' :  /* I only use -x asg-with-cpp, so pass to comp only */
            if (at->len == 2) /* -S alone means emit assembly, not object */
            {
               DOASM = 0;
               KillArg(at);
            }
            else  /* otherwise, must be an fko flag */
            {
               cp->next = at;
               cp = cp->next;
            }
            break;
         case 'E' :  /* I only use -x asg-with-cpp, so pass to comp only */
            if (at->len == 2) /* -E alone means emit assembly, not object */
            {
               DOCMP = 0;
               KillArg(at);
            }
            else  /* otherwise, must be an fko flag */
            {
               cp->next = at;
               cp = cp->next;
            }
            break;
         case 'c' :      /* -c is flag meaning same thing as default */
            KillArg(at); /* so don't remember seeing it, and continue */
            break;
         default:  /* fko flag */
            cp->next = at;
            cp = cp->next;
         }
      }
   }
   if (!filb)
   {
      fprintf(stderr, "%s: no files specified!\n\n", args[0]);
      exit(1);
   }
   *FLAG = (DOASM<<FKO_DOASM) | (DOCMP<<FKO_DOCMP) | (SVTMP<<FKO_SVTMP);
   *files = filb;
   *CMP = cmpb;
   *ASM = asmb;
   *PP = ppb;
   if (outnm && filb->next)  /* must use particular output name */
   {                         /* can only have one file when renaming! */
      fprintf(stderr, "%s: can't use -o for multiple source files!\n",
              args[0]);
      exit(2);
   }
   return(outnm);
}

char GetNextFileExt(ATL_arg_t *ap)
{
   int i = ap->len - 1;
   char ch = ap->arg[i];
   assert(ap->arg[i-1] == '.');
   if (ch == 'B')
      return('b');
   else if (ch == 'b')
      return('s');
   else if (ch == 's')
      return('o');
   return(0);
}

char *FindNameInPath(char *ptnm)
/*
 * RETURNS: ptr in ptnm one past last '/'
 */
{
   int i, isl=0;
   for (i=0; ptnm[i]; i++)
      if (ptnm[i] == '/')
         isl = i+1;
   return(ptnm+isl);
}

char *GetOutFileName(ATL_arg_t *ap, char extc)
{
   char *sp, *fn;
   int i;

   sp = FindNameInPath(ap->arg);  /* ignore path of input file */
   i = strlen(sp);
   assert(i > 2);         /* must end in .[B,b,S] */
   fn = malloc(i+1);
   assert(fn);
   i -= 2;               /* get past [.B,b,S] */
   assert(sp[i] == '.'); /* must be '.' for proper fkoc file extension */
   strncpy(fn, sp, i+1);   /* copy everything except extension */
   fn[i+1] = extc;
   fn[i+2] = '\0';
   return(fn);
}
/*
 * ap->arg should be a file name that ends in ".[B,b,s]".  We will create
 * a temporary file that has the same name plus a random integer (in hex),
 * with file extension ".<extc>", and open the new file, returning the
 * stream in *FPOUT.
 * NOTE: like tmpnam, not completely safe!  In particular, imagine two
 *       instances of fkoc running at same time, with same input file
 *       name.  Both could use same integer, and get same proposed filename.
 *       then both processes check that no such file exists, see that it
 *       does not, and then both do an fopen, which only one of them "wins".
 *       To minimize likelihood of this happening, we to fopen immediately
 *       after checking, and use a random integer rather than a count.
 */
char *GetTempFile(ATL_arg_t *ap, FILE **FPOUT, char extc)
{
   FILE *fp;
   char *fn, *sp;
   int i, cnt;
   char ch;

   sp = FindNameInPath(ap->arg);  /* ignore path of input file */
   i = strlen(sp);
   assert(i > 2);         /* must end in .[B,b,S] */
   fn = malloc(i+9);
   assert(fn);
   i -= 2;               /* get past [.B,b,S] */
   assert(sp[i] == '.'); /* must be '.' for proper fkoc file extension */
   strncpy(fn, sp, i);   /* copy everything except .[B,b,s] */
   ch = sp[i+1];
   if (ch != 'B' && ch != 'b' && ch != 's')
   {
      printf("UNKNOWN FILE EXTENSION: '%c' (FILE='%s')\n", ch, ap->arg);
      exit(10);
   }
   sp = fn+i;
   cnt = 0;
   do
   {
      int k;
      k = rand();
      cnt++;
      sprintf(sp, "%8.8x.%c", k, extc);
      fp = fopen(fn, "r");
      if (fp)            /* if file already exists */
      {
         fclose(fp);
         fp = NULL;      /* try again with different integer */
      }
      else
         fp = fopen(fn, "w");
   }
   while(!fp && cnt < 10000);
   assert(fp);
   *FPOUT = fp;
   return(fn);
}

int main(int narg, char **args)
{
   int flag, DOCOMP, DOASM, SVTMP;
   ATL_arg_t *fileb, *ppb, *cmpb, *asmb, *fp, *inf;
   char *outnm;

   outnm = ParseCompFlags(narg, args, &flag, &fileb, &ppb, &cmpb, &asmb);
   printf("flag=%x\n", flag);
   DOCOMP = flag & (1<<FKO_DOCMP);
   DOASM = flag & (1<<FKO_DOASM);
   SVTMP = flag & (1<<FKO_SVTMP);
   for (fp=fileb; fp; fp = fp->next)
   {
      char extc, *bf=NULL, *sf=NULL, *of=NULL, *pf=NULL;
      FILE *fpout;
      extc = GetNextFileExt(fp);
      pf = fp->arg;
      if (extc == 'b')
      {
         char *sp;
         ATL_arg_t *at;

         at = NewArg("-b");
         at->next = NewArg(fp->arg);
         at->next->next = NewArg("-o");
         pf = bf = GetTempFile(fp, &fpout, extc);
         at->next->next->next = NewArg(bf);
         sp = ArgsToStrNQ(ppb, at);
         printf("BPP='%s'\n", sp);
         assert(!system(sp));
         free(sp);
         KillAllArgs(at);
         fclose(fpout);
         if (!DOCOMP)  /* if this is final stage */
         {
            if (outnm)
            {
               if (rename(bf, outnm))
               {
                  fprintf(stderr, "Can't rename '%s' to '%s'!\n",
                          bf, outnm);
                  exit(-1);
               }
               free(outnm);
            }
            else  /* final stage w/o renaming gets orig filename .b */
            {
               sp = GetOutFileName(fp, 'b');
               assert(!rename(bf, sp));
               free(sp);
            }
            free(bf);
            continue;
         }
         else
            extc = 's';
      }
      if (extc == 's')
      {
         char *sp;
         ATL_arg_t *at;

         at = NewArg(pf);
         at->next = NewArg("-o");
         if (!DOASM)  /* this is final stage */
         {
            if (outnm)
               sp = outnm;
            else
               sp = GetOutFileName(fp, 's');
            fpout = fopen(sp, "w");
            at->next->next = NewArg(sp);
            free(sp);
         }
         else
         {
            pf = sf = GetTempFile(fp, &fpout, extc);
            at->next->next = NewArg(sf);
         }
         sp = ArgsToStrNQ(cmpb, at);
         printf("FKO='%s'\n", sp);
         assert(!system(sp));
         free(sp);
         KillAllArgs(at);
         fclose(fpout);
         if (bf)
         {
            if (!SVTMP)
               assert(!remove(bf));  /* delete temporary file */
            free(bf);
         }
         if (!DOASM)  /* if this is final stage */
            continue;
         extc = 'o';
      }
      if (extc == 'o')
      {
         char *sp;
         ATL_arg_t *at;

         at = NewArg(pf);
         if (outnm)
            sp = outnm;
         else
            sp = GetOutFileName(fp, 'o');
         at->next = NewArg("-c -o");
         at->next->next = NewArg(sp);
         free(sp);
         sp = ArgsToStrNQ(asmb, at);
         KillAllArgs(at);
         printf("ASM='%s'\n", sp);
         assert(!system(sp));
         free(sp);
         if (pf != fp->arg)
         {
            assert(!remove(pf));
            free(pf);
         }
      }
   }
   KillAllArgs(ppb);
   KillAllArgs(cmpb);
   KillAllArgs(asmb);
   KillAllArgs(fileb);
   return(0);
}

