/*  $Id: AutoController.h,v 1.11 2024/04/27 05:24:34 sarrazip Exp $

    cosmosmash - A space rock shooting video game.
    Copyright (C) 2007-2011 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public
    License along with this program; if not, write to the Free
    Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
    Boston, MA  02110-1301, USA.
*/

#ifndef _H_AutoController
#define _H_AutoController

#include "Controller.h"

#include <stdlib.h>
#include <limits.h>


class AutoController : public Controller
{
public:
    AutoController()
      : requestStart(true),
        playerPos(),
        closestRockPos(),
        closestRockDist(HUGE_VAL),
        lowestSpinnerPos(),
        closestPulsarPos(),
        closestPulsarDist(HUGE_VAL),
        goLeft(false),
        goRight(false),
        goHyperspace(false)
    {
    }

    /** Indicates if the player has just asked to start a game.
        Returns true if yes, and then 'isExtendedMode' indicates
        if the non-Intellivision extensions must be activated.
        Otherwise, returns false and 'isExtendedMode' is false.
    */
    virtual bool isStartRequested(bool &isExtendedMode) override
    {
        isExtendedMode = false;
        bool ret = requestStart;
        requestStart = false;
        return ret;
    }

    virtual bool isResumeRequested() override
    {
        return true;
    }

    virtual bool isShootingActive() override
    {
        return true;
    }

    virtual bool isHyperspaceRequested() override
    {
        return goHyperspace;
    }

    virtual bool isLeftMoveRequested() override
    {
        return goLeft;
    }

    virtual bool isRightMoveRequested() override
    {
        return goRight;
    }

    virtual bool isFullScreenToggleRequested() override
    {
        return false;
    }

    virtual void beginGameState() override
    {
        closestRockDist = HUGE_VAL;
        closestRockPos = flatzebra::Couple(-1, -1);

        lowestSpinnerPos = flatzebra::Couple(-1, -1);

        closestPulsarDist = HUGE_VAL;
        closestPulsarPos = flatzebra::Couple(-1, -1);
    }

    virtual void setPlayerPos(flatzebra::Couple pos) override
    {
        playerPos = pos;
    }

    virtual void addRockPos(flatzebra::Couple pos) override
    {
        flatzebra::Couple delta = playerPos - pos;
        double thisDist = delta.length();
        if (thisDist < closestRockDist)
        {
            closestRockDist = thisDist;
            closestRockPos = pos;
        }
    }

    virtual void addSpinnerPos(flatzebra::Couple pos) override
    {
        // The automated player can sometimes start oscillating
        // below a small spinnier that falls vertically.
        // This patch introduces a bit of randomness to help
        // avoid these situations.
        pos.x += rand() % 11 - 5;

        if (pos.y > lowestSpinnerPos.y)
            lowestSpinnerPos = pos;
    }

    virtual void addPulsarPos(flatzebra::Couple pos) override
    {
        flatzebra::Couple delta = playerPos - pos;
        double thisDist = delta.length();
        if (thisDist < closestPulsarDist)
        {
            closestPulsarDist = thisDist;
            closestPulsarPos = pos;
        }
    }

    virtual void endGameState() override
    {
        goLeft = goRight = goHyperspace = false;

        // If a pulsar is very close, use hyperspace.
        if (closestPulsarDist < 100)
            goHyperspace = true;

        // Move if a rock is very close and at a side.
        if (closestRockPos.y >= 0 && closestRockDist < 100)
        {
            goLeft  = playerPos.x <= closestRockPos.x;
            goRight = !goLeft;
            return;
        }

        // Move if there is at least one spinner.
        if (lowestSpinnerPos.y >= 0)
        {
            goLeft  = playerPos.x > lowestSpinnerPos.x;
            goRight = playerPos.x < lowestSpinnerPos.x;
            return;
        }

        // Assume there is no imminent danger,
        // so pursue the nearest rock.
        goLeft  = closestRockPos.y >= 0 && playerPos.x > closestRockPos.x;
        goRight = closestRockPos.y >= 0 && playerPos.x < closestRockPos.x;
    }

private:

    bool requestStart;

    flatzebra::Couple playerPos;

    flatzebra::Couple closestRockPos;  // y == -1 means no rock
    double closestRockDist;

    flatzebra::Couple lowestSpinnerPos;  // y == -1 means no spinner

    flatzebra::Couple closestPulsarPos;  // y == -1 means no rock
    double closestPulsarDist;

    bool goLeft;
    bool goRight;
    bool goHyperspace;
};


#endif  /* _H_AutoController */
