// Copyright 2020-2025 Consensys Software Inc.
// Licensed under the Apache License, Version 2.0. See the LICENSE file for details.

// Code generated by consensys/gnark-crypto DO NOT EDIT

package shplonk

import (
	"errors"
	"hash"
	"math/big"

	"github.com/consensys/gnark-crypto/ecc"
	"github.com/consensys/gnark-crypto/ecc/bls12-377"
	"github.com/consensys/gnark-crypto/ecc/bls12-377/fr"
	"github.com/consensys/gnark-crypto/ecc/bls12-377/kzg"
	fiatshamir "github.com/consensys/gnark-crypto/fiat-shamir"
)

var (
	ErrInvalidNumberOfPoints  = errors.New("number of digests should be equal to the number of points")
	ErrVerifyOpeningProof     = errors.New("can't verify batch opening proof")
	ErrInvalidNumberOfDigests = errors.New("number of digests should be equal to the number of polynomials")
	ErrPairingCheck           = errors.New("pairing product is not 1")
)

// OpeningProof KZG proof for opening (fᵢ)_{i} at a different points (xᵢ)_{i}.
//
// implements io.ReaderFrom and io.WriterTo
type OpeningProof struct {

	// W = ∑ᵢ γⁱZ_{T\Sᵢ}(f_i(X)-r)/Z_{T} where Z_{T} is the vanishing polynomial on the (Sᵢ)_{i}
	// and r interpolates fᵢ(Sᵢ) on (Sᵢ)
	W bls12377.G1Affine

	// L(X)/(X-z) where L(X)=∑ᵢγⁱZ_{T\xᵢ}(f_i(X)-rᵢ) - Z_{T}W(X)
	WPrime bls12377.G1Affine

	// ClaimedValues[i] are the values of fᵢ on Sᵢ
	ClaimedValues [][]fr.Element
}

// BatchOpen opens the list of polynomials on points, where the i-th polynomials is opend at points[i].
func BatchOpen(polynomials [][]fr.Element, digests []kzg.Digest, points [][]fr.Element, hf hash.Hash, pk kzg.ProvingKey, dataTranscript ...[]byte) (OpeningProof, error) {

	var res OpeningProof

	nbPolynomials := len(polynomials)
	if len(polynomials) != len(points) {
		return res, ErrInvalidNumberOfPoints
	}
	if len(polynomials) != len(digests) {
		return res, ErrInvalidNumberOfDigests
	}

	// transcript
	fs := fiatshamir.NewTranscript(hf, "gamma", "z")

	// derive γ
	gamma, err := deriveChallenge("gamma", points, digests, fs, dataTranscript...)
	if err != nil {
		return res, err
	}

	// compute the size of the linear combination
	maxSizePolys := len(polynomials[0])
	for i := 1; i < len(polynomials); i++ {
		if maxSizePolys < len(polynomials[i]) {
			maxSizePolys = len(polynomials[i])
		}
	}
	for i := range points {
		if len(points[i])+1 > maxSizePolys {
			maxSizePolys = len(points[i]) + 1
		}
	}
	nbPoints := 0
	sizeSi := make([]int, len(points))
	for i := 0; i < nbPolynomials; i++ {
		nbPoints += len(points[i])
		sizeSi[i] = len(points[i])
	}
	totalSize := maxSizePolys + nbPoints // upper bound of the size of f := ∑ᵢ γⁱZ_{T\Sᵢ}(f_i(X)-r)

	bufMaxSizePolynomials := make([]fr.Element, maxSizePolys)
	bufTotalSize := make([]fr.Element, totalSize)
	f := make([]fr.Element, totalSize) // cf https://eprint.iacr.org/2020/081.pdf page 11 for notation
	res.ClaimedValues = make([][]fr.Element, nbPolynomials)
	for i := 0; i < nbPolynomials; i++ {
		res.ClaimedValues[i] = make([]fr.Element, len(points[i]))
	}
	var accGamma fr.Element
	accGamma.SetOne()

	ztMinusSi := make([][]fr.Element, nbPolynomials)
	ri := make([][]fr.Element, nbPolynomials)
	for i := 0; i < nbPolynomials; i++ {

		for j := 0; j < len(points[i]); j++ {
			res.ClaimedValues[i][j] = eval(polynomials[i], points[i][j])
		}

		ztMinusSi[i] = buildZtMinusSi(points, i)

		copy(bufMaxSizePolynomials, polynomials[i])
		ri[i] = interpolate(points[i], res.ClaimedValues[i])
		sub(bufMaxSizePolynomials, ri[i])

		bufTotalSize = mul(bufMaxSizePolynomials, ztMinusSi[i], bufTotalSize)
		bufTotalSize = mulByConstant(bufTotalSize, accGamma)
		for j := 0; j < len(bufTotalSize); j++ {
			f[j].Add(&f[j], &bufTotalSize[j])
		}

		accGamma.Mul(&accGamma, &gamma)
		setZero(bufMaxSizePolynomials)
	}

	zt := buildVanishingPoly(flatten(points))
	w := div(f, zt) // cf https://eprint.iacr.org/2020/081.pdf page 11 for notation page 11 for notation
	res.W, err = kzg.Commit(w, pk)
	if err != nil {
		return res, err
	}

	// derive z
	z, err := deriveChallenge("z", nil, []kzg.Digest{res.W}, fs)
	if err != nil {
		return res, err
	}

	// compute L = ∑ᵢγⁱZ_{T\Sᵢ}(z)(fᵢ-rᵢ(z))-Z_{T}(z)W
	accGamma.SetOne()
	var gammaiZtMinusSiZ fr.Element
	l := make([]fr.Element, totalSize) // cf https://eprint.iacr.org/2020/081.pdf page 11 for notation page 11 for notation
	for i := 0; i < len(polynomials); i++ {

		ztMinusSiZ := eval(ztMinusSi[i], z)          // Z_{T\Sᵢ}(z)
		gammaiZtMinusSiZ.Mul(&accGamma, &ztMinusSiZ) // γⁱZ_{T\Sᵢ}(z)

		copy(bufMaxSizePolynomials, polynomials[i])
		riz := eval(ri[i], z)
		bufMaxSizePolynomials[0].Sub(&bufMaxSizePolynomials[0], &riz)                // (fᵢ-rᵢ(z))
		mulByConstant(bufMaxSizePolynomials[:len(polynomials[i])], gammaiZtMinusSiZ) // γⁱZ_{T\Sᵢ}(z)(fᵢ-rᵢ(z))
		for j := 0; j < len(bufMaxSizePolynomials); j++ {
			l[j].Add(&l[j], &bufMaxSizePolynomials[j])
		}

		setZero(bufMaxSizePolynomials)
		accGamma.Mul(&accGamma, &gamma)
	}
	ztz := eval(zt, z)
	setZero(bufTotalSize)
	copy(bufTotalSize, w)
	mulByConstant(bufTotalSize, ztz) // Z_{T}(z)W
	for i := 0; i < totalSize-maxSizePolys; i++ {
		l[totalSize-1-i].Neg(&bufTotalSize[totalSize-1-i])
	}
	for i := 0; i < maxSizePolys; i++ {
		l[i].Sub(&l[i], &bufTotalSize[i])
	} // L <- L-Z_{T}(z)W

	xMinusZ := buildVanishingPoly([]fr.Element{z})
	wPrime := div(l, xMinusZ)

	res.WPrime, err = kzg.Commit(wPrime, pk)
	if err != nil {
		return res, err
	}

	return res, nil
}

// BatchVerify uses proof to check that the commitments correctly open to proof.ClaimedValues
// at points. The order matters: the proof validates that the i-th commitment is correctly opened
// at the i-th point
// dataTranscript is some extra data that might be needed for Fiat Shamir, and is appended at the end
// of the original transcript.
func BatchVerify(proof OpeningProof, digests []kzg.Digest, points [][]fr.Element, hf hash.Hash, vk kzg.VerifyingKey, dataTranscript ...[]byte) error {

	if len(digests) != len(proof.ClaimedValues) {
		return ErrInvalidNumberOfPoints
	}
	if len(digests) != len(points) {
		return ErrInvalidNumberOfPoints
	}

	// transcript
	fs := fiatshamir.NewTranscript(hf, "gamma", "z")

	// derive γ
	gamma, err := deriveChallenge("gamma", points, digests, fs, dataTranscript...)
	if err != nil {
		return err
	}

	// derive z
	// TODO seems ok that z depend only on W, need to check that carefully
	z, err := deriveChallenge("z", nil, []kzg.Digest{proof.W}, fs)
	if err != nil {
		return err
	}

	// check that e(F + zW', [1]_{2})=e(W',[x]_{2})
	// where F = ∑ᵢγⁱZ_{T\xᵢ}[Com]_{i}-[∑ᵢγⁱZ_{T\xᵢ}(z)fᵢ(z)]_{1}-Z_{T}(z)[W]
	var sumGammaiZTminusSiRiz, tmp, accGamma fr.Element
	nbInstances := len(points)
	gammaiZTminusSiz := make([]fr.Element, nbInstances)
	accGamma.SetOne()
	ri := make([][]fr.Element, nbInstances)
	for i := 0; i < len(points); i++ {

		ztMinusSi := buildZtMinusSi(points, i)                   // Z_{T-S_{i}}(X)
		gammaiZTminusSiz[i] = eval(ztMinusSi, z)                 // Z_{T-S_{i}}(z)
		gammaiZTminusSiz[i].Mul(&accGamma, &gammaiZTminusSiz[i]) // \gamma^{i} Z_{T-S_{i}}(z)

		ri[i] = interpolate(points[i], proof.ClaimedValues[i])
		riz := eval(ri[i], z)               // r_{i}(z)
		tmp.Mul(&gammaiZTminusSiz[i], &riz) // Z_{T-S_{i}}(z)r_{i}(z)
		sumGammaiZTminusSiRiz.Add(&sumGammaiZTminusSiRiz, &tmp)

		accGamma.Mul(&accGamma, &gamma)
	}

	// ∑ᵢγⁱZ_{T\xᵢ}[Com]_{i}
	config := ecc.MultiExpConfig{}
	var sumGammaiZtMinusSiComi kzg.Digest
	_, err = sumGammaiZtMinusSiComi.MultiExp(digests, gammaiZTminusSiz, config)
	if err != nil {
		return err
	}

	var bufBigInt big.Int

	// [∑ᵢZ_{T\xᵢ}fᵢ(z)]_{1}
	var sumGammaiZTminusSiRizCom kzg.Digest
	var sumGammaiZTminusSiRizBigInt big.Int
	sumGammaiZTminusSiRiz.BigInt(&sumGammaiZTminusSiRizBigInt)
	sumGammaiZTminusSiRizCom.ScalarMultiplication(&vk.G1, &sumGammaiZTminusSiRizBigInt)

	// Z_{T}(z)[W]
	zt := buildVanishingPoly(flatten(points))
	ztz := eval(zt, z)
	var ztW kzg.Digest
	ztz.BigInt(&bufBigInt)
	ztW.ScalarMultiplication(&proof.W, &bufBigInt)

	// F = ∑ᵢγⁱZ_{T\xᵢ}[Com]_{i} - [∑ᵢγⁱZ_{T\xᵢ}fᵢ(z)]_{1} - Z_{T}(z)[W]
	var f kzg.Digest
	f.Sub(&sumGammaiZtMinusSiComi, &sumGammaiZTminusSiRizCom).
		Sub(&f, &ztW)

	// F+zW'
	var zWPrime kzg.Digest
	z.BigInt(&bufBigInt)
	zWPrime.ScalarMultiplication(&proof.WPrime, &bufBigInt)
	f.Add(&f, &zWPrime)
	f.Neg(&f)

	// check that e(F+zW',[1]_{2})=e(W',[x]_{2})
	check, err := bls12377.PairingCheckFixedQ(
		[]bls12377.G1Affine{f, proof.WPrime},
		vk.Lines[:],
	)
	if err != nil {
		return ErrPairingCheck
	}

	if !check {
		return ErrVerifyOpeningProof
	}

	return nil
}

// deriveChallenge derives a challenge using Fiat Shamir to polynomials.
// The arguments are added to the transcript in the order in which they are given.
func deriveChallenge(name string, points [][]fr.Element, digests []kzg.Digest, t *fiatshamir.Transcript, dataTranscript ...[]byte) (fr.Element, error) {

	// derive the challenge gamma, binded to the point and the commitments
	for i := range points {
		for j := range points[i] {
			if err := t.Bind(name, points[i][j].Marshal()); err != nil {
				return fr.Element{}, err
			}
		}
	}
	for i := range digests {
		if err := t.Bind(name, digests[i].Marshal()); err != nil {
			return fr.Element{}, err
		}
	}

	for i := 0; i < len(dataTranscript); i++ {
		if err := t.Bind(name, dataTranscript[i]); err != nil {
			return fr.Element{}, err
		}
	}

	challengeByte, err := t.ComputeChallenge(name)
	if err != nil {
		return fr.Element{}, err
	}
	var challenge fr.Element
	challenge.SetBytes(challengeByte)

	return challenge, nil
}

// ------------------------------
// utils

func flatten(x [][]fr.Element) []fr.Element {
	nbPoints := 0
	for i := 0; i < len(x); i++ {
		nbPoints += len(x[i])
	}
	res := make([]fr.Element, 0, nbPoints)
	for i := 0; i < len(x); i++ {
		res = append(res, x[i]...)
	}
	return res
}

// sets f to zero
func setZero(f []fr.Element) {
	for i := 0; i < len(f); i++ {
		f[i].SetZero()
	}
}

func eval(f []fr.Element, x fr.Element) fr.Element {
	var y fr.Element
	for i := len(f) - 1; i >= 0; i-- {
		y.Mul(&y, &x).Add(&y, &f[i])
	}
	return y
}

// returns γ*f, re-using f
func mulByConstant(f []fr.Element, gamma fr.Element) []fr.Element {
	for i := 0; i < len(f); i++ {
		f[i].Mul(&f[i], &gamma)
	}
	return f
}

// computes f <- (x-a)*f
// memory of f is re used, need to pass a copy to not modify it
func multiplyLinearFactor(f []fr.Element, a fr.Element) []fr.Element {
	s := len(f)
	var tmp fr.Element
	f = append(f, fr.NewElement(0))
	f[s] = f[s-1]
	for i := s - 1; i >= 1; i-- {
		tmp.Mul(&f[i], &a)
		f[i].Sub(&f[i-1], &tmp)
	}
	f[0].Mul(&f[0], &a).Neg(&f[0])
	return f
}

// returns S_{T\Sᵢ} where Sᵢ=x[i]
func buildZtMinusSi(x [][]fr.Element, i int) []fr.Element {
	nbPoints := 0
	for i := 0; i < len(x); i++ {
		nbPoints += len(x[i])
	}
	bufPoints := make([]fr.Element, 0, nbPoints-len(x[i]))
	for j := 0; j < i; j++ {
		bufPoints = append(bufPoints, x[j]...)
	}
	for j := i + 1; j < len(x); j++ {
		bufPoints = append(bufPoints, x[j]...)
	}
	ztMinusSi := buildVanishingPoly(bufPoints)
	return ztMinusSi
}

// returns πᵢ(X-xᵢ)
func buildVanishingPoly(x []fr.Element) []fr.Element {
	res := make([]fr.Element, 1, len(x)+1)
	res[0].SetOne()
	for i := 0; i < len(x); i++ {
		res = multiplyLinearFactor(res, x[i])
	}
	return res
}

// returns f such that f(xᵢ)=yᵢ, x and y are assumed to be of the same size
func interpolate(x, y []fr.Element) []fr.Element {

	res := make([]fr.Element, len(x))
	for i := 0; i < len(x); i++ {
		li := buildLagrangeFromDomain(x, i)
		li = mulByConstant(li, y[i])
		for j := 0; j < len(x); j++ {
			res[j].Add(&res[j], &li[j])
		}
	}

	return res
}

// returns f such that f(xⱼ)=δⁱⱼ
func buildLagrangeFromDomain(x []fr.Element, i int) []fr.Element {
	xx := make([]fr.Element, len(x)-1)
	copy(xx, x[:i])
	copy(xx[i:], x[i+1:])
	res := buildVanishingPoly(xx)
	d := eval(res, x[i])
	d.Inverse(&d)
	res = mulByConstant(res, d)
	return res
}

// returns f-g, the memory of f is re used, deg(g) << deg(f) here
func sub(f, g []fr.Element) []fr.Element {
	for i := 0; i < len(g); i++ {
		f[i].Sub(&f[i], &g[i])
	}
	return f
}

// returns f*g using naive multiplication
// deg(f)>>deg(g), deg(small) =~ 10 max
// buf is used as a buffer and should not be f or g
// f and g are not modified
func mul(f, g []fr.Element, res []fr.Element) []fr.Element {

	sizeRes := len(f) + len(g) - 1
	if len(res) < sizeRes {
		s := make([]fr.Element, sizeRes-len(res))
		res = append(res, s...)
	}
	setZero(res)

	var tmp fr.Element
	for i := 0; i < len(g); i++ {
		for j := 0; j < len(f); j++ {
			tmp.Mul(&f[j], &g[i])
			res[j+i].Add(&res[j+i], &tmp)
		}
	}
	return res
}

// returns f/g (assuming g divides f)
// OK to not use fft if deg(g) is small
// g's leading coefficient is assumed to be 1
// f memory is re-used for the result, need to pass a copy to not modify it
func div(f, g []fr.Element) []fr.Element {
	sizef := len(f)
	sizeg := len(g)
	stop := sizeg - +1
	var t fr.Element
	for i := sizef - 2; i >= stop; i-- {
		for j := 0; j < sizeg-1; j++ {
			t.Mul(&f[i+1], &g[sizeg-2-j])
			f[i-j].Sub(&f[i-j], &t)
		}
	}
	return f[sizeg-1:]
}
