pub const ext = @import("ext.zig");
const xmlb = @This();

const std = @import("std");
const compat = @import("compat");
const gio = @import("gio2");
const gobject = @import("gobject2");
const glib = @import("glib2");
const gmodule = @import("gmodule2");
pub const Builder = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = xmlb.BuilderClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new builder.
    ///
    /// The `xmlb.Silo` returned by the methods of this `xmlb.Builder` will use the
    /// thread-default main context at the time of calling this function for its
    /// future signal emissions.
    extern fn xb_builder_new() *xmlb.Builder;
    pub const new = xb_builder_new;

    /// Adds a function that will get run on every `xmlb.BuilderNode` compile creates
    /// for the silo. This is run after all the `xmlb.BuilderSource` fixups have been
    /// run.
    extern fn xb_builder_add_fixup(p_self: *Builder, p_fixup: *xmlb.BuilderFixup) void;
    pub const addFixup = xb_builder_add_fixup;

    /// Adds a locale to the builder. Locales added first will be prioritised over
    /// locales added later.
    extern fn xb_builder_add_locale(p_self: *Builder, p_locale: [*:0]const u8) void;
    pub const addLocale = xb_builder_add_locale;

    /// Adds the GUID to the internal correctness hash.
    extern fn xb_builder_append_guid(p_self: *Builder, p_guid: [*:0]const u8) void;
    pub const appendGuid = xb_builder_append_guid;

    /// Compiles a `xmlb.Silo`.
    extern fn xb_builder_compile(p_self: *Builder, p_flags: xmlb.BuilderCompileFlags, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*xmlb.Silo;
    pub const compile = xb_builder_compile;

    /// Ensures `file` is up to date, and returns a compiled `xmlb.Silo`.
    ///
    /// If `silo` is being used by a query (e.g. in another thread) then all node
    /// data is immediately invalid.
    ///
    /// The returned `xmlb.Silo` will use the thread-default main context at the time of
    /// calling this function for its future signal emissions.
    extern fn xb_builder_ensure(p_self: *Builder, p_file: *gio.File, p_flags: xmlb.BuilderCompileFlags, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*xmlb.Silo;
    pub const ensure = xb_builder_ensure;

    /// Adds a node tree to the builder.
    ///
    /// If you are manually adding dynamic data sourced from a non-static source then you MUST use
    /// `xmlb.Builder.appendGuid` with the appropriate GUID value, e.g. the file name and mtime.
    ///
    /// If no appropriate value is available, the caller can use something like:
    ///
    ///     g_autofree gchar *tmp = xb_builder_node_export(bn, XB_NODE_EXPORT_FLAG_NONE, NULL);
    ///     xb_builder_append_guid(builder, tmp);
    ///
    /// Failure to include an appropriate GUID value would allow an out-of-data silo to be used.
    extern fn xb_builder_import_node(p_self: *Builder, p_bn: *xmlb.BuilderNode) void;
    pub const importNode = xb_builder_import_node;

    /// Adds a `xmlb.BuilderSource` to the `xmlb.Builder`.
    extern fn xb_builder_import_source(p_self: *Builder, p_source: *xmlb.BuilderSource) void;
    pub const importSource = xb_builder_import_source;

    /// Enables or disables the collection of profiling data.
    extern fn xb_builder_set_profile_flags(p_self: *Builder, p_profile_flags: xmlb.SiloProfileFlags) void;
    pub const setProfileFlags = xb_builder_set_profile_flags;

    extern fn xb_builder_get_type() usize;
    pub const getGObjectType = xb_builder_get_type;

    extern fn g_object_ref(p_self: *xmlb.Builder) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *xmlb.Builder) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Builder, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const BuilderFixup = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = xmlb.BuilderFixupClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a function that will get run on every `xmlb.BuilderNode` compile creates.
    extern fn xb_builder_fixup_new(p_id: [*:0]const u8, p_func: xmlb.BuilderFixupFunc, p_user_data: ?*anyopaque, p_user_data_free: ?glib.DestroyNotify) *xmlb.BuilderFixup;
    pub const new = xb_builder_fixup_new;

    /// Gets the maximum depth used for this fixup, if each node is being visited.
    extern fn xb_builder_fixup_get_max_depth(p_self: *BuilderFixup) c_int;
    pub const getMaxDepth = xb_builder_fixup_get_max_depth;

    /// Sets the maximum depth used for this fixup. Use a `max_depth` of 0 to only
    /// visit the root node.
    ///
    /// Setting a maximum depth may increase performance considerably if using
    /// fixup functions on large and deeply nested XML files.
    extern fn xb_builder_fixup_set_max_depth(p_self: *BuilderFixup, p_max_depth: c_int) void;
    pub const setMaxDepth = xb_builder_fixup_set_max_depth;

    extern fn xb_builder_fixup_get_type() usize;
    pub const getGObjectType = xb_builder_fixup_get_type;

    extern fn g_object_ref(p_self: *xmlb.BuilderFixup) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *xmlb.BuilderFixup) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *BuilderFixup, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const BuilderNode = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = xmlb.BuilderNodeClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new builder node.
    extern fn xb_builder_node_new(p_element: [*:0]const u8) *xmlb.BuilderNode;
    pub const new = xb_builder_node_new;

    /// Adds a child builder node.
    extern fn xb_builder_node_add_child(p_self: *BuilderNode, p_child: *xmlb.BuilderNode) void;
    pub const addChild = xb_builder_node_add_child;

    /// Adds a flag to the builder node.
    extern fn xb_builder_node_add_flag(p_self: *BuilderNode, p_flag: xmlb.BuilderNodeFlags) void;
    pub const addFlag = xb_builder_node_add_flag;

    /// Adds a token to the builder node.
    extern fn xb_builder_node_add_token(p_self: *BuilderNode, p_token: [*:0]const u8) void;
    pub const addToken = xb_builder_node_add_token;

    /// Gets the depth of the node tree, where 0 is the root node.
    extern fn xb_builder_node_depth(p_self: *BuilderNode) c_uint;
    pub const depth = xb_builder_node_depth;

    /// Exports the node to XML.
    extern fn xb_builder_node_export(p_self: *BuilderNode, p_flags: xmlb.NodeExportFlags, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const @"export" = xb_builder_node_export;

    /// Gets an attribute from the builder node.
    extern fn xb_builder_node_get_attr(p_self: *BuilderNode, p_name: [*:0]const u8) [*:0]const u8;
    pub const getAttr = xb_builder_node_get_attr;

    /// Gets an attribute from the builder node.
    extern fn xb_builder_node_get_attr_as_uint(p_self: *BuilderNode, p_name: [*:0]const u8) u64;
    pub const getAttrAsUint = xb_builder_node_get_attr_as_uint;

    /// Finds a child builder node by the element name, and optionally text value.
    extern fn xb_builder_node_get_child(p_self: *BuilderNode, p_element: [*:0]const u8, p_text: ?[*:0]const u8) *xmlb.BuilderNode;
    pub const getChild = xb_builder_node_get_child;

    /// Gets the children of the builder node.
    extern fn xb_builder_node_get_children(p_self: *BuilderNode) *glib.PtrArray;
    pub const getChildren = xb_builder_node_get_children;

    /// Gets the element from the builder node.
    extern fn xb_builder_node_get_element(p_self: *BuilderNode) [*:0]const u8;
    pub const getElement = xb_builder_node_get_element;

    /// Gets the first child of the builder node.
    extern fn xb_builder_node_get_first_child(p_self: *BuilderNode) *xmlb.BuilderNode;
    pub const getFirstChild = xb_builder_node_get_first_child;

    /// Gets the last child of the builder node.
    extern fn xb_builder_node_get_last_child(p_self: *BuilderNode) *xmlb.BuilderNode;
    pub const getLastChild = xb_builder_node_get_last_child;

    /// Gets the parent node for the current node.
    extern fn xb_builder_node_get_parent(p_self: *BuilderNode) *xmlb.BuilderNode;
    pub const getParent = xb_builder_node_get_parent;

    /// Gets the tail from the builder node.
    extern fn xb_builder_node_get_tail(p_self: *BuilderNode) [*:0]const u8;
    pub const getTail = xb_builder_node_get_tail;

    /// Gets the text from the builder node.
    extern fn xb_builder_node_get_text(p_self: *BuilderNode) [*:0]const u8;
    pub const getText = xb_builder_node_get_text;

    /// Gets the text from the builder node.
    extern fn xb_builder_node_get_text_as_uint(p_self: *BuilderNode) u64;
    pub const getTextAsUint = xb_builder_node_get_text_as_uint;

    /// Gets the tokens of the builder node.
    extern fn xb_builder_node_get_tokens(p_self: *BuilderNode) ?*glib.PtrArray;
    pub const getTokens = xb_builder_node_get_tokens;

    /// Checks a flag on the builder node.
    extern fn xb_builder_node_has_flag(p_self: *BuilderNode, p_flag: xmlb.BuilderNodeFlags) c_int;
    pub const hasFlag = xb_builder_node_has_flag;

    /// Creates a new builder node.
    extern fn xb_builder_node_insert(p_parent: *BuilderNode, p_element: [*:0]const u8, ...) *xmlb.BuilderNode;
    pub const insert = xb_builder_node_insert;

    /// Creates a new builder node with optional node text.
    extern fn xb_builder_node_insert_text(p_parent: *BuilderNode, p_element: [*:0]const u8, p_text: ?[*:0]const u8, ...) void;
    pub const insertText = xb_builder_node_insert_text;

    /// Removes an attribute from the builder node.
    extern fn xb_builder_node_remove_attr(p_self: *BuilderNode, p_name: [*:0]const u8) void;
    pub const removeAttr = xb_builder_node_remove_attr;

    /// Removes a child builder node.
    extern fn xb_builder_node_remove_child(p_self: *BuilderNode, p_child: *xmlb.BuilderNode) void;
    pub const removeChild = xb_builder_node_remove_child;

    /// Adds an attribute to the builder node.
    extern fn xb_builder_node_set_attr(p_self: *BuilderNode, p_name: [*:0]const u8, p_value: [*:0]const u8) void;
    pub const setAttr = xb_builder_node_set_attr;

    /// Sets the element name on the builder node.
    extern fn xb_builder_node_set_element(p_self: *BuilderNode, p_element: [*:0]const u8) void;
    pub const setElement = xb_builder_node_set_element;

    /// Sets the tail on the builder node.
    extern fn xb_builder_node_set_tail(p_self: *BuilderNode, p_tail: ?[*:0]const u8, p_tail_len: isize) void;
    pub const setTail = xb_builder_node_set_tail;

    /// Sets the text on the builder node.
    extern fn xb_builder_node_set_text(p_self: *BuilderNode, p_text: ?[*:0]const u8, p_text_len: isize) void;
    pub const setText = xb_builder_node_set_text;

    /// Sorts the node children using a custom sort function.
    extern fn xb_builder_node_sort_children(p_self: *BuilderNode, p_func: xmlb.BuilderNodeSortFunc, p_user_data: ?*anyopaque) void;
    pub const sortChildren = xb_builder_node_sort_children;

    /// Tokenize text added with `xmlb.BuilderNode.setText`.
    ///
    /// When searching, libxmlb often has to tokenize strings before they can be
    /// compared. This is done in the "fast path" and makes searching for non-ASCII
    /// text much slower.
    ///
    /// Adding the tokens to the deduplicated string table allows much faster
    /// searching at the expense of a ~5% size increase of the silo.
    ///
    /// This function adds all valid UTF-8 and ASCII search words generated from
    /// the value of `xmlb.BuilderNode.setText`.
    ///
    /// The transliteration locale (e.g. `en_GB`) is read from the `xml:lang`
    /// node attribute if set.
    extern fn xb_builder_node_tokenize_text(p_self: *BuilderNode) void;
    pub const tokenizeText = xb_builder_node_tokenize_text;

    /// Traverses a tree starting from `self`. It calls the given function for each
    /// node visited.
    ///
    /// The traversal can be halted at any point by returning TRUE from `func`.
    extern fn xb_builder_node_traverse(p_self: *BuilderNode, p_order: glib.TraverseType, p_flags: glib.TraverseFlags, p_max_depth: c_int, p_func: xmlb.BuilderNodeTraverseFunc, p_user_data: ?*anyopaque) void;
    pub const traverse = xb_builder_node_traverse;

    /// Unlinks a `xmlb.BuilderNode` from a tree, resulting in two separate trees.
    ///
    /// This should not be used from the function called by `xmlb.BuilderNode.traverse`
    /// otherwise the entire tree will not be traversed.
    ///
    /// Instead use xb_builder_node_add_flag(bn,XB_BUILDER_NODE_FLAG_IGNORE);
    extern fn xb_builder_node_unlink(p_self: *BuilderNode) void;
    pub const unlink = xb_builder_node_unlink;

    extern fn xb_builder_node_get_type() usize;
    pub const getGObjectType = xb_builder_node_get_type;

    extern fn g_object_ref(p_self: *xmlb.BuilderNode) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *xmlb.BuilderNode) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *BuilderNode, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const BuilderSource = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = xmlb.BuilderSourceClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new builder source.
    extern fn xb_builder_source_new() *xmlb.BuilderSource;
    pub const new = xb_builder_source_new;

    /// Adds a function that can be used to convert streams loaded with
    /// `xmlb.BuilderSource.loadXml`.
    ///
    /// This will decompress multiple layers of content, for instance decompressing a gzip stream into a
    /// different content type that can then be parsed. Use `xmlb.BuilderSource.addSimpleAdapter` when
    /// this recursive behaviour is not desired.
    extern fn xb_builder_source_add_adapter(p_self: *BuilderSource, p_content_types: [*:0]const u8, p_func: xmlb.BuilderSourceAdapterFunc, p_user_data: ?*anyopaque, p_user_data_free: ?glib.DestroyNotify) void;
    pub const addAdapter = xb_builder_source_add_adapter;

    /// Adds a function that will get run on every `xmlb.BuilderNode` compile creates
    /// with this source.
    extern fn xb_builder_source_add_fixup(p_self: *BuilderSource, p_fixup: *xmlb.BuilderFixup) void;
    pub const addFixup = xb_builder_source_add_fixup;

    /// Adds a function that can be used to convert streams loaded with
    /// `xmlb.BuilderSource.loadXml`.
    ///
    /// This function is similar to `xmlb.BuilderSource.addAdapter` but is limited to one "layer" of
    /// content, for instance handling application/xml or a single simple type added using
    /// `xmlb.BuilderSource.addAdapter`.
    extern fn xb_builder_source_add_simple_adapter(p_self: *BuilderSource, p_content_types: [*:0]const u8, p_func: xmlb.BuilderSourceAdapterFunc, p_user_data: ?*anyopaque, p_user_data_free: ?glib.DestroyNotify) void;
    pub const addSimpleAdapter = xb_builder_source_add_simple_adapter;

    /// Loads XML data and begins to build a `xmlb.Silo`.
    extern fn xb_builder_source_load_bytes(p_self: *BuilderSource, p_bytes: *glib.Bytes, p_flags: xmlb.BuilderSourceFlags, p_error: ?*?*glib.Error) c_int;
    pub const loadBytes = xb_builder_source_load_bytes;

    /// Loads an optionally compressed XML file to build a `xmlb.Silo`.
    extern fn xb_builder_source_load_file(p_self: *BuilderSource, p_file: *gio.File, p_flags: xmlb.BuilderSourceFlags, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const loadFile = xb_builder_source_load_file;

    /// Loads XML data and begins to build a `xmlb.Silo`.
    extern fn xb_builder_source_load_xml(p_self: *BuilderSource, p_xml: [*:0]const u8, p_flags: xmlb.BuilderSourceFlags, p_error: ?*?*glib.Error) c_int;
    pub const loadXml = xb_builder_source_load_xml;

    /// Sets an optional information metadata node on the root node.
    extern fn xb_builder_source_set_info(p_self: *BuilderSource, p_info: ?*xmlb.BuilderNode) void;
    pub const setInfo = xb_builder_source_set_info;

    /// Sets an optional prefix on the root node. This makes any nodes added
    /// using this source reside under a common shared parent node.
    extern fn xb_builder_source_set_prefix(p_self: *BuilderSource, p_prefix: ?[*:0]const u8) void;
    pub const setPrefix = xb_builder_source_set_prefix;

    extern fn xb_builder_source_get_type() usize;
    pub const getGObjectType = xb_builder_source_get_type;

    extern fn g_object_ref(p_self: *xmlb.BuilderSource) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *xmlb.BuilderSource) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *BuilderSource, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const BuilderSourceCtx = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = xmlb.BuilderSourceCtxClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Returns the data currently being processed.
    ///
    /// If the `xmlb.BuilderSourceCtx` is backed by a file, the returned `glib.Bytes` may be
    /// memory-mapped, and the backing file must not be modified until the `glib.Bytes` is
    /// destroyed.
    extern fn xb_builder_source_ctx_get_bytes(p_self: *BuilderSourceCtx, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) ?*glib.Bytes;
    pub const getBytes = xb_builder_source_ctx_get_bytes;

    /// Returns the basename of the file currently being processed.
    extern fn xb_builder_source_ctx_get_filename(p_self: *BuilderSourceCtx) ?[*:0]const u8;
    pub const getFilename = xb_builder_source_ctx_get_filename;

    /// Returns the input stream currently being processed.
    extern fn xb_builder_source_ctx_get_stream(p_self: *BuilderSourceCtx) *gio.InputStream;
    pub const getStream = xb_builder_source_ctx_get_stream;

    extern fn xb_builder_source_ctx_get_type() usize;
    pub const getGObjectType = xb_builder_source_ctx_get_type;

    extern fn g_object_ref(p_self: *xmlb.BuilderSourceCtx) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *xmlb.BuilderSourceCtx) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *BuilderSourceCtx, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const Machine = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = xmlb.MachineClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a new virtual machine.
    extern fn xb_machine_new() *xmlb.Machine;
    pub const new = xb_machine_new;

    /// Adds a new function to the virtual machine. Registered functions can then be
    /// used as methods.
    ///
    /// The `method_cb` must not modify the stack it’s passed unless it’s going to
    /// succeed. In particular, if a method call is not optimisable, it must not
    /// modify the stack it’s passed.
    ///
    /// You need to add a custom function using `xmlb.Machine.addMethod` before using
    /// methods that may reference it, for example `xmlb.Machine.addOpcodeFixup`.
    extern fn xb_machine_add_method(p_self: *Machine, p_name: [*:0]const u8, p_n_opcodes: c_uint, p_method_cb: xmlb.MachineMethodFunc, p_user_data: ?*anyopaque, p_user_data_free: ?glib.DestroyNotify) void;
    pub const addMethod = xb_machine_add_method;

    /// Adds an opcode fixup. Fixups can be used to optimize the stack of opcodes or
    /// to add support for a nonstandard feature, for instance supporting missing
    /// attributes to functions.
    extern fn xb_machine_add_opcode_fixup(p_self: *Machine, p_opcodes_sig: [*:0]const u8, p_fixup_cb: xmlb.MachineOpcodeFixupFunc, p_user_data: ?*anyopaque, p_user_data_free: ?glib.DestroyNotify) void;
    pub const addOpcodeFixup = xb_machine_add_opcode_fixup;

    /// Adds a new operator to the virtual machine. Operators can then be used
    /// instead of explicit methods like ``eq``.
    ///
    /// You need to add a custom operator using `xmlb.Machine.addOperator` before
    /// using `xmlb.Machine.parse`. Common operators like `<=` and `=` are built-in
    /// and do not have to be added manually.
    extern fn xb_machine_add_operator(p_self: *Machine, p_str: [*:0]const u8, p_name: [*:0]const u8) void;
    pub const addOperator = xb_machine_add_operator;

    /// Adds a text handler. This allows the virtual machine to support nonstandard
    /// encoding or shorthand mnemonics for standard functions.
    extern fn xb_machine_add_text_handler(p_self: *Machine, p_handler_cb: xmlb.MachineTextHandlerFunc, p_user_data: ?*anyopaque, p_user_data_free: ?glib.DestroyNotify) void;
    pub const addTextHandler = xb_machine_add_text_handler;

    /// Gets the maximum stack size used for the machine.
    extern fn xb_machine_get_stack_size(p_self: *Machine) c_uint;
    pub const getStackSize = xb_machine_get_stack_size;

    /// Initialises a stack allocated `xmlb.Opcode` for a registered function.
    /// Some standard functions are registered by default, for instance `eq` or `ge`.
    /// Other functions have to be added using `xmlb.Machine.addMethod`.
    extern fn xb_machine_opcode_func_init(p_self: *Machine, p_opcode: *xmlb.Opcode, p_func_name: [*:0]const u8) c_int;
    pub const opcodeFuncInit = xb_machine_opcode_func_init;

    /// Parses an XPath predicate. Not all of XPath 1.0 or XPath 1.0 is supported,
    /// and new functions and mnemonics can be added using `xmlb.Machine.addMethod`
    /// and `xmlb.Machine.addTextHandler`.
    extern fn xb_machine_parse(p_self: *Machine, p_text: [*:0]const u8, p_text_len: isize, p_error: ?*?*glib.Error) ?*xmlb.Stack;
    pub const parse = xb_machine_parse;

    /// Parses an XPath predicate. Not all of XPath 1.0 or XPath 1.0 is supported,
    /// and new functions and mnemonics can be added using `xmlb.Machine.addMethod`
    /// and `xmlb.Machine.addTextHandler`.
    extern fn xb_machine_parse_full(p_self: *Machine, p_text: [*:0]const u8, p_text_len: isize, p_flags: xmlb.MachineParseFlags, p_error: ?*?*glib.Error) ?*xmlb.Stack;
    pub const parseFull = xb_machine_parse_full;

    /// Runs a set of opcodes on the virtual machine.
    ///
    /// It is safe to call this function from a different thread to the one that
    /// created the `xmlb.Machine`.
    extern fn xb_machine_run(p_self: *Machine, p_opcodes: *xmlb.Stack, p_result: *c_int, p_exec_data: ?*anyopaque, p_error: ?*?*glib.Error) c_int;
    pub const run = xb_machine_run;

    /// Runs a set of opcodes on the virtual machine, using the bound values given in
    /// `bindings` to substitute for bound opcodes.
    ///
    /// It is safe to call this function from a different thread to the one that
    /// created the `xmlb.Machine`.
    extern fn xb_machine_run_with_bindings(p_self: *Machine, p_opcodes: *xmlb.Stack, p_bindings: ?*xmlb.ValueBindings, p_result: *c_int, p_exec_data: ?*anyopaque, p_error: ?*?*glib.Error) c_int;
    pub const runWithBindings = xb_machine_run_with_bindings;

    /// Sets the debug level of the virtual machine.
    extern fn xb_machine_set_debug_flags(p_self: *Machine, p_flags: xmlb.MachineDebugFlags) void;
    pub const setDebugFlags = xb_machine_set_debug_flags;

    /// Sets the maximum stack size used for the machine.
    ///
    /// The stack size will be affective for new jobs started with `xmlb.Machine.run`
    /// and `xmlb.Machine.parse`.
    extern fn xb_machine_set_stack_size(p_self: *Machine, p_stack_size: c_uint) void;
    pub const setStackSize = xb_machine_set_stack_size;

    /// Pops an opcode from the stack.
    extern fn xb_machine_stack_pop(p_self: *Machine, p_stack: *xmlb.Stack, p_opcode_out: ?*xmlb.Opcode, p_error: ?*?*glib.Error) c_int;
    pub const stackPop = xb_machine_stack_pop;

    /// Pushes a new empty opcode onto the end of the stack. A pointer to the opcode
    /// is returned in `opcode_out` so that the caller can initialise it.
    ///
    /// If the stack reaches its maximum size, `G_IO_ERROR_NO_SPACE` will be returned.
    extern fn xb_machine_stack_push(p_self: *Machine, p_stack: *xmlb.Stack, p_opcode_out: ?**xmlb.Opcode, p_error: ?*?*glib.Error) c_int;
    pub const stackPush = xb_machine_stack_push;

    /// Adds an integer literal to the stack.
    ///
    /// Errors are as for `xmlb.Machine.stackPush`.
    extern fn xb_machine_stack_push_integer(p_self: *Machine, p_stack: *xmlb.Stack, p_val: u32, p_error: ?*?*glib.Error) c_int;
    pub const stackPushInteger = xb_machine_stack_push_integer;

    /// Adds a text literal to the stack, copying `str`.
    ///
    /// Errors are as for `xmlb.Machine.stackPush`.
    extern fn xb_machine_stack_push_text(p_self: *Machine, p_stack: *xmlb.Stack, p_str: [*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const stackPushText = xb_machine_stack_push_text;

    /// Adds static text literal to the stack.
    ///
    /// Errors are as for `xmlb.Machine.stackPush`.
    extern fn xb_machine_stack_push_text_static(p_self: *Machine, p_stack: *xmlb.Stack, p_str: [*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const stackPushTextStatic = xb_machine_stack_push_text_static;

    /// Adds a stolen text literal to the stack.
    ///
    /// Errors are as for `xmlb.Machine.stackPush`.
    extern fn xb_machine_stack_push_text_steal(p_self: *Machine, p_stack: *xmlb.Stack, p_str: [*:0]u8, p_error: ?*?*glib.Error) c_int;
    pub const stackPushTextSteal = xb_machine_stack_push_text_steal;

    extern fn xb_machine_get_type() usize;
    pub const getGObjectType = xb_machine_get_type;

    extern fn g_object_ref(p_self: *xmlb.Machine) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *xmlb.Machine) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Machine, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const Node = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = xmlb.NodeClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Exports the node back to XML.
    extern fn xb_node_export(p_self: *Node, p_flags: xmlb.NodeExportFlags, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const @"export" = xb_node_export;

    /// Gets some attribute text data for a specific node.
    extern fn xb_node_get_attr(p_self: *Node, p_name: [*:0]const u8) [*:0]const u8;
    pub const getAttr = xb_node_get_attr;

    /// Gets some attribute text data for a specific node.
    extern fn xb_node_get_attr_as_uint(p_self: *Node, p_name: [*:0]const u8) u64;
    pub const getAttrAsUint = xb_node_get_attr_as_uint;

    /// Gets the first child node for the current node.
    extern fn xb_node_get_child(p_self: *Node) *xmlb.Node;
    pub const getChild = xb_node_get_child;

    /// Gets all the children for the current node.
    extern fn xb_node_get_children(p_self: *Node) *glib.PtrArray;
    pub const getChildren = xb_node_get_children;

    /// Gets any data that has been set on the node using `xmlb.Node.setData`.
    ///
    /// This will only work across queries to the associated silo if the silo has
    /// its `xmlb.Silo.properties.enable`-node-cache property set to `TRUE`. Otherwise a new `xmlb.Node`
    /// may be constructed for future queries which return the same element as a
    /// result.
    extern fn xb_node_get_data(p_self: *Node, p_key: [*:0]const u8) *glib.Bytes;
    pub const getData = xb_node_get_data;

    /// Gets the depth of the node to a root.
    extern fn xb_node_get_depth(p_self: *Node) c_uint;
    pub const getDepth = xb_node_get_depth;

    /// Gets the element name for a specific node.
    extern fn xb_node_get_element(p_self: *Node) [*:0]const u8;
    pub const getElement = xb_node_get_element;

    /// Gets the next sibling node for the current node.
    extern fn xb_node_get_next(p_self: *Node) *xmlb.Node;
    pub const getNext = xb_node_get_next;

    /// Gets the parent node for the current node.
    extern fn xb_node_get_parent(p_self: *Node) *xmlb.Node;
    pub const getParent = xb_node_get_parent;

    /// Gets the root node for the node.
    extern fn xb_node_get_root(p_self: *Node) *xmlb.Node;
    pub const getRoot = xb_node_get_root;

    /// Gets the `xmlb.Silo` for the node.
    extern fn xb_node_get_silo(p_self: *Node) *xmlb.Silo;
    pub const getSilo = xb_node_get_silo;

    /// Gets the tail data for a specific node.
    extern fn xb_node_get_tail(p_self: *Node) [*:0]const u8;
    pub const getTail = xb_node_get_tail;

    /// Gets the text data for a specific node.
    extern fn xb_node_get_text(p_self: *Node) [*:0]const u8;
    pub const getText = xb_node_get_text;

    /// Gets some attribute text data for a specific node.
    extern fn xb_node_get_text_as_uint(p_self: *Node) u64;
    pub const getTextAsUint = xb_node_get_text_as_uint;

    /// Searches the silo using an XPath query, returning up to `limit` results.
    ///
    /// It is safe to call this function from a different thread to the one that
    /// created the `xmlb.Silo`.
    ///
    /// Please note: Only a subset of XPath is supported.
    extern fn xb_node_query(p_self: *Node, p_xpath: [*:0]const u8, p_limit: c_uint, p_error: ?*?*glib.Error) ?*glib.PtrArray;
    pub const query = xb_node_query;

    /// Searches the node using an XPath query, returning up to one result.
    ///
    /// It is safe to call this function from a different thread to the one that
    /// created the `xmlb.Silo`.
    ///
    /// Please note: Only a subset of XPath is supported.
    extern fn xb_node_query_attr(p_self: *Node, p_xpath: [*:0]const u8, p_name: [*:0]const u8, p_error: ?*?*glib.Error) ?[*:0]const u8;
    pub const queryAttr = xb_node_query_attr;

    /// Searches the node using an XPath query, returning up to one result.
    ///
    /// It is safe to call this function from a different thread to the one that
    /// created the `xmlb.Silo`.
    ///
    /// Please note: Only a subset of XPath is supported.
    extern fn xb_node_query_attr_as_uint(p_self: *Node, p_xpath: [*:0]const u8, p_name: [*:0]const u8, p_error: ?*?*glib.Error) u64;
    pub const queryAttrAsUint = xb_node_query_attr_as_uint;

    /// Searches the node using an XPath query, returning an XML string of the
    /// result and any children.
    ///
    /// It is safe to call this function from a different thread to the one that
    /// created the `xmlb.Silo`.
    ///
    /// Please note: Only a subset of XPath is supported.
    extern fn xb_node_query_export(p_self: *Node, p_xpath: [*:0]const u8, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const queryExport = xb_node_query_export;

    /// Searches the node using an XPath query, returning up to one result.
    ///
    /// Please note: Only a tiny subset of XPath 1.0 is supported.
    extern fn xb_node_query_first(p_self: *Node, p_xpath: [*:0]const u8, p_error: ?*?*glib.Error) ?*xmlb.Node;
    pub const queryFirst = xb_node_query_first;

    /// Searches the silo using a prepared query, returning up to one result. To
    /// search using a query with bound values, use
    /// `xmlb.Node.queryFirstWithContext`.
    ///
    /// It is safe to call this function from a different thread to the one that
    /// created the `xmlb.Silo`.
    ///
    /// Please note: Only a subset of XPath is supported.
    extern fn xb_node_query_first_full(p_self: *Node, p_query: *xmlb.Query, p_error: ?*?*glib.Error) ?*xmlb.Node;
    pub const queryFirstFull = xb_node_query_first_full;

    /// Searches the silo using a prepared query, returning up to one result.
    ///
    /// It is safe to call this function from a different thread to the one that
    /// created the `xmlb.Silo`.
    ///
    /// Please note: Only a subset of XPath is supported.
    extern fn xb_node_query_first_with_context(p_self: *Node, p_query: *xmlb.Query, p_context: ?*xmlb.QueryContext, p_error: ?*?*glib.Error) ?*xmlb.Node;
    pub const queryFirstWithContext = xb_node_query_first_with_context;

    /// Searches the silo using a prepared query. To search using a query with
    /// bound values, use `xmlb.Node.queryWithContext`.
    ///
    /// It is safe to call this function from a different thread to the one that
    /// created the `xmlb.Silo`.
    ///
    /// Please note: Only a subset of XPath is supported.
    extern fn xb_node_query_full(p_self: *Node, p_query: *xmlb.Query, p_error: ?*?*glib.Error) ?*glib.PtrArray;
    pub const queryFull = xb_node_query_full;

    /// Searches the node using an XPath query, returning up to one result.
    ///
    /// It is safe to call this function from a different thread to the one that
    /// created the `xmlb.Silo`.
    ///
    /// Please note: Only a subset of XPath is supported.
    extern fn xb_node_query_text(p_self: *Node, p_xpath: [*:0]const u8, p_error: ?*?*glib.Error) ?[*:0]const u8;
    pub const queryText = xb_node_query_text;

    /// Searches the node using an XPath query, returning up to one result.
    ///
    /// It is safe to call this function from a different thread to the one that
    /// created the `xmlb.Silo`.
    ///
    /// Please note: Only a subset of XPath is supported.
    extern fn xb_node_query_text_as_uint(p_self: *Node, p_xpath: [*:0]const u8, p_error: ?*?*glib.Error) u64;
    pub const queryTextAsUint = xb_node_query_text_as_uint;

    /// Searches the silo using a prepared query, substituting values from the
    /// bindings in `context` for bound opcodes as needed.
    ///
    /// It is safe to call this function from a different thread to the one that
    /// created the `xmlb.Silo`.
    ///
    /// Please note: Only a subset of XPath is supported.
    extern fn xb_node_query_with_context(p_self: *Node, p_query: *xmlb.Query, p_context: ?*xmlb.QueryContext, p_error: ?*?*glib.Error) ?*glib.PtrArray;
    pub const queryWithContext = xb_node_query_with_context;

    /// Sets some data on the node which can be retrieved using `xmlb.Node.getData`.
    ///
    /// This will only work across queries to the associated silo if the silo has
    /// its `xmlb.Silo.properties.enable`-node-cache property set to `TRUE`. Otherwise a new `xmlb.Node`
    /// may be constructed for future queries which return the same element as a
    /// result.
    extern fn xb_node_set_data(p_self: *Node, p_key: [*:0]const u8, p_data: *glib.Bytes) void;
    pub const setData = xb_node_set_data;

    /// Traverses a tree starting from `self`. It calls the given functions for each
    /// node visited. This allows transmogrification of the source, for instance
    /// converting the XML description to PangoMarkup or even something completely
    /// different like markdown.
    ///
    /// The traversal can be halted at any point by returning TRUE from `func`.
    extern fn xb_node_transmogrify(p_self: *Node, p_func_text: xmlb.NodeTransmogrifyFunc, p_func_tail: xmlb.NodeTransmogrifyFunc, p_user_data: ?*anyopaque) c_int;
    pub const transmogrify = xb_node_transmogrify;

    extern fn xb_node_get_type() usize;
    pub const getGObjectType = xb_node_get_type;

    extern fn g_object_ref(p_self: *xmlb.Node) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *xmlb.Node) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Node, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const Query = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = xmlb.QueryClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {};

    pub const properties = struct {};

    pub const signals = struct {};

    /// Creates a query to be used by `silo`. It may be quicker to create a query
    /// manually and re-use it multiple times.
    extern fn xb_query_new(p_silo: *xmlb.Silo, p_xpath: [*:0]const u8, p_error: ?*?*glib.Error) ?*xmlb.Query;
    pub const new = xb_query_new;

    /// Creates a query to be used by `silo`. It may be quicker to create a query
    /// manually and re-use it multiple times.
    ///
    /// The query will point to strings inside `silo`, so the lifetime of `silo` must
    /// exceed the lifetime of the returned query.
    extern fn xb_query_new_full(p_silo: *xmlb.Silo, p_xpath: [*:0]const u8, p_flags: xmlb.QueryFlags, p_error: ?*?*glib.Error) ?*xmlb.Query;
    pub const newFull = xb_query_new_full;

    /// Assigns a string to a bound value specified using `?`.
    extern fn xb_query_bind_str(p_self: *Query, p_idx: c_uint, p_str: [*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const bindStr = xb_query_bind_str;

    /// Assigns a string to a bound value specified using `?`.
    extern fn xb_query_bind_val(p_self: *Query, p_idx: c_uint, p_val: u32, p_error: ?*?*glib.Error) c_int;
    pub const bindVal = xb_query_bind_val;

    /// Gets the flags used for this query.
    extern fn xb_query_get_flags(p_self: *Query) xmlb.QueryFlags;
    pub const getFlags = xb_query_get_flags;

    /// Gets the results limit on this query, where 0 is 'all'.
    extern fn xb_query_get_limit(p_self: *Query) c_uint;
    pub const getLimit = xb_query_get_limit;

    /// Gets the XPath string that created the query.
    extern fn xb_query_get_xpath(p_self: *Query) [*:0]const u8;
    pub const getXpath = xb_query_get_xpath;

    /// Sets the flags to use for this query.
    extern fn xb_query_set_flags(p_self: *Query, p_flags: xmlb.QueryFlags) void;
    pub const setFlags = xb_query_set_flags;

    /// Sets the results limit on this query, where 0 is 'all'.
    extern fn xb_query_set_limit(p_self: *Query, p_limit: c_uint) void;
    pub const setLimit = xb_query_set_limit;

    extern fn xb_query_get_type() usize;
    pub const getGObjectType = xb_query_get_type;

    extern fn g_object_ref(p_self: *xmlb.Query) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *xmlb.Query) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Query, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const Silo = extern struct {
    pub const Parent = gobject.Object;
    pub const Implements = [_]type{};
    pub const Class = xmlb.SiloClass;
    f_parent_instance: gobject.Object,

    pub const virtual_methods = struct {};

    pub const properties = struct {
        /// Whether to cache all `xmlb.Node` instances ever constructed in a single
        /// cache in the `xmlb.Silo`, so that the same `xmlb.Node` instance is always
        /// returned in query results for a given XPath. This is a form of
        /// memoisation, and allows `xmlb.Node.getData` and `xmlb.Node.setData` to
        /// be used.
        ///
        /// This is enabled by default to preserve compatibility with older
        /// versions of libxmlb, but most clients will want to disable it. It
        /// adds a large memory overhead (no `xmlb.Node` is ever finalised) but
        /// achieves moderately low hit rates for typical XML parsing workloads
        /// where most nodes are accessed only once or twice as they are
        /// processed and then processing moves on to other nodes.
        ///
        /// This property can only be changed before the `xmlb.Silo` is passed
        /// between threads. Changing it is not thread-safe.
        pub const enable_node_cache = struct {
            pub const name = "enable-node-cache";

            pub const Type = c_int;
        };

        pub const guid = struct {
            pub const name = "guid";

            pub const Type = ?[*:0]u8;
        };

        pub const valid = struct {
            pub const name = "valid";

            pub const Type = c_int;
        };
    };

    pub const signals = struct {};

    /// Creates a new silo.
    extern fn xb_silo_new() *xmlb.Silo;
    pub const new = xb_silo_new;

    /// Creates a new silo from an XML string.
    extern fn xb_silo_new_from_xml(p_xml: [*:0]const u8, p_error: ?*?*glib.Error) ?*xmlb.Silo;
    pub const newFromXml = xb_silo_new_from_xml;

    /// Exports the silo back to XML.
    extern fn xb_silo_export(p_self: *Silo, p_flags: xmlb.NodeExportFlags, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const @"export" = xb_silo_export;

    /// Exports the silo back to an XML file.
    extern fn xb_silo_export_file(p_self: *Silo, p_file: *gio.File, p_flags: xmlb.NodeExportFlags, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const exportFile = xb_silo_export_file;

    /// Gets the backing object that created the blob.
    ///
    /// You should never *ever* modify this data.
    extern fn xb_silo_get_bytes(p_self: *Silo) *glib.Bytes;
    pub const getBytes = xb_silo_get_bytes;

    /// Get `xmlb.Silo.properties.enable`-node-cache.
    extern fn xb_silo_get_enable_node_cache(p_self: *Silo) c_int;
    pub const getEnableNodeCache = xb_silo_get_enable_node_cache;

    /// Gets the GUID used to identify this silo.
    extern fn xb_silo_get_guid(p_self: *Silo) [*:0]const u8;
    pub const getGuid = xb_silo_get_guid;

    /// Returns the profiling data. This will only return profiling text if
    /// `xmlb.Silo.setProfileFlags` was used with `XB_SILO_PROFILE_FLAG_APPEND`.
    extern fn xb_silo_get_profile_string(p_self: *Silo) [*:0]const u8;
    pub const getProfileString = xb_silo_get_profile_string;

    /// Gets the root node for the silo. (MIGHT BE MORE).
    extern fn xb_silo_get_root(p_self: *Silo) *xmlb.Node;
    pub const getRoot = xb_silo_get_root;

    /// Gets the number of nodes in the silo.
    extern fn xb_silo_get_size(p_self: *Silo) c_uint;
    pub const getSize = xb_silo_get_size;

    /// Invalidates a silo. Future calls `xmlb.Silo.isValid` will return `FALSE`.
    extern fn xb_silo_invalidate(p_self: *Silo) void;
    pub const invalidate = xb_silo_invalidate;

    /// Checks is the silo is valid. The usual reason the silo is invalidated is
    /// when the backing mmapped file has changed, or one of the imported files have
    /// been modified.
    extern fn xb_silo_is_valid(p_self: *Silo) c_int;
    pub const isValid = xb_silo_is_valid;

    /// Loads a silo from memory location.
    extern fn xb_silo_load_from_bytes(p_self: *Silo, p_blob: *glib.Bytes, p_flags: xmlb.SiloLoadFlags, p_error: ?*?*glib.Error) c_int;
    pub const loadFromBytes = xb_silo_load_from_bytes;

    /// Loads a silo from file.
    extern fn xb_silo_load_from_file(p_self: *Silo, p_file: *gio.File, p_flags: xmlb.SiloLoadFlags, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const loadFromFile = xb_silo_load_from_file;

    /// Create an `xmlb.Query` from the given `xpath` XPath string, or return it from the
    /// query cache in the `xmlb.Silo`.
    ///
    /// `xpath` must be valid: it is a programmer error if creating the query fails
    /// (i.e. if `xmlb.Query.new` returns an error).
    ///
    /// This function is thread-safe.
    extern fn xb_silo_lookup_query(p_self: *Silo, p_xpath: [*:0]const u8) *xmlb.Query;
    pub const lookupQuery = xb_silo_lookup_query;

    /// Searches the silo using an XPath query, returning up to `limit` results.
    ///
    /// It is safe to call this function from a different thread to the one that
    /// created the `xmlb.Silo`.
    ///
    /// Please note: Only a subset of XPath is supported.
    extern fn xb_silo_query(p_self: *Silo, p_xpath: [*:0]const u8, p_limit: c_uint, p_error: ?*?*glib.Error) ?*glib.PtrArray;
    pub const query = xb_silo_query;

    /// Adds the ``attr`` or ``text`` results of a query to the index.
    extern fn xb_silo_query_build_index(p_self: *Silo, p_xpath: [*:0]const u8, p_attr: ?[*:0]const u8, p_error: ?*?*glib.Error) c_int;
    pub const queryBuildIndex = xb_silo_query_build_index;

    /// Searches the silo using an XPath query, returning up to one result.
    ///
    /// It is safe to call this function from a different thread to the one that
    /// created the `xmlb.Silo`.
    ///
    /// Please note: Only a tiny subset of XPath 1.0 is supported.
    extern fn xb_silo_query_first(p_self: *Silo, p_xpath: [*:0]const u8, p_error: ?*?*glib.Error) ?*xmlb.Node;
    pub const queryFirst = xb_silo_query_first;

    /// Searches the silo using an XPath query, returning up to one result.
    ///
    /// It is safe to call this function from a different thread to the one that
    /// created the `xmlb.Silo`.
    ///
    /// Please note: Only a tiny subset of XPath 1.0 is supported.
    extern fn xb_silo_query_first_full(p_self: *Silo, p_query: *xmlb.Query, p_error: ?*?*glib.Error) ?*xmlb.Node;
    pub const queryFirstFull = xb_silo_query_first_full;

    /// Searches the silo using an XPath query, returning up to one result.
    ///
    /// It is safe to call this function from a different thread to the one that
    /// created the `xmlb.Silo`.
    ///
    /// Please note: Only a tiny subset of XPath 1.0 is supported.
    extern fn xb_silo_query_first_with_context(p_self: *Silo, p_query: *xmlb.Query, p_context: ?*xmlb.QueryContext, p_error: ?*?*glib.Error) ?*xmlb.Node;
    pub const queryFirstWithContext = xb_silo_query_first_with_context;

    /// Searches the silo using an XPath query.
    ///
    /// It is safe to call this function from a different thread to the one that
    /// created the `xmlb.Silo`.
    ///
    /// Please note: Only a subset of XPath is supported.
    extern fn xb_silo_query_full(p_self: *Silo, p_query: *xmlb.Query, p_error: ?*?*glib.Error) ?*glib.PtrArray;
    pub const queryFull = xb_silo_query_full;

    /// Searches the silo using an XPath query.
    ///
    /// It is safe to call this function from a different thread to the one that
    /// created the `xmlb.Silo`.
    ///
    /// Please note: Only a subset of XPath is supported.
    extern fn xb_silo_query_with_context(p_self: *Silo, p_query: *xmlb.Query, p_context: ?*xmlb.QueryContext, p_error: ?*?*glib.Error) ?*glib.PtrArray;
    pub const queryWithContext = xb_silo_query_with_context;

    /// Saves a silo to a file.
    extern fn xb_silo_save_to_file(p_self: *Silo, p_file: *gio.File, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const saveToFile = xb_silo_save_to_file;

    /// Set `xmlb.Silo.properties.enable`-node-cache.
    ///
    /// This is not thread-safe, and can only be called before the `xmlb.Silo` is passed
    /// between threads.
    extern fn xb_silo_set_enable_node_cache(p_self: *Silo, p_enable_node_cache: c_int) void;
    pub const setEnableNodeCache = xb_silo_set_enable_node_cache;

    /// Enables or disables the collection of profiling data.
    extern fn xb_silo_set_profile_flags(p_self: *Silo, p_profile_flags: xmlb.SiloProfileFlags) void;
    pub const setProfileFlags = xb_silo_set_profile_flags;

    /// Converts the silo to an internal string representation. This is only
    /// really useful for debugging `xmlb.Silo` itself.
    extern fn xb_silo_to_string(p_self: *Silo, p_error: ?*?*glib.Error) ?[*:0]u8;
    pub const toString = xb_silo_to_string;

    /// Adds a file monitor to the silo. If the file or directory for `file` changes
    /// then the silo will be invalidated.
    ///
    /// The monitor will internally use the `glib.MainContext` which was the thread
    /// default when the `xmlb.Silo` was created, so that `glib.MainContext` must be iterated
    /// for monitoring to work.
    extern fn xb_silo_watch_file(p_self: *Silo, p_file: *gio.File, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) c_int;
    pub const watchFile = xb_silo_watch_file;

    extern fn xb_silo_get_type() usize;
    pub const getGObjectType = xb_silo_get_type;

    extern fn g_object_ref(p_self: *xmlb.Silo) void;
    pub const ref = g_object_ref;

    extern fn g_object_unref(p_self: *xmlb.Silo) void;
    pub const unref = g_object_unref;

    pub fn as(p_instance: *Silo, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const BuilderClass = extern struct {
    pub const Instance = xmlb.Builder;

    f_parent_class: gobject.ObjectClass,
    f__xb_reserved1: ?*const fn () callconv(.C) void,
    f__xb_reserved2: ?*const fn () callconv(.C) void,
    f__xb_reserved3: ?*const fn () callconv(.C) void,
    f__xb_reserved4: ?*const fn () callconv(.C) void,
    f__xb_reserved5: ?*const fn () callconv(.C) void,
    f__xb_reserved6: ?*const fn () callconv(.C) void,
    f__xb_reserved7: ?*const fn () callconv(.C) void,

    pub fn as(p_instance: *BuilderClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const BuilderFixupClass = extern struct {
    pub const Instance = xmlb.BuilderFixup;

    f_parent_class: gobject.ObjectClass,
    f__xb_reserved1: ?*const fn () callconv(.C) void,
    f__xb_reserved2: ?*const fn () callconv(.C) void,
    f__xb_reserved3: ?*const fn () callconv(.C) void,
    f__xb_reserved4: ?*const fn () callconv(.C) void,
    f__xb_reserved5: ?*const fn () callconv(.C) void,
    f__xb_reserved6: ?*const fn () callconv(.C) void,
    f__xb_reserved7: ?*const fn () callconv(.C) void,

    pub fn as(p_instance: *BuilderFixupClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const BuilderNodeClass = extern struct {
    pub const Instance = xmlb.BuilderNode;

    f_parent_class: gobject.ObjectClass,
    f__xb_reserved1: ?*const fn () callconv(.C) void,
    f__xb_reserved2: ?*const fn () callconv(.C) void,
    f__xb_reserved3: ?*const fn () callconv(.C) void,
    f__xb_reserved4: ?*const fn () callconv(.C) void,
    f__xb_reserved5: ?*const fn () callconv(.C) void,
    f__xb_reserved6: ?*const fn () callconv(.C) void,
    f__xb_reserved7: ?*const fn () callconv(.C) void,

    pub fn as(p_instance: *BuilderNodeClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const BuilderSourceClass = extern struct {
    pub const Instance = xmlb.BuilderSource;

    f_parent_class: gobject.ObjectClass,
    f__xb_reserved1: ?*const fn () callconv(.C) void,
    f__xb_reserved2: ?*const fn () callconv(.C) void,
    f__xb_reserved3: ?*const fn () callconv(.C) void,
    f__xb_reserved4: ?*const fn () callconv(.C) void,
    f__xb_reserved5: ?*const fn () callconv(.C) void,
    f__xb_reserved6: ?*const fn () callconv(.C) void,
    f__xb_reserved7: ?*const fn () callconv(.C) void,

    pub fn as(p_instance: *BuilderSourceClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const BuilderSourceCtxClass = extern struct {
    pub const Instance = xmlb.BuilderSourceCtx;

    f_parent_class: gobject.ObjectClass,
    f__xb_reserved1: ?*const fn () callconv(.C) void,
    f__xb_reserved2: ?*const fn () callconv(.C) void,
    f__xb_reserved3: ?*const fn () callconv(.C) void,
    f__xb_reserved4: ?*const fn () callconv(.C) void,
    f__xb_reserved5: ?*const fn () callconv(.C) void,
    f__xb_reserved6: ?*const fn () callconv(.C) void,
    f__xb_reserved7: ?*const fn () callconv(.C) void,

    pub fn as(p_instance: *BuilderSourceCtxClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const MachineClass = extern struct {
    pub const Instance = xmlb.Machine;

    f_parent_class: gobject.ObjectClass,
    f__xb_reserved1: ?*const fn () callconv(.C) void,
    f__xb_reserved2: ?*const fn () callconv(.C) void,
    f__xb_reserved3: ?*const fn () callconv(.C) void,
    f__xb_reserved4: ?*const fn () callconv(.C) void,
    f__xb_reserved5: ?*const fn () callconv(.C) void,
    f__xb_reserved6: ?*const fn () callconv(.C) void,
    f__xb_reserved7: ?*const fn () callconv(.C) void,

    pub fn as(p_instance: *MachineClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// A `xmlb.NodeAttrIter` structure represents an iterator that can be used
/// to iterate over the attributes of a `xmlb.Node`. `xmlb.NodeAttrIter`
/// structures are typically allocated on the stack and then initialized
/// with `xmlb.NodeAttrIter.init`.
///
/// The iteration order of a `xmlb.NodeAttrIter` is not defined.
pub const NodeAttrIter = extern struct {
    f_dummy1: ?*anyopaque,
    f_dummy2: u8,
    f_dummy3: ?*anyopaque,
    f_dummy4: ?*anyopaque,
    f_dummy5: ?*anyopaque,
    f_dummy6: ?*anyopaque,

    /// Initializes a name/value pair iterator for the node attributes
    /// and associates it with `self`.
    /// The `xmlb.NodeAttrIter` structure is typically allocated on the stack
    /// and does not need to be freed explicitly.
    extern fn xb_node_attr_iter_init(p_iter: *NodeAttrIter, p_self: *xmlb.Node) void;
    pub const init = xb_node_attr_iter_init;

    /// Returns the current attribute name and value and advances the iterator.
    /// Example:
    /// ```
    /// XbNodeAttrIter iter;
    /// const gchar *attr_name, *attr_value;
    ///
    /// xb_node_attr_iter_init (&iter, node);
    /// while (xb_node_attr_iter_next (&iter, &attr_name, &attr_value)) {
    ///     // use attr_name and attr_value; no need to free them
    /// }
    /// ```
    extern fn xb_node_attr_iter_next(p_iter: *NodeAttrIter, p_name: ?*[*:0]const u8, p_value: ?*[*:0]const u8) c_int;
    pub const next = xb_node_attr_iter_next;
};

/// A `xmlb.NodeChildIter` structure represents an iterator that can be used
/// to iterate over the children of a `xmlb.Node`. `xmlb.NodeChildIter`
/// structures are typically allocated on the stack and then initialized
/// with `xmlb.NodeChildIter.init`.
pub const NodeChildIter = extern struct {
    f_dummy1: ?*anyopaque,
    f_dummy2: ?*anyopaque,
    f_dummy3: c_int,
    f_dummy4: ?*anyopaque,
    f_dummy5: ?*anyopaque,
    f_dummy6: ?*anyopaque,

    /// Initializes a child iterator for the node's children and associates
    /// it with `self`.
    /// The `xmlb.NodeChildIter` structure is typically allocated on the stack
    /// and does not need to be freed explicitly.
    extern fn xb_node_child_iter_init(p_iter: *NodeChildIter, p_self: *xmlb.Node) void;
    pub const init = xb_node_child_iter_init;

    /// Returns the current child and advances the iterator.
    /// On the first call to this function, the `child` pointer is assumed to point
    /// at uninitialised memory.
    /// On any later calls, it is assumed that the same pointers
    /// will be given and that they will point to the memory as set by the
    /// previous call to this function. This allows the previous values to
    /// be freed, as appropriate.
    ///
    /// Example:
    /// ```
    /// XbNodeChildIter iter;
    /// XbNode *child;
    ///
    /// xb_node_child_iter_init (&iter, node);
    /// while (xb_node_child_iter_loop (&iter, &child)) {
    ///     // do something with the node child
    ///     // no need to free 'child' unless breaking out of this loop
    /// }
    /// ```
    extern fn xb_node_child_iter_loop(p_iter: *NodeChildIter, p_child: ?**xmlb.Node) c_int;
    pub const loop = xb_node_child_iter_loop;

    /// Returns the current child and advances the iterator.
    /// The retrieved `xmlb.Node` child needs to be dereferenced with `gobject.Object.unref`.
    /// Example:
    /// ```
    /// XbNodeChildIter iter;
    /// g_autoptr(XbNode) child = NULL;
    ///
    /// xb_node_child_iter_init (&iter, node);
    /// while (xb_node_child_iter_next (&iter, &child)) {
    ///     // do something with the node child
    ///     g_clear_pointer (&child, g_object_unref);
    /// }
    /// ```
    extern fn xb_node_child_iter_next(p_iter: *NodeChildIter, p_child: ?**xmlb.Node) c_int;
    pub const next = xb_node_child_iter_next;
};

pub const NodeClass = extern struct {
    pub const Instance = xmlb.Node;

    f_parent_class: gobject.ObjectClass,
    f__xb_reserved1: ?*const fn () callconv(.C) void,
    f__xb_reserved2: ?*const fn () callconv(.C) void,
    f__xb_reserved3: ?*const fn () callconv(.C) void,
    f__xb_reserved4: ?*const fn () callconv(.C) void,
    f__xb_reserved5: ?*const fn () callconv(.C) void,
    f__xb_reserved6: ?*const fn () callconv(.C) void,
    f__xb_reserved7: ?*const fn () callconv(.C) void,

    pub fn as(p_instance: *NodeClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const Opcode = opaque {
    /// Converts a string to an opcode kind.
    extern fn xb_opcode_kind_from_string(p_str: [*:0]const u8) xmlb.OpcodeKind;
    pub const kindFromString = xb_opcode_kind_from_string;

    /// Converts the opcode kind to a string.
    extern fn xb_opcode_kind_to_string(p_kind: xmlb.OpcodeKind) [*:0]const u8;
    pub const kindToString = xb_opcode_kind_to_string;

    /// Checks if the opcode can be compared using the string value.
    extern fn xb_opcode_cmp_str(p_self: *Opcode) c_int;
    pub const cmpStr = xb_opcode_cmp_str;

    /// Checks if the opcode can be compared using the integer value.
    extern fn xb_opcode_cmp_val(p_self: *Opcode) c_int;
    pub const cmpVal = xb_opcode_cmp_val;

    /// Initialises a stack allocated `xmlb.Opcode` to contain a specific function.
    /// Custom functions can be registered using `xb_machine_add_func` and retrieved
    /// using `xb_machine_opcode_func_new`.
    extern fn xb_opcode_func_init(p_self: *Opcode, p_func: u32) void;
    pub const funcInit = xb_opcode_func_init;

    /// Gets the opcode kind.
    extern fn xb_opcode_get_kind(p_self: *Opcode) xmlb.OpcodeKind;
    pub const getKind = xb_opcode_get_kind;

    /// Gets the string value stored on the opcode.
    extern fn xb_opcode_get_str(p_self: *Opcode) [*:0]const u8;
    pub const getStr = xb_opcode_get_str;

    /// Gets the integer value stored in the opcode. This may be a function ID,
    /// a index into the string table or a literal integer.
    extern fn xb_opcode_get_val(p_self: *Opcode) u32;
    pub const getVal = xb_opcode_get_val;

    /// Initialises a stack allocated `xmlb.Opcode` to contain an integer literal.
    extern fn xb_opcode_integer_init(p_self: *Opcode, p_val: u32) void;
    pub const integerInit = xb_opcode_integer_init;

    /// Initialises a stack allocated `xmlb.Opcode` to contain a text literal.
    /// The `str` argument is copied internally and is not tied to the lifecycle of
    /// the `xmlb.Opcode`.
    extern fn xb_opcode_text_init(p_self: *Opcode, p_str: [*:0]const u8) void;
    pub const textInit = xb_opcode_text_init;

    /// Initialises a stack allocated `xmlb.Opcode` to contain a text literal, where
    /// `str` is either static text or will outlive the `xmlb.Opcode` lifecycle.
    extern fn xb_opcode_text_init_static(p_self: *Opcode, p_str: [*:0]const u8) void;
    pub const textInitStatic = xb_opcode_text_init_static;

    /// Initialises a stack allocated `xmlb.Opcode` to contain a text literal, stealing
    /// the `str`. Once the opcode is finalized `glib.free` will be called on `str`.
    extern fn xb_opcode_text_init_steal(p_self: *Opcode, p_str: [*:0]u8) void;
    pub const textInitSteal = xb_opcode_text_init_steal;

    /// Returns a string representing the specific opcode.
    extern fn xb_opcode_to_string(p_self: *Opcode) [*:0]u8;
    pub const toString = xb_opcode_to_string;
};

pub const QueryClass = extern struct {
    pub const Instance = xmlb.Query;

    f_parent_class: gobject.ObjectClass,
    f__xb_reserved1: ?*const fn () callconv(.C) void,
    f__xb_reserved2: ?*const fn () callconv(.C) void,
    f__xb_reserved3: ?*const fn () callconv(.C) void,
    f__xb_reserved4: ?*const fn () callconv(.C) void,
    f__xb_reserved5: ?*const fn () callconv(.C) void,
    f__xb_reserved6: ?*const fn () callconv(.C) void,
    f__xb_reserved7: ?*const fn () callconv(.C) void,

    pub fn as(p_instance: *QueryClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

/// An opaque struct which contains context for executing a query in, such as the
/// number of results to return, or values to bind to query placeholders.
pub const QueryContext = extern struct {
    f_dummy0: c_int,
    f_dummy1: c_uint,
    f_dummy2: xmlb.ValueBindings,
    f_dummy3: [5]*anyopaque,

    /// Clear an `xmlb.QueryContext`, freeing any allocated memory it points to.
    ///
    /// After this function has been called, the contents of the `xmlb.QueryContext` are
    /// undefined, and it’s only safe to call `xmlb.QueryContext.init` on it.
    extern fn xb_query_context_clear(p_self: *QueryContext) void;
    pub const clear = xb_query_context_clear;

    /// Copy `self` into a new heap-allocated `xmlb.QueryContext` instance.
    extern fn xb_query_context_copy(p_self: *QueryContext) *xmlb.QueryContext;
    pub const copy = xb_query_context_copy;

    /// Free a heap-allocated `xmlb.QueryContext` instance. This should be used on
    /// `xmlb.QueryContext` instances created with `xmlb.QueryContext.copy`.
    ///
    /// For stack-allocated instances, `xmlb.QueryContext.clear` should be used
    /// instead.
    extern fn xb_query_context_free(p_self: *QueryContext) void;
    pub const free = xb_query_context_free;

    /// Get the `xmlb.ValueBindings` for this query context.
    extern fn xb_query_context_get_bindings(p_self: *QueryContext) *xmlb.ValueBindings;
    pub const getBindings = xb_query_context_get_bindings;

    /// Get the flags set on the context. See `xmlb.QueryContext.setFlags`.
    extern fn xb_query_context_get_flags(p_self: *QueryContext) xmlb.QueryFlags;
    pub const getFlags = xb_query_context_get_flags;

    /// Get the limit on the number of query results. See
    /// `xmlb.QueryContext.setLimit`.
    extern fn xb_query_context_get_limit(p_self: *QueryContext) c_uint;
    pub const getLimit = xb_query_context_get_limit;

    /// Initialise a stack-allocated `xmlb.QueryContext` struct so it can be used.
    ///
    /// Stack-allocated `xmlb.QueryContext` instances should be freed once finished
    /// with, using `xmlb.QueryContext.clear` (or `g_auto(XbQueryContext)`, which is
    /// equivalent).
    extern fn xb_query_context_init(p_self: *QueryContext) void;
    pub const init = xb_query_context_init;

    /// Set flags which affect the behaviour of the query.
    extern fn xb_query_context_set_flags(p_self: *QueryContext, p_flags: xmlb.QueryFlags) void;
    pub const setFlags = xb_query_context_set_flags;

    /// Set the limit on the number of results to return from the query.
    extern fn xb_query_context_set_limit(p_self: *QueryContext, p_limit: c_uint) void;
    pub const setLimit = xb_query_context_set_limit;

    extern fn xb_query_context_get_type() usize;
    pub const getGObjectType = xb_query_context_get_type;
};

pub const SiloClass = extern struct {
    pub const Instance = xmlb.Silo;

    f_parent_class: gobject.ObjectClass,
    f__xb_reserved1: ?*const fn () callconv(.C) void,
    f__xb_reserved2: ?*const fn () callconv(.C) void,
    f__xb_reserved3: ?*const fn () callconv(.C) void,
    f__xb_reserved4: ?*const fn () callconv(.C) void,
    f__xb_reserved5: ?*const fn () callconv(.C) void,
    f__xb_reserved6: ?*const fn () callconv(.C) void,
    f__xb_reserved7: ?*const fn () callconv(.C) void,

    pub fn as(p_instance: *SiloClass, comptime P_T: type) *P_T {
        return gobject.ext.as(P_T, p_instance);
    }
};

pub const Stack = opaque {
    /// Pops an opcode off the stack.
    extern fn xb_stack_pop(p_self: *Stack, p_opcode_out: ?*xmlb.Opcode, p_error: ?*?*glib.Error) c_int;
    pub const pop = xb_stack_pop;

    /// Pushes a new empty opcode onto the end of the stack. A pointer to the opcode
    /// is returned in `opcode_out` so that the caller can initialise it. This must be
    /// done before the stack is next used as, for performance reasons, the newly
    /// pushed opcode is not zero-initialised.
    extern fn xb_stack_push(p_self: *Stack, p_opcode_out: ?**xmlb.Opcode, p_error: ?*?*glib.Error) c_int;
    pub const push = xb_stack_push;

    /// Returns a string representing a stack.
    extern fn xb_stack_to_string(p_self: *Stack) [*:0]u8;
    pub const toString = xb_stack_to_string;

    extern fn xb_stack_get_type() usize;
    pub const getGObjectType = xb_stack_get_type;
};

/// An opaque struct which contains values bound to a query.
pub const ValueBindings = extern struct {
    f_dummy0: u8,
    f_dummy1: u32,
    f_dummy2: [2]*anyopaque,
    f_dummy3: u8,
    f_dummy4: u32,
    f_dummy5: [2]*anyopaque,
    f_dummy6: u8,
    f_dummy7: u32,
    f_dummy8: [2]*anyopaque,
    f_dummy9: u8,
    f_dummy10: u32,
    f_dummy11: [2]*anyopaque,
    f_dummy12: [3]*anyopaque,

    /// Bind `str` to `idx` in the value bindings.
    ///
    /// This will overwrite any previous binding at `idx`. It will take ownership of
    /// `str`, and an appropriate `destroy_func` must be provided to free `str` once the
    /// binding is no longer needed. `destroy_func` will be called exactly once at
    /// some point before the `xmlb.ValueBindings` is cleared or freed.
    extern fn xb_value_bindings_bind_str(p_self: *ValueBindings, p_idx: c_uint, p_str: [*:0]const u8, p_destroy_func: ?glib.DestroyNotify) void;
    pub const bindStr = xb_value_bindings_bind_str;

    /// Bind `val` to `idx` in the value bindings.
    ///
    /// This will overwrite any previous binding at `idx`.
    extern fn xb_value_bindings_bind_val(p_self: *ValueBindings, p_idx: c_uint, p_val: u32) void;
    pub const bindVal = xb_value_bindings_bind_val;

    /// Clear an `xmlb.ValueBindings`, freeing any allocated memory it points to.
    ///
    /// After this function has been called, the contents of the `xmlb.ValueBindings` are
    /// undefined, and it’s only safe to call `xmlb.ValueBindings.init` on it.
    extern fn xb_value_bindings_clear(p_self: *ValueBindings) void;
    pub const clear = xb_value_bindings_clear;

    /// Copy `self` into a new heap-allocated `xmlb.ValueBindings` instance.
    extern fn xb_value_bindings_copy(p_self: *ValueBindings) *xmlb.ValueBindings;
    pub const copy = xb_value_bindings_copy;

    /// Copies the value bound at `idx` on `self` to `dest_idx` on `dest`. If no value is
    /// bound at `idx`, `dest` is not modified and `FALSE` is returned.
    ///
    /// `dest` must be initialised. If a binding already exists at `dest_idx`, it will
    /// be overwritten.
    extern fn xb_value_bindings_copy_binding(p_self: *ValueBindings, p_idx: c_uint, p_dest: *xmlb.ValueBindings, p_dest_idx: c_uint) c_int;
    pub const copyBinding = xb_value_bindings_copy_binding;

    /// Free a heap-allocated `xmlb.ValueBindings` instance. This should be used on
    /// `xmlb.ValueBindings` instances created with `xmlb.ValueBindings.copy`.
    ///
    /// For stack-allocated instances, `xmlb.ValueBindings.clear` should be used
    /// instead.
    extern fn xb_value_bindings_free(p_self: *ValueBindings) void;
    pub const free = xb_value_bindings_free;

    /// Initialise a stack-allocated `xmlb.ValueBindings` struct so it can be used.
    ///
    /// Stack-allocated `xmlb.ValueBindings` instances should be freed once finished
    /// with, using `xmlb.ValueBindings.clear` (or `g_auto(XbValueBindings)`, which is
    /// equivalent).
    extern fn xb_value_bindings_init(p_self: *ValueBindings) void;
    pub const init = xb_value_bindings_init;

    /// Check whether a value has been bound to the given index using (for example)
    /// `xmlb.ValueBindings.bindStr`.
    extern fn xb_value_bindings_is_bound(p_self: *ValueBindings, p_idx: c_uint) c_int;
    pub const isBound = xb_value_bindings_is_bound;

    /// Initialises an `xmlb.Opcode` with the value bound to `idx`, if a value is bound.
    /// If no value is bound, `opcode_out` is not touched and `FALSE` is returned.
    ///
    /// `opcode_out` is initialised to point to the data inside the `xmlb.ValueBindings`,
    /// so must have a shorter lifetime than the `xmlb.ValueBindings`. It will be of kind
    /// `XB_OPCODE_KIND_BOUND_TEXT` or `XB_OPCODE_KIND_BOUND_INTEGER`.
    extern fn xb_value_bindings_lookup_opcode(p_self: *ValueBindings, p_idx: c_uint, p_opcode_out: *xmlb.Opcode) c_int;
    pub const lookupOpcode = xb_value_bindings_lookup_opcode;

    extern fn xb_value_bindings_get_type() usize;
    pub const getGObjectType = xb_value_bindings_get_type;
};

pub const OpcodeKind = enum(c_int) {
    unknown = 0,
    integer = 1,
    text = 2,
    function = 5,
    bound_unset = 8,
    bound_integer = 9,
    bound_text = 10,
    indexed_text = 3,
    boolean = 17,
    bound_indexed_text = 11,
    _,
};

/// The flags for converting to XML.
pub const BuilderCompileFlags = packed struct(c_uint) {
    _padding0: bool = false,
    native_langs: bool = false,
    ignore_invalid: bool = false,
    single_lang: bool = false,
    watch_blob: bool = false,
    ignore_guid: bool = false,
    single_root: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_none: BuilderCompileFlags = @bitCast(@as(c_uint, 0));
    const flags_native_langs: BuilderCompileFlags = @bitCast(@as(c_uint, 2));
    const flags_ignore_invalid: BuilderCompileFlags = @bitCast(@as(c_uint, 4));
    const flags_single_lang: BuilderCompileFlags = @bitCast(@as(c_uint, 8));
    const flags_watch_blob: BuilderCompileFlags = @bitCast(@as(c_uint, 16));
    const flags_ignore_guid: BuilderCompileFlags = @bitCast(@as(c_uint, 32));
    const flags_single_root: BuilderCompileFlags = @bitCast(@as(c_uint, 64));
};

/// The flags used when building a node.
pub const BuilderNodeFlags = packed struct(c_uint) {
    ignore: bool = false,
    literal_text: bool = false,
    has_text: bool = false,
    has_tail: bool = false,
    tokenize_text: bool = false,
    strip_text: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_none: BuilderNodeFlags = @bitCast(@as(c_uint, 0));
    const flags_ignore: BuilderNodeFlags = @bitCast(@as(c_uint, 1));
    const flags_literal_text: BuilderNodeFlags = @bitCast(@as(c_uint, 2));
    const flags_has_text: BuilderNodeFlags = @bitCast(@as(c_uint, 4));
    const flags_has_tail: BuilderNodeFlags = @bitCast(@as(c_uint, 8));
    const flags_tokenize_text: BuilderNodeFlags = @bitCast(@as(c_uint, 16));
    const flags_strip_text: BuilderNodeFlags = @bitCast(@as(c_uint, 32));
};

/// The flags for converting to XML.
pub const BuilderSourceFlags = packed struct(c_uint) {
    literal_text: bool = false,
    watch_file: bool = false,
    watch_directory: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_none: BuilderSourceFlags = @bitCast(@as(c_uint, 0));
    const flags_literal_text: BuilderSourceFlags = @bitCast(@as(c_uint, 1));
    const flags_watch_file: BuilderSourceFlags = @bitCast(@as(c_uint, 2));
    const flags_watch_directory: BuilderSourceFlags = @bitCast(@as(c_uint, 4));
};

/// The flags to control the amount of debugging is generated.
pub const MachineDebugFlags = packed struct(c_uint) {
    show_stack: bool = false,
    show_parsing: bool = false,
    show_optimizer: bool = false,
    show_slow_path: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_none: MachineDebugFlags = @bitCast(@as(c_uint, 0));
    const flags_show_stack: MachineDebugFlags = @bitCast(@as(c_uint, 1));
    const flags_show_parsing: MachineDebugFlags = @bitCast(@as(c_uint, 2));
    const flags_show_optimizer: MachineDebugFlags = @bitCast(@as(c_uint, 4));
    const flags_show_slow_path: MachineDebugFlags = @bitCast(@as(c_uint, 8));
};

/// The flags to control the parsing behaviour.
pub const MachineParseFlags = packed struct(c_uint) {
    optimize: bool = false,
    _padding1: bool = false,
    _padding2: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_none: MachineParseFlags = @bitCast(@as(c_uint, 0));
    const flags_optimize: MachineParseFlags = @bitCast(@as(c_uint, 1));
};

/// The flags for converting to XML.
pub const NodeExportFlags = packed struct(c_uint) {
    add_header: bool = false,
    format_multiline: bool = false,
    format_indent: bool = false,
    include_siblings: bool = false,
    only_children: bool = false,
    collapse_empty: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_none: NodeExportFlags = @bitCast(@as(c_uint, 0));
    const flags_add_header: NodeExportFlags = @bitCast(@as(c_uint, 1));
    const flags_format_multiline: NodeExportFlags = @bitCast(@as(c_uint, 2));
    const flags_format_indent: NodeExportFlags = @bitCast(@as(c_uint, 4));
    const flags_include_siblings: NodeExportFlags = @bitCast(@as(c_uint, 8));
    const flags_only_children: NodeExportFlags = @bitCast(@as(c_uint, 16));
    const flags_collapse_empty: NodeExportFlags = @bitCast(@as(c_uint, 32));
};

/// The opcode flags. The values have been carefully chosen so that a simple
/// bitmask can be done to know how to compare for equality.
///
/// function─┐ ┌─string
/// bound──┐ │ │ ┌──integer
/// token┐ │ │ │ │
///  X X X X X X X
///        8 4 2 1
pub const OpcodeFlags = packed struct(c_uint) {
    integer: bool = false,
    text: bool = false,
    function: bool = false,
    bound: bool = false,
    boolean: bool = false,
    tokenized: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_unknown: OpcodeFlags = @bitCast(@as(c_uint, 0));
    const flags_integer: OpcodeFlags = @bitCast(@as(c_uint, 1));
    const flags_text: OpcodeFlags = @bitCast(@as(c_uint, 2));
    const flags_function: OpcodeFlags = @bitCast(@as(c_uint, 4));
    const flags_bound: OpcodeFlags = @bitCast(@as(c_uint, 8));
    const flags_boolean: OpcodeFlags = @bitCast(@as(c_uint, 16));
    const flags_tokenized: OpcodeFlags = @bitCast(@as(c_uint, 32));
};

/// The flags used for queries.
pub const QueryFlags = packed struct(c_uint) {
    optimize: bool = false,
    use_indexes: bool = false,
    reverse: bool = false,
    force_node_cache: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_none: QueryFlags = @bitCast(@as(c_uint, 0));
    const flags_optimize: QueryFlags = @bitCast(@as(c_uint, 1));
    const flags_use_indexes: QueryFlags = @bitCast(@as(c_uint, 2));
    const flags_reverse: QueryFlags = @bitCast(@as(c_uint, 4));
    const flags_force_node_cache: QueryFlags = @bitCast(@as(c_uint, 8));
};

/// The flags for loading a silo.
pub const SiloLoadFlags = packed struct(c_uint) {
    no_magic: bool = false,
    watch_blob: bool = false,
    _padding2: bool = false,
    _padding3: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_none: SiloLoadFlags = @bitCast(@as(c_uint, 0));
    const flags_no_magic: SiloLoadFlags = @bitCast(@as(c_uint, 1));
    const flags_watch_blob: SiloLoadFlags = @bitCast(@as(c_uint, 2));
};

/// The flags used when profiling a silo.
pub const SiloProfileFlags = packed struct(c_uint) {
    debug: bool = false,
    append: bool = false,
    xpath: bool = false,
    optimizer: bool = false,
    _padding4: bool = false,
    _padding5: bool = false,
    _padding6: bool = false,
    _padding7: bool = false,
    _padding8: bool = false,
    _padding9: bool = false,
    _padding10: bool = false,
    _padding11: bool = false,
    _padding12: bool = false,
    _padding13: bool = false,
    _padding14: bool = false,
    _padding15: bool = false,
    _padding16: bool = false,
    _padding17: bool = false,
    _padding18: bool = false,
    _padding19: bool = false,
    _padding20: bool = false,
    _padding21: bool = false,
    _padding22: bool = false,
    _padding23: bool = false,
    _padding24: bool = false,
    _padding25: bool = false,
    _padding26: bool = false,
    _padding27: bool = false,
    _padding28: bool = false,
    _padding29: bool = false,
    _padding30: bool = false,
    _padding31: bool = false,

    const flags_none: SiloProfileFlags = @bitCast(@as(c_uint, 0));
    const flags_debug: SiloProfileFlags = @bitCast(@as(c_uint, 1));
    const flags_append: SiloProfileFlags = @bitCast(@as(c_uint, 2));
    const flags_xpath: SiloProfileFlags = @bitCast(@as(c_uint, 4));
    const flags_optimizer: SiloProfileFlags = @bitCast(@as(c_uint, 8));
};

/// Appends an XPath query into the string, automatically adding the union
/// operator (`|`) if required.
extern fn xb_string_append_union(p_xpath: *glib.String, p_fmt: [*:0]const u8, ...) void;
pub const stringAppendUnion = xb_string_append_union;

/// Escapes XPath control sequences such as newlines, tabs, and forward slashes.
extern fn xb_string_escape(p_str: [*:0]const u8) [*:0]u8;
pub const stringEscape = xb_string_escape;

/// Gets the XMLb installed runtime version.
extern fn xb_version_string() [*:0]const u8;
pub const versionString = xb_version_string;

pub const BuilderFixupFunc = *const fn (p_self: *xmlb.BuilderFixup, p_bn: *xmlb.BuilderNode, p_user_data: ?*anyopaque, p_error: ?*?*glib.Error) callconv(.C) c_int;

pub const BuilderNodeSortFunc = *const fn (p_bn1: *xmlb.BuilderNode, p_bn2: *xmlb.BuilderNode, p_user_data: ?*anyopaque) callconv(.C) c_int;

pub const BuilderNodeTraverseFunc = *const fn (p_bn: *xmlb.BuilderNode, p_user_data: ?*anyopaque) callconv(.C) c_int;

pub const BuilderSourceAdapterFunc = *const fn (p_self: *xmlb.BuilderSource, p_ctx: *xmlb.BuilderSourceCtx, p_user_data: ?*anyopaque, p_cancellable: ?*gio.Cancellable, p_error: ?*?*glib.Error) callconv(.C) ?*gio.InputStream;

pub const BuilderSourceNodeFunc = *const fn (p_self: *xmlb.BuilderSource, p_bn: *xmlb.BuilderNode, p_user_data: ?*anyopaque, p_error: ?*?*glib.Error) callconv(.C) c_int;

pub const MachineMethodFunc = *const fn (p_self: *xmlb.Machine, p_stack: *xmlb.Stack, p_result_unused: *c_int, p_exec_data: ?*anyopaque, p_user_data: ?*anyopaque, p_error: ?*?*glib.Error) callconv(.C) c_int;

pub const MachineOpcodeFixupFunc = *const fn (p_self: *xmlb.Machine, p_opcodes: *xmlb.Stack, p_user_data: ?*anyopaque, p_error: ?*?*glib.Error) callconv(.C) c_int;

pub const MachineTextHandlerFunc = *const fn (p_self: *xmlb.Machine, p_opcodes: *xmlb.Stack, p_text: [*:0]const u8, p_handled: *c_int, p_user_data: ?*anyopaque, p_error: ?*?*glib.Error) callconv(.C) c_int;

pub const NodeTransmogrifyFunc = *const fn (p_self: *xmlb.Node, p_user_data: ?*anyopaque) callconv(.C) c_int;

/// The compile-time major version
pub const MAJOR_VERSION = 0;
/// The compile-time micro version
pub const MICRO_VERSION = 19;
/// The compile-time minor version
pub const MINOR_VERSION = 3;
