#include "config.h"
#include <sys/types.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include <sys/fcntl.h>
#include "uostr.h"
#include "uoio.h"
#include "uosock.h"
#include "dns.h"
#include "smtp.h"
#include "smtptools.h"

static void
wchan_logsub(const char *id, const char *host,uostr_t *logstr) 
{
	uostr_xdup_cstrmulti(logstr,"#",id,": ",host,": cannot write, deferred: ", strerror(errno));
}

#define WRITECHAN(host,chan,x,jmp) do {  \
	if ((x)==-1 || uoio_flush(chan)) { wchan_logsub(id,host,logstr); goto jmp; }; \
	} while(0)

static void 
logreply(uostr_t *s)
{
	int len;
	char *t;
	if (!reply_log_chan)
		return;
	t=memchr(s->data,'\n',s->len);
	if (t) {
		len=t-s->data;
		if (len && s->data[len]=='\r')
			len--;
	} else
		len=s->len;
	if (len)
		uoio_write_mem(reply_log_chan,s->data,len);
	uoio_write_char(reply_log_chan,'\n');
	uoio_flush(reply_log_chan);
}

static int
smtp_read_answer(uoio_t *rop,uostr_t *str)
{
	while (1) {
		int c;
		ssize_t got;
		got=uoio_getdelim_uostr(rop,str,'\n');
		if (got==-1)
			return -1;
		logreply(str);
		if (got<4)
			return -1;
		if (str->data[3]==' ') {
			c=(str->data[0]-'0')*100;
			c+=(str->data[1]-'0')*10;
			c+=(str->data[2]-'0');
			return c;
		}
	}
}

int 
smtp_open(const char *id, const char *host,unsigned int port,
		const char *greet,uoio_t *rop,uoio_t *wop, uostr_t *logstr)
{
	struct sockaddr_in s_in;
	dns_t *dns=NULL;
	dns_t *lauf;
	int c;
	if (strspn(host,"0123456789.")==strlen(host)) {
		dns=malloc(sizeof(*dns));
		if (!dns) { /* let xdup do it's job */
			uostr_xdup_cstrmulti(logstr,host,": soft DNS error",NULL);
			return SMTP_DEFER;
		}
		dns->name=0;
		dns->pref=0;
		dns->next=0;
		if (!inet_aton(host,&dns->ip)) {
			free(dns);
			dns=0;
		}
	}
	if (!dns) {
		c=dns_a(host,&dns);
		if (c==0 || c==DNS_SOFT || c==DNS_HARD) {
			int retcode=0; /* keep compiler quiet */
			if (c==0) {
				uostr_xdup_cstrmulti(logstr,host,": has no address",NULL);
				retcode=SMTP_FATAL;
			} else if (c==DNS_SOFT) {
				uostr_xdup_cstrmulti(logstr,host,": soft DNS error",NULL);
				retcode=SMTP_DEFER;
			} else if (c==DNS_HARD) {
				uostr_xdup_cstrmulti(logstr,host,": hard DNS error",NULL);
				retcode=SMTP_FATAL;
			}
			return retcode;
		}
	}
	for (lauf=dns;lauf;lauf=lauf->next) {
		int fd;
		ssize_t got;
		int rop_ok=0;
		int wop_ok=0;
		uostr_t ansstr;
		ansstr.data=0;

		s_in.sin_family=AF_INET;
		s_in.sin_addr=lauf->ip;
		s_in.sin_port=htons(port);

		fd=uosock_connect(&s_in, connect_timeout);
		if (fd==-1) {
			char *t=inet_ntoa(s_in.sin_addr);
			uostr_xdup_cstrmulti(logstr,host, " (",t,"): connect failed: ",strerror(errno),NULL);
			continue;
		}
		uoio_assign_r(rop,fd,read,0);
		rop->timeout=read_timeout;
		rop_ok=1;
		got=smtp_read_answer(rop,&ansstr);
		if (got==-1)  {
			char *t=inet_ntoa(s_in.sin_addr);
			uostr_xdup_cstrmulti(logstr,host, " (",t,"): greeting read failed: ",strerror(errno),NULL);
			goto do_continue;
		}
		if (got!=220) {	
			char *t=inet_ntoa(s_in.sin_addr);
			uostr_xdup_cstrmulti(logstr,host, " (",t,"): greeting failed: ",NULL);
			uostr_xadd_uostr(logstr,&ansstr);
			goto do_continue;
		}
		uoio_assign_w(wop,fd,write,0);
		rop->timeout=write_timeout;
		wop_ok=1;
		WRITECHAN(host,wop,uoio_write_cstrmulti(wop,"HELO ",greet ? greet : "NULL","\r\n",0),do_continue);
		got=smtp_read_answer(rop,&ansstr);
		if (got==-1) {
			char *t=inet_ntoa(s_in.sin_addr);
			uostr_xdup_cstrmulti(logstr,host, " (",t,"): greeting reply read failed: ",strerror(errno),NULL);
			goto do_continue;
		}
		/* ok, ready for the job */
	  	uostr_freedata(&ansstr);
		return SMTP_SUCCESS;
	  do_continue:
	  	uostr_freedata(&ansstr);
		if (rop_ok) uoio_destroy(rop);
		if (wop_ok) uoio_destroy(wop);
		close(fd);
		continue;
	}
	return SMTP_DEFER;
}

static void
rchan_logsub1(const char *id, const char *host, const char *where, uostr_t *logstr)
{
	uostr_xdup_cstrmulti(logstr,"#",id,": ",host,": cannot read ", where, " answer, deferred: ", 
		strerror(errno),NULL);
}

static int
rchan_logsub2(const char *id, const char *host, const char *who, int code, uostr_t *str, uostr_t *logstr)
{
	int retcode=SMTP_DEFER;
	if (str->len>0 && str->data[str->len-1]=='\n') str->len--;
	if (str->len>0 && str->data[str->len-1]=='\r') str->len--;
	uostr_xdup_cstrmulti(logstr,"#",id,": ", host,": ", who, " rejected, ", 
		(code>499)? "giving up: " : "deferred: ",NULL);
	uostr_xadd_uostr(logstr,str);
	if (code>499) retcode=SMTP_FATAL;
	return retcode;
}

int
smtp_send(const char *id, const char *host, uoio_t *rop, uoio_t *wop,
	const char *from, const char *to, int inputfd,
	uostr_t *logstr)
{
	int lines=0;
	int need_rset=0;
	int retcode=SMTP_DEFER;
	uoio_t fop;
	int code;
	uostr_t ansstr;
	ansstr.data=0;
#define REPLYHANDLER(where,who) do { \
	code=smtp_read_answer(rop,&ansstr); \
	if (code==-1) { rchan_logsub1(id,host, where,logstr); goto bailoutquit; } \
	else if (code>399) { \
		if (rchan_logsub2(id,host,who,code,&ansstr,logstr)==SMTP_FATAL) retcode=SMTP_FATAL; \
		goto bailout; \
	} } while(0)

	WRITECHAN(host,wop,uoio_write_cstrmulti(wop,"MAIL FROM: <",from ? from : "",">\r\n",NULL),bailoutquit);
	need_rset=1;
	REPLYHANDLER("MAIL","sender");

	WRITECHAN(host,wop,uoio_write_cstrmulti(wop,"RCPT TO: <",to,">\r\n",NULL),bailoutquit);
	REPLYHANDLER("RCPT","recipient");

	WRITECHAN(host,wop,uoio_write_cstr(wop,"DATA\r\n"),bailoutquit);
	REPLYHANDLER("DATA","message");

	uoio_assign_r(&fop,inputfd,read,0);

	while (1) {	
		char *p;
		ssize_t got=uoio_getdelim_zc(&fop,&p,'\n');
		if (got==-1) {
			/* shit. Can`t do anything except ... */
			uostr_xdup_cstrmulti(logstr,host,": cannot read message: ",strerror(errno),NULL);
			uoio_destroy(&fop);
			goto bailoutquit; /* no RSET in DATA possible */
		}
		if (got==0)
			break;
		lines++;
		if (got>0 && p[0]=='.') {
			uoio_write_char(wop,'.');
		}
		if (got>1)
			uoio_write_mem(wop,p,got-1);
		uoio_write_mem(wop,"\r\n",2);
	}
	if (uoio_flush(wop)) { wchan_logsub(id,host,logstr); goto bailoutquit; }
	uoio_destroy(&fop);
	if (!lines)
		WRITECHAN(host,wop,uoio_write_mem(wop,"\r\n",2),bailoutquit);
	WRITECHAN(host,wop,uoio_write_mem(wop,".\r\n",3),bailoutquit);
	REPLYHANDLER("message","message");
	ansstr.len--; /* kill \n */
	if (ansstr.data[ansstr.len-1]=='\r') ansstr.len--; /* just in case */
	uostr_xdup_uostr(logstr,&ansstr);
	uostr_freedata(&ansstr);
	return SMTP_SUCCESS;

	bailoutquit: {
		int fd2=rop->fd;
		uostr_freedata(&ansstr);
		uoio_destroy(rop);
		uoio_destroy(wop);
		close(fd2);
		return (retcode|SMTP_CLOSED);
	}
	bailout: {	
		if (need_rset) {
			WRITECHAN(host,wop,uoio_write_mem(wop,"RSET\r\n",6),bailoutquit);
			code=smtp_read_answer(rop,&ansstr); 
			if (code < 0) {
				rchan_logsub1(id, host, "RSET", logstr);
				goto bailoutquit;
			} else if (code >= 400) {
				uostr_xdup_cstrmulti(logstr,host,": RSET rejected: ",NULL);
				if (ansstr.len>0 && ansstr.data[ansstr.len-1]=='\n') ansstr.len--;
				if (ansstr.len>0 && ansstr.data[ansstr.len-1]=='\r') ansstr.len--;
				uostr_xadd_uostr(logstr,&ansstr);
				goto bailoutquit;
			}
		}
		uostr_freedata(&ansstr);
		return retcode;
	}
}

int
smtp_close(const char *id, const char *host, uoio_t *rop, uoio_t *wop)
{
	/* note that the logstring here is thrown away, it's useless */
	int code;
	int fd;
	uostr_t ls;
	uostr_t *logstr=&ls;
	uostr_t str;
	ls.data=str.data=0;
	WRITECHAN(host,wop,uoio_write_cstr(wop,"QUIT\r\n"),bailout);
	uoio_flush(wop); 
	code=smtp_read_answer(rop,&str); 
	uostr_freedata(&str);
	uostr_freedata(&ls);
  bailout:
    fd=rop->fd;
	uoio_destroy(rop);
	uoio_destroy(wop);
	close(fd);
	return 0;
}

