/*
** Copyright (C) 2001-2025 Zabbix SIA
**
** Permission is hereby granted, free of charge, to any person obtaining a copy of this software and associated
** documentation files (the "Software"), to deal in the Software without restriction, including without limitation the
** rights to use, copy, modify, merge, publish, distribute, sublicense, and/or sell copies of the Software, and to
** permit persons to whom the Software is furnished to do so, subject to the following conditions:
**
** The above copyright notice and this permission notice shall be included in all copies or substantial portions
** of the Software.
**
** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE
** WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
** COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
** TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
** SOFTWARE.
**/

package tlsconfig

import (
	"testing"

	"github.com/google/go-cmp/cmp"
)

func TestWithTLSConnect(t *testing.T) {
	t.Parallel()

	tests := []struct {
		name       string
		tlsConnect TLSConnectionType
		want       Details
	}{
		{
			name:       "+verifyCa",
			tlsConnect: "verify_ca",
			want:       Details{TLSConnect: "verify_ca"},
		},
		{
			name:       "+insecure",
			tlsConnect: "insecure",
			want:       Details{TLSConnect: "insecure"},
		},
	}

	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			gotDetails := &Details{AllowedConnections: make(map[string]bool)}
			opt := WithTLSConnect(tt.tlsConnect)
			opt(gotDetails)

			wantDetails := &tt.want
			if wantDetails.AllowedConnections == nil {
				wantDetails.AllowedConnections = make(map[string]bool)
			}

			if diff := cmp.Diff(wantDetails, gotDetails); diff != "" {
				t.Fatalf("WithTLSConnect() mismatch (-want +got):\n%s", diff)
			}
		})
	}
}

func TestWithTLSCaFile(t *testing.T) {
	t.Parallel()

	tests := []struct {
		name   string
		caFile string
		want   Details
	}{
		{
			name:   "+withPath",
			caFile: "/path/to/ca.pem",
			want:   Details{TLSCaFile: "/path/to/ca.pem"},
		},
		{
			name:   "+emptyPath",
			caFile: "",
			want:   Details{TLSCaFile: ""},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			gotDetails := &Details{AllowedConnections: make(map[string]bool)}
			opt := WithTLSCaFile(tt.caFile)
			opt(gotDetails)

			wantDetails := &tt.want
			if wantDetails.AllowedConnections == nil {
				wantDetails.AllowedConnections = make(map[string]bool)
			}

			if diff := cmp.Diff(wantDetails, gotDetails); diff != "" {
				t.Fatalf("WithTLSCaFile() mismatch (-want +got):\n%s", diff)
			}
		})
	}
}

func TestWithTLSCertFile(t *testing.T) {
	t.Parallel()

	tests := []struct {
		name     string
		certFile string
		want     Details
	}{
		{
			name:     "+withPath",
			certFile: "/path/to/cert.pem",
			want:     Details{TLSCertFile: "/path/to/cert.pem"},
		},
		{
			name:     "+emptyPath",
			certFile: "",
			want:     Details{TLSCertFile: ""},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			gotDetails := &Details{AllowedConnections: make(map[string]bool)}
			opt := WithTLSCertFile(tt.certFile)
			opt(gotDetails)

			wantDetails := &tt.want
			if wantDetails.AllowedConnections == nil {
				wantDetails.AllowedConnections = make(map[string]bool)
			}

			if diff := cmp.Diff(wantDetails, gotDetails); diff != "" {
				t.Fatalf("WithTLSCertFile() mismatch (-want +got):\n%s", diff)
			}
		})
	}
}

func TestWithTLSKeyFile(t *testing.T) {
	t.Parallel()

	tests := []struct {
		name    string
		keyFile string
		want    Details
	}{
		{
			name:    "+withPath",
			keyFile: "/path/to/key.pem",
			want:    Details{TLSKeyFile: "/path/to/key.pem"},
		},
		{
			name:    "+emptyPath",
			keyFile: "",
			want:    Details{TLSKeyFile: ""},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			gotDetails := &Details{AllowedConnections: make(map[string]bool)}
			opt := WithTLSKeyFile(tt.keyFile)
			opt(gotDetails)

			wantDetails := &tt.want
			if wantDetails.AllowedConnections == nil {
				wantDetails.AllowedConnections = make(map[string]bool)
			}

			if diff := cmp.Diff(wantDetails, gotDetails); diff != "" {
				t.Fatalf("WithTLSKeyFile() mismatch (-want +got):\n%s", diff)
			}
		})
	}
}

func TestWithAllowedConnections(t *testing.T) {
	t.Parallel()

	tests := []struct {
		name            string
		connectionTypes []string
		want            Details
	}{
		{
			name:            "+single",
			connectionTypes: []string{"verify_full"},
			want:            Details{AllowedConnections: map[string]bool{"verify_full": true}},
		},
		{
			name:            "+multiple",
			connectionTypes: []string{"verify_ca", "insecure"},
			want:            Details{AllowedConnections: map[string]bool{"verify_ca": true, "insecure": true}},
		},
		{
			name:            "+empty",
			connectionTypes: []string{},
			want:            Details{AllowedConnections: map[string]bool{}},
		},
		{
			name:            "+nilInput",
			connectionTypes: nil,
			want:            Details{AllowedConnections: map[string]bool{}},
		},
	}
	for _, tt := range tests {
		t.Run(tt.name, func(t *testing.T) {
			t.Parallel()

			gotDetails := &Details{AllowedConnections: make(map[string]bool)}
			opt := WithAllowedConnections(tt.connectionTypes...)
			opt(gotDetails)

			wantDetails := &tt.want
			if wantDetails.AllowedConnections == nil {
				wantDetails.AllowedConnections = make(map[string]bool)
			}

			if diff := cmp.Diff(wantDetails, gotDetails); diff != "" {
				t.Fatalf("WithAllowedConnections() mismatch (-want +got):\n%s", diff)
			}
		})
	}
}
