//  Linux ulog support

#include "linux_ulog.h"
#ifdef __HAVE_ULOG
#ifdef __OS_LINUX_LIKE

#include <net/if.h>
#include "debug.h"
#include "ldefs.h"
#include "iptonf.h"

#define BUFSIZE 65535

/// create ulog group mask
u_int32_t create_group_mask(u_int32_t grp){
	if (grp < 1 || grp > 32)
	{
		return 0;
	}
	return (1 << (grp - 1));
};

/// main working function ...
/// bind to ulog socket, receive packets and send them to nf
void * linux_ulog_process(void * ptr){
	char logbuf[256];
    char * dev = "ulog_iface0";
    if_list * iface = ifl_iface( dev ); /// reserver iface name!
    if(!iface){
        iface = ifl_add( dev );
        writelog("ULOG:dev added");
    };

    int fd = socket(PF_NETLINK, SOCK_RAW, NETLINK_NFLOG);
    if(fd == -1){
        writelog("ULOG:can't open socket !");
        pthread_exit(0);
    };

    struct sockaddr_nl sin;
    memset(&sin, 0, sizeof(sin));
    sin.nl_family = AF_NETLINK;
    sin.nl_pid = getpid();

    struct sockaddr_nl sin_kernel;
    memset(&sin_kernel, 0, sizeof(sin_kernel));
    sin_kernel.nl_family = AF_NETLINK;
    sin_kernel.nl_pid = 0;

	std::vector<u_int32_t>::iterator it = cfg.ulog_groups.begin();
	for(; it != cfg.ulog_groups.end(); it++){
		u_int32_t mask = create_group_mask(*it);
		snprintf( logbuf, 256, "ULOG:bind to ulog group  <%d> mask <%u>", (*it), mask);
		writelog(logbuf);

		sin.nl_groups  = mask;
		sin_kernel.nl_groups  = mask;
	};

    if(bind(fd, (struct sockaddr *)&sin, sizeof(sin)) == -1) {
		snprintf( logbuf, 256, "ULOG:can't bind. Error <%s>", strerror(errno));
        writelog(logbuf);
		pthread_exit(0);
    }

    /// receive packets
    char * packet = new char[BUFSIZE];
    socklen_t addrlen = sizeof(sin_kernel);
    struct nlmsghdr *nlheader;
    ulog_packet_msg_t *upacket;
	struct timeval ts;

    while (1){
        int data_size = recvfrom(fd, packet, BUFSIZE, 0, (struct sockaddr *)&sin_kernel, &addrlen);
        if(data_size <= 0){
            writelog("ULOG:can't recvfrom!");
            goto LOOP_END;
        };

#ifdef __DEBUG__
        DEB(fprintf( stderr, "packet received. size <%d>: ", data_size));
#endif

        nlheader = (struct nlmsghdr *)packet;
        if (nlheader->nlmsg_flags & MSG_TRUNC || data_size > BUFSIZE) {
            writelog("ULOG:packet truncated!");
            goto LOOP_END;
        }

        while(NLMSG_OK(nlheader, (size_t)BUFSIZE)) {
            upacket = (ulog_packet_msg_t *)NLMSG_DATA(nlheader);
			if( gettimeofday(&ts, NULL) == -1){
			};

			iptonf((u_char *)iface, &ts, upacket->payload);

            if(nlheader->nlmsg_type == NLMSG_DONE || !(nlheader->nlmsg_flags & NLM_F_MULTI)) {
                goto LOOP_END;
            }
            nlheader = NLMSG_NEXT(nlheader, data_size);
        }

LOOP_END:
        continue;
    };
};

/// thread starter
void run_linux_ulog(){
    /// run linux ulog process if specified in config
    pthread_t th;
    char * pbuf = new char[64];
    writelog( "Starting worker thread for linux ulog socket.");
    pthread_create( &th, 0, linux_ulog_process, ( void * ) pbuf );
    pthread_detach( th );
};
#endif ///  __OS_LINUX_LIKE
#endif // __HAVE_ULOG
