searchData={"items":[{"type":"module","title":"erl_tar","doc":"Unix 'tar' utility for reading and writing tar archives.\n\nThis module archives and extract files to and from a tar file. This module\nsupports reading most common tar formats, namely v7, STAR, USTAR, and PAX, as\nwell as some of GNU tar's extensions to the USTAR format (sparse files most\nnotably). It produces tar archives in USTAR format, unless the files being\narchived require PAX format due to restrictions in USTAR (such as unicode\nmetadata, filename length, and more). As such, `erl_tar` supports tar archives\nproduced by most all modern tar utilities, and produces tarballs which should be\nsimilarly portable.\n\nBy convention, the name of a tar file is to end in \"`.tar`\". To abide to the\nconvention, add \"`.tar`\" to the name.\n\nTar files can be created in one operation using function `create/2` or\n`create/3`.\n\nAlternatively, for more control, use functions `open/2`, [`add/3,4`](`add/3`),\nand `close/1`.\n\nTo extract all files from a tar file, use function `extract/1`. To extract only\nsome files or to be able to specify some more options, use function `extract/2`.\n\nTo return a list of the files in a tar file, use function `table/1` or\n`table/2`. To print a list of files to the Erlang shell, use function `t/1` or\n`tt/1`.\n\nTo convert an error term returned from one of the functions above to a readable\nmessage, use function `format_error/1`.","ref":"erl_tar.html"},{"type":"module","title":"Unicode Support - erl_tar","doc":"If `file:native_name_encoding/0` returns `utf8`, path names are encoded in UTF-8\nwhen creating tar files, and path names are assumed to be encoded in UTF-8 when\nextracting tar files.\n\nIf `file:native_name_encoding/0` returns `latin1`, no translation of path names\nis done.\n\nUnicode metadata stored in PAX headers is preserved","ref":"erl_tar.html#module-unicode-support"},{"type":"module","title":"Other Storage Media - erl_tar","doc":"The `m:ftp` module normally accesses the tar file on disk using the `m:file`\nmodule. When other needs arise, you can define your own low-level Erlang\nfunctions to perform the writing and reading on the storage media; use function\n`init/3`.\n\nAn example of this is the SFTP support in `ssh_sftp:open_tar/3`. This function\nopens a tar file on a remote machine using an SFTP channel.","ref":"erl_tar.html#module-other-storage-media"},{"type":"module","title":"Limitations - erl_tar","doc":"- If you must remain compatible with the USTAR tar format, you must ensure file\n  paths being stored are less than 255 bytes in total, with a maximum filename\n  component length of 100 bytes. USTAR uses a header field (prefix) in addition\n  to the name field, and splits file paths longer than 100 bytes into two parts.\n  This split is done on a directory boundary, and is done in such a way to make\n  the best use of the space available in those two fields, but in practice this\n  will often mean that you have less than 255 bytes for a path. `erl_tar` will\n  automatically upgrade the format to PAX to handle longer filenames, so this is\n  only an issue if you need to extract the archive with an older implementation\n  of `erl_tar` or `tar` which does not support PAX. In this case, the PAX\n  headers will be extracted as regular files, and you will need to apply them\n  manually.\n- Like the above, if you must remain USTAR compatible, you must also ensure than\n  paths for symbolic/hard links are no more than 100 bytes, otherwise PAX\n  headers will be used.","ref":"erl_tar.html#module-limitations"},{"type":"function","title":"erl_tar.add/3","doc":"Equivalent to `add/4`.\n\nIf `Name` is `t:name_in_archive/0`, then [`add(TarDescriptor, Name, Name, Options)`](`add/4`) is called.\n\nIf `Name` is a two tuple then [`add(TarDescriptor, NameInArchive, Name, Options)`](`add/4`) is called.","ref":"erl_tar.html#add/3"},{"type":"function","title":"erl_tar.add/4","doc":"Adds a file to a tar file that has been opened for writing by\n[`open/1`](`open/2`).\n\n`NameInArchive` is the name under which the file becomes stored in the tar file.\nThe file gets this name when it is extracted from the tar file.\n\nOptions:\n\n- **`dereference`** - By default, symbolic links are stored as symbolic links in\n  the tar file. To override the default and store the file that the symbolic\n  link points to into the tar file, use option `dereference`.\n\n- **`verbose`** - Prints an informational message about the added file.\n\n- **`{chunks,ChunkSize}`** - Reads data in parts from the file. This is intended\n  for memory-limited machines that, for example, builds a tar file on a remote\n  machine over SFTP, see `ssh_sftp:open_tar/3`.\n\n- **`{atime,non_neg_integer()}`** - Sets the last time, as\n  [POSIX time](`e:erts:time_correction.md#posix-time`), when the file was read.\n  See also `file:read_file_info/1`.\n\n- **`{mtime,non_neg_integer()}`** - Sets the last time, as\n  [POSIX time](`e:erts:time_correction.md#posix-time`), when the file was\n  written. See also `file:read_file_info/1`.\n\n- **`{ctime,non_neg_integer()}`** - Sets the time, as\n  [POSIX time](`e:erts:time_correction.md#posix-time`), when the file was\n  created. See also `file:read_file_info/1`.\n\n- **`{uid,non_neg_integer()}`** - Sets the file owner. `file:read_file_info/1`.\n\n- **`{gid,non_neg_integer()}`** - Sets the group that the file owner belongs to.\n  `file:read_file_info/1`.","ref":"erl_tar.html#add/4"},{"type":"type","title":"erl_tar.add_opt/0","doc":"","ref":"erl_tar.html#t:add_opt/0"},{"type":"function","title":"erl_tar.close/1","doc":"Closes a tar file opened by `open/2`.","ref":"erl_tar.html#close/1"},{"type":"function","title":"erl_tar.create/2","doc":"Creates a tar file and archives the files whose names are specified in\n`FileList` into it. The files can either be read from disk or be specified as\nbinaries.","ref":"erl_tar.html#create/2"},{"type":"function","title":"erl_tar.create/3","doc":"Creates a tar file and archives the files whose names are specified in\n`FileList` into it. The files can either be read from disk or be specified as\nbinaries.\n\nThe options in `OptionList` modify the defaults as follows:\n\n- **`compressed`** - The entire tar file is compressed, as if it has been run\n  through the `gzip` program. To abide to the convention that a compressed tar\n  file is to end in \"`.tar.gz`\" or \"`.tgz`\", add the appropriate extension.\n\n- **`cooked`** - By default, function [`open/2`](`open/2`) opens the tar file in\n  `raw` mode, which is faster but does not allow a remote (Erlang) file server\n  to be used. Adding `cooked` to the mode list overrides the default and opens\n  the tar file without option `raw`.\n\n- **`dereference`** - By default, symbolic links are stored as symbolic links in\n  the tar file. To override the default and store the file that the symbolic\n  link points to into the tar file, use option `dereference`.\n\n- **`verbose`** - Prints an informational message about each added file.","ref":"erl_tar.html#create/3"},{"type":"type","title":"erl_tar.create_opt/0","doc":"","ref":"erl_tar.html#t:create_opt/0"},{"type":"function","title":"erl_tar.extract/1","doc":"Extracts all files from a tar archive.\n\nIf argument `Name` is specified as `{binary,Binary}`, the contents of the binary\nis assumed to be a tar archive.\n\nIf argument `Name` is specified as `{file,Fd}`, `Fd` is assumed to be a file\ndescriptor returned from function `file:open/2`.\n\nOtherwise, `Name` is to be a filename.\n\n> #### Note {: .info }\n>\n> Leading slashes in tar member names will be removed before writing the file.\n> That is, absolute paths will be turned into relative paths. There will be an\n> info message written to the error logger when paths are changed in this way.\n\n> #### Warning {: .warning }\n>\n> The `compressed` and `cooked` flags are invalid when passing a file descriptor\n> with `{file,Fd}`. The file is assumed to have been opened with the appropriate\n> flags.","ref":"erl_tar.html#extract/1"},{"type":"function","title":"erl_tar.extract/2","doc":"Extracts files from a tar archive.\n\nIf argument `Name` is specified as `{binary,Binary}`, the contents of the binary\nis assumed to be a tar archive.\n\nIf argument `Name` is specified as `{file,Fd}`, `Fd` is assumed to be a file\ndescriptor returned from function `file:open/2`.\n\nOtherwise, `Name` is to be a filename.\n\nThe following options modify the defaults for the extraction as follows:\n\n- **`{cwd,Cwd}`** - Files with relative filenames are by default extracted to\n  the current working directory. With this option, files are instead extracted\n  into directory `Cwd`.\n\n- **`{files,FileList}`** - By default, all files are extracted from the tar\n  file. With this option, only those files are extracted whose names are\n  included in `FileList`.\n\n- **`compressed`** - With this option, the file is uncompressed while\n  extracting. If the tar file is not compressed, this option is ignored.\n\n- **`cooked`** - By default, function [`open/2`](`open/2`) function opens the\n  tar file in `raw` mode, which is faster but does not allow a remote (Erlang)\n  file server to be used. Adding `cooked` to the mode list overrides the default\n  and opens the tar file without option `raw`.\n\n- **`memory`** - Instead of extracting to a directory, this option gives the\n  result as a list of tuples `{Filename, Binary}`, where `Binary` is a binary\n  containing the extracted data of the file named `Filename` in the tar file.\n\n- **`keep_old_files`** - By default, all existing files with the same name as\n  files in the tar file are overwritten. With this option, existing files are\n  not overwriten.\n\n- **`verbose`** - Prints an informational message for each extracted file.\n\n> #### Warning {: .warning }\n>\n> The `compressed` and `cooked` flags are invalid when passing a file descriptor\n> with `{file,Fd}`. The file is assumed to have been opened with the appropriate\n> flags.","ref":"erl_tar.html#extract/2"},{"type":"type","title":"erl_tar.extract_opt/0","doc":"","ref":"erl_tar.html#t:extract_opt/0"},{"type":"type","title":"erl_tar.file_op/0","doc":"","ref":"erl_tar.html#t:file_op/0"},{"type":"type","title":"erl_tar.filelist/0","doc":"","ref":"erl_tar.html#t:filelist/0"},{"type":"function","title":"erl_tar.format_error/1","doc":"Converts an error reason term to a human-readable error message string.","ref":"erl_tar.html#format_error/1"},{"type":"type","title":"erl_tar.gid/0","doc":"","ref":"erl_tar.html#t:gid/0"},{"type":"function","title":"erl_tar.init/3","doc":"The `Fun` is the definition of what to do when the different storage operations\nfunctions are to be called from the higher tar handling functions (such as\n[`add/3`](`add/3`), [`add/4`](`add/4`), and [`close/1`](`close/1`)).\n\nThe `Fun` is called when the tar function wants to do a low-level operation,\nlike writing a block to a file. The `Fun` is called as\n`Fun(Op, {UserData,Parameters...})`, where `Op` is the operation name,\n`UserData` is the term passed as the first argument to `init/1` and\n`Parameters...` are the data added by the tar function to be passed down to the\nstorage handling function.\n\nParameter `UserData` is typically the result of opening a low-level structure\nlike a file descriptor or an SFTP channel id. The different `Fun` clauses\noperate on that very term.\n\nThe following are the fun clauses parameter lists:\n\n- **`(write, {UserData,DataToWrite})`** - Writes term `DataToWrite` using\n  `UserData`.\n\n- **`(close, UserData)`** - Closes the access.\n\n- **`(read2, {UserData,Size})`** - Reads using `UserData` but only `Size` bytes.\n  Notice that there is only an arity-2 read function, not an arity-1 function.\n\n- **`(position,{UserData,Position})`** - Sets the position of `UserData` as\n  defined for files in `file:position/2`\n\n_Example:_\n\nThe following is a complete `Fun` parameter for reading and writing on files\nusing the `m:file` module:\n\n```erlang\nExampleFun =\n   fun(write, {Fd,Data}) ->  file:write(Fd, Data);\n      (position, {Fd,Pos}) -> file:position(Fd, Pos);\n      (read2, {Fd,Size}) -> file:read(Fd, Size);\n      (close, Fd) -> file:close(Fd)\n   end\n```\n\nHere `Fd` was specified to function [`init/3`](`init/3`) as:\n\n```erlang\n{ok,Fd} = file:open(Name, ...).\n{ok,TarDesc} = erl_tar:init(Fd, [write], ExampleFun),\n```\n\n`TarDesc` is then used:\n\n```erlang\nerl_tar:add(TarDesc, SomeValueIwantToAdd, FileNameInTarFile),\n...,\nerl_tar:close(TarDesc)\n```\n\nWhen the `erl_tar` core wants to, for example, write a piece of `Data`, it would\ncall `ExampleFun(write, {UserData,Data})`.\n\n> #### Note {: .info }\n>\n> This example with the `file` module operations is not necessary to use\n> directly, as that is what function `open/2` in principle does.\n\n> #### Warning {: .warning }\n>\n> The `TarDescriptor` term is not a file descriptor. You are advised not to rely\n> on the specific contents of this term, as it can change in future Erlang/OTP\n> releases when more features are added to this module.","ref":"erl_tar.html#init/3"},{"type":"type","title":"erl_tar.mode/0","doc":"","ref":"erl_tar.html#t:mode/0"},{"type":"type","title":"erl_tar.name_in_archive/0","doc":"","ref":"erl_tar.html#t:name_in_archive/0"},{"type":"function","title":"erl_tar.open/2","doc":"Creates a tar file for writing (any existing file with the same name is\ntruncated).\n\nBy convention, the name of a tar file is to end in \"`.tar`\". To abide to the\nconvention, add \"`.tar`\" to the name.\n\nExcept for the `write` atom, the following atoms can be added to `OpenModeList`:\n\n- **`compressed`** - The entire tar file is compressed, as if it has been run\n  through the `gzip` program. To abide to the convention that a compressed tar\n  file is to end in \"`.tar.gz`\" or \"`.tgz`\", add the appropriate extension.\n\n- **`cooked`** - By default, the tar file is opened in `raw` mode, which is\n  faster but does not allow a remote (Erlang) file server to be used. Adding\n  `cooked` to the mode list overrides the default and opens the tar file without\n  option `raw`.\n\nTo add one file at the time into an opened tar file, use function\n[`add/3,4`](`add/3`). When you are finished adding files, use function `close/1`\nto close the tar file.\n\n> #### Warning {: .warning }\n>\n> The `compressed` and `cooked` flags are invalid when passing a file descriptor\n> with `{file,Fd}`. The file must already be opened with the appropriate flags.\n\n> #### Warning {: .warning }\n>\n> The `TarDescriptor` term is not a file descriptor. You are advised not to rely\n> on the specific contents of this term, as it can change in future Erlang/OTP\n> releases when more features are added to this module.","ref":"erl_tar.html#open/2"},{"type":"type","title":"erl_tar.open_type/0","doc":"","ref":"erl_tar.html#t:open_type/0"},{"type":"function","title":"erl_tar.t/1","doc":"Prints the names of all files in the tar file `Name` to the Erlang shell\n(similar to \"`tar t`\").","ref":"erl_tar.html#t/1"},{"type":"function","title":"erl_tar.table/1","doc":"","ref":"erl_tar.html#table/1"},{"type":"function","title":"erl_tar.table/2","doc":"Retrieves the names of all files in the tar file `Name`.","ref":"erl_tar.html#table/2"},{"type":"opaque","title":"erl_tar.tar_descriptor/0","doc":"","ref":"erl_tar.html#t:tar_descriptor/0"},{"type":"type","title":"erl_tar.tar_entry/0","doc":"","ref":"erl_tar.html#t:tar_entry/0"},{"type":"type","title":"erl_tar.tar_time/0","doc":"","ref":"erl_tar.html#t:tar_time/0"},{"type":"function","title":"erl_tar.tt/1","doc":"Prints names and information about all files in the tar file `Name` to the\nErlang shell (similar to \"`tar tv`\").","ref":"erl_tar.html#tt/1"},{"type":"type","title":"erl_tar.typeflag/0","doc":"","ref":"erl_tar.html#t:typeflag/0"},{"type":"type","title":"erl_tar.uid/0","doc":"","ref":"erl_tar.html#t:uid/0"},{"type":"type","title":"erl_tar.user_data/0","doc":"","ref":"erl_tar.html#t:user_data/0"},{"type":"module","title":"rand","doc":"Pseudo random number generation\n\nThis module provides pseudo random number generation and implements\na number of base generator algorithms.  Most are provided through\na [plug-in framework](#plug-in-framework) that adds\nfeatures to the base generators.\n\nAt the end of this module documentation there are some\n[niche algorithms](#niche-algorithms) that don't use\nthis module's normal [plug-in framework](#plug-in-framework).\nThey may be useful for special purposes like short generation time\nwhen quality is not essential, for seeding other generators, and such.\n\n[](){: #plug-in-framework } Plug-in framework\n---------------------------------------------\n\nThe [plug-in framework](#plug-in-framework-api) implements\na common [API](#plug-in-framework-api) to, and enhancements\nof the base generators:\n\n* Operating on a generator state in the\n  [process dictionary](#generator-state).\n* [Automatic](#generator-state) [seeding](`seed/1`).\n* Manual [seeding support](`seed/2`) to avoid common pitfalls.\n* Generating [integers](`t:integer/0`) in any range, with\n  [uniform distribution](`uniform/1`), without noticable bias.\n* Generating [integers](`t:integer/0`) in any range, larger than\n  the base generator's, with [uniform distribution](`uniform/1`).\n* Generating [floating-point numbers](`t:float/0`) with\n  [uniform distribution](`uniform/0`).\n* Generating [floating-point numbers](`t:float/0`) with\n  [normal distribution](`normal/0`).\n* Generating any number of [bytes](`bytes/1`).\n\nThe base generator algorithms implements the\n[Xoroshiro and Xorshift algorithms](http://xorshift.di.unimi.it)\nby Sebastiano Vigna.  During an iteration they generate a large integer\n(at least 58-bit) and operate on a state of several large integers.\n\nTo create numbers with normal distribution the\n[Ziggurat Method by Marsaglia and Tsang](http://www.jstatsoft.org/v05/i08)\nis used on the output from a base generator.\n\nFor most algorithms, jump functions are provided for generating\nnon-overlapping sequences. A jump function perform a calculation\nequivalent to a large number of repeated state iterations,\nbut execute in a time roughly equivalent to one regular iteration\nper generator bit.\n\n[](){: #algorithms } The following algorithms are provided:\n\n- **`exsss`**, the [_default algorithm_](#default-algorithm)\n  *(Since OTP 22.0)*  \n  Xorshift116\\*\\*, 58 bits precision and period of 2^116-1\n\n  Jump function: equivalent to 2^64 calls\n\n  This is the Xorshift116 generator combined with the StarStar scrambler from\n  the 2018 paper by David Blackman and Sebastiano Vigna:\n  [Scrambled Linear Pseudorandom Number Generators](http://vigna.di.unimi.it/ftp/papers/ScrambledLinear.pdf)\n\n  The generator doesn't use 58-bit rotates so it is faster than the\n  Xoroshiro116 generator, and when combined with the StarStar scrambler\n  it doesn't have any weak low bits like `exrop` (Xoroshiro116+).\n\n  Alas, this combination is about 10% slower than `exrop`, but despite that\n  it is the [_default algorithm_](#default-algorithm) thanks to\n  its statistical qualities.\n\n- **`exro928ss`** *(Since OTP 22.0)*  \n  Xoroshiro928\\*\\*, 58 bits precision and a period of 2^928-1\n\n  Jump function: equivalent to 2^512 calls\n\n  This is a 58 bit version of Xoroshiro1024\\*\\*, from the 2018 paper by\n  David Blackman and Sebastiano Vigna:\n  [Scrambled Linear Pseudorandom Number Generators](http://vigna.di.unimi.it/ftp/papers/ScrambledLinear.pdf)\n  that on a 64 bit Erlang system executes only about 40% slower than the\n  [*default `exsss` algorithm*](#default-algorithm)\n  but with much longer period and better statistical properties,\n  but on the flip side a larger state.\n\n  Many thanks to Sebastiano Vigna for his help with the 58 bit adaption.\n\n- **`exrop`** *(Since OTP 20.0)*  \n  Xoroshiro116+, 58 bits precision and period of 2^116-1\n\n  Jump function: equivalent to 2^64 calls\n\n- **`exs1024s`** *(Since OTP 20.0)*  \n  Xorshift1024\\*, 64 bits precision and a period of 2^1024-1\n\n  Jump function: equivalent to 2^512 calls\n\n- **`exsp`** *(Since OTP 20.0)*  \n  Xorshift116+, 58 bits precision and period of 2^116-1\n\n  Jump function: equivalent to 2^64 calls\n\n  This is a corrected version of a previous\n  [_default algorithm_](#default-algorithm) (`exsplus`, _deprecated_),\n  that was superseded by Xoroshiro116+ (`exrop`).  Since this algorithm\n  doesn't use rotate it executes a little (say < 15%) faster than `exrop`\n  (that has to do a 58 bit rotate, for which there is no native instruction).\n  See the [algorithms' homepage](http://xorshift.di.unimi.it).\n\n[](){: #default-algorithm }\n#### Default Algorithm\n\nThe current _default algorithm_ is\n[`exsss` (Xorshift116\\*\\*)](#algorithms). If a specific algorithm is\nrequired, ensure to always use `seed/1` to initialize the state.\n\nWhich algorithm that is the default may change between Erlang/OTP releases,\nand is selected to be one with high speed, small state and \"good enough\"\nstatistical properties.\n\n#### Old Algorithms\n\nUndocumented (old) algorithms are deprecated but still implemented so old code\nrelying on them will produce the same pseudo random sequences as before.\n\n> #### Note {: .info }\n>\n> There were a number of problems in the implementation of\n> the now undocumented algorithms, which is why they are deprecated.\n> The new algorithms are a bit slower but do not have these problems:\n>\n> Uniform integer ranges had a skew in the probability distribution\n> that was not noticable for small ranges but for large ranges\n> less than the generator's precision the probability to produce\n> a low number could be twice the probability for a high.\n>\n> Uniform integer ranges larger than or equal to the generator's precision\n> used a floating point fallback that only calculated with 52 bits\n> which is smaller than the requested range and therefore all numbers\n> in the requested range weren't even possible to produce.\n>\n> Uniform floats had a non-uniform density so small values for example\n> less than 0.5 had got smaller intervals decreasing as the generated value\n> approached 0.0 although still uniformly distributed for sufficiently large\n> subranges. The new algorithms produces uniformly distributed floats\n> on the form `N * 2.0^(-53)` hence they are equally spaced.\n\n[](){: #generator-state }\n#### Generator State\n\nEvery time a random number is generated, a state is used to calculate it,\nproducing a new state. The state can either be implicit\nor be an explicit argument and return value.\n\nThe functions with implicit state operates on a state stored\nin the process dictionary under the key `rand_seed`.  If that key\ndoesn't exist when the function is called, `seed/1` is called automatically\nwith the [_default algorithm_](#default-algorithm) and creates\na reasonably unpredictable seed.\n\nThe functions with explicit state don't use the process dictionary.\n\n#### _Examples_\n\nSimple use; create and seed the\n[_default algorithm_](#default-algorithm) with a non-fixed seed,\nif not already done, and generate two uniformly distibuted\nfloating point numbers.\n\n```erlang\nR0 = rand:uniform(),\nR1 = rand:uniform(),\n```\n\nUse a specified algorithm:\n\n```erlang\n_ = rand:seed(exro928ss),\nR2 = rand:uniform(),\n```\n\nUse a specified algorithm with a fixed seed:\n\n```erlang\n_ = rand:seed(exro928ss, {123, 123534, 345345}),\nR3 = rand:uniform(),\n```\n\nUse the functional API with a non-fixed seed:\n\n```erlang\nS0 = rand:seed_s(exsss),\n{R4, S1} = rand:uniform_s(S0),\n```\n\nGenerate a textbook basic form Box-Muller standard normal distribution number:\n\n```erlang\nR5 = rand:uniform_real(),\nR6 = rand:uniform(),\nSND0 = math:sqrt(-2 * math:log(R5)) * math:cos(math:pi() * R6)\n```\n\nGenerate a standard normal distribution number:\n\n```erlang\n{SND1, S2} = rand:normal_s(S1),\n```\n\nGenerate a normal distribution number with with mean -3 and variance 0.5:\n\n```erlang\n{ND0, S3} = rand:normal_s(-3, 0.5, S2),\n```\n\n#### Quality of the Generated Numbers\n\n> #### Note {: .info }\n>\n> The builtin random number generator algorithms are not cryptographically\n> strong. If a cryptographically strong random number generator is needed,\n> use something like `crypto:rand_seed/0`.\n\nFor all these generators except `exro928ss` and `exsss` the lowest bit(s)\nhave got a slightly less random behaviour than all other bits.\n1 bit for `exrop` (and `exsp`), and 3 bits for `exs1024s`. See for example\nthis explanation in the\n[Xoroshiro128+](http://xoroshiro.di.unimi.it/xoroshiro128plus.c)\ngenerator source code:\n\n> Beside passing BigCrush, this generator passes the PractRand test suite\n> up to (and included) 16TB, with the exception of binary rank tests,\n> which fail due to the lowest bit being an LFSR; all other bits pass all\n> tests. We suggest to use a sign test to extract a random Boolean value.\n\nIf this is a problem; to generate a boolean with these algorithms,\nuse something like this:\n\n```erlang\n(rand:uniform(256) > 128) % -> boolean()\n```\n\n```erlang\n((rand:uniform(256) - 1) bsr 7) % -> 0 | 1\n```\n\nFor a general range, with `N = 1` for `exrop`, and `N = 3` for `exs1024s`:\n\n```erlang\n(((rand:uniform(Range bsl N) - 1) bsr N) + 1)\n```\n\nThe floating point generating functions in this module waste the lowest bits\nwhen converting from an integer so they avoid this snag.\n\n\n[](){: #niche-algorithms } Niche algorithms\n-------------------------------------------\n\nThe [niche algorithms API](#niche-algorithms-api) contains\nspecial purpose algorithms that don't use the\n[plug-in framework](#plug-in-framework), mainly for performance reasons.\n\nSince these algorithms lack the plug-in framework support, generating numbers\nin a range other than the base generator's range may become a problem.\n\nThere are at least four ways to do this, assuming the `Range` is less than\nthe generator's range:\n\n[](){: #modulo-method }\n- **Modulo**  \n  To generate a number `V` in the range `0..Range-1`:\n\n  > Generate a number `X`.  \n  > Use `V = X rem Range` as your value.\n\n  This method uses `rem`, that is, the remainder of an integer division,\n  which is a slow operation.\n\n  Low bits from the generator propagate straight through to\n  the generated value, so if the generator has got weaknesses\n  in the low bits this method propagates them too.\n\n  If `Range` is not a divisor of the generator range, the generated numbers\n  have a bias.  Example:\n\n  Say the generator generates a byte, that is, the generator range\n  is `0..255`, and the desired range is `0..99` (`Range = 100`).\n  Then there are 3 generator outputs that produce the value `0`,\n  these are; `0`, `100` and `200`.\n  But there are only 2 generator outputs that produce the value `99`,\n  which are; `99` and `199`. So the probability for a value `V` in `0..55`\n  is 3/2 times the probability for the other values `56..99`.\n\n  If `Range` is much smaller than the generator range, then this bias\n  gets hard to detect. The rule of thumb is that if `Range` is smaller\n  than the square root of the generator range, the bias is small enough.\n  Example:\n\n  A byte generator when `Range = 20`. There are 12 (`256 div 20`)\n  possibilities to generate the highest numbers and one more to generate\n  a number `V < 16` (`256 rem 20`). So the probability is 13/12\n  for a low number versus a high. To detect that difference with\n  some confidence you would need to generate a lot more numbers\n  than the generator range, `256` in this small example.\n\n[](){: #truncated-multiplication-method }\n- **Truncated multiplication**  \n  To generate a number `V` in the range `0..Range-1`, when you have\n  a generator with a power of 2 range (`0..2^Bits-1`):\n\n  > Generate a number `X`.  \n  > Use `V = X * Range bsr Bits` as your value.\n\n  If the multiplication `X * Range` creates a bignum\n  this method becomes very slow.\n\n  High bits from the generator propagate through to the generated value,\n  so if the generator has got weaknesses in the high bits this method\n  propagates them too.\n\n  If `Range` is not a divisor of the generator range, the generated numbers\n  have a bias, pretty much as for the [Modulo](#modulo-method) method above.\n\n[](){: #shift-or-mask-method }\n- **Shift or mask**  \n  To generate a number in a power of 2 range (`0..2^RBits-1`),\n  when you have a generator with a power of 2 range (`0..2^Bits`):\n\n  > Generate a number `X`.  \n  > Use `V = X band ((1 bsl RBits)-1)` or `V = X bsr (Bits-RBits)`\n  > as your value.\n\n  Masking with `band` preserves the low bits, and right shifting\n  with `bsr` preserves the high, so if the generator has got weaknesses\n  in high or low bits; choose the right operator.\n\n  If the generator has got a range that is not a power of 2\n  and this method is used anyway, it introduces bias in the same way\n  as for the [Modulo](#modulo-method) method above.\n\n[](){: #rejection-method }\n- **Rejection**  \n\n  > Generate a number `X`.  \n  > If `X` is in the range, use it as your value,\n  > otherwise reject it and repeat.\n\n  In theory it is not certain that this method will ever complete,\n  but in practice you ensure that the probability of rejection is low.\n  Then the probability for yet another iteration decreases exponentially\n  so the expected mean number of iterations will often be between 1 and 2.\n  Also, since the base generator is a full length generator,\n  a value that will break the loop must eventually be generated.\n\n  These methods can be combined, such as using\n  the [Modulo](#modulo-method) method and only if the generator value\n  would create bias use [Rejection](#rejection-method).\n  Or using [Shift or mask](#shift-or-mask-method) to reduce the size\n  of a generator value so that\n  [Truncated multiplication](#truncated-multiplication-method)\n  will not create a bignum.\n\n  The recommended way to generate a floating point number\n  (IEEE 745 Double, that has got a 53-bit mantissa) in the range\n  `0..1`, that is `0.0 =< V < 1.0` is to generate a 53-bit number `X`\n  and then use `V = X * (1.0/((1 bsl 53)))` as your value.\n  This will create a value on the form N*2^-53 with equal probability\n  for every possible N for the range.","ref":"rand.html"},{"type":"type","title":"rand.alg/0","doc":"","ref":"rand.html#t:alg/0"},{"type":"type","title":"rand.alg_handler/0","doc":"","ref":"rand.html#t:alg_handler/0"},{"type":"type","title":"rand.alg_state/0","doc":"","ref":"rand.html#t:alg_state/0"},{"type":"type","title":"rand.builtin_alg/0","doc":"","ref":"rand.html#t:builtin_alg/0"},{"type":"function","title":"rand.bytes/1","doc":"Generate random bytes as a `t:binary()`,\nusing the state in the process dictionary.\n\nLike `bytes_s/2` but operates on the state stored in\nthe process dictionary.  Returns the generated [`Bytes`](`t:binary/0`).","ref":"rand.html#bytes/1"},{"type":"function","title":"rand.bytes_s/2","doc":"Generate random bytes as a `t:binary()`.\n\nFor a specified integer `N >= 0`, generates a `t:binary/0`\nwith that number of random bytes.\n\nThe selected algorithm is used to generate as many random numbers\nas required to compose the `t:binary/0`.  Returns the generated\n[`Bytes`](`t:binary/0`) and a [`NewState`](`t:state/0`).","ref":"rand.html#bytes_s/2"},{"type":"type","title":"rand.dummy_state/0","doc":"Algorithm specific internal state","ref":"rand.html#t:dummy_state/0"},{"type":"function","title":"rand.export_seed/0","doc":"Export the seed value.\n\nReturns the random number state in an external format.\nTo be used with `seed/1`.","ref":"rand.html#export_seed/0"},{"type":"function","title":"rand.export_seed_s/1","doc":"Export the seed value.\n\nReturns the random number generator state in an external format.\nTo be used with `seed/1`.","ref":"rand.html#export_seed_s/1"},{"type":"type","title":"rand.export_state/0","doc":"Algorithm-dependent state that can be printed or saved to file.","ref":"rand.html#t:export_state/0"},{"type":"opaque","title":"rand.exro928_state/0","doc":"Algorithm specific internal state","ref":"rand.html#t:exro928_state/0"},{"type":"opaque","title":"rand.exrop_state/0","doc":"Algorithm specific internal state","ref":"rand.html#t:exrop_state/0"},{"type":"opaque","title":"rand.exs64_state/0","doc":"Algorithm specific internal state","ref":"rand.html#t:exs64_state/0"},{"type":"opaque","title":"rand.exs1024_state/0","doc":"Algorithm specific internal state","ref":"rand.html#t:exs1024_state/0"},{"type":"function","title":"rand.exsp_jump/1","doc":"Jump the generator state forward.\n\nPerforms a [`State`](`t:state/0`) jump calculation\nthat is equvalent to a 2^64 state iterations.\n\nReturns the [`NewState`](`t:state/0`).\n\nThis feature can be used to create many non-overlapping\nrandom number sequences from one start state.\n\nSee the description of jump functions at the top of this module description.\n\nSee `exsp_next/1` about why this internal implementation function\nhas been exposed.","ref":"rand.html#exsp_jump/1"},{"type":"function","title":"rand.exsp_next/1","doc":"Generate an Xorshift116+ random integer and new algorithm state.\n\nFrom the specified [`AlgState`](`t:exsplus_state/0`),\ngenerates a random 58-bit integer [`X`](`t:uint58/0`)\nand a new algorithm state [`NewAlgState`](`t:exsplus_state/0`),\naccording to the Xorshift116+ algorithm.\n\nThis is an API function exposing the internal implementation of the\n[`exsp`](#algorithms) algorithm that enables using it without the\noverhead of the plug-in framework, which might be useful for time critial\napplications. On a typical 64 bit Erlang VM this approach executes\nin just above 30% (1/3) of the time for the default algorithm through\nthis module's normal plug-in framework.\n\nTo seed this generator use [`{_, AlgState} = rand:seed_s(exsp)`](`seed_s/1`)\nor [`{_, AlgState} = rand:seed_s(exsp, Seed)`](`seed_s/1`)\nwith a specific [`Seed`](`t:seed/0`).\n\n> #### Note {: .info }\n>\n> This function offers no help in generating a number on a selected range,\n> nor in generating floating point numbers.  It is easy to accidentally\n> mess up the statistical properties of this generator or to loose\n> the performance advantage when doing either.\n> See the recepies at the start of this\n> [Niche algorithms API](#niche-algorithms-api) description.\n>\n> Note also the caveat about weak low bits that this generator suffers from.\n>\n> The generator is exported in this form primarily for performance reasons.","ref":"rand.html#exsp_next/1"},{"type":"opaque","title":"rand.exsplus_state/0","doc":"Algorithm specific internal state","ref":"rand.html#t:exsplus_state/0"},{"type":"function","title":"rand.jump/0","doc":"Jump the generator state forward.\n\nLike `jump/1` but operates on the state stored in\nthe process dictionary.  Returns the [`NewState`](`t:state/0`).","ref":"rand.html#jump/0"},{"type":"function","title":"rand.jump/1","doc":"Jump the generator state forward.\n\nPerforms an algorithm specific [`State`](`t:state/0`) jump calculation\nthat is equvalent to a large number of state iterations.\nSee this module's [algorithms list](#algorithms).\n\nReturns the [`NewState`](`t:state/0`).\n\nThis feature can be used to create many non-overlapping\nrandom number sequences from one start state.\n\nThis function raises a `not_implemented` error exception if there is\nno jump function implemented for the [`State`](`t:state/0`)'s algorithm.","ref":"rand.html#jump/1"},{"type":"function","title":"rand.mwc59/1","doc":"Generate a new MWC59 state.\n\nFrom the specified generator state [`CX0`](`t:mwc59_state/0`) generate\na new state [`CX1`](`t:mwc59_state/0`), according to a Multiply With Carry\ngenerator, which is an efficient implementation of\na Multiplicative Congruential Generator with a power of 2 multiplier\nand a prime modulus.\n\nThis generator uses the multiplier `2^32` and the modulus\n`16#7fa6502 * 2^32 - 1`, which have been selected, in collaboration with\nSebastiano Vigna, to avoid bignum operations and still get\ngood statistical quality. It has been named \"MWC59\" and can be written as:\n\n```erlang\nC = CX0 bsr 32\nX = CX0 band ((1 bsl 32)-1))\nCX1 = 16#7fa6502 * X + C\n```\n\nBecause the generator uses a multiplier that is a power of 2 it gets\nstatistical flaws for collision tests and birthday spacings tests\nin 2 and 3 dimensions, and these caveats apply even when looking\nonly at the MWC \"digit\", that is the low 32 bits (the multiplier)\nof the generator state.  The higher bits of the state are worse.\n\nThe quality of the output value improves much by using a scrambler,\ninstead of just taking the low bits.\nFunction [`mwc59_value32`](`mwc59_value32/1`) is a fast scrambler\nthat returns a decent 32-bit number. The slightly slower\n[`mwc59_value`](`mwc59_value/1`) scrambler returns 59 bits of\nvery good quality, and [`mwc59_float`](`mwc59_float/1`) returns\na `t:float/0` of very good quality.\n\nThe low bits of the base generator are surprisingly good, so the lowest\n16 bits actually pass fairly strict PRNG tests, despite the generator's\nweaknesses that lie in the high bits of the 32-bit MWC \"digit\".\nIt is recommended to use `rem` on the the generator state, or bit mask\nextracting the lowest bits to produce numbers in a range 16 bits or less.\nSee the recepies at the start of this\n[Niche algorithms API](#niche-algorithms-api) description.\n\nOn a typical 64 bit Erlang VM this generator executes in below 8% (1/13)\nof the time for the default algorithm in the\n[plug-in framework API](#plug-in-framework-api) of this module.\nWith the [`mwc59_value32`](`mwc59_value32/1`) scrambler the total time\nbecomes 16% (1/6), and with [`mwc59_value`](`mwc59_value/1`)\nit becomes 20% (1/5) of the time for the default algorithm.\nWith [`mwc59_float`](`mwc59_float/1`) the total time\nis 60% of the time for the default algorithm generating a `t:float/0`.\n\n> #### Note {: .info }\n>\n> This generator is a niche generator for high speed applications.\n> It has a much shorter period than the default generator, which in itself\n> is a quality concern, although when used with the value scramblers\n> it passes strict PRNG tests.  The generator is much faster than\n> `exsp_next/1` but with a bit lower quality and much shorter period.","ref":"rand.html#mwc59/1"},{"type":"function","title":"rand.mwc59_float/1","doc":"Calculate a scrambled `t:float/0` from a [MWC59 state](`t:mwc59_state/0`).\n\nReturns a value `V ::` `t:float/0` from a generator state `CX`,\nin the range `0.0 =< V < 1.0` like for example `uniform_s/1`.\n\nThe generator state is scrambled as with\n[`mwc59_value/1`](`mwc59_value/1`) before converted to a `t:float/0`.","ref":"rand.html#mwc59_float/1"},{"type":"function","title":"rand.mwc59_seed/0","doc":"Create a [MWC59 generator state](`t:mwc59_state/0`).\n\nLike `mwc59_seed/1` but it hashes the default seed value\nof [`seed_s(atom())`](`seed_s/1`).","ref":"rand.html#mwc59_seed/0"},{"type":"function","title":"rand.mwc59_seed/1","doc":"Create a [MWC59 generator state](`t:mwc59_state/0`).\n\nReturns a generator state [`CX`](`t:mwc59_state/0`).\nThe 58-bit seed value `S` is hashed to create the generator state,\nto avoid that similar seeds create similar sequences.","ref":"rand.html#mwc59_seed/1"},{"type":"type","title":"rand.mwc59_state/0","doc":"`1 .. (16#1ffb072 bsl 29) - 2`","ref":"rand.html#t:mwc59_state/0"},{"type":"function","title":"rand.mwc59_value32/1","doc":"Calculate a 32-bit scrambled value from a [MWC59 state](`t:mwc59_state/0`).\n\nReturns a 32-bit value [`V`](`t:integer/0`) from a generator state `CX`.\nThe generator state is scrambled using an 8-bit xorshift which masks\nthe statistical imperfecions of the base generator [`mwc59`](`mwc59/1`)\nenough to produce numbers of decent quality. Still some problems\nin 2- and 3-dimensional birthday spacing and collision tests show through.\n\nWhen using this scrambler it is in general better to use the high bits of the\nvalue than the low. The lowest 8 bits are of good quality and are passed\nright through from the base generator. They are combined with the next 8\nin the xorshift making the low 16 good quality, but in the range\n16..31 bits there are weaker bits that should not become high bits\nof the generated values.\n\nTherefore it is in general safer to shift out low bits. See the recepies\nat the start of this [Niche algorithms API](#niche-algorithms-api)\ndescription.\n\nFor a non power of 2 range less than about 16 bits (to not get\ntoo much bias and to avoid bignums) truncated multiplication can be used,\nthat is: `(Range*V) bsr 32`, which is much faster than using `rem`.","ref":"rand.html#mwc59_value32/1"},{"type":"function","title":"rand.mwc59_value/1","doc":"Calculate a 59-bit scrambled value from a [MWC59 state](`t:mwc59_state/0`).\n\nReturns a 59-bit value [`V`](`t:integer/0`) from a generator state `CX`.\nThe generator state is scrambled using an 4-bit followed by a 27-bit xorshift,\nwhich masks the statistical imperfecions of the [MWC59](`mwc59/1`)\nbase generator enough that all 59 bits are of very good quality.\n\nBe careful to not accidentaly create a bignum when handling the value `V`.\n\nIt is in general general better to use the high bits from this scrambler than\nthe low. See the recepies at the start of this\n[Niche algorithms API](#niche-algorithms-api) description.\n\nFor a non power of 2 range less than about 29 bits (to not get\ntoo much bias and to avoid bignums) truncated multiplication can be used,\nwhich is much faster than using `rem`. Example for range 1'000'000'000;\nthe range is 30 bits, we use 29 bits from the generator,\nadding up to 59 bits, which is not a bignum (on a 64-bit VM ):\n`(1000000000 * (V bsr (59-29))) bsr 29`.","ref":"rand.html#mwc59_value/1"},{"type":"function","title":"rand.normal/0","doc":"Generate a random number with standard normal distribution.\n\nLike `normal_s/1` but operates on the state stored in\nthe process dictionary.  Returns the generated number `X`.","ref":"rand.html#normal/0"},{"type":"function","title":"rand.normal/2","doc":"Generate a random number with specified normal distribution 𝒩 *(μ, σ²)*.\n\nLike `normal_s/3` but operates on the state stored in\nthe process dictionary.  Returns the generated number `X`.","ref":"rand.html#normal/2"},{"type":"function","title":"rand.normal_s/1","doc":"Generate a random number with standard normal distribution.\n\nFrom the specified `State`, generates a random number `X ::` `t:float/0`,\nwith standard normal distribution, that is with mean value `0.0`\nand variance `1.0`.\n\nReturns the generated number [`X`](`t:float/0`)\nand the [`NewState`](`t:state/0`).","ref":"rand.html#normal_s/1"},{"type":"function","title":"rand.normal_s/3","doc":"Generate a random number with specified normal distribution 𝒩 *(μ, σ²)*.\n\nFrom the specified `State`, generates a random number `X ::` `t:float/0`,\nwith normal distribution 𝒩 *(μ, σ²)*, that is 𝒩 (Mean, Variance)\nwhere `Variance >= 0.0`.\n\nReturns [`X`](`t:float/0`) and the [`NewState`](`t:state/0`).","ref":"rand.html#normal_s/3"},{"type":"type","title":"rand.seed/0","doc":"Generator seed value.\n\nA list of integers sets the generator's internal state directly, after\nalgorithm-dependent checks of the value and masking to the proper word size.\nThe number of integers must be equal to the number of state words\nin the generator.\n\nA single integer is used as the initial state for a SplitMix64 generator.\nThe sequential output values of that is then used for setting\nthe generator's internal state after masking to the proper word size\nand if needed avoiding zero values.\n\nA traditional 3-tuple of integers seed is passed through algorithm-dependent\nhashing functions to create the generator's initial state.","ref":"rand.html#t:seed/0"},{"type":"function","title":"rand.seed/1","doc":"Seed the random number generator and select algorithm.\n\nThe same as [`seed_s(Alg_or_State)`](`seed_s/1`),\nbut also stores the generated state in the process dictionary.\n\nThe argument `default` is an alias for the\n[_default algorithm_](#default-algorithm)\nthat has been implemented *(Since OTP 24.0)*.","ref":"rand.html#seed/1"},{"type":"function","title":"rand.seed/2","doc":"Seed the random number generator and select algorithm.\n\nThe same as [`seed_s(Alg, Seed)`](`seed_s/2`),\nbut also stores the generated state in the process dictionary.\n\n`Alg = default` is an alias for the\n[_default algorithm_](#default-algorithm)\nthat has been implemented *(Since OTP 24.0)*.","ref":"rand.html#seed/2"},{"type":"function","title":"rand.seed_s/1","doc":"Seed the random number generator and select algorithm.\n\nWith the argument `Alg`, select that algorithm and seed random number\ngeneration with reasonably unpredictable time dependent data.\n\n`Alg = default` is an alias for the\n[_default algorithm_](#default-algorithm)\n*(Since OTP 24.0)*.\n\nWith the argument `State`, re-creates the state and returns it.\nSee also `export_seed/0`.","ref":"rand.html#seed_s/1"},{"type":"function","title":"rand.seed_s/2","doc":"Seed the random number generator and select algorithm.\n\nCreates and returns a generator state for the specified algorithm\nfrom the specified `t:seed/0` integers.\n\n`Alg = default` is an alias for the [_default algorithm_](#default-algorithm)\nthat has been implemented *since OTP 24.0*.","ref":"rand.html#seed_s/2"},{"type":"function","title":"rand.splitmix64_next/1","doc":"Generate a SplitMix64 random 64-bit integer and new algorithm state.\n\nFrom the specified `AlgState` generates a random 64-bit integer\n[`X`](`t:uint64/0`) and a new generator state\n[`NewAlgState`](`t:splitmix64_state/0`),\naccording to the SplitMix64 algorithm.\n\nThis generator is used internally in the `rand` module for seeding other\ngenerators since it is of a quite different breed which reduces\nthe probability for creating an accidentally bad seed.","ref":"rand.html#splitmix64_next/1"},{"type":"type","title":"rand.splitmix64_state/0","doc":"Algorithm specific state","ref":"rand.html#t:splitmix64_state/0"},{"type":"type","title":"rand.state/0","doc":"Algorithm-dependent state.","ref":"rand.html#t:state/0"},{"type":"type","title":"rand.uint58/0","doc":"`0 .. (2^58 - 1)`","ref":"rand.html#t:uint58/0"},{"type":"type","title":"rand.uint64/0","doc":"`0 .. (2^64 - 1)`","ref":"rand.html#t:uint64/0"},{"type":"function","title":"rand.uniform/0","doc":"Generate a uniformly distributed random number `0.0 =< X < 1.0`,\nusing the state in the process dictionary.\n\nLike `uniform_s/1` but operates on the state stored in\nthe process dictionary.  Returns the generated number `X`.","ref":"rand.html#uniform/0"},{"type":"function","title":"rand.uniform/1","doc":"Generate a uniformly distributed random integer `1 =< X =< N`,\nusing the state in the process dictionary.\n\nLike `uniform_s/2` but operates on the state stored in\nthe process dictionary.  Returns the generated number `X`.","ref":"rand.html#uniform/1"},{"type":"function","title":"rand.uniform_real/0","doc":"Generate a uniformly distributed random number `0.0 < X < 1.0`,\nusing the state in the process dictionary.\n\nLike `uniform_real_s/1` but operates on the state stored in\nthe process dictionary.  Returns the generated number `X`.\n\nSee `uniform_real_s/1`.","ref":"rand.html#uniform_real/0"},{"type":"function","title":"rand.uniform_real_s/1","doc":"Generate a uniformly distributed random number `0.0 < X < 1.0`.\n\nFrom the specified state, generates a random float, uniformly distributed\nin the value range `DBL_MIN =  #### Note {: .info }\n>\n> The generated numbers from this function has got better granularity\n> for small numbers than the regular `uniform_s/1` because all bits\n> in the mantissa are random. This property, in combination with the fact\n> that exactly zero is never returned is useful for algorithms doing\n> for example `1.0 / X` or `math:log(X)`.\n\nThe concept implicates that the probability to get exactly zero is extremely\nlow; so low that this function in fact never returns `0.0`.\nThe smallest number that it might return is `DBL_MIN`,\nwhich is `2.0^(-1022)`.\n\nThe value range stated at the top of this function description is\ntechnically correct, but `0.0 =< X < 1.0` is a better description\nof the generated numbers' statistical distribution, and that\nthis function never returns exactly `0.0` is impossible to observe.\n\nFor all sub ranges `N*2.0^(-53) =< X < (N+1)*2.0^(-53)` where\n`0 =< integer(N) < 2.0^53`, the probability to generate a number\nin the range is the same.  Compare with the numbers\ngenerated by `uniform_s/1`.\n\nHaving to generate extra random bits for occasional small numbers\ncosts a little performance. This function is about 20% slower\nthan the regular `uniform_s/1`","ref":"rand.html#uniform_real_s/1"},{"type":"function","title":"rand.uniform_s/1","doc":"Generate a uniformly distributed random number `0.0 =< X < 1.0`.\n\nFrom the specified `State`, generates a random number `X ::` `t:float/0`,\nuniformly distributed in the value range `0.0 =  #### Warning {: .warning }\n>\n> This function may return exactly `0.0` which can be fatal for certain\n> applications. If that is undesired you can use `(1.0 - rand:uniform())`\n> to get the interval `0.0 < X =< 1.0`, or instead use `uniform_real/0`.\n>\n> If neither endpoint is desired you can achieve the range\n> `0.0 < X < 1.0` using test and re-try like this:\n>\n> ```erlang\n> my_uniform() ->\n>     case rand:uniform() of\n>         X when 0.0   X;\n>         _ -> my_uniform()\n>     end.\n> ```","ref":"rand.html#uniform_s/1"},{"type":"function","title":"rand.uniform_s/2","doc":"Generate a uniformly distributed random integer `1 =< X =< N`.\n\nFrom the specified `State`, generates a random number `X ::` `t:integer/0`,\nuniformly distributed in the specified range `1 =< X =< N`.\nReturns the number `X` and the updated `NewState`.","ref":"rand.html#uniform_s/2"},{"type":"module","title":"random","doc":"Pseudo-random number generation.\n\nThis module provides a random number generator. The method is attributed to B.A.\nWichmann and I.D. Hill in 'An efficient and portable pseudo-random number\ngenerator', Journal of Applied Statistics. AS183. 1982. Also Byte March 1987.\n\nThe algorithm is a modification of the version attributed to Richard A. O'Keefe\nin the standard Prolog library.\n\nEvery time a random number is requested, a state is used to calculate it, and a\nnew state is produced. The state can either be implicit (kept in the process\ndictionary) or be an explicit argument and return value. In this implementation,\nthe state (the type `t:ran/0`) consists of a tuple of three integers.\n\n> #### Note {: .info }\n>\n> This random number generator is not cryptographically strong. If a strong\n> cryptographic random number generator is needed, use one of functions in the\n> `m:crypto` module, for example, [`crypto:strong_rand_bytes/1`](`m:crypto`).\n\n> #### Note {: .info }\n>\n> The improved `m:rand` module is to be used instead of this module.","ref":"random.html"},{"type":"module","title":"Note - random","doc":"Some of the functions use the process dictionary variable `random_seed` to\nremember the current seed.\n\nIf a process calls `uniform/0` or `uniform/1` without setting a seed first,\n`seed/0` is called automatically.\n\nThe implementation changed in Erlang/OTP R15. Upgrading to R15 breaks\napplications that expect a specific output for a specified seed. The output is\nstill deterministic number series, but different compared to releases older than\nR15. Seed `{0,0,0}` does, for example, no longer produce a flawed series of only\nzeros.","ref":"random.html#module-note"},{"type":"type","title":"random.ran/0","doc":"The state.","ref":"random.html#t:ran/0"},{"type":"function","title":"random.seed0/0","doc":"Returns the default state.","ref":"random.html#seed0/0"},{"type":"function","title":"random.seed/0","doc":"Seeds random number generation with default (fixed) values in the process\ndictionary and returns the old state.","ref":"random.html#seed/0"},{"type":"function","title":"random.seed/1","doc":"[`seed({A1, A2, A3})`](`seed/1`) is equivalent to\n[`seed(A1, A2, A3)`](`seed/3`).","ref":"random.html#seed/1"},{"type":"function","title":"random.seed/3","doc":"Seeds random number generation with integer values in the process dictionary and\nreturns the old state.\n\nThe following is an easy way of obtaining a unique value to seed with:\n\n```erlang\nrandom:seed(erlang:phash2([node()]),\n            erlang:monotonic_time(),\n            erlang:unique_integer())\n```\n\nFor details, see `erlang:phash2/1`, `erlang:node/0`, `erlang:monotonic_time/0`,\nand `erlang:unique_integer/0`.","ref":"random.html#seed/3"},{"type":"function","title":"random.uniform/0","doc":"Returns a random float uniformly distributed between `0.0` and `1.0`, updating\nthe state in the process dictionary.","ref":"random.html#uniform/0"},{"type":"function","title":"random.uniform/1","doc":"Returns, for a specified integer `N >= 1`, a random integer uniformly\ndistributed between `1` and `N`, updating the state in the process dictionary.","ref":"random.html#uniform/1"},{"type":"function","title":"random.uniform_s/1","doc":"Returns, for a specified state, a random float uniformly distributed between\n`0.0` and `1.0`, and a new state.","ref":"random.html#uniform_s/1"},{"type":"function","title":"random.uniform_s/2","doc":"Returns, for a specified integer `N >= 1` and a state, a random integer\nuniformly distributed between `1` and `N`, and a new state.","ref":"random.html#uniform_s/2"},{"type":"module","title":"zip","doc":"Utility for reading and creating 'zip' archives.\n\nThis module archives and extracts files to and from a zip archive. The zip\nformat is specified by the \"ZIP Appnote.txt\" file, available on the PKWARE web\nsite [www.pkware.com](http://www.pkware.com).\n\nThe zip module supports zip archive versions up to 6.1. However,\npassword-protection is not supported.\n\nBy convention, the name of a zip file is to end with `.zip`. To abide to the\nconvention, add `.zip` to the filename.\n\n- To create zip archives, use function `zip/2` or `zip/3`. They are\n  also available as [`create/2,3`](`create/3`), to resemble the `m:erl_tar` module.\n- To extract files from a zip archive, use function `unzip/1` or `unzip/2`. They\n  are also available as [`extract/1,2`](`extract/1`), to resemble the `m:erl_tar` module.\n- To fold a function over all files in a zip archive, use function `foldl/3`.\n- To return a list of the files in a zip archive, use function `list_dir/1` or\n  `list_dir/2`. They are also available as [`table/1,2`](`table/1`), to resemble the\n  `m:erl_tar` module.\n- To print a list of files to the Erlang shell, use function `t/1` or `tt/1`.\n- Sometimes it is desirable to open a zip archive, and to unzip files from it\n  file by file, without having to reopen the archive. This can be done by\n  functions [`zip_open/1,2`](`zip_open/1`), [`zip_get/1,2`](`zip_get/1`),\n  `zip_list_dir/1`, and `zip_close/1`.\n- The ZIP extensions 0x5355 \"extended timestamps\" and 0x7875 \"UID+GID handling\"\n  are supported. Both extensions are by default enabled when creating an archive,\n  but only \"extended timestamps\" are enabled when extracting. Use the `t:extra/0`\n  option to change how these extensions are used.","ref":"zip.html"},{"type":"module","title":"Limitations - zip","doc":"- Password-protected and encrypted archives are not supported.\n- Only the DEFLATE (zlib-compression) and the STORE (uncompressed data) zip\n  methods are supported.\n- Comments for individual files are not supported when creating zip archives.\n  The zip archive comment for the whole zip archive is supported.\n- Changing a zip archive is not supported. To add or remove a file from an\n  archive, the whole archive must be recreated.","ref":"zip.html#module-limitations"},{"type":"function","title":"zip.create/2","doc":"","ref":"zip.html#create/2"},{"type":"function","title":"zip.create/3","doc":"","ref":"zip.html#create/3"},{"type":"type","title":"zip.create_option/0","doc":"These options are described in [`create/3`](`m:zip#zip_options`).","ref":"zip.html#t:create_option/0"},{"type":"type","title":"zip.extension/0","doc":"A filename extension, for example \".txt\".","ref":"zip.html#t:extension/0"},{"type":"type","title":"zip.extension_spec/0","doc":"","ref":"zip.html#t:extension_spec/0"},{"type":"type","title":"zip.extra/0","doc":"The possible extra extension that can be used.\n\n- **`extended_timestamp`** - enables the 0x5455 \"extended timestamps\" zip extension\n  that embeds POSIX timestamps for access and modification times for each file in the\n  archive. This makes the timestamps to be in UTC instead of local time and also increases\n  the time resolution from 2 seconds to 1 second.\n- **`uid_gid`** - enables 0x7875 \"UNIX 3rd generation\" zip extension that embeds the\n  UID and GID for each file into the archive.","ref":"zip.html#t:extra/0"},{"type":"function","title":"zip.extract/1","doc":"","ref":"zip.html#extract/1"},{"type":"function","title":"zip.extract/2","doc":"","ref":"zip.html#extract/2"},{"type":"type","title":"zip.filename/0","doc":"The name of a zip file.","ref":"zip.html#t:filename/0"},{"type":"function","title":"zip.foldl/3","doc":"Calls `Fun(FileInArchive, GetInfo, GetBin, AccIn)` on successive files in the\n`Archive`, starting with `AccIn == Acc0`.\n\n`FileInArchive` is the name that the file has in the archive.\n\n`GetInfo` is a fun that returns information about the file.\n\n`GetBin` returns the file contents.\n\nBoth `GetInfo` and `GetBin` must be called within the `Fun`. Their behavior is\nundefined if they are called outside the context of `Fun`.\n\nThe `Fun` must return a new accumulator, which is passed to the next call.\n[`foldl/3`](`foldl/3`) returns the final accumulator value. `Acc0` is returned\nif the archive is empty. It is not necessary to iterate over all files in the\narchive. The iteration can be ended prematurely in a controlled manner by\nthrowing an exception.\n\n_Example:_\n\n```erlang\n> Name = \"dummy.zip\".\n\"dummy.zip\"\n> {ok, {Name, Bin}} = zip:create(Name, [{\"foo\", <<\"FOO\">>}, {\"bar\", <<\"BAR\">>}], [memory]).\n{ok,{\"dummy.zip\",\n     <<80,75,3,4,20,0,0,0,0,0,74,152,97,60,171,39,212,26,3,0,\n       0,0,3,0,0,...>>}}\n> {ok, FileSpec} = zip:foldl(fun(N, I, B, Acc) -> [{N, B(), I()} | Acc] end, [], {Name, Bin}).\n{ok,[{\"bar\",<<\"BAR\">>,\n      {file_info,3,regular,read_write,\n                 {{2010,3,1},{19,2,10}},\n                 {{2010,3,1},{19,2,10}},\n                 {{2010,3,1},{19,2,10}},\n                 54,1,0,0,0,0,0}},\n     {\"foo\",<<\"FOO\">>,\n      {file_info,3,regular,read_write,\n                 {{2010,3,1},{19,2,10}},\n                 {{2010,3,1},{19,2,10}},\n                 {{2010,3,1},{19,2,10}},\n                 54,1,0,0,0,0,0}}]}\n> {ok, {Name, Bin}} = zip:create(Name, lists:reverse(FileSpec), [memory]).\n{ok,{\"dummy.zip\",\n     <<80,75,3,4,20,0,0,0,0,0,74,152,97,60,171,39,212,26,3,0,\n       0,0,3,0,0,...>>}}\n> catch zip:foldl(fun(\"foo\", _, B, _) -> throw(B()); (_,_,_,Acc) -> Acc end, [], {Name, Bin}).\n<<\"FOO\">>\n```","ref":"zip.html#foldl/3"},{"type":"opaque","title":"zip.handle/0","doc":"As returned by `zip_open/2`.","ref":"zip.html#t:handle/0"},{"type":"function","title":"zip.list_dir/1","doc":"","ref":"zip.html#list_dir/1"},{"type":"function","title":"zip.list_dir/2","doc":"Retrieves all filenames in the zip archive `Archive`.\n\nThe result value is the tuple `{ok, List}`, where `List` contains the zip\narchive comment as the first element.\n\nOne option is available:\n\n- **`cooked`** - By default, this function opens the zip file in `raw` mode,\n  which is faster but does not allow a remote (Erlang) file server to be used.\n  Adding `cooked` to the mode list overrides the default and opens the zip file\n  without option `raw`.\n\n- **`skip_directories`** - By default empty directories within zip archives are\n  listed. With option `skip_directories` set, empty directories are no longer\n  listed.\n\n- **`{extra, Extras}`** - The zip \"extra\" features to respect. The supported\n  \"extra\" features are \"extended timestamps\" and \"UID and GID\" handling.\n  By default only \"extended timestamps\" is enabled when listing files.\n  See `t:extra/0` for more details.","ref":"zip.html#list_dir/2"},{"type":"function","title":"zip.t/1","doc":"Prints all filenames in the zip archive `Archive` to the Erlang shell. (Similar\nto `tar t`.)","ref":"zip.html#t/1"},{"type":"function","title":"zip.table/1","doc":"","ref":"zip.html#table/1"},{"type":"function","title":"zip.table/2","doc":"","ref":"zip.html#table/2"},{"type":"function","title":"zip.tt/1","doc":"Prints filenames and information about all files in the zip archive `Archive` to\nthe Erlang shell. (Similar to `tar tv`.)","ref":"zip.html#tt/1"},{"type":"function","title":"zip.unzip/1","doc":"","ref":"zip.html#unzip/1"},{"type":"function","title":"zip.unzip/2","doc":"Extracts all files from a zip archive.\n\nIf argument `Archive` is specified as a `t:binary/0`, the contents of the binary is\nassumed to be a zip archive, otherwise a filename.\n\nOptions:\n\n- **`{file_list, FileList}`** - By default, all files are extracted from the zip\n  archive. With option `{file_list, FileList}`, function [`unzip/2`](`unzip/2`)\n  only extracts the files whose names are included in `FileList`. The full\n  paths, including the names of all subdirectories within the zip archive, must\n  be specified.\n\n- **`cooked`** - By default, this function opens the zip file in `raw` mode,\n  which is faster but does not allow a remote (Erlang) file server to be used.\n  Adding `cooked` to the mode list overrides the default and opens the zip file\n  without option `raw`. The same applies for the files extracted.\n\n- **`keep_old_files`** - By default, all files with the same name as files in\n  the zip archive are overwritten. With option `keep_old_files` set, function\n  [`unzip/2`](`unzip/2`) does not overwrite existing files. Notice that even\n  with option `memory` specified, which means that no files are overwritten,\n  existing files are excluded from the result.\n\n- **`skip_directories`** - By default empty directories within zip archives are\n  extracted. With option `skip_directories` set, empty directories are no longer\n  created.\n\n- **`{extra, Extras}`** - The zip \"extra\" features to respect. The supported\n  \"extra\" features are \"extended timestamps\" and \"UID and GID\" handling.\n  By default only \"extended timestamps\" is enabled when unzipping.\n  See `t:extra/0` for more details.\n\n- **`verbose`** - Prints an informational message for each extracted file.\n\n- **`memory`** - Instead of extracting to the current directory, the result is\n  given as a list of tuples `{Filename, Binary}`, where `Binary` is a binary\n  containing the extracted data of file `Filename` in the zip archive.\n\n- **`{cwd, CWD}`** - Uses the specified directory as current directory. It is\n  prepended to filenames when extracting them from the zip archive. (Acting like\n  `file:set_cwd/1` in Kernel, but without changing the global `cwd` property.)","ref":"zip.html#unzip/2"},{"type":"function","title":"zip.zip/2","doc":"","ref":"zip.html#zip/2"},{"type":"function","title":"zip.zip/3","doc":"Creates a zip archive containing the files specified in `FileList`.\n\n`FileList` is a list of files, with paths relative to the current directory,\nwhich are stored with this path in the archive. File system operations are\nperformed to read the file metadata and, when compression is enabled, to stream\nthe file contents without loading whole files into memory. Files can also be\nspecified as binaries to create an archive directly from data. In such cases, no\nmetadata or file system reads are performed.\n\nFiles are compressed using the DEFLATE compression, as described in the\n\"Appnote.txt\" file. However, files are stored without compression if they are\nalready compressed. [`zip/2`](`zip/2`) and [`zip/3`](`zip/3`) check the file\nextension to determine if the file is to be stored without compression. Files\nwith the following extensions are not compressed: `.Z`, `.zip`, `.zoo`, `.arc`,\n`.lzh`, `.arj`.\n\nIt is possible to override the default behavior and control what types of files\nthat are to be compressed by using options `{compress, What}` and\n`{uncompress, What}`. It is also possible to use many `compress` and\n`uncompress` options.\n\nTo trigger file compression, its extension must match with the `compress`\ncondition and must not match the `uncompress` condition. For example, if\n`compress` is set to `[\"gif\", \"jpg\"]` and `uncompress` is set to `[\"jpg\"]`, only\nfiles with extension `\"gif\"` are compressed.\n\n[](){: #zip_options }\n\nOptions:\n\n- **`cooked`** - By default, this function opens the zip file in mode `raw`,\n  which is faster but does not allow a remote (Erlang) file server to be used.\n  Adding `cooked` to the mode list overrides the default and opens the zip file\n  without the `raw` option. The same applies for the files added.\n\n- **`verbose`** - Prints an informational message about each added file.\n\n- **`memory`** - The output is not to a file, but instead as a tuple\n  `{FileName, binary()}`. The binary is a full zip archive with header and can\n  be extracted with, for example, `unzip/2`.\n\n- **`{comment, Comment}`** - Adds a comment to the zip archive.\n\n- **`{cwd, CWD}`** - Uses the specified directory as current work directory\n  (`cwd`). This is prepended to filenames when adding them, although not in the\n  zip archive (acting like `file:set_cwd/1` in Kernel, but without changing the\n  global `cwd` property.).\n\n- **`{extra, Extras}`** - The zip \"extra\" features to respect. The supported\n  \"extra\" features are \"extended timestamps\" and \"UID and GID\" handling.\n  By default both these \"extra\" features are enabled.\n  See `t:extra/0` for more details.\n\n- **`{compress, What}`** - Controls what types of files to be compressed.\n  Defaults to `all`. The following values of `What` are allowed:\n\n  - **`all`** - All files are compressed (as long as they pass the `uncompress`\n    condition).\n\n  - **`[Extension]`** - Only files with exactly these extensions are compressed.\n\n  - **`{add,[Extension]}`** - Adds these extensions to the list of compress\n    extensions.\n\n  - **`{del,[Extension]}`** - Deletes these extensions from the list of compress\n    extensions.\n\n- **`{uncompress, What}`** - Controls what types of files to be uncompressed.\n  Defaults to `[\".Z\", \".zip\", \".zoo\", \".arc\", \".lzh\", \".arj\"]`. The following\n  values of `What` are allowed:\n\n  - **`all`** - No files are compressed.\n\n  - **`[Extension]`** - Files with these extensions are uncompressed.\n\n  - **`{add,[Extension]}`** - Adds these extensions to the list of uncompress\n    extensions.\n\n  - **`{del,[Extension]}`** - Deletes these extensions from the list of\n    uncompress extensions.","ref":"zip.html#zip/3"},{"type":"function","title":"zip.zip_close/1","doc":"Closes a zip archive, previously opened with [`zip_open/1,2`](`zip_open/1`). All\nresources are closed, and the handle is not to be used after closing.","ref":"zip.html#zip_close/1"},{"type":"type","title":"zip.zip_comment/0","doc":"The record `zip_comment` only contains the archive comment for a zip archive.","ref":"zip.html#t:zip_comment/0"},{"type":"type","title":"zip.zip_file/0","doc":"The record `zip_file` contains the following fields:\n\n- **`name`** - The filename\n\n- **`info`** - File information as in `file:read_file_info/1` in Kernel.\n  `mtime`, `atime` and `ctime` are expected to be\n  in [`local time`](`erlang:localtime/0`) if represented using `t:calendar:datetime/0`,\n  or in [OS system time](`e:erts:time_correction.md#os-system-time`) if represented by an integer.\n\n- **`comment`** - The comment for the file in the zip archive\n\n- **`offset`** - The file offset in the zip archive (used internally)\n\n- **`comp_size`** - The size of the compressed file (the size of the\n  uncompressed file is found in `info`)","ref":"zip.html#t:zip_file/0"},{"type":"function","title":"zip.zip_get/1","doc":"","ref":"zip.html#zip_get/1"},{"type":"function","title":"zip.zip_get/2","doc":"Extracts one or all files from an open archive.\n\nThe files are unzipped to memory or to file, depending on the options specified\nto function [`zip_open/1,2`](`zip_open/1`) when opening the archive.","ref":"zip.html#zip_get/2"},{"type":"function","title":"zip.zip_get_crc32/2","doc":"Extracts one crc32 checksum from an open archive.","ref":"zip.html#zip_get_crc32/2"},{"type":"function","title":"zip.zip_list_dir/1","doc":"Returns the file list of an open zip archive. The first returned element is the\nzip archive comment.","ref":"zip.html#zip_list_dir/1"},{"type":"function","title":"zip.zip_open/1","doc":"","ref":"zip.html#zip_open/1"},{"type":"function","title":"zip.zip_open/2","doc":"Opens a zip archive, and reads and saves its directory. This means that later\nreading files from the archive is faster than unzipping files one at a time with\n[`unzip/1,2`](`unzip/1`).\n\nThe options are equivalent to those in `unzip/2`.\n\nThe archive must be closed with `zip_close/1`.\n\nThe `ZipHandle` is closed if the process that originally opened the archive\ndies.","ref":"zip.html#zip_open/2"},{"type":"module","title":"beam_lib","doc":"This module provides an interface to files created by the BEAM Compiler (\"BEAM\nfiles\").\n\nThe format used, a variant of \"EA IFF 1985\" Standard for Interchange Format Files,\ndivides data into chunks.\n\nChunk data can be returned as binaries or as compound terms. Compound terms are\nreturned when chunks are referenced by names (atoms) rather than identifiers\n(strings). The recognized names and the corresponding identifiers are as\nfollows:\n\n- `atoms (\"Atom\")`\n- `attributes (\"Attr\")`\n- `compile_info (\"CInf\")`\n- `debug_info (\"Dbgi\")`\n- `exports (\"ExpT\")`\n- `imports (\"ImpT\")`\n- `indexed_imports (\"ImpT\")`\n- `labeled_exports (\"ExpT\")`\n- `labeled_locals (\"LocT\")`\n- `locals (\"LocT\")`\n- `documentation (\"Docs\")`\n\n[](){: #debug_info }","ref":"beam_lib.html"},{"type":"module","title":"Debug Information/Abstract Code - beam_lib","doc":"Option `debug_info` can be specified to the Compiler (see\n[`compile`](`m:compile#debug_info`)) to have debug information, such as\n[Erlang Abstract Format](`e:erts:absform.md`), stored in the `debug_info` chunk.\nTools such as Debugger and Xref require the debug information to be included.\n\n> #### Warning {: .warning }\n>\n> Source code can be reconstructed from the debug information. To prevent this,\n> use encrypted debug information (see below).\n\nThe debug information can also be removed from BEAM files using `strip/1`,\n`strip_files/1`, and/or `strip_release/1`.","ref":"beam_lib.html#module-debug-information-abstract-code"},{"type":"module","title":"Reconstruct Source Code - beam_lib","doc":"The following example shows how to reconstruct Erlang source code from the debug\ninformation in a BEAM file `Beam`:\n\n```erlang\n{ok,{_,[{abstract_code,{_,AC}}]}} = beam_lib:chunks(Beam,[abstract_code]).\nio:fwrite(\"~s~n\", [erl_prettypr:format(erl_syntax:form_list(AC))]).\n```","ref":"beam_lib.html#module-reconstruct-source-code"},{"type":"module","title":"Encrypted Debug Information - beam_lib","doc":"The debug information can be encrypted to keep the source code secret, but still\nbe able to use tools such as Debugger or Xref.\n\nTo use encrypted debug information, a key must be provided to the compiler and\n`beam_lib`. The key is specified as a string. It is recommended that the string\ncontains at least 32 characters and that both upper and lower case letters as\nwell as digits and special characters are used.\n\nThe default type (and currently the only type) of crypto algorithm is\n`des3_cbc`, three rounds of DES. The key string is scrambled using\n`erlang:md5/1` to generate the keys used for `des3_cbc`.\n\n> #### Note {: .info }\n>\n> As far as we know by the time of writing, it is infeasible to break `des3_cbc`\n> encryption without any knowledge of the key. Therefore, as long as the key is\n> kept safe and is unguessable, the encrypted debug information _should_ be safe\n> from intruders.\n\nThe key can be provided in the following two ways:\n\n1. Use Compiler option `{debug_info_key,Key}`, see\n   [`compile`](`m:compile#debug_info_key`) and function `crypto_key_fun/1` to\n   register a fun that returns the key whenever `beam_lib` must decrypt the\n   debug information.\n\nIf no such fun is registered, `beam_lib` instead searches for an `.erlang.crypt`\nfile, see the next section.\n\n1. Store the key in a text file named `.erlang.crypt`.\n\nIn this case, Compiler option `encrypt_debug_info` can be used, see\n[`compile`](`m:compile#encrypt_debug_info`).\n\n## .erlang.crypt\n\n`beam_lib` searches for `.erlang.crypt` in the current directory, then the\n[user's home directory](`m:init#home`) and then\n[`filename:basedir(user_config, \"erlang\")`](`m:filename#user_config`). If the\nfile is found and contains a key, `beam_lib` implicitly creates a crypto key fun\nand registers it.\n\nFile `.erlang.crypt` is to contain a single list of tuples:\n\n```erlang\n{debug_info, Mode, Module, Key}\n```\n\n`Mode` is the type of crypto algorithm; currently, the only allowed value is\n`des3_cbc`. `Module` is either an atom, in which case `Key` is only used for the\nmodule `Module`, or `[]`, in which case `Key` is used for all modules. `Key` is\nthe non-empty key string.\n\n`Key` in the first tuple where both `Mode` and `Module` match is used.\n\nThe following is an example of an `.erlang.crypt` file that returns the same key\nfor all modules:\n\n```erlang\n[{debug_info, des3_cbc, [], \"%>7}|pc/DM6Cga*68$Mw]L#&_Gejr]G^\"}].\n```\n\nThe following is a slightly more complicated example of an `.erlang.crypt`\nproviding one key for module `t` and another key for all other modules:\n\n```erlang\n[{debug_info, des3_cbc, t, \"My KEY\"},\n {debug_info, des3_cbc, [], \"%>7}|pc/DM6Cga*68$Mw]L#&_Gejr]G^\"}].\n```\n\n> #### Note {: .info }\n>\n> Do not use any of the keys in these examples. Use your own keys.","ref":"beam_lib.html#module-encrypted-debug-information"},{"type":"type","title":"beam_lib.abst_code/0","doc":"It is not checked that the forms conform to the abstract format indicated by\n`AbstVersion`. `no_abstract_code` means that chunk `\"Abst\"` is present, but\nempty.\n\nFor modules compiled with OTP 20 onwards, the `abst_code` chunk is automatically\ncomputed from the `debug_info` chunk.","ref":"beam_lib.html#t:abst_code/0"},{"type":"function","title":"beam_lib.all_chunks/1","doc":"Reads chunk data for all chunks.","ref":"beam_lib.html#all_chunks/1"},{"type":"type","title":"beam_lib.attrib_entry/0","doc":"","ref":"beam_lib.html#t:attrib_entry/0"},{"type":"type","title":"beam_lib.beam/0","doc":"Each of the functions described below accept either the filename (as a string)\nor a binary containing the BEAM module.","ref":"beam_lib.html#t:beam/0"},{"type":"function","title":"beam_lib.build_module/1","doc":"Builds a BEAM module (as a binary) from a list of chunks.","ref":"beam_lib.html#build_module/1"},{"type":"type","title":"beam_lib.chnk_rsn/0","doc":"","ref":"beam_lib.html#t:chnk_rsn/0"},{"type":"type","title":"beam_lib.chunkdata/0","doc":"The list of attributes is sorted on `Attribute` (in `t:attrib_entry/0`) and each\nattribute name occurs once in the list. The attribute values occur in the same\norder as in the file. The lists of functions are also sorted.","ref":"beam_lib.html#t:chunkdata/0"},{"type":"type","title":"beam_lib.chunkid/0","doc":"`\"Attr\" | \"CInf\" | \"Dbgi\" | \"ExpT\" | \"ImpT\" | \"LocT\" | \"AtU8\" | \"Docs\"`","ref":"beam_lib.html#t:chunkid/0"},{"type":"type","title":"beam_lib.chunkname/0","doc":"","ref":"beam_lib.html#t:chunkname/0"},{"type":"type","title":"beam_lib.chunkref/0","doc":"","ref":"beam_lib.html#t:chunkref/0"},{"type":"function","title":"beam_lib.chunks/2","doc":"Reads chunk data for selected chunks references. The order of the returned list\nof chunk data is determined by the order of the list of chunks references.","ref":"beam_lib.html#chunks/2"},{"type":"function","title":"beam_lib.chunks/3","doc":"Reads chunk data for selected chunks references. The order of the returned list\nof chunk data is determined by the order of the list of chunks references.\n\nBy default, if any requested chunk is missing in `Beam`, an `error` tuple is\nreturned. However, if option `allow_missing_chunks` is specified, a result is\nreturned even if chunks are missing. In the result list, any missing chunks are\nrepresented as `{ChunkRef,missing_chunk}`. Notice however that if chunk `\"Atom\"`\nis missing, that is considered a fatal error and the return value is an `error`\ntuple.","ref":"beam_lib.html#chunks/3"},{"type":"function","title":"beam_lib.clear_crypto_key_fun/0","doc":"Unregisters the crypto key fun and terminates the process holding it, started by\n`crypto_key_fun/1`.\n\nReturns either `{ok, undefined}` if no crypto key fun is registered, or\n`{ok, Term}`, where `Term` is the return value from `CryptoKeyFun(clear)`, see\n[`crypto_key_fun/1`](`crypto_key_fun/1`).","ref":"beam_lib.html#clear_crypto_key_fun/0"},{"type":"function","title":"beam_lib.cmp/2","doc":"Compares the contents of two BEAM files.\n\nIf the module names are the same, and all chunks except for chunk `\"CInf\"`\n(the chunk containing the compilation information that is returned by\n`Module:module_info(compile)`) have the same contents in both files, `ok` is\nreturned. Otherwise an error message is returned.","ref":"beam_lib.html#cmp/2"},{"type":"function","title":"beam_lib.cmp_dirs/2","doc":"Compares the BEAM files in two directories.\n\nOnly files with extension `\".beam\"` are compared. BEAM files that exist only in\ndirectory `Dir1` (`Dir2`) are returned in `Only1` (`Only2`). BEAM files that\nexist in both directories but are considered different by [`cmp/2`](`cmp/2`) are\n returned as pairs \\{`Filename1`, `Filename2`\\}, where `Filename1` (`Filename2`)\nexists in directory `Dir1` (`Dir2`).","ref":"beam_lib.html#cmp_dirs/2"},{"type":"type","title":"beam_lib.cmp_rsn/0","doc":"","ref":"beam_lib.html#t:cmp_rsn/0"},{"type":"type","title":"beam_lib.compinfo_entry/0","doc":"","ref":"beam_lib.html#t:compinfo_entry/0"},{"type":"type","title":"beam_lib.crypto_fun/0","doc":"","ref":"beam_lib.html#t:crypto_fun/0"},{"type":"type","title":"beam_lib.crypto_fun_arg/0","doc":"","ref":"beam_lib.html#t:crypto_fun_arg/0"},{"type":"function","title":"beam_lib.crypto_key_fun/1","doc":"Registers an unary fun that is called if `beam_lib` must read an `debug_info`\nchunk that has been encrypted. The fun is held in a process that is started by\nthe function.\n\nIf a fun is already registered when attempting to register a fun,\n`{error, exists}` is returned.\n\nThe fun must handle the following arguments:\n\n```erlang\nCryptoKeyFun(init) -> ok | {ok, NewCryptoKeyFun} | {error, Term}\n```\n\nCalled when the fun is registered, in the process that holds the fun. Here the\ncrypto key fun can do any necessary initializations. If `{ok, NewCryptoKeyFun}`\nis returned, `NewCryptoKeyFun` is registered instead of `CryptoKeyFun`. If\n`{error, Term}` is returned, the registration is aborted and\n[`crypto_key_fun/1`](`crypto_key_fun/1`) also returns `{error, Term}`.\n\n```erlang\nCryptoKeyFun({debug_info, Mode, Module, Filename}) -> Key\n```\n\nCalled when the key is needed for module `Module` in the file named `Filename`.\n`Mode` is the type of crypto algorithm; currently, the only possible value is\n`des3_cbc`. The call is to fail (raise an exception) if no key is available.\n\n```text\nCryptoKeyFun(clear) -> term()\n```\n\nCalled before the fun is unregistered. Here any cleaning up can be done. The\nreturn value is not important, but is passed back to the caller of\n`clear_crypto_key_fun/0` as part of its return value.","ref":"beam_lib.html#crypto_key_fun/1"},{"type":"type","title":"beam_lib.dataB/0","doc":"","ref":"beam_lib.html#t:dataB/0"},{"type":"type","title":"beam_lib.debug_info/0","doc":"The format stored in the `debug_info` chunk.\n\nTo retrieve particular code representation from the backend,\n`Backend:debug_info(Format, Module, Data, Opts)` must be invoked. `Format` is an\natom, such as `erlang_v1` for the Erlang Abstract Format or `core_v1` for Core\nErlang. `Module` is the module represented by the beam file and `Data` is the\nvalue stored in the debug info chunk. `Opts` is any list of values supported by\nthe `Backend`. `Backend:debug_info/4` must return `{ok, Code}` or\n`{error, Term}`.\n\nDevelopers must always invoke the `debug_info/4` function and never rely on the\n`Data` stored in the `debug_info` chunk, as it is opaque and may change at any\nmoment. `no_debug_info` means that chunk `\"Dbgi\"` is present, but empty.","ref":"beam_lib.html#t:debug_info/0"},{"type":"function","title":"beam_lib.diff_dirs/2","doc":"Compares the BEAM files in two directories as `cmp_dirs/2`, but the names of\nfiles that exist in only one directory or are different are presented on\nstandard output.","ref":"beam_lib.html#diff_dirs/2"},{"type":"type","title":"beam_lib.docs/0","doc":"[EEP-48 documentation format](`e:kernel:eep48_chapter.md#the-docs-format`)","ref":"beam_lib.html#t:docs/0"},{"type":"function","title":"beam_lib.format_error/1","doc":"For a specified error returned by any function in this module, this function\nreturns a descriptive string of the error in English. For file errors, function\n[`file:format_error(Posix)`](`file:format_error/1`) is to be called.","ref":"beam_lib.html#format_error/1"},{"type":"type","title":"beam_lib.forms/0","doc":"","ref":"beam_lib.html#t:forms/0"},{"type":"type","title":"beam_lib.index/0","doc":"","ref":"beam_lib.html#t:index/0"},{"type":"function","title":"beam_lib.info/1","doc":"Returns a list containing some information about a BEAM file as tuples\n`{Item, Info}`:\n\n- **`{file, Filename} | {binary, Binary}`** - The name (string) of the BEAM\n  file, or the binary from which the information was extracted.\n\n- **`{module, Module}`** - The name (atom) of the module.\n\n- **`{chunks, [{ChunkId, Pos, Size}]}`** - For each chunk, the identifier\n  (string) and the position and size of the chunk data, in bytes.","ref":"beam_lib.html#info/1"},{"type":"type","title":"beam_lib.info_rsn/0","doc":"","ref":"beam_lib.html#t:info_rsn/0"},{"type":"type","title":"beam_lib.label/0","doc":"","ref":"beam_lib.html#t:label/0"},{"type":"type","title":"beam_lib.labeled_entry/0","doc":"","ref":"beam_lib.html#t:labeled_entry/0"},{"type":"function","title":"beam_lib.md5/1","doc":"Calculates an MD5 redundancy check for the code of the module (compilation date\nand other attributes are not included).","ref":"beam_lib.html#md5/1"},{"type":"type","title":"beam_lib.mode/0","doc":"","ref":"beam_lib.html#t:mode/0"},{"type":"function","title":"beam_lib.strip/1","doc":"Removes all chunks from a BEAM file except those used by the loader.\n\nIn particular, the debug information (chunk `debug_info` and `abstract_code`) is\nremoved.","ref":"beam_lib.html#strip/1"},{"type":"function","title":"beam_lib.strip/2","doc":"Removes all chunks from a BEAM file except those used by the loader or mentioned\nin `AdditionalChunks`.\n\nIn particular, the debug information (chunk `debug_info` and `abstract_code`) is removed.","ref":"beam_lib.html#strip/2"},{"type":"function","title":"beam_lib.strip_files/1","doc":"Removes all chunks except those used by the loader from `Files`.\n\nIn particular, the debug information (chunk `debug_info` and `abstract_code`) is\nremoved. The returned list contains one element for each specified filename, in\nthe same order as in `Files`.","ref":"beam_lib.html#strip_files/1"},{"type":"function","title":"beam_lib.strip_files/2","doc":"Removes all chunks except those used by the loader or mentioned in\n`AdditionalChunks` from `Files`.\n\nIn particular, the debug information (chunk `debug_info` and `abstract_code`) is\nremoved. The returned list contains one element for each specified filename,\nin the same order as in `Files`.","ref":"beam_lib.html#strip_files/2"},{"type":"function","title":"beam_lib.strip_release/1","doc":"Removes all chunks except those used by the loader from the BEAM files of a\nrelease.\n\n`Dir` is to be the installation root directory. For example, the current OTP\nrelease can be stripped with the call `beam_lib:strip_release(code:root_dir())`.","ref":"beam_lib.html#strip_release/1"},{"type":"function","title":"beam_lib.strip_release/2","doc":"Removes all chunks except those used by the loader or mentioned in\n`AdditionalChunks`.\n\n`Dir` is to be the installation root directory. For example, the current OTP\nrelease can be stripped with the call `beam_lib:strip_release(code:root_dir(),[documentation])`.","ref":"beam_lib.html#strip_release/2"},{"type":"function","title":"beam_lib.version/1","doc":"Returns the module version or versions. A version is defined by module attribute\n`-vsn(Vsn)`.\n\nIf this attribute is not specified, the version defaults to the\nchecksum of the module. Notice that if version `Vsn` is not a list, it is made\ninto one, that is `{ok,{Module,[Vsn]}}` is returned. If there are many `-vsn`\nmodule attributes, the result is the concatenated list of versions.\n\n_Examples:_\n\n```erlang\n1> beam_lib:version(a). % -vsn(1).\n{ok,{a,[1]}}\n2> beam_lib:version(b). % -vsn([1]).\n{ok,{b,[1]}}\n3> beam_lib:version(c). % -vsn([1]). -vsn(2).\n{ok,{c,[1,2]}}\n4> beam_lib:version(d). % no -vsn attribute\n{ok,{d,[275613208176997377698094100858909383631]}}\n```","ref":"beam_lib.html#version/1"},{"type":"module","title":"epp","doc":"An Erlang code preprocessor.\n\nThe Erlang code preprocessor includes functions that are used by the `m:compile`\nmodule to preprocess macros and include files before the parsing takes place.\n\nThe Erlang source file _encoding_{: #encoding } is selected by a comment in one\nof the first two lines of the source file. The first string matching the regular\nexpression `coding\\s*[:=]\\s*([-a-zA-Z0-9])+` selects the encoding. If the\nmatching string is not a valid encoding, it is ignored. The valid encodings are\n`Latin-1` and `UTF-8`, where the case of the characters can be chosen freely.\n\n_Examples:_\n\n```erlang\n%% coding: utf-8\n```\n\n```erlang\n%% For this file we have chosen encoding = Latin-1\n```\n\n```erlang\n%% -*- coding: latin-1 -*-\n```","ref":"epp.html"},{"type":"module","title":"Error Information - epp","doc":"`ErrorInfo` is the standard `ErrorInfo` structure that is returned from all I/O\nmodules. The format is as follows:\n\n```erlang\n{ErrorLine, Module, ErrorDescriptor}\n```\n\nA string describing the error is obtained with the following call:\n\n```erlang\nModule:format_error(ErrorDescriptor)\n```","ref":"epp.html#module-error-information"},{"type":"module","title":"See Also - epp","doc":"`m:erl_parse`","ref":"epp.html#module-see-also"},{"type":"function","title":"epp.close/1","doc":"Closes the preprocessing of a file.","ref":"epp.html#close/1"},{"type":"function","title":"epp.default_encoding/0","doc":"Returns the default encoding of Erlang source files.","ref":"epp.html#default_encoding/0"},{"type":"function","title":"epp.encoding_to_string/1","doc":"Returns a string representation of an encoding. The string is recognized by\n[`read_encoding/1,2`](`read_encoding/1`),\n[`read_encoding_from_binary/1,2`](`read_encoding_from_binary/1`), and\n[`set_encoding/1,2`](`set_encoding/1`) as a valid encoding.","ref":"epp.html#encoding_to_string/1"},{"type":"type","title":"epp.epp_handle/0","doc":"Handle to the `epp` server.","ref":"epp.html#t:epp_handle/0"},{"type":"function","title":"epp.format_error/1","doc":"Takes an `ErrorDescriptor` and returns a string that describes the error or\nwarning. This function is usually called implicitly when processing an\n`ErrorInfo` structure (see section [Error Information](`m:epp#module-error-information`)).","ref":"epp.html#format_error/1"},{"type":"type","title":"epp.macros/0","doc":"","ref":"epp.html#t:macros/0"},{"type":"function","title":"epp.open/1","doc":"Opens a file for preprocessing.\n\nIf you want to change the file name of the implicit -file() attributes inserted\nduring preprocessing, you can do with `{source_name, SourceName}`. If unset it\nwill default to the name of the opened file.\n\nSetting `{deterministic, Enabled}` will additionally reduce the file name of the\nimplicit -file() attributes inserted during preprocessing to only the basename\nof the path.\n\nIf `extra` is specified in `Options`, the return value is `{ok, Epp, Extra}`\ninstead of `{ok, Epp}`.\n\nThe option `location` is forwarded to the Erlang token scanner, see\n[`erl_scan:tokens/3,4`](`erl_scan:tokens/3`).\n\nThe `{compiler_internal,term()}` option is forwarded to the Erlang token\nscanner, see [`{compiler_internal,term()}`](`m:erl_scan#compiler_interal`).","ref":"epp.html#open/1"},{"type":"function","title":"epp.open/2","doc":"Equivalent to `epp:open([{name, FileName}, {includes, IncludePath}])`.","ref":"epp.html#open/2"},{"type":"function","title":"epp.open/3","doc":"Equivalent to\n`epp:open([{name, FileName}, {includes, IncludePath}, {macros, PredefMacros}])`.","ref":"epp.html#open/3"},{"type":"function","title":"epp.parse_erl_form/1","doc":"Returns the next Erlang form from the opened Erlang source file. Tuple\n`{eof, Location}` is returned at the end of the file. The first form corresponds\nto an implicit attribute `-file(File,1).`, where `File` is the file name.","ref":"epp.html#parse_erl_form/1"},{"type":"function","title":"epp.parse_file/2","doc":"Preprocesses and parses an Erlang source file. Notice that tuple\n`{eof, Location}` returned at the end of the file is included as a \"form\".\n\nIf you want to change the file name of the implicit -file() attributes inserted\nduring preprocessing, you can do with `{source_name, SourceName}`. If unset it\nwill default to the name of the opened file.\n\nIf `extra` is specified in `Options`, the return value is `{ok, [Form], Extra}`\ninstead of `{ok, [Form]}`.\n\nThe option `location` is forwarded to the Erlang token scanner, see\n[`erl_scan:tokens/3,4`](`erl_scan:tokens/3`).\n\nThe `{compiler_internal,term()}` option is forwarded to the Erlang token\nscanner, see [`{compiler_internal,term()}`](`m:erl_scan#compiler_interal`).","ref":"epp.html#parse_file/2"},{"type":"function","title":"epp.parse_file/3","doc":"Equivalent to\n`epp:parse_file(FileName, [{includes, IncludePath}, {macros, PredefMacros}])`.","ref":"epp.html#parse_file/3"},{"type":"function","title":"epp.read_encoding/1","doc":"","ref":"epp.html#read_encoding/1"},{"type":"function","title":"epp.read_encoding/2","doc":"Read the [encoding](`m:epp#encoding`) from a file. Returns the read encoding, or\n`none` if no valid encoding is found.\n\nOption `in_comment_only` is `true` by default, which is correct for Erlang\nsource files. If set to `false`, the encoding string does not necessarily have\nto occur in a comment.","ref":"epp.html#read_encoding/2"},{"type":"function","title":"epp.read_encoding_from_binary/1","doc":"","ref":"epp.html#read_encoding_from_binary/1"},{"type":"function","title":"epp.read_encoding_from_binary/2","doc":"Read the [encoding](`m:epp#encoding`) from a binary. Returns the read encoding,\nor `none` if no valid encoding is found.\n\nOption `in_comment_only` is `true` by default, which is correct for Erlang\nsource files. If set to `false`, the encoding string does not necessarily have\nto occur in a comment.","ref":"epp.html#read_encoding_from_binary/2"},{"type":"function","title":"epp.scan_erl_form/1","doc":"Returns the raw tokens of the next Erlang form from the opened Erlang source\nfile. A tuple `{eof, Line}` is returned at the end of the file. The first form\ncorresponds to an implicit attribute `-file(File,1).`, where `File` is the file\nname.","ref":"epp.html#scan_erl_form/1"},{"type":"function","title":"epp.scan_file/2","doc":"Preprocesses an Erlang source file returning a list of the lists of raw tokens\nof each form. Notice that the tuple `{eof, Line}` returned at the end of the\nfile is included as a \"form\", and any failures to scan a form are included in\nthe list as tuples `{error, ErrorInfo}`.","ref":"epp.html#scan_file/2"},{"type":"function","title":"epp.set_encoding/1","doc":"Reads the [encoding](`m:epp#encoding`) from an I/O device and sets the encoding\nof the device accordingly. The position of the I/O device referenced by `File`\nis not affected. If no valid encoding can be read from the I/O device, the\nencoding of the I/O device is set to the default encoding.\n\nReturns the read encoding, or `none` if no valid encoding is found.","ref":"epp.html#set_encoding/1"},{"type":"function","title":"epp.set_encoding/2","doc":"Reads the [encoding](`m:epp#encoding`) from an I/O device and sets the encoding\nof the device accordingly. The position of the I/O device referenced by `File`\nis not affected. If no valid encoding can be read from the I/O device, the\nencoding of the I/O device is set to the [encoding](`m:epp#encoding`) specified\nby `Default`.\n\nReturns the read encoding, or `none` if no valid encoding is found.","ref":"epp.html#set_encoding/2"},{"type":"type","title":"epp.source_encoding/0","doc":"","ref":"epp.html#t:source_encoding/0"},{"type":"type","title":"epp.warning_info/0","doc":"","ref":"epp.html#t:warning_info/0"},{"type":"module","title":"erl_anno","doc":"Abstract datatype for the annotations of the Erlang Compiler.\n\nThis module provides an abstract type that is used by the Erlang Compiler and\nits helper modules for holding data such as column, line number, and text. The\ndata type is a collection of _annotations_{: #annotations } as described in the\nfollowing.\n\nThe Erlang Token Scanner returns tokens with a subset of the following\nannotations, depending on the options:\n\n- **`column`** - The column where the token begins.\n\n- **`location`** - The line and column where the token begins, or just the line\n  if the column is unknown.\n\n- **`text`** - The token's text.\n\nFrom this, the following annotation is derived:\n\n- **`line`** - The line where the token begins.\n\nThis module also supports the following annotations, which are used by various\nmodules:\n\n- **`file`** - A filename.\n\n- **`generated`** - A Boolean indicating if the abstract code is\n  compiler-generated. The Erlang Compiler does not emit warnings for such code.\n\n- **`record`** - A Boolean indicating if the origin of the abstract code is a\n  record. Used by [Dialyzer](`m:dialyzer`) to assign types to tuple elements.\n\nThe functions [`column()`](`erl_scan:column/1`),\n[`end_location()`](`erl_scan:end_location/1`), [`line()`](`erl_scan:line/1`),\n[`location()`](`erl_scan:location/1`), and [`text()`](`erl_scan:text/1`) in the\n`erl_scan` module can be used for inspecting annotations in tokens.\n\nThe functions [`anno_from_term()`](`erl_parse:anno_from_term/1`),\n[`anno_to_term()`](`erl_parse:anno_to_term/1`),\n[`fold_anno()`](`erl_parse:fold_anno/3`),\n[`map_anno()`](`erl_parse:map_anno/2`),\n[`mapfold_anno()`](`erl_parse:mapfold_anno/3`), and\n[`new_anno()`](`erl_parse:new_anno/1`), in the `erl_parse` module can be used\nfor manipulating annotations in abstract code.","ref":"erl_anno.html"},{"type":"module","title":"See Also - erl_anno","doc":"`m:erl_parse`, `m:erl_scan`","ref":"erl_anno.html#module-see-also"},{"type":"opaque","title":"erl_anno.anno/0","doc":"A collection of annotations.","ref":"erl_anno.html#t:anno/0"},{"type":"type","title":"erl_anno.anno_term/0","doc":"The term representing a collection of annotations. It is either a `t:location/0`\nor a list of key-value pairs.","ref":"erl_anno.html#t:anno_term/0"},{"type":"type","title":"erl_anno.column/0","doc":"","ref":"erl_anno.html#t:column/0"},{"type":"function","title":"erl_anno.column/1","doc":"Returns the column of the annotations Anno.","ref":"erl_anno.html#column/1"},{"type":"function","title":"erl_anno.end_location/1","doc":"Returns the end location of the text of the annotations Anno. If there is no\ntext, `undefined` is returned.","ref":"erl_anno.html#end_location/1"},{"type":"function","title":"erl_anno.file/1","doc":"Returns the filename of the annotations Anno. If there is no filename,\n`undefined` is returned.","ref":"erl_anno.html#file/1"},{"type":"type","title":"erl_anno.filename/0","doc":"","ref":"erl_anno.html#t:filename/0"},{"type":"function","title":"erl_anno.from_term/1","doc":"Returns annotations with representation Term.\n\nSee also [to_term()](`to_term/1`).","ref":"erl_anno.html#from_term/1"},{"type":"type","title":"erl_anno.generated/0","doc":"","ref":"erl_anno.html#t:generated/0"},{"type":"function","title":"erl_anno.generated/1","doc":"Returns `true` if annotations Anno is marked as generated. The default is to\nreturn `false`.","ref":"erl_anno.html#generated/1"},{"type":"function","title":"erl_anno.is_anno/1","doc":"Returns `true` if Term is a collection of annotations, otherwise `false`.","ref":"erl_anno.html#is_anno/1"},{"type":"type","title":"erl_anno.line/0","doc":"","ref":"erl_anno.html#t:line/0"},{"type":"function","title":"erl_anno.line/1","doc":"Returns the line of the annotations Anno.","ref":"erl_anno.html#line/1"},{"type":"type","title":"erl_anno.location/0","doc":"","ref":"erl_anno.html#t:location/0"},{"type":"function","title":"erl_anno.location/1","doc":"Returns the location of the annotations Anno.","ref":"erl_anno.html#location/1"},{"type":"function","title":"erl_anno.new/1","doc":"Creates a new collection of annotations given a location.","ref":"erl_anno.html#new/1"},{"type":"type","title":"erl_anno.record/0","doc":"","ref":"erl_anno.html#t:record/0"},{"type":"function","title":"erl_anno.set_file/2","doc":"Modifies the filename of the annotations Anno.","ref":"erl_anno.html#set_file/2"},{"type":"function","title":"erl_anno.set_generated/2","doc":"Modifies the generated marker of the annotations Anno.","ref":"erl_anno.html#set_generated/2"},{"type":"function","title":"erl_anno.set_line/2","doc":"Modifies the line of the annotations Anno.","ref":"erl_anno.html#set_line/2"},{"type":"function","title":"erl_anno.set_location/2","doc":"Modifies the location of the annotations Anno.","ref":"erl_anno.html#set_location/2"},{"type":"function","title":"erl_anno.set_record/2","doc":"Modifies the record marker of the annotations Anno.","ref":"erl_anno.html#set_record/2"},{"type":"function","title":"erl_anno.set_text/2","doc":"Modifies the text of the annotations Anno.","ref":"erl_anno.html#set_text/2"},{"type":"type","title":"erl_anno.text/0","doc":"","ref":"erl_anno.html#t:text/0"},{"type":"function","title":"erl_anno.text/1","doc":"Returns the text of the annotations Anno. If there is no text, `undefined` is\nreturned.","ref":"erl_anno.html#text/1"},{"type":"function","title":"erl_anno.to_term/1","doc":"Returns the term representing the annotations Anno.\n\nSee also [from_term()](`from_term/1`).","ref":"erl_anno.html#to_term/1"},{"type":"module","title":"erl_eval","doc":"The Erlang meta interpreter.\n\nThis module provides an interpreter for Erlang expressions. The expressions are\nin the abstract syntax as returned by `m:erl_parse`, the Erlang parser, or\n`m:io`.","ref":"erl_eval.html"},{"type":"module","title":"Local Function Handler - erl_eval","doc":"During evaluation of a function, no calls can be made to local functions. An\nundefined function error would be generated. However, the optional argument\n`LocalFunctionHandler` can be used to define a function that is called when\nthere is a call to a local function. The argument can have the following\nformats:\n\n- **`{value,Func}`** - This defines a local function handler that is called\n  with:\n\n  ```erlang\n  Func(Name, Arguments)\n  ```\n\n  `Name` is the name of the local function (an atom) and `Arguments` is a list\n  of the _evaluated_ arguments. The function handler returns the value of the\n  local function. In this case, the current bindings cannot be accessed. To\n  signal an error, the function handler calls [`exit/1`](`exit/1`) with a\n  suitable exit value.\n\n- **`{eval,Func}`** - This defines a local function handler that is called with:\n\n  ```erlang\n  Func(Name, Arguments, Bindings)\n  ```\n\n  `Name` is the name of the local function (an atom), `Arguments` is a list of\n  the _unevaluated_ arguments, and `Bindings` are the current variable bindings.\n  The function handler returns:\n\n  ```erlang\n  {value,Value,NewBindings}\n  ```\n\n  `Value` is the value of the local function and `NewBindings` are the updated\n  variable bindings. In this case, the function handler must itself evaluate all\n  the function arguments and manage the bindings. To signal an error, the\n  function handler calls [`exit/1`](`exit/1`) with a suitable exit value.\n\n- **`none`** - There is no local function handler.","ref":"erl_eval.html#module-local-function-handler"},{"type":"module","title":"Non-Local Function Handler - erl_eval","doc":"The optional argument `NonLocalFunctionHandler` can be used to define a function\nthat is called in the following cases:\n\n- A functional object (fun) is called.\n- A built-in function is called.\n- A function is called using the `M:F` syntax, where `M` and `F` are atoms or\n  expressions.\n- An operator `Op/A` is called (this is handled as a call to function\n  `erlang:Op/A`).\n\nExceptions are calls to `erlang:apply/2,3`; neither of the function handlers are\ncalled for such calls. The argument can have the following formats:\n\n- **`{value,Func}`** - This defines a non-local function handler. The function\n  may be called with two arguments:\n\n  ```erlang\n  Func(FuncSpec, Arguments)\n  ```\n\n  or three arguments:\n\n  ```erlang\n  Func(Anno, FuncSpec, Arguments)\n  ```\n\n  `Anno` is the [`erl_anno:anno()`](`t:erl_anno:anno/0`) of the node, `FuncSpec`\n  is the name of the function on the form `{Module,Function}` or a fun, and\n  `Arguments` is a list of the _evaluated_ arguments. The function handler\n  returns the value of the function. To signal an error, the function handler\n  calls [`exit/1`](`exit/1`) with a suitable exit value.\n\n- **`none`** - There is no non-local function handler.\n\n> #### Note {: .info }\n>\n> For calls such as `erlang:apply(Fun, Args)` or\n> `erlang:apply(Module, Function, Args)`, the call of the non-local function\n> handler corresponding to the call to `erlang:apply/2,3` itself\n> (`Func({erlang, apply}, [Fun, Args])` or\n> `Func({erlang, apply}, [Module, Function, Args])`) never takes place.\n>\n> The non-local function handler _is_ however called with the evaluated\n> arguments of the call to `erlang:apply/2,3`: `Func(Fun, Args)` or\n> `Func({Module, Function}, Args)` (assuming that `{Module, Function}` is not\n> `{erlang, apply}`).\n>\n> Calls to functions defined by evaluating fun expressions `\"fun ... end\"` are\n> also hidden from non-local function handlers.\n\nThe non-local function handler argument is probably not used as frequently as\nthe local function handler argument. A possible use is to call\n[`exit/1`](`exit/1`) on calls to functions that for some reason are not allowed\nto be called.","ref":"erl_eval.html#module-non-local-function-handler"},{"type":"function","title":"erl_eval.add_binding/3","doc":"Adds binding `Name=Value` to `BindingStruct`. Returns an updated binding\nstructure.","ref":"erl_eval.html#add_binding/3"},{"type":"function","title":"erl_eval.binding/2","doc":"Returns the binding of `Name` in `BindingStruct`.","ref":"erl_eval.html#binding/2"},{"type":"type","title":"erl_eval.binding_struct/0","doc":"A binding structure. It is either a `map` or an `orddict`. `erl_eval` will\nalways return the same type as the one given.","ref":"erl_eval.html#t:binding_struct/0"},{"type":"type","title":"erl_eval.bindings/0","doc":"","ref":"erl_eval.html#t:bindings/0"},{"type":"function","title":"erl_eval.bindings/1","doc":"Returns the list of bindings contained in the binding structure.","ref":"erl_eval.html#bindings/1"},{"type":"function","title":"erl_eval.del_binding/2","doc":"Removes the binding of `Name` in `BindingStruct`. Returns an updated binding\nstructure.","ref":"erl_eval.html#del_binding/2"},{"type":"function","title":"erl_eval.expr/2","doc":"","ref":"erl_eval.html#expr/2"},{"type":"function","title":"erl_eval.expr/3","doc":"","ref":"erl_eval.html#expr/3"},{"type":"function","title":"erl_eval.expr/4","doc":"","ref":"erl_eval.html#expr/4"},{"type":"function","title":"erl_eval.expr/5","doc":"Evaluates `Expression` with the set of bindings `Bindings`. `Expression` is an\nexpression in abstract syntax.\n\nFor an explanation of when and how to use arguments `LocalFunctionHandler` and\n`NonLocalFunctionHandler`, see sections\n[Local Function Handler](`m:erl_eval#module-local-function-handler`) and\n[Non-Local Function Handler](`m:erl_eval#module-non-local-function-handler`) in this\nmodule.\n\nReturns `{value, Value, NewBindings}` by default. If `ReturnFormat` is `value`,\nonly `Value` is returned.","ref":"erl_eval.html#expr/5"},{"type":"function","title":"erl_eval.expr_list/2","doc":"","ref":"erl_eval.html#expr_list/2"},{"type":"function","title":"erl_eval.expr_list/3","doc":"","ref":"erl_eval.html#expr_list/3"},{"type":"function","title":"erl_eval.expr_list/4","doc":"Evaluates a list of expressions in parallel, using the same initial bindings for\neach expression. Attempts are made to merge the bindings returned from each\nevaluation.\n\nThis function is useful in `LocalFunctionHandler`, see section\n[Local Function Handler](`m:erl_eval#module-local-function-handler`) in this module.\n\nReturns `{ValueList, NewBindings}`.","ref":"erl_eval.html#expr_list/4"},{"type":"type","title":"erl_eval.expression/0","doc":"","ref":"erl_eval.html#t:expression/0"},{"type":"type","title":"erl_eval.expression_list/0","doc":"","ref":"erl_eval.html#t:expression_list/0"},{"type":"type","title":"erl_eval.expressions/0","doc":"As returned by `erl_parse:parse_exprs/1` or `io:parse_erl_exprs/2`.","ref":"erl_eval.html#t:expressions/0"},{"type":"function","title":"erl_eval.exprs/2","doc":"","ref":"erl_eval.html#exprs/2"},{"type":"function","title":"erl_eval.exprs/3","doc":"","ref":"erl_eval.html#exprs/3"},{"type":"function","title":"erl_eval.exprs/4","doc":"Evaluates `Expressions` with the set of bindings `Bindings`, where `Expressions`\nis a sequence of expressions (in abstract syntax) of a type that can be returned\nby `io:parse_erl_exprs/2`.\n\nFor an explanation of when and how to use arguments\n`LocalFunctionHandler` and `NonLocalFunctionHandler`, see sections\n[Local Function Handler](`m:erl_eval#module-local-function-handler`) and\n[Non-Local Function Handler](`m:erl_eval#module-non-local-function-handler`) in this\nmodule.\n\nReturns `{value, Value, NewBindings}`","ref":"erl_eval.html#exprs/4"},{"type":"type","title":"erl_eval.func_spec/0","doc":"","ref":"erl_eval.html#t:func_spec/0"},{"type":"type","title":"erl_eval.lfun_eval_handler/0","doc":"","ref":"erl_eval.html#t:lfun_eval_handler/0"},{"type":"type","title":"erl_eval.lfun_value_handler/0","doc":"","ref":"erl_eval.html#t:lfun_value_handler/0"},{"type":"type","title":"erl_eval.local_function_handler/0","doc":"Further described in section\n[Local Function Handler](`m:erl_eval#module-local-function-handler`) in this module","ref":"erl_eval.html#t:local_function_handler/0"},{"type":"type","title":"erl_eval.name/0","doc":"","ref":"erl_eval.html#t:name/0"},{"type":"function","title":"erl_eval.new_bindings/0","doc":"Returns an empty binding structure.","ref":"erl_eval.html#new_bindings/0"},{"type":"type","title":"erl_eval.nlfun_handler/0","doc":"","ref":"erl_eval.html#t:nlfun_handler/0"},{"type":"type","title":"erl_eval.non_local_function_handler/0","doc":"Further described in section\n[Non-Local Function Handler](`m:erl_eval#module-non-local-function-handler`) in this\nmodule.","ref":"erl_eval.html#t:non_local_function_handler/0"},{"type":"type","title":"erl_eval.value/0","doc":"","ref":"erl_eval.html#t:value/0"},{"type":"module","title":"erl_expand_records","doc":"This module expands records in a module.","ref":"erl_expand_records.html"},{"type":"module","title":"See Also - erl_expand_records","doc":"Section [The Abstract Format](`e:erts:absform.md`) in ERTS User's Guide.","ref":"erl_expand_records.html#module-see-also"},{"type":"function","title":"erl_expand_records.module/2","doc":"Expands all records in a module to use explicit tuple operations and adds\nexplicit module names to calls to BIFs and imported functions. The returned\nmodule has no references to records, attributes, or code.","ref":"erl_expand_records.html#module/2"},{"type":"module","title":"erl_features","doc":"This module contains functions for supporting features that can be\nenabled/disabled in Erlang.\n\nIt should be considered as mostly for internal use, although there are some\nfunctions that might be useful when writing tools.","ref":"erl_features.html"},{"type":"function","title":"erl_features.all/0","doc":"Return a list of all known features. This list will include features that have\nbeen removed (status `rejected`) and features that are no longer configurable\n(status `permanent`).","ref":"erl_features.html#all/0"},{"type":"function","title":"erl_features.configurable/0","doc":"Return a list of all configurable features, that is, features with status\n`experimental` or `approved`. These are the features that can be enabled or\ndisabled.","ref":"erl_features.html#configurable/0"},{"type":"function","title":"erl_features.enabled/0","doc":"Return a list of the features that are currently enabled. Note that the set of\nenabled is set during startup and can then not be changed.","ref":"erl_features.html#enabled/0"},{"type":"type","title":"erl_features.feature/0","doc":"","ref":"erl_features.html#t:feature/0"},{"type":"function","title":"erl_features.info/1","doc":"Return a map containing information about the given feature.","ref":"erl_features.html#info/1"},{"type":"type","title":"erl_features.release/0","doc":"","ref":"erl_features.html#t:release/0"},{"type":"type","title":"erl_features.status/0","doc":"","ref":"erl_features.html#t:status/0"},{"type":"type","title":"erl_features.type/0","doc":"","ref":"erl_features.html#t:type/0"},{"type":"function","title":"erl_features.used/1","doc":"Return the list of features enabled when compiling the module. The module need\nnot be loaded, but is found if it exists in the loadpath. If not all features\nused by the module are enabled in the runtime, loading the module is not\nallowed.","ref":"erl_features.html#used/1"},{"type":"module","title":"erl_id_trans","doc":"This module performs an identity parse transformation of Erlang code.\n\nIt is included as an example for users who wants to write their own\nparse transformers. If option `{parse_transform,Module}` is passed\nto the compiler, a user-written function `parse_transform/2`\nis called by the compiler before the code is checked for errors.\n\nBefore the function `parse_transform/2` is called, the Erlang\nCompiler checks if the parse transformation can handle abstract code\nwith column numbers: If the function `parse_transform_info/0`\nis implemented and returns a map where the key `error_location` is\nassociated with the value `line`, the compiler removes\ncolumn numbers from the abstract code before calling the parse\ntransform. Otherwise, the compiler passes the abstract code on\nwithout modification.","ref":"erl_id_trans.html"},{"type":"module","title":"Parse Transformations - erl_id_trans","doc":"Parse transformations are used if a programmer wants to use\nErlang syntax, but with different semantics. The original Erlang\ncode is then transformed into other Erlang code.\n\n> #### Note {: .info }\n>\n> Programmers are strongly advised not to engage in parse\n> transformations. No support is offered for problems encountered.\n>","ref":"erl_id_trans.html#module-parse-transformations"},{"type":"module","title":"See Also - erl_id_trans","doc":"`m:erl_parse` and `m:compile`.","ref":"erl_id_trans.html#module-see-also"},{"type":"function","title":"erl_id_trans.parse_transform/2","doc":"Performs an identity transformation on Erlang forms, as an example.","ref":"erl_id_trans.html#parse_transform/2"},{"type":"function","title":"erl_id_trans.parse_transform_info/0","doc":"Returns information about the parse transform itself.","ref":"erl_id_trans.html#parse_transform_info/0"},{"type":"module","title":"erl_internal","doc":"Internal Erlang definitions.\n\nThis module defines Erlang BIFs, guard tests, and operators. This module is only\nof interest to programmers who manipulate Erlang code.","ref":"erl_internal.html"},{"type":"function","title":"erl_internal.add_predefined_functions/1","doc":"Adds to `Forms` the code for the standard pre-defined functions (such as\n`module_info/0`) that are to be included in every module.","ref":"erl_internal.html#add_predefined_functions/1"},{"type":"function","title":"erl_internal.arith_op/2","doc":"Returns `true` if `OpName/Arity` is an arithmetic operator, otherwise `false`.","ref":"erl_internal.html#arith_op/2"},{"type":"function","title":"erl_internal.bif/2","doc":"Returns `true` if `Name/Arity` is an Erlang BIF that is automatically recognized\nby the compiler, otherwise `false`.","ref":"erl_internal.html#bif/2"},{"type":"function","title":"erl_internal.bool_op/2","doc":"Returns `true` if `OpName/Arity` is a Boolean operator, otherwise `false`.","ref":"erl_internal.html#bool_op/2"},{"type":"function","title":"erl_internal.comp_op/2","doc":"Returns `true` if `OpName/Arity` is a comparison operator, otherwise `false`.","ref":"erl_internal.html#comp_op/2"},{"type":"function","title":"erl_internal.guard_bif/2","doc":"Returns `true` if `Name/Arity` is an Erlang BIF that is allowed in guards,\notherwise `false`.","ref":"erl_internal.html#guard_bif/2"},{"type":"function","title":"erl_internal.list_op/2","doc":"Returns `true` if `OpName/Arity` is a list operator, otherwise `false`.","ref":"erl_internal.html#list_op/2"},{"type":"function","title":"erl_internal.op_type/2","doc":"Returns the `Type` of operator that `OpName/Arity` belongs to, or generates a\n`function_clause` error if it is not an operator.","ref":"erl_internal.html#op_type/2"},{"type":"function","title":"erl_internal.send_op/2","doc":"Returns `true` if `OpName/Arity` is a send operator, otherwise `false`.","ref":"erl_internal.html#send_op/2"},{"type":"function","title":"erl_internal.type_test/2","doc":"Returns `true` if `Name/Arity` is a valid Erlang type test, otherwise `false`.","ref":"erl_internal.html#type_test/2"},{"type":"module","title":"erl_lint","doc":"The Erlang code linter.\n\nThis module is used to check Erlang code for illegal syntax and other bugs. It\nalso warns against coding practices that are not recommended.\n\nThe errors detected include:\n\n- Redefined and undefined functions\n- Unbound and unsafe variables\n- Illegal record use\n\nThe warnings detected include:\n\n- Unused functions and imports\n- Unused variables\n- Variables imported into matches\n- Variables exported from `if`/`case`/`receive`\n- Variables shadowed in funs and list comprehensions\n\nSome of the warnings are optional, and can be turned on by specifying the\nappropriate option, described below.\n\nThe functions in this module are invoked automatically by the Erlang compiler.\nThere is no reason to invoke these functions separately unless you have written\nyour own Erlang compiler.","ref":"erl_lint.html"},{"type":"module","title":"Error Information - erl_lint","doc":"`ErrorInfo` is the standard `ErrorInfo` structure that is returned from all I/O\nmodules. The format is as follows:\n\n```erlang\n{ErrorLine, Module, ErrorDescriptor}\n```\n\nA string describing the error is obtained with the following call:\n\n```erlang\nModule:format_error(ErrorDescriptor)\n```","ref":"erl_lint.html#module-error-information"},{"type":"module","title":"See Also - erl_lint","doc":"`m:epp`, `m:erl_parse`","ref":"erl_lint.html#module-see-also"},{"type":"type","title":"erl_lint.error_description/0","doc":"","ref":"erl_lint.html#t:error_description/0"},{"type":"type","title":"erl_lint.error_info/0","doc":"","ref":"erl_lint.html#t:error_info/0"},{"type":"type","title":"erl_lint.fa/0","doc":"","ref":"erl_lint.html#t:fa/0"},{"type":"function","title":"erl_lint.format_error/1","doc":"Takes an `ErrorDescriptor` and returns a string that describes the error or\nwarning. This function is usually called implicitly when processing an\n`ErrorInfo` structure (see section [Error Information](`m:erl_lint#module-error-information`)).","ref":"erl_lint.html#format_error/1"},{"type":"type","title":"erl_lint.fun_used_vars/0","doc":"","ref":"erl_lint.html#t:fun_used_vars/0"},{"type":"function","title":"erl_lint.is_guard_test/1","doc":"Tests if `Expr` is a legal guard test. `Expr` is an Erlang term representing the\nabstract form for the expression.\n[`erl_parse:parse_exprs(Tokens)`](`erl_parse:parse_exprs/1`) can be used to\ngenerate a list of `Expr`.","ref":"erl_lint.html#is_guard_test/1"},{"type":"function","title":"erl_lint.module/1","doc":"","ref":"erl_lint.html#module/1"},{"type":"function","title":"erl_lint.module/2","doc":"","ref":"erl_lint.html#module/2"},{"type":"function","title":"erl_lint.module/3","doc":"Checks all the forms in a module for errors. It returns:\n\n- **`{ok,Warnings}`** - There are no errors in the module.\n\n- **`{error,Errors,Warnings}`** - There are errors in the module.\n\nAs this module is of interest only to the maintainers of the compiler, and to\navoid the same description in two places, the elements of `Options` that control\nthe warnings are only described in the [`compile`](`m:compile#erl_lint_options`)\nmodule.\n\n`AbsForms` of a module, which comes from a file that is read through `epp`, the\nErlang preprocessor, can come from many files. This means that any references to\nerrors must include the filename, see the `m:epp` module or parser (see the\n`m:erl_parse` module). The returned errors and warnings have the following\nformat:\n\n```text\n[{SourceFile,[ErrorInfo]}]\n```\n\nThe errors and warnings are listed in the order in which they are encountered in\nthe forms. The errors from one file can therefore be split into different\nentries in the list of errors.","ref":"erl_lint.html#module/3"},{"type":"module","title":"erl_parse","doc":"This module is the basic Erlang parser that converts tokens into the abstract\nform of either forms (that is, top-level constructs), expressions, or terms.\n\nThe Abstract Format is described in the ERTS User's Guide. Notice that a token\nlist must end with the dot token to be acceptable to the parse functions\n(see the `m:erl_scan`) module.","ref":"erl_parse.html"},{"type":"module","title":"Error Information - erl_parse","doc":"ErrorInfo is the standard ErrorInfo structure that is returned from all I/O modules.\nThe format is as follows:\n\n```\n{ErrorLine, Module, ErrorDescriptor}\n```\n\nA string describing the error is obtained with the following call:\n\n```\nModule:format_error(ErrorDescriptor)\n```","ref":"erl_parse.html#module-error-information"},{"type":"module","title":"See Also - erl_parse","doc":"`m:erl_anno`, `m:erl_scan`, `m:io`, section [The Abstract Format](`e:erts:absform`)\nin the ERTS User's Guide.","ref":"erl_parse.html#module-see-also"},{"type":"function","title":"erl_parse.abstract/1","doc":"Converts the Erlang data structure `Data` into an abstract form of type\n`AbsTerm`. This function is the inverse of `normalise/1`.\n\n`erl_parse:abstract(T)` is equivalent to `erl_parse:abstract(T, 0)`.","ref":"erl_parse.html#abstract/1"},{"type":"function","title":"erl_parse.abstract/2","doc":"Converts the Erlang data structure `Data` into an abstract form of type\n`AbsTerm`.\n\nEach node of `AbsTerm` is assigned an annotation, see `m:erl_anno`. The\nannotation contains the location given by option `location` or by option `line`.\nOption `location` overrides option `line`. If neither option `location` nor\noption `line` is given, `0` is used as location.\n\nOption `Encoding` is used for selecting which integer lists to be considered as\nstrings. The default is to use the encoding returned by function\n`epp:default_encoding/0`. Value `none` means that no integer lists are\nconsidered as strings. `encoding_func()` is called with one integer of a list at\na time; if it returns `true` for every integer, the list is considered a string.","ref":"erl_parse.html#abstract/2"},{"type":"type","title":"erl_parse.abstract_clause/0","doc":"Abstract form of an Erlang clause.","ref":"erl_parse.html#t:abstract_clause/0"},{"type":"type","title":"erl_parse.abstract_expr/0","doc":"Abstract form of an Erlang expression.","ref":"erl_parse.html#t:abstract_expr/0"},{"type":"type","title":"erl_parse.abstract_form/0","doc":"Abstract form of an Erlang form.","ref":"erl_parse.html#t:abstract_form/0"},{"type":"type","title":"erl_parse.abstract_type/0","doc":"Abstract form of an Erlang type.","ref":"erl_parse.html#t:abstract_type/0"},{"type":"type","title":"erl_parse.af_anno/0","doc":"","ref":"erl_parse.html#t:af_anno/0"},{"type":"type","title":"erl_parse.af_annotated_type/0","doc":"","ref":"erl_parse.html#t:af_annotated_type/0"},{"type":"type","title":"erl_parse.af_args/0","doc":"","ref":"erl_parse.html#t:af_args/0"},{"type":"type","title":"erl_parse.af_assoc/1","doc":"","ref":"erl_parse.html#t:af_assoc/1"},{"type":"type","title":"erl_parse.af_assoc_exact/1","doc":"","ref":"erl_parse.html#t:af_assoc_exact/1"},{"type":"type","title":"erl_parse.af_assoc_type/0","doc":"","ref":"erl_parse.html#t:af_assoc_type/0"},{"type":"type","title":"erl_parse.af_atom/0","doc":"","ref":"erl_parse.html#t:af_atom/0"},{"type":"type","title":"erl_parse.af_behavior/0","doc":"","ref":"erl_parse.html#t:af_behavior/0"},{"type":"type","title":"erl_parse.af_behaviour/0","doc":"","ref":"erl_parse.html#t:af_behaviour/0"},{"type":"type","title":"erl_parse.af_bin/1","doc":"","ref":"erl_parse.html#t:af_bin/1"},{"type":"type","title":"erl_parse.af_binary_comprehension/0","doc":"","ref":"erl_parse.html#t:af_binary_comprehension/0"},{"type":"type","title":"erl_parse.af_binary_op/1","doc":"","ref":"erl_parse.html#t:af_binary_op/1"},{"type":"type","title":"erl_parse.af_binelement/1","doc":"Abstract representation of an element of a bitstring.","ref":"erl_parse.html#t:af_binelement/1"},{"type":"type","title":"erl_parse.af_binelement_size/0","doc":"","ref":"erl_parse.html#t:af_binelement_size/0"},{"type":"type","title":"erl_parse.af_bitstring_type/0","doc":"","ref":"erl_parse.html#t:af_bitstring_type/0"},{"type":"type","title":"erl_parse.af_block/0","doc":"","ref":"erl_parse.html#t:af_block/0"},{"type":"type","title":"erl_parse.af_body/0","doc":"","ref":"erl_parse.html#t:af_body/0"},{"type":"type","title":"erl_parse.af_case/0","doc":"","ref":"erl_parse.html#t:af_case/0"},{"type":"type","title":"erl_parse.af_catch/0","doc":"","ref":"erl_parse.html#t:af_catch/0"},{"type":"type","title":"erl_parse.af_character/0","doc":"","ref":"erl_parse.html#t:af_character/0"},{"type":"type","title":"erl_parse.af_clause/0","doc":"","ref":"erl_parse.html#t:af_clause/0"},{"type":"type","title":"erl_parse.af_clause_seq/0","doc":"","ref":"erl_parse.html#t:af_clause_seq/0"},{"type":"type","title":"erl_parse.af_compile/0","doc":"","ref":"erl_parse.html#t:af_compile/0"},{"type":"type","title":"erl_parse.af_cons/1","doc":"","ref":"erl_parse.html#t:af_cons/1"},{"type":"type","title":"erl_parse.af_constrained_function_type/0","doc":"","ref":"erl_parse.html#t:af_constrained_function_type/0"},{"type":"type","title":"erl_parse.af_constraint/0","doc":"","ref":"erl_parse.html#t:af_constraint/0"},{"type":"type","title":"erl_parse.af_empty_list_type/0","doc":"","ref":"erl_parse.html#t:af_empty_list_type/0"},{"type":"type","title":"erl_parse.af_export/0","doc":"","ref":"erl_parse.html#t:af_export/0"},{"type":"type","title":"erl_parse.af_export_type/0","doc":"","ref":"erl_parse.html#t:af_export_type/0"},{"type":"type","title":"erl_parse.af_fa_list/0","doc":"","ref":"erl_parse.html#t:af_fa_list/0"},{"type":"type","title":"erl_parse.af_field/0","doc":"","ref":"erl_parse.html#t:af_field/0"},{"type":"type","title":"erl_parse.af_field_decl/0","doc":"Abstract representation of a record field.","ref":"erl_parse.html#t:af_field_decl/0"},{"type":"type","title":"erl_parse.af_field_name/0","doc":"","ref":"erl_parse.html#t:af_field_name/0"},{"type":"type","title":"erl_parse.af_file/0","doc":"","ref":"erl_parse.html#t:af_file/0"},{"type":"type","title":"erl_parse.af_filter/0","doc":"","ref":"erl_parse.html#t:af_filter/0"},{"type":"type","title":"erl_parse.af_float/0","doc":"","ref":"erl_parse.html#t:af_float/0"},{"type":"type","title":"erl_parse.af_fun/0","doc":"","ref":"erl_parse.html#t:af_fun/0"},{"type":"type","title":"erl_parse.af_fun_type/0","doc":"","ref":"erl_parse.html#t:af_fun_type/0"},{"type":"type","title":"erl_parse.af_function_constraint/0","doc":"","ref":"erl_parse.html#t:af_function_constraint/0"},{"type":"type","title":"erl_parse.af_function_decl/0","doc":"","ref":"erl_parse.html#t:af_function_decl/0"},{"type":"type","title":"erl_parse.af_function_spec/0","doc":"","ref":"erl_parse.html#t:af_function_spec/0"},{"type":"type","title":"erl_parse.af_function_type/0","doc":"","ref":"erl_parse.html#t:af_function_type/0"},{"type":"type","title":"erl_parse.af_function_type_list/0","doc":"","ref":"erl_parse.html#t:af_function_type_list/0"},{"type":"type","title":"erl_parse.af_generator/0","doc":"Abstract representation of a generator or a bitstring generator.","ref":"erl_parse.html#t:af_generator/0"},{"type":"type","title":"erl_parse.af_guard/0","doc":"","ref":"erl_parse.html#t:af_guard/0"},{"type":"type","title":"erl_parse.af_guard_call/0","doc":"","ref":"erl_parse.html#t:af_guard_call/0"},{"type":"type","title":"erl_parse.af_guard_seq/0","doc":"","ref":"erl_parse.html#t:af_guard_seq/0"},{"type":"type","title":"erl_parse.af_guard_test/0","doc":"","ref":"erl_parse.html#t:af_guard_test/0"},{"type":"type","title":"erl_parse.af_if/0","doc":"","ref":"erl_parse.html#t:af_if/0"},{"type":"type","title":"erl_parse.af_import/0","doc":"","ref":"erl_parse.html#t:af_import/0"},{"type":"type","title":"erl_parse.af_integer/0","doc":"","ref":"erl_parse.html#t:af_integer/0"},{"type":"type","title":"erl_parse.af_integer_range_type/0","doc":"","ref":"erl_parse.html#t:af_integer_range_type/0"},{"type":"type","title":"erl_parse.af_list_comprehension/0","doc":"","ref":"erl_parse.html#t:af_list_comprehension/0"},{"type":"type","title":"erl_parse.af_lit_atom/1","doc":"","ref":"erl_parse.html#t:af_lit_atom/1"},{"type":"type","title":"erl_parse.af_literal/0","doc":"","ref":"erl_parse.html#t:af_literal/0"},{"type":"type","title":"erl_parse.af_local_call/0","doc":"","ref":"erl_parse.html#t:af_local_call/0"},{"type":"type","title":"erl_parse.af_local_fun/0","doc":"","ref":"erl_parse.html#t:af_local_fun/0"},{"type":"type","title":"erl_parse.af_local_function/0","doc":"","ref":"erl_parse.html#t:af_local_function/0"},{"type":"type","title":"erl_parse.af_map_comprehension/0","doc":"","ref":"erl_parse.html#t:af_map_comprehension/0"},{"type":"type","title":"erl_parse.af_map_creation/1","doc":"","ref":"erl_parse.html#t:af_map_creation/1"},{"type":"type","title":"erl_parse.af_map_pattern/0","doc":"","ref":"erl_parse.html#t:af_map_pattern/0"},{"type":"type","title":"erl_parse.af_map_type/0","doc":"","ref":"erl_parse.html#t:af_map_type/0"},{"type":"type","title":"erl_parse.af_map_update/1","doc":"","ref":"erl_parse.html#t:af_map_update/1"},{"type":"type","title":"erl_parse.af_match/1","doc":"","ref":"erl_parse.html#t:af_match/1"},{"type":"type","title":"erl_parse.af_maybe/0","doc":"","ref":"erl_parse.html#t:af_maybe/0"},{"type":"type","title":"erl_parse.af_maybe_else/0","doc":"","ref":"erl_parse.html#t:af_maybe_else/0"},{"type":"type","title":"erl_parse.af_maybe_match/0","doc":"","ref":"erl_parse.html#t:af_maybe_match/0"},{"type":"type","title":"erl_parse.af_module/0","doc":"","ref":"erl_parse.html#t:af_module/0"},{"type":"type","title":"erl_parse.af_named_fun/0","doc":"","ref":"erl_parse.html#t:af_named_fun/0"},{"type":"type","title":"erl_parse.af_nil/0","doc":"","ref":"erl_parse.html#t:af_nil/0"},{"type":"type","title":"erl_parse.af_pattern/0","doc":"","ref":"erl_parse.html#t:af_pattern/0"},{"type":"type","title":"erl_parse.af_predefined_type/0","doc":"","ref":"erl_parse.html#t:af_predefined_type/0"},{"type":"type","title":"erl_parse.af_qualifier/0","doc":"","ref":"erl_parse.html#t:af_qualifier/0"},{"type":"type","title":"erl_parse.af_qualifier_seq/0","doc":"","ref":"erl_parse.html#t:af_qualifier_seq/0"},{"type":"type","title":"erl_parse.af_receive/0","doc":"","ref":"erl_parse.html#t:af_receive/0"},{"type":"type","title":"erl_parse.af_record_creation/1","doc":"","ref":"erl_parse.html#t:af_record_creation/1"},{"type":"type","title":"erl_parse.af_record_decl/0","doc":"","ref":"erl_parse.html#t:af_record_decl/0"},{"type":"type","title":"erl_parse.af_record_field/1","doc":"","ref":"erl_parse.html#t:af_record_field/1"},{"type":"type","title":"erl_parse.af_record_field_access/1","doc":"","ref":"erl_parse.html#t:af_record_field_access/1"},{"type":"type","title":"erl_parse.af_record_field_type/0","doc":"","ref":"erl_parse.html#t:af_record_field_type/0"},{"type":"type","title":"erl_parse.af_record_index/0","doc":"","ref":"erl_parse.html#t:af_record_index/0"},{"type":"type","title":"erl_parse.af_record_type/0","doc":"","ref":"erl_parse.html#t:af_record_type/0"},{"type":"type","title":"erl_parse.af_record_update/1","doc":"","ref":"erl_parse.html#t:af_record_update/1"},{"type":"type","title":"erl_parse.af_remote_call/0","doc":"","ref":"erl_parse.html#t:af_remote_call/0"},{"type":"type","title":"erl_parse.af_remote_fun/0","doc":"","ref":"erl_parse.html#t:af_remote_fun/0"},{"type":"type","title":"erl_parse.af_remote_function/0","doc":"Abstract representation of a remote function call.","ref":"erl_parse.html#t:af_remote_function/0"},{"type":"type","title":"erl_parse.af_remote_guard_call/0","doc":"","ref":"erl_parse.html#t:af_remote_guard_call/0"},{"type":"type","title":"erl_parse.af_remote_type/0","doc":"","ref":"erl_parse.html#t:af_remote_type/0"},{"type":"type","title":"erl_parse.af_singleton_integer_type/0","doc":"","ref":"erl_parse.html#t:af_singleton_integer_type/0"},{"type":"type","title":"erl_parse.af_string/0","doc":"","ref":"erl_parse.html#t:af_string/0"},{"type":"type","title":"erl_parse.af_ta_list/0","doc":"","ref":"erl_parse.html#t:af_ta_list/0"},{"type":"type","title":"erl_parse.af_template/0","doc":"","ref":"erl_parse.html#t:af_template/0"},{"type":"type","title":"erl_parse.af_try/0","doc":"","ref":"erl_parse.html#t:af_try/0"},{"type":"type","title":"erl_parse.af_tuple/1","doc":"","ref":"erl_parse.html#t:af_tuple/1"},{"type":"type","title":"erl_parse.af_tuple_type/0","doc":"","ref":"erl_parse.html#t:af_tuple_type/0"},{"type":"type","title":"erl_parse.af_type_decl/0","doc":"","ref":"erl_parse.html#t:af_type_decl/0"},{"type":"type","title":"erl_parse.af_type_union/0","doc":"","ref":"erl_parse.html#t:af_type_union/0"},{"type":"type","title":"erl_parse.af_type_variable/0","doc":"","ref":"erl_parse.html#t:af_type_variable/0"},{"type":"type","title":"erl_parse.af_typed_field/0","doc":"","ref":"erl_parse.html#t:af_typed_field/0"},{"type":"type","title":"erl_parse.af_unary_op/1","doc":"","ref":"erl_parse.html#t:af_unary_op/1"},{"type":"type","title":"erl_parse.af_user_defined_type/0","doc":"","ref":"erl_parse.html#t:af_user_defined_type/0"},{"type":"type","title":"erl_parse.af_variable/0","doc":"","ref":"erl_parse.html#t:af_variable/0"},{"type":"type","title":"erl_parse.af_wild_attribute/0","doc":"","ref":"erl_parse.html#t:af_wild_attribute/0"},{"type":"type","title":"erl_parse.anno/0","doc":"","ref":"erl_parse.html#t:anno/0"},{"type":"function","title":"erl_parse.anno_from_term/1","doc":"Assumes that `Term` is a term with the same structure as a `erl_parse` tree, but\nwith terms, say `T`, where a `erl_parse` tree has collections of annotations.\n\nReturns a `erl_parse` tree where each term `T` is replaced by the value returned\nby [`erl_anno:from_term(T)`](`erl_anno:from_term/1`). The term `Term` is\ntraversed in a depth-first, left-to-right fashion.","ref":"erl_parse.html#anno_from_term/1"},{"type":"function","title":"erl_parse.anno_to_term/1","doc":"Returns a term where each collection of annotations `Anno` of the nodes of the\n`erl_parse` tree `Abstr` is replaced by the term returned by\n[`erl_anno:to_term(Anno)`](`erl_anno:to_term/1`). The `erl_parse` tree is\ntraversed in a depth-first, left-to-right fashion.","ref":"erl_parse.html#anno_to_term/1"},{"type":"type","title":"erl_parse.behaviour/0","doc":"","ref":"erl_parse.html#t:behaviour/0"},{"type":"type","title":"erl_parse.binary_op/0","doc":"","ref":"erl_parse.html#t:binary_op/0"},{"type":"type","title":"erl_parse.encoding_func/0","doc":"","ref":"erl_parse.html#t:encoding_func/0"},{"type":"type","title":"erl_parse.endianness/0","doc":"","ref":"erl_parse.html#t:endianness/0"},{"type":"type","title":"erl_parse.erl_parse_tree/0","doc":"","ref":"erl_parse.html#t:erl_parse_tree/0"},{"type":"type","title":"erl_parse.error_description/0","doc":"","ref":"erl_parse.html#t:error_description/0"},{"type":"type","title":"erl_parse.error_info/0","doc":"","ref":"erl_parse.html#t:error_info/0"},{"type":"function","title":"erl_parse.fold_anno/3","doc":"Updates an accumulator by applying `Fun` on each collection of annotations of\nthe `erl_parse` tree `Abstr`.\n\nThe first call to `Fun` has `AccIn` as argument, the returned accumulator\n`AccOut` is passed to the next call, and so on. The\nfinal value of the accumulator is returned. The `erl_parse` tree is traversed in\na depth-first, left-to-right fashion.","ref":"erl_parse.html#fold_anno/3"},{"type":"type","title":"erl_parse.form_info/0","doc":"Tuples `{error, error_info()}` and `{warning, error_info()}`, denoting\nsyntactically incorrect forms and warnings, and `{eof, line()}`, denoting an\nend-of-stream encountered before a complete form had been parsed.","ref":"erl_parse.html#t:form_info/0"},{"type":"function","title":"erl_parse.format_error/1","doc":"Uses an ErrorDescriptor and returns a string that describes the error.\n\nThis function is usually called implicitly when an ErrorInfo structure is\nprocessed (see section [Error Information](#module-error-information)).","ref":"erl_parse.html#format_error/1"},{"type":"type","title":"erl_parse.fun_name/0","doc":"","ref":"erl_parse.html#t:fun_name/0"},{"type":"type","title":"erl_parse.function_name/0","doc":"","ref":"erl_parse.html#t:function_name/0"},{"type":"function","title":"erl_parse.map_anno/2","doc":"Modifies the `erl_parse` tree `Abstr` by applying `Fun` on each collection of\nannotations of the nodes of the `erl_parse` tree. The `erl_parse` tree is\ntraversed in a depth-first, left-to-right fashion.","ref":"erl_parse.html#map_anno/2"},{"type":"function","title":"erl_parse.mapfold_anno/3","doc":"Modifies the `erl_parse` tree `Abstr` by applying `Fun` on each collection of\nannotations of the nodes of the `erl_parse` tree, while at the same time\nupdating an accumulator.\n\nThe first call to `Fun` has `AccIn` as second argument,\nthe returned accumulator `AccOut` is passed to the next call, and so on. The\nmodified `erl_parse` tree and the final value of the accumulator are returned.\nThe `erl_parse` tree is traversed in a depth-first, left-to-right fashion.","ref":"erl_parse.html#mapfold_anno/3"},{"type":"function","title":"erl_parse.new_anno/1","doc":"Assumes that `Term` is a term with the same structure as a `erl_parse` tree, but\nwith [locations](`t:erl_anno:location/0`) where a `erl_parse` tree has\ncollections of annotations.\n\nReturns a `erl_parse` tree where each location `L`\nis replaced by the value returned by [`erl_anno:new(L)`](`erl_anno:new/1`). The\nterm `Term` is traversed in a depth-first, left-to-right fashion.","ref":"erl_parse.html#new_anno/1"},{"type":"function","title":"erl_parse.normalise/1","doc":"Converts the abstract form `AbsTerm` of a term into a conventional Erlang data\nstructure (that is, the term itself). This function is the inverse of\n`abstract/1`.","ref":"erl_parse.html#normalise/1"},{"type":"function","title":"erl_parse.parse_exprs/1","doc":"Parses `Tokens` as if it was a list of expressions.\n\nReturns one of the following:\n\n- **`{ok, ExprList}`** - The parsing was successful. `ExprList` is a list of the\n  abstract forms of the parsed expressions.\n\n- **`{error, ErrorInfo}`** - An error occurred.","ref":"erl_parse.html#parse_exprs/1"},{"type":"function","title":"erl_parse.parse_form/1","doc":"Parses `Tokens` as if it was a form.\n\nReturns one of the following:\n\n- **`{ok, AbsForm}`** - The parsing was successful. `AbsForm` is the abstract\n  form of the parsed form.\n\n- **`{error, ErrorInfo}`** - An error occurred.","ref":"erl_parse.html#parse_form/1"},{"type":"function","title":"erl_parse.parse_term/1","doc":"Parses `Tokens` as if it was a term.\n\nReturns one of the following:\n\n- **`{ok, Term}`** - The parsing was successful. `Term` is the Erlang term\n  corresponding to the token list.\n\n- **`{error, ErrorInfo}`** - An error occurred.","ref":"erl_parse.html#parse_term/1"},{"type":"type","title":"erl_parse.record_name/0","doc":"","ref":"erl_parse.html#t:record_name/0"},{"type":"type","title":"erl_parse.signedness/0","doc":"","ref":"erl_parse.html#t:signedness/0"},{"type":"type","title":"erl_parse.spec_attr/0","doc":"","ref":"erl_parse.html#t:spec_attr/0"},{"type":"type","title":"erl_parse.token/0","doc":"","ref":"erl_parse.html#t:token/0"},{"type":"function","title":"erl_parse.tokens/1","doc":"","ref":"erl_parse.html#tokens/1"},{"type":"function","title":"erl_parse.tokens/2","doc":"Generates a list of tokens representing the abstract form `AbsTerm` of an\nexpression. Optionally, `MoreTokens` is appended.","ref":"erl_parse.html#tokens/2"},{"type":"type","title":"erl_parse.type/0","doc":"","ref":"erl_parse.html#t:type/0"},{"type":"type","title":"erl_parse.type_attr/0","doc":"","ref":"erl_parse.html#t:type_attr/0"},{"type":"type","title":"erl_parse.type_name/0","doc":"","ref":"erl_parse.html#t:type_name/0"},{"type":"type","title":"erl_parse.type_specifier/0","doc":"","ref":"erl_parse.html#t:type_specifier/0"},{"type":"type","title":"erl_parse.type_specifier_list/0","doc":"","ref":"erl_parse.html#t:type_specifier_list/0"},{"type":"type","title":"erl_parse.unary_op/0","doc":"","ref":"erl_parse.html#t:unary_op/0"},{"type":"type","title":"erl_parse.unit/0","doc":"","ref":"erl_parse.html#t:unit/0"},{"type":"module","title":"erl_pp","doc":"The Erlang pretty printer.\n\nThe functions in this module are used to generate aesthetically attractive\nrepresentations of abstract forms, which are suitable for printing. All\nfunctions return (possibly deep) lists of characters and generate an error if\nthe form is wrong.\n\nAll functions can have an optional argument, which specifies a hook that is\ncalled if an attempt is made to print an unknown form.\n\nNote that if the functions in this module are used to convert abstract code back\nto Erlang source code, the enclosing function should first be processed by\n`legalize_vars/1` in order to ensure that the output is semantically equivalent\nto the abstract code.","ref":"erl_pp.html"},{"type":"module","title":"Known Limitations - erl_pp","doc":"It is not possible to have hook functions for unknown forms at other places than\nexpressions.","ref":"erl_pp.html#module-known-limitations"},{"type":"module","title":"See Also - erl_pp","doc":"`m:erl_eval`, `m:erl_parse`, `m:io`","ref":"erl_pp.html#module-see-also"},{"type":"function","title":"erl_pp.attribute/1","doc":"","ref":"erl_pp.html#attribute/1"},{"type":"function","title":"erl_pp.attribute/2","doc":"Same as [`form/1,2`](`form/1`), but only for attribute `Attribute`.","ref":"erl_pp.html#attribute/2"},{"type":"function","title":"erl_pp.expr/1","doc":"","ref":"erl_pp.html#expr/1"},{"type":"function","title":"erl_pp.expr/2","doc":"","ref":"erl_pp.html#expr/2"},{"type":"function","title":"erl_pp.expr/3","doc":"","ref":"erl_pp.html#expr/3"},{"type":"function","title":"erl_pp.expr/4","doc":"Prints one expression.\n\nIt is useful for implementing hooks (see section\n[Known Limitations](`m:erl_pp#module-known-limitations`)).","ref":"erl_pp.html#expr/4"},{"type":"function","title":"erl_pp.exprs/1","doc":"","ref":"erl_pp.html#exprs/1"},{"type":"function","title":"erl_pp.exprs/2","doc":"","ref":"erl_pp.html#exprs/2"},{"type":"function","title":"erl_pp.exprs/3","doc":"Same as [`form/1,2`](`form/1`), but only for the sequence of expressions in\n`Expressions`.","ref":"erl_pp.html#exprs/3"},{"type":"function","title":"erl_pp.form/1","doc":"","ref":"erl_pp.html#form/1"},{"type":"function","title":"erl_pp.form/2","doc":"Pretty prints a `Form`, which is an abstract form of a type that is returned by\n`erl_parse:parse_form/1`.","ref":"erl_pp.html#form/2"},{"type":"function","title":"erl_pp.function/1","doc":"","ref":"erl_pp.html#function/1"},{"type":"function","title":"erl_pp.function/2","doc":"Same as [`form/1,2`](`form/1`), but only for function `Function`.","ref":"erl_pp.html#function/2"},{"type":"function","title":"erl_pp.guard/1","doc":"","ref":"erl_pp.html#guard/1"},{"type":"function","title":"erl_pp.guard/2","doc":"Same as [`form/1,2`](`form/1`), but only for the guard test `Guard`.","ref":"erl_pp.html#guard/2"},{"type":"type","title":"erl_pp.hook_function/0","doc":"Optional argument `HookFunction`{: #hook_function }, shown in the functions\ndescribed in this module, defines a function that is called when an unknown form\noccurs where there is to be a valid expression. If `HookFunction` is equal to\n`none`, there is no hook function.\n\nThe called hook function is to return a (possibly deep) list of characters.\nFunction `expr/4` is useful in a hook.\n\nIf `CurrentIndentation` is negative, there are no line breaks and only a space\nis used as a separator.","ref":"erl_pp.html#t:hook_function/0"},{"type":"function","title":"erl_pp.legalize_vars/1","doc":"The Erlang compiler will, when expanding records to tuples, introduce new\nvariables in the abstract representation. As the expansion is done on the\nabstract representation, the compiler can safely name the new variables with\nnames that are not syntactically valid in Erlang source code (the name starts\nwith a lowercase letter), thus ensuring the uniqueness of the new names.\n\nThe above strategy leads to problems if a user wants to convert the abstract\nrepresentation, using the functions of this module back to Erlang source code.\nTypically, pattern variables are output as atoms thus changing the sematics of\nthe program. To solve this problem [`legalize_vars/1`](`legalize_vars/1`), when\nrun on the abstract representation of a function, will return an equivalent\nfunction where all variables will have syntactically valid names.","ref":"erl_pp.html#legalize_vars/1"},{"type":"type","title":"erl_pp.option/0","doc":"The option `quote_singleton_atom_types` is used to add quotes to all singleton\natom types.\n\nThe option `linewidth` controls the maximum line width for formatted lines\n(defaults to 72 characters).\n\nThe option `indent` controls the indention for formatted lines (defaults to 4\nspaces).","ref":"erl_pp.html#t:option/0"},{"type":"type","title":"erl_pp.options/0","doc":"","ref":"erl_pp.html#t:options/0"},{"type":"module","title":"erl_scan","doc":"The Erlang token scanner.\n\nThis module contains functions for tokenizing (scanning) characters into Erlang\ntokens.","ref":"erl_scan.html"},{"type":"module","title":"Error Information - erl_scan","doc":"`ErrorInfo` is the standard `ErrorInfo` structure that is returned from all I/O\nmodules. The format is as follows:\n\n```erlang\n{ErrorLocation, Module, ErrorDescriptor}\n```\n\nA string describing the error is obtained with the following call:\n\n```erlang\nModule:format_error(ErrorDescriptor)\n```","ref":"erl_scan.html#module-error-information"},{"type":"module","title":"Notes - erl_scan","doc":"The continuation of the first call to the re-entrant input functions must be\n`[]`. For a complete description of how the re-entrant input scheme works, see\nArmstrong, Virding and Williams: 'Concurrent Programming in Erlang', Chapter 13.","ref":"erl_scan.html#module-notes"},{"type":"module","title":"See Also - erl_scan","doc":"`m:erl_anno`, `m:erl_parse`, `m:io`","ref":"erl_scan.html#module-see-also"},{"type":"type","title":"erl_scan.category/0","doc":"","ref":"erl_scan.html#t:category/0"},{"type":"function","title":"erl_scan.category/1","doc":"Returns the category of `Token`.","ref":"erl_scan.html#category/1"},{"type":"type","title":"erl_scan.char_spec/0","doc":"","ref":"erl_scan.html#t:char_spec/0"},{"type":"function","title":"erl_scan.column/1","doc":"Returns the column of `Token`'s collection of annotations.","ref":"erl_scan.html#column/1"},{"type":"function","title":"erl_scan.end_location/1","doc":"Returns the end location of the text of `Token`'s collection of annotations. If\nthere is no text, `undefined` is returned.","ref":"erl_scan.html#end_location/1"},{"type":"type","title":"erl_scan.error_description/0","doc":"","ref":"erl_scan.html#t:error_description/0"},{"type":"type","title":"erl_scan.error_info/0","doc":"","ref":"erl_scan.html#t:error_info/0"},{"type":"function","title":"erl_scan.format_error/1","doc":"Uses an `ErrorDescriptor` and returns a string that describes the error or\nwarning. This function is usually called implicitly when an `ErrorInfo`\nstructure is processed (see section\n[Error Information](`m:erl_scan#module-error-information`)).","ref":"erl_scan.html#format_error/1"},{"type":"function","title":"erl_scan.line/1","doc":"Returns the line of `Token`'s collection of annotations.","ref":"erl_scan.html#line/1"},{"type":"function","title":"erl_scan.location/1","doc":"Returns the location of `Token`'s collection of annotations.","ref":"erl_scan.html#location/1"},{"type":"type","title":"erl_scan.option/0","doc":"","ref":"erl_scan.html#t:option/0"},{"type":"type","title":"erl_scan.options/0","doc":"","ref":"erl_scan.html#t:options/0"},{"type":"function","title":"erl_scan.reserved_word/1","doc":"Returns `true` if `Atom` is an Erlang reserved word, otherwise `false`.","ref":"erl_scan.html#reserved_word/1"},{"type":"type","title":"erl_scan.resword_fun/0","doc":"","ref":"erl_scan.html#t:resword_fun/0"},{"type":"opaque","title":"erl_scan.return_cont/0","doc":"","ref":"erl_scan.html#t:return_cont/0"},{"type":"function","title":"erl_scan.string/1","doc":"","ref":"erl_scan.html#string/1"},{"type":"function","title":"erl_scan.string/2","doc":"","ref":"erl_scan.html#string/2"},{"type":"function","title":"erl_scan.string/3","doc":"Takes the list of characters `String` and tries to scan (tokenize) them.\n\nReturns one of the following:\n\n- **`{ok, Tokens, EndLocation}`** - `Tokens` are the Erlang tokens from\n  `String`. `EndLocation` is the first location after the last token.\n\n- **`{error, ErrorInfo, ErrorLocation}`** - An error occurred. `ErrorLocation`\n  is the first location after the erroneous token.\n\n`StartLocation` indicates the initial location when scanning starts. If\n`StartLocation` is a line, `Anno`, `EndLocation`, and `ErrorLocation` are lines.\nIf `StartLocation` is a pair of a line and a column, `Anno` takes the form of an\nopaque compound data type, and `EndLocation` and `ErrorLocation` are pairs of a\nline and a column. The _token annotations_ contain information about the column\nand the line where the token begins, as well as the text of the token (if option\n`text` is specified), all of which can be accessed by calling `column/1`,\n`line/1`, `location/1`, and `text/1`.\n\nA _token_ is a tuple containing information about syntactic category, the token\nannotations, and the terminal symbol. For punctuation characters (such as `;`\nand `|`) and reserved words, the category and the symbol coincide, and the token\nis represented by a two-tuple. Three-tuples have one of the following forms:\n\n- `{atom, Anno, atom()}`\n- `{char, Anno, char()}`\n- `{comment, Anno, string()}`\n- `{float, Anno, float()}`\n- `{integer, Anno, integer()}`\n- `{var, Anno, atom()}`\n- `{white_space, Anno, string()}`\n\nValid options:\n\n- **`{reserved_word_fun, reserved_word_fun()}`** - A callback function that is\n  called when the scanner has found an unquoted atom. If the function returns\n  `true`, the unquoted atom itself becomes the category of the token. If the\n  function returns `false`, `atom` becomes the category of the unquoted atom.\n\n- **`return_comments`** - Return comment tokens.\n\n- **`return_white_spaces`** - Return white space tokens. By convention, a\n  newline character, if present, is always the first character of the text\n  (there cannot be more than one newline in a white space token).\n\n- **`return`** - Short for `[return_comments, return_white_spaces]`.\n\n- **`text`{: #text }** - Include the token text in the token annotation. The\n  text is the part of the input corresponding to the token. See also\n  [`text_fun`](`m:erl_scan#text_fun`).\n\n- **`{text_fun, text_fun()}`{: #text_fun }** - A callback function used to\n  determine whether the full text for the token shall be included in the token\n  annotation. Arguments of the function are the category of the token and the\n  full token string. This is only used when [`text`](`m:erl_scan#text`) is not\n  present. If neither are present the text will not be saved in the token\n  annotation.\n\n- **`{compiler_internal, term()}`{: #compiler_interal }** - Pass\n  compiler-internal options to the scanner. The set of internal options\n  understood by the scanner should be considered experimental and can thus be\n  changed at any time without prior warning.\n\n  The following options are currently understood:\n\n  - **`ssa_checks`** - Tokenizes source code annotations used for encoding tests\n    on the BEAM SSA code produced by the compiler.","ref":"erl_scan.html#string/3"},{"type":"type","title":"erl_scan.symbol/0","doc":"","ref":"erl_scan.html#t:symbol/0"},{"type":"function","title":"erl_scan.symbol/1","doc":"Returns the symbol of `Token`.","ref":"erl_scan.html#symbol/1"},{"type":"function","title":"erl_scan.text/1","doc":"Returns the text of `Token`'s collection of annotations. If there is no text,\n`undefined` is returned.","ref":"erl_scan.html#text/1"},{"type":"type","title":"erl_scan.text_fun/0","doc":"","ref":"erl_scan.html#t:text_fun/0"},{"type":"type","title":"erl_scan.token/0","doc":"","ref":"erl_scan.html#t:token/0"},{"type":"type","title":"erl_scan.tokens/0","doc":"","ref":"erl_scan.html#t:tokens/0"},{"type":"function","title":"erl_scan.tokens/3","doc":"","ref":"erl_scan.html#tokens/3"},{"type":"function","title":"erl_scan.tokens/4","doc":"This is the re-entrant scanner, which scans characters until either a _dot_ ('.'\nfollowed by a white space) or `eof` is reached.\n\nIt returns:\n\n- **`{done, Result, LeftOverChars}`** - Indicates that there is sufficient input\n  data to get a result. `Result` is:\n\n  - **`{ok, Tokens, EndLocation}`** - The scanning was successful. `Tokens` is\n    the list of tokens including _dot_.\n\n  - **`{eof, EndLocation}`** - End of file was encountered before any more\n    tokens.\n\n  - **`{error, ErrorInfo, EndLocation}`** - An error occurred. `LeftOverChars`\n    is the remaining characters of the input data, starting from `EndLocation`.\n\n- **`{more, Continuation1}`** - More data is required for building a term.\n  `Continuation1` must be passed in a new call to `tokens/3,4` when more data is\n  available.\n\nThe `CharSpec` `eof` signals end of file. `LeftOverChars` then takes the value\n`eof` as well.\n\nFor a description of the options, see `string/3`.","ref":"erl_scan.html#tokens/4"},{"type":"type","title":"erl_scan.tokens_result/0","doc":"","ref":"erl_scan.html#t:tokens_result/0"},{"type":"module","title":"ms_transform","doc":"A parse transformation that translates fun syntax into match specifications.\n\nThis module provides the parse transformation that makes calls to `m:ets` and\n`dbg:fun2ms/1` translate into literal match specifications. It also provides the\nback end for the same functions when called from the Erlang shell.\n\nThe translation from funs to match specifications is accessed through the two\n\"pseudo functions\" `ets:fun2ms/1` and `dbg:fun2ms/1`.\n\nAs everyone trying to use [`ets:select/2`](`ets:select/1`) or `m:dbg` seems to\nend up reading this manual page, this description is an introduction to the\nconcept of match specifications.\n\nRead the whole manual page if it is the first time you are using the\ntransformations.\n\nMatch specifications are used more or less as filters. They resemble usual\nErlang matching in a list comprehension or in a fun used with `lists:foldl/3`,\nand so on. However, the syntax of pure match specifications is awkward, as they\nare made up purely by Erlang terms, and the language has no syntax to make the\nmatch specifications more readable.\n\nAs the execution and structure of the match specifications are like that of a\nfun, it is more straightforward to write it using the familiar fun syntax and to\nhave that translated into a match specification automatically. A real fun is\nclearly more powerful than the match specifications allow, but bearing the match\nspecifications in mind, and what they can do, it is still more convenient to\nwrite it all as a fun. This module contains the code that translates the fun\nsyntax into match specification terms.","ref":"ms_transform.html"},{"type":"module","title":"Example 1 - ms_transform","doc":"Using `ets:select/2` and a match specification, one can filter out rows of a\ntable and construct a list of tuples containing relevant parts of the data in\nthese rows. One can use `ets:foldl/3` instead, but the `ets:select/2` call is\nfar more efficient. Without the translation provided by `ms_transform`, one must\nstruggle with writing match specifications terms to accommodate this.\n\nConsider a simple table of employees:\n\n```erlang\n-record(emp, {empno,     %Employee number as a string, the key\n              surname,   %Surname of the employee\n              givenname, %Given name of employee\n              dept,      %Department, one of {dev,sales,prod,adm}\n              empyear}). %Year the employee was employed\n```\n\nWe create the table using:\n\n```text\nets:new(emp_tab, [{keypos,#emp.empno},named_table,ordered_set]).\n```\n\nWe fill the table with randomly chosen data:\n\n```erlang\n[{emp,\"011103\",\"Black\",\"Alfred\",sales,2000},\n {emp,\"041231\",\"Doe\",\"John\",prod,2001},\n {emp,\"052341\",\"Smith\",\"John\",dev,1997},\n {emp,\"076324\",\"Smith\",\"Ella\",sales,1995},\n {emp,\"122334\",\"Weston\",\"Anna\",prod,2002},\n {emp,\"535216\",\"Chalker\",\"Samuel\",adm,1998},\n {emp,\"789789\",\"Harrysson\",\"Joe\",adm,1996},\n {emp,\"963721\",\"Scott\",\"Juliana\",dev,2003},\n {emp,\"989891\",\"Brown\",\"Gabriel\",prod,1999}]\n```\n\nAssuming that we want the employee numbers of everyone in the sales department,\nthere are several ways.\n\n`ets:match/2` can be used:\n\n```erlang\n1> ets:match(emp_tab, {'_', '$1', '_', '_', sales, '_'}).\n[[\"011103\"],[\"076324\"]]\n```\n\n`ets:match/2` uses a simpler type of match specification, but it is still\nunreadable, and one has little control over the returned result. It is always a\nlist of lists.\n\n`ets:foldl/3` or `ets:foldr/3` can be used to avoid the nested lists:\n\n```erlang\nets:foldr(fun(#emp{empno = E, dept = sales},Acc) -> [E | Acc];\n             (_,Acc) -> Acc\n          end,\n          [],\n          emp_tab).\n```\n\nThe result is `[\"011103\",\"076324\"]`. The fun is straightforward, so the only\nproblem is that all the data from the table must be transferred from the table\nto the calling process for filtering. That is inefficient compared to the\n`ets:match/2` call where the filtering can be done \"inside\" the emulator and\nonly the result is transferred to the process.\n\nConsider a \"pure\" `ets:select/2` call that does what `ets:foldr` does:\n\n```erlang\nets:select(emp_tab, [{#emp{empno = '$1', dept = sales, _='_'},[],['$1']}]).\n```\n\nAlthough the record syntax is used, it is still hard to read and even harder to\nwrite. The first element of the tuple,\n`#emp{empno = '$1', dept = sales, _='_'}`, tells what to match. Elements not\nmatching this are not returned, as in the `ets:match/2` example. The second\nelement, the empty list, is a list of guard expressions, which we do not need.\nThe third element is the list of expressions constructing the return value (in\nETS this is almost always a list containing one single term). In our case `'$1'`\nis bound to the employee number in the head (first element of the tuple), and\nhence the employee number is returned. The result is `[\"011103\",\"076324\"]`, as\nin the `ets:foldr/3` example, but the result is retrieved much more efficiently\nin terms of execution speed and memory consumption.\n\nUsing `ets:fun2ms/1`, we can combine the ease of use of the `ets:foldr/3` and\nthe efficiency of the pure `ets:select/2` example:\n\n```erlang\n-include_lib(\"stdlib/include/ms_transform.hrl\").\n\nets:select(emp_tab, ets:fun2ms(\n                      fun(#emp{empno = E, dept = sales}) ->\n                              E\n                      end)).\n```\n\nThis example requires no special knowledge of match specifications to\nunderstand. The head of the fun matches what you want to filter out and the body\nreturns what you want returned. As long as the fun can be kept within the limits\nof the match specifications, there is no need to transfer all table data to the\nprocess for filtering as in the `ets:foldr/3` example. It is easier to read than\nthe `ets:foldr/3` example, as the select call in itself discards anything that\ndoes not match, while the fun of the `ets:foldr/3` call needs to handle both the\nelements matching and the ones not matching.\n\nIn the `ets:fun2ms/1` example above, it is needed to include `ms_transform.hrl`\nin the source code, as this is what triggers the parse transformation of the\n`ets:fun2ms/1` call to a valid match specification. This also implies that the\ntransformation is done at compile time (except when called from the shell) and\ntherefore takes no resources in runtime. That is, although you use the more\nintuitive fun syntax, it gets as efficient in runtime as writing match\nspecifications by hand.","ref":"ms_transform.html#module-example-1"},{"type":"module","title":"Example 2 - ms_transform","doc":"Assume that we want to get all the employee numbers of employees hired before\nyear 2000. Using `ets:match/2` is not an alternative here, as relational\noperators cannot be expressed there. Once again, `ets:foldr/3` can do it\n(slowly, but correct):\n\n```erlang\nets:foldr(fun(#emp{empno = E, empyear = Y},Acc) when Y < 2000 -> [E | Acc];\n                  (_,Acc) -> Acc\n          end,\n          [],\n          emp_tab).\n```\n\nThe result is `[\"052341\",\"076324\",\"535216\",\"789789\",\"989891\"]`, as expected. The\nequivalent expression using a handwritten match specification would look like\nthis:\n\n```erlang\nets:select(emp_tab, [{#emp{empno = '$1', empyear = '$2', _='_'},\n                     [{'<', '$2', 2000}],\n                     ['$1']}]).\n```\n\nThis gives the same result. `[{'<', '$2', 2000}]` is in the guard part and\ntherefore discards anything that does not have an `empyear` (bound to `'$2'` in\nthe head) less than 2000, as the guard in the `foldr/3` example.\n\nWe write it using `ets:fun2ms/1`:\n\n```erlang\n-include_lib(\"stdlib/include/ms_transform.hrl\").\n\nets:select(emp_tab, ets:fun2ms(\n                      fun(#emp{empno = E, empyear = Y}) when Y < 2000 ->\n                           E\n                      end)).\n```","ref":"ms_transform.html#module-example-2"},{"type":"module","title":"Example 3 - ms_transform","doc":"Assume that we want the whole object matching instead of only one element. One\nalternative is to assign a variable to every part of the record and build it up\nonce again in the body of the fun, but the following is easier:\n\n```erlang\nets:select(emp_tab, ets:fun2ms(\n                      fun(Obj = #emp{empno = E, empyear = Y})\n                         when Y < 2000 ->\n                              Obj\n                      end)).\n```\n\nAs in ordinary Erlang matching, you can bind a variable to the whole matched\nobject using a \"match inside the match\", that is, a `=`. Unfortunately in funs\ntranslated to match specifications, it is allowed only at the \"top-level\", that\nis, matching the _whole_ object arriving to be matched into a separate variable.\nIf you are used to writing match specifications by hand, we mention that\nvariable A is simply translated into '$\\_'. Alternatively, pseudo function\n`object/0` also returns the whole matched object, see section\n[Warnings and Restrictions](`m:ms_transform#warnings_and_restrictions`).","ref":"ms_transform.html#module-example-3"},{"type":"module","title":"Example 4 - ms_transform","doc":"This example concerns the body of the fun. Assume that all employee numbers\nbeginning with zero (`0`) must be changed to begin with one (`1`) instead, and\nthat we want to create the list `[{ , }]`:\n\n```erlang\nets:select(emp_tab, ets:fun2ms(\n                      fun(#emp{empno = [$0 | Rest] }) ->\n                              {[$0|Rest],[$1|Rest]}\n                      end)).\n```\n\nThis query hits the feature of partially bound keys in table type `ordered_set`,\nso that not the whole table needs to be searched, only the part containing keys\nbeginning with `0` is looked into.","ref":"ms_transform.html#module-example-4"},{"type":"module","title":"Example 5 - ms_transform","doc":"The fun can have many clauses. Assume that we want to do the following:\n\n- If an employee started before 1997, return the tuple\n  `{inventory,  }`.\n- If an employee started 1997 or later, but before 2001, return\n  `{rookie,  }`.\n- For all other employees, return `{newbie,  }`, except for\n  those named `Smith` as they would be affronted by anything other than the tag\n  `guru` and that is also what is returned for their numbers:\n  `{guru,  }`.\n\nThis is accomplished as follows:\n\n```erlang\nets:select(emp_tab, ets:fun2ms(\n                      fun(#emp{empno = E, surname = \"Smith\" }) ->\n                              {guru,E};\n                         (#emp{empno = E, empyear = Y}) when Y < 1997  ->\n                              {inventory, E};\n                         (#emp{empno = E, empyear = Y}) when Y > 2001  ->\n                              {newbie, E};\n                         (#emp{empno = E, empyear = Y}) -> % 1997 -- 2001\n                              {rookie, E}\n                      end)).\n```\n\nThe result is as follows:\n\n```erlang\n[{rookie,\"011103\"},\n {rookie,\"041231\"},\n {guru,\"052341\"},\n {guru,\"076324\"},\n {newbie,\"122334\"},\n {rookie,\"535216\"},\n {inventory,\"789789\"},\n {newbie,\"963721\"},\n {rookie,\"989891\"}]\n```","ref":"ms_transform.html#module-example-5"},{"type":"module","title":"Useful BIFs - ms_transform","doc":"What more can you do? A simple answer is: see the documentation of\n[match specifications](`e:erts:match_spec.md`) in ERTS User's Guide. However,\nthe following is a brief overview of the most useful \"built-in functions\" that\nyou can use when the fun is to be translated into a match specification by\n`ets:fun2ms/1`. It is not possible to call other functions than those allowed in\nmatch specifications. No \"usual\" Erlang code can be executed by the fun that is\ntranslated by `ets:fun2ms/1`. The fun is limited exactly to the power of the\nmatch specifications, which is unfortunate, but the price one must pay for the\nexecution speed of `ets:select/2` compared to `ets:foldl/foldr`.\n\nThe head of the fun is a head matching (or mismatching) _one_ parameter, one\nobject of the table we select from. The object is always a single variable (can\nbe `_`) or a tuple, as ETS, Dets, and Mnesia tables include that. The match\nspecification returned by `ets:fun2ms/1` can be used with `dets:select/2` and\n`mnesia:select/2`, and with `ets:select/2`. The use of `=` in the head is\nallowed (and encouraged) at the top-level.\n\nThe guard section can contain any guard expression of Erlang. The following is a\nlist of BIFs and expressions:\n\n- Type tests: `is_atom`, `is_float`, `is_integer`, `is_list`, `is_number`,\n  `is_pid`, `is_port`, `is_reference`, `is_tuple`, `is_binary`, `is_function`,\n  `is_record`\n- Boolean operators: `not`, `and`, `or`, `andalso`, `orelse`\n- Relational operators: `>`, `>=`, `<`, `=<`, `=:=`, `==`, `=/=`, `/=`\n- Arithmetic: `+`, `-`, `*`, `div`, `rem`\n- Bitwise operators: `band`, `bor`, `bxor`, `bnot`, `bsl`, `bsr`\n- The guard BIFs: `abs`, `element`, `hd`, `length`, `node`, `round`, `size`,\n  `byte_size`, `tl`, `trunc`, `binary_part`, `self`\n\nContrary to the fact with \"handwritten\" match specifications, the `is_record`\nguard works as in ordinary Erlang code.\n\nSemicolons (`;`) in guards are allowed, the result is (as expected) one \"match\nspecification clause\" for each semicolon-separated part of the guard. The\nsemantics is identical to the Erlang semantics.\n\nThe body of the fun is used to construct the resulting value. When selecting\nfrom tables, one usually construct a suiting term here, using ordinary Erlang\nterm construction, like tuple parentheses, list brackets, and variables matched\nout in the head, possibly with the occasional constant. Whatever expressions are\nallowed in guards are also allowed here, but no special functions exist except\n`object` and `bindings` (see further down), which returns the whole matched\nobject and all known variable bindings, respectively.\n\nThe `dbg` variants of match specifications have an imperative approach to the\nmatch specification body, the ETS dialect has not. The fun body for\n`ets:fun2ms/1` returns the result without side effects. As matching (`=`) in the\nbody of the match specifications is not allowed (for performance reasons) the\nonly thing left, more or less, is term construction.","ref":"ms_transform.html#module-useful-bifs"},{"type":"module","title":"Example with dbg - ms_transform","doc":"This section describes the slightly different match specifications translated by\n`dbg:fun2ms/1`.\n\nThe same reasons for using the parse transformation apply to `dbg`, maybe even\nmore, as filtering using Erlang code is not a good idea when tracing (except\nafterwards, if you trace to file). The concept is similar to that of\n`ets:fun2ms/1` except that you usually use it directly from the shell (which can\nalso be done with `ets:fun2ms/1`).\n\nThe following is an example module to trace on:\n\n```erlang\n-module(toy).\n\n-export([start/1, store/2, retrieve/1]).\n\nstart(Args) ->\n    toy_table = ets:new(toy_table, Args).\n\nstore(Key, Value) ->\n    ets:insert(toy_table, {Key,Value}).\n\nretrieve(Key) ->\n    [{Key, Value}] = ets:lookup(toy_table, Key),\n    Value.\n```\n\nDuring model testing, the first test results in `{badmatch,16}` in\n`{toy,start,1}`, why?\n\nWe suspect the `ets:new/2` call, as we match hard on the return value, but want\nonly the particular `new/2` call with `toy_table` as first parameter. So we\nstart a default tracer on the node:\n\n```text\n1> dbg:tracer().\n{ok,<0.88.0>}\n```\n\nWe turn on call tracing for all processes, we want to make a pretty restrictive\ntrace pattern, so there is no need to call trace only a few processes (usually\nit is not):\n\n```text\n2> dbg:p(all,call).\n{ok,[{matched,nonode@nohost,25}]}\n```\n\nWe specify the filter, we want to view calls that resemble\n`ets:new(toy_table,  )`:\n\n```erlang\n3> dbg:tp(ets,new,dbg:fun2ms(fun([toy_table,_]) -> true end)).\n{ok,[{matched,nonode@nohost,1},{saved,1}]}\n```\n\nAs can be seen, the fun used with `dbg:fun2ms/1` takes a single list as\nparameter instead of a single tuple. The list matches a list of the parameters\nto the traced function. A single variable can also be used. The body of the fun\nexpresses, in a more imperative way, actions to be taken if the fun head (and\nthe guards) matches. `true` is returned here, only because the body of a fun\ncannot be empty. The return value is discarded.\n\nThe following trace output is received during test:\n\n```erlang\n(<0.86.0>) call ets:new(toy_table, [ordered_set])\n```\n\nAssume that we have not found the problem yet, and want to see what `ets:new/2`\nreturns. We use a slightly different trace pattern:\n\n```erlang\n4> dbg:tp(ets,new,dbg:fun2ms(fun([toy_table,_]) -> return_trace() end)).\n```\n\nThe following trace output is received during test:\n\n```erlang\n(<0.86.0>) call ets:new(toy_table,[ordered_set])\n(<0.86.0>) returned from ets:new/2 -> 24\n```\n\nThe call to `return_trace` results in a trace message when the function returns.\nIt applies only to the specific function call triggering the match specification\n(and matching the head/guards of the match specification). This is by far the\nmost common call in the body of a `dbg` match specification.\n\nThe test now fails with `{badmatch,24}` because the atom `toy_table` does not\nmatch the number returned for an unnamed table. So, the problem is found, the\ntable is to be named, and the arguments supplied by the test program do not\ninclude `named_table`. We rewrite the start function:\n\n```erlang\nstart(Args) ->\n    toy_table = ets:new(toy_table, [named_table|Args]).\n```\n\nWith the same tracing turned on, the following trace output is received:\n\n```erlang\n(<0.86.0>) call ets:new(toy_table,[named_table,ordered_set])\n(<0.86.0>) returned from ets:new/2 -> toy_table\n```\n\nAssume that the module now passes all testing and goes into the system. After a\nwhile, it is found that table `toy_table` grows while the system is running and\nthat there are many elements with atoms as keys. We expected only integer keys\nand so does the rest of the system, but clearly not the entire system. We turn\non call tracing and try to see calls to the module with an atom as the key:\n\n```erlang\n1> dbg:tracer().\n{ok,<0.88.0>}\n2> dbg:p(all,call).\n{ok,[{matched,nonode@nohost,25}]}\n3> dbg:tpl(toy,store,dbg:fun2ms(fun([A,_]) when is_atom(A) -> true end)).\n{ok,[{matched,nonode@nohost,1},{saved,1}]}\n```\n\nWe use `dbg:tpl/3` to ensure to catch local calls (assume that the module has\ngrown since the smaller version and we are unsure if this inserting of atoms is\nnot done locally). When in doubt, always use local call tracing.\n\nAssume that nothing happens when tracing in this way. The function is never\ncalled with these parameters. We conclude that someone else (some other module)\nis doing it and realize that we must trace on `ets:insert/2` and want to see the\ncalling function. The calling function can be retrieved using the match\nspecification function `caller`. To get it into the trace message, the match\nspecification function `message` must be used. The filter call looks like this\n(looking for calls to `ets:insert/2`):\n\n```erlang\n4> dbg:tpl(ets,insert,dbg:fun2ms(fun([toy_table,{A,_}]) when is_atom(A) ->\n                                    message(caller())\n                                  end)).\n{ok,[{matched,nonode@nohost,1},{saved,2}]}\n```\n\nThe caller is now displayed in the \"additional message\" part of the trace\noutput, and the following is displayed after a while:\n\n```erlang\n(<0.86.0>) call ets:insert(toy_table,{garbage,can}) ({evil_mod,evil_fun,2})\n```\n\nYou have realized that function `evil_fun` of the `evil_mod` module, with arity\n`2`, is causing all this trouble.\n\nThis example illustrates the most used calls in match specifications for `dbg`.\nThe other, more esoteric, calls are listed and explained in\n[Match specifications in Erlang](`e:erts:match_spec.md`) in ERTS User's Guide,\nas they are beyond the scope of this description.","ref":"ms_transform.html#module-example-with-dbg"},{"type":"module","title":"Warnings and Restrictions - ms_transform","doc":"[](){: #warnings_and_restrictions }\n\nThe following warnings and restrictions apply to the funs used in with\n`ets:fun2ms/1` and `dbg:fun2ms/1`.\n\n> #### Warning {: .warning }\n>\n> To use the pseudo functions triggering the translation, ensure to include the\n> header file `ms_transform.hrl` in the source code. Failure to do so possibly\n> results in runtime errors rather than compile time, as the expression can be\n> valid as a plain Erlang program without translation.\n\n> #### Warning {: .warning }\n>\n> The fun must be literally constructed inside the parameter list to the pseudo\n> functions. The fun cannot be bound to a variable first and then passed to\n> `ets:fun2ms/1` or `dbg:fun2ms/1`. For example, `ets:fun2ms(fun(A) -> A end)`\n> works, but not `F = fun(A) -> A end, ets:fun2ms(F)`. The latter results in a\n> compile-time error if the header is included, otherwise a runtime error.\n\nMany restrictions apply to the fun that is translated into a match\nspecification. To put it simple: you cannot use anything in the fun that you\ncannot use in a match specification. This means that, among others, the\nfollowing restrictions apply to the fun itself:\n\n- Functions written in Erlang cannot be called, neither can local functions,\n  global functions, or real funs.\n- Everything that is written as a function call is translated into a match\n  specification call to a built-in function, so that the call\n  [`is_list(X)`](`is_list/1`) is translated to `{'is_list', '$1'}` (`'$1'` is\n  only an example, the numbering can vary). If one tries to call a function that\n  is not a match specification built-in, it causes an error.\n- Variables occurring in the head of the fun are replaced by match specification\n  variables in the order of occurrence, so that fragment `fun({A,B,C})` is\n  replaced by `{'$1', '$2', '$3'}`, and so on. Every occurrence of such a\n  variable in the match specification is replaced by a match specification\n  variable in the same way, so that the fun\n  `fun({A,B}) when is_atom(A) -> B end` is translated into\n  `[{{'$1','$2'},[{is_atom,'$1'}],['$2']}]`.\n- Variables that are not included in the head are imported from the environment\n  and made into match specification `const` expressions. Example from the shell:\n\n  ```erlang\n  1> X = 25.\n  25\n  2> ets:fun2ms(fun({A,B}) when A > X -> B end).\n  [{{'$1','$2'},[{'>','$1',{const,25}}],['$2']}]\n  ```\n\n- Matching with `=` cannot be used in the body. It can only be used on the\n  top-level in the head of the fun. Example from the shell again:\n\n  ```erlang\n  1> ets:fun2ms(fun({A,[B|C]} = D) when A > B -> D end).\n  [{{'$1',['$2'|'$3']},[{'>','$1','$2'}],['$_']}]\n  2> ets:fun2ms(fun({A,[B|C]=D}) when A > B -> D end).\n  Error: fun with head matching ('=' in head) cannot be translated into\n  match_spec\n  {error,transform_error}\n  3> ets:fun2ms(fun({A,[B|C]}) when A > B -> D = [B|C], D end).\n  Error: fun with body matching ('=' in body) is illegal as match_spec\n  {error,transform_error}\n  ```\n\n  All variables are bound in the head of a match specification, so the\n  translator cannot allow multiple bindings. The special case when matching is\n  done on the top-level makes the variable bind to `'$_'` in the resulting match\n  specification. It is to allow a more natural access to the whole matched\n  object. Pseudo function `object()` can be used instead, see below.\n\n  The following expressions are translated equally:\n\n  ```erlang\n  ets:fun2ms(fun({a,_} = A) -> A end).\n  ets:fun2ms(fun({a,_}) -> object() end).\n  ```\n\n- The special match specification variables `'$_'` and `'$*'` can be accessed\n  through the pseudo functions `object()` (for `'$_'`) and `bindings()` (for\n  `'$*'`). As an example, one can translate the following `ets:match_object/2`\n  call to a `ets:select/2` call:\n\n  ```erlang\n  ets:match_object(Table, {'$1',test,'$2'}).\n  ```\n\n  This is the same as:\n\n  ```erlang\n  ets:select(Table, ets:fun2ms(fun({A,test,B}) -> object() end)).\n  ```\n\n  In this simple case, the former expression is probably preferable in terms of\n  readability.\n\n  The `ets:select/2` call conceptually looks like this in the resulting code:\n\n  ```erlang\n  ets:select(Table, [{{'$1',test,'$2'},[],['$_']}]).\n  ```\n\n  Matching on the top-level of the fun head can be a more natural way to access\n  `'$_'`, see above.\n\n- Term constructions/literals are translated as much as is needed to get them\n  into valid match specification. This way tuples are made into match\n  specification tuple constructions (a one element tuple containing the tuple)\n  and constant expressions are used when importing variables from the\n  environment. Records are also translated into plain tuple constructions, calls\n  to element, and so on. The guard test [`is_record/2`](`is_record/2`) is\n  translated into match specification code using the three parameter version\n  that is built into match specification, so that\n  [`is_record(A,t)`](`is_record/2`) is translated into `{is_record,'$1',t,5}` if\n  the record size of record type `t` is 5.\n- Language constructions such as `case`, `if`, and `catch` that are not present\n  in match specifications are not allowed.\n- If header file `ms_transform.hrl` is not included, the fun is not translated,\n  which can result in a _runtime error_ (depending on whether the fun is valid\n  in a pure Erlang context).\n\n  Ensure that the header is included when using `ets` and `dbg:fun2ms/1` in\n  compiled code.\n\n- If pseudo function triggering the translation is `ets:fun2ms/1`, the head of\n  the fun must contain a single variable or a single tuple. If the pseudo\n  function is `dbg:fun2ms/1`, the head of the fun must contain a single variable\n  or a single list.\n\nThe translation from funs to match specifications is done at compile time, so\nruntime performance is not affected by using these pseudo functions.\n\nFor more information about match specifications, see the\n[Match specifications in Erlang](`e:erts:match_spec.md`) in ERTS User's Guide.","ref":"ms_transform.html#module-warnings-and-restrictions"},{"type":"function","title":"ms_transform.format_error/1","doc":"Takes an error code returned by one of the other functions in the module and\ncreates a textual description of the error.","ref":"ms_transform.html#format_error/1"},{"type":"function","title":"ms_transform.parse_transform/2","doc":"Implements the transformation at compile time. This function is called by the\ncompiler to do the source code transformation if and when header file\n`ms_transform.hrl` is included in the source code.\n\nFor information about how to use this parse transformation, see `m:ets` and\n`dbg:fun2ms/1`.\n\nFor a description of match specifications, see section\n[Match Specification in Erlang](`e:erts:match_spec.md`) in ERTS User's Guide.","ref":"ms_transform.html#parse_transform/2"},{"type":"function","title":"ms_transform.transform_from_shell/3","doc":"Implements the transformation when the `fun2ms/1` functions are called from the\nshell. In this case, the abstract form is for one single fun (parsed by the\nErlang shell). All imported variables are to be in the key-value list passed as\n`BoundEnvironment`. The result is a term, normalized, that is, not in abstract\nformat.","ref":"ms_transform.html#transform_from_shell/3"},{"type":"module","title":"array","doc":"Functional, extendible arrays.\n\nArrays can have fixed size, or can grow automatically as needed. A default value\nis used for entries that have not been explicitly set.\n\nArrays uses _zero_-based indexing. This is a deliberate design choice and\ndiffers from other Erlang data structures, for example, tuples.\n\nUnless specified by the user when the array is created, the default value is the\natom `undefined`. There is no difference between an unset entry and an entry\nthat has been explicitly set to the same value as the default one (compare\n`reset/2`). If you need to differentiate between unset and set entries, ensure\nthat the default value cannot be confused with the values of set entries.\n\nThe array never shrinks automatically. If an index `I` has been used to set an\nentry successfully, all indices in the range `[0,I]` stay accessible unless the\narray size is explicitly changed by calling `resize/2`.\n\n_Examples:_\n\nCreate a fixed-size array with entries 0-9 set to `undefined`:\n\n```\nA0 = array:new(10).\n10 = array:size(A0).\n```\n\nCreate an extendible array and set entry 17 to `true`, causing the array to grow\nautomatically:\n\n```\nA1 = array:set(17, true, array:new()).\n18 = array:size(A1).\n```\n\nRead back a stored value:\n\n```\ntrue = array:get(17, A1).\n```\n\nAccessing an unset entry returns default value:\n\n```\nundefined = array:get(3, A1)\n```\n\nAccessing an entry beyond the last set entry also returns the default value, if\nthe array does not have fixed size:\n\n```\nundefined = array:get(18, A1).\n```\n\n\"Sparse\" functions ignore default-valued entries:\n\n```\nA2 = array:set(4, false, A1).\n[{4, false}, {17, true}] = array:sparse_to_orddict(A2).\n```\n\nAn extendible array can be made fixed-size later:\n\n```\nA3 = array:fix(A2).\n```\n\nA fixed-size array does not grow automatically and does not allow accesses\nbeyond the last set entry:\n\n```\n{'EXIT',{badarg,_}} = (catch array:set(18, true, A3)).\n{'EXIT',{badarg,_}} = (catch array:get(18, A3)).\n```","ref":"array.html"},{"type":"type","title":"array.array/0","doc":"","ref":"array.html#t:array/0"},{"type":"opaque","title":"array.array/1","doc":"A functional, extendible array. The representation is not documented and is\nsubject to change without notice. Notice that arrays cannot be directly compared\nfor equality.","ref":"array.html#t:array/1"},{"type":"type","title":"array.array_indx/0","doc":"","ref":"array.html#t:array_indx/0"},{"type":"type","title":"array.array_opt/0","doc":"","ref":"array.html#t:array_opt/0"},{"type":"type","title":"array.array_opts/0","doc":"","ref":"array.html#t:array_opts/0"},{"type":"function","title":"array.default/1","doc":"Gets the value used for uninitialized entries.\n\nSee also `new/2`.","ref":"array.html#default/1"},{"type":"function","title":"array.fix/1","doc":"Fixes the array size. This prevents it from growing automatically upon\ninsertion.\n\nSee also `set/3` and `relax/1`.","ref":"array.html#fix/1"},{"type":"function","title":"array.foldl/3","doc":"Folds the array elements using the specified function and initial accumulator\nvalue. The elements are visited in order from the lowest index to the highest.\n\nIf `Function` is not a function, the call fails with reason `badarg`.\n\nSee also `foldr/3`, `map/2`, `sparse_foldl/3`.","ref":"array.html#foldl/3"},{"type":"function","title":"array.foldr/3","doc":"Folds the array elements right-to-left using the specified function and initial\naccumulator value. The elements are visited in order from the highest index to\nthe lowest.\n\nIf `Function` is not a function, the call fails with reason `badarg`.\n\nSee also `foldl/3`, `map/2`.","ref":"array.html#foldr/3"},{"type":"function","title":"array.from_list/1","doc":"Equivalent to [`from_list(List, undefined)`](`from_list/2`).","ref":"array.html#from_list/1"},{"type":"function","title":"array.from_list/2","doc":"Converts a list to an extendible array. `Default` is used as the value for\nuninitialized entries of the array.\n\nIf `List` is not a proper list, the call fails with reason `badarg`.\n\nSee also `new/2`, `to_list/1`.","ref":"array.html#from_list/2"},{"type":"function","title":"array.from_orddict/1","doc":"Equivalent to [`from_orddict(Orddict, undefined)`](`from_orddict/2`).","ref":"array.html#from_orddict/1"},{"type":"function","title":"array.from_orddict/2","doc":"Converts an ordered list of pairs `{Index, Value}` to a corresponding extendible\narray. `Default` is used as the value for uninitialized entries of the array.\n\nIf `Orddict` is not a proper, ordered list of pairs whose first elements are\nnon-negative integers, the call fails with reason `badarg`.\n\nSee also `new/2`, `to_orddict/1`.","ref":"array.html#from_orddict/2"},{"type":"function","title":"array.get/2","doc":"Gets the value of entry `I`.\n\nIf `I` is not a non-negative integer, or if the array has fixed size and `I` is\nlarger than the maximum index, the call fails with reason `badarg`.\n\nIf the array does not have fixed size, the default value for any index `I`\ngreater than `size(Array)-1` is returned.\n\nSee also `set/3`.","ref":"array.html#get/2"},{"type":"type","title":"array.indx_pair/1","doc":"","ref":"array.html#t:indx_pair/1"},{"type":"type","title":"array.indx_pairs/1","doc":"","ref":"array.html#t:indx_pairs/1"},{"type":"function","title":"array.is_array/1","doc":"Returns `true` if `X` is an array, otherwise `false`.\n\nNotice that the check is only shallow, as there is no guarantee that `X` is a\nwell-formed array representation even if this function returns `true`.","ref":"array.html#is_array/1"},{"type":"function","title":"array.is_fix/1","doc":"Checks if the array has fixed size. Returns `true` if the array is fixed,\notherwise `false`.\n\nSee also `fix/1`.","ref":"array.html#is_fix/1"},{"type":"function","title":"array.map/2","doc":"Maps the specified function onto each array element. The elements are visited in\norder from the lowest index to the highest.\n\nIf `Function` is not a function, the call fails with reason `badarg`.\n\nSee also `foldl/3`, `foldr/3`, `sparse_map/2`.","ref":"array.html#map/2"},{"type":"function","title":"array.new/0","doc":"Creates a new, extendible array with initial size zero.\n\nSee also `new/1`, `new/2`.","ref":"array.html#new/0"},{"type":"function","title":"array.new/1","doc":"Creates a new array according to the specified options. By default, the array is\nextendible and has initial size zero. Array indices start at `0`.\n\n`Options` is a single term or a list of terms, selected from the following:\n\n- **`N::integer() >= 0` or `{size, N::integer() >= 0}`** - Specifies the initial\n  array size; this also implies `{fixed, true}`. If `N` is not a non-negative\n  integer, the call fails with reason `badarg`.\n\n- **`fixed` or `{fixed, true}`** - Creates a fixed-size array. See also `fix/1`.\n\n- **`{fixed, false}`** - Creates an extendible (non-fixed-size) array.\n\n- **`{default, Value}`** - Sets the default value for the array to `Value`.\n\nOptions are processed in the order they occur in the list, that is, later\noptions have higher precedence.\n\nThe default value is used as the value of uninitialized entries, and cannot be\nchanged once the array has been created.\n\n_Examples:_\n\n```\narray:new(100)\n```\n\ncreates a fixed-size array of size 100.\n\n```\narray:new({default,0})\n```\n\ncreates an empty, extendible array whose default value is `0`.\n\n```\narray:new([{size,10},{fixed,false},{default,-1}])\n```\n\ncreates an extendible array with initial size 10 whose default value is `-1`.\n\nSee also `fix/1`, `from_list/2`, `get/2`, `new/0`, `new/2`, `set/3`.","ref":"array.html#new/1"},{"type":"function","title":"array.new/2","doc":"Creates a new array according to the specified size and options.\n\nIf `Size` is not a non-negative integer, the call fails with reason `badarg`.\nBy default, the array has fixed size. Notice that any size specifications in\n`Options` override parameter `Size`.\n\nIf `Options` is a list, this is equivalent to\n[`new([{size, Size} | Options])`](`new/1`), otherwise it is equivalent to\n[`new([{size, Size} | [Options]])`](`new/1`). However, using this function\ndirectly is more efficient.\n\n_Example:_\n\n```\narray:new(100, {default,0})\n```\n\ncreates a fixed-size array of size 100, whose default value is `0`.\n\nSee also `new/1`.","ref":"array.html#new/2"},{"type":"function","title":"array.relax/1","doc":"Makes the array resizable. (Reverses the effects of `fix/1`.)\n\nSee also `fix/1`.","ref":"array.html#relax/1"},{"type":"function","title":"array.reset/2","doc":"Resets entry `I` to the default value for the array. If the value of entry `I`\nis the default value, the array is returned unchanged.\n\nReset never changes the array size. Shrinking can be done explicitly by calling\n`resize/2`.\n\nIf `I` is not a non-negative integer, or if the array has fixed size and `I` is\nlarger than the maximum index, the call fails with reason `badarg`; compare\n`set/3`\n\nSee also `new/2`, `set/3`.","ref":"array.html#reset/2"},{"type":"function","title":"array.resize/1","doc":"Changes the array size to that reported by `sparse_size/1`. If the specified\narray has fixed size, also the resulting array has fixed size.\n\nSee also `resize/2`, `sparse_size/1`.","ref":"array.html#resize/1"},{"type":"function","title":"array.resize/2","doc":"Change the array size.\n\nIf `Size` is not a non-negative integer, the call fails with reason `badarg`. If\nthe specified array has fixed size, also the resulting array has fixed size.","ref":"array.html#resize/2"},{"type":"function","title":"array.set/3","doc":"Sets entry `I` of the array to `Value`.\n\nIf `I` is not a non-negative integer, or if the array has fixed size and `I` is\nlarger than the maximum index, the call fails with reason `badarg`.\n\nIf the array does not have fixed size, and `I` is greater than `size(Array)-1`,\nthe array grows to size `I+1`.\n\nSee also `get/2`, `reset/2`.","ref":"array.html#set/3"},{"type":"function","title":"array.size/1","doc":"Gets the number of entries in the array. Entries are numbered from `0` to\n`size(Array)-1`. Hence, this is also the index of the first entry that is\nguaranteed to not have been previously set.\n\nSee also `set/3`, `sparse_size/1`.","ref":"array.html#size/1"},{"type":"function","title":"array.sparse_foldl/3","doc":"Folds the array elements using the specified function and initial accumulator\nvalue, skipping default-valued entries. The elements are visited in order from\nthe lowest index to the highest.\n\nIf `Function` is not a function, the call fails with reason `badarg`.\n\nSee also `foldl/3`, `sparse_foldr/3`.","ref":"array.html#sparse_foldl/3"},{"type":"function","title":"array.sparse_foldr/3","doc":"Folds the array elements right-to-left using the specified function and initial\naccumulator value, skipping default-valued entries. The elements are visited in\norder from the highest index to the lowest.\n\nIf `Function` is not a function, the call fails with reason `badarg`.\n\nSee also `foldr/3`, `sparse_foldl/3`.","ref":"array.html#sparse_foldr/3"},{"type":"function","title":"array.sparse_map/2","doc":"Maps the specified function onto each array element, skipping default-valued\nentries. The elements are visited in order from the lowest index to the highest.\n\nIf `Function` is not a function, the call fails with reason `badarg`.\n\nSee also `map/2`.","ref":"array.html#sparse_map/2"},{"type":"function","title":"array.sparse_size/1","doc":"Gets the number of entries in the array up until the last non-default-valued\nentry. That is, returns `I+1` if `I` is the last non-default-valued entry in the\narray, or zero if no such entry exists.\n\nSee also `resize/1`, `size/1`.","ref":"array.html#sparse_size/1"},{"type":"function","title":"array.sparse_to_list/1","doc":"Converts the array to a list, skipping default-valued entries.\n\nSee also `to_list/1`.","ref":"array.html#sparse_to_list/1"},{"type":"function","title":"array.sparse_to_orddict/1","doc":"Converts the array to an ordered list of pairs `{Index, Value}`, skipping\ndefault-valued entries.\n\nSee also `to_orddict/1`.","ref":"array.html#sparse_to_orddict/1"},{"type":"function","title":"array.to_list/1","doc":"Converts the array to a list.\n\nSee also `from_list/2`, `sparse_to_list/1`.","ref":"array.html#to_list/1"},{"type":"function","title":"array.to_orddict/1","doc":"Converts the array to an ordered list of pairs `{Index, Value}`.\n\nSee also `from_orddict/2`, `sparse_to_orddict/1`.","ref":"array.html#to_orddict/1"},{"type":"module","title":"dets","doc":"A disk-based term storage.\n\nThis module provides a term storage on file. The stored terms, in this module\ncalled _objects_, are tuples such that one element is defined to be the key. A\nDets _table_ is a collection of objects with the key at the same position stored\non a file.\n\nThis module is used by the Mnesia application, and is provided \"as is\" for users\nwho are interested in efficient storage of Erlang terms on disk only. Many\napplications only need to store some terms in a file. Mnesia adds transactions,\nqueries, and distribution. The size of Dets files cannot exceed 2 GB. If larger\ntables are needed, table fragmentation in Mnesia can be used.\n\nThree types of Dets tables exist:\n\n- `set`. A table of this type has at most one object with a given key. If an\n  object with a key already present in the table is inserted, the existing\n  object is overwritten by the new object.\n- `bag`. A table of this type has zero or more different objects with a given\n  key.\n- `duplicate_bag`. A table of this type has zero or more possibly matching\n  objects with a given key.\n\nDets tables must be opened before they can be updated or read, and when finished\nthey must be properly closed. If a table is not properly closed, Dets\nautomatically repairs the table. This can take a substantial time if the table\nis large. A Dets table is closed when the process which opened the table\nterminates. If many Erlang processes (users) open the same Dets table, they\nshare the table. The table is properly closed when all users have either\nterminated or closed the table. Dets tables are not properly closed if the\nErlang runtime system terminates abnormally.\n\n> #### Note {: .info }\n>\n> A `^C` command abnormally terminates an Erlang runtime system in a Unix\n> environment with a break-handler.\n\nAs all operations performed by Dets are disk operations, it is important to\nrealize that a single look-up operation involves a series of disk seek and read\noperations. The Dets functions are therefore much slower than the corresponding\n`m:ets` functions, although Dets exports a similar interface.\n\nDets organizes data as a linear hash list and the hash list grows gracefully as\nmore data is inserted into the table. Space management on the file is performed\nby what is called a buddy system. The current implementation keeps the entire\nbuddy system in RAM, which implies that if the table gets heavily fragmented,\nquite some memory can be used up. The only way to defragment a table is to close\nit and then open it again with option `repair` set to `force`.\n\nNotice that type `ordered_set` in Ets is not yet provided by Dets, neither is\nthe limited support for concurrent updates that makes a sequence of `first` and\n`next` calls safe to use on fixed ETS tables. Both these features may be\nprovided by Dets in a future release of Erlang/OTP. Until then, the Mnesia\napplication (or some user-implemented method for locking) must be used to\nimplement safe concurrency. Currently, no Erlang/OTP library has support for\nordered disk-based term storage.\n\nAll Dets functions return `{error, Reason}` if an error occurs (`first/1` and\n`next/2` are exceptions, they exit the process with the error tuple). If badly\nformed arguments are specified, all functions exit the process with a `badarg`\nmessage.","ref":"dets.html"},{"type":"module","title":"See Also - dets","doc":"`m:ets`, `m:mnesia`, `m:qlc`","ref":"dets.html#module-see-also"},{"type":"type","title":"dets.access/0","doc":"","ref":"dets.html#t:access/0"},{"type":"function","title":"dets.all/0","doc":"Returns a list of the names of all open tables on this node.","ref":"dets.html#all/0"},{"type":"type","title":"dets.auto_save/0","doc":"","ref":"dets.html#t:auto_save/0"},{"type":"function","title":"dets.bchunk/2","doc":"Returns a list of objects stored in a table. The exact representation of the\nreturned objects is not public.\n\nThe lists of data can be used for initializing a table by specifying value\n`bchunk` to option `format` of function `init_table/3`. The Mnesia application\nuses this function for copying open tables.\n\nUnless the table is protected using [`safe_fixtable/2`](`safe_fixtable/2`),\ncalls to [`bchunk/2`](`bchunk/2`) do possibly not work as expected if concurrent\nupdates are made to the table.\n\nThe first time [`bchunk/2`](`bchunk/2`) is called, an initial continuation, the\natom `start`, must be provided.\n\n[`bchunk/2`](`bchunk/2`) returns a tuple `{Continuation2, Data}`, where `Data`\nis a list of objects. `Continuation2` is another continuation that is to be\npassed on to a subsequent call to [`bchunk/2`](`bchunk/2`). With a series of\ncalls to [`bchunk/2`](`bchunk/2`), all table objects can be extracted.\n\n[`bchunk/2`](`bchunk/2`) returns `'$end_of_table'` when all objects are\nreturned, or `{error, Reason}` if an error occurs.","ref":"dets.html#bchunk/2"},{"type":"opaque","title":"dets.bindings_cont/0","doc":"Opaque continuation used by `match/1` and `match/3`.","ref":"dets.html#t:bindings_cont/0"},{"type":"function","title":"dets.close/1","doc":"Closes a table. Only processes that have opened a table are allowed to close it.\n\nAll open tables must be closed before the system is stopped. If an attempt is\nmade to open a table that is not properly closed, Dets automatically tries to\nrepair it.","ref":"dets.html#close/1"},{"type":"opaque","title":"dets.cont/0","doc":"Opaque continuation used by `bchunk/2`.","ref":"dets.html#t:cont/0"},{"type":"function","title":"dets.delete/2","doc":"Deletes all objects with key `Key` from table `Name`.","ref":"dets.html#delete/2"},{"type":"function","title":"dets.delete_all_objects/1","doc":"Deletes all objects from a table in almost constant time. However, if the table\nif fixed, [`delete_all_objects(T)`](`delete_all_objects/1`) is equivalent to\n[`match_delete(T, '_')`](`match_delete/2`).","ref":"dets.html#delete_all_objects/1"},{"type":"function","title":"dets.delete_object/2","doc":"Deletes all instances of a specified object from a table. If a table is of type\n`bag` or `duplicate_bag`, this function can be used to delete only some of the\nobjects with a specified key.","ref":"dets.html#delete_object/2"},{"type":"function","title":"dets.first/1","doc":"Returns the first key stored in table `Name` according to the internal order of\nthe table, or `'$end_of_table'` if the table is empty.\n\nUnless the table is protected using [`safe_fixtable/2`](`safe_fixtable/2`),\nsubsequent calls to `next/2` do possibly not work as expected if concurrent\nupdates are made to the table.\n\nIf an error occurs, the process is exited with an error tuple `{error, Reason}`.\nThe error tuple is not returned, as it cannot be distinguished from a key.\n\nThere are two reasons why [`first/1`](`first/1`) and [`next/2`](`next/2`) are\nnot to be used: they are not efficient, and they prevent the use of key\n`'$end_of_table'`, as this atom is used to indicate the end of the table. If\npossible, use functions [`match`](`match/1`),\n[`match_object`](`match_object/1`), and [`select`](`select/1`) for traversing\ntables.","ref":"dets.html#first/1"},{"type":"function","title":"dets.foldl/3","doc":"","ref":"dets.html#foldl/3"},{"type":"function","title":"dets.foldr/3","doc":"Calls `Function` on successive elements of table `Name` together with an extra\nargument `AccIn`. The table elements are traversed in unspecified order.\n`Function` must return a new accumulator that is passed to the next call. `Acc0`\nis returned if the table is empty.","ref":"dets.html#foldr/3"},{"type":"function","title":"dets.from_ets/2","doc":"Deletes all objects of table `Name` and then inserts all the objects of the ETS\ntable `EtsTab`. The objects are inserted in unspecified order. As\n`ets:safe_fixtable/2` is called, the ETS table must be public or owned by the\ncalling process.","ref":"dets.html#from_ets/2"},{"type":"function","title":"dets.info/1","doc":"Returns information about table `Name` as a list of tuples:\n\n- `{file_size, integer() >= 0}}` \\- The file size, in bytes.\n- `{filename,` `t:file:name/0` `}` \\- The name of the file where objects are\n  stored.\n- `{keypos,` `t:keypos/0` `}` \\- The key position.\n- `{size, integer() >= 0}` \\- The number of objects stored in the table.\n- `{type,` `t:type/0` `}` \\- The table type.","ref":"dets.html#info/1"},{"type":"function","title":"dets.info/2","doc":"Returns the information associated with `Item` for table `Name`. In addition to\nthe `{Item, Value}` pairs defined for `info/1`, the following items are allowed:\n\n- `{access,` `t:access/0` `}` \\- The access mode.\n- `{auto_save,` `t:auto_save/0` `}` \\- The autosave interval.\n- `{bchunk_format, binary()}` \\- An opaque binary describing the format of the\n  objects returned by [`bchunk/2`](`bchunk/2`). The binary can be used as\n  argument to\n  [`is_compatible_bchunk_format/2`](`is_compatible_bchunk_format/2`).\n- `{hash, Hash}` \\- Describes which BIF is used to calculate the hash values of\n  the objects stored in the Dets table. Possible values of `Hash`:\n\n  - `phash` \\- Implies that the `erlang:phash/2` BIF is used.\n  - `phash2` \\- Implies that the `erlang:phash2/1` BIF is used.\n\n- `{memory, integer() >= 0}` \\- The file size, in bytes. The same value is\n  associated with item `file_size`.\n- `{no_keys, integer >= 0()}` \\- The number of different keys stored in the\n  table.\n- `{no_objects, integer >= 0()}` \\- The number of objects stored in the table.\n- `{no_slots, {Min, Used, Max}}` \\- The number of slots of the table. `Min` is\n  the minimum number of slots, `Used` is the number of currently used slots, and\n  `Max` is the maximum number of slots.\n- `{owner, pid()}` \\- The pid of the process that handles requests to the Dets\n  table.\n- `{ram_file, boolean()}` \\- Whether the table is kept in RAM.\n- `{safe_fixed_monotonic_time, SafeFixed}` \\- If the table is fixed, `SafeFixed`\n  is a tuple `{FixedAtTime, [{Pid,RefCount}]}`. `FixedAtTime` is the time when\n  the table was first fixed, and `Pid` is the pid of the process that fixes the\n  table `RefCount` times. There can be any number of processes in the list. If\n  the table is not fixed, `SafeFixed` is the atom `false`.\n\n  `FixedAtTime` corresponds to the result returned by `erlang:monotonic_time/0`\n  at the time of fixation. The use of `safe_fixed_monotonic_time` is\n  [time warp safe](`e:erts:time_correction.md#time-warp-safe-code`).\n\n- `{safe_fixed, SafeFixed}` \\- The same as\n  `{safe_fixed_monotonic_time, SafeFixed}` except the format and value of\n  `FixedAtTime`.\n\n  `FixedAtTime` corresponds to the result returned by `erlang:timestamp/0` at\n  the time of fixation. Notice that when the system uses single or multi\n  [time warp modes](`e:erts:time_correction.md#time-warp-modes`), this can\n  produce strange results. This is because the use of `safe_fixed` is not\n  [time warp safe](`e:erts:time_correction.md#time-warp-safe-code`). Time warp\n  safe code must use `safe_fixed_monotonic_time` instead.","ref":"dets.html#info/2"},{"type":"function","title":"dets.init_table/2","doc":"","ref":"dets.html#init_table/2"},{"type":"function","title":"dets.init_table/3","doc":"Replaces the existing objects of table `Name` with objects created by calling\nthe input function `InitFun`.\n\nThe reason for using this function rather than calling [`insert/2`](`insert/2`)\nis that of efficiency. Notice that the input functions are called by the process\nthat handles requests to the Dets table, not by the calling process.\n\nWhen called with argument `read`, function `InitFun` is assumed to return\n`end_of_input` when there is no more input, or `{Objects, Fun}`, where `Objects`\nis a list of objects and `Fun` is a new input function. Any other value `Value`\nis returned as an error `{error, {init_fun, Value}}`. Each input function is\ncalled exactly once, and if an error occurs, the last function is called with\nargument `close`, the reply of which is ignored.\n\nIf the table type is `set` and more than one object exists with a given key, one\nof the objects is chosen. This is not necessarily the last object with the given\nkey in the sequence of objects returned by the input functions. Avoid duplicate\nkeys, otherwise the file becomes unnecessarily fragmented. This holds also for\nduplicated objects stored in tables of type `bag`.\n\nIt is important that the table has a sufficient number of slots for the objects.\nIf not, the hash list starts to grow when [`init_table/2`](`init_table/2`)\nreturns, which significantly slows down access to the table for a period of\ntime. The minimum number of slots is set by the [`open_file/2`](`open_file/2`)\noption `min_no_slots` and returned by the [`info/2`](`info/2`) item `no_slots`.\nSee also option `min_no_slots` below.\n\nArgument `Options` is a list of `{Key, Val}` tuples, where the following values\nare allowed:\n\n- `{min_no_slots, no_slots()}` \\- Specifies the estimated number of different\n  keys to be stored in the table. The [`open_file/2`](`open_file/2`) option with\n  the same name is ignored, unless the table is created, in which case\n  performance can be enhanced by supplying an estimate when initializing the\n  table.\n- `{format, Format}` \\- Specifies the format of the objects returned by function\n  `InitFun`. If `Format` is `term` (the default), `InitFun` is assumed to return\n  a list of tuples. If `Format` is `bchunk`, `InitFun` is assumed to return\n  `Data` as returned by `bchunk/2`. This option overrides option `min_no_slots`.","ref":"dets.html#init_table/3"},{"type":"function","title":"dets.insert/2","doc":"Inserts one or more objects into the table `Name`. If there already exists an\nobject with a key matching the key of some of the given objects and the table\ntype is `set`, the old object will be replaced.","ref":"dets.html#insert/2"},{"type":"function","title":"dets.insert_new/2","doc":"Inserts one or more objects into table `Name`. If there already exists some\nobject with a key matching the key of any of the specified objects, the table is\nnot updated and `false` is returned. Otherwise the objects are inserted and\n`true` returned.","ref":"dets.html#insert_new/2"},{"type":"function","title":"dets.is_compatible_bchunk_format/2","doc":"Returns `true` if it would be possible to initialize table `Name`, using\n`init_table/3` with option `{format, bchunk}`, with objects read with `bchunk/2`\nfrom some table `T`, such that calling [`info(T, bchunk_format)`](`info/2`)\nreturns `BchunkFormat`.","ref":"dets.html#is_compatible_bchunk_format/2"},{"type":"function","title":"dets.is_dets_file/1","doc":"Returns `true` if file `Filename` is a Dets table, otherwise `false`.","ref":"dets.html#is_dets_file/1"},{"type":"type","title":"dets.keypos/0","doc":"","ref":"dets.html#t:keypos/0"},{"type":"function","title":"dets.lookup/2","doc":"Returns a list of all objects with key `Key` stored in table `Name`, for\nexample:\n\n```erlang\n2> dets:open_file(abc, [{type, bag}]).\n{ok,abc}\n3> dets:insert(abc, {1,2,3}).\nok\n4> dets:insert(abc, {1,3,4}).\nok\n5> dets:lookup(abc, 1).\n[{1,2,3},{1,3,4}]\n```\n\nIf the table type is `set`, the function returns either the empty list or a list\nwith one object, as there cannot be more than one object with a given key. If\nthe table type is `bag` or `duplicate_bag`, the function returns a list of\narbitrary length.\n\nNotice that the order of objects returned is unspecified. In particular, the\norder in which objects were inserted is not reflected.","ref":"dets.html#lookup/2"},{"type":"function","title":"dets.match/1","doc":"Matches some objects stored in a table and returns a non-empty list of the\nbindings matching a specified pattern in some unspecified order. The table, the\npattern, and the number of objects that are matched are all defined by\n`Continuation`, which has been returned by a previous call to\n[`match/1`](`match/1`) or [`match/3`](`match/3`).\n\nWhen all table objects are matched, `'$end_of_table'` is returned.","ref":"dets.html#match/1"},{"type":"function","title":"dets.match/2","doc":"Returns for each object of table `Name` that matches `Pattern` a list of\nbindings in some unspecified order. For a description of patterns, see\n`ets:match/2`. If the keypos'th element of `Pattern` is unbound, all table\nobjects are matched. If the keypos'th element is bound, only the objects with\nthe correct key are matched.","ref":"dets.html#match/2"},{"type":"function","title":"dets.match/3","doc":"Matches some or all objects of table `Name` and returns a non-empty list of the\nbindings that match `Pattern` in some unspecified order. For a description of\npatterns, see `ets:match/2`.\n\nA tuple of the bindings and a continuation is returned, unless the table is\nempty, in which case `'$end_of_table'` is returned. The continuation is to be\nused when matching further objects by calling `match/1`.\n\nIf the keypos'th element of `Pattern` is bound, all table objects are matched.\nIf the keypos'th element is unbound, all table objects are matched, `N` objects\nat a time, until at least one object matches or the end of the table is reached.\nThe default, indicated by giving `N` the value `default`, is to let the number\nof objects vary depending on the sizes of the objects. All objects with the same\nkey are always matched at the same time, which implies that more than N objects\ncan sometimes be matched.\n\nThe table is always to be protected using `safe_fixtable/2` before calling\n[`match/3`](`match/3`), otherwise errors can occur when calling\n[`match/1`](`match/1`).","ref":"dets.html#match/3"},{"type":"function","title":"dets.match_delete/2","doc":"Deletes all objects that match `Pattern` from table `Name`. For a description of\npatterns, see `ets:match/2`.\n\nIf the keypos'th element of `Pattern` is bound, only the objects with the\ncorrect key are matched.","ref":"dets.html#match_delete/2"},{"type":"function","title":"dets.match_object/1","doc":"Returns a non-empty list of some objects stored in a table that match a given\npattern in some unspecified order. The table, the pattern, and the number of\nobjects that are matched are all defined by `Continuation`, which has been\nreturned by a previous call to [`match_object/1`](`match_object/1`) or\n[`match_object/3`](`match_object/3`).\n\nWhen all table objects are matched, `'$end_of_table'` is returned.","ref":"dets.html#match_object/1"},{"type":"function","title":"dets.match_object/2","doc":"Returns a list of all objects of table `Name` that match `Pattern` in some\nunspecified order. For a description of patterns, see `ets:match/2`.\n\nIf the keypos'th element of `Pattern` is unbound, all table objects are matched.\nIf the keypos'th element of `Pattern` is bound, only the objects with the\ncorrect key are matched.\n\nUsing the `match_object` functions for traversing all table objects is more\nefficient than calling [`first/1`](`first/1`) and [`next/2`](`next/2`) or\n[`slot/2`](`slot/2`).","ref":"dets.html#match_object/2"},{"type":"function","title":"dets.match_object/3","doc":"Matches some or all objects stored in table `Name` and returns a non-empty list\nof the objects that match `Pattern` in some unspecified order. For a description\nof patterns, see `ets:match/2`.\n\nA list of objects and a continuation is returned, unless the table is empty, in\nwhich case `'$end_of_table'` is returned. The continuation is to be used when\nmatching further objects by calling `match_object/1`.\n\nIf the keypos'th element of `Pattern` is bound, all table objects are matched.\nIf the keypos'th element is unbound, all table objects are matched, `N` objects\nat a time, until at least one object matches or the end of the table is reached.\nThe default, indicated by giving `N` the value `default`, is to let the number\nof objects vary depending on the sizes of the objects. All matching objects with\nthe same key are always returned in the same reply, which implies that more than\nN objects can sometimes be returned.\n\nThe table is always to be protected using `safe_fixtable/2` before calling\n[`match_object/3`](`match_object/3`), otherwise errors can occur when calling\n[`match_object/1`](`match_object/1`).","ref":"dets.html#match_object/3"},{"type":"type","title":"dets.match_spec/0","doc":"Match specifications, see section\n[Match Specification in Erlang](`e:erts:match_spec.md`) in ERTS User's Guide and\nthe `m:ms_transform` module.","ref":"dets.html#t:match_spec/0"},{"type":"function","title":"dets.member/2","doc":"Works like `lookup/2`, but does not return the objects. Returns `true` if one or\nmore table elements has key `Key`, otherwise `false`.","ref":"dets.html#member/2"},{"type":"function","title":"dets.next/2","doc":"Returns either the key following `Key1` in table `Name` according to the\ninternal order of the table, or `'$end_of_table'` if there is no next key.\n\nIf an error occurs, the process is exited with an error tuple `{error, Reason}`.\n\nTo find the first key in the table, use `first/1`.","ref":"dets.html#next/2"},{"type":"type","title":"dets.no_slots/0","doc":"","ref":"dets.html#t:no_slots/0"},{"type":"type","title":"dets.object/0","doc":"","ref":"dets.html#t:object/0"},{"type":"opaque","title":"dets.object_cont/0","doc":"Opaque continuation used by `match_object/1` and `match_object/3`.","ref":"dets.html#t:object_cont/0"},{"type":"function","title":"dets.open_file/1","doc":"Opens an existing table. If the table is not properly closed, it is repaired.\nThe returned reference is to be used as the table name. This function is most\nuseful for debugging purposes.","ref":"dets.html#open_file/1"},{"type":"function","title":"dets.open_file/2","doc":"Opens a table. An empty Dets table is created if no file exists.\n\nThe atom `Name` is the table name. The table name must be provided in all\nsubsequent operations on the table. The name can be used by other processes as\nwell, and many processes can share one table.\n\nIf two processes open the same table by giving the same name and arguments, the\ntable has two users. If one user closes the table, it remains open until the\nsecond user closes it.\n\nArgument `Args` is a list of `{Key, Val}` tuples, where the following values are\nallowed:\n\n- `{access,` `t:access/0` `}` \\- Existing tables can be opened in read-only mode.\n  A table that is opened in read-only mode is not subjected to the automatic\n  file reparation algorithm if it is later opened after a crash. Defaults to\n  `read_write`.\n- `{auto_save,` `t:auto_save/0` `}` \\- The autosave interval. If the interval is\n  an integer `Time`, the table is flushed to disk whenever it is not accessed\n  for `Time` milliseconds. A table that has been flushed requires no reparation\n  when reopened after an uncontrolled emulator halt. If the interval is the atom\n  `infinity`, autosave is disabled. Defaults to 180000 (3 minutes).\n- `{estimated_no_objects,` `t:no_slots/0` `}` \\- Equivalent to option\n  `min_no_slots`.\n- `{file,` `t:file:name/0` `}` \\- The name of the file to be opened. Defaults to\n  the table name.\n- `{max_no_slots,` `t:no_slots/0` `}` \\- The maximum number of slots to be used.\n  Defaults to 32 M, which is the maximal value. Notice that a higher value can\n  increase the table fragmentation, and a smaller value can decrease the\n  fragmentation, at the expense of execution time.\n- `{min_no_slots,` `t:no_slots/0` `}` \\- Application performance can be enhanced\n  with this flag by specifying, when the table is created, the estimated number\n  of different keys to be stored in the table. Defaults to 256, which is the\n  minimum value.\n- `{keypos,` `t:keypos/0` `}` \\- The position of the element of each object to be\n  used as key. Defaults to 1. The ability to explicitly state the key position\n  is most convenient when we want to store Erlang records in which the first\n  position of the record is the name of the record type.\n- `{ram_file, boolean()}` \\- Whether the table is to be kept in RAM. Keeping the\n  table in RAM can sound like an anomaly, but can enhance the performance of\n  applications that open a table, insert a set of objects, and then close the\n  table. When the table is closed, its contents are written to the disk file.\n  Defaults to `false`.\n- `{repair, Value}` \\- `Value` can be either a `t:boolean/0` or the atom\n  `force`. The flag specifies if the Dets server is to invoke the automatic file\n  reparation algorithm. Defaults to `true`. If `false` is specified, no attempt\n  is made to repair the file, and `{error, {needs_repair, FileName}}` is\n  returned if the table must be repaired.\n\n  Value `force` means that a reparation is made even if the table is properly\n  closed. This is a seldom needed option.\n\n  Option `repair` is ignored if the table is already open.\n\n- `{type,` `t:type/0` `}` \\- The table type. Defaults to `set`.","ref":"dets.html#open_file/2"},{"type":"type","title":"dets.pattern/0","doc":"For a description of patterns, see `ets:match/2`.","ref":"dets.html#t:pattern/0"},{"type":"function","title":"dets.pid2name/1","doc":"Returns the table name given the pid of a process that handles requests to a\ntable, or `undefined` if there is no such table.\n\nThis function is meant to be used for debugging only.","ref":"dets.html#pid2name/1"},{"type":"function","title":"dets.repair_continuation/2","doc":"This function can be used to restore an opaque continuation returned by\n`select/3` or `select/1` if the continuation has passed through external term\nformat (been sent between nodes or stored on disk).\n\nThe reason for this function is that continuation terms contain compiled match\nspecifications and therefore are invalidated if converted to external term\nformat. Given that the original match specification is kept intact, the\ncontinuation can be restored, meaning it can once again be used in subsequent\n[`select/1`](`select/1`) calls even though it has been stored on disk or on\nanother node.\n\nFor more information and examples, see the `m:ets` module.\n\n> #### Note {: .info }\n>\n> This function is rarely needed in application code. It is used by application\n> Mnesia to provide distributed [`select/3`](`select/3`) and\n> [`select/1`](`select/1`) sequences. A normal application would either use\n> Mnesia or keep the continuation from being converted to external format.\n>\n> The reason for not having an external representation of compiled match\n> specifications is performance. It can be subject to change in future releases,\n> while this interface remains for backward compatibility.","ref":"dets.html#repair_continuation/2"},{"type":"function","title":"dets.safe_fixtable/2","doc":"If `Fix` is `true`, table `Name` is fixed (once more) by the calling process,\notherwise the table is released. The table is also released when a fixing\nprocess terminates.\n\nIf many processes fix a table, the table remains fixed until all processes have\nreleased it or terminated. A reference counter is kept on a per process basis,\nand N consecutive fixes require N releases to release the table.\n\nIt is not guaranteed that calls to [`first/1`](`first/1`), [`next/2`](`next/2`),\nor select and match functions work as expected even if the table is fixed; the\nlimited support for concurrency provided by the `m:ets` module is not yet\nprovided by Dets. Fixing a table currently only disables resizing of the hash\nlist of the table.\n\nIf objects have been added while the table was fixed, the hash list starts to\ngrow when the table is released, which significantly slows down access to the\ntable for a period of time.","ref":"dets.html#safe_fixtable/2"},{"type":"function","title":"dets.select/1","doc":"Applies a match specification to some objects stored in a table and returns a\nnon-empty list of the results. The table, the match specification, and the\nnumber of objects that are matched are all defined by `Continuation`, which is\nreturned by a previous call to `select/1` or `select/3`.\n\nWhen all objects of the table have been matched, `'$end_of_table'` is returned.","ref":"dets.html#select/1"},{"type":"function","title":"dets.select/2","doc":"Returns the results of applying match specification `MatchSpec` to all or some\nobjects stored in table `Name`. The order of the objects is not specified. For a\ndescription of match specifications, see the\n[ERTS User's Guide](`e:erts:match_spec.md`).\n\nIf the keypos'th element of `MatchSpec` is unbound, the match specification is\napplied to all objects of the table. If the keypos'th element is bound, the\nmatch specification is applied to the objects with the correct key(s) only.\n\nUsing the `select` functions for traversing all objects of a table is more\nefficient than calling [`first/1`](`first/1`) and [`next/2`](`next/2`) or\n[`slot/2`](`slot/2`).","ref":"dets.html#select/2"},{"type":"function","title":"dets.select/3","doc":"Returns the results of applying match specification `MatchSpec` to some or all\nobjects stored in table `Name`. The order of the objects is not specified. For a\ndescription of match specifications, see the\n[ERTS User's Guide](`e:erts:match_spec.md`).\n\nA tuple of the results of applying the match specification and a continuation is\nreturned, unless the table is empty, in which case `'$end_of_table'` is\nreturned. The continuation is to be used when matching more objects by calling\n`select/1`.\n\nIf the keypos'th element of `MatchSpec` is bound, the match specification is\napplied to all objects of the table with the correct key(s). If the keypos'th\nelement of `MatchSpec` is unbound, the match specification is applied to all\nobjects of the table, `N` objects at a time, until at least one object matches\nor the end of the table is reached. The default, indicated by giving `N` the\nvalue `default`, is to let the number of objects vary depending on the sizes of\nthe objects. All objects with the same key are always handled at the same time,\nwhich implies that the match specification can be applied to more than N\nobjects.\n\nThe table is always to be protected using `safe_fixtable/2` before calling\n[`select/3`](`select/3`), otherwise errors can occur when calling\n[`select/1`](`select/1`).","ref":"dets.html#select/3"},{"type":"opaque","title":"dets.select_cont/0","doc":"Opaque continuation used by `select/1` and `select/3`.","ref":"dets.html#t:select_cont/0"},{"type":"function","title":"dets.select_delete/2","doc":"Deletes each object from table `Name` such that applying match specification\n`MatchSpec` to the object returns value `true`. For a description of match\nspecifications, see the [ERTS User's Guide](`e:erts:match_spec.md`). Returns the\nnumber of deleted objects.\n\nIf the keypos'th element of `MatchSpec` is bound, the match specification is\napplied to the objects with the correct key(s) only.","ref":"dets.html#select_delete/2"},{"type":"function","title":"dets.slot/2","doc":"The objects of a table are distributed among slots, starting with slot `0` and\nending with slot `n`. Returns the list of objects associated with slot `I`. If\n`I` > `n`, `'$end_of_table'` is returned.","ref":"dets.html#slot/2"},{"type":"function","title":"dets.sync/1","doc":"Ensures that all updates made to table `Name` are written to disk. This also\napplies to tables that have been opened with flag `ram_file` set to `true`. In\nthis case, the contents of the RAM file are flushed to disk.\n\nNotice that the space management data structures kept in RAM, the buddy system,\nis also written to the disk. This can take some time if the table is fragmented.","ref":"dets.html#sync/1"},{"type":"type","title":"dets.tab_name/0","doc":"","ref":"dets.html#t:tab_name/0"},{"type":"function","title":"dets.table/1","doc":"","ref":"dets.html#table/1"},{"type":"function","title":"dets.table/2","doc":"Returns a Query List Comprehension (QLC) query handle. The `m:qlc` module\nprovides a query language aimed mainly for Mnesia, but ETS tables, Dets tables,\nand lists are also recognized by `qlc` as sources of data. Calling\n[`dets:table/1,2`](`table/1`) is the means to make Dets table `Name` usable to\n`qlc`.\n\nWhen there are only simple restrictions on the key position, `qlc` uses\n[`dets:lookup/2`](`lookup/2`) to look up the keys. When that is not possible,\nthe whole table is traversed. Option `traverse` determines how this is done:\n\n- `first_next` \\- The table is traversed one key at a time by calling\n  `dets:first/1` and `dets:next/2`.\n- `select` \\- The table is traversed by calling [`dets:select/3`](`select/3`)\n  and [`dets:select/1`](`select/1`). Option `n_objects` determines the number of\n  objects returned (the third argument of [`select/3`](`select/3`)). The match\n  specification (the second argument of [`select/3`](`select/3`)) is assembled\n  by `qlc`:\n\n  - Simple filters are translated into equivalent match specifications.\n  - More complicated filters must be applied to all objects returned by\n    [`select/3`](`select/3`) given a match specification that matches all\n    objects.\n\n- `{select,` `t:match_spec/0` `}` \\- As for `select`, the table is traversed by\n  calling `dets:select/3` and `dets:select/1`. The difference is that the match\n  specification is specified explicitly. This is how to state match\n  specifications that cannot easily be expressed within the syntax provided by\n  `qlc`.\n\nThe following example uses an explicit match specification to traverse the\ntable:\n\n```erlang\n1> dets:open_file(t, []),\nok = dets:insert(t, [{1,a},{2,b},{3,c},{4,d}]),\nMS = ets:fun2ms(fun({X,Y}) when (X > 1) or (X < 5) -> {Y} end),\nQH1 = dets:table(t, [{traverse, {select, MS}}]).\n```\n\nAn example with implicit match specification:\n\n```erlang\n2> QH2 = qlc:q([{Y} || {X,Y} <- dets:table(t), (X > 1) or (X < 5)]).\n```\n\nThe latter example is equivalent to the former, which can be verified using\nfunction `qlc:info/1`:\n\n```erlang\n3> qlc:info(QH1) =:= qlc:info(QH2).\ntrue\n```\n\n`qlc:info/1` returns information about a query handle. In this case identical\ninformation is returned for the two query handles.","ref":"dets.html#table/2"},{"type":"function","title":"dets.to_ets/2","doc":"Inserts the objects of the Dets table `Name` into the ETS table `EtsTab`. The\norder in which the objects are inserted is not specified. The existing objects\nof the ETS table are kept unless overwritten.","ref":"dets.html#to_ets/2"},{"type":"function","title":"dets.traverse/2","doc":"Applies `Fun` to each object stored in table `Name` in some unspecified order.\nDifferent actions are taken depending on the return value of `Fun`. The\nfollowing `Fun` return values are allowed:\n\n- **`continue`** - Continue to perform the traversal. For example, the following\n  function can be used to print the contents of a table:\n\n  ```erlang\n  fun(X) -> io:format(\"~p~n\", [X]), continue end.\n  ```\n\n- **`{continue, Val}`** - Continue the traversal and accumulate `Val`. The\n  following function is supplied to collect all objects of a table in a list:\n\n  ```text\n  fun(X) -> {continue, X} end.\n  ```\n\n- **`{done, Value}`** - Terminate the traversal and return `[Value | Acc]`.\n\nAny other value `OtherValue` returned by `Fun` terminates the traversal and is\nreturned immediately.","ref":"dets.html#traverse/2"},{"type":"type","title":"dets.type/0","doc":"","ref":"dets.html#t:type/0"},{"type":"function","title":"dets.update_counter/3","doc":"Updates the object with key `Key` stored in table `Name` of type `set` by adding\n`Incr` to the element at the `Pos`:th position. The new counter value is\nreturned. If no position is specified, the element directly following the key is\nupdated.\n\nThis functions provides a way of updating a counter, without having to look up\nan object, update the object by incrementing an element, and insert the\nresulting object into the table again.","ref":"dets.html#update_counter/3"},{"type":"module","title":"dict","doc":"A Key-value dictionary.\n\nThe representation of a dictionary is not defined.\n\nThis module provides the same interface as the `m:orddict` module. One\ndifference is that while this module considers two keys as different if they do\nnot match (`=:=`), `orddict` considers two keys as different if and only if they\ndo not compare equal (`==`).","ref":"dict.html"},{"type":"module","title":"Notes - dict","doc":"[](){: #notes }\n\nFunctions `append` and `append_list` are included so that keyed values can be\nstored in a list _accumulator_, for example:\n\n```erlang\n> D0 = dict:new(),\n  D1 = dict:store(files, [], D0),\n  D2 = dict:append(files, f1, D1),\n  D3 = dict:append(files, f2, D2),\n  D4 = dict:append(files, f3, D3),\n  dict:fetch(files, D4).\n[f1,f2,f3]\n```\n\nThis saves the trouble of first fetching a keyed value, appending a new value to\nthe list of stored values, and storing the result.\n\nFunction `fetch` is to be used if the key is known to be in the dictionary,\notherwise function `find`.","ref":"dict.html#module-notes"},{"type":"module","title":"See Also - dict","doc":"`m:gb_trees`, `m:orddict`","ref":"dict.html#module-see-also"},{"type":"function","title":"dict.append/3","doc":"Appends a new `Value` to the current list of values associated with `Key`.\n\nSee also section [Notes](`m:dict#module-notes`).","ref":"dict.html#append/3"},{"type":"function","title":"dict.append_list/3","doc":"Appends a list of values `ValList` to the current list of values associated with\n`Key`. An exception is generated if the initial value associated with `Key` is\nnot a list of values.\n\nSee also section [Notes](`m:dict#module-notes`).","ref":"dict.html#append_list/3"},{"type":"type","title":"dict.dict/0","doc":"","ref":"dict.html#t:dict/0"},{"type":"opaque","title":"dict.dict/2","doc":"Dictionary as returned by `new/0`.","ref":"dict.html#t:dict/2"},{"type":"function","title":"dict.erase/2","doc":"Erases all items with a given key from a dictionary.","ref":"dict.html#erase/2"},{"type":"function","title":"dict.fetch/2","doc":"Returns the value associated with `Key` in dictionary `Dict`. This function\nassumes that `Key` is present in dictionary `Dict`, and an exception is\ngenerated if `Key` is not in the dictionary.\n\nSee also section [Notes](`m:dict#module-notes`).","ref":"dict.html#fetch/2"},{"type":"function","title":"dict.fetch_keys/1","doc":"Returns a list of all keys in dictionary `Dict`.","ref":"dict.html#fetch_keys/1"},{"type":"function","title":"dict.filter/2","doc":"`Dict2` is a dictionary of all keys and values in `Dict1` for which\n`Pred(Key, Value)` is `true`.","ref":"dict.html#filter/2"},{"type":"function","title":"dict.find/2","doc":"Searches for a key in dictionary `Dict`. Returns `{ok, Value}`, where `Value` is\nthe value associated with `Key`, or `error` if the key is not present in the\ndictionary.\n\nSee also section [Notes](`m:dict#module-notes`).","ref":"dict.html#find/2"},{"type":"function","title":"dict.fold/3","doc":"Calls `Fun` on successive keys and values of dictionary `Dict` together with an\nextra argument `Acc` (short for accumulator). `Fun` must return a new\naccumulator that is passed to the next call. `Acc0` is returned if the\ndictionary is empty. The evaluation order is undefined.","ref":"dict.html#fold/3"},{"type":"function","title":"dict.from_list/1","doc":"Converts the `Key`-`Value` list `List` to dictionary `Dict`.","ref":"dict.html#from_list/1"},{"type":"function","title":"dict.is_empty/1","doc":"Returns `true` if dictionary `Dict` has no elements, otherwise `false`.","ref":"dict.html#is_empty/1"},{"type":"function","title":"dict.is_key/2","doc":"Tests if `Key` is contained in dictionary `Dict`.","ref":"dict.html#is_key/2"},{"type":"function","title":"dict.map/2","doc":"Calls `Fun` on successive keys and values of dictionary `Dict1` to return a new\nvalue for each key. The evaluation order is undefined.","ref":"dict.html#map/2"},{"type":"function","title":"dict.merge/3","doc":"Merges two dictionaries, `Dict1` and `Dict2`, to create a new dictionary. All\nthe `Key`-`Value` pairs from both dictionaries are included in the new\ndictionary. If a key occurs in both dictionaries, `Fun` is called with the key\nand both values to return a new value. `merge` can be defined as follows, but is\nfaster:\n\n```erlang\nmerge(Fun, D1, D2) ->\n    fold(fun (K, V1, D) ->\n                 update(K, fun (V2) -> Fun(K, V1, V2) end, V1, D)\n         end, D2, D1).\n```","ref":"dict.html#merge/3"},{"type":"function","title":"dict.new/0","doc":"Creates a new dictionary.","ref":"dict.html#new/0"},{"type":"function","title":"dict.size/1","doc":"Returns the number of elements in dictionary `Dict`.","ref":"dict.html#size/1"},{"type":"function","title":"dict.store/3","doc":"Stores a `Key`-`Value` pair in dictionary `Dict2`. If `Key` already exists in\n`Dict1`, the associated value is replaced by `Value`.","ref":"dict.html#store/3"},{"type":"function","title":"dict.take/2","doc":"This function returns value from dictionary and a new dictionary without this\nvalue. Returns `error` if the key is not present in the dictionary.","ref":"dict.html#take/2"},{"type":"function","title":"dict.to_list/1","doc":"Converts dictionary `Dict` to a list representation.","ref":"dict.html#to_list/1"},{"type":"function","title":"dict.update/3","doc":"Updates a value in a dictionary by calling `Fun` on the value to get a new\nvalue. An exception is generated if `Key` is not present in the dictionary.","ref":"dict.html#update/3"},{"type":"function","title":"dict.update/4","doc":"Updates a value in a dictionary by calling `Fun` on the value to get a new\nvalue. If `Key` is not present in the dictionary, `Initial` is stored as the\nfirst value. For example, [`append/3`](`append/3`) can be defined as:\n\n```erlang\nappend(Key, Val, D) ->\n    update(Key, fun (Old) -> Old ++ [Val] end, [Val], D).\n```","ref":"dict.html#update/4"},{"type":"function","title":"dict.update_counter/3","doc":"Adds `Increment` to the value associated with `Key` and stores this value. If\n`Key` is not present in the dictionary, `Increment` is stored as the first\nvalue.\n\nThis can be defined as follows, but is faster:\n\n```erlang\nupdate_counter(Key, Incr, D) ->\n    update(Key, fun (Old) -> Old + Incr end, Incr, D).\n```","ref":"dict.html#update_counter/3"},{"type":"module","title":"digraph","doc":"This module provides a version of labeled directed graphs (\"digraphs\").\n\nThe digraphs managed by this module are stored in [ETS tables](`m:ets`). That\nimplies the following:\n\n- Only the process that created the digraph is allowed to update it.\n- Digraphs will not be garbage collected. The ETS tables used for a digraph will\n  only be deleted when `delete/1` is called or the process that created the\n  digraph terminates.\n- A digraph is a mutable data structure.\n\nWhat makes the graphs provided here non-proper directed graphs is that multiple\nedges between vertices are allowed. However, the customary definition of\ndirected graphs is used here.\n\n- A _directed graph_{: #digraph } (or just \"digraph\") is a pair (V, E) of a\n  finite set V of _vertices_{: #vertex } and a finite set E of _directed\n  edges_{: #edge } (or just \"edges\"). The set of edges E is a subset of V × V\n  (the Cartesian product of V with itself).\n\n  In this module, V is allowed to be empty. The so obtained unique digraph is\n  called the _empty digraph_{: #empty_digraph }. Both vertices and edges are\n  represented by unique Erlang terms.\n\n- Digraphs can be annotated with more information. Such information can be\n  attached to the vertices and to the edges of the digraph. An annotated digraph\n  is called a _labeled digraph_, and the information attached to a vertex or an\n  edge is called a _label_{: #label }. Labels are Erlang terms.\n- An edge e = (v, w) is said to _emanate_{: #emanate } from vertex v and to be\n  _incident_{: #incident } on vertex w.\n- The _out-degree_{: #out_degree } of a vertex is the number of edges emanating\n  from that vertex.\n- The _in-degree_{: #in_degree } of a vertex is the number of edges incident on\n  that vertex.\n- If an edge is emanating from v and incident on w, then w is said to be an\n  _out-neighbor_{: #out_neighbour } of v, and v is said to be an _in-neighbor_{:\n  #in_neighbour } of w.\n- A _path_{: #path } P from v\\[1] to v\\[k] in a digraph (V, E) is a non-empty\n  sequence v\\[1], v\\[2], ..., v\\[k] of vertices in V such that there is an edge\n  (v\\[i],v\\[i+1]) in E for 1 <= i < k.\n- The _length_{: #length } of path P is k-1.\n- Path P is _simple_{: #simple_path } if all vertices are distinct, except that\n  the first and the last vertices can be the same.\n- Path P is a _cycle_{: #cycle } if the length of P is not zero and v\\[1] =\n  v\\[k].\n- A _loop_{: #loop } is a cycle of length one.\n- A _simple cycle_{: #simple_cycle } is a path that is both a cycle and simple.\n- An _acyclic digraph_{: #acyclic_digraph } is a digraph without cycles.","ref":"digraph.html"},{"type":"module","title":"See Also - digraph","doc":"`m:digraph_utils`, `m:ets`","ref":"digraph.html#module-see-also"},{"type":"function","title":"digraph.add_edge/3","doc":"","ref":"digraph.html#add_edge/3"},{"type":"function","title":"digraph.add_edge/4","doc":"Equivalent to [`add_edge(G, E, V1, V2, Label)`](`add_edge/5`), where `E` is a created edge.\n\nThe created edge is represented by term `['$e' | N]`, where `N` is an integer >= 0.\n\nSee `t:add_edge_err_rsn/0` for details on possible errors.","ref":"digraph.html#add_edge/4"},{"type":"function","title":"digraph.add_edge/5","doc":"Creates (or modifies) an edge with the identifier\n`E` of digraph `G`, using `Label` as the (new) [label](`m:digraph#label`) of the\nedge. The edge is [emanating](`m:digraph#emanate`) from `V1` and\n[incident](`m:digraph#incident`) on `V2`. Returns `E`.\n\nSee `t:add_edge_err_rsn/0` for details on possible errors.","ref":"digraph.html#add_edge/5"},{"type":"type","title":"digraph.add_edge_err_rsn/0","doc":"The error reason for when an edge could not be added to a graph.\n\nIf the edge would create a cycle in an\n[acyclic digraph](`m:digraph#acyclic_digraph`), `{error, {bad_edge, Path}}` is\nreturned. If `G` already has an edge with value `E` connecting a different pair\nof vertices, `{error, {bad_edge, [V1, V2]}}` is returned. If either of `V1` or\n`V2` is not a vertex of digraph `G`, `{error, {bad_vertex, `V`}}` is returned,\nV = `V1` or V = `V2`.","ref":"digraph.html#t:add_edge_err_rsn/0"},{"type":"function","title":"digraph.add_vertex/1","doc":"Creates a vertex using the empty list as label, and returns the created vertex.\n\nThe created vertex is represented by term `['$v' | N]`, where `N` is an integer >= 0.","ref":"digraph.html#add_vertex/1"},{"type":"function","title":"digraph.add_vertex/2","doc":"","ref":"digraph.html#add_vertex/2"},{"type":"function","title":"digraph.add_vertex/3","doc":"Creates (or modifies) vertex `V` of digraph `G`, using `Label` as the (new)\n[label](`m:digraph#label`) of the vertex. Returns the new vertex `V`.","ref":"digraph.html#add_vertex/3"},{"type":"type","title":"digraph.d_cyclicity/0","doc":"","ref":"digraph.html#t:d_cyclicity/0"},{"type":"type","title":"digraph.d_protection/0","doc":"","ref":"digraph.html#t:d_protection/0"},{"type":"type","title":"digraph.d_type/0","doc":"","ref":"digraph.html#t:d_type/0"},{"type":"function","title":"digraph.del_edge/2","doc":"Deletes edge `E` from digraph `G`.","ref":"digraph.html#del_edge/2"},{"type":"function","title":"digraph.del_edges/2","doc":"Deletes the edges in list `Edges` from digraph `G`.","ref":"digraph.html#del_edges/2"},{"type":"function","title":"digraph.del_path/3","doc":"Deletes edges from digraph `G` until there are no [paths](`m:digraph#path`) from\nvertex `V1` to vertex `V2`.\n\nA sketch of the procedure employed:\n\n- Find an arbitrary [simple path](`m:digraph#simple_path`)\n  v\\[1], v\\[2], ..., v\\[k] from `V1` to `V2` in `G`.\n- Remove all edges of `G` [emanating](`m:digraph#emanate`) from v\\[i] and\n  [incident](`m:digraph#incident`) to v\\[i+1] for 1 <= i < k (including multiple\n  edges).\n- Repeat until there is no path between `V1` and `V2`.","ref":"digraph.html#del_path/3"},{"type":"function","title":"digraph.del_vertex/2","doc":"Deletes vertex `V` from digraph `G`. Any edges [emanating](`m:digraph#emanate`)\nfrom `V` or [incident](`m:digraph#incident`) on `V` are also deleted.","ref":"digraph.html#del_vertex/2"},{"type":"function","title":"digraph.del_vertices/2","doc":"Deletes the vertices in list `Vertices` from digraph `G`.","ref":"digraph.html#del_vertices/2"},{"type":"function","title":"digraph.delete/1","doc":"Deletes digraph `G`. This call is important as digraphs are implemented with\nETS. There is no garbage collection of ETS tables. However, the digraph is\ndeleted if the process that created the digraph terminates.","ref":"digraph.html#delete/1"},{"type":"type","title":"digraph.edge/0","doc":"Serves as the identifier or \"name\" of an edge. This is distinct from an edge\n\"label\" which attaches ancillary information to the edge rather than identifying\nthe edge itself.","ref":"digraph.html#t:edge/0"},{"type":"function","title":"digraph.edge/2","doc":"Returns `{E, V1, V2, Label}`, where `Label` is the [label](`m:digraph#label`) of\nedge `E` [emanating](`m:digraph#emanate`) from `V1` and\n[incident](`m:digraph#incident`) on `V2` of digraph `G`. If no edge `E` of\ndigraph `G` exists, `false` is returned.","ref":"digraph.html#edge/2"},{"type":"function","title":"digraph.edges/1","doc":"Returns a list of all edges of digraph `G`, in some unspecified order.","ref":"digraph.html#edges/1"},{"type":"function","title":"digraph.edges/2","doc":"Returns a list of all edges [emanating](`m:digraph#emanate`) from or\n[incident](`m:digraph#incident`) on `V` of digraph `G`, in some unspecified\norder.","ref":"digraph.html#edges/2"},{"type":"function","title":"digraph.get_cycle/2","doc":"If a [simple cycle](`m:digraph#simple_cycle`) of length two or more exists\nthrough vertex `V`, the cycle is returned as a list `[V, ..., V]` of vertices.\nIf a [loop](`m:digraph#loop`) through `V` exists, the loop is returned as a list\n`[V]`. If no cycles through `V` exist, `false` is returned.\n\n`get_path/3` is used for finding a simple cycle through `V`.","ref":"digraph.html#get_cycle/2"},{"type":"function","title":"digraph.get_path/3","doc":"Tries to find a [simple path](`m:digraph#simple_path`) from vertex `V1` to\nvertex `V2` of digraph `G`. Returns the path as a list `[V1, ..., V2]` of\nvertices, or `false` if no simple path from `V1` to `V2` of length one or more\nexists.\n\nDigraph `G` is traversed in a depth-first manner, and the first found path is\nreturned.","ref":"digraph.html#get_path/3"},{"type":"function","title":"digraph.get_short_cycle/2","doc":"Tries to find an as short as possible [simple cycle](`m:digraph#simple_cycle`)\nthrough vertex `V` of digraph `G`. Returns the cycle as a list `[V, ..., V]` of\nvertices, or `false` if no simple cycle through `V` exists. Notice that a\n[loop](`m:digraph#loop`) through `V` is returned as list `[V, V]`.\n\n`get_short_path/3` is used for finding a simple cycle through `V`.","ref":"digraph.html#get_short_cycle/2"},{"type":"function","title":"digraph.get_short_path/3","doc":"Tries to find an as short as possible [simple path](`m:digraph#simple_path`)\nfrom vertex `V1` to vertex `V2` of digraph `G`. Returns the path as a list\n`[V1, ..., V2]` of vertices, or `false` if no simple path from `V1` to `V2` of\nlength one or more exists.\n\nDigraph `G` is traversed in a breadth-first manner, and the first found path is\nreturned.","ref":"digraph.html#get_short_path/3"},{"type":"opaque","title":"digraph.graph/0","doc":"A digraph as returned by [`new/0,1`](`new/0`).","ref":"digraph.html#t:graph/0"},{"type":"function","title":"digraph.in_degree/2","doc":"Returns the [in-degree](`m:digraph#in_degree`) of vertex `V` of digraph `G`.","ref":"digraph.html#in_degree/2"},{"type":"function","title":"digraph.in_edges/2","doc":"Returns a list of all edges [incident](`m:digraph#incident`) on `V` of digraph\n`G`, in some unspecified order.","ref":"digraph.html#in_edges/2"},{"type":"function","title":"digraph.in_neighbours/2","doc":"Returns a list of all [in-neighbors](`m:digraph#in_neighbour`) of `V` of digraph\n`G`, in some unspecified order.","ref":"digraph.html#in_neighbours/2"},{"type":"function","title":"digraph.info/1","doc":"Returns a list of `{Tag, Value}` pairs describing digraph `G`. The following\npairs are returned:\n\n- `{cyclicity, Cyclicity}`, where `Cyclicity` is `cyclic` or `acyclic`,\n  according to the options given to `new`.\n- `{memory, NoWords}`, where `NoWords` is the number of words allocated to the\n  ETS tables.\n- `{protection, Protection}`, where `Protection` is `protected` or `private`,\n  according to the options given to `new`.","ref":"digraph.html#info/1"},{"type":"type","title":"digraph.label/0","doc":"","ref":"digraph.html#t:label/0"},{"type":"function","title":"digraph.new/0","doc":"","ref":"digraph.html#new/0"},{"type":"function","title":"digraph.new/1","doc":"Returns an [empty digraph](`m:digraph#empty_digraph`) with properties according\nto the options in `Type`:\n\n- **`cyclic`** - Allows [cycles](`m:digraph#cycle`) in the digraph (default).\n\n- **`acyclic`** - The digraph is to be kept\n  [acyclic](`m:digraph#acyclic_digraph`).\n\n- **`protected`** - Other processes can read the digraph (default).\n\n- **`private`** - The digraph can be read and modified by the creating process\n  only.\n\nIf an unrecognized type option `T` is specified or `Type` is not a proper list,\na `badarg` exception is raised.","ref":"digraph.html#new/1"},{"type":"function","title":"digraph.no_edges/1","doc":"Returns the number of edges of digraph `G`.","ref":"digraph.html#no_edges/1"},{"type":"function","title":"digraph.no_vertices/1","doc":"Returns the number of vertices of digraph `G`.","ref":"digraph.html#no_vertices/1"},{"type":"function","title":"digraph.out_degree/2","doc":"Returns the [out-degree](`m:digraph#out_degree`) of vertex `V` of digraph `G`.","ref":"digraph.html#out_degree/2"},{"type":"function","title":"digraph.out_edges/2","doc":"Returns a list of all edges [emanating](`m:digraph#emanate`) from `V` of digraph\n`G`, in some unspecified order.","ref":"digraph.html#out_edges/2"},{"type":"function","title":"digraph.out_neighbours/2","doc":"Returns a list of all [out-neighbors](`m:digraph#out_neighbour`) of `V` of\ndigraph `G`, in some unspecified order.","ref":"digraph.html#out_neighbours/2"},{"type":"type","title":"digraph.vertex/0","doc":"","ref":"digraph.html#t:vertex/0"},{"type":"function","title":"digraph.vertex/2","doc":"Returns `{V, Label}`, where `Label` is the [label](`m:digraph#label`) of the\nvertex `V` of digraph `G`, or `false` if no vertex `V` of digraph `G` exists.","ref":"digraph.html#vertex/2"},{"type":"function","title":"digraph.vertices/1","doc":"Returns a list of all vertices of digraph `G`, in some unspecified order.","ref":"digraph.html#vertices/1"},{"type":"module","title":"digraph_utils","doc":"This module provides algorithms based on depth-first traversal of directed\ngraphs.\n\nFor basic functions on directed graphs, see the `m:digraph` module.\n\n- A _directed graph_{: #digraph } (or just \"digraph\") is a pair (V, E) of a\n  finite set V of _vertices_{: #vertex } and a finite set E of _directed\n  edges_{: #edge } (or just \"edges\"). The set of edges E is a subset of V × V\n  (the Cartesian product of V with itself).\n- Digraphs can be annotated with more information. Such information can be\n  attached to the vertices and to the edges of the digraph. An annotated digraph\n  is called a _labeled digraph_, and the information attached to a vertex or an\n  edge is called a _label_{: #label }.\n- An edge e = (v, w) is said to _emanate_{: #emanate } from vertex v and to be\n  _incident_{: #incident } on vertex w.\n- If an edge is emanating from v and incident on w, then w is said to be an\n  _out-neighbor_{: #out_neighbour } of v, and v is said to be an _in-neighbor_{:\n  #in_neighbour } of w.\n- A _path_{: #path } P from v\\[1] to v\\[k] in a digraph (V, E) is a non-empty\n  sequence v\\[1], v\\[2], ..., v\\[k] of vertices in V such that there is an edge\n  (v\\[i],v\\[i+1]) in E for 1 <= i < k.\n- The _length_{: #length } of path P is k-1.\n- Path P is a _cycle_{: #cycle } if the length of P is not zero and v\\[1] =\n  v\\[k].\n- A _loop_{: #loop } is a cycle of length one.\n- An _acyclic digraph_{: #acyclic_digraph } is a digraph without cycles.\n- A _depth-first traversal_{: #depth_first_traversal } of a directed digraph can\n  be viewed as a process that visits all vertices of the digraph. Initially, all\n  vertices are marked as unvisited. The traversal starts with an arbitrarily\n  chosen vertex, which is marked as visited, and follows an edge to an unmarked\n  vertex, marking that vertex. The search then proceeds from that vertex in the\n  same fashion, until there is no edge leading to an unvisited vertex. At that\n  point the process backtracks, and the traversal continues as long as there are\n  unexamined edges. If unvisited vertices remain when all edges from the first\n  vertex have been examined, some so far unvisited vertex is chosen, and the\n  process is repeated.\n- A _partial ordering_{: #partial_ordering } of a set S is a transitive,\n  antisymmetric, and reflexive relation between the objects of S.\n- The problem of _topological sorting_{: #topsort } is to find a total ordering\n  of S that is a superset of the partial ordering. A digraph G = (V, E) is\n  equivalent to a relation E on V (we neglect that the version of directed\n  graphs provided by the `digraph` module allows multiple edges between\n  vertices). If the digraph has no cycles of length two or more, the reflexive\n  and transitive closure of E is a partial ordering.\n- A _subgraph_{: #subgraph } G' of G is a digraph whose vertices and edges form\n  subsets of the vertices and edges of G.\n- G' is _maximal_ with respect to a property P if all other subgraphs that\n  include the vertices of G' do not have property P.\n- A _strongly connected component_{: #strong_components } is a maximal subgraph\n  such that there is a path between each pair of vertices.\n- A _connected component_{: #components } is a maximal subgraph such that there\n  is a path between each pair of vertices, considering all edges undirected.\n- An _arborescence_{: #arborescence } is an acyclic digraph with a vertex V, the\n  _root_{: #root }, such that there is a unique path from V to every other\n  vertex of G.\n- A _tree_{: #tree } is an acyclic non-empty digraph such that there is a unique\n  path between every pair of vertices, considering all edges undirected.","ref":"digraph_utils.html"},{"type":"module","title":"See Also - digraph_utils","doc":"`m:digraph`","ref":"digraph_utils.html#module-see-also"},{"type":"function","title":"digraph_utils.arborescence_root/1","doc":"Returns `{yes, Root}` if `Root` is the [root](`m:digraph_utils#root`) of the\narborescence `Digraph`, otherwise `no`.","ref":"digraph_utils.html#arborescence_root/1"},{"type":"function","title":"digraph_utils.components/1","doc":"Returns a list of [connected components](`m:digraph_utils#components`). Each\ncomponent is represented by its vertices. The order of the vertices and the\norder of the components are arbitrary. Each vertex of digraph `Digraph` occurs\nin exactly one component.","ref":"digraph_utils.html#components/1"},{"type":"function","title":"digraph_utils.condensation/1","doc":"Creates a digraph where the vertices are the\n[strongly connected components](`m:digraph_utils#strong_components`) of\n`Digraph` as returned by `strong_components/1`. If X and Y are two different\nstrongly connected components, and vertices x and y exist in X and Y,\nrespectively, such that there is an edge [emanating](`m:digraph_utils#emanate`)\nfrom x and [incident](`m:digraph_utils#incident`) on y, then an edge emanating\nfrom X and incident on Y is created.\n\nThe created digraph has the same type as `Digraph`. All vertices and edges have\nthe default [label](`m:digraph_utils#label`) `[]`.\n\nEach [cycle](`m:digraph_utils#cycle`) is included in some strongly connected\ncomponent, which implies that a\n[topological ordering](`m:digraph_utils#topsort`) of the created digraph always\nexists.","ref":"digraph_utils.html#condensation/1"},{"type":"function","title":"digraph_utils.cyclic_strong_components/1","doc":"Returns a list of\n[strongly connected components](`m:digraph_utils#strong_components`). Each\nstrongly component is represented by its vertices. The order of the vertices and\nthe order of the components are arbitrary. Only vertices that are included in\nsome [cycle](`m:digraph_utils#cycle`) in `Digraph` are returned, otherwise the\nreturned list is equal to that returned by `strong_components/1`.","ref":"digraph_utils.html#cyclic_strong_components/1"},{"type":"function","title":"digraph_utils.is_acyclic/1","doc":"Returns `true` if and only if digraph `Digraph` is\n[acyclic](`m:digraph_utils#acyclic_digraph`).","ref":"digraph_utils.html#is_acyclic/1"},{"type":"function","title":"digraph_utils.is_arborescence/1","doc":"Returns `true` if and only if digraph `Digraph` is an\n[arborescence](`m:digraph_utils#arborescence`).","ref":"digraph_utils.html#is_arborescence/1"},{"type":"function","title":"digraph_utils.is_tree/1","doc":"Returns `true` if and only if digraph `Digraph` is a\n[tree](`m:digraph_utils#tree`).","ref":"digraph_utils.html#is_tree/1"},{"type":"function","title":"digraph_utils.loop_vertices/1","doc":"Returns a list of all vertices of `Digraph` that are included in some\n[loop](`m:digraph_utils#loop`).","ref":"digraph_utils.html#loop_vertices/1"},{"type":"function","title":"digraph_utils.postorder/1","doc":"Returns all vertices of digraph `Digraph`. The order is given by a\n[depth-first traversal](`m:digraph_utils#depth_first_traversal`) of the digraph,\ncollecting visited vertices in postorder. More precisely, the vertices visited\nwhile searching from an arbitrarily chosen vertex are collected in postorder,\nand all those collected vertices are placed before the subsequently visited\nvertices.","ref":"digraph_utils.html#postorder/1"},{"type":"function","title":"digraph_utils.preorder/1","doc":"Returns all vertices of digraph `Digraph`. The order is given by a\n[depth-first traversal](`m:digraph_utils#depth_first_traversal`) of the digraph,\ncollecting visited vertices in preorder.","ref":"digraph_utils.html#preorder/1"},{"type":"function","title":"digraph_utils.reachable/2","doc":"Returns an unsorted list of digraph vertices such that for each vertex in the\nlist, there is a [path](`m:digraph_utils#path`) in `Digraph` from some vertex of\n`Vertices` to the vertex. In particular, as paths can have length zero, the\nvertices of `Vertices` are included in the returned list.","ref":"digraph_utils.html#reachable/2"},{"type":"function","title":"digraph_utils.reachable_neighbours/2","doc":"Returns an unsorted list of digraph vertices such that for each vertex in the\nlist, there is a [path](`m:digraph_utils#path`) in `Digraph` of length one or\nmore from some vertex of `Vertices` to the vertex. As a consequence, only those\nvertices of `Vertices` that are included in some\n[cycle](`m:digraph_utils#cycle`) are returned.","ref":"digraph_utils.html#reachable_neighbours/2"},{"type":"function","title":"digraph_utils.reaching/2","doc":"Returns an unsorted list of digraph vertices such that for each vertex in the\nlist, there is a [path](`m:digraph_utils#path`) from the vertex to some vertex\nof `Vertices`. In particular, as paths can have length zero, the vertices of\n`Vertices` are included in the returned list.","ref":"digraph_utils.html#reaching/2"},{"type":"function","title":"digraph_utils.reaching_neighbours/2","doc":"Returns an unsorted list of digraph vertices such that for each vertex in the\nlist, there is a [path](`m:digraph_utils#path`) of length one or more from the\nvertex to some vertex of `Vertices`. Therefore only those vertices of `Vertices`\nthat are included in some [cycle](`m:digraph_utils#cycle`) are returned.","ref":"digraph_utils.html#reaching_neighbours/2"},{"type":"function","title":"digraph_utils.strong_components/1","doc":"Returns a list of\n[strongly connected components](`m:digraph_utils#strong_components`). Each\nstrongly component is represented by its vertices. The order of the vertices and\nthe order of the components are arbitrary. Each vertex of digraph `Digraph`\noccurs in exactly one strong component.","ref":"digraph_utils.html#strong_components/1"},{"type":"function","title":"digraph_utils.subgraph/2","doc":"","ref":"digraph_utils.html#subgraph/2"},{"type":"function","title":"digraph_utils.subgraph/3","doc":"Creates a maximal [subgraph](`m:digraph_utils#subgraph`) of `Digraph` having as\nvertices those vertices of `Digraph` that are mentioned in `Vertices`.\n\nIf the value of option `type` is `inherit`, which is the default, the type of\n`Digraph` is used for the subgraph as well. Otherwise the option value of `type`\nis used as argument to `digraph:new/1`.\n\nIf the value of option `keep_labels` is `true`, which is the default, the\n[labels](`m:digraph_utils#label`) of vertices and edges of `Digraph` are used\nfor the subgraph as well. If the value is `false`, default label `[]` is used\nfor the vertices and edges of the subgroup.\n\n[`subgraph(Digraph, Vertices)`](`subgraph/2`) is equivalent to\n[`subgraph(Digraph, Vertices, [])`](`subgraph/3`).\n\nIf any of the arguments are invalid, a `badarg` exception is raised.","ref":"digraph_utils.html#subgraph/3"},{"type":"function","title":"digraph_utils.topsort/1","doc":"Returns a [topological ordering](`m:digraph_utils#topsort`) of the vertices of\ndigraph `Digraph` if such an ordering exists, otherwise `false`. For each vertex\nin the returned list, no [out-neighbors](`m:digraph_utils#out_neighbour`) occur\nearlier in the list.","ref":"digraph_utils.html#topsort/1"},{"type":"module","title":"ets","doc":"Built-in term storage.\n\nThis module is an interface to the Erlang built-in term storage BIFs. These\nprovide the ability to store very large quantities of data in an Erlang runtime\nsystem, and to have constant access time to the data. (In the case of\n`ordered_set`, see below, access time is proportional to the logarithm of the\nnumber of stored objects.)\n\nData is organized as a set of dynamic tables, which can store tuples. Each table\nis created by a process. When the process terminates, the table is automatically\ndestroyed. Every table has access rights set at creation.\n\nTables are divided into four different types, `set`, `ordered_set`, `bag`, and\n`duplicate_bag`. A `set` or `ordered_set` table can only have one object\nassociated with each key. A `bag` or `duplicate_bag` table can have many objects\nassociated with each key.\n\nInsert and lookup times in tables of type `set` are constant, regardless of the\ntable size. For table types `bag` and `duplicate_bag` time is proportional to\nthe number of objects with the same key. Even seemingly unrelated keys may\ninflict linear search to be skipped past while looking for the key of interest\n(due to hash collision).\n\n> #### Warning {: .warning }\n>\n> For tables of type `bag` and `duplicate_bag`, avoid inserting an extensive\n> amount of objects with the same key. It will hurt insert and lookup\n> performance as well as real time characteristics of the runtime environment\n> (hash bucket linear search do not yield).\n\nThe `ordered_set` table type uses a binary search tree. Insert and lookup times\nare proportional to the logarithm of the number of objects in the table.\n\n[](){: #max_ets_tables }\n\n> #### Note {: .info }\n>\n> The number of tables stored at one Erlang node _used_ to be limited. This is\n> no longer the case (except by memory usage). The previous default limit was\n> about 1400 tables and could be increased by setting the environment variable\n> `ERL_MAX_ETS_TABLES` or the command line option\n> [`+e`](`e:erts:erl_cmd.md#%2Be`) before starting the Erlang runtime system.\n> This hard limit has been removed, but it is currently useful to set the\n> `ERL_MAX_ETS_TABLES` anyway. It should be set to an approximate of the maximum\n> amount of tables used since an internal table for named tables is sized using\n> this value. If large amounts of named tables are used and `ERL_MAX_ETS_TABLES`\n> hasn't been increased, the performance of named table lookup will degrade.\n\nNotice that there is no automatic garbage collection for tables. Even if there\nare no references to a table from any process, it is not automatically destroyed\nunless the owner process terminates. To destroy a table explicitly, use function\n`delete/1`. The default owner is the process that created the table. To transfer\ntable ownership at process termination, use option [`heir`](`m:ets#heir`) or\ncall `give_away/3`.\n\nSome implementation details:\n\n- In the current implementation, every object insert and look-up operation\n  results in a copy of the object.\n- `'$end_of_table'` is not to be used as a key, as this atom is used to mark the\n  end of the table when using functions `first/1` and `next/2`.\n\nNotice the subtle difference between _matching_ and _comparing equal_, which is\ndemonstrated by table types `set` and `ordered_set`:\n\n- Two Erlang terms `match` if they are of the same type and have the same value,\n  so that `1` matches `1`, but not `1.0` (as `1.0` is a `t:float/0` and not an\n  `t:integer/0`).\n- Two Erlang terms _compare equal_ if they either are of the same type and\n  value, or if both are numeric types and extend to the same value, so that `1`\n  compares equal to both `1` and `1.0`.\n- The `ordered_set` works on the _Erlang term order_ and no defined order exists\n  between an `t:integer/0` and a `t:float/0` that extends to the same value.\n  Hence the key `1` and the key `1.0` are regarded as equal in an `ordered_set`\n  table.\n\n[](){: #ets_failures }","ref":"ets.html"},{"type":"module","title":"Failures - ets","doc":"Functions in this module fail by raising an error exception with error reason:\n\n- **`badarg`** - If any argument has the wrong format.\n\n- **`badarg`** - If the table identifier is invalid.\n\n- **`badarg`** - If the operation is denied because of table access rights\n  ([protected](`m:ets#protected`) or [private](`m:ets#private`)).\n\n- **`system_limit`** - Modification of a value causes it to not be representable\n  internally in the VM. For example, incrementation of a counter past the\n  largest integer representable.\n\n- **`system_limit`** - If a match specification passed as argument has excessive\n  nesting which causes scheduler stack exhaustion for the scheduler that the\n  calling process is executing on.\n  [Scheduler stack size](`e:erts:erl_cmd.md#sched_thread_stack_size`) can be\n  configured when starting the runtime system.\n\n[](){: #concurrency }","ref":"ets.html#module-failures"},{"type":"module","title":"Concurrency - ets","doc":"This module provides some limited support for concurrent access. All updates to\nsingle objects are guaranteed to be both _atomic_ and _isolated_. This means\nthat an updating operation to a single object either succeeds or fails\ncompletely without any effect (atomicity) and that no intermediate results of\nthe update can be seen by other processes (isolation). Some functions that\nupdate many objects state that they even guarantee atomicity and isolation for\nthe entire operation. In database terms the isolation level can be seen as\n\"serializable\", as if all isolated operations are carried out serially, one\nafter the other in a strict order.\n\n[](){: #traversal }","ref":"ets.html#module-concurrency"},{"type":"module","title":"Table traversal - ets","doc":"There are different ways to traverse through the objects of a table.\n\n- _Single-step_ traversal one key at at time, using `first/1`, `next/2`,\n  `last/1` and `prev/2`.\n- _Single-step_ traversal one key at at time, but using `first_lookup/1`,\n  `next_lookup/2`, `last_lookup/1` and `prev_lookup/2`. This is more efficient\n  when you also need to lookup the objects for the keys.\n- Search with simple _match patterns_, using [`match/1/2/3`](`match/1`),\n  `match_delete/2` and [`match_object/1/2/3`](`match_object/1`).\n- Search with more powerful _match specifications_, using\n  [`select/1/2/3`](`select/1`), `select_count/2`, `select_delete/2`,\n  `select_replace/2` and [`select_reverse/1/2/3`](`select_reverse/1`).\n- _Table conversions_, using [`tab2file/2/3`](`tab2file/2`) and `tab2list/1`.\n\nNo table traversal will guarantee a consistent snapshot of the entire table if\nthe table is also updated by concurrent processes during the traversal. The\nresult of each concurrently updated object may be seen (or not) depending on if\nit has happened when the traversal visits that part of the table. The only way\nto guarantee a full consistent table snapshot (if you really need that) is to\ndisallow concurrent updates during the entire traversal.\n\nMoreover, traversals not done in a _safe_ way, on tables where keys are inserted\nor deleted during the traversal, may yield the following undesired effects:\n\n- Any key may be missed.\n- Any key may be found more than once.\n- The traversal may fail with `badarg` exception if keys are deleted.\n\nA table traversal is _safe_ if either\n\n- the table is of type `ordered_set`.\n- the entire table traversal is done within one ETS function call.\n- function `safe_fixtable/2` is used to keep the table fixated during the entire\n  traversal.\n\n> #### Note {: .info }\n>\n> Even though the access of a single object is always guaranteed to be\n> [atomic and isolated](`m:ets#module-concurrency`), each traversal through a table to\n> find the next key is not done with such guarantees. This is often not a\n> problem, but may cause rare subtle \"unexpected\" effects if a concurrent\n> process inserts objects during a traversal. For example, consider one process\n> doing\n>\n> ```erlang\n> ets:new(t, [ordered_set, named_table]),\n> ets:insert(t, {1}),\n> ets:insert(t, {2}),\n> ets:insert(t, {3}),\n> ```\n>\n> A concurrent call to `ets:first(t)`, done by another process, may then in rare\n> cases return `2` even though `2` has never existed in the table ordered as the\n> first key. In the same way, a concurrent call to `ets:next(t, 1)` may return\n> `3` even though `3` never existed in the table ordered directly after `1`.\n>\n> Effects like this are improbable but possible. The probability will further be\n> reduced (if not vanish) if table option\n> [`write_concurrency`](`m:ets#new_2_write_concurrency`) is not enabled. This\n> can also only be a potential concern for `ordered_set` where the traversal\n> order is defined.\n\nTraversals using `match` and `select` functions may not need to scan the entire\ntable depending on how the key is specified. A match pattern with a _fully bound\nkey_ (without any match variables) will optimize the operation to a single key\nlookup without any table traversal at all. For `ordered_set` a _partially bound\nkey_ will limit the traversal to only scan a subset of the table based on term\norder. A partially bound key is either a list or a tuple with a prefix that is\nfully bound. Example:\n\n```erlang\n1> T = ets:new(t,[ordered_set]), ets:insert(T, {\"555-1234\", \"John Smith\"}).\ntrue\n2> %% Efficient search of all with area code 555\n2> ets:match(T,{[$5,$5,$5,$- |'$1'],'$2'}).\n[[\"1234\",\"John Smith\"]]\n```\n\n[](){: #match_spec }","ref":"ets.html#module-table-traversal"},{"type":"module","title":"Match Specifications - ets","doc":"Some of the functions use a _match specification_, `match_spec`. For a brief\nexplanation, see `select/2`. For a detailed description, see section\n[Match Specifications in Erlang](`e:erts:match_spec.md`) in ERTS User's Guide.\n\nA match specifications with excessive nesting will cause a\n[`system_limit`](`m:ets#ets_failures`) error exception to be raised.","ref":"ets.html#module-match-specifications"},{"type":"function","title":"ets.all/0","doc":"Returns a list of all tables at the node. Named tables are specified by their\nnames, unnamed tables are specified by their table identifiers.\n\nThere is no guarantee of consistency in the returned list. Tables created or\ndeleted by other processes \"during\" the `ets:all()` call either are or are not\nincluded in the list. Only tables created/deleted _before_ `ets:all()` is called\nare guaranteed to be included/excluded.","ref":"ets.html#all/0"},{"type":"type","title":"ets.comp_match_spec/0","doc":"","ref":"ets.html#t:comp_match_spec/0"},{"type":"opaque","title":"ets.compiled_match_spec/0","doc":"A compiled match specification.","ref":"ets.html#t:compiled_match_spec/0"},{"type":"type","title":"ets.continuation/0","doc":"Opaque continuation used by [`select/1,3`](`select/1`),\n[`select_reverse/1,3`](`select_reverse/1`), [`match/1,3`](`match/1`), and\n[`match_object/1,3`](`match_object/1`).","ref":"ets.html#t:continuation/0"},{"type":"function","title":"ets.delete/1","doc":"Deletes the entire table `Table`.","ref":"ets.html#delete/1"},{"type":"function","title":"ets.delete/2","doc":"Deletes all objects with key `Key` from table `Table`. This function succeeds\neven if no objects with key `Key` exist.","ref":"ets.html#delete/2"},{"type":"function","title":"ets.delete_all_objects/1","doc":"Delete all objects in the ETS table `Table`. The operation is guaranteed to be\n[atomic and isolated](`m:ets#module-concurrency`).","ref":"ets.html#delete_all_objects/1"},{"type":"function","title":"ets.delete_object/2","doc":"Delete the exact object `Object` from the ETS table, leaving objects with the\nsame key but other differences (useful for type `bag`). In a `duplicate_bag`\ntable, all instances of the object are deleted.","ref":"ets.html#delete_object/2"},{"type":"function","title":"ets.file2tab/1","doc":"Reads a file produced by `tab2file/2` or `tab2file/3` and creates the\ncorresponding table `Table`.\n\nEquivalent to [`file2tab(Filename, [])`](`file2tab/2`).","ref":"ets.html#file2tab/1"},{"type":"function","title":"ets.file2tab/2","doc":"Reads a file produced by `tab2file/2` or `tab2file/3` and creates the\ncorresponding table `Table`.\n\nThe only supported option is `{verify,boolean()}`. If verification is turned on\n(by specifying `{verify,true}`), the function uses whatever information is\npresent in the file to assert that the information is not damaged. How this is\ndone depends on which `extended_info` was written using `tab2file/3`.\n\nIf no `extended_info` is present in the file and `{verify,true}` is specified,\nthe number of objects written is compared to the size of the original table when\nthe dump was started. This can make verification fail if the table was `public`\nand objects were added or removed while the table was dumped to file. To avoid\nthis problem, either do not verify files dumped while updated simultaneously or\nuse option `{extended_info, [object_count]}` to `tab2file/3`, which extends the\ninformation in the file with the number of objects written.\n\nIf verification is turned on and the file was written with option\n`{extended_info, [md5sum]}`, reading the file is slower and consumes radically\nmore CPU time than otherwise.\n\n`{verify,false}` is the default.","ref":"ets.html#file2tab/2"},{"type":"function","title":"ets.first/1","doc":"Returns the first key `Key` in table `Table`. For an `ordered_set` table, the\nfirst key in Erlang term order is returned. For other table types, the first key\naccording to the internal order of the table is returned. If the table is empty,\n`'$end_of_table'` is returned.\n\nTo find subsequent keys in the table, use `next/2`.","ref":"ets.html#first/1"},{"type":"function","title":"ets.first_lookup/1","doc":"Similar to `first/1` except that it returns the object(s) along with the key\nstored in the table. This is equivalent to doing `first/1` followed by a\n`lookup/2`. If the table is empty, `'$end_of_table'` is returned.\n\nTo find subsequent objects in the table, use `next_lookup/2`.","ref":"ets.html#first_lookup/1"},{"type":"function","title":"ets.foldl/3","doc":"`Acc0` is returned if the table is empty. This function is similar to\n`lists:foldl/3`. The table elements are traversed in an unspecified order,\nexcept for `ordered_set` tables, where they are traversed first to last.\n\nIf `Function` inserts objects into the table, or another process inserts objects\ninto the table, those objects _can_ (depending on key ordering) be included in\nthe traversal.","ref":"ets.html#foldl/3"},{"type":"function","title":"ets.foldr/3","doc":"`Acc0` is returned if the table is empty. This function is similar to\n`lists:foldr/3`. The table elements are traversed in an unspecified order,\nexcept for `ordered_set` tables, where they are traversed last to first.\n\nIf `Function` inserts objects into the table, or another process inserts objects\ninto the table, those objects _can_ (depending on key ordering) be included in\nthe traversal.","ref":"ets.html#foldr/3"},{"type":"function","title":"ets.from_dets/2","doc":"Fills an already created ETS table with the objects in the already opened Dets\ntable `DetsTab`. Existing objects in the ETS table are kept unless overwritten.\n\nIf any of the tables does not exist or the Dets table is not open, a `badarg`\nexception is raised.","ref":"ets.html#from_dets/2"},{"type":"function","title":"ets.fun2ms/1","doc":"Pseudo function that by a `parse_transform` translates `LiteralFun` typed as\nparameter in the function call to a [match specification](`m:ets#match_spec`).\nWith \"literal\" is meant that the fun must textually be written as the parameter\nof the function, it cannot be held in a variable that in turn is passed to the\nfunction.\n\nThe parse transform is provided in the `ms_transform` module and the source\n_must_ include file `ms_transform.hrl` in STDLIB for this pseudo function to\nwork. Failing to include the hrl file in the source results in a runtime error,\nnot a compile time error. The include file is easiest included by adding line\n`-include_lib(\"stdlib/include/ms_transform.hrl\").` to the source file.\n\nThe fun is very restricted, it can take only a single parameter (the object to\nmatch): a sole variable or a tuple. It must use the `is_` guard tests. Language\nconstructs that have no representation in a match specification (`if`, `case`,\n`receive`, and so on) are not allowed.\n\nThe return value is the resulting match specification.\n\n_Example:_\n\n```erlang\n1> ets:fun2ms(fun({M,N}) when N > 3 -> M end).\n[{{'$1','$2'},[{'>','$2',3}],['$1']}]\n```\n\nVariables from the environment can be imported, so that the following works:\n\n```erlang\n2> X=3.\n3\n3> ets:fun2ms(fun({M,N}) when N > X -> M end).\n[{{'$1','$2'},[{'>','$2',{const,3}}],['$1']}]\n```\n\nThe imported variables are replaced by match specification `const` expressions,\nwhich is consistent with the static scoping for Erlang funs. However, local or\nglobal function calls cannot be in the guard or body of the fun. Calls to\nbuilt-in match specification functions is of course allowed:\n\n```erlang\n4> ets:fun2ms(fun({M,N}) when N > X, my_fun(M) -> M end).\nError: fun containing local Erlang function calls\n('my_fun' called in guard) cannot be translated into match_spec\n{error,transform_error}\n5> ets:fun2ms(fun({M,N}) when N > X, is_atom(M) -> M end).\n[{{'$1','$2'},[{'>','$2',{const,3}},{is_atom,'$1'}],['$1']}]\n```\n\nAs shown by the example, the function can be called from the shell also. The fun\nmust be literally in the call when used from the shell as well.\n\n> #### Warning {: .warning }\n>\n> If the `parse_transform` is not applied to a module that calls this pseudo\n> function, the call fails in runtime (with a `badarg`). The `ets` module\n> exports a function with this name, but it is never to be called except when\n> using the function in the shell. If the `parse_transform` is properly applied\n> by including header file `ms_transform.hrl`, compiled code never calls the\n> function, but the function call is replaced by a literal match specification.\n\nFor more information, see [`ms_transform`](`m:ms_transform`).","ref":"ets.html#fun2ms/1"},{"type":"function","title":"ets.give_away/3","doc":"Make process `Pid` the new owner of table `Table`. If successful, message\n`{'ETS-TRANSFER',Table,FromPid,GiftData}` is sent to the new owner.\n\nThe process `Pid` must be alive, local, and not already the owner of the table.\nThe calling process must be the table owner.\n\nNotice that this function does not affect option [`heir`](`m:ets#heir`) of the\ntable. A table owner can, for example, set `heir` to itself, give the table\naway, and then get it back if the receiver terminates.","ref":"ets.html#give_away/3"},{"type":"function","title":"ets.i/0","doc":"Displays information about all ETS tables on a terminal.","ref":"ets.html#i/0"},{"type":"function","title":"ets.i/1","doc":"Browses table `Table` on a terminal.","ref":"ets.html#i/1"},{"type":"function","title":"ets.info/1","doc":"Returns information about table `Table` as a list of tuples. If `Table` has the\ncorrect type for a table identifier, but does not refer to an existing ETS\ntable, `undefined` is returned. If `Table` is not of the correct type, a\n`badarg` exception is raised.\n\n- **`{compressed, boolean()}`** - Indicates if the table is compressed.\n\n- **`{decentralized_counters, boolean()}`** - Indicates whether the table uses\n  `decentralized_counters`.\n\n- **`{heir, pid() | none}`** - The pid of the heir of the table, or `none` if no\n  heir is set.\n\n- **`{id,`[ `tid()`](`t:tid/0`)`}`** - The table identifier.\n\n- **`{keypos, integer() >= 1}`** - The key position.\n\n- **`{memory, integer() >= 0}`** - The number of words allocated to the table.\n\n- **`{name, atom()}`** - The table name.\n\n- **`{named_table, boolean()}`** - Indicates if the table is named.\n\n- **`{node, node()}`** - The node where the table is stored. This field is no\n  longer meaningful, as tables cannot be accessed from other nodes.\n\n- **`{owner, pid()}`** - The pid of the owner of the table.\n\n- **`{protection,` [`access()`](`t:table_access/0`)`}`** - The table access\n  rights.\n\n- **`{size, integer() >= 0}`** - The number of objects inserted in the table.\n\n- **`{type,` [`type()`](`t:table_type/0`)`}`** - The table type.\n\n- **`{read_concurrency, boolean()}`** - Indicates whether the table uses\n  `read_concurrency` or not.\n\n- **`{write_concurrency, WriteConcurrencyAlternative}`** - Indicates which\n  `write_concurrency` option the table uses.\n\n> #### Note {: .info }\n>\n> The execution time of this function is affected by the\n> [`decentralized_counters`](`m:ets#new_2_decentralized_counters`) table option.\n> The execution time is much longer when the `decentralized_counters` option is\n> set to `true` than when the `decentralized_counters` option is set to `false`.","ref":"ets.html#info/1"},{"type":"function","title":"ets.info/2","doc":"Returns the information associated with `Item` for table `Table`, or returns\n`undefined` if `Table` does not refer an existing ETS table. If `Table` is not\nof the correct type, or if `Item` is not one of the allowed values, a `badarg`\nexception is raised.\n\nIn addition to the `{Item,Value}` pairs defined for `info/1`, the following\nitems are allowed:\n\n- `Item=binary, Value=BinInfo`\n\n  `BinInfo` is a list containing miscellaneous information about binaries kept\n  by the table. This `Item` can be changed or removed without prior notice. In\n  the current implementation `BinInfo` is a list of tuples\n  `{BinaryId,BinarySize,BinaryRefcCount}`.\n\n- `Item=fixed, Value=boolean()`\n\n  Indicates if the table is fixed by any process.\n\n- [](){: #info_2_safe_fixed_monotonic_time }\n\n  `Item=safe_fixed|safe_fixed_monotonic_time, Value={FixationTime,Info}|false`\n\n  If the table is fixed using `safe_fixtable/2`, the call returns a tuple where\n  `FixationTime` is the last time when the table changed from unfixed to fixed.\n\n  The format and value of `FixationTime` depends on `Item`:\n\n  - **`safe_fixed`** - `FixationTime` corresponds to the result returned by\n    `erlang:timestamp/0` at the time of fixation. Notice that when the system\n    uses single or multi\n    [time warp modes](`e:erts:time_correction.md#time-warp-modes`) this can\n    produce strange results, as the use of `safe_fixed` is not\n    [time warp safe](`e:erts:time_correction.md#time-warp-safe-code`). Time warp\n    safe code must use `safe_fixed_monotonic_time` instead.\n\n  - **`safe_fixed_monotonic_time`** - `FixationTime` corresponds to the result\n    returned by `erlang:monotonic_time/0` at the time of fixation. The use of\n    `safe_fixed_monotonic_time` is\n    [time warp safe](`e:erts:time_correction.md#time-warp-safe-code`).\n\n  `Info` is a possibly empty lists of tuples `{Pid,RefCount}`, one tuple for\n  every process the table is fixed by now. `RefCount` is the value of the\n  reference counter and it keeps track of how many times the table has been\n  fixed by the process.\n\n  Table fixations are not limited to `safe_fixtable/2`. Temporary fixations may\n  also be done by for example [traversing functions](`m:ets#traversal`) like\n  `select` and `match`. Such table fixations are automatically released before\n  the corresponding functions returns, but they may be seen by a concurrent call\n  to `ets:info(T,safe_fixed|safe_fixed_monotonic_time)`.\n\n  If the table is not fixed at all, the call returns `false`.\n\n- `Item=stats, Value=tuple()`\n\n  Returns internal statistics about tables on an internal format used by OTP\n  test suites. Not for production use.\n\n> #### Note {: .info }\n>\n> The execution time of this function is affected by the\n> [`decentralized_counters`](`m:ets#new_2_decentralized_counters`) table option\n> when the second argument of the function is `size` or `memory`. The execution\n> time is much longer when the `decentralized_counters` option is set to `true`\n> than when the `decentralized_counters` option is set to `false`.","ref":"ets.html#info/2"},{"type":"function","title":"ets.init_table/2","doc":"Replaces the existing objects of table `Table` with objects created by calling\nthe input function `InitFun`, see below. This function is provided for\ncompatibility with the `dets` module, it is not more efficient than filling a\ntable by using `insert/2`.\n\nWhen called with argument `read`, the function `InitFun` is assumed to return\n`end_of_input` when there is no more input, or `{Objects, Fun}`, where `Objects`\nis a list of objects and `Fun` is a new input function. Any other value `Value`\nis returned as an error `{error, {init_fun, Value}}`. Each input function is\ncalled exactly once, and if an error occur, the last function is called with\nargument `close`, the reply of which is ignored.\n\nIf the table type is `set` and more than one object exists with a given key, one\nof the objects is chosen. This is not necessarily the last object with the given\nkey in the sequence of objects returned by the input functions. This holds also\nfor duplicated objects stored in tables of type `bag`.","ref":"ets.html#init_table/2"},{"type":"function","title":"ets.insert/2","doc":"Inserts the object or all of the objects in list `ObjectOrObjects` into table\n`Table`.\n\n- If the table type is `set` and the key of the inserted objects _matches_ the\n  key of any object in the table, the old object is replaced.\n- If the table type is `ordered_set` and the key of the inserted object\n  _compares equal_ to the key of any object in the table, the old object is\n  replaced.\n- If the table type is `bag` and the object _matches_ any whole object in the\n  table, the object is not inserted.\n- If the list contains more than one object with _matching_ keys and the table\n  type is `set`, one is inserted, which one is not defined. The same holds for\n  table type `ordered_set` if the keys _compare equal_.\n\nThe entire operation is guaranteed to be\n[atomic and isolated](`m:ets#module-concurrency`), even when a list of objects is\ninserted.\n\n[](){: #insert_list_order }\n\nFor `bag` and `duplicate_bag`, objects in the list with identical keys will be\ninserted in list order (from head to tail). That is, a subsequent call to\n[`lookup(T,Key)`](`lookup/2`) will return them in that inserted order.\n\n> #### Note {: .info }\n>\n> For `bag` the insertion order of indentical keys described above was\n> accidentally reverted in OTP 23.0 and later fixed in OTP 25.3. That is, from\n> OTP 23.0 up until OTP 25.3 the objects in a list are inserted in reverse order\n> (from tail to head).\n>\n> For `duplicate_bag` the same faulty reverse insertion exist from OTP 23.0\n> until OTP 25.3. However, it is unpredictable and may or may not happen. A\n> longer list will increase the probabiliy of the insertion being done in\n> reverse.","ref":"ets.html#insert/2"},{"type":"function","title":"ets.insert_new/2","doc":"Same as `insert/2` except that instead of overwriting objects with the same key\n(for `set` or `ordered_set`) or adding more objects with keys already existing\nin the table (for `bag` and `duplicate_bag`), `false` is returned.\n\nIf `ObjectOrObjects` is a list, the function checks _every_ key before inserting\nanything. Nothing is inserted unless _all_ keys present in the list are absent\nfrom the table. Like [`insert/2`](`insert/2`), the entire operation is\nguaranteed to be [atomic and isolated](`m:ets#module-concurrency`).","ref":"ets.html#insert_new/2"},{"type":"function","title":"ets.is_compiled_ms/1","doc":"Checks if a term represent a valid compiled\n[match specification](`m:ets#match_spec`). A compiled match specification is\nonly valid on the Erlang node where it was compiled by calling\n`match_spec_compile/1`.\n\n> #### Note {: .info }\n>\n> Before STDLIB 3.4 (OTP 20.0) compiled match specifications did not have an\n> external representation. If passed through\n> [`binary_to_term(term_to_binary(CMS))`](`binary_to_term/1`) or sent to another\n> node and back, the result was always an empty binary `<<>>`.\n>\n> After STDLIB 3.4 (OTP 20.0) compiled match specifications have an external\n> representation as a node specific reference to the original compiled match\n> specification. If passed through\n> [`binary_to_term(term_to_binary(CMS))`](`binary_to_term/1`) or sent to another\n> node and back, the result _may or may not_ be a valid compiled match\n> specification depending on if the original compiled match specification was\n> still alive.","ref":"ets.html#is_compiled_ms/1"},{"type":"function","title":"ets.last/1","doc":"Returns the last key `Key` according to Erlang term order in table `Table` of\ntype `ordered_set`. For other table types, the function is synonymous to\n`first/1`. If the table is empty, `'$end_of_table'` is returned.\n\nTo find preceding keys in the table, use `prev/2`.","ref":"ets.html#last/1"},{"type":"function","title":"ets.last_lookup/1","doc":"Similar to `last/1` except that it returns the object(s) along with the key\nstored in the table. This is equivalent to doing `last/1` followed by a\n`lookup/2`. If the table is empty, `'$end_of_table'` is returned.\n\nTo find preceding objects in the table, use `prev_lookup/2`.","ref":"ets.html#last_lookup/1"},{"type":"function","title":"ets.lookup/2","doc":"Returns a list of all objects with key `Key` in table `Table`.\n\n- For tables of type `set`, `bag`, or `duplicate_bag`, an object is returned\n  only if the specified key _matches_ the key of the object in the table.\n- For tables of type `ordered_set`, an object is returned if the specified key\n  _compares equal_ to the key of an object in the table.\n\nThe difference is the same as between `=:=` and `==`.\n\nAs an example, one can insert an object with `t:integer/0` `1` as a key in an\n`ordered_set` and get the object returned as a result of doing a\n[`lookup/2`](`lookup/2`) with `t:float/0` `1.0` as the key to search for.\n\nFor tables of type `set` or `ordered_set`, the function returns either the empty\nlist or a list with one element, as there cannot be more than one object with\nthe same key. For tables of type `bag` or `duplicate_bag`, the function returns\na list of arbitrary length.\n\nNotice that the sequential order of object insertions is preserved; the first\nobject inserted with the specified key is the first in the resulting list, and\nso on. See also the note about\n[list insertion order](`m:ets#insert_list_order`).","ref":"ets.html#lookup/2"},{"type":"function","title":"ets.lookup_element/3","doc":"For a table `Table` of type `set` or `ordered_set`, the function returns the\n`Pos`:th element of the object with key `Key`.\n\nFor tables of type `bag` or `duplicate_bag`, the functions returns a list with\nthe `Pos`:th element of every object with key `Key`.\n\nIf no object with key `Key` exists, the function exits with reason `badarg`.\n\nIf `Pos` is larger than the size of the tuple, the function exits with reason\n`badarg`.\n\nThe difference between `set`, `bag`, and `duplicate_bag` on one hand, and\n`ordered_set` on the other, regarding the fact that `ordered_set` view keys as\nequal when they _compare equal_ whereas the other table types regard them equal\nonly when they _match_, holds for [`lookup_element/3`](`lookup_element/3`).","ref":"ets.html#lookup_element/3"},{"type":"function","title":"ets.lookup_element/4","doc":"For a table `Table` of type `set` or `ordered_set`, the function returns the\n`Pos`:th element of the object with key `Key`.\n\nFor tables of type `bag` or `duplicate_bag`, the functions returns a list with\nthe `Pos`:th element of every object with key `Key`.\n\nIf no object with key `Key` exists, the function returns `Default`.\n\nIf `Pos` is larger than the size of any tuple with a matching key, the function\nexits with reason `badarg`.\n\nThe difference between `set`, `bag`, and `duplicate_bag` on one hand, and\n`ordered_set` on the other, regarding the fact that `ordered_set` view keys as\nequal when they _compare equal_ whereas the other table types regard them equal\nonly when they _match_, holds for [`lookup_element/4`](`lookup_element/4`).","ref":"ets.html#lookup_element/4"},{"type":"function","title":"ets.match/1","doc":"Continues a match started with `match/3`. The next chunk of the size specified\nin the initial [`match/3`](`match/3`) call is returned together with a new\n`Continuation`, which can be used in subsequent calls to this function.\n\nWhen there are no more objects in the table, `'$end_of_table'` is returned.","ref":"ets.html#match/1"},{"type":"function","title":"ets.match/2","doc":"Matches the objects in table `Table` against pattern `Pattern`.\n\nA pattern is a term that can contain:\n\n- Bound parts (Erlang terms)\n- `'_'` that matches any Erlang term\n- Pattern variables `'$N'`, where `N`=0,1,...\n\nThe function returns a list with one element for each matching object, where\neach element is an ordered list of pattern variable bindings, for example:\n\n```erlang\n6> ets:match(T, '$1'). % Matches every object in table\n[[{rufsen,dog,7}],[{brunte,horse,5}],[{ludde,dog,5}]]\n7> ets:match(T, {'_',dog,'$1'}).\n[[7],[5]]\n8> ets:match(T, {'_',cow,'$1'}).\n[]\n```\n\nIf the key is specified in the pattern, the match is very efficient. If the key\nis not specified, that is, if it is a variable or an underscore, the entire\ntable must be searched. The search time can be substantial if the table is very\nlarge.\n\nFor tables of type `ordered_set`, the result is in the same order as in a\n`first`/`next` traversal.","ref":"ets.html#match/2"},{"type":"function","title":"ets.match/3","doc":"Works like `match/2`, but returns only a limited (`Limit`) number of matching\nobjects. Term `Continuation` can then be used in subsequent calls to `match/1`\nto get the next chunk of matching objects. This is a space-efficient way to work\non objects in a table, which is faster than traversing the table object by\nobject using `first/1` and `next/2`.\n\nIf the table is empty, `'$end_of_table'` is returned.\n\nUse `safe_fixtable/2` to guarantee [safe traversal](`m:ets#traversal`) for\nsubsequent calls to `match/1`.","ref":"ets.html#match/3"},{"type":"function","title":"ets.match_delete/2","doc":"Deletes all objects that match pattern `Pattern` from table `Table`. For a\ndescription of patterns, see `match/2`.","ref":"ets.html#match_delete/2"},{"type":"function","title":"ets.match_object/1","doc":"Continues a match started with `match_object/3`. The next chunk of the size\nspecified in the initial [`match_object/3`](`match_object/3`) call is returned\ntogether with a new `Continuation`, which can be used in subsequent calls to\nthis function.\n\nWhen there are no more objects in the table, `'$end_of_table'` is returned.","ref":"ets.html#match_object/1"},{"type":"function","title":"ets.match_object/2","doc":"Matches the objects in table `Table` against pattern `Pattern`. For a\ndescription of patterns, see `match/2`. The function returns a list of all\nobjects that match the pattern.\n\nIf the key is specified in the pattern, the match is very efficient. If the key\nis not specified, that is, if it is a variable or an underscore, the entire\ntable must be searched. The search time can be substantial if the table is very\nlarge.\n\nFor tables of type `ordered_set`, the result is in the same order as in a\n`first`/`next` traversal.","ref":"ets.html#match_object/2"},{"type":"function","title":"ets.match_object/3","doc":"Works like `match_object/2`, but only returns a limited (`Limit`) number of\nmatching objects. Term `Continuation` can then be used in subsequent calls to\n`match_object/1` to get the next chunk of matching objects. This is a\nspace-efficient way to work on objects in a table, which is faster than\ntraversing the table object by object using `first/1` and `next/2`.\n\nIf the table is empty, `'$end_of_table'` is returned.\n\nUse `safe_fixtable/2` to guarantee [safe traversal](`m:ets#traversal`) for\nsubsequent calls to `match_object/1`.","ref":"ets.html#match_object/3"},{"type":"type","title":"ets.match_pattern/0","doc":"","ref":"ets.html#t:match_pattern/0"},{"type":"type","title":"ets.match_spec/0","doc":"A match specification, see [Match Specifications](`m:ets#match_spec`).","ref":"ets.html#t:match_spec/0"},{"type":"function","title":"ets.match_spec_compile/1","doc":"Transforms a [match specification](`m:ets#match_spec`) into an internal\nrepresentation that can be used in subsequent calls to `match_spec_run/2`. The\ninternal representation is opaque. To check the validity of a compiled match\nspecification, use `is_compiled_ms/1`.\n\nIf term `MatchSpec` does not represent a valid match specification, a `badarg`\nexception is raised.\n\n> #### Note {: .info }\n>\n> This function has limited use in normal code. It is used by the `m:dets`\n> module to perform the `dets:select/1` operations.","ref":"ets.html#match_spec_compile/1"},{"type":"function","title":"ets.match_spec_run/2","doc":"Executes the matching specified in a compiled\n[match specification](`m:ets#match_spec`) on a list of terms. Term\n`CompiledMatchSpec` is to be the result of a call to `match_spec_compile/1` and\nis hence the internal representation of the match specification one wants to\nuse.\n\nThe matching is executed on each element in `List` and the function returns a\nlist containing all results. If an element in `List` does not match, nothing is\nreturned for that element. The length of the result list is therefore equal or\nless than the length of parameter `List`.\n\n_Example:_\n\nThe following two calls give the same result (but certainly not the same\nexecution time):\n\n```erlang\nTable = ets:new...\nMatchSpec = ...\n% The following call...\nets:match_spec_run(ets:tab2list(Table),\n                   ets:match_spec_compile(MatchSpec)),\n% ...gives the same result as the more common (and more efficient)\nets:select(Table, MatchSpec),\n```\n\n> #### Note {: .info }\n>\n> This function has limited use in normal code. It is used by the `m:dets`\n> module to perform the `dets:select/1` operations and by Mnesia during\n> transactions.","ref":"ets.html#match_spec_run/2"},{"type":"function","title":"ets.member/2","doc":"Works like `lookup/2`, but does not return the objects. Returns `true` if one or\nmore elements in the table has key `Key`, otherwise `false`.","ref":"ets.html#member/2"},{"type":"function","title":"ets.new/2","doc":"Creates a new table and returns a table identifier that can be used in\nsubsequent operations. The table identifier can be sent to other processes so\nthat a table can be shared between different processes within a node.\n\nParameter `Options` is a list of options that specifies table type, access\nrights, key position, and whether the table is named. Default values are used\nfor omitted options. This means that not specifying any options (`[]`) is the\nsame as specifying\n`[set, protected, {keypos,1}, {heir,none}, {write_concurrency,false}, {read_concurrency,false}, {decentralized_counters,false}]`.\n\n- **`set`** - The table is a `set` table: one key, one object, no order among\n  objects. This is the default table type.\n\n- **`ordered_set`** - The table is a `ordered_set` table: one key, one object,\n  ordered in Erlang term order, which is the order implied by the  \n  operators. Tables of this type have a somewhat different behavior in some\n  situations than tables of other types. Most notably, the `ordered_set` tables\n  regard keys as equal when they _compare equal_, not only when they match. This\n  means that to an `ordered_set` table, `t:integer/0` `1` and `t:float/0` `1.0`\n  are regarded as equal. This also means that the key used to lookup an element\n  does not necessarily _match_ the key in the returned elements, if\n  `t:float/0`'s and `t:integer/0`'s are mixed in keys of a table.\n\n- **`bag`** - The table is a `bag` table, which can have many objects, but only\n  one instance of each object, per key.\n\n- **`duplicate_bag`** - The table is a `duplicate_bag` table, which can have\n  many objects, including multiple copies of the same object, per key.\n\n- **`public`** - Any process can read or write to the table.\n\n  [](){: #protected }\n\n- **`protected`** - The owner process can read and write to the table. Other\n  processes can only read the table. This is the default setting for the access\n  rights.\n\n  [](){: #private }\n\n- **`private`** - Only the owner process can read or write to the table.\n\n- **`named_table`** - If this option is present, the table is registered under\n  its `Name` which can then be used instead of the table identifier in\n  subsequent operations.\n\n  The function will also return the `Name` instead of the table identifier. To\n  get the table identifier of a named table, use `whereis/1`.\n\n- **`{keypos,Pos}`** - Specifies which element in the stored tuples to use as\n  key. By default, it is the first element, that is, `Pos=1`. However, this is\n  not always appropriate. In particular, we do not want the first element to be\n  the key if we want to store Erlang records in a table.\n\n  Notice that any tuple stored in the table must have at least `Pos` number of\n  elements.\n\n  [](){: #heir }\n\n- **`{heir,Pid,HeirData} | {heir,none}`** - Set a process as heir. The heir\n  inherits the table if the owner terminates. Message\n  `{'ETS-TRANSFER',tid(),FromPid,HeirData}` is sent to the heir when that\n  occurs. The heir must be a local process. Default heir is `none`, which\n  destroys the table when the owner terminates.\n\n  [](){: #new_2_write_concurrency }\n\n- **`{write_concurrency,WriteConcurrencyAlternative}`** - Performance tuning.\n  Defaults to `false`, in which case an operation that mutates (writes to) the\n  table obtains exclusive access, blocking any concurrent access of the same\n  table until finished. If set to `true`, the table is optimized for concurrent\n  write access. Different objects of the same table can be mutated (and read) by\n  concurrent processes. This is achieved to some degree at the expense of memory\n  consumption and the performance of sequential access and concurrent reading.\n\n  The `auto` alternative for the `write_concurrency` option is similar to the\n  `true` option but automatically adjusts the synchronization granularity during\n  runtime depending on how the table is used. This is the recommended\n  `write_concurrency` option when using Erlang/OTP 25 and above as it performs\n  well in most scenarios.\n\n  The `write_concurrency` option can be combined with the options\n  [`read_concurrency`](`m:ets#new_2_read_concurrency`) and\n  [`decentralized_counters`](`m:ets#new_2_decentralized_counters`). You\n  typically want to combine `write_concurrency` with `read_concurrency` when\n  large concurrent read bursts and large concurrent write bursts are common; for\n  more information, see option\n  [`read_concurrency`](`m:ets#new_2_read_concurrency`). It is almost always a\n  good idea to combine the `write_concurrency` option with the\n  [`decentralized_counters`](`m:ets#new_2_decentralized_counters`) option.\n\n  Notice that this option does not change any guarantees about\n  [atomicity and isolation](`m:ets#module-concurrency`). Functions that makes such\n  promises over many objects (like `insert/2`) gain less (or nothing) from this\n  option.\n\n  The memory consumption inflicted by both `write_concurrency` and\n  `read_concurrency` is a constant overhead per table for `set`, `bag` and\n  `duplicate_bag` when the `true` alternative for the `write_concurrency` option\n  is not used. For all tables with the `auto` alternative and `ordered_set`\n  tables with `true` alternative the memory overhead depends on the amount of\n  actual detected concurrency during runtime. The memory overhead can be\n  especially large when both `write_concurrency` and `read_concurrency` are\n  combined.\n\n  > #### Note {: .info }\n  >\n  > Prior to stdlib-3.7 (OTP-22.0) `write_concurrency` had no effect on\n  > `ordered_set`.\n\n  > #### Note {: .info }\n  >\n  > The `auto` alternative for the `write_concurrency` option is only available\n  > in OTP-25.0 and above.\n\n  [](){: #new_2_read_concurrency }\n\n- **`{read_concurrency,boolean()}`**(Since OTP R14B)  \n  Performance tuning. Defaults to `false`. When set to `true`, the table is\n  optimized for concurrent read operations. When this option is enabled read\n  operations become much cheaper; especially on systems with multiple physical\n  processors. However, switching between read and write operations becomes more\n  expensive.\n\n  You typically want to enable this option when concurrent read operations are\n  much more frequent than write operations, or when concurrent reads and writes\n  comes in large read and write bursts (that is, many reads not interrupted by\n  writes, and many writes not interrupted by reads).\n\n  You typically do _not_ want to enable this option when the common access\n  pattern is a few read operations interleaved with a few write operations\n  repeatedly. In this case, you would get a performance degradation by enabling\n  this option.\n\n  Option `read_concurrency` can be combined with option\n  [`write_concurrency`](`m:ets#new_2_write_concurrency`). You typically want to\n  combine these when large concurrent read bursts and large concurrent write\n  bursts are common.\n\n  [](){: #new_2_decentralized_counters }\n\n- **`{decentralized_counters,boolean()}`**(Since OTP 23.0)  \n  Performance tuning. Defaults to `true` for all tables with the\n  `write_concurrency` option set to `auto`. For tables of type `ordered_set` the\n  option also defaults to true when the `write_concurrency` option is set to\n  `true`. The option defaults to `false` for all other configurations. This\n  option has no effect if the `write_concurrency` option is set to `false`.\n\n  When this option is set to `true`, the table is optimized for frequent\n  concurrent calls to operations that modify the tables size and/or its memory\n  consumption (e.g., `insert/2` and `delete/2`). The drawback is that calls to\n  `info/1` and `info/2` with `size` or `memory` as the second argument can get\n  much slower when the `decentralized_counters` option is turned on.\n\n  When this option is enabled the counters for the table size and memory\n  consumption are distributed over several cache lines and the scheduling\n  threads are mapped to one of those cache lines. The `erl` option\n  [`+dcg`](`e:erts:erl_cmd.md#%2Bdcg`) can be used to control the number of\n  cache lines that the counters are distributed over.\n\n  [](){: #new_2_compressed }\n\n- **`compressed`**(Since OTP R14B01)  \n  If this option is present, the table data is stored in a more compact format\n  to consume less memory. However, it will make table operations slower.\n  Especially operations that need to inspect entire objects, such as `match` and\n  `select`, get much slower. The key element is not compressed.","ref":"ets.html#new/2"},{"type":"function","title":"ets.next/2","doc":"Returns the next key `Key2`, following key `Key1` in table `Table`. For table\ntype `ordered_set`, the next key in Erlang term order is returned. For other\ntable types, the next key according to the internal order of the table is\nreturned. If no next key exists, `'$end_of_table'` is returned.\n\nTo find the first key in the table, use `first/1`.\n\nUnless a table of type `set`, `bag`, or `duplicate_bag` is fixated using\n`safe_fixtable/2`, a call to [`next/2`](`next/2`) will fail if `Key1` no longer\nexists in the table. For table type `ordered_set`, the function always returns\nthe next key after `Key1` in term order, regardless whether `Key1` ever existed\nin the table.","ref":"ets.html#next/2"},{"type":"function","title":"ets.next_lookup/2","doc":"Similar to `next/2` except that it returns the object(s) along with the key\nstored in the table. This is equivalent to doing `next/2` followed by a\n`lookup/2`. If no next key exists, `'$end_of_table'` is returned.\n\nIt can be interleaved with `next/2` during traversal.","ref":"ets.html#next_lookup/2"},{"type":"function","title":"ets.prev/2","doc":"Returns the previous key `Key2`, preceding key `Key1` according to Erlang term\norder in table `Table` of type `ordered_set`. For other table types, the\nfunction is synonymous to `next/2`. If no previous key exists, `'$end_of_table'`\nis returned.\n\nTo find the last key in an `ordered_set` table, use `last/1`.","ref":"ets.html#prev/2"},{"type":"function","title":"ets.prev_lookup/2","doc":"Similar to `prev/2` except that it returns the object(s) along with the key\nstored in the table. This is equivalent to doing `prev/2` followed by a\n`lookup/2`. If no previous key exists, `'$end_of_table'` is returned.\n\nIt can be interleaved with `prev/2` during traversal.","ref":"ets.html#prev_lookup/2"},{"type":"function","title":"ets.rename/2","doc":"Renames the named table `Table` to the new name `Name`. Afterwards, the old name\ncannot be used to access the table. Renaming an unnamed table has no effect.","ref":"ets.html#rename/2"},{"type":"function","title":"ets.repair_continuation/2","doc":"Restores an opaque continuation returned by `select/3` or `select/1` if the\ncontinuation has passed through external term format (been sent between nodes or\nstored on disk).\n\nThe reason for this function is that continuation terms contain compiled match\nspecifications and may therefore be invalidated if converted to external term\nformat. Given that the original match specification is kept intact, the\ncontinuation can be restored, meaning it can once again be used in subsequent\n[`select/1`](`select/1`) calls even though it has been stored on disk or on\nanother node.\n\n_Examples:_\n\nThe following sequence of calls may fail:\n\n```erlang\nT=ets:new(x,[]),\n...\nMS = ets:fun2ms(fun({N,_}=A) when (N rem 10) =:= 0 -> A end),\n{_,C} = ets:select(T, MS, 10),\nMaybeBroken = binary_to_term(term_to_binary(C)),\nets:select(MaybeBroken).\n```\n\nThe following sequence works, as the call to\n[`repair_continuation/2`](`repair_continuation/2`) reestablishes the\n`MaybeBroken` continuation.\n\n```erlang\nT=ets:new(x,[]),\n...\nMS = ets:fun2ms(fun({N,_}=A) when (N rem 10) =:= 0 -> A end),\n{_,C} = ets:select(T,MS,10),\nMaybeBroken = binary_to_term(term_to_binary(C)),\nets:select(ets:repair_continuation(MaybeBroken,MS)).\n```\n\n> #### Note {: .info }\n>\n> This function is rarely needed in application code. It is used by Mnesia to\n> provide distributed [`select/3`](`select/3`) and [`select/1`](`select/1`)\n> sequences. A normal application would either use Mnesia or keep the\n> continuation from being converted to external format.\n>\n> The actual behavior of compiled match specifications when recreated from\n> external format has changed and may change in future releases, but this\n> interface remains for backward compatibility. See `is_compiled_ms/1`.","ref":"ets.html#repair_continuation/2"},{"type":"function","title":"ets.safe_fixtable/2","doc":"Fixes a table of type `set`, `bag`, or `duplicate_bag` for\n[safe traversal](`m:ets#traversal`) using `first/1` & `next/2`, `match/3` &\n`match/1`, `match_object/3` & `match_object/1`, or `select/3` & `select/1`.\n\nA process fixes a table by calling\n[`safe_fixtable(Table, true)`](`safe_fixtable/2`). The table remains fixed until\nthe process releases it by calling\n[`safe_fixtable(Table, false)`](`safe_fixtable/2`), or until the process\nterminates.\n\nIf many processes fix a table, the table remains fixed until all processes have\nreleased it (or terminated). A reference counter is kept on a per process basis,\nand N consecutive fixes requires N releases to release the table.\n\nWhen a table is fixed, a sequence of `first/1` and `next/2` calls are guaranteed\nto succeed even if keys are removed during the traversal. The keys for objects\ninserted or deleted during a traversal may or may not be returned by\n[`next/2`](`next/2`) depending on the ordering of keys within the table and if\nthe key exists at the time [`next/2`](`next/2`) is called.\n\n_Example:_\n\n```erlang\nclean_all_with_value(Table,X) ->\n    safe_fixtable(Table,true),\n    clean_all_with_value(Table,X,ets:first(Table)),\n    safe_fixtable(Table,false).\n\nclean_all_with_value(Table,X,'$end_of_table') ->\n    true;\nclean_all_with_value(Table,X,Key) ->\n    case ets:lookup(Table,Key) of\n        [{Key,X}] ->\n            ets:delete(Table,Key);\n        _ ->\n            true\n    end,\n    clean_all_with_value(Table,X,ets:next(Table,Key)).\n```\n\nNotice that deleted objects are not freed from a fixed table until it has been\nreleased. If a process fixes a table but never releases it, the memory used by\nthe deleted objects is never freed. The performance of operations on the table\nalso degrades significantly.\n\nTo retrieve information about which processes have fixed which tables, use\n[`info(Table, safe_fixed_monotonic_time)`](`m:ets#info_2_safe_fixed_monotonic_time`).\nA system with many processes fixing tables can need a monitor that sends alarms\nwhen tables have been fixed for too long.\n\nNotice that [`safe_fixtable/2`](`safe_fixtable/2`) is not necessary for table\ntype `ordered_set` and for traversals done by a single ETS function call, like\n`select/2`.","ref":"ets.html#safe_fixtable/2"},{"type":"function","title":"ets.select/1","doc":"Continues a match started with `select/3`. The next chunk of the size specified\nin the initial [`select/3`](`select/3`) call is returned together with a new\n`Continuation`, which can be used in subsequent calls to this function.\n\nWhen there are no more objects in the table, `'$end_of_table'` is returned.","ref":"ets.html#select/1"},{"type":"function","title":"ets.select/2","doc":"Matches the objects in table `Table` using a\n[match specification](`m:ets#match_spec`). This is a more general call than\n`match/2` and `match_object/2` calls. In its simplest form, the match\nspecification is as follows:\n\n```text\nMatchSpec = [MatchFunction]\nMatchFunction = {MatchHead, [Guard], [Result]}\nMatchHead = \"Pattern as in ets:match\"\nGuard = {\"Guardtest name\", ...}\nResult = \"Term construct\"\n```\n\nThis means that the match specification is always a list of one or more tuples\n(of arity 3). The first element of the tuple is to be a pattern as described in\n`match/2`. The second element of the tuple is to be a list of 0 or more guard\ntests (described below). The third element of the tuple is to be a list\ncontaining a description of the value to return. In almost all normal cases, the\nlist contains exactly one term that fully describes the value to return for each\nobject.\n\nThe return value is constructed using the \"match variables\" bound in `MatchHead`\nor using the special match variables `'$_'` (the whole matching object) and\n`'$$'` (all match variables in a list), so that the following\n[`match/2`](`match/2`) expression:\n\n```text\nets:match(Table,{'$1','$2','$3'})\n```\n\nis exactly equivalent to:\n\n```text\nets:select(Table,[{{'$1','$2','$3'},[],['$$']}])\n```\n\nAnd that the following [`match_object/2`](`match_object/2`) call:\n\n```text\nets:match_object(Table,{'$1','$2','$1'})\n```\n\nis exactly equivalent to\n\n```text\nets:select(Table,[{{'$1','$2','$1'},[],['$_']}])\n```\n\nComposite terms can be constructed in the `Result` part either by simply writing\na list, so that the following code:\n\n```text\nets:select(Table,[{{'$1','$2','$3'},[],['$$']}])\n```\n\ngives the same output as:\n\n```text\nets:select(Table,[{{'$1','$2','$3'},[],[['$1','$2','$3']]}])\n```\n\nThat is, all the bound variables in the match head as a list. If tuples are to\nbe constructed, one has to write a tuple of arity 1 where the single element in\nthe tuple is the tuple one wants to construct (as an ordinary tuple can be\nmistaken for a `Guard`).\n\nTherefore the following call:\n\n```text\nets:select(Table,[{{'$1','$2','$1'},[],['$_']}])\n```\n\ngives the same output as:\n\n```erlang\nets:select(Table,[{{'$1','$2','$1'},[],[{{'$1','$2','$3'}}]}])\n```\n\nThis syntax is equivalent to the syntax used in the trace patterns (see the\n`m:dbg`) module in Runtime_Tools.\n\nThe `Guard`s are constructed as tuples, where the first element is the test name\nand the remaining elements are the test parameters. To check for a specific type\n(say a list) of the element bound to the match variable `'$1'`, one would write\nthe test as `{is_list, '$1'}`. If the test fails, the object in the table does\nnot match and the next `MatchFunction` (if any) is tried. Most guard tests\npresent in Erlang can be used, but only the new versions prefixed `is_` are\nallowed (`is_float`, `is_atom`, and so on).\n\nThe `Guard` section can also contain logic and arithmetic operations, which are\nwritten with the same syntax as the guard tests (prefix notation), so that the\nfollowing guard test written in Erlang:\n\n```text\nis_integer(X), is_integer(Y), X + Y < 4711\n```\n\nis expressed as follows (`X` replaced with `'$1'` and `Y` with `'$2'`):\n\n```text\n[{is_integer, '$1'}, {is_integer, '$2'}, {'<', {'+', '$1', '$2'}, 4711}]\n```\n\nFor tables of type `ordered_set`, objects are visited in the same order as in a\n`first`/`next` traversal. This means that the match specification is executed\nagainst objects with keys in the `first`/`next` order and the corresponding\nresult list is in the order of that execution.","ref":"ets.html#select/2"},{"type":"function","title":"ets.select/3","doc":"Works like `select/2`, but only returns a limited (`Limit`) number of matching\nobjects. Term `Continuation` can then be used in subsequent calls to `select/1`\nto get the next chunk of matching objects. This is a space-efficient way to work\non objects in a table, which is still faster than traversing the table object by\nobject using `first/1` and `next/2`.\n\nIf the table is empty, `'$end_of_table'` is returned.\n\nUse `safe_fixtable/2` to guarantee [safe traversal](`m:ets#traversal`) for\nsubsequent calls to `select/1`.","ref":"ets.html#select/3"},{"type":"function","title":"ets.select_count/2","doc":"Matches the objects in table `Table` using a\n[match specification](`m:ets#match_spec`). If and only if the match specification\nreturns `true` for an object, that object is considered a match and is counted.\nFor any other result from the match specification the object is not considered a\nmatch and is therefore not counted.\n\nThe function returns the number of objects matched.","ref":"ets.html#select_count/2"},{"type":"function","title":"ets.select_delete/2","doc":"Matches the objects in table `Table` using a\n[match specification](`m:ets#match_spec`). If and only if the match\nspecification returns `true` for an object, that object is removed from the\ntable. For any other result from the match specification the object is\nretained. This is a more general function than `match_delete/2`.\n\nThe function returns the number of objects deleted from the table.\n\n> #### Note {: .info }\n>\n> The match specification has to return the atom `true` if the object is to be\n> deleted. No other return value gets the object deleted. So one cannot use the\n> same match specification for looking up elements as for deleting them.","ref":"ets.html#select_delete/2"},{"type":"function","title":"ets.select_replace/2","doc":"Matches the objects in the table `Table` using a\n[match specification](`m:ets#match_spec`). For each matched object, the existing\nobject is replaced with the match specification result.\n\nThe match-and-replace operation for each individual object is guaranteed to be\n[atomic and isolated](`m:ets#module-concurrency`). The `select_replace` table traversal\nas a whole, like all other select functions, does not give such guarantees.\n\nThe match specification must be guaranteed to _retain the key_ of any matched\nobject. If not, `select_replace` will fail with `badarg` without updating any\nobjects.\n\nFor the moment, due to performance and semantic constraints, tables of type\n`bag` are not yet supported.\n\nThe function returns the total number of replaced objects.\n\n_Example_\n\nFor all 2-tuples with a list in second position, add atom `'marker'` first in\nthe list:\n\n```erlang\n1> T = ets:new(x,[]), ets:insert(T, {key, [1, 2, 3]}).\ntrue\n2> MS = ets:fun2ms(fun({K, L}) when is_list(L) -> {K, [marker | L]} end).\n[{{'$1','$2'},[{is_list,'$2'}],[{{'$1',[marker|'$2']}}]}]\n3> ets:select_replace(T, MS).\n1\n4> ets:tab2list(T).\n[{key,[marker,1,2,3]}]\n```\n\nA generic single object compare-and-swap operation:\n\n```erlang\n[Old] = ets:lookup(T, Key),\nNew = update_object(Old),\nSuccess = (1 =:= ets:select_replace(T, [{Old, [], [{const, New}]}])),\n```","ref":"ets.html#select_replace/2"},{"type":"function","title":"ets.select_reverse/1","doc":"Continues a match started with `select_reverse/3`. For tables of type\n`ordered_set`, the traversal of the table continues to objects with keys earlier\nin the Erlang term order. The returned list also contains objects with keys in\nreverse order. For all other table types, the behavior is exactly that of\n`select/1`.\n\n_Example:_\n\n```erlang\n1> T = ets:new(x,[ordered_set]).\n2> [ ets:insert(T,{N}) || N <- lists:seq(1,10) ].\n...\n3> {R0,C0} = ets:select_reverse(T,[{'_',[],['$_']}],4).\n...\n4> R0.\n[{10},{9},{8},{7}]\n5> {R1,C1} = ets:select_reverse(C0).\n...\n6> R1.\n[{6},{5},{4},{3}]\n7> {R2,C2} = ets:select_reverse(C1).\n...\n8> R2.\n[{2},{1}]\n9> '$end_of_table' = ets:select_reverse(C2).\n...\n```","ref":"ets.html#select_reverse/1"},{"type":"function","title":"ets.select_reverse/2","doc":"Works like `select/2`, but returns the list in reverse order for table type\n`ordered_set`. For all other table types, the return value is identical to that\nof [`select/2`](`select/2`).","ref":"ets.html#select_reverse/2"},{"type":"function","title":"ets.select_reverse/3","doc":"Works like `select/3`, but for table type `ordered_set` traversing is done\nstarting at the last object in Erlang term order and moves to the first. For all\nother table types, the return value is identical to that of\n[`select/3`](`select/3`).\n\nNotice that this is _not_ equivalent to reversing the result list of a\n[`select/3`](`select/3`) call, as the result list is not only reversed, but also\ncontains the last `Limit` matching objects in the table, not the first.","ref":"ets.html#select_reverse/3"},{"type":"function","title":"ets.setopts/2","doc":"Sets table options. The only allowed option to be set after the table has been\ncreated is [`heir`](`m:ets#heir`). The calling process must be the table owner.","ref":"ets.html#setopts/2"},{"type":"function","title":"ets.slot/2","doc":"This function is mostly for debugging purposes, normally `first`/`next` or\n`last`/`prev` are to be used instead.\n\nReturns all objects in slot `I` of table `Table`. A table can be traversed by\nrepeatedly calling the function, starting with the first slot `I=0` and ending\nwhen `'$end_of_table'` is returned. If argument `I` is out of range, the\nfunction fails with reason `badarg`.\n\nUnless a table of type `set`, `bag`, or `duplicate_bag` is protected using\n`safe_fixtable/2`, a traversal can fail if concurrent updates are made to the\ntable. For table type `ordered_set`, the function returns a list containing\nobject `I` in Erlang term order.","ref":"ets.html#slot/2"},{"type":"function","title":"ets.tab2file/2","doc":"Dumps table `Table` to file `Filename`.\n\nEquivalent to [`tab2file(Table, Filename,[])`](`tab2file/3`)","ref":"ets.html#tab2file/2"},{"type":"function","title":"ets.tab2file/3","doc":"Dumps table `Table` to file `Filename`.\n\nWhen dumping the table, some information about the table is dumped to a header\nat the beginning of the dump. This information contains data about the table\ntype, name, protection, size, version, and if it is a named table. It also\ncontains notes about what extended information is added to the file, which can\nbe a count of the objects in the file or a MD5 sum of the header and records in\nthe file.\n\nThe size field in the header might not correspond to the number of records in\nthe file if the table is public and records are added or removed from the table\nduring dumping. Public tables updated during dump, and that one wants to verify\nwhen reading, needs at least one field of extended information for the read\nverification process to be reliable later.\n\nOption `extended_info` specifies what extra information is written to the table\ndump:\n\n- **`object_count`** - The number of objects written to the file is noted in the\n  file footer, so file truncation can be verified even if the file was updated\n  during dump.\n\n- **`md5sum`** - The header and objects in the file are checksummed using the\n  built-in MD5 functions. The MD5 sum of all objects is written in the file\n  footer, so that verification while reading detects the slightest bitflip in\n  the file data. Using this costs a fair amount of CPU time.\n\nWhenever option `extended_info` is used, it results in a file not readable by\nversions of ETS before that in STDLIB 1.15.1\n\nIf option `sync` is set to `true`, it ensures that the content of the file is\nwritten to the disk before `tab2file` returns. Defaults to `{sync, false}`.","ref":"ets.html#tab2file/3"},{"type":"function","title":"ets.tab2list/1","doc":"Returns a list of all objects in table `Table`.","ref":"ets.html#tab2list/1"},{"type":"type","title":"ets.tab/0","doc":"","ref":"ets.html#t:tab/0"},{"type":"function","title":"ets.tabfile_info/1","doc":"Returns information about the table dumped to file by `tab2file/2` or\n`tab2file/3`.\n\nThe following items are returned:\n\n- **`name`** - The name of the dumped table. If the table was a named table, a\n  table with the same name cannot exist when the table is loaded from file with\n  `file2tab/2`. If the table is not saved as a named table, this field has no\n  significance when loading the table from file.\n\n- **`type`** - The ETS type of the dumped table (that is, `set`, `bag`,\n  `duplicate_bag`, or `ordered_set`). This type is used when loading the table\n  again.\n\n- **`protection`** - The protection of the dumped table (that is, `private`,\n  `protected`, or `public`). A table loaded from the file gets the same\n  protection.\n\n- **`named_table`** - `true` if the table was a named table when dumped to file,\n  otherwise `false`. Notice that when a named table is loaded from a file, there\n  cannot exist a table in the system with the same name.\n\n- **`keypos`** - The `keypos` of the table dumped to file, which is used when\n  loading the table again.\n\n- **`size`** - The number of objects in the table when the table dump to file\n  started. For a `public` table, this number does not need to correspond to the\n  number of objects saved to the file, as objects can have been added or deleted\n  by another process during table dump.\n\n- **`extended_info`** - The extended information written in the file footer to\n  allow stronger verification during table loading from file, as specified to\n  `tab2file/3`. Notice that this function only tells _which_ information is\n  present, not the values in the file footer. The value is a list containing one\n  or more of the atoms `object_count` and `md5sum`.\n\n- **`version`** - A tuple `{Major,Minor}` containing the major and minor version\n  of the file format for ETS table dumps. This version field was added beginning\n  with STDLIB 1.5.1. Files dumped with older versions return `{0,0}` in this\n  field.\n\nAn error is returned if the file is inaccessible, badly damaged, or not produced\nwith `tab2file/2` or `tab2file/3`.","ref":"ets.html#tabfile_info/1"},{"type":"type","title":"ets.table/0","doc":"","ref":"ets.html#t:table/0"},{"type":"function","title":"ets.table/1","doc":"","ref":"ets.html#table/1"},{"type":"function","title":"ets.table/2","doc":"Returns a Query List Comprehension (QLC) query handle. The `m:qlc` module\nprovides a query language aimed mainly at Mnesia, but ETS tables, Dets tables,\nand lists are also recognized by QLC as sources of data. Calling `table/1,2` is\nthe means to make the ETS table `Table` usable to QLC.\n\nWhen there are only simple restrictions on the key position, QLC uses `lookup/2`\nto look up the keys. When that is not possible, the whole table is traversed.\nOption `traverse` determines how this is done:\n\n- **`first_next`** - The table is traversed one key at a time by calling\n  `first/1` and `next/2`.\n\n- **`last_prev`** - The table is traversed one key at a time by calling `last/1`\n  and `prev/2`.\n\n- **`select`** - The table is traversed by calling `select/3` and `select/1`.\n  Option `n_objects` determines the number of objects returned (the third\n  argument of [`select/3`](`select/3`)); the default is to return `100` objects\n  at a time. The [match specification](`m:ets#match_spec`) (the second argument\n  of [`select/3`](`select/3`)) is assembled by QLC: simple filters are\n  translated into equivalent match specifications while more complicated filters\n  must be applied to all objects returned by [`select/3`](`select/3`) given a\n  match specification that matches all objects.\n\n- **`{select, MatchSpec}`** - As for `select`, the table is traversed by calling\n  `select/3` and `select/1`. The difference is that the match specification is\n  explicitly specified. This is how to state match specifications that cannot\n  easily be expressed within the syntax provided by QLC.\n\n_Examples:_\n\nAn explicit match specification is here used to traverse the table:\n\n```erlang\n9> true = ets:insert(Table = ets:new(t, []), [{1,a},{2,b},{3,c},{4,d}]),\nMS = ets:fun2ms(fun({X,Y}) when (X > 1) or (X < 5) -> {Y} end),\nQH1 = ets:table(Table, [{traverse, {select, MS}}]).\n```\n\nAn example with an implicit match specification:\n\n```erlang\n10> QH2 = qlc:q([{Y} || {X,Y} <- ets:table(Table), (X > 1) or (X < 5)]).\n```\n\nThe latter example is equivalent to the former, which can be verified using\nfunction `qlc:info/1`:\n\n```erlang\n11> qlc:info(QH1) =:= qlc:info(QH2).\ntrue\n```\n\n`qlc:info/1` returns information about a query handle, and in this case\nidentical information is returned for the two query handles.","ref":"ets.html#table/2"},{"type":"type","title":"ets.table_access/0","doc":"","ref":"ets.html#t:table_access/0"},{"type":"type","title":"ets.table_type/0","doc":"","ref":"ets.html#t:table_type/0"},{"type":"function","title":"ets.take/2","doc":"Returns and removes a list of all objects with key `Key` in table `Table`.\n\nThe specified `Key` is used to identify the object by either _comparing equal_\nthe key of an object in an `ordered_set` table, or _matching_ in other types of\ntables (for details on the difference, see `lookup/2` and `new/2`).","ref":"ets.html#take/2"},{"type":"function","title":"ets.test_ms/2","doc":"This function is a utility to test a [match specification](`m:ets#match_spec`)\nused in calls to `select/2`. The function both tests `MatchSpec` for \"syntactic\"\ncorrectness and runs the match specification against object `Tuple`.\n\nIf the match specification is syntactically correct, the function either returns\n`{ok,Result}`, where `Result` is what would have been the result in a real\n[`select/2`](`select/2`) call, or `false` if the match specification does not\nmatch object `Tuple`.\n\nIf the match specification contains errors, tuple `{error, Errors}` is returned,\nwhere `Errors` is a list of natural language descriptions of what was wrong with\nthe match specification.\n\nThis is a useful debugging and test tool, especially when writing complicated\n[`select/2`](`select/2`) calls.\n\nSee also: `erlang:match_spec_test/3`.","ref":"ets.html#test_ms/2"},{"type":"opaque","title":"ets.tid/0","doc":"A table identifier, as returned by `new/2`.","ref":"ets.html#t:tid/0"},{"type":"function","title":"ets.to_dets/2","doc":"Fills an already created/opened Dets table with the objects in the already\nopened ETS table named `Table`. The Dets table is emptied before the objects are\ninserted.","ref":"ets.html#to_dets/2"},{"type":"function","title":"ets.update_counter/3","doc":"","ref":"ets.html#update_counter/3"},{"type":"function","title":"ets.update_counter/4","doc":"This function provides an efficient way to update one or more counters, without\nthe trouble of having to look up an object, update the object by incrementing an\nelement, and insert the resulting object into the table again. The operation is\nguaranteed to be [atomic and isolated](`m:ets#module-concurrency`).\n\nThis function destructively updates the object with key `Key` in table `Table`\nby adding `Incr` to the element at position `Pos`. The new counter value is\nreturned. If no position is specified, the element directly following key\n(` +1`) is updated.\n\nIf a `Threshold` is specified, the counter is reset to value `SetValue` if the\nfollowing conditions occur:\n\n- `Incr` is not negative (`>= 0`) and the result would be greater than (`>`)\n  `Threshold`.\n- `Incr` is negative (`< 0`) and the result would be less than (`<`)\n  `Threshold`.\n\nA list of `UpdateOp` can be supplied to do many update operations within the\nobject. The operations are carried out in the order specified in the list. If\nthe same counter position occurs more than once in the list, the corresponding\ncounter is thus updated many times, each time based on the previous result. The\nreturn value is a list of the new counter values from each update operation in\nthe same order as in the operation list. If an empty list is specified, nothing\nis updated and an empty list is returned. If the function fails, no updates are\ndone.\n\nThe specified `Key` is used to identify the object by either _matching_ the key\nof an object in a `set` table, or _compare equal_ to the key of an object in an\n`ordered_set` table (for details on the difference, see `lookup/2` and `new/2`).\n\nIf a default object `Default` is specified, it is used as the object to be\nupdated if the key is missing from the table. The value in place of the key is\nignored and replaced by the proper key value. The return value is as if the\ndefault object had not been used, that is, a single updated element or a list of\nthem.\n\nThe function fails with reason `badarg` in the following situations:\n\n- The table type is not `set` or `ordered_set`.\n- No object with the correct key exists and no default object was supplied.\n- The object has the wrong arity.\n- The default object arity is smaller than ` `.\n- Any field from the default object that is updated is not an integer.\n- The element to update is not an integer.\n- The element to update is also the key.\n- Any of `Pos`, `Incr`, `Threshold`, or `SetValue` is not an integer.","ref":"ets.html#update_counter/4"},{"type":"function","title":"ets.update_element/3","doc":"","ref":"ets.html#update_element/3"},{"type":"function","title":"ets.update_element/4","doc":"This function provides an efficient way to update one or more elements within an\nobject, without the trouble of having to look up, update, and write back the\nentire object.\n\nThis function destructively updates the object with key `Key` in table `Table`.\nThe element at position `Pos` is given the value `Value`.\n\nA list of `{Pos,Value}` can be supplied to update many elements within the same\nobject. If the same position occurs more than once in the list, the last value\nin the list is written. If the list is empty or the function fails, no updates\nare done. The function is also atomic in the sense that other processes can\nnever see any intermediate results.\n\nReturns `true` if an object with key `Key` is found, otherwise `false`.\n\nThe specified `Key` is used to identify the object by either _matching_ the key\nof an object in a `set` table, or _compare equal_ to the key of an object in an\n`ordered_set` table (for details on the difference, see `lookup/2` and `new/2`).\n\nIf a default object `Default` is specified, it is used as the object to be\nupdated if the key is missing from the table. The value in place of the key is\nignored and replaced by the proper key value.\n\nThe function fails with reason `badarg` in the following situations:\n\n- The table type is not `set` or `ordered_set`.\n- `Pos` < 1.\n- `Pos` > object arity.\n- The default object arity is smaller than ` `.\n- The element to update is also the key.","ref":"ets.html#update_element/4"},{"type":"function","title":"ets.whereis/1","doc":"This function returns the `t:tid/0` of the named table identified by\n`TableName`, or `undefined` if no such table exists. The `t:tid/0` can be used\nin place of the table name in all operations, which is slightly faster since the\nname does not have to be resolved on each call.\n\nIf the table is deleted, the `t:tid/0` will be invalid even if another named\ntable is created with the same name.","ref":"ets.html#whereis/1"},{"type":"module","title":"gb_sets","doc":"Sets represented by general balanced trees.\n\nThis module provides ordered sets using Prof. Arne Andersson's General Balanced\nTrees. Ordered sets can be much more efficient than using ordered lists, for\nlarger sets, but depends on the application.\n\nThe data representing a set as used by this module is to be regarded as opaque\nby other modules. In abstract terms, the representation is a composite type of\nexisting Erlang terms. See note on\n[data types](`e:system:data_types.md#no_user_types`). Any code assuming\nknowledge of the format is running on thin ice.\n\nThis module considers two elements as different if and only if they do not\ncompare equal (`==`).","ref":"gb_sets.html"},{"type":"module","title":"Complexity Note - gb_sets","doc":"The complexity on set operations is bounded by either _O(|S|)_ or _O(|T| _\nlog(|S|))\\*, where S is the largest given set, depending on which is fastest for\nany particular function call. For operating on sets of almost equal size, this\nimplementation is about 3 times slower than using ordered-list sets directly.\nFor sets of very different sizes, however, this solution can be arbitrarily much\nfaster; in practical cases, often 10-100 times. This implementation is\nparticularly suited for accumulating elements a few at a time, building up a\nlarge set (> 100-200 elements), and repeatedly testing for membership in the\ncurrent set.\n\nAs with normal tree structures, lookup (membership testing), insertion, and\ndeletion have logarithmic complexity.","ref":"gb_sets.html#module-complexity-note"},{"type":"module","title":"Compatibility - gb_sets","doc":"See the [Compatibility Section in the `sets` module](`m:sets#module-compatibility`)\nfor information about the compatibility of the different implementations of sets\nin the Standard Library.","ref":"gb_sets.html#module-compatibility"},{"type":"module","title":"See Also - gb_sets","doc":"`m:gb_trees`, `m:ordsets`, `m:sets`","ref":"gb_sets.html#module-see-also"},{"type":"function","title":"gb_sets.add/2","doc":"","ref":"gb_sets.html#add/2"},{"type":"function","title":"gb_sets.add_element/2","doc":"Returns a new set formed from `Set1` with `Element` inserted. If `Element` is\nalready an element in `Set1`, nothing is changed.","ref":"gb_sets.html#add_element/2"},{"type":"function","title":"gb_sets.balance/1","doc":"Rebalances the tree representation of `Set1`.\n\nNotice that this is rarely necessary, but can be motivated when a large number of\nelements have been deleted from the tree without further insertions. Rebalancing\n can then be forced to minimise lookup times, as deletion does not rebalance the\ntree.","ref":"gb_sets.html#balance/1"},{"type":"function","title":"gb_sets.del_element/2","doc":"","ref":"gb_sets.html#del_element/2"},{"type":"function","title":"gb_sets.delete/2","doc":"Returns a new set formed from `Set1` with `Element` removed. Assumes that\n`Element` is present in `Set1`.","ref":"gb_sets.html#delete/2"},{"type":"function","title":"gb_sets.delete_any/2","doc":"Returns a new set formed from `Set1` with `Element` removed. If `Element` is not\nan element in `Set1`, nothing is changed.","ref":"gb_sets.html#delete_any/2"},{"type":"function","title":"gb_sets.difference/2","doc":"","ref":"gb_sets.html#difference/2"},{"type":"function","title":"gb_sets.empty/0","doc":"Returns a new empty set.","ref":"gb_sets.html#empty/0"},{"type":"function","title":"gb_sets.filter/2","doc":"Filters elements in `Set1` using predicate function `Pred`.","ref":"gb_sets.html#filter/2"},{"type":"function","title":"gb_sets.filtermap/2","doc":"Filters and maps elements in `Set1` using function `Fun`.","ref":"gb_sets.html#filtermap/2"},{"type":"function","title":"gb_sets.fold/3","doc":"Folds `Function` over every element in `Set` returning the final value of the\naccumulator.","ref":"gb_sets.html#fold/3"},{"type":"function","title":"gb_sets.from_list/1","doc":"Returns a set of the elements in `List`, where `List` can be unordered and\ncontain duplicates.","ref":"gb_sets.html#from_list/1"},{"type":"function","title":"gb_sets.from_ordset/1","doc":"Turns an ordered-set list `List` into a set. The list must not contain\nduplicates.","ref":"gb_sets.html#from_ordset/1"},{"type":"function","title":"gb_sets.insert/2","doc":"Returns a new set formed from `Set1` with `Element` inserted. Assumes that\n`Element` is not present in `Set1`.","ref":"gb_sets.html#insert/2"},{"type":"function","title":"gb_sets.intersection/1","doc":"Returns the intersection of the non-empty list of sets.","ref":"gb_sets.html#intersection/1"},{"type":"function","title":"gb_sets.intersection/2","doc":"Returns the intersection of `Set1` and `Set2`.","ref":"gb_sets.html#intersection/2"},{"type":"function","title":"gb_sets.is_disjoint/2","doc":"Returns `true` if `Set1` and `Set2` are disjoint (have no elements in common),\notherwise `false`.","ref":"gb_sets.html#is_disjoint/2"},{"type":"function","title":"gb_sets.is_element/2","doc":"","ref":"gb_sets.html#is_element/2"},{"type":"function","title":"gb_sets.is_empty/1","doc":"Returns `true` if `Set` is an empty set, otherwise `false`.","ref":"gb_sets.html#is_empty/1"},{"type":"function","title":"gb_sets.is_equal/2","doc":"Returns `true` if `Set1` and `Set2` are equal, that is when every element of one\nset is also a member of the respective other set, otherwise `false`.","ref":"gb_sets.html#is_equal/2"},{"type":"function","title":"gb_sets.is_member/2","doc":"Returns `true` if `Element` is an member of `Set`, otherwise `false`.","ref":"gb_sets.html#is_member/2"},{"type":"function","title":"gb_sets.is_set/1","doc":"Returns `true` if `Term` appears to be a set, otherwise `false`. This function\nwill return `true` for any term that coincides with the representation of a\n`gb_set`, while not really being a `gb_set`, thus it might return false positive\nresults. See also note on [data types](`e:system:data_types.md#no_user_types`).","ref":"gb_sets.html#is_set/1"},{"type":"function","title":"gb_sets.is_subset/2","doc":"Returns `true` when every element of `Set1` is also a member of `Set2`,\notherwise `false`.","ref":"gb_sets.html#is_subset/2"},{"type":"type","title":"gb_sets.iter/0","doc":"","ref":"gb_sets.html#t:iter/0"},{"type":"opaque","title":"gb_sets.iter/1","doc":"A general balanced set iterator.","ref":"gb_sets.html#t:iter/1"},{"type":"function","title":"gb_sets.iterator/1","doc":"Returns an iterator that can be used for traversing the entries of `Set`; see\n`next/1`.\n\nEquivalent to [`iterator(Set, ordered)`](`iterator/2`).","ref":"gb_sets.html#iterator/1"},{"type":"function","title":"gb_sets.iterator/2","doc":"Returns an iterator that can be used for traversing the entries of `Set` in\neither `ordered` or `reversed` direction; see `next/1`.\n\nThe implementation of this is very efficient; traversing the whole set using\n[`next/1`](`next/1`) is only slightly slower than getting the list of all\n elements using `to_list/1` and traversing that. The main advantage of the\niterator approach is that it does not require the complete list of all elements\nto be built in memory at one time.","ref":"gb_sets.html#iterator/2"},{"type":"function","title":"gb_sets.iterator_from/2","doc":"Returns an iterator that can be used for traversing the entries of `Set`; see\n`next/1`. The difference as compared to the iterator returned by `iterator/1` is\nthat the iterator starts with the first element greater than or equal to\n`Element`.\n\nEquivalent to [`iterator_from(Element, Set, ordered)`](`iterator_from/3`).","ref":"gb_sets.html#iterator_from/2"},{"type":"function","title":"gb_sets.iterator_from/3","doc":"Returns an iterator that can be used for traversing the entries of `Set`; see\n`next/1`. The difference as compared to the iterator returned by `iterator/2` is\nthat the iterator starts with the first element next to or equal to `Element`.","ref":"gb_sets.html#iterator_from/3"},{"type":"function","title":"gb_sets.larger/2","doc":"Returns `{found, Element2}`, where `Element2` is the least element strictly\ngreater than `Element1`.\n\nReturns `none` if no such element exists.","ref":"gb_sets.html#larger/2"},{"type":"function","title":"gb_sets.largest/1","doc":"Returns the largest element in `Set`. Assumes that `Set` is not empty.","ref":"gb_sets.html#largest/1"},{"type":"function","title":"gb_sets.map/2","doc":"Maps elements in `Set1` using mapping function `Fun`.","ref":"gb_sets.html#map/2"},{"type":"function","title":"gb_sets.new/0","doc":"Returns a new empty set.","ref":"gb_sets.html#new/0"},{"type":"function","title":"gb_sets.next/1","doc":"Returns `{Element, Iter2}`, where `Element` is the smallest element referred to\nby iterator `Iter1`, and `Iter2` is the new iterator to be used for traversing\nthe remaining elements, or the atom `none` if no elements remain.","ref":"gb_sets.html#next/1"},{"type":"type","title":"gb_sets.set/0","doc":"","ref":"gb_sets.html#t:set/0"},{"type":"opaque","title":"gb_sets.set/1","doc":"A general balanced set.","ref":"gb_sets.html#t:set/1"},{"type":"function","title":"gb_sets.singleton/1","doc":"Returns a set containing only element `Element`.","ref":"gb_sets.html#singleton/1"},{"type":"function","title":"gb_sets.size/1","doc":"Returns the number of elements in `Set`.","ref":"gb_sets.html#size/1"},{"type":"function","title":"gb_sets.smaller/2","doc":"Returns `{found, Element2}`, where `Element2` is the greatest element strictly\nless than `Element1`.\n\nReturns `none` if no such element exists.","ref":"gb_sets.html#smaller/2"},{"type":"function","title":"gb_sets.smallest/1","doc":"Returns the smallest element in `Set`. Assumes that `Set` is not empty.","ref":"gb_sets.html#smallest/1"},{"type":"function","title":"gb_sets.subtract/2","doc":"Returns only the elements of `Set1` that are not also elements of `Set2`.","ref":"gb_sets.html#subtract/2"},{"type":"function","title":"gb_sets.take_largest/1","doc":"Returns `{Element, Set2}`, where `Element` is the largest element in `Set1`, and\n`Set2` is this set with `Element` deleted. Assumes that `Set1` is not empty.","ref":"gb_sets.html#take_largest/1"},{"type":"function","title":"gb_sets.take_smallest/1","doc":"Returns `{Element, Set2}`, where `Element` is the smallest element in `Set1`,\nand `Set2` is this set with `Element` deleted. Assumes that `Set1` is not empty.","ref":"gb_sets.html#take_smallest/1"},{"type":"function","title":"gb_sets.to_list/1","doc":"Returns the elements of `Set` as a list.","ref":"gb_sets.html#to_list/1"},{"type":"function","title":"gb_sets.union/1","doc":"Returns the merged (union) set of the list of sets.","ref":"gb_sets.html#union/1"},{"type":"function","title":"gb_sets.union/2","doc":"Returns the merged (union) set of `Set1` and `Set2`.","ref":"gb_sets.html#union/2"},{"type":"module","title":"gb_trees","doc":"General balanced trees.\n\nThis module provides Prof. Arne Andersson's General Balanced Trees. These have\nno storage overhead compared to unbalanced binary trees, and their performance\nis better than AVL trees.\n\nThis module considers two keys as different if and only if they do not compare\nequal (`==`).","ref":"gb_trees.html"},{"type":"module","title":"Data Structure - gb_trees","doc":"Trees and iterators are built using opaque data structures that should not be\npattern-matched from outside this module.\n\nThere is no attempt to balance trees after deletions. As deletions do not\nincrease the height of a tree, this should be OK.\n\nThe original balance condition `h(T) <= ceil(c * log(|T|))` has been changed to\nthe similar (but not quite equivalent) condition `2 ^ h(T) <= |T| ^ c`. This\nshould also be OK.","ref":"gb_trees.html#module-data-structure"},{"type":"module","title":"See Also - gb_trees","doc":"`m:dict`, `m:gb_sets`","ref":"gb_trees.html#module-see-also"},{"type":"function","title":"gb_trees.balance/1","doc":"Rebalances `Tree1`.\n\nNotice that this is rarely necessary, but can be motivated\nwhen many nodes have been deleted from the tree without further insertions.\nRebalancing can then be forced to minimize lookup times, as deletion does not\nrebalance the tree.","ref":"gb_trees.html#balance/1"},{"type":"function","title":"gb_trees.delete/2","doc":"Removes the node with key `Key` from `Tree1` and returns the new tree. Assumes\nthat the key is present in the tree, crashes otherwise.","ref":"gb_trees.html#delete/2"},{"type":"function","title":"gb_trees.delete_any/2","doc":"Removes the node with key `Key` from `Tree1` if the key is present in the tree,\notherwise does nothing. Returns the new tree.","ref":"gb_trees.html#delete_any/2"},{"type":"function","title":"gb_trees.empty/0","doc":"Returns a new empty tree.","ref":"gb_trees.html#empty/0"},{"type":"function","title":"gb_trees.enter/3","doc":"Inserts `Key` with value `Value` into `Tree1` if the key is not present in the\ntree, otherwise updates `Key` to value `Value` in `Tree1`. Returns the new tree.","ref":"gb_trees.html#enter/3"},{"type":"function","title":"gb_trees.from_orddict/1","doc":"Turns an ordered list `List` of key-value tuples into a tree. The list must not\ncontain duplicate keys.","ref":"gb_trees.html#from_orddict/1"},{"type":"function","title":"gb_trees.get/2","doc":"Retrieves the value stored with `Key` in `Tree`. Assumes that the key is present\nin the tree, crashes otherwise.","ref":"gb_trees.html#get/2"},{"type":"function","title":"gb_trees.insert/3","doc":"Inserts `Key` with value `Value` into `Tree1` and returns the new tree. Assumes\nthat the key is not present in the tree, crashes otherwise.","ref":"gb_trees.html#insert/3"},{"type":"function","title":"gb_trees.is_defined/2","doc":"Returns `true` if `Key` is present in `Tree`, otherwise `false`.","ref":"gb_trees.html#is_defined/2"},{"type":"function","title":"gb_trees.is_empty/1","doc":"Returns `true` if `Tree` is an empty tree, othwewise `false`.","ref":"gb_trees.html#is_empty/1"},{"type":"type","title":"gb_trees.iter/0","doc":"","ref":"gb_trees.html#t:iter/0"},{"type":"opaque","title":"gb_trees.iter/2","doc":"A general balanced tree iterator.","ref":"gb_trees.html#t:iter/2"},{"type":"function","title":"gb_trees.iterator/1","doc":"Returns an iterator that can be used for traversing the entries of `Tree`; see\n`next/1`.\n\nEquivalent to [`iterator(Tree, ordered)`](`iterator/2`).","ref":"gb_trees.html#iterator/1"},{"type":"function","title":"gb_trees.iterator/2","doc":"Returns an iterator that can be used for traversing the entries of `Tree` in\neither `ordered` or `reversed` direction; see `next/1`.\n\nThe implementation of this is very efficient; traversing the whole tree using\n[`next/1`](`next/1`) is only slightly slower than getting the list of all\nelements using `to_list/1` and traversing that. The main advantage of the\niterator approach is that it does not require the complete list of all elements\nto be built in memory at one time.","ref":"gb_trees.html#iterator/2"},{"type":"function","title":"gb_trees.iterator_from/2","doc":"Returns an iterator that can be used for traversing the entries of `Tree`; see\n`next/1`. The difference as compared to the iterator returned by `iterator/1` is\nthat the iterator starts with the first key greater than or equal to `Key`.\n\nEquivalent to [`iterator_from(Key, Tree, ordered)`](`iterator_from/3`).","ref":"gb_trees.html#iterator_from/2"},{"type":"function","title":"gb_trees.iterator_from/3","doc":"Returns an iterator that can be used for traversing the entries of `Tree` in\neither `ordered` or `reversed` direction; see `next/1`. The difference as\ncompared to the iterator returned by `iterator/2` is that the iterator starts\nwith the first key next to or equal to `Key`.","ref":"gb_trees.html#iterator_from/3"},{"type":"function","title":"gb_trees.keys/1","doc":"Returns the keys in `Tree` as an ordered list.","ref":"gb_trees.html#keys/1"},{"type":"function","title":"gb_trees.larger/2","doc":"Returns `{Key2, Value}`, where `Key2` is the least key strictly greater than\n`Key1`, `Value` is the value associated with this key.\n\nReturns `none` if no such pair exists.","ref":"gb_trees.html#larger/2"},{"type":"function","title":"gb_trees.largest/1","doc":"Returns `{Key, Value}`, where `Key` is the largest key in `Tree`, and `Value` is\nthe value associated with this key. Assumes that the tree is not empty.","ref":"gb_trees.html#largest/1"},{"type":"function","title":"gb_trees.lookup/2","doc":"Looks up `Key` in `Tree`. Returns `{value, Value}`, or `none` if `Key` is not\npresent.","ref":"gb_trees.html#lookup/2"},{"type":"function","title":"gb_trees.map/2","doc":"Maps function F(K, V1) -> V2 to all key-value pairs of tree `Tree1`. Returns a\nnew tree `Tree2` with the same set of keys as `Tree1` and the new set of values\n`V2`.","ref":"gb_trees.html#map/2"},{"type":"function","title":"gb_trees.next/1","doc":"Returns `{Key, Value, Iter2}`, where `Key` is the next key referred to by\niterator `Iter1`, and `Iter2` is the new iterator to be used for traversing the\nremaining nodes, or the atom `none` if no nodes remain.","ref":"gb_trees.html#next/1"},{"type":"function","title":"gb_trees.size/1","doc":"Returns the number of nodes in `Tree`.","ref":"gb_trees.html#size/1"},{"type":"function","title":"gb_trees.smaller/2","doc":"Returns `{Key2, Value}`, where `Key2` is the greatest key strictly less than\n`Key1`, `Value` is the value associated with this key.\n\nReturns `none` if no such pair exists.","ref":"gb_trees.html#smaller/2"},{"type":"function","title":"gb_trees.smallest/1","doc":"Returns `{Key, Value}`, where `Key` is the smallest key in `Tree`, and `Value`\nis the value associated with this key. Assumes that the tree is not empty.","ref":"gb_trees.html#smallest/1"},{"type":"function","title":"gb_trees.take/2","doc":"Returns a value `Value` from node with key `Key` and new `Tree2` without the\nnode with this value. Assumes that the node with key is present in the tree,\ncrashes otherwise.","ref":"gb_trees.html#take/2"},{"type":"function","title":"gb_trees.take_any/2","doc":"Returns a value `Value` from node with key `Key` and new `Tree2` without the\nnode with this value. Returns `error` if the node with the key is not present in\nthe tree.","ref":"gb_trees.html#take_any/2"},{"type":"function","title":"gb_trees.take_largest/1","doc":"Returns `{Key, Value, Tree2}`, where `Key` is the largest key in `Tree1`,\n`Value` is the value associated with this key, and `Tree2` is this tree with the\ncorresponding node deleted. Assumes that the tree is not empty.","ref":"gb_trees.html#take_largest/1"},{"type":"function","title":"gb_trees.take_smallest/1","doc":"Returns `{Key, Value, Tree2}`, where `Key` is the smallest key in `Tree1`,\n`Value` is the value associated with this key, and `Tree2` is this tree with the\ncorresponding node deleted. Assumes that the tree is not empty.","ref":"gb_trees.html#take_smallest/1"},{"type":"function","title":"gb_trees.to_list/1","doc":"Converts a tree into an ordered list of key-value tuples.","ref":"gb_trees.html#to_list/1"},{"type":"type","title":"gb_trees.tree/0","doc":"","ref":"gb_trees.html#t:tree/0"},{"type":"opaque","title":"gb_trees.tree/2","doc":"A general balanced tree.","ref":"gb_trees.html#t:tree/2"},{"type":"function","title":"gb_trees.update/3","doc":"Updates `Key` to value `Value` in `Tree1` and returns the new tree. Assumes that\nthe key is present in the tree.","ref":"gb_trees.html#update/3"},{"type":"function","title":"gb_trees.values/1","doc":"Returns the values in `Tree` as an ordered list, sorted by their corresponding\nkeys. Duplicates are not removed.","ref":"gb_trees.html#values/1"},{"type":"module","title":"json","doc":"A library for encoding and decoding JSON.\n\nThis module implements [EEP68](https://github.com/erlang/eep/blob/master/eeps/eep-0068.md).\n\nBoth encoder and decoder fully conform to\n[RFC 8259](https://tools.ietf.org/html/rfc8259) and\n[ECMA 404](https://ecma-international.org/publications-and-standards/standards/ecma-404/)\nstandards. The decoder is tested using [JSONTestSuite](https://github.com/nst/JSONTestSuite).","ref":"json.html"},{"type":"type","title":"json.array_finish_fun/0","doc":"","ref":"json.html#t:array_finish_fun/0"},{"type":"type","title":"json.array_push_fun/0","doc":"","ref":"json.html#t:array_push_fun/0"},{"type":"type","title":"json.array_start_fun/0","doc":"","ref":"json.html#t:array_start_fun/0"},{"type":"opaque","title":"json.continuation_state/0","doc":"","ref":"json.html#t:continuation_state/0"},{"type":"function","title":"json.decode/1","doc":"Parses a JSON value from `Binary`.\n\nSupports basic data mapping:\n\n| **JSON** | **Erlang**             |\n|----------|------------------------|\n| Number   | `integer() \\| float()` |\n| Boolean  | `true \\| false`        |\n| Null     | `null`                 |\n| String   | `binary()`             |\n| Object   | `#{binary() => _}`     |","ref":"json.html#decode/1"},{"type":"function","title":"Errors - json.decode/1","doc":"* `error(unexpected_end)` if `Binary` contains incomplete JSON value\n* `error({invalid_byte, Byte})` if `Binary` contains unexpected byte or invalid UTF-8 byte\n* `error({unexpected_sequence, Bytes})` if `Binary` contains invalid UTF-8 escape","ref":"json.html#decode/1-errors"},{"type":"function","title":"Example - json.decode/1","doc":"```erlang\n> json:decode(<<\"{\\\"foo\\\": 1}\">>).\n#{<<\"foo\">> => 1}\n```","ref":"json.html#decode/1-example"},{"type":"function","title":"json.decode/3","doc":"Parses a JSON value from `Binary`.\n\nSimilar to `decode/1` except the decoding process\ncan be customized with the callbacks specified in\n`Decoders`. The callbacks will use the `Acc` value\nas the initial accumulator.\n\nAny leftover, unparsed data in `Binary` will be returned.","ref":"json.html#decode/3"},{"type":"function","title":"Default callbacks - json.decode/3","doc":"All callbacks are optional. If not provided, they will fall back to\nimplementations used by the `decode/1` function:\n\n* for `array_start`: `fun(_) -> [] end`\n* for `array_push`: `fun(Elem, Acc) -> [Elem | Acc] end`\n* for `array_finish`: `fun(Acc, OldAcc) -> {lists:reverse(Acc), OldAcc} end`\n* for `object_start`: `fun(_) -> [] end`\n* for `object_push`: `fun(Key, Value, Acc) -> [{Key, Value} | Acc] end`\n* for `object_finish`: `fun(Acc, OldAcc) -> {maps:from_list(Acc), OldAcc} end`\n* for `float`: `fun erlang:binary_to_float/1`\n* for `integer`: `fun erlang:binary_to_integer/1`\n* for `string`: `fun (Value) -> Value end`\n* for `null`: the atom `null`","ref":"json.html#decode/3-default-callbacks"},{"type":"function","title":"Errors - json.decode/3","doc":"* `error({invalid_byte, Byte})` if `Binary` contains unexpected byte or invalid UTF-8 byte\n* `error({unexpected_sequence, Bytes})` if `Binary` contains invalid UTF-8 escape\n* `error(unexpected_end)` if `Binary` contains incomplete JSON value","ref":"json.html#decode/3-errors"},{"type":"function","title":"Example - json.decode/3","doc":"Decoding object keys as atoms:\n\n```erlang\n> Push = fun(Key, Value, Acc) -> [{binary_to_existing_atom(Key), Value} | Acc] end.\n> json:decode(<<\"{\\\"foo\\\": 1}\">>, ok, #{object_push => Push}).\n{#{foo => 1},ok,<<>>}\n```","ref":"json.html#decode/3-example"},{"type":"function","title":"json.decode_continue/2","doc":"Continue parsing a stream of bytes of a JSON value.\n\nSimilar to `decode_start/3`, if the function returns `{continue, State}` and\nthere is no more data, use `end_of_input` instead of a binary.\n\n```erlang\n> {continue, State} = json:decode_start(<<\"{\\\"foo\\\":\">>, ok, #{}).\n> json:decode_continue(<<\"1}\">>, State).\n{#{foo => 1},ok,<<>>}\n```\n```erlang\n> {continue, State} = json:decode_start(<<\"123\">>, ok, #{}).\n> json:decode_continue(end_of_input, State).\n{123,ok,<<>>}\n```","ref":"json.html#decode_continue/2"},{"type":"function","title":"json.decode_start/3","doc":"Begin parsing a stream of bytes of a JSON value.\n\nSimilar to `decode/3` but returns when a complete JSON value can be parsed or\nreturns `{continue, State}` for incomplete data,\nthe `State` can be fed to the `decode_continue/2` function when more data is available.","ref":"json.html#decode_start/3"},{"type":"type","title":"json.decode_value/0","doc":"","ref":"json.html#t:decode_value/0"},{"type":"type","title":"json.decoders/0","doc":"","ref":"json.html#t:decoders/0"},{"type":"function","title":"json.encode/1","doc":"Generates JSON corresponding to `Term`.\n\nSupports basic data mapping:\n\n| **Erlang**             | **JSON** |\n|------------------------|----------|\n| `integer() \\| float()` | Number   |\n| `true \\| false `       | Boolean  |\n| `null`                 | Null     |\n| `binary()`             | String   |\n| `atom()`               | String   |\n| `list()`               | Array    |\n| `#{binary() => _}`     | Object   |\n| `#{atom() => _}`       | Object   |\n| `#{integer() => _}`    | Object   |\n\nThis is equivalent to `encode(Term, fun json:encode_value/2)`.","ref":"json.html#encode/1"},{"type":"function","title":"Examples - json.encode/1","doc":"```erlang\n> iolist_to_binary(json:encode(#{foo => <<\"bar\">>})).\n<<\"{\\\"foo\\\":\\\"bar\\\"}\">>\n```","ref":"json.html#encode/1-examples"},{"type":"function","title":"json.encode/2","doc":"Generates JSON corresponding to `Term`.\n\nCan be customised with the `Encoder` callback.\nThe callback will be recursively called for all the data\nto be encoded and is expected to return the corresponding\nencoded JSON as iodata.\n\nVarious `encode_*` functions in this module can be used\nto help in constructing such callbacks.","ref":"json.html#encode/2"},{"type":"function","title":"Examples - json.encode/2","doc":"An encoder that uses a heuristic to differentiate object-like\nlists of key-value pairs from plain lists:\n\n```erlang\n> encoder([{_, _} | _] = Value, Encode) -> json:encode_key_value_list(Value, Encode);\n> encoder(Other, Encode) -> json:encode_value(Other, Encode).\n> custom_encode(Value) -> json:encode(Value, fun(Value, Encode) -> encoder(Value, Encode) end).\n> iolist_to_binary(custom_encode([{a, []}, {b, 1}])).\n<<\"{\\\"a\\\":[],\\\"b\\\":1}\">>\n```","ref":"json.html#encode/2-examples"},{"type":"function","title":"json.encode_atom/2","doc":"Default encoder for atoms used by `json:encode/1`.\n\nEncodes the atom `null` as JSON `null`,\natoms `true` and `false` as JSON booleans,\nand everything else as JSON strings calling the `Encode`\ncallback with the corresponding binary.","ref":"json.html#encode_atom/2"},{"type":"function","title":"json.encode_binary/1","doc":"Default encoder for binaries as JSON strings used by `json:encode/1`.","ref":"json.html#encode_binary/1"},{"type":"function","title":"Errors - json.encode_binary/1","doc":"* `error(unexpected_end)` if the binary contains incomplete UTF-8 sequences.\n* `error({invalid_byte, Byte})` if the binary contains invalid UTF-8 sequences.","ref":"json.html#encode_binary/1-errors"},{"type":"function","title":"json.encode_binary_escape_all/1","doc":"Encoder for binaries as JSON strings producing pure-ASCII JSON.\n\nFor any non-ASCII unicode character, a corresponding `\\\\uXXXX` sequence is used.","ref":"json.html#encode_binary_escape_all/1"},{"type":"function","title":"Errors - json.encode_binary_escape_all/1","doc":"* `error(unexpected_end)` if the binary contains incomplete UTF-8 sequences.\n* `error({invalid_byte, Byte})` if the binary contains invalid UTF-8 sequences.","ref":"json.html#encode_binary_escape_all/1-errors"},{"type":"function","title":"json.encode_float/1","doc":"Default encoder for floats as JSON numbers used by `json:encode/1`.","ref":"json.html#encode_float/1"},{"type":"function","title":"json.encode_integer/1","doc":"Default encoder for integers as JSON numbers used by `json:encode/1`.","ref":"json.html#encode_integer/1"},{"type":"function","title":"json.encode_key_value_list/2","doc":"Encoder for lists of key-value pairs as JSON objects.\n\nAccepts lists with atom, binary, integer, or float keys.","ref":"json.html#encode_key_value_list/2"},{"type":"function","title":"json.encode_key_value_list_checked/2","doc":"Encoder for lists of key-value pairs as JSON objects.\n\nAccepts lists with atom, binary, integer, or float keys.\nVerifies that no duplicate keys will be produced in the\nresulting JSON object.","ref":"json.html#encode_key_value_list_checked/2"},{"type":"function","title":"Errors - json.encode_key_value_list_checked/2","doc":"Raises `error({duplicate_key, Key})` if there are duplicates.","ref":"json.html#encode_key_value_list_checked/2-errors"},{"type":"function","title":"json.encode_list/2","doc":"Default encoder for lists as JSON arrays used by `json:encode/1`.","ref":"json.html#encode_list/2"},{"type":"type","title":"json.encode_map/1","doc":"","ref":"json.html#t:encode_map/1"},{"type":"function","title":"json.encode_map/2","doc":"Default encoder for maps as JSON objects used by `json:encode/1`.\n\nAccepts maps with atom, binary, integer, or float keys.","ref":"json.html#encode_map/2"},{"type":"function","title":"json.encode_map_checked/2","doc":"Encoder for maps as JSON objects.\n\nAccepts maps with atom, binary, integer, or float keys.\nVerifies that no duplicate keys will be produced in the\nresulting JSON object.","ref":"json.html#encode_map_checked/2"},{"type":"function","title":"Errors - json.encode_map_checked/2","doc":"Raises `error({duplicate_key, Key})` if there are duplicates.","ref":"json.html#encode_map_checked/2-errors"},{"type":"type","title":"json.encode_value/0","doc":"Simple JSON value encodeable with `json:encode/1`.","ref":"json.html#t:encode_value/0"},{"type":"function","title":"json.encode_value/2","doc":"Default encoder used by `json:encode/1`.\n\nRecursively calls `Encode` on all the values in `Value`.","ref":"json.html#encode_value/2"},{"type":"type","title":"json.encoder/0","doc":"","ref":"json.html#t:encoder/0"},{"type":"function","title":"json.format/1","doc":"Generates formatted JSON corresponding to `Term`.\n\nSimiliar to `encode/1` but with added whitespaces for formatting.\n\n```erlang\n> io:put_chars(json:format(#{foo => <<\"bar\">>, baz => 52})).\n{\n  \"baz\": 52,\n  \"foo\": \"bar\"\n}\nok\n```","ref":"json.html#format/1"},{"type":"function","title":"json.format/2","doc":"Generates formatted JSON corresponding to `Term`.\n\nEquivalent to `format(Term, fun json:format_value/3, Options)` or `format(Term, Encoder, #{})`","ref":"json.html#format/2"},{"type":"function","title":"json.format/3","doc":"Generates formatted JSON corresponding to `Term`.\n\nSimilar to `encode/2`, can be customised with the `Encoder` callback and `Options`.\n\n`Options` can include 'indent' to specify number of spaces per level and 'max' which loosely limits\nthe width of lists.\n\nThe `Encoder` will get a 'State' argument which contains the 'Options' maps merged with other data\nwhen recursing through 'Term'.\n\n`format_value/3` or various `encode_*` functions in this module can be used\nto help in constructing such callbacks.\n\n```erlang\n> formatter({posix_time, SysTimeSecs}, Encode, State) ->\n    TimeStr = calendar:system_time_to_rfc3339(SysTimeSecs, [{offset, \"Z\"}]),\n    json:format_value(unicode:characters_to_binary(TimeStr), Encode, State);\n> formatter(Other, Encode, State) -> json:format_value(Other, Encode, State).\n>\n> Fun = fun(Value, Encode, State) -> formatter(Value, Encode, State) end.\n> Options = #{indent => 4}.\n> Term = #{id => 1, time => {posix_time, erlang:system_time(seconds)}}.\n>\n> io:put_chars(json:format(Term, Fun, Options)).\n{\n    \"id\": 1,\n    \"time\": \"2024-05-23T16:07:48Z\"\n}\nok\n```","ref":"json.html#format/3"},{"type":"function","title":"json.format_key_value_list/3","doc":"Format function for lists of key-value pairs as JSON objects.\n\nAccepts lists with atom, binary, integer, or float keys.","ref":"json.html#format_key_value_list/3"},{"type":"function","title":"json.format_key_value_list_checked/3","doc":"Format function for lists of key-value pairs as JSON objects.\n\nAccepts lists with atom, binary, integer, or float keys.\nVerifies that no duplicate keys will be produced in the\nresulting JSON object.","ref":"json.html#format_key_value_list_checked/3"},{"type":"function","title":"Errors - json.format_key_value_list_checked/3","doc":"Raises `error({duplicate_key, Key})` if there are duplicates.","ref":"json.html#format_key_value_list_checked/3-errors"},{"type":"function","title":"json.format_value/3","doc":"Default format function used by `json:format/1`.\n\nRecursively calls `Encode` on all the values in `Value`,\nand indents objects and lists.","ref":"json.html#format_value/3"},{"type":"type","title":"json.formatter/0","doc":"","ref":"json.html#t:formatter/0"},{"type":"type","title":"json.from_binary_fun/0","doc":"","ref":"json.html#t:from_binary_fun/0"},{"type":"type","title":"json.object_finish_fun/0","doc":"","ref":"json.html#t:object_finish_fun/0"},{"type":"type","title":"json.object_push_fun/0","doc":"","ref":"json.html#t:object_push_fun/0"},{"type":"type","title":"json.object_start_fun/0","doc":"","ref":"json.html#t:object_start_fun/0"},{"type":"module","title":"orddict","doc":"Key-value dictionary as ordered list.\n\nThis module provides a `Key`-`Value` dictionary. An `orddict` is a\nrepresentation of a dictionary, where a list of pairs is used to store the keys\nand values. The list is ordered after the keys in the\n[Erlang term order](`e:system:expressions.md#term-comparisons`).\n\nThis module provides the same interface as the `m:dict` module but with a\ndefined representation. One difference is that while `dict` considers two keys\nas different if they do not match (`=:=`), this module considers two keys as\ndifferent if and only if they do not compare equal (`==`).","ref":"orddict.html"},{"type":"module","title":"Notes - orddict","doc":"[](){: #notes }\n\nFunctions [`append/3`](`append/3`) and [`append_list/3`](`append_list/3`) are\nincluded so that keyed values can be stored in a list _accumulator_, for\nexample:\n\n```erlang\n> D0 = orddict:new(),\n  D1 = orddict:store(files, [], D0),\n  D2 = orddict:append(files, f1, D1),\n  D3 = orddict:append(files, f2, D2),\n  D4 = orddict:append(files, f3, D3),\n  orddict:fetch(files, D4).\n[f1,f2,f3]\n```\n\nThis saves the trouble of first fetching a keyed value, appending a new value to\nthe list of stored values, and storing the result.\n\nFunction [`fetch/2`](`fetch/2`) is to be used if the key is known to be in the\ndictionary, otherwise function [`find/2`](`find/2`).","ref":"orddict.html#module-notes"},{"type":"module","title":"See Also - orddict","doc":"`m:dict`, `m:gb_trees`","ref":"orddict.html#module-see-also"},{"type":"function","title":"orddict.append/3","doc":"Appends a new `Value` to the current list of values associated with `Key`. An\nexception is generated if the initial value associated with `Key` is not a list\nof values.\n\nSee also section [Notes](`m:orddict#module-notes`).\n\n_Example 1:_\n\n```erlang\n1> OrdDict1 = orddict:from_list([{x, []}]).\n[{x,[]}]\n2> OrdDict2 = orddict:append(x, 1, OrdDict1).\n[{x,[1]}]\n3> OrdDict3 = orddict:append(x, 2, OrdDict2).\n[{x,[1,2]}]\n4> orddict:append(y, 3, OrdDict3).\n[{x,[1,2]},{y,[3]}]\n```\n\n_Example 2:_\n\n```erlang\n1> OrdDict1 = orddict:from_list([{a, no_list}]).\n[{a,no_list}]\n2> orddict:append(a, 1, OrdDict1).\n** exception error: bad argument\n     in operator  ++/2\n        called as no_list ++ [1]\n```","ref":"orddict.html#append/3"},{"type":"function","title":"orddict.append_list/3","doc":"Appends a list of values `ValList` to the current list of values associated with\n`Key`. An exception is generated if the initial value associated with `Key` is\nnot a list of values.\n\nSee also section [Notes](`m:orddict#module-notes`).\n\n_Example:_\n\n```erlang\n1> OrdDict1 = orddict:from_list([{x, []}]).\n[{x,[]}]\n2> OrdDict2 = orddict:append_list(x, [1,2], OrdDict1).\n[{x,[1,2]}]\n3> OrdDict3 = orddict:append_list(y, [3,4], OrdDict2).\n[{x,[1,2]},{y,[3,4]}]\n```","ref":"orddict.html#append_list/3"},{"type":"function","title":"orddict.erase/2","doc":"Erases all items with a specified key from a dictionary.\n\n_Example:_\n\n```erlang\n1> OrdDict1 = orddict:from_list([{a, 1}, {b, 2}]).\n[{a,1},{b,2}]\n2> orddict:erase(a, OrdDict1).\n[{b,2}]\n```","ref":"orddict.html#erase/2"},{"type":"function","title":"orddict.fetch/2","doc":"Returns the value associated with `Key` in dictionary `Orddict`. This function\nassumes that the `Key` is present in the dictionary. An exception is generated\nif `Key` is not in the dictionary.\n\nSee also section [Notes](`m:orddict#module-notes`).\n\n_Example:_\n\n```erlang\n1> OrdDict1 = orddict:from_list([{a, 1}, {b, 2}]).\n[{a,1},{b,2}]\n2> orddict:fetch(a, OrdDict1).\n1\n3> orddict:fetch(missing, OrdDict1).\n** exception error: no function clause matching orddict:fetch(missing,[])\n```","ref":"orddict.html#fetch/2"},{"type":"function","title":"orddict.fetch_keys/1","doc":"Returns a list of all keys in a dictionary.\n\n_Example:_\n\n```erlang\n1> OrdDict1 = orddict:from_list([{a, 1}, {b, 2}]).\n[{a,1},{b,2}]\n2> orddict:fetch_keys(OrdDict1).\n[a,b]\n```","ref":"orddict.html#fetch_keys/1"},{"type":"function","title":"orddict.filter/2","doc":"`Orddict2` is a dictionary of all keys and values in `Orddict1` for which\n`Pred(Key, Value)` is `true`.\n\n_Example:_\n\n```erlang\n1> OrdDict1 = orddict:from_list([{a, 1}, {b, 2}]).\n[{a,1},{b,2}]\n2> orddict:filter(fun (K, V) -> V > 1 end, OrdDict1).\n[{b,2}]\n```","ref":"orddict.html#filter/2"},{"type":"function","title":"orddict.find/2","doc":"Searches for a key in a dictionary. Returns `{ok, Value}`, where `Value` is the\nvalue associated with `Key`, or `error` if the key is not present in the\ndictionary.\n\nSee also section [Notes](`m:orddict#module-notes`).\n\n_Example:_\n\n```erlang\n1> OrdDict1 = orddict:from_list([{a, 1}, {b, 2}]).\n[{a,1},{b,2}]\n2> orddict:find(a, OrdDict1).\n{ok,1}\n3> orddict:find(c, OrdDict1).\nerror\n```","ref":"orddict.html#find/2"},{"type":"function","title":"orddict.fold/3","doc":"Calls `Fun` on successive keys and values of `Orddict` together with an extra\nargument `Acc` (short for accumulator). `Fun` must return a new accumulator that\nis passed to the next call. `Acc0` is returned if the list is empty.\n\n_Example:_\n\n```erlang\n1> OrdDict1 = orddict:from_list([{a, 1}, {b, 2}]).\n[{a,1},{b,2}]\n2> orddict:fold(fun (K, V, Acc) -> [{K, V+100} | Acc] end, [], OrdDict1).\n[{b,102},{a,101}]\n```","ref":"orddict.html#fold/3"},{"type":"function","title":"orddict.from_list/1","doc":"Converts the `Key`-`Value` list `List` to a dictionary.","ref":"orddict.html#from_list/1"},{"type":"function","title":"orddict.is_empty/1","doc":"Returns `true` if `Orddict` has no elements, otherwise `false`.","ref":"orddict.html#is_empty/1"},{"type":"function","title":"orddict.is_key/2","doc":"Tests if `Key` is contained in dictionary `Orddict`.","ref":"orddict.html#is_key/2"},{"type":"function","title":"orddict.map/2","doc":"Calls `Fun` on successive keys and values of `Orddict1` to return a new value\nfor each key.\n\n_Example:_\n\n```erlang\n1> OrdDict1 = orddict:from_list([{a, 1}, {b, 2}]).\n[{a,1},{b,2}]\n2> orddict:map(fun (_K, V) -> V + 100 end, OrdDict1).\n[{a,101},{b,102}]\n```","ref":"orddict.html#map/2"},{"type":"function","title":"orddict.merge/3","doc":"Merges two dictionaries, `Orddict1` and `Orddict2`, to create a new dictionary.\nAll the `Key`-`Value` pairs from both dictionaries are included in the new\ndictionary.\n\nIf a key occurs in both dictionaries, `Fun` is called with the key\nand both values to return a new value.\n\n[`merge/3`](`merge/3`) can be defined as follows, but is faster:\n\n```erlang\nmerge(Fun, D1, D2) ->\n    fold(fun (K, V1, D) ->\n                 update(K, fun (V2) -> Fun(K, V1, V2) end, V1, D)\n         end, D2, D1).\n```\n\n_Example:_\n\n```erlang\n1> OrdDict1 = orddict:from_list([{a, 1}, {b, 2}]).\n[{a,1},{b,2}]\n2> OrdDict2 = orddict:from_list([{b, 7}, {c, 8}]).\n[{b,7},{c,8}]\n3> orddict:merge(fun (K, V1, V2) -> V1 * V2 end, OrdDict1, OrdDict2).\n[{a,1},{b,14},{c,8}]\n```","ref":"orddict.html#merge/3"},{"type":"function","title":"orddict.new/0","doc":"Creates a new dictionary.","ref":"orddict.html#new/0"},{"type":"type","title":"orddict.orddict/0","doc":"","ref":"orddict.html#t:orddict/0"},{"type":"type","title":"orddict.orddict/2","doc":"Dictionary as returned by `new/0`.","ref":"orddict.html#t:orddict/2"},{"type":"function","title":"orddict.size/1","doc":"Returns the number of elements in an `Orddict`.","ref":"orddict.html#size/1"},{"type":"function","title":"orddict.store/3","doc":"Stores a `Key`-`Value` pair in a dictionary. If the `Key` already exists in\n`Orddict1`, the associated value is replaced by `Value`.\n\n_Example:_\n\n```erlang\n1> OrdDict1 = orddict:from_list([{a, 1}, {b, 2}]).\n[{a,1},{b,2}]\n2> orddict:store(a, 99, OrdDict1).\n[{a,99},{b,2}]\n3> orddict:store(c, 100, OrdDict1).\n[{a,1},{b,2},{c,100}]\n```","ref":"orddict.html#store/3"},{"type":"function","title":"orddict.take/2","doc":"This function returns value from dictionary and new dictionary without this\nvalue. Returns `error` if the key is not present in the dictionary.\n\n_Example:_\n\n```erlang\n1> OrdDict1 = orddict:from_list([{a, 1}, {b, 2}]).\n[{a,1},{b,2}]\n2> orddict:take(a, OrdDict1).\n{1,[{b,2}]}\n3> orddict:take(missing, OrdDict1).\nerror\n```","ref":"orddict.html#take/2"},{"type":"function","title":"orddict.to_list/1","doc":"Converts a dictionary to a list representation.","ref":"orddict.html#to_list/1"},{"type":"function","title":"orddict.update/3","doc":"Updates a value in a dictionary by calling `Fun` on the value to get a new\nvalue. An exception is generated if `Key` is not present in the dictionary.\n\n_Example:_\n\n```erlang\n1> OrdDict1 = orddict:from_list([{a, 1}, {b, 2}]).\n[{a,1},{b,2}]\n2> orddict:update(a, fun (V) -> V + 100 end, OrdDict1).\n[{a,101},{b,2}]\n```","ref":"orddict.html#update/3"},{"type":"function","title":"orddict.update/4","doc":"Updates a value in a dictionary by calling `Fun` on the value to get a new\nvalue. If `Key` is not present in the dictionary, `Initial` is stored as the\nfirst value.\n\nFor example, [`append/3`](`append/3`) can be defined as follows:\n\n```erlang\nappend(Key, Val, D) ->\n    update(Key, fun (Old) -> Old ++ [Val] end, [Val], D).\n```\n\n_Example 1:_\n\n```erlang\n1> OrdDict1 = orddict:from_list([{a, 1}, {b, 2}]).\n[{a,1},{b,2}]\n2> orddict:update(c, fun (V) -> V + 100 end, 99, OrdDict1).\n[{a,1},{b,2},{c,99}]\n```\n\n_Example 2:_\n\n```erlang\n1> OrdDict1 = orddict:from_list([{a, 1}, {b, 2}]).\n[{a,1},{b,2}]\n2> orddict:update(a, fun (V) -> V + 100 end, 99, OrdDict1).\n[{a,101},{b,2}]\n```","ref":"orddict.html#update/4"},{"type":"function","title":"orddict.update_counter/3","doc":"Adds `Increment` to the value associated with `Key` and store this value. If\n`Key` is not present in the dictionary, `Increment` is stored as the first\nvalue.\n\nThis can be defined as follows, but is faster:\n\n```erlang\nupdate_counter(Key, Incr, D) ->\n    update(Key, fun (Old) -> Old + Incr end, Incr, D).\n```","ref":"orddict.html#update_counter/3"},{"type":"module","title":"ordsets","doc":"Functions for manipulating sets as ordered lists.\n\nSets are collections of elements with no duplicate elements. An `ordset` is a\nrepresentation of a set, where an ordered list is used to store the elements of\nthe set. An ordered list is more efficient than an unordered list. Elements are\nordered according to the _Erlang term order_.\n\nThis module provides the same interface as the `m:sets` module but with a\ndefined representation. One difference is that while `sets` considers two\nelements as different if they do not match (`=:=`), this module considers two\nelements as different if and only if they do not compare equal (`==`).\n\nSee the [Compatibility Section in the `sets` module](`m:sets#module-compatibility`)\nfor more information about the compatibility of the different implementations of\nsets in the Standard Library.","ref":"ordsets.html"},{"type":"module","title":"See Also - ordsets","doc":"`m:gb_sets`, `m:sets`","ref":"ordsets.html#module-see-also"},{"type":"function","title":"ordsets.add_element/2","doc":"Returns a new ordered set formed from `Ordset1` with `Element` inserted.","ref":"ordsets.html#add_element/2"},{"type":"function","title":"ordsets.del_element/2","doc":"Returns `Ordset1`, but with `Element` removed.","ref":"ordsets.html#del_element/2"},{"type":"function","title":"ordsets.filter/2","doc":"Filters elements in `Ordset1` with boolean function `Pred`.","ref":"ordsets.html#filter/2"},{"type":"function","title":"ordsets.filtermap/2","doc":"Filters and maps elements in `Ordset1` with function `Fun`.","ref":"ordsets.html#filtermap/2"},{"type":"function","title":"ordsets.fold/3","doc":"Folds `Function` over every element in `Ordset` and returns the final value of\nthe accumulator.","ref":"ordsets.html#fold/3"},{"type":"function","title":"ordsets.from_list/1","doc":"Returns an ordered set of the elements in `List`.","ref":"ordsets.html#from_list/1"},{"type":"function","title":"ordsets.intersection/1","doc":"Returns the intersection of the non-empty list of sets.","ref":"ordsets.html#intersection/1"},{"type":"function","title":"ordsets.intersection/2","doc":"Returns the intersection of `Ordset1` and `Ordset2`.","ref":"ordsets.html#intersection/2"},{"type":"function","title":"ordsets.is_disjoint/2","doc":"Returns `true` if `Ordset1` and `Ordset2` are disjoint (have no elements in\ncommon), otherwise `false`.","ref":"ordsets.html#is_disjoint/2"},{"type":"function","title":"ordsets.is_element/2","doc":"Returns `true` if `Element` is an element of `Ordset`, otherwise `false`.","ref":"ordsets.html#is_element/2"},{"type":"function","title":"ordsets.is_empty/1","doc":"Returns `true` if `Ordset` is an empty set, otherwise `false`.","ref":"ordsets.html#is_empty/1"},{"type":"function","title":"ordsets.is_equal/2","doc":"Returns `true` if `Ordset1` and `Ordset2` are equal, that is when every element\nof one set is also a member of the respective other set, otherwise `false`.","ref":"ordsets.html#is_equal/2"},{"type":"function","title":"ordsets.is_set/1","doc":"Returns `true` if `Ordset` is an ordered set of elements, otherwise `false`.\nThis function will return `true` for any ordered list, even when not constructed\nby the functions in this module.","ref":"ordsets.html#is_set/1"},{"type":"function","title":"ordsets.is_subset/2","doc":"Returns `true` when every element of `Ordset1` is also a member of `Ordset2`,\notherwise `false`.","ref":"ordsets.html#is_subset/2"},{"type":"function","title":"ordsets.map/2","doc":"Maps elements in `Ordset1` with mapping function `Fun`.","ref":"ordsets.html#map/2"},{"type":"function","title":"ordsets.new/0","doc":"Returns a new empty ordered set.","ref":"ordsets.html#new/0"},{"type":"type","title":"ordsets.ordset/1","doc":"As returned by `new/0`.","ref":"ordsets.html#t:ordset/1"},{"type":"function","title":"ordsets.size/1","doc":"Returns the number of elements in `Ordset`.","ref":"ordsets.html#size/1"},{"type":"function","title":"ordsets.subtract/2","doc":"Returns only the elements of `Ordset1` that are not also elements of `Ordset2`.","ref":"ordsets.html#subtract/2"},{"type":"function","title":"ordsets.to_list/1","doc":"Returns the elements of `Ordset` as a list.","ref":"ordsets.html#to_list/1"},{"type":"function","title":"ordsets.union/1","doc":"Returns the merged (union) set of the list of sets.","ref":"ordsets.html#union/1"},{"type":"function","title":"ordsets.union/2","doc":"Returns the merged (union) set of `Ordset1` and `Ordset2`.","ref":"ordsets.html#union/2"},{"type":"module","title":"proplists","doc":"Support functions for property lists.\n\nProperty lists are ordinary lists containing entries in the form of either\ntuples, whose first elements are keys used for lookup and insertion, or atoms,\nwhich work as shorthand for tuples `{Atom, true}`. (Other terms are allowed in\nthe lists, but are ignored by this module.) If there is more than one entry in a\nlist for a certain key, the first occurrence normally overrides any later\n(irrespective of the arity of the tuples).\n\nProperty lists are useful for representing inherited properties, such as options\npassed to a function where a user can specify options overriding the default\nsettings, object properties, annotations, and so on.\n\nTwo keys are considered equal if they match (`=:=`). That is, numbers are\ncompared literally rather than by value, so that, for example, `1` and `1.0` are\ndifferent keys.","ref":"proplists.html"},{"type":"function","title":"proplists.append_values/2","doc":"Similar to `get_all_values/2`, but each value is wrapped in a list unless it is\nalready itself a list. The resulting list of lists is concatenated. This is\noften useful for \"incremental\" options.\n\n_Example:_\n\n```erlang\nappend_values(a, [{a, [1,2]}, {b, 0}, {a, 3}, {c, -1}, {a, [4]}])\n```\n\nreturns:\n\n```erlang\n[1,2,3,4]\n```","ref":"proplists.html#append_values/2"},{"type":"function","title":"proplists.compact/1","doc":"Minimizes the representation of all entries in the list. This is equivalent to\n`[property(P) || P <- ListIn]`.\n\nSee also `property/1`, `unfold/1`.","ref":"proplists.html#compact/1"},{"type":"function","title":"proplists.delete/2","doc":"Deletes all entries associated with `Key` from `List`.","ref":"proplists.html#delete/2"},{"type":"function","title":"proplists.expand/2","doc":"Expands particular properties to corresponding sets of properties (or other\nterms).\n\nFor each pair `{Property, Expansion}` in `Expansions`: if `E` is the\nfirst entry in `ListIn` with the same key as `Property`, and `E` and `Property`\nhave equivalent normal forms, then `E` is replaced with the terms in\n`Expansion`, and any following entries with the same key are deleted from\n`ListIn`.\n\nFor example, the following expressions all return `[fie, bar, baz, fum]`:\n\n```erlang\nexpand([{foo, [bar, baz]}], [fie, foo, fum])\nexpand([{{foo, true}, [bar, baz]}], [fie, foo, fum])\nexpand([{{foo, false}, [bar, baz]}], [fie, {foo, false}, fum])\n```\n\nHowever, no expansion is done in the following call because `{foo, false}`\nshadows `foo`:\n\n```erlang\nexpand([{{foo, true}, [bar, baz]}], [{foo, false}, fie, foo, fum])\n```\n\nNotice that if the original property term is to be preserved in the result when\nexpanded, it must be included in the expansion list. The inserted terms are not\nexpanded recursively. If `Expansions` contains more than one property with the\nsame key, only the first occurrence is used.\n\nSee also `normalize/2`.","ref":"proplists.html#expand/2"},{"type":"function","title":"proplists.from_map/1","doc":"Converts the map `Map` to a property list.","ref":"proplists.html#from_map/1"},{"type":"function","title":"proplists.get_all_values/2","doc":"Similar to `get_value/2`, but returns the list of values for _all_ entries\n`{Key, Value}` in `List`. If no such entry exists, the result is the empty list.","ref":"proplists.html#get_all_values/2"},{"type":"function","title":"proplists.get_bool/2","doc":"Returns the value of a boolean key/value option. If\n[`lookup(Key, List)`](`lookup/2`) would yield `{Key, true}`, this function\nreturns `true`, otherwise `false`.\n\nSee also `get_value/2`, `lookup/2`.","ref":"proplists.html#get_bool/2"},{"type":"function","title":"proplists.get_keys/1","doc":"Returns an unordered list of the keys used in `List`, not containing duplicates.","ref":"proplists.html#get_keys/1"},{"type":"function","title":"proplists.get_value/2","doc":"","ref":"proplists.html#get_value/2"},{"type":"function","title":"proplists.get_value/3","doc":"Returns the value of a simple key/value property in `List`. If\n[`lookup(Key, List)`](`lookup/2`) would yield `{Key, Value}`, this function\nreturns the corresponding `Value`, otherwise `Default`.\n\nSee also `get_all_values/2`, `get_bool/2`, `get_value/2`, `lookup/2`.","ref":"proplists.html#get_value/3"},{"type":"function","title":"proplists.is_defined/2","doc":"Returns `true` if `List` contains at least one entry associated with `Key`,\notherwise `false`.","ref":"proplists.html#is_defined/2"},{"type":"function","title":"proplists.lookup/2","doc":"Returns the first entry associated with `Key` in `List`, if one exists,\notherwise returns `none`. For an atom `A` in the list, the tuple `{A, true}` is\nthe entry associated with `A`.\n\nSee also `get_bool/2`, `get_value/2`, `lookup_all/2`.","ref":"proplists.html#lookup/2"},{"type":"function","title":"proplists.lookup_all/2","doc":"Returns the list of all entries associated with `Key` in `List`. If no such\nentry exists, the result is the empty list.\n\nSee also `lookup/2`.","ref":"proplists.html#lookup_all/2"},{"type":"function","title":"proplists.normalize/2","doc":"Passes `ListIn` through a sequence of substitution/expansion stages. For an\n`aliases` operation, function `substitute_aliases/2` is applied using the\nspecified list of aliases:\n\n- For a `negations` operation,\n  [`substitute_negations/2`](`substitute_negations/2`) is applied using the\n  specified negation list.\n- For an `expand` operation, function `expand/2` is applied using the specified\n  list of expansions.\n\nThe final result is automatically compacted (compare `compact/1`).\n\nTypically you want to substitute negations first, then aliases, then perform one\nor more expansions (sometimes you want to pre-expand particular entries before\ndoing the main expansion). You might want to substitute negations and/or aliases\nrepeatedly, to allow such forms in the right-hand side of aliases and expansion\nlists.\n\nSee also `substitute_negations/2`.","ref":"proplists.html#normalize/2"},{"type":"type","title":"proplists.property/0","doc":"A property item within a list","ref":"proplists.html#t:property/0"},{"type":"function","title":"proplists.property/1","doc":"Creates a normal form (minimal) representation of a property. If `PropertyIn` is\n`{Key, true}`, where `Key` is an atom, `Key` is returned, otherwise the whole\nterm `PropertyIn` is returned.\n\nSee also `property/2`.","ref":"proplists.html#property/1"},{"type":"function","title":"proplists.property/2","doc":"Creates a normal form (minimal) representation of a simple key/value property.\nReturns `Key` if `Value` is `true` and `Key` is an atom, otherwise a tuple\n`{Key, Value}` is returned.\n\nSee also `property/1`.","ref":"proplists.html#property/2"},{"type":"type","title":"proplists.proplist/0","doc":"A list of `t:property/0`, also knows as a proplist.","ref":"proplists.html#t:proplist/0"},{"type":"function","title":"proplists.split/2","doc":"Partitions `List` into a list of sublists and a remainder.\n\n`Lists` contains one sublist for each key in `Keys`, in the corresponding order.\nThe relative order of the elements in each sublist is preserved from the original `List`.\n`Rest` contains the elements in `List` that are not associated with any of the\nspecified keys, also with their original relative order preserved.\n\n_Example:_\n\n```erlang\nsplit([{c, 2}, {e, 1}, a, {c, 3, 4}, d, {b, 5}, b], [a, b, c])\n```\n\nreturns:\n\n```erlang\n{[[a], [{b, 5}, b],[{c, 2}, {c, 3, 4}]], [{e, 1}, d]}\n```","ref":"proplists.html#split/2"},{"type":"function","title":"proplists.substitute_aliases/2","doc":"Substitutes keys of properties. For each entry in `ListIn`, if it is associated\nwith some key `K1` such that `{K1, K2}` occurs in `Aliases`, the key of the\nentry is changed to `K2`. If the same `K1` occurs more than once in `Aliases`,\nonly the first occurrence is used.\n\nFor example,\n[`substitute_aliases([{color, colour}], L)`](`substitute_aliases/2`) replaces\nall tuples `{color, ...}` in `L` with `{colour, ...}`, and all atoms `color`\nwith `colour`.\n\nSee also `normalize/2`, `substitute_negations/2`.","ref":"proplists.html#substitute_aliases/2"},{"type":"function","title":"proplists.substitute_negations/2","doc":"Substitutes keys of boolean-valued properties and simultaneously negates their\nvalues.\n\nFor each entry in `ListIn`, if it is associated with some key `K1` such\nthat `{K1, K2}` occurs in `Negations`: if the entry was `{K1, true}`, it is\nreplaced with `{K2, false}`, otherwise with `K2`, thus changing the name of the\noption and simultaneously negating the value specified by\n[`get_bool(Key, ListIn)`](`get_bool/2`). If the same `K1` occurs more than once\nin `Negations`, only the first occurrence is used.\n\nFor example,\n[`substitute_negations([{no_foo, foo}], L)`](`substitute_negations/2`) replaces\nany atom `no_foo` or tuple `{no_foo, true}` in `L` with `{foo, false}`, and any\nother tuple `{no_foo, ...}` with `foo`.\n\nSee also `get_bool/2`, `normalize/2`, `substitute_aliases/2`.","ref":"proplists.html#substitute_negations/2"},{"type":"function","title":"proplists.to_map/1","doc":"Converts the property list `List` to a map.\n\nShorthand atom values in `List` will be expanded to an association of the form\n`Atom => true`. Tuples of the form `{Key, Value}` in `List` will be converted to\nan association of the form `Key => Value`. Anything else will be silently\nignored.\n\nIf the same key appears in `List` multiple times, the value of the one appearing\nnearest to the head of `List` will be in the result map, that is the value that\nwould be returned by a call to [`get_value(Key, List)`](`get_value/2`).\n\n_Example:_\n\n```erlang\nto_map([a, {b, 1}, {c, 2}, {c, 3}])\n```\n\nreturns:\n\n```erlang\n#{a => true, b => 1, c => 2}\n```","ref":"proplists.html#to_map/1"},{"type":"function","title":"proplists.to_map/2","doc":"Converts the property list `List` to a map after applying the normalizations\ngiven in `Stages`.\n\nSee also `normalize/2`, `to_map/1`.","ref":"proplists.html#to_map/2"},{"type":"function","title":"proplists.unfold/1","doc":"Unfolds all occurrences of atoms in `ListIn` to tuples `{Atom, true}`.\n\nSee also `compact/1`.","ref":"proplists.html#unfold/1"},{"type":"module","title":"qlc","doc":"This module provides a query interface to [Mnesia](`m:mnesia`), [ETS](`m:ets`),\n[Dets](`m:dets`), and other data structures that provide an iterator style\ntraversal of objects.","ref":"qlc.html"},{"type":"module","title":"Overview - qlc","doc":"This module provides a query interface to _QLC tables_. Typical QLC tables are\nMnesia, ETS, and Dets tables. Support is also provided for user-defined tables,\nsee section [Implementing a QLC Table](`m:qlc#implementing_a_qlc_table`). [](){:\n#query_list_comprehension } A _query_ is expressed using _Query List\nComprehensions_ (QLCs). The answers to a query are determined by data in QLC\ntables that fulfill the constraints expressed by the QLCs of the query. QLCs are\nsimilar to ordinary list comprehensions as described in\n[Erlang Reference Manual](`e:system:expressions.md#lcs`) and\n[Programming Examples](`e:system:list_comprehensions.md`), except that variables\nintroduced in patterns cannot be used in list expressions. In the absence of\noptimizations and options such as `cache` and `unique` (see section\n[Common Options](`m:qlc#common_options`), every QLC free of QLC tables evaluates\nto the same list of answers as the identical ordinary list comprehension.\n\nWhile ordinary list comprehensions evaluate to lists, calling [`q/1,2`](`q/1`)\nreturns a _query handle_{: #query_handle }. To obtain all the answers to a\nquery, [`eval/1,2`](`eval/1`) is to be called with the query handle as first\nargument. Query handles are essentially functional objects (funs) created in the\nmodule calling `q/1,2`. As the funs refer to the module code, be careful not to\nkeep query handles too long if the module code is to be replaced. Code\nreplacement is described in section\n[Compilation and Code Loading](`e:system:code_loading.md`) in the Erlang\nReference Manual. The list of answers can also be traversed in chunks by use of\na _query cursor_{: #query_cursor }. Query cursors are created by calling\n[`cursor/1,2`](`cursor/1`) with a query handle as first argument. Query cursors\nare essentially Erlang processes. One answer at a time is sent from the query\ncursor process to the process that created the cursor.","ref":"qlc.html#module-overview"},{"type":"module","title":"Syntax - qlc","doc":"Syntactically QLCs have the same parts as ordinary list comprehensions:\n\n```text\n[Expression || Qualifier1, Qualifier2, ...]\n```\n\n`Expression` (the _template_) is any Erlang expression. Qualifiers are either\n_filters_ or _generators_. Filters are Erlang expressions returning\n`t:boolean/0`. Generators have the form `Pattern <- ListExpression`, where\n`ListExpression` is an expression evaluating to a query handle or a list. Query\nhandles are returned from [`append/1,2`](`append/1`),\n[`keysort/2,3`](`keysort/2`), [`q/1,2`](`q/1`), [`sort/1,2`](`sort/1`),\n[`string_to_handle/1,2,3`](`string_to_handle/1`), and `table/2`.","ref":"qlc.html#module-syntax"},{"type":"module","title":"Evaluation - qlc","doc":"A query handle is evaluated in the following order:\n\n- Inspection of options and the collection of information about tables. As a\n  result, qualifiers are modified during the optimization phase.\n- All list expressions are evaluated. If a cursor has been created, evaluation\n  takes place in the cursor process. For list expressions that are QLCs, the\n  list expressions of the generators of the QLCs are evaluated as well. Be\n  careful if list expressions have side effects, as list expressions are\n  evaluated in unspecified order.\n- The answers are found by evaluating the qualifiers from left to right,\n  backtracking when some filter returns `false`, or collecting the template when\n  all filters return `true`.\n\nFilters that do not return `t:boolean/0` but fail are handled differently\ndepending on their syntax: if the filter is a guard, it returns `false`,\notherwise the query evaluation fails. This behavior makes it possible for the\n`qlc` module to do some optimizations without affecting the meaning of a query.\nFor example, when testing some position of a table and one or more constants for\nequality, only the objects with equal values are candidates for further\nevaluation. The other objects are guaranteed to make the filter return `false`,\nbut never fail. The (small) set of candidate objects can often be found by\nlooking up some key values of the table or by traversing the table using a match\nspecification. It is necessary to place the guard filters immediately after the\ntable generator, otherwise the candidate objects are not restricted to a small\nset. The reason is that objects that could make the query evaluation fail must\nnot be excluded by looking up a key or running a match specification.","ref":"qlc.html#module-evaluation"},{"type":"module","title":"Join - qlc","doc":"The `qlc` module supports fast join of two query handles. Fast join is possible\nif some position `P1` of one query handler and some position `P2` of another\nquery handler are tested for equality. Two fast join methods are provided:\n\n- _Lookup join_ traverses all objects of one query handle and finds objects of\n  the other handle (a QLC table) such that the values at `P1` and `P2` match or\n  compare equal. The `qlc` module does not create any indexes but looks up\n  values using the key position and the indexed positions of the QLC table.\n- _Merge join_ sorts the objects of each query handle if necessary and filters\n  out objects where the values at `P1` and `P2` do not compare equal. If many\n  objects with the same value of `P2` exist, a temporary file is used for the\n  equivalence classes.\n\nThe `qlc` module warns at compile time if a QLC combines query handles in such a\nway that more than one join is possible. That is, no query planner is provided\nthat can select a good order between possible join operations. It is up to the\nuser to order the joins by introducing query handles.\n\nThe join is to be expressed as a guard filter. The filter must be placed\nimmediately after the two joined generators, possibly after guard filters that\nuse variables from no other generators but the two joined generators. The `qlc`\nmodule inspects the operands of `=:=/2`, `==/2`, [`is_record/2`](`is_record/2`),\n[`element/2`](`element/2`), and logical operators (`and/2`, `or/2`, `andalso/2`,\n`orelse/2`, `xor/2`) when determining which joins to consider.\n\n[](){: #common_options }","ref":"qlc.html#module-join"},{"type":"module","title":"Common Options - qlc","doc":"The following options are accepted by `cursor/2`, `eval/2`, `fold/4`, and\n`info/2`:\n\n- `{cache_all, Cache}`, where `Cache` is equal to `ets` or `list` adds a\n  `{cache, Cache}` option to every list expression of the query except tables\n  and lists. Defaults to `{cache_all, no}`. Option `cache_all` is equivalent to\n  `{cache_all, ets}`.\n- `{max_list_size, MaxListSize}`{: #max_list_size }, where `MaxListSize` is the\n  size in bytes of terms on the external format. If the accumulated size of\n  collected objects exceeds `MaxListSize`, the objects are written onto a\n  temporary file. This option is used by option `{cache, list}` and by the merge\n  join method. Defaults to 512\\*1024 bytes.\n- `{tmpdir_usage, TmpFileUsage}` determines the action taken when `qlc` is about\n  to create temporary files on the directory set by option `tmpdir`. If the\n  value is `not_allowed`, an error tuple is returned, otherwise temporary files\n  are created as needed. Default is `allowed`, which means that no further\n  action is taken. The values `info_msg`, `warning_msg`, and `error_msg` mean\n  that the function with the corresponding name in module `m:error_logger` is\n  called for printing some information (currently the stacktrace).\n- `{tmpdir, TempDirectory}` sets the directory used by merge join for temporary\n  files and by option `{cache, list}`. The option also overrides option `tmpdir`\n  of `keysort/3` and `sort/2`. Defaults to `\"\"`, which means that the directory\n  returned by `file:get_cwd()` is used.\n- `{unique_all, true}` adds a `{unique, true}` option to every list expression\n  of the query. Defaults to `{unique_all, false}`. Option `unique_all` is\n  equivalent to `{unique_all, true}`.\n\n[](){: #getting_started }","ref":"qlc.html#module-common-options"},{"type":"module","title":"Getting Started - qlc","doc":"As mentioned earlier, queries are expressed in the list comprehension syntax as\ndescribed in section [Expressions](`e:system:expressions.md`) in Erlang\nReference Manual. In the following, some familiarity with list comprehensions is\nassumed. The examples in section\n[List Comprehensions](`e:system:list_comprehensions.md`) in Programming Examples\ncan get you started. Notice that list comprehensions do not add any\ncomputational power to the language; anything that can be done with list\ncomprehensions can also be done without them. But they add syntax for expressing\nsimple search problems, which is compact and clear once you get used to it.\n\nMany list comprehension expressions can be evaluated by the `qlc` module.\nExceptions are expressions, such that variables introduced in patterns (or\nfilters) are used in some generator later in the list comprehension. As an\nexample, consider an implementation of `lists:append(L)`:\n`[X ||Y <- L, X <- Y]`. `Y` is introduced in the first generator and used in the\nsecond. The ordinary list comprehension is normally to be preferred when there\nis a choice as to which to use. One difference is that [`eval/1,2`](`eval/1`)\ncollects answers in a list that is finally reversed, while list comprehensions\ncollect answers on the stack that is finally unwound.\n\nWhat the `qlc` module primarily adds to list comprehensions is that data can be\nread from QLC tables in small chunks. A QLC table is created by calling\n[`qlc:table/2`](`table/2`). Usually `qlc:table/2` is not called directly from\nthe query but through an interface function of some data structure. Erlang/OTP\nincludes a few examples of such functions:\n[`mnesia:table/1,2`](`mnesia:table/1`), [`ets:table/1,2`](`ets:table/1`), and\n[`dets:table/1,2`](`dets:table/1`). For a given data structure, many functions\ncan create QLC tables, but common for these functions is that they return a\nquery handle created by [`qlc:table/2`](`table/2`). Using the QLC tables\nprovided by Erlang/OTP is usually probably sufficient, but for the more advanced\nuser section [Implementing a QLC Table](`m:qlc#implementing_a_qlc_table`)\ndescribes the implementation of a function calling `qlc:table/2`.\n\nBesides `qlc:table/2`, other functions return query handles. They are used more\nseldom than tables, but are sometimes useful. [`qlc:append/1,2`](`append/1`)\ntraverses objects from many tables or lists after each other. If, for example,\nyou want to traverse all answers to a query `QH` and then finish off by a term\n`{finished}`, you can do that by calling `qlc:append(QH, [{finished}])`.\n[`append/2`](`append/2`) first returns all objects of `QH`, then `{finished}`.\nIf a tuple `{finished}` exists among the answers to `QH`, it is returned twice\nfrom [`append/2`](`append/2`).\n\nAs another example, consider concatenating the answers to two queries `QH1` and\n`QH2` while removing all duplicates. This is accomplished by using option\n`unique`:\n\n```erlang\nqlc:q([X || X <- qlc:append(QH1, QH2)], {unique, true})\n```\n\nThe cost is substantial: every returned answer is stored in an ETS table. Before\nreturning an answer, it is looked up in the ETS table to check if it has already\nbeen returned. Without the `unique` option, all answers to `QH1` would be\nreturned followed by all answers to `QH2`. The `unique` option keeps the order\nbetween the remaining answers.\n\nIf the order of the answers is not important, there is an alternative to the\n`unique` option, namely to sort the answers uniquely:\n\n```erlang\nqlc:sort(qlc:q([X || X <- qlc:append(QH1, QH2)], {unique, true})).\n```\n\nThis query also removes duplicates but the answers are sorted. If there are many\nanswers, temporary files are used. Notice that to get the first unique answer,\nall answers must be found and sorted. Both alternatives find duplicates by\ncomparing answers, that is, if `A1` and `A2` are answers found in that order,\nthen `A2` is a removed if `A1 == A2`.\n\nTo return only a few answers, cursors can be used. The following code returns no\nmore than five answers using an ETS table for storing the unique answers:\n\n```erlang\nC = qlc:cursor(qlc:q([X || X <- qlc:append(QH1, QH2)],{unique,true})),\nR = qlc:next_answers(C, 5),\nok = qlc:delete_cursor(C),\nR.\n```\n\nQLCs are convenient for stating constraints on data from two or more tables. The\nfollowing example does a natural join on two query handles on position 2:\n\n```erlang\nqlc:q([{X1,X2,X3,Y1} ||\n          {X1,X2,X3} <- QH1,\n          {Y1,Y2} <- QH2,\n          X2 =:= Y2])\n```\n\nThe `qlc` module evaluates this differently depending on the query handles `QH1`\nand `QH2`. If, for example, `X2` is matched against the key of a QLC table, the\nlookup join method traverses the objects of `QH2` while looking up key values in\nthe table. However, if not `X2` or `Y2` is matched against the key or an indexed\nposition of a QLC table, the merge join method ensures that `QH1` and `QH2` are\nboth sorted on position 2 and next do the join by traversing the objects one by\none.\n\nOption `join` can be used to force the `qlc` module to use a certain join\nmethod. For the rest of this section it is assumed that the excessively slow\njoin method called \"nested loop\" has been chosen:\n\n```erlang\nqlc:q([{X1,X2,X3,Y1} ||\n          {X1,X2,X3} <- QH1,\n          {Y1,Y2} <- QH2,\n          X2 =:= Y2],\n      {join, nested_loop})\n```\n\nIn this case the filter is applied to every possible pair of answers to `QH1`\nand `QH2`, one at a time. If there are M answers to `QH1` and N answers to\n`QH2`, the filter is run M\\*N times.\n\nIf `QH2` is a call to the function for `m:gb_trees`, as defined in section\n[Implementing a QLC Table](`m:qlc#implementing_a_qlc_table`), then\n[`gb_table:table/1` ](`m:qlc#gb_table`), the iterator for the gb-tree is\ninitiated for each answer to `QH1`. The objects of the gb-tree are then returned\none by one. This is probably the most efficient way of traversing the table in\nthat case, as it takes minimal computational power to get the following object.\nBut if `QH2` is not a table but a more complicated QLC, it can be more efficient\nto use some RAM memory for collecting the answers in a cache, particularly if\nthere are only a few answers. It must then be assumed that evaluating `QH2` has\nno side effects so that the meaning of the query does not change if `QH2` is\nevaluated only once. One way of caching the answers is to evaluate `QH2` first\nof all and substitute the list of answers for `QH2` in the query. Another way is\nto use option `cache`. It is expressed like this:\n\n```erlang\nQH2' = qlc:q([X || X <- QH2], {cache, ets})\n```\n\nor only\n\n```text\nQH2' = qlc:q([X || X <- QH2], cache)\n```\n\nThe effect of option `cache` is that when generator `QH2'` is run the first\ntime, every answer is stored in an ETS table. When the next answer of `QH1` is\ntried, answers to `QH2'` are copied from the ETS table, which is very fast. As\nfor option `unique` the cost is a possibly substantial amount of RAM memory.\n\nOption `{cache, list}` offers the possibility to store the answers in a list on\nthe process heap. This has the potential of being faster than ETS tables, as\nthere is no need to copy answers from the table. However, it can often result in\nslower evaluation because of more garbage collections of the process heap and\nincreased RAM memory consumption because of larger heaps. Another drawback with\ncache lists is that if the list size exceeds a limit, a temporary file is used.\nReading the answers from a file is much slower than copying them from an ETS\ntable. But if the available RAM memory is scarce, setting the\n[limit](`m:qlc#max_list_size`) to some low value is an alternative.\n\nOption `cache_all` can be set to `ets` or `list` when evaluating a query. It\nadds a `cache` or `{cache, list}` option to every list expression except QLC\ntables and lists on all levels of the query. This can be used for testing if\ncaching would improve efficiency at all. If the answer is yes, further testing\nis needed to pinpoint the generators that are to be cached.\n\n[](){: #implementing_a_qlc_table }","ref":"qlc.html#module-getting-started"},{"type":"module","title":"Implementing a QLC Table - qlc","doc":"As an example of how to use function `table/2`, the implementation of a QLC\ntable for the `m:gb_trees` module is given:\n\n[](){: #gb_table }\n\n```erlang\n-module(gb_table).\n\n-export([table/1]).\n\ntable(T) ->\n    TF = fun() -> qlc_next(gb_trees:next(gb_trees:iterator(T))) end,\n    InfoFun = fun(num_of_objects) -> gb_trees:size(T);\n                 (keypos) -> 1;\n                 (is_sorted_key) -> true;\n                 (is_unique_objects) -> true;\n                 (_) -> undefined\n              end,\n    LookupFun =\n        fun(1, Ks) ->\n                lists:flatmap(fun(K) ->\n                                      case gb_trees:lookup(K, T) of\n                                          {value, V} -> [{K,V}];\n                                          none -> []\n                                      end\n                              end, Ks)\n        end,\n    FormatFun =\n        fun({all, NElements, ElementFun}) ->\n                ValsS = io_lib:format(\"gb_trees:from_orddict(~w)\",\n                                      [gb_nodes(T, NElements, ElementFun)]),\n                io_lib:format(\"gb_table:table(~s)\", [ValsS]);\n           ({lookup, 1, KeyValues, _NElements, ElementFun}) ->\n                ValsS = io_lib:format(\"gb_trees:from_orddict(~w)\",\n                                      [gb_nodes(T, infinity, ElementFun)]),\n                io_lib:format(\"lists:flatmap(fun(K) -> \"\n                              \"case gb_trees:lookup(K, ~s) of \"\n                              \"{value, V} -> [{K,V}];none -> [] end \"\n                              \"end, ~w)\",\n                              [ValsS, [ElementFun(KV) || KV <- KeyValues]])\n        end,\n    qlc:table(TF, [{info_fun, InfoFun}, {format_fun, FormatFun},\n                   {lookup_fun, LookupFun},{key_equality,'=='}]).\n\nqlc_next({X, V, S}) ->\n    [{X,V} | fun() -> qlc_next(gb_trees:next(S)) end];\nqlc_next(none) ->\n    [].\n\ngb_nodes(T, infinity, ElementFun) ->\n    gb_nodes(T, -1, ElementFun);\ngb_nodes(T, NElements, ElementFun) ->\n    gb_iter(gb_trees:iterator(T), NElements, ElementFun).\n\ngb_iter(_I, 0, _EFun) ->\n    '...';\ngb_iter(I0, N, EFun) ->\n    case gb_trees:next(I0) of\n        {X, V, I} ->\n            [EFun({X,V}) | gb_iter(I, N-1, EFun)];\n        none ->\n            []\n    end.\n```\n\n`TF` is the traversal function. The `qlc` module requires that there is a way of\ntraversing all objects of the data structure. `gb_trees` has an iterator\nfunction suitable for that purpose. Notice that for each object returned, a new\nfun is created. As long as the list is not terminated by `[]`, it is assumed\nthat the tail of the list is a nullary function and that calling the function\nreturns further objects (and functions).\n\nThe lookup function is optional. It is assumed that the lookup function always\nfinds values much faster than it would take to traverse the table. The first\nargument is the position of the key. As `qlc_next/1` returns the objects as\n`{Key, Value}` pairs, the position is 1. Notice that the lookup function is to\nreturn `{Key, Value}` pairs, as the traversal function does.\n\nThe format function is also optional. It is called by [`info/1,2`](`info/1`) to\ngive feedback at runtime of how the query is to be evaluated. Try to give as\ngood feedback as possible without showing too much details. In the example, at\nmost seven objects of the table are shown. The format function handles two\ncases: `all` means that all objects of the table are traversed;\n`{lookup, 1, KeyValues}` means that the lookup function is used for looking up\nkey values.\n\nWhether the whole table is traversed or only some keys looked up depends on how\nthe query is expressed. If the query has the form\n\n```text\nqlc:q([T || P <- LE, F])\n```\n\nand `P` is a tuple, the `qlc` module analyzes `P` and `F` in compile time to\nfind positions of tuple `P` that are tested for equality to constants. If such a\nposition at runtime turns out to be the key position, the lookup function can be\nused, otherwise all objects of the table must be traversed. The info function\n`InfoFun` returns the key position. There can be indexed positions as well, also\nreturned by the info function. An index is an extra table that makes lookup on\nsome position fast. Mnesia maintains indexes upon request, and introduces so\ncalled secondary keys. The `qlc` module prefers to look up objects using the key\nbefore secondary keys regardless of the number of constants to look up.","ref":"qlc.html#module-implementing-a-qlc-table"},{"type":"module","title":"Key Equality - qlc","doc":"Erlang/OTP has two operators for testing term equality: `==/2` and `=:=/2`. The\ndifference is all about the integers that can be represented by floats. For\nexample, `2 == 2.0` evaluates to `true` while `2 =:= 2.0` evaluates to `false`.\nNormally this is a minor issue, but the `qlc` module cannot ignore the\ndifference, which affects the user's choice of operators in QLCs.\n\nIf the `qlc` module at compile time can determine that some constant is free of\nintegers, it does not matter which one of `==/2` or `=:=/2` is used:\n\n```erlang\n1> E1 = ets:new(t, [set]), % uses =:=/2 for key equality\nQ1 = qlc:q([K ||\n{K} <- ets:table(E1),\nK == 2.71 orelse K == a]),\nio:format(\"~s~n\", [qlc:info(Q1)]).\nets:match_spec_run(\n       lists:flatmap(fun(V) ->\n\t\t\t    ets:lookup(#Ref<0.3098908599.2283929601.256025>,\n\t\t\t\t       V)\n\t\t     end,\n\t\t     [a, 2.71]),\n       ets:match_spec_compile([{{'$1'}, [], ['$1']}]))\n```\n\nIn the example, operator `==/2` has been handled exactly as `=:=/2` would have\nbeen handled. However, if it cannot be determined at compile time that some\nconstant is free of integers, and the table uses `=:=/2` when comparing keys for\nequality (see option [key_equality](`m:qlc#key_equality`)), then the `qlc`\nmodule does not try to look up the constant. The reason is that there is in the\ngeneral case no upper limit on the number of key values that can compare equal\nto such a constant; every combination of integers and floats must be looked up:\n\n```erlang\n2> E2 = ets:new(t, [set]),\ntrue = ets:insert(E2, [{{2,2},a},{{2,2.0},b},{{2.0,2},c}]),\nF2 = fun(I) ->\nqlc:q([V || {K,V} <- ets:table(E2), K == I])\nend,\nQ2 = F2({2,2}),\nio:format(\"~s~n\", [qlc:info(Q2)]).\nets:table(#Ref<0.3098908599.2283929601.256125>,\n          [{traverse,\n            {select,\n             [{{'$1', '$2'}, [{'==', '$1', {const, {2, 2}}}], ['$2']}]}}])\n3> lists:sort(qlc:e(Q2)).\n[a,b,c]\n```\n\nLooking up only `{2,2}` would not return `b` and `c`.\n\nIf the table uses `==/2` when comparing keys for equality, the `qlc` module\nlooks up the constant regardless of which operator is used in the QLC. However,\n`==/2` is to be preferred:\n\n```erlang\n4> E3 = ets:new(t, [ordered_set]), % uses ==/2 for key equality\ntrue = ets:insert(E3, [{{2,2.0},b}]),\nF3 = fun(I) ->\nqlc:q([V || {K,V} <- ets:table(E3), K == I])\nend,\nQ3 = F3({2,2}),\nio:format(\"~s~n\", [qlc:info(Q3)]).\nets:match_spec_run(ets:lookup(#Ref<0.3098908599.2283929601.256211>,\n                              {2, 2}),\n                   ets:match_spec_compile([{{'$1', '$2'}, [], ['$2']}]))\n5> qlc:e(Q3).\n[b]\n```\n\nLookup join is handled analogously to lookup of constants in a table: if the\njoin operator is `==/2`, and the table where constants are to be looked up uses\n`=:=/2` when testing keys for equality, then the `qlc` module does not consider\nlookup join for that table.","ref":"qlc.html#module-key-equality"},{"type":"module","title":"See Also - qlc","doc":"`m:dets`, `m:erl_eval`, `m:erlang`, `m:error_logger`, `m:ets`, `m:file`,\n`m:file_sorter`, `m:mnesia`, `m:shell`,\n[Erlang Reference Manual](`e:system:index.html`),\n[Programming Examples](`e:system:index.html`)","ref":"qlc.html#module-see-also"},{"type":"type","title":"qlc.abstract_expr/0","doc":"Parse trees for Erlang expression, see section\n[The Abstract Format](`e:erts:absform.md`) in the ERTS User's Guide.","ref":"qlc.html#t:abstract_expr/0"},{"type":"type","title":"qlc.answer/0","doc":"","ref":"qlc.html#t:answer/0"},{"type":"type","title":"qlc.answers/0","doc":"","ref":"qlc.html#t:answers/0"},{"type":"function","title":"qlc.append/1","doc":"Returns a query handle. When evaluating query handle `QH`, all answers to the\nfirst query handle in `QHL` are returned, followed by all answers to the\nremaining query handles in `QHL`.","ref":"qlc.html#append/1"},{"type":"function","title":"qlc.append/2","doc":"Returns a query handle. When evaluating query handle `QH3`, all answers to `QH1`\nare returned, followed by all answers to `QH2`.\n\n[`append(QH1, QH2)`](`append/2`) is equivalent to\n[`append([QH1, QH2])`](`append/1`).","ref":"qlc.html#append/2"},{"type":"type","title":"qlc.cache/0","doc":"","ref":"qlc.html#t:cache/0"},{"type":"function","title":"qlc.cursor/1","doc":"","ref":"qlc.html#cursor/1"},{"type":"function","title":"qlc.cursor/2","doc":"Creates a query cursor and makes the calling process the owner of the cursor.\n\nThe cursor is to be used as argument to [`next_answers/1,2`](`next_answers/1`)\nand (eventually) `delete_cursor/1`. Calls `erlang:spawn_opt/2` to spawn and link\nto a process that evaluates the query handle. The value of option\n`spawn_options` is used as last argument when calling\n[`spawn_opt/2`](`spawn_opt/2`). Defaults to `[link]`.\n\n_Example:_\n\n```erlang\n1> QH = qlc:q([{X,Y} || X <- [a,b], Y <- [1,2]]),\nQC = qlc:cursor(QH),\nqlc:next_answers(QC, 1).\n[{a,1}]\n2> qlc:next_answers(QC, 1).\n[{a,2}]\n3> qlc:next_answers(QC, all_remaining).\n[{b,1},{b,2}]\n4> qlc:delete_cursor(QC).\nok\n```","ref":"qlc.html#cursor/2"},{"type":"function","title":"qlc.delete_cursor/1","doc":"Deletes a query cursor. Only the owner of the cursor can delete the cursor.","ref":"qlc.html#delete_cursor/1"},{"type":"function","title":"qlc.e/1","doc":"","ref":"qlc.html#e/1"},{"type":"function","title":"qlc.e/2","doc":"","ref":"qlc.html#e/2"},{"type":"function","title":"qlc.eval/1","doc":"","ref":"qlc.html#eval/1"},{"type":"function","title":"qlc.eval/2","doc":"Evaluates a query handle in the calling process and collects all answers in a\nlist.\n\n_Example:_\n\n```erlang\n1> QH = qlc:q([{X,Y} || X <- [a,b], Y <- [1,2]]),\nqlc:eval(QH).\n[{a,1},{a,2},{b,1},{b,2}]\n```","ref":"qlc.html#eval/2"},{"type":"function","title":"qlc.fold/3","doc":"","ref":"qlc.html#fold/3"},{"type":"function","title":"qlc.fold/4","doc":"Calls `Function` on successive answers to the query handle together with an\nextra argument `AccIn`.\n\nThe query handle and the function are evaluated in the\ncalling process. `Function` must return a new accumulator, which is passed to\nthe next call. `Acc0` is returned if there are no answers to the query handle.\n\n_Example:_\n\n```erlang\n1> QH = [1,2,3,4,5,6],\nqlc:fold(fun(X, Sum) -> X + Sum end, 0, QH).\n21\n```","ref":"qlc.html#fold/4"},{"type":"function","title":"qlc.format_error/1","doc":"Returns a descriptive string in English of an error tuple returned by some of\nthe functions of the `qlc` module or the parse transform. This function is\nmainly used by the compiler invoking the parse transform.","ref":"qlc.html#format_error/1"},{"type":"function","title":"qlc.info/1","doc":"","ref":"qlc.html#info/1"},{"type":"function","title":"qlc.info/2","doc":"Returns information about a query handle. The information describes the\nsimplifications and optimizations that are the results of preparing the query\nfor evaluation. This function is probably mainly useful during debugging.\n\nThe information has the form of an Erlang expression where QLCs most likely\noccur. Depending on the format functions of mentioned QLC tables, it is not\ncertain that the information is absolutely accurate.\n\nOptions:\n\n- The default is to return a sequence of QLCs in a block, but if option\n  `{flat, false}` is specified, one single QLC is returned.\n- The default is to return a string, but if option `{format, abstract_code}` is\n  specified, abstract code is returned instead. In the abstract code, port\n  identifiers, references, and pids are represented by strings.\n- The default is to return all elements in lists, but if option\n  `{n_elements, NElements}` is specified, only a limited number of elements are\n  returned.\n- The default is to show all parts of objects and match specifications, but if\n  option `{depth, Depth}` is specified, parts of terms below a certain depth are\n  replaced by `'...'`.\n\n_Examples:_\n\nIn the following example two simple QLCs are inserted only to hold option\n`{unique, true}`:\n\n```erlang\n1> QH = qlc:q([{X,Y} || X <- [x,y], Y <- [a,b]]),\nio:format(\"~s~n\", [qlc:info(QH, unique_all)]).\nbegin\n    V1 =\n        qlc:q([\n               SQV ||\n                   SQV <- [x, y]\n              ],\n              [{unique, true}]),\n    V2 =\n        qlc:q([\n               SQV ||\n                   SQV <- [a, b]\n              ],\n              [{unique, true}]),\n    qlc:q([\n           {X,Y} ||\n               X <- V1,\n               Y <- V2\n          ],\n          [{unique, true}])\nend\n```\n\nIn the following example QLC `V2` has been inserted to show the joined\ngenerators and the join method chosen. A convention is used for lookup join: the\nfirst generator (`G2`) is the one traversed, the second (`G1`) is the table\nwhere constants are looked up.\n\n```erlang\n1> E1 = ets:new(e1, []),\nE2 = ets:new(e2, []),\ntrue = ets:insert(E1, [{1,a},{2,b}]),\ntrue = ets:insert(E2, [{a,1},{b,2}]),\nQ = qlc:q([{X,Z,W} ||\n{X, Z} <- ets:table(E1),\n{W, Y} <- ets:table(E2),\nX =:= Y]),\nio:format(\"~s~n\", [qlc:info(Q)]).\nbegin\n    V1 =\n        qlc:q([\n               P0 ||\n                   P0 = {W, Y} <-\n                       ets:table(#Ref<0.3098908599.2283929601.256549>)\n              ]),\n    V2 =\n        qlc:q([\n               [G1 | G2] ||\n                   G2 <- V1,\n                   G1 <-\n                       ets:table(#Ref<0.3098908599.2283929601.256548>),\n                   element(2, G1) =:= element(1, G2)\n              ],\n              [{join, lookup}]),\n    qlc:q([\n           {X, Z, W} ||\n               [{X, Z} | {W, Y}] <- V2\n          ])\nend\n```","ref":"qlc.html#info/2"},{"type":"type","title":"qlc.key_pos/0","doc":"","ref":"qlc.html#t:key_pos/0"},{"type":"function","title":"qlc.keysort/2","doc":"","ref":"qlc.html#keysort/2"},{"type":"function","title":"qlc.keysort/3","doc":"Returns a query handle. When evaluating query handle `QH2`, the answers to query\nhandle `QH1` are sorted by `file_sorter:keysort/4` according to the options.\n\nThe sorter uses temporary files only if `QH1` does not evaluate to a list and\nthe size of the binary representation of the answers exceeds `Size` bytes, where\n`Size` is the value of option `size`.","ref":"qlc.html#keysort/3"},{"type":"type","title":"qlc.match_expression/0","doc":"Match specification, see section\n[Match Specifications in Erlang](`e:erts:match_spec.md`) in the ERTS User's\nGuide and `m:ms_transform`.","ref":"qlc.html#t:match_expression/0"},{"type":"type","title":"qlc.max_list_size/0","doc":"","ref":"qlc.html#t:max_list_size/0"},{"type":"function","title":"qlc.next_answers/1","doc":"","ref":"qlc.html#next_answers/1"},{"type":"function","title":"qlc.next_answers/2","doc":"Returns some or all of the remaining answers to a query cursor. Only the owner\nof `QueryCursor` can retrieve answers.\n\nArgument `NumberOfAnswers` determines the maximum number of answers\nreturned. If less than the requested number of answers is\nreturned, subsequent calls to `next_answers` return `[]`.","ref":"qlc.html#next_answers/2"},{"type":"type","title":"qlc.no_files/0","doc":"An integer > 1.","ref":"qlc.html#t:no_files/0"},{"type":"type","title":"qlc.order/0","doc":"","ref":"qlc.html#t:order/0"},{"type":"type","title":"qlc.order_fun/0","doc":"","ref":"qlc.html#t:order_fun/0"},{"type":"function","title":"qlc.q/1","doc":"","ref":"qlc.html#q/1"},{"type":"function","title":"qlc.q/2","doc":"Returns a query handle for a QLC. The QLC must be the first argument to this\nfunction, otherwise it is evaluated as an ordinary list comprehension. It is\nalso necessary to add the following line to the source code:\n\n```erlang\n-include_lib(\"stdlib/include/qlc.hrl\").\n```\n\nThis causes a parse transform to substitute a fun for the QLC. The (compiled)\nfun is called when the query handle is evaluated.\n\nWhen calling `qlc:q/1,2` from the Erlang shell, the parse transform is\nautomatically called. When this occurs, the fun substituted for the QLC is not\ncompiled but is evaluated by `m:erl_eval`. This is also true when expressions\nare evaluated by `file:eval/1,2` or in the debugger.\n\nTo be explicit, this does not work:\n\n```erlang\n...\nA = [X || {X} <- [{1},{2}]],\nQH = qlc:q(A),\n...\n```\n\nVariable `A` is bound to the evaluated value of the list comprehension\n(`[1,2]`). The compiler complains with an error message (\"argument is not a\nquery list comprehension\"); the shell process stops with a `badarg` reason.\n\nOptions:\n\n- Option `{cache, ets}` can be used to cache the answers to a QLC. The answers\n  are stored in one ETS table for each cached QLC. When a cached QLC is\n  evaluated again, answers are fetched from the table without any further\n  computations. Therefore, when all answers to a cached QLC have been found, the\n  ETS tables used for caching answers to the qualifiers of the QLC can be\n  emptied. Option `cache` is equivalent to `{cache, ets}`.\n- Option `{cache, list}` can be used to cache the answers to a QLC like\n  `{cache, ets}`. The difference is that the answers are kept in a list (on the\n  process heap). If the answers would occupy more than a certain amount of RAM\n  memory, a temporary file is used for storing the answers. Option\n  `max_list_size` sets the limit in bytes and the temporary file is put on the\n  directory set by option `tmpdir`.\n\n  Option `cache` has no effect if it is known that the QLC is to be evaluated at\n  most once. This is always true for the top-most QLC and also for the list\n  expression of the first generator in a list of qualifiers. Notice that in the\n  presence of side effects in filters or callback functions, the answers to QLCs\n  can be affected by option `cache`.\n\n- Option `{unique, true}` can be used to remove duplicate answers to a QLC. The\n  unique answers are stored in one ETS table for each QLC. The table is emptied\n  every time it is known that there are no more answers to the QLC. Option\n  `unique` is equivalent to `{unique, true}`. If option `unique` is combined\n  with option `{cache, ets}`, two ETS tables are used, but the full answers are\n  stored in one table only. If option `unique` is combined with option\n  `{cache, list}`, the answers are sorted twice using `keysort/3`; once to\n  remove duplicates and once to restore the order.\n\nOptions `cache` and `unique` apply not only to the QLC itself but also to the\nresults of looking up constants, running match specifications, and joining\nhandles.\n\n_Example:_\n\nIn the following example the cached results of the merge join are traversed for\neach value of `A`. Notice that without option `cache` the join would have been\ncarried out three times, once for each value of `A`.\n\n```erlang\n1> Q = qlc:q([{A,X,Z,W} ||\nA <- [a,b,c],\n{X,Z} <- [{a,1},{b,4},{c,6}],\n{W,Y} <- [{2,a},{3,b},{4,c}],\nX =:= Y],\n{cache, list}),\nio:format(\"~s~n\", [qlc:info(Q)]).\nbegin\n    V1 =\n        qlc:q([\n               P0 ||\n                   P0 = {X, Z} <-\n                       qlc:keysort(1, [{a, 1}, {b, 4}, {c, 6}], [])\n              ]),\n    V2 =\n        qlc:q([\n               P0 ||\n                   P0 = {W, Y} <-\n                       qlc:keysort(2, [{2, a}, {3, b}, {4, c}], [])\n              ]),\n    V3 =\n        qlc:q([\n               [G1 | G2] ||\n                   G1 <- V1,\n                   G2 <- V2,\n                   element(1, G1) == element(2, G2)\n              ],\n              [{join, merge}, {cache, list}]),\n    qlc:q([\n           {A, X, Z, W} ||\n               A <- [a, b, c],\n               [{X, Z} | {W, Y}] <- V3,\n               X =:= Y\n          ])\nend\n```\n\n[`sort/1,2`](`sort/1`) and [`keysort/2,3`](`keysort/2`) can also be used for\ncaching answers and for removing duplicates. When sorting answers are cached in\na list, possibly stored on a temporary file, and no ETS tables are used.\n\nSometimes (see `table/2`) traversal of tables can be done by looking up key\nvalues, which is assumed to be fast. Under certain (rare) circumstances there\ncan be too many key values to look up. [](){: #max_lookup } Option\n`{max_lookup, MaxLookup}` can then be used to limit the number of lookups: if\nmore than `MaxLookup` lookups would be required, no lookups are done but the\ntable is traversed instead. Defaults to `infinity`, which means that there is no\nlimit on the number of keys to look up.\n\n_Example:_\n\nIn the following example, using the `gb_table` module from section\n[Implementing a QLC Table](`m:qlc#implementing_a_qlc_table`), there are six keys\nto look up: `{1,a}`, `{1,b}`, `{1,c}`, `{2,a}`, `{2,b}`, and `{2,c}`. The reason\nis that the two elements of key `{X, Y}` are compared separately.\n\n```erlang\n1> T = gb_trees:empty(),\nQH = qlc:q([X || {{X,Y},_} <- gb_table:table(T),\n((X == 1) or (X == 2)) andalso\n((Y == a) or (Y == b) or (Y == c))]),\nio:format(\"~s~n\", [qlc:info(QH)]).\nets:match_spec_run(\n       lists:flatmap(fun(K) ->\n                            case\n                                gb_trees:lookup(K,\n                                                gb_trees:from_orddict([]))\n                            of\n                                {value, V} ->\n                                    [{K, V}];\n                                none ->\n                                    []\n                            end\n                     end,\n                     [{1, a},\n                      {1, b},\n                      {1, c},\n                      {2, a},\n                      {2, b},\n                      {2, c}]),\n       ets:match_spec_compile([{{{'$1', '$2'}, '_'},\n                                [],\n                                ['$1']}]))\n```\n\nOptions:\n\n- Option `{lookup, true}` can be used to ensure that the `qlc` module looks up\n  constants in some QLC table. If there are more than one QLC table among the\n  list expressions of the generators, constants must be looked up in at least\n  one of the tables. The evaluation of the query fails if there are no constants\n  to look up. This option is useful when it would be unacceptable to traverse\n  all objects in some table. Setting option `lookup` to `false` ensures that no\n  constants are looked up (`{max_lookup, 0}` has the same effect). Defaults to\n  `any`, which means that constants are looked up whenever possible.\n- Option `{join, Join}` can be used to ensure that a certain join method is\n  used:\n\n  - `{join, lookup}` invokes the lookup join method.\n  - `{join, merge}` invokes the merge join method.\n  - `{join, nested_loop}` invokes the method of matching every pair of objects\n    from two handles. This method is mostly very slow.\n\n  The evaluation of the query fails if the `qlc` module cannot carry out the\n  chosen join method. Defaults to `any`, which means that some fast join method\n  is used if possible.","ref":"qlc.html#q/2"},{"type":"opaque","title":"qlc.query_cursor/0","doc":"A [query cursor](`m:qlc#query_cursor`).","ref":"qlc.html#t:query_cursor/0"},{"type":"opaque","title":"qlc.query_handle/0","doc":"A [query handle](`m:qlc#query_handle`).","ref":"qlc.html#t:query_handle/0"},{"type":"type","title":"qlc.query_handle_or_list/0","doc":"","ref":"qlc.html#t:query_handle_or_list/0"},{"type":"type","title":"qlc.query_list_comprehension/0","doc":"A literal [query list comprehension](`m:qlc#query_list_comprehension`).","ref":"qlc.html#t:query_list_comprehension/0"},{"type":"function","title":"qlc.sort/1","doc":"","ref":"qlc.html#sort/1"},{"type":"function","title":"qlc.sort/2","doc":"Returns a query handle. When evaluating query handle `QH2`, the answers to query\nhandle `QH1` are sorted by `file_sorter:sort/3` according to the options.\n\nThe sorter uses temporary files only if `QH1` does not evaluate to a list and\nthe size of the binary representation of the answers exceeds `Size` bytes, where\n`Size` is the value of option `size`.","ref":"qlc.html#sort/2"},{"type":"type","title":"qlc.sort_option/0","doc":"See `m:file_sorter` for a description of the options.","ref":"qlc.html#t:sort_option/0"},{"type":"type","title":"qlc.sort_options/0","doc":"","ref":"qlc.html#t:sort_options/0"},{"type":"type","title":"qlc.spawn_options/0","doc":"","ref":"qlc.html#t:spawn_options/0"},{"type":"function","title":"qlc.string_to_handle/1","doc":"","ref":"qlc.html#string_to_handle/1"},{"type":"function","title":"qlc.string_to_handle/2","doc":"","ref":"qlc.html#string_to_handle/2"},{"type":"function","title":"qlc.string_to_handle/3","doc":"A string version of [`q/1,2`](`q/1`). When the query handle is evaluated, the\nfun created by the parse transform is interpreted by `m:erl_eval`. The query\nstring is to be one single QLC terminated by a period.\n\n_Example:_\n\n```erlang\n1> L = [1,2,3],\nBs = erl_eval:add_binding('L', L, erl_eval:new_bindings()),\nQH = qlc:string_to_handle(\"[X+1 || X <- L].\", [], Bs),\nqlc:eval(QH).\n[2,3,4]\n```\n\nThis function is probably mainly useful when called from outside of Erlang, for\nexample from a driver written in C.\n\n> #### Note {: .info }\n>\n> Query handles created this way may have worse performance than when created\n> directly via [`q/1,2`](`q/1`).","ref":"qlc.html#string_to_handle/3"},{"type":"function","title":"qlc.table/2","doc":"Returns a query handle for a QLC table. In Erlang/OTP there is support for ETS,\nDets, and Mnesia tables, but many other data structures can be turned into QLC\ntables. This is accomplished by letting function(s) in the module implementing\nthe data structure create a query handle by calling `qlc:table/2`.\n\nThe different ways to traverse the table and properties of the table are handled\nby callback functions provided as options to `qlc:table/2`.\n\n- Callback function `TraverseFun` is used for traversing the table. It is to\n  return a list of objects terminated by either `[]` or a nullary fun to be used\n  for traversing the not yet traversed objects of the table. Any other return\n  value is immediately returned as value of the query evaluation. Unary\n  `TraverseFun`s are to accept a match specification as argument. The match\n  specification is created by the parse transform by analyzing the pattern of\n  the generator calling `qlc:table/2` and filters using variables introduced in\n  the pattern. If the parse transform cannot find a match specification\n  equivalent to the pattern and filters, `TraverseFun` is called with a match\n  specification returning every object.\n\n  - Modules that can use match specifications for optimized traversal of tables\n    are to call `qlc:table/2` with an unary `TraverseFun`. An example is\n    `ets:table/2`.\n  - Other modules can provide a nullary `TraverseFun`. An example is\n    [`gb_table:table/1`](`m:qlc#gb_table`) in section\n    [Implementing a QLC Table](`m:qlc#implementing_a_qlc_table`).\n\n- Unary callback function `PreFun` is called once before the table is read for\n  the first time. If the call fails, the query evaluation fails.\n\n  Argument `PreArgs` is a list of tagged values. There are two tags,\n  `parent_value` and `stop_fun`, used by Mnesia for managing transactions.\n\n  - The value of `parent_value` is the value returned by `ParentFun`, or\n    `undefined` if there is no `ParentFun`. `ParentFun` is called once just\n    before the call of `PreFun` in the context of the process calling\n    [`eval/1,2`](`eval/1`), [`fold/3,4`](`fold/3`), or\n    [`cursor/1,2`](`cursor/1`).\n  - The value of `stop_fun` is a nullary fun that deletes the cursor if called\n    from the parent, or `undefined` if there is no cursor.\n\n- Nullary callback function `PostFun` is called once after the table was last\n  read. The return value, which is caught, is ignored. If `PreFun` has been\n  called for a table, `PostFun` is guaranteed to be called for that table, even\n  if the evaluation of the query fails for some reason.\n\n  The pre (post) functions for different tables are evaluated in unspecified\n  order.\n\n  Other table access than reading, such as calling `InfoFun`, is assumed to be\n  OK at any time.\n\n- [](){: #lookup_fun } Binary callback function `LookupFun` is used for looking\n  up objects in the table. The first argument `Position` is the key position or\n  an indexed position and the second argument `Keys` is a sorted list of unique\n  values. The return value is to be a list of all objects (tuples), such that\n  the element at `Position` is a member of `Keys`. Any other return value is\n  immediately returned as value of the query evaluation. `LookupFun` is called\n  instead of traversing the table if the parse transform at compile time can\n  determine that the filters match and compare the element at `Position` in such\n  a way that only `Keys` need to be looked up to find all potential answers.\n\n  The key position is obtained by calling `InfoFun(keypos)` and the indexed\n  positions by calling `InfoFun(indices)`. If the key position can be used for\n  lookup, it is always chosen, otherwise the indexed position requiring the\n  least number of lookups is chosen. If there is a tie between two indexed\n  positions, the one occurring first in the list returned by `InfoFun` is\n  chosen. Positions requiring more than [max_lookup](`m:qlc#max_lookup`) lookups\n  are ignored.\n\n- Unary callback function `InfoFun` is to return information about the table.\n  `undefined` is to be returned if the value of some tag is unknown:\n\n  - **`indices`** - Returns a list of indexed positions, a list of positive\n    integers.\n\n  - **`is_unique_objects`** - Returns `true` if the objects returned by\n    `TraverseFun` are unique.\n\n  - **`keypos`** - Returns the position of the table key, a positive integer.\n\n  - **`is_sorted_key`** - Returns `true` if the objects returned by\n    `TraverseFun` are sorted on the key.\n\n  - **`num_of_objects`** - Returns the number of objects in the table, a\n    non-negative integer.\n\n- Unary callback function `FormatFun` is used by [`info/1,2`](`info/1`) for\n  displaying the call that created the query handle of the table. Defaults to\n  `undefined`, which means that `info/1,2` displays a call to `'$MOD':'$FUN'/0`.\n  It is up to `FormatFun` to present the selected objects of the table in a\n  suitable way. However, if a character list is chosen for presentation, it must\n  be an Erlang expression that can be scanned and parsed (a trailing dot is\n  added by `info/1,2` though).\n\n  `FormatFun` is called with an argument that describes the selected objects\n  based on optimizations done as a result of analyzing the filters of the QLC\n  where the call to `qlc:table/2` occurs. The argument can have the following\n  values:\n\n  - **`{lookup, Position, Keys, NElements, DepthFun}`.** - `LookupFun` is used\n    for looking up objects in the table.\n\n  - **`{match_spec, MatchExpression}`** - No way of finding all possible answers\n    by looking up keys was found, but the filters could be transformed into a\n    match specification. All answers are found by calling\n    `TraverseFun(MatchExpression)`.\n\n  - **`{all, NElements, DepthFun}`** - No optimization was found. A match\n    specification matching all objects is used if `TraverseFun` is unary.\n\n    `NElements` is the value of the `info/1,2` option `n_elements`.\n\n    `DepthFun` is a function that can be used for limiting the size of terms;\n    calling `DepthFun(Term)` substitutes `'...'` for parts of `Term` below the\n    depth specified by the `info/1,2` option `depth`.\n\n    If calling `FormatFun` with an argument including `NElements` and `DepthFun`\n    fails, `FormatFun` is called once again with an argument excluding\n    `NElements` and `DepthFun` (`{lookup, Position, Keys}` or `all`).\n\n- [](){: #key_equality } The value of option `key_equality` is to be `'=:='` if\n  the table considers two keys equal if they match, and to be `'=='` if two keys\n  are equal if they compare equal. Defaults to `'=:='`.\n\nFor the various options recognized by `table/1,2` in respective module, see\n[`ets`](`ets:table/1`), [`dets`](`dets:table/1`), and\n[`mnesia`](`mnesia:table/1`).","ref":"qlc.html#table/2"},{"type":"type","title":"qlc.tmp_directory/0","doc":"","ref":"qlc.html#t:tmp_directory/0"},{"type":"type","title":"qlc.tmp_file_usage/0","doc":"","ref":"qlc.html#t:tmp_file_usage/0"},{"type":"module","title":"queue","doc":"Abstract data type for FIFO queues.\n\nThis module provides (double-ended) FIFO queues in an efficient manner.\n\nAll functions fail with reason `badarg` if arguments are of wrong type, for\nexample, queue arguments are not queues, indexes are not integers, and list\narguments are not lists. Improper lists cause internal crashes. An index out of\nrange for a queue also causes a failure with reason `badarg`.\n\nSome functions, where noted, fail with reason `empty` for an empty queue.\n\nThe data representing a queue as used by this module is to be regarded as opaque\nby other modules. In abstract terms, the representation is a composite type of\nexisting Erlang terms. See note on\n[data types](`e:system:data_types.md#no_user_types`). Any code assuming\nknowledge of the format is running on thin ice.\n\nAll operations have an amortized O(1) running time, except `all/2`, `any/2`,\n`delete/2`, `delete_r/2`, `delete_with/2`, `delete_with_r/2`, `filter/2`,\n`filtermap/2`, `fold/3`, `join/2`, `len/1`, `member/2`, `split/2` that have\nO(n). To minimize the size of a queue minimizing the amount of garbage built by\nqueue operations, the queues do not contain explicit length information, and\nthat is why [`len/1`](`len/1`) is O(n). If better performance for this\nparticular operation is essential, it is easy for the caller to keep track of\nthe length.\n\nQueues are double-ended. The mental picture of a queue is a line of people\n(items) waiting for their turn. The queue front is the end with the item that\nhas waited the longest. The queue rear is the end an item enters when it starts\nto wait. If instead using the mental picture of a list, the front is called head\nand the rear is called tail.\n\nEntering at the front and exiting at the rear are reverse operations on the\nqueue.\n\nThis module has three sets of interface functions: the _\"Original API\"_, the\n_\"Extended API\"_, and the _\"Okasaki API\"_.\n\nThe \"Original API\" and the \"Extended API\" both use the mental picture of a\nwaiting line of items. Both have reverse operations suffixed \"\\_r\".\n\nThe \"Original API\" item removal functions return compound terms with both the\nremoved item and the resulting queue. The \"Extended API\" contains alternative\nfunctions that build less garbage and functions for just inspecting the queue\nends. Also the \"Okasaki API\" functions build less garbage.\n\nThe \"Okasaki API\" is inspired by \"Purely Functional Data Structures\" by Chris\nOkasaki. It regards queues as lists. This API is by many regarded as strange and\navoidable. For example, many reverse operations have lexically reversed names,\nsome with more readable but perhaps less understandable aliases.","ref":"queue.html"},{"type":"function","title":"queue.all/2","doc":"Returns `true` if `Pred(Item)` returns `true` for all items `Item` in `Q`,\notherwise `false`.\n\n_Example:_\n\n```erlang\n1> Queue = queue:from_list([1,2,3,4,5]).\n2> queue:all(fun (E) -> E > 3 end, Queue).\nfalse\n3> queue:all(fun (E) -> E > 0 end, Queue).\ntrue\n```","ref":"queue.html#all/2"},{"type":"function","title":"queue.any/2","doc":"Returns `true` if `Pred(Item)` returns `true` for at least one item `Item` in\n`Q`, otherwise `false`.\n\n_Example:_\n\n```erlang\n1> Queue = queue:from_list([1,2,3,4,5]).\n2> queue:any(fun (E) -> E > 10 end, Queue).\nfalse\n3> queue:any(fun (E) -> E > 3 end, Queue).\ntrue\n```","ref":"queue.html#any/2"},{"type":"function","title":"queue.cons/2","doc":"Inserts `Item` at the head of queue `Q1`. Returns the new queue `Q2`.\n\n_Example:_\n\n```erlang\n1> Queue = queue:cons(0, queue:from_list([1,2,3])).\n{[3,2],[0,1]}\n2> queue:to_list(Queue).\n[0,1,2,3]\n```","ref":"queue.html#cons/2"},{"type":"function","title":"queue.daeh/1","doc":"Returns the tail item of queue `Q`.\n\nFails with reason `empty` if `Q` is empty.\n\n_Example 1:_\n\n```erlang\n1> queue:daeh(queue:from_list([1,2,3])).\n3\n```","ref":"queue.html#daeh/1"},{"type":"function","title":"queue.delete/2","doc":"Returns a copy of `Q1` where the first item matching `Item` is deleted, if there\nis such an item.\n\n_Example:_\n\n```erlang\n1> Queue = queue:from_list([1,2,3,4,5]).\n2> Queue1 = queue:delete(3, Queue).\n3> queue:member(3, Queue1).\nfalse\n```","ref":"queue.html#delete/2"},{"type":"function","title":"queue.delete_r/2","doc":"Returns a copy of `Q1` where the last item matching `Item` is deleted, if there\nis such an item.\n\n_Example:_\n\n```erlang\n1> Queue = queue:from_list([1,2,3,4,3,5]).\n2> Queue1 = queue:delete_r(3, Queue).\n3> queue:to_list(Queue1).\n[1,2,3,4,5]\n```","ref":"queue.html#delete_r/2"},{"type":"function","title":"queue.delete_with/2","doc":"Returns a copy of `Q1` where the first item for which `Pred` returns `true` is\ndeleted, if there is such an item.\n\n_Example:_\n\n```erlang\n1> Queue = queue:from_list([100,1,2,3,4,5]).\n2> Queue1 = queue:delete_with(fun (E) -> E > 0, Queue).\n3> queue:to_list(Queue1).\n[1,2,3,4,5]\n```","ref":"queue.html#delete_with/2"},{"type":"function","title":"queue.delete_with_r/2","doc":"Returns a copy of `Q1` where the last item for which `Pred` returns `true` is\ndeleted, if there is such an item.\n\n_Example:_\n\n```erlang\n1> Queue = queue:from_list([1,2,3,4,5,100]).\n2> Queue1 = queue:delete_with(fun (E) -> E > 10, Queue).\n3> queue:to_list(Queue1).\n[1,2,3,4,5]\n```","ref":"queue.html#delete_with_r/2"},{"type":"function","title":"queue.drop/1","doc":"Returns a queue `Q2` that is the result of removing the front item from `Q1`.\n\nFails with reason `empty` if `Q1` is empty.\n\n_Example:_\n\n```erlang\n1> Queue = queue:from_list([1,2,3,4,5]).\n{[5,4,3],[1,2]}\n2> Queue = queue:drop(Queue).\n{[5,4,3],[2]}\n3> queue:to_list(Queue1).\n[2,3,4,5]\n```","ref":"queue.html#drop/1"},{"type":"function","title":"queue.drop_r/1","doc":"Returns a queue `Q2` that is the result of removing the rear item from `Q1`.\n\nFails with reason `empty` if `Q1` is empty.\n\n_Example:_\n\n```erlang\n1> Queue = queue:from_list([1,2,3,4,5]).\n{[5,4,3],[1,2]}\n2> Queue = queue:drop_r(Queue).\n{[4,3],[1,2]}\n3> queue:to_list(Queue1).\n[1,2,3,4]\n```","ref":"queue.html#drop_r/1"},{"type":"function","title":"queue.filter/2","doc":"Returns a queue `Q2` that is the result of calling `Fun(Item)` on all items in\n`Q1`.\n\nIf `Fun(Item)` returns `true`, `Item` is copied to the result queue. If it\nreturns `false`, `Item` is not copied. If it returns a list, the list elements\nare inserted instead of `Item` in the result queue.\n\n_Example 1:_\n\n```erlang\n1> Queue = queue:from_list([1,2,3,4,5]).\n{[5,4,3],[1,2]}\n2> Queue1 = queue:filter(fun (E) -> E > 2 end, Queue).\n{[5],[3,4]}\n3> queue:to_list(Queue1).\n[3,4,5]\n```\n\nSo, `Fun(Item)` returning `[Item]` is thereby semantically equivalent to\nreturning `true`, just as returning `[]` is semantically equivalent to returning\n`false`. But returning a list builds more garbage than returning an atom.\n\n_Example 2:_\n\n```erlang\n1> Queue = queue:from_list([1,2,3,4,5]).\n{[5,4,3],[1,2]}\n2> Queue1 = queue:filter(fun (E) -> [E, E+1] end, Queue).\n{[6,5,5,4,4,3],[1,2,2,3]}\n3> queue:to_list(Queue1).\n[1,2,2,3,3,4,4,5,5,6]\n```","ref":"queue.html#filter/2"},{"type":"function","title":"queue.filtermap/2","doc":"Returns a queue `Q2` that is the result of calling `Fun(Item)` on all items in\n`Q1`.\n\nIf `Fun(Item)` returns `true`, `Item` is copied to the result queue. If it\nreturns `false`, `Item` is not copied. If it returns `{true, NewItem}`, the\nqueue element at this position is replaced with `NewItem` in the result queue.\n\n_Example 1:_\n\n```erlang\n1> Queue = queue:from_list([1,2,3,4,5]).\n{[5,4,3],[1,2]}\n2> Queue1 = queue:filtermap(fun (E) -> E > 2 end, Queue).\n{[5],[3,4]}\n3> queue:to_list(Queue1).\n[3,4,5]\n4> Queue1 = queue:filtermap(fun (E) -> {true, E+100} end, Queue).\n{\"ihg\",\"ef\"}\n5> queue:to_list(Queue1).\n\"efghi\n```","ref":"queue.html#filtermap/2"},{"type":"function","title":"queue.fold/3","doc":"Calls `Fun(Item, AccIn)` on successive items `Item` of `Queue`, starting with\n`AccIn == Acc0`. The queue is traversed in queue order, that is, from front to\nrear. `Fun/2` must return a new accumulator, which is passed to the next call.\nThe function returns the final value of the accumulator. `Acc0` is returned if\nthe queue is empty.\n\n_Example:_\n\n```erlang\n1> queue:fold(fun(X, Sum) -> X + Sum end, 0, queue:from_list([1,2,3,4,5])).\n15\n2> queue:fold(fun(X, Prod) -> X * Prod end, 1, queue:from_list([1,2,3,4,5])).\n120\n```","ref":"queue.html#fold/3"},{"type":"function","title":"queue.from_list/1","doc":"Returns a queue containing the items in `L` in the same order; the head item of\nthe list becomes the front item of the queue.","ref":"queue.html#from_list/1"},{"type":"function","title":"queue.get/1","doc":"Returns `Item` at the front of queue `Q`.\n\nFails with reason `empty` if `Q` is empty.\n\n_Example 1:_\n\n```erlang\n1> Queue = queue:from_list([1,2,3,4,5]).\n{[5,4,3],[1,2]}\n2> 1 == queue:get(Queue).\ntrue\n```","ref":"queue.html#get/1"},{"type":"function","title":"queue.get_r/1","doc":"Returns `Item` at the rear of queue `Q`.\n\nFails with reason `empty` if `Q` is empty.\n\n_Example 1:_\n\n```erlang\n1> Queue = queue:from_list([1,2,3,4,5]).\n{[5,4,3],[1,2]}\n2> 5 == queue:get_r(Queue).\ntrue\n```","ref":"queue.html#get_r/1"},{"type":"function","title":"queue.head/1","doc":"Returns `Item` from the head of queue `Q`.\n\nFails with reason `empty` if `Q` is empty.\n\n_Example 1:_\n\n```erlang\n1> queue:head(queue:from_list([1,2,3])).\n1\n```","ref":"queue.html#head/1"},{"type":"function","title":"queue.in/2","doc":"Inserts `Item` at the rear of queue `Q1`. Returns the resulting queue `Q2`.\n\n_Example:_\n\n```erlang\n1> Queue = queue:from_list([1,2,3,4,5]).\n{[5,4,3],[1,2]}\n2> Queue1 = queue:in(100, Queue).\n{[100,5,4,3],[1,2]}\n3> queue:to_list(Queue1).\n[1,2,3,4,5,100]\n```","ref":"queue.html#in/2"},{"type":"function","title":"queue.in_r/2","doc":"Inserts `Item` at the front of queue `Q1`. Returns the resulting queue `Q2`.\n\n_Example:_\n\n```erlang\n1> Queue = queue:from_list([1,2,3,4,5]).\n{[5,4,3],[1,2]}\n2> Queue1 = queue:in_r(100, Queue).\n{[5,4,3],[100,1,2]}\n3> queue:to_list(Queue1).\n[100,1,2,3,4,5]\n```","ref":"queue.html#in_r/2"},{"type":"function","title":"queue.init/1","doc":"Returns a queue `Q2` that is the result of removing the tail item from `Q1`.\n\nFails with reason `empty` if `Q1` is empty.\n\n_Example:_\n\n```erlang\n1> Queue = queue:init(queue:from_list([1,2,3])).\n{[2],[1]}\n2> queue:to_list(Queue).\n[1,2]\n```","ref":"queue.html#init/1"},{"type":"function","title":"queue.is_empty/1","doc":"Tests if `Q` is empty and returns `true` if so, otherwise `false`.","ref":"queue.html#is_empty/1"},{"type":"function","title":"queue.is_queue/1","doc":"Tests if `Term` is a queue and returns `true` if so, otherwise `false`. Note\nthat the test will return `true` for a term coinciding with the representation\nof a queue, even when not constructed by thus module. See also note on\n[data types](`e:system:data_types.md#no_user_types`).","ref":"queue.html#is_queue/1"},{"type":"function","title":"queue.join/2","doc":"Returns a queue `Q3` that is the result of joining `Q1` and `Q2` with `Q1` in\nfront of `Q2`.\n\n_Example:_\n\n```erlang\n1> Queue1 = queue:from_list([1,3]).\n{[3],[1]}\n2> Queue2 = queue:from_list([2,4]).\n{[4],[2]}\n3> queue:to_list(queue:join(Queue1, Queue2)).\n[1,3,2,4]\n```","ref":"queue.html#join/2"},{"type":"function","title":"queue.lait/1","doc":"Returns a queue `Q2` that is the result of removing the tail item from `Q1`.\n\nFails with reason `empty` if `Q1` is empty.\n\nThe name [`lait/1`](`lait/1`) is a misspelling - do not use it anymore.","ref":"queue.html#lait/1"},{"type":"function","title":"queue.last/1","doc":"Returns the tail item of queue `Q`.\n\nFails with reason `empty` if `Q` is empty.\n\n_Example:_\n\n```erlang\n1> queue:last(queue:from_list([1,2,3])).\n3\n```","ref":"queue.html#last/1"},{"type":"function","title":"queue.len/1","doc":"Calculates and returns the length of queue `Q`.","ref":"queue.html#len/1"},{"type":"function","title":"queue.liat/1","doc":"Returns a queue `Q2` that is the result of removing the tail item from `Q1`.\n\nFails with reason `empty` if `Q1` is empty.\n\n_Example:_\n\n```erlang\n1> Queue = queue:liat(queue:from_list([1,2,3])).\n{[2],[1]}\n2> queue:to_list(Queue).\n[1,2]\n```","ref":"queue.html#liat/1"},{"type":"function","title":"queue.member/2","doc":"Returns `true` if `Item` matches some element in `Q`, otherwise `false`.","ref":"queue.html#member/2"},{"type":"function","title":"queue.new/0","doc":"Returns an empty queue.","ref":"queue.html#new/0"},{"type":"function","title":"queue.out/1","doc":"Removes the item at the front of queue `Q1`. Returns tuple\n`{{value, Item}, Q2}`, where `Item` is the item removed and `Q2` is the\nresulting queue. If `Q1` is empty, tuple `{empty, Q1}` is returned.\n\n_Example:_\n\n```erlang\n1> Queue = queue:from_list([1,2,3,4,5]).\n{[5,4,3],[1,2]}\n2> {{value, 1=Item}, Queue1} = queue:out(Queue).\n{{value,1},{[5,4,3],[2]}}\n3> queue:to_list(Queue1).\n[2,3,4,5]\n```","ref":"queue.html#out/1"},{"type":"function","title":"queue.out_r/1","doc":"Removes the item at the rear of queue `Q1`. Returns tuple `{{value, Item}, Q2}`,\nwhere `Item` is the item removed and `Q2` is the new queue. If `Q1` is empty,\ntuple `{empty, Q1}` is returned.\n\n_Example:_\n\n```erlang\n1> Queue = queue:from_list([1,2,3,4,5]).\n{[5,4,3],[1,2]}\n2> {{value, 5=Item}, Queue1} = queue:out_r(Queue).\n{{value,5},{[4,3],[1,2]}}\n3> queue:to_list(Queue1).\n[1,2,3,4]\n```","ref":"queue.html#out_r/1"},{"type":"function","title":"queue.peek/1","doc":"Returns tuple `{value, Item}`, where `Item` is the front item of `Q`, or `empty`\nif `Q` is empty.\n\n_Example 1:_\n\n```erlang\n1> queue:peek(queue:new()).\nempty\n2> Queue = queue:from_list([1,2,3,4,5]).\n{[5,4,3],[1,2]}\n3> queue:peek(Queue).\n{value, 1}\n```","ref":"queue.html#peek/1"},{"type":"function","title":"queue.peek_r/1","doc":"Returns tuple `{value, Item}`, where `Item` is the rear item of `Q`, or `empty`\nif `Q` is empty.\n\n_Example 1:_\n\n```erlang\n1> queue:peek_r(queue:new()).\nempty\n2> Queue = queue:from_list([1,2,3,4,5]).\n{[5,4,3],[1,2]}\n3> queue:peek_r(Queue).\n{value, 5}\n```","ref":"queue.html#peek_r/1"},{"type":"type","title":"queue.queue/0","doc":"","ref":"queue.html#t:queue/0"},{"type":"opaque","title":"queue.queue/1","doc":"As returned by `new/0`.","ref":"queue.html#t:queue/1"},{"type":"function","title":"queue.reverse/1","doc":"Returns a queue `Q2` containing the items of `Q1` in the reverse order.","ref":"queue.html#reverse/1"},{"type":"function","title":"queue.snoc/2","doc":"Inserts `Item` as the tail item of queue `Q1`. Returns the new queue `Q2`.\n\n_Example:_\n\n```erlang\n1> Queue = queue:snoc(queue:from_list([1,2,3]), 4).\n{[4,3,2],[1]}\n2> queue:to_list(Queue).\n[1,2,3,4]\n```","ref":"queue.html#snoc/2"},{"type":"function","title":"queue.split/2","doc":"Splits `Q1` in two. The `N` front items are put in `Q2` and the rest in `Q3`.","ref":"queue.html#split/2"},{"type":"function","title":"queue.tail/1","doc":"Returns a queue `Q2` that is the result of removing the head item from `Q1`.\n\nFails with reason `empty` if `Q1` is empty.","ref":"queue.html#tail/1"},{"type":"function","title":"queue.to_list/1","doc":"Returns a list of the items in the queue in the same order; the front item of\nthe queue becomes the head of the list.\n\n_Example:_\n\n```erlang\n1> Queue = queue:from_list([1,2,3,4,5]).\n{[5,4,3],[1,2]}\n2> List == queue:to_list(Queue).\ntrue\n```","ref":"queue.html#to_list/1"},{"type":"module","title":"sets","doc":"Sets are collections of elements with no duplicate elements.\n\nThe data representing a set as used by this module is to be regarded as opaque\nby other modules. In abstract terms, the representation is a composite type of\nexisting Erlang terms. See note on\n[data types](`e:system:data_types.md#no_user_types`). Any code assuming\nknowledge of the format is running on thin ice.\n\nThis module provides the same interface as the `m:ordsets` module but with an\nundefined representation. One difference is that while this module considers two\nelements as different if they do not match (`=:=`), `ordsets` considers two\nelements as different if and only if they do not compare equal (`==`).\n\nErlang/OTP 24.0 introduced a new internal representation for sets which is more\nperformant. Developers can use this new representation by passing the\n`{version, 2}` flag to `new/1` and `from_list/2`, such as\n`sets:new([{version, 2}])`. This new representation will become the default in\nfuture Erlang/OTP versions. Functions that work on two sets, such as `union/2`\nand similar, will work with sets of different versions. In such cases, there is\nno guarantee about the version of the returned set. Explicit conversion from the\nold version to the new one can be done with\n`sets:from_list(sets:to_list(Old), [{version,2}])`.","ref":"sets.html"},{"type":"module","title":"Compatibility - sets","doc":"The following functions in this module also exist and provide the same\nfunctionality in the `m:gb_sets` and `m:ordsets` modules. That is, by only\nchanging the module name for each call, you can try out different set\nrepresentations.\n\n- `add_element/2`\n- `del_element/2`\n- `filter/2`\n- `filtermap/2`\n- `fold/3`\n- `from_list/1`\n- `intersection/1`\n- `intersection/2`\n- `is_element/2`\n- `is_empty/1`\n- `is_equal/2`\n- `is_set/1`\n- `is_subset/2`\n- `map/2`\n- `new/0`\n- `size/1`\n- `subtract/2`\n- `to_list/1`\n- `union/1`\n- `union/2`\n\n> #### Note {: .info }\n>\n> While the three set implementations offer the same _functionality_ with\n> respect to the aforementioned functions, their overall _behavior_ may differ.\n> As mentioned, this module considers elements as different if and only if they\n> do not match (`=:=`), while both `m:ordsets` and `m:gb_sets` consider elements\n> as different if and only if they do not compare equal (`==`).\n>\n> _Example:_\n>\n> ```erlang\n> 1> sets:is_element(1.0, sets:from_list([1])).\n> false\n> 2> ordsets:is_element(1.0, ordsets:from_list([1])).\n> true\n> 2> gb_sets:is_element(1.0, gb_sets:from_list([1])).\n> true\n> ```","ref":"sets.html#module-compatibility"},{"type":"module","title":"See Also - sets","doc":"`m:gb_sets`, `m:ordsets`","ref":"sets.html#module-see-also"},{"type":"function","title":"sets.add_element/2","doc":"Returns a new set formed from `Set1` with `Element` inserted.","ref":"sets.html#add_element/2"},{"type":"function","title":"sets.del_element/2","doc":"Returns `Set1`, but with `Element` removed.","ref":"sets.html#del_element/2"},{"type":"function","title":"sets.filter/2","doc":"Filters elements in `Set1` with boolean function `Pred`.","ref":"sets.html#filter/2"},{"type":"function","title":"sets.filtermap/2","doc":"Filters and maps elements in `Set1` with function `Fun`.","ref":"sets.html#filtermap/2"},{"type":"function","title":"sets.fold/3","doc":"Folds `Function` over every element in `Set` and returns the final value of the\naccumulator. The evaluation order is undefined.","ref":"sets.html#fold/3"},{"type":"function","title":"sets.from_list/1","doc":"Returns a set of the elements in `List`.","ref":"sets.html#from_list/1"},{"type":"function","title":"sets.from_list/2","doc":"Returns a set of the elements in `List` at the given version.","ref":"sets.html#from_list/2"},{"type":"function","title":"sets.intersection/1","doc":"Returns the intersection of the non-empty list of sets.","ref":"sets.html#intersection/1"},{"type":"function","title":"sets.intersection/2","doc":"Returns the intersection of `Set1` and `Set2`.","ref":"sets.html#intersection/2"},{"type":"function","title":"sets.is_disjoint/2","doc":"Returns `true` if `Set1` and `Set2` are disjoint (have no elements in common),\notherwise `false`.","ref":"sets.html#is_disjoint/2"},{"type":"function","title":"sets.is_element/2","doc":"Returns `true` if `Element` is an element of `Set`, otherwise `false`.","ref":"sets.html#is_element/2"},{"type":"function","title":"sets.is_empty/1","doc":"Returns `true` if `Set` is an empty set, otherwise `false`.","ref":"sets.html#is_empty/1"},{"type":"function","title":"sets.is_equal/2","doc":"Returns `true` if `Set1` and `Set2` are equal, that is when every element of one\nset is also a member of the respective other set, otherwise `false`.","ref":"sets.html#is_equal/2"},{"type":"function","title":"sets.is_set/1","doc":"Returns `true` if `Set` appears to be a set of elements, otherwise `false`.\n\nNote that the test is shallow and will return `true` for any term that coincides with\nthe possible representations of a set. See also note on [data types](`e:system:data_types.md#no_user_types`).","ref":"sets.html#is_set/1"},{"type":"function","title":"sets.is_subset/2","doc":"Returns `true` when every element of `Set1` is also a member of `Set2`,\notherwise `false`.","ref":"sets.html#is_subset/2"},{"type":"function","title":"sets.map/2","doc":"Maps elements in `Set1` with mapping function `Fun`.","ref":"sets.html#map/2"},{"type":"function","title":"sets.new/0","doc":"Returns a new empty set.","ref":"sets.html#new/0"},{"type":"function","title":"sets.new/1","doc":"Returns a new empty set at the given version.","ref":"sets.html#new/1"},{"type":"type","title":"sets.set/0","doc":"","ref":"sets.html#t:set/0"},{"type":"opaque","title":"sets.set/1","doc":"As returned by `new/0`.","ref":"sets.html#t:set/1"},{"type":"function","title":"sets.size/1","doc":"Returns the number of elements in `Set`.","ref":"sets.html#size/1"},{"type":"function","title":"sets.subtract/2","doc":"Returns only the elements of `Set1` that are not also elements of `Set2`.","ref":"sets.html#subtract/2"},{"type":"function","title":"sets.to_list/1","doc":"Returns the elements of `Set` as a list. The order of the returned elements is\nundefined.","ref":"sets.html#to_list/1"},{"type":"function","title":"sets.union/1","doc":"Returns the merged (union) set of the list of sets.","ref":"sets.html#union/1"},{"type":"function","title":"sets.union/2","doc":"Returns the merged (union) set of `Set1` and `Set2`.","ref":"sets.html#union/2"},{"type":"module","title":"sofs","doc":"Functions for manipulating sets of sets.\n\nThis module provides operations on finite sets and relations represented as\nsets. Intuitively, a set is a collection of elements; every element belongs to\nthe set, and the set contains every element.\n\nThe data representing `sofs` as used by this module is to be regarded as opaque\nby other modules. In abstract terms, the representation is a composite type of\nexisting Erlang terms. See note on\n[data types](`e:system:data_types.md#no_user_types`). Any code assuming\nknowledge of the format is running on thin ice.\n\nGiven a set A and a sentence S(x), where x is a free variable, a new set B whose\nelements are exactly those elements of A for which S(x) holds can be formed,\nthis is denoted B = \\{x in A : S(x)\\}. Sentences are expressed using the logical\noperators \"for some\" (or \"there exists\"), \"for all\", \"and\", \"or\", \"not\". If the\nexistence of a set containing all the specified elements is known (as is always\nthe case in this module), this is denoted B = \\{x : S(x)\\}.\n\n- The _unordered set_ containing the elements a, b, and c is denoted\n  \\{a, b, c\\}. This notation is not to be confused with tuples.\n\n  The _ordered pair_ of a and b, with first _coordinate_ a and second coordinate\n  b, is denoted (a, b). An ordered pair is an _ordered set_ of two elements. In\n  this module, ordered sets can contain one, two, or more elements, and\n  parentheses are used to enclose the elements.\n\n  Unordered sets and ordered sets are orthogonal, again in this module; there is\n  no unordered set equal to any ordered set.\n\n- The _empty set_ contains no elements.\n\n  Set A is _equal_{: #equal } to set B if they contain the same elements, which\n  is denoted A = B. Two ordered sets are equal if they contain the same number\n  of elements and have equal elements at each coordinate.\n\n  Set B is a _subset_{: #subset } of set A if A contains all elements that B\n  contains.\n\n  The _union_{: #union } of two sets A and B is the smallest set that contains\n  all elements of A and all elements of B.\n\n  The _intersection_{: #intersection } of two sets A and B is the set that\n  contains all elements of A that belong to B.\n\n  Two sets are _disjoint_{: #disjoint } if their intersection is the empty set.\n\n  The _difference_{: #difference } of two sets A and B is the set that contains\n  all elements of A that do not belong to B.\n\n  The _symmetric difference_{: #symmetric_difference } of two sets is the set\n  that contains those element that belong to either of the two sets, but not\n  both.\n\n  The _union_{: #union_n } of a collection of sets is the smallest set that\n  contains all the elements that belong to at least one set of the collection.\n\n  The _intersection_{: #intersection_n } of a non-empty collection of sets is\n  the set that contains all elements that belong to every set of the collection.\n\n- The _Cartesian product_{: #Cartesian_product } of two sets X and Y, denoted\n  X × Y, is the set \\{a : a = (x, y) for some x in X and for some y in Y\\}.\n\n  A _relation_{: #relation } is a subset of X × Y. Let R be a relation. The fact\n  that (x, y) belongs to R is written as x R y. As relations are sets, the\n  definitions of the last item (subset, union, and so on) apply to relations as\n  well.\n\n  The _domain_{: #domain } of R is the set \\{x : x R y for some y in Y\\}.\n\n  The _range_{: #range } of R is the set \\{y : x R y for some x in X\\}.\n\n  The _converse_{: #converse } of R is the set \\{a : a = (y, x) for some\n  (x, y) in R\\}.\n\n  If A is a subset of X, the _image_{: #image } of A under R is the set \\{y :\n  x R y for some x in A\\}. If B is a subset of Y, the _inverse image_{:\n  #inverse_image } of B is the set \\{x : x R y for some y in B\\}.\n\n  If R is a relation from X to Y, and S is a relation from Y to Z, the _relative\n  product_{: #relative_product } of R and S is the relation T from X to Z\n  defined so that x T z if and only if there exists an element y in Y such that\n  x R y and y S z.\n\n  The _restriction_{: #restriction } of R to A is the set S defined so that\n  x S y if and only if there exists an element x in A such that x R y.\n\n  If S is a restriction of R to A, then R is an _extension_{: #extension } of S\n  to X.\n\n  If X = Y, then R is called a relation _in_ X.\n\n  The _field_{: #field } of a relation R in X is the union of the domain of R\n  and the range of R.\n\n  If R is a relation in X, and if S is defined so that x S y if x R y and not\n  x = y, then S is the _strict_{: #strict_relation } relation corresponding to\n  R. Conversely, if S is a relation in X, and if R is defined so that x R y if\n  x S y or x = y, then R is the _weak_{: #weak_relation } relation corresponding\n  to S.\n\n  A relation R in X is _reflexive_ if x R x for every element x of X, it is\n  _symmetric_ if x R y implies that y R x, and it is _transitive_ if x R y and\n  y R z imply that x R z.\n\n- A _function_{: #function } F is a relation, a subset of X × Y, such that the\n  domain of F is equal to X and such that for every x in X there is a unique\n  element y in Y with (x, y) in F. The latter condition can be formulated as\n  follows: if x F y and x F z, then y = z. In this module, it is not required\n  that the domain of F is equal to X for a relation to be considered a function.\n\n  Instead of writing (x, y) in F or x F y, we write F(x) = y when F is a\n  function, and say that F maps x onto y, or that the value of F at x is y.\n\n  As functions are relations, the definitions of the last item (domain, range,\n  and so on) apply to functions as well.\n\n  If the converse of a function F is a function F', then F' is called the\n  _inverse_{: #inverse } of F.\n\n  The relative product of two functions F1 and F2 is called the _composite_{:\n  #composite } of F1 and F2 if the range of F1 is a subset of the domain of F2.\n\n- Sometimes, when the range of a function is more important than the function\n  itself, the function is called a _family_.\n\n  The domain of a family is called the _index set_, and the range is called the\n  _indexed set_.\n\n  If x is a family from I to X, then x\\[i] denotes the value of the function at\n  index i. The notation \"a family in X\" is used for such a family.\n\n  When the indexed set is a set of subsets of a set X, we call x a _family of\n  subsets_{: #family } of X.\n\n  If x is a family of subsets of X, the union of the range of x is called the\n  _union of the family_ x.\n\n  If x is non-empty (the index set is non-empty), the _intersection of the\n  family_ x is the intersection of the range of x.\n\n  In this module, the only families that are considered are families of subsets\n  of some set X; in the following, the word \"family\" is used for such families\n  of subsets.\n\n- A _partition_{: #partition } of a set X is a collection S of non-empty subsets\n  of X whose union is X and whose elements are pairwise disjoint.\n\n  A relation in a set is an _equivalence relation_ if it is reflexive,\n  symmetric, and transitive.\n\n  If R is an equivalence relation in X, and x is an element of X, the\n  _equivalence class_{: #equivalence_class } of x with respect to R is the set\n  of all those elements y of X for which x R y holds. The equivalence classes\n  constitute a partitioning of X. Conversely, if C is a partition of X, the\n  relation that holds for any two elements of X if they belong to the same\n  equivalence class, is an equivalence relation induced by the partition C.\n\n  If R is an equivalence relation in X, the _canonical map_{: #canonical_map }\n  is the function that maps every element of X onto its equivalence class.\n\n- [](){: #binary_relation } Relations as defined above (as sets of ordered\n  pairs) are from now on referred to as _binary relations_.\n\n  We call a set of ordered sets (x\\[1], ..., x\\[n]) an _(n-ary) relation_{:\n  #n_ary_relation }, and say that the relation is a subset of the [](){:\n  #Cartesian_product_tuple } Cartesian product X\\[1] × ... × X\\[n], where x\\[i]\n  is an element of X\\[i], 1 <= i <= n.\n\n  The _projection_{: #projection } of an n-ary relation R onto coordinate i is\n  the set \\{x\\[i] : (x\\[1], ..., x\\[i], ..., x\\[n]) in R for some\n  x\\[j] in X\\[j], 1 <= j <= n and not i = j\\}. The projections of a binary\n  relation R onto the first and second coordinates are the domain and the range\n  of R, respectively.\n\n  The relative product of binary relations can be generalized to n-ary relations\n  as follows. Let TR be an ordered set (R\\[1], ..., R\\[n]) of binary relations\n  from X to Y\\[i] and S a binary relation from (Y\\[1] × ... × Y\\[n]) to Z. The\n  _relative product_{: #tuple_relative_product } of TR and S is the binary\n  relation T from X to Z defined so that x T z if and only if there exists an\n  element y\\[i] in Y\\[i] for each 1 <= i <= n such that x R\\[i] y\\[i] and\n  (y\\[1], ..., y\\[n]) S z. Now let TR be a an ordered set (R\\[1], ..., R\\[n]) of\n  binary relations from X\\[i] to Y\\[i] and S a subset of X\\[1] × ... × X\\[n].\n  The _multiple relative product_{: #multiple_relative_product } of TR and S is\n  defined to be the set \\{z : z = ((x\\[1], ..., x\\[n]), (y\\[1],...,y\\[n])) for\n  some (x\\[1], ..., x\\[n]) in S and for some (x\\[i], y\\[i]) in R\\[i],\n  1 <= i <= n\\}.\n\n  The _natural join_{: #natural_join } of an n-ary relation R and an m-ary\n  relation S on coordinate i and j is defined to be the set \\{z : z =\n  (x\\[1], ..., x\\[n],  y\\[1], ..., y\\[j-1], y\\[j+1], ..., y\\[m]) for some\n  (x\\[1], ..., x\\[n]) in R and for some (y\\[1], ..., y\\[m]) in S such that\n  x\\[i] = y\\[j]\\}.\n\n- [](){: #sets_definition } The sets recognized by this module are represented\n  by elements of the relation Sets, which is defined as the smallest set such\n  that:\n\n  - For every atom T, except '\\_', and for every term X, (T, X) belongs to Sets\n    (_atomic sets_).\n  - (\\['\\_'], []) belongs to Sets (the _untyped empty set_).\n  - For every tuple T = \\{T\\[1], ..., T\\[n]\\} and for every tuple X =\n    \\{X\\[1], ..., X\\[n]\\}, if (T\\[i], X\\[i]) belongs to Sets for every\n    1 <= i <= n, then (T, X) belongs to Sets (_ordered sets_).\n  - For every term T, if X is the empty list or a non-empty sorted list\n    \\[X[1], ..., X\\[n]] without duplicates such that (T, X\\[i]) belongs to Sets\n    for every 1 <= i <= n, then (\\[T], X) belongs to Sets (_typed unordered\n    sets_).\n\n  An _external set_{: #external_set } is an element of the range of Sets.\n\n  A _type_{: #type } is an element of the domain of Sets.\n\n  If S is an element (T, X) of Sets, then T is a _valid type_{: #valid_type } of\n  X, T is the type of S, and X is the external set of S. `from_term/2` creates a\n  set from a type and an Erlang term turned into an external set.\n\n  The sets represented by Sets are the elements of the range of function Set\n  from Sets to Erlang terms and sets of Erlang terms:\n\n  - Set(T,Term) = Term, where T is an atom\n  - Set(\\{T\\[1], ..., T\\[n]\\}, \\{X\\[1], ...,  X\\[n]\\}) =\n    (Set(T\\[1], X\\[1]), ...,  Set(T\\[n], X\\[n]))\n  - Set(\\[T], \\[X[1], ..., X\\[n]]) = \\{Set(T, X\\[1]), ..., Set(T, X\\[n])\\}\n  - Set(\\[T], []) = \\{\\}\n\n  When there is no risk of confusion, elements of Sets are identified with the\n  sets they represent. For example, if U is the result of calling `union/2` with\n  S1 and S2 as arguments, then U is said to be the union of S1 and S2. A more\n  precise formulation is that Set(U) is the union of Set(S1) and Set(S2).\n\nThe types are used to implement the various conditions that sets must fulfill.\nAs an example, consider the relative product of two sets R and S, and recall\nthat the relative product of R and S is defined if R is a binary relation to Y\nand S is a binary relation from Y. The function that implements the relative\nproduct, `relative_product/2`, checks that the arguments represent binary\nrelations by matching \\[\\{A,B\\}] against the type of the first argument (Arg1\nsay), and \\[\\{C,D\\}] against the type of the second argument (Arg2 say). The\nfact that \\[\\{A,B\\}] matches the type of Arg1 is to be interpreted as Arg1\nrepresenting a binary relation from X to Y, where X is defined as all sets\nSet(x) for some element x in Sets the type of which is A, and similarly for Y.\nIn the same way Arg2 is interpreted as representing a binary relation from W to\nZ. Finally it is checked that B matches C, which is sufficient to ensure that W\nis equal to Y. The untyped empty set is handled separately: its type, \\['\\_'],\nmatches the type of any unordered set.\n\nA few functions of this module (`drestriction/3`, `family_projection/2`,\n`partition/2`, `partition_family/2`, `projection/2`, `restriction/3`,\n`substitution/2`) accept an Erlang function as a means to modify each element of\na given unordered set. [](){: #set_fun } Such a function, called SetFun in the\nfollowing, can be specified as a functional object (fun), a tuple\n`{external, Fun}`, or an integer:\n\n- If SetFun is specified as a fun, the fun is applied to each element of the\n  given set and the return value is assumed to be a set.\n- If SetFun is specified as a tuple `{external, Fun}`, Fun is applied to the\n  external set of each element of the given set and the return value is assumed\n  to be an external set. Selecting the elements of an unordered set as external\n  sets and assembling a new unordered set from a list of external sets is in the\n  present implementation more efficient than modifying each element as a set.\n  However, this optimization can only be used when the elements of the unordered\n  set are atomic or ordered sets. It must also be the case that the type of the\n  elements matches some clause of Fun (the type of the created set is the result\n  of applying Fun to the type of the given set), and that Fun does nothing but\n  selecting, duplicating, or rearranging parts of the elements.\n- Specifying a SetFun as an integer I is equivalent to specifying\n  `{external, fun(X) -> element(I, X) end}`, but is to be preferred, as it makes\n  it possible to handle this case even more efficiently.\n\nExamples of SetFuns:\n\n```erlang\nfun sofs:union/1\nfun(S) -> sofs:partition(1, S) end\n{external, fun(A) -> A end}\n{external, fun({A,_,C}) -> {C,A} end}\n{external, fun({_,{_,C}}) -> C end}\n{external, fun({_,{_,{_,E}=C}}) -> {E,{E,C}} end}\n2\n```\n\nThe order in which a SetFun is applied to the elements of an unordered set is\nnot specified, and can change in future versions of this module.\n\nThe execution time of the functions of this module is dominated by the time it\ntakes to sort lists. When no sorting is needed, the execution time is in the\nworst case proportional to the sum of the sizes of the input arguments and the\nreturned value. A few functions execute in constant time: `from_external/2`,\n`is_empty_set/1`, `is_set/1`, `is_sofs_set/1`, `to_external/1` `type/1`.\n\nThe functions of this module exit the process with a `badarg`, `bad_function`,\nor `type_mismatch` message when given badly formed arguments or sets the types\nof which are not compatible.\n\nWhen comparing external sets, operator `==/2` is used.","ref":"sofs.html"},{"type":"module","title":"See Also - sofs","doc":"`m:dict`, `m:digraph`, `m:orddict`, `m:ordsets`, `m:sets`","ref":"sofs.html#module-see-also"},{"type":"type","title":"sofs.a_function/0","doc":"A [function](`m:sofs#function`).","ref":"sofs.html#t:a_function/0"},{"type":"function","title":"sofs.a_function/1","doc":"","ref":"sofs.html#a_function/1"},{"type":"function","title":"sofs.a_function/2","doc":"Creates a [function](`m:sofs#function`).\n\n[`a_function(F, T)`](`a_function/2`) is equivalent to\n[`from_term(F, T)`](`from_term/2`) if the result is a function.","ref":"sofs.html#a_function/2"},{"type":"opaque","title":"sofs.a_set/0","doc":"An [unordered set](`m:sofs#sets_definition`).","ref":"sofs.html#t:a_set/0"},{"type":"type","title":"sofs.anyset/0","doc":"Any kind of set (also included are the atomic sets).","ref":"sofs.html#t:anyset/0"},{"type":"type","title":"sofs.binary_relation/0","doc":"A [binary relation](`m:sofs#binary_relation`).","ref":"sofs.html#t:binary_relation/0"},{"type":"function","title":"sofs.canonical_relation/1","doc":"Returns the binary relation containing the elements (E, Set) such that Set\nbelongs to `SetOfSets` and E belongs to Set.\n\nIf `SetOfSets` is a [partition](`m:sofs#partition`) of a set X and R is the\nequivalence relation in X induced by `SetOfSets`, then the returned relation is the\n[canonical map](`m:sofs#canonical_map`) from X onto the equivalence classes with\nrespect to R.\n\n```erlang\n1> Ss = sofs:from_term([[a,b],[b,c]]),\nCR = sofs:canonical_relation(Ss),\nsofs:to_external(CR).\n[{a,[a,b]},{b,[a,b]},{b,[b,c]},{c,[b,c]}]\n```","ref":"sofs.html#canonical_relation/1"},{"type":"function","title":"sofs.composite/2","doc":"Returns the [composite](`m:sofs#composite`) of the functions `Function1` and\n`Function2`.\n\n```erlang\n1> F1 = sofs:a_function([{a,1},{b,2},{c,2}]),\nF2 = sofs:a_function([{1,x},{2,y},{3,z}]),\nF = sofs:composite(F1, F2),\nsofs:to_external(F).\n[{a,x},{b,y},{c,y}]\n```","ref":"sofs.html#composite/2"},{"type":"function","title":"sofs.constant_function/2","doc":"Creates the [function](`m:sofs#function`) that maps each element of set `Set`\nonto `AnySet`.\n\n```erlang\n1> S = sofs:set([a,b]),\nE = sofs:from_term(1),\nR = sofs:constant_function(S, E),\nsofs:to_external(R).\n[{a,1},{b,1}]\n```","ref":"sofs.html#constant_function/2"},{"type":"function","title":"sofs.converse/1","doc":"Returns the [converse](`m:sofs#converse`) of the binary relation `BinRel1`.\n\n```erlang\n1> R1 = sofs:relation([{1,a},{2,b},{3,a}]),\nR2 = sofs:converse(R1),\nsofs:to_external(R2).\n[{a,1},{a,3},{b,2}]\n```","ref":"sofs.html#converse/1"},{"type":"function","title":"sofs.difference/2","doc":"Returns the [difference](`m:sofs#difference`) of the sets `Set1` and `Set2`.","ref":"sofs.html#difference/2"},{"type":"function","title":"sofs.digraph_to_family/1","doc":"","ref":"sofs.html#digraph_to_family/1"},{"type":"function","title":"sofs.digraph_to_family/2","doc":"Creates a [family](`m:sofs#family`) from the directed graph `Graph`. Each vertex\na of `Graph` is represented by a pair (a, \\{b\\[1], ..., b\\[n]\\}), where the\nb\\[i]:s are the out-neighbors of a. It is assumed that `Type` is\na [valid type](`m:sofs#valid_type`) of the external set of the family.\n\nIf G is a directed graph, it holds that the vertices and edges of G are the same\nas the vertices and edges of\n[`family_to_digraph(digraph_to_family(G))`](`family_to_digraph/1`).","ref":"sofs.html#digraph_to_family/2"},{"type":"function","title":"sofs.domain/1","doc":"Returns the [domain](`m:sofs#domain`) of the binary relation `BinRel`.\n\n```erlang\n1> R = sofs:relation([{1,a},{1,b},{2,b},{2,c}]),\nS = sofs:domain(R),\nsofs:to_external(S).\n[1,2]\n```","ref":"sofs.html#domain/1"},{"type":"function","title":"sofs.drestriction/2","doc":"Returns the difference between the binary relation `BinRel1` and the\n[restriction](`m:sofs#restriction`) of `BinRel1` to `Set`.\n\n```erlang\n1> R1 = sofs:relation([{1,a},{2,b},{3,c}]),\nS = sofs:set([2,4,6]),\nR2 = sofs:drestriction(R1, S),\nsofs:to_external(R2).\n[{1,a},{3,c}]\n```\n\n[`drestriction(R, S)`](`drestriction/2`) is equivalent to\n[`difference(R, restriction(R, S))`](`difference/2`).","ref":"sofs.html#drestriction/2"},{"type":"function","title":"sofs.drestriction/3","doc":"Returns a subset of `Set1` containing those elements that do not give an element\nin `Set2` as the result of applying `SetFun`.\n\n```erlang\n1> SetFun = {external, fun({_A,B,C}) -> {B,C} end},\nR1 = sofs:relation([{a,aa,1},{b,bb,2},{c,cc,3}]),\nR2 = sofs:relation([{bb,2},{cc,3},{dd,4}]),\nR3 = sofs:drestriction(SetFun, R1, R2),\nsofs:to_external(R3).\n[{a,aa,1}]\n```\n\n[`drestriction(F, S1, S2)`](`drestriction/3`) is equivalent to\n[`difference(S1, restriction(F, S1, S2))`](`difference/2`).","ref":"sofs.html#drestriction/3"},{"type":"function","title":"sofs.empty_set/0","doc":"Returns the [untyped empty set](`m:sofs#sets_definition`). `empty_set/0` is\nequivalent to [`from_term([], ['_'])`](`from_term/2`).","ref":"sofs.html#empty_set/0"},{"type":"function","title":"sofs.extension/3","doc":"Returns the [extension](`m:sofs#extension`) of `BinRel1` such that for each\nelement E in `Set` that does not belong to the [domain](`m:sofs#domain`) of\n`BinRel1`, `BinRel2` contains the pair (E, `AnySet`).\n\n```erlang\n1> S = sofs:set([b,c]),\nA = sofs:empty_set(),\nR = sofs:family([{a,[1,2]},{b,[3]}]),\nX = sofs:extension(R, S, A),\nsofs:to_external(X).\n[{a,[1,2]},{b,[3]},{c,[]}]\n```","ref":"sofs.html#extension/3"},{"type":"type","title":"sofs.external_set/0","doc":"An [external set](`m:sofs#external_set`).","ref":"sofs.html#t:external_set/0"},{"type":"type","title":"sofs.family/0","doc":"A [family](`m:sofs#family`) (of subsets).","ref":"sofs.html#t:family/0"},{"type":"function","title":"sofs.family/1","doc":"","ref":"sofs.html#family/1"},{"type":"function","title":"sofs.family/2","doc":"Creates a [family of subsets](`m:sofs#family`). [`family(F, T)`](`family/2`) is\nequivalent to [`from_term(F, T)`](`from_term/2`) if the result is a family.","ref":"sofs.html#family/2"},{"type":"function","title":"sofs.family_difference/2","doc":"If `Family1` and `Family2` are [families](`m:sofs#family`), then `Family3` is\nthe family such that the index set is equal to the index set of `Family1`, and\n`Family3`\\[i] is the difference between `Family1`\\[i] and `Family2`\\[i] if\n`Family2` maps i, otherwise `Family1[i]`.\n\n```erlang\n1> F1 = sofs:family([{a,[1,2]},{b,[3,4]}]),\nF2 = sofs:family([{b,[4,5]},{c,[6,7]}]),\nF3 = sofs:family_difference(F1, F2),\nsofs:to_external(F3).\n[{a,[1,2]},{b,[3]}]\n```","ref":"sofs.html#family_difference/2"},{"type":"function","title":"sofs.family_domain/1","doc":"If `Family1` is a [family](`m:sofs#family`) and `Family1`\\[i] is a binary\nrelation for every i in the index set of `Family1`, then `Family2` is the family\nwith the same index set as `Family1` such that `Family2`\\[i] is the\n[domain](`m:sofs#domain`) of `Family1[i]`.\n\n```erlang\n1> FR = sofs:from_term([{a,[{1,a},{2,b},{3,c}]},{b,[]},{c,[{4,d},{5,e}]}]),\nF = sofs:family_domain(FR),\nsofs:to_external(F).\n[{a,[1,2,3]},{b,[]},{c,[4,5]}]\n```","ref":"sofs.html#family_domain/1"},{"type":"function","title":"sofs.family_field/1","doc":"If `Family1` is a [family](`m:sofs#family`) and `Family1`\\[i] is a binary\nrelation for every i in the index set of `Family1`, then `Family2` is the family\nwith the same index set as `Family1` such that `Family2`\\[i] is the\n[field](`m:sofs#field`) of `Family1`\\[i].\n\n```erlang\n1> FR = sofs:from_term([{a,[{1,a},{2,b},{3,c}]},{b,[]},{c,[{4,d},{5,e}]}]),\nF = sofs:family_field(FR),\nsofs:to_external(F).\n[{a,[1,2,3,a,b,c]},{b,[]},{c,[4,5,d,e]}]\n```\n\n[`family_field(Family1)`](`family_field/1`) is equivalent to\n[`family_union(family_domain(Family1), family_range(Family1))`](`family_union/2`).","ref":"sofs.html#family_field/1"},{"type":"function","title":"sofs.family_intersection/1","doc":"If `Family1` is a [family](`m:sofs#family`) and `Family1`\\[i] is a set of sets\nfor every i in the index set of `Family1`, then `Family2` is the family with the\nsame index set as `Family1` such that `Family2`\\[i] is the\n[intersection](`m:sofs#intersection_n`) of `Family1`\\[i].\n\nIf `Family1`\\[i] is an empty set for some i, the process exits with a `badarg`\nmessage.\n\n```erlang\n1> F1 = sofs:from_term([{a,[[1,2,3],[2,3,4]]},{b,[[x,y,z],[x,y]]}]),\nF2 = sofs:family_intersection(F1),\nsofs:to_external(F2).\n[{a,[2,3]},{b,[x,y]}]\n```","ref":"sofs.html#family_intersection/1"},{"type":"function","title":"sofs.family_intersection/2","doc":"If `Family1` and `Family2` are [families](`m:sofs#family`), then `Family3` is\nthe family such that the index set is the intersection of `Family1`:s and\n`Family2`:s index sets, and `Family3`\\[i] is the intersection of `Family1`\\[i]\nand `Family2`\\[i].\n\n```erlang\n1> F1 = sofs:family([{a,[1,2]},{b,[3,4]},{c,[5,6]}]),\nF2 = sofs:family([{b,[4,5]},{c,[7,8]},{d,[9,10]}]),\nF3 = sofs:family_intersection(F1, F2),\nsofs:to_external(F3).\n[{b,[4]},{c,[]}]\n```","ref":"sofs.html#family_intersection/2"},{"type":"function","title":"sofs.family_projection/2","doc":"If `Family1` is a [family](`m:sofs#family`), then `Family2` is the family with\nthe same index set as `Family1` such that `Family2`\\[i] is the result of calling\n`SetFun` with `Family1`\\[i] as argument.\n\n```erlang\n1> F1 = sofs:from_term([{a,[[1,2],[2,3]]},{b,[[]]}]),\nF2 = sofs:family_projection(fun sofs:union/1, F1),\nsofs:to_external(F2).\n[{a,[1,2,3]},{b,[]}]\n```","ref":"sofs.html#family_projection/2"},{"type":"function","title":"sofs.family_range/1","doc":"If `Family1` is a [family](`m:sofs#family`) and `Family1`\\[i] is a binary\nrelation for every i in the index set of `Family1`, then `Family2` is the family\nwith the same index set as `Family1` such that `Family2`\\[i] is the\n[range](`m:sofs#range`) of `Family1`\\[i].\n\n```erlang\n1> FR = sofs:from_term([{a,[{1,a},{2,b},{3,c}]},{b,[]},{c,[{4,d},{5,e}]}]),\nF = sofs:family_range(FR),\nsofs:to_external(F).\n[{a,[a,b,c]},{b,[]},{c,[d,e]}]\n```","ref":"sofs.html#family_range/1"},{"type":"function","title":"sofs.family_specification/2","doc":"If `Family1` is a [family](`m:sofs#family`), then `Family2` is the\n[restriction](`m:sofs#restriction`) of `Family1` to those elements i of the\nindex set for which `Fun` applied to `Family1`\\[i] returns `true`. If `Fun` is a\ntuple `{external, Fun2}`, then `Fun2` is applied to the\n[external set](`m:sofs#external_set`) of `Family1`\\[i], otherwise `Fun` is\napplied to `Family1`\\[i].\n\n```erlang\n1> F1 = sofs:family([{a,[1,2,3]},{b,[1,2]},{c,[1]}]),\nSpecFun = fun(S) -> sofs:no_elements(S) =:= 2 end,\nF2 = sofs:family_specification(SpecFun, F1),\nsofs:to_external(F2).\n[{b,[1,2]}]\n```","ref":"sofs.html#family_specification/2"},{"type":"function","title":"sofs.family_to_digraph/1","doc":"","ref":"sofs.html#family_to_digraph/1"},{"type":"function","title":"sofs.family_to_digraph/2","doc":"Creates a directed graph from [family](`m:sofs#family`) `Family`. For each pair\n(a, \\{b\\[1], ..., b\\[n]\\}) of `Family`, vertex a and the edges (a, b\\[i]) for\n1 <= i <= n are added to a newly created directed graph.\n\n`GraphType` is passed on to `digraph:new/1`.\n\nIt F is a family, it holds that F is a subset of\n[`digraph_to_family(family_to_digraph(F), type(F))`](`digraph_to_family/2`).\nEquality holds if [`union_of_family(F)`](`union_of_family/1`) is a subset of\n[`domain(F)`](`domain/1`).\n\nCreating a cycle in an acyclic graph exits the process with a `cyclic` message.","ref":"sofs.html#family_to_digraph/2"},{"type":"function","title":"sofs.family_to_relation/1","doc":"If `Family` is a [family](`m:sofs#family`), then `BinRel` is the binary relation\ncontaining all pairs (i, x) such that i belongs to the index set of `Family` and\nx belongs to `Family`\\[i].\n\n```erlang\n1> F = sofs:family([{a,[]}, {b,[1]}, {c,[2,3]}]),\nR = sofs:family_to_relation(F),\nsofs:to_external(R).\n[{b,1},{c,2},{c,3}]\n```","ref":"sofs.html#family_to_relation/1"},{"type":"function","title":"sofs.family_union/1","doc":"If `Family1` is a [family](`m:sofs#family`) and `Family1`\\[i] is a set of sets\nfor each i in the index set of `Family1`, then `Family2` is the family with the\nsame index set as `Family1` such that `Family2`\\[i] is the\n[union](`m:sofs#union_n`) of `Family1`\\[i].\n\n```erlang\n1> F1 = sofs:from_term([{a,[[1,2],[2,3]]},{b,[[]]}]),\nF2 = sofs:family_union(F1),\nsofs:to_external(F2).\n[{a,[1,2,3]},{b,[]}]\n```\n\n[`family_union(F)`](`family_union/1`) is equivalent to\n[`family_projection(fun sofs:union/1, F)`](`family_projection/2`).","ref":"sofs.html#family_union/1"},{"type":"function","title":"sofs.family_union/2","doc":"If `Family1` and `Family2` are [families](`m:sofs#family`), then `Family3` is\nthe family such that the index set is the union of `Family1`:s and `Family2`:s\nindex sets, and `Family3`\\[i] is the union of `Family1`\\[i] and `Family2`\\[i] if\nboth map i, otherwise `Family1`\\[i] or `Family2`\\[i].\n\n```erlang\n1> F1 = sofs:family([{a,[1,2]},{b,[3,4]},{c,[5,6]}]),\nF2 = sofs:family([{b,[4,5]},{c,[7,8]},{d,[9,10]}]),\nF3 = sofs:family_union(F1, F2),\nsofs:to_external(F3).\n[{a,[1,2]},{b,[3,4,5]},{c,[5,6,7,8]},{d,[9,10]}]\n```","ref":"sofs.html#family_union/2"},{"type":"function","title":"sofs.field/1","doc":"Returns the [field](`m:sofs#field`) of the binary relation `BinRel`.\n\n```erlang\n1> R = sofs:relation([{1,a},{1,b},{2,b},{2,c}]),\nS = sofs:field(R),\nsofs:to_external(S).\n[1,2,a,b,c]\n```\n\n[`field(R)`](`field/1`) is equivalent to\n[`union(domain(R), range(R))`](`union/2`).","ref":"sofs.html#field/1"},{"type":"function","title":"sofs.from_external/2","doc":"Creates a set from the [external set](`m:sofs#external_set`) `ExternalSet` and\nthe [type](`m:sofs#type`) `Type`. It is assumed that `Type` is a\n[valid type](`m:sofs#valid_type`) of `ExternalSet`.","ref":"sofs.html#from_external/2"},{"type":"function","title":"sofs.from_sets/1","doc":"Returns the [unordered set](`m:sofs#sets_definition`) containing the sets of\nlist `ListOfSets`.\n\n```erlang\n1> S1 = sofs:relation([{a,1},{b,2}]),\nS2 = sofs:relation([{x,3},{y,4}]),\nS = sofs:from_sets([S1,S2]),\nsofs:to_external(S).\n[[{a,1},{b,2}],[{x,3},{y,4}]]\n```\n\nReturns the [ordered set](`m:sofs#sets_definition`) containing the sets of the\nnon-empty tuple `TupleOfSets`.","ref":"sofs.html#from_sets/1"},{"type":"function","title":"sofs.from_term/1","doc":"","ref":"sofs.html#from_term/1"},{"type":"function","title":"sofs.from_term/2","doc":"Creates an element of [Sets](`m:sofs#sets_definition`) by\ntraversing term `Term`, sorting lists, removing duplicates, and deriving or\nverifying a [valid type](`m:sofs#valid_type`) for the so obtained external set.\n\nAn explicitly specified [type](`m:sofs#type`) `Type` can be used to limit the\ndepth of the traversal; an atomic type stops the traversal, as shown by the\nfollowing example where `\"foo\"` and `{\"foo\"}` are left unmodified:\n\n```erlang\n1> S = sofs:from_term([{{\"foo\"},[1,1]},{\"foo\",[2,2]}],\n                      [{atom,[atom]}]),\n   sofs:to_external(S).\n[{{\"foo\"},[1]},{\"foo\",[2]}]\n```\n\n`from_term` can be used for creating atomic or ordered sets. The only purpose of\nsuch a set is that of later building unordered sets, as all functions in this\nmodule that _do_ anything operate on unordered sets. Creating unordered sets\nfrom a collection of ordered sets can be the way to go if the ordered sets are\nbig and one does not want to waste heap by rebuilding the elements of the\nunordered set. The following example shows that a set can be built \"layer by\nlayer\":\n\n```erlang\n1> A = sofs:from_term(a),\nS = sofs:set([1,2,3]),\nP1 = sofs:from_sets({A,S}),\nP2 = sofs:from_term({b,[6,5,4]}),\nSs = sofs:from_sets([P1,P2]),\nsofs:to_external(Ss).\n[{a,[1,2,3]},{b,[4,5,6]}]\n```\n\nOther functions that create sets are `from_external/2` and `from_sets/1`.\nSpecial cases of [`from_term/2`](`from_term/2`) are\n[`a_function/1,2`](`a_function/1`), `empty_set/0`, [`family/1,2`](`family/1`),\n[`relation/1,2`](`relation/1`), and [`set/1,2`](`set/1`).","ref":"sofs.html#from_term/2"},{"type":"function","title":"sofs.image/2","doc":"Returns the [image](`m:sofs#image`) of set `Set1` under the binary relation\n`BinRel`.\n\n```erlang\n1> R = sofs:relation([{1,a},{2,b},{2,c},{3,d}]),\nS1 = sofs:set([1,2]),\nS2 = sofs:image(R, S1),\nsofs:to_external(S2).\n[a,b,c]\n```","ref":"sofs.html#image/2"},{"type":"function","title":"sofs.intersection/1","doc":"Returns the [intersection](`m:sofs#intersection_n`) of the set of sets\n`SetOfSets`.\n\nIntersecting an empty set of sets exits the process with a `badarg` message.","ref":"sofs.html#intersection/1"},{"type":"function","title":"sofs.intersection/2","doc":"Returns the [intersection](`m:sofs#intersection`) of `Set1` and `Set2`.","ref":"sofs.html#intersection/2"},{"type":"function","title":"sofs.intersection_of_family/1","doc":"Returns the intersection of [family](`m:sofs#family`) `Family`.\n\nIntersecting an empty family exits the process with a `badarg` message.\n\n```erlang\n1> F = sofs:family([{a,[0,2,4]},{b,[0,1,2]},{c,[2,3]}]),\nS = sofs:intersection_of_family(F),\nsofs:to_external(S).\n[2]\n```","ref":"sofs.html#intersection_of_family/1"},{"type":"function","title":"sofs.inverse/1","doc":"Returns the [inverse](`m:sofs#inverse`) of function `Function1`.\n\n```erlang\n1> R1 = sofs:relation([{1,a},{2,b},{3,c}]),\nR2 = sofs:inverse(R1),\nsofs:to_external(R2).\n[{a,1},{b,2},{c,3}]\n```","ref":"sofs.html#inverse/1"},{"type":"function","title":"sofs.inverse_image/2","doc":"Returns the [inverse image](`m:sofs#inverse_image`) of `Set1` under the binary\nrelation `BinRel`.\n\n```erlang\n1> R = sofs:relation([{1,a},{2,b},{2,c},{3,d}]),\nS1 = sofs:set([c,d,e]),\nS2 = sofs:inverse_image(R, S1),\nsofs:to_external(S2).\n[2,3]\n```","ref":"sofs.html#inverse_image/2"},{"type":"function","title":"sofs.is_a_function/1","doc":"Returns `true` if the binary relation `BinRel` is a\n[function](`m:sofs#function`) or the untyped empty set, otherwise `false`.","ref":"sofs.html#is_a_function/1"},{"type":"function","title":"sofs.is_disjoint/2","doc":"Returns `true` if `Set1` and `Set2` are [disjoint](`m:sofs#disjoint`), otherwise\n`false`.","ref":"sofs.html#is_disjoint/2"},{"type":"function","title":"sofs.is_empty_set/1","doc":"Returns `true` if `AnySet` is an empty unordered set, otherwise `false`.","ref":"sofs.html#is_empty_set/1"},{"type":"function","title":"sofs.is_equal/2","doc":"Returns `true` if `AnySet1` and `AnySet2` are [equal](`m:sofs#equal`), otherwise\n`false`. The following example shows that `==/2` is used when comparing sets for\nequality:\n\n```erlang\n1> S1 = sofs:set([1.0]),\nS2 = sofs:set([1]),\nsofs:is_equal(S1, S2).\ntrue\n```","ref":"sofs.html#is_equal/2"},{"type":"function","title":"sofs.is_set/1","doc":"Returns `true` if `AnySet` appears to be an\n[unordered set](`m:sofs#sets_definition`), and `false` if `AnySet` is an ordered\nset or an atomic set or any other term.\n\nNote that the test is shallow and this function will return `true` for any term\nthat coincides with the representation of an unordered set. See also note on\n[data types](`e:system:data_types.md#no_user_types`).","ref":"sofs.html#is_set/1"},{"type":"function","title":"sofs.is_sofs_set/1","doc":"Returns `true` if `Term` appears to be an\n[unordered set](`m:sofs#sets_definition`), an ordered set, or an atomic set,\notherwise `false`.\n\nNote that this function will return `true` for any term that\ncoincides with the representation of a `sofs` set. See also note on\n[data types](`e:system:data_types.md#no_user_types`).","ref":"sofs.html#is_sofs_set/1"},{"type":"function","title":"sofs.is_subset/2","doc":"Returns `true` if `Set1` is a [subset](`m:sofs#subset`) of `Set2`, otherwise\n`false`.","ref":"sofs.html#is_subset/2"},{"type":"function","title":"sofs.is_type/1","doc":"Returns `true` if term `Term` is a [type](`m:sofs#type`).","ref":"sofs.html#is_type/1"},{"type":"function","title":"sofs.join/4","doc":"Returns the [natural join](`m:sofs#natural_join`) of the relations `Relation1`\nand `Relation2` on coordinates `I` and `J`.\n\n```erlang\n1> R1 = sofs:relation([{a,x,1},{b,y,2}]),\nR2 = sofs:relation([{1,f,g},{1,h,i},{2,3,4}]),\nJ = sofs:join(R1, 3, R2, 1),\nsofs:to_external(J).\n[{a,x,1,f,g},{a,x,1,h,i},{b,y,2,3,4}]\n```","ref":"sofs.html#join/4"},{"type":"function","title":"sofs.multiple_relative_product/2","doc":"If `TupleOfBinRels` is a non-empty tuple \\{R\\[1], ..., R\\[n]\\} of binary\nrelations and `BinRel1` is a binary relation, then `BinRel2` is the\n[multiple relative product](`m:sofs#multiple_relative_product`) of the ordered\nset (R\\[i], ..., R\\[n]) and `BinRel1`.\n\n```erlang\n1> Ri = sofs:relation([{a,1},{b,2},{c,3}]),\nR = sofs:relation([{a,b},{b,c},{c,a}]),\nMP = sofs:multiple_relative_product({Ri, Ri}, R),\nsofs:to_external(sofs:range(MP)).\n[{1,2},{2,3},{3,1}]\n```","ref":"sofs.html#multiple_relative_product/2"},{"type":"function","title":"sofs.no_elements/1","doc":"Returns the number of elements of the ordered or unordered set `ASet`.","ref":"sofs.html#no_elements/1"},{"type":"opaque","title":"sofs.ordset/0","doc":"An [ordered set](`m:sofs#sets_definition`).","ref":"sofs.html#t:ordset/0"},{"type":"function","title":"sofs.partition/1","doc":"Returns the [partition](`m:sofs#partition`) of the union of the set of sets\n`SetOfSets` such that two elements are considered equal if they belong to the\nsame elements of `SetOfSets`.\n\n```erlang\n1> Sets1 = sofs:from_term([[a,b,c],[d,e,f],[g,h,i]]),\nSets2 = sofs:from_term([[b,c,d],[e,f,g],[h,i,j]]),\nP = sofs:partition(sofs:union(Sets1, Sets2)),\nsofs:to_external(P).\n[[a],[b,c],[d],[e,f],[g],[h,i],[j]]\n```","ref":"sofs.html#partition/1"},{"type":"function","title":"sofs.partition/2","doc":"Returns the [partition](`m:sofs#partition`) of `Set` such that two elements are\nconsidered equal if the results of applying `SetFun` are equal.\n\n```erlang\n1> Ss = sofs:from_term([[a],[b],[c,d],[e,f]]),\nSetFun = fun(S) -> sofs:from_term(sofs:no_elements(S)) end,\nP = sofs:partition(SetFun, Ss),\nsofs:to_external(P).\n[[[a],[b]],[[c,d],[e,f]]]\n```","ref":"sofs.html#partition/2"},{"type":"function","title":"sofs.partition/3","doc":"Returns a pair of sets that, regarded as constituting a set, forms a\n[partition](`m:sofs#partition`) of `Set1`. If the result of applying `SetFun` to\nan element of `Set1` gives an element in `Set2`, the element belongs to `Set3`,\notherwise the element belongs to `Set4`.\n\n```erlang\n1> R1 = sofs:relation([{1,a},{2,b},{3,c}]),\nS = sofs:set([2,4,6]),\n{R2,R3} = sofs:partition(1, R1, S),\n{sofs:to_external(R2),sofs:to_external(R3)}.\n{[{2,b}],[{1,a},{3,c}]}\n```\n\n[`partition(F, S1, S2)`](`partition/3`) is equivalent to\n`{restriction(F, S1, S2), drestriction(F, S1, S2)}`.","ref":"sofs.html#partition/3"},{"type":"function","title":"sofs.partition_family/2","doc":"Returns [family](`m:sofs#family`) `Family` where the indexed set is a\n[partition](`m:sofs#partition`) of `Set` such that two elements are considered\nequal if the results of applying `SetFun` are the same value i. This i is the\nindex that `Family` maps onto the\n[equivalence class](`m:sofs#equivalence_class`).\n\n```erlang\n1> S = sofs:relation([{a,a,a,a},{a,a,b,b},{a,b,b,b}]),\nSetFun = {external, fun({A,_,C,_}) -> {A,C} end},\nF = sofs:partition_family(SetFun, S),\nsofs:to_external(F).\n[{{a,a},[{a,a,a,a}]},{{a,b},[{a,a,b,b},{a,b,b,b}]}]\n```","ref":"sofs.html#partition_family/2"},{"type":"function","title":"sofs.product/1","doc":"Returns the [Cartesian product](`m:sofs#Cartesian_product_tuple`) of the\nnon-empty tuple of sets `TupleOfSets`. If (x\\[1], ..., x\\[n]) is an element of\nthe n-ary relation `Relation`, then x\\[i] is drawn from element i of\n`TupleOfSets`.\n\n```erlang\n1> S1 = sofs:set([a,b]),\nS2 = sofs:set([1,2]),\nS3 = sofs:set([x,y]),\nP3 = sofs:product({S1,S2,S3}),\nsofs:to_external(P3).\n[{a,1,x},{a,1,y},{a,2,x},{a,2,y},{b,1,x},{b,1,y},{b,2,x},{b,2,y}]\n```","ref":"sofs.html#product/1"},{"type":"function","title":"sofs.product/2","doc":"Returns the [Cartesian product](`m:sofs#Cartesian_product`) of `Set1` and\n`Set2`.\n\n```erlang\n1> S1 = sofs:set([1,2]),\nS2 = sofs:set([a,b]),\nR = sofs:product(S1, S2),\nsofs:to_external(R).\n[{1,a},{1,b},{2,a},{2,b}]\n```\n\n[`product(S1, S2)`](`product/2`) is equivalent to\n[`product({S1, S2})`](`product/1`).","ref":"sofs.html#product/2"},{"type":"function","title":"sofs.projection/2","doc":"Returns the set created by substituting each element of `Set1` by the result of\napplying `SetFun` to the element.\n\nIf `SetFun` is a number i >= 1 and `Set1` is a relation, then the returned set\nis the [projection](`m:sofs#projection`) of `Set1` onto coordinate i.\n\n```erlang\n1> S1 = sofs:from_term([{1,a},{2,b},{3,a}]),\nS2 = sofs:projection(2, S1),\nsofs:to_external(S2).\n[a,b]\n```","ref":"sofs.html#projection/2"},{"type":"function","title":"sofs.range/1","doc":"Returns the [range](`m:sofs#range`) of the binary relation `BinRel`.\n\n```erlang\n1> R = sofs:relation([{1,a},{1,b},{2,b},{2,c}]),\nS = sofs:range(R),\nsofs:to_external(S).\n[a,b,c]\n```","ref":"sofs.html#range/1"},{"type":"type","title":"sofs.relation/0","doc":"An [n-ary relation](`m:sofs#n_ary_relation`).","ref":"sofs.html#t:relation/0"},{"type":"function","title":"sofs.relation/1","doc":"Equivalent to [`relation(Tuples, Type)`](`relation/2`) where `Type` is the size\nof the first tuple of `Tuples` is used if there is such a tuple.\n\nIf tuples is `[]`, then `Type` is `2`.","ref":"sofs.html#relation/1"},{"type":"function","title":"sofs.relation/2","doc":"Creates a [relation](`m:sofs#relation`). [`relation(R, T)`](`relation/2`) is\nequivalent to [`from_term(R, T)`](`from_term/2`), if T is a\n[type](`m:sofs#type`) and the result is a relation.\n\nIf `Type` is an integer N, then `[{atom, ..., atom}])`, where the tuple size is N,\nis used as type of the relation.","ref":"sofs.html#relation/2"},{"type":"function","title":"sofs.relation_to_family/1","doc":"Returns [family](`m:sofs#family`) `Family` such that the index set is equal to\nthe [domain](`m:sofs#domain`) of the binary relation `BinRel`, and `Family`\\[i]\nis the [image](`m:sofs#image`) of the set of i under `BinRel`.\n\n```erlang\n1> R = sofs:relation([{b,1},{c,2},{c,3}]),\nF = sofs:relation_to_family(R),\nsofs:to_external(F).\n[{b,[1]},{c,[2,3]}]\n```","ref":"sofs.html#relation_to_family/1"},{"type":"function","title":"sofs.relative_product1/2","doc":"Returns the [relative product](`m:sofs#relative_product`) of the\n[converse](`m:sofs#converse`) of the binary relation `BinRel1` and the binary\nrelation `BinRel2`.\n\n```erlang\n1> R1 = sofs:relation([{1,a},{1,aa},{2,b}]),\nR2 = sofs:relation([{1,u},{2,v},{3,c}]),\nR3 = sofs:relative_product1(R1, R2),\nsofs:to_external(R3).\n[{a,u},{aa,u},{b,v}]\n```\n\n[`relative_product1(R1, R2)`](`relative_product1/2`) is equivalent to\n[`relative_product(converse(R1), R2)`](`relative_product/2`).","ref":"sofs.html#relative_product1/2"},{"type":"function","title":"sofs.relative_product/1","doc":"","ref":"sofs.html#relative_product/1"},{"type":"function","title":"sofs.relative_product/2","doc":"If `ListOfBinRels` is a non-empty list \\[R[1], ..., R\\[n]] of binary relations\nand `BinRel1` is a binary relation, then `BinRel2` is the\n[relative product](`m:sofs#tuple_relative_product`) of the ordered set\n(R\\[i], ..., R\\[n]) and `BinRel1`.\n\nIf `BinRel1` is omitted, the relation of equality between the elements of the\n[Cartesian product](`m:sofs#Cartesian_product_tuple`) of the ranges of R\\[i],\nrange R\\[1] × ... × range R\\[n], is used instead (intuitively, nothing is\n\"lost\").\n\n```erlang\n1> TR = sofs:relation([{1,a},{1,aa},{2,b}]),\nR1 = sofs:relation([{1,u},{2,v},{3,c}]),\nR2 = sofs:relative_product([TR, R1]),\nsofs:to_external(R2).\n[{1,{a,u}},{1,{aa,u}},{2,{b,v}}]\n```\n\nNotice that [`relative_product([R1], R2)`](`relative_product/2`) is different\nfrom [`relative_product(R1, R2)`](`relative_product/2`); the list of one element\nis not identified with the element itself.\n\nReturns the [relative product](`m:sofs#relative_product`) of the binary\nrelations `BinRel1` and `BinRel2`.","ref":"sofs.html#relative_product/2"},{"type":"function","title":"sofs.restriction/2","doc":"Returns the [restriction](`m:sofs#restriction`) of the binary relation `BinRel1`\nto `Set`.\n\n```erlang\n1> R1 = sofs:relation([{1,a},{2,b},{3,c}]),\nS = sofs:set([1,2,4]),\nR2 = sofs:restriction(R1, S),\nsofs:to_external(R2).\n[{1,a},{2,b}]\n```","ref":"sofs.html#restriction/2"},{"type":"function","title":"sofs.restriction/3","doc":"Returns a subset of `Set1` containing those elements that gives an element in\n`Set2` as the result of applying `SetFun`.\n\n```erlang\n1> S1 = sofs:relation([{1,a},{2,b},{3,c}]),\nS2 = sofs:set([b,c,d]),\nS3 = sofs:restriction(2, S1, S2),\nsofs:to_external(S3).\n[{2,b},{3,c}]\n```","ref":"sofs.html#restriction/3"},{"type":"function","title":"sofs.set/1","doc":"","ref":"sofs.html#set/1"},{"type":"function","title":"sofs.set/2","doc":"Creates an [unordered set](`m:sofs#sets_definition`). [`set(L, T)`](`set/2`) is\nequivalent to [`from_term(L, T)`](`from_term/2`), if the result is an unordered\nset.","ref":"sofs.html#set/2"},{"type":"type","title":"sofs.set_fun/0","doc":"A [SetFun](`m:sofs#set_fun`).","ref":"sofs.html#t:set_fun/0"},{"type":"type","title":"sofs.set_of_sets/0","doc":"An [unordered set](`m:sofs#sets_definition`) of unordered sets.","ref":"sofs.html#t:set_of_sets/0"},{"type":"type","title":"sofs.spec_fun/0","doc":"","ref":"sofs.html#t:spec_fun/0"},{"type":"function","title":"sofs.specification/2","doc":"Returns the set containing every element of `Set1` for which `Fun` returns\n`true`. If `Fun` is a tuple `{external, Fun2}`, `Fun2` is applied to the\n[external set](`m:sofs#external_set`) of each element, otherwise `Fun` is\napplied to each element.\n\n```erlang\n1> R1 = sofs:relation([{a,1},{b,2}]),\nR2 = sofs:relation([{x,1},{x,2},{y,3}]),\nS1 = sofs:from_sets([R1,R2]),\nS2 = sofs:specification(fun sofs:is_a_function/1, S1),\nsofs:to_external(S2).\n[[{a,1},{b,2}]]\n```","ref":"sofs.html#specification/2"},{"type":"function","title":"sofs.strict_relation/1","doc":"Returns the [strict relation](`m:sofs#strict_relation`) corresponding to the\nbinary relation `BinRel1`.\n\n```erlang\n1> R1 = sofs:relation([{1,1},{1,2},{2,1},{2,2}]),\nR2 = sofs:strict_relation(R1),\nsofs:to_external(R2).\n[{1,2},{2,1}]\n```","ref":"sofs.html#strict_relation/1"},{"type":"function","title":"sofs.substitution/2","doc":"Returns a function, the domain of which is `Set1`. The value of an element of\nthe domain is the result of applying `SetFun` to the element.\n\n```erlang\n1> L = [{a,1},{b,2}].\n[{a,1},{b,2}]\n2> sofs:to_external(sofs:projection(1,sofs:relation(L))).\n[a,b]\n3> sofs:to_external(sofs:substitution(1,sofs:relation(L))).\n[{{a,1},a},{{b,2},b}]\n4> SetFun = {external, fun({A,_}=E) -> {E,A} end},\nsofs:to_external(sofs:projection(SetFun,sofs:relation(L))).\n[{{a,1},a},{{b,2},b}]\n```\n\nThe relation of equality between the elements of \\{a,b,c\\}:\n\n```erlang\n1> I = sofs:substitution(fun(A) -> A end, sofs:set([a,b,c])),\nsofs:to_external(I).\n[{a,a},{b,b},{c,c}]\n```\n\nLet `SetOfSets` be a set of sets and `BinRel` a binary relation. The function\nthat maps each element `Set` of `SetOfSets` onto the [image](`m:sofs#image`) of\n`Set` under `BinRel` is returned by the following function:\n\n```erlang\nimages(SetOfSets, BinRel) ->\n   Fun = fun(Set) -> sofs:image(BinRel, Set) end,\n   sofs:substitution(Fun, SetOfSets).\n```\n\nExternal unordered sets are represented as sorted lists. So, creating the image\nof a set under a relation R can traverse all elements of R (to that comes the\nsorting of results, the image). In `image/2`, `BinRel` is traversed once for\neach element of `SetOfSets`, which can take too long. The following efficient\nfunction can be used instead under the assumption that the image of each element\nof `SetOfSets` under `BinRel` is non-empty:\n\n```erlang\nimages2(SetOfSets, BinRel) ->\n   CR = sofs:canonical_relation(SetOfSets),\n   R = sofs:relative_product1(CR, BinRel),\n   sofs:relation_to_family(R).\n```","ref":"sofs.html#substitution/2"},{"type":"function","title":"sofs.symdiff/2","doc":"Returns the [symmetric difference](`m:sofs#symmetric_difference`) (or the\nBoolean sum) of `Set1` and `Set2`.\n\n```erlang\n1> S1 = sofs:set([1,2,3]),\nS2 = sofs:set([2,3,4]),\nP = sofs:symdiff(S1, S2),\nsofs:to_external(P).\n[1,4]\n```","ref":"sofs.html#symdiff/2"},{"type":"function","title":"sofs.symmetric_partition/2","doc":"Returns a triple of sets:\n\n- `Set3` contains the elements of `Set1` that do not belong to `Set2`.\n- `Set4` contains the elements of `Set1` that belong to `Set2`.\n- `Set5` contains the elements of `Set2` that do not belong to `Set1`.","ref":"sofs.html#symmetric_partition/2"},{"type":"function","title":"sofs.to_external/1","doc":"Returns the [external set](`m:sofs#external_set`) of an atomic, ordered, or\nunordered set.","ref":"sofs.html#to_external/1"},{"type":"function","title":"sofs.to_sets/1","doc":"Returns the elements of the ordered set `ASet` as a tuple of sets, and the\nelements of the unordered set `ASet` as a sorted list of sets without\nduplicates.","ref":"sofs.html#to_sets/1"},{"type":"type","title":"sofs.tuple_of/1","doc":"A tuple where the elements are of type `T`.","ref":"sofs.html#t:tuple_of/1"},{"type":"type","title":"sofs.type/0","doc":"A [type](`m:sofs#type`).","ref":"sofs.html#t:type/0"},{"type":"function","title":"sofs.type/1","doc":"Returns the [type](`m:sofs#type`) of an atomic, ordered, or unordered set.","ref":"sofs.html#type/1"},{"type":"function","title":"sofs.union/1","doc":"Returns the [union](`m:sofs#union_n`) of the set of sets `SetOfSets`.","ref":"sofs.html#union/1"},{"type":"function","title":"sofs.union/2","doc":"Returns the [union](`m:sofs#union`) of `Set1` and `Set2`.","ref":"sofs.html#union/2"},{"type":"function","title":"sofs.union_of_family/1","doc":"Returns the union of [family](`m:sofs#family`) `Family`.\n\n```erlang\n1> F = sofs:family([{a,[0,2,4]},{b,[0,1,2]},{c,[2,3]}]),\nS = sofs:union_of_family(F),\nsofs:to_external(S).\n[0,1,2,3,4]\n```","ref":"sofs.html#union_of_family/1"},{"type":"function","title":"sofs.weak_relation/1","doc":"Returns a subset S of the [weak relation](`m:sofs#weak_relation`) W\ncorresponding to the binary relation `BinRel1`. Let F be the\n[field](`m:sofs#field`) of `BinRel1`. The subset S is defined so that x S y if x\nW y for some x in F and for some y in F.\n\n```erlang\n1> R1 = sofs:relation([{1,1},{1,2},{3,1}]),\nR2 = sofs:weak_relation(R1),\nsofs:to_external(R2).\n[{1,1},{1,2},{2,2},{3,1},{3,3}]\n```","ref":"sofs.html#weak_relation/1"},{"type":"module","title":"binary","doc":"Library for handling binary data.\n\nThis module contains functions for manipulating byte-oriented binaries. Although\nthe majority of functions could be provided using bit-syntax, the functions in\nthis library are highly optimized and are expected to either execute faster or\nconsume less memory, or both, than a counterpart written in pure Erlang.\n\nThe module is provided according to Erlang Enhancement Proposal (EEP) 31.\n\n> #### Note {: .info }\n>\n> The library handles byte-oriented data. For bitstrings that are not binaries\n> (does not contain whole octets of bits) a `badarg` exception is thrown from\n> any of the functions in this module.","ref":"binary.html"},{"type":"function","title":"binary.at/2","doc":"Returns the byte at position `Pos` (zero-based) in binary `Subject` as an\ninteger.\n\nIf `Pos` >= [`byte_size(Subject)`](`byte_size/1`), a `badarg` exception\nis raised.","ref":"binary.html#at/2"},{"type":"function","title":"binary.bin_to_list/1","doc":"Converts `Subject` to a list of `t:byte/0`s, each representing the value of one byte.\n\n_Example:_\n\n```erlang\n1> binary:bin_to_list(<<\"erlang\">>).\n\"erlang\"\n%% or [101,114,108,97,110,103] in list notation.\n```","ref":"binary.html#bin_to_list/1"},{"type":"function","title":"binary.bin_to_list/2","doc":"","ref":"binary.html#bin_to_list/2"},{"type":"function","title":"binary.bin_to_list/3","doc":"Converts `Subject` to a list of `t:byte/0`s, each representing the value of one\nbyte. `PosLen` or alternatively `Pos` and `Len` denote which part of the\n`Subject` binary to convert. By default, the entire `Subject` binary is\nconverted.\n\n_Example:_\n\n```erlang\n1> binary:bin_to_list(<<\"erlang\">>, {1,3}).\n\"rla\"\n%% or [114,108,97] in list notation.\n```\n\nIf `PosLen` or alternatively `Pos` and `Len` in any way reference outside the\nbinary, a `badarg` exception is raised.","ref":"binary.html#bin_to_list/3"},{"type":"function","title":"binary.compile_pattern/1","doc":"Builds an internal structure representing a compilation of a search pattern,\nlater to be used in functions `match/3`, `matches/3`, `split/3`, or `replace/4`.\n\nThe `t:cp/0` returned is guaranteed to be a `t:tuple/0` to allow programs to\ndistinguish it from non-precompiled search patterns.\n\nWhen a list of binaries is specified, it denotes a set of alternative binaries\nto search for. For example, if `[<<\"functional\">>,<<\"programming\">>]` is\nspecified as `Pattern`, this means either `<<\"functional\">>` or\n`<<\"programming\">>`\". The pattern is a set of alternatives; when only a single\nbinary is specified, the set has only one element. The order of alternatives in\na pattern is not significant.\n\nThe list of binaries used for search alternatives must be flat, proper and\nnon-empty.\n\nIf `Pattern` is not a binary or a flat proper non-empty list of binaries with\nlength > 0, a `badarg` exception is raised.","ref":"binary.html#compile_pattern/1"},{"type":"function","title":"binary.copy/1","doc":"","ref":"binary.html#copy/1"},{"type":"function","title":"binary.copy/2","doc":"Creates a binary with the content of `Subject` duplicated `N` times.\n\nThis function always creates a new binary, even if `N = 1`. By using `copy/1` on\na binary referencing a larger binary, one can free up the larger binary for\ngarbage collection.\n\n> #### Note {: .info }\n>\n> By deliberately copying a single binary to avoid referencing a larger binary,\n> one can, instead of freeing up the larger binary for later garbage collection,\n> create much more binary data than needed. Sharing binary data is usually good.\n> Only in special cases, when small parts reference large binaries and the large\n> binaries are no longer used in any process, deliberate copying can be a good\n> idea.","ref":"binary.html#copy/2"},{"type":"opaque","title":"binary.cp/0","doc":"Opaque data type representing a compiled search pattern.\n\nGuaranteed to be a `t:tuple/0` to allow programs to distinguish it from\nnon-precompiled search patterns.","ref":"binary.html#t:cp/0"},{"type":"function","title":"binary.decode_hex/1","doc":"Decodes a hex encoded binary into a binary.\n\n_Example_\n\n```erlang\n1> binary:decode_hex(<<\"66\">>).\n<<\"f\">>\n```","ref":"binary.html#decode_hex/1"},{"type":"function","title":"binary.decode_unsigned/1","doc":"","ref":"binary.html#decode_unsigned/1"},{"type":"function","title":"binary.decode_unsigned/2","doc":"Converts the binary digit representation, in big endian or little endian, of a\npositive integer in `Subject` to an Erlang `t:integer/0`.\n\n_Example:_\n\n```erlang\n1> binary:decode_unsigned(<<169,138,199>>).\n11111111\n2> binary:decode_unsigned(<<169,138,199>>, big).\n11111111\n3> binary:decode_unsigned(<<169,138,199>>, little).\n13077161\n```","ref":"binary.html#decode_unsigned/2"},{"type":"function","title":"binary.encode_hex/1","doc":"","ref":"binary.html#encode_hex/1"},{"type":"function","title":"binary.encode_hex/2","doc":"Encodes a binary into a hex encoded binary using the specified case for the\nhexadecimal digits \"a\" to \"f\".\n\nThe default case is `uppercase`.\n\n_Example:_\n\n```erlang\n1> binary:encode_hex(<<\"f\">>).\n<<\"66\">>\n2> binary:encode_hex(<<\"/\">>).\n<<\"2F\">>\n3> binary:encode_hex(<<\"/\">>, lowercase).\n<<\"2f\">>\n4> binary:encode_hex(<<\"/\">>, uppercase).\n<<\"2F\">>\n```","ref":"binary.html#encode_hex/2"},{"type":"function","title":"binary.encode_unsigned/1","doc":"","ref":"binary.html#encode_unsigned/1"},{"type":"function","title":"binary.encode_unsigned/2","doc":"Converts a positive integer to the smallest possible representation in a binary\ndigit representation, either big endian or little endian.\n\n_Example:_\n\n```erlang\n1> binary:encode_unsigned(11111111).\n<<169,138,199>>\n2> binary:encode_unsigned(11111111, big).\n<<169,138,199>>\n2> binary:encode_unsigned(11111111, little).\n<<199,138,169>>\n```","ref":"binary.html#encode_unsigned/2"},{"type":"function","title":"binary.first/1","doc":"Returns the first byte of binary `Subject` as an integer. If the size of\n`Subject` is zero, a `badarg` exception is raised.","ref":"binary.html#first/1"},{"type":"function","title":"binary.last/1","doc":"Returns the last byte of binary `Subject` as an integer. If the size of\n`Subject` is zero, a `badarg` exception is raised.","ref":"binary.html#last/1"},{"type":"function","title":"binary.list_to_bin/1","doc":"Works exactly as `erlang:list_to_binary/1`, added for completeness.","ref":"binary.html#list_to_bin/1"},{"type":"function","title":"binary.longest_common_prefix/1","doc":"Returns the length of the longest common prefix of the binaries in list\n`Binaries`.\n\n_Example:_\n\n```erlang\n1> binary:longest_common_prefix([<<\"erlang\">>, <<\"ergonomy\">>]).\n2\n2> binary:longest_common_prefix([<<\"erlang\">>, <<\"perl\">>]).\n0\n```\n\nIf `Binaries` is not a flat non-empty list of binaries, a `badarg` exception is\nraised.","ref":"binary.html#longest_common_prefix/1"},{"type":"function","title":"binary.longest_common_suffix/1","doc":"Returns the length of the longest common suffix of the binaries in list\n`Binaries`.\n\n_Example:_\n\n```erlang\n1> binary:longest_common_suffix([<<\"erlang\">>, <<\"fang\">>]).\n3\n2> binary:longest_common_suffix([<<\"erlang\">>, <<\"perl\">>]).\n0\n```\n\nIf `Binaries` is not a flat non-empty list of binaries, a `badarg` exception is\nraised.","ref":"binary.html#longest_common_suffix/1"},{"type":"function","title":"binary.match/2","doc":"","ref":"binary.html#match/2"},{"type":"function","title":"binary.match/3","doc":"Searches for the first occurrence of `Pattern` in `Subject` and returns the\nposition and length.\n\nThe function returns `{Pos, Length}` for the binary in `Pattern`, starting at\nthe lowest position in `Subject`.\n\n_Example:_\n\n```erlang\n1> binary:match(<<\"abcde\">>, [<<\"bcde\">>, <<\"cd\">>],[]).\n{1,4}\n```\n\nEven though `<<\"cd\">>` ends before `<<\"bcde\">>`, `<<\"bcde\">>` begins first and\nis therefore the first match. If two overlapping matches begin at the same\nposition, the longest is returned.\n\nSummary of the options:\n\n- **\\{scope, \\{Start, Length\\}\\}** - Only the specified part is searched. Return\n  values still have offsets from the beginning of `Subject`. A negative `Length`\n  is allowed as described in section Data Types in this manual.\n\nIf none of the strings in `Pattern` is found, the atom `nomatch` is returned.\n\nFor a description of `Pattern`, see function `compile_pattern/1`.\n\nIf `{scope, {Start,Length}}` is specified in the options such that `Start` >\nsize of `Subject`, `Start` \\+ `Length` < 0 or `Start` \\+ `Length` > size of\n`Subject`, a `badarg` exception is raised.","ref":"binary.html#match/3"},{"type":"function","title":"binary.matches/2","doc":"","ref":"binary.html#matches/2"},{"type":"function","title":"binary.matches/3","doc":"As `match/2`, but `Subject` is searched until exhausted and a list of all\nnon-overlapping parts matching `Pattern` is returned (in order).\n\nThe first and longest match is preferred to a shorter, which is illustrated by\nthe following example:\n\n```erlang\n1> binary:matches(<<\"abcde\">>,\n                  [<<\"bcde\">>,<<\"bc\">>,<<\"de\">>],[]).\n[{1,4}]\n```\n\nThe result shows that <<\"bcde\">> is selected instead of the shorter match\n<<\"bc\">> (which would have given raise to one more match, <<\"de\">>). This\ncorresponds to the behavior of POSIX regular expressions (and programs like\nawk), but is not consistent with alternative matches in `re` (and Perl), where\ninstead lexical ordering in the search pattern selects which string matches.\n\nIf none of the strings in a pattern is found, an empty list is returned.\n\nFor a description of `Pattern`, see `compile_pattern/1`. For a description of\navailable options, see `match/3`.\n\nIf `{scope, {Start,Length}}` is specified in the options such that `Start` >\nsize of `Subject`, `Start + Length` < 0 or `Start + Length` is > size of\n`Subject`, a `badarg` exception is raised.","ref":"binary.html#matches/3"},{"type":"type","title":"binary.part/0","doc":"A representation of a part (or range) in a binary. `Start` is a zero-based\noffset into a `t:binary/0` and `Length` is the length of that part.\n\nAs input to functions in this module, a reverse part specification is allowed, constructed\nwith a negative `Length`, so that the part of the binary begins at `Start` \\+\n`Length` and is -`Length` long. This is useful for referencing the last `N`\nbytes of a binary as `{size(Binary), -N}`. The functions in this module always\nreturn `t:part/0`s with positive `Length`.","ref":"binary.html#t:part/0"},{"type":"function","title":"binary.part/2","doc":"","ref":"binary.html#part/2"},{"type":"function","title":"binary.part/3","doc":"Extracts the part of binary `Subject` described by `PosLen`.\n\nA negative length can be used to extract bytes at the end of a binary:\n\n```erlang\n1> Bin = <<1,2,3,4,5,6,7,8,9,10>>.\n2> binary:part(Bin, {byte_size(Bin), -5}).\n<<6,7,8,9,10>>\n```\n\n> #### Note {: .info }\n>\n> `part/2` and `part/3` are also available in the `m:erlang` module under the\n> names [`binary_part/2`](`binary_part/2`) and\n> [`binary_part/3`](`binary_part/3`). Those BIFs are allowed in guard tests.\n\nIf `PosLen` in any way references outside the binary, a `badarg` exception is\nraised.","ref":"binary.html#part/3"},{"type":"function","title":"binary.referenced_byte_size/1","doc":"Get the size of the underlying binary referenced by `Binary`.\n\nIf a binary references a larger binary (often described as being a subbinary),\nit can be useful to get the size of the referenced binary. This function can be\nused in a program to trigger the use of `copy/1`. By copying\n a binary, one can dereference the original, possibly large, binary that a\nsmaller binary is a reference to.\n\n_Example:_\n\n```erlang\nstore(Binary, GBSet) ->\n  NewBin =\n      case binary:referenced_byte_size(Binary) of\n          Large when Large > 2 * byte_size(Binary) ->\n             binary:copy(Binary);\n          _ ->\n             Binary\n      end,\n  gb_sets:insert(NewBin,GBSet).\n```\n\nIn this example, we chose to copy the binary content before inserting it in\n[`gb_sets:set()`](`t:gb_sets:set/0`) if it references a binary more than twice\nthe data size we want to keep. Of course, different rules apply when copying to\ndifferent programs.\n\nBinary sharing occurs whenever binaries are taken apart. This is the fundamental\nreason why binaries are fast, decomposition can always be done with O(1)\ncomplexity. In rare circumstances this data sharing is however undesirable, why\nthis function together with [`copy/1`](`copy/1`) can be useful when optimizing\nfor memory use.\n\nExample of binary sharing:\n\n```erlang\n1> A = binary:copy(<<1>>, 100).\n<<1,1,1,1,1 ...\n2> byte_size(A).\n100\n3> binary:referenced_byte_size(A).\n100\n4> < > = A.\n<<1,1,1,1,1 ...\n5> {byte_size(B), binary:referenced_byte_size(B)}.\n{10,10}\n6> {byte_size(C), binary:referenced_byte_size(C)}.\n{90,100}\n```\n\nIn the above example, the small binary `B` was copied while the larger binary\n`C` references binary `A`.\n\n> #### Note {: .info }\n>\n> Binary data is shared among processes. If another process still references the\n> larger binary, copying the part this process uses only consumes more memory\n> and does not free up the larger binary for garbage collection. Use this kind\n> of intrusive functions with extreme care and only if a real problem is\n> detected.","ref":"binary.html#referenced_byte_size/1"},{"type":"function","title":"binary.replace/3","doc":"","ref":"binary.html#replace/3"},{"type":"function","title":"binary.replace/4","doc":"Constructs a new binary by replacing the parts in `Subject` matching `Pattern`\nwith `Replacement` if given as a literal `t:binary/0` or with the result of\napplying `Replacement` to a matching subpart if given as a `fun`.\n\nIf `Replacement` is given as a `t:binary/0` and the matching subpart of\n`Subject` giving raise to the replacement is to be inserted in the result,\noption `{insert_replaced, InsPos}` inserts the matching part into `Replacement`\nat the specified position (or positions) before inserting `Replacement` into\n`Subject`. If `Replacement` is given as a `fun` instead, this option is ignored.\n\nIf any position specified in `InsPos` > size of the replacement binary, a\n`badarg` exception is raised.\n\nOptions `global` and `{scope, part()}` work as for `split/3`. The return type is\nalways a `t:binary/0`.\n\nFor a description of `Pattern`, see `compile_pattern/1`.\n\n_Examples:_\n\n```erlang\n1> binary:replace(<<\"abcde\">>, [<<\"b\">>, <<\"d\">>], <<\"X\">>, []).\n<<\"aXcde\">>\n\n2> binary:replace(<<\"abcde\">>, [<<\"b\">>, <<\"d\">>], <<\"X\">>, [global]).\n<<\"aXcXe\">>\n\n3> binary:replace(<<\"abcde\">>, <<\"b\">>, <<\"[]\">>, [{insert_replaced, 1}]).\n<<\"a[b]cde\">>\n\n4> binary:replace(<<\"abcde\">>, [<<\"b\">>, <<\"d\">>], <<\"[]\">>, [global, {insert_replaced, 1}]).\n<<\"a[b]c[d]e\">>\n\n5> binary:replace(<<\"abcde\">>, [<<\"b\">>, <<\"d\">>], <<\"[]\">>, [global, {insert_replaced, [1, 1]}]).\n<<\"a[bb]c[dd]e\">>\n\n6> binary:replace(<<\"abcde\">>, [<<\"b\">>, <<\"d\">>], <<\"[-]\">>, [global, {insert_replaced, [1, 2]}]).\n<<\"a[b-b]c[d-d]e\">>\n\n7> binary:replace(<<\"abcde\">>, [<<\"b\">>, <<\"d\">>], fun(M) -> <<$[, M/binary, $]>> end, []).\n<<\"a[b]cde\">>\n\n8> binary:replace(<<\"abcde\">>, [<<\"b\">>, <<\"d\">>], fun(M) -> <<$[, M/binary, $]>> end, [global]).\n<<\"a[b]c[d]e\">>\n```","ref":"binary.html#replace/4"},{"type":"function","title":"binary.split/2","doc":"","ref":"binary.html#split/2"},{"type":"function","title":"binary.split/3","doc":"Splits `Subject` into a list of binaries based on `Pattern`.\n\nIf option `global` is not specified, only the first occurrence of `Pattern` in\n`Subject` gives rise to a split.\n\nThe parts of `Pattern` found in `Subject` are not included in the result.\n\n_Example:_\n\n```erlang\n1> binary:split(<<1,255,4,0,0,0,2,3>>, [<<0,0,0>>,<<2>>],[]).\n[<<1,255,4>>, <<2,3>>]\n2> binary:split(<<0,1,0,0,4,255,255,9>>, [<<0,0>>, <<255,255>>],[global]).\n[<<0,1>>,<<4>>,<<9>>]\n```\n\nSummary of options:\n\n- **\\{scope, part()\\}** - Works as in `match/3` and `matches/3`. Notice that\n  this only defines the scope of the search for matching strings, it does not\n  cut the binary before splitting. The bytes before and after the scope are kept\n  in the result. See the example below.\n\n- **trim** - Removes trailing empty parts of the result (as does `trim` in\n  `re:split/3`.\n\n- **trim_all** - Removes all empty parts of the result.\n\n- **global** - Repeats the split until `Subject` is exhausted. Conceptually\n  option `global` makes split work on the positions returned by `matches/3`,\n  while it normally works on the position returned by `match/3`.\n\nExample of the difference between a scope and taking the binary apart before\nsplitting:\n\n```erlang\n1> binary:split(<<\"banana\">>, [<<\"a\">>],[{scope,{2,3}}]).\n[<<\"ban\">>,<<\"na\">>]\n2> binary:split(binary:part(<<\"banana\">>,{2,3}), [<<\"a\">>],[]).\n[<<\"n\">>,<<\"n\">>]\n```\n\nThe return type is always a list of binaries that are all referencing `Subject`.\nThis means that the data in `Subject` is not copied to new binaries, and that\n`Subject` cannot be garbage collected until the results of the split are no\nlonger referenced.\n\nFor a description of `Pattern`, see `compile_pattern/1`.","ref":"binary.html#split/3"},{"type":"module","title":"lists","doc":"List processing functions.\n\nThis module contains functions for list processing.\n\nUnless otherwise stated, all functions assume that position numbering starts\nat 1. That is, the first element of a list is at position 1.\n\nTwo terms `T1` and `T2` compare equal if `T1 == T2` evaluates to `true`. They\nmatch if `T1 =:= T2` evaluates to `true`.\n\nWhenever an _ordering function_{: #ordering_function } `F` is expected as\nargument, it is assumed that the following properties hold of `F` for all x, y,\nand z:\n\n- If x `F` y and y `F` x, then x = y (`F` is antisymmetric).\n- If x `F` y and y `F` z, then x `F` z (`F` is transitive).\n- x `F` y or y `F` x (`F` is total).\n\nAn example of a typical ordering function is less than or equal to: `=</2`.","ref":"lists.html"},{"type":"function","title":"lists.all/2","doc":"Returns `true` if `Pred(Elem)` returns `true` for all elements `Elem` in `List`,\notherwise `false`. The `Pred` function must return a boolean.","ref":"lists.html#all/2"},{"type":"function","title":"lists.any/2","doc":"Returns `true` if `Pred(Elem)` returns `true` for at least one element `Elem` in\n`List`. The `Pred` function must return a boolean.","ref":"lists.html#any/2"},{"type":"function","title":"lists.append/1","doc":"Returns a list in which all the sublists of `ListOfLists` have been appended.\n\n_Example:_\n\n```erlang\n> lists:append([[1, 2, 3], [a, b], [4, 5, 6]]).\n[1,2,3,a,b,4,5,6]\n```","ref":"lists.html#append/1"},{"type":"function","title":"lists.append/2","doc":"Returns a new list `List3`, which is made from the elements of `List1` followed\nby the elements of `List2`.\n\n_Example:_\n\n```erlang\n> lists:append(\"abc\", \"def\").\n\"abcdef\"\n```\n\n`lists:append(A, B)` is equivalent to `A ++ B`.","ref":"lists.html#append/2"},{"type":"function","title":"lists.concat/1","doc":"Concatenates the text representation of the elements of `Things`. The elements\nof `Things` can be atoms, integers, floats, or strings.\n\n_Example:_\n\n```erlang\n> lists:concat([doc, '/', file, '.', 3]).\n\"doc/file.3\"\n```","ref":"lists.html#concat/1"},{"type":"function","title":"lists.delete/2","doc":"Returns a copy of `List1` where the first element matching `Elem` is deleted, if\nthere is such an element.","ref":"lists.html#delete/2"},{"type":"function","title":"lists.droplast/1","doc":"Drops the last element of a `List`. The list is to be non-empty, otherwise the\nfunction crashes with a `function_clause`.","ref":"lists.html#droplast/1"},{"type":"function","title":"lists.dropwhile/2","doc":"Drops elements `Elem` from `List1` while `Pred(Elem)` returns `true` and returns\nthe remaining list. The `Pred` function must return a boolean.","ref":"lists.html#dropwhile/2"},{"type":"function","title":"lists.duplicate/2","doc":"Returns a list containing `N` copies of term `Elem`.\n\n_Example:_\n\n```erlang\n> lists:duplicate(5, xx).\n[xx,xx,xx,xx,xx]\n```","ref":"lists.html#duplicate/2"},{"type":"function","title":"lists.enumerate/1","doc":"","ref":"lists.html#enumerate/1"},{"type":"function","title":"lists.enumerate/2","doc":"","ref":"lists.html#enumerate/2"},{"type":"function","title":"lists.enumerate/3","doc":"Returns `List1` with each element `H` replaced by a tuple of form `{I, H}` where\n`I` is the position of `H` in `List1`. The enumeration starts with `Index` and\nincreases by `Step` in each step.\n\nThat is, [`enumerate/3`](`enumerate/3`) behaves as if it had been defined as\nfollows:\n\n```erlang\nenumerate(I, S, List) ->\n  {List1, _ } = lists:mapfoldl(fun(T, Acc) -> {{Acc, T}, Acc+S} end, I, List),\n  List1.\n```\n\nThe default values for `Index` and `Step` are both `1`.\n\n_Examples:_\n\n```erlang\n> lists:enumerate([a,b,c]).\n[{1,a},{2,b},{3,c}]\n```\n\n```erlang\n> lists:enumerate(10, [a,b,c]).\n[{10,a},{11,b},{12,c}]\n```\n\n```erlang\n> lists:enumerate(0, -2, [a,b,c]).\n[{0,a},{-2,b},{-4,c}]\n```","ref":"lists.html#enumerate/3"},{"type":"function","title":"lists.filter/2","doc":"`List2` is a list of all elements `Elem` in `List1` for which `Pred(Elem)`\nreturns `true`. The `Pred` function must return a boolean.","ref":"lists.html#filter/2"},{"type":"function","title":"lists.filtermap/2","doc":"Calls `Fun(Elem)` on successive elements `Elem` of `List1` in order to update or\nremove elements from `List1`.\n\n`Fun/1` must return either a Boolean or a tuple `{true, Value}`. The function\nreturns the list of elements for which `Fun` returns a new value, where a value\nof `true` is synonymous with `{true, Elem}`.\n\nThat is, `filtermap` behaves as if it had been defined as follows:\n\n```erlang\nfiltermap(Fun, List1) ->\n    lists:foldr(fun(Elem, Acc) ->\n                       case Fun(Elem) of\n                           false -> Acc;\n                           true -> [Elem|Acc];\n                           {true,Value} -> [Value|Acc]\n                       end\n                end, [], List1).\n```\n\n_Example:_\n\n```erlang\n> lists:filtermap(fun(X) -> case X rem 2 of 0 -> {true, X div 2}; _ -> false end end, [1,2,3,4,5]).\n[1,2]\n```","ref":"lists.html#filtermap/2"},{"type":"function","title":"lists.flatlength/1","doc":"Equivalent to [`length(flatten(DeepList))`](`length/1`), but more efficient.","ref":"lists.html#flatlength/1"},{"type":"function","title":"lists.flatmap/2","doc":"Takes a function from `A`s to lists of `B`s, and a list of `A`s (`List1`) and\nproduces a list of `B`s by applying the function to every element in `List1` and\nappending the resulting lists.\n\nThat is, `flatmap` behaves as if it had been defined as follows:\n\n```erlang\nflatmap(Fun, List1) ->\n    append(map(Fun, List1)).\n```\n\n_Example:_\n\n```erlang\n> lists:flatmap(fun(X)->[X,X] end, [a,b,c]).\n[a,a,b,b,c,c]\n```","ref":"lists.html#flatmap/2"},{"type":"function","title":"lists.flatten/1","doc":"Returns a flattened version of `DeepList`.","ref":"lists.html#flatten/1"},{"type":"function","title":"lists.flatten/2","doc":"Returns a flattened version of `DeepList` with tail `Tail` appended.","ref":"lists.html#flatten/2"},{"type":"function","title":"lists.foldl/3","doc":"Calls `Fun(Elem, AccIn)` on successive elements `A` of `List`, starting with\n`AccIn == Acc0`. `Fun/2` must return a new accumulator, which is passed to the\nnext call. The function returns the final value of the accumulator. `Acc0` is\nreturned if the list is empty.\n\n_Example:_\n\n```erlang\n> lists:foldl(fun(X, Sum) -> X + Sum end, 0, [1,2,3,4,5]).\n15\n> lists:foldl(fun(X, Prod) -> X * Prod end, 1, [1,2,3,4,5]).\n120\n```","ref":"lists.html#foldl/3"},{"type":"function","title":"lists.foldr/3","doc":"Like `foldl/3`, but the list is traversed from right to left.\n\n_Example:_\n\n```erlang\n> P = fun(A, AccIn) -> io:format(\"~p \", [A]), AccIn end.\n#Fun \n> lists:foldl(P, void, [1,2,3]).\n1 2 3 void\n> lists:foldr(P, void, [1,2,3]).\n3 2 1 void\n```\n\n[`foldl/3`](`foldl/3`) is tail recursive and is usually preferred to\n[`foldr/3`](`foldr/3`).","ref":"lists.html#foldr/3"},{"type":"function","title":"lists.foreach/2","doc":"Calls `Fun(Elem)` for each element `Elem` in `List`. This function is used for\nits side effects and the evaluation order is defined to be the same as the order\nof the elements in the list.","ref":"lists.html#foreach/2"},{"type":"function","title":"lists.join/2","doc":"Inserts `Sep` between each element in `List1`. Has no effect on the empty list\nand on a singleton list. For example:\n\n```erlang\n> lists:join(x, [a,b,c]).\n[a,x,b,x,c]\n> lists:join(x, [a]).\n[a]\n> lists:join(x, []).\n[]\n```","ref":"lists.html#join/2"},{"type":"function","title":"lists.keydelete/3","doc":"Returns a copy of `TupleList1` where the first occurrence of a tuple whose `N`th\nelement compares equal to `Key` is deleted, if there is such a tuple.","ref":"lists.html#keydelete/3"},{"type":"function","title":"lists.keyfind/3","doc":"Searches the list of tuples `TupleList` for a tuple whose `N`th element compares\nequal to `Key`. Returns `Tuple` if such a tuple is found, otherwise `false`.","ref":"lists.html#keyfind/3"},{"type":"function","title":"lists.keymap/3","doc":"Returns a list of tuples where, for each tuple in `TupleList1`, the `N`th\nelement `Term1` of the tuple has been replaced with the result of calling\n`Fun(Term1)`.\n\n_Examples:_\n\n```erlang\n> Fun = fun(Atom) -> atom_to_list(Atom) end.\n#Fun \n2> lists:keymap(Fun, 2, [{name,jane,22},{name,lizzie,20},{name,lydia,15}]).\n[{name,\"jane\",22},{name,\"lizzie\",20},{name,\"lydia\",15}]\n```","ref":"lists.html#keymap/3"},{"type":"function","title":"lists.keymember/3","doc":"Returns `true` if there is a tuple in `TupleList` whose `N`th element compares\nequal to `Key`, otherwise `false`.","ref":"lists.html#keymember/3"},{"type":"function","title":"lists.keymerge/3","doc":"Returns the sorted list formed by merging `TupleList1` and `TupleList2`.\n\nThe merge is performed on the `N`th element of each tuple. Both `TupleList1` and\n`TupleList2` must be key-sorted before evaluating this function. When two tuples\ncompare equal, the tuple from `TupleList1` is picked before the tuple from\n`TupleList2`.","ref":"lists.html#keymerge/3"},{"type":"function","title":"lists.keyreplace/4","doc":"Returns a copy of `TupleList1` where the first occurrence of a `T` tuple whose\n`N`th element compares equal to `Key` is replaced with `NewTuple`, if there is\nsuch a tuple `T`.","ref":"lists.html#keyreplace/4"},{"type":"function","title":"lists.keysearch/3","doc":"Searches the list of tuples `TupleList` for a tuple whose `N`th element compares\nequal to `Key`. Returns `{value, Tuple}` if such a tuple is found, otherwise\n`false`.\n\n> #### Note {: .info }\n>\n> This function is retained for backward compatibility. Function `keyfind/3` is\n> usually more convenient.","ref":"lists.html#keysearch/3"},{"type":"function","title":"lists.keysort/2","doc":"Returns a list containing the sorted elements of list `TupleList1`. Sorting is\nperformed on the `N`th element of the tuples. The sort is stable.","ref":"lists.html#keysort/2"},{"type":"function","title":"lists.keystore/4","doc":"Returns a copy of `TupleList1` where the first occurrence of a tuple `T` whose\n`N`th element compares equal to `Key` is replaced with `NewTuple`, if there is\nsuch a tuple `T`. If there is no such tuple `T`, a copy of `TupleList1` where\n[`NewTuple`] has been appended to the end is returned.","ref":"lists.html#keystore/4"},{"type":"function","title":"lists.keytake/3","doc":"Searches the list of tuples `TupleList1` for a tuple whose `N`th element\ncompares equal to `Key`. Returns `{value, Tuple, TupleList2}` if such a tuple is\nfound, otherwise `false`. `TupleList2` is a copy of `TupleList1` where the first\noccurrence of `Tuple` has been removed.","ref":"lists.html#keytake/3"},{"type":"function","title":"lists.last/1","doc":"Returns the last element in `List`.","ref":"lists.html#last/1"},{"type":"function","title":"lists.map/2","doc":"Takes a function from `A`s to `B`s, and a list of `A`s and produces a list of\n`B`s by applying the function to every element in the list. This function is\nused to obtain the return values. The evaluation order depends on the\nimplementation.","ref":"lists.html#map/2"},{"type":"function","title":"lists.mapfoldl/3","doc":"Combines the operations of `map/2` and `foldl/3` into one pass.\n\n_Example:_\n\nSumming the elements in a list and double them at the same time:\n\n```erlang\n> lists:mapfoldl(fun(X, Sum) -> {2*X, X+Sum} end,\n0, [1,2,3,4,5]).\n{[2,4,6,8,10],15}\n```","ref":"lists.html#mapfoldl/3"},{"type":"function","title":"lists.mapfoldr/3","doc":"Combines the operations of `map/2` and `foldr/3` into one pass.","ref":"lists.html#mapfoldr/3"},{"type":"function","title":"lists.max/1","doc":"Returns the first element of `List` that compares greater than or equal to all\nother elements of `List`.","ref":"lists.html#max/1"},{"type":"function","title":"lists.member/2","doc":"Returns `true` if `Elem` matches some element of `List`, otherwise `false`.","ref":"lists.html#member/2"},{"type":"function","title":"lists.merge3/3","doc":"Returns the sorted list formed by merging `List1`, `List2`, and `List3`. All of\n`List1`, `List2`, and `List3` must be sorted before evaluating this function.\n\nWhen two elements compare equal, the element from `List1`, if there is such an\nelement, is picked before the other element, otherwise the element from `List2`\nis picked before the element from `List3`.","ref":"lists.html#merge3/3"},{"type":"function","title":"lists.merge/1","doc":"Returns the sorted list formed by merging all the sublists of `ListOfLists`. All\nsublists must be sorted before evaluating this function.\n\nWhen two elements compare equal, the element from the sublist with the lowest\nposition in `ListOfLists` is picked before the other element.","ref":"lists.html#merge/1"},{"type":"function","title":"lists.merge/2","doc":"Returns the sorted list formed by merging `List1` and `List2`. Both `List1` and\n`List2` must be sorted before evaluating this function.\n\nWhen two elements compare equal, the element from `List1` is picked before the\nelement from `List2`.","ref":"lists.html#merge/2"},{"type":"function","title":"lists.merge/3","doc":"Returns the sorted list formed by merging `List1` and `List2`. Both `List1` and\n`List2` must be sorted according to the\n[ordering function](`m:lists#ordering_function`) `Fun` before evaluating this\nfunction.\n\n`Fun(A, B)` is to return `true` if `A` compares less than or equal to\n`B` in the ordering, otherwise `false`. When two elements compare equal, the\nelement from `List1` is picked before the element from `List2`.","ref":"lists.html#merge/3"},{"type":"function","title":"lists.min/1","doc":"Returns the first element of `List` that compares less than or equal to all\nother elements of `List`.","ref":"lists.html#min/1"},{"type":"function","title":"lists.nth/2","doc":"Returns the `N`th element of `List`.\n\n_Example:_\n\n```erlang\n> lists:nth(3, [a, b, c, d, e]).\nc\n```","ref":"lists.html#nth/2"},{"type":"function","title":"lists.nthtail/2","doc":"Returns the `N`th tail of `List`, that is, the sublist of `List` starting at\n`N+1` and continuing up to the end of the list.\n\n_Example_\n\n```erlang\n> lists:nthtail(3, [a, b, c, d, e]).\n[d,e]\n> tl(tl(tl([a, b, c, d, e]))).\n[d,e]\n> lists:nthtail(0, [a, b, c, d, e]).\n[a,b,c,d,e]\n> lists:nthtail(5, [a, b, c, d, e]).\n[]\n```","ref":"lists.html#nthtail/2"},{"type":"function","title":"lists.partition/2","doc":"Partitions `List` into two lists, where the first list contains all elements for\nwhich `Pred(Elem)` returns `true`, and the second list contains all elements for\nwhich `Pred(Elem)` returns `false`.\n\n_Examples:_\n\n```erlang\n> lists:partition(fun(A) -> A rem 2 == 1 end, [1,2,3,4,5,6,7]).\n{[1,3,5,7],[2,4,6]}\n> lists:partition(fun(A) -> is_atom(A) end, [a,b,1,c,d,2,3,4,e]).\n{[a,b,c,d,e],[1,2,3,4]}\n```\n\nFor a different way to partition a list, see `splitwith/2`.","ref":"lists.html#partition/2"},{"type":"function","title":"lists.prefix/2","doc":"Returns `true` if `List1` is a prefix of `List2`, otherwise `false`.","ref":"lists.html#prefix/2"},{"type":"function","title":"lists.reverse/1","doc":"Returns a list with the elements in `List1` in reverse order.","ref":"lists.html#reverse/1"},{"type":"function","title":"lists.reverse/2","doc":"Returns a list with the elements in `List1` in reverse order, with tail `Tail`\nappended.\n\n_Example:_\n\n```erlang\n> lists:reverse([1, 2, 3, 4], [a, b, c]).\n[4,3,2,1,a,b,c]\n```","ref":"lists.html#reverse/2"},{"type":"function","title":"lists.search/2","doc":"If there is a `Value` in `List` such that `Pred(Value)` returns `true`, returns\n`{value, Value}` for the first such `Value`, otherwise returns `false`. The\n`Pred` function must return a boolean.","ref":"lists.html#search/2"},{"type":"function","title":"lists.seq/2","doc":"","ref":"lists.html#seq/2"},{"type":"function","title":"lists.seq/3","doc":"Returns a sequence of integers that starts with `From` and contains the\nsuccessive results of adding `Incr` to the previous element, until `To` is\nreached or passed (in the latter case, `To` is not an element of the sequence).\n`Incr` defaults to 1.\n\nFailures:\n\n- If `To   0`.\n- If `To > From - Incr` and `Incr < 0`.\n- If `Incr =:= 0` and `From =/= To`.\n\nThe following equalities hold for all sequences:\n\n```erlang\nlength(lists:seq(From, To)) =:= To - From + 1\nlength(lists:seq(From, To, Incr)) =:= (To - From + Incr) div Incr\n```\n\n_Examples:_\n\n```erlang\n> lists:seq(1, 10).\n[1,2,3,4,5,6,7,8,9,10]\n> lists:seq(1, 20, 3).\n[1,4,7,10,13,16,19]\n> lists:seq(1, 0, 1).\n[]\n> lists:seq(10, 6, 4).\n[]\n> lists:seq(1, 1, 0).\n[1]\n```","ref":"lists.html#seq/3"},{"type":"function","title":"lists.sort/1","doc":"Returns a list containing the sorted elements of `List1`.","ref":"lists.html#sort/1"},{"type":"function","title":"lists.sort/2","doc":"Returns a list containing the sorted elements of `List1`, according to the\n[ordering function](`m:lists#ordering_function`) `Fun`. `Fun(A, B)` is to return\n`true` if `A` compares less than or equal to `B` in the ordering, otherwise\n`false`.","ref":"lists.html#sort/2"},{"type":"function","title":"lists.split/2","doc":"Splits `List1` into `List2` and `List3`. `List2` contains the first `N` elements\nand `List3` the remaining elements (the `N`th tail).","ref":"lists.html#split/2"},{"type":"function","title":"lists.splitwith/2","doc":"Partitions `List` into two lists according to `Pred`.\n[`splitwith/2`](`splitwith/2`) behaves as if it is defined as follows:\n\n```erlang\nsplitwith(Pred, List) ->\n    {takewhile(Pred, List), dropwhile(Pred, List)}.\n```\n\n_Examples:_\n\n```erlang\n> lists:splitwith(fun(A) -> A rem 2 == 1 end, [1,2,3,4,5,6,7]).\n{[1],[2,3,4,5,6,7]}\n> lists:splitwith(fun(A) -> is_atom(A) end, [a,b,1,c,d,2,3,4,e]).\n{[a,b],[1,c,d,2,3,4,e]}\n```\n\nThe `Pred` function must return a boolean. For a different way to partition a\nlist, see `partition/2`.","ref":"lists.html#splitwith/2"},{"type":"function","title":"lists.sublist/2","doc":"Returns the sublist of `List1` starting at position 1 and with (maximum) `Len`\nelements. It is not an error for `Len` to exceed the length of the list, in that\ncase the whole list is returned.","ref":"lists.html#sublist/2"},{"type":"function","title":"lists.sublist/3","doc":"Returns the sublist of `List1` starting at `Start` and with (maximum) `Len`\nelements. It is not an error for `Start+Len` to exceed the length of the list.\n\n_Examples:_\n\n```erlang\n> lists:sublist([1,2,3,4], 2, 2).\n[2,3]\n> lists:sublist([1,2,3,4], 2, 5).\n[2,3,4]\n> lists:sublist([1,2,3,4], 5, 2).\n[]\n```","ref":"lists.html#sublist/3"},{"type":"function","title":"lists.subtract/2","doc":"Returns a new list `List3` that is a copy of `List1`, subjected to the following\nprocedure: for each element in `List2`, its first occurrence in `List1` is\ndeleted.\n\n_Example:_\n\n```erlang\n> lists:subtract(\"123212\", \"212\").\n\"312\".\n```\n\n`lists:subtract(A, B)` is equivalent to `A -- B`.","ref":"lists.html#subtract/2"},{"type":"function","title":"lists.suffix/2","doc":"Returns `true` if `List1` is a suffix of `List2`, otherwise `false`.","ref":"lists.html#suffix/2"},{"type":"function","title":"lists.sum/1","doc":"Returns the sum of the elements in `List`.","ref":"lists.html#sum/1"},{"type":"function","title":"lists.takewhile/2","doc":"Takes elements `Elem` from `List1` while `Pred(Elem)` returns `true`, that is,\nthe function returns the longest prefix of the list for which all elements\nsatisfy the predicate. The `Pred` function must return a boolean.","ref":"lists.html#takewhile/2"},{"type":"function","title":"lists.ukeymerge/3","doc":"Returns the sorted list formed by merging `TupleList1` and `TupleList2`. The\nmerge is performed on the `N`th element of each tuple. Both `TupleList1` and\n`TupleList2` must be key-sorted without duplicates before evaluating this\nfunction.\n\nWhen two tuples compare equal, the tuple from `TupleList1` is picked\nand the one from `TupleList2` is deleted.","ref":"lists.html#ukeymerge/3"},{"type":"function","title":"lists.ukeysort/2","doc":"Returns a list containing the sorted elements of list `TupleList1` where all\nexcept the first tuple of the tuples comparing equal have been deleted. Sorting\nis performed on the `N`th element of the tuples.","ref":"lists.html#ukeysort/2"},{"type":"function","title":"lists.umerge3/3","doc":"Returns the sorted list formed by merging `List1`, `List2`, and `List3`. All of\n`List1`, `List2`, and `List3` must be sorted and contain no duplicates before\nevaluating this function.\n\nWhen two elements compare equal, the element from\n`List1` is picked if there is such an element, otherwise the element from\n`List2` is picked, and the other is deleted.","ref":"lists.html#umerge3/3"},{"type":"function","title":"lists.umerge/1","doc":"Returns the sorted list formed by merging all the sublists of `ListOfLists`. All\nsublists must be sorted and contain no duplicates before evaluating this\nfunction.\n\nWhen two elements compare equal, the element from the sublist with the\nlowest position in `ListOfLists` is picked and the other is deleted.","ref":"lists.html#umerge/1"},{"type":"function","title":"lists.umerge/2","doc":"Returns the sorted list formed by merging `List1` and `List2`. Both `List1` and\n`List2` must be sorted and contain no duplicates before evaluating this\nfunction.\n\nWhen two elements compare equal, the element from `List1` is picked\nand the one from `List2` is deleted.","ref":"lists.html#umerge/2"},{"type":"function","title":"lists.umerge/3","doc":"Returns the sorted list formed by merging `List1` and `List2`. Both `List1` and\n`List2` must be sorted according to the\n[ordering function](`m:lists#ordering_function`) `Fun` and contain no duplicates\nbefore evaluating this function.\n\n`Fun(A, B)` is to return `true` if `A` compares\nless than or equal to `B` in the ordering, otherwise `false`. When two elements\ncompare equal, the element from `List1` is picked and the one from `List2` is\ndeleted.","ref":"lists.html#umerge/3"},{"type":"function","title":"lists.uniq/1","doc":"Returns a list containing the elements of `List1` with duplicated elements\nremoved (preserving the order of the elements). The first occurrence of each\nelement is kept.\n\n_Examples:_\n\n```erlang\n> lists:uniq([3,3,1,2,1,2,3]).\n[3,1,2]\n> lists:uniq([a, a, 1, b, 2, a, 3]).\n[a, 1, b, 2, 3]\n```","ref":"lists.html#uniq/1"},{"type":"function","title":"lists.uniq/2","doc":"Returns a list containing the elements of `List1` without the elements for which\n`Fun` returned duplicate values (preserving the order of the elements). The\nfirst occurrence of each element is kept.\n\n_Examples:_\n\n```erlang\n> lists:uniq(fun({X, _}) -> X end, [{b, 2}, {a, 1}, {c, 3}, {a, 2}]).\n[{b, 2}, {a, 1}, {c, 3}]\n```","ref":"lists.html#uniq/2"},{"type":"function","title":"lists.unzip3/1","doc":"\"Unzips\" a list of three-tuples into three lists, where the first list contains\nthe first element of each tuple, the second list contains the second element of\neach tuple, and the third list contains the third element of each tuple.","ref":"lists.html#unzip3/1"},{"type":"function","title":"lists.unzip/1","doc":"\"Unzips\" a list of two-tuples into two lists, where the first list contains the\nfirst element of each tuple, and the second list contains the second element of\neach tuple.","ref":"lists.html#unzip/1"},{"type":"function","title":"lists.usort/1","doc":"Returns a list containing the sorted elements of `List1` where all except the\nfirst element of the elements comparing equal have been deleted.","ref":"lists.html#usort/1"},{"type":"function","title":"lists.usort/2","doc":"Returns a list containing the sorted elements of `List1` where all except the\nfirst element of the elements comparing equal according to the\n[ordering function](`m:lists#ordering_function`) `Fun` have been deleted.\n`Fun(A, B)` is to return `true` if `A` compares less than or equal to `B` in the\nordering, otherwise `false`.","ref":"lists.html#usort/2"},{"type":"function","title":"lists.zip3/3","doc":"","ref":"lists.html#zip3/3"},{"type":"function","title":"lists.zip3/4","doc":"\"Zips\" three lists into one list of three-tuples, where the first element of\neach tuple is taken from the first list, the second element is taken from the\ncorresponding element in the second list, and the third element is taken from\nthe corresponding element in the third list.\n\nFor a description of the `How` parameter, see `zip/3`.","ref":"lists.html#zip3/4"},{"type":"function","title":"lists.zip/2","doc":"","ref":"lists.html#zip/2"},{"type":"function","title":"lists.zip/3","doc":"\"Zips\" two lists into one list of two-tuples, where the first element of each\ntuple is taken from the first list and the second element is taken from the\ncorresponding element in the second list.\n\nThe `How` parameter specifies the behavior if the given lists are of different\nlengths.\n\n- **`fail`** - The call will fail if the given lists are not of equal length.\n  This is the default.\n\n- **`trim`** - Surplus elements from the longer list will be ignored.\n\n  _Examples:_\n\n  ```erlang\n  > lists:zip([a, b], [1, 2, 3], trim).\n  [{a,1},{b,2}]\n  > lists:zip([a, b, c], [1, 2], trim).\n  [{a,1},{b,2}]\n  ```\n\n- **`{pad, Defaults}`** - The shorter list will be padded to the length of the\n  longer list, using the respective elements from the given `Defaults` tuple.\n\n  _Examples:_\n\n  ```erlang\n  > lists:zip([a, b], [1, 2, 3], {pad, {x, 0}}).\n  [{a,1},{b,2},{x,3}]\n  > lists:zip([a, b, c], [1, 2], {pad, {x, 0}}).\n  [{a,1},{b,2},{c,0}]\n  ```","ref":"lists.html#zip/3"},{"type":"function","title":"lists.zipwith3/4","doc":"","ref":"lists.html#zipwith3/4"},{"type":"function","title":"lists.zipwith3/5","doc":"Combines the elements of three lists into one list. For each triple `X, Y, Z` of\nlist elements from the three lists, the element in the result list is\n`Combine(X, Y, Z)`.\n\nFor a description of the `How` parameter, see `zip/3`.\n\n[`zipwith3(fun(X, Y, Z) -> {X,Y,Z} end, List1, List2, List3)`](`zipwith3/4`) is\nequivalent to [`zip3(List1, List2, List3)`](`zip3/3`).\n\n_Examples:_\n\n```erlang\n> lists:zipwith3(fun(X, Y, Z) -> X+Y+Z end, [1,2,3], [4,5,6], [7,8,9]).\n[12,15,18]\n> lists:zipwith3(fun(X, Y, Z) -> [X,Y,Z] end, [a,b,c], [x,y,z], [1,2,3]).\n[[a,x,1],[b,y,2],[c,z,3]]\n```","ref":"lists.html#zipwith3/5"},{"type":"function","title":"lists.zipwith/3","doc":"","ref":"lists.html#zipwith/3"},{"type":"function","title":"lists.zipwith/4","doc":"Combines the elements of two lists into one list. For each pair `X, Y` of list\nelements from the two lists, the element in the result list is `Combine(X, Y)`.\n\nFor a description of the `How` parameter, see `zip/3`.\n\n[`zipwith(fun(X, Y) -> {X,Y} end, List1, List2)`](`zipwith/3`) is equivalent to\n[`zip(List1, List2)`](`zip/2`).\n\n_Example:_\n\n```erlang\n> lists:zipwith(fun(X, Y) -> X+Y end, [1,2,3], [4,5,6]).\n[5,7,9]\n```","ref":"lists.html#zipwith/4"},{"type":"module","title":"maps","doc":"Maps processing functions.\n\nThis module contains functions for maps processing. The Efficiency Guide\ncontains a chapter that describes\n[how to use maps efficiently](`e:system:maps.md`).","ref":"maps.html"},{"type":"function","title":"maps.filter/2","doc":"Returns a map `Map` for which predicate `Pred` holds true in `MapOrIter`.\n\nThe call fails with a `{badmap,Map}` exception if `MapOrIter` is not a map or\nvalid iterator, or with `badarg` if `Pred` is not a function of arity 2.\n\n_Example:_\n\n```erlang\n> M = #{a => 2, b => 3, c=> 4, \"a\" => 1, \"b\" => 2, \"c\" => 4},\n  Pred = fun(K,V) -> is_atom(K) andalso (V rem 2) =:= 0 end,\n  maps:filter(Pred,M).\n#{a => 2,c => 4}\n```","ref":"maps.html#filter/2"},{"type":"function","title":"maps.filtermap/2","doc":"Returns a map `Map` that is the result of calling `Fun(Key, Value1)` for every\n`Key` to value `Value1` association in `MapOrIter` in any order.\n\nIf `Fun(Key, Value1)` returns `true`, the association is copied to the result\nmap. If it returns `false`, the association is not copied. If it returns\n`{true, NewValue}`, the value for `Key` is replaced with `NewValue` in the\nresult map.\n\nThe call fails with a `{badmap,Map}` exception if `MapOrIter` is not a map or\nvalid iterator, or with `badarg` if `Fun` is not a function of arity 2.\n\n_Example:_\n\n```erlang\n> Fun = fun(K,V) when is_atom(K) -> {true, V*2}; (_,V) -> (V rem 2) =:= 0 end,\n  Map = #{k1 => 1, \"k2\" => 2, \"k3\" => 3},\n  maps:filtermap(Fun,Map).\n#{k1 => 2,\"k2\" => 2}\n```","ref":"maps.html#filtermap/2"},{"type":"function","title":"maps.find/2","doc":"Returns a tuple `{ok, Value}`, where `Value` is the value associated with `Key`,\nor `error` if no value is associated with `Key` in `Map`.\n\nThe call fails with a `{badmap,Map}` exception if `Map` is not a map.\n\n_Example:_\n\n```erlang\n> Map = #{\"hi\" => 42},\n  Key = \"hi\",\n  maps:find(Key,Map).\n{ok,42}\n```","ref":"maps.html#find/2"},{"type":"function","title":"maps.fold/3","doc":"Calls `F(Key, Value, AccIn)` for every `Key` to value `Value` association in\n`MapOrIter` in any order. Function `fun F/3` must return a new accumulator,\nwhich is passed to the next successive call. This function returns the final\nvalue of the accumulator. The initial accumulator value `Init` is returned if\nthe map is empty.\n\nThe call fails with a `{badmap,Map}` exception if `MapOrIter` is not a map or\nvalid iterator, or with `badarg` if `Fun` is not a function of arity 3.\n\n_Example:_\n\n```erlang\n> Fun = fun(K,V,AccIn) when is_list(K) -> AccIn + V end,\n  Map = #{\"k1\" => 1, \"k2\" => 2, \"k3\" => 3},\n  maps:fold(Fun,0,Map).\n6\n```","ref":"maps.html#fold/3"},{"type":"function","title":"maps.foreach/2","doc":"Calls `fun F(Key, Value)` for every `Key` to value `Value` association in\n`MapOrIter` in any order.\n\nThe call fails with a `{badmap,Map}` exception if `MapOrIter` is not a map or\nvalid iterator, or with `badarg` if `Fun` is not a function of arity 2.","ref":"maps.html#foreach/2"},{"type":"function","title":"maps.from_keys/2","doc":"Takes a list of keys and a value and builds a map where all keys point to the\nsame value. The key can be in any order, and keys and value can be of any term.\n\n_Example:_\n\n```erlang\n> Keys = [\"a\", \"b\", \"c\"], maps:from_keys(Keys, ok).\n#{\"a\" => ok,\"b\" => ok,\"c\" => ok}\n```","ref":"maps.html#from_keys/2"},{"type":"function","title":"maps.from_list/1","doc":"Takes a list of key-value tuples elements and builds a map. The associations can\nbe in any order, and both keys and values in the association can be of any term.\n\n\nIf the same key appears more than once, the latter (right-most) value is used\nand the previous values are ignored.\n\n_Example:_\n\n```erlang\n> List = [{\"a\",ignored},{1337,\"value two\"},{42,value_three},{\"a\",1}],\n  maps:from_list(List).\n#{42 => value_three,1337 => \"value two\",\"a\" => 1}\n```","ref":"maps.html#from_list/1"},{"type":"function","title":"maps.get/2","doc":"Returns value `Value` associated with `Key` if `Map` contains `Key`.\n\nThe call fails with a `{badmap,Map}` exception if `Map` is not a map, or with a\n`{badkey,Key}` exception if no value is associated with `Key`.\n\n_Example:_\n\n```erlang\n> Key = 1337,\n  Map = #{42 => value_two,1337 => \"value one\",\"a\" => 1},\n  maps:get(Key,Map).\n\"value one\"\n```","ref":"maps.html#get/2"},{"type":"function","title":"maps.get/3","doc":"Returns value `Value` associated with `Key` if `Map` contains `Key`. If no value\nis associated with `Key`, `Default` is returned.\n\nThe call fails with a `{badmap,Map}` exception if `Map` is not a map.\n\n_Example:_\n\n```erlang\n> Map = #{ key1 => val1, key2 => val2 }.\n#{key1 => val1,key2 => val2}\n> maps:get(key1, Map, \"Default value\").\nval1\n> maps:get(key3, Map, \"Default value\").\n\"Default value\"\n```","ref":"maps.html#get/3"},{"type":"function","title":"maps.groups_from_list/2","doc":"Partitions the given `List` into a map of groups.\n\nThe result is a map where each key is given by `KeyFun` and each value is a list\nof elements from the given `List` for which `KeyFun` returned the same key.\n\nThe order of elements within each group list is preserved from the original\nlist.\n\n_Examples:_\n\n```erlang\n> EvenOdd = fun(X) -> case X rem 2 of 0 -> even; 1 -> odd end end,\nmaps:groups_from_list(EvenOdd, [1, 2, 3]).\n#{even => [2], odd => [1, 3]}\n> maps:groups_from_list(fun erlang:length/1, [\"ant\", \"buffalo\", \"cat\", \"dingo\"]).\n#{3 => [\"ant\", \"cat\"], 5 => [\"dingo\"], 7 => [\"buffalo\"]}\n```","ref":"maps.html#groups_from_list/2"},{"type":"function","title":"maps.groups_from_list/3","doc":"Partitions the given `List` into a map of groups.\n\nThe result is a map where each key is given by `KeyFun` and each value is a list\nof elements from the given `List`, mapped via `ValueFun`, for which `KeyFun`\nreturned the same key.\n\nThe order of elements within each group list is preserved from the original\nlist.\n\n_Examples:_\n\n```erlang\n> EvenOdd = fun(X) -> case X rem 2 of 0 -> even; 1 -> odd end end,\n> Square = fun(X) -> X * X end,\n> maps:groups_from_list(EvenOdd, Square, [1, 2, 3]).\n#{even => [4], odd => [1, 9]}\n> maps:groups_from_list(\n    fun erlang:length/1,\n    fun lists:reverse/1,\n    [\"ant\", \"buffalo\", \"cat\", \"dingo\"]).\n#{3 => [\"tna\", \"tac\"],5 => [\"ognid\"],7 => [\"olaffub\"]}\n```","ref":"maps.html#groups_from_list/3"},{"type":"function","title":"maps.intersect/2","doc":"Intersects two maps into a single map `Map3`. If a key exists in both maps, the\nvalue in `Map1` is superseded by the value in `Map2`.\n\nThe call fails with a `{badmap,Map}` exception if `Map1` or `Map2` is not a map.\n\n_Example:_\n\n```erlang\n> Map1 = #{a => \"value_one\", b => \"value_two\"},\n  Map2 = #{a => 1, c => 2},\n  maps:intersect(Map1,Map2).\n#{a => 1}\n```","ref":"maps.html#intersect/2"},{"type":"function","title":"maps.intersect_with/3","doc":"Intersects two maps into a single map `Map3`. If a key exists in both maps, the\nvalue in `Map1` is combined with the value in `Map2` by the `Combiner` fun.\n\nWhen `Combiner` is applied the key that exists in both maps is the first parameter,\nthe value from `Map1` is the second parameter, and the value from `Map2` is the\nthird parameter.\n\nThe call fails with a `{badmap,Map}` exception if `Map1` or `Map2` is not a map.\nThe call fails with a `badarg` exception if `Combiner` is not a fun that takes\nthree arguments.\n\n_Example:_\n\n```erlang\n> Map1 = #{a => \"value_one\", b => \"value_two\"},\n  Map2 = #{a => 1, c => 2},\n  maps:intersect_with(fun(_Key, Value1, Value2) -> {Value1, Value2} end, Map1, Map2).\n#{a => {\"value_one\",1}}\n```","ref":"maps.html#intersect_with/3"},{"type":"function","title":"maps.is_key/2","doc":"Returns `true` if map `Map` contains `Key` and returns `false` if it does not\ncontain the `Key`.\n\nThe call fails with a `{badmap,Map}` exception if `Map` is not a map.\n\n_Example:_\n\n```erlang\n> Map = #{\"42\" => value}.\n#{\"42\" => value}\n> maps:is_key(\"42\",Map).\ntrue\n> maps:is_key(value,Map).\nfalse\n```","ref":"maps.html#is_key/2"},{"type":"type","title":"maps.iterator/0","doc":"","ref":"maps.html#t:iterator/0"},{"type":"function","title":"maps.iterator/1","doc":"Returns a map iterator `Iterator` that can be used by [`maps:next/1`](`next/1`)\nto traverse the key-value associations in a map. When iterating over a map, the\nmemory usage is guaranteed to be bounded no matter the size of the map.\n\nThe call fails with a `{badmap,Map}` exception if `Map` is not a map.\n\n_Example:_\n\n```erlang\n> M = #{ a => 1, b => 2 }.\n#{a => 1,b => 2}\n> I = maps:iterator(M), ok.\nok\n> {K1, V1, I2} = maps:next(I), {K1, V1}.\n{a,1}\n> {K2, V2, I3} = maps:next(I2),{K2, V2}.\n{b,2}\n> maps:next(I3).\nnone\n```","ref":"maps.html#iterator/1"},{"type":"opaque","title":"maps.iterator/2","doc":"An iterator representing the associations in a map with keys of type `Key` and\nvalues of type `Value`.\n\nCreated using [`maps:iterator/1`](`iterator/1`) or\n[`maps:iterator/2`](`iterator/2`).\n\nConsumed by:\n\n- [`maps:next/1`](`next/1`)\n- [`maps:filter/2`](`filter/2`)\n- [`maps:filtermap/2`](`filtermap/2`)\n- [`maps:fold/3`](`fold/3`)\n- [`maps:foreach/2`](`foreach/2`)\n- [`maps:map/2`](`map/2`)\n- [`maps:to_list/1`](`to_list/1`)","ref":"maps.html#t:iterator/2"},{"type":"function","title":"maps.iterator/2","doc":"Returns a map iterator `Iterator` that can be used by [`maps:next/1`](`next/1`)\nto traverse the key-value associations in a map sorted by key using the given\n`Order`.\n\nThe call fails with a `{badmap,Map}` exception if `Map` is not a map or if\n`Order` is invalid.\n\n_Example (when _`Order`_ is _`ordered`_):_\n\n```erlang\n> M = #{ a => 1, b => 2 }.\n#{a => 1,b => 2}\n> OrdI = maps:iterator(M, ordered), ok.\nok\n> {K1, V1, OrdI2} = maps:next(OrdI), {K1, V1}.\n{a,1}\n> {K2, V2, OrdI3} = maps:next(OrdI2),{K2, V2}.\n{b,2}\n> maps:next(OrdI3).\nnone\n```\n\n_Example (when _`Order`_ is _`reversed`_):_\n\n```erlang\n> M = #{ a => 1, b => 2 }.\n#{a => 1,b => 2}\n> RevI = maps:iterator(M, reversed), ok.\nok\n> {K2, V2, RevI2} = maps:next(RevI), {K2, V2}.\n{b,2}\n> {K1, V1, RevI3} = maps:next(RevI2),{K1, V1}.\n{a,1}\n> maps:next(RevI3).\nnone\n```\n\n_Example (when _`Order`_ is an arithmetic sorting function):_\n\n```erlang\n> M = #{ -1 => a, -1.0 => b, 0 => c, 0.0 => d }.\n#{-1 => a,0 => c,-1.0 => b,0.0 => d}\n> ArithOrdI = maps:iterator(M, fun(A, B) -> A =  maps:to_list(ArithOrdI).\n[{-1,a},{-1.0,b},{0,c},{0.0,d}]\n> ArithRevI = maps:iterator(M, fun(A, B) -> B   maps:to_list(ArithRevI).\n[{0.0,d},{0,c},{-1.0,b},{-1,a}]\n```","ref":"maps.html#iterator/2"},{"type":"type","title":"maps.iterator_order/0","doc":"","ref":"maps.html#t:iterator_order/0"},{"type":"type","title":"maps.iterator_order/1","doc":"Key-based iterator order option that can be one of `undefined` (default for\n[`maps:iterator/1`](`iterator/1`)), `ordered` (sorted in map-key order),\n`reversed`, or a custom sorting function.\n\nUsed by [`maps:iterator/2`](`iterator/2`).\n\nThe [Expressions section](`e:system:expressions.md#term-comparisons`) contains\ndescriptions of how terms are ordered.","ref":"maps.html#t:iterator_order/1"},{"type":"function","title":"maps.keys/1","doc":"Returns a complete list of keys, in any order, which resides within `Map`.\n\nThe call fails with a `{badmap,Map}` exception if `Map` is not a map.\n\n_Example:_\n\n```erlang\n> Map = #{42 => value_three,1337 => \"value two\",\"a\" => 1},\n  maps:keys(Map).\n[42,1337,\"a\"]\n```","ref":"maps.html#keys/1"},{"type":"function","title":"maps.map/2","doc":"Produces a new map `Map` by calling function `fun F(Key, Value1)` for every\n`Key` to value `Value1` association in `MapOrIter` in any order. Function\n`fun Fun/2` must return value `Value2` to be associated with key `Key` for the\nnew map `Map`.\n\nThe call fails with a `{badmap,Map}` exception if `MapOrIter` is not a map or\nvalid iterator, or with `badarg` if `Fun` is not a function of arity 2.\n\n_Example:_\n\n```erlang\n> Fun = fun(K,V1) when is_list(K) -> V1*2 end,\n  Map = #{\"k1\" => 1, \"k2\" => 2, \"k3\" => 3},\n  maps:map(Fun,Map).\n#{\"k1\" => 2,\"k2\" => 4,\"k3\" => 6}\n```","ref":"maps.html#map/2"},{"type":"function","title":"maps.merge/2","doc":"Merges two maps into a single map `Map3`. If two keys exist in both maps, the\nvalue in `Map1` is superseded by the value in `Map2`.\n\nThe call fails with a `{badmap,Map}` exception if `Map1` or `Map2` is not a map.\n\n_Example:_\n\n```erlang\n> Map1 = #{a => \"value_one\", b => \"value_two\"},\n  Map2 = #{a => 1, c => 2},\n  maps:merge(Map1,Map2).\n#{a => 1,b => \"value_two\",c => 2}\n```","ref":"maps.html#merge/2"},{"type":"function","title":"maps.merge_with/3","doc":"Merges two maps into a single map `Map3`. If a key exists in both maps, the\nvalue in `Map1` is combined with the value in `Map2` by the `Combiner` fun.\n\nWhen `Combiner` is applied the key that exists in both maps is the first parameter,\nthe value from `Map1` is the second parameter, and the value from `Map2` is the\nthird parameter.\n\nThe call fails with a `{badmap,Map}` exception if `Map1` or `Map2` is not a map.\nThe call fails with a `badarg` exception if `Combiner` is not a fun that takes\nthree arguments.\n\n_Example:_\n\n```erlang\n> Map1 = #{a => \"value_one\", b => \"value_two\"},\n  Map2 = #{a => 1, c => 2},\n  maps:merge_with(fun(_Key, Value1, Value2) -> {Value1, Value2} end, Map1, Map2).\n#{a => {\"value_one\",1},b => \"value_two\",c => 2}\n```","ref":"maps.html#merge_with/3"},{"type":"function","title":"maps.new/0","doc":"Returns a new empty map.\n\n_Example:_\n\n```text\n> maps:new().\n#{}\n```","ref":"maps.html#new/0"},{"type":"function","title":"maps.next/1","doc":"Returns the next key-value association in `Iterator` and a new iterator for the\nremaining associations in the iterator.\n\nIf there are no more associations in the iterator, `none` is returned.\n\n_Example:_\n\n```erlang\n> Map = #{a => 1, b => 2, c => 3}.\n#{a => 1,b => 2,c => 3}\n> I = maps:iterator(Map), ok.\nok\n> {K1, V1, I1} = maps:next(I), {K1, V1}.\n{a,1}\n> {K2, V2, I2} = maps:next(I1), {K2, V2}.\n{b,2}\n> {K3, V3, I3} = maps:next(I2), {K3, V3}.\n{c,3}\n> maps:next(I3).\nnone\n```","ref":"maps.html#next/1"},{"type":"function","title":"maps.put/3","doc":"Associates `Key` with value `Value` and inserts the association into map `Map2`.\nIf key `Key` already exists in map `Map1`, the old associated value is replaced\nby value `Value`. The function returns a new map `Map2` containing the new\nassociation and the old associations in `Map1`.\n\nThe call fails with a `{badmap,Map}` exception if `Map1` is not a map.\n\n_Example:_\n\n```erlang\n> Map = #{\"a\" => 1}.\n#{\"a\" => 1}\n> maps:put(\"a\", 42, Map).\n#{\"a\" => 42}\n> maps:put(\"b\", 1337, Map).\n#{\"a\" => 1,\"b\" => 1337}\n```","ref":"maps.html#put/3"},{"type":"function","title":"maps.remove/2","doc":"Removes the `Key`, if it exists, and its associated value from `Map1` and\nreturns a new map `Map2` without key `Key`.\n\nThe call fails with a `{badmap,Map}` exception if `Map1` is not a map.\n\n_Example:_\n\n```erlang\n> Map = #{\"a\" => 1}.\n#{\"a\" => 1}\n> maps:remove(\"a\",Map).\n#{}\n> maps:remove(\"b\",Map).\n#{\"a\" => 1}\n```","ref":"maps.html#remove/2"},{"type":"function","title":"maps.size/1","doc":"Returns the number of key-value associations in `Map`. This operation occurs in\nconstant time.\n\n_Example:_\n\n```erlang\n> Map = #{42 => value_two,1337 => \"value one\",\"a\" => 1},\n  maps:size(Map).\n3\n```","ref":"maps.html#size/1"},{"type":"function","title":"maps.take/2","doc":"The function removes the `Key`, if it exists, and its associated value from\n`Map1` and returns a tuple with the removed `Value` and the new map `Map2`\nwithout key `Key`. If the key does not exist `error` is returned.\n\nThe call will fail with a `{badmap,Map}` exception if `Map1` is not a map.\n\nExample:\n\n```erlang\n> Map = #{\"a\" => \"hello\", \"b\" => \"world\"}.\n#{\"a\" => \"hello\", \"b\" => \"world\"}\n> maps:take(\"a\",Map).\n{\"hello\",#{\"b\" => \"world\"}}\n> maps:take(\"does not exist\",Map).\nerror\n```","ref":"maps.html#take/2"},{"type":"function","title":"maps.to_list/1","doc":"Returns a list of pairs representing the key-value associations of\n`MapOrIterator`, where the pairs `[{K1,V1}, ..., {Kn,Vn}]` are returned in\narbitrary order.\n\nThe call fails with a `{badmap,Map}` exception if `MapOrIterator` is not a map\nor an iterator obtained by a call to `iterator/1` or `iterator/2`.\n\n_Example:_\n\n```erlang\n> Map = #{42 => value_three,1337 => \"value two\",\"a\" => 1},\n  maps:to_list(Map).\n[{42,value_three},{1337,\"value two\"},{\"a\",1}]\n```\n\n_Example (using _`iterator/2`_):_\n\n```erlang\n> Map = #{ z => 1, y => 2, x => 3 }.\n#{x => 3,y => 2,z => 1}\n> maps:to_list(maps:iterator(Map, ordered)).\n[{x,3},{y,2},{z,1}]\n```","ref":"maps.html#to_list/1"},{"type":"function","title":"maps.update/3","doc":"If `Key` exists in `Map1`, the old associated value is replaced by value\n`Value`. The function returns a new map `Map2` containing the new associated\nvalue.\n\nThe call fails with a `{badmap,Map}` exception if `Map1` is not a map, or with a\n`{badkey,Key}` exception if no value is associated with `Key`.\n\n_Example:_\n\n```erlang\n> Map = #{\"a\" => 1}.\n#{\"a\" => 1}\n> maps:update(\"a\", 42, Map).\n#{\"a\" => 42}\n```","ref":"maps.html#update/3"},{"type":"function","title":"maps.update_with/3","doc":"Update a value in a `Map1` associated with `Key` by calling `Fun` on the old\nvalue to get a new value. An exception `{badkey,Key}` is generated if `Key` is\nnot present in the map.\n\nExample:\n\n```erlang\n> Map = #{\"counter\" => 1},\n  Fun = fun(V) -> V + 1 end,\n  maps:update_with(\"counter\",Fun,Map).\n#{\"counter\" => 2}\n```","ref":"maps.html#update_with/3"},{"type":"function","title":"maps.update_with/4","doc":"Update a value in a `Map1` associated with `Key` by calling `Fun` on the old\nvalue to get a new value. If `Key` is not present in `Map1` then `Init` will be\nassociated with `Key`.\n\nExample:\n\n```erlang\n> Map = #{\"counter\" => 1},\n  Fun = fun(V) -> V + 1 end,\n  maps:update_with(\"new counter\",Fun,42,Map).\n#{\"counter\" => 1,\"new counter\" => 42}\n```","ref":"maps.html#update_with/4"},{"type":"function","title":"maps.values/1","doc":"Returns a complete list of values, in arbitrary order, contained in map `Map`.\n\nThe call fails with a `{badmap,Map}` exception if `Map` is not a map.\n\n_Example:_\n\n```erlang\n> Map = #{42 => value_three,1337 => \"value two\",\"a\" => 1},\n  maps:values(Map).\n[value_three,\"value two\",1]\n```","ref":"maps.html#values/1"},{"type":"function","title":"maps.with/2","doc":"Returns a new map `Map2` with the keys `K1` through `Kn` and their associated\nvalues from map `Map1`. Any key in `Ks` that does not exist in `Map1` is\nignored.\n\n_Example:_\n\n```erlang\n> Map = #{42 => value_three,1337 => \"value two\",\"a\" => 1},\n  Ks = [\"a\",42,\"other key\"],\n  maps:with(Ks,Map).\n#{42 => value_three,\"a\" => 1}\n```","ref":"maps.html#with/2"},{"type":"function","title":"maps.without/2","doc":"Returns a new map `Map2` without keys `K1` through `Kn` and their associated\nvalues from map `Map1`. Any key in `Ks` that does not exist in `Map1` is ignored\n\n_Example:_\n\n```erlang\n> Map = #{42 => value_three,1337 => \"value two\",\"a\" => 1},\n  Ks = [\"a\",42,\"other key\"],\n  maps:without(Ks,Map).\n#{1337 => \"value two\"}\n```","ref":"maps.html#without/2"},{"type":"module","title":"math","doc":"Mathematical functions.\n\nThis module provides an interface to a number of mathematical functions.\nFor details about what each function does, see the the C library documentation\non your system. On Unix systems the easiest way it to run `man sin`. On\nWindows you should check the [Math and floating-point support](https://learn.microsoft.com/en-us/cpp/c-runtime-library/floating-point-support)\ndocumentation.","ref":"math.html"},{"type":"module","title":"Limitations - math","doc":"As these are the C library, the same limitations apply.","ref":"math.html#module-limitations"},{"type":"function","title":"math.acos/1","doc":"Inverse cosine of `X`, return value is in radians.","ref":"math.html#acos/1"},{"type":"function","title":"math.acosh/1","doc":"Inverse hyperbolic cosine of `X`.","ref":"math.html#acosh/1"},{"type":"function","title":"math.asin/1","doc":"Inverse sine of `X`, return value is in radians.","ref":"math.html#asin/1"},{"type":"function","title":"math.asinh/1","doc":"Inverse hyperbolic sine of `X`.","ref":"math.html#asinh/1"},{"type":"function","title":"math.atan2/2","doc":"Inverse 2-argument tangent of `X`, return value is in radians.","ref":"math.html#atan2/2"},{"type":"function","title":"math.atan/1","doc":"Inverse tangent of `X`, return value is in radians.","ref":"math.html#atan/1"},{"type":"function","title":"math.atanh/1","doc":"Inverse hyperbolic tangent of `X`.","ref":"math.html#atanh/1"},{"type":"function","title":"math.ceil/1","doc":"The ceiling of `X`.","ref":"math.html#ceil/1"},{"type":"function","title":"math.cos/1","doc":"The cosine of `X` in radians.","ref":"math.html#cos/1"},{"type":"function","title":"math.cosh/1","doc":"The hyperbolic cosine of `X`.","ref":"math.html#cosh/1"},{"type":"function","title":"math.erf/1","doc":"Returns the error function (or Gauss error function) of `X`.\n\nWhere:\n\n```text\nerf(X) = 2/sqrt(pi)*integral from 0 to X of exp(-t*t) dt.\n```","ref":"math.html#erf/1"},{"type":"function","title":"math.erfc/1","doc":"[`erfc(X)`](`erfc/1`) returns `1.0` - [`erf(X)`](`erf/1`), computed by methods\nthat avoid cancellation for large `X`.","ref":"math.html#erfc/1"},{"type":"function","title":"math.exp/1","doc":"Raise e by `X`, that is `eˣ`.\n\nWhere e is the base of the natural logarithm.","ref":"math.html#exp/1"},{"type":"function","title":"math.floor/1","doc":"The floor of `X`.","ref":"math.html#floor/1"},{"type":"function","title":"math.fmod/2","doc":"Returns `X` modulus `Y`.","ref":"math.html#fmod/2"},{"type":"function","title":"math.log2/1","doc":"The base-2 logarithm of `X`.","ref":"math.html#log2/1"},{"type":"function","title":"math.log10/1","doc":"The base-10 logarithm of `X`.","ref":"math.html#log10/1"},{"type":"function","title":"math.log/1","doc":"The natural (base-e) logarithm of `X`.","ref":"math.html#log/1"},{"type":"function","title":"math.pi/0","doc":"Ratio of the circumference of a circle to its diameter.\n\nFloating point approximation of mathematical constant pi.","ref":"math.html#pi/0"},{"type":"function","title":"math.pow/2","doc":"Raise `X` by `N`, that is `xⁿ`.","ref":"math.html#pow/2"},{"type":"function","title":"math.sin/1","doc":"Sine of `X` in radians.","ref":"math.html#sin/1"},{"type":"function","title":"math.sinh/1","doc":"Hyperbolic sine of `X`.","ref":"math.html#sinh/1"},{"type":"function","title":"math.sqrt/1","doc":"Square root of `X`.","ref":"math.html#sqrt/1"},{"type":"function","title":"math.tan/1","doc":"Tangent of `X` in radians.","ref":"math.html#tan/1"},{"type":"function","title":"math.tanh/1","doc":"Hyperbolic tangent of `X`.","ref":"math.html#tanh/1"},{"type":"function","title":"math.tau/0","doc":"Ratio of the circumference of a circle to its radius.\n\nThis constant is equivalent to a full turn when described in radians.\n\nThe same as `2 * pi()`.","ref":"math.html#tau/0"},{"type":"module","title":"calendar","doc":"Local and universal time, day of the week, date and time conversions.\n\nThis module provides computation of local and universal time, day of the week,\nand many time conversion functions.\n\nTime is local when it is adjusted in accordance with the current time zone and\ndaylight saving. Time is universal when it reflects the time at longitude zero,\nwithout any adjustment for daylight saving. Universal Coordinated Time (UTC)\ntime is also called Greenwich Mean Time (GMT).\n\nThe time functions `local_time/0` and `universal_time/0` in this module both\nreturn date and time. This is because separate functions for date and time can\nresult in a date/time combination that is displaced by 24 hours. This occurs if\none of the functions is called before midnight, and the other after midnight.\nThis problem also applies to the Erlang BIFs `date/0` and `time/0`, and their\nuse is strongly discouraged if a reliable date/time stamp is required.\n\nAll dates conform to the Gregorian calendar. This calendar was introduced by\nPope Gregory XIII in 1582 and was used in all Catholic countries from this year.\nProtestant parts of Germany and the Netherlands adopted it in 1698, England\nfollowed in 1752, and Russia in 1918 (the October revolution of 1917 took place\nin November according to the Gregorian calendar).\n\nThe Gregorian calendar in this module is extended back to year 0. For a given\ndate, the _gregorian days_ is the number of days up to and including the date\nspecified. Similarly, the _gregorian seconds_ for a specified date and time is\nthe number of seconds up to and including the specified date and time.\n\nFor computing differences between epochs in time, use the functions counting\ngregorian days or seconds. If epochs are specified as local time, they must be\nconverted to universal time to get the correct value of the elapsed time between\nepochs. Use of function [`time_difference/2`](`time_difference/2`) is\ndiscouraged.\n\nDifferent definitions exist for the week of the year. This module contains a\nweek of the year implementation conforming to the ISO 8601 standard. As the week\nnumber for a specified date can fall on the previous, the current, or on the\nnext year, it is important to specify both the year and the week number.\nFunctions `iso_week_number/0` and [`iso_week_number/1`](`iso_week_number/1`)\nreturn a tuple of the year and the week number.","ref":"calendar.html"},{"type":"module","title":"Leap Years - calendar","doc":"The notion that every fourth year is a leap year is not completely true. By the\nGregorian rule, a year Y is a leap year if one of the following rules is valid:\n\n- Y is divisible by 4, but not by 100.\n- Y is divisible by 400.\n\nHence, 1996 is a leap year, 1900 is not, but 2000 is.","ref":"calendar.html#module-leap-years"},{"type":"module","title":"Date and Time Source - calendar","doc":"Local time is obtained from the Erlang BIF `localtime/0`. Universal time is\ncomputed from the BIF `universaltime/0`.\n\nThe following apply:\n\n- There are 86400 seconds in a day.\n- There are 365 days in an ordinary year.\n- There are 366 days in a leap year.\n- There are 1461 days in a 4 year period.\n- There are 36524 days in a 100 year period.\n- There are 146097 days in a 400 year period.\n- There are 719528 days between Jan 1, 0 and Jan 1, 1970.","ref":"calendar.html#module-date-and-time-source"},{"type":"type","title":"calendar.date/0","doc":"A date using the Gregorian calendar.\n\nAll APIs expect this to be a valid date. If the source of the date\nis unknown, then verify that is it valid by calling `valid_date/1`\nbefore using it.","ref":"calendar.html#t:date/0"},{"type":"function","title":"calendar.date_to_gregorian_days/1","doc":"Computes the number of gregorian days starting with year 0 and ending at the\nspecified date.","ref":"calendar.html#date_to_gregorian_days/1"},{"type":"function","title":"calendar.date_to_gregorian_days/3","doc":"","ref":"calendar.html#date_to_gregorian_days/3"},{"type":"type","title":"calendar.datetime1970/0","doc":"","ref":"calendar.html#t:datetime1970/0"},{"type":"type","title":"calendar.datetime/0","doc":"","ref":"calendar.html#t:datetime/0"},{"type":"function","title":"calendar.datetime_to_gregorian_seconds/1","doc":"Computes the number of gregorian seconds starting with year 0 and ending at the\nspecified date and time.","ref":"calendar.html#datetime_to_gregorian_seconds/1"},{"type":"type","title":"calendar.day/0","doc":"","ref":"calendar.html#t:day/0"},{"type":"function","title":"calendar.day_of_the_week/1","doc":"Computes the day of the week from the specified `Year`, `Month`, and `Day`.\nReturns the day of the week as `1`: Monday, `2`: Tuesday, and so on.","ref":"calendar.html#day_of_the_week/1"},{"type":"function","title":"calendar.day_of_the_week/3","doc":"","ref":"calendar.html#day_of_the_week/3"},{"type":"type","title":"calendar.daynum/0","doc":"","ref":"calendar.html#t:daynum/0"},{"type":"function","title":"calendar.gregorian_days_to_date/1","doc":"Computes the date from the specified number of gregorian days.","ref":"calendar.html#gregorian_days_to_date/1"},{"type":"function","title":"calendar.gregorian_seconds_to_datetime/1","doc":"Computes the date and time from the specified number of gregorian seconds.","ref":"calendar.html#gregorian_seconds_to_datetime/1"},{"type":"type","title":"calendar.hour/0","doc":"","ref":"calendar.html#t:hour/0"},{"type":"function","title":"calendar.is_leap_year/1","doc":"Checks if the specified year is a leap year.","ref":"calendar.html#is_leap_year/1"},{"type":"function","title":"calendar.iso_week_number/0","doc":"Returns tuple `{Year, WeekNum}` representing the ISO week number for the actual\ndate. To determine the actual date, use function `local_time/0`.","ref":"calendar.html#iso_week_number/0"},{"type":"function","title":"calendar.iso_week_number/1","doc":"Returns tuple `{Year, WeekNum}` representing the ISO week number for the\nspecified date.","ref":"calendar.html#iso_week_number/1"},{"type":"function","title":"calendar.last_day_of_the_month/2","doc":"Computes the number of days in a month.","ref":"calendar.html#last_day_of_the_month/2"},{"type":"type","title":"calendar.ldom/0","doc":"The last day of the month.","ref":"calendar.html#t:ldom/0"},{"type":"function","title":"calendar.local_time/0","doc":"Returns the local time reported by the underlying operating system.","ref":"calendar.html#local_time/0"},{"type":"function","title":"calendar.local_time_to_universal_time/1","doc":"Converts from local time to Universal Coordinated Time (UTC). `DateTime1` must\nrefer to a local date after Jan 1, 1970.\n\n> #### Warning {: .warning }\n>\n> This function is deprecated. Use `local_time_to_universal_time_dst/1` instead,\n> as it gives a more correct and complete result. Especially for the period that\n> does not exist, as it is skipped during the switch _to_ daylight saving time,\n> this function still returns a result.","ref":"calendar.html#local_time_to_universal_time/1"},{"type":"function","title":"calendar.local_time_to_universal_time_dst/1","doc":"Converts from local time to Universal Coordinated Time (UTC). `DateTime1` must\nrefer to a local date after Jan 1, 1970.\n\nThe return value is a list of 0, 1, or 2 possible UTC times:\n\n- **`[]`** - For a local `{Date1, Time1}` during the period that is skipped when\n  switching _to_ daylight saving time, there is no corresponding UTC, as the\n  local time is illegal (it has never occured).\n\n- **`[DstDateTimeUTC, DateTimeUTC]`** - For a local `{Date1, Time1}` during the\n  period that is repeated when switching _from_ daylight saving time, two\n  corresponding UTCs exist; one for the first instance of the period when\n  daylight saving time is still active, and one for the second instance.\n\n- **`[DateTimeUTC]`** - For all other local times only one corresponding UTC\n  exists.","ref":"calendar.html#local_time_to_universal_time_dst/1"},{"type":"type","title":"calendar.minute/0","doc":"","ref":"calendar.html#t:minute/0"},{"type":"type","title":"calendar.month/0","doc":"","ref":"calendar.html#t:month/0"},{"type":"function","title":"calendar.now_to_datetime/1","doc":"Returns Universal Coordinated Time (UTC) converted from the return value from\n`erlang:timestamp/0`.","ref":"calendar.html#now_to_datetime/1"},{"type":"function","title":"calendar.now_to_local_time/1","doc":"Returns local date and time converted from the return value from\n`erlang:timestamp/0`.","ref":"calendar.html#now_to_local_time/1"},{"type":"function","title":"calendar.now_to_universal_time/1","doc":"Returns Universal Coordinated Time (UTC) converted from the return value from\n`erlang:timestamp/0`.","ref":"calendar.html#now_to_universal_time/1"},{"type":"type","title":"calendar.offset/0","doc":"","ref":"calendar.html#t:offset/0"},{"type":"type","title":"calendar.rfc3339_string/0","doc":"","ref":"calendar.html#t:rfc3339_string/0"},{"type":"type","title":"calendar.rfc3339_time_unit/0","doc":"The time unit used by the rfc3339 conversion functions.\n\n> #### Note {: .info }\n>\n> The `native` time unit was added to `t:rfc3339_time_unit/0` in OTP 25.0.","ref":"calendar.html#t:rfc3339_time_unit/0"},{"type":"function","title":"calendar.rfc3339_to_system_time/1","doc":"","ref":"calendar.html#rfc3339_to_system_time/1"},{"type":"function","title":"calendar.rfc3339_to_system_time/2","doc":"Converts an RFC 3339 timestamp into system time. The data format of RFC 3339\ntimestamps is described by [RFC 3339](https://www.ietf.org/rfc/rfc3339.txt).\nStarting from OTP 25.1, the minutes part of the time zone is optional.\n\nValid option:\n\n- **`{unit, Unit}`** - The time unit of the return value. The default is\n  `second`.\n\n```erlang\n1> calendar:rfc3339_to_system_time(\"2018-02-01T16:17:58+01:00\").\n1517498278\n2> calendar:rfc3339_to_system_time(\"2018-02-01 15:18:02.088Z\",\n   [{unit, nanosecond}]).\n1517498282088000000\n```","ref":"calendar.html#rfc3339_to_system_time/2"},{"type":"type","title":"calendar.second/0","doc":"","ref":"calendar.html#t:second/0"},{"type":"function","title":"calendar.seconds_to_daystime/1","doc":"Converts a specified number of seconds into days, hours, minutes, and seconds.\n`Time` is always non-negative, but `Days` is negative if argument `Seconds` is.","ref":"calendar.html#seconds_to_daystime/1"},{"type":"function","title":"calendar.seconds_to_time/1","doc":"Computes the time from the specified number of seconds. `Seconds` must be less\nthan the number of seconds per day (86400).","ref":"calendar.html#seconds_to_time/1"},{"type":"type","title":"calendar.secs_per_day/0","doc":"","ref":"calendar.html#t:secs_per_day/0"},{"type":"function","title":"calendar.system_time_to_local_time/2","doc":"Converts a specified system time into local date and time.","ref":"calendar.html#system_time_to_local_time/2"},{"type":"function","title":"calendar.system_time_to_rfc3339/1","doc":"","ref":"calendar.html#system_time_to_rfc3339/1"},{"type":"function","title":"calendar.system_time_to_rfc3339/2","doc":"Converts a system time into an RFC 3339 timestamp.\n\nThe data format of RFC 3339 timestamps is described by [RFC 3339].\nThe data format of offsets is also described by [RFC 3339].\n\nValid options:\n\n- **`{offset, Offset}`** - The offset, either a string or an integer, to be\n  included in the formatted string. An empty string, which is the default, is\n  interpreted as local time. A non-empty string is included as is. The time unit\n  of the integer is the same as the one of `Time`.\n\n- **`{time_designator, Character}`** - The character used as time designator,\n  that is, the date and time separator. The default is `$T`.\n\n- **`{unit, Unit}`** - The time unit of `Time`. The default is `second`. If some\n  other unit is given (`millisecond`, `microsecond`, `nanosecond`, or `native`),\n  the formatted string includes a fraction of a second. The number of fractional\n  second digits is three, six, or nine depending on what time unit is chosen.\n  For `native` three fractional digits are included. Notice that trailing zeros\n  are not removed from the fraction.\n\n```erlang\n1> calendar:system_time_to_rfc3339(erlang:system_time(second)).\n\"2018-04-23T14:56:28+02:00\"\n2> calendar:system_time_to_rfc3339(erlang:system_time(second),\n   [{offset, \"-02:00\"}]).\n\"2018-04-23T10:56:52-02:00\"\n3> calendar:system_time_to_rfc3339(erlang:system_time(second),\n   [{offset, -7200}]).\n\"2018-04-23T10:57:05-02:00\"\n4> calendar:system_time_to_rfc3339(erlang:system_time(millisecond),\n   [{unit, millisecond}, {time_designator, $\\s}, {offset, \"Z\"}]).\n\"2018-04-23 12:57:20.482Z\"\n```\n[RFC 3339]: https://www.ietf.org/rfc/rfc3339.txt","ref":"calendar.html#system_time_to_rfc3339/2"},{"type":"function","title":"calendar.system_time_to_universal_time/2","doc":"Converts a specified system time into universal date and time.","ref":"calendar.html#system_time_to_universal_time/2"},{"type":"type","title":"calendar.time/0","doc":"","ref":"calendar.html#t:time/0"},{"type":"function","title":"calendar.time_difference/2","doc":"Returns the difference between two `{Date, Time}` tuples. `T2` is to refer to an\nepoch later than `T1`.\n\n> #### Warning {: .warning }\n>\n> This function is obsolete. Use the conversion functions for gregorian days and\n> seconds instead.","ref":"calendar.html#time_difference/2"},{"type":"function","title":"calendar.time_to_seconds/1","doc":"Returns the number of seconds since midnight up to the specified time.","ref":"calendar.html#time_to_seconds/1"},{"type":"function","title":"calendar.universal_time/0","doc":"Returns the Universal Coordinated Time (UTC) reported by the underlying\noperating system. Returns local time if universal time is unavailable.","ref":"calendar.html#universal_time/0"},{"type":"function","title":"calendar.universal_time_to_local_time/1","doc":"Converts from Universal Coordinated Time (UTC) to local time. `DateTime` must\nrefer to a date after Jan 1, 1970.","ref":"calendar.html#universal_time_to_local_time/1"},{"type":"function","title":"calendar.valid_date/1","doc":"This function checks if a date is a valid.","ref":"calendar.html#valid_date/1"},{"type":"function","title":"calendar.valid_date/3","doc":"","ref":"calendar.html#valid_date/3"},{"type":"type","title":"calendar.weeknum/0","doc":"","ref":"calendar.html#t:weeknum/0"},{"type":"type","title":"calendar.year1970/0","doc":"","ref":"calendar.html#t:year1970/0"},{"type":"type","title":"calendar.year/0","doc":"The year using the Gregorian calendar.\n\nYear cannot be abbreviated. For example, 93 denotes year 93, not 1993. The valid\nrange depends on the underlying operating system.","ref":"calendar.html#t:year/0"},{"type":"type","title":"calendar.yearweeknum/0","doc":"","ref":"calendar.html#t:yearweeknum/0"},{"type":"module","title":"timer","doc":"Timer functions.\n\nThis module provides useful functions related to time. Unless otherwise stated,\ntime is always measured in _milliseconds_. All timer functions return\nimmediately, regardless of work done by another process.\n\nSuccessful evaluations of the timer functions give return values containing a\ntimer reference, denoted `TRef`. By using `cancel/1`, the returned reference can\nbe used to cancel any requested action. A `TRef` is an Erlang term, which\ncontents must not be changed.\n\nThe time-outs are not exact, but are _at least_ as long as requested.\n\nCreating timers using `erlang:send_after/3` and `erlang:start_timer/3` is more\nefficient than using the timers provided by this module. However, the timer\nmodule has been improved in OTP 25, making it more efficient and less\nsusceptible to being overloaded. See\n[the Timer Module section in the Efficiency Guide](`e:system:commoncaveats.md#timer-module`).\n\nFor more information on timers in Erlang in general, see the\n[*Timers*](`e:erts:time_correction.md#timers`) section of the\n[*Time and Time Correction in Erlang*](`e:erts:time_correction.md`)\nERTS User's guide.","ref":"timer.html"},{"type":"module","title":"Examples - timer","doc":"_Example 1_\n\nThe following example shows how to print \"Hello World\\!\" in 5 seconds:\n\n```erlang\n1> timer:apply_after(5000, io, format, [\"~nHello World!~n\", []]).\n{ok,TRef}\nHello World!\n```\n\n_Example 2_\n\nThe following example shows a process performing a certain action, and if this\naction is not completed within a certain limit, the process is killed:\n\n```erlang\nPid = spawn(mod, fun, [foo, bar]),\n%% If pid is not finished in 10 seconds, kill him\n{ok, R} = timer:kill_after(timer:seconds(10), Pid),\n...\n%% We change our mind...\ntimer:cancel(R),\n...\n```","ref":"timer.html#module-examples"},{"type":"module","title":"Notes - timer","doc":"A timer can always be removed by calling `cancel/1`.\n\nAn interval timer, that is, a timer created by evaluating any of the functions\n`apply_interval/2`, `apply_interval/3`, `apply_interval/4`,\n`apply_repeatedly/2`, `apply_repeatedly/3`, `apply_repeatedly/4`,\n`send_interval/2`, and `send_interval/3` is linked to the process to which the\ntimer performs its task.\n\nA one-shot timer, that is, a timer created by evaluating any of the functions\n`apply_after/2`, `apply_after/3`, `apply_after/4`, `send_after/2`,\n`send_after/3`, `exit_after/2`, `exit_after/3`, `kill_after/1`, and\n`kill_after/2` is not linked to any process. Hence, such a timer is removed only\nwhen it reaches its time-out, or if it is explicitly removed by a call to\n`cancel/1`.\n\nThe functions given to `apply_after/2`, `apply_after/3`, `apply_interval/2`,\n`apply_interval/3`, `apply_repeatedly/2`, and `apply_repeatedly/3`, or denoted\nby `Module`, `Function` and `Arguments` given to `apply_after/4`,\n`apply_interval/4`, and `apply_repeatedly/4` are executed in a freshly-spawned\nprocess, and therefore calls to `self/0` in those functions will return the Pid\nof this process, which is different from the process that called\n`timer:apply_*`.\n\n_Example_\n\nIn the following example, the intention is to set a timer to execute a function\nafter 1 second, which performs a fictional task, and then wants to inform the\nprocess which set the timer about its completion, by sending it a `done`\nmessage.\n\nUsing `self/0` _inside_ the timed function, the code below does not work as\nintended. The task gets done, but the `done` message gets sent to the wrong\nprocess and is lost.\n\n```erlang\n1> timer:apply_after(1000, fun() -> do_something(), self() ! done end).\n{ok,TRef}\n2> receive done -> done after 5000 -> timeout end.\n%% ... 5s pass...\ntimeout\n```\n\nThe code below calls `self/0` in the process which sets the timer and assigns it\nto a variable, which is then used in the function to send the `done` message to,\nand so works as intended.\n\n```erlang\n1> Target = self()\n<0.82.0>\n2> timer:apply_after(1000, fun() -> do_something(), Target ! done end).\n{ok,TRef}\n3> receive done -> done after 5000 -> timeout end.\n%% ... 1s passes...\ndone\n```\n\nAnother option is to pass the message target as a parameter to the function.\n\n```erlang\n1> timer:apply_after(1000, fun(Target) -> do_something(), Target ! done end, [self()]).\n{ok,TRef}\n2> receive done -> done after 5000 -> timeout end.\n%% ... 1s passes...\ndone\n```","ref":"timer.html#module-notes"},{"type":"function","title":"timer.apply_after/2","doc":"Evaluates [`spawn(erlang, apply, [Function, []])`](`spawn/3`) after `Time`\nmilliseconds.","ref":"timer.html#apply_after/2"},{"type":"function","title":"timer.apply_after/3","doc":"Evaluates [`spawn(erlang, apply, [Function, Arguments])`](`spawn/3`) after\n`Time` milliseconds.","ref":"timer.html#apply_after/3"},{"type":"function","title":"timer.apply_after/4","doc":"Evaluates [`spawn(Module, Function, Arguments)`](`spawn/3`) after `Time`\nmilliseconds.","ref":"timer.html#apply_after/4"},{"type":"function","title":"timer.apply_interval/2","doc":"Evaluates [`spawn(erlang, apply, [Function, []])`](`spawn/3`) repeatedly at\nintervals of `Time`, irrespective of whether a previously spawned process has\nfinished or not.","ref":"timer.html#apply_interval/2"},{"type":"function","title":"timer.apply_interval/3","doc":"Evaluates [`spawn(erlang, apply, [Function, Arguments])`](`spawn/3`) repeatedly\nat intervals of `Time`, irrespective of whether a previously spawned process has\nfinished or not.","ref":"timer.html#apply_interval/3"},{"type":"function","title":"timer.apply_interval/4","doc":"Evaluates [`spawn(Module, Function, Arguments)`](`spawn/3`) repeatedly at\nintervals of `Time`, irrespective of whether a previously spawned process has\nfinished or not.\n\n> #### Warning {: .warning }\n>\n> If the execution time of the spawned process is, on average, greater than the\n> given `Time`, multiple such processes will run at the same time. With long\n> execution times, short intervals, and many interval timers running, this may\n> even lead to exceeding the number of allowed processes. As an extreme example,\n> consider\n> `[timer:apply_interval(1, timer, sleep, [1000]) || _ <- lists:seq(1, 1000)]`,\n> that is, 1,000 interval timers executing a process that takes 1s to complete,\n> started in intervals of 1ms, which would result in 1,000,000 processes running\n> at the same time, far more than a node started with default settings allows\n> (see the\n> [System Limits section in the Effiency Guide](`e:system:system_limits.md`)).","ref":"timer.html#apply_interval/4"},{"type":"function","title":"timer.apply_repeatedly/2","doc":"Evaluates [`spawn(erlang, apply, [Function, []])`](`spawn/3`) repeatedly at\nintervals of `Time`, waiting for the spawned process to finish before starting\nthe next.","ref":"timer.html#apply_repeatedly/2"},{"type":"function","title":"timer.apply_repeatedly/3","doc":"Evaluates [`spawn(erlang, apply, [Function, Arguments])`](`spawn/3`) repeatedly\nat intervals of `Time`, waiting for the spawned process to finish before\nstarting the next.","ref":"timer.html#apply_repeatedly/3"},{"type":"function","title":"timer.apply_repeatedly/4","doc":"Evaluates [`spawn(Module, Function, Arguments)`](`spawn/3`) repeatedly at\nintervals of `Time`, waiting for the spawned process to finish before starting\nthe next.\n\nIf the execution time of the spawned process is greater than the given `Time`,\nthe next process is spawned immediately after the one currently running has\nfinished. Assuming that execution times of the spawned processes performing the\napplies on average are smaller than `Time`, the amount of applies made over a\nlarge amount of time will be the same even if some individual execution times\nare larger than `Time`. The system will try to catch up as soon as possible. For\nexample, if one apply takes `2.5*Time`, the following two applies will be made\nimmediately one after the other in sequence.","ref":"timer.html#apply_repeatedly/4"},{"type":"function","title":"timer.cancel/1","doc":"Cancels a previously requested time-out. `TRef` is a unique timer reference\nreturned by the related timer function.\n\nReturns `{ok, cancel}`, or `{error, Reason}` when `TRef` is not a timer\nreference.","ref":"timer.html#cancel/1"},{"type":"function","title":"timer.exit_after/2","doc":"","ref":"timer.html#exit_after/2"},{"type":"function","title":"timer.exit_after/3","doc":"Sends an exit signal with reason `Reason1` to `Target`, which can be a local\nprocess identifier or an atom of a registered name.","ref":"timer.html#exit_after/3"},{"type":"function","title":"timer.hms/3","doc":"Returns the number of milliseconds in `Hours + Minutes + Seconds`.","ref":"timer.html#hms/3"},{"type":"function","title":"timer.hours/1","doc":"Returns the number of milliseconds in `Hours`.","ref":"timer.html#hours/1"},{"type":"function","title":"timer.kill_after/1","doc":"","ref":"timer.html#kill_after/1"},{"type":"function","title":"timer.kill_after/2","doc":"","ref":"timer.html#kill_after/2"},{"type":"function","title":"timer.minutes/1","doc":"Returns the number of milliseconds in `Minutes`.","ref":"timer.html#minutes/1"},{"type":"function","title":"timer.now_diff/2","doc":"Calculates the time difference `Tdiff = T2 - T1` in _microseconds_, where `T1`\nand `T2` are time-stamp tuples on the same format as returned from\n`erlang:timestamp/0` or `os:timestamp/0`.","ref":"timer.html#now_diff/2"},{"type":"function","title":"timer.seconds/1","doc":"Returns the number of milliseconds in `Seconds`.","ref":"timer.html#seconds/1"},{"type":"function","title":"timer.send_after/2","doc":"","ref":"timer.html#send_after/2"},{"type":"function","title":"timer.send_after/3","doc":"Evaluates `Destination ! Message` after `Time` milliseconds.\n\n`Destination` can be a remote or local process identifier, an atom of a\nregistered name or a tuple `{RegName, Node}` for a registered name at another node.\n\nSee also [the Timer Module section in the Efficiency Guide](`e:system:commoncaveats.md#timer-module`).","ref":"timer.html#send_after/3"},{"type":"function","title":"timer.send_interval/2","doc":"","ref":"timer.html#send_interval/2"},{"type":"function","title":"timer.send_interval/3","doc":"Evaluates `Destination ! Message` repeatedly after `Time` milliseconds.\n\n`Destination` can be a remote or local process identifier, an atom of a registered\nname or a tuple `{RegName, Node}` for a registered name at another node.","ref":"timer.html#send_interval/3"},{"type":"function","title":"timer.sleep/1","doc":"Suspends the process calling this function for `Time` milliseconds and then\nreturns `ok`, or suspends the process forever if `Time` is the atom `infinity`.\nNaturally, this function does _not_ return immediately.\n\n> #### Note {: .info }\n>\n> Before OTP 25, `timer:sleep/1` did not accept integer timeout values greater\n> than `16#ffffffff`, that is, `2^32-1`. Since OTP 25, arbitrarily high integer\n> values are accepted.","ref":"timer.html#sleep/1"},{"type":"function","title":"timer.start/0","doc":"Starts the timer server.\n\nNormally, the server does not need to be started explicitly. It is started dynamically\nif it is needed. This is useful during development, but in a target system the server\nis to be started explicitly. Use configuration parameters for [Kernel](`e:kernel:index.html`)\nfor this.","ref":"timer.html#start/0"},{"type":"function","title":"timer.tc/1","doc":"","ref":"timer.html#tc/1"},{"type":"function","title":"timer.tc/2","doc":"Measures the execution time of `Fun`.\n\nEquivalent to [`tc(Fun, Arguments, microsecond)`](`tc/3`) if called as `tc(Fun, Arguments)`.\n\nMeasures the execution time of `Fun` in `TimeUnit` if called as `tc(Fun, TimeUnit)`. Added in OTP 26.0.","ref":"timer.html#tc/2"},{"type":"function","title":"timer.tc/3","doc":"Measures the execution time of `Fun` or `apply(Module, Function, Arguments)`.\n\nEquivalent to [`tc(Module, Function, Arguments, microsecond)`](`tc/4`) if called as `tc(Module, Function, Arguments)`.\n\nEquivalent to [`tc(erlang, apply, [Fun, Arguments], TimeUnit)`](`tc/4`) if called as `tc(Fun, Arguments, TimeUnit)`. Added in OTP 26.0","ref":"timer.html#tc/3"},{"type":"function","title":"timer.tc/4","doc":"Evaluates [`apply(Module, Function, Arguments)`](`apply/3`) and measures the elapsed\nreal time as reported by `erlang:monotonic_time/0`.\n\nReturns `{Time, Value}`, where `Time` is the elapsed real time in the\nspecified `TimeUnit`, and `Value` is what is returned from the apply.","ref":"timer.html#tc/4"},{"type":"type","title":"timer.time/0","doc":"Time in milliseconds.","ref":"timer.html#t:time/0"},{"type":"opaque","title":"timer.tref/0","doc":"A timer reference.","ref":"timer.html#t:tref/0"},{"type":"module","title":"argparse","doc":"Command line arguments parser.\n\nThis module implements command line parser. Parser operates with _commands_ and\n_arguments_ represented as a tree. Commands are branches, and arguments are\nleaves of the tree. Parser always starts with the root command, named after\n`progname` (the name of the program which started Erlang).\n\nA [`command specification`](`t:command/0`) may contain handler definition for\neach command, and a number argument specifications. When parser is successful,\n`argparse` calls the matching handler, passing arguments extracted from the\ncommand line. Arguments can be positional (occupying specific position in the\ncommand line), and optional, residing anywhere but prefixed with a specified\ncharacter.\n\n`argparse` automatically generates help and usage messages. It will also issue\nerrors when users give the program invalid arguments.","ref":"argparse.html"},{"type":"module","title":"Quick start - argparse","doc":"`argparse` is designed to work with [`escript`](`e:erts:escript_cmd.md`). The\nexample below is a fully functioning Erlang program accepting two command line\narguments and printing their product.\n\n```erlang\n#!/usr/bin/env escript\n\nmain(Args) ->\n    argparse:run(Args, cli(), #{progname => mul}).\n\ncli() ->\n    #{\n        arguments => [\n            #{name => left, type => integer},\n            #{name => right, type => integer}\n        ],\n        handler =>\n            fun (#{left := Left, right := Right}) ->\n                io:format(\"~b~n\", [Left * Right])\n            end\n    }.\n```\n\nRunning this script with no arguments results in an error, accompanied by the\nusage information.\n\nThe `cli` function defines a single command with embedded handler accepting a\nmap. Keys of the map are argument names as defined by the `argument` field of\nthe command, `left` and `right` in the example. Values are taken from the\ncommand line, and converted into integers, as requested by the type\nspecification. Both arguments in the example above are required (and therefore\ndefined as positional).","ref":"argparse.html#module-quick-start"},{"type":"module","title":"Command hierarchy - argparse","doc":"A command may contain nested commands, forming a hierarchy. Arguments defined at\nthe upper level command are automatically added to all nested commands. Nested\ncommands example (assuming `progname` is `nested`):\n\n```erlang\ncli() ->\n  #{\n    %% top level argument applicable to all commands\n    arguments => [#{name => top}],\n      commands => #{\n        \"first\" => #{\n          %% argument applicable to \"first\" command and\n          %%  all commands nested into \"first\"\n          arguments => [#{name => mid}],\n          commands => #{\n            \"second\" => #{\n              %% argument only applicable for \"second\" command\n              arguments => [#{name => bottom}],\n              handler => fun (A) -> io:format(\"~p~n\", [A]) end\n          }\n        }\n      }\n    }\n  }.\n```\n\nIn the example above, a 3-level hierarchy is defined. First is the script itself\n(`nested`), accepting the only argument `top`. Since it has no associated\nhandler, `run/3` will not accept user input omitting nested command selection.\nFor this example, user has to supply 5 arguments in the command line, two being\ncommand names, and another 3 - required positional arguments:\n\n```text\n./nested.erl one first second two three\n#{top => \"one\",mid => \"two\",bottom => \"three\"}\n```\n\nCommands have preference over positional argument values. In the example above,\ncommands and positional arguments are interleaving, and `argparse` matches\ncommand name first.","ref":"argparse.html#module-command-hierarchy"},{"type":"module","title":"Arguments - argparse","doc":"`argparse` supports positional and optional arguments. Optional arguments, or\noptions for short, must be prefixed with a special character (`-` is the default\non all operating systems). Both options and positional arguments have 1 or more\nassociated values. See [`argument specification`](`t:argument/0`) to find more\ndetails about supported combinations.\n\nIn the user input, short options may be concatenated with their values. Long\noptions support values separated by `=`. Consider this definition:\n\n```erlang\ncli() ->\n  #{\n    arguments => [\n      #{name => long, long => \"-long\"},\n      #{name => short, short => $s}\n    ],\n    handler => fun (Args) -> io:format(\"~p~n\", [Args]) end\n  }.\n```\n\nRunning `./args --long=VALUE` prints `#{long => \"VALUE\"}`, running\n`./args -sVALUE` prints `#{short => \"VALUE\"}`\n\n`argparse` supports boolean flags concatenation: it is possible to shorten\n`-r -f -v` to `-rfv`.\n\nShortened option names are not supported: it is not possible to use `--my-argum`\ninstead of `--my-argument-name` even when such option can be unambiguously\nfound.","ref":"argparse.html#module-arguments"},{"type":"type","title":"argparse.arg_map/0","doc":"Arguments map is the map of argument names to the values extracted from the\ncommand line. It is passed to the matching command handler. If an argument is\nomitted, but has the default value is specified, it is added to the map. When no\ndefault value specified, and argument is not present in the command line,\ncorresponding key is not present in the resulting map.","ref":"argparse.html#t:arg_map/0"},{"type":"type","title":"argparse.arg_type/0","doc":"Defines type conversion applied to the string retrieved from the user input. If\nthe conversion is successful, resulting value is validated using optional\n`Choices`, or minimums and maximums (for integer and floating point values\nonly). Strings and binary values may be validated using regular expressions.\nIt's possible to define custom type conversion function, accepting a string and\nreturning Erlang term. If this function raises error with `badarg` reason,\nargument is treated as invalid.","ref":"argparse.html#t:arg_type/0"},{"type":"type","title":"argparse.argument/0","doc":"Argument specification. Defines a single named argument that is returned in the\n[`argument map`](`t:arg_map/0`). The only required field is `name`, all other\nfields have defaults.\n\nIf either of the `short` or `long` fields is specified, the argument is treated\nas optional. Optional arguments do not have specific order and may appear\nanywhere in the command line. Positional arguments are ordered the same way as\nthey appear in the arguments list of the command specification.\n\nBy default, all positional arguments must be present in the command line. The\nparser will return an error otherwise. Options, however, may be omitted, in\nwhich case resulting argument map will either contain the default value, or not\nhave the key at all.\n\n- **`name`** - Sets the argument name in the parsed argument map. If `help` is\n  not defined, name is also used to generate the default usage message.\n\n- **`short`** - Defines a short (single character) form of an optional argument.\n\n  ```erlang\n  %% Define a command accepting argument named myarg, with short form $a:\n  1> Cmd = #{arguments => [#{name => myarg, short => $a}]}.\n  %% Parse command line \"-a str\":\n  2> {ok, ArgMap, _, _} = argparse:parse([\"-a\", \"str\"], Cmd), ArgMap.\n\n  #{myarg => \"str\"}\n\n  %% Option value can be concatenated with the switch: \"-astr\"\n  3> {ok, ArgMap, _, _} = argparse:parse([\"-astr\"], Cmd), ArgMap.\n\n  #{myarg => \"str\"}\n  ```\n\n  By default all options expect a single value following the option switch. The\n  only exception is an option of a boolean type.\n\n- **`long`** - Defines a long form of an optional argument.\n\n  ```erlang\n  1> Cmd = #{arguments => [#{name => myarg, long => \"name\"}]}.\n  %% Parse command line \"-name Erlang\":\n  2> {ok, ArgMap, _, _} = argparse:parse([\"-name\", \"Erlang\"], Cmd), ArgMap.\n\n  #{myarg => \"Erlang\"}\n  %% Or use \"=\" to separate the switch and the value:\n  3> {ok, ArgMap, _, _} = argparse:parse([\"-name=Erlang\"], Cmd), ArgMap.\n\n  #{myarg => \"Erlang\"}\n  ```\n\n  If neither `short` not `long` is defined, the argument is treated as\n  positional.\n\n- **`required`** - Forces the parser to expect the argument to be present in the\n  command line. By default, all positional argument are required, and all\n  options are not.\n\n- **`default`** - Specifies the default value to put in the parsed argument map\n  if the value is not supplied in the command line.\n\n  ```erlang\n  1> argparse:parse([], #{arguments => [#{name => myarg, short => $m}]}).\n\n  {ok,#{}, ...\n  2> argparse:parse([], #{arguments => [#{name => myarg, short => $m, default => \"def\"}]}).\n\n  {ok,#{myarg => \"def\"}, ...\n  ```\n\n- **`type`** - Defines type conversion and validation routine. The default is\n  `string`, assuming no conversion.\n\n- **`nargs`** - Defines the number of following arguments to consume from the\n  command line. By default, the parser consumes the next argument and converts\n  it into an Erlang term according to the specified type.\n\n  - **`t:pos_integer/0`** - Consume exactly this number of positional arguments,\n    fail if there is not enough. Value in the argument map contains a list of\n    exactly this length. Example, defining a positional argument expecting 3\n    integer values:\n\n    ```erlang\n    1> Cmd = #{arguments => [#{name => ints, type => integer, nargs => 3}]},\n    argparse:parse([\"1\", \"2\", \"3\"], Cmd).\n\n    {ok, #{ints => [1, 2, 3]}, ...\n    ```\n\n    Another example defining an option accepted as `-env` and expecting two\n    string arguments:\n\n    ```erlang\n    1> Cmd = #{arguments => [#{name => env, long => \"env\", nargs => 2}]},\n    argparse:parse([\"-env\", \"key\", \"value\"], Cmd).\n\n    {ok, #{env => [\"key\", \"value\"]}, ...\n    ```\n\n  - **`list`** - Consume all following arguments until hitting the next option\n    (starting with an option prefix). May result in an empty list added to the\n    arguments map.\n\n    ```erlang\n    1> Cmd = #{arguments => [\n      #{name => nodes, long => \"nodes\", nargs => list},\n      #{name => verbose, short => $v, type => boolean}\n    ]},\n    argparse:parse([\"-nodes\", \"one\", \"two\", \"-v\"], Cmd).\n\n    {ok, #{nodes => [\"one\", \"two\"], verbose => true}, ...\n    ```\n\n  - **`nonempty_list`** - Same as `list`, but expects at least one argument.\n    Returns an error if the following command line argument is an option switch\n    (starting with the prefix).\n\n  - **`'maybe'`** - Consumes the next argument from the command line, if it does\n    not start with an option prefix. Otherwise, adds a default value to the\n    arguments map.\n\n    ```erlang\n    1> Cmd = #{arguments => [\n      #{name => level, short => $l, nargs => 'maybe', default => \"error\"},\n      #{name => verbose, short => $v, type => boolean}\n    ]},\n    argparse:parse([\"-l\", \"info\", \"-v\"], Cmd).\n\n    {ok,#{level => \"info\",verbose => true}, ...\n\n    %% When \"info\" is omitted, argument maps receives the default \"error\"\n    2> argparse:parse([\"-l\", \"-v\"], Cmd).\n\n    {ok,#{level => \"error\",verbose => true}, ...\n    ```\n\n  - **`{'maybe', term()}`** - Consumes the next argument from the command line,\n    if it does not start with an option prefix. Otherwise, adds a specified\n    Erlang term to the arguments map.\n\n  - **`all`** - Fold all remaining command line arguments into a list, ignoring\n    any option prefixes or switches. Useful for proxying arguments into another\n    command line utility.\n\n    ```erlang\n    1> Cmd = #{arguments => [\n        #{name => verbose, short => $v, type => boolean},\n        #{name => raw, long => \"-\", nargs => all}\n    ]},\n    argparse:parse([\"-v\", \"--\", \"-kernel\", \"arg\", \"opt\"], Cmd).\n\n    {ok,#{raw => [\"-kernel\",\"arg\",\"opt\"],verbose => true}, ...\n    ```\n\n- **`action`** - Defines an action to take when the argument is found in the\n  command line. The default action is `store`.\n\n  - **`store`** - Store the value in the arguments map. Overwrites the value\n    previously written.\n\n    ```erlang\n    1> Cmd = #{arguments => [#{name => str, short => $s}]},\n    argparse:parse([\"-s\", \"one\", \"-s\", \"two\"], Cmd).\n\n    {ok, #{str => \"two\"}, ...\n    ```\n\n  - **`{store, term()}`** - Stores the specified term instead of reading the\n    value from the command line.\n\n    ```erlang\n    1> Cmd = #{arguments => [#{name => str, short => $s, action => {store, \"two\"}}]},\n    argparse:parse([\"-s\"], Cmd).\n\n    {ok, #{str => \"two\"}, ...\n    ```\n\n  - **`append`** - Appends the repeating occurrences of the argument instead of\n    overwriting.\n\n    ```erlang\n    1> Cmd = #{arguments => [#{name => node, short => $n, action => append}]},\n    argparse:parse([\"-n\", \"one\", \"-n\", \"two\", \"-n\", \"three\"], Cmd).\n\n    {ok, #{node => [\"one\", \"two\", \"three\"]}, ...\n\n    %% Always produces a list - even if there is one occurrence\n    2> argparse:parse([\"-n\", \"one\"], Cmd).\n\n    {ok, #{node => [\"one\"]}, ...\n    ```\n\n  - **`{append, term()}`** - Same as `append`, but instead of consuming the\n    argument from the command line, appends a provided `t:term/0`.\n\n  - **`count`** - Puts a counter as a value in the arguments map. Useful for\n    implementing verbosity option:\n\n    ```erlang\n    1> Cmd = #{arguments => [#{name => verbose, short => $v, action => count}]},\n    argparse:parse([\"-v\"], Cmd).\n\n    {ok, #{verbose => 1}, ...\n\n    2> argparse:parse([\"-vvvv\"], Cmd).\n\n    {ok, #{verbose => 4}, ...\n    ```\n\n  - **`extend`** - Works as `append`, but flattens the resulting list. Valid\n    only for `nargs` set to `list`, `nonempty_list`, `all` or `t:pos_integer/0`.\n\n    ```erlang\n    1> Cmd = #{arguments => [#{name => duet, short => $d, nargs => 2, action => extend}]},\n    argparse:parse([\"-d\", \"a\", \"b\", \"-d\", \"c\", \"d\"], Cmd).\n\n    {ok, #{duet => [\"a\", \"b\", \"c\", \"d\"]}, ...\n\n    %% 'append' would result in {ok, #{duet => [[\"a\", \"b\"],[\"c\", \"d\"]]},\n    ```\n\n- **`help`** - Specifies help/usage text for the argument. `argparse` provides\n  automatic generation based on the argument name, type and default value, but\n  for better usability it is recommended to have a proper description. Setting\n  this field to `hidden` suppresses usage output for this argument.","ref":"argparse.html#t:argument/0"},{"type":"type","title":"argparse.argument_help/0","doc":"User-defined help template to print in the command usage. First element of a\ntuple must be a string. It is printed as a part of the usage header. Second\nelement of the tuple can be either a list containing strings, `type` and\n`default` atoms, or a user-defined function that must return a string. A plain\nstring should be wrapped as a list such as `[\"string is nested\"]`.","ref":"argparse.html#t:argument_help/0"},{"type":"type","title":"argparse.argument_name/0","doc":"Argument name is used to populate argument map.","ref":"argparse.html#t:argument_name/0"},{"type":"type","title":"argparse.cmd_path/0","doc":"Path to the nested command. First element is always the `progname`, subsequent\nelements are nested command names.","ref":"argparse.html#t:cmd_path/0"},{"type":"type","title":"argparse.command/0","doc":"Command specification. May contain nested commands, forming a hierarchy.\n\n- **`commands`** - Maps of nested commands. Keys must be strings, matching\n  command line input. Basic utilities do not need to specify any nested\n  commands.\n\n- **`arguments`** - List of arguments accepted by this command, and all nested\n  commands in the hierarchy.\n\n- **`help`** - Specifies help/usage text for this command. Pass `hidden` to\n  remove this command from the usage output.\n\n- **`handler`** - Specifies a callback function to call by `run/3` when the\n  parser is successful.","ref":"argparse.html#t:command/0"},{"type":"type","title":"argparse.command_help/0","doc":"User-defined help template. Use this option to mix custom and predefined usage\ntext. Help template may contain unicode strings, and following atoms:\n\n- **usage** - Formatted command line usage text, e.g. `rm [-rf]  `.\n\n- **commands** - Expanded list of sub-commands.\n\n- **arguments** - Detailed description of positional arguments.\n\n- **options** - Detailed description of optional arguments.","ref":"argparse.html#t:command_help/0"},{"type":"function","title":"argparse.format_error/1","doc":"Generates human-readable text for [`parser error`](`t:parser_error/0`). Does not\ninclude help/usage information, and does not provide localisation.","ref":"argparse.html#format_error/1"},{"type":"type","title":"argparse.handler/0","doc":"Command handler specification. Called by [`run/3` ](`run/3`)upon successful\nparser return.\n\n- **`fun((arg_map()) -> term())`** - Function accepting\n  [`argument map`](`t:arg_map/0`). See the basic example in the\n  [Quick Start](`m:argparse#module-quick-start`) section.\n\n- **`{Module :: module(), Function :: atom()}`** - Function named `Function`,\n  exported from `Module`, accepting [`argument map`](`t:arg_map/0`).\n\n- **`{fun(() -> term()), Default :: term()}`** - Function accepting as many\n  arguments as there are in the `arguments` list for this command. Arguments\n  missing from the parsed map are replaced with the `Default`. Convenient way to\n  expose existing functions.\n\n  ```erlang\n  1> Cmd = #{arguments => [\n          #{name => x, type => float},\n          #{name => y, type => float, short => $p}],\n      handler => {fun math:pow/2, 1}},\n  argparse:run([\"2\", \"-p\", \"3\"], Cmd, #{}).\n\n  8.0\n\n  %% default term 1 is passed to math:pow/2\n  2> argparse:run([\"2\"], Cmd, #{}).\n\n  2.0\n  ```\n\n- **`{Module :: module(), Function :: atom(), Default :: term()}`** - Function\n  named `Function`, exported from `Module`, accepting as many arguments as\n  defined for this command. Arguments missing from the parsed map are replaced\n  with the `Default`. Effectively, just a different syntax to the same\n  functionality as demonstrated in the code above.","ref":"argparse.html#t:handler/0"},{"type":"function","title":"argparse.help/1","doc":"","ref":"argparse.html#help/1"},{"type":"function","title":"argparse.help/2","doc":"Generates help/usage information text for the command supplied, or any nested\ncommand when `command` option is specified. Arguments are displayed in the same\norder as specified in `Command`. Does not provide localisation. Expects\n`progname` to be set, otherwise defaults to return value of\n`init:get_argument(progname)`.","ref":"argparse.html#help/2"},{"type":"function","title":"argparse.parse/2","doc":"","ref":"argparse.html#parse/2"},{"type":"function","title":"argparse.parse/3","doc":"Parses command line arguments according to the command specification. Raises an\nexception if the command specification is not valid. Use\n[`erl_error:format_exception/3,4` ](`erl_error:format_exception/3`)to see a\nfriendlier message. Invalid command line input does not raise an exception, but\nmakes `parse/2,3` to return a tuple\n[`{error, parser_error()}`](`t:parser_error/0`).\n\nThis function does not call command handler.","ref":"argparse.html#parse/3"},{"type":"type","title":"argparse.parse_result/0","doc":"Returned from [`parse/2,3`](`parse/3`). Contains arguments extracted from the\ncommand line, path to the nested command (if any), and a (potentially nested)\ncommand specification that was considered when the parser finished successfully.\nIt is expected that the command contains a handler definition, that will be\ncalled passing the argument map.","ref":"argparse.html#t:parse_result/0"},{"type":"type","title":"argparse.parser_error/0","doc":"Returned from [`parse/2,3`](`parse/3`) when the user input cannot be parsed\naccording to the command specification.\n\nFirst element is the path to the command that was considered when the parser\ndetected an error. Second element, `Expected`, is the argument specification\nthat caused an error. It could be `undefined`, meaning that `Actual` argument\nhad no corresponding specification in the arguments list for the current\ncommand.\n\nWhen `Actual` is set to `undefined`, it means that a required argument is\nmissing from the command line. If both `Expected` and `Actual` have values, it\nmeans validation error.\n\nUse `format_error/1` to generate a human-readable error description, unless\nthere is a need to provide localised error messages.","ref":"argparse.html#t:parser_error/0"},{"type":"type","title":"argparse.parser_options/0","doc":"Options changing parser behaviour.\n\n- **`prefixes`** - Changes the option prefix (the default is `-`).\n\n- **`default`** - Specifies the default value for all optional arguments. When\n  this field is set, resulting argument map will contain all argument names.\n  Useful for easy pattern matching on the argument map in the handler function.\n\n- **`progname`** - Specifies the program (root command) name. Returned as the\n  first element of the command path, and printed in help/usage text. It is\n  recommended to have this value set, otherwise the default one is determined\n  with `init:get_argument(progname)` and is often set to `erl` instead of the\n  actual script name.\n\n- **`command`** - Specifies the path to the nested command for `help/2`. Useful\n  to limit output for complex utilities with multiple commands, and used by the\n  default error handling logic.\n\n- **`columns`** - Specifies the help/usage text width (characters) for `help/2`.\n  Default value is 80.","ref":"argparse.html#t:parser_options/0"},{"type":"function","title":"argparse.run/3","doc":"Parses command line arguments and calls the matching command handler. Prints\nhuman-readable error, help/usage information for the discovered command, and\nhalts the emulator with code 1 if there is any error in the command\nspecification or user-provided command line input.\n\n> #### Warning {: .warning }\n>\n> This function is designed to work as an entry point to a standalone\n> [`escript`](`e:erts:escript_cmd.md`). Therefore, it halts the emulator for any\n> error detected. Do not use this function through remote procedure call, or it\n> may result in an unexpected shutdown of a remote node.","ref":"argparse.html#run/3"},{"type":"module","title":"escript","doc":"This module provides functions to create and inspect escripts.\n\nSee the [escript](`e:erts:escript_cmd.md`) program documentation\nfor more details on how to use escripts.","ref":"escript.html"},{"type":"type","title":"escript.comment/0","doc":"","ref":"escript.html#t:comment/0"},{"type":"function","title":"escript.create/2","doc":"Creates an escript from a list of sections.\n\nThe sections can be specified in any order. An escript begins with an optional\n`Header` followed by a mandatory `Body`. If the header is present, it does always\n begin with a `shebang`, possibly followed by a `comment` and `emu_args`. The\n`shebang` defaults to `\"/usr/bin/env escript\"`. The `comment` defaults to\n`\"This is an -*- erlang -*- file\"`. The created escript can either be returned\nas a binary or written to file.\n\nAs an example of how the function can be used, we create an interpreted escript\nthat uses `emu_args` to set some emulator flag. In this case, it happens to set\nnumber of schedulers with `+S3`. We also extract the different sections from the\nnewly created script:\n\n```erlang\n> Source = \"%% Demo\\nmain(_Args) ->\\n    io:format(\\\"~p\\\",[erlang:system_info(schedulers)]).\\n\".\n\"%% Demo\\nmain(_Args) ->\\n    io:format(erlang:system_info(schedulers)).\\n\"\n> io:format(\"~s\\n\", [Source]).\n%% Demo\nmain(_Args) ->\n    io:format(erlang:system_info(schedulers)).\n\nok\n> {ok, Bin} = escript:create(binary, [shebang, comment, {emu_args, \"+S3\"},\n                                      {source, list_to_binary(Source)}]).\n{ok,<<\"#!/usr/bin/env escript\\n%% This is an -*- erlang -*- file\\n%%!+S3\"...>>}\n> file:write_file(\"demo.escript\", Bin).\nok\n> os:cmd(\"escript demo.escript\").\n\"3\"\n> escript:extract(\"demo.escript\", []).\n{ok,[{shebang,default}, {comment,default}, {emu_args,\"+S3\"},\n     {source,<<\"%% Demo\\nmain(_Args) ->\\n    io:format(erlang:system_info(schedu\"...>>}]}\n```\n\nAn escript without header can be created as follows:\n\n```erlang\n> file:write_file(\"demo.erl\",\n                  [\"%% demo.erl\\n-module(demo).\\n-export([main/1]).\\n\\n\", Source]).\nok\n> {ok, _, BeamCode} = compile:file(\"demo.erl\", [binary, debug_info]).\n{ok,demo,\n    <<70,79,82,49,0,0,2,208,66,69,65,77,65,116,111,109,0,0,0,\n      79,0,0,0,9,4,100,...>>}\n> escript:create(\"demo.beam\", [{beam, BeamCode}]).\nok\n> escript:extract(\"demo.beam\", []).\n{ok,[{shebang,undefined}, {comment,undefined}, {emu_args,undefined},\n     {beam,<<70,79,82,49,0,0,3,68,66,69,65,77,65,116,\n             111,109,0,0,0,83,0,0,0,9,...>>}]}\n> os:cmd(\"escript demo.beam\").\n\"true\"\n```\n\nHere we create an archive script containing both Erlang code and Beam code, then\nwe iterate over all files in the archive and collect their contents and some\ninformation about them:\n\n```erlang\n> {ok, SourceCode} = file:read_file(\"demo.erl\").\n{ok,<<\"%% demo.erl\\n-module(demo).\\n-export([main/1]).\\n\\n%% Demo\\nmain(_Arg\"...>>}\n> escript:create(\"demo.escript\",\n                 [shebang,\n                  {archive, [{\"demo.erl\", SourceCode},\n                             {\"demo.beam\", BeamCode}], []}]).\nok\n> {ok, [{shebang,default}, {comment,undefined}, {emu_args,undefined},\n     {archive, ArchiveBin}]} = escript:extract(\"demo.escript\", []).\n{ok,[{shebang,default}, {comment,undefined}, {emu_args,undefined},\n     {{archive,<<80,75,3,4,20,0,0,0,8,0,118,7,98,60,105,\n                152,61,93,107,0,0,0,118,0,...>>}]}\n> file:write_file(\"demo.zip\", ArchiveBin).\nok\n> zip:foldl(fun(N, I, B, A) -> [{N, I(), B()} | A] end, [], \"demo.zip\").\n{ok,[{\"demo.beam\",\n      {file_info,748,regular,read_write,\n                 {{2010,3,2},{0,59,22}},\n                 {{2010,3,2},{0,59,22}},\n                 {{2010,3,2},{0,59,22}},\n                 54,1,0,0,0,0,0},\n      <<70,79,82,49,0,0,2,228,66,69,65,77,65,116,111,109,0,0,0,\n        83,0,0,...>>},\n     {\"demo.erl\",\n      {file_info,118,regular,read_write,\n                 {{2010,3,2},{0,59,22}},\n                 {{2010,3,2},{0,59,22}},\n                 {{2010,3,2},{0,59,22}},\n                 54,1,0,0,0,0,0},\n      <<\"%% demo.erl\\n-module(demo).\\n-export([main/1]).\\n\\n%% Demo\\nmain(_Arg\"...>>}]}\n```","ref":"escript.html#create/2"},{"type":"type","title":"escript.emu_args/0","doc":"Any arguments that should be passed to [erl](`e:erts:erl_cmd.md`) when starting.","ref":"escript.html#t:emu_args/0"},{"type":"function","title":"escript.extract/2","doc":"Parses an escript and extracts its sections. This is the reverse of `create/2`.\n\nAll sections are returned even if they do not exist in the escript. If a\nparticular section happens to have the same value as the default value, the\nextracted value is set to the atom `default`. If a section is missing, the\nextracted value is set to the atom `undefined`.\n\nOption `compile_source` only affects the result if the escript contains `source`\ncode. In this case the Erlang code is automatically compiled and\n`{source, BeamCode}` is returned instead of `{source, SourceCode}`.\n\nExample:\n\n```erlang\n> escript:create(\"demo.escript\",\n                 [shebang, {archive, [{\"demo.erl\", SourceCode},\n                                      {\"demo.beam\", BeamCode}], []}]).\nok\n> {ok, [{shebang,default}, {comment,undefined}, {emu_args,undefined},\n     {archive, ArchiveBin}]} =\n              escript:extract(\"demo.escript\", []).\n{ok,[{{archive,<<80,75,3,4,20,0,0,0,8,0,118,7,98,60,105,\n                152,61,93,107,0,0,0,118,0,...>>}\n     {emu_args,undefined}]}\n```","ref":"escript.html#extract/2"},{"type":"type","title":"escript.extract_option/0","doc":"","ref":"escript.html#t:extract_option/0"},{"type":"function","title":"escript.script_name/0","doc":"Returns the name of the escript that is executed.\n\nIf the function is invoked outside the context of an escript,\nthe behavior is undefined.","ref":"escript.html#script_name/0"},{"type":"type","title":"escript.section/0","doc":"","ref":"escript.html#t:section/0"},{"type":"type","title":"escript.section_name/0","doc":"","ref":"escript.html#t:section_name/0"},{"type":"type","title":"escript.shebang/0","doc":"The initial `#!` line.\n\nFor example:\n\n```text\n#!/usr/bin/env escript\n```","ref":"escript.html#t:shebang/0"},{"type":"type","title":"escript.zip_file/0","doc":"","ref":"escript.html#t:zip_file/0"},{"type":"module","title":"peer","doc":"Start and control linked Erlang nodes.\n\nThis module provides functions for starting linked Erlang nodes. The node\nspawning new nodes is called _origin_, and newly started nodes are _peer_ nodes,\nor peers. A peer node automatically terminates when it loses the _control\nconnection_ to the origin. This connection could be an Erlang distribution\nconnection, or an alternative - TCP or standard I/O. The alternative connection\nprovides a way to execute remote procedure calls even when Erlang Distribution\nis not available, allowing to test the distribution itself.\n\nPeer node terminal input/output is relayed through the origin. If a standard I/O\nalternative connection is requested, console output also goes via the origin,\nallowing debugging of node startup and boot script execution (see\n[`-init_debug`](`e:erts:erl_cmd.md#init_debug`)). File I/O is not redirected,\ncontrary to `m:slave` behaviour.\n\nThe peer node can start on the same or a different host (via `ssh`) or in a\nseparate container (for example Docker). When the peer starts on the same host\nas the origin, it inherits the current directory and environment variables from\nthe origin.\n\n> #### Note {: .info }\n>\n> This module is designed to facilitate multi-node testing with Common Test. Use\n> the `?CT_PEER()` macro to start a linked peer node according to Common Test\n> conventions: crash dumps written to specific location, node name prefixed with\n> module name, calling function, and origin OS process ID). Use `random_name/1`\n> to create sufficiently unique node names if you need more control.\n>\n> A peer node started without alternative connection behaves similarly to\n> `m:slave`. When an alternative connection is requested, the behaviour is\n> similar to `test_server:start_node(Name, peer, Args).`","ref":"peer.html"},{"type":"module","title":"Example - peer","doc":"The following example implements a test suite starting extra Erlang nodes. It\nemploys a number of techniques to speed up testing and reliably shut down peer\nnodes:\n\n- peers start linked to test runner process. If the test case fails, the peer\n  node is stopped automatically, leaving no rogue nodes running in the\n  background\n- arguments used to start the peer are saved in the control process state for\n  manual analysis. If the test case fails, the CRASH REPORT contains these\n  arguments\n- multiple test cases can run concurrently speeding up overall testing process,\n  peer node names are unique even when there are multiple instances of the same\n  test suite running in parallel\n\n```erlang\n-module(my_SUITE).\n-behaviour(ct_suite).\n-export([all/0, groups/0]).\n-export([basic/1, args/1, named/1, restart_node/1, multi_node/1]).\n\n-include_lib(\"common_test/include/ct.hrl\").\n\ngroups() ->\n    [{quick, [parallel],\n        [basic, args, named, restart_node, multi_node]}].\n\nall() ->\n    [{group, quick}].\n\nbasic(Config) when is_list(Config) ->\n    {ok, Peer, _Node} = ?CT_PEER(),\n    peer:stop(Peer).\n\nargs(Config) when is_list(Config) ->\n    %% specify additional arguments to the new node\n    {ok, Peer, _Node} = ?CT_PEER([\"-emu_flavor\", \"smp\"]),\n    peer:stop(Peer).\n\nnamed(Config) when is_list(Config) ->\n    %% pass test case name down to function starting nodes\n    Peer = start_node_impl(named_test),\n    peer:stop(Peer).\n\nstart_node_impl(ActualTestCase) ->\n    {ok, Peer, Node} = ?CT_PEER(#{name => ?CT_PEER_NAME(ActualTestCase)}),\n    %% extra setup needed for multiple test cases\n    ok = rpc:call(Node, application, set_env, [kernel, key, value]),\n    Peer.\n\nrestart_node(Config) when is_list(Config) ->\n    Name = ?CT_PEER_NAME(),\n    {ok, Peer, Node} = ?CT_PEER(#{name => Name}),\n    peer:stop(Peer),\n    %% restart the node with the same name as before\n    {ok, Peer2, Node} = ?CT_PEER(#{name => Name, args => [\"+fnl\"]}),\n    peer:stop(Peer2).\n```\n\nThe next example demonstrates how to start multiple nodes concurrently:\n\n```erlang\nmulti_node(Config) when is_list(Config) ->\n    Peers = [?CT_PEER(#{wait_boot => {self(), tag}})\n        || _ <- lists:seq(1, 4)],\n    %% wait for all nodes to complete boot process, get their names:\n    _Nodes = [receive {tag, {started, Node, Peer}} -> Node end\n        || {ok, Peer} <- Peers],\n    [peer:stop(Peer) || {ok, Peer} <- Peers].\n```\n\nStart a peer on a different host. Requires `ssh` key-based authentication set\nup, allowing \"another_host\" connection without password prompt.\n\n```erlang\nSsh = os:find_executable(\"ssh\"),\npeer:start_link(#{exec => {Ssh, [\"another_host\", \"erl\"]},\n    connection => standard_io}),\n```\n\nThe following Common Test case demonstrates Docker integration, starting two\ncontainers with hostnames \"one\" and \"two\". In this example Erlang nodes running\ninside containers form an Erlang cluster.\n\n```erlang\ndocker(Config) when is_list(Config) ->\n    Docker = os:find_executable(\"docker\"),\n    PrivDir = proplists:get_value(priv_dir, Config),\n    build_release(PrivDir),\n    build_image(PrivDir),\n\n    %% start two Docker containers\n    {ok, Peer, Node} = peer:start_link(#{name => lambda,\n        connection => standard_io,\n        exec => {Docker, [\"run\", \"-h\", \"one\", \"-i\", \"lambda\"]}}),\n    {ok, Peer2, Node2} = peer:start_link(#{name => lambda,\n        connection => standard_io,\n        exec => {Docker, [\"run\", \"-h\", \"two\", \"-i\", \"lambda\"]}}),\n\n    %% find IP address of the second node using alternative connection RPC\n    {ok, Ips} = peer:call(Peer2, inet, getifaddrs, []),\n    {\"eth0\", Eth0} = lists:keyfind(\"eth0\", 1, Ips),\n    {addr, Ip} = lists:keyfind(addr, 1, Eth0),\n\n    %% make first node to discover second one\n    ok = peer:call(Peer, inet_db, set_lookup, [[file]]),\n    ok = peer:call(Peer, inet_db, add_host, [Ip, [\"two\"]]),\n\n    %% join a cluster\n    true = peer:call(Peer, net_kernel, connect_node, [Node2]),\n    %% verify that second peer node has only the first node visible\n    [Node] = peer:call(Peer2, erlang, nodes, []),\n\n    %% stop peers, causing containers to also stop\n    peer:stop(Peer2),\n    peer:stop(Peer).\n\nbuild_release(Dir) ->\n    %% load sasl.app file, otherwise application:get_key will fail\n    application:load(sasl),\n    %% create *.rel - release file\n    RelFile = filename:join(Dir, \"lambda.rel\"),\n    Release = {release, {\"lambda\", \"1.0.0\"},\n        {erts, erlang:system_info(version)},\n        [{App, begin {ok, Vsn} = application:get_key(App, vsn), Vsn end}\n            || App <- [kernel, stdlib, sasl]]},\n    ok = file:write_file(RelFile, list_to_binary(lists:flatten(\n        io_lib:format(\"~tp.\", [Release])))),\n    RelFileNoExt = filename:join(Dir, \"lambda\"),\n\n    %% create boot script\n    {ok, systools_make, []} = systools:make_script(RelFileNoExt,\n        [silent, {outdir, Dir}]),\n    %% package release into *.tar.gz\n    ok = systools:make_tar(RelFileNoExt, [{erts, code:root_dir()}]).\n\nbuild_image(Dir) ->\n    %% Create Dockerfile example, working only for Ubuntu 20.04\n    %% Expose port 4445, and make Erlang distribution to listen\n    %%  on this port, and connect to it without EPMD\n    %% Set cookie on both nodes to be the same.\n    BuildScript = filename:join(Dir, \"Dockerfile\"),\n    Dockerfile =\n      \"FROM ubuntu:20.04 as runner\\n\"\n      \"EXPOSE 4445\\n\"\n      \"WORKDIR /opt/lambda\\n\"\n      \"COPY lambda.tar.gz /tmp\\n\"\n      \"RUN tar -zxvf /tmp/lambda.tar.gz -C /opt/lambda\\n\"\n      \"ENTRYPOINT [\\\"/opt/lambda/erts-\" ++ erlang:system_info(version) ++\n      \"/bin/dyn_erl\\\", \\\"-boot\\\", \\\"/opt/lambda/releases/1.0.0/start\\\",\"\n      \" \\\"-kernel\\\", \\\"inet_dist_listen_min\\\", \\\"4445\\\",\"\n      \" \\\"-erl_epmd_port\\\", \\\"4445\\\",\"\n      \" \\\"-setcookie\\\", \\\"secret\\\"]\\n\",\n    ok = file:write_file(BuildScript, Dockerfile),\n    os:cmd(\"docker build -t lambda \" ++ Dir).\n```","ref":"peer.html#module-example"},{"type":"function","title":"peer.call/4","doc":"","ref":"peer.html#call/4"},{"type":"function","title":"peer.call/5","doc":"Uses the alternative connection to evaluate\n[`apply(Module, Function, Args)`](`apply/3`) on the peer node and returns the\ncorresponding value `Result`.\n\n`Timeout` is an integer representing the timeout in milliseconds or the atom\n`infinity` which prevents the operation from ever timing out.\n\nWhen an alternative connection is not requested, this function will raise `exit`\nsignal with the `noconnection` reason. Use `m:erpc` module to communicate over\nErlang distribution.","ref":"peer.html#call/5"},{"type":"function","title":"peer.cast/4","doc":"Uses the alternative connection to evaluate\n[`apply(Module, Function, Args)`](`apply/3`) on the peer node. No response is\ndelivered to the calling process.\n\n`peer:cast/4` fails silently when the alternative connection is not configured.\nUse `m:erpc` module to communicate over Erlang distribution.","ref":"peer.html#cast/4"},{"type":"type","title":"peer.connection/0","doc":"Alternative connection between the origin and the peer. When the connection\ncloses, the peer node terminates automatically.\n\nIf the `peer_down` startup flag is set to `crash`, the controlling process on\nthe origin node exits with corresponding reason, effectively providing a two-way link.\n\nWhen `connection` is set to a port number, the origin starts listening on the\nrequested TCP port, and the peer node connects to the port. When it is set to an\n`{IP, Port}` tuple, the origin listens only on the specified IP. The port number\ncan be set to 0 for automatic selection.\n\nUsing the `standard_io` alternative connection starts the peer attached to the\norigin (other connections use `-detached` flag to erl). In this mode peer and\norigin communicate via stdin/stdout.","ref":"peer.html#t:connection/0"},{"type":"type","title":"peer.disconnect_timeout/0","doc":"Disconnect timeout. See [`stop()`](`stop/1`).","ref":"peer.html#t:disconnect_timeout/0"},{"type":"type","title":"peer.exec/0","doc":"Overrides executable to start peer nodes with.\n\nBy default it is the path to \"erl\", taken from `init:get_argument(progname)`.\nIf `progname` is not known, `peer` makes best guess given the current ERTS version.\n\nWhen a tuple is passed, the first element is the path to executable, and the\nsecond element is prepended to the final command line. This can be used to start\npeers on a remote host or in a Docker container. See the examples above.\n\nThis option is useful for testing backwards compatibility with previous\nreleases, installed at specific paths, or when the Erlang installation location\nis missing from the `PATH`.","ref":"peer.html#t:exec/0"},{"type":"function","title":"peer.get_state/1","doc":"Returns the peer node state.\n\nThe initial state is `booting`; the node stays in that state until then boot\nscript is complete, and then the node progresses to `running`. If the node stops\n(gracefully or not), the state changes to `down`.","ref":"peer.html#get_state/1"},{"type":"type","title":"peer.peer_state/0","doc":"Peer node state.","ref":"peer.html#t:peer_state/0"},{"type":"function","title":"peer.random_name/0","doc":"","ref":"peer.html#random_name/0"},{"type":"function","title":"peer.random_name/1","doc":"Creates a sufficiently unique node name for the current host, combining a\nprefix, a unique number, and the current OS process ID.\n\n> #### Note {: .info }\n>\n> Use the `?CT_PEER([\"erl_arg1\"])` macro provided by Common Test\n> `-include_lib(\"common_test/include/ct.hrl\")` for convenience. It starts a new\n> peer using Erlang distribution as the control channel, supplies thes calling\n> module's code path to the peer, and uses the calling function name for the\n> name prefix.","ref":"peer.html#random_name/1"},{"type":"function","title":"peer.send/3","doc":"Uses the alternative connection to send Message to a process on the the peer node.\n\nSilently fails if no alternative connection is configured. The process can\nbe referenced by process ID or registered name.","ref":"peer.html#send/3"},{"type":"type","title":"peer.server_ref/0","doc":"Identifies the controlling process of a peer node.","ref":"peer.html#t:server_ref/0"},{"type":"function","title":"peer.start/1","doc":"Starts a peer node with the specified `t:start_options/0`. Returns the\ncontrolling process and the full peer node name, unless `wait_boot` is not\nrequested and the host name is not known in advance.","ref":"peer.html#start/1"},{"type":"function","title":"peer.start_link/0","doc":"The same as [`start_link(#{name => random_name()})`](`start_link/1`).","ref":"peer.html#start_link/0"},{"type":"function","title":"peer.start_link/1","doc":"Starts a peer node in the same way as `start/1`, except that the peer node is\nlinked to the currently executing process. If that process terminates, the peer\nnode also terminates.\n\nAccepts `t:start_options/0`. Returns the controlling process and the full peer\nnode name, unless `wait_boot` is not requested and host name is not known in\nadvance.\n\nWhen the `standard_io` alternative connection is requested, and `wait_boot` is\nnot set to `false`, a failed peer boot sequence causes the caller to exit with\nthe `{boot_failed, {exit_status, ExitCode}}` reason.","ref":"peer.html#start_link/1"},{"type":"type","title":"peer.start_options/0","doc":"Options that can be used when starting a `peer` node through `start/1` and\n[`start_link/0,1`](`start_link/0`).\n\n- **`name`** - Node name (the part before \"@\"). When `name` is not specified,\n  but `host` is, `peer` follows compatibility behaviour and uses the origin node\n  name.\n\n- **`longnames`** - Use long names to start a node. Default is taken from the\n  origin using `net_kernel:longnames()`. If the origin is not distributed, short\n  names is the default.\n\n- **`host`** - Enforces a specific host name. Can be used to override the\n  default behaviour and start \"node@localhost\" instead of \"node@realhostname\".\n\n- **`peer_down`** - Defines the peer control process behaviour when the control\n  connection is closed from the peer node side (for example when the peer\n  crashes or dumps core). When set to `stop` (default), a lost control\n  connection causes the control process to exit normally. Setting `peer_down` to\n  `continue` keeps the control process running, and `crash` will cause the\n  controlling process to exit abnormally.\n\n- **`connection`** - Alternative connection specification. See the\n  [`connection` datatype](`t:connection/0`).\n\n- **`exec`** - Alternative mechanism to start peer nodes with, for example, ssh\n  instead of the default bash.\n\n- **`detached`** - Defines whether to pass the `-detached` flag to the started\n  peer. This option cannot be set to `false` using the `standard_io` alternative\n  connection type. Default is `true`.\n\n- **`args`** - Extra command line arguments to append to the \"erl\" command.\n  Arguments are passed as is, no escaping or quoting is needed or accepted.\n\n- **`post_process_args`** - Allows the user to change the arguments passed to\n  `exec` before the peer is started. This can for example be useful when the\n  `exec` program wants the arguments to \"erl\" as a single argument. Example:\n\n  ```erlang\n  peer:start(#{ name => peer:random_name(),\n    exec => {os:find_executable(\"bash\"),[\"-c\",\"erl\"]},\n    post_process_args =>\n       fun([\"-c\"|Args]) -> [\"-c\", lists:flatten(lists:join($\\s, Args))] end\n    }).\n  ```\n\n- **`env`** - List of environment variables with their values. This list is\n  applied to a locally started executable. If you need to change the environment\n  of the remote peer, adjust `args` to contain `-env ENV_KEY ENV_VALUE`.\n\n- **`wait_boot`** - Specifies the start/start_link timeout. See\n  [`wait_boot` datatype](`t:wait_boot/0`).\n\n- **`shutdown`** - Specifies the peer node stopping behaviour. See\n  [`stop()`](`stop/1`).","ref":"peer.html#t:start_options/0"},{"type":"function","title":"peer.stop/1","doc":"Stops a peer node. How the node is stopped depends on the\n[`shutdown`](`t:start_options/0`) option passed when starting the peer node.\nCurrently the following `shutdown` options are supported:\n\n- **`halt`** - This is the default shutdown behavior. It behaves as `shutdown`\n  option `{halt, DefaultTimeout}` where `DefaultTimeout` currently equals\n  `5000`.\n\n- **`{halt, Timeout :: disconnect_timeout()}`** - Triggers a call to\n  [`erlang:halt()`](`erlang:halt/0`) on the peer node and then waits for the\n  Erlang distribution connection to the peer node to be taken down. If this\n  connection has not been taken down after `Timeout` milliseconds, it will\n  forcefully be taken down by `peer:stop/1`. See the\n  [warning](`m:peer#dist_connection_close`) below for more info about this.\n\n- **`Timeout :: disconnect_timeout()`** - Triggers a call to\n  [`init:stop()`](`init:stop/0`) on the peer node and then waits for the Erlang\n  distribution connection to the peer node to be taken down. If this connection\n  has not been taken down after `Timeout` milliseconds, it will forcefully be\n  taken down by `peer:stop/1`. See the [warning](`m:peer#dist_connection_close`)\n  below for more info about this.\n\n- **`close`** - Close the _control connection_ to the peer node and return. This\n  is the fastest way for the caller of `peer:stop/1` to stop a peer node.\n\n  Note that if the Erlang distribution connection is not used as control\n  connection it might not have been taken down when `peer:stop/1` returns. Also\n  note that the [warning](`m:peer#dist_connection_close`) below applies when the\n  Erlang distribution connection is used as control connection.\n\n[](){: #dist_connection_close }\n\n> #### Warning {: .warning }\n>\n> In the cases where the Erlang distribution connection is taken down by\n> `peer:stop/1`, other code independent of the peer code might react to the\n> connection loss before the peer node is stopped which might cause undesirable\n> effects. For example, [`global`](`m:global#prevent_overlapping_partitions`)\n> might trigger even more Erlang distribution connections to other nodes to be\n> taken down. The potential undesirable effects are, however, not limited to\n> this. It is hard to say what the effects will be since these effects can be\n> caused by any code with links or monitors to something on the origin node, or\n> code monitoring the connection to the origin node.","ref":"peer.html#stop/1"},{"type":"type","title":"peer.wait_boot/0","doc":"Specifies start/start_link timeout in milliseconds. Can be set to `false`,\nallowing the peer to start asynchronously. If `{Pid, Tag}` is specified instead\nof a timeout, the peer will send `Tag` to the requested process.\n\nThe default is `15_000` ms.","ref":"peer.html#t:wait_boot/0"},{"type":"module","title":"slave","doc":"This module provides functions for starting Erlang slave nodes.\n\nAll slave nodes that are started by a master terminate automatically when the\nmaster terminates. All terminal output produced at the slave is sent back to\nthe master node. File I/O is done through the master.\n\nSlave nodes on other hosts than the current one are started with the `ssh`\nprogram. The user must be allowed to `ssh` to the remote hosts without being\nprompted for a password. This can be arranged in a number of ways (for details,\nsee the `ssh` documentation). A slave node started on the same host as the\nmaster inherits certain environment values from the master, such as the current\ndirectory and the environment variables. For what can be assumed about the\nenvironment when a slave is started on another host, see the documentation for\nthe `ssh` program.\n\nAn alternative to the `ssh` program can be specified on the command line to\n[`erl(1)`](`e:erts:erl_cmd.md`) as follows:\n\n```text\n-rsh Program\n```\n\nNote that the command specified with the `-rsh` flag is treated as a file name\nwhich may contain spaces. It is thus not possible to include any command line\noptions. The remote node will be launched as\n`\"$RSH\" \"$REMOTE_HOSTNAME\" erl -detached -noinput ...`, so the `erl` command\nmust be found in the path on the remote host.\n\nThe slave node is to use the same file system at the master. At least,\nErlang/OTP is to be installed in the same place on both computers and the same\nversion of Erlang is to be used.\n\nA node running on Windows can only start slave nodes on the host on which it is\nrunning.\n\nThe master node must be alive.","ref":"slave.html"},{"type":"function","title":"slave.pseudo/1","doc":"Calls [`pseudo(Master, ServerList)`](`pseudo/2`). If you want to start a node\nfrom the command line and set up a number of pseudo servers, an Erlang runtime\nsystem can be started as follows:\n\n```text\n% erl -name abc -s slave pseudo klacke@super x --\n```","ref":"slave.html#pseudo/1"},{"type":"function","title":"slave.pseudo/2","doc":"Starts a number of pseudo servers. A pseudo server is a server with a registered\nname that does nothing but pass on all message to the real server that executes\nat a master node. A pseudo server is an intermediary that only has the same\nregistered name as the real server.\n\nFor example, if you have started a slave node `N` and want to execute `pxw`\ngraphics code on this node, you can start server `pxw_server` as a pseudo server\nat the slave node. This is illustrated as follows:\n\n```erlang\nrpc:call(N, slave, pseudo, [node(), [pxw_server]]).\n```","ref":"slave.html#pseudo/2"},{"type":"function","title":"slave.relay/1","doc":"Runs a pseudo server. This function never returns any value and the process that\nexecutes the function receives messages. All messages received are simply passed\non to `Pid`.","ref":"slave.html#relay/1"},{"type":"function","title":"slave.start/1","doc":"Equivalent to [`start(Host, Name)`](`start/2`) where `Name` is the same\nas the node that executes this call.","ref":"slave.html#start/1"},{"type":"function","title":"slave.start/2","doc":"","ref":"slave.html#start/2"},{"type":"function","title":"slave.start/3","doc":"Starts a slave node on host `Host`. Host names need not necessarily be specified\nas fully qualified names; short names can also be used. This is the same\ncondition that applies to names of distributed Erlang nodes.\n\nThe name of the started node becomes `Name@Host`.\n\nThe slave node resets its `t:io:user/0` process so that all terminal I/O that is\nproduced at the slave is automatically relayed to the master. Also, the file\nserver is relayed to the master.\n\nArgument `Args` is used to set `erl` command-line arguments. It is\npassed to the new node and can be used for a variety of purposes; see\n[`erl(1)`](`e:erts:erl_cmd.md`).\n\nAs an example, suppose that you want to start a slave node at host `H` with node\nname `Name@H` and want the slave node to have the following properties:\n\n- Directory `Dir` is to be added to the code path.\n- The Mnesia directory is to be set to `M`.\n- The Unix `DISPLAY` environment variable is to be set to the display of the\n  master node.\n\nThe following code is executed to achieve this:\n\n```erlang\nE = \" -env DISPLAY \" ++ net_adm:localhost() ++ \":0 \",\nArg = \"-mnesia_dir \" ++ M ++ \" -pa \" ++ Dir ++ E,\nslave:start(H, Name, Arg).\n```\n\nThe function returns `{ok, Node}`, where `Node` is the name of the new node,\notherwise `{error, Reason}`, where `Reason` can be one of:\n\n- **`timeout`** - The master node failed to get in contact with the slave node.\n  This can occur in a number of circumstances:\n\n  - Erlang/OTP is not installed on the remote host.\n  - The file system on the other host has a different structure to the the\n    master.\n  - The Erlang nodes have different cookies.\n\n- **`no_rsh`** - No remote shell program was found on the computer. Note that\n  `ssh` is used by default, but this can be overridden with the `-rsh` flag.\n\n- **`{already_running, Node}`** - A node with name `Name@Host` already exists.","ref":"slave.html#start/3"},{"type":"function","title":"slave.start_link/1","doc":"","ref":"slave.html#start_link/1"},{"type":"function","title":"slave.start_link/2","doc":"","ref":"slave.html#start_link/2"},{"type":"function","title":"slave.start_link/3","doc":"Starts a slave node in the same way as `start/1,2,3`, except that the slave node\nis linked to the currently executing process. If that process terminates, the\nslave node also terminates.\n\nFor a description of arguments and return values, see\n[`start/1,2,3`](`start/1`).","ref":"slave.html#start_link/3"},{"type":"function","title":"slave.stop/1","doc":"Stops (kills) a node.","ref":"slave.html#stop/1"},{"type":"module","title":"win32reg","doc":"Provides access to the registry on Windows.\n\nThis module provides read and write access to the registry on Windows. It is\nessentially a port driver wrapped around the Win32 API calls for accessing the\nregistry.\n\nThe registry is a hierarchical database, used to store various system and\nsoftware information in Windows. It contains installation data, and is updated\nby installers and system programs. The Erlang installer updates the registry by\nadding data that Erlang needs.\n\nThe registry contains keys and values. Keys are like the directories in a file\nsystem, they form a hierarchy. Values are like files, they have a name and a\nvalue, and also a type.\n\nPaths to keys are left to right, with subkeys to the right and backslash between\nkeys. (Remember that backslashes must be doubled in Erlang strings.) Case is\npreserved but not significant.\n\nFor example, `\"\\\\hkey_local_machine\\\\software\\\\Ericsson\\\\Erlang\\\\5.0\"` is the\nkey for the installation data for the latest Erlang release.\n\nThere are six entry points in the Windows registry, top-level keys. They can be\nabbreviated in this module as follows:\n\n```text\nAbbreviation     Registry key\n============     ============\nhkcr             HKEY_CLASSES_ROOT\ncurrent_user     HKEY_CURRENT_USER\nhkcu             HKEY_CURRENT_USER\nlocal_machine    HKEY_LOCAL_MACHINE\nhklm             HKEY_LOCAL_MACHINE\nusers            HKEY_USERS\nhku              HKEY_USERS\ncurrent_config   HKEY_CURRENT_CONFIG\nhkcc             HKEY_CURRENT_CONFIG\ndyn_data         HKEY_DYN_DATA\nhkdd             HKEY_DYN_DATA\n```\n\nThe key above can be written as `\"\\\\hklm\\\\software\\\\ericsson\\\\erlang\\\\5.0\"`.\n\nThis module uses a current key. It works much like the current directory. From\nthe current key, values can be fetched, subkeys can be listed, and so on.\n\nUnder a key, any number of named values can be stored. They have names, types,\nand data.\n\n`win32reg` supports storing of the following types:\n\n- `REG_DWORD`, which is an integer\n- `REG_SZ`, which is a string\n- `REG_BINARY`, which is a binary\n\nOther types can be read, and are returned as binaries.\n\nThere is also a \"default\" value, which has the empty string as name. It is read\nand written with the atom `default` instead of the name.\n\nSome registry values are stored as strings with references to environment\nvariables, for example, `%SystemRoot%Windows`. `SystemRoot` is an environment\nvariable, and is to be replaced with its value. Function `expand/1` is provided\nso that environment variables surrounded by `%` can be expanded to their values.\n\nFor more information on the Windows registry, see consult the Win32 Programmer's\nReference.","ref":"win32reg.html"},{"type":"module","title":"See Also - win32reg","doc":"`erl_posix_msg`, The Windows 95 Registry (book from O'Reilly), Win32\nProgrammer's Reference (from Microsoft)","ref":"win32reg.html#module-see-also"},{"type":"function","title":"win32reg.change_key/2","doc":"Changes the current key to another key. Works like `cd`. The key can be\nspecified as a relative path or as an absolute path, starting with `\\.`","ref":"win32reg.html#change_key/2"},{"type":"function","title":"win32reg.change_key_create/2","doc":"Creates a key, or just changes to it, if it is already there. Works like a\ncombination of `mkdir` and `cd`. Calls the Win32 API function\n`RegCreateKeyEx()`.\n\nThe registry must have been opened in write mode.","ref":"win32reg.html#change_key_create/2"},{"type":"function","title":"win32reg.close/1","doc":"Closes the registry. After that, the `RegHandle` cannot be used.","ref":"win32reg.html#close/1"},{"type":"function","title":"win32reg.current_key/1","doc":"Returns the path to the current key. This is the equivalent of `pwd`.\n\nNotice that the current key is stored in the driver, and can be invalid (for\nexample, if the key has been removed).","ref":"win32reg.html#current_key/1"},{"type":"function","title":"win32reg.delete_key/1","doc":"Deletes the current key, if it is valid. Calls the Win32 API function\n`RegDeleteKey()`. Notice that this call does not change the current key (unlike\n`change_key_create/2`). This means that after the call, the current key is\ninvalid.","ref":"win32reg.html#delete_key/1"},{"type":"function","title":"win32reg.delete_value/2","doc":"Deletes a named value on the current key. The atom `default` is used for the\ndefault value.\n\nThe registry must have been opened in write mode.","ref":"win32reg.html#delete_value/2"},{"type":"function","title":"win32reg.expand/1","doc":"Expands a string containing environment variables between percent characters.\nAnything between two `%` is taken for an environment variable, and is replaced\nby the value. Two consecutive `%` are replaced by one `%`.\n\nA variable name that is not in the environment results in an error.","ref":"win32reg.html#expand/1"},{"type":"function","title":"win32reg.format_error/1","doc":"Converts a POSIX error code to a string (by calling `file:format_error/1`).","ref":"win32reg.html#format_error/1"},{"type":"type","title":"win32reg.name/0","doc":"","ref":"win32reg.html#t:name/0"},{"type":"function","title":"win32reg.open/1","doc":"Opens the registry for reading or writing. The current key is the root\n(`HKEY_CLASSES_ROOT`). Flag `read` in the mode list can be omitted.\n\nUse `change_key/2` with an absolute path after [`open`](`open/1`).","ref":"win32reg.html#open/1"},{"type":"opaque","title":"win32reg.reg_handle/0","doc":"As returned by `open/1`.","ref":"win32reg.html#t:reg_handle/0"},{"type":"function","title":"win32reg.set_value/3","doc":"Sets the named (or default) value to `value`. Calls the Win32 API function\n`RegSetValueEx()`. The value can be of three types, and the corresponding\nregistry type is used. The supported types are the following:\n\n- `REG_DWORD` for integers\n- `REG_SZ` for strings\n- `REG_BINARY` for binaries\n\nOther types cannot be added or changed.\n\nThe registry must have been opened in write mode.","ref":"win32reg.html#set_value/3"},{"type":"function","title":"win32reg.sub_keys/1","doc":"Returns a list of subkeys to the current key. Calls the Win32 API function\n`EnumRegKeysEx()`.\n\nAvoid calling this on the root keys, as it can be slow.","ref":"win32reg.html#sub_keys/1"},{"type":"type","title":"win32reg.value/0","doc":"","ref":"win32reg.html#t:value/0"},{"type":"function","title":"win32reg.value/2","doc":"Retrieves the named value (or default) on the current key. Registry values of\ntype `REG_SZ` are returned as strings. Type `REG_DWORD` values are returned as\nintegers. All other types are returned as binaries.","ref":"win32reg.html#value/2"},{"type":"function","title":"win32reg.values/1","doc":"Retrieves a list of all values on the current key. The values have types\ncorresponding to the registry types, see `value/2`. Calls the Win32 API function\n`EnumRegValuesEx()`.","ref":"win32reg.html#values/1"},{"type":"behaviour","title":"gen_event","doc":"Generic event handling behavior.\n\nThis behavior module provides event handling functionality.\nIt consists of a generic event manager process with any number of\nevent handlers that are added and deleted dynamically.\n\nAn event manager implemented using this module has a standard set of\ninterface functions and includes functionality for tracing\nand error reporting.  It also fits into an OTP supervision tree.\nFor more information, see [OTP Design Principles](`e:system:events.md`).\n\nEach event handler is implemented as a callback module\nexporting a predefined set of functions. The relationship between\nthe behavior functions and the callback functions is as follows:\n\n```text\ngen_event module                   Callback module\n----------------                   ---------------\ngen_event:start\ngen_event:start_monitor\ngen_event:start_link       ----->  -\n\ngen_event:add_handler\ngen_event:add_sup_handler  ----->  Module:init/1\n\ngen_event:notify\ngen_event:sync_notify      ----->  Module:handle_event/2\n\ngen_event:send_request\ngen_event:call             ----->  Module:handle_call/2\n\n-                          ----->  Module:handle_info/2\n\ngen_event:delete_handler   ----->  Module:terminate/2\n\ngen_event:swap_handler\ngen_event:swap_sup_handler ----->  Module1:terminate/2\n                                   Module2:init/1\n\ngen_event:which_handlers   ----->  -\n\ngen_event:stop             ----->  Module:terminate/2\n\n-                          ----->  Module:code_change/3\n```\n\nAs each event handler is one callback module, an event manager\nhas many callback modules that are added and deleted dynamically.\n`gen_event` is therefore more tolerant of callback module errors\nthan the other behaviors.  If a callback function for an installed\nevent handler fails with `Reason`, or returns a bad value `Term`,\nthe event manager does not fail.  It deletes the event handler\nby calling callback function [`Module:terminate/2`](`c:terminate/2`),\ngiving as argument `{error, {'EXIT', Reason}}` or `{error, Term}`,\nrespectively.  No other event handler is affected.\n\nA `gen_event` process handles system messages as described in `m:sys`.\nThe `sys` module can be used for debugging an event manager.\n\nNotice that an event manager _does_ trap exit signals automatically.\n\nThe `gen_event` process can go into hibernation\n(see `erlang:hibernate/3`) if a callback function in a handler module\nspecifies `hibernate` in its return value.  This can be useful\nif the server is expected to be idle for a long time.\nHowever, use this feature with care, as hibernation implies\nat least two garbage collections (when hibernating\nand shortly after waking up) and is not something you want to do\nbetween each event handled by a busy event manager.\n\nNotice that when multiple event handlers are invoked,\nit is sufficient that one single event handler returns a `hibernate`\nrequest for the whole event manager to go into hibernation.\n\nUnless otherwise stated, all functions in this module fail\nif the specified event manager does not exist\nor if bad arguments are specified.\n\n> #### Note {: .info }\n>\n> For some important information about distributed signals, see the\n> [_Blocking Signaling Over Distribution_\n> ](`e:system:ref_man_processes.md#blocking-signaling-over-distribution`)\n> section in the _Processes_ chapter of the _Erlang Reference Manual_.\n> Blocking signaling can, for example, cause call timeouts in `gen_event`\n> to be significantly delayed.","ref":"gen_event.html"},{"type":"behaviour","title":"See Also - gen_event","doc":"`m:supervisor`, `m:sys`","ref":"gen_event.html#module-see-also"},{"type":"function","title":"gen_event.add_handler/3","doc":"Add a new event handler to an event manager.\n\nThe new event handler is added to event manager `EventMgrRef`.\nThe event manager calls [`Module:init/1`](`c:init/1`)\nto initiate the event handler and its internal state.\n\n`Handler` is the name of the callback module `Module`\nor a tuple `{Module, Id}`, where `Id` is any term.\nThe `{Module, Id}` representation makes it possible to\nidentify a specific event handler, when many event handlers\nuse the same callback module.\n\n`Args` is any term that is passed as the argument to\n[`Module:init/1`](`c:init/1`).\n\nIf [`Module:init/1`](`c:init/1`) returns a correct value\nindicating successful completion, the event manager\nadds the event handler and this function returns `ok`.\nIf [`Module:init/1`](`c:init/1`) fails with `Reason` or returns\n`{error,Reason}`, the event handler is ignored and this function\nreturns `{'EXIT',Reason}` or `{error,Reason}`, respectively.","ref":"gen_event.html#add_handler/3"},{"type":"type","title":"gen_event.add_handler_ret/0","doc":"","ref":"gen_event.html#t:add_handler_ret/0"},{"type":"function","title":"gen_event.add_sup_handler/3","doc":"Add a new event handler to an event manager, supervised.\n\nThe new event handler is added as for `add_handler/3`,\nbut the event manager also supervises the connection\nby linking the event handler and the calling process.\n\n- If the calling process later terminates with `Reason`,\n  the event manager deletes any supervised event handlers by calling\n  [`Module:terminate/2`](`c:terminate/2`), then calls\n  [`Module:handle_info/2`](`c:handle_info/2`) for each remaining handler.\n\n- If the event handler is deleted later, the event manager\n  sends a message `{gen_event_EXIT,Handler,Reason}`\n  to the calling process. `Reason` is one of the following:\n\n  + `normal`, if the event handler has been removed because of\n    a call to [`delete_handler/3`](`delete_handler/3`),\n    or `remove_handler` has been returned by a callback function\n    (see below).\n  + `shutdown`, if the event handler has been removed\n    because the event manager is terminating.\n  + `{swapped, NewHandler, Pid}`, if the process `Pid` has replaced\n    the event handler with another event handler `NewHandler`,\n    through a call to `swap_handler/3` or `swap_sup_handler/3`.\n  + Other `t:term/0`, if the event handler is removed\n    because of an error.  Which term depends on the error.\n\nFor a description of the arguments and return values, see `add_handler/3`.","ref":"gen_event.html#add_sup_handler/3"},{"type":"function","title":"gen_event.call/3","doc":"","ref":"gen_event.html#call/3"},{"type":"function","title":"gen_event.call/4","doc":"Make a synchronous call to an event handler.\n\nThe call is sent to `Handler`, installed in event manager `EventMgrRef`,\nby sending a request and waiting until a reply arrives,\nor a time-out occurs.  The event manager calls\n[`Module:handle_call/2`](`c:handle_call/2`) to handle the request.\n\n`Request` is any term that is passed as one of the arguments to\n[`Module:handle_call/2`](`c:handle_call/2`).\n\n`Timeout` is an integer greater than zero that specifies\nhow many milliseconds to wait for a reply, or the atom `infinity`\nto wait indefinitely.  Defaults to 5000.  If no reply is received\nwithin the specified time, the function call fails.\n\nThe return value `Reply` is defined in the return value of\n[`Module:handle_call/2`](`c:handle_call/2`).  If the specified\nevent handler is not installed, the function returns\n`{error, bad_module}`.  If the callback function fails with `Reason`,\nor returns an unexpected value `Term`, this function returns\n`{error, {'EXIT', Reason}}` or `{error, Term}`, respectively.\n\nWhen this call fails it [exits](`erlang:exit/1`) the calling process.\nThe exit term is on the form `{Reason, Location}` where\n`Location = {gen_event, call, ArgList}`. See `gen_server:call/3`\nthat has a description of relevant values for the `Reason`\nin the exit term.","ref":"gen_event.html#call/4"},{"type":"function","title":"gen_event.check_response/2","doc":"Check if a received message is a request response.\n\nCheck if `Msg` is a response corresponding to\nthe request identifier `ReqId`.  The request must have been made\nby `send_request/3`, and by the same process calling this function.\n\nIf `Msg` is a response corresponding to `ReqId` the response is returned\nin `Reply`.  Otherwise this function returns `no_reply`\nand no cleanup is done.  Thus this function must be invoked repeatedly\nuntil a response is returned.\n\nIf the specified event handler is not installed, the function returns\n`{error, bad_module}`.  If the callback function fails with `Reason`\nor returns an unexpected value `Term`, this function returns\n`{error, {'EXIT', Reason}}` or `{error, Term}`, respectively.\nIf the event manager has died before this function is called,\nthat is; `Msg` reports the server's death, this function returns\n`{error,{Reason, EventMgrRef}}` where `Reason` is the exit reason.","ref":"gen_event.html#check_response/2"},{"type":"function","title":"gen_event.check_response/3","doc":"Check if a received message is a request response in a collection.\n\nCheck if `Msg` is a response corresponding to a request identifier\nstored in `ReqIdCollection`.  All request identifiers of `ReqIdCollection`\nmust correspond to requests that have been made using `send_request/3`\nor `send_request/5`, and all requests must have been made\nby the process calling this function.\n\nThe `Label` in the response is the `Label` associated with\nthe request identifier that the response corresponds to.\nThe `Label` of a request identifier is associated\nwhen [storing the request id](`reqids_add/3`) in a collection,\nor when sending the request using `send_request/5`.\n\nCompared to `check_response/2`, the returned result or exception\nassociated with a specific request identifier will be wrapped\nin a 3-tuple `{Response, Label, NewReqIdCollection}`.\n`Response` is the value that would have been produced\nby `check_response/2`, `Label` is the value associated with\nthe specific [request identifier](`t:request_id/0`)\nand `NewReqIdCollection` is a possibly modified\nrequest identifier collection.\n\nIf `ReqIdCollection` is empty, `no_request` will be returned.\n\nIf `Msg` does not correspond to any of the request identifiers\nin `ReqIdCollection`, `no_reply` is returned.\n\nIf `Delete` is `true`, the association with `Label` has been deleted\nfrom `ReqIdCollection` in the resulting `NewReqIdCollection`.\nIf `Delete` is `false`, `NewReqIdCollection` will equal `ReqIdCollection`.\nNote that deleting an association is not for free and that\na collection containing already handled requests\ncan still be used by subsequent calls to `check_response/3`),\n`receive_response/3`, and `wait_response/3`.\n\nHowever, without deleting handled associations, the above calls\nwill not be able to detect when there are no more outstanding requests\nto handle, so you will have to keep track of this some other way\nthan relying on a `no_request` return.  Note that if you pass\na collection only containing associations of already handled\nor abandoned requests to `check_response/3`,\nit will always return `no_reply`.","ref":"gen_event.html#check_response/3"},{"type":"callback","title":"gen_event.code_change/3","doc":"Update the event handler state after code change.\n\nThis function is called for an installed event handler\nthat is to update its internal state during a release upgrade/downgrade,\nthat is, when the instruction `{update, Module, Change,...}`,\nis specified in the [`appup`](`e:sasl:appup.md`) file.\n\nFor more information, see [OTP Design Principles](`e:system:index.html`).\n\nFor an upgrade, `OldVsn` is `Vsn`, and for a downgrade,\n`OldVsn` is `{down, Vsn}`.  `Vsn` is defined by the `vsn` attribute(s)\nof the old version of the callback module `Module`.  If no such attribute\nis defined, the version is the checksum of the Beam file.\n\n`State` is the internal state of the event handler.\n\n`Extra` is passed \"as is\" from the `{advanced, Extra}` part\nof the update instruction.\n\nThe function is to return the updated internal state.\n\n> #### Note {: .info }\n>\n> If a release upgrade/downgrade with `Change={advanced, Extra}`\n> specified in the [`.appup`](`e:sasl:appup.md`) file is made\n> when `c:code_change/3` is not implemented the event handler will crash\n> with an `undef` error reason.","ref":"gen_event.html#c:code_change/3"},{"type":"type","title":"gen_event.debug_flag/0","doc":"","ref":"gen_event.html#t:debug_flag/0"},{"type":"type","title":"gen_event.del_handler_ret/0","doc":"","ref":"gen_event.html#t:del_handler_ret/0"},{"type":"function","title":"gen_event.delete_handler/3","doc":"Deletes an event handler from an event manager.\n\nThis function deletes event handler `Handler` from event manager\n`EventMgrRef`. The event manager calls\n[`Module:terminate/2`](`c:terminate/2`) to terminate the event handler.\n\n`Args` is any term that is passed as one of the arguments to\n[`Module:terminate/2`](`c:terminate/2`).\n\nThe return value is the return value of\n[`Module:terminate/2`](`c:terminate/2`).  If the specified\nevent handler is not installed, the function returns\n`{error, module_not_found}`. If the callback function fails\nwith `Reason`, the function returns `{'EXIT', Reason}`.","ref":"gen_event.html#delete_handler/3"},{"type":"type","title":"gen_event.emgr_name/0","doc":"Event manager name specification: `local`, `global`, or `via` registered.\n\n- *`{local, Name}`* - the event manager is registered locally as\n  `Name` using [`register/2`](`register/2`).\n- *`{global, GlobalName}`* - The event manager is registered\n  globally as `GlobalName` using `global:register_name/2`.\n  If no name is provided, the event manager is not registered.\n- *`{via, Module, ViaName}`*, the event manager registers with the\n  registry represented by `Module`. The `Module` callback is to export\n  the functions `register_name/2`, `unregister_name/1`, `whereis_name/1`,\n  and `send/2`, which are to behave as the corresponding functions\n  in `m:global`.  Thus, `{via, global, GlobalName}` is a valid reference.","ref":"gen_event.html#t:emgr_name/0"},{"type":"type","title":"gen_event.emgr_ref/0","doc":"A reference used to locate an event manager.\n\nThe reference can be any of the following:\n\n- The pid of the event manager\n- `Name`, if the event manager is locally registered\n- `{Name, Node}`, if the event manager is locally registered\n  at another node\n- `{global, GlobalName}`, if the event manager is globally registered\n- `{via, Module, ViaName}`, if the event manager is registered through\n  an alternative process registry","ref":"gen_event.html#t:emgr_ref/0"},{"type":"type","title":"gen_event.format_status/0","doc":"A map that describes the `gen_event` process status.\n\nThe keys are:\n- **`state`** - The internal state of the event handler.\n- **`message`** - The message that caused the event handler to terminate.\n- **`reason`** - The reason that caused the event handler to terminate.\n- **`log`** - The [sys log](`sys:log/2`) of the server.\n\nNew associations may be added into the status map without prior notice.","ref":"gen_event.html#t:format_status/0"},{"type":"callback","title":"gen_event.format_status/1","doc":"Format/limit the status value.\n\nThis function is called by a `gen_event` process in in order to\nformat/limit the server state for debugging and logging purposes.\n\nIt is called in the following situations:\n\n- One of [`sys:get_status/1,2`](`sys:get_status/1`) is invoked\n  to get the `gen_event` status.\n\n- The event handler terminates abnormally and `gen_event` logs an error.\n\nThis callback is used to limit the status of the event handler returned by\n[`sys:get_status/1,2`](`sys:get_status/1`) or sent to `m:logger`.\n\nThe callback gets a map `Status` describing the current status\nand shall return a map `NewStatus` with the same keys,\nbut it may transform some values.\n\nTwo possible use cases for this callback is to remove\nsensitive information from the state to prevent it from being printed\nin log files, or to compact large irrelevant status items\nthat would only clutter the logs.\n\n_Example_:\n\n```erlang\nformat_status(Status) ->\n  maps:map(\n    fun(state,State) ->\n            maps:remove(private_key, State);\n       (message,{password, _Pass}) ->\n            {password, removed};\n       (_,Value) ->\n            Value\n    end, Status).\n```\n\n> #### Note {: .info }\n>\n> This callback is optional, so event handler modules need not export it.\n> If a handler does not export this function, the `gen_event` module\n> uses the handler state directly for the purposes described below.\n>\n> If this callback is exported but fails, to hide possibly sensitive data,\n> the default function will instead return the fact that\n> [`format_status/1`](`c:format_status/1`) has crashed.","ref":"gen_event.html#c:format_status/1"},{"type":"callback","title":"gen_event.format_status/2","doc":"Format/limit the status value.\n\nThis function is called by a `gen_event` process in in order to\nformat/limit the server state for debugging and logging purposes.\n\nIt is called in the following situations:\n\n- One of [`sys:get_status/1,2`](`sys:get_status/1`) is invoked\n  to get the `gen_event` status. `Opt` is set to the atom `normal`\n  for this case.\n\n- The event handler terminates abnormally and `gen_event` logs an error.\n  `Opt` is set to the atom `terminate` for this case.\n\nThis function is useful for changing the form and appearance of the event\nhandler state for these cases. An event handler callback module\nwishing to change the `sys:get_status/1,2` return value as well as\nhow its state appears in termination error logs, exports an instance of\n[`format_status/2`](`c:format_status/2`) that returns a term\ndescribing the current state of the event handler.\n\n`PDict` is the current value of the process dictionary of `gen_event`.\n\n`State` is the internal state of the event handler.\n\nThe function is to return `Status`, a term that change the details of\nthe current state of the event handler. Any term is allowed for `Status`.\nThe `gen_event` module uses `Status` as follows:\n\n- When `sys:get_status/1,2` is called, `gen_event` ensures that\n  its return value contains `Status` in place of the state term\n  of the event handler.\n\n- When an event handler terminates abnormally, `gen_event` logs `Status`\n  in place of the state term of the event handler.\n\nOne use for this function is to return compact alternative\nstate representations to avoid that large state terms\nare printed in log files.\n\n> #### Note {: .info }\n>\n> This callback is optional, so event handler modules need not export it.\n> If a handler does not export this function, the `gen_event` module\n> uses the handler state directly for the purposes described below.","ref":"gen_event.html#c:format_status/2"},{"type":"callback","title":"gen_event.handle_call/2","doc":"Handle a call.\n\nWhenever an event manager receives a request sent using\n[`call/3,4`](`call/3`), this function is called\nfor the specified event handler to handle the request.\n\n`Request` is the `Request` argument of `call/3,4`.\n\n`State` is the internal state of the event handler.\n\nThe return values are the same as for\n[`Module:handle_event/2`](`c:handle_event/2`) except that\nthey also contain a term `Reply`, which is the reply to the client\nas the return value of `call/3,4`.","ref":"gen_event.html#c:handle_call/2"},{"type":"callback","title":"gen_event.handle_event/2","doc":"Handle an event.\n\nWhenever an event manager receives an event sent using `notify/2` or\n`sync_notify/2`, this function is called for each installed event handler\nto handle the event.\n\n`Event` is the `Event` argument of `notify/2` / `sync_notify/2`.\n\n`State` is the internal state of the event handler.\n\n- If `{ok, NewState}` or `{ok, NewState, hibernate}` is returned,\n  the event handler remains in the event manager with the possibly\n  updated internal state `NewState`.\n\n- If `{ok, NewState, hibernate}` is returned, the event manager\n  also goes into hibernation (by calling `proc_lib:hibernate/3`),\n  waiting for the next event to occur.  It is sufficient\n  that one of the event handlers return `{ok, NewState, hibernate}`\n  for the whole event manager process to hibernate.\n\n- If `{swap_handler, Args1, NewState, Handler2, Args2}` is returned,\n  the event handler is replaced by `Handler2` by first calling\n  [`Module:terminate(Args1, NewState)`](`c:terminate/2`) and then\n  [`Module2:init({Args2, Term})`](`c:init/1`), where `Term`\n  is the return value of [`Module:terminate/2`](`c:terminate/2`).\n  For more information, see `swap_handler/3`.\n\n- If `remove_handler` is returned, the event handler is deleted by calling\n  [`Module:terminate(remove_handler, State)`](`c:terminate/2`).","ref":"gen_event.html#c:handle_event/2"},{"type":"callback","title":"gen_event.handle_info/2","doc":"Handle an info message (regular process message).\n\nThis function is called for each installed event handler when\nan event manager receives any other message than an event\nor a synchronous request (or a system message).\n\n`Info` is the received message.\n\nIn particular, this callback will be made when a process terminated\nafter calling `add_sup_handler/3`. Any event handler attached to\nan event manager which in turn has a supervised handler\nshould expect callbacks of the shape\n[`Module:handle_info({'EXIT', Pid, Reason}, State)`](`c:handle_info/2`).\n\nFor a description of `State` and possible return values,\nsee [`Module:handle_event/2`](`c:handle_event/2`).\n\n> #### Note {: .info }\n>\n> This callback is optional, so callback modules need not export it.\n> The `gen_event` module provides a default implementation\n> of this function that logs about the unexpected `Info` message,\n> drops it and returns `{ok, State}`.","ref":"gen_event.html#c:handle_info/2"},{"type":"type","title":"gen_event.handler/0","doc":"","ref":"gen_event.html#t:handler/0"},{"type":"type","title":"gen_event.handler_args/0","doc":"","ref":"gen_event.html#t:handler_args/0"},{"type":"callback","title":"gen_event.init/1","doc":"Initialize the event handler.\n\nWhenever a new event handler is added to an event manager,\nthis function is called to initialize the event handler.\n\nIf the event handler is added because of a call to `add_handler/3` or\n`add_sup_handler/3`, `InitArgs` is the `Args` argument of these functions.\n\nIf the event handler replaces another event handler because of\na call to `swap_handler/3` or `swap_sup_handler/3`, or because of\na `swap` return tuple from one of the other callback functions,\n`InitArgs` is a tuple `{Args, Term}`, where `Args` is the argument\nprovided in the function call/return tuple and `Term` is the result\nof terminating the old event handler, see `swap_handler/3`.\n\nIf successful, the function returns `{ok, State}` or\n`{ok, State, hibernate}`, where `State` is the initial internal state\nof the event handler.\n\nIf `{ok, State, hibernate}` is returned, the event manager\ngoes into hibernation (by calling `proc_lib:hibernate/3`),\nwaiting for the next event to occur.","ref":"gen_event.html#c:init/1"},{"type":"function","title":"gen_event.notify/2","doc":"Send an asynchronous event notification to an event manager.\n\nThe event is sent to `EventMgrRef`, that calls\n[`Module:handle_event/2`](`c:handle_event/2`) for each installed\nevent handler to handle the event.\n\n`Event` is any term that is passed as one of the arguments to\n[`Module:handle_event/2`](`c:handle_event/2`).\n\n`notify/1` does not fail even if the specified event manager\ndoes not exist, unless it is specified as `Name`.","ref":"gen_event.html#notify/2"},{"type":"type","title":"gen_event.options/0","doc":"Options that can be used to configure an event handler\nwhen it is started.","ref":"gen_event.html#t:options/0"},{"type":"function","title":"gen_event.receive_response/2","doc":"Receive a request response.\n\nReceive a response corresponding to the request identifier `ReqId`.\nThe request must have been made by `send_request/3`,\nand it must have been made from the same process calling this function.\n\n`Timeout` specifies how long to wait for a response.\nIf no response is received within the specified time,\nthis function returns `timeout`. Assuming that the\nserver executes on a node supporting aliases (introduced in OTP 24)\nthe request will also be abandoned.  That is,\nno response will be received after a timeout.\nOtherwise, a stray response might be received at a later time.\n\nThe return value `Reply` is defined in the return value of\n[`Module:handle_call/2`](`c:handle_call/2`).\n\nIf the specified event handler is not installed, this function returns\n`{error, bad_module}`. If the callback function fails\nwith `Reason` or returns an unexpected value `Term`,\nthis function returns `{error, {'EXIT', Reason}}` or`{error,Term}`,\nrespectively.  If the event manager dies before or during the\nrequest this function returns `{error, {Reason, EventMgrRef}}`.\n\nThe difference between `wait_response/2` and `receive_response/2`\nis that `receive_response/2` abandons the request at time-out\nso that a potential future response is ignored,\nwhile [`wait_response/2`](`wait_response/2`) does not.","ref":"gen_event.html#receive_response/2"},{"type":"function","title":"gen_event.receive_response/3","doc":"Receive a request response in a collection.\n\nReceive a response in `ReqIdCollection`. All request identifiers\nof `ReqIdCollection` must correspond to requests that have been\nmade using `send_request/3` or `send_request/5`, and all requests\nmust have been made by the process calling this function.\n\nThe `Label` in the response is the `Label` associated with\nthe request identifier that the response corresponds to.\nThe `Label` of a request identifier is associated\nwhen [adding the request id](`reqids_add/3`) to a collection,\nor when sending the request using `send_request/5`.\n\nCompared to `receive_response/2`, the returned result or exception\nassociated with a specific request identifier will be wrapped\nin a 3-tuple `{Response, Label, NewReqIdCollection}`.\n`Response` is the value that would have been produced\nby `receive_response/2`, `Label` is the value associated with\nthe specific [request identifier](`t:request_id/0`)\nand `NewReqIdCollection` is a possibly modified\nrequest identifier collection.\n\nIf `ReqIdCollection` is empty, `no_request` will be returned.\n\n`Timeout` specifies how long to wait for a response.  If no response\nis received within the specified time, the function returns `timeout`.\nAssuming that the server executes on a node supporting aliases\n(introduced in OTP 24) all requests identified by `ReqIdCollection`\nwill also be abandoned.  That is, no responses will be received\nafter a time-out.  Otherwise, stray responses might be received\nat a later time.\n\nThe difference between `receive_response/3` and `wait_response/3`\nis that `receive_response/3` abandons the requests at time-out\nso that potential future responses are ignored,\nwhile `wait_response/3` does not.\n\nIf `Delete` is `true`, the association with `Label`\nis deleted from `ReqIdCollection` in the resulting\n`NewReqIdCollection`.  If `Delete` is `false`, `NewReqIdCollection`\nwill equal `ReqIdCollection`.  Note that deleting an association\nis not for free and that a collection containing already handled\nrequests can still be used by subsequent calls to\n`receive_response/3`, `check_response/3`, and `wait_response/3`.\n\nHowever, without deleting handled associations,\nthe above calls will not be able to detect when there are\nno more outstanding requests to handle, so you will have to keep track\nof this some other way than relying on a `no_request` return.\nNote that if you pass a collection only containing\nassociations of already handled or abandoned requests to\n`receive_response/3`, it will always block until `Timeout` expires\nand then return `timeout`.","ref":"gen_event.html#receive_response/3"},{"type":"function","title":"gen_event.reqids_add/3","doc":"Store a request identifier in a colletion.\n\nStores `ReqId` and associates a `Label` with the request identifier\nby adding this information to `ReqIdCollection` and returning\nthe resulting request identifier collection.","ref":"gen_event.html#reqids_add/3"},{"type":"function","title":"gen_event.reqids_new/0","doc":"Create an empty request identifier collection.\n\nReturns a new empty request identifier collection.\nA request identifier collection can be utilized to handle\nmultiple outstanding requests.\n\nRequest identifiers of requests made by `send_request/3`\ncan be saved in a request identifier collection using `reqids_add/3`.\nSuch a collection of request identifiers can later be used\nin order to get one response corresponding to a request\nin the collection by passing the collection as argument to\n`receive_response/3`, `wait_response/3`, or, `check_response/3`.\n\n`reqids_size/1` can be used to determine the number of\nrequest identifiers in a collection.","ref":"gen_event.html#reqids_new/0"},{"type":"function","title":"gen_event.reqids_size/1","doc":"Returns the number of request identifiers in `ReqIdCollection`.","ref":"gen_event.html#reqids_size/1"},{"type":"function","title":"gen_event.reqids_to_list/1","doc":"Convert a request identifier collection to a list.\n\nReturns a list of `{ReqId, Label}` tuples which corresponds to\nall request identifiers with their associated labels\nin [`ReqIdCollection`](`t:request_id_collection/0`).","ref":"gen_event.html#reqids_to_list/1"},{"type":"opaque","title":"gen_event.request_id/0","doc":"An opaque request identifier. See `send_request/3` for details.","ref":"gen_event.html#t:request_id/0"},{"type":"opaque","title":"gen_event.request_id_collection/0","doc":"An opaque collection of request identifiers (`t:request_id/0`).\n\nEach request identifier can be associated with a label\nchosen by the user.  For more information see `reqids_new/0`.","ref":"gen_event.html#t:request_id_collection/0"},{"type":"type","title":"gen_event.response_timeout/0","doc":"Response time-out for an asynchronous call.\n\nUsed to set a time limit on how long to wait for a response using either\n`receive_response/2`, `receive_response/3`, `wait_response/2`, or\n`wait_response/3`. The time unit used is `millisecond`.\nCurrently valid values:\n\n- **`0..4294967295`** - Timeout relative to current time in milliseconds.\n\n- **`infinity`** - Infinite timeout. That is, the operation\n  will never time out.\n\n- **`{abs, Timeout}`** - An absolute\n  [Erlang monotonic time](`erlang:monotonic_time/1`) timeout\n  in milliseconds.  That is, the operation will time out when\n  [`erlang:monotonic_time(millisecond)`](`erlang:monotonic_time/1`)\n  returns a value larger than or equal to `Timeout`.\n `Timeout` is not allowed to identify a time further into the future\n  than `4294967295` milliseconds. Identifying the timeout using\n  an absolute timeout value is especially handy when you have a\n  deadline for responses corresponding to a complete collection\n  of requests (`t:request_id_collection/0`) , since you do not have to\n  recalculate the relative time until the deadline over and over again.","ref":"gen_event.html#t:response_timeout/0"},{"type":"function","title":"gen_event.send_request/3","doc":"Send an asynchronous `call` request to an event handler.\n\nThis function sends the call request `Request` to the event handler\n`Handler` installed in the event manager identified by `EventMgrRef`,\nand returns a request identifier `ReqId`.  The return value `ReqId`\nshall later be used with `receive_response/2`, `wait_response/2`,\nor `check_response/2` to fetch the actual result of the request.\n\nBesides passing the request identifier directly to these functions,\nit can also be stored in a request identifier collection\nusing `reqids_add/3`.  Such a collection of request identifiers\ncan later be used in order to get one response corresponding to\na request in the collection by passing the collection as argument to\n`receive_response/3`, `wait_response/3`, or `check_response/3`.\nIf you are about to store the request identifier in a collection,\nyou may want to consider using `send_request/5` instead.\n\nThe calls\n`gen_event:receive_response(gen_event:send_request(EventMgrRef,\nHandler, Request), Timeout)`\ncan be seen as equivalent to\n[`gen_event:call(EventMgrRef, Handler, Request, Timeout)`](`call/3`),\nignoring the error handling.\n\nThe event manager calls [`Module:handle_call/2`](`c:handle_call/2`)\nto handle the request.\n\n`Request` may be any term and is passed as one of the arguments to\n[`Module:handle_call/2`](`c:handle_call/2`).","ref":"gen_event.html#send_request/3"},{"type":"function","title":"gen_event.send_request/5","doc":"Send an asynchronous `call` request to an event handler,\nstoring it in a request identifier collection.\n\nThis function sends the call request `Request` to the event handler\n`Handler` installed in the event manager identified by `EventMgrRef`.\nThe `Label` will be associated with the request identifier\nof the operation and added to the returned\nrequest identifier collection `NewReqIdCollection`.\n\nThe collection can later be used in order to get one response\ncorresponding to a request in the collection by passing the collection\nas argument to `receive_response/3`, `wait_response/3`,\nor `check_response/3`.\n\nThe same as calling\n[`gen_event:reqids_add`](`reqids_add/3`)`(`[`gen_event:send_request`](`send_request/3`)`(EventMgrRef, Handler, Request), Label, ReqIdCollection)`,\nbut slightly more efficient.","ref":"gen_event.html#send_request/5"},{"type":"function","title":"gen_event.start/0","doc":"","ref":"gen_event.html#start/0"},{"type":"function","title":"gen_event.start/1","doc":"Create a stand-alone event manager process, possibly nameless.\n\nEquivalent to [`start(EventMgrName, Options)`](`start/2`).\n\nWith argument `EventMgrName`, `Options` is `[]`.\n\nWith argument `Options` a nameless event manager is created.\n\nFor a description of the arguments and return values, see `start_link/2`.","ref":"gen_event.html#start/1"},{"type":"function","title":"gen_event.start/2","doc":"Create a stand-alone event manager process.\n\nThe created event manager process is not part of a supervision tree\nand thus has no supervisor.\n\nFor a description of the arguments and return values, see `start_link/2`.","ref":"gen_event.html#start/2"},{"type":"function","title":"gen_event.start_link/0","doc":"","ref":"gen_event.html#start_link/0"},{"type":"function","title":"gen_event.start_link/1","doc":"Create an event manager process as part of a supervision tree,\npossibly nameless.\n\nEquivalent to [`start_link(EventMgrName, Options)`](`start_link/2`).\n\nWith argument `EventMgrName`, `Options` is `[]`.\n\nWith argument `Options` a nameless event manager is created.\n\nFor a description of the arguments and return values, see `start_link/2`.","ref":"gen_event.html#start_link/1"},{"type":"function","title":"gen_event.start_link/2","doc":"Create an event manager process as part of a supervision tree.\n\nThe function is to be called, directly or indirectly, by the supervisor.\nFor example, it ensures that the event manager is linked\nto the caller (supervisor).\n\n- If option `{hibernate_after, HibernateAfterTimeout}` is present, the\n  `gen_event` process awaits any message for `HibernateAfterTimeout`\n  milliseconds and if no message is received, the process\n  goes into hibernation automatically (by calling `proc_lib:hibernate/3`).\n\nIf the event manager is successfully created,\nthe function returns `{ok, Pid}` where `Pid` is the `t:pid/0`\nof the event manager.\n\nIf a process with the specified `EventMgrName` exists already,\nthe function returns `{error,{already_started,OtherPid}}`,\nwhere `OtherPid` is the pid of that process, and the event manager process\nexits with reason `normal`.\n\nIf the event manager fails to start within the specified start timeout\n`{timeout, Time}`, which is very unlikely since the start\ndoes not interact with other processes, the function returns\n`{error, timeout}` and the failed event manager is killed with\n[`exit(_, kill)`](`erlang:exit/2`).\n\nIf `start_link/1,2` returns `{error, _}`, the started event manager process\nhas terminated.  If an `'EXIT'` message was delivered\nto the calling process (due to the process link), that message\nhas been consumed.\n\n> #### Warning {: .warning }\n>\n> Before OTP 26.0, if the started event manager failed to register\n> its name, this founction could return\n> `{error, {already_started, OtherPid}}` _before_\n> the started event manager process had terminated,\n> so starting again might fail because the registered name\n> was not yet unregistered, and an `'EXIT'` message could arrive later\n> to the process calling this function.\n>\n> But if the start timed out, this function killed\n> the started event manager process and returned `{error, timeout}`,\n> and then the process link `{'EXIT', Pid, killed}` message _was_ consumed.\n>\n> The start was made synchronous in OTP 26.0 and a guarantee\n> was implemented that no process link `'EXIT'` message\n> from a failed start will linger in the caller's inbox.","ref":"gen_event.html#start_link/2"},{"type":"type","title":"gen_event.start_mon_ret/0","doc":"","ref":"gen_event.html#t:start_mon_ret/0"},{"type":"function","title":"gen_event.start_monitor/0","doc":"","ref":"gen_event.html#start_monitor/0"},{"type":"function","title":"gen_event.start_monitor/1","doc":"Creates a stand-alone event manager process,\nmonitored, possibly nameless.\n\nEquivalent to [`start_monitor(EventMgrName, Options)`](`start_monitor/2`).\n\nWith argument `EventMgrName`, `Options` is `[]`.\n\nWith argument `Options` a nameless event manager is created.\n\nFor a description of the arguments and return values,\nsee `start_monitor/2` and `start_link/1`.","ref":"gen_event.html#start_monitor/1"},{"type":"function","title":"gen_event.start_monitor/2","doc":"Creates a stand-alone event manager process, monitored.\n\nThe created event manager process is not part of a supervision tree\nand thus has no supervisor.  A monitor is atomically set up\nto the newly created process.\n\nFor a description of the arguments and return values, see\n[`start_link/2`](`start_link/2`). Note that the return value\nfor a successful start differs from `start_link/2`.\n`start_monitor/0,1,2` will return `{ok, {Pid, Mon}}`\nwhere `Pid` is the process identifier of the process,\nand `Mon` is a reference to the monitor set up to monitor the process.\nIf the start is not successful, the caller will be blocked\nuntil the `DOWN` message has been received and removed\nfrom the message queue.","ref":"gen_event.html#start_monitor/2"},{"type":"type","title":"gen_event.start_ret/0","doc":"","ref":"gen_event.html#t:start_ret/0"},{"type":"function","title":"gen_event.stop/1","doc":"","ref":"gen_event.html#stop/1"},{"type":"function","title":"gen_event.stop/3","doc":"Stop an event manager.\n\nOrders event manager `EventMgrRef` to exit with the specifies `Reason`,\nand waits for it to terminate.  Before terminating, `gen_event` calls\n[`Module:terminate(stop,...)`](`c:terminate/2`)\nfor each installed event handler.\n\nThe function returns `ok` if the event manager terminates\nwith the expected reason.  Any other reason than `normal`,\n`shutdown`, or `{shutdown, Term}` causes an error report\nto be issued using `m:logger`.\n\n`Timeout` is an integer greater than zero that specifies\nhow many milliseconds to wait for the event manager to terminate,\nor the atom `infinity` to wait indefinitely.  If the event manager\nhas not terminated within the specified time, the call exits\nthe calling process with reason `timeout`.\n\nIf the process does not exist,\nthe call exits the calling process with reason `noproc`,\nand with reason `{nodedown, Node}` if the connection fails\nto the remote `Node` where the server runs.","ref":"gen_event.html#stop/3"},{"type":"function","title":"gen_event.swap_handler/3","doc":"Replace an event handler.\n\nThis function replaces an event handler in event manager `EventMgrRef`.\n\nFor a description of `OldHandler` and `NewHandler`, see `add_handler/3`.\n\nFirst the old event handler `OldHandler` is deleted. The event manager\ncalls `OldModule:terminate(Args1, ...)`, where `OldModule`\nis the callback module of `OldHandler`, and collects the return value.\n\nThen the new event handler `NewHandler` is added and initiated\nby calling [`NewModule:init({Args2,Term})`](`c:init/1`), where `NewModule`\nis the callback module of `NewHandler`, and `Term` is the return value\nof [`OldModule:terminate/2`](`c:terminate/2`).  This makes it possible\nto transfer information from `OldHandler` to `NewHandler`.\n\nThe new handler is added even if the the specified old event handler\nis not installed, in which case `Term = error`, or if\n[`OldModule:terminate/2`](`c:terminate/2`) fails with `Reason`,\nin which case `Term = {'EXIT', Reason}`.  The old handler\nis deleted even if [`NewModule:init/1`](`c:init/1`) fails.\n\nIf there was a supervised connection\nbetween `OldHandler` and a process `Pid`,\nthere is a supervised connection between `NewHandler` and `Pid` instead.\n\nIf [`NewModule:init/1`](`c:init/1`) returns a correct value,\nthis function returns `ok`. If [`NewModule:init/1`](`c:init/1`) fails\nwith `Reason` or returns an unexpected value `Term`,\nthis function returns `{error, {'EXIT', Reason}}` or\n`{error, Term}`, respectively.","ref":"gen_event.html#swap_handler/3"},{"type":"function","title":"gen_event.swap_sup_handler/3","doc":"Replace an event handler, and supervise it.\n\nReplaces an event handler in event manager `EventMgrRef`\nin the same way as [`swap_handler/3`](`swap_handler/3`),\nbut also supervises the connection between `NewHandler`\nand the calling process.\n\nFor a description of the arguments and return values, see `swap_handler/3`.","ref":"gen_event.html#swap_sup_handler/3"},{"type":"function","title":"gen_event.sync_notify/2","doc":"Send a synchronous event notification to an event manager.\n\nThe event is sent to `EventMgrRef` that callsr calls\n[`Module:handle_event/2`](`c:handle_event/2`) for each installed\nevent handler to handle the event. This function will return `ok`\nafter the event has been handled by all event handlers.\n\n`Event` is any term that is passed as one of the arguments to\n[`Module:handle_event/2`](`c:handle_event/2`).","ref":"gen_event.html#sync_notify/2"},{"type":"callback","title":"gen_event.terminate/2","doc":"Handle event handler termination.\n\nWhenever an event handler is deleted from an event manager,\nthis function is called. It is to be the opposite\nof [`Module:init/1`](`c:init/1`) and do any necessary cleaning up.\n\nIf the event handler is deleted because of a call to `delete_handler/3`,\n`swap_handler/3`, or `swap_sup_handler/3`, `Arg` is\nthe `Args` argument of this function call.\n\n`Arg = {stop, Reason}` if the event handler has a supervised connection\nto a process that has terminated with reason `Reason`.\n\n`Arg = stop` if the event handler is deleted because\nthe event manager is terminating.\n\nThe event manager terminates if it is part of a supervision tree\nand it is ordered by its supervisor to terminate.  Even if\nit is _not_ part of a supervision tree, it terminates if it receives\nan `'EXIT'` message from its parent.\n\n`Arg = remove_handler` if the event handler is deleted\nbecause another callback function has returned `remove_handler`\nor `{remove_handler, Reply}`.\n\n`Arg = {error, Term}` if the event handler is deleted because\na callback function returned an unexpected value `Term`,\nor `Arg = {error, {'EXIT', Reason}}` if a callback function failed.\n\n`State` is the internal state of the event handler.\n\nThe function can return any term.  If the event handler\nis deleted because of a call to `gen_event:delete_handler/3`,\nthe return value of that function becomes the return value\nof this function. If the event handler is to be replaced with\nanother event handler because of a swap, the return value\nis passed to the `init` function of the new event handler.\nOtherwise the return value is ignored.\n\n> #### Note {: .info }\n>\n> This callback is optional, so callback modules need not export it.\n> The `gen_event` module provides a default implementation\n> without cleanup.","ref":"gen_event.html#c:terminate/2"},{"type":"function","title":"gen_event.wait_response/2","doc":"Wait for a request resonse.\n\nWait for the response to the request identifier `ReqId`. The request\nmust have been made by `send_request/3`, from the same process\nthat called `send_request/3`.\n\n`WaitTime` specifies how long to wait for a response.\nIf no response is received within the specified time,\nthe function returns `timeout` and no cleanup is done,\nThus the function can be invoked repeatedly until a reply is returned.\n\nThe return value `Reply` is defined in the return value of\n[`Module:handle_call/2`](`c:handle_call/2`).\n\nIf the specified event handler is not installed, the function returns\n`{error, bad_module}`.  If the callback function fails with `Reason`,\nor returns an unexpected value `Term`, this function returns\n`{error,{'EXIT',Reason}}` or `{error,Term}`, respectively.\nIf the event manager dies before or during the request\nthis function returns `{error, {Reason, EventMgrRef}}`.\n\nThe difference between `receive_response/2` and\n`wait_response/2` is that `receive_response/2` abandons the request\nat timeout so that a potential future response is ignored,\nwhile [`wait_response/2`](`wait_response/2`) does not.","ref":"gen_event.html#wait_response/2"},{"type":"function","title":"gen_event.wait_response/3","doc":"Wait for any request response in a collection.\n\nWait for a response in a `ReqIdCollection`.  All request identifiers\nof `ReqIdCollection` must correspond to requests that have been made\nusing `send_request/3` or `send_request/5`, and all requests\nmust have been made by the process calling this function.\n\nThe `Label` in the response is the `Label` associated with\nthe request identifier that the response corresponds to.\nThe `Label` of a request identifier is associated\nwhen [adding the request id](`reqids_add/3`) to a collection,\nor when sending the request using `send_request/5`.\n\nCompared to `wait_response/2`, the returned result or exception\nassociated with a specific request identifier will be wrapped\nin a 3-tuple `{Response, Label, NewReqIdCollection}`.\n`Response` is the value that would have been produced\nby `wait_response/2`, `Label` is the value associated with\nthe specific [request identifier](`t:request_id/0`)\nand `NewReqIdCollection` is a possibly modified\nrequest identifier collection.\n\nIf `ReqIdCollection` is empty, `no_request` will be returned.\n\nIf no response is received before the `WaitTime` has expired,\n`timeout` is returned.  It is valid to continue waiting\nfor a response as many times as needed up until a response\nhas been received and completed by `check_response()`,\n`receive_response()`, or `wait_response()`.\n\nThe difference between `receive_response/3` and `wait_response/3`\nis that `receive_response/3` abandons requests at time-out\nso that potential future responses are ignored, while\n[`wait_response/3`](`wait_response/3`) does not.\n\nIf `Delete` is `true`, the association with `Label`\nhas been deleted from `ReqIdCollection` in the resulting\n`NewReqIdCollection`. If `Delete` is`false`, `NewReqIdCollection`\nwill equal `ReqIdCollection`.  Note that deleting an association\nis not for free and that a collection containing already handled\nrequests can still be used by subsequent calls to\n`wait_response/3`, `check_response/3`, and `receive_response/3`.\n\nHowever, without deleting handled associations, the above\ncalls will not be able to detect when there are\nno more outstanding requests to handle, so you will have to keep track\nof this some other way than relying on a `no_request` return.\nNote that if you pass a collection only containing\nassociations of already handled or abandoned requests\nto this function, it will always block until `WaitTime` expires\nand then return `timeout`.","ref":"gen_event.html#wait_response/3"},{"type":"function","title":"gen_event.which_handlers/1","doc":"Return all event handlers in an event manager.\n\nThis function returns a list of all event handlers\ninstalled in event manager `EventMgrRef`.\n\nFor a description of `Handler`, see `add_handler/3`.","ref":"gen_event.html#which_handlers/1"},{"type":"behaviour","title":"gen_fsm","doc":"Deprecated and replaced by `m:gen_statem` in OTP 20.\n\nMigration to gen_statem\n-----------------------\n\nHere follows a simple example of turning a gen_fsm into a `m:gen_statem`.\nThe example comes from the previous User's Guide for `gen_fsm`\n\n```erlang\n-module(code_lock).\n-define(NAME, code_lock).\n%-define(BEFORE_REWRITE, true).\n\n-ifdef(BEFORE_REWRITE).\n-behaviour(gen_fsm).\n-else.\n-behaviour(gen_statem).\n-endif.\n\n-export([start_link/1, button/1, stop/0]).\n\n-ifdef(BEFORE_REWRITE).\n-export([init/1, locked/2, open/2, handle_sync_event/4, handle_event/3,\n     handle_info/3, terminate/3, code_change/4]).\n-else.\n-export([init/1, callback_mode/0, locked/3, open/3,\n     terminate/3, code_change/4]).\n%% Add callback__mode/0\n%% Change arity of the state functions\n%% Remove handle_info/3\n-endif.\n\n-ifdef(BEFORE_REWRITE).\nstart_link(Code) ->\n    gen_fsm:start_link({local, ?NAME}, ?MODULE, Code, []).\n-else.\nstart_link(Code) ->\n    gen_statem:start_link({local,?NAME}, ?MODULE, Code, []).\n-endif.\n\n-ifdef(BEFORE_REWRITE).\nbutton(Digit) ->\n    gen_fsm:send_event(?NAME, {button, Digit}).\n-else.\nbutton(Digit) ->\n    gen_statem:cast(?NAME, {button,Digit}).\n    %% send_event is asynchronous and becomes a cast\n-endif.\n\n-ifdef(BEFORE_REWRITE).\nstop() ->\n    gen_fsm:sync_send_all_state_event(?NAME, stop).\n-else.\nstop() ->\n    gen_statem:call(?NAME, stop).\n    %% sync_send is synchronous and becomes call\n    %% all_state is handled by callback code in gen_statem\n-endif.\n\ninit(Code) ->\n    do_lock(),\n    Data = #{code => Code, remaining => Code},\n    {ok, locked, Data}.\n\n-ifdef(BEFORE_REWRITE).\n-else.\ncallback_mode() ->\n    state_functions.\n%% state_functions mode is the mode most similar to\n%% gen_fsm. There is also handle_event mode which is\n%% a fairly different concept.\n-endif.\n\n-ifdef(BEFORE_REWRITE).\nlocked({button, Digit}, Data0) ->\n    case analyze_lock(Digit, Data0) of\n    {open = StateName, Data} ->\n        {next_state, StateName, Data, 10000};\n    {StateName, Data} ->\n        {next_state, StateName, Data}\n    end.\n-else.\nlocked(cast, {button,Digit}, Data0) ->\n    case analyze_lock(Digit, Data0) of\n    {open = StateName, Data} ->\n        {next_state, StateName, Data, 10000};\n    {StateName, Data} ->\n        {next_state, StateName, Data}\n    end;\nlocked({call, From}, Msg, Data) ->\n    handle_call(From, Msg, Data);\nlocked({info, Msg}, StateName, Data) ->\n    handle_info(Msg, StateName, Data).\n%% Arity differs\n%% All state events are dispatched to handle_call and handle_info help\n%% functions. If you want to handle a call or cast event specifically\n%% for this state you would add a special clause for it above.\n-endif.\n\n-ifdef(BEFORE_REWRITE).\nopen(timeout, State) ->\n     do_lock(),\n    {next_state, locked, State};\nopen({button,_}, Data) ->\n    {next_state, locked, Data}.\n-else.\nopen(timeout, _, Data) ->\n    do_lock(),\n    {next_state, locked, Data};\nopen(cast, {button,_}, Data) ->\n    {next_state, locked, Data};\nopen({call, From}, Msg, Data) ->\n    handle_call(From, Msg, Data);\nopen(info, Msg, Data) ->\n    handle_info(Msg, open, Data).\n%% Arity differs\n%% All state events are dispatched to handle_call and handle_info help\n%% functions. If you want to handle a call or cast event specifically\n%% for this state you would add a special clause for it above.\n-endif.\n\n-ifdef(BEFORE_REWRITE).\nhandle_sync_event(stop, _From, _StateName, Data) ->\n    {stop, normal, ok, Data}.\n\nhandle_event(Event, StateName, Data) ->\n    {stop, {shutdown, {unexpected, Event, StateName}}, Data}.\n\nhandle_info(Info, StateName, Data) ->\n    {stop, {shutdown, {unexpected, Info, StateName}}, StateName, Data}.\n-else.\n-endif.\n\nterminate(_Reason, State, _Data) ->\n    State =/= locked andalso do_lock(),\n    ok.\ncode_change(_Vsn, State, Data, _Extra) ->\n    {ok, State, Data}.\n\n%% Internal functions\n-ifdef(BEFORE_REWRITE).\n-else.\nhandle_call(From, stop, Data) ->\n     {stop_and_reply, normal,  {reply, From, ok}, Data}.\n\nhandle_info(Info, StateName, Data) ->\n    {stop, {shutdown, {unexpected, Info, StateName}}, StateName, Data}.\n%% These are internal functions for handling all state events\n%% and not behaviour callbacks as in gen_fsm\n-endif.\n\nanalyze_lock(Digit, #{code := Code, remaining := Remaining} = Data) ->\n     case Remaining of\n         [Digit] ->\n         do_unlock(),\n         {open,  Data#{remaining := Code}};\n         [Digit|Rest] -> % Incomplete\n             {locked, Data#{remaining := Rest}};\n         _Wrong ->\n             {locked, Data#{remaining := Code}}\n     end.\n\ndo_lock() ->\n    io:format(\"Lock~n\", []).\ndo_unlock() ->\n    io:format(\"Unlock~n\", []).\n```\n\nOTP 19 Documentation\n--------------------","ref":"gen_fsm.html"},{"type":"behaviour","title":"Module - gen_fsm","doc":"`gen_fsm`","ref":"gen_fsm.html#module-module"},{"type":"behaviour","title":"Module Summary - gen_fsm","doc":"Generic finite state machine behavior.","ref":"gen_fsm.html#module-module-summary"},{"type":"behaviour","title":"Description - gen_fsm","doc":"This behavior module provides a finite state machine.\nA generic finite state machine process (`gen_fsm`) implemented\nusing this module has a standard set of interface functions\nand includes functionality for tracing and error reporting.\nIt also fits into an OTP supervision tree.  For more information,\nsee [OTP Design Principles](`e:system:design_principles`).\n\nA `gen_fsm` process assumes all specific parts to be located\nin a callback module exporting a predefined set of functions.\nThe relationship between the behavior functions\nand the callback functions is as follows:\n\n``` text\ngen_fsm module                    Callback module\n--------------                    ---------------\ngen_fsm:start\ngen_fsm:start_link                -----> Module:init/1\n\ngen_fsm:stop                      -----> Module:terminate/3\n\ngen_fsm:send_event                -----> Module:StateName/2\n\ngen_fsm:send_all_state_event      -----> Module:handle_event/3\n\ngen_fsm:sync_send_event           -----> Module:StateName/3\n\ngen_fsm:sync_send_all_state_event -----> Module:handle_sync_event/4\n\n-                                 -----> Module:handle_info/3\n\n-                                 -----> Module:terminate/3\n\n-                                 -----> Module:code_change/4\n```\n\nIf a callback function fails or returns a bad value,\nthe `gen_fsm` process terminates.\n\nA `gen_fsm` process handles system messages as described\nin [sys(3)](`m:sys`).  The sys module can be used for\ndebugging a `gen_fsm` process.\n\nNotice that a `gen_fsm` process does not trap exit signals automatically,\nthis must be explicitly initiated in the callback module.\n\nUnless otherwise stated, all functions in this module fail\nif the specified `gen_fsm` process does not exist\nor if bad arguments are specified.\n\nThe gen_fsm process can go into hibernation (see `erlang:hibernate/3`)\nif a callback function specifies `hibernate` instead of a time-out value.\nThis can be useful if the server is expected to be idle for a long time.\nHowever, use this feature with care, as hibernation implies at least\ntwo garbage collections (when hibernating and shortly after waking up)\nand is not something you want to do between each call\nto a busy state machine.","ref":"gen_fsm.html#module-description"},{"type":"behaviour","title":"Callback Functions - gen_fsm","doc":"See the [Callback Functions](#callbacks-deprecated) section\nfor the functions to be exported from a `gen_fsm` callback module.\n\n[]() {: #state-name }\n**State name** denotes a state of the state machine.\n\n[]() {: #state-data }\n**State data** denotes the internal state of the Erlang process\nthat implements the state machine.","ref":"gen_fsm.html#module-callback-functions"},{"type":"function","title":"gen_fsm.cancel_timer/1","doc":"Cancel an internal timer in a generic FSM.\n\nCancels an internal timer referred by `Ref` in the `gen_fsm` process\nthat calls this function.\n\n`Ref` is a reference returned from `send_event_after/2`\nor `start_timer/2`.\n\nIf the timer has already timed out, but the event not yet been delivered,\nit is cancelled as if it had not timed out, so there is no false\ntimer event after returning from this function.\n\nReturns the remaining time in milliseconds until the timer\nwould have expired if `Ref` referred to an active timer,\notherwise `false`.","ref":"gen_fsm.html#cancel_timer/1"},{"type":"callback","title":"gen_fsm.code_change/4","doc":"Update the internal [*state data*](#state-data) during upgrade/downgrade.\n\nThis function is called by a `gen_fsm` process when it is to update\nits internal [*state data*](#state-data)\nduring a release upgrade/downgrade, that is,\nwhen instruction `{update, Module, Change, ...}`,\nwhere `Change = {advanced, Extra}`, is given in the appup file;\nsee section Release Handling Instructions in OTP Design Principles.\n[OTP Design Principles](`e:system:release_handling.md#instr`).\n\nFor an upgrade, `OldVsn` is `Vsn`, and for a downgrade,\n`OldVsn` is `{down, Vsn}`. `Vsn` is defined by the vsn attribute(s)\nof the old version of the callback module `Module`.  If no such\n attribute is defined, the version is the checksum of the Beam file.\n\n`StateName` is the current [*state name*](#state-name)\n and `StateData` the internal [*state data*](#state-data)\n of the `gen_fsm` process.\n\n`Extra` is passed \"as is\" from the `{advanced, Extra}` part\n of the update instruction.\n\nThe function is to return the new current [*state name*](#state-name)\nand updated internal data.","ref":"gen_fsm.html#c:code_change/4"},{"type":"function","title":"gen_fsm.enter_loop/4","doc":"Enter the `gen_fsm` receive loop.\n\nEquivalent to `enter_loop/6` with `Timeout = infinity`\nbut the started server is not registered as for `start_link/3`.","ref":"gen_fsm.html#enter_loop/4"},{"type":"function","title":"gen_fsm.enter_loop/5","doc":"Enter the `gen_fsm` receive loop.\n\nWith argument `FsmName` equivalent to `enter_loop/6`\nwith `Timeout = infinity`.\n\nWith argument `Timeout` equivalent to `enter_loop/6`\nbut the started server is not registered as for `start_link/3`.","ref":"gen_fsm.html#enter_loop/5"},{"type":"function","title":"gen_fsm.enter_loop/6","doc":"Enter the `gen_fsm` receive loop.\n\nMakes an existing process into a `gen_fsm` process.  Does not return,\ninstead the calling process enters the `gen_fsm` receive loop\nand becomes a `gen_fsm` process.  The process must have been started\nusing one of the start functions in `m:proc_lib`.  The user is responsible\nfor any initialization of the process, including registering a name for it.\n\nThis function is useful when a more complex initialization procedure\nis needed than the `gen_fsm` behavior provides.\n\n`Module`, `Options`, and `FsmName` have the same meanings\nas when calling [`start[_link]/3,4`](`start_link/4`).\nHowever, the process must have been registered according to\n`FsmName` before this function is called.\n\n`StateName`, `StateData`, and `Timeout` have the same meanings\nas in the return value of [`Module:init/1`](`c:init/1`).\nThe callback module `Module` does not need to export\nan `c:init/1` function.\n\nThe function fails if the calling process was not started\nby a `m:proc_lib` start function, or if it is not registered\naccording to `FsmName`.","ref":"gen_fsm.html#enter_loop/6"},{"type":"type","title":"gen_fsm.enter_loop_opt/0","doc":"[Start options](#start-options) for the\n[`enter_loop/4,5,6`](`enter_loop/6`), [`start/3,4`](`start/3`),\nand [`start_link/3,4`](`start_link/3`) functions.\n\nSee `start_link/4`.","ref":"gen_fsm.html#t:enter_loop_opt/0"},{"type":"callback","title":"gen_fsm.format_status/2","doc":"Optional function for providing a term describing\nthe current `gen_fsm` process status.\n\nThe second argument is `[PDict, StateData]`, that is, a list\nwith the 2 elements, in that order.\n\n> #### Note {: .info }\n>\n> This callback is optional, so callback modules need not export it.\n> The `gen_fsm` module provides a default implementation\n> of this function that returns the callback module\n> [*state data*](#state-data).\n\nThis function is called by a `gen_fsm` process\nin the following situations:\n\n- One of [`sys:get_status/1,2`](`sys:get_status/1`) is invoked to get\n  the `gen_fsm` status. `Opt` is set to the atom `normal` for this case.\n- The `gen_fsm` process terminates abnormally and logs an error.\n  `Opt` is set to the atom terminate for this case.\n\nThis function is useful for changing the form and appearance\nof the `gen_fsm` status for these cases.  A callback module\nwishing to change the [`sys:get_status/1,2`](`sys:get_status/1`)\nreturn value as well as how its status appears in termination error logs,\nexports an instance of `c:format_status/2` that returns a term\ndescribing the current status of the `gen_fsm` process.\n\n`PDict` is the current value of the process dictionary\nof the `gen_fsm` process.\n\n`StateData` is the internal [*state data*](#state-data)\nof the `gen_fsm` process.\n\nThe function is to return `Status`, a term that change the details\nof the current state and status of the `gen_fsm` process.\nThere are no restrictions on the form `Status` can take,\nbut for the [`sys:get_status/1,2`](`sys:get_status/1`) case\n(when `Opt` is `normal`), the recommended form for the `Status` value\nis `[{data, [{\"StateData\", Term}]}]`, where `Term` provides\nrelevant details of the `gen_fsm` [*state data*](#state-data).\nFollowing this recommendation is not required, but it makes\nthe callback module status consistent with the rest of\nthe [`sys:get_status/1,2`](`sys:get_status/1`) return value.\n\nOne use for this function is to return compact alternative\n[*state data*](#state-data) representations to avoid\nthat large state terms are printed in log files.","ref":"gen_fsm.html#c:format_status/2"},{"type":"type","title":"gen_fsm.from/0","doc":"Reply destination. See `reply/2`","ref":"gen_fsm.html#t:from/0"},{"type":"type","title":"gen_fsm.fsm_name/0","doc":"[FSM name](#fsm-name) specification:\n`local`, `global`, or `via` registered.\n\nTo be used when starting a `gen_fsm`. See `start_link/4`.","ref":"gen_fsm.html#t:fsm_name/0"},{"type":"type","title":"gen_fsm.fsm_ref/0","doc":"[FSM reference](#fsm-ref) `t:pid/0` or registered `t:fsm_name/0`.\n\nTo be used in for example `send_event/2` to specify the server.","ref":"gen_fsm.html#t:fsm_ref/0"},{"type":"callback","title":"gen_fsm.handle_event/3","doc":"Handle an asynchronous event.\n\nWhenever a `gen_fsm` process receives an event sent using\n`send_all_state_event/2`, this function is called to handle the event.\n\n`StateName` is the current [*state name*](#state-name)\nof the `gen_fsm` process.\n\nFor a description of the other arguments and possible return values,\nsee [`Module:StateName/2`](`c:'StateName'/2`).","ref":"gen_fsm.html#c:handle_event/3"},{"type":"callback","title":"gen_fsm.handle_info/3","doc":"Handle an incoming message\n\nThis function is called by a `gen_fsm` process when it receives\nany other message than a synchronous or asynchronous event\n(or a system message).\n\n`Info` is the received message.\n\nFor a description of the other arguments and possible return values,\nsee [`Module:StateName/2`](`c:'StateName'/2`).","ref":"gen_fsm.html#c:handle_info/3"},{"type":"callback","title":"gen_fsm.handle_sync_event/4","doc":"Handle a synchronous event.\n\nWhenever a `gen_fsm` process receives an event sent using\n[`sync_send_all_state_event/2,3`](`sync_send_all_state_event/3`),\nthis function is called to handle the event.\n\n`StateName` is the current [*state name*](#state-name)\nof the `gen_fsm` process.\n\nFor a description of the other arguments and possible return values,\nsee [`Module:StateName/3`](`c:'StateName'/3`).","ref":"gen_fsm.html#c:handle_sync_event/4"},{"type":"callback","title":"gen_fsm.init/1","doc":"Initialize process and internal [*state name*](#state-name)\nand [*state data*](#state-data).\n\nWhenever a `gen_fsm` process is started using\n[`start/3,4`](`start/4`) or [`start_link/3,4`](`start_link/4`),\nthis function is called by the new process to initialize.\n\n`Args` is the `Args` argument provided to the start function.\n\nIf initialization is successful, the function is to return\n{ok, StateName, StateData}, {ok, StateName, StateData, Timeout},\nor {ok, StateName, StateData, hibernate}, where `StateName`\nis the initial [*state name*](#state-name) and `StateData`\nthe initial [*state data*](#state-data) of the `gen_fsm` process.\n\nIf an `t:integer/0` time-out value is provided, a time-out occurs\nunless an event or a message is received within `Timeout` milliseconds.\nA time-out is represented by the atom `timeout` and is to be handled\nby the [`Module:StateName/2`](`c:'StateName'/2`) callback functions.\nThe atom `infinity` can be used to wait indefinitely, this is\nthe default value.\n\nIf `hibernate` is specified instead of a time-out value,\nthe process goes into hibernation when waiting for the next message\nto arrive (by calling `proc_lib:hibernate/3`).\n\nIf the initialization fails, the function returns `{stop, Reason}`,\nwhere `Reason` is any term, or `ignore`.","ref":"gen_fsm.html#c:init/1"},{"type":"function","title":"gen_fsm.reply/2","doc":"Send a reply to a caller.\n\nThis function can be used by a `gen_fsm` process to explicitly send\na reply to a client process that called\n[`sync_send_event/2,3`](`sync_send_event/3`) or\n[`sync_send_all_state_event/2,3`](`sync_send_all_state_event/3`)\nwhen the reply cannot be defined in the return value of\n[`Module:StateName/3`](`c:'StateName'/3`) or\n[`Module:handle_sync_event/4`](`c:handle_sync_event/4`).\n\n`Caller` must be the `From` argument provided to the callback function.\n`Reply` is any term given back to the client as the return value of\n[`sync_send_event/2,3`](`sync_send_event/3`) or\n[`sync_send_all_state_event/2,3`](`sync_send_all_state_event/3`).\n\nReturn value `Result` is not further defined, and is always to be ignored.","ref":"gen_fsm.html#reply/2"},{"type":"callback","title":"gen_fsm.StateName/2","doc":"Handle an asynchronous event.\n\nThere is to be one instance of this function\nfor each possible [*state name*](#state-name).\nWhenever a `gen_fsm` process receives an event sent using `send_event/2`,\nthe instance of this function with the same name as the current\n[*state name*](#state-name) `StateName` is called to handle the event.\nIt is also called if a time-out occurs.\n\n`Event` is either the atom `timeout`, if a time-out has occurred,\nor the `Event` argument provided to `send_event/2`.\n\n`StateData` is the [*state data*](#state-data) of the `gen_fsm` process.\n\nIf the function returns `{next_state, NextStateName, NewStateData}`,\n`{next_state, NextStateName, NewStateData, Timeout}`,\nor `{next_state, NextStateName, NewStateData, hibernate}`,\nthe `gen_fsm` process continues executing with\nthe current [*state name*](#state-name) set to `NextStateName`\nand with the possibly updated [*state data*](#state-data)\n`NewStateData`.  For a description of `Timeout` and `hibernate`,\nsee [`Module:init/1`](`c:init/1`).\n\nIf the function returns `{stop ,Reason, NewStateData}`,\nthe `gen_fsm` process calls\n[`Module:terminate(Reason, StateName, NewStateData)`](`c:terminate/3`)\nand terminates.","ref":"gen_fsm.html#c:StateName/2"},{"type":"callback","title":"gen_fsm.StateName/3","doc":"Handle a synchronous event.\n\nThere is to be one instance of this function\nfor each possible [*state name*](#state-name).\nWhenever a `gen_fsm` process receives an event sent using\n[`sync_send_event/2,3`](`sync_send_event/3`),\nthe instance of this function with the same name\nas the current [*state name*](#state-name) `StateName` is called\nto handle the event.\n\n`Event` is the `Event` argument provided to\n[`sync_send_event/2,3`](`sync_send_event/3`).\n\n`From` is a tuple `{Pid, Tag}` where `Pid` is the `t:pid/0`\nof the process that called [`sync_send_event/2,3`](`sync_send_event/3`),\n`Tag` is a unique tag.\n\n`StateData` is the [*state data*](#state-data) of the `gen_fsm` process.\n\n- If `{reply, Reply, NextStateName, NewStateData}`,\n  `{reply, Reply, NextStateName, NewStateData, Timeout}`,\n  or `{reply, Reply, NextStateName, NewStateData, hibernate}` is returned,\n  `Reply` is given back to `From` as the return value of\n  [`sync_send_event/2,3`](`sync_send_event/3`).\n  The `gen_fsm` process then continues executing\n  with the current [*state name*](#state-name) set to `NextStateName`\n  and with the possibly updated [*state data*](#state-data) `NewStateData`.\n  For a description of `Timeout` and `hibernate`,\n  see [`Module:init/1`](`c:init/1`).\n\n- If `{next_state, NextStateName, NewStateData}`,\n  `{next_state, NextStateName, NewStateData, Timeout}`,\n  or `{next_state, NextStateName, NewStateData, hibernate}` is returned,\n  the `gen_fsm` process continues executing in `NextStateName`\n  with `NewStateData`.  Any reply to `From`\n  must be specified explicitly using `reply/2`.\n\n- If the function returns `{stop, Reason, Reply, NewStateData}`,\n  `Reply` is given back to `From`.  If the function returns\n  {stop, Reason, NewStateData}, any reply to `From` must be specified\n  explicitly using `reply/2`.  The `gen_fsm` process then calls\n  [`Module:terminate(Reason, StateName, NewStateData)`](`c:terminate/3`)\n  and terminates.","ref":"gen_fsm.html#c:StateName/3"},{"type":"function","title":"gen_fsm.send_all_state_event/2","doc":"Send an event asynchronously to a generic FSM.\n\nSends an event asynchronously to the `FsmRef` of the `gen_fsm` process\nand returns `ok` immediately.  The `gen_fsm` process calls\n[`Module:handle_event/3`](`c:handle_event/3`) to handle the event.\n\nFor a description of the arguments, see `send_event/2`.\n\nThe difference between `send_event/2` and `send_all_state_event/2`\nis which callback function is used to handle the event.\nThis function is useful when sending events that are handled\nthe same way in every state, as only one `handle_event` clause\nis needed to handle the event instead of one clause\nin each state name function.","ref":"gen_fsm.html#send_all_state_event/2"},{"type":"function","title":"gen_fsm.send_event/2","doc":"Send an event asynchronously to a generic FSM.\n\nSends `Event` to the `FsmRef` of the `gen_fsm` process\nand returns `ok` immediately.  The `gen_fsm` process calls\n[`Module:StateName/2`](`c:'StateName'/2`) to handle the event,\nwhere `StateName` is the name of the current state\nof the `gen_fsm` process.\n\n[](){: #fsm-ref }\n`FsmRef` can be any of the following:\n\n- The `t:pid/0`\n- `Name`, if the `gen_fsm` process is locally registered\n- `{Name, Node}`, if the `gen_fsm` process is locally registered\n  at another node\n- `{global, GlobalName}`, if the `gen_fsm` process is globally registered\n- `{via, Module, ViaName}`, if the `gen_fsm` process is registered\n  through an alternative process registry\n\n`Event` is any term that is passed as one of the arguments\nto `Module:StateName/2`.","ref":"gen_fsm.html#send_event/2"},{"type":"function","title":"gen_fsm.send_event_after/2","doc":"Send a delayed event internally in a generic FSM.\n\nSends a delayed event internally in the `gen_fsm` process\nthat calls this function after `Time` milliseconds.\nReturns immediately a reference that can be used to cancel\nthe delayed send using `cancel_timer/1`.\n\nThe `gen_fsm` process calls [`Module:StateName/2`](`c:'StateName'/2`)\nto handle the event, where `'StateName'` is the name of\nthe current state of the `gen_fsm` process at the time\nthe delayed event is delivered.\n\n`Event` is any term that is passed as one of the arguments\nto [`Module:StateName/2`](`c:'StateName'/2`).","ref":"gen_fsm.html#send_event_after/2"},{"type":"function","title":"gen_fsm.start/3","doc":"Create a standalone `gen_fsm` process, not registered.\n\nEquivalent to [`start(Name, Mod, Args, Options)`](`start/4`)\nwithout registering a `Name`.\n\nFor a description of arguments and return values,\nsee [`start_link/3,4`](`start_link/3`).","ref":"gen_fsm.html#start/3"},{"type":"function","title":"gen_fsm.start/4","doc":"Create a standalone `gen_fsm` process.\n\nThe created process is not part of a supervision tree\nand thus has no supervisor.\n\nFor a description of arguments and return values,\nsee [`start_link/3,4`](`start_link/4`).","ref":"gen_fsm.html#start/4"},{"type":"function","title":"gen_fsm.start_link/3","doc":"Create a `gen_fsm` process in a supervision tree, not registered.\n\nEquivalent to [`start_link(Name, Mod, Args, Options)`](`start_link/4`)\nwithout registering a `Name`.","ref":"gen_fsm.html#start_link/3"},{"type":"function","title":"gen_fsm.start_link/4","doc":"Create a `gen_fsm` process in a supervision tree.\n\nThe process is created as part of a supervision tree.  The function\nis to be called, directly or indirectly, by the supervisor.\nFor example, it ensures that the `gen_fsm` process\nis linked to the supervisor.\n\nThe `gen_fsm` process calls [`Module:init/1`](`c:init/1`) to initialize.\nTo ensure a synchronized startup procedure,\n[`start_link/3,4`](`start_link/4`) does not return\nuntil `Module:init/1` has returned.\n\n[]() {: #fsm-name }\n\n- If **`FsmName = {local, Name}`**, the `gen_fsm` process\n  is registered locally as `Name` using `register/2`.\n\n- If **`FsmName = {global, GlobalName}`**, the `gen_fsm` process\n  is registered globally as `GlobalName` using `global:register_name/2`.\n\n- If **`FsmName = {via, Module, ViaName}`**,\n  the `gen_fsm` process registers with the registry\n  represented by `Module`.  The `Module` callback is to export\n  the functions `register_name/2`, `unregister_name/1`,\n  `whereis_name/1`, and `send/2`, which are to behave like\n  the corresponding functions in `m:global`.\n  Thus, `{via, global, GlobalName}` is a valid reference.\n\n`Module` is the name of the callback module.\n\n`Args` is any term that is passed as the argument to `Module:init/1`.\n\n[]() {: #start-options }\n\nIf option **`{timeout, Time}`** is present, the `gen_fsm` process\nis allowed to spend `Time` milliseconds initializing or it terminates\nand the start function returns `{error, timeout}`.\n\nIf option **`{debug, Dbgs}`** is present, the corresponding `sys` function\nis called for each item in `Dbgs`; see [`sys(3)`](`m:sys`).\n\nIf option **`{spawn_opt, SOpts}`** is present, `SOpts` is passed\nas option list to the `spawn_opt` BIF that is used\nto spawn the `gen_fsm` process; see `spawn_opt/2`.\n\n> #### Note {: .info }\n> Using spawn option `monitor` is not allowed, it causes\n> the function to fail with reason `badarg`.\n\nIf the `gen_fsm` process is successfully created and initialized,\nthe function returns `{ok, Pid}`, where `Pid` is the pid\nof the `gen_fsm` process.  If a process with the specified `FsmName`\nexists already, the function returns `{error, {already_started, Pid}}`,\nwhere `Pid` is the pid of that process.\n\nIf `Module:init/1` fails with `Reason`, the function returns\n`{error, Reason}`.  If `Module:init/1` returns `{stop, Reason}`\nor `ignore`, the process is terminated and the function returns\n`{error, Reason}` or `ignore`, respectively.","ref":"gen_fsm.html#start_link/4"},{"type":"type","title":"gen_fsm.start_opt/0","doc":"[Start options](#start-options) for the [`start/3,4`](`start/3`),\nand [`start_link/3,4`](`start_link/3`) functions.\n\nSee `start_link/4`.","ref":"gen_fsm.html#t:start_opt/0"},{"type":"function","title":"gen_fsm.start_timer/2","doc":"Send a time-out event internally in a generic FSM.\n\nSends a time-out event internally in the `gen_fsm process`\nthat calls this function after `Time` milliseconds.\nReturns immediately a reference that can be used to cancel the timer\nusing `cancel_timer/1`.\n\nThe `gen_fsm` process calls [`Module:StateName/2`](`c:'StateName'/2`)\nto handle the event, where `'StateName'` is the name\nof the current state of the `gen_fsm` process at the time\nthe time-out message is delivered.\n\n`Msg` is any term that is passed in the time-out message,\n`{timeout, Ref, Msg}`, as one of the arguments\nto [`Module:StateName/2`](`c:'StateName'/2`).","ref":"gen_fsm.html#start_timer/2"},{"type":"function","title":"gen_fsm.stop/1","doc":"","ref":"gen_fsm.html#stop/1"},{"type":"function","title":"gen_fsm.stop/3","doc":"Synchronously stop a generic FSM.\n\nOrders a generic finite state machine to exit with the specified `Reason`\nand waits for it to terminate.  The `gen_fsm` process calls\n[`Module:terminate/3`](`c:terminate/3`) before exiting.\n\nThe function returns `ok` if the generic finite state machine terminates\nwith the expected reason.  Any other reason than `normal`, `shutdown`,\nor `{shutdown, Term}` causes an error report to be issued using\n`error_logger:format/2`.\n\n`Timeout` is an integer greater than zero that specifies\nhow many milliseconds to wait for the generic FSM to terminate,\nor the atom `infinity` to wait indefinitely.\nIf the generic finite state machine has not terminated\nwithin the specified time, a `timeout` exception is raised.\n\nIf the process does not exist, a `noproc` exception is raised.","ref":"gen_fsm.html#stop/3"},{"type":"function","title":"gen_fsm.sync_send_all_state_event/2","doc":"","ref":"gen_fsm.html#sync_send_all_state_event/2"},{"type":"function","title":"gen_fsm.sync_send_all_state_event/3","doc":"Send an event synchronously to a generic FSM.\n\nSends an event to the `FsmRef` of the `gen_fsm` process and waits\nuntil a reply arrives or a time-out occurs.  The `gen_fsm` process calls\n[`Module:handle_sync_event/4`](`c:handle_sync_event/4`)\nto handle the event.\n\nFor a description of `FsmRef` and `Event`, see `send_event/2`.\nFor a description of `Timeout` and `Reply`, see `sync_send_event/3`.\n\nFor a discussion about the difference between `sync_send_event`\nand `sync_send_all_state_event`, see `send_all_state_event/2`.","ref":"gen_fsm.html#sync_send_all_state_event/3"},{"type":"function","title":"gen_fsm.sync_send_event/2","doc":"","ref":"gen_fsm.html#sync_send_event/2"},{"type":"function","title":"gen_fsm.sync_send_event/3","doc":"Send an event synchronously to a generic FSM.\n\nSends an event to the `FsmRef` of the `gen_fsm` process\nand waits until a reply arrives or a time-out occurs.\nThe `gen_fsm` process calls [`Module:StateName/3`](`c:'StateName'/3`)\nto handle the event, where `'StateName'` is the name\nof the current state of the `gen_fsm` process.\n\nFor a description of `FsmRef` and `Event`, see `send_event/2`.\n\n`Timeout` is an integer greater than zero that specifies\nhow many milliseconds to wait for a reply, or the atom `infinity`\nto wait indefinitely.  If no reply is received within the specified time,\nthe function call fails.\n\nReturn value `Reply` is defined in the return value of\n[`Module:StateName/3`](`c:'StateName'/3`)\n\n> #### Note {: .info }\n> The ancient behavior of sometimes consuming the server exit message\n> if the server died during the call while linked to the client\n> was removed in Erlang 5.6/OTP R12B.","ref":"gen_fsm.html#sync_send_event/3"},{"type":"callback","title":"gen_fsm.terminate/3","doc":"Clean up before termination.\n\nThis function is called by a `gen_fsm` process\nwhen it is about to terminate.  It is to be the opposite of\n[`Module:init/1`](`c:init/1`) and do any necessary cleaning up.\nWhen it returns, the `gen_fsm` process terminates with `Reason`.\nThe return value is ignored.\n\n`Reason` is a term denoting the stop reason, `StateName` is\nthe current [*state name*](#state-name),\nand `StateData` is the [*state data*](#state-data)\nof the `gen_fsm` process.\n\n`Reason` depends on why the `gen_fsm` process is terminating.\nIf it is because another callback function has returned a stop tuple\n`{stop, ...}`, `Reason` has the value specified in that tuple.\nIf it is because of a failure, `Reason` is the error reason.\n\nIf the `gen_fsm` process is part of a supervision tree\nand is ordered by its supervisor to terminate, this function\nis called with `Reason = shutdown` if the following conditions apply:\n\n- The gen_fsm process has been set to trap exit signals.\n\n- The shutdown strategy as defined in the child specification\n  of the supervisor is an integer time-out value, not brutal_kill.\n\nEven if the gen_fsm process is **not** part of a supervision tree,\nthis function is called if it receives an `'EXIT'` message\nfrom its parent. `Reason` is the same as in the `'EXIT'` message.\n\nOtherwise, the gen_fsm process terminates immediately.\n\nNotice that for any other reason than `normal`, `shutdown`,\nor `{shutdown, Term}` the `gen_fsm` process is assumed to terminate\nbecause of an error and an error report is issued\nusing `error_logger:format/2`.","ref":"gen_fsm.html#c:terminate/3"},{"type":"behaviour","title":"gen_server","doc":"Generic server behavior.\n\nThis behavior module provides the server in a client-server relation.\nA generic server process (`gen_server`) implemented using this module\nhas a standard set of interface functions and includes functionality\nfor tracing and error reporting.  It also fits into\nan OTP supervision tree. For more information, see section\n[gen_server Behaviour](`e:system:gen_server_concepts.md`)\nin OTP Design Principles.\n\nA `gen_server` process assumes all specific parts to be located\nin a callback module exporting a predefined set of functions.\nThe relationship between the behavior functions\nand the callback functions is as follows:\n\n```text\ngen_server module            Callback module\n-----------------            ---------------\ngen_server:start\ngen_server:start_monitor\ngen_server:start_link -----> Module:init/1\n\ngen_server:stop       -----> Module:terminate/2\n\ngen_server:call\ngen_server:send_request\ngen_server:multi_call -----> Module:handle_call/3\n\ngen_server:cast\ngen_server:abcast     -----> Module:handle_cast/2\n\n-                     -----> Module:handle_info/2\n\n-                     -----> Module:handle_continue/2\n\n-                     -----> Module:terminate/2\n\n-                     -----> Module:code_change/3\n```\n\nIf a callback function fails or returns a bad value,\nthe `gen_server` process terminates.  However, an exception of class\n[`throw`](`erlang:throw/1`) is not regarded as an error\nbut as a valid return, from all callback functions.\n\nA `gen_server` process handles system messages as described in `m:sys`.\nThe `m:sys` module can be used for debugging a `gen_server` process.\n\nNotice that a `gen_server` process does not trap exit signals\nautomatically, this must be explicitly initiated in the callback module.\n\nUnless otherwise stated, all functions in this module fail\nif the specified `gen_server` process does not exist\nor if bad arguments are specified.\n\nThe `gen_server` process can go into hibernation (see `erlang:hibernate/3`)\nif a callback function specifies `'hibernate'` instead of a time-out value.\nThis can be useful if the server is expected to be idle for a long time.\nHowever, use this feature with care, as hibernation implies at least\ntwo garbage collections (when hibernating and shortly after waking up)\nand is not something you want to do between each call to a busy server.\n\nIf the `gen_server` process needs to perform an action after\ninitialization or to break the execution of a callback into multiple steps,\nit can return `{continue, Continue}` in place of\nthe time-out or hibernation value, which will invoke\nthe [`Module:handle_continue/2`](`c:handle_continue/2`) callback,\nbefore receiving any external message / request.\n\nIf the `gen_server` process terminates, e.g. as a result of a function\nin the callback module returning `{stop,Reason,NewState}`,\nan exit signal with this `Reason` is sent to linked processes and ports.\nSee [Processes](`e:system:ref_man_processes.md#errors`)\nin the Reference Manual for details regarding error handling\nusing exit signals.\n\n> #### Note {: .info }\n>\n> For some important information about distributed signals, see the\n> [_Blocking Signaling Over Distribution_][1]\n> section in the _Processes_ chapter of the _Erlang Reference Manual_.\n> Blocking signaling can, for example, cause call time-outs\n> in `gen_server` to be significantly delayed.\n\n[1]: `e:system:ref_man_processes.md#blocking-signaling-over-distribution`","ref":"gen_server.html"},{"type":"behaviour","title":"See Also - gen_server","doc":"`m:gen_event`, `m:gen_statem`, `m:proc_lib`, `m:supervisor`, `m:sys`","ref":"gen_server.html#module-see-also"},{"type":"function","title":"gen_server.abcast/2","doc":"Cast a request to multiple nodes.\n\nEquivalent to [`abcast(Nodes, Name, Request)`](`abcast/3`)\nwhere `Nodes` is all nodes connected to the calling node,\nincluding the calling node itself.","ref":"gen_server.html#abcast/2"},{"type":"function","title":"gen_server.abcast/3","doc":"Cast a request to multiple nodes.\n\nSends an asynchronous request to the `gen_server` processes\nlocally registered as `Name` at the specified nodes.\nThe function returns immediately and ignores nodes that do not exist,\nor where the `gen_server` `Name` does not exist.  The  `gen_server`\nprocesses call [`Module:handle_cast/2`](`c:handle_cast/2`)\nto handle the request.\n\nFor a description of the arguments,\nsee [`multi_call/2,3,4`](`multi_call/2`).","ref":"gen_server.html#abcast/3"},{"type":"function","title":"gen_server.call/2","doc":"","ref":"gen_server.html#call/2"},{"type":"function","title":"gen_server.call/3","doc":"Call a server: send request and wait for response.\n\nMakes a synchronous call to the `ServerRef` of the `gen_server` process\nby sending a request and waiting until a reply arrives\nor a time-out occurs.  The `gen_server` process calls\n[`Module:handle_call/3`](`c:handle_call/3`) to handle the request.\n\nSee also `ServerRef`'s type `t:server_ref/0`.\n\n`Request` is any term that is passed as the first argument to\n[`Module:handle_call/3`](`c:handle_call/3`).\n\n`Timeout` is an integer that specifies how many milliseconds to wait\nfor a reply, or the atom `infinity` to wait indefinitely.  If no reply\nis received within the specified time, this function exits the calling\nprocess with an exit term containing `Reason = timeout` as described below.\n\n> #### Note {: .info }\n>\n> Before OTP 24, if the caller uses (`try`...)`catch`\n> to avoid process exit, and the server happens to just be late\n> with the reply, it may arrive to the process message queue\n> any time later. The calling process must therefore after\n> catching a time-out exit be prepared to receive garbage message(s)\n> on the form `{reference(), _}` and deal with them appropriately\n> (discard them) so they do not clog the process message queue,\n> or gets mistaken for other messages.\n>\n> Starting with OTP 24, `gen_server:call` uses process aliases,\n> so late replies will not be received.\n\nThe return value `Reply` is passed from the return value of\n[`Module:handle_call/3`](`c:handle_call/3`).\n\nThis call may exit the calling process with an exit term on the form\n`{Reason, Location}` where `Location = {gen_server, call, ArgList}`\nand `Reason` can be (at least) one of:\n\n- **`timeout`** - The call was aborted after waiting `Timeout` milliseconds\n  for a reply, as described above.\n\n- **`noproc`** - The `ServerRef` refers to a server by name (it is not a\n  `t:pid/0`) and looking up the server process failed, or the `t:pid/0`\n  was already terminated.\n\n- **`{nodedown,Node}`** - The `ServerRef` refers to a server\n  on the remote node `Node` and the connection to that node failed.\n\n- **`calling_self`** - A call to `self/0` would hang indefinitely.\n\n- **`shutdown`** - The server was stopped during the call\n  by its supervisor.  See also `stop/3`.\n\n- **`normal`\\\n  `{shutdown,Term}`** - The server stopped during the call\n  by returning `{stop,Reason,_}` from one of its callbacks\n  without replying to this call. See also `stop/3`.\n\n- **`_OtherTerm`** - The server process exited during the call,\n  with reason `Reason`. Either by returning `{stop,Reason,_}`\n  from one of its callbacks (without replying to this call),\n  by raising an exception, or due to getting an exit signal\n  it did not trap.","ref":"gen_server.html#call/3"},{"type":"function","title":"gen_server.cast/2","doc":"Cast a request to a server.\n\nSends an asynchronous request to the `gen_server`\n[`ServerRef`](`t:server_ref/0`) and returns `ok` immediately,\nignoring if the destination node or `gen_server`\nprocess does not exist.\n\nThe `gen_server` process calls\n[`Module:handle_cast(Request, _)`](`c:handle_cast/2`)\nto handle the request.","ref":"gen_server.html#cast/2"},{"type":"function","title":"gen_server.check_response/2","doc":"Check if a received message is a request response.\n\nChecks if `Msg` is a response corresponding to\nthe request identifier `ReqId`.  The request must have been made\nby `send_request/2`, and by the same process calling this function.\n\nIf `Msg` is a reply to the handle `ReqId` the result of the request\nis returned in `Reply`.  Otherwise this function returns `no_reply`\nand no cleanup is done, and thus the function shall be invoked repeatedly\nuntil the response is returned.\n\nThe return value `Reply` is passed from the return value of\n[`Module:handle_call/3`](`c:handle_call/3`).\n\nIf the `gen_statem` server process has died when this function\nis called, that is; `Msg` reports the server's death,\nthis function returns an `error` return with the exit `Reason`.","ref":"gen_server.html#check_response/2"},{"type":"function","title":"gen_server.check_response/3","doc":"Check if a received message is a request response in a collection.\n\nCheck if `Msg` is a response corresponding to a request identifier\nstored in `ReqIdCollection`.  All request identifiers of `ReqIdCollection`\nmust correspond to requests that have been made using `send_request/2`\nor `send_request/4`, by the process calling this function.\n\nThe `Label` in the response equals the `Label` associated\nwith the request identifier that the response corresponds to.\nThe `Label` of a request identifier is associated\nwhen [storing the request id](`reqids_add/3`) in a collection,\nor when sending the request using `send_request/4`.\n\nCompared to `check_response/2`, the returned result or exception\nassociated with a specific request identifier will be wrapped\nin a 3-tuple `{Response, Label, NewReqIdCollection}`.\n`Response` is the value that would have been produced\nby `check_response/2`, `Label` is the value associated with\nthe specific [request identifier](`t:request_id/0`)\nand `NewReqIdCollection` is a possibly modified\nrequest identifier collection.\n\nIf `ReqIdCollection` is empty, `no_request` will be returned.\n\nIf `Msg` does not correspond to any of the request identifiers\nin `ReqIdCollection`, `no_reply` is returned.\n\nIf `Delete` is `true`, the association with `Label`\nhas been deleted from `ReqIdCollection` in the resulting\n`NewReqIdCollection`.  If `Delete` is `false`, `NewReqIdCollection`\nwill equal `ReqIdCollection`.  Note that deleting an association\nis not for free and that a collection containing already handled\nrequests can still be used by subsequent calls to\n`check_response/3`, `receive_response/3`, and `wait_response/3`.\n\nHowever, without deleting handled associations,\nthe above calls will not be able to detect when there are\nno more outstanding requests to handle, so you will have to keep track\nof this some other way than relying on a `no_request` return.\nNote that if you pass a collection only containing\nassociations of already handled or abandoned requests to\nthis function, it will always return `no_reply`.","ref":"gen_server.html#check_response/3"},{"type":"callback","title":"gen_server.code_change/3","doc":"Update the server state after code change.\n\nThis function is called by a `gen_server` process when it is to update\nits internal state during a release upgrade/downgrade, that is,\nwhen the instruction `{update, Module, Change, ...}`, is specified\nin the [`appup`](`e:sasl:appup.md`) file.\n\nFor more information, see section\n[Release Handling Instructions](`e:system:release_handling.md#instr`)\nin OTP Design Principles.\n\nFor an upgrade, `OldVsn` is `Vsn`, and for a downgrade, `OldVsn` is\n`{down,Vsn}`.  `Vsn` is defined by the `vsn` attribute(s)\nof the old version of the callback module `Module`.  If no such attribute\nis defined, the version is the checksum of the Beam file.\n\n`State` is the internal state of the `gen_server` process.\n\n`Extra` is passed \"as is\" from the `{advanced,Extra}` part\nof the update instruction.\n\nIf successful, the function must return the updated internal state.\n\nIf the function returns `{error,Reason}`,\nthe ongoing upgrade fails and rolls back to the old release.\n\n> #### Note {: .info }\n>\n> If a release upgrade/downgrade with `Change = {advanced, Extra}`\n> specified in the [`.appup`](`e:sasl:appup.md`) file is made when\n> [`Module:code_change/3`](`c:code_change/3`) is not implemented,\n> the callback call will crash with an `undef` error reason.","ref":"gen_server.html#c:code_change/3"},{"type":"function","title":"gen_server.enter_loop/3","doc":"","ref":"gen_server.html#enter_loop/3"},{"type":"function","title":"gen_server.enter_loop/4","doc":"Make the calling process become a `gen_server` process.\n\nWith argument `ServerName` equivalent to\n[`enter_loop(Module, Options,\n  State, ServerName, infinity)`](`enter_loop/5`).\n\nWith argument `How` equivalent to\n[`enter_loop(Module, Options, State, self(), How)`](`enter_loop/5`).","ref":"gen_server.html#enter_loop/4"},{"type":"function","title":"gen_server.enter_loop/5","doc":"Make the calling process become a `gen_server` process.\n\nDoes not return, instead the calling process enters the `gen_server`\nprocess receive loop and becomes a `gen_server` process.\nThe process _must_ have been started using one of the start functions\nin `m:proc_lib`.  The user is responsible for any initialization\nof the process, including registering a name for it.\n\nThis function is useful when a more complex initialization procedure\nis needed than the `gen_server` [`Module:init/1`](`c:init/1`);\ncallback provides.\n\n`Module`, `Options`, and `ServerName` have the same meanings\nas when calling [`start[_link|_monitor]/3,4`](`start_link/3`)\nor `ServerName` can be `self/0` for an anonymous server,\nwhich is the same as calling an `enter_loop/3,4` function\nwithout a `ServerName` argument.  However, if `ServerName`\nis specified (and not as `self/0`), the process must have been registered\naccordingly _before_ this function is called.\n\n`State`, `Timeout`, `Hibernate` and `Cont` have the same meanings\nas in the return value of [`Module:init/1`](`c:init/1`),\nwhich is _not_ called when `enter_loop/3,4,5` is used.  Note that\nto adhere to the [gen_server Behaviour](`e:system:gen_server_concepts.md`)\nsuch a callback function needs to be defined, and it might as well\nbe the one used when starting the `gen_server` process\nthrough `proc_lib`, and then be the one that calls `enter_loop/3,4,5`.\nBut if such a [`Module:init/1`](`c:init/1`) function,\nin for example error cases, cannot call `enter_loop/3,4,5`,\nit should return a value that follows the type specification\nfor [`Module:init/1`](`c:init/1`) such as `ignore`,\nalthough that value will be lost when returning to the spawning function.\n\nThis function fails if the calling process was not started\nby a `proc_lib` start function, or if it is not registered\naccording to `ServerName`.","ref":"gen_server.html#enter_loop/5"},{"type":"type","title":"gen_server.enter_loop_opt/0","doc":"Server start options for the [`start`](`start_link/4`) or\n[`enter_loop`](`enter_loop/5`) functions.\n\nOptions that can be used when starting a `gen_server` server through\n[`enter_loop/3-5`](`enter_loop/5`) or the start functions such as\n[`start_link/3,4`](`start_link/4`).\n\n- **`{hibernate_after, HibernateAfterTimeout}`** - Specifies that the\n  `gen_server` process awaits any message for `HibernateAfterTimeout`\n  milliseconds and if no message is received, the process goes into\n  hibernation automatically (by calling `proc_lib:hibernate/3`).\n\n- **`{debug, Dbgs}`** - For every entry in `Dbgs`,\n  the corresponding function in `m:sys` is called.","ref":"gen_server.html#t:enter_loop_opt/0"},{"type":"type","title":"gen_server.format_status/0","doc":"A map that describes the `gen_server` status.\n\nThe keys are:\n- **`state`** - The internal state of the `gen_server` process.\n- **`message`** - The message that caused the server to terminate.\n- **`reason`** - The reason that caused the server to terminate.\n- **`log`** - The [sys log](`sys:log/2`) of the server.\n\nNew associations may be added to the status map without prior notice.","ref":"gen_server.html#t:format_status/0"},{"type":"callback","title":"gen_server.format_status/1","doc":"Format/limit the status value.\n\nThis function is called by a `gen_server` process in in order to\nformat/limit the server state for debugging and logging purposes.\n\nIt is called in the following situations:\n\n- [`sys:get_status/1,2`](`sys:get_status/1`) is invoked\n  to get the `gen_server` status.\n- The `gen_server` process terminates abnormally and logs an error.\n\nThis callback is used to limit the status of the process returned by\n[`sys:get_status/1,2`](`sys:get_status/1`) or sent to `m:logger`.\n\nThe callback gets a map `Status` describing the current status\nand shall return a map `NewStatus` with the same keys,\nbut it may transform some values.\n\nTwo possible use cases for this callback is to remove\nsensitive information from the state to prevent it from being printed\nin log files, or to compact large irrelevant status items\nthat would only clutter the logs.\n\nExample:\n\n```erlang\nformat_status(Status) ->\n  maps:map(\n    fun(state,State) ->\n            maps:remove(private_key, State);\n       (message,{password, _Pass}) ->\n            {password, removed};\n       (_,Value) ->\n            Value\n    end, Status).\n```\n\n> #### Note {: .info }\n>\n> This callback is optional, so callback modules need not export it. The\n> `gen_server` module provides a default implementation\n> of this function that returns the callback module state.\n>\n> If this callback is exported but fails,\n> to hide possibly sensitive data,\n> the default function will instead return the fact that\n> [`Module:format_status/1`](`c:format_status/1`) has crashed.","ref":"gen_server.html#c:format_status/1"},{"type":"callback","title":"gen_server.format_status/2","doc":"Format/limit the status value.\n\nThis function is called by a `gen_server` process\nin in order to format/limit the server state\nfor debugging and logging purposes.\n\nIt is called in the following situations:\n\n- One of [`sys:get_status/1,2`](`sys:get_status/1`) is invoked to get the\n  `gen_server` status. `Opt` is set to the atom `normal`.\n- The `gen_server` process terminates abnormally and logs an error.\n  `Opt` is set to the atom `terminate`.\n\nThis function is useful for changing the form and appearance\nof the `gen_server` status for these cases. A callback module\nwishing to change the `sys:get_status/1,2` return value,\nas well as how its status appears in termination error logs,\nexports an instance of [`Module:format_status/2`](`c:format_status/2`)\nthat returns a term describing the current status\nof the `gen_server` process.\n\n`PDict` is the current value of the process dictionary\nof the `gen_server` process..\n\n`State` is the internal state of the `gen_server` process.\n\nThe function is to return `Status`, a term that changes the details\nof the current state and status of the `gen_server` process.\nThere are no restrictions on the form `Status` can take,\nbut for the `sys:get_status/1,2` case (when `Opt` is `normal`),\nthe recommended form for the `Status` value is\n`[{data, [{\"State\", Term}]}]`, where `Term` provides relevant details\nof the `gen_server` state.  Following this recommendation is not required,\nbut it makes the callback module status consistent with the rest of\nthe `sys:get_status/1,2` return value.\n\nOne use for this function is to return compact alternative\nstate representations to avoid that large state terms are printed\nin log files.\n\n> #### Note {: .info }\n>\n> This callback is optional, so callback modules need not export it.\n> The `gen_server` module provides a default implementation\n> of this function that returns the callback module state.","ref":"gen_server.html#c:format_status/2"},{"type":"type","title":"gen_server.from/0","doc":"A call's reply destination.\n\nDestination, given to the `gen_server` as the first argument\nto the callback function [`Module:handle_call/3`](`c:handle_call/3`),\nto be used by the when replying through `reply/2` (instead of\nthrough the callback function's return value), to the process `Client`\nthat has called the `gen_server` using [`call/2,3`](`call/2`).\n`Tag` is a term that is unique for this call/request instance.","ref":"gen_server.html#t:from/0"},{"type":"callback","title":"gen_server.handle_call/3","doc":"Handle a call.\n\nWhenever a `gen_server` process receives a request sent using\n[`call/2,3`](`call/3`), [`multi_call/2,3,4`](`multi_call/4`),\nor [`send_request/2,4`](`send_request/4`), this function is called\nto handle the request.\n\n`State` is the internal state of the `gen_server` process,\nand `NewState` a possibly updated one.\n\n`Request` is passed from the same argument provided\nto `call` or `multi_call`.\n\nThe return value `Result` is interpreted as follows:\n\n- **`{reply,Reply,NewState}`\\\n  `{reply,Reply,NewState,_}`** - The `Reply` value is sent back\n to the client request and there becomes its return value.\n\n  The `gen_server` process continues executing with the possibly updated\n  internal state `NewState`.\n\n- **`{noreply,NewState}`\\\n  `{noreply,NewState,_}`** - The `gen_server` process\n  continues executing with the possibly updated internal state `NewState`.\n\n  A reply to the client request has to be created by calling\n  [`reply(From, Reply)`](`reply/2`), either in this\n  or in a later callback.\n\n- **`{reply,_,_,Timeout}`\\\n  `{noreply,_,Timeout}`** - If an integer `Timeout` is provided,\n  a time-out occurs unless a request or a message is received\n  within that many milliseconds. A time-out is represented\n  by the atom `timeout` to be handled by the\n  [`Module:handle_info/2`](`c:handle_info/2`) callback function.\n  `Timeout =:= infinity` can be used to wait indefinitely,\n  which is the same as returning a value without a `Timeout` member.\n\n- **`{reply,_,_,hibernate}`\\\n  `{noreply,_,hibernate}`** - The process goes into hibernation,\n  by calling `proc_lib:hibernate/3`, waiting for\n  the next message to arrive\n\n- **`{reply,_,_,{continue,Continue}}`\\\n  `{noreply,_,{continue,Continue}}`** - The process will execute the\n  [`Module:handle_continue/2`](`c:handle_continue/2`) callback function,\n  with `Continue` as the first argument.\n\n- **`{stop,Reason,NewState}`\\\n  `{stop,Reason,Reply,NewState}`** - The `gen_server` process will call\n  [`Module:terminate(Reason,NewState)`](`c:terminate/2`),\n  and then terminate.\n\n  `{stop,_,Reply,_}` will create a reply to the client request just as\n  `{reply,Reply,...}` while `{stop,_,_}` will not, so just as for\n  `{noreply,NewState,...}` a reply has to be created by calling\n  [`reply(From, Reply)`](`reply/2`) before returning `{stop,_,_}`.","ref":"gen_server.html#c:handle_call/3"},{"type":"callback","title":"gen_server.handle_cast/2","doc":"Handle a cast message.\n\nWhenever a `gen_server` process receives a request sent using `cast/2`\nor [`abcast/2,3`](`abcast/2`), this function is called\nto handle the request.\n\nFor a description of the arguments and possible return values,\nsee [`Module:handle_call/3`](`c:handle_call/3`).","ref":"gen_server.html#c:handle_cast/2"},{"type":"callback","title":"gen_server.handle_continue/2","doc":"Handle a callback continuation.\n\nThis function is called by a `gen_server` process whenever\na previous callback returns one of the tuples containing\n`{continue, Continue}`.  The call is invoked immediately after\nthe previous callback, which makes it useful for performing work\nafter initialization or, for splitting the work in a callback\ninto multiple steps, updating the process state along the way.\n\nFor a description of the other arguments and possible return values,\nsee [`Module:handle_call/3`](`c:handle_call/3`).\n\n> #### Note {: .info }\n>\n> This callback is optional, so callback modules need to export it\n> only if theyreturn one of the tuples containing `{continue,Continue}`\n> from another callback.  If such a `{continue,_}` tuple is used\n> and the callback is not implemented, the process will exit\n> with `undef` error.","ref":"gen_server.html#c:handle_continue/2"},{"type":"callback","title":"gen_server.handle_info/2","doc":"Handle an info message (regular process message).\n\nThis function is called by a `gen_server` process when a time-out occurs\nor when it receives any other message than a synchronous\nor asynchronous request (or a system message).\n\n`Info` is either the atom `timeout`, if a time-out has occurred,\nor the received message.\n\nFor a description of the other arguments and possible return values,\nsee [`Module:handle_call/3`](`c:handle_call/3`).\n\n> #### Note {: .info }\n>\n> This callback is optional, so callback modules need not export it.\n> The `gen_server` module provides a default implementation\n> of this function that logs about the unexpected `Info` message,\n> drops it and returns `{noreply, State}`.","ref":"gen_server.html#c:handle_info/2"},{"type":"callback","title":"gen_server.init/1","doc":"Initialize the server.\n\nWhenever a `gen_server` process is started using [`start/3,4`](`start/3`),\n[`start_monitor/3,4`](`start_monitor/3`),\nor [`start_link/3,4`](`start_link/3`), this function is called\nby the new process to initialize the server.\n\n`Args` is the `Args` argument provided to the start function.\n\nThe return value `Result` is interpreted as follows:\n\n- **`{ok,State}`\\\n  `{ok,State,_}`** - Initialization was succesful\n   and `State` is the internal state of the `gen_server` process.\n\n- **`{ok,_,Timeout}`\\\n  `{ok,_,hibernate}`\\\n  `{ok,_,{continue,Continue}}`** - See the corresponding return values from\n  [`Module:handle_call/3`](`c:handle_call/3`) for a description\n  of this tuple member.\n\n- **`{stop,Reason}`** - Initialization failed.  The `gen_server`\n  process exits with reason `Reason`.\n\n- **`{error,Reason}` _since OTP 26.0_\\\n  `ignore`** - Initialization failed. The `gen_server` process exits\n  with reason `normal`.\n\nSee function [`start_link/3,4`](`start_link/3`)'s return value\n`t:start_ret/0` in these different cases.","ref":"gen_server.html#c:init/1"},{"type":"function","title":"gen_server.multi_call/2","doc":"Call servers on multiple nodes in parallel.\n\nEquivalent to [`multi_call(Nodes, Name, Request)`](`multi_call/3`)\nwhere `Nodes` is all nodes connected to the calling node,\nincluding the calling node itself.","ref":"gen_server.html#multi_call/2"},{"type":"function","title":"gen_server.multi_call/3","doc":"","ref":"gen_server.html#multi_call/3"},{"type":"function","title":"gen_server.multi_call/4","doc":"Call servers on multiple nodes in parallel.\n\nMakes a synchronous call to all `gen_server` processes\nlocally registered as `Name` at the specified nodes,\nby first sending the request to the nodes, and then waiting\nfor the replies. The `gen_server` processes on the nodes call\n[`Module:handle_call/3`](`c:handle_call/3`) to handle the request.\n\nThe function returns a tuple `{Replies, BadNodes}`,\nwhere `Replies` is a list of `{Node, Reply}` tuples,\nand `BadNodes` is a list of nodes that either did not exist,\nwhere `Name` was not a registered `gen_server`,\nor where it did not reply.\n\n`Nodes` is a list of node names to which the request is to be sent.\n\n`Name` is the locally registered name for each `gen_server` process.\n\n`Request` is any term that is passed as the first argument to\n[`Module:handle_call/3`](`c:handle_call/3`).\n\n`Timeout` is an integer that specifies how many milliseconds\nto wait for all replies, or the atom `infinity` to wait indefinitely.\nIf no reply is received from a node within the specified time,\nthe node is added to `BadNodes`.\n\nWhen a reply `Reply` is received from the `gen_server` process\nat a node `Node`, `{Node,Reply}` is added to `Replies`.\n`Reply` is passed from the return value of\n[`Module:handle_call/3`](`c:handle_call/3`).\n\n> #### Warning {: .warning }\n>\n> If one of the nodes cannot process monitors, for example,\n> C or Java nodes, and the `gen_server` process is not started\n> when the requests are sent, but starts within 2 seconds,\n> this function waits the whole `Timeout`, which may be infinity.\n>\n> This problem does not exist if all nodes are Erlang nodes.\n\nTo prevent late answers (after the time-out)\nfrom polluting the message queue of the caller,\na middleman process is used to do the calls.\nLate answers are then discarded when they arrive to\nthe terminated middleman process.","ref":"gen_server.html#multi_call/4"},{"type":"function","title":"gen_server.receive_response/2","doc":"Receive a request response.\n\nReceive a response corresponding to the request identifier `ReqId`.\nThe request must have been made by `send_request/2`,\nand it must have been made by the same process calling this function.\n\n`Timeout` specifies how long to wait for a response.\nIf no response is received within the specified time,\nthis function returns `timeout`.  Assuming that the\nserver executes on a node supporting aliases (introduced in OTP 24)\nthe request will also be abandoned.  That is,\nno response will be received after a time-out.\nOtherwise, a stray response might be received at a later time.\n\nThe return value `Reply` is passed from the return value of\n[`Module:handle_call/3`](`c:handle_call/3`).\n\nThe function returns an error if the `gen_server` died\nbefore a reply was sent.\n\nThe difference between `receive_response/2` and `wait_response/2`\nis that `receive_response/2` abandons the request at time-out\nso that a potential future response is ignored,\nwhile `wait_response/2` does not.","ref":"gen_server.html#receive_response/2"},{"type":"function","title":"gen_server.receive_response/3","doc":"Receive a request response in a collection.\n\nReceive a response in `ReqIdCollection`. All request identifiers\nof `ReqIdCollection` must correspond to requests that have been made\nusing `send_request/2` or `send_request/4`, and all requests\nmust have been made by the process calling this function.\n\nThe `Label` in the response is the `Label` associated with\nthe request identifier that the response corresponds to.\nThe `Label` of a request identifier is associated\nwhen [adding the request id](`reqids_add/3`) to a collection,\nor when sending the request using `send_request/4`.\n\nCompared to `receive_response/2`, the returned result or exception\nassociated with a specific request identifier will be wrapped\nin a 3-tuple `{Response, Label, NewReqIdCollection}`.\n`Response` is the value that would have been produced\nby `receive_response/2`, `Label` is the value associated with\nthe specific [request identifier](`t:request_id/0`)\nand `NewReqIdCollection` is a possibly modified\nrequest identifier collection.\n\nIf `ReqIdCollection` is empty, `no_request` will be returned.\n\n`Timeout` specifies how long to wait for a response.  If no response\nis received within the specified time, the function returns `timeout`.\nAssuming that the server executes on a node supporting aliases\n(introduced in OTP 24) all requests identified by `ReqIdCollection`\nwill also be abandoned.  That is, no responses will be received\nafter a time-out.  Otherwise, stray responses might be received\nat a later time.\n\nThe difference between `receive_response/3` and `wait_response/3`\nis that `receive_response/3` abandons the requests at time-out\nso that potential future responses are ignored,\nwhile [`wait_response/3`](`wait_response/3`) does not.\n\nIf `Delete` is `true`, the association with `Label`\nis deleted from `ReqIdCollection` in the resulting\n`NewReqIdCollection`. If `Delete` is `false`, `NewReqIdCollection`\nwill equal `ReqIdCollection`.  Note that deleting an association\nis not for free and that a collection containing already handled\nrequests can still be used by subsequent calls to\n`receive_response/3`, `check_response/3`, and `wait_response/3`.\n\nHowever, without deleting handled associations,\nthe above calls will not be able to detect when there are\nno more outstanding requests to handle, so you will have to keep track\nof this some other way than relying on a `no_request` return.\nNote that if you pass a collection only containing\nassociations of already handled or abandoned requests to\nthis function, it will always block until `Timeout` expires\nand then return `timeout`.","ref":"gen_server.html#receive_response/3"},{"type":"function","title":"gen_server.reply/2","doc":"Send a reply to a client.\n\nThis function can be used by a `gen_server` process to explicitly send\na reply to a client that called [`call/2,3`](`call/2`) or\n[`multi_call/2,3,4`](`multi_call/2`), when the reply cannot be passed\nin the return value of [`Module:handle_call/3`](`c:handle_call/3`).\n\n`Client` must be the `From` argument provided to the `c:handle_call/3`\ncallback function. `Reply` is any term passed back to the client\nas the return value of `call/2,3` or `multi_call/2,3,4`.","ref":"gen_server.html#reply/2"},{"type":"opaque","title":"gen_server.reply_tag/0","doc":"A handle that associates a reply to the corresponding request.","ref":"gen_server.html#t:reply_tag/0"},{"type":"function","title":"gen_server.reqids_add/3","doc":"Store a request identifier in a colletion.\n\nStores `ReqId` and associates a `Label` with the request identifier\nby adding this information to `ReqIdCollection` and returning\nthe resulting request identifier collection.","ref":"gen_server.html#reqids_add/3"},{"type":"function","title":"gen_server.reqids_new/0","doc":"Create an empty request identifier collection.\n\nReturns a new empty request identifier collection.\nA request identifier collection can be utilized to handle\nmultiple outstanding requests.\n\nRequest identifiers of requests made by `send_request/2`\ncan be stored in a collection using `reqids_add/3`.\nSuch a collection of request identifiers can later be used\nin order to get one response corresponding to a request\nin the collection by passing the collection as argument to\n`receive_response/3`, `wait_response/3`, or, `check_response/3`.\n\n`reqids_size/1` can be used to determine the number of\nrequest identifiers in a collection.","ref":"gen_server.html#reqids_new/0"},{"type":"function","title":"gen_server.reqids_size/1","doc":"Returns the number of request identifiers in `ReqIdCollection`.","ref":"gen_server.html#reqids_size/1"},{"type":"function","title":"gen_server.reqids_to_list/1","doc":"Convert a request identifier collection to a list.\n\nReturns a list of `{ReqId, Label}` tuples which corresponds to\nall request identifiers with their associated labels\nin [`ReqIdCollection`](`t:request_id_collection/0`).","ref":"gen_server.html#reqids_to_list/1"},{"type":"opaque","title":"gen_server.request_id/0","doc":"An opaque request identifier. See `send_request/2` for details.","ref":"gen_server.html#t:request_id/0"},{"type":"opaque","title":"gen_server.request_id_collection/0","doc":"An opaque collection of request identifiers (`t:request_id/0`).\n\nEach request identifier can be associated with a label\nchosen by the user.  For more information see `reqids_new/0`.","ref":"gen_server.html#t:request_id_collection/0"},{"type":"type","title":"gen_server.response_timeout/0","doc":"Response time-out for an asynchronous call.\n\nUsed to set a time limit on how long to wait for a response using either\n`receive_response/2`, `receive_response/3`, `wait_response/2`, or\n`wait_response/3`. The time unit used is `millisecond`.\n\nCurrently valid values:\n\n- **`0..4294967295`** - Time-out relative to current time in milliseconds.\n\n- **`infinity`** - Infinite time-out. That is,\n  the operation will never time out.\n\n- **`{abs, Timeout}`** - An absolute\n  [Erlang monotonic time](`erlang:monotonic_time/1`)\n  time-out in milliseconds. That is, the operation will time out when\n  [`erlang:monotonic_time(millisecond)`](`erlang:monotonic_time/1`)\n  returns a value larger than or equal to `Timeout`.\n  `Timeout` is not allowed to identify a time further into the future\n  than `4294967295` milliseconds.  Specifying the time-out\n  using an absolute value is especially handy when you have\n  a deadline for responses corresponding to a complete collection\n  of requests (`t:request_id_collection/0`), since you do not have to\n  recalculate the relative time until the deadline over and over again.","ref":"gen_server.html#t:response_timeout/0"},{"type":"function","title":"gen_server.send_request/2","doc":"Send an asynchronous `call` request.\n\nSends `Request` to the `gen_server` process identified by `ServerRef`\nand returns a request identifier `ReqId`.\n\nThe return value `ReqId` shall later be used with `receive_response/2`,\n`wait_response/2`, or `check_response/2` to fetch the actual result\nof the request.  Besides passing the request identifier directly\nto these functions, it can also be stored in\na request identifier collection using `reqids_add/3`.\nSuch a collection of request identifiers can later be used\nin order to get one response corresponding to a\nrequest in the collection by passing the collection\nas argument to `receive_response/3`, `wait_response/3`,\nor `check_response/3`.  If you are about to store the request identifier\nin a collection, you may want to consider using `send_request/4` instead.\n\nThe call\n`gen_server:receive_response(gen_server:send_request(ServerRef, Request), Timeout)`\ncan be seen as equivalent to\n[`gen_server:call(ServerRef, Request, Timeout)`](`call/3`),\nignoring the error handling.\n\nThe `gen_server` process calls [`Module:handle_call/3`](`c:handle_call/3`) to\nhandle the request.\n\nSee the type `t:server_ref/0` for the possible values for `ServerRef`.\n\n`Request` is any term that is passed as the first argument to\n[`Module:handle_call/3`](`c:handle_call/3`).","ref":"gen_server.html#send_request/2"},{"type":"function","title":"gen_server.send_request/4","doc":"Send an asynchronous `call` request and add it\nto a request identifier collection.\n\nSends `Request` to the `gen_server` process identified by `ServerRef`.\nThe `Label` will be associated with the request identifier\nof the operation and added to the returned request identifier collection\n`NewReqIdCollection`.  The collection can later be used in order to\nget one response corresponding to a request in the collection\nby passing the collection as argument to `receive_response/3`,\n`wait_response/3`, or `check_response/3`.\n\nThe same as calling\n[`reqids_add`](`reqids_add/3`)`(`[`send_request`](`send_request/2`)`(ServerRef, Request), Label, ReqIdCollection)`,\nbut slightly more efficient.","ref":"gen_server.html#send_request/4"},{"type":"type","title":"gen_server.server_name/0","doc":"Server name specification: `local`, `global`, or `via` registered.\n\nTo be used when starting a `gen_server`.  See functions\n[`start/3,4`](`start/3`),\n[`start_link/3,4`](`start_link/3`),\n[`start_monitor/3,4`](`start_monitor/3`),\n[`enter_loop/3,4,5`](`enter_loop/3`), and the type `t:server_ref/0`.\n\n- **`{local, LocalName}`** - Register the `gen_server` locally\n  as `LocalName` using [`register/2`](`erlang:register/2`).\n\n- **`{global, GlobalName}`** - Register the `gen_server` process id\n  globally as `GlobalName` using `global:register_name/2`.\n\n- **`{via, RegMod, ViaName}`** - Register the `gen_server` process\n  with the registry represented by `RegMod`. The `RegMod` callback\n  is to export the functions `register_name/2`, `unregister_name/1`,\n  `whereis_name/1`, and `send/2`, which are to behave like\n  the corresponding functions in `m:global`.\n  Thus, `{via, global, GlobalName}` is a valid reference\n  equivalent to `{global, GlobalName}`.","ref":"gen_server.html#t:server_name/0"},{"type":"type","title":"gen_server.server_ref/0","doc":"Server specification: `t:pid/0` or registered `t:server_name/0`.\n\nTo be used when addressing a `gen_server`.  See [`call/2,3`](`call/2`),\n`cast/2`, `send_request/2`, `check_response/2`, `wait_response/2`,\n[`stop/2,3`](`stop/1`) and the type `t:server_name/0`.\n\nIt can be:\n\n- **`t:pid/0`** - The `gen_server`'s process identifier.\n\n- **`LocalName`** - The `gen_server` is locally registered\n  as `LocalName` with [`register/2`](`erlang:register/2`).\n\n- **`{Name,Node}`** - The `gen_server` is locally registered\n  on another node.\n\n- **`{global, GlobalName}`** - The `gen_server` is globally registered\n  in `m:global`.\n\n- **`{via, RegMod, ViaName}`** - The `gen_server` is registered\n  in an alternative process registry.  See the same term\n  described for `t:server_name/0`.","ref":"gen_server.html#t:server_ref/0"},{"type":"function","title":"gen_server.start/3","doc":"Start a server, neither linked nor registered.\n\nEquivalent to `start/4` except that the `gen_server` process is not\nregistered with any [name service](`t:server_name/0`).","ref":"gen_server.html#start/3"},{"type":"function","title":"gen_server.start/4","doc":"Start a server, neither linked nor registered.\n\nCreates a standalone `gen_server` process, that is,\na `gen_server` process that is not part of a supervision tree,\nand thus has no supervisor.\n\nOther than that see `start_link/4`.","ref":"gen_server.html#start/4"},{"type":"function","title":"gen_server.start_link/3","doc":"Start a server, linked but not registered.\n\nEquivalent to `start_link/4` except that the `gen_server` process is\nnot registered with any [name service](`t:server_name/0`).","ref":"gen_server.html#start_link/3"},{"type":"function","title":"gen_server.start_link/4","doc":"Start a server, linked but not registered.\n\nCreates a `gen_server` process as part of a supervision tree.\nThis function is to be called, directly or indirectly, by the supervisor.\nFor example, it ensures that the `gen_server` process is spawned\nas linked to the caller (supervisor).\n\nThe `gen_server` process calls [`Module:init/1`](`c:init/1`)\nto initialize.  To ensure a synchronized startup procedure,\n`start_link/3,4` does not return until [`Module:init/1`](`c:init/1`)\nhas returned or failed.\n\n[`ServerName`](`t:server_name/0`) specifies with what name\nand now to register the server name.  See type `t:server_name/0`\nfor different name registrations.\n\n`Module` is the name of the callback module.\n\n`Args` is any term that is passed as the argument to\n[`Module:init/1`](`c:init/1`).\n\nSee type `t:start_opt/0` for `Options` for starting\nthe `gen_server` process.\n\nSee type `t:start_ret/0` for a description this function's return values.\n\nIf `start_link/3,4` returns `ignore` or `{error, _}`,\nthe started `gen_server` process has terminated.  If an `'EXIT'` message\nwas delivered to the calling process (due to the process link),\nthat message has been consumed.\n\n> #### Warning {: .warning }\n>\n> Before OTP 26.0, if the started `gen_server` process returned e.g.\n> `{stop, Reason}` from [`Module:init/1`](`c:init/1`), this function\n> could return `{error, Reason}` _before_ the started `m:gen_server` process\n> had terminated so starting again might fail because VM resources\n> such as the registered name was not yet unregistered. An `'EXIT'` message\n> could arrive later to the process calling this function.\n>\n> But if the started `gen_server` process instead failed during\n> [`Module:init/1`](`c:init/1`), a process link `{'EXIT', Pid, Reason}`\n> message caused this function to return `{error, Reason}`,\n> so the `'EXIT'` message had been consumed and the started\n> `m:gen_server` process had terminated.\n>\n> Since it was impossible to tell the difference between these two cases\n> from `start_link/3,4`'s return value, this inconsistency was cleaned up\n> in OTP 26.0.\n\nThe difference between returning `{stop, _}` and `{error, _}` from\n[`Module:init/1`](`c:init/1`), is that `{error, _}` results in a graceful\n(\"silent\") termination since the `gen_server` process exits\nwith reason `normal`.","ref":"gen_server.html#start_link/4"},{"type":"type","title":"gen_server.start_mon_ret/0","doc":"Return value from the [`start_monitor/3,4`](`start_monitor/3`) functions.\n\nThe same as type `t:start_ret/0` except that for a succesful start\nit returns both the process identifier `Pid`\nand a [`monitor/2,3`](`erlang:monitor/2`) [`MonRef`](`t:reference/0`).","ref":"gen_server.html#t:start_mon_ret/0"},{"type":"function","title":"gen_server.start_monitor/3","doc":"Start a server, monitored but neither linked nor registered.\n\nEquivalent to `start_monitor/4` except that the `gen_server` process\nis not registered with any [name service](`t:server_name/0`).","ref":"gen_server.html#start_monitor/3"},{"type":"function","title":"gen_server.start_monitor/4","doc":"Start a server, monitored and registered, but not linked.\n\nCreates a standalone `gen_server` process, that is,\na `gen_server` process that is not part of a supervision tree\n(and thus has no supervisor) and atomically sets up a monitor\nto the newly created server.\n\nOther than that see [`start_link/3,4`](`start_link/3`).\nNote that the return value for a successful start differs in that\nit returns a monitor `reference`.  See type `t:start_mon_ret/0`.\n\nIf the start is not successful, the caller will be blocked\nuntil the monitor's `'DOWN'` message has been received\nand removed from the message queue.","ref":"gen_server.html#start_monitor/4"},{"type":"type","title":"gen_server.start_opt/0","doc":"Server start options for the [`start` functions](`start_link/3`).\n\nOptions that can be used when starting a `gen_server` server through,\nfor example, [`start_link/3,4`](`start_link/4`).\n\n- **`{timeout, Timeout}`** - How many milliseconds\n  the `gen_server` process is allowed to spend initializing\n  or it is terminated and the start function returns `{error, timeout}`.\n\n- **`{spawn_opt, SpawnOptions}`** - The `SpawnOptions` option list\n  is passed to the function used to spawn the `gen_server`;\n  see `t:proc_lib:start_spawn_option/0`).\n\n  > #### Note {: .info }\n  >\n  > Using spawn option `monitor` is not allowed -\n  > it causes a `badarg` failure.\n\n- **`t:enter_loop_opt/0`** - See the type `t:enter_loop_opt/0`\n  below for more start options that are also allowed\n  by [`enter_loop/3,4,5`](`enter_loop/3`).","ref":"gen_server.html#t:start_opt/0"},{"type":"type","title":"gen_server.start_ret/0","doc":"Return value from the [`start/3,4`](`start/3`) and\n[`start_link/3,4`](`start_link/3`) functions.\n\n- **`{ok, Pid}`** - The `gen_server` process was succesfully created and\n  initialized, with the process identifier `Pid`.\n\n- **`{error, {already_started, OtherPid}}`** - A process with the specified\n  `ServerName` exists already with the process identifier `OtherPid`.\n  This function failed to start a `gen_server`.  It exited with reason\n  `normal` before calling [`Module:init/1`](`c:init/1`).\n\n- **`{error, timeout}`** - The `gen_server` process failed to initialize\n  since [`Module:init/1`](`c:init/1`) did not return within the\n  [start time-out](`t:start_opt/0`). The `gen_server` process was killed\n  with [`exit(_, kill)`](`erlang:exit/2`).\n\n- **`ignore`** - The `gen_server` process failed to initialize since\n  [`Module:init/1`](`c:init/1`) returned `ignore`.\n\n- **`{error,Reason}`** - The `gen_server` process failed to initialize since\n  [`Module:init/1`](`c:init/1`) returned `{stop,Reason}`, `{error,Reason}`,\n  or it failed with reason `Reason`.\n\nSee [`Module:init/1`](`c:init/1`) about the exit reason\nfor the `gen_server` process when it fails to initialize.","ref":"gen_server.html#t:start_ret/0"},{"type":"function","title":"gen_server.stop/1","doc":"","ref":"gen_server.html#stop/1"},{"type":"function","title":"gen_server.stop/3","doc":"Stop a server.\n\nOrders the generic server specified by `ServerRef` to exit\nwith the specified `Reason` and waits for it to terminate.\nThe `gen_server` process calls [`Module:terminate/2`](`c:terminate/2`)\nbefore exiting.\n\nThe function returns `ok` if the server terminates\nwith the expected reason. Any other reason than `normal`, `shutdown`,\nor `{shutdown,Term}` causes an error report to be issued using `m:logger`.\nAn exit signal with the same reason is sent to linked processes and ports.\n\n`Timeout` is an integer that specifies how many milliseconds to wait\nfor the server to terminate, or the atom `infinity` to wait indefinitely.\nIf the server has not terminated within the specified time,\nthe call exits the calling process with reason `timeout`.\n\nIf the process does not exist, the call exits the calling process\nwith reason `noproc`, or with reason `{nodedown,Node}`\nif the connection fails to the remote `Node` where the server runs.","ref":"gen_server.html#stop/3"},{"type":"callback","title":"gen_server.terminate/2","doc":"Handle server termination.\n\nThis function is called by a `gen_server` process\nwhen it is about to terminate.\n\nIt is to be the opposite of [`Module:init/1`](`c:init/1`)\nand do any necessary cleaning up.  When it returns,\nthe `gen_server` process terminates with `Reason`.\nThe return value is ignored.\n\n`Reason` is a term denoting the stop reason and `State`\nis the internal state of the `gen_server` process.\n\n`Reason` depends on why the `gen_server` process is terminating.\nIf it is because another callback function has returned a stop tuple\n`{stop,..}`, `Reason` has the value specified in that tuple.\nIf it is because of a failure, `Reason` is the error reason.\n\nIf the `gen_server` process is part of a supervision tree\nand is ordered by its supervisor to terminate, this function is called\nwith `Reason=shutdown` if the following conditions apply:\n\n- The `gen_server` process has been set to trap exit signals.\n- The shutdown strategy as defined in the child specification\n  of the supervisor is an integer time-out value, not `brutal_kill`.\n\nEven if the `gen_server` process is _not_ part of a supervision tree,\nthis function is called if it receives an `'EXIT'` message from its parent.\n`Reason` is the same as in the `'EXIT'` message.\n\nIf the `gen_server` process does not trap exits,\nthe `gen_server` process terminates immediately.\n\nNotice that for any other reason than `normal`, `shutdown`, or\n`{shutdown,Term}`, see `stop/3`, the `gen_server` process is assumed\nto terminate because of an error, and an error report is issued\nusing `m:logger`.\n\nWhen the gen_server process exits, an exit signal with the same reason\nis sent to linked processes and ports.\n\n> #### Note {: .info }\n>\n> This callback is optional, so callback modules need not export it.\n> The `gen_server` module provides a default implementation\n> with no cleanup.","ref":"gen_server.html#c:terminate/2"},{"type":"function","title":"gen_server.wait_response/2","doc":"Wait for a request response.\n\nWait for the response to the request identifier `ReqId`. The request\nmust have been made by `send_request/2`, and it must have been made\nby the same process calling this function.\n\n`WaitTime` specifies how long to wait for a reply.\nIf no reply is received within the specified time,\nthe function returns `timeout` and no cleanup is done.\nThus the function can be invoked repeatedly until a reply is returned.\n\nThe return value `Reply` is passed from the return value of\n[`Module:handle_call/3`](`c:handle_call/3`).\n\nThe function returns an error if the `gen_server`\ndied before a reply was sent.\n\nThe difference between `receive_response/2` and\n`wait_response/2` is that `receive_response/2` abandons\nthe request at time-out so that a potential future response is ignored,\nwhile [`wait_response/2`](`wait_response/2`) does not.","ref":"gen_server.html#wait_response/2"},{"type":"function","title":"gen_server.wait_response/3","doc":"Wait for any request response in a collection.\n\nWait for a response in a `ReqIdCollection`.  All request identifiers\nof `ReqIdCollection` must correspond to requests that have been made\nusing `send_request/2` or `send_request/4`, and all requests\nmust have been made by the process calling this function.\n\nThe `Label` in the response is the `Label` associated with\nthe request identifier that the response corresponds to.\nThe `Label` of a request identifier is associated\nwhen [adding the request id](`reqids_add/3`) to a collection,\nor when sending the request using `send_request/4`.\n\nCompared to `wait_response/2`, the returned result or exception\nassociated with a specific request identifier will be wrapped\nin a 3-tuple `{Response, Label, NewReqIdCollection}`.\n`Response` is the value that would have been produced\nby `wait_response/2`, `Label` is the value associated with\nthe specific [request identifier](`t:request_id/0`)\nand `NewReqIdCollection` is a possibly modified\nrequest identifier collection.\n\nIf `ReqIdCollection` is empty, `no_request` will be returned.\n\nIf no response is received before `WaitTime` has expired,\n`timeout` is returned.  It is valid to continue waiting\nfor a response as many times as needed up until a response\nhas been received and completed by `check_response()`,\n`receive_response()`, or `wait_response()`.\n\nThe difference between `receive_response/3` and `wait_response/3`\nis that `receive_response/3` abandons requests at time-out\nso that potential future responses are ignored, while\n`wait_response/3` does not.\n\nIf `Delete` is `true`, the association with `Label`\nhas been deleted from `ReqIdCollection` in the resulting\n`NewReqIdCollection`.  If `Delete` is `false`, `NewReqIdCollection`\nwill equal `ReqIdCollection`.  Note that deleting an association\nis not for free and that a collection containing already handled\nrequests can still be used by subsequent calls to\n`wait_response/3`, `check_response/3`, and `receive_response/3`.\n\nHowever, without deleting handled associations, the above\ncalls will not be able to detect when there are\nno more outstanding requests to handle, so you will have to keep track\nof this some other way than relying on a `no_request` return.\nNote that if you pass a collection only containing\nassociations of already handled or abandoned requests\nto this function, it will always block until `WaitTime` expires\nand then return `timeout`.","ref":"gen_server.html#wait_response/3"},{"type":"behaviour","title":"gen_statem","doc":"Generic state machine behavior.\n\n`gen_statem` provides a generic state machine behaviour\nthat since Erlang/OTP 20.0 replaces its predecessor `m:gen_fsm`,\nand should be used for new code.  The `gen_fsm` behaviour\nremains in OTP \"as is\" to not break old code using it.\n\nA generic state machine server process (`gen_statem`) implemented\nusing this module has a standard set of interface functions\nand includes functionality for tracing and error reporting.\nIt also fits into an OTP supervision tree.  For more information,\nsee [OTP Design Principles](`e:system:statem.md`).\n\n> #### Note {: .info }\n>\n> If you are new to `gen_statem` and want an overview\n> of concepts and operation the section\n> [`gen_statem` Behaviour](`e:system:statem.md`) located in\n> the User's Guide [OTP Design Principles](`e:system:index.html`)\n> is recommended to read.  This reference manual focuses on\n> being correct and complete, which might make it hard to see\n> the forest for all the trees.\n\n#### Features\n\n`gen_statem` has got the same features that `m:gen_fsm` had\nand adds some really useful:\n\n- [Co-located state code](#state_functions)\n- [Arbitrary term state](#handle_event_function)\n- [Event postponing](#event-postponing)\n- [Self-generated events](#event-insertion)\n- [State time-out](`t:state_timeout/0`)\n- [Multiple generic named time-outs](`t:generic_timeout/0`)\n- [Absolute time-out time](`t:timeout_option/0`)\n- [Automatic state enter calls](#state-enter-calls)\n- [Reply from other state than the request](#reply-to-a-call),\n  traceable with `m:sys`\n- [Multiple replies](#reply-to-a-call), traceable with `m:sys`\n- [Changing the callback module](#change_callback_module)\n\nTwo [_callback modes_](`t:callback_mode/0`) are supported:\n\n- `state_functions` - for finite-state machines (`m:gen_fsm` like),\n  which requires the state to be an atom and uses that state\n  as the name of the current callback function, arity 3.\n- `handle_event_function` - that allows the state to be any term\n  and that uses `c:handle_event/4` as callback function for all states.\n\nThe callback modes for `gen_statem` differs from the one for\n`gen_fsm`, but it is still fairly easy to\nrewrite from `gen_fsm` to `gen_statem`.  See the\n[rewrite guide](`m:gen_fsm#module-migration-to-gen_statem`)\nat the start of the `m:gen_fsm` documentation.\n\n#### Callback module\n\nA `gen_statem` assumes all specific parts to be located\nin a callback module exporting a predefined set of functions.\nThe relationship between the behavior functions\nand the callback functions is as follows:\n\n```\ngen_statem module            Callback module\n-----------------            ---------------\ngen_statem:start\ngen_statem:start_monitor\ngen_statem:start_link -----> Module:init/1\n\nServer start or code change\n                      -----> Module:callback_mode/0\n                      selects callback mode\n\ngen_statem:stop\nSupervisor exit\nCallback failure      -----> Module:terminate/3\n\ngen_statem:call\ngen_statem:cast\ngen_statem:send_request\nerlang:send\nerlang:'!'            -----> Module:StateName/3\n                   or -----> Module:handle_event/4\n                   depending on callback mode\n\nRelease upgrade/downgrade\n(code change)\n                      -----> Module:code_change/4\n```\n\n#### State callback {: #state-callback }\n\nThe _state callback_ for a specific [state](`t:state/0`) in a `gen_statem`\nis the callback function that is called for all events in this state.\nIt is selected depending on which [_callback mode_](`t:callback_mode/0`)\nthat the callback module defines with the callback function\n[`Module:callback_mode/0`](`c:callback_mode/0`).\n\n[](){: #state_functions }\nWhen the [_callback mode_](`t:callback_mode/0`) is `state_functions`,\nthe state must be an atom and is used as the _state callback_ name;\nsee [`Module:StateName/3`](`c:'StateName'/3`).  This co-locates all code\nfor a specific state in one function as the `gen_statem` engine branches\ndepending on state name.  Note the fact that the callback function\n[`Module:terminate/3`](`c:terminate/3`) makes the state name `terminate`\nunusable in this mode.\n\n[](){: #handle_event_function }\nWhen the [_callback mode_](`t:callback_mode/0`) is `handle_event_function`,\nthe state can be any term and the _state callback_ name is\n[`Module:handle_event/4`](`c:handle_event/4`). This makes it easy\nto branch depending on state or event as you desire. Be careful about\nwhich events you handle in which states so that you do not accidentally\npostpone an event forever creating an infinite busy loop.\n\n#### Event types\n\nEvents are of different [types](`t:event_type/0`),\ntherefore the callback functions can know the origin of an event\nwhen handling it.  [External events](`t:external_event_type/0`) are\n`call`,  `cast`, and  `info`. Internal events are\n[`timeout`](`t:timeout_event_type/0`) and `internal`.\n\n#### Event handling\n\nWhen `gen_statem` receives a process message it is transformed\ninto an event and the [_state callback_](#state-callback)\nis called with the event as two arguments: type and content. When the\n[_state callback_](#state-callback) has processed the event\nit returns to `gen_statem` which does a _state transition_. If this\n_state transition_ is to a different state, that is: `NextState =/= State`,\nit is a _state change_.\n\n#### Transition actions\n\nThe [_state callback_](#state-callback) may return\n[_transition actions_](`t:action/0`) for `gen_statem` to execute\nduring the _state transition_, for example to set a time-out\nor reply to a call.\n\n#### Reply to a call {: #reply-to-a-call }\n\nSee [`gen_statem:call/2,3`](#call-reply) about how to reply\nto a call.  A reply can be sent from any _state callback_,\nnot just the one that got the request event.\n\n#### Event postponing {: #event-postponing }\n\nOne of the possible _transition actions_ is to postpone the current event.\nThen it will not be handled in the current state.  The `gen_statem` engine\nkeeps a queue of events divided into postponed events and\nevents still to process (not presented yet).  After a _state change_\nthe queue restarts with the postponed events.\n\nThe `gen_statem` event queue model is sufficient to emulate\nthe normal process message queue with selective receive.\nPostponing an event corresponds to not matching it\nin a receive statement, and changing states corresponds to\nentering a new receive statement.\n\n#### Event insertion {: #event-insertion }\n\nThe [_state callback_](#state-callback) can insert\nevents using the [_transition action_](`t:action/0`) `next_event`,\nand such an event is inserted in the event queue as the next to call the\n[_state callback_](#state-callback) with. That is,\nas if it is the oldest incoming event. A dedicated `t:event_type/0`\n`internal` can be used for such events making it possible to\nsafely distinguish them from external events.\n\nInserting an event replaces the trick of calling your own state handling\nfunctions that you often would have to resort to in, for example,\n`m:gen_fsm` to force processing an inserted event before others.\n\n> #### Note {: .info }\n>\n> If you postpone an event and (against good practice) directly call\n> a different _state callback_, the postponed event is not retried,\n> since there was no _state change_.\n>\n> Instead of directly calling a _state callback_, do a _state change_.\n> This makes the `gen_statem` engine retry postponed events.\n>\n> Inserting an event in a _state change_ also triggers\n> the new  _state callback_ to be called with that event\n> before receiving any external events.\n\n#### State enter calls {: #state-enter-calls }\n\nThe `gen_statem` engine can automatically make a special call to the\n[_state callback_](#state-callback) whenever a new state is\nentered; see `t:state_enter/0`. This is for writing code common\nto all state entries.  Another way to do it is to explicitly insert\nan event at the _state transition_, and/or to use a dedicated\n_state transition_ function, but that is something you will have to\nremember at every _state transition_ to the state(s) that need it.\n\nFor the details of a _state transition_, see type `t:transition_option/0`.\n\n#### Hibernation\n\nThe `gen_statem` process can go into hibernation;\nsee `proc_lib:hibernate/3`. It is done when\na [_state callback_](#state-callback) or\n[`Module:init/1`](`c:init/1`) specifies `hibernate`\nin the returned [`Actions`](`t:enter_action/0`) list. This feature\ncan be useful to reclaim process heap memory while the server\nis expected to be idle for a long time. However, use it with care,\nas hibernation can be too costly to use after every event;\nsee `erlang:hibernate/3`.\n\nThere is also a server start option\n[`{hibernate_after, Timeout}`](`t:enter_loop_opt/0`)\nfor [`start/3,4`](`start/3`), [`start_link/3,4`](`start_link/3`),\n[`start_monitor/3,4`](`start_monitor/3`),\nor [`enter_loop/4,5,6`](`enter_loop/6`), that may be used\nto automatically hibernate the server.\n\n#### Callback failure\n\nIf a callback function fails or returns a bad value,\nthe `gen_statem` terminates.  However, an exception of class\n[`throw`](`erlang:throw/1`) is not regarded as an error\nbut as a valid return, from all callback functions.\n\n#### System messages and the `m:sys` module\n\nA `gen_statem` handles system messages as described in `m:sys`.\nThe `m:sys` module can be used for debugging a `gen_statem`.\nReplies sent through [_transition actions_](`t:action/0`)\ngets logged, but not replies sent through [`reply/1,2`](`reply/2`).\n\n#### Trapping exit\n\nA `gen_statem` process, like all `gen_`\\* behaviours,\ndoes not trap exit signals automatically;\nthis must be explicitly initiated in the callback module\n(by calling [`process_flag(trap_exit, true)`](`erlang:process_flag/2`)\npreferably from `c:init/1`.\n\n#### Server termination\n\nIf the `gen_statem` process terminates, e.g. as a result\nof a callback function returning `{stop, Reason}`, an exit signal\nwith this `Reason` is sent to linked processes and ports.\nSee [Processes](`e:system:ref_man_processes.md#errors`)\nin the Reference Manual for details regarding error handling\nusing exit signals.\n\n> #### Note {: .info }\n>\n> For some important information about distributed signals, see the\n> [_Blocking Signaling Over Distribution_\n> ](`e:system:ref_man_processes.md#blocking-signaling-over-distribution`)\n> section in the _Processes_ chapter of the _Erlang Reference Manual_.\n> Blocking signaling can, for example, cause call time-outs in `gen_statem`\n> to be significantly delayed.\n\n#### Bad argument\n\nUnless otherwise stated, all functions in this module fail if the specified\n`gen_statem` does not exist or if bad arguments are specified.","ref":"gen_statem.html"},{"type":"behaviour","title":"Example - gen_statem","doc":"The following example shows a simple pushbutton model\nfor a toggling pushbutton implemented with\n[_callback mode_](`t:callback_mode/0`) `state_functions`.\nYou can push the button and it replies if it went on or off,\nand you can ask for a count of how many times it has been pushed\nto switch on.","ref":"gen_statem.html#module-example"},{"type":"behaviour","title":"Pushbutton State Diagram - gen_statem","doc":"```mermaid\n---\ntitle: Pushbutton State Diagram\n---\nstateDiagram-v2\n    [*]  --> off\n    off  --> on  : push\\n* Increment count\\n* Reply 'on'\n    on   --> off : push\\n* Reply 'off'\n```\n\nNot shown in the state diagram:\n* The API function `push()` generates an event `push` of type `call`.\n* The API function `get_count()` generates an event `get_count`\n  of type `call` that is handled in all states by replying with\n  the current count value.\n* Unknown events are ignored and discarded.\n* There is boilerplate code for start, stop, terminate, code change,\n  init, to set the callback mode to `state_functions`, etc...","ref":"gen_statem.html#module-pushbutton-state-diagram"},{"type":"behaviour","title":"Pushbutton Code - gen_statem","doc":"The following is the complete callback module file `pushbutton.erl`:\n\n```erlang\n-module(pushbutton).\n-behaviour(gen_statem).\n\n-export([start/0,push/0,get_count/0,stop/0]).\n-export([terminate/3,code_change/4,init/1,callback_mode/0]).\n-export([on/3,off/3]).\n\nname() -> pushbutton_statem. % The registered server name\n\n%% API.  This example uses a registered name name()\n%% and does not link to the caller.\nstart() ->\n    gen_statem:start({local,name()}, ?MODULE, [], []).\npush() ->\n    gen_statem:call(name(), push).\nget_count() ->\n    gen_statem:call(name(), get_count).\nstop() ->\n    gen_statem:stop(name()).\n\n%% Mandatory callback functions\nterminate(_Reason, _State, _Data) ->\n    void.\ncode_change(_Vsn, State, Data, _Extra) ->\n    {ok,State,Data}.\ninit([]) ->\n    %% Set the initial state + data.  Data is used only as a counter.\n    State = off, Data = 0,\n    {ok,State,Data}.\ncallback_mode() -> state_functions.\n\n%%% state callback(s)\n\noff({call,From}, push, Data) ->\n    %% Go to 'on', increment count and reply\n    %% that the resulting status is 'on'\n    {next_state,on,Data+1,[{reply,From,on}]};\noff(EventType, EventContent, Data) ->\n    handle_event(EventType, EventContent, Data).\n\non({call,From}, push, Data) ->\n    %% Go to 'off' and reply that the resulting status is 'off'\n    {next_state,off,Data,[{reply,From,off}]};\non(EventType, EventContent, Data) ->\n    handle_event(EventType, EventContent, Data).\n\n%% Handle events common to all states\nhandle_event({call,From}, get_count, Data) ->\n    %% Reply with the current count\n    {keep_state,Data,[{reply,From,Data}]};\nhandle_event(_, _, Data) ->\n    %% Ignore all other events\n    {keep_state,Data}.\n```\n\nThe following is a shell session when running it:\n\n```erlang\n1> pushbutton:start().\n{ok,<0.36.0>}\n2> pushbutton:get_count().\n0\n3> pushbutton:push().\non\n4> pushbutton:get_count().\n1\n5> pushbutton:push().\noff\n6> pushbutton:get_count().\n1\n7> pushbutton:stop().\nok\n8> pushbutton:push().\n** exception exit: {noproc,{gen_statem,call,[pushbutton_statem,push,infinity]}}\n     in function  gen:do_for_proc/2 (gen.erl, line 261)\n     in call from gen_statem:call/3 (gen_statem.erl, line 386)\n```\n\nTo compare styles, here follows the same example using\n[_callback mode_](`t:callback_mode/0`) `handle_event_function`,\nor rather, the code to replace after function [`init/1`](`c:init/1`)\nof the `pushbutton.erl` example file above:\n\n```erlang\ncallback_mode() -> handle_event_function.\n\n%%% state callback(s)\n\nhandle_event({call,From}, push, off, Data) ->\n    %% Go to 'on', increment count and reply\n    %% that the resulting status is 'on'\n    {next_state,on,Data+1,[{reply,From,on}]};\nhandle_event({call,From}, push, on, Data) ->\n    %% Go to 'off' and reply that the resulting status is 'off'\n    {next_state,off,Data,[{reply,From,off}]};\n%%\n%% Event handling common to all states\nhandle_event({call,From}, get_count, State, Data) ->\n    %% Reply with the current count\n    {next_state,State,Data,[{reply,From,Data}]};\nhandle_event(_, _, State, Data) ->\n    %% Ignore all other events\n    {next_state,State,Data}.\n```\n\n> #### Note {: .info }\n>","ref":"gen_statem.html#module-pushbutton-code"},{"type":"behaviour","title":"API changes - gen_statem","doc":"> - This behavior appeared in Erlang/OTP 19.0 as experimental.\n> - In OTP 19.1 a backwards incompatible change of the return tuple from\n>   [`Module:init/1`](`c:init/1`) was made,\n>   the mandatory callback function\n>   [`Module:callback_mode/0`](`c:callback_mode/0`) was introduced,\n>   and `enter_loop/4` was added.\n> - In OTP 19.2 [_state enter calls_](`t:state_enter/0`) were added.\n> - In OTP 19.3 [state time-outs](`t:state_timeout/0`) were added.\n> - In OTP 20.0 [generic time-outs](`t:generic_timeout/0`) were added\n>   and `gen_statem` was stated to be no longer experimental and\n>   preferred over `gen_fsm`.\n> - In OTP 22.1 time-out content [`update`](`t:timeout_update_action/0`)\n>   and explicit time-out [`cancel`](`t:timeout_cancel_action/0`)\n>   were added.\n> - In OTP 22.3 the possibility to change the callback module with actions\n>   [`change_callback_module`](#change_callback_module),\n>   [`push_callback_module`](#push_callback_module) and\n>   [`pop_callback_module`](#pop_callback_module), was added.\n> - In OTP 23.0 [`start_monitor/3,4`](`start_monitor/3`) were added,\n>   as well as functions for asynchronous calls: `send_request/2`,\n>   [`wait_response/1,2`](`wait_response/2`), and `check_response/2`.\n> - In OTP 24.0 [`receive_response/1,2`](`receive_response/2`) were added.\n> - In OTP 25.0 [`Module:format_status/1`](`c:format_status/1`)\n>   was added to replace [`Module:format_status/1`](`c:format_status/1`),\n>   as well as functions for collections of asynchronous calls:\n>   `send_request/4`, `wait_response/3`, `receive_response/3`,\n>   `check_response/3`, `reqids_new/0`, `reqids_size/1`,\n>   `reqids_add/3`, `reqids_to_list/1`.\n> - In OTP 26.0 the possibility to return `{error, Reason}` from\n>   [`Module:init/1`](`c:init/1`) was added.\n> - In OTP 27.0 [`Module:format_status/1`](`c:format_status/1`)\n>   was deprecated.","ref":"gen_statem.html#module-api-changes"},{"type":"behaviour","title":"See Also - gen_statem","doc":"`m:gen_event`, `m:gen_fsm`, `m:gen_server`, `m:proc_lib`, `m:supervisor`,\n`m:sys`.","ref":"gen_statem.html#module-see-also"},{"type":"type","title":"gen_statem.action/0","doc":"Actions for a _state transition_, or when starting the server.\n\nThese _transition actions_ can be invoked by returning them from the\n[_state callback_](#state-callback) when it is called\nwith an [event](`t:event_type/0`), from [`Module:init/1`](`c:init/1`)\nor by passing them to [`enter_loop/4,5,6`](`enter_loop/6`).\nThey are **not allowed** from _state enter calls_.\n\nActions are executed in the containing list order.\n\nActions that set [transition options](`t:transition_option/0`)\noverride any previous of the same type, so the last\nin the containing list wins.  For example, the last `t:postpone/0`\noverrides any previous `t:postpone/0` in the list.\n\n- **`{postpone, Value}`** - Sets the\n  [`transition_option()` ](`t:transition_option/0`)`t:postpone/0`\n  for this _state transition_.  This action is ignored when returned from\n  [`Module:init/1`](`c:init/1`) or passed to\n  [`enter_loop/4,5,6`](`enter_loop/6`), as there is no event to postpone\n  in those cases.\n\n  `postpone` is equivalent to `{postpone, true}`.\n\n- **`{next_event, EventType, EventContent}`** - This action\n  does not set any [`transition_option()`](`t:transition_option/0`)\n  but instead stores the specified `EventType` and `EventContent`\n  for insertion after all actions have been executed.\n\n  The stored events are inserted in the queue as the next to process\n  before any already queued events. The order of these stored events\n  is preserved, so the first `next_event` in the containing list\n  becomes the first to process.\n\n  An event of type [`internal`](`t:event_type/0`) should be used\n  when you want to reliably distinguish an event inserted this way\n  from any external event.\n\n- **`{change_callback_module, NewModule}`** {: #change_callback_module } -\n  Changes the callback module to `NewModule` which will be used\n  when calling all subsequent [state callbacks](#state-callback).\\\n  **Since OTP 22.3.**\n\n  The `gen_statem` engine will find out the\n  [_callback mode_](`t:callback_mode/0`) of `NewModule` by calling\n  [`NewModule:callback_mode/0`](`c:callback_mode/0`) before the next\n  [state callback](#state-callback).\n\n  Changing the callback module does not affect the _state transition_\n  in any way, it only changes which module that handles the events.\n  Be aware that all relevant callback functions in `NewModule` such as\n  the [state callback](#state-callback),\n  [`NewModule:code_change/4`](`c:code_change/4`),\n  [`NewModule:format_status/1`](`c:format_status/1`) and\n  [`NewModule:terminate/3`](`c:terminate/3`) must be able to handle\n  the state and data from the old module.\n\n- **`{push_callback_module, NewModule}`** {: #push_callback_module } -\n   Pushes the current callback module to the top of an internal stack\n   of callback modules, and changes the callback module to `NewModule`.\n   Otherwise like `{change_callback_module, NewModule}` above.\\\n  **Since OTP 22.3.**\n\n- **`pop_callback_module`** {: #pop_callback_module } -\n  Pops the top module from the internal stack of callback modules\n  and changes the callback module to be the popped module.\n  If the stack is empty the server fails.\n  Otherwise like `{change_callback_module, NewModule}` above.\\\n  **Since OTP 22.3.**","ref":"gen_statem.html#t:action/0"},{"type":"function","title":"gen_statem.call/2","doc":"","ref":"gen_statem.html#call/2"},{"type":"function","title":"gen_statem.call/3","doc":"Call a server: send request and wait for response.\n\nMakes a synchronous call to the `gen_statem`\n[`ServerRef`](`t:server_ref/0`) by sending a request\nand waiting until the response arrives.\n\n[](){: #call-reply }\nThe `gen_statem` calls the\n[_state callback_](#state-callback)\nwith `t:event_type/0` `{call, From}` and event content `Request`.\n\nThe server's reply is sent from a [_state callback_](#state-callback),\nby returning a [_transition action_](`t:action/0`) `{reply, From, Reply}`,\ncalling [`reply(Replies)`](`reply/1`) with such a reply action\nin the `Replies` list, or calling [`reply(From, Reply)`](`reply/2`).\n\n`Timeout` is an integer > 0, which specifies how many milliseconds\nto wait for a reply, or the atom `infinity` to wait indefinitely,\nwhich is the default. If no reply is received within the specified time,\nthe function call fails.\n\nPrevious issue with late replies that could occur\nwhen having network issues or using `dirty_timeout`\nis now prevented by use of\n[_process aliases_](`e:system:ref_man_processes.md#process-aliases`).\n`{clean_timeout, T}` and `{dirty_timeout, T}` therefore\nno longer serves any purpose and will work the same as `Timeout`\nwhile all of them also being equally efficient.\n\nThe call can also fail, for example, if the `gen_statem`\ndies before or during this function call.\n\nWhen this call fails it [exits](`erlang:exit/1`)\nthe calling process.  The exit term is on the form\n`{Reason, Location}` where `Location = {gen_statem, call, ArgList}`.\nSee [`gen_server:call/3`](`gen_server:call/3`) that has a description\nof relevant values for the `Reason` in the exit term.","ref":"gen_statem.html#call/3"},{"type":"type","title":"gen_statem.callback_mode/0","doc":"One function per state or one common event handler.\n\nThe _callback mode_ is selected with the return value from\n[`Module:callback_mode/0`](`c:callback_mode/0`):\n\n- **`state_functions`** - The state must be of type `t:state_name/0`\n  and one callback function per state, that is,\n  [`Module:StateName/3`](`c:'StateName'/3`), is used.\n\n- **`handle_event_function`** - The state can be any term and the callback\n  function [`Module:handle_event/4`](`c:handle_event/4`)\n  is used for all states.\n\nThe function [`Module:callback_mode/0`](`c:callback_mode/0`) is called\nwhen starting the `gen_statem`, after code change and after changing\nthe callback module with any of the actions\n[`change_callback_module`](#change_callback_module),\n[`push_callback_module`](#push_callback_module),\nor [`pop_callback_module`](#pop_callback_module).\nThe result is cached for subsequent calls to\n[_state callbacks_](#state-callback).","ref":"gen_statem.html#t:callback_mode/0"},{"type":"callback","title":"gen_statem.callback_mode/0","doc":"Select the _callback mode_ and possibly\n[_state enter calls_](`t:state_enter/0`).\n\nThis function is called by a `gen_statem` when it needs to find out the\n[_callback mode_](`t:callback_mode/0`) of the callback module.\n\nThe value is cached by `gen_statem` for efficiency reasons,\nso this function is only called once after server start,\nafter code change, and after changing the callback module,\nbut before the first [_state callback_](#state-callback)\nin the current callback module's code is called.  More occasions may be\nadded in future versions of `gen_statem`.\n\nServer start happens either when [`Module:init/1`](`c:init/1`)\nreturns or when [`enter_loop/4,5,6`](`enter_loop/6`) is called.\nCode change happens when [`Module:code_change/4`](`c:code_change/4`)\nreturns.  A change of the callback module happens when\na [_state callback_](#state-callback) returns\nany of the actions [`change_callback_module`](#push_callback_module),\n[`push_callback_module`](#push_callback_module) or\n[`pop_callback_module`](#pop_callback_module).\n\nThe `CallbackMode` is either just `t:callback_mode/0`\nor a list containing `t:callback_mode/0` and possibly\nthe atom [`state_enter`](`t:state_enter/0`).\n\n> #### Note {: .info }\n>\n> If this function's body does not return an inline constant value\n> the callback module is doing something strange.","ref":"gen_statem.html#c:callback_mode/0"},{"type":"type","title":"gen_statem.callback_mode_result/0","doc":"Return value from [`Module:callback_mode/0`](`c:callback_mode/0`).\n\nThis is the return type from\n[`Module:callback_mode/0`](`c:callback_mode/0`)\nwhich selects [_callback mode_](`t:callback_mode/0`)\nand whether to do [_state enter calls_](`t:state_enter/0`),\nor not.","ref":"gen_statem.html#t:callback_mode_result/0"},{"type":"function","title":"gen_statem.cast/2","doc":"Cast an event to a server.\n\nSends an asynchronous `cast` event to the `gen_statem`\n[`ServerRef`](`t:server_ref/0`) and returns `ok` immediately,\nignoring if the destination node or `gen_statem` does not exist.\n\nThe `gen_statem` calls the\n[_state callback_](#state-callback)\nwith `t:event_type/0` `cast` and event content `Msg`.","ref":"gen_statem.html#cast/2"},{"type":"function","title":"gen_statem.check_response/2","doc":"Check if a received message is a request response.\n\nChecks if `Msg` is a response corresponding to\nthe request identifier `ReqId`.  The request must have been made\nby `send_request/2` and by the same process calling this function.\n\nIf `Msg` is a reply to the handle `ReqId` the result of the request\nis returned in `Reply`.  Otherwise this function returns `no_reply`\nand no cleanup is done, and thus the function shall be invoked repeatedly\nuntil the response is returned.\n\nSee [`call/3`](#call-reply) about how the request is handled\nand the `Reply` is sent by the `gen_statem` server.\n\nIf the `gen_statem` server process has died when this function\nis called, that is; `Msg` reports the server's death,\nthis function returns an `error` return with the exit `Reason`.","ref":"gen_statem.html#check_response/2"},{"type":"function","title":"gen_statem.check_response/3","doc":"Check if a received message is a request response in a collection.\n\nCheck if `Msg` is a response corresponding to a request identifier\nstored in `ReqIdCollection`.  All request identifiers of `ReqIdCollection`\nmust correspond to requests that have been made using `send_request/2`\nor `send_request/4`, by the process calling this function.\n\nThe `Label` in the response equals the `Label` associated\nwith the request identifier that the response corresponds to.\nThe `Label` of a request identifier is associated\nwhen [storing the request id](`reqids_add/3`) in a collection,\nor when sending the request using `send_request/4`.\n\nCompared to `check_response/2`, the returned result or exception\nassociated with a specific request identifier will be wrapped\nin a 3-tuple `{Response, Label, NewReqIdCollection}`.\n`Response` is the value that would have been produced\nby `check_response/2`, `Label` is the value associated with\nthe specific [request identifier](`t:request_id/0`)\nand `NewReqIdCollection` is a possibly modified\nrequest identifier collection.\n\nIf `ReqIdCollection` is empty, `no_request` is returned.\n\nIf `Msg` does not correspond to any of the request identifiers\nin `ReqIdCollection`, `no_reply` is returned.\n\nIf `Delete` equals `true`, the association with `Label`\nhas been deleted from `ReqIdCollection` in the resulting\n`NewReqIdCollection`. If `Delete` is `false`, `NewReqIdCollection`\nwill equal `ReqIdCollection`. Note that deleting an association\nis not for free and that a collection containing already handled\nrequests can still be used by subsequent calls to\n`wait_response/3`, `check_response/3`, and `receive_response/3`.\n\nHowever, without deleting handled associations,\nthe above calls will not be able to detect when there are\nno more outstanding requests to handle, so you will have to keep track\nof this some other way than relying on a `no_request` return.\nNote that if you pass a collection only containing\nassociations of already handled or abandoned requests to\nthis function, it will always return `no_reply`.","ref":"gen_statem.html#check_response/3"},{"type":"callback","title":"gen_statem.code_change/4","doc":"Update the [state](`t:state/0`) and [data](`t:data/0`)\nafter code change.\n\nThis function is called by a `gen_statem` when it is to update\nits internal state during a release upgrade/downgrade, that is,\nwhen the instruction `{update, Module, Change, ...}`,\nwhere `Change = {advanced, Extra}`, is specified in\nthe [`appup`](`e:sasl:appup.md`) file.  For more information, see\n[OTP Design Principles](`e:system:release_handling.md#instr`).\n\nFor an upgrade, `OldVsn` is `Vsn`, and for a downgrade, `OldVsn` is\n`{down, Vsn}`. `Vsn` is defined by the `vsn` attribute(s)\nof the old version of the callback module `Module`.\nIf no such attribute is defined, the version is the checksum\nof the Beam file.\n\n`OldState` and `OldData` is the internal state of the `gen_statem`.\n\n`Extra` is passed \"as is\" from the `{advanced, Extra}` part\nof the update instruction.\n\nIf successful, the function must return the updated internal state\nin an `{ok, NewState, NewData}` tuple.\n\nIf the function returns a failure `Reason`, the ongoing upgrade fails\nand rolls back to the old release. Note that `Reason` cannot be\nan `{ok, _, _}` tuple since that will be regarded\nas a `{ok, NewState, NewData}` tuple, and that a tuple matching `{ok, _}`\nis an also invalid failure `Reason`.  It is recommended to use\nan atom as `Reason` since it will be wrapped in an `{error, Reason}` tuple.\n\nAlso note when upgrading a `gen_statem`, this function and hence the\n`Change = {advanced, Extra}` parameter\nin the [`appup`](`e:sasl:appup.md`) file is not only needed\nto update the internal state or to act on the `Extra`\nargument.  It is also needed if an upgrade or downgrade should change\n[_callback mode_](`t:callback_mode/0`), or else the _callback mode_\nafter the code change will not be honoured, most probably causing\na server crash.\n\nIf the server changes callback module using any of the actions\n[`change_callback_module`](#change_callback_module),\n[`push_callback_module`](#push_callback_module), or\n[`pop_callback_module`](#pop_callback_module), be aware that it is always\nthe current callback module that will get this callback call.\nThat the current callback module handles the current\nstate and data update should be no surprise, but it\nmust be able to handle even parts of the state and data\nthat it is not familiar with, somehow.\n\nIn the supervisor\n[child specification](`e:system:sup_princ.md#child-specification`)\nthere is a list of modules which is recommended to contain\nonly the callback module.  For a `gen_statem`\nwith multiple callback modules there is no real need to list\nall of them, it may not even be possible since the list could change\nafter code upgrade.  If this list would contain only\nthe start callback module, as recommended, what is important\nis to upgrade _that_ module whenever\na _synchronized code replacement_ is done.\nThen the release handler concludes that\nan upgrade that upgrades _that_ module needs to suspend,\ncode change, and resume any server whose child specification declares\nthat it is using _that_ module.\nAnd again; the _current_ callback module will get the\n[`Module:code_change/4`](`c:code_change/4`) call.\n\n> #### Note {: .info }\n>\n> If a release upgrade/downgrade with `Change = {advanced, Extra}`\n> specified in the `.appup` file is made\n> when [`Module:code_change/4`](`c:code_change/4`) is not implemented\n> the process will crash with exit reason `undef`.","ref":"gen_statem.html#c:code_change/4"},{"type":"type","title":"gen_statem.data/0","doc":"Generic state data for the server.\n\nA term in which the state machine implementation is to store\nany server data it needs. The difference between this and the `t:state/0`\nitself is that a change in this data does not cause postponed events\nto be retried. Hence, if a change in this data would change\nthe set of events that are handled, then that data item\nshould be part of the `t:state/0` instead.","ref":"gen_statem.html#t:data/0"},{"type":"type","title":"gen_statem.enter_action/0","doc":"Actions for any callback: hibernate, time-outs or replies.\n\nThese _transition actions_ are allowed when a `t:action/0` is allowed,\nand also from a _state enter call_, and can be invoked\nby returning them from the [_state callback_](#state-callback), from\n[`Module:init/1`](`c:init/1`) or by passing them to\n[`enter_loop/4,5,6`](`enter_loop/6`).\n\nActions are executed in the containing list order.\n\nActions that set [transition options](`t:transition_option/0`)\noverride any previous of the same type,\nso the last in the containing list wins. For example,\nthe last `t:event_timeout/0` overrides any previous\n`t:event_timeout/0` in the list.\n\n- **`{hibernate, Value}`** - Sets the `t:transition_option/0`\n  `t:hibernate/0` for this _state transition_.\n\n  `hibernate` is equivalent to `{hibernate, true}`.","ref":"gen_statem.html#t:enter_action/0"},{"type":"function","title":"gen_statem.enter_loop/4","doc":"","ref":"gen_statem.html#enter_loop/4"},{"type":"function","title":"gen_statem.enter_loop/5","doc":"Make the calling process become a `gen_statem` server.\n\nWith argument `Actions`, equivalent to\n[`enter_loop(Module, Opts, State, Data, self(), Actions)`](`enter_loop/6`).\n\nOtherwise equivalent to\n[`enter_loop(Module, Opts, State, Data, Server, [])`](`enter_loop/6`).","ref":"gen_statem.html#enter_loop/5"},{"type":"function","title":"gen_statem.enter_loop/6","doc":"Make the calling process become a `gen_statem` server.\n\nDoes not return, instead the calling process enters the `gen_statem`\nreceive loop and becomes a `gen_statem` server.  The process\n_must_ have been started using one of the start functions\nin `m:proc_lib`.  The user is responsible for any initialization\nof the process, including registering a name for it.\n\nThis function is useful when a more complex initialization procedure\nis needed than the `gen_statem` [`Module:init/1`](`c:init/1`)\ncallback offers.\n\n`Module` and `Opts` have the same meanings as when calling\n[`start[link | monitor]/3,4`](`start_link/3`).\n\nIf `Server` is `self/0` an anonymous server is created just as when using\n[`start[link |_monitor]/3`](`start_link/3`).  If `Server`\nis a `t:server_name/0` a named server is created just as when using\n[`start[link |_monitor]/4`](`start_link/4`).  However,\nthe `t:server_name/0` name must have been registered accordingly\n_before_ this function is called.\n\n`State`, `Data`, and `Actions` have the same meanings\nas in the return value of [`Module:init/1`](`c:init/1`).\nAlso, the callback module does not need to export\na [`Module:init/1`](`c:init/1`) function.\n\nThe function fails if the calling process was not started\nby a `m:proc_lib` start function, or if it is not registered\naccording to `t:server_name/0`.","ref":"gen_statem.html#enter_loop/6"},{"type":"type","title":"gen_statem.enter_loop_opt/0","doc":"Server [start options](#start-options) for the\n[`enter_loop/4,5,6`](`enter_loop/6`),\n[`start/3,4`](`start/3`), [`start_link/3,4`](`start_link/3`),\nand [`start_monitor/3,4`](`start_monitor/3`), functions.\n\nSee [`start_link/4`](#start-options).","ref":"gen_statem.html#t:enter_loop_opt/0"},{"type":"type","title":"gen_statem.event_content/0","doc":"Event payload from the event's origin, delivered to\nthe [_state callback_](#state-callback).\n\nSee [`event_type`](`t:event_type/0`) that describes the origins of\nthe different event types, which is also where the event's content\ncomes from.","ref":"gen_statem.html#t:event_content/0"},{"type":"type","title":"gen_statem.event_handler_result/1","doc":"","ref":"gen_statem.html#t:event_handler_result/1"},{"type":"type","title":"gen_statem.event_handler_result/2","doc":"Return value from a [_state callback_](#state-callback)\nafter handling an event.\n\n`StateType` is `t:state_name/0`\nif [_callback mode_](`t:callback_mode/0`) is `state_functions`,\nor `t:state/0`\nif [_callback mode_](`t:callback_mode/0`) is `handle_event_function`.\n\n- **`{next_state, NextState, NewData [, Actions]}`** -\n  The `gen_statem` does a _state transition_ to `NextState`\n  (which may be the same as the current state), sets `NewData`\n  as the current server `t:data/0`, and executes all `Actions`.\n  If `NextState =/= CurrentState` the _state transition_\n  is a _state change_.","ref":"gen_statem.html#t:event_handler_result/2"},{"type":"type","title":"gen_statem.event_timeout/0","doc":"How long to wait for an event.\n\nStarts a timer set by `t:timeout_action/0`\n`Time`, or `{timeout, Time, EventContent [, Options]}`.\n\nWhen the timer expires an event of `t:event_type/0` `timeout`\nwill be generated. See `erlang:start_timer/4` for how `Time`\nand [`Options`](`t:timeout_option/0`) are interpreted.  Future\n`erlang:start_timer/4` `Options` will not necessarily be supported.\n\nAny event that arrives cancels this time-out. Note that a retried\nor inserted event counts as arrived. So does a state time-out zero event,\nif it was generated before this time-out is requested.\n\nIf `Time` is `infinity`, no timer is started,\nas it never would expire anyway.\n\nIf `Time` is relative and `0` no timer is actually started,\ninstead the the time-out event is enqueued to ensure\nthat it gets processed before any not yet received external event,\nbut after already queued events.\n\nNote that it is not possible nor needed to cancel this time-out,\nas it is cancelled automatically by any other event, meaning that\nwhenever a callback is invoked that may want to cancel this time-out,\nthe timer is already cancelled or expired.\n\nThe timer `EventContent` can be updated with the\n[`{timeout, update, NewEventContent}`](`t:timeout_update_action/0`)\naction without affecting the time of expiry.","ref":"gen_statem.html#t:event_timeout/0"},{"type":"type","title":"gen_statem.event_type/0","doc":"All event types: [external](`t:external_event_type/0`),\n[time-out](`t:timeout_event_type/0`), or `internal`.\n\n`internal` events can only be generated by the state machine itself\nthrough the _transition action_ [`next_event`](`t:action/0`).","ref":"gen_statem.html#t:event_type/0"},{"type":"type","title":"gen_statem.external_event_type/0","doc":"Event from a [call](`call/3`), [cast](`cast/2`),\nor regular process message; \"info\".\n\nType `{call, From}` originates from the API functions\n[`call/2,3`](`call/3`) or `send_request/2`.  The event contains\n[`From`](`t:from/0`), which is whom to reply to\nby a `t:reply_action/0` or [`reply/2,3`](`reply/2`) call.\n\nType `cast` originates from the API function `cast/2`.\n\nType `info` originates from regular process messages\nsent to the `gen_statem` process.","ref":"gen_statem.html#t:external_event_type/0"},{"type":"type","title":"gen_statem.format_status/0","doc":"A map that describes the server's status.\n\nThe keys are:\n- **`state`** - The current state.\n- **`data`** - The state data.\n- **`reason`** - The reason that caused the process to terminate.\n- **`queue`** - The event queue.\n- **`postponed`** - The queue of [postponed](`t:postpone/0`) events.\n- **`timeouts`** - The active [time-outs](`t:timeout_action/0`).\n- **`log`** - The [sys log](`sys:log/2`) of the server.\n\nNew associations may be added to the status map without prior notice.","ref":"gen_statem.html#t:format_status/0"},{"type":"callback","title":"gen_statem.format_status/1","doc":"Format/limit the status value.\n\nThis function is called by a `gen_statem` process\nin order to format/limit the server status\nfor debugging and logging purposes.\n\nIt is called in the following situations:\n\n- [`sys:get_status/1,2`](`sys:get_status/1`) is invoked\n  to get the `gen_statem` status.\n- The `gen_statem` process terminates abnormally and logs an error.\n\nThis function is useful for changing the form and appearance\nof the `gen_statem` status for these cases.  A callback module\nwishing to change the [`sys:get_status/1,2`](`sys:get_status/1`)\nreturn value and how its status appears in termination error logs,\nexports an instance of [`Module:format_status/1`](`c:format_status/1`),\nwhich will get a map `Status` that describes the current state\nof the `gen_statem`, and shall return a map `NewStatus`\ncontaining the same keys as the input map,\nbut it may transform some values.\n\nOne use case for this function is to return compact alternative state\nrepresentations to avoid having large state terms printed in log files.\nAnother is to hide sensitive data from being written to the error log.\n\nExample:\n\n```erlang\nformat_status(Status) ->\n  maps:map(\n    fun(state,State) ->\n            maps:remove(private_key, State);\n       (message,{password, _Pass}) ->\n            {password, removed};\n       (_,Value) ->\n            Value\n    end, Status).\n```\n\n> #### Note {: .info }\n>\n> This callback is optional, so a callback module does not need\n> to export it.  The `gen_statem` module provides\n> a default implementation of this function that returns `{State, Data}`.\n>\n> If this callback is exported but fails, to hide possibly sensitive data,\n> the default function will instead return `{State, Info}`,\n> where `Info` says nothing but the fact that\n> [`Module:format_status/2`](`c:format_status/2`) has crashed.","ref":"gen_statem.html#c:format_status/1"},{"type":"callback","title":"gen_statem.format_status/2","doc":"Format/limit the status value.\n\nThis function is called by a `gen_statem` process\nin in order to format/limit the server state\nfor debugging and logging purposes.\n\nIt is called in the following situations:\n\n- One of [`sys:get_status/1,2`](`sys:get_status/1`) is invoked to get the\n  `gen_statem` status. `Opt` is set to the atom `normal` for this case.\n\n- The `gen_statem` terminates abnormally and logs an error.\n  `Opt` is set to the atom `terminate` for this case.\n\nThis function is useful for changing the form and appearance of\nthe `gen_statem` status for these cases.  A callback module wishing to\nchange the [`sys:get_status/1,2`](`sys:get_status/1`) return value\nand how its status appears in termination error logs, should export\nan instance of [`Module:format_status/2`](`c:format_status/2`),\nthat returns a term describing the current status of the `gen_statem`.\n\n`PDict` is the current value of the process dictionary of the `gen_statem`.\n\n[`State`](`t:state/0`) is the internal state of the `gen_statem`.\n\n[`Data`](`t:data/0`) is the internal server data of the `gen_statem`.\n\nThe function is to return `Status`, a term that contains\nthe appropriate details of the current state and status\nof the `gen_statem`.  There are no restrictions on the form `Status`\ncan take, but for the [`sys:get_status/1,2`](`sys:get_status/1`) case\n(when `Opt` is `normal`), the recommended form for the `Status` value\nis `[{data, [{\"State\", Term}]}]`, where `Term` provides relevant details\nof the `gen_statem` state.  Following this recommendation is not required,\nbut it makes the callback module status consistent\nwith the rest of the [`sys:get_status/1,2`](`sys:get_status/1`)\nreturn value.\n\nOne use for this function is to return compact alternative\nstate representations to avoid having large state terms printed\nin log files. Another use is to hide sensitive data\nfrom being written to the error log.\n\n> #### Note {: .info }\n>\n> This callback is optional, so a callback module does not need\n> to export it.  The `gen_statem` module provides a default\n> implementation of this function that returns `{State, Data}`.\n>\n> If this callback is exported but fails, to hide possibly sensitive data,\n> the default function will instead return `{State, Info}`,\n> where `Info` says nothing but the fact that\n> [`Module:format_status/2`](`c:format_status/2`) has crashed.","ref":"gen_statem.html#c:format_status/2"},{"type":"type","title":"gen_statem.from/0","doc":"A [`call`](`t:external_event_type/0`) event's reply destination.\n\nDestination to use when replying through, for example,\nthe action [`{reply, From, Reply}`](`t:reply_action/0`)\nto a process that has called the `gen_statem` server\nusing [`call/2,3`](`call/3`).","ref":"gen_statem.html#t:from/0"},{"type":"type","title":"gen_statem.generic_timeout/0","doc":"How long to wait for a named time-out event.\n\nStarts a timer set by `t:timeout_action/0`\n`{{timeout, Name}, Time, EventContent [, Options]}`.\n\nWhen the timer expires an event of `t:event_type/0` `{timeout, Name}`\nwill be generated. See `erlang:start_timer/4` for how `Time`\nand [`Options`](`t:timeout_option/0`) are interpreted. Future\n`erlang:start_timer/4` `Options` will not necessarily be supported.\n\nIf `Time` is `infinity`, no timer is started,\nas it never would expire anyway.\n\nIf `Time` is relative and `0` no timer is actually started,\ninstead the time-out event is enqueued to ensure\nthat it gets processed before any not yet received external event.\n\nSetting a timer with the same `Name` while it is running\nwill restart it with the new time-out value.  Therefore it is possible\nto cancel a specific time-out by setting it to `infinity`.\nIt can also be cancelled more explicitly with the\n[`{{timeout, Name}, cancel}`](`t:timeout_cancel_action/0`) action.\n\nThe timer `EventContent` can be updated with the\n[`{{timeout, Name}, update, NewEventContent}`](`t:timeout_update_action/0`)\naction without affecting the time of expiry.","ref":"gen_statem.html#t:generic_timeout/0"},{"type":"callback","title":"gen_statem.handle_event/4","doc":"[_State callback_](#state-callback) in\n[_callback mode_](`t:callback_mode/0`) `handle_event_function`.\n\nWhenever a `gen_statem` receives an event from [`call/2,3`](`call/3`),\n`cast/2`, or as a normal process message, this function is called.\n\nIf `EventType` is [`{call, From}`](`t:event_type/0`),\nthe caller waits for a reply.  The reply can be sent from this\nor from any other [_state callback_](#state-callback)\nby returning with `{reply, From, Reply}` in [`Actions`](`t:action/0`),\nin [`Replies`](`t:reply_action/0`), or by calling\n[`reply(From, Reply)`](`reply/2`).\n\nIf this function returns with a next state\nthat does not match equal (`=/=`) to the current state,\nall postponed events are retried in the next state.\n\nFor options that can be set and actions that can be done\nby `gen_statem` after returning from this function, see `t:action/0`.\n\nWhen the `gen_statem` runs with [_state enter calls_](`t:state_enter/0`),\nthis function is also called with arguments `(enter, OldState, ...)`\nduring every _state change_.  In this case there are some restrictions\non the [actions](`t:action/0`) that may be returned:\n\n- `t:postpone/0` is not allowed since a _state enter call_\n  is not an event so there is no event to postpone.\n- [`{next_event, _, _}`](`t:action/0`) is not allowed since\n  using _state enter calls_ should not affect how events\n  are consumed and produced.\n- It is not allowed to change states from this call.\n  Should you return `{next_state, NextState, ...}`\n  with `NextState =/= State` the `gen_statem` crashes.\n\n  Note that it is actually allowed to use `{repeat_state, NewData, ...}`\n  although it makes little sense since you immediately\n  will be called again with a new _state enter call_ making this\n  just a weird way of looping, and there are better ways to loop in Erlang.\n\n  If you do not update `NewData` and have some loop termination condition,\n  or if you use `{repeat_state_and_data, _}` or `repeat_state_and_data`\n  you have an infinite loop\\!\n\n  You are advised to use `{keep_state, ...}`, `{keep_state_and_data, _}`\n  or `keep_state_and_data` since changing states\n  from a _state enter call_ is not possible anyway.\n\nNote the fact that you can use [`throw`](`erlang:throw/1`)\nto return the result, which can be useful.  For example to bail out with\n[`throw(keep_state_and_data)`](`throw/1`) from deep within complex code\nthat cannot return `{next_state, State, Data}` because `State` or `Data`\nis no longer in scope.","ref":"gen_statem.html#c:handle_event/4"},{"type":"type","title":"gen_statem.hibernate/0","doc":"Hibernate the server process.\n\nIf `true`, hibernates the `gen_statem` by calling `proc_lib:hibernate/3`\nbefore going into `receive` to wait for a new external event.\n\nThere is also a server start option\n[`{hibernate_after, Timeout}`](`t:enter_loop_opt/0`)\nfor automatic hibernation.\n\n> #### Note {: .info }\n>\n> If there are enqueued events to process when hibernation is requested,\n> this is optimized by not hibernating but instead calling\n> [`erlang:garbage_collect/0`](`erlang:garbage_collect/0`) to simulate,\n> in a more effective way, that the `gen_statem` entered hibernation\n> and immediately got awakened by an enqueued event.","ref":"gen_statem.html#t:hibernate/0"},{"type":"callback","title":"gen_statem.init/1","doc":"Initialize the state machine.\n\nWhenever a `gen_statem` is started using\n[`start_link/3,4`](`start_link/3`),\n[`start_monitor/3,4`](`start_monitor/3`), or\n[`start/3,4`](`start/3`), this function is called by the new process\nto initialize the implementation state and server data.\n\n`Args` is the `Args` argument provided to that start function.\n\n> #### Note {: .info }\n>\n> Note that if the `gen_statem` is started through `m:proc_lib`\n> and [`enter_loop/4,5,6`](`enter_loop/6`), this callback\n> will never be called.  Since this callback is not optional\n> it can in that case be implemented as:\n>\n> ```erlang\n> -spec init(_) -> no_return().\n> init(Args) -> erlang:error(not_implemented, [Args]).\n> ```","ref":"gen_statem.html#c:init/1"},{"type":"type","title":"gen_statem.init_result/1","doc":"","ref":"gen_statem.html#t:init_result/1"},{"type":"type","title":"gen_statem.init_result/2","doc":"The return value from [`Module:init/1`](`c:init/1`).\n\nFor a succesful initialization, `State` is the initial `t:state/0`,\nand `Data` the initial server `t:data/0` of the `gen_statem`.\n\nThe [`Actions`](`t:action/0`) are executed when entering the first\n[state](`t:state/0`) just as for a\n[_state callback_](#state-callback), except that the action\n`postpone` is forced to `false` since there is no event to postpone.\n\nFor an unsuccesful initialization, `{stop, Reason}`, `{error, Reason}`,\nor `ignore` should be used; see [`start_link/3,4`](`start_link/3`).\n\n`{error, Reason}` has been allowed **since OTP 26.0**.\n\nThe `{ok, ...}` tuples have existed **since OTP 19.1**,\nbefore that they were not `ok` tagged.  This was before\n`gen_statem` replaced `gen_fsm` in OTP 20.0.","ref":"gen_statem.html#t:init_result/2"},{"type":"type","title":"gen_statem.postpone/0","doc":"Postpone an event to handle it later.\n\nIf `true`, postpones the current event.\nAfter a _state change_ (`NextState =/= State`), it is retried.","ref":"gen_statem.html#t:postpone/0"},{"type":"function","title":"gen_statem.receive_response/1","doc":"","ref":"gen_statem.html#receive_response/1"},{"type":"function","title":"gen_statem.receive_response/2","doc":"Receive a request response.\n\nReceive a response corresponding to the request identifier `ReqId`.\nThe request must have been made by `send_request/2`\nto the `gen_statem` process.  This function must be called\nfrom the same process from which `send_request/2` was made.\n\n`Timeout` specifies how long to wait for a response.\nIf no response is received within the specified time,\nthis function returns `timeout`.  Assuming that the server executes\non a node supporting aliases (introduced in OTP 24)\nthe request will also be abandoned.  That is,\nno response will be received after a time-out.\nOtherwise, a stray response might be received at a later time.\n\nSee [`call/3`](#call-reply) about how the request is handled\nand the `Reply` is sent by the `gen_statem` server.\n\nIf the `gen_statem` server process is dead or dies while\nthis function waits for the reply, it returns an `error` return\nwith the exit `Reason`.\n\nThe difference between `wait_response/2` and `receive_response/2`\nis that `receive_response/2` abandons the request at time-out\nso that a potential future response is ignored,\nwhile `wait_response/2` does not.","ref":"gen_statem.html#receive_response/2"},{"type":"function","title":"gen_statem.receive_response/3","doc":"Receive a request response in a collection.\n\nReceive a response in `ReqIdCollection`.  All request identifiers\nof `ReqIdCollection` must correspond to requests that have been made\nusing `send_request/2` or `send_request/4`, and all requests\nmust have been made by the process calling this function.\n\nThe `Label` in the response is the `Label` associated with\nthe request identifier that the response corresponds to.\nThe `Label` of a request identifier is associated\nwhen [adding the request id](`reqids_add/3`) to a collection,\nor when sending the request using `send_request/4`.\n\nCompared to `receive_response/2`, the returned result or exception\nassociated with a specific request identifier will be wrapped\nin a 3-tuple `{Response, Label, NewReqIdCollection}`.\n`Response` is the value that would have been produced\nby `receive_response/2`, `Label` is the value associated with\nthe specific [request identifier](`t:request_id/0`)\nand `NewReqIdCollection` is a possibly modified\nrequest identifier collection.\n\nIf `ReqIdCollection` is empty, `no_request` will be returned.\n\n`Timeout` specifies how long to wait for a response.  If no response\nis received within the specified time, the function returns `timeout`.\nAssuming that the server executes on a node supporting aliases\n(introduced in OTP 24) all requests identified by `ReqIdCollection`\nwill also be abandoned. That is, no responses will be received\nafter a time-out.  Otherwise, stray responses might be received\nat a later time.\n\nThe difference between `receive_response/3` and `wait_response/3`\nis that `receive_response/3` abandons requests at time-out\nso that potential future responses are ignored,\nwhile `wait_response/3` does not.\n\nIf `Delete` is `true`, the association with `Label`\nis deleted from `ReqIdCollection` in the resulting\n`NewReqIdCollection`.  If `Delete` is `false`, `NewReqIdCollection`\nwill equal`ReqIdCollection`.  Note that deleting an association\nis not for free and that a collection containing already handled\nrequests can still be used by subsequent calls to\n`wait_response/3`, `check_response/3`, and `receive_response/3`.\n\nHowever, without deleting handled associations,\nthe above calls will not be able to detect when there are\nno more outstanding requests to handle, so you will have to keep track\nof this some other way than relying on a `no_request` return.\nNote that if you pass a collection only containing\nassociations of already handled or abandoned requests to\nthis function, it will always block until `Timeout` expires\nand then return `timeout`.","ref":"gen_statem.html#receive_response/3"},{"type":"function","title":"gen_statem.reply/1","doc":"Send one or multiple `call` replies.\n\nThis funcion can be used by a `gen_statem` callback to explicitly send\none or multiple replies to processes waiting for `call` requests' replies,\nwhen it is impractical or impossible to return `t:reply_action/0`s\nfrom a [_state callback_](#state-callback).\n\n> #### Note {: .info }\n>\n> A reply sent with this function is not visible in `m:sys` debug output.","ref":"gen_statem.html#reply/1"},{"type":"function","title":"gen_statem.reply/2","doc":"Send a `call` `Reply` to `From`.\n\nThis funcion can be used by a `gen_statem` callback to explicitly send\na reply to a process waiting for a `call` requests' reply,\nwhen it is impractical or impossible to return a `t:reply_action/0`\nfrom a [_state callback_](#state-callback).\n\n> #### Note {: .info }\n>\n> A reply sent with this function is not visible in `m:sys` debug output.","ref":"gen_statem.html#reply/2"},{"type":"type","title":"gen_statem.reply_action/0","doc":"Reply to a [`call/2,3`](`call/3`).\n\nThis _transition action_ can be invoked by returning it from the\n[_state callback_](#state-callback), from\n[`Module:init/1`](`c:init/1`) or by passing it to\n[`enter_loop/4,5,6`](`enter_loop/6`).\n\nIt does not set any [`transition_option()`](`t:transition_option/0`)\nbut instead replies to a caller waiting for a reply in `call/3`.\n`From` must be the term from argument [`{call, From}`](`t:event_type/0`)\nin a call to a [_state callback_](#state-callback).\n\nNote that using this action from [`Module:init/1`](`c:init/1`) or\n[`enter_loop/4,5,6`](`enter_loop/6`) would be weird\non the border of witchcraft since there has been no earlier call to a\n[_state callback_](#state-callback) in this server.","ref":"gen_statem.html#t:reply_action/0"},{"type":"opaque","title":"gen_statem.reply_tag/0","doc":"A handle that associates a reply to the corresponding request.","ref":"gen_statem.html#t:reply_tag/0"},{"type":"function","title":"gen_statem.reqids_add/3","doc":"Store a request identifier in a colletion.\n\nStores `ReqId` and associates a `Label` with the request identifier\nby adding this information to `ReqIdCollection` and returning\nthe resulting request identifier collection.","ref":"gen_statem.html#reqids_add/3"},{"type":"function","title":"gen_statem.reqids_new/0","doc":"Create an empty request identifier collection.\n\nReturns a new empty request identifier collection.\nA request identifier collection can be used to handle\nmultiple outstanding requests.\n\nRequest identifiers of requests made by `send_request/2`\ncan be stored in a collection using `reqids_add/3`.\nSuch a collection of request identifiers can later be used\nin order to get one response corresponding to a request\nin the collection by passing the collection as argument to\n`receive_response/3`, `wait_response/3`, or, `check_response/3`.\n\n`reqids_size/1` can be used to determine the number of\nrequest identifiers in a collection.","ref":"gen_statem.html#reqids_new/0"},{"type":"function","title":"gen_statem.reqids_size/1","doc":"Return the number of request identifiers in `ReqIdCollection`.","ref":"gen_statem.html#reqids_size/1"},{"type":"function","title":"gen_statem.reqids_to_list/1","doc":"Convert a request identifier collection to a list.\n\nReturns a list of `{ReqId, Label}` tuples which corresponds to\nall request identifiers with their associated labels\nin [`ReqIdCollection`](`t:request_id_collection/0`).","ref":"gen_statem.html#reqids_to_list/1"},{"type":"opaque","title":"gen_statem.request_id/0","doc":"An opaque request identifier. See `send_request/2` for details.","ref":"gen_statem.html#t:request_id/0"},{"type":"opaque","title":"gen_statem.request_id_collection/0","doc":"An opaque collection of request identifiers (`t:request_id/0`).\n\nEach request identifier can be associated with\na label chosen by the user.  For more information see `reqids_new/0`.","ref":"gen_statem.html#t:request_id_collection/0"},{"type":"type","title":"gen_statem.response_timeout/0","doc":"Response time-out for an asynchronous call.\n\nUsed to set a time limit on how long to wait for a response using either\n`receive_response/2`, `receive_response/3`, `wait_response/2`, or\n`wait_response/3`.  The time unit used is `millisecond`.\n\n Currently valid values:\n\n- **`0..4294967295`** - Time-out relative to current time in milliseconds.\n\n- **`infinity`** - Infinite time-out. That is,\n  the operation will never time out.\n\n- **`{abs, Timeout}`** - An absolute\n  [Erlang monotonic time](`erlang:monotonic_time/1`)\n  time-out in milliseconds.  That is, the operation will time out when\n  [`erlang:monotonic_time(millisecond)`](`erlang:monotonic_time/1`)\n  returns a value larger than or equal to `Timeout`.\n  `Timeout` is not allowed to identify a time further into the future\n  than `4294967295` milliseconds.  Specifying the time-out\n  using an absolute value is especially handy when you have\n  a deadline for responses corresponding to a complete collection\n  of requests (`t:request_id_collection/0`), since you do not have to\n  recalculate the relative time until the deadline over and over again.","ref":"gen_statem.html#t:response_timeout/0"},{"type":"callback","title":"gen_statem.StateName/3","doc":"[_State callback_](#state-callback) in\n[_callback mode_](`t:callback_mode/0`) `state_functions`.\n\nState callback that handles all events in state `StateName`, where\n[`StateName :: state_name()`](`t:state_name/0`)\nhas to be an `t:atom/0`.\n\n`StateName` cannot be `terminate` since that would collide\nwith the callback function [`Module:terminate/3`](`c:terminate/3`).\n\nBesides that when doing a [_state change_](#state-callback)\nthe next state always has to be an `t:atom/0`,\nthis function is equivalent to\n[`Module:handle_event(​EventType, EventContent,\n?FUNCTION_NAME, Data)`](`c:handle_event/4`),\nwhich is the [_state callback_](#state-callback) in\n[_callback mode_](`t:callback_mode/0`) `handle_event_function`.","ref":"gen_statem.html#c:StateName/3"},{"type":"function","title":"gen_statem.send_request/2","doc":"Send an asynchronous `call` request.\n\nSends `Request` to the `gen_statem` process identified by `ServerRef`\nand returns a request identifier `ReqId`.\n\nThe return value `ReqId` shall later be used with `receive_response/2`,\n`wait_response/2`, or `check_response/2` to fetch the actual result\nof the request.  Besides passing the request identifier directly\nto these functions, it can also be stored in\na request identifier collection using `reqids_add/3`.\nSuch a collection of request identifiers can later be used\nin order to get one response corresponding to a\nrequest in the collection by passing the collection\nas argument to `receive_response/3`, `wait_response/3`,\nor `check_response/3`.  If you are about to store the request identifier\nin a collection, you may want to consider using `send_request/4` instead.\n\nThe call\n`gen_statem:wait_response(gen_statem:send_request(ServerRef,\nRequest), Timeout)` can be seen as equivalent to\n[`gen_statem:call(Server, Request, Timeout)`](`call/3`),\nignoring the error handling.\n\nSee [`call/3`](#call-reply) about how the request is handled\nand the `Reply` is sent by the `gen_statem` server.\n\nThe server's `Reply` is returned by one of the\n[`receive_response/1,2`](`receive_response/2`),\n[`wait_response/1,2`](`wait_response/2`),\nor `check_response/2` functions.","ref":"gen_statem.html#send_request/2"},{"type":"function","title":"gen_statem.send_request/4","doc":"Send an asynchronous `call` request and add it\nto a request identifier collection.\n\nSends `Request` to the `gen_statem` process identified by `ServerRef`.\nThe `Label` will be associated with the request identifier\nof the operation and added to the returned request identifier collection\n`NewReqIdCollection`.  The collection can later be used in order to\nget one response corresponding to a request in the collection\nby passing the collection as argument to `receive_response/3`,\n`wait_response/3`, or `check_response/3`.\n\nThe same as calling\n[`reqids_add(​`](`reqids_add/3`)[`send_request(ServerRef, Request),\n`](`send_request/2`)[`Label, ReqIdCollection)`](`reqids_add/3`),\nbut slightly more efficient.","ref":"gen_statem.html#send_request/4"},{"type":"type","title":"gen_statem.server_name/0","doc":"Server name specification: `local`, `global`, or `via` registered.\n\nName specification to use when starting a `gen_statem` server.\nSee `start_link/3` and `t:server_ref/0` below.","ref":"gen_statem.html#t:server_name/0"},{"type":"type","title":"gen_statem.server_ref/0","doc":"Server specification: `t:pid/0` or registered `t:server_name/0`.\n\nTo be used in [`call/2,3`](`call/3`) to specify the server.\n\nIt can be:\n\n- **`pid() | LocalName`** - The `gen_statem` is locally registered.\n\n- **`{Name, Node}`** - The `gen_statem` is locally registered\n  on another node.\n\n- **`{global, GlobalName}`** - The `gen_statem` is globally registered\n  in `m:global`.\n\n- **`{via, RegMod, ViaName}`** - The `gen_statem` is registered\n  in an alternative process registry.  The registry callback module\n  `RegMod` is to export functions `register_name/2`, `unregister_name/1`,\n  `whereis_name/1`, and `send/2`, which are to behave like\n  the corresponding functions in `m:global`.\n  Thus, `{via, global, GlobalName}` is the same as `{global, GlobalName}`.","ref":"gen_statem.html#t:server_ref/0"},{"type":"function","title":"gen_statem.start/3","doc":"Start a server, neither linked nor registered.\n\nEquivalent to `start/4` except that the `gen_statem` process\nis not registered with any [name service](`t:server_name/0`).","ref":"gen_statem.html#start/3"},{"type":"function","title":"gen_statem.start/4","doc":"Start a server, registered but not linked.\n\nCreates a standalone `gen_statem` process according to\nOTP design principles (using `m:proc_lib` primitives).\nAs it does not get linked to the calling process,\nthis start function cannot be used by a supervisor to start a child.\n\nFor a description of arguments and return values,\nsee [`start_link/4`](`start_link/4`).","ref":"gen_statem.html#start/4"},{"type":"function","title":"gen_statem.start_link/3","doc":"Start a server, linked but not registered.\n\nEquivalent to `start_link/4` except that the `gen_statem` process\nis not registered with any [name service](`t:server_name/0`).","ref":"gen_statem.html#start_link/3"},{"type":"function","title":"gen_statem.start_link/4","doc":"Start a server, linked and registered.\n\nCreates a `gen_statem` process according to OTP design principles\n(using `m:proc_lib` primitives) that is spawned linked to\nthe calling process.  This is essential when the `gen_statem`\nmust be part of a supervision tree so it gets linked to its supervisor.\n\nThe spawned `gen_statem` process calls [`Module:init/1`](`c:init/1`)\nto initialize the server.  To ensure a synchronized startup procedure,\n`start_link/3,4` does not return until [`Module:init/1`](`c:init/1`)\nhas returned or failed.\n\n`ServerName` specifies the `t:server_name/0` to register\nfor the `gen_statem` process.  If the `gen_statem` process is started with\n[`start_link/3`](`start_link/3`), no `ServerName` is provided and the\n`gen_statem` process is not registered.\n\n`Module` is the name of the callback module.\n\n`Args` is an arbitrary term that is passed as the argument to\n[`Module:init/1`](`c:init/1`).\n\n#### Start options in `Opts` {: #start-options }\n\n- **[`{timeout, Time}`](`t:start_opt/0`)** - The `gen_statem` process\n  is allowed to spend `Time` milliseconds before returning\n  from [`Module:init/1`](`c:init/1`), or it is terminated\n  and this start function returns [`{error, timeout}`](`t:start_ret/0`).\n\n- **[`{spawn_opt, SpawnOpts}`](`t:start_opt/0`)** -\n  `SpawnOpts` is passed as option list to `erlang:spawn_opt/2`,\n  which is used to spawn the `gen_statem` process.\n  See `t:proc_lib:start_spawn_option/0`.\n\n  > #### Note {: .info }\n  >\n  > Using spawn option `monitor` is not allowed,\n  > it causes a `badarg` failure.\n\n- **[`{hibernate_after, HibernateAfterTimeout}`](`t:enter_loop_opt/0`)** -\n  When the `gen_statem` process waits for a message, if no message\n  is received within `HibernateAfterTimeout` milliseconds,\n  the process goes into hibernation automatically\n  (by calling `proc_lib:hibernate/3`).  This option is also\n  allowed for the [`enter_loop`](`enter_loop/6`) functions.\n\n  Note that there is also a `t:transition_option/0`\n  to explicitly hibernate the server from a\n  [_state callback_](#state-callback).\n\n- **[`{debug, Dbgs}`](`t:enter_loop_opt/0`)** - Activates\n  debugging through `m:sys`.  For every entry in `Dbgs`,\n  the corresponding function in `m:sys` is called. This option is also\n  allowed for the [`enter_loop`](`enter_loop/6`) functions.\n\n#### Return values {: #start-return-values }\n\n- **[`{ok, Pid}`](`t:start_ret/0`)** -\n  The `gen_statem` server process was successfully created and\n  initialized.  `Pid` is the `t:pid/0` of the process.\n\n- **[`ignore`](`t:start_ret/0`)** -\n  [`Module:init/1`](`c:init/1`) returned [`ignore`](`t:init_result/1`).\n  The `gen_statem` process has exited with reason `normal`.\n\n- **[`{error, {already_started, OtherPid}}`](`t:start_ret/0`)** -\n  A process with the specified [`ServerName`](`t:server_name/0`)\n  already exists.  `OtherPid` is the `t:pid/0` of that process.\n  The `gen_statem` process exited with reason `normal`\n  before calling [`Module:init/1`](`c:init/1`).\n\n- **[`{error, timeout}`](`t:start_ret/0`)** -\n  [`Module:init/1`](`c:init/1`) did not return within\n  the [start time-out](`t:start_opt/0`).  The `gen_statem` process\n  has been killed with [`exit(_, kill)`](`erlang:exit/2`).\n\n- **[`{error, Reason}`](`t:start_ret/0`)**\n  + Either [`Module:init/1`](`c:init/1`) returned\n    [`{stop, Reason}`](`t:init_result/1`) or failed with reason `Reason`,\n    The `gen_statem` process exited with reason `Reason`.\n  + Or [`Module:init/1`](`c:init/1`) returned\n    [`{error, Reason}`](`t:init_result/1`).\n    The `gen_statem` process did a graceful exit with reason `normal`.\n\nIf the return value is `ignore` or `{error, _}`, the started\n`gen_statem` process has terminated.  If an `'EXIT'` message\nwas delivered to the calling process (due to the process link),\nthat message has been consumed.\n\n> #### Warning {: .warning }\n>\n> Before OTP 26.0, if the started `gen_statem` process returned e.g.\n> `{stop, Reason}` from [`Module:init/1`](`c:init/1`),\n> this function could return `{error, Reason}`\n> _before_ the started `gen_statem` process had terminated,\n> so starting again might fail because VM resources\n> such as the registered name was not yet unregistered,\n> and an `'EXIT'` message could arrive later to the\n> process calling this function.\n>\n> But if the started `gen_statem` process instead failed during\n> [`Module:init/1`](`c:init/1`), a process link `{'EXIT', Pid, Reason}`\n> message caused this function to return `{error, Reason}`,\n> so the `'EXIT'` message had been consumed and\n> the started `gen_statem` process had terminated.\n>\n> Since it was impossible to tell the difference between these two cases\n> from `start_link/3,4`'s return value, this inconsistency\n> was cleaned up in OTP 26.0.","ref":"gen_statem.html#start_link/4"},{"type":"type","title":"gen_statem.start_mon_ret/0","doc":"Return value from the [`start_monitor/3,4`](`start_monitor/3`) functions.\n\nAs for [`start_link/4`](#start-return-values) but a succesful return\nwraps the process ID and the [monitor reference](`erlang:monitor/2`) in a\n`{ok, {`[`pid()`](`t:pid/0`)`, `[`reference()`](`t:reference/0`)`}}`\ntuple.","ref":"gen_statem.html#t:start_mon_ret/0"},{"type":"function","title":"gen_statem.start_monitor/3","doc":"Start a server, monitored but neither linked nor registered.\n\nEquivalent to `start_monitor/4` except that the `gen_statem`\nprocess is not registered with any [name service](`t:server_name/0`).","ref":"gen_statem.html#start_monitor/3"},{"type":"function","title":"gen_statem.start_monitor/4","doc":"Start a server, monitored and registered, but not linked.\n\nCreates a standalone `gen_statem` process according to\nOTP design principles (using `m:proc_lib` primitives),\nand atomically sets up a monitor to the newly created process.\n\nAs the started process does not get linked to the calling process,\nthis start function cannot be used by a supervisor to start a child.\n\nFor a description of arguments and return values, see\n[`start_link/4`](`start_link/4`), but note that for a succesful start\nthe return value differs since this function returns `{ok, {Pid, Mon}}`,\nwhere `Pid` is the process identifier of the process,\nand `Mon` is the monitor reference for the process.\nIf the start is not successful, the caller will be blocked\nuntil the `DOWN` message has been received\nand removed from the caller's message queue.","ref":"gen_statem.html#start_monitor/4"},{"type":"type","title":"gen_statem.start_opt/0","doc":"Server [start options](#start-options) for the\n[`start/3,4`](`start/3`), [`start_link/3,4`](`start_link/3`),\nand [`start_monitor/3,4`](`start_monitor/3`) functions.\n\nSee [`start_link/4`](#start-options).","ref":"gen_statem.html#t:start_opt/0"},{"type":"type","title":"gen_statem.start_ret/0","doc":"[Return value](#start-return-values) from the [`start/3,4`](`start/3`)\nand [`start_link/3,4`](`start_link/3`) functions.\n\nSee [`start_link/4`](#start-return-values).","ref":"gen_statem.html#t:start_ret/0"},{"type":"type","title":"gen_statem.state/0","doc":"State name or state term.\n\nIf the [_callback mode_](`t:callback_mode/0`) is `handle_event_function`,\nthe state can be any term. After a _state change_ (`NextState =/= State`),\nall postponed events are retried.\n\nComparing two states for strict equality is assumed to be a fast operation,\nsince for every _state transition_ the `gen_statem` engine has to deduce\nif it is a  _state change_.\n\n> #### Note {: .info }\n> The smaller the state term, in general, the faster the comparison.\n>\n> Note that if the \"same\" state term is returned for a state transition\n> (or a return action without a `NextState` field is used),\n> the comparison for equality is always fast because that can be seen\n> from the term handle.\n>\n> But if a newly constructed state term is returned,\n> both the old and the new state terms will have to be traversed\n> until an inequality is found, or until both terms\n> have been fully traversed.\n>\n> So it is possible to use large state terms that are fast to compare,\n> but very easy to accidentally mess up.  Using small state terms is\n> the safe choice.","ref":"gen_statem.html#t:state/0"},{"type":"type","title":"gen_statem.state_callback_result/2","doc":"Return value from any [_state callback_](#state-callback).\n\n`ActionType` is `t:enter_action/0` if the state callback\nwas called with a [_state enter call_](`t:state_enter/0`),\nand `t:action/0` if the state callback was called with an event.\n\n- **`{keep_state, NewData [, Actions]}`** - The same as\n  `{next_state, CurrentState, NewData [, Actions]}`.\n\n- **`keep_state_and_data | {keep_state_and_data, Actions}`** -\n  The same as `{keep_state, CurrentData [, Actions]}`.\n\n- **`{repeat_state, NewData [, Actions]}`** - If the `gen_statem`\n  runs with [_state enter calls_](`t:state_enter/0`),\n  the _state enter call_ is repeated, see type `t:transition_option/0`.\n  Other than that `{repeat_state, NewData [, Actions]}` is the same as\n  `{keep_state, NewData [, Actions]}`.\n\n- **`repeat_state_and_data | {repeat_state_and_data, Actions}`** -\n  The same as `{repeat_state, CurrentData [, Actions]}`.\n\n- **`{stop, Reason [, NewData]}`** - Terminates the `gen_statem`\n  by calling [`Module:terminate/3`](`c:terminate/3`)\n  with `Reason` and `NewData`, if specified. An exit signal\n  with this reason is sent to linked processes and ports.\n\n- **`stop`** - The same as `{stop, normal}`.\n\n- **`{stop_and_reply, Reason, Replies [, NewData]}`** -\n  Sends all `Replies`, then terminates the `gen_statem`\n  like with `{stop, Reason [, NewData]}`.\n\nAll these terms are tuples or atoms and will be so\nin all future versions of `gen_statem`.","ref":"gen_statem.html#t:state_callback_result/2"},{"type":"type","title":"gen_statem.state_enter/0","doc":"[_Callback mode_](`t:callback_mode/0`) modifier\nfor _state enter calls_: the atom `state_enter`.\n\nBoth _callback modes_ can use _state enter calls_,\nand this is selected by adding this `state_enter` flag\nto the [_callback mode_](`t:callback_mode/0`) return value from\n[`Module:callback_mode/0`](`c:callback_mode/0`).\n\nIf [`Module:callback_mode/0`](`c:callback_mode/0`) returns\na list containing `state_enter`, the `gen_statem` engine will,\nat every _state change_, that is; `NextState =/= CurrentState`,\ncall the [_state callback_](#state-callback) with arguments\n`(enter, OldState, Data)` or `(enter, OldState, State, Data)`,\ndepending on the [_callback mode_](`t:callback_mode/0`).\n\nThis may look like an event but is really a call performed\nafter the previous [_state callback_](#state-callback) returned,\nand before any event is delivered to the new\n[_state callback_](#state-callback).\nSee [`Module:StateName/3`](`c:'StateName'/3`) and\n[`Module:handle_event/4`](`c:handle_event/4`).  A _state enter call_\nmay be repeated without doing a _state change_ by returning\na [`repeat_state`](`t:state_callback_result/2`) or\n[`repeat_state_and_data`](`t:state_callback_result/2`) action\nfrom the _state callback_.\n\nIf [`Module:callback_mode/0`](`c:callback_mode/0`) does not return\na list containing `state_enter`, no _state enter calls_ are done.\n\nIf [`Module:code_change/4`](`c:code_change/4`) should transform the state,\nit is regarded as a state rename and not a _state change_,\nwhich will not cause a _state enter call_.\n\nNote that a _state enter call_ **will** be done right before entering\nthe initial state, which may be seen as a state change from no state\nto the initial state. In this case `OldState =:= State`,\nwhich cannot happen for a subsequent state change,\nbut will happen when repeating the _state enter call_.","ref":"gen_statem.html#t:state_enter/0"},{"type":"type","title":"gen_statem.state_enter_result/1","doc":"","ref":"gen_statem.html#t:state_enter_result/1"},{"type":"type","title":"gen_statem.state_enter_result/2","doc":"Return value from a [_state callback_](#state-callback)\nafter a _state enter call_.\n\n`State` is the current state and it cannot be changed\nsince the state callback  was called with a\n[_state enter call_](`t:state_enter/0`).\n\n- **`{next_state, State, NewData [, Actions]}`** -\n  The `gen_statem` does a state transition to `State`, which has to be\n  equal to the current state, sets `NewData`, and executes all `Actions`.","ref":"gen_statem.html#t:state_enter_result/2"},{"type":"type","title":"gen_statem.state_name/0","doc":"State name in [_callback mode_](`t:callback_mode/0`) `state_functions`.\n\nIf the [_callback mode_](`t:callback_mode/0`) is `state_functions`,\nthe state must be an atom. After a _state change_ (`NextState =/= State`),\nall postponed events are retried.  Note that the state `terminate`\nis not possible to use since it would collide with the optional\ncallback function [`Module:terminate/3`](`c:terminate/3`).","ref":"gen_statem.html#t:state_name/0"},{"type":"type","title":"gen_statem.state_timeout/0","doc":"How long to wait in the current state.\n\nStarts a timer set by `t:timeout_action/0`, or\n`{state_timeout, Time, EventContent [, Options]}`.\n\nWhen the timer expires an event of `t:event_type/0` `state_timeout`\nwill be generated. See `erlang:start_timer/4` for how `Time`\nand [`Options`](`t:timeout_option/0`) are interpreted. Future\n`erlang:start_timer/4` `Options` will not necessarily be supported.\n\nA _state change_ cancels this timer, if it is running.\nThat is, if the `t:timeout_action/0` that starts this timer\nis part of a list of `t:action/0`s for a _state change_,\n`NextState =/= CurrentState`, the timer runs in the **`NextState`**.\n\nIf the state machine stays in that new state, now the current state,\nthe timer will run until it expires, which creates the time-out event.\nIf the state machine changes states from the now current state,\nthe timer is cancelled.  During the _state change_ from\nthe now current state, a new _state time-out_ may be started\nfor the next **`NextState`**.\n\nIf the `t:timeout_action/0` that starts this timer\nis part of a list of `t:action/0`s for a _state transition_\nthat is not a _state change_, the timer runs in the current state.\n\nIf `Time` is `infinity`, no timer is started,\nas it never would expire anyway.\n\nIf `Time` is relative and `0` no timer is actually started,\ninstead the the time-out event is enqueued to ensure\nthat it gets processed before any not yet received external event.\n\nSetting this timer while it is running will restart it\nwith the new time-out value.  Therefore it is possible\nto cancel this time-out by setting it to `infinity`.\nIt can also be cancelled more explicitly with\n[`{state_timeout, cancel}`](`t:timeout_cancel_action/0`).\n\nThe timer `EventContent` can be updated with the\n[`{state_timeout, update, NewEventContent}`](`t:timeout_update_action/0`)\naction without affecting the time of expiry.","ref":"gen_statem.html#t:state_timeout/0"},{"type":"function","title":"gen_statem.stop/1","doc":"","ref":"gen_statem.html#stop/1"},{"type":"function","title":"gen_statem.stop/3","doc":"Stop a server.\n\nOrders the `gen_statem` [`ServerRef`](`t:server_ref/0`) to exit with the\nspecified `Reason` and waits for it to terminate. The `gen_statem` calls\n[`Module:terminate/3`](`c:terminate/3`) before exiting.\n\nThis function returns `ok` if the server terminates\nwith the expected reason.  Any other reason than `normal`, `shutdown`,\nor `{shutdown, Term}` causes an error report to be issued\nthrough `m:logger`.  An exit signal with the same reason is\nsent to linked processes and ports. The default `Reason` is `normal`.\n\n`Timeout` is an integer > 0, which specifies how many milliseconds\nto wait for the server to terminate, or the atom `infinity`\nto wait indefinitely.  Defaults to `infinity`.\nIf the server does not terminate within the specified time,\nthe call exits the calling process with reason `timeout`.\n\nIf the process does not exist, the call exits the calling process\nwith reason `noproc`, or with reason `{nodedown, Node}`\nif the connection fails to the remote `Node` where the server runs.","ref":"gen_statem.html#stop/3"},{"type":"callback","title":"gen_statem.terminate/3","doc":"Handle state machine termination.\n\nThis function is called by a `gen_statem` when it is about to terminate.\nIt is to be the opposite of [`Module:init/1`](`c:init/1`)\nand do any necessary cleaning up.  When it returns, the `gen_statem`\nterminates with `Reason`.  The return value is ignored.\n\n`Reason` is a term denoting the stop reason and [`State`](`t:state/0`)\nis the internal state of the `gen_statem`.\n\n`Reason` depends on why the `gen_statem` is terminating.  If it is because\nanother callback function has returned, a stop tuple `{stop, Reason}` in\n[`Actions`](`t:action/0`), `Reason` has the value specified in that tuple.\nIf it is because of a failure, `Reason` is the error reason.\n\nIf the `gen_statem` is part of a supervision tree and is ordered by its\nsupervisor to terminate, this function is called with `Reason = shutdown`\nif both the following conditions apply:\n\n- The `gen_statem` process has been set to trap exit signals.\n- The shutdown strategy as defined in the supervisor's\n  child specification is an integer time-out value, not `brutal_kill`.\n\nEven if the `gen_statem` is _not_ part of a supervision tree,\nthis function is called if it receives an `'EXIT'` message\nfrom its parent. `Reason` is the same as in the `'EXIT'` message.\n\nIf the `gen_statem` process is not set up to trap\nexit signals it is immediately terminated, just like any process,\nand this function is not called.\n\nNotice that for any other reason than `normal`, `shutdown`, or\n`{shutdown, Term}`, the `gen_statem` is assumed to terminate\nbecause of an error and an error report is issued using `m:logger`.\n\nWhen the `gen_statem` process exits, an exit signal\nwith the same reason is sent to linked processes and ports,\njust as for any process.","ref":"gen_statem.html#c:terminate/3"},{"type":"type","title":"gen_statem.timeout_action/0","doc":"Event time-out, generic time-outs or state time-out.\n\nThese _transition actions_ can be invoked by returning them from the\n[_state callback_](#state-callback), from\n[`Module:init/1`](`c:init/1`) or by passing them to\n[`enter_loop/4,5,6`](`enter_loop/6`).\n\nThese time-out actions sets time-out\n[transition options](`t:transition_option/0`).\n\n- **`Time`** - Short for `{timeout, Time, Time}`, that is,\n  the time-out message is the time-out time. This form exists to allow the\n  [_state callback_](#state-callback) return value\n  `{next_state, NextState, NewData, Time}` like in `gen_fsm`.\n\n- **`{timeout, Time, EventContent [, Options]}`** -\n  Sets the `t:transition_option/0` `t:event_timeout/0` to `Time`\n  with `EventContent`, and time-out options\n  [`Options`](`t:timeout_option/0`).\n\n- **`{{timeout,Name}, Time, EventContent [, Options]}`** -\n  Sets the `t:transition_option/0` `t:generic_timeout/0` to `Time`\n  for time-out `Name` with `EventContent`, and time-out options\n  [`Options`](`t:timeout_option/0`).\\\n  **Since OTP 20.0**.\n\n- **`{state_timeout, Time, EventContent [, Options]}`** -\n  Sets the `t:transition_option/0` `t:state_timeout/0` to `Time`\n  with `EventContent`, and time-out options\n  [`Options`](`t:timeout_option/0`).\\\n  **Since OTP 19.3**.","ref":"gen_statem.html#t:timeout_action/0"},{"type":"type","title":"gen_statem.timeout_cancel_action/0","doc":"Clearer way to cancel a time-out than the original\nsetting it to 'infinity'.\n\nIt has always been possible to cancel a time-out using\n`t:timeout_action/0` with `Time = infinity`, since setting a new\ntime-out time overrides a running timer, and since setting the time\nto `infinity` is optimized to not setting a timer (that never\nwill expire).  Using this action shows the intention more clearly.","ref":"gen_statem.html#t:timeout_cancel_action/0"},{"type":"type","title":"gen_statem.timeout_event_type/0","doc":"[Event time-out](`t:event_timeout/0`),\n[generic time-out](`t:generic_timeout/0`),\nor [state time-out](`t:state_timeout/0`).\n\nThe time-out event types that the state machine can generate\nfor itself with the corresponding `t:timeout_action/0`s:\n\n| Time-out type     | Action                         | Event type        |\n|-------------------|--------------------------------|-------------------|\n| Event time-out    | `{timeout, Time, ...}`         | `timeout`         |\n| Generic time-out  | `{{timeout, Name}, Time, ...}` | `{timeout, Name}` |\n| State time-out    | `{state_timeout, Time, ...}`   | `state_timeout`   |\n\nIn short; the action to set a time-out with\n[`EventType`](`t:timeout_event_type/0`) is `{EventType, Time, ...}`.","ref":"gen_statem.html#t:timeout_event_type/0"},{"type":"type","title":"gen_statem.timeout_option/0","doc":"Time-out timer start option, to select absolute time of expiry.\n\nIf `Abs` is `true` an absolute timer is started,\nand if it is `false` a relative, which is the default.\nSee [`erlang:start_timer/4`](`erlang:start_timer/4`) for details.","ref":"gen_statem.html#t:timeout_option/0"},{"type":"type","title":"gen_statem.timeout_update_action/0","doc":"Update the `EventContent` without affecting the time of expiry.\n\nSets a new `EventContent` for a running time-out timer.\nSee [timeout_action()](`t:timeout_action/0`) for how to start a time-out.\n\nIf no time-out of this type is active, instead inserts\nthe time-out event just like when starting a time-out\nwith relative `Time = 0`.  This is a time-out autostart with\nimmediate expiry, so there will be noise for example\nif a generic time-out name was misspelled.","ref":"gen_statem.html#t:timeout_update_action/0"},{"type":"type","title":"gen_statem.transition_option/0","doc":"_State transition_ options set by [actions](`t:action/0`).\n\nThese determine what happens during the _state transition_.\nThe _state transition_ takes place when the\n[_state callback_](#state-callback) has processed an event\nand returns. Here are the sequence of steps for a _state transition_:\n\n1. All returned [actions](`t:action/0`) are processed\n   in order of appearance.  In this step all replies generated\n   by any `t:reply_action/0` are sent.  Other actions set\n   `t:transition_option/0`s that come into play in subsequent steps.\n\n2. If [_state enter calls_](`t:state_enter/0`) are used,\n   it is either the initial state or one of the callback results\n   [`repeat_state`](`t:state_callback_result/2`) or\n   [`repeat_state_and_data`](`t:state_callback_result/2`) is used the\n   `gen_statem` engine calls the current _state callback_ with arguments\n   [`(enter, State, Data)`](`t:state_enter/0`) or\n   [`(enter, State, State, Data)`](`t:state_enter/0`) (depending on\n   [_callback mode_](`t:callback_mode/0`)) and when it returns\n   starts again from the top of this sequence.\n\n   If [_state enter calls_](`t:state_enter/0`) are used,\n   and the state changes, the `gen_statem` engine calls\n   the new _state callback_ with arguments\n   [`(enter, OldState, Data)`](`t:state_enter/0`) or\n   [`(enter, OldState, State, Data)`](`t:state_enter/0`) (depending on\n   [_callback mode_](`t:callback_mode/0`)) and when it returns\n   starts again from the top of this sequence.\n\n3. If `t:postpone/0` is `true`, the current event is postponed.\n\n4. If this is a _state change_, the queue of incoming events is reset\n   to start with the oldest postponed.\n\n5. All events stored with `t:action/0` `next_event` are inserted\n   to be processed before previously queued events.\n\n6. Time-out timers `t:event_timeout/0`, `t:generic_timeout/0` and\n   `t:state_timeout/0` are handled.  Time-outs with zero time\n   are guaranteed to be delivered to the state machine\n   before any external not yet received event so if there is\n   such a time-out requested, the corresponding time-out zero event\n   is enqueued as the newest received event; that is after\n   already queued events such as inserted and postponed events.\n\n   Any event cancels an `t:event_timeout/0` so a zero time event time-out\n   is only generated if the event queue is empty.\n\n   A _state change_ cancels a `t:state_timeout/0` and any new transition\n   option of this type belongs to the new state, that is;\n   a `t:state_timeout/0` applies to the state the state machine enters.\n\n7. If there are enqueued events the\n   [_state callback_](#state-callback) for the possibly\n   new state is called with the oldest enqueued event, and we start again\n   from the top of this sequence.\n\n8. Otherwise the `gen_statem` goes into `receive` or hibernation\n   (if `t:hibernate/0` is `true`) to wait for the next message.\n   In hibernation the next non-system event awakens the `gen_statem`,\n   or rather the next incoming message awakens the `gen_statem`,\n   but if it is a system event it goes right back into hibernation.\n   When a new message arrives the\n   [_state callback_](#state-callback) is called with\n   the corresponding event, and we start again\n   from the top of this sequence.\n\n> #### Note {: .info }\n> The behaviour of a zero time-out (a time-out with time `0`)\n> differs subtly from Erlang's `receive ... after 0 ... end`.\n>\n> The latter receives one message if there is one,\n> while using the `t:timeout_action/0` `{timeout, 0}` does not\n> receive any external event.\n>\n> `m:gen_server`'s time-out works like Erlang's\n> `receive ... after 0 ... end`, in contrast to `gen_statem`.","ref":"gen_statem.html#t:transition_option/0"},{"type":"function","title":"gen_statem.wait_response/1","doc":"","ref":"gen_statem.html#wait_response/1"},{"type":"function","title":"gen_statem.wait_response/2","doc":"Wait for a request response.\n\nWaits for the response to the request identifier `ReqId`.  The request\nmust have been made by `send_request/2` to the `gen_statem` process.\nThis function must be called from the same process from which\n`send_request/2` was called.\n\n`WaitTime` specifies how long to wait for a reply.\nIf no reply is received within the specified time,\nthe function returns `timeout` and no cleanup is done,\nThus the function can be invoked repeatedly until a reply is returned.\n\nSee [`call/3`](#call-reply) about how the request is handled\nand the `Reply` is sent by the `gen_statem` server.\n\nIf the `gen_statem` server process is dead or dies while\nthis function waits for the reply, it returns an `error` return\nwith the exit `Reason`.\n\nThe difference between `receive_response/2` and\n`wait_response/2` is that `receive_response/2` abandons\nthe request at time-out so that a potential future response is ignored,\nwhile `wait_response/2` does not.","ref":"gen_statem.html#wait_response/2"},{"type":"function","title":"gen_statem.wait_response/3","doc":"Wait for any request response in a collection.\n\nWaits for a response in `ReqIdCollection`.  All request identifiers\nof `ReqIdCollection` must correspond to requests that have been made\nusing `send_request/2` or `send_request/4`, and all requests\nmust have been made by the process calling this function.\n\nThe `Label` in the response is the `Label` associated with\nthe request identifier that the response corresponds to.\nThe `Label` of a request identifier is associated\nwhen [adding the request id](`reqids_add/3`) to a collection,\nor when sending the request using `send_request/4`.\n\nCompared to `wait_response/2`, the returned result or exception\nassociated with a specific request identifier will be wrapped\nin a 3-tuple `{Response, Label, NewReqIdCollection}`.\n`Response` is the value that would have been produced\nby `wait_response/2`, `Label` is the value associated with\nthe specific [request identifier](`t:request_id/0`)\nand `NewReqIdCollection` is a possibly modified\nrequest identifier collection.\n\nIf `ReqIdCollection` is empty, `no_request` is returned.\n\nIf no response is received before `WaitTime` has expired,\n`timeout` is returned.  It is valid to continue waiting\nfor a response as many times as needed up until a response\nhas been received and completed by `check_response()`,\n`receive_response()`, or `wait_response()`.\n\nThe difference between `receive_response/3` and `wait_response/3`\nis that `receive_response/3` abandons requests at time-out\nso that potential future responses are ignored,\nwhile `wait_response/3` does not.\n\nIf `Delete` is `true`, the association with `Label`\nhas been deleted from `ReqIdCollection` in the resulting\n`NewReqIdCollection`.  If `Delete` is `false`, `NewReqIdCollection`\nwill equal`ReqIdCollection`.  Note that deleting an association\nis not for free and that a collection containing already handled\nrequests can still be used by subsequent calls to\n`wait_response/3`, `check_response/3`, and `receive_response/3`.\n\nHowever, without deleting handled associations,\nthe above calls will not be able to detect when there are\nno more outstanding requests to handle, so you will have to keep track\nof this some other way than relying on a `no_request` return.\nNote that if you pass a collection only containing\nassociations of already handled or abandoned requests\nto this function, it will always block until `WaitTime` expires\nand then return `timeout`.","ref":"gen_statem.html#wait_response/3"},{"type":"module","title":"log_mf_h","doc":"An event handler that logs events to disk.\n\nThis module is a `gen_event` handler module that can be installed in any\n`gen_event` process. It logs onto disk all events that are sent to an event\nmanager. Each event is written as a binary, which makes the logging very fast.\nHowever, a tool such as the Report Browser (`m:rb`) must be used to read the\nfiles. The events are written to multiple files. When all files have been used,\nthe first one is reused and overwritten. The directory location, the number of\nfiles, and the size of each file are configurable. The directory will include\none file called `index`, and report files `1, 2, ...`.","ref":"log_mf_h.html"},{"type":"module","title":"See Also - log_mf_h","doc":"`m:gen_event`, `m:rb`","ref":"log_mf_h.html#module-see-also"},{"type":"opaque","title":"log_mf_h.args/0","doc":"Term to be sent to `gen_event:add_handler/3`.","ref":"log_mf_h.html#t:args/0"},{"type":"type","title":"log_mf_h.b/0","doc":"","ref":"log_mf_h.html#t:b/0"},{"type":"type","title":"log_mf_h.f/0","doc":"","ref":"log_mf_h.html#t:f/0"},{"type":"function","title":"log_mf_h.init/3","doc":"","ref":"log_mf_h.html#init/3"},{"type":"function","title":"log_mf_h.init/4","doc":"Initiates the event handler. Returns `Args`, which is to be used in a call to\n[`gen_event:add_handler(EventMgr, log_mf_h, Args)`](`gen_event:add_handler/3`).\n\n`Dir` specifies which directory to use for the log files. `MaxBytes` specifies\nthe size of each individual file. `MaxFiles` specifies how many files are used.\n`Pred` is a predicate function used to filter the events. If no predicate\nfunction is specified, all events are logged.","ref":"log_mf_h.html#init/4"},{"type":"type","title":"log_mf_h.pred/0","doc":"","ref":"log_mf_h.html#t:pred/0"},{"type":"module","title":"pool","doc":"Load distribution facility.\n\nThis module can be used to run a set of Erlang nodes as a pool of computational\nprocessors. It is organized as a master and a set of slave nodes and includes\nthe following features:\n\n- The slave nodes send regular reports to the master about their current load.\n- Queries can be sent to the master to determine which node will have the least\n  load.\n\nThe BIF [`statistics(run_queue)`](`statistics/1`) is used for estimating future\nloads. It returns the length of the queue of ready to run processes in the\nErlang runtime system.\n\nThe slave nodes are started with the `m:slave` module. This effects terminal\nI/O, file I/O, and code loading.\n\nIf the master node fails, the entire pool exits.\n\n[](){: #files }","ref":"pool.html"},{"type":"module","title":"Files - pool","doc":"`.hosts.erlang` is used to pick hosts where nodes can be started. For\ninformation about format and location of this file, see `net_adm:host_file/0`.\n\n`$HOME/.erlang.slave.out.HOST` is used for all extra I/O that can come from the\nslave nodes on standard I/O. If the startup procedure does not work, this file\ncan indicate the reason.","ref":"pool.html#module-files"},{"type":"function","title":"pool.attach/1","doc":"Ensures that a pool master is running and includes `Node` in the pool master's\npool of nodes.","ref":"pool.html#attach/1"},{"type":"function","title":"pool.get_node/0","doc":"Returns the node with the expected lowest future load.","ref":"pool.html#get_node/0"},{"type":"function","title":"pool.get_nodes/0","doc":"Returns a list of the current member nodes of the pool.","ref":"pool.html#get_nodes/0"},{"type":"function","title":"pool.pspawn/3","doc":"Spawns a process on the pool node that is expected to have the lowest future\nload.","ref":"pool.html#pspawn/3"},{"type":"function","title":"pool.pspawn_link/3","doc":"Spawns and links to a process on the pool node that is expected to have the\nlowest future load.","ref":"pool.html#pspawn_link/3"},{"type":"function","title":"pool.start/1","doc":"","ref":"pool.html#start/1"},{"type":"function","title":"pool.start/2","doc":"Starts a new pool.\n\nThe file `.hosts.erlang` is read to find host names where the\npool nodes can be started; see section [Files](`m:pool#module-files`). The startup\nprocedure fails if the file is not found.\n\nThe slave nodes are started with [`slave:start/2,3`](`slave:start/2`), passing\nalong `Name` and, if provided, `Args`. `Name` is used as the first part of the\nnode names, `Args` is used to specify command-line arguments.\n\nAccess rights must be set so that all nodes in the pool have the authority to\naccess each other.\n\nThe function is synchronous and all the nodes, and all the system servers, are\nrunning when it returns a value.","ref":"pool.html#start/2"},{"type":"function","title":"pool.stop/0","doc":"Stops the pool and kills all the slave nodes.","ref":"pool.html#stop/0"},{"type":"module","title":"proc_lib","doc":"Functions for asynchronous and synchronous start of processes adhering to the\nOTP design principles.\n\nThis module is used to start processes adhering to the\n[OTP Design Principles](`e:system:design_principles.md`). Specifically, the\nfunctions in this module are used by the OTP standard behaviors (for example,\n`m:gen_server` and `m:gen_statem`) when starting new processes. The functions can\nalso be used to start _special processes_, user-defined processes that comply to\nthe OTP design principles. For an example, see section\n[sys and proc_lib](`e:system:spec_proc.md`) in OTP Design Principles.\n\nSome useful information is initialized when a process starts. The registered\nnames, or the process identifiers, of the parent process, and the parent\nancestors, are stored together with information about the function initially\ncalled in the process.\n\nWhile in \"plain Erlang\", a process is said to terminate normally only for exit\nreason `normal`, a process started using `m:proc_lib` is also said to terminate\nnormally if it exits with reason `shutdown` or `{shutdown,Term}`. `shutdown` is\nthe reason used when an application (supervision tree) is stopped.\n\nWhen a process that is started using `m:proc_lib` terminates abnormally (that is,\nwith another exit reason than `normal`, `shutdown`, or `{shutdown,Term}`), a\n_crash report_ is generated, which is written to terminal by the default logger\nhandler setup by Kernel. For more information about how crash reports were\nlogged prior to Erlang/OTP 21.0, see\n[SASL Error Logging](`e:sasl:error_logging.md`) in the SASL User's Guide.\n\nUnlike in \"plain Erlang\", `m:proc_lib` processes will not generate _error\nreports_, which are written to the terminal by the emulator. All exceptions are\nconverted to _exits_ which are ignored by the default `logger` handler.\n\nThe crash report contains the previously stored information, such as ancestors\nand initial function, the termination reason, and information about other\nprocesses that terminate as a result of this process terminating.","ref":"proc_lib.html"},{"type":"module","title":"See Also - proc_lib","doc":"`m:logger`","ref":"proc_lib.html#module-see-also"},{"type":"type","title":"proc_lib.dict_or_pid/0","doc":"","ref":"proc_lib.html#t:dict_or_pid/0"},{"type":"type","title":"proc_lib.exception/0","doc":"An exception passed to `init_fail/3`. See `erlang:raise/3` for a description\nof `Class`, `Reason` and `Stacktrace`.","ref":"proc_lib.html#t:exception/0"},{"type":"function","title":"proc_lib.format/1","doc":"Equivalent to [`format(CrashReport, latin1)`](`format/2`).","ref":"proc_lib.html#format/1"},{"type":"function","title":"proc_lib.format/2","doc":"> #### Note {: .info }\n>\n> This function is deprecated in the sense that the `error_logger` is no longer\n> the preferred interface for logging in Erlang/OTP. A new\n> [logging API](`e:kernel:logger_chapter.md`) was added in Erlang/OTP 21.0, but\n> legacy `error_logger` handlers can still be used. New Logger handlers do not\n> need to use this function, since the formatting callback (`report_cb`) is\n> included as metadata in the log event.\n\nThis function can be used by a user-defined legacy `error_logger` event handler\nto format a crash report. The crash report is sent using `m:logger`, and the\nevent to be handled is of the format\n`{error_report, GL, {Pid, crash_report, CrashReport}}`, where `GL` is the group\nleader pid of process `Pid` that sent the crash report.","ref":"proc_lib.html#format/2"},{"type":"function","title":"proc_lib.format/3","doc":"> #### Note {: .info }\n>\n> This function is deprecated in the sense that the `error_logger` is no longer\n> the preferred interface for logging in Erlang/OTP. A new\n> [logging API](`e:kernel:logger_chapter.md`) was added in Erlang/OTP 21.0, but\n> legacy `error_logger` handlers can still be used. New Logger handlers do not\n> need to used this function, since the formatting callback (`report_cb`) is\n> included as metadata in the log event.\n\nThis function can be used by a user-defined legacy `error_logger` event handler\nto format a crash report. When Depth is specified as a positive integer, it is\nused in the format string to limit the output as follows:\n`io_lib:format(\"~P\", [Term,Depth])`.","ref":"proc_lib.html#format/3"},{"type":"function","title":"proc_lib.get_label/1","doc":"Returns either `undefined` or the label for the process Pid set with\n[`proc_lib:set_label/1`](`set_label/1`).","ref":"proc_lib.html#get_label/1"},{"type":"function","title":"proc_lib.hibernate/3","doc":"This function does the same as (and does call) the\n[`hibernate/3`](`erlang:hibernate/3`) BIF, but ensures that exception handling\nand logging continues to work as expected when the process wakes up.\n\nAlways use this function instead of the BIF for processes started using\n`proc_lib` functions.","ref":"proc_lib.html#hibernate/3"},{"type":"function","title":"proc_lib.init_ack/1","doc":"Equivalent to [`init_ack(Parent, Ret)`](`init_ack/2`) where `Parent` is\nthe process that called `start/5`.","ref":"proc_lib.html#init_ack/1"},{"type":"function","title":"proc_lib.init_ack/2","doc":"This function must only be used by a process that has been started by a\n[`start[_link|_monitor]/3,4,5`](`start/5`) function. It tells `Parent` that the\nprocess has initialized itself and started.\n\nFunction [`init_ack/1`](`init_ack/1`) uses the parent value previously stored by\nthe start function used.\n\nIf neither this function nor [`init_fail/2,3`](`init_fail/3`) is called by the\nstarted process, the start function returns an error tuple when the started\nprocess exits, or when the start function time-out (if used) has passed, see\n[`start/3,4,5`](`start/5`).\n\n> #### Warning {: .warning }\n>\n> Do not use this function to return an error indicating that the process start\n> failed. When doing so the start function can return before the failing process\n> has exited, which may block VM resources required for a new start attempt to\n> succeed. Use [`init_fail/2,3`](`init_fail/3`) for that purpose.\n\nThe following example illustrates how this function and `proc_lib:start_link/3`\nare used:\n\n```erlang\n-module(my_proc).\n-export([start_link/0]).\n-export([init/1]).\n\nstart_link() ->\n    proc_lib:start_link(my_proc, init, [self()]).\n\ninit(Parent) ->\n    case do_initialization() of\n        ok ->\n            proc_lib:init_ack(Parent, {ok, self()});\n        {error, Reason} ->\n            exit(Reason)\n    end,\n    loop().\n\n...\n```","ref":"proc_lib.html#init_ack/2"},{"type":"function","title":"proc_lib.init_fail/2","doc":"Equivalent to [`init_fail(Parent, Return, Exception)`](`init_fail/3`) where\n`Parent` is the process that called `start/5`.","ref":"proc_lib.html#init_fail/2"},{"type":"function","title":"proc_lib.init_fail/3","doc":"This function must only be used by a process that has been started by a\n[`start[_link|_monitor]/3,4,5`](`start/3`) function. It tells `Parent` that the\nprocess has failed to initialize, and immediately raises an exception according\nto `Exception`. The start function then returns `Ret`.\n\nSee `erlang:raise/3` for a description of `Class`, `Reason` and `Stacktrace`.\n\n> #### Warning {: .warning }\n>\n> Do not consider catching the exception from this function. That would defeat\n> its purpose. A process started by a [`start[_link|_monitor]/3,4,5`](`start/3`)\n> function should end in a value (that will be ignored) or an exception that\n> will be handled by this module. See [Description](`m:proc_lib`).\n\nIf neither this function nor [`init_ack/1,2`](`init_ack/1`) is called by the\nstarted process, the start function returns an error tuple when the started\nprocess exits, or when the start function time-out (if used) has passed, see\n[`start/3,4,5`](`start/3`).\n\nThe following example illustrates how this function and `proc_lib:start_link/3`\ncan be used:\n\n```erlang\n-module(my_proc).\n-export([start_link/0]).\n-export([init/1]).\n\nstart_link() ->\n    proc_lib:start_link(my_proc, init, [self()]).\n\ninit(Parent) ->\n    case do_initialization() of\n        ok ->\n            proc_lib:init_ack(Parent, {ok, self()});\n        {error, Reason} = Error ->\n            proc_lib:init_fail(Parent, Error, {exit, normal})\n    end,\n    loop().\n\n...\n```","ref":"proc_lib.html#init_fail/3"},{"type":"function","title":"proc_lib.initial_call/1","doc":"Extracts the initial call of a process that was started using one of the spawn\nor start functions in this module. `Process` can either be a pid, an integer\ntuple (from which a pid can be created), or the process information of a process\n`Pid` fetched through an `erlang:process_info(Pid)` function call.\n\n> #### Note {: .info }\n>\n> The list `Args` no longer contains the arguments, but the same number of atoms\n> as the number of arguments; the first atom is `'Argument__1'`, the second\n> `'Argument__2'`, and so on. The reason is that the argument list could waste a\n> significant amount of memory, and if the argument list contained funs, it\n> could be impossible to upgrade the code for the module.\n>\n> If the process was spawned using a fun, [`initial_call/1`](`initial_call/1`)\n> no longer returns the fun, but the module, function for the local function\n> implementing the fun, and the arity, for example,\n> `{some_module,-work/3-fun-0-,0}` (meaning that the fun was created in function\n> `some_module:work/3`). The reason is that keeping the fun would prevent code\n> upgrade for the module, and that a significant amount of memory could be\n> wasted.","ref":"proc_lib.html#initial_call/1"},{"type":"function","title":"proc_lib.set_label/1","doc":"Set a label for the current process. The primary purpose is to aid in debugging\nunregistered processes. The process label can be used in tools and crash reports\nto identify processes but it doesn't have to be unique or an atom, as a\nregistered name needs to be. The process label can be any term, for example\n`{worker_process, 1..N}`.\n\nUse [`proc_lib:get_label/1`](`get_label/1`) to lookup the process description.","ref":"proc_lib.html#set_label/1"},{"type":"function","title":"proc_lib.spawn/1","doc":"","ref":"proc_lib.html#spawn/1"},{"type":"function","title":"proc_lib.spawn/2","doc":"","ref":"proc_lib.html#spawn/2"},{"type":"function","title":"proc_lib.spawn/3","doc":"","ref":"proc_lib.html#spawn/3"},{"type":"function","title":"proc_lib.spawn/4","doc":"Spawns a new process and initializes it as described in the beginning of this\nmanual page. The process is spawned using the [`spawn`](`erlang:spawn/1`) BIFs.","ref":"proc_lib.html#spawn/4"},{"type":"function","title":"proc_lib.spawn_link/1","doc":"","ref":"proc_lib.html#spawn_link/1"},{"type":"function","title":"proc_lib.spawn_link/2","doc":"","ref":"proc_lib.html#spawn_link/2"},{"type":"function","title":"proc_lib.spawn_link/3","doc":"","ref":"proc_lib.html#spawn_link/3"},{"type":"function","title":"proc_lib.spawn_link/4","doc":"Spawns a new process and initializes it as described in the beginning of this\nmanual page. The process is spawned using the\n[`spawn_link`](`erlang:spawn_link/1`) BIFs.","ref":"proc_lib.html#spawn_link/4"},{"type":"function","title":"proc_lib.spawn_opt/2","doc":"","ref":"proc_lib.html#spawn_opt/2"},{"type":"function","title":"proc_lib.spawn_opt/3","doc":"","ref":"proc_lib.html#spawn_opt/3"},{"type":"function","title":"proc_lib.spawn_opt/4","doc":"","ref":"proc_lib.html#spawn_opt/4"},{"type":"function","title":"proc_lib.spawn_opt/5","doc":"Spawns a new process and initializes it as described in the beginning of this\nmanual page. The process is spawned using the\n[`erlang:spawn_opt`](`erlang:spawn_opt/2`) BIFs.","ref":"proc_lib.html#spawn_opt/5"},{"type":"type","title":"proc_lib.spawn_option/0","doc":"Equivalent to `t:erlang:spawn_opt_option/0`.","ref":"proc_lib.html#t:spawn_option/0"},{"type":"function","title":"proc_lib.start/3","doc":"","ref":"proc_lib.html#start/3"},{"type":"function","title":"proc_lib.start/4","doc":"","ref":"proc_lib.html#start/4"},{"type":"function","title":"proc_lib.start/5","doc":"Starts a new process synchronously. Spawns the process and waits for it to\nstart.\n\nTo indicate a succesful start, the started process _must_ call\n[`init_ack(Parent, Ret)`](`init_ack/2`) where `Parent` is the process that\nevaluates this function, or [`init_ack(Ret)`](`init_ack/1`). `Ret` is then\nreturned by this function.\n\nIf the process fails to start, it _must_ fail; preferably by calling\n[`init_fail(Parent, Ret, Exception)` ](`init_fail/3`) where `Parent` is the\nprocess that evaluates this function, or\n[`init_fail(Ret, Exception)`](`init_fail/2`). `Ret` is then returned by this\nfunction, and the started process fails with `Exception`.\n\nIf the process instead fails before calling `init_ack/1,2` or `init_fail/2,3`,\nthis function returns `{error, Reason}` where `Reason` depends a bit on the\nexception just like for a process link `{'EXIT',Pid,Reason}` message.\n\nIf `Time` is specified as an integer, this function waits for `Time`\nmilliseconds for the new process to call `init_ack/1,2` or `init_fail/2,3`,\notherwise the process gets killed and `Ret = {error, timeout}` is returned.\n\nArgument `SpawnOpts`, if specified, is passed as the last argument to the\n[`spawn_opt/4`](`erlang:spawn_opt/4`) BIF.\n\n> #### Note {: .info }\n>\n> Using spawn option `monitor` is not allowed. It causes the function to fail\n> with reason `badarg`.\n>\n> Using spawn option `link` will set a link to the spawned process, just like\n> [start_link/3,4,5](`start_link/3`).","ref":"proc_lib.html#start/5"},{"type":"function","title":"proc_lib.start_link/3","doc":"","ref":"proc_lib.html#start_link/3"},{"type":"function","title":"proc_lib.start_link/4","doc":"","ref":"proc_lib.html#start_link/4"},{"type":"function","title":"proc_lib.start_link/5","doc":"Starts a new process synchronously. Spawns the process and waits for it to\nstart. A link is atomically set on the newly spawned process.\n\n> #### Note {: .info }\n>\n> If the started process gets killed or crashes with a reason that is not\n> `normal`, the process link will kill the calling process so this function does\n> not return, unless the calling process traps exits. For example, if this\n> function times out it will kill the spawned process, and then the link might\n> kill the calling process.\n\nBesides setting a link on the spawned process this function behaves like\n[start/5](`start/5`).\n\nWhen the calling process traps exits; if this function returns due to the\nspawned process exiting (any error return), this function receives (consumes)\nthe `'EXIT'` message, also when this function times out and kills the spawned\nprocess.\n\n> #### Note {: .info }\n>\n> Using spawn option `monitor` is not allowed. It causes the function to fail\n> with reason `badarg`.","ref":"proc_lib.html#start_link/5"},{"type":"function","title":"proc_lib.start_monitor/3","doc":"","ref":"proc_lib.html#start_monitor/3"},{"type":"function","title":"proc_lib.start_monitor/4","doc":"","ref":"proc_lib.html#start_monitor/4"},{"type":"function","title":"proc_lib.start_monitor/5","doc":"Starts a new process synchronously. Spawns the process and waits for it to\nstart. A monitor is atomically set on the newly spawned process.\n\nBesides setting a monitor on the spawned process this function behaves like\n[start/5](`start/5`).\n\nThe return value is `{Ret, Mon}` where `Ret` corresponds to the `Ret` argument\nin the call to `init_ack/1,2` or `init_fail/2,3`, and `Mon` is the monitor\nreference of the monitor that has been set up.\n\nIf this function returns due to the spawned process exiting, that is returns any\nerror value, a `'DOWN'` message will be delivered to the calling process, also\nwhen this function times out and kills the spawned process.\n\n> #### Note {: .info }\n>\n> Using spawn option `monitor` is not allowed. It causes the function to fail\n> with reason `badarg`.\n>\n> Using spawn option `link` will set a link to the spawned process, just like\n> [start_link/3,4,5](`start_link/3`).","ref":"proc_lib.html#start_monitor/5"},{"type":"type","title":"proc_lib.start_spawn_option/0","doc":"A restricted set of [spawn options](`t:spawn_option/0`). Most notably `monitor`\nis _not_ part of these options.","ref":"proc_lib.html#t:start_spawn_option/0"},{"type":"function","title":"proc_lib.stop/1","doc":"Equivalent to [`stop(Process, normal, infinity)`](`stop/3`).","ref":"proc_lib.html#stop/1"},{"type":"function","title":"proc_lib.stop/3","doc":"Orders the process to exit with the specified `Reason` and waits for it to\nterminate.\n\nReturns `ok` if the process exits with the specified `Reason` within `Timeout`\nmilliseconds.\n\nIf the call times out, a `timeout` exception is raised.\n\nIf the process does not exist, a `noproc` exception is raised.\n\nThe implementation of this function is based on the `terminate` system message,\nand requires that the process handles system messages correctly. For information\nabout system messages, see `m:sys` and section\n[sys and proc_lib](`e:system:spec_proc.md`) in OTP Design Principles.","ref":"proc_lib.html#stop/3"},{"type":"function","title":"proc_lib.translate_initial_call/1","doc":"This function is used by functions `\\c:i/0` and `\\c:regs/0` to present process\ninformation.\n\nThis function extracts the initial call of a process that was started using one\nof the spawn or start functions in this module, and translates it to more useful\ninformation. `Process` can either be a pid, an integer tuple (from which a pid\ncan be created), or the process information of a process `Pid` fetched through\nan `erlang:process_info(Pid)` function call.\n\nIf the initial call is to one of the system-defined behaviors such as\n`gen_server` or `gen_event`, it is translated to more useful information. If a\n`gen_server` is spawned, the returned `Module` is the name of the callback\nmodule and `Function` is `init` (the function that initiates the new server).\n\nA `supervisor` and a `supervisor_bridge` are also `gen_server` processes. To\nreturn information that this process is a supervisor and the name of the\ncallback module, `Module` is `supervisor` and `Function` is the name of the\nsupervisor callback module. `Arity` is `1`, as the `init/1` function is called\ninitially in the callback module.\n\nBy default, `{proc_lib,init_p,5}` is returned if no information about the\ninitial call can be found. It is assumed that the caller knows that the process\nhas been spawned with the `proc_lib` module.","ref":"proc_lib.html#translate_initial_call/1"},{"type":"behaviour","title":"supervisor","doc":"Generic supervisor behavior.\n\nThis behavior module provides a supervisor, a process that supervises other\nprocesses called child processes. A child process can either be another\nsupervisor or a worker process. Worker processes are normally implemented using\none of the `m:gen_event`, `m:gen_server`, or `m:gen_statem` behaviors. A\nsupervisor implemented using this module has a standard set of interface\nfunctions and includes functionality for tracing and error reporting.\nSupervisors are used to build a hierarchical process structure called a\nsupervision tree, a nice way to structure a fault-tolerant application. For more\ninformation, see [Supervisor Behaviour](`e:system:sup_princ.md`) in OTP Design\nPrinciples.\n\nA supervisor expects the definition of which child processes to supervise to be\nspecified in a callback module exporting a predefined set of functions.\n\nUnless otherwise stated, all functions in this module fail if the specified\nsupervisor does not exist or if bad arguments are specified.\n\n[](){: #supervision_princ }","ref":"supervisor.html"},{"type":"behaviour","title":"Supervision Principles - supervisor","doc":"The supervisor is responsible for starting, stopping, and monitoring its child\nprocesses. The basic idea of a supervisor is that it must keep its child\nprocesses alive by restarting them when necessary.\n\nThe children of a supervisor are defined as a list of _child specifications_.\nWhen the supervisor is started, the child processes are started in order from\nleft to right according to this list. When the supervisor is going to terminate,\nit first terminates its child processes in reversed start order, from right to\nleft.\n\n[](){: #sup_flags }","ref":"supervisor.html#module-supervision-principles"},{"type":"behaviour","title":"Supervisor flags - supervisor","doc":"The supervisor properties are defined by the supervisor flags. The type\ndefinition for the supervisor flags is as follows:\n\n```erlang\nsup_flags() = #{strategy => strategy(),           % optional\n                intensity => non_neg_integer(),   % optional\n                period => pos_integer(),          % optional\n                auto_shutdown => auto_shutdown()} % optional\n```\n\n#### Restart Strategies\n\nA supervisor can have one of the following _restart strategies_ specified with\nthe `strategy` key in the above map:\n\n- `one_for_one` \\- If one child process terminates and is to be restarted, only\n  that child process is affected. This is the default restart strategy.\n- `one_for_all` \\- If one child process terminates and is to be restarted, all\n  other child processes are terminated and then all child processes are\n  restarted.\n- `rest_for_one` \\- If one child process terminates and is to be restarted, the\n  'rest' of the child processes (that is, the child processes after the\n  terminated child process in the start order) are terminated. Then the\n  terminated child process and all child processes after it are restarted.\n- `simple_one_for_one` \\- A simplified `one_for_one` supervisor, where all child\n  processes are dynamically added instances of the same process type, that is,\n  running the same code.\n\n  Functions `delete_child/2` and `restart_child/2` are invalid for\n  `simple_one_for_one` supervisors and return `{error,simple_one_for_one}` if\n  the specified supervisor uses this restart strategy.\n\n  Function `terminate_child/2` can be used for children under\n  `simple_one_for_one` supervisors by specifying the child's `t:pid/0` as the\n  second argument. If instead the child specification identifier is used,\n  [`terminate_child/2`](`terminate_child/2`) return\n  `{error,simple_one_for_one}`.\n\n  As a `simple_one_for_one` supervisor can have many children, it shuts them all\n  down asynchronously. This means that the children do their cleanup in\n  parallel, and therefore the order in which they are stopped is not defined.\n\n#### Restart intensity and period\n\nTo prevent a supervisor from getting into an infinite loop of child process\nterminations and restarts, a _maximum restart intensity_ is defined using two\ninteger values specified with keys `intensity` and `period` in the above map.\nAssuming the values `MaxR` for `intensity` and `MaxT` for `period`, then, if\nmore than `MaxR` restarts occur within `MaxT` seconds, the supervisor terminates\nall child processes and then itself. The termination reason for the supervisor\nitself in that case will be `shutdown`. `intensity` defaults to `1` and `period`\ndefaults to `5`.\n\n[](){: #auto_shutdown }\n\n#### Automatic Shutdown\n\nA supervisor can be configured to automatically shut itself down with exit\nreason `shutdown` when [significant children](`m:supervisor#significant_child`)\nterminate with the `auto_shutdown` key in the above map:\n\n- `never` \\- Automic shutdown is disabled. This is the default setting.\n\n  With `auto_shutdown` set to `never`, child specs with the `significant` flag\n  set to `true` are considered invalid and will be rejected.\n\n- `any_significant` \\- The supervisor will shut itself down when _any_\n  significant child terminates, that is, when a `transient` significant child\n  terminates normally or when a `temporary` significant child terminates\n  normally or abnormally.\n- `all_significant` \\- The supervisor will shut itself down when _all_\n  significant children have terminated, that is, when the _last active_\n  significant child terminates. The same rules as for `any_significant` apply.\n\nFor more information, see the section\n[Automatic Shutdown](`e:system:sup_princ.md#automatic-shutdown`) in Supervisor\nBehavior in OTP Design Principles.\n\n> #### Warning {: .warning }\n>\n> The automatic shutdown feature appeared in OTP 24.0, but applications using\n> this feature will also compile and run with older OTP versions.\n>\n> However, such applications, when compiled with an OTP version that predates\n> the appearance of the automatic shutdown feature, will leak processes because\n> the automatic shutdowns they rely on will not happen.\n>\n> It is up to implementors to take proper precautions if they expect that their\n> applications may be compiled with older OTP versions.\n\n[](){: #child_spec }","ref":"supervisor.html#module-supervisor-flags"},{"type":"behaviour","title":"Child specification - supervisor","doc":"The type definition of a child specification is as follows:\n\n```erlang\nchild_spec() = #{id => child_id(),             % mandatory\n                 start => mfargs(),            % mandatory\n                 restart => restart(),         % optional\n                 significant => significant(), % optional\n                 shutdown => shutdown(),       % optional\n                 type => worker(),             % optional\n                 modules => modules()}         % optional\n```\n\nThe old tuple format is kept for backwards compatibility, see `t:child_spec/0`,\nbut the map is preferred.\n\n- `id` is used to identify the child specification internally by the supervisor.\n\n  The `id` key is mandatory.\n\n  Notice that this identifier on occations has been called \"name\". As far as\n  possible, the terms \"identifier\" or \"id\" are now used but to keep backward\n  compatibility, some occurences of \"name\" can still be found, for example in\n  error messages.\n\n- `start` defines the function call used to start the child process. It must be\n  a module-function-arguments tuple `{M,F,A}` used as\n  [`apply(M,F,A)`](`apply/3`).\n\n  The start function _must create and link to_ the child process, and must\n  return `{ok,Child}` or `{ok,Child,Info}`, where `Child` is the pid of the\n  child process and `Info` any term that is ignored by the supervisor.\n\n  The start function can also return `ignore` if the child process for some\n  reason cannot be started, in which case the child specification is kept by the\n  supervisor (unless it is a temporary child) but the non-existing child process\n  is ignored.\n\n  If something goes wrong, the function can also return an error tuple\n  `{error,Error}`.\n\n  Notice that the `start_link` functions of the different behavior modules\n  fulfill the above requirements.\n\n  The `start` key is mandatory.\n\n- [](){: #restart } `restart` defines when a terminated child process must be\n  restarted. A `permanent` child process is always restarted. A `temporary`\n  child process is never restarted (even when the supervisor's restart strategy\n  is `rest_for_one` or `one_for_all` and a sibling's death causes the temporary\n  process to be terminated). A `transient` child process is restarted only if it\n  terminates abnormally, that is, with another exit reason than `normal`,\n  `shutdown`, or `{shutdown,Term}`.\n\n  The `restart` key is optional. If it is not specified, it defaults to\n  `permanent`.\n\n- [](){: #significant_child } `significant` defines if a child is considered\n  significant for [automatic self-shutdown](`m:supervisor#auto_shutdown`) of the\n  supervisor.\n\n  Setting this option to `true` when the [restart type](`m:supervisor#restart`)\n  is `permanent` is invalid. Also, it is considered invalid to start children\n  with this option set to `true` in a supervisor when the\n  [`auto_shutdown`](`m:supervisor#auto_shutdown`) supervisor flag is set to\n  `never`.\n\n  The `significant` key is optional. If it is not specified, it defaults to\n  `false`.\n\n- `shutdown` defines how a child process must be terminated. `brutal_kill` means\n  that the child process is unconditionally terminated using\n  [`exit(Child,kill)`](`exit/2`). An integer time-out value means that the\n  supervisor tells the child process to terminate by calling\n  [`exit(Child,shutdown)`](`exit/2`) and then wait for an exit signal with\n  reason `shutdown` back from the child process. If no exit signal is received\n  within the specified number of milliseconds, the child process is\n  unconditionally terminated using [`exit(Child,kill)`](`exit/2`).\n\n  If the child process is another supervisor, the shutdown time must be set to\n  `infinity` to give the subtree ample time to shut down.\n\n  > #### Warning {: .warning }\n  >\n  > Setting the shutdown time to anything other than `infinity` for a child of\n  > type `supervisor` can cause a race condition where the child in question\n  > unlinks its own children, but fails to terminate them before it is killed.\n\n  It is also allowed to set it to `infinity`, if the child process is a worker.\n\n  > #### Warning {: .warning }\n  >\n  > Be careful when setting the shutdown time to `infinity` when the child\n  > process is a worker. Because, in this situation, the termination of the\n  > supervision tree depends on the child process, it must be implemented in a\n  > safe way and its cleanup procedure must always return.\n\n  Notice that all child processes implemented using the standard OTP behavior\n  modules automatically adhere to the shutdown protocol.\n\n  The `shutdown` key is optional. If it is not specified, it defaults to `5000`\n  if the child is of type `worker` and it defaults to `infinity` if the child is\n  of type `supervisor`.\n\n- `type` specifies if the child process is a supervisor or a worker.\n\n  The `type` key is optional. If it is not specified, it defaults to `worker`.\n\n- `modules` is used by the release handler during code replacement to determine\n  which processes are using a certain module. As a rule of thumb, if the child\n  process is a `m:supervisor`, `m:gen_server` or, `m:gen_statem`, this is to be a list\n  with one element `[Module]`, where `Module` is the callback module. If the\n  child process is an event manager (`m:gen_event`) with a dynamic set of callback\n  modules, value `dynamic` must be used. For more information about release\n  handling, see [Release Handling](`e:system:release_handling.md`) in OTP Design\n  Principles.\n\n  The `modules` key is optional. If it is not specified, it defaults to `[M]`,\n  where `M` comes from the child's start `{M,F,A}`.\n\n- Internally, the supervisor also keeps track of the pid `Child` of the child\n  process, or `undefined` if no pid exists.","ref":"supervisor.html#module-child-specification"},{"type":"behaviour","title":"See Also - supervisor","doc":"`m:gen_event`, `m:gen_statem`, `m:gen_server`, `m:sys`","ref":"supervisor.html#module-see-also"},{"type":"type","title":"supervisor.auto_shutdown/0","doc":"","ref":"supervisor.html#t:auto_shutdown/0"},{"type":"function","title":"supervisor.check_childspecs/1","doc":"","ref":"supervisor.html#check_childspecs/1"},{"type":"function","title":"supervisor.check_childspecs/2","doc":"Takes a list of child specification as argument and returns `ok` if all of them\nare syntactically correct, otherwise `{error,Error}`.\n\nIf the `AutoShutdown` argument is not `undefined`, also\nchecks if the child specifications are allowed for the given\n[auto_shutdown](`m:supervisor#auto_shutdown`) option.","ref":"supervisor.html#check_childspecs/2"},{"type":"type","title":"supervisor.child/0","doc":"","ref":"supervisor.html#t:child/0"},{"type":"type","title":"supervisor.child_id/0","doc":"Not a `t:pid/0`.","ref":"supervisor.html#t:child_id/0"},{"type":"type","title":"supervisor.child_rec/0","doc":"","ref":"supervisor.html#t:child_rec/0"},{"type":"type","title":"supervisor.child_spec/0","doc":"The tuple format is kept for backward compatibility only. A map is preferred;\nsee more details [above](`m:supervisor#child_spec`).","ref":"supervisor.html#t:child_spec/0"},{"type":"type","title":"supervisor.children/0","doc":"","ref":"supervisor.html#t:children/0"},{"type":"function","title":"supervisor.count_children/1","doc":"Returns a [property list](`t:proplists:proplist/0`) containing the counts for each of\nthe following elements of the supervisor's child specifications and managed\nprocesses:\n\n- `specs` \\- The total count of children, dead or alive.\n- `active` \\- The count of all actively running child processes managed by this\n  supervisor. For a `simple_one_for_one` supervisors, no check is done to ensure\n  that each child process is still alive, although the result provided here is\n  likely to be very accurate unless the supervisor is heavily overloaded.\n- `supervisors` \\- The count of all children marked as `child_type = supervisor`\n  in the specification list, regardless if the child process is still alive.\n- `workers` \\- The count of all children marked as `child_type = worker` in the\n  specification list, regardless if the child process is still alive.","ref":"supervisor.html#count_children/1"},{"type":"function","title":"supervisor.delete_child/2","doc":"Tells supervisor `SupRef` to delete the child specification identified by `Id`.\nThe corresponding child process must not be running. Use `terminate_child/2` to\nterminate it.\n\nIf successful, the function returns `ok`. If the child specification identified\nby `Id` exists but the corresponding child process is running or is about to be\nrestarted, the function returns `{error,running}` or `{error,restarting}`,\nrespectively. If the child specification identified by `Id` does not exist, the\nfunction returns `{error,not_found}`.","ref":"supervisor.html#delete_child/2"},{"type":"function","title":"supervisor.get_childspec/2","doc":"Returns the child specification map for the child identified by `Id` under\nsupervisor `SupRef`. The returned map contains all keys, both mandatory and\noptional.","ref":"supervisor.html#get_childspec/2"},{"type":"callback","title":"supervisor.init/1","doc":"Whenever a supervisor is started using [`start_link/2,3`](`start_link/2`), this\nfunction is called by the new process to find out about restart strategy,\nmaximum restart intensity, and child specifications.\n\n`Args` is the `Args` argument provided to the start function.\n\n`SupFlags` is the supervisor flags defining the restart strategy and maximum\nrestart intensity for the supervisor. `[ChildSpec]` is a list of valid child\nspecifications defining which child processes the supervisor must start and\nmonitor. See the discussion in section\n[`Supervision Principles`](`m:supervisor#supervision_princ`) earlier.\n\nNotice that when the restart strategy is `simple_one_for_one`, the list of child\nspecifications must be a list with one child specification only. (The child\nspecification identifier is ignored.) No child process is then started during\nthe initialization phase, but all children are assumed to be started dynamically\nusing `start_child/2`.\n\nThe function can also return `ignore`.\n\nNotice that this function can also be called as a part of a code upgrade\nprocedure. Therefore, the function is not to have any side effects. For more\ninformation about code upgrade of supervisors, see section\n[Changing a Supervisor](`e:system:appup_cookbook.md#sup`) in OTP Design\nPrinciples.","ref":"supervisor.html#c:init/1"},{"type":"type","title":"supervisor.mfargs/0","doc":"Value `undefined` for `A` (the argument list) is only to be used internally in\n`m:supervisor`. If the restart type of the child is `temporary`, the process is\nnever to be restarted and therefore there is no need to store the real argument\nlist. Value `undefined` is then stored instead.","ref":"supervisor.html#t:mfargs/0"},{"type":"type","title":"supervisor.modules/0","doc":"","ref":"supervisor.html#t:modules/0"},{"type":"type","title":"supervisor.restart/0","doc":"","ref":"supervisor.html#t:restart/0"},{"type":"function","title":"supervisor.restart_child/2","doc":"Tells supervisor `SupRef` to restart a child process corresponding to the child\nspecification identified by `Id`. The child specification must exist, and the\ncorresponding child process must not be running.\n\nNotice that for temporary children, the child specification is automatically\ndeleted when the child terminates; thus, it is not possible to restart such\nchildren.\n\nIf the child specification identified by `Id` does not exist, the function\nreturns `{error,not_found}`. If the child specification exists but the\ncorresponding process is already running, the function returns\n`{error,running}`.\n\nIf the child process start function returns `{ok,Child}` or `{ok,Child,Info}`,\nthe pid is added to the supervisor and the function returns the same value.\n\nIf the child process start function returns `ignore`, the pid remains set to\n`undefined` and the function returns `{ok,undefined}`.\n\nIf the child process start function returns an error tuple or an erroneous\nvalue, or if it fails, the function returns `{error,Error}`, where `Error` is a\nterm containing information about the error.","ref":"supervisor.html#restart_child/2"},{"type":"type","title":"supervisor.shutdown/0","doc":"","ref":"supervisor.html#t:shutdown/0"},{"type":"type","title":"supervisor.significant/0","doc":"","ref":"supervisor.html#t:significant/0"},{"type":"function","title":"supervisor.start_child/2","doc":"Dynamically adds a child specification to supervisor `SupRef`, which starts the\ncorresponding child process.\n\nFor `one_for_one`, `one_for_all` and `rest_for_one` supervisors, the second\nargument must be a valid child specification `ChildSpec`. The child process\nis started by using the start function as defined in the child specification.\n\nFor `simple_one_for_one` supervisors, the child specification defined in\n[`Module:init/1`](`c:init/1`) is used, and the second argument must instead\nbe an arbitrary list of terms `ExtraArgs`. The child process is then started\nby appending `ExtraArgs` to the existing start function arguments, that is, by\ncalling [`apply(M, F, A++ExtraArgs)`](`apply/3`), where `{M,F,A}` is the start\nfunction defined in the child specification.\n\n- If there already exists a child specification with the specified identifier,\n  `ChildSpec` is discarded, and the function returns `{error,already_present}`\n  or `{error,{already_started,Child}}`, depending on if the corresponding child\n  process is running or not.\n- If the child process start function returns `{ok,Child}` or `{ok,Child,Info}`,\n  the child specification and pid are added to the supervisor and the function\n  returns the same value.\n- If the child process start function returns `ignore`, the child specification\n  `ChildSpec` is added to the supervisor if it is an `one_for_one`, `one_for_all`\n  or `rest_for_one` supervisor, and the pid is set to `undefined`. For\n  `simple_one_for_one` supervisors, no child is added to the supervisor. The\n  function returns `{ok,undefined}`.\n\nIf the child process start function returns an error tuple or an erroneous\nvalue, or if it fails, the child specification is discarded, and the function\nreturns `{error,Error}`, where `Error` is a term containing information about\nthe error and child specification.","ref":"supervisor.html#start_child/2"},{"type":"function","title":"supervisor.start_link/2","doc":"Creates a nameless supervisor process as part of a supervision tree.\n\nEquivalent to `start_link/3` except that the supervisor process is not\n[`registered`](`erlang:register/2`).","ref":"supervisor.html#start_link/2"},{"type":"function","title":"supervisor.start_link/3","doc":"Creates a supervisor process as part of a supervision tree.\n\nFor example, the function ensures that the supervisor is linked to the calling\nprocess (its supervisor).\n\nThe created supervisor process calls [`Module:init/1`](`c:init/1`) to find out\nabout restart strategy, maximum restart intensity, and child processes. To\nensure a synchronized startup procedure, `start_link/2,3` does not return until\n[`Module:init/1`](`c:init/1`) has returned and all child processes have been\nstarted.\n\n- If `SupName={local,Name}`, the supervisor is registered locally as `Name`\n  using [`register/2`](`register/2`).\n- If `SupName={global,Name}`, the supervisor is registered globally as `Name`\n  using `global:register_name/2`.\n- If `SupName={via,Module,Name}`, the supervisor is registered as `Name` using\n  the registry represented by `Module`. The `Module` callback must export the\n  functions `register_name/2`, `unregister_name/1`, and `send/2`, which must\n  behave like the corresponding functions in `m:global`. Thus,\n  `{via,global,Name}` is a valid reference.\n\n`Module` is the name of the callback module.\n\n`Args` is any term that is passed as the argument to\n[`Module:init/1`](`c:init/1`).\n\n- If the supervisor and its child processes are successfully created (that is,\n  if all child process start functions return `{ok,Child}`, `{ok,Child,Info}`,\n  or `ignore`), the function returns `{ok,Pid}`, where `Pid` is the pid of the\n  supervisor.\n- If there already exists a process with the specified `SupName`, the function\n  returns `{error,{already_started,Pid}}`, where `Pid` is the pid of that\n  process.\n- If [`Module:init/1`](`c:init/1`) returns `ignore`, this function returns\n  `ignore` as well, and the supervisor terminates with reason `normal`.\n- If [`Module:init/1`](`c:init/1`) fails or returns an incorrect value, this\n  function returns `{error,Term}`, where `Term` is a term with information about\n  the error, and the supervisor terminates with reason `Term`.\n- If any child process start function fails or returns an error tuple or an\n  erroneous value, the supervisor first terminates all already started child\n  processes with reason `shutdown` and then terminate itself and returns\n  `{error, {shutdown, Reason}}`.","ref":"supervisor.html#start_link/3"},{"type":"type","title":"supervisor.startchild_err/0","doc":"","ref":"supervisor.html#t:startchild_err/0"},{"type":"type","title":"supervisor.startchild_ret/0","doc":"","ref":"supervisor.html#t:startchild_ret/0"},{"type":"type","title":"supervisor.startlink_err/0","doc":"","ref":"supervisor.html#t:startlink_err/0"},{"type":"type","title":"supervisor.startlink_ret/0","doc":"","ref":"supervisor.html#t:startlink_ret/0"},{"type":"type","title":"supervisor.strategy/0","doc":"","ref":"supervisor.html#t:strategy/0"},{"type":"type","title":"supervisor.sup_flags/0","doc":"The tuple format is kept for backward compatibility only. A map is preferred;\nsee more details [above](`m:supervisor#sup_flags`).","ref":"supervisor.html#t:sup_flags/0"},{"type":"type","title":"supervisor.sup_name/0","doc":"Name specification to use when starting a `supervisor`. See function\n[`start_link/2,3`](`start_link/2`) and the type `t:sup_ref/0` below.\n\n- **`{local,LocalName}`** - Register the `supervisor` locally as `LocalName`\n  using [`register/2`](`erlang:register/2`).\n\n- **`{global,GlobalName}`** - Register the `supervisor` process id globally as\n  `GlobalName` using `global:register_name/2`.\n\n- **`{via,RegMod,ViaName}`** - Register the `supervisor` process with the\n  registry represented by `RegMod`. The `RegMod` callback is to export the\n  functions `register_name/2`, `unregister_name/1`, `whereis_name/1`, and\n  `send/2`, which are to behave like the corresponding functions in `m:global`.\n  Thus, `{via,global,GlobalName}` is a valid reference equivalent to\n  `{global,GlobalName}`.","ref":"supervisor.html#t:sup_name/0"},{"type":"type","title":"supervisor.sup_ref/0","doc":"Supervisor specification to use when addressing a `supervisor`. See\n[`count_children/1`](`count_children/1`), [`delete_child/2`](`delete_child/2`),\n[`get_childspec/2`](`get_childspec/2`), [`restart_child/2`](`restart_child/2`),\n[`start_child/2`](`start_child/2`), [`terminate_child/2`](`terminate_child/2`),\n[`which_children/1`](`which_children/1`) and the type `t:sup_name/0` above.\n\nIt can be:\n\n- **`t:pid/0`** - The `supervisor`'s process identifier.\n\n- **`LocalName`** - The `supervisor` is locally registered as `LocalName` with\n  [`register/2`](`erlang:register/2`).\n\n- **`{Name,Node}`** - The `supervisor` is locally registered on another node.\n\n- **`{global,GlobalName}`** - The `supervisor` is globally registered in\n  `m:global`.\n\n- **`{via,RegMod,ViaName}`** - The `supervisor` is registered in an alternative\n  process registry. The registry callback module `RegMod` is to export functions\n  `register_name/2`, `unregister_name/1`, `whereis_name/1`, and `send/2`, which\n  are to behave like the corresponding functions in `m:global`. Thus,\n  `{via,global,GlobalName}` is the same as `{global,GlobalName}`.","ref":"supervisor.html#t:sup_ref/0"},{"type":"function","title":"supervisor.terminate_child/2","doc":"Tells supervisor `SupRef` to terminate the specified child.\n\nIf the supervisor is not `simple_one_for_one`, `Id` must be the child\nspecification identifier. The process, if any, is terminated and, unless it is a\ntemporary child, the child specification is kept by the supervisor. The child\nprocess can later be restarted by the supervisor. The child process can also be\nrestarted explicitly by calling `restart_child/2`. Use `delete_child/2` to\nremove the child specification.\n\nIf the child is temporary, the child specification is deleted as soon as the\nprocess terminates. This means that [`delete_child/2`](`delete_child/2`) has no\nmeaning and [`restart_child/2`](`restart_child/2`) cannot be used for these\nchildren.\n\nIf the supervisor is `simple_one_for_one`, `Id` must be the `t:pid/0` of the\nchild process. If the specified process is alive, but is not a child of the\nspecified supervisor, the function returns `{error,not_found}`. If the child\nspecification identifier is specified instead of a `t:pid/0`, the function\nreturns `{error,simple_one_for_one}`.\n\nIf successful, the function returns `ok`. If there is no child specification\nwith the specified `Id`, the function returns `{error,not_found}`.","ref":"supervisor.html#terminate_child/2"},{"type":"function","title":"supervisor.which_children/1","doc":"Returns a newly created list with information about all child specifications and\nchild processes belonging to supervisor `SupRef`.\n\nNotice that calling this function when supervising many children under low\nmemory conditions can cause an out of memory exception.\n\nThe following information is given for each child specification/process:\n\n- `Id` \\- As defined in the child specification or `undefined` for a\n  `simple_one_for_one` supervisor.\n- `Child` \\- The pid of the corresponding child process, the atom `restarting`\n  if the process is about to be restarted, or `undefined` if there is no such\n  process.\n- `Type` \\- As defined in the child specification.\n- `Modules` \\- As defined in the child specification.","ref":"supervisor.html#which_children/1"},{"type":"type","title":"supervisor.worker/0","doc":"","ref":"supervisor.html#t:worker/0"},{"type":"behaviour","title":"supervisor_bridge","doc":"Generic supervisor bridge behavior.\n\nThis behavior module provides a supervisor bridge, a process that connects a\nsubsystem not designed according to the OTP design principles to a supervision\ntree. The supervisor bridge sits between a supervisor and the subsystem. It\nbehaves like a real supervisor to its own supervisor, but has a different\ninterface than a real supervisor to the subsystem. For more information, see\n[Supervisor Behaviour](`e:system:sup_princ.md`) in OTP Design Principles.\n\nA supervisor bridge assumes the functions for starting and stopping the\nsubsystem to be located in a callback module exporting a predefined set of\nfunctions.\n\nThe `m:sys` module can be used for debugging a supervisor bridge.\n\nUnless otherwise stated, all functions in this module fail if the specified\nsupervisor bridge does not exist or if bad arguments are specified.","ref":"supervisor_bridge.html"},{"type":"behaviour","title":"See Also - supervisor_bridge","doc":"`m:supervisor`, `m:sys`","ref":"supervisor_bridge.html#module-see-also"},{"type":"callback","title":"supervisor_bridge.init/1","doc":"Whenever a supervisor bridge is started using\n[`start_link/2,3`](`start_link/2`), this function is called by the new process\nto start the subsystem and initialize.\n\n`Args` is the `Args` argument provided to the start function.\n\nThe function is to return `{ok,Pid,State}`, where `Pid` is the pid of the main\nprocess in the subsystem and `State` is any term.\n\nIf later `Pid` terminates with a reason `Reason`, the supervisor bridge\nterminates with reason `Reason` as well. If later the supervisor bridge is\nstopped by its supervisor with reason `Reason`, it calls\n[`Module:terminate(Reason,State)`](`c:terminate/2`) to terminate.\n\nIf the initialization fails, the function is to return `{error,Error}`, where\n`Error` is any term, or `ignore`.","ref":"supervisor_bridge.html#c:init/1"},{"type":"function","title":"supervisor_bridge.start_link/2","doc":"Creates a nameless supervisor bridge process as part of a supervision tree.\n\nEquivalent to `start_link/3` except that the supervisor process is not\n[`registered`](`erlang:register/2`).","ref":"supervisor_bridge.html#start_link/2"},{"type":"function","title":"supervisor_bridge.start_link/3","doc":"Creates a supervisor bridge process, linked to the calling process, which calls\n[`Module:init/1`](`c:init/1`) to start the subsystem.\n\nTo ensure a synchronized startup procedure, this function does not return until\n[`Module:init/1`](`c:init/1`) has returned.\n\n- If `SupBridgeName={local,Name}`, the supervisor bridge is registered locally\n  as `Name` using [`register/2`](`register/2`).\n- If `SupBridgeName={global,GlobalName}`, the supervisor bridge is registered\n  globally as `GlobalName` using `global:register_name/2`.\n- If `SupBridgeName={via,Module,ViaName}`, the supervisor bridge is registered\n  as `ViaName` using a registry represented by Module. The `Module` callback is\n  to export functions `register_name/2`, `unregister_name/1`, and `send/2`,\n  which are to behave like the corresponding functions in `m:global`. Thus,\n  `{via,global,GlobalName}` is a valid reference.\n\n`Module` is the name of the callback module.\n\n`Args` is an arbitrary term that is passed as the argument to\n[`Module:init/1`](`c:init/1`).\n\n- If the supervisor bridge and the subsystem are successfully started, the\n  function returns `{ok,Pid}`, where `Pid` is is the pid of the supervisor\n  bridge.\n- If there already exists a process with the specified `SupBridgeName`, the\n  function returns `{error,{already_started,Pid}}`, where `Pid` is the pid of\n  that process.\n- If [`Module:init/1`](`c:init/1`) returns `ignore`, this function returns\n  `ignore` as well and the supervisor bridge terminates with reason `normal`.\n- If [`Module:init/1`](`c:init/1`) fails or returns an error tuple or an\n  incorrect value, this function returns `{error,Error}`, where `Error` is a\n  term with information about the error, and the supervisor bridge terminates\n  with reason `Error`.","ref":"supervisor_bridge.html#start_link/3"},{"type":"callback","title":"supervisor_bridge.terminate/2","doc":"This function is called by the supervisor bridge when it is about to terminate.\nIt is to be the opposite of [`Module:init/1`](`c:init/1`) and stop the subsystem\nand do any necessary cleaning up. The return value is ignored.\n\n`Reason` is `shutdown` if the supervisor bridge is terminated by its supervisor.\nIf the supervisor bridge terminates because a a linked process (apart from the\nmain process of the subsystem) has terminated with reason `Term`, then `Reason`\nbecomes `Term`.\n\n`State` is taken from the return value of [`Module:init/1`](`c:init/1`).","ref":"supervisor_bridge.html#c:terminate/2"},{"type":"behaviour","title":"sys","doc":"A functional interface to system messages.\n\nThis module contains functions for sending system messages used by programs, and\nmessages used for debugging purposes.\n\nFunctions used for implementation of processes are also expected to understand\nsystem messages, such as debug messages and code change. These functions must be\nused to implement the use of system messages for a process; either directly, or\nthrough standard behaviors, such as `m:gen_server`.\n\nThe default time-out is 5000 ms, unless otherwise specified. `timeout` defines\nthe time to wait for the process to respond to a request. If the process does\nnot respond, the function evaluates [`exit({timeout, {M, F, A}})`](`exit/1`).\n\n[](){: #dbg_opt }\n\nThe functions make references to a debug structure. The debug structure is a\nlist of `t:dbg_opt/0`, which is an internal data type used by function\n`handle_system_msg/6`. No debugging is performed if it is an empty list.","ref":"sys.html"},{"type":"behaviour","title":"System Messages - sys","doc":"Processes that are not implemented as one of the standard behaviors must still\nunderstand system messages. The following three messages must be understood:\n\n- Plain system messages. These are received as `{system, From, Msg}`. The\n  content and meaning of this message are not interpreted by the receiving\n  process module. When a system message is received, function\n  `handle_system_msg/6` is called to handle the request.\n- Shutdown messages. If the process traps exits, it must be able to handle a\n  shutdown request from its parent, the supervisor. The message\n  `{'EXIT', Parent, Reason}` from the parent is an order to terminate. The\n  process must terminate when this message is received, normally with the same\n  `Reason` as `Parent`.\n- If the modules used to implement the process change dynamically during\n  runtime, the process must understand one more message. An example is the\n  `m:gen_event` processes. The message is `{_Label, {From, Ref}, get_modules}`.\n  The reply to this message is `From ! {Ref, Modules}`, where `Modules` is a\n  list of the currently active modules in the process.\n\n  This message is used by the release handler to find which processes that\n  execute a certain module. The process can later be suspended and ordered to\n  perform a code change for one of its modules.","ref":"sys.html#module-system-messages"},{"type":"behaviour","title":"System Events - sys","doc":"When debugging a process with the functions of this module, the process\ngenerates _system_events_, which are then treated in the debug function. For\nexample, `trace` formats the system events to the terminal.\n\nFour predefined system events are used when a process receives or sends a\nmessage. The process can also define its own system events. It is always up to\nthe process itself to format these events.","ref":"sys.html#module-system-events"},{"type":"function","title":"sys.change_code/4","doc":"","ref":"sys.html#change_code/4"},{"type":"function","title":"sys.change_code/5","doc":"Tells the process to change code.\n\nThe process must be suspended to handle this message.\nArgument `Extra` is reserved for each process to use as its own.\nFunction [`Module:system_code_change/4`](`c:system_code_change/4`) is called.\n`OldVsn` is the old version of the `Module`.","ref":"sys.html#change_code/5"},{"type":"type","title":"sys.dbg_fun/0","doc":"","ref":"sys.html#t:dbg_fun/0"},{"type":"opaque","title":"sys.dbg_opt/0","doc":"See the introduction of this manual page.","ref":"sys.html#t:dbg_opt/0"},{"type":"type","title":"sys.debug_option/0","doc":"","ref":"sys.html#t:debug_option/0"},{"type":"function","title":"sys.debug_options/1","doc":"Can be used by a process that initiates a debug structure from a list of\noptions. The values of argument `Opt` are the same as for the corresponding\nfunctions.","ref":"sys.html#debug_options/1"},{"type":"type","title":"sys.format_fun/0","doc":"","ref":"sys.html#t:format_fun/0"},{"type":"function","title":"sys.get_debug/3","doc":"Gets the data associated with a debug option. `Default` is returned if `Item` is\nnot found. Can be used by the process to retrieve debug data for printing before\nit terminates.","ref":"sys.html#get_debug/3"},{"type":"function","title":"sys.get_log/1","doc":"Returns the logged system events in the debug structure, that is the last\nargument to `handle_debug/4`.","ref":"sys.html#get_log/1"},{"type":"function","title":"sys.get_state/1","doc":"","ref":"sys.html#get_state/1"},{"type":"function","title":"sys.get_state/2","doc":"Gets the state of the process.\n\n> #### Note {: .info }\n>\n> These functions are intended only to help with debugging. They are provided\n> for convenience, allowing developers to avoid having to create their own state\n> extraction functions and also avoid having to interactively extract the state\n> from the return values of `get_status/1` or `get_status/2` while debugging.\n\nThe value of `State` varies for different types of processes, as follows:\n\n- For a `m:gen_server` process, the returned `State` is the state of the\n  callback module.\n- For a `m:gen_statem` process, `State` is the tuple\n  `{CurrentState,CurrentData}`.\n- For a `m:gen_event` process, `State` is a list of tuples, where each tuple\n  corresponds to an event handler registered in the process and contains\n  `{Module, Id, HandlerState}`, as follows:\n\n  - **`Module`** - The module name of the event handler.\n\n  - **`Id`** - The ID of the handler (which is `false` if it was registered\n    without an ID).\n\n  - **`HandlerState`** - The state of the handler.\n\nIf the callback module exports a function\n[`system_get_state/1`](`c:system_get_state/1`), it is called in the target\nprocess to get its state. Its argument is the same as the `Misc` value returned\nby [`get_status/1,2`](`get_status/1`), and function\n[`Module:system_get_state/1`](`c:system_get_state/1`) is expected to extract the\nstate of the callback module from it. Function\n[`system_get_state/1`](`c:system_get_state/1`) must return `{ok, State}`, where\n`State` is the state of the callback module.\n\nIf the callback module does not export a\n[`system_get_state/1`](`c:system_get_state/1`) function, `get_state/1,2` assumes\nthat the `Misc` value is the state of the callback module and returns it\ndirectly instead.\n\nIf the callback module's [`system_get_state/1`](`c:system_get_state/1`) function\ncrashes or throws an exception, the caller exits with error\n`{callback_failed, {Module, system_get_state}, {Class, Reason}}`, where `Module`\nis the name of the callback module and `Class` and `Reason` indicate details of\nthe exception.\n\nFunction [`system_get_state/1`](`c:system_get_state/1`) is primarily useful for\nuser-defined behaviors and modules that implement OTP\n[special processes](`m:sys#process-implementation-functions`). The `m:gen_server`,\n`m:gen_statem`, and `m:gen_event` OTP behavior modules export this function, so\ncallback modules for those behaviors need not to supply their own.\n\nFor more information about a process, including its state, see `get_status/1`\nand `get_status/2`.","ref":"sys.html#get_state/2"},{"type":"function","title":"sys.get_status/1","doc":"","ref":"sys.html#get_status/1"},{"type":"function","title":"sys.get_status/2","doc":"Gets the status of the process.\n\nThe value of `Misc` varies for different types of processes, for example:\n\n- A `m:gen_server` process returns the state of the callback module.\n- A `m:gen_statem` process returns information, such as its current state name\n  and state data.\n- A `m:gen_event` process returns information about each of its registered\n  handlers.\n- A bare `m:sys` process returns the value passed as `Misc` to\n  `handle_system_message/6`.\n\nCallback modules for `m:gen_server`, `m:gen_statem`, and `m:gen_event` can also change\nthe value of `Misc` by exporting a function `format_status/1`, which contributes\nmodule-specific information. For details, see `c:gen_server:format_status/1`,\n`c:gen_statem:format_status/1`, and `c:gen_event:format_status/1`.","ref":"sys.html#get_status/2"},{"type":"function","title":"sys.handle_debug/4","doc":"This function is called by a process when it generates a system event.\n`FormFunc` is a formatting function, called as `FormFunc(Device, Event, Extra)`\nto print the events, which is necessary if tracing is activated. `Extra` is any\nextra information that the process needs in the format function, for example,\nthe process name.","ref":"sys.html#handle_debug/4"},{"type":"function","title":"sys.handle_system_msg/6","doc":"This function is used by a process module to take care of system messages. The\nprocess receives a `{system, From, Msg}` message and passes `Msg` and `From` to\nthis function.\n\nThis function _never_ returns. It calls either of the following functions:\n\n- [`Module:system_continue(Parent, NDebug, Misc)`](`c:system_continue/3`), where\n  the process continues the execution.\n- [`Module:system_terminate(Reason, Parent, Debug, Misc)`](`c:system_terminate/4`),\n  if the process is to terminate.\n\n`Module` must export the following:\n\n- [`system_continue/3`](`c:system_continue/3`)\n- [`system_terminate/4`](`c:system_terminate/4`)\n- [`system_code_change/4`](`c:system_code_change/4`)\n- [`system_get_state/1`](`c:system_get_state/1`)\n- [`system_replace_state/2`](`c:system_replace_state/2`)\n\nArgument `Misc` can be used to save internal data in a process, for example, its\nstate. It is sent to [`Module:system_continue/3`](`c:system_continue/3`) or\n[`Module:system_terminate/4`](`c:system_terminate/4`).","ref":"sys.html#handle_system_msg/6"},{"type":"function","title":"sys.install/2","doc":"","ref":"sys.html#install/2"},{"type":"function","title":"sys.install/3","doc":"Enables installation of alternative debug functions. An example of such a\nfunction is a trigger, a function that waits for some special event and performs\nsome action when the event is generated. For example, turning on low-level\ntracing.\n\n`Func` is called whenever a system event is generated. This function is to\nreturn `done`, or a new `Func` state. In the first case, the function is\nremoved. It is also removed if the function fails. If one debug function should\nbe installed more times, a unique `FuncId` must be specified for each\ninstallation.","ref":"sys.html#install/3"},{"type":"function","title":"sys.log/2","doc":"","ref":"sys.html#log/2"},{"type":"function","title":"sys.log/3","doc":"Turns the logging of system events on or off. If on, a maximum of `N` events are\nkept in the debug structure (default is 10).\n\nIf `Flag` is `get`, a list of all logged events is returned.\n\nIf `Flag` is `print`, the logged events are printed to\n[`standard_io`](`t:io:standard_io/0`).\n\nThe events are formatted with a function that is defined by the process that\ngenerated the event (with a call to [`handle_debug/4`)](`handle_debug/4`).","ref":"sys.html#log/3"},{"type":"function","title":"sys.log_to_file/2","doc":"","ref":"sys.html#log_to_file/2"},{"type":"function","title":"sys.log_to_file/3","doc":"Enables or disables the logging of all system events in text format to the file.\nThe events are formatted with a function that is defined by the process that\ngenerated the event (with a call to `handle_debug/4`). The file is opened with\nencoding UTF-8.","ref":"sys.html#log_to_file/3"},{"type":"type","title":"sys.name/0","doc":"","ref":"sys.html#t:name/0"},{"type":"function","title":"sys.no_debug/1","doc":"","ref":"sys.html#no_debug/1"},{"type":"function","title":"sys.no_debug/2","doc":"Turns off all debugging for the process. This includes functions that are\ninstalled explicitly with function [`install/2,3`](`install/2`), for example,\ntriggers.","ref":"sys.html#no_debug/2"},{"type":"function","title":"sys.print_log/1","doc":"Prints the logged system events in the debug structure, using `FormFunc` as\ndefined when the event was generated by a call to `handle_debug/4`.","ref":"sys.html#print_log/1"},{"type":"function","title":"sys.remove/2","doc":"","ref":"sys.html#remove/2"},{"type":"function","title":"sys.remove/3","doc":"Removes an installed debug function from the process. `Func` or `FuncId` must be\nthe same as previously installed.","ref":"sys.html#remove/3"},{"type":"function","title":"sys.replace_state/2","doc":"","ref":"sys.html#replace_state/2"},{"type":"function","title":"sys.replace_state/3","doc":"Replaces the state of the process, and returns the new state.\n\n> #### Note {: .info }\n>\n> These functions are intended only to help with debugging, and are not to be\n> called from normal code. They are provided for convenience, allowing\n> developers to avoid having to create their own custom state replacement\n> functions.\n\nFunction `StateFun` provides a new state for the process. Argument `State` and\nthe `NewState` return value of `StateFun` vary for different types of processes\nas follows:\n\n- For a `m:gen_server` process, `State` is the state of the callback module and\n  `NewState` is a new instance of that state.\n- For a `m:gen_statem` process, `State` is the tuple\n  `{CurrentState,CurrentData}`, and `NewState` is a similar tuple, which can\n  contain a new current state, new state data, or both.\n- For a `m:gen_event` process, `State` is the tuple `{Module, Id, HandlerState}`\n  as follows:\n\n  - **`Module`** - The module name of the event handler.\n\n  - **`Id`** - The ID of the handler (which is `false` if it was registered\n    without an ID).\n\n  - **`HandlerState`** - The state of the handler.\n\n  `NewState` is a similar tuple where `Module` and `Id` are to have the same\n  values as in `State`, but the value of `HandlerState` can be different.\n  Returning a `NewState`, whose `Module` or `Id` values differ from those of\n  `State`, leaves the state of the event handler unchanged. For a `m:gen_event`\n  process, `StateFun` is called once for each event handler registered in the\n  `m:gen_event` process.\n\nIf a `StateFun` function decides not to effect any change in process state, then\nregardless of process type, it can return its `State` argument.\n\nIf a `StateFun` function crashes or throws an exception, the original state of\nthe process is unchanged for `m:gen_server`, and `m:gen_statem` processes. For\n`m:gen_event` processes, a crashing or failing `StateFun` function means that only\nthe state of the particular event handler it was working on when it failed or\ncrashed is unchanged; it can still succeed in changing the states of other event\nhandlers registered in the same `m:gen_event` process.\n\nIf the callback module exports a `c:system_replace_state/2` function, it is\ncalled in the target process to replace its state using `StateFun`. Its two\narguments are `StateFun` and `Misc`, where `Misc` is the same as the `Misc`\nvalue returned by [`get_status/1,2`](`get_status/1`). A\n[`system_replace_state/2`](`c:system_replace_state/2`) function is expected to\nreturn `{ok, NewState, NewMisc}`, where `NewState` is the new state of the\ncallback module, obtained by calling `StateFun`, and `NewMisc` is a possibly new\nvalue used to replace the original `Misc` (required as `Misc` often contains the\nstate of the callback module within it).\n\nIf the callback module does not export a\n[`system_replace_state/2`](`c:system_replace_state/2`) function,\n[`replace_state/2,3`](`replace_state/2`) assumes that `Misc` is the state of the\ncallback module, passes it to `StateFun` and uses the return value as both the\nnew state and as the new value of `Misc`.\n\nIf the callback module's function\n[`system_replace_state/2`](`c:system_replace_state/2`) crashes or throws an\nexception, the caller exits with error\n`{callback_failed, {Module, system_replace_state}, {Class, Reason}}`, where\n`Module` is the name of the callback module and `Class` and `Reason` indicate\ndetails of the exception. If the callback module does not provide a\n[`system_replace_state/2`](`c:system_replace_state/2`) function and `StateFun`\ncrashes or throws an exception, the caller exits with error\n`{callback_failed, StateFun, {Class, Reason}}`.\n\nFunction [`system_replace_state/2`](`c:system_replace_state/2`) is primarily\nuseful for user-defined behaviors and modules that implement OTP\n[special processes](`m:sys#process-implementation-functions`). The OTP behavior\nmodules `m:gen_server`, `m:gen_statem`, and `m:gen_event` export this function, so\ncallback modules for those behaviors need not to supply their own.","ref":"sys.html#replace_state/3"},{"type":"function","title":"sys.resume/1","doc":"","ref":"sys.html#resume/1"},{"type":"function","title":"sys.resume/2","doc":"Resumes a suspended process.","ref":"sys.html#resume/2"},{"type":"function","title":"sys.statistics/2","doc":"","ref":"sys.html#statistics/2"},{"type":"function","title":"sys.statistics/3","doc":"Enables or disables the collection of statistics. If `Flag` is `get`, the\nstatistical collection is returned.","ref":"sys.html#statistics/3"},{"type":"function","title":"sys.suspend/1","doc":"","ref":"sys.html#suspend/1"},{"type":"function","title":"sys.suspend/2","doc":"Suspends the process. When the process is suspended, it only responds to other\nsystem messages, but not other messages.","ref":"sys.html#suspend/2"},{"type":"callback","title":"sys.system_code_change/4","doc":"Called from `handle_system_msg/6` when the process is to perform a code change.\nThe code change is used when the internal data structure has changed. This\nfunction converts argument `Misc` to the new data structure. `OldVsn` is\nattribute _vsn_ of the old version of the `Module`. If no such attribute is\ndefined, the atom `undefined` is sent.","ref":"sys.html#c:system_code_change/4"},{"type":"callback","title":"sys.system_continue/3","doc":"Called from `handle_system_msg/6` when the process is to continue its execution\n(for example, after it has been suspended). This function never returns.","ref":"sys.html#c:system_continue/3"},{"type":"type","title":"sys.system_event/0","doc":"Debug events produced by `m:gen_server`, `m:gen_statem` and `m:gen_event`\n\n- **`{in,Msg}`** - Is produced by `m:gen_server` and `m:gen_event` when the message\n  `Msg` arrives.\n\n- **`{in,Msg,State}`** - Is produced by `m:gen_statem` when the message `Msg`\n  arrives in state `State`.\n\n  For `m:gen_statem` the `Msg` term is an `{EventType,EventContent}` tuple.\n\n- **`{out,Msg,To}`** - Is produced by `m:gen_statem` when the reply `Msg` is sent\n  back to `To` by returning a `{reply,To,Msg}` action from the callback module.\n\n  `To` is of the same type as the first argument to `gen_statem:reply/2`.\n\n- **`{out,Msg,To,State}`** - Is produced by `m:gen_server` when the reply `Msg` is\n  sent back to `To` by returning a `{reply,...}` tuple from the callback module.\n\n  `To` is of the same type as the first argument to `gen_server:reply/2`.\n\n  `State` is the new server state.\n\n- **`{noreply,State}`** - Is produced by `m:gen_server` when a `{noreply,...}`\n  tuple is returned from the callback module.\n\n  `State` is the new server state.\n\n- **`{continue,Continuation}`** - Is produced by `m:gen_server` when a\n  `{continue,Continuation}` tuple is returned from the callback module.\n\n- **`{postpone,Event,State,NextState}`** - Is produced by `m:gen_statem` when the\n  message `Event` is postponed in state `State`. `NextState` is the new state.\n\n  `Event` is an `{EventType,EventContent}` tuple.\n\n- **`{consume,Event,State,NextState}`** - Is produced by `m:gen_statem` when the\n  message `Event` is consumed in state `State`. `NextState` is the new state.\n\n  `Event` is an `{EventType,EventContent}` tuple.\n\n- **`{start_timer,Action,State}`** - Is produced by `m:gen_statem` when the action\n  `Action` starts a timer in state `State`.\n\n- **`{insert_timeout,Event,State}`** - Is produced by `m:gen_statem` when a\n  timeout zero action inserts event `Event` in state `State`.\n\n  `Event` is an `{EventType,EventContent}` tuple.\n\n- **`{enter,Module,State}`** - Is produced by `m:gen_statem` when module `Module`\n  enters the first state `State`.\n\n- **`{module,Module,State}`** - Is produced by `m:gen_statem` when setting module\n  `Module` in state `State`.\n\n- **`{terminate,Reason,State}`** - Is produced by `m:gen_statem` when it\n  terminates with reason `Reason` in state `State`.","ref":"sys.html#t:system_event/0"},{"type":"callback","title":"sys.system_get_state/1","doc":"Called from `handle_system_msg/6` when the process is to return a term that\nreflects its current state. `State` is the value returned by `get_state/2`.","ref":"sys.html#c:system_get_state/1"},{"type":"callback","title":"sys.system_replace_state/2","doc":"Called from `handle_system_msg/6` when the process is to replace its current\nstate. `NState` is the value returned by `replace_state/3`.","ref":"sys.html#c:system_replace_state/2"},{"type":"callback","title":"sys.system_terminate/4","doc":"Called from `handle_system_msg/6` when the process is to terminate. For example,\nthis function is called when the process is suspended and its parent orders\nshutdown. It gives the process a chance to do a cleanup. This function never\nreturns.","ref":"sys.html#c:system_terminate/4"},{"type":"function","title":"sys.terminate/2","doc":"","ref":"sys.html#terminate/2"},{"type":"function","title":"sys.terminate/3","doc":"Orders the process to terminate with the specified `Reason`. The termination is\ndone asynchronously, so it is not guaranteed that the process is terminated when\nthe function returns.","ref":"sys.html#terminate/3"},{"type":"function","title":"sys.trace/2","doc":"","ref":"sys.html#trace/2"},{"type":"function","title":"sys.trace/3","doc":"Prints all system events on [`standard_io`](`t:io:standard_io/0`). The events\nare formatted with a function that is defined by the process that generated the\nevent (with a call to `handle_debug/4`).","ref":"sys.html#trace/3"},{"type":"module","title":"c","doc":"Command line interface module.\n\nThis module enables users to enter the short form of some commonly used\ncommands.\n\n> #### Note {: .info }\n>\n> These functions are intended for interactive use in the Erlang shell only. The\n> module prefix can be omitted.","ref":"c.html"},{"type":"module","title":"See Also - c","doc":"`m:filename`, `m:compile`, `m:erlang`, `m:yecc`, `m:xref`","ref":"c.html#module-see-also"},{"type":"function","title":"c.bt/1","doc":"Stack backtrace for a process. Equivalent to\n`erlang:process_display(Pid, backtrace)`.","ref":"c.html#bt/1"},{"type":"function","title":"c.c/1","doc":"Works like [`c(Module, [])`](`c/2`).","ref":"c.html#c/1"},{"type":"function","title":"c.c/2","doc":"Compiles and then purges and loads the code for a module. `Module` can be either\na module name or a source file path, with or without `.erl` extension.\n\nIf `Module` is a string, it is assumed to be a source file path, and the\ncompiler will attempt to compile the source file with the options `Options`. If\ncompilation fails, the old object file (if any) is deleted.\n\nIf `Module` is an atom, a source file with that exact name or with `.erl`\nextension will be looked for. If found, the source file is compiled with the\noptions `Options`. If compilation fails, the old object file (if any) is\ndeleted.\n\nIf `Module` is an atom and is not the path of a source file, then the code path\nis searched to locate the object file for the module and extract its original\ncompiler options and source path. If the source file is not found in the\noriginal location, `filelib:find_source/1` is used to search for it relative to\nthe directory of the object file.\n\nThe source file is compiled with the the original options appended to the given\n`Options`, the output replacing the old object file if and only if compilation\nsucceeds.\n\nNotice that purging the code means that any processes lingering in old code for\nthe module are killed without warning. For more information, see the `m:code`\nmodule.","ref":"c.html#c/2"},{"type":"function","title":"c.c/3","doc":"Compiles and then purges and loads the code for module `Module`, which must be\nan atom.\n\nThe code path is searched to locate the object file for module `Module` and\nextract its original compiler options and source path. If the source file is not\nfound in the original location, `filelib:find_source/1` is used to search for it\nrelative to the directory of the object file.\n\nThe source file is compiled with the the original options appended to the given\n`Options`, the output replacing the old object file if and only if compilation\nsucceeds. The function `Filter` specifies which elements to remove from the\noriginal compiler options before the new options are added. The `Filter` fun\nshould return `true` for options to keep, and `false` for options to remove.\n\nNotice that purging the code means that any processes lingering in old code for\nthe module are killed without warning. For more information, see the `m:code`\nmodule.","ref":"c.html#c/3"},{"type":"function","title":"c.cd/1","doc":"Changes working directory to `Dir`, which can be a relative name, and then\nprints the name of the new working directory.\n\n_Example:_\n\n```text\n2> cd(\"../erlang\").\n/home/ron/erlang\n```","ref":"c.html#cd/1"},{"type":"type","title":"c.cmd_line_arg/0","doc":"","ref":"c.html#t:cmd_line_arg/0"},{"type":"function","title":"c.erlangrc/1","doc":"Search `PathList` and load `.erlang` resource file if found.","ref":"c.html#erlangrc/1"},{"type":"function","title":"c.flush/0","doc":"Flushes any messages sent to the shell.","ref":"c.html#flush/0"},{"type":"function","title":"c.h/1","doc":"Print the documentation for `Module`","ref":"c.html#h/1"},{"type":"function","title":"c.h/2","doc":"Print the documentation for all `Module:Function`s (regardless of arity).","ref":"c.html#h/2"},{"type":"function","title":"c.h/3","doc":"Print the documentation for `Module:Function/Arity`.","ref":"c.html#h/3"},{"type":"type","title":"c.h_return/0","doc":"","ref":"c.html#t:h_return/0"},{"type":"function","title":"c.hcb/1","doc":"Print the callback documentation for `Module`","ref":"c.html#hcb/1"},{"type":"function","title":"c.hcb/2","doc":"Print the callback documentation for all `Module:Callback`s (regardless of\narity).","ref":"c.html#hcb/2"},{"type":"function","title":"c.hcb/3","doc":"Print the callback documentation for `Module:Callback/Arity`.","ref":"c.html#hcb/3"},{"type":"type","title":"c.hcb_return/0","doc":"","ref":"c.html#t:hcb_return/0"},{"type":"function","title":"c.help/0","doc":"Displays help information: all valid shell internal commands, and commands in\nthis module.","ref":"c.html#help/0"},{"type":"type","title":"c.hf_return/0","doc":"","ref":"c.html#t:hf_return/0"},{"type":"function","title":"c.ht/1","doc":"Print the type documentation for `Module`","ref":"c.html#ht/1"},{"type":"function","title":"c.ht/2","doc":"Print the type documentation for `Type` in `Module` regardless of arity.","ref":"c.html#ht/2"},{"type":"function","title":"c.ht/3","doc":"Print the type documentation for `Type/Arity` in `Module`.","ref":"c.html#ht/3"},{"type":"type","title":"c.ht_return/0","doc":"","ref":"c.html#t:ht_return/0"},{"type":"function","title":"c.i/0","doc":"","ref":"c.html#i/0"},{"type":"function","title":"c.i/3","doc":"Displays information about a process, Equivalent to\n[`process_info(pid(X, Y, Z))`](`process_info/1`), but location transparent.","ref":"c.html#i/3"},{"type":"function","title":"c.l/1","doc":"Purges and loads, or reloads, a module by calling `code:purge(Module)` followed\nby `code:load_file(Module)`.\n\nNotice that purging the code means that any processes lingering in old code for\nthe module are killed without warning. For more information, see `code/3`.","ref":"c.html#l/1"},{"type":"function","title":"c.lc/1","doc":"Compiles a list of files by calling\n`compile:file(File, [report_errors, report_warnings])` for each `File` in\n`Files`.\n\nFor information about `File`, see `t:file:filename/0`.","ref":"c.html#lc/1"},{"type":"function","title":"c.lm/0","doc":"Reloads all currently loaded modules that have changed on disk (see `mm/0`).\nReturns the list of results from calling [`l(M)`](`l/1`) for each such `M`.","ref":"c.html#lm/0"},{"type":"function","title":"c.ls/0","doc":"Lists files in the current directory.","ref":"c.html#ls/0"},{"type":"function","title":"c.ls/1","doc":"Lists files in directory `Dir` or, if `Dir` is a file, only lists it.","ref":"c.html#ls/1"},{"type":"function","title":"c.m/0","doc":"Displays information about the loaded modules, including the files from which\nthey have been loaded.","ref":"c.html#m/0"},{"type":"function","title":"c.m/1","doc":"Displays information about `Module`.","ref":"c.html#m/1"},{"type":"function","title":"c.memory/0","doc":"Memory allocation information. Equivalent to `erlang:memory/0`.","ref":"c.html#memory/0"},{"type":"function","title":"c.memory/1","doc":"Memory allocation information. Equivalent to `erlang:memory/1`.","ref":"c.html#memory/1"},{"type":"function","title":"c.mm/0","doc":"Lists all modified modules. Shorthand for `code:modified_modules/0`.","ref":"c.html#mm/0"},{"type":"function","title":"c.nc/1","doc":"","ref":"c.html#nc/1"},{"type":"function","title":"c.nc/2","doc":"Compiles and then loads the code for a file on all nodes. `Options` defaults to\n`[]`. Compilation is equivalent to:\n\n```erlang\ncompile:file(File, Options ++ [report_errors, report_warnings])\n```","ref":"c.html#nc/2"},{"type":"function","title":"c.ni/0","doc":"`i/0` displays system information, listing information about all processes.\n`ni/0` does the same, but for all nodes in the network.","ref":"c.html#ni/0"},{"type":"function","title":"c.nl/1","doc":"Loads `Module` on all nodes.","ref":"c.html#nl/1"},{"type":"function","title":"c.nregs/0","doc":"","ref":"c.html#nregs/0"},{"type":"function","title":"c.pid/3","doc":"Converts `X`, `Y`, `Z` to pid ` `. This function is only to be used when\ndebugging.","ref":"c.html#pid/3"},{"type":"function","title":"c.pwd/0","doc":"Prints the name of the working directory.","ref":"c.html#pwd/0"},{"type":"function","title":"c.q/0","doc":"This function is shorthand for `init:stop()`, that is, it causes the node to\nstop in a controlled fashion.","ref":"c.html#q/0"},{"type":"function","title":"c.regs/0","doc":"`regs/0` displays information about all registered processes. `nregs/0` does the\nsame, but for all nodes in the network.","ref":"c.html#regs/0"},{"type":"function","title":"c.uptime/0","doc":"Prints the node uptime (as specified by `erlang:statistics(wall_clock)`) in\nhuman-readable form.","ref":"c.html#uptime/0"},{"type":"function","title":"c.xm/1","doc":"Finds undefined functions, unused functions, and calls to deprecated functions\nin a module by calling `xref:m/1`.","ref":"c.html#xm/1"},{"type":"function","title":"c.y/1","doc":"Generates an LALR-1 parser. Equivalent to:\n\n```text\nyecc:file(File)\n```\n\nFor information about `File = name()`, see `m:filename`. For information about\n`YeccRet`, see [`yecc:file/2`](`yecc:file/1`).","ref":"c.html#y/1"},{"type":"function","title":"c.y/2","doc":"Generates an LALR-1 parser. Equivalent to:\n\n```text\nyecc:file(File, Options)\n```\n\nFor information about `File = name()`, see `m:filename`. For information about\n`Options` and `YeccRet`, see [`yecc:file/2`](`yecc:file/1`).","ref":"c.html#y/2"},{"type":"module","title":"edlin","doc":"Line and input interpretter for the erlang shell.\n\nThis module reads input, handles any escape sequences that have been configured\nvia edlin_key and outputs action requests. The action requests are handled\neither by modules `group` or the `user_drv`.","ref":"edlin.html"},{"type":"module","title":"Key configuration - edlin","doc":"You can setup a custom key configuration that overrides the default key\nconfiguration. This is done by setting the stdlib application parameter\n[`shell_keymap`](stdlib_app.md#shell_keymap) before Erlang is started. If you\nwant to have the same keymap in all Erlang shells you can do so by putting a\n[config](`e:kernel:config.md`) file in your user's home directory and then set\n[ERL_FLAGS](`e:erts:erl_cmd.md#ERL_FLAGS`) to load it at startup. For example:\n\n```text\n$ cat $HOME/.erlang_keymap.config\n[{stdlib,\n  [{shell_keymap,\n    #{ normal => #{ \"\\^[A\" => clear } }\n  }]\n}].\n$ ERL_FLAGS=\"-config $HOME/.erlang_keymap\" erl\n```\n\nThe current keymap configuration can be fetched through\n[edlin:keymap()](`keymap/0`). If a custom keymap or keymap file is specified,\nthen it will be merged with the default keymap.\n\nThe keymap is a map of maps where the keys in the parent map corresponds to\ndifferent editing modes in the shell. The valid modes currently supported are\n`normal` and `search`.\n\nThe keys in the child maps are the escape sequences that are sent from the\nterminal when a key is pressed and each value is a valid action as seen below.\n\nThe default atom is used to specify that an action should happen when a key is\npressed that does not have any mapping. Typically used to exit a mode.\n\nSee [tty - A Command-Line Interface](`e:erts:tty.md`) for more information about\nthe default keymap.","ref":"edlin.html#module-key-configuration"},{"type":"module","title":"Actions - edlin","doc":"The commands below are the built-in action requests for switching input modes on\nthe normal shell or navigating, or manipulating the line feed. The line feed\nsupports multiple lines.\n\n- **`auto_blink`** - Automatically close the closest matching opening\n  parenthesis.\n\n- **`backward_char`** - Move backward one character.\n\n- **`backward_delete_char`** - Delete the character behind the cursor.\n\n- **`backward_delete_word`** - Delete the word behind the cursor.\n\n- **`backward_kill_line`** - Delete all characters from the cursor to the\n  beginning of the line and save them in the kill buffer.\n\n- **`backward_kill_word`** - Delete the word behind the cursor and save it in\n  the kill buffer.\n\n- **`backward_line`** - Move backward one line.\n\n- **`backward_word`** - Move backward one word.\n\n- **`beginning_of_expression`** - Move to the beginning of the expression.\n\n- **`beginning_of_line`** - Move to the beginning of the line.\n\n- **`clear`** - Clear the screen.\n\n- **`clear_line`** - Clear the current expression.\n\n- **`end_of_expression`** - Move to the end of the expression.\n\n- **`end_of_line`** - Move to the end of the line.\n\n- **`forward_char`** - Move forward one character.\n\n- **`forward_delete_char`** - Delete the character under the cursor.\n\n- **`forward_line`** - Move forward one line.\n\n- **`forward_word`** - Move forward one word.\n\n- **`help`** - Display help for the module or function closest on the left of\n  the cursor.\n\n- **`help_full`** - Display the whole help text for the module or function closest on the left of\n  the cursor.\n\n- **`history_down`** - Move to the next item in the history.\n\n- **`history_up`** - Move to the previous item in the history.\n\n- **`kill_line`** - Delete all characters from the cursor to the end of the line\n  and save them in the kill buffer.\n\n- **`kill_word`** - Delete the word under the cursor and save it in the kill\n  buffer.\n\n- **`move_expand_down`** - Move down one line in the expand area e.g. help or\n  tab completion pager.\n\n- **`move_expand_up`** - Move up one line in the expand area e.g. help or tab\n  completion pager.\n\n- **`new_line_finish`** - Add a newline at the end of the line and try to\n  evaluate the current expression.\n\n- **`newline`** - Add a newline at the cursor position.\n\n- **`open_editor`** - Open the current line in an editor e.g. EDITOR=\"code -w\"\n  opens a buffer in vs code. Note that you need to pass a flag to the editor so\n  that it signals the shell when you close the buffer.\n\n- **`redraw_line`** - Redraw the current line.\n\n- **`scroll_expand_down`** - Scroll down five lines in the expand area e.g. help\n  or tab completion pager.\n\n- **`scroll_expand_up`** - Scroll up five lines in the expand area e.g. help or\n  tab completion pager.\n\n- **`search_cancel`** - Cancel the current search.\n\n- **`search_found`** - Accept the current search result and submit it.\n\n- **`search_quit`** - Accept the current search result, but edit it before\n  submitting.\n\n- **`search`** - Enter search mode, search the history.\n\n- **`skip_down`** - Skip to the next line in the history that matches the\n  current search expression.\n\n- **`skip_up`** - Skip to the previous line in the history that matches the\n  current search expression.\n\n- **`tab_expand_full`** - Output all possible tab completions.\n\n- **`tab_expand_quit`** - Go back to normal mode.\n\n- **`tab_expand`** - Autocomplete the current word, or show 5 lines of possible\n  completions.\n\n- **`transpose_char`** - Swap the character behind the cursor with the one in\n  front of it.\n\n- **`transpose_word`** - Swap the word behind the cursor with the one in front\n  of it.\n\n- **`yank`** - Insert the contents of the kill buffer at the cursor position.","ref":"edlin.html#module-actions"},{"type":"type","title":"edlin.keymap/0","doc":"A map of maps for each shell mode containing key, action pairs.","ref":"edlin.html#t:keymap/0"},{"type":"function","title":"edlin.keymap/0","doc":"Get the current keymap used in the shell. Each key in the parent map represents\na _shell mode_ e.g. `normal` or `search`. Each map associated with the _shell\nmodes_ contains _key sequences_ represented as strings, paired with an _action_,\nwhich is one of the valid actions mentioned above.","ref":"edlin.html#keymap/0"},{"type":"module","title":"edlin_expand","doc":"Shell expansion and formatting of expansion suggestions.\n\nThis module provides an expand_fun for the erlang shell\n[`expand/1,2`](`expand/1`). It is possible to override this expand_fun\n[`io:setopts/1,2`](`io:setopts/1`).","ref":"edlin_expand.html"},{"type":"function","title":"edlin_expand.expand/1","doc":"","ref":"edlin_expand.html#expand/1"},{"type":"function","title":"edlin_expand.expand/2","doc":"The standard expansion function is able to expand strings to valid erlang terms.\nThis includes module names:\n\n```text\n1> erla\nmodules\nerlang:\n```\n\nfunction names:\n\n```text\n1> is_ato\nfunctions\nis_atom(\n2> erlang:is_ato\nfunctions\nis_atom(\n```\n\nfunction types:\n\n```text\n1> erlang:is_atom(\ntypespecs\nerlang:is_atom(Term)\nany()\n```\n\nand automatically add , or closing parenthesis when no other valid expansion is\npossi