(*
 * BatRandom - Additional randomization operations
 * Copyright (C) 1996 Damien Doligez
 *               2008 David Teller
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version,
 * with the special exception on linking described in file LICENSE.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *)

(** Pseudo-random number generators (PRNG).

    This module extends Stdlib's
    {{:http://caml.inria.fr/pub/docs/manual-ocaml/libref/Random.html}Random}
    module, go there for documentation on the rest of the functions
    and types.

    @author Damien Doligez (base library)
    @author David Teller
    @author Pierre Chambart

    @documents Random
*)

(** {6 Basic functions} *)

val init : int -> unit
(** Initialize the generator, using the argument as a seed.
     The same seed will always yield the same sequence of numbers. *)

val full_init : int array -> unit
(** Same as {!Random.init} but takes more data as seed. *)

val self_init : unit -> unit
(** Initialize the generator with a more-or-less random seed chosen
    in a system-dependent way. *)

val bits : unit -> int
(** Return 30 random bits in a nonnegative integer. *)

val int : int -> int
(** [Random.int bound] returns a random integer between 0 (inclusive)
     and [bound] (exclusive).  [bound] must be greater than 0 and less
     than 2{^30}. *)

##V>=4.13##val full_int : int -> int
##V>=4.13##(** [Random.full_int bound] returns a random integer between 0 (inclusive)
##V>=4.13##     and [bound] (exclusive). [bound] may be any positive integer.
##V>=4.13##
##V>=4.13##     If [bound] is less than 2{^30}, [Random.full_int bound] is equal to
##V>=4.13##     {!Random.int}[ bound]. If [bound] is greater than 2{^30} (on 64-bit systems
##V>=4.13##     or non-standard environments, such as JavaScript), [Random.full_int]
##V>=4.13##     returns a value, where {!Random.int} raises {!Invalid_argument}.
##V>=4.13##
##V>=4.13##    @since 3.4.0 and OCaml 4.13.0 *)

##V>=5.2##val int_in_range : min:int -> max:int -> int

val int32 : Int32.t -> Int32.t
(** [Random.int32 bound] returns a random integer between 0 (inclusive)
     and [bound] (exclusive).  [bound] must be greater than 0. *)

##V>=5.2##val int32_in_range : min:int32 -> max:int32 -> int32

val nativeint : Nativeint.t -> Nativeint.t
(** [Random.nativeint bound] returns a random integer between 0 (inclusive)
     and [bound] (exclusive).  [bound] must be greater than 0. *)

##V>=5.2##val nativeint_in_range : min:nativeint -> max:nativeint -> nativeint

val int64 : Int64.t -> Int64.t
(** [Random.int64 bound] returns a random integer between 0 (inclusive)
     and [bound] (exclusive).  [bound] must be greater than 0. *)

val float : float -> float
(** [Random.float bound] returns a random floating-point number
    between 0 (inclusive) and [bound] (exclusive).  If [bound] is
    negative, the result is negative or zero.  If [bound] is 0,
    the result is 0. *)

##V>=5.2##val int64_in_range : min:int64 -> max:int64 -> int64

val bool : unit -> bool
(** [Random.bool ()] returns [true] or [false] with probability 0.5 each. *)

val char : unit -> char
(** Return a random Latin-1 character.*)

(*val uchar : unit -> UChar.t
  (** Return a random Unicode character.*)*)

val full_range_int : unit -> int
(** [full_range_int ()] returns the maximum entropy possible in a
    single int: 31 bits on 32-bit platforms and 63 bits on 64-bit
    platforms.  Intentionally gives different results on different
    platforms, so is not portable. *)


(** {6 Enumerations of random values.} *)

val enum_bits  : unit    -> int BatEnum.t

val enum_int   : int     -> int BatEnum.t

val enum_bool  : unit    -> bool BatEnum.t

val enum_float : float   -> float BatEnum.t

val enum_int32 : Int32.t -> Int32.t BatEnum.t

val enum_int64 : Int64.t -> Int64.t BatEnum.t

val enum_nativeint : Nativeint.t -> Nativeint.t BatEnum.t

val enum_char : unit -> char BatEnum.t

(*val enum_uchar : unit -> UChar.t BatEnum.t*)

(** {6 Working with data structures.} *)

val choice : 'a BatEnum.t -> 'a
(** [choice e] returns a randomly-chosen element of [e].

    @raise Empty when applied to an empty enum.
    This function only works on finite enumerations with
    less than 2{^30} elements. *)

val multi_choice : int -> 'a BatEnum.t -> 'a BatEnum.t
(** [multi_choice n e] returns an enumeration of [n]
    randomly-chosen elements of [e]. *)

val shuffle: 'a BatEnum.t -> 'a array
(** [shuffle e] returns a new array, containing the
    same set of elements as [e], but in a random order.

    Shuffling is implemented using the Fisher-Yates
    algorithm and works in O(n), where n is the number
    of elements of [e].

    This function only works on finite enumerations with
    less than 2{^30} elements. *)

##V>=4.14##val bits32 : unit -> Int32.t
##V>=4.14##(** [Random.bits32 ()] returns 32 random bits as an integer between
##V>=4.14##    {!Int32.min_int} and {!Int32.max_int}.
##V>=4.14##    @since 4.14.0 *)

##V>=4.14##val bits64 : unit -> Int64.t
##V>=4.14##(** [Random.bits64 ()] returns 64 random bits as an integer between
##V>=4.14##    {!Int64.min_int} and {!Int64.max_int}.
##V>=4.14##    @since 4.14.0 *)

##V>=4.14##val nativebits : unit -> Nativeint.t
##V>=4.14##(** [Random.nativebits ()] returns 32 or 64 random bits (depending on
##V>=4.14##    the bit width of the platform) as an integer between
##V>=4.14##    {!Nativeint.min_int} and {!Nativeint.max_int}.
##V>=4.14##    @since 4.14.0 *)

(** {6 Advanced functions} *)

(** Manipulate the current state of the random generator.

    This allows using one or several deterministic PRNGs, even in a
    multi-threaded program, without interference from other parts of
    the program.
*)
module State : sig
  type t = Random.State.t
  (** The type of PRNG states. *)

  val make : int array -> t
  (** Create a new state and initialize it with the given seed. *)

  val make_self_init : unit -> t
  (** Create a new state and initialize it with a system-dependent
      low-entropy seed. *)

  val copy : t -> t
  (** Return a copy of the given state. *)

  val bits       : t -> int
  val int        : t -> int -> int
##V>=4.13##val full_int : t -> int -> int
##V>=5.2##val int_in_range : t -> min:int -> max:int -> int
  val int32      : t -> Int32.t -> Int32.t
##V>=5.2##val int32_in_range : t -> min:int32 -> max:int32 -> int32
  val nativeint  : t -> Nativeint.t -> Nativeint.t
##V>=5.2##val nativeint_in_range : t -> min:nativeint -> max:nativeint -> nativeint
  val int64      : t -> Int64.t -> Int64.t
##V>=5.2##val int64_in_range : t -> min:int64 -> max:int64 -> int64
  val float      : t -> float -> float
  val bool       : t -> bool
  val char       : t -> char
  (*  val uchar      : t -> UChar.t*)
  val enum_bits  : t -> unit    -> int BatEnum.t
  val enum_int   : t -> int     -> int BatEnum.t
  val enum_bool  : t -> unit    -> bool BatEnum.t
  val enum_float : t -> float   -> float BatEnum.t
  val enum_int32 : t -> Int32.t -> Int32.t BatEnum.t
  val enum_int64 : t -> Int64.t -> Int64.t BatEnum.t
  val enum_nativeint : t -> Nativeint.t -> Nativeint.t BatEnum.t
  val enum_char  : t -> unit    -> char BatEnum.t
      (*  val enum_uchar : t -> unit    -> UChar.t BatEnum.t*)
      (** These functions are the same as the basic functions, except that they
          use (and update) the given PRNG state instead of the default one.
      *)

##V>=4.14##val bits32 : t -> Int32.t
##V>=4.14##val bits64 : t -> Int64.t
##V>=4.14##val nativebits : t -> Nativeint.t

##V>=5##val split : t -> t
##V>=5##(** Draw a fresh PRNG state from the given PRNG state.
##V>=5##    (The given PRNG state is modified.)
##V>=5##    The new PRNG is statistically independent from the given PRNG.
##V>=5##    Data can be drawn from both PRNGs, in any order, without risk of
##V>=5##    correlation.  Both PRNGs can be split later, arbitrarily many times.
##V>=5##    @since 3.6.0 and OCaml 5.0.0 *)

##V>=5.1##val to_binary_string : t -> string
##V>=5.1##val of_binary_string : string -> t

end


val get_state : unit -> State.t
(** Return the current state of the generator used by the basic functions. *)

val set_state : State.t -> unit
(** Set the state of the generator used by the basic functions. *)

##V>=5##val split : unit -> State.t
##V>=5##(** Draw a fresh PRNG state from the current state of the domain-local
##V>=5##    generator used by the default functions.
##V>=5##    (The state of the domain-local generator is modified.)
##V>=5##    See {!Random.State.split}.
##V>=5##    @since 3.6.0 and OCaml 5.0.0 *)

module Incubator : sig
  module Private_state_enums : sig
    module State : sig (** same as BatRandom.State *)
      type t = Random.State.t
      (** The type of PRNG states. *)

      val make : int array -> t
      (** Create a new state and initialize it with the given seed. *)

      val make_self_init : unit -> t
      (** Create a new state and initialize it with a system-dependent
          low-entropy seed. *)

      val copy : t -> t
      (** Return a copy of the given state. *)

      val bits       : t -> int
      val int        : t -> int -> int
      val int32      : t -> Int32.t -> Int32.t
      val nativeint  : t -> Nativeint.t -> Nativeint.t
      val int64      : t -> Int64.t -> Int64.t
      val float      : t -> float -> float
      val bool       : t -> bool
      val char       : t -> char

      (** A copy of the input state is made to start these generators;
          the input state is not modified.  This means that two enums
          constructed from the same state will produce the same value
          sequence. *)
      val enum_bits  : t -> unit    -> int BatEnum.t
      val enum_int   : t -> int     -> int BatEnum.t
      val enum_bool  : t -> unit    -> bool BatEnum.t
      val enum_float : t -> float   -> float BatEnum.t
      val enum_int32 : t -> Int32.t -> Int32.t BatEnum.t
      val enum_int64 : t -> Int64.t -> Int64.t BatEnum.t
      val enum_nativeint : t -> Nativeint.t -> Nativeint.t BatEnum.t
      val enum_char  : t -> unit    -> char BatEnum.t

      (** [perturb s] returns a new state based on the given state
          that is, in a sense, the hash of the input state.  This new
          state should be quite different from the input. *)
      val perturb : t -> t

    end

    (** These enumerations are built on a copy of the global RNG
        state.  To keep successive constructions from using the same RNG
        state, when any of these functions is called, the global RNG state
        is perturbed by using its current internal state as seed to
        construct a new state. *)
    val enum_bits  : unit    -> int BatEnum.t
    val enum_int   : int     -> int BatEnum.t
    val enum_bool  : unit    -> bool BatEnum.t
    val enum_float : float   -> float BatEnum.t
    val enum_int32 : Int32.t -> Int32.t BatEnum.t
    val enum_int64 : Int64.t -> Int64.t BatEnum.t
    val enum_nativeint : Nativeint.t -> Nativeint.t BatEnum.t
    val enum_char : unit -> char BatEnum.t
##V>=4.14## val bits32 : unit -> Int32.t
##V>=4.14## val bits64 : unit -> Int64.t
##V>=4.14## val nativebits : unit -> Nativeint.t
  end
end
