package addmask

import (
	"encoding/base64"
	"encoding/json"
	"encoding/xml"
	"net/url"
	"slices"
	"strings"
)

var escaper = strings.NewReplacer("%0D", "\r", "%0A", "\n", "%25", "%")

func encode(inputStr string) []string {
	var encoded []string

	// unlike GitHub, we're going to also add the unencoded version
	encoded = append(encoded, inputStr)

	// add-mask espects \n, \r and % to be encoded.
	inputStr = escaper.Replace(inputStr)
	input := []byte(inputStr)

	// base64 encodings, shifted so that data concatenated with a password is
	// still masked (such as base64(user:password))
	encoded = append(encoded, base64Encode(input, 0))
	encoded = append(encoded, base64Encode(input, 1))
	encoded = append(encoded, base64Encode(input, 2))

	// command-line escape
	encoded = append(encoded, strings.ReplaceAll(inputStr, "\"", "\\\""))

	// json escape
	buf, _ := json.Marshal(inputStr)
	if len(buf) > 2 {
		encoded = append(encoded, string(buf[1:len(buf)-1]))
	}

	// uri data escape
	encoded = append(encoded, strings.ReplaceAll(url.QueryEscape(inputStr), "+", "%20"))

	// xml data escape
	encoded = append(encoded, xmlEscape(input))

	// trim double quotes
	if len(inputStr) > 8 && strings.HasPrefix(inputStr, `"`) && strings.HasSuffix(inputStr, `"`) {
		encoded = append(encoded, inputStr[1:len(inputStr)-1])
	}

	slices.Sort(encoded)

	return slices.Compact(slices.DeleteFunc(encoded, func(v string) bool { return v == "" }))
}

func base64Encode(input []byte, shift int) string {
	if len(input) > shift {
		return base64.StdEncoding.EncodeToString(input[shift:])
	}

	return ""
}

func xmlEscape(input []byte) string {
	var sb strings.Builder
	xml.EscapeText(&sb, input)
	return sb.String()
}
