/*
  This file is part of TALER
  (C) 2020--2026 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify
  it under the terms of the GNU Affero General Public License as
  published by the Free Software Foundation; either version 3,
  or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public
  License along with TALER; see the file COPYING.  If not,
  see <http://www.gnu.org/licenses/>
*/

/**
 * @file taler-merchant-httpd_private-patch-products-ID.c
 * @brief implementing PATCH /products/$ID request handling
 * @author Christian Grothoff
 */
#include "platform.h"
#include "taler-merchant-httpd_private-patch-products-ID.h"
#include "taler-merchant-httpd_helper.h"
#include <taler/taler_json_lib.h>


/**
 * PATCH configuration of an existing instance, given its configuration.
 *
 * @param rh context of the handler
 * @param connection the MHD connection to handle
 * @param[in,out] hc context with further information about the request
 * @return MHD result code
 */
MHD_RESULT
TMH_private_patch_products_ID (
  const struct TMH_RequestHandler *rh,
  struct MHD_Connection *connection,
  struct TMH_HandlerContext *hc)
{
  struct TMH_MerchantInstance *mi = hc->instance;
  const char *product_id = hc->infix;
  struct TALER_MERCHANTDB_ProductDetails pd = {0};
  const json_t *categories = NULL;
  int64_t total_stock;
  const char *unit_total_stock = NULL;
  bool unit_total_stock_missing;
  bool total_stock_missing;
  struct TALER_Amount price;
  bool price_missing;
  bool unit_price_missing;
  bool unit_allow_fraction;
  bool unit_allow_fraction_missing;
  uint32_t unit_precision_level;
  bool unit_precision_missing;
  enum GNUNET_DB_QueryStatus qs;
  struct GNUNET_JSON_Specification spec[] = {
    /* new in protocol v20, thus optional for backwards-compatibility */
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_string ("product_name",
                               (const char **) &pd.product_name),
      NULL),
    GNUNET_JSON_spec_string ("description",
                             (const char **) &pd.description),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_json ("description_i18n",
                             &pd.description_i18n),
      NULL),
    GNUNET_JSON_spec_string ("unit",
                             (const char **) &pd.unit),
    // FIXME: deprecated API
    GNUNET_JSON_spec_mark_optional (
      TALER_JSON_spec_amount_any ("price",
                                  &price),
      &price_missing),
    GNUNET_JSON_spec_mark_optional (
      TALER_JSON_spec_amount_any_array ("unit_price",
                                        &pd.price_array_length,
                                        &pd.price_array),
      &unit_price_missing),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_string ("image",
                               (const char **) &pd.image),
      NULL),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_json ("taxes",
                             &pd.taxes),
      NULL),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_array_const ("categories",
                                    &categories),
      NULL),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_string ("unit_total_stock",
                               &unit_total_stock),
      &unit_total_stock_missing),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_int64 ("total_stock",
                              &total_stock),
      &total_stock_missing),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_bool ("unit_allow_fraction",
                             &unit_allow_fraction),
      &unit_allow_fraction_missing),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_uint32 ("unit_precision_level",
                               &unit_precision_level),
      &unit_precision_missing),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_uint64 ("total_lost",
                               &pd.total_lost),
      NULL),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_uint64 ("product_group_id",
                               &pd.product_group_id),
      NULL),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_uint64 ("money_pot_id",
                               &pd.money_pot_id),
      NULL),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_json ("address",
                             &pd.address),
      NULL),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_timestamp ("next_restock",
                                  &pd.next_restock),
      NULL),
    GNUNET_JSON_spec_mark_optional (
      GNUNET_JSON_spec_uint32 ("minimum_age",
                               &pd.minimum_age),
      NULL),
    GNUNET_JSON_spec_end ()
  };
  MHD_RESULT ret;
  size_t num_cats = 0;
  uint64_t *cats = NULL;
  bool no_instance;
  ssize_t no_cat;
  bool no_product;
  bool lost_reduced;
  bool sold_reduced;
  bool stock_reduced;
  bool no_group;
  bool no_pot;

  GNUNET_assert (NULL != mi);
  GNUNET_assert (NULL != product_id);
  {
    enum GNUNET_GenericReturnValue res;

    res = TALER_MHD_parse_json_data (connection,
                                     hc->request_body,
                                     spec);
    if (GNUNET_OK != res)
      return (GNUNET_NO == res)
             ? MHD_YES
             : MHD_NO;
    /* For pre-v20 clients, we use the description given as the
       product name; remove once we make product_name mandatory. */
    if (NULL == pd.product_name)
      pd.product_name = pd.description;
  }
  if (! unit_price_missing)
  {
    if (! price_missing)
    {
      if (0 != TALER_amount_cmp (&price,
                                 &pd.price_array[0]))
      {
        ret = TALER_MHD_reply_with_error (connection,
                                          MHD_HTTP_BAD_REQUEST,
                                          TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                          "price,unit_price mismatch");
        goto cleanup;
      }
    }
    if (GNUNET_OK !=
        TMH_validate_unit_price_array (pd.price_array,
                                       pd.price_array_length))
    {
      ret = TALER_MHD_reply_with_error (connection,
                                        MHD_HTTP_BAD_REQUEST,
                                        TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                        "unit_price");
      goto cleanup;
    }
  }
  else
  {
    if (price_missing)
    {
      ret = TALER_MHD_reply_with_error (connection,
                                        MHD_HTTP_BAD_REQUEST,
                                        TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                        "price missing");
      goto cleanup;
    }
    pd.price_array = GNUNET_new_array (1,
                                       struct TALER_Amount);
    pd.price_array[0] = price;
    pd.price_array_length = 1;
  }
  if (! unit_precision_missing)
  {
    if (unit_precision_level > TMH_MAX_FRACTIONAL_PRECISION_LEVEL)
    {
      ret = TALER_MHD_reply_with_error (connection,
                                        MHD_HTTP_BAD_REQUEST,
                                        TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                        "unit_precision_level");
      goto cleanup;
    }
  }
  {
    bool default_allow_fractional;
    uint32_t default_precision_level;

    if (GNUNET_OK !=
        TMH_unit_defaults_for_instance (mi,
                                        pd.unit,
                                        &default_allow_fractional,
                                        &default_precision_level))
    {
      GNUNET_break (0);
      ret = TALER_MHD_reply_with_error (connection,
                                        MHD_HTTP_INTERNAL_SERVER_ERROR,
                                        TALER_EC_GENERIC_DB_FETCH_FAILED,
                                        "unit defaults");
      goto cleanup;
    }
    if (unit_allow_fraction_missing)
      unit_allow_fraction = default_allow_fractional;
    if (unit_precision_missing)
      unit_precision_level = default_precision_level;

    if (! unit_allow_fraction)
      unit_precision_level = 0;
    pd.fractional_precision_level = unit_precision_level;
  }
  {
    const char *eparam;
    if (GNUNET_OK !=
        TALER_MERCHANT_vk_process_quantity_inputs (
          TALER_MERCHANT_VK_STOCK,
          unit_allow_fraction,
          total_stock_missing,
          total_stock,
          unit_total_stock_missing,
          unit_total_stock,
          &pd.total_stock,
          &pd.total_stock_frac,
          &eparam))
    {
      ret = TALER_MHD_reply_with_error (
        connection,
        MHD_HTTP_BAD_REQUEST,
        TALER_EC_GENERIC_PARAMETER_MALFORMED,
        eparam);
      goto cleanup;
    }
    pd.allow_fractional_quantity = unit_allow_fraction;
  }
  if (NULL == pd.address)
    pd.address = json_object ();

  if (! TMH_location_object_valid (pd.address))
  {
    GNUNET_break_op (0);
    ret = TALER_MHD_reply_with_error (connection,
                                      MHD_HTTP_BAD_REQUEST,
                                      TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                      "address");
    goto cleanup;
  }
  num_cats = json_array_size (categories);
  cats = GNUNET_new_array (num_cats,
                           uint64_t);
  {
    size_t idx;
    json_t *val;

    json_array_foreach (categories, idx, val)
    {
      if (! json_is_integer (val))
      {
        GNUNET_break_op (0);
        ret = TALER_MHD_reply_with_error (connection,
                                          MHD_HTTP_BAD_REQUEST,
                                          TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                          "categories");
        goto cleanup;
      }
      cats[idx] = json_integer_value (val);
    }
  }

  if (NULL == pd.description_i18n)
    pd.description_i18n = json_object ();

  if (! TALER_JSON_check_i18n (pd.description_i18n))
  {
    GNUNET_break_op (0);
    ret = TALER_MHD_reply_with_error (connection,
                                      MHD_HTTP_BAD_REQUEST,
                                      TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                      "description_i18n");
    goto cleanup;
  }

  if (NULL == pd.taxes)
    pd.taxes = json_array ();
  /* check taxes is well-formed */
  if (! TALER_MERCHANT_taxes_array_valid (pd.taxes))
  {
    GNUNET_break_op (0);
    ret = TALER_MHD_reply_with_error (connection,
                                      MHD_HTTP_BAD_REQUEST,
                                      TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                      "taxes");
    goto cleanup;
  }

  if (NULL == pd.image)
    pd.image = (char *) "";
  if (! TALER_MERCHANT_image_data_url_valid (pd.image))
  {
    GNUNET_break_op (0);
    ret = TALER_MHD_reply_with_error (connection,
                                      MHD_HTTP_BAD_REQUEST,
                                      TALER_EC_GENERIC_PARAMETER_MALFORMED,
                                      "image");
    goto cleanup;
  }

  if ( (pd.total_stock < pd.total_sold + pd.total_lost) ||
       (pd.total_sold + pd.total_lost < pd.total_sold) /* integer overflow */)
  {
    GNUNET_break_op (0);
    ret = TALER_MHD_reply_with_error (
      connection,
      MHD_HTTP_BAD_REQUEST,
      TALER_EC_MERCHANT_PRIVATE_PATCH_PRODUCTS_TOTAL_LOST_EXCEEDS_STOCKS,
      NULL);
    goto cleanup;
  }

  qs = TMH_db->update_product (TMH_db->cls,
                               mi->settings.id,
                               product_id,
                               &pd,
                               num_cats,
                               cats,
                               &no_instance,
                               &no_cat,
                               &no_product,
                               &lost_reduced,
                               &sold_reduced,
                               &stock_reduced,
                               &no_group,
                               &no_pot);
  switch (qs)
  {
  case GNUNET_DB_STATUS_HARD_ERROR:
    GNUNET_break (0);
    ret = TALER_MHD_reply_with_error (connection,
                                      MHD_HTTP_INTERNAL_SERVER_ERROR,
                                      TALER_EC_GENERIC_DB_STORE_FAILED,
                                      NULL);
    goto cleanup;
  case GNUNET_DB_STATUS_SOFT_ERROR:
    GNUNET_break (0);
    ret = TALER_MHD_reply_with_error (connection,
                                      MHD_HTTP_INTERNAL_SERVER_ERROR,
                                      TALER_EC_GENERIC_INTERNAL_INVARIANT_FAILURE,
                                      "unexpected serialization problem");
    goto cleanup;
  case GNUNET_DB_STATUS_SUCCESS_NO_RESULTS:
    GNUNET_break (0);
    ret = TALER_MHD_reply_with_error (connection,
                                      MHD_HTTP_INTERNAL_SERVER_ERROR,
                                      TALER_EC_GENERIC_INTERNAL_INVARIANT_FAILURE,
                                      "unexpected problem in stored procedure");
    goto cleanup;
  case GNUNET_DB_STATUS_SUCCESS_ONE_RESULT:
    break;
  }

  if (no_instance)
  {
    ret = TALER_MHD_reply_with_error (connection,
                                      MHD_HTTP_NOT_FOUND,
                                      TALER_EC_MERCHANT_GENERIC_INSTANCE_UNKNOWN,
                                      mi->settings.id);
    goto cleanup;
  }
  if (-1 != no_cat)
  {
    char cat_str[24];

    GNUNET_snprintf (cat_str,
                     sizeof (cat_str),
                     "%llu",
                     (unsigned long long) no_cat);
    ret = TALER_MHD_reply_with_error (connection,
                                      MHD_HTTP_NOT_FOUND,
                                      TALER_EC_MERCHANT_GENERIC_CATEGORY_UNKNOWN,
                                      cat_str);
    goto cleanup;
  }
  if (no_product)
  {
    ret = TALER_MHD_reply_with_error (connection,
                                      MHD_HTTP_NOT_FOUND,
                                      TALER_EC_MERCHANT_GENERIC_PRODUCT_UNKNOWN,
                                      product_id);
    goto cleanup;
  }
  if (no_group)
  {
    ret = TALER_MHD_reply_with_error (
      connection,
      MHD_HTTP_NOT_FOUND,
      TALER_EC_MERCHANT_GENERIC_PRODUCT_GROUP_UNKNOWN,
      NULL);
    goto cleanup;
  }
  if (no_pot)
  {
    ret = TALER_MHD_reply_with_error (
      connection,
      MHD_HTTP_NOT_FOUND,
      TALER_EC_MERCHANT_GENERIC_MONEY_POT_UNKNOWN,
      NULL);
    goto cleanup;
  }
  if (lost_reduced)
  {
    ret = TALER_MHD_reply_with_error (
      connection,
      MHD_HTTP_CONFLICT,
      TALER_EC_MERCHANT_PRIVATE_PATCH_PRODUCTS_TOTAL_LOST_REDUCED,
      NULL);
    goto cleanup;
  }
  if (sold_reduced)
  {
    ret = TALER_MHD_reply_with_error (
      connection,
      MHD_HTTP_CONFLICT,
      TALER_EC_MERCHANT_PRIVATE_PATCH_PRODUCTS_TOTAL_SOLD_REDUCED,
      NULL);
    goto cleanup;
  }
  if (stock_reduced)
  {
    ret = TALER_MHD_reply_with_error (
      connection,
      MHD_HTTP_CONFLICT,
      TALER_EC_MERCHANT_PRIVATE_PATCH_PRODUCTS_TOTAL_STOCKED_REDUCED,
      NULL);
    goto cleanup;
  }
  /* success! */
  ret = TALER_MHD_reply_static (connection,
                                MHD_HTTP_NO_CONTENT,
                                NULL,
                                NULL,
                                0);
cleanup:
  GNUNET_free (cats);
  GNUNET_free (pd.price_array);
  GNUNET_JSON_parse_free (spec);
  return ret;
}


/* end of taler-merchant-httpd_private-patch-products-ID.c */
