/*
  This file is part of TALER
  Copyright (C) 2025 Taler Systems SA

  TALER is free software; you can redistribute it and/or modify it under the
  terms of the GNU General Public License as published by the Free Software
  Foundation; either version 3, or (at your option) any later version.

  TALER is distributed in the hope that it will be useful, but WITHOUT ANY
  WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
  A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

  You should have received a copy of the GNU General Public License along with
  TALER; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
*/
/**
 * @file testing/testing_api_cmd_post_categories.c
 * @brief command to test POST /private/categories
 * @author Bohdan Potuzhnyi
 */
#include "platform.h"
#include <jansson.h>
#include <microhttpd.h>
#include <taler/taler_testing_lib.h>
#include "taler_merchant_service.h"
#include "taler_merchant_testing_lib.h"

/**
 * State of a "POST /private/categories" CMD.
 */
struct PostCategoriesState
{
  /**
   * Handle for a "POST /private/categories" request.
   */
  struct TALER_MERCHANT_CategoriesPostHandle *cph;

  /**
   * The interpreter state.
   */
  struct TALER_TESTING_Interpreter *is;

  /**
   * Base URL of the merchant serving the request.
   */
  const char *merchant_url;

  /**
   * Category name.
   */
  const char *name;

  /**
   * Category name translations.
   */
  json_t *name_i18n;

  /**
   * Expected HTTP response code.
   */
  unsigned int http_status;

  /**
   * Expected category id (0 to ignore).
   */
  uint64_t expected_category_id;

  /**
   * Category id returned by the backend.
   */
  uint64_t category_id;
};

/**
 * Callback for POST /private/categories.
 *
 * @param cls closure
 * @param cpr response details
 */
static void
post_categories_cb (void *cls,
                    const struct TALER_MERCHANT_CategoriesPostResponse *cpr)
{
  struct PostCategoriesState *pcs = cls;

  pcs->cph = NULL;
  if (pcs->http_status != cpr->hr.http_status)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                "Unexpected response code %u (%d) to command %s\n",
                cpr->hr.http_status,
                (int) cpr->hr.ec,
                TALER_TESTING_interpreter_get_current_label (pcs->is));
    TALER_TESTING_interpreter_fail (pcs->is);
    return;
  }
  if (MHD_HTTP_OK == cpr->hr.http_status)
  {
    pcs->category_id = cpr->category_id;
    if ( (0 != pcs->expected_category_id) &&
         (pcs->expected_category_id != pcs->category_id) )
    {
      GNUNET_log (GNUNET_ERROR_TYPE_ERROR,
                  "Unexpected category_id %lu (expected %lu)\n",
                  pcs->category_id,
                  pcs->expected_category_id);
      TALER_TESTING_interpreter_fail (pcs->is);
      return;
    }
  }
  TALER_TESTING_interpreter_next (pcs->is);
}


/**
 * Run the "POST /private/categories" CMD.
 *
 * @param cls closure.
 * @param cmd command being run now.
 * @param is interpreter state.
 */
static void
post_categories_run (void *cls,
                     const struct TALER_TESTING_Command *cmd,
                     struct TALER_TESTING_Interpreter *is)
{
  struct PostCategoriesState *pcs = cls;

  pcs->is = is;
  pcs->cph = TALER_MERCHANT_categories_post (
    TALER_TESTING_interpreter_get_context (is),
    pcs->merchant_url,
    pcs->name,
    pcs->name_i18n,
    &post_categories_cb,
    pcs);
  GNUNET_assert (NULL != pcs->cph);
}


/**
 * Offer internal data to other commands.
 *
 * @param cls closure
 * @param[out] ret result
 * @param trait name of the trait
 * @param index index number of the object to extract.
 * @return #GNUNET_OK on success
 */
static enum GNUNET_GenericReturnValue
post_categories_traits (void *cls,
                        const void **ret,
                        const char *trait,
                        unsigned int index)
{
  struct PostCategoriesState *pcs = cls;
  struct TALER_TESTING_Trait traits[] = {
    TALER_TESTING_make_trait_category_id (&pcs->category_id),
    TALER_TESTING_trait_end ()
  };

  return TALER_TESTING_get_trait (traits,
                                  ret,
                                  trait,
                                  index);
}


/**
 * Free the state of a "POST /private/categories" CMD, and possibly
 * cancel a pending operation thereof.
 *
 * @param cls closure.
 * @param cmd command being run.
 */
static void
post_categories_cleanup (void *cls,
                         const struct TALER_TESTING_Command *cmd)
{
  struct PostCategoriesState *pcs = cls;

  if (NULL != pcs->cph)
  {
    GNUNET_log (GNUNET_ERROR_TYPE_WARNING,
                "POST /private/categories operation did not complete\n");
    TALER_MERCHANT_categories_post_cancel (pcs->cph);
  }
  json_decref (pcs->name_i18n);
  GNUNET_free (pcs);
}


struct TALER_TESTING_Command
TALER_TESTING_cmd_merchant_post_categories (
  const char *label,
  const char *merchant_url,
  const char *name,
  json_t *name_i18n,
  uint64_t expected_category_id,
  unsigned int http_status)
{
  struct PostCategoriesState *pcs;

  GNUNET_assert ((NULL == name_i18n) ||
                 json_is_object (name_i18n));
  pcs = GNUNET_new (struct PostCategoriesState);
  pcs->merchant_url = merchant_url;
  pcs->name = name;
  pcs->name_i18n = name_i18n; /* ownership taken */
  pcs->http_status = http_status;
  pcs->expected_category_id = expected_category_id;
  {
    struct TALER_TESTING_Command cmd = {
      .cls = pcs,
      .label = label,
      .run = &post_categories_run,
      .cleanup = &post_categories_cleanup,
      .traits = &post_categories_traits
    };

    return cmd;
  }
}


/* end of testing_api_cmd_post_categories.c */
