/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 *  Copyright (C) 2006 Juernjakob Harder
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more edit_char.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifdef HAVE_CONFIG_H
#   include <config.h>
#endif

#include <string.h>
#include <glib/gi18n-lib.h>
#include <tomoe.h>

#include "tomoe-edit-char.h"
#include "tomoe-canvas.h"
#include "tomoe-edit-meta.h"

typedef struct _TomoeEditCharPrivate	TomoeEditCharPrivate;
struct _TomoeEditCharPrivate
{
    GtkWidget    *ok_button;
    GtkWidget    *input;
    TomoeChar    *character;
    TomoeDict    *dict;
};

enum
{
    PROP_0,
    PROP_TOMOE_CHAR,
    PROP_TOMOE_DICT
};
#define TOMOE_EDIT_CHAR_GET_PRIVATE(obj) (G_TYPE_INSTANCE_GET_PRIVATE ((obj), TOMOE_TYPE_EDIT_CHAR, TomoeEditCharPrivate))

G_DEFINE_TYPE (TomoeEditChar, tomoe_edit_char, GTK_TYPE_DIALOG)

static void     dispose      (GObject              *object);
static GObject *constructor  (GType                  type,
                              guint                  n_props,
                              GObjectConstructParam *props);
static void     set_property (GObject               *object,
                              guint                  prop_id,
                              const GValue          *value,
                              GParamSpec            *pspec);
static void     get_property (GObject               *object,
                              guint                  prop_id,
                              GValue                *value,
                              GParamSpec            *pspec);

static void on_ok_button_clicked             (GtkButton            *button,
                                              gpointer              user_data);
static void on_cancel_button_clicked         (GtkButton            *button,
                                              gpointer              user_data);
static void on_input_changed                 (GtkEntry             *input,
                                              gpointer              user_data);

static void
tomoe_edit_char_class_init (TomoeEditCharClass *klass)
{
    GObjectClass *gobject_class = G_OBJECT_CLASS (klass);

    gobject_class->dispose      = dispose;
    gobject_class->constructor  = constructor;
    gobject_class->set_property = set_property;
    gobject_class->get_property = get_property;

    g_object_class_install_property (gobject_class,
                                     PROP_TOMOE_CHAR,
                                     g_param_spec_object ("tomoe-char",
                                         N_("TomoeChar object"),
                                         N_("The Object of TomoeChar"),
                                         TOMOE_TYPE_CHAR,
                                         G_PARAM_READWRITE |
                                         G_PARAM_CONSTRUCT_ONLY));
    g_object_class_install_property (gobject_class,
                                     PROP_TOMOE_DICT,
                                     g_param_spec_object ("tomoe-dict",
                                         N_("TomoeDict object"),
                                         N_("The Object of TomoeDict"),
                                         TOMOE_TYPE_DICT,
                                         G_PARAM_READWRITE |
                                         G_PARAM_CONSTRUCT_ONLY));
    g_type_class_add_private (gobject_class, sizeof (TomoeEditCharPrivate));
}

static GObject *
constructor (GType type, guint n_props,
             GObjectConstructParam *props)
{

    GObject *object;
    GtkWidget *hbox, *vbox, *alignment;
    GtkWidget *widget;
    TomoeEditChar *dialog;
    GObjectClass *klass = G_OBJECT_CLASS (tomoe_edit_char_parent_class);

    object = klass->constructor (type, n_props, props);
    dialog = TOMOE_EDIT_CHAR (object);

    TomoeEditCharPrivate *priv = TOMOE_EDIT_CHAR_GET_PRIVATE (object);

    gtk_window_set_title (GTK_WINDOW (dialog),
                          _("Change character"));

    gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
    gtk_window_set_modal (GTK_WINDOW (dialog), TRUE);
    gtk_container_set_border_width (GTK_CONTAINER (dialog), 5);
    gtk_dialog_set_default_response (GTK_DIALOG (dialog), GTK_RESPONSE_DELETE_EVENT);
    gtk_dialog_set_has_separator (GTK_DIALOG (dialog), FALSE);

    vbox = gtk_vbox_new (FALSE, 4);
    gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), vbox);
    gtk_widget_set_size_request (vbox, 240, -1);
    gtk_widget_show (vbox);

    widget = gtk_label_new (_("New character:"));
    gtk_box_pack_start (GTK_BOX (vbox), widget, FALSE, FALSE, 4);
    gtk_widget_show (widget);

    widget = gtk_entry_new ();
    priv->input = widget;
    gtk_box_pack_start (GTK_BOX (vbox), widget, TRUE, TRUE, 4);
    g_signal_connect (G_OBJECT (widget), "changed",
                      G_CALLBACK (on_input_changed),
                      (gpointer) dialog);
    gtk_entry_set_text (GTK_ENTRY (widget), 
                        tomoe_char_get_utf8 (priv->character));
    gtk_widget_show (widget);

    /* button area */
    alignment = gtk_alignment_new (0.5, 0.5, 1.0, 1.0);
    gtk_alignment_set_padding (GTK_ALIGNMENT (alignment), 0, 0, 8, 0);
    gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), alignment, FALSE, FALSE, 0);
    gtk_widget_show (alignment);

    hbox = gtk_hbox_new (FALSE, 0);
    gtk_container_add (GTK_CONTAINER (alignment), hbox);
    gtk_widget_show (hbox);

    widget = gtk_button_new_from_stock (GTK_STOCK_CANCEL);
    gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (widget), "clicked",
                      G_CALLBACK (on_cancel_button_clicked),
                      (gpointer) dialog);
    gtk_widget_show (widget);

    widget = gtk_button_new_from_stock (GTK_STOCK_OK);
    priv->ok_button = widget;
    gtk_box_pack_start (GTK_BOX (hbox), widget, FALSE, FALSE, 4);
    g_signal_connect (G_OBJECT (widget), "clicked",
                      G_CALLBACK (on_ok_button_clicked),
                      (gpointer) dialog);
    gtk_widget_show (widget);
    return object;
}

static void
tomoe_edit_char_init (TomoeEditChar *dialog)
{
}

static void
dispose (GObject *object)
{
    TomoeEditCharPrivate *priv = TOMOE_EDIT_CHAR_GET_PRIVATE (object);

    if (priv->character)
        g_object_unref (priv->character);
    if (priv->dict)
        g_object_unref (priv->dict);

    priv->character = NULL;
    priv->dict      = NULL;

    if (G_OBJECT_CLASS(tomoe_edit_char_parent_class)->dispose)
        G_OBJECT_CLASS(tomoe_edit_char_parent_class)->dispose(object);
}

static void
set_property (GObject      *object,
              guint         prop_id,
              const GValue *value,
              GParamSpec   *pspec)
{
    TomoeEditCharPrivate *priv = TOMOE_EDIT_CHAR_GET_PRIVATE (object);

    switch (prop_id) {
        case PROP_TOMOE_CHAR:
            priv->character = g_object_ref (g_value_get_object (value));
            break;
        case PROP_TOMOE_DICT:
            priv->dict = g_object_ref (g_value_get_object (value));
            break;

        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

static void
get_property (GObject    *object,
              guint       prop_id,
              GValue     *value,
              GParamSpec *pspec)
{
    TomoeEditCharPrivate *priv = TOMOE_EDIT_CHAR_GET_PRIVATE (object);

    switch (prop_id) {
        case PROP_TOMOE_CHAR:
            g_value_set_object (value, priv->character);
            break;
        case PROP_TOMOE_DICT:
            g_value_set_object (value, priv->dict);
            break;

        default:
            G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
            break;
    }
}

GtkWidget *
tomoe_edit_char_new (TomoeChar *chr, TomoeDict *dict)
{
    GtkWidget *w = GTK_WIDGET(g_object_new (TOMOE_TYPE_EDIT_CHAR,
                                            "tomoe-char", chr,
                                            "tomoe-dict", dict,
                                            NULL));
    return w;
}

static void
on_ok_button_clicked (GtkButton *button, gpointer user_data)
{
    TomoeEditChar *dialog = TOMOE_EDIT_CHAR (user_data);
    TomoeEditCharPrivate *priv = TOMOE_EDIT_CHAR_GET_PRIVATE (dialog);

    g_return_if_fail (TOMOE_IS_EDIT_CHAR (dialog));

    tomoe_char_set_utf8 (priv->character,
                         gtk_entry_get_text (GTK_ENTRY (priv->input)));
    gtk_dialog_response (GTK_DIALOG (dialog), GTK_RESPONSE_OK);
}

static void
on_cancel_button_clicked (GtkButton *button, gpointer user_data)
{
    GtkDialog *dialog = GTK_DIALOG (user_data);

    g_return_if_fail (GTK_IS_DIALOG (dialog));

    gtk_dialog_response (dialog, GTK_RESPONSE_CANCEL);
}

static void
on_input_changed (GtkEntry *input, gpointer user_data)
{
    TomoeEditChar *dialog = TOMOE_EDIT_CHAR (user_data);
    TomoeEditCharPrivate *priv = TOMOE_EDIT_CHAR_GET_PRIVATE (dialog);
    gboolean ok = FALSE;

    g_return_if_fail (TOMOE_IS_EDIT_CHAR (dialog));

    ok = (strcmp (gtk_entry_get_text (GTK_ENTRY (priv->input)), "") != 0);

    gtk_widget_set_sensitive (priv->ok_button, ok);
}
/*
 * vi:ts=4:nowrap:ai:expandtab
 */
