import subprocess
import glob
import os
import re
import libcalamares
from libcalamares.utils import target_env_process_output

def find_latest_gentoo_initramfs():
    root_mount_point = libcalamares.globalstorage.value("rootMountPoint")
    if not root_mount_point:
        raise ValueError("rootMountPoint not set in global storage")
    
    target_boot_path = os.path.join(root_mount_point, 'boot')
    search_pattern = os.path.join(target_boot_path, 'initramfs-*-gentoo-dist.img')
    candidates = glob.glob(search_pattern)
    
    if not candidates:
        raise FileNotFoundError(f"No Gentoo dist initramfs found in {target_boot_path}")

    def extract_version(path):
        basename = os.path.basename(path)
        match = re.search(r'initramfs-(\d+\.\d+\.\d+)-gentoo-dist\.img', basename)
        if match:
            return tuple(map(int, match.group(1).split('.')))
        return (0, 0, 0)

    candidates.sort(key=lambda x: extract_version(x), reverse=True)
    return candidates[0]

def extract_kernel_simple_version(initramfs_path):
    basename = os.path.basename(initramfs_path)
    match = re.search(r'initramfs-(\d+\.\d+\.\d+)-gentoo-dist\.img', basename)
    if match:
        return match.group(1)
    raise ValueError(f"Could not extract simple version from initramfs filename: {basename}")

def is_systemd_stage3():
    stage_name = libcalamares.globalstorage.value("STAGE_NAME_TAR")
    if stage_name and "systemd" in stage_name.lower():
        return True
    return False

def is_root_encrypted():
    partitions = libcalamares.globalstorage.value("partitions")
    if not partitions:
        return False
    for partition in partitions:
        if partition.get("mountPoint") == "/" and "luksMapperName" in partition:
            return True
    return False

def ensure_cryptsetup_for_openrc():
    """Ensure cryptsetup is installed for OpenRC encrypted systems.
    
    Systemd dracut includes its own cryptsetup implementation,
    but OpenRC dracut needs the sys-apps/cryptsetup package.
    """
    if is_systemd_stage3():
        return
    
    root_mount_point = libcalamares.globalstorage.value("rootMountPoint")
    if not root_mount_point:
        return
    
    try:
        check_result = target_env_process_output(['which', 'cryptsetup'])
        libcalamares.utils.debug("cryptsetup already installed")
    except:
        libcalamares.utils.debug("Installing cryptsetup for OpenRC encryption")
        try:
            target_env_process_output(['emerge', '--quiet', 'sys-apps/cryptsetup'])
        except subprocess.CalledProcessError as e:
            libcalamares.utils.warning(f"Failed to install cryptsetup: {e}")

def configure_dracut_persistent():
    """Write persistent dracut configuration for future kernel upgrades.
    This ensures that when users upgrade gentoo-kernel-bin in the future,
    the installkernel dracut plugin will use the correct options.
    """
    root_mount_point = libcalamares.globalstorage.value("rootMountPoint")
    if not root_mount_point:
        return
    
    dracut_conf_dir = os.path.join(root_mount_point, "etc/dracut.conf.d")
    dracut_conf_path = os.path.join(dracut_conf_dir, "10-calamares.conf")
    
    if os.path.exists(dracut_conf_path):
        libcalamares.utils.debug(f"Dracut config already exists at {dracut_conf_path}, skipping")
        return
    
    os.makedirs(dracut_conf_dir, exist_ok=True)
    
    with open(dracut_conf_path, 'w') as conf_file:
        conf_file.write("# Generated by Calamares installer\n")
        conf_file.write("# Configuration for dracut initramfs generation\n\n")
        
        conf_file.write('hostonly="yes"\n')
        conf_file.write('hostonly_cmdline="yes"\n\n')
        
        if not is_systemd_stage3():
            conf_file.write("# Exclude systemd modules for OpenRC\n")
            conf_file.write('omit_dracutmodules+=" systemd systemd-initrd systemd-networkd dracut-systemd plymouth "\n\n')
        
        if is_root_encrypted():
            conf_file.write("# Add encryption support\n")
            conf_file.write('add_dracutmodules+=" crypt dm rootfs-block "\n')
    
    libcalamares.utils.debug(f"Created persistent dracut configuration at {dracut_conf_path}")

def configure_openrc_dmcrypt():
    """Configure OpenRC dmcrypt service for encrypted partitions (non-root)."""
    if is_systemd_stage3():
        return
    
    partitions = libcalamares.globalstorage.value("partitions")
    if not partitions:
        return
    
    root_mount_point = libcalamares.globalstorage.value("rootMountPoint")
    if not root_mount_point:
        return
    
    dmcrypt_conf_path = os.path.join(root_mount_point, "etc/conf.d/dmcrypt")
    unencrypted_separate_boot = any(
        p.get("mountPoint") == "/boot" and "luksMapperName" not in p 
        for p in partitions
    )
    
    for partition in partitions:
        has_luks = "luksMapperName" in partition
        skip_partitions = partition.get("mountPoint") == "/" or partition.get("fs") == "linuxswap"
        
        if has_luks and not skip_partitions:
            crypto_target = partition["luksMapperName"]
            crypto_source = f"/dev/disk/by-uuid/{partition['luksUuid']}"
            
            libcalamares.utils.debug(
                f"Writing OpenRC LUKS configuration for partition {partition.get('mountPoint')}"
            )
            
            with open(dmcrypt_conf_path, 'a+') as dmcrypt_file:
                dmcrypt_file.write(f"\ntarget={crypto_target}")
                dmcrypt_file.write(f"\nsource={crypto_source}")
                if not unencrypted_separate_boot:
                    dmcrypt_file.write("\nkey=/crypto_keyfile.bin")
                dmcrypt_file.write("\n")

def run():
    try:
        dracut_options = [
            "-H", "-f",
            "-o", "systemd", "-o", "systemd-initrd", "-o", "systemd-networkd",
            "-o", "dracut-systemd", "-o", "plymouth",
            "--early-microcode"
        ]
        
        if is_root_encrypted():
            dracut_options.extend([
                "--add", "crypt",
                "--add", "dm",
                "--add", "rootfs-block"
            ])
        
        latest_initramfs = find_latest_gentoo_initramfs()
        simple_version = extract_kernel_simple_version(latest_initramfs)
        dracut_options.append(f'--kver={simple_version}-gentoo-dist')
        
        if is_root_encrypted():
            ensure_cryptsetup_for_openrc()
            
            # Write persistent dracut configuration BEFORE generating initramfs
            # This ensures future kernel upgrades use the correct options
            # Only write for encrypted systems to maintain original behavior for non-encrypted
            configure_dracut_persistent()
        
        result = target_env_process_output(['dracut'] + dracut_options)
        libcalamares.utils.debug(f"Successfully created initramfs for kernel {simple_version}-gentoo-dist")
        
        if is_root_encrypted():
            configure_openrc_dmcrypt()
        
    except FileNotFoundError as e:
        libcalamares.utils.warning(f"No Gentoo initramfs found: {e}")
        return 1
    except ValueError as e:
        libcalamares.utils.warning(f"Failed to extract kernel version: {e}")
        return 1
    except subprocess.CalledProcessError as cpe:
        libcalamares.utils.warning(f"Dracut failed with output: {cpe.output}")
        return cpe.returncode
    
    return None