/*
    SPDX-FileCopyrightText: 2009, 2010, 2011, 2012 Alex Richardson <alex.richardson@gmx.de>

    SPDX-License-Identifier: LGPL-2.1-only OR LGPL-3.0-only OR LicenseRef-KDE-Accepted-LGPL
*/

#include "enumdatainformation.hpp"

// lib
#include <topleveldatainformation.hpp>
#include <classes/enumscriptclass.hpp>
#include <scriptlogger.hpp>
// KF
#include <KComboBox>
#include <KLocalizedString>
// Std
#include <utility>

EnumDataInformation::EnumDataInformation(const QString& name,
                                         std::unique_ptr<PrimitiveDataInformation>&& type,
                                         const std::shared_ptr<EnumDefinition>& enumDef, DataInformation* parent)
    : PrimitiveDataInformationWrapper(name, std::move(type), parent)
    , mEnum(enumDef)
{
    if (enumDef->type() != mValue->type()) {
        logWarn().nospace() << "Incompatible types in definition and value: '"
                  << enumDef->type() << "' and '" << mValue->type() << "'.";
    }
    mValue->setParent(this);
}

EnumDataInformation::~EnumDataInformation() = default;

QWidget* EnumDataInformation::createEditWidget(QWidget* parent) const
{
    auto* const box = new KComboBox(false, parent);

    const std::map<AllPrimitiveTypes, QString>& enumValues = mEnum->values();

    // TODO: have an option to edit custom values not defined in the enum
    if (enumValues.find(mValue->value()) == enumValues.end()) {
        const QString text = i18n("%1 (value not in enum)", mValue->valueString());
        box->addItem(text, mValue->valueToQVariant());
        box->insertSeparator(1);
    }

    for (const auto& [enumeralValue, enumeralName] : enumValues) {
        const QString text = i18n("%1 (%2)", enumeralName, mValue->valueToQString(enumeralValue));
        box->addItem(text, mValue->valueToQVariant(enumeralValue));
    }
    return box;
}

QVariant EnumDataInformation::dataFromWidget(const QWidget* w) const
{
    const auto* const box = qobject_cast<const KComboBox*>(w);
    return box->currentData();
}

void EnumDataInformation::setWidgetData(QWidget* w) const
{
    auto* const box = qobject_cast<KComboBox*>(w);
    const int dataIndex = box->findData(mValue->valueToQVariant());
    if (dataIndex != -1) {
        box->setCurrentIndex(dataIndex);
    }
}

QString EnumDataInformation::valueStringImpl() const
{
    Q_ASSERT(mWasAbleToRead);
    QString enumVal = mEnum->value(mValue->value());
    if (!enumVal.isEmpty()) {
        return i18n("%1 (%2)", enumVal, mValue->valueString());
    }

    return i18n("%1 (value not in enum)", mValue->valueString());
}

QString EnumDataInformation::typeNameImpl() const
{
    return i18nc("Displayed in the type column. first comes the name "
                 "of the enum, then the underlying type (e.g. uint32)",
                 "enum %1 (%2)", mEnum->name(), mValue->typeName());
}

QScriptClass* EnumDataInformation::scriptClass(ScriptHandlerInfo* handlerInfo) const
{
    return handlerInfo->enumClass();
}

bool EnumDataInformation::isEnum() const
{
    return true;
}
