"use strict";
// Copyright 2021-2024 Buf Technologies, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.
Object.defineProperty(exports, "__esModule", { value: true });
exports.getFieldZeroValueExpression = exports.getFieldDefaultValueExpression = exports.getFieldTypeInfo = void 0;
const protobuf_1 = require("@bufbuild/protobuf");
const ecmascript_1 = require("@bufbuild/protoplugin/ecmascript");
function getFieldTypeInfo(field) {
    const typing = [];
    let typingInferrableFromZeroValue;
    let optional = false;
    switch (field.fieldKind) {
        case "scalar":
            typing.push(scalarTypeScriptType(field.scalar, field.longType));
            optional =
                field.optional ||
                    field.proto.label === protobuf_1.FieldDescriptorProto_Label.REQUIRED;
            typingInferrableFromZeroValue = true;
            break;
        case "message": {
            const baseType = protobuf_1.codegenInfo.getUnwrappedFieldType(field);
            if (baseType !== undefined) {
                typing.push(scalarTypeScriptType(baseType, protobuf_1.LongType.BIGINT));
            }
            else {
                typing.push({
                    kind: "es_ref_message",
                    type: field.message,
                    typeOnly: true,
                });
            }
            optional = true;
            typingInferrableFromZeroValue = true;
            break;
        }
        case "enum":
            typing.push({
                kind: "es_ref_enum",
                type: field.enum,
                typeOnly: true,
            });
            optional =
                field.optional ||
                    field.proto.label === protobuf_1.FieldDescriptorProto_Label.REQUIRED;
            typingInferrableFromZeroValue = true;
            break;
        case "map": {
            let keyType;
            switch (field.mapKey) {
                case protobuf_1.ScalarType.INT32:
                case protobuf_1.ScalarType.FIXED32:
                case protobuf_1.ScalarType.UINT32:
                case protobuf_1.ScalarType.SFIXED32:
                case protobuf_1.ScalarType.SINT32:
                    keyType = "number";
                    break;
                default:
                    keyType = "string";
                    break;
            }
            let valueType;
            switch (field.mapValue.kind) {
                case "scalar":
                    valueType = scalarTypeScriptType(field.mapValue.scalar, protobuf_1.LongType.BIGINT);
                    break;
                case "message":
                    valueType = {
                        kind: "es_ref_message",
                        type: field.mapValue.message,
                        typeOnly: true,
                    };
                    break;
                case "enum":
                    valueType = {
                        kind: "es_ref_enum",
                        type: field.mapValue.enum,
                        typeOnly: true,
                    };
                    break;
            }
            typing.push("{ [key: ", keyType, "]: ", valueType, " }");
            typingInferrableFromZeroValue = false;
            optional = false;
            break;
        }
    }
    if (field.repeated) {
        typing.push("[]");
        optional = false;
        typingInferrableFromZeroValue = false;
    }
    return { typing, optional, typingInferrableFromZeroValue };
}
exports.getFieldTypeInfo = getFieldTypeInfo;
/**
 * Return a printable expression for the default value of a field.
 * Only applicable for singular scalar and enum fields.
 */
function getFieldDefaultValueExpression(field, enumAs = "enum_value_as_is") {
    if (field.repeated) {
        return undefined;
    }
    if (field.fieldKind !== "enum" && field.fieldKind !== "scalar") {
        return undefined;
    }
    const defaultValue = field.getDefaultValue();
    if (defaultValue === undefined) {
        return undefined;
    }
    switch (field.fieldKind) {
        case "enum": {
            const enumValue = field.enum.values.find((value) => value.number === defaultValue);
            if (enumValue === undefined) {
                throw new Error(`invalid enum default value: ${String(defaultValue)} for ${enumValue}`);
            }
            return literalEnumValue(enumValue, enumAs);
        }
        case "scalar":
            return literalScalarValue(defaultValue, field);
    }
}
exports.getFieldDefaultValueExpression = getFieldDefaultValueExpression;
/**
 * Return a printable expression for the zero value of a field.
 *
 * Returns either:
 * - empty array literal for repeated fields
 * - empty object literal for maps
 * - undefined for message fields
 * - an enums first value
 * - scalar zero value
 */
function getFieldZeroValueExpression(field, enumAs = "enum_value_as_is") {
    if (field.repeated) {
        return "[]";
    }
    switch (field.fieldKind) {
        case "message":
            return undefined;
        case "map":
            return "{}";
        case "enum": {
            // In proto3, the first enum value must be zero.
            // In proto2, protobuf-go returns the first value as the default.
            if (field.enum.values.length < 1) {
                throw new Error("invalid enum: missing at least one value");
            }
            const zeroValue = field.enum.values[0];
            return literalEnumValue(zeroValue, enumAs);
        }
        case "scalar": {
            const defaultValue = protobuf_1.codegenInfo.scalarZeroValue(field.scalar, field.longType);
            return literalScalarValue(defaultValue, field);
        }
    }
}
exports.getFieldZeroValueExpression = getFieldZeroValueExpression;
function literalScalarValue(value, field) {
    switch (field.scalar) {
        case protobuf_1.ScalarType.DOUBLE:
        case protobuf_1.ScalarType.FLOAT:
        case protobuf_1.ScalarType.INT32:
        case protobuf_1.ScalarType.FIXED32:
        case protobuf_1.ScalarType.UINT32:
        case protobuf_1.ScalarType.SFIXED32:
        case protobuf_1.ScalarType.SINT32:
            if (typeof value != "number") {
                throw new Error(`Unexpected value for ${protobuf_1.ScalarType[field.scalar]} ${field.toString()}: ${String(value)}`);
            }
            return value;
        case protobuf_1.ScalarType.BOOL:
            if (typeof value != "boolean") {
                throw new Error(`Unexpected value for ${protobuf_1.ScalarType[field.scalar]} ${field.toString()}: ${String(value)}`);
            }
            return value;
        case protobuf_1.ScalarType.STRING:
            if (typeof value != "string") {
                throw new Error(`Unexpected value for ${protobuf_1.ScalarType[field.scalar]} ${field.toString()}: ${String(value)}`);
            }
            return { kind: "es_string", value };
        case protobuf_1.ScalarType.BYTES:
            if (!(value instanceof Uint8Array)) {
                throw new Error(`Unexpected value for ${protobuf_1.ScalarType[field.scalar]} ${field.toString()}: ${String(value)}`);
            }
            return value;
        case protobuf_1.ScalarType.INT64:
        case protobuf_1.ScalarType.SINT64:
        case protobuf_1.ScalarType.SFIXED64:
        case protobuf_1.ScalarType.UINT64:
        case protobuf_1.ScalarType.FIXED64:
            if (typeof value != "bigint" && typeof value != "string") {
                throw new Error(`Unexpected value for ${protobuf_1.ScalarType[field.scalar]} ${field.toString()}: ${String(value)}`);
            }
            return {
                kind: "es_proto_int64",
                type: field.scalar,
                longType: field.longType,
                value,
            };
    }
}
function literalEnumValue(value, enumAs) {
    switch (enumAs) {
        case "enum_value_as_is":
            return [
                { kind: "es_ref_enum", type: value.parent, typeOnly: false },
                ".",
                (0, ecmascript_1.localName)(value),
            ];
        case "enum_value_as_integer":
            return [
                value.number,
                " /* ",
                value.parent.typeName,
                ".",
                value.name,
                " */",
            ];
        case "enum_value_as_cast_integer":
            return [
                value.number,
                " as ",
                { kind: "es_ref_enum", type: value.parent, typeOnly: true },
                ".",
                (0, ecmascript_1.localName)(value),
            ];
    }
}
function scalarTypeScriptType(type, longType) {
    switch (type) {
        case protobuf_1.ScalarType.STRING:
            return "string";
        case protobuf_1.ScalarType.BOOL:
            return "boolean";
        case protobuf_1.ScalarType.UINT64:
        case protobuf_1.ScalarType.SFIXED64:
        case protobuf_1.ScalarType.FIXED64:
        case protobuf_1.ScalarType.SINT64:
        case protobuf_1.ScalarType.INT64:
            if (longType === protobuf_1.LongType.STRING) {
                return "string";
            }
            return "bigint";
        case protobuf_1.ScalarType.BYTES:
            return "Uint8Array";
        default:
            return "number";
    }
}
