/*
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.j2objc;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringWriter;
import java.net.URL;
import java.util.Enumeration;
import junit.framework.TestCase;

/** Tests that embedded resources are found and returned. */
public final class EmbeddedResourceTest extends TestCase {

  private static final String RESOURCE_NAME = "embedded_resource.data";
  private static final String RESOURCE_NAME_WITH_DASH = "another-resource.data";
  private static final String RESOURCE_PATH = "com/google/j2objc/embedded_resource.data";

  private void assertResourceData(InputStream resource) throws IOException {
    Reader in = new InputStreamReader(resource);
    StringWriter out = new StringWriter();
    char[] buffer = new char[64];
    int n;
    while ((n = in.read(buffer)) != -1) {
      out.write(buffer, 0, n);
    }
    assertEquals("Success!\n", out.toString());
  }

  public void testClassGetResourceAsStream() throws IOException {
    InputStream resource = getClass().getResourceAsStream(RESOURCE_NAME);
    assertNotNull("resource not found", resource);
    assertResourceData(resource);
  }

  public void testClassLoaderGetResourceAsStream() throws IOException {
    InputStream resource = getClass().getClassLoader().getResourceAsStream(RESOURCE_PATH);
    assertNotNull("resource not found", resource);
    assertResourceData(resource);
  }

  public void testClassGetResource() throws IOException {
    URL url = getClass().getResource(RESOURCE_NAME);
    assertNotNull("resource not found", url);
    assertResourceData(url.openStream());
  }

  public void testClassLoaderGetResource() throws IOException {
    URL url = getClass().getClassLoader().getResource(RESOURCE_PATH);
    assertNotNull("resource not found", url);
    assertResourceData(url.openStream());
  }

  public void testClassLoaderGetResources() throws IOException {
    Enumeration<URL> urls = getClass().getClassLoader().getResources(RESOURCE_PATH);
    assertTrue("resource not found", urls.hasMoreElements());
    assertResourceData(urls.nextElement().openStream());
  }

  public void testClassGetResourceWithDashedName() throws IOException {
    URL url = getClass().getResource(RESOURCE_NAME_WITH_DASH);
    assertNotNull("resource not found", url);
    assertResourceData(url.openStream());
  }

  /*-[
    //
    // Generated by gen_java_source_jar.py
    //

    #include "J2ObjC_source.h"

    static jbyte _com_google_j2objc_embedded_resource_data[] = {
      0x53, 0x75, 0x63, 0x63, 0x65, 0x73, 0x73, 0x21, 0x0A,  // "Success!\n"
    };
    J2OBJC_RESOURCE(_com_google_j2objc_embedded_resource_data, 9, 0x20822135)

    static jbyte _com_google_j2objc_another_resource_data[] = {
      0x53, 0x75, 0x63, 0x63, 0x65, 0x73, 0x73, 0x21, 0x0A,  // "Success!\n"
    };
    J2OBJC_RESOURCE(_com_google_j2objc_another_resource_data, 9, 0x11fd530f)
  ]-*/
}
