Example of illegal code:

```ada
X : Integer;

function F return Integer is
begin
   X := 1; -- error: write to global variable in function
   return 2;
end F;
```

This error is issued on functions which may write variables in the enclosing
scopes. This side-effect is not allowed in functions in SPARK, as this could
lead to different behaviors depending on the choice made by the compiler in the
order of evaluation of sub-expressions.

The fix is to declare that the function has side-effects with the aspect
`Side_Effects`:

```ada
X : Integer;

function F return Integer
  with Side_Effects
is
begin
   X := 1; -- ok
   return 2;
end F;
```

As a result, calls to `F` can only appear as the right-hand side of
assignments, which may require changing `F`'s callers:

```ada
declare
   Y : Integer := F; -- error
begin
   Y := F; -- ok
   Y := F + 1; -- error
end;
```

Note that the same error is issued when reading a volatile variable in the
enclosing scope, unless its volatile property `Effective_Reads` is set to
`False`, as just a read of the variable is considered to have an effect on the
value of the variable in that case.

Another possibility, if the side-effect is not relevant for analysis (say, only
for logging or debugging) is to hide it by marking the body of `F` as not in
SPARK with aspect `SPARK_Mode => Off`:

```ada
X : Integer;

function F return Integer
  with Global => null, Always_Terminates;

function F return Integer
  with SPARK_Mode => Off
is
begin
   X := 1; -- hidden
   return 2;
end F;
```

Note the contract on `F` to indicate that the function does not read or write
any global variable and always terminates, which should be validated by review
on the body of `F`.
