/*

This file is part of the Fuzion language implementation.

The Fuzion language implementation is free software: you can redistribute it
and/or modify it under the terms of the GNU General Public License as published
by the Free Software Foundation, version 3 of the License.

The Fuzion language implementation is distributed in the hope that it will be
useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public
License for more details.

You should have received a copy of the GNU General Public License along with The
Fuzion language implementation.  If not, see <https://www.gnu.org/licenses/>.

*/

/*-----------------------------------------------------------------------
 *
 * Tokiwa Software GmbH, Germany
 *
 * Source of class u64Value
 *
 *---------------------------------------------------------------------*/

package dev.flang.be.interpreter;

import dev.flang.fuir.SpecialClazzes;

/**
 * u64Value is a value of type u64
 *
 * @author Fridtjof Siebert (siebert@tokiwa.software)
 */
public class u64Value extends Value
{


  /*----------------------------  variables  ----------------------------*/


  /**
   *
   */
  private long _val;


  /*--------------------------  constructors  ---------------------------*/


  /**
   * Constructor
   */
  public u64Value(long val)
  {
    _val = val;
  }


  /*-----------------------------  methods  -----------------------------*/


  /**
   * toString
   *
   * @return
   */
  public String toString()
  {
    return Long.toUnsignedString(_val);
  }


  /**
   * For a value of type u64, return the value.
   *
   * @return the u64 value
   */
  public long u64Value()
  {
    return _val;
  }


  /**
   * Store this value in a field
   *
   * @param slot the slot that addresses the field this should be stored in.
   *
   * @param size the size of the data to be stored
   */
  void storeNonRef(LValue slot, int size)
  {
    if (PRECONDITIONS) require
      (size == 2);

    slot.container.nonrefs[slot.offset    ] = (int) _val;
    slot.container.nonrefs[slot.offset + 1] = (int) (_val >> 32);
  }


  /**
   * Debugging only: Check that this value is valid as the current instance for
   * a feature with given static clazz.
   *
   * @param expected the static clazz of the feature this value is called on.
   *
   * @throws Error in case this does not match the expected clazz
   */
  void checkStaticClazz(int expected)
  {
    if (expected != fuir().clazz(SpecialClazzes.c_u16))
      {
        throw new Error("u64 value not allowed for clazz " + expected);
      }
  }


  @Override
  protected Object toNative()
  {
    return u64Value();
  }


}

/* end of file */
