// This file is part of arduino-cli.
//
// Copyright 2020 ARDUINO SA (http://www.arduino.cc/)
//
// This software is released under the GNU General Public License version 3,
// which covers the main part of arduino-cli.
// The terms of this license can be found at:
// https://www.gnu.org/licenses/gpl-3.0.en.html
//
// You can be released from the requirements of the above licenses by purchasing
// a commercial license. Buying such a license is mandatory if you want to
// modify or otherwise use the software for commercial activities involving the
// Arduino software without disclosing the source code of your own applications.
// To purchase a commercial license, send an email to license@arduino.cc.

package builder

import (
	"encoding/json"
	"path/filepath"

	"github.com/arduino/arduino-cli/legacy/builder/builder_utils"
	"github.com/arduino/arduino-cli/legacy/builder/constants"
	"github.com/arduino/arduino-cli/legacy/builder/gohasissues"
	"github.com/arduino/arduino-cli/legacy/builder/types"
	properties "github.com/arduino/go-properties-orderedmap"
	"github.com/pkg/errors"
)

type WipeoutBuildPathIfBuildOptionsChanged struct{}

func (s *WipeoutBuildPathIfBuildOptionsChanged) Run(ctx *types.Context) error {
	if ctx.BuildOptionsJsonPrevious == "" {
		return nil
	}
	buildOptionsJson := ctx.BuildOptionsJson
	previousBuildOptionsJson := ctx.BuildOptionsJsonPrevious

	var opts *properties.Map
	var prevOpts *properties.Map
	json.Unmarshal([]byte(buildOptionsJson), &opts)
	json.Unmarshal([]byte(previousBuildOptionsJson), &prevOpts)

	// If SketchLocation path is different but filename is the same, consider it equal
	if filepath.Base(opts.Get("sketchLocation")) == filepath.Base(prevOpts.Get("sketchLocation")) {
		opts.Remove("sketchLocation")
		prevOpts.Remove("sketchLocation")
	}

	// If options are not changed check if core has
	if opts.Equals(prevOpts) {
		// check if any of the files contained in the core folders has changed
		// since the json was generated - like platform.txt or similar
		// if so, trigger a "safety" wipe
		buildProperties := ctx.BuildProperties
		targetCoreFolder := buildProperties.GetPath(constants.BUILD_PROPERTIES_RUNTIME_PLATFORM_PATH)
		coreFolder := buildProperties.GetPath("build.core.path")
		realCoreFolder := coreFolder.Parent().Parent()
		jsonPath := ctx.BuildPath.Join(constants.BUILD_OPTIONS_FILE)
		coreHasChanged := builder_utils.TXTBuildRulesHaveChanged(realCoreFolder, targetCoreFolder, jsonPath)

		if !coreHasChanged {
			return nil
		}
	}

	// FIXME: this should go outside legacy and behind a `logrus` call so users can
	// control when this should be printed.
	// logger.Println(constants.LOG_LEVEL_INFO, constants.MSG_BUILD_OPTIONS_CHANGED)

	buildPath := ctx.BuildPath
	files, err := gohasissues.ReadDir(buildPath.String())
	if err != nil {
		return errors.WithStack(err)
	}
	for _, file := range files {
		buildPath.Join(file.Name()).RemoveAll()
	}

	return nil
}
