//==========================================
//  xhkeys_cdaudio.c
//------------------------------------------
// CD Audio control
// A plug-in module for xhkeys
// Michael Glickman  Jun-2004
//==========================================

#include <X11/Xlib.h>	// True False
#include "xhkeys_conf.h"

#ifdef CDROM_SUPPORT
#include <stdio.h>	// NULL
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <stdarg.h>
#include <ctype.h>
#include <linux/cdrom.h>
#include <sys/ioctl.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <errno.h>
#define XHKEYS_PLUGIN_CODE
#include "xhkeys_plugin.h"
#include "plugin_common.h"

static const char *parmNames[] = {"devname", "devgrab", "osd"};
static const int parmCount = sizeof(parmNames) / sizeof(const char *);

#define CD_MIN_VOLUME  0
#define CD_MAX_VOLUME 255

enum CmdCodes {
    CDAUDIO_PLAYPAUSE,  CDAUDIO_PLAY, 
    CDAUDIO_PAUSE,      CDAUDIO_STOP,
    CDAUDIO_LOADEJECT,  CDAUDIO_LOAD, 
    CDAUDIO_EJECT,      CDAUDIO_SLOT,
    CDAUDIO_STATUS,   
    CDAUDIO_VOLUME,  CDAUDIO_BALANCE, 
    CDAUDIO_CODE_COUNT
};    


static const struct 
{
    const char *opname;
    int flags;

}
OpData[] =
{
    { "PlayPause", XHKEYS_PLUGOPTYPE_PARMNONE |XHKEYS_PLUGOPTYPE_PERSISTENT},
    { "Play", XHKEYS_PLUGOPTYPE_PARMOPTIONAL | XHKEYS_PLUGOPTYPE_PERSISTENT},
    { "Pause", XHKEYS_PLUGOPTYPE_PARMNONE},
    { "Stop", XHKEYS_PLUGOPTYPE_PARMNONE},
    { "LoadEject", XHKEYS_PLUGOPTYPE_PARMNONE},
    { "Load", XHKEYS_PLUGOPTYPE_PARMNONE},
    { "Eject", XHKEYS_PLUGOPTYPE_PARMNONE},
    { "Slot", XHKEYS_PLUGOPTYPE_PARMREQUIRED},
    { "Status", XHKEYS_PLUGOPTYPE_PARMNONE | XHKEYS_PLUGOPTYPE_CANCONTINUE},
    { "Volume", XHKEYS_PLUGOPTYPE_PARMOPTIONAL | XHKEYS_PLUGOPTYPE_PERSISTENT},
    { "Balance", XHKEYS_PLUGOPTYPE_PARMOPTIONAL | XHKEYS_PLUGOPTYPE_PERSISTENT}
};


const char *Description = "CD Audio Control";
const char *OpenErrorMessage = "CD open error code %d";
const char *IoctlErrorMessage = "CD open error code %d";
const char *InvArgErrorMessage = "Invalid '%s' argument";
// const char *CantContinueErrorMessage = "%s can't run continuosly with non-zero increment!";

typedef struct 
{
    char CDDevName[256];
    int cdrom_fd;
    Bool grabCD;
    Bool osd;
} BufferData;



static int get_cdrom_fd(BufferData *buffer)
{
    int cdrom_fd = buffer->cdrom_fd;
    
    if (cdrom_fd < 0)
	buffer->cdrom_fd =
	   cdrom_fd = open(buffer->CDDevName, O_RDONLY | O_NONBLOCK);

    if (cdrom_fd >= 0) {
  	 ioctl(cdrom_fd, CDROMEJECT_SW, NULL);
    }

    return cdrom_fd;  	  
}	  

static void release_cdrom_fd(BufferData *buffer, Bool force)
{
    int cdrom_fd = buffer->cdrom_fd;
    
    if (cdrom_fd >= 0 && (force || (buffer->grabCD)==False)) {
//  if (cdrom_fd >= 0) {
	close(cdrom_fd);
	buffer->cdrom_fd = -1;
    }	
}

//----------------------------------------------------------
// Playing routines
static void lbatomsf(int lba, short *minute, short *second, short *frame)
{
	lba += CD_MSF_OFFSET;
	*frame = lba % CD_FRAMES;
	lba /= CD_FRAMES;
	*second = lba % CD_SECS;
	*minute = lba / CD_SECS;
}

/*
static int msftolba(short minute, short second, short frame)
{
	return (minute*CD_SECS+second)*CD_FRAMES+frame-CD_MSF_OFFSET; 	
}
*/

static Bool cd_get_status_text(BufferData *buffer, char *message, int msglen) 
{
    struct cdrom_subchnl subchn;
    char statusText[31];
    char positionText[81];
    Bool statusOK = False;

    int cdrom_fd = get_cdrom_fd(buffer);
    
    if (cdrom_fd < 0) {
	snprintf(message, msglen,  OpenErrorMessage, errno);
        return False;
    }
    
    subchn.cdsc_format = CDROM_MSF;

    if (ioctl(cdrom_fd, CDROMSUBCHNL, &subchn) == -1) 
        strcpy(statusText, "Ejected");
    else {
	switch(subchn.cdsc_audiostatus)
	{
	    case CDROM_AUDIO_PLAY:
		strcpy(statusText, "Playing");
		statusOK = True;
		break;
	    
	    case CDROM_AUDIO_PAUSED:
		strcpy(statusText, "Paused");
		statusOK = True;
		break;

	    case CDROM_AUDIO_COMPLETED:
		strcpy(statusText, "Completed");
		statusOK = True;
		break;

	    case CDROM_AUDIO_NO_STATUS:
		strcpy(statusText, "Stopped");
		statusOK = True;
		break;

	    default:
		sprintf(statusText, "Error [%d]", subchn.cdsc_audiostatus);
		break;
    	}
    }	
    
    if (statusOK) {
	int trk = subchn.cdsc_trk;
	if (trk == CDROM_LEADOUT)
	    sprintf(positionText, "End of disc / session");
	else        
	    sprintf(positionText, "Track %d  Pos %02d:%02d", trk,
	    subchn.cdsc_reladdr.msf.minute, subchn.cdsc_reladdr.msf.second);
    } else
	strcpy(positionText, "");
	
    snprintf(message, msglen, "CDA %s  %s", statusText, positionText);	   
    release_cdrom_fd(buffer, False);
    return True;
    
}

static Bool cd_play_from_lba(int cdrom_fd, int lba, int dir) {
  struct cdrom_msf playmsf;
  int max_lba;

  {
    struct cdrom_tocentry te;
	te.cdte_track = CDROM_LEADOUT;
	te.cdte_format = CDROM_LBA;
    if(ioctl(cdrom_fd, CDROMREADTOCENTRY, &te) == -1) return False; 
	max_lba = te.cdte_addr.lba;
  }
    
  if (lba < 0 || lba >= max_lba)	return False;
  {  
	short minute, second, frame;
	lbatomsf(lba, &minute, &second, &frame);
	playmsf.cdmsf_min0 = minute;
        playmsf.cdmsf_sec0 = second;
	playmsf.cdmsf_frame0 = frame;
	lbatomsf(max_lba, &minute, &second, &frame);
	playmsf.cdmsf_min1 = minute;
	playmsf.cdmsf_sec1 = second;
	playmsf.cdmsf_frame1 = frame;
  }	

  return (ioctl(cdrom_fd, CDROMPLAYMSF, &playmsf) != -1);
}

static Bool cd_play_from_trk(int cdrom_fd, short trk, short index, short dir) {
  struct cdrom_ti playti;
  struct cdrom_tochdr thdr;

  if (ioctl(cdrom_fd, CDROMREADTOCHDR, &thdr) == 1) return False;

  if (trk < thdr.cdth_trk0)  trk = (dir < 0) ? thdr.cdth_trk1 : thdr.cdth_trk0;
  if (trk > thdr.cdth_trk1)  trk = (dir > 0) ? thdr.cdth_trk0 : thdr.cdth_trk1;
    
  playti.cdti_trk0 = trk;
  playti.cdti_trk1 = index;
  playti.cdti_trk1 = CDROM_LEADOUT;
  playti.cdti_ind1 = 0;

  return (ioctl(cdrom_fd, CDROMPLAYTRKIND, &playti) != -1);
}

static Bool cd_play_parsed(int cdrom_fd, short addrType, int value, short dir) 
{
    if (addrType == 2)
    {	// Start of disk
	struct cdrom_tochdr thdr;
	if (ioctl(cdrom_fd, CDROMREADTOCHDR, &thdr) == 1) return False;
	return cd_play_from_trk(cdrom_fd, thdr.cdth_trk0, 0, 1);
    }


    if (dir != 0) {
	struct cdrom_subchnl subch;
	subch.cdsc_format = CDROM_LBA;
	if (ioctl(cdrom_fd, CDROMSUBCHNL, &subch) == -1) return -1;
	if (dir < 0) value = -value;
	value += (addrType == 0) ? (int) subch.cdsc_trk & 0xff : subch.cdsc_absaddr.lba;
    }

    if (addrType == 0)
	return cd_play_from_trk(cdrom_fd, value, 0, dir);

  return cd_play_from_lba(cdrom_fd, value, dir);
  	
}


// Returns the type of value:
//	 0 - TRACK absolute,
//   1 -  LBA absolute,
//   2 - start of disk
//   (-1) - invalid
static short cd_parse_play_parm(char *parm, int *valPtr, short *offsTypePtr)
{
  char *sepPtr;
  int addrType = -1;		// We don't know the type yet
  int offsType = 0;			// 0 -abs addr, 1 - plus, (-1) - minus,
  int value = 0;
  int crnt = 0;

  
  while (*parm == ' ') parm++;
  
  switch(toupper(*parm))
  {
	case '+':
	  offsType = 1; parm++;
	  break;
	  
	case '-':
	  offsType = -1; parm++;
	  break;

	case 'S':
	case 'D':
	  addrType = 2;
	  break;
  
	case '*':			// Current position
	case '\0':
	  addrType = 1;
	  offsType = 1;
	  break;
	  
	default:
	  offsType = 0;
  }	  

  if (addrType != 2)
	while (*parm != '\0') {
  	  crnt = strtol(parm, &sepPtr, 0);

	  switch(toupper(*sepPtr))
	  {
		case 'M':					// minutes
		  crnt *= CD_SECS;		// convert minus to seconds
	                            // no break!  					  

		case 'S':					// seconds 
		  crnt *= CD_FRAMES;		// convert seconds to frames
	                            // no break!  					  

		case 'F':					// frames
		  if (addrType == 0) return -1;  // No contradiction
		  addrType = 1;			// MSF
	  	  value += crnt;
		  break;	                // At last !		
		
		case 'T':					// tracks
		case '\0':
		  if (addrType == 1) return -1;	 // No contradiction
		  addrType = 0;			// Track
	  	  value = crnt;
		break;

                          // We just use the following break :=)				
	  default:		
		return -1;				// Invalid character	
	}
	
	if (*sepPtr == '\0') break;
	parm = sepPtr+1;	
	while (*parm == ' ') parm++;
  }

  if (offsType == 0 && addrType == 1)
	value -= CD_MSF_OFFSET; 	 // For absolute MSF take away 2 secs

  *offsTypePtr = offsType;
  *valPtr = value;
  return addrType;

}


static Bool cd_play_parm(BufferData *buffer, const char *parm, char *message, int msglen)
{
    int value = 0;
    short offsetType=0, dir = 0;
    Bool success = False;

    int cdrom_fd = get_cdrom_fd(buffer);
      
    if (cdrom_fd < 0) {
	snprintf(message, msglen, OpenErrorMessage, errno);
        return False;
    }

    if (parm != NULL) {    
      char *parmCopy = strdup(parm);
      offsetType = cd_parse_play_parm(parmCopy, &value, &dir);
      free(parmCopy);
      if (offsetType == -1) {
	    snprintf(message,  msglen, InvArgErrorMessage, "CD Play");    
    	    goto BadLuck;
      }	
    }

    success = cd_play_parsed(cdrom_fd, offsetType, value, dir);
      
    if (success == False)
	success = cd_play_parsed(cdrom_fd, 2, 0, 1);        // Play from disc start

    if (success == False && errno != 0) 
	    snprintf(message,  msglen, IoctlErrorMessage, errno);    

BadLuck:  
    release_cdrom_fd(buffer, False);
    return success;
 
}


static Bool cd_pauseresume(BufferData *buffer, char *message, int msglen)
{
    Bool success = False;
    Bool needPause, needResume;
    struct cdrom_subchnl subch;
  
    int cdrom_fd = get_cdrom_fd(buffer);

    if (cdrom_fd < 0) {
	snprintf(message, msglen, OpenErrorMessage, errno);
        return False;
    }

    needPause = needResume = False;
    subch.cdsc_format = CDROM_LBA;

    if (ioctl(cdrom_fd, CDROMSUBCHNL, &subch) != -1) {
        switch(subch.cdsc_audiostatus) {
	    case CDROM_AUDIO_PAUSED:
		needResume = True;
	        break;
		  
	    case CDROM_AUDIO_PLAY:
		needPause = True;
		break;

        }  // switch (...)
	
	if (needPause) 
    	    success = (ioctl(cdrom_fd, CDROMPAUSE) != -1);
	else {
    	    if (needResume)      
        	success = (ioctl(cdrom_fd, CDROMRESUME) != -1);

	    if (success == False)
    	        success = cd_play_parsed(cdrom_fd, 1, 0, 1);	    // Play from current

    	    if (success == False)
        	success = cd_play_parsed(cdrom_fd, 2, 0, 1);        // Play from disc start
        }
	
    } // (if (cdrom_fd >= 0)

    
    if (success == False && errno != 0) 
    	snprintf(message, msglen, IoctlErrorMessage, errno);

    release_cdrom_fd(buffer, False);
    return success;
}

//----------------------------------------------------------
static Bool cd_slot(BufferData *buffer, const char *parm, char *message, int msglen)
{
  int val= 0, crnt;
  short dir = 0;
  Bool success = False;
  int cdrom_fd;    

  if (parm != NULL) {    
     char *parmCopy = strdup(parm);     
     success = parse_parm_plusminus(parmCopy, &val, &dir, NULL);
     free(parmCopy);
     if (success == False) {
	snprintf(message, msglen,  InvArgErrorMessage, "CD Slot");
        return False;
     }
  }     
  if (dir == 2) dir = 0;

  cdrom_fd = get_cdrom_fd(buffer);
  if (cdrom_fd < 0) {
	snprintf(message, msglen,  OpenErrorMessage, errno);
        return False;
  }

  crnt = ioctl(cdrom_fd, CDROM_SELECT_DISC, CDSL_CURRENT);
  if (crnt >= 0) {
	if (dir != 0) {
	  if (dir < 0) val = -val;
	  val += crnt;
	}

	if (val < 0) val = 0;
	else {  
	  int maxval = ioctl(cdrom_fd, CDROM_CHANGER_NSLOTS);
	  if (val >= maxval) val = maxval-1;
	}
  
	success = (ioctl(cdrom_fd, CDROM_SELECT_DISC, val) != -1);
  }	

 if (success == False && errno != 0) 
	snprintf(message, msglen,  IoctlErrorMessage, errno);

  release_cdrom_fd(buffer, False);
  return success;
	
}

static Bool cd_justioctl(BufferData *buffer, int request, void *parm, char *message, int msglen)
{
    int cdrom_fd = get_cdrom_fd(buffer);
    Bool success = False;
    
    if (cdrom_fd < 0) {
    	snprintf(message, msglen, OpenErrorMessage, errno);
        return False;
    }

    if (ioctl(cdrom_fd, request, parm) == -1) 
    	snprintf(message, msglen, IoctlErrorMessage, errno);
    else
	success = True;

    release_cdrom_fd(buffer, False);
     
    return success;
}



static Bool cd_eject(BufferData *buffer, char *message, int msglen)
{
    Bool success = False;
    int cdrom_fd = get_cdrom_fd(buffer);

    if (cdrom_fd < 0) {
	snprintf(message, msglen, OpenErrorMessage, errno);
        return False;
    }

    ioctl(cdrom_fd, CDROM_LOCKDOOR, 0);
    success = (ioctl(cdrom_fd, CDROMEJECT) != -1);
    release_cdrom_fd(buffer, True);

    return success;
}

static Bool cd_loadeject(BufferData *buffer, char *message, int msglen)
{

    Bool success = False;
    int cdrom_fd = get_cdrom_fd(buffer);
    int res;
      
    if (cdrom_fd < 0) {
	snprintf(message, msglen, OpenErrorMessage, errno);
        return False;
    }

    res = ioctl(cdrom_fd, CDROM_DRIVE_STATUS, CDSL_CURRENT);

    if (res == CDS_DISC_OK || res == CDS_NO_INFO) {
        ioctl(cdrom_fd, CDROM_LOCKDOOR, 0);
        success = (ioctl(cdrom_fd, CDROMEJECT) != -1);
        release_cdrom_fd(buffer, True);
    }	
    else  {
        success = (ioctl(cdrom_fd, CDROMCLOSETRAY) != -1);
        release_cdrom_fd(buffer, False);
    }	

    if (success == False && errno != 0) 
	snprintf(message, msglen,  IoctlErrorMessage, errno);

    release_cdrom_fd(buffer, True);
    return success;
}

//----------------------------------------------------------
static Bool cd_get_volumes(int cdrom_fd, int *volLeft, int *volRight)
{
    struct cdrom_volctrl rawvols;
    Bool success =(ioctl(cdrom_fd, CDROMVOLREAD, &rawvols) != -1);
    if (success) {
	*volLeft  =  rawvols.channel0;
	*volRight =  rawvols.channel1;
    }	

    return success;	
}

static Bool cd_set_volumes(int cdrom_fd, int volLeft, int volRight)
{
    struct cdrom_volctrl rawvols;


    if (volLeft < CD_MIN_VOLUME) volLeft=CD_MIN_VOLUME;
    else
    if (volLeft > CD_MAX_VOLUME) volLeft=CD_MAX_VOLUME;
  
    if (volRight < CD_MIN_VOLUME) volRight=CD_MIN_VOLUME;
    else
    if (volRight > CD_MAX_VOLUME) volRight=CD_MAX_VOLUME;
    
    rawvols.channel0 = (__u8) volLeft;
    rawvols.channel1 = (__u8) volRight;

    return (ioctl(cdrom_fd, CDROMVOLCTRL, &rawvols) != -1);
}

void left_right_to_vol_balance(int volLeft, int volRight, int *volPtr, int *balPtr)
{
    int volume, balance;    

    if (volRight >= volLeft) {
	// Non-negative
	volume = volRight;
	balance = volRight - volLeft;
	// If balance > 0, then volRight is guaranteed to be positive
	if (balance > 0) balance = (balance * 50) / volRight;
    } else {
	// Negative balance (volLeft > volRight, therefore
	// volLeft is guaranteed to be positive
	volume = volLeft;
	balance = (volRight - volLeft) * 50 / volLeft; 
    } 

    *volPtr = volume * 100 / CD_MAX_VOLUME;
    *balPtr = balance;
}


void vol_balance_to_left_right(int volume, int balance, int *volLeftPtr, int *volRightPtr)
{
    int left, right;

    if (balance == 0)
	left = right = volume;
    else 
    if (balance > 0) {
	right = volume;
        left = volume - (volume * balance) / 50;    
    } else {
	left = volume;
	right = volume + (volume * balance) / 50;     
    }	

    *volLeftPtr = left * CD_MAX_VOLUME / 100;
    *volRightPtr = right * CD_MAX_VOLUME / 100;
}

static Bool cd_show_volumes(int cdrom_fd, char *message, int msglen)
{
    int volLeft, volRight;
    int volume, balance;

    if (cd_get_volumes(cdrom_fd, &volLeft, &volRight) == False)
	return False;

    left_right_to_vol_balance(volLeft, volRight, &volume, &balance);
	
    snprintf (message, msglen, 
              "CD Intern: volume %d  balance %d", volume, balance);
    return True;	      
}

static Bool cd_volume(BufferData *buffer, const char *parm, int behaviour, char *message, int msglen)
{
    Bool success = False;
    int volLeftOld, volRightOld;	
    int volLeft, volRight, value;
    int volume, balance;
    short dir;
    int cdrom_fd;
    Bool osd = (behaviour == XHK_BEHAVIOUR_ONCE) ? buffer->osd : True;
    

    // A repetative call only updates values
    if (parm == NULL || *parm == '\0' || behaviour == XHK_BEHAVIOUR_REPEATED) {
	value = 0;
	dir = 1;
	success = True;
    } else {	
        char *parmCopy = strdup(parm);
	if (parmCopy != NULL) {
    	    success = parse_parm_plusminus(parmCopy, &value, &dir, NULL);
	    free(parmCopy);
        }
    }	
    	
    if (success == False) {
	snprintf(message, msglen,  InvArgErrorMessage, "CD Volume");
        return False;
    }
/*
    if (behaviour == XHK_BEHAVIOUR_CONTINUOUS &&	
         (value != 0 || dir == 0 || dir == 2)) {
        snprintf(message, msglen, CantContinueErrorMessage, "CD Volume");
	return False;
    }	 
*/
    cdrom_fd = get_cdrom_fd(buffer);
    if (cdrom_fd < 0) {
	snprintf(message, msglen, OpenErrorMessage, errno);
        return False;
    }

    if (cd_get_volumes(cdrom_fd, &volLeftOld, &volRightOld) == False) {
        snprintf(message, msglen, IoctlErrorMessage, errno);
	success = False;
	goto BadLuck;
    }

    left_right_to_vol_balance(volLeftOld, volRightOld, &volume, &balance);


    if (dir == 2 || dir == 0) {
	// Absolute volume
	if (value < 0 || value > 100) {
	    snprintf(message, msglen,  InvArgErrorMessage, "CD Volume");
	    goto BadLuck;
	}
	volume = value;    
        vol_balance_to_left_right(volume, balance, &volLeft, &volRight);
	
    } else 
	// Relative zero - no change
    if (value == 0) {
	osd = True;	// Force OSD
	goto JustShowIt;
    } else {
	// Relative nonzero - avoid dead lock
        int iter = 3;
	Bool needBreak = False;


        if (dir < 0) value = -value;

        do {
	    volume += value;
	    
	    if (volume < 0) {
		volume = 0;
		if (value < 0) needBreak = True;
	    } 
	    else
	    if (volume > 100) {
		volume = 100;
		if (value > 0) needBreak = True;
	    }	 
		
	    vol_balance_to_left_right(volume, balance, &volLeft, &volRight);
		
	} while (volLeft == volLeftOld && volRight == volRightOld &&
	         needBreak == False && --iter > 0);
	
    }
    
    success = cd_set_volumes(cdrom_fd,  volLeft, volRight);
    
JustShowIt:
    if (success && osd)  
	cd_show_volumes(cdrom_fd, message, msglen);

BadLuck:
    release_cdrom_fd(buffer, False);
    return success;

}

static Bool cd_balance(BufferData *buffer, const char *parm, int behaviour, char *message, int msglen)
{
    Bool success = False;
    int volLeftOld, volRightOld;
    int volLeft, volRight, value;
    int volume, balance;
    short dir;
    int cdrom_fd;
    Bool osd = (behaviour == XHK_BEHAVIOUR_ONCE) ? buffer->osd : True;

    if (parm == NULL || *parm == '\0' || behaviour == XHK_BEHAVIOUR_REPEATED) {
	value = 0;
	dir = 1;
	success = True;
    } else {	
        char *parmCopy = strdup(parm);
	if (parmCopy != NULL) {
    	    success = parse_parm_plusminus(parmCopy, &value, &dir, NULL);
	    free(parmCopy);
        }
    }	
    	
    if (success == False) {
	snprintf(message, msglen,  InvArgErrorMessage, "CD Balance");
        return False;
    }
/*
    if (behaviour == XHK_BEHAVIOUR_CONTINUOUS &&	
         (value != 0 || dir == 0 || dir == 2)) {
        snprintf(message, msglen, CantContinueErrorMessage, "CD Balance");
	return False;    
    }	 
*/
    cdrom_fd = get_cdrom_fd(buffer);
    if (cdrom_fd < 0) {
	snprintf(message, msglen, OpenErrorMessage, errno);
        return False;
    }

    if (cd_get_volumes(cdrom_fd, &volLeftOld, &volRightOld) == False) {
        snprintf(message, msglen, IoctlErrorMessage, errno);
	success = False;
	goto BadLuck;
    }

    left_right_to_vol_balance(volLeftOld, volRightOld, &volume, &balance);

    if (dir == 2 || dir == 0) {
	// Absolute volume
	if (value < -50 || value > 50) {
	    snprintf(message, msglen,  InvArgErrorMessage, "CD Balance");
	    goto BadLuck;
	}
	balance = value;    
        vol_balance_to_left_right(volume, balance, &volLeft, &volRight);
    } else 
	// Relative zero - no change
    if (value == 0) {
	osd = True;	// Force OSD
	goto JustShowIt;
    } else {
	// Relative nonzero - avoid dead lock
        int iter = 3;
	Bool needBreak = False;
        if (dir < 0) value = -value;

        do {
	    balance += value;
	    
	    if (balance < -50) {
		balance = -50;
		if (value < 0) needBreak = True;
	    }	
	    else
	    if (balance > 50) {
		balance = 50;
		if (value > 0) needBreak = True;
	    }	
		
	    vol_balance_to_left_right(volume, balance, &volLeft, &volRight);

		
	} while (volLeft == volLeftOld && volRight == volRightOld &&
	         needBreak == False && --iter > 0);
    }
    
    success = cd_set_volumes(cdrom_fd,  volLeft, volRight);

JustShowIt:
    if (success && osd)  
	cd_show_volumes(cdrom_fd, message, msglen);

BadLuck:
    release_cdrom_fd(buffer, False);
    return success;

}


//==============================================================
// Exported functions
//==============================================================
Bool init(void **bufferReturn, const char *command, char *message, int msglen)
{
    Bool result;
    BufferData *buffer;
    char CDDevName[256] = "/dev/cdrom";
    Bool grabCD = False;
    Bool osd = True;

    const int  parmAttr[] = {ATTRIB_CHAR_CUSTOM_SEP | sizeof(CDDevName),
                                ATTRIB_BOOL, ATTRIB_BOOL};
    void *parmValPtr[] = {CDDevName, &grabCD, &osd};
    
    //printf("Init %s\n", command);
    
    result = True;
    if (command != NULL && *command != '\0')
	result = lookup_parms(command, parmCount,
	     parmNames, parmAttr, parmValPtr, message, msglen);


    *bufferReturn = NULL;
    if (result == False) return False;
    
    /*	 
    	printf ("CDDevName = '%s' grabCD = %d, osd = %d\n",
	    CDDevName, grabCD, osd);
    */    
    
    buffer = (BufferData *) malloc(sizeof(BufferData));
    if (buffer == NULL) return False;

    *bufferReturn = (void *) buffer;
    strcpy(buffer->CDDevName,  CDDevName);   
    buffer->grabCD = grabCD;
    buffer->osd = osd;
    buffer->cdrom_fd = -1;

    return result;
}

int getopcount(void)
{
    return CDAUDIO_CODE_COUNT;

}

Bool getopdata(int code, char opname[XHKEYS_PLUGIN_OPNAME_LENGTH], int *flags)
{
    if (code < 0 || code >= CDAUDIO_CODE_COUNT) return False;
    
    if (opname != NULL) {
	strncpy(opname, OpData[code].opname, XHKEYS_PLUGIN_OPNAME_LENGTH);
	opname[XHKEYS_PLUGIN_OPNAME_LENGTH-1] = '\0';
    }	

    if (flags != NULL) *flags = OpData[code].flags;
    return True;
}   

Bool process(void *bufferVoidPtr, int code, const char *parm, int behaviour,
		 char *message, int msglen)
{
    Bool success = False;
    Bool force_osd = False;
    BufferData *buffer = (BufferData *) bufferVoidPtr;
//    int retCode = XHKEYS_PLUGPROCRC_ERROR;

    *message = '\0';

    if (behaviour == XHK_BEHAVIOUR_DISCONTINUE) return 0;

    switch(code) {
	  case CDAUDIO_PLAYPAUSE:
	  case CDAUDIO_PLAY:
		if (behaviour == XHK_BEHAVIOUR_REPEATED) 
		    force_osd = success = True;
		else {
		    success = (code == CDAUDIO_PLAYPAUSE) ?
		               cd_pauseresume(buffer, message, msglen) :
			       cd_play_parm(buffer, parm, message, msglen);
	    	    force_osd = (behaviour == XHK_BEHAVIOUR_CONTINUOUS);
		}    
		break;

	  case CDAUDIO_STATUS:
		// We just need status
		force_osd = success = True;
		break;

	  case CDAUDIO_STOP:
//		if (behaviour != XHK_BEHAVIOUR_ONCE) return retCode;
		if (behaviour == XHK_BEHAVIOUR_ONCE) 
		    success = cd_justioctl(buffer, CDROMSTOP, 0, message, msglen);
		break;


	  case CDAUDIO_EJECT:
//		if (behaviour != XHK_BEHAVIOUR_ONCE) return retCode;
		if (behaviour == XHK_BEHAVIOUR_ONCE)
		    success = cd_eject(buffer, message, msglen);
		break;

	  case CDAUDIO_LOADEJECT:
//		if (behaviour != XHK_BEHAVIOUR_ONCE) return retCode;
		if (behaviour == XHK_BEHAVIOUR_ONCE)
		    success = cd_loadeject(buffer, message, msglen);
		break;

	  case CDAUDIO_LOAD:
//		if (behaviour != XHK_BEHAVIOUR_ONCE) return retCode;
		if (behaviour == XHK_BEHAVIOUR_ONCE)
		    success = cd_justioctl(buffer, CDROMCLOSETRAY, 0, message, msglen);
		break;

	  case CDAUDIO_SLOT:
//		if (behaviour != XHK_BEHAVIOUR_ONCE) return retCode;
		if (behaviour == XHK_BEHAVIOUR_ONCE)
		    success = cd_slot(buffer, parm, message, msglen);
		break;

	  case CDAUDIO_VOLUME:
		// Avoid standard OSD
		return cd_volume(buffer, parm, behaviour, message, msglen);

	  case CDAUDIO_BALANCE:
		// Avoid standard OSD
		return cd_balance(buffer, parm, behaviour, message, msglen);
		
  }

   if (success) {
//	retCode = 0;
	
	if (buffer->osd || force_osd) {
	    usleep(200000);
	    cd_get_status_text(buffer, message, msglen);
//	    retCode = XHKEYS_PLUGPROCRC_OSDTEXT;
	}    
    } 

//    return retCode;
    return success;
  
}


void term(void *buffer)
{

    int cdrom_fd = ((BufferData *)buffer)->cdrom_fd;
    if (cdrom_fd >= 0) close(cdrom_fd);
}




Bool property(void *buffer, XHKEYS_PLUGIN_PROPERTY code, ...)
{
    Bool success = True;
    int  intVal;
    int  *intPtr;
    char *charPtr;
    
    va_list arg_list;
    
    va_start(arg_list, code);
    
    switch(code)
    {
	case XHK_PLUGIN_VERSION:
	    if ((intPtr = va_arg(arg_list, int *)) != NULL) {
		*intPtr = XHKEYS_PLUGIN_VERSION;
		success = True;
	    }
	    break;
    
	case XHK_PLUGIN_DESCRIPTION:
	    if ((charPtr = va_arg(arg_list, char *)) != NULL &&
		(intVal = va_arg(arg_list, int)) > 0) {
	        strncpy(charPtr, Description, intVal);
		success = True;
	    }	
	    // Break is not necessary
	    
	default:
	    break;    
    }

    va_end(arg_list);
    
    return success;
}

#endif
