//
// Syd: rock-solid application kernel
// src/cookie.rs: Syscall argument cookies
//
// Copyright (c) 2025 Ali Polatel <alip@chesswob.org>
//
// SPDX-License-Identifier: GPL-3.0

// We allow this to easily write portable code.
// FIXME: Do not be lazy.
#![allow(dead_code)]

use std::os::fd::{AsFd, AsRawFd, FromRawFd, OwnedFd, RawFd};

use libseccomp::ScmpSyscall;
use nix::{
    errno::Errno,
    fcntl::{AtFlags, OpenHow},
    unistd::UnlinkatFlags,
    NixPath,
};
use once_cell::sync::Lazy;

use crate::{compat::RenameFlags, fs::fillrandom, path::XPath};

/// A platform-sized secure cookie
///
/// 32 bits on 32-bit, 64 bits on 64-bit targets.
#[cfg(target_pointer_width = "32")]
pub(crate) type Cookie = u32;
#[cfg(target_pointer_width = "64")]
pub(crate) type Cookie = u64;

/// Generate a random `Cookie` using the OS's secure RNG.
///
/// This uses `syd::fs::fillrandom` under the hood to pull in
/// exactly the number of bytes needed for `Cookie`,
/// interprets them in little-endian order, and returns the result.
pub(crate) fn getcookie() -> Result<Cookie, Errno> {
    #[cfg(target_pointer_width = "32")]
    {
        const N: usize = 4;
        let mut buf = [0u8; N];
        fillrandom(&mut buf)?;
        Ok(Cookie::from_le_bytes(buf))
    }

    #[cfg(target_pointer_width = "64")]
    {
        const N: usize = 8;
        let mut buf = [0u8; N];
        fillrandom(&mut buf)?;
        Ok(Cookie::from_le_bytes(buf))
    }
}

// These cookies are confined by seccomp for use with openat2(2).
#[allow(clippy::disallowed_methods)]
pub(crate) static OPENAT2_COOKIE_ARG4: Lazy<Cookie> = Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static OPENAT2_COOKIE_ARG5: Lazy<Cookie> = Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static SOCKET_COOKIE_ARG3: Lazy<Cookie> = Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static SOCKET_COOKIE_ARG4: Lazy<Cookie> = Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static SOCKET_COOKIE_ARG5: Lazy<Cookie> = Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static MEMFD_CREATE_COOKIE_ARG2: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static MEMFD_CREATE_COOKIE_ARG3: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static MEMFD_CREATE_COOKIE_ARG4: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static MEMFD_CREATE_COOKIE_ARG5: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static RENAMEAT2_COOKIE_ARG5: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static TRUNCATE_COOKIE_ARG2: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static TRUNCATE_COOKIE_ARG3: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static TRUNCATE_COOKIE_ARG4: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static TRUNCATE_COOKIE_ARG5: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static TRUNCATE64_COOKIE_ARG3: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static TRUNCATE64_COOKIE_ARG4: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static TRUNCATE64_COOKIE_ARG5: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static FTRUNCATE_COOKIE_ARG2: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static FTRUNCATE_COOKIE_ARG3: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static FTRUNCATE_COOKIE_ARG4: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static FTRUNCATE_COOKIE_ARG5: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static FTRUNCATE64_COOKIE_ARG3: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static FTRUNCATE64_COOKIE_ARG4: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static FTRUNCATE64_COOKIE_ARG5: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static UNLINKAT_COOKIE_ARG3: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static UNLINKAT_COOKIE_ARG4: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static UNLINKAT_COOKIE_ARG5: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static LINKAT_COOKIE_ARG5: Lazy<Cookie> = Lazy::new(|| getcookie().expect("getcookie"));

/// These are used in `syd::fs::seccomp_notify_addfd`.
#[allow(clippy::disallowed_methods)]
pub(crate) static SECCOMP_IOCTL_NOTIF_ADDFD_COOKIE_ARG3: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static SECCOMP_IOCTL_NOTIF_ADDFD_COOKIE_ARG4: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static SECCOMP_IOCTL_NOTIF_ADDFD_COOKIE_ARG5: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));

/// These are used in `syd::fs::seccomp_notify_respond`.
#[allow(clippy::disallowed_methods)]
pub(crate) static SECCOMP_IOCTL_NOTIF_SEND_COOKIE_ARG3: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static SECCOMP_IOCTL_NOTIF_SEND_COOKIE_ARG4: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static SECCOMP_IOCTL_NOTIF_SEND_COOKIE_ARG5: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));

/// These are used in `syd::proc::procmap_query`.
#[allow(clippy::disallowed_methods)]
pub(crate) static PROCMAP_QUERY_COOKIE_ARG3: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static PROCMAP_QUERY_COOKIE_ARG4: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));
#[allow(clippy::disallowed_methods)]
pub(crate) static PROCMAP_QUERY_COOKIE_ARG5: Lazy<Cookie> =
    Lazy::new(|| getcookie().expect("getcookie"));

/// Safe openat2 confined by syscall cookies.
#[inline(always)]
pub(crate) fn safe_openat2<Fd: AsFd>(
    dirfd: Fd,
    path: &XPath,
    mut how: OpenHow,
) -> Result<OwnedFd, Errno> {
    // SAFETY: In libc we trust.
    #[allow(clippy::cast_possible_truncation)]
    let fd = path.with_nix_path(|cstr| unsafe {
        libc::syscall(
            libc::SYS_openat2,
            dirfd.as_fd().as_raw_fd(),
            cstr.as_ptr(),
            std::ptr::addr_of_mut!(how),
            std::mem::size_of::<libc::open_how>(),
            *OPENAT2_COOKIE_ARG4,
            *OPENAT2_COOKIE_ARG5,
        )
    })? as RawFd;

    Errno::result(fd)?;

    // SAFETY:
    //
    // `openat2(2)` should return a valid owned fd on success
    Ok(unsafe { OwnedFd::from_raw_fd(fd) })
}

/// socket(2) may be multiplexed by socketcall(2).
pub(crate) static SYS_SOCKET: Lazy<Option<libc::c_long>> = Lazy::new(|| {
    match ScmpSyscall::from_name("socket")
        .map(i32::from)
        .map(libc::c_long::from)
        .ok()
    {
        Some(n) if n < 0 => None,
        Some(n) => Some(n),
        None => None,
    }
});

/// Safe socket confined by syscall cookies.
#[inline(always)]
pub(crate) fn safe_socket(
    domain: libc::c_int,
    stype: libc::c_int,
    proto: libc::c_int,
) -> Result<OwnedFd, Errno> {
    if let Some(sys_socket) = *SYS_SOCKET {
        // SAFETY: In libc we trust.
        #[allow(clippy::cast_possible_truncation)]
        Errno::result(unsafe {
            libc::syscall(
                sys_socket,
                domain,
                stype,
                proto,
                *SOCKET_COOKIE_ARG3,
                *SOCKET_COOKIE_ARG4,
                *SOCKET_COOKIE_ARG5,
            )
        })
        .map(|fd| fd as RawFd)
    } else {
        // SAFETY:
        // socketcall(2) on multiplexed architecture.
        // We use libc version for convenience.
        Errno::result(unsafe { libc::socket(domain, stype, proto) })
    }
    .map(|fd| {
        // SAFETY: socket returns a valid FD on success.
        unsafe { OwnedFd::from_raw_fd(fd) }
    })
}

/// Safe memfd_create confined by syscall cookies.
#[inline(always)]
pub(crate) fn safe_memfd_create(name: &[u8], flags: libc::c_uint) -> Result<OwnedFd, Errno> {
    // SAFETY: In libc we trust.
    #[allow(clippy::cast_possible_truncation)]
    let fd = Errno::result(unsafe {
        libc::syscall(
            libc::SYS_memfd_create,
            name.as_ptr(),
            flags,
            *MEMFD_CREATE_COOKIE_ARG2,
            *MEMFD_CREATE_COOKIE_ARG3,
            *MEMFD_CREATE_COOKIE_ARG4,
            *MEMFD_CREATE_COOKIE_ARG5,
        )
    })? as RawFd;

    // SAFETY:
    //
    // `memfd_create(2)` should return a valid owned fd on success
    Ok(unsafe { OwnedFd::from_raw_fd(fd) })
}

/// Safe renameat2(2) confined by syscall cookies.
#[inline(always)]
pub(crate) fn safe_renameat2<Fd1: AsFd, Fd2: AsFd>(
    old_dirfd: Fd1,
    old_path: &XPath,
    new_dirfd: Fd2,
    new_path: &XPath,
    flags: RenameFlags,
) -> Result<(), Errno> {
    let res = old_path.with_nix_path(|old_cstr| {
        // SAFETY: In libc we trust.
        new_path.with_nix_path(|new_cstr| unsafe {
            libc::syscall(
                libc::SYS_renameat2,
                old_dirfd.as_fd().as_raw_fd(),
                old_cstr.as_ptr(),
                new_dirfd.as_fd().as_raw_fd(),
                new_cstr.as_ptr(),
                flags.bits(),
                *RENAMEAT2_COOKIE_ARG5,
            )
        })
    })??;

    Errno::result(res).map(drop)
}

/// truncate(2) may be aliased to truncate64(2) by libc.
static SYS_TRUNCATE: Lazy<Option<libc::c_long>> = Lazy::new(|| {
    match ScmpSyscall::from_name("truncate")
        .map(i32::from)
        .map(libc::c_long::from)
        .ok()
    {
        Some(n) if n < 0 => None,
        Some(n) => Some(n),
        None => None,
    }
});

/// truncate64(2) may not always be available via libc.
static SYS_TRUNCATE64: Lazy<Option<libc::c_long>> = Lazy::new(|| {
    match ScmpSyscall::from_name("truncate64")
        .map(i32::from)
        .map(libc::c_long::from)
        .ok()
    {
        Some(n) if n < 0 => None,
        Some(n) => Some(n),
        None => None,
    }
});

/// ftruncate(2) may be aliased to ftruncate64(2) by libc.
static SYS_FTRUNCATE: Lazy<Option<libc::c_long>> = Lazy::new(|| {
    match ScmpSyscall::from_name("ftruncate")
        .map(i32::from)
        .map(libc::c_long::from)
        .ok()
    {
        Some(n) if n < 0 => None,
        Some(n) => Some(n),
        None => None,
    }
});

/// ftruncate64(2) may not always be available via libc.
static SYS_FTRUNCATE64: Lazy<Option<libc::c_long>> = Lazy::new(|| {
    match ScmpSyscall::from_name("ftruncate64")
        .map(i32::from)
        .map(libc::c_long::from)
        .ok()
    {
        Some(n) if n < 0 => None,
        Some(n) => Some(n),
        None => None,
    }
});

/// Safe truncate(2) confined by syscall cookies.
pub(crate) fn safe_truncate(path: &XPath, len: libc::off_t) -> Result<(), Errno> {
    let sys_truncate = SYS_TRUNCATE.ok_or(Errno::ENOSYS)?;

    // SAFETY: In libc we trust.
    let res = path.with_nix_path(|cstr| unsafe {
        libc::syscall(
            sys_truncate,
            cstr.as_ptr(),
            len,
            *TRUNCATE_COOKIE_ARG2,
            *TRUNCATE_COOKIE_ARG3,
            *TRUNCATE_COOKIE_ARG4,
            *TRUNCATE_COOKIE_ARG5,
        )
    })?;

    Errno::result(res).map(drop)
}

/// Safe truncate64(2) confined by syscall cookies.
pub(crate) fn safe_truncate64(path: &XPath, len: libc::off64_t) -> Result<(), Errno> {
    #[cfg(not(any(
        target_pointer_width = "64",
        all(target_arch = "x86_64", target_pointer_width = "32"),
        target_arch = "x86",
        target_arch = "arm",
        target_arch = "powerpc",
        target_arch = "m68k",
        target_arch = "mips",
        target_arch = "mips32r6",
    )))]
    {
        compile_error!("BUG: safe_truncate64 is not implemented for this architecture!");
    }

    #[cfg(any(
        target_pointer_width = "64",
        all(target_arch = "x86_64", target_pointer_width = "32"),
    ))]
    {
        safe_truncate(path, len)
    }

    #[cfg(any(target_arch = "m68k", target_arch = "x86",))]
    {
        let sys_truncate64 = SYS_TRUNCATE64.ok_or(Errno::ENOSYS)?;

        // i386: low, high
        let val = len as u64;
        let low = (val & 0xFFFF_FFFF) as libc::c_long;
        let high = (val >> 32) as libc::c_long;

        // SAFETY: In libc we trust.
        Errno::result(path.with_nix_path(|cstr| unsafe {
            libc::syscall(
                sys_truncate64,
                cstr.as_ptr(),
                low,
                high,
                *TRUNCATE64_COOKIE_ARG3,
                *TRUNCATE64_COOKIE_ARG4,
                *TRUNCATE64_COOKIE_ARG5,
            )
        })?)
        .map(drop)
    }

    #[cfg(any(
        target_arch = "arm",
        target_arch = "powerpc",
        target_arch = "mips",
        target_arch = "mips32r6"
    ))]
    {
        let sys_truncate64 = SYS_TRUNCATE64.ok_or(Errno::ENOSYS)?;

        // 32-bit ARM/ppc/mips: 0, low, high
        let val = len as u64;
        let low = (val & 0xFFFF_FFFF) as libc::c_long;
        let high = (val >> 32) as libc::c_long;

        // SAFETY: In libc we trust.
        Errno::result(path.with_nix_path(|cstr| unsafe {
            libc::syscall(
                sys_truncate64,
                cstr.as_ptr(),
                0 as libc::c_long,
                low,
                high,
                *TRUNCATE64_COOKIE_ARG4,
                *TRUNCATE64_COOKIE_ARG5,
            )
        })?)
        .map(drop)
    }
}

/// Safe ftruncate(2) confined by syscall cookies.
pub(crate) fn safe_ftruncate<Fd: AsFd>(fd: Fd, len: libc::off_t) -> Result<(), Errno> {
    let sys_ftruncate = SYS_FTRUNCATE.ok_or(Errno::ENOSYS)?;

    // SAFETY: In libc we trust.
    Errno::result(unsafe {
        libc::syscall(
            sys_ftruncate,
            fd.as_fd().as_raw_fd(),
            len,
            *FTRUNCATE_COOKIE_ARG2,
            *FTRUNCATE_COOKIE_ARG3,
            *FTRUNCATE_COOKIE_ARG4,
            *FTRUNCATE_COOKIE_ARG5,
        )
    })
    .map(drop)
}

/// Safe ftruncate64(2) confined by syscall cookies.
pub(crate) fn safe_ftruncate64<Fd: AsFd>(fd: Fd, len: libc::off64_t) -> Result<(), Errno> {
    #[cfg(not(any(
        target_pointer_width = "64",
        all(target_arch = "x86_64", target_pointer_width = "32"),
        target_arch = "x86",
        target_arch = "arm",
        target_arch = "powerpc",
        target_arch = "m68k",
        target_arch = "mips",
        target_arch = "mips32r6",
    )))]
    {
        compile_error!("BUG: safe_ftruncate64 is not implemented for this architecture!");
    }

    #[cfg(any(
        target_pointer_width = "64",
        all(target_arch = "x86_64", target_pointer_width = "32"),
    ))]
    {
        safe_ftruncate(fd, len)
    }

    #[cfg(any(target_arch = "m68k", target_arch = "x86",))]
    {
        let sys_ftruncate64 = SYS_FTRUNCATE64.ok_or(Errno::ENOSYS)?;

        // i386: low, high
        let val = len as u64;
        let low = (val & 0xFFFF_FFFF) as libc::c_long;
        let high = (val >> 32) as libc::c_long;

        // SAFETY: In libc we trust.
        Errno::result(unsafe {
            libc::syscall(
                sys_ftruncate64,
                fd.as_fd().as_raw_fd(),
                low,
                high,
                *FTRUNCATE64_COOKIE_ARG3,
                *FTRUNCATE64_COOKIE_ARG4,
                *FTRUNCATE64_COOKIE_ARG5,
            )
        })
        .map(drop)
    }

    #[cfg(any(
        target_arch = "arm",
        target_arch = "powerpc",
        target_arch = "mips",
        target_arch = "mips32r6"
    ))]
    {
        let sys_ftruncate64 = SYS_FTRUNCATE64.ok_or(Errno::ENOSYS)?;

        // 32-bit ARM/ppc/mips: 0, low, high
        let val = len as u64;
        let low = (val & 0xFFFF_FFFF) as libc::c_long;
        let high = (val >> 32) as libc::c_long;

        // SAFETY: In libc we trust.
        Errno::result(unsafe {
            libc::syscall(
                sys_ftruncate64,
                fd.as_fd().as_raw_fd(),
                0 as libc::c_long,
                low,
                high,
                *FTRUNCATE64_COOKIE_ARG4,
                *FTRUNCATE64_COOKIE_ARG5,
            )
        })
        .map(drop)
    }
}

/// Safe unlinkat(2) confined by syscall cookies.
#[inline(always)]
pub(crate) fn safe_unlinkat<Fd: AsFd>(
    dirfd: Fd,
    path: &XPath,
    flag: UnlinkatFlags,
) -> Result<(), Errno> {
    let atflag = match flag {
        UnlinkatFlags::RemoveDir => AtFlags::AT_REMOVEDIR,
        UnlinkatFlags::NoRemoveDir => AtFlags::empty(),
    };

    // SAFETY: In libc we trust.
    let res = path.with_nix_path(|cstr| unsafe {
        libc::syscall(
            libc::SYS_unlinkat,
            dirfd.as_fd().as_raw_fd(),
            cstr.as_ptr(),
            atflag.bits(),
            *UNLINKAT_COOKIE_ARG3,
            *UNLINKAT_COOKIE_ARG4,
            *UNLINKAT_COOKIE_ARG5,
        )
    })?;

    Errno::result(res).map(drop)
}

/// Safe linkat(2) confined by syscall cookies.
#[inline(always)]
pub(crate) fn safe_linkat<Fd1: AsFd, Fd2: AsFd>(
    olddirfd: Fd1,
    oldpath: &XPath,
    newdirfd: Fd2,
    newpath: &XPath,
    flag: AtFlags,
) -> Result<(), Errno> {
    let res = oldpath.with_nix_path(|oldcstr| {
        // SAFETY: In libc we trust.
        newpath.with_nix_path(|newcstr| unsafe {
            libc::syscall(
                libc::SYS_linkat,
                olddirfd.as_fd().as_raw_fd(),
                oldcstr.as_ptr(),
                newdirfd.as_fd().as_raw_fd(),
                newcstr.as_ptr(),
                flag.bits(),
                *LINKAT_COOKIE_ARG5,
            )
        })
    })??;

    Errno::result(res).map(drop)
}
