/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.action.datastreams.autosharding;

import java.util.List;
import java.util.Objects;
import java.util.OptionalDouble;
import java.util.function.Function;
import java.util.function.LongSupplier;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.elasticsearch.action.datastreams.autosharding.AutoShardingResult;
import org.elasticsearch.action.datastreams.autosharding.AutoShardingType;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.metadata.DataStream;
import org.elasticsearch.cluster.metadata.IndexMetadata;
import org.elasticsearch.cluster.metadata.IndexMetadataStats;
import org.elasticsearch.cluster.metadata.IndexWriteLoad;
import org.elasticsearch.cluster.metadata.Metadata;
import org.elasticsearch.cluster.service.ClusterService;
import org.elasticsearch.common.regex.Regex;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.core.Nullable;
import org.elasticsearch.core.TimeValue;
import org.elasticsearch.features.FeatureService;
import org.elasticsearch.features.NodeFeature;
import org.elasticsearch.index.Index;

public class DataStreamAutoShardingService {
    private static final Logger logger = LogManager.getLogger(DataStreamAutoShardingService.class);
    public static final String DATA_STREAMS_AUTO_SHARDING_ENABLED = "data_streams.auto_sharding.enabled";
    public static final NodeFeature DATA_STREAM_AUTO_SHARDING_FEATURE = new NodeFeature("data_stream.auto_sharding", true);
    public static final Setting<List<String>> DATA_STREAMS_AUTO_SHARDING_EXCLUDES_SETTING = Setting.listSetting("data_streams.auto_sharding.excludes", List.of(), Function.identity(), Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<TimeValue> DATA_STREAMS_AUTO_SHARDING_INCREASE_SHARDS_COOLDOWN = Setting.timeSetting("data_streams.auto_sharding.increase_shards.cooldown", TimeValue.timeValueSeconds((long)270L), TimeValue.timeValueSeconds((long)0L), Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<TimeValue> DATA_STREAMS_AUTO_SHARDING_DECREASE_SHARDS_COOLDOWN = Setting.timeSetting("data_streams.auto_sharding.decrease_shards.cooldown", TimeValue.timeValueDays((long)3L), TimeValue.timeValueSeconds((long)0L), Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<Integer> CLUSTER_AUTO_SHARDING_MIN_WRITE_THREADS = Setting.intSetting("cluster.auto_sharding.min_write_threads", 2, 1, Setting.Property.Dynamic, Setting.Property.NodeScope);
    public static final Setting<Integer> CLUSTER_AUTO_SHARDING_MAX_WRITE_THREADS = Setting.intSetting("cluster.auto_sharding.max_write_threads", 32, 1, Setting.Property.Dynamic, Setting.Property.NodeScope);
    private final ClusterService clusterService;
    private final boolean isAutoShardingEnabled;
    private final FeatureService featureService;
    private final LongSupplier nowSupplier;
    private volatile TimeValue increaseShardsCooldown;
    private volatile TimeValue reduceShardsCooldown;
    private volatile int minWriteThreads;
    private volatile int maxWriteThreads;
    private volatile List<String> dataStreamExcludePatterns;

    public DataStreamAutoShardingService(Settings settings, ClusterService clusterService, FeatureService featureService, LongSupplier nowSupplier) {
        this.clusterService = clusterService;
        this.isAutoShardingEnabled = settings.getAsBoolean(DATA_STREAMS_AUTO_SHARDING_ENABLED, false);
        this.increaseShardsCooldown = DATA_STREAMS_AUTO_SHARDING_INCREASE_SHARDS_COOLDOWN.get(settings);
        this.reduceShardsCooldown = DATA_STREAMS_AUTO_SHARDING_DECREASE_SHARDS_COOLDOWN.get(settings);
        this.minWriteThreads = CLUSTER_AUTO_SHARDING_MIN_WRITE_THREADS.get(settings);
        this.maxWriteThreads = CLUSTER_AUTO_SHARDING_MAX_WRITE_THREADS.get(settings);
        this.dataStreamExcludePatterns = DATA_STREAMS_AUTO_SHARDING_EXCLUDES_SETTING.get(settings);
        this.featureService = featureService;
        this.nowSupplier = nowSupplier;
    }

    public void init() {
        this.clusterService.getClusterSettings().addSettingsUpdateConsumer(DATA_STREAMS_AUTO_SHARDING_INCREASE_SHARDS_COOLDOWN, this::updateIncreaseShardsCooldown);
        this.clusterService.getClusterSettings().addSettingsUpdateConsumer(DATA_STREAMS_AUTO_SHARDING_DECREASE_SHARDS_COOLDOWN, this::updateReduceShardsCooldown);
        this.clusterService.getClusterSettings().addSettingsUpdateConsumer(CLUSTER_AUTO_SHARDING_MIN_WRITE_THREADS, this::updateMinWriteThreads);
        this.clusterService.getClusterSettings().addSettingsUpdateConsumer(CLUSTER_AUTO_SHARDING_MAX_WRITE_THREADS, this::updateMaxWriteThreads);
        this.clusterService.getClusterSettings().addSettingsUpdateConsumer(DATA_STREAMS_AUTO_SHARDING_EXCLUDES_SETTING, this::updateDataStreamExcludePatterns);
    }

    public AutoShardingResult calculate(ClusterState state, DataStream dataStream, @Nullable Double writeIndexLoad) {
        Metadata metadata = state.metadata();
        if (!this.isAutoShardingEnabled) {
            logger.debug("Data stream auto sharding service is not enabled.");
            return AutoShardingResult.NOT_APPLICABLE_RESULT;
        }
        if (!this.featureService.clusterHasFeature(state, DATA_STREAM_AUTO_SHARDING_FEATURE)) {
            logger.debug("Data stream auto sharding service cannot compute the optimal number of shards for data stream [{}] because the cluster doesn't have the auto sharding feature", (Object)dataStream.getName());
            return AutoShardingResult.NOT_APPLICABLE_RESULT;
        }
        if (this.dataStreamExcludePatterns.stream().anyMatch(pattern -> Regex.simpleMatch(pattern, dataStream.getName()))) {
            logger.debug("Data stream [{}] is excluded from auto sharding via the [{}] setting", (Object)dataStream.getName(), (Object)DATA_STREAMS_AUTO_SHARDING_EXCLUDES_SETTING.getKey());
            return AutoShardingResult.NOT_APPLICABLE_RESULT;
        }
        if (writeIndexLoad == null) {
            logger.debug("Data stream auto sharding service cannot compute the optimal number of shards for data stream [{}] as the write index load is not available", (Object)dataStream.getName());
            return AutoShardingResult.NOT_APPLICABLE_RESULT;
        }
        return this.innerCalculate(metadata, dataStream, writeIndexLoad, this.nowSupplier);
    }

    private AutoShardingResult innerCalculate(Metadata metadata, DataStream dataStream, double writeIndexLoad, LongSupplier nowSupplier) {
        IndexMetadata writeIndex = metadata.index(dataStream.getWriteIndex());
        assert (writeIndex != null) : "the data stream write index must exist in the provided cluster metadata";
        AutoShardingResult increaseShardsResult = this.getIncreaseShardsResult(dataStream, writeIndexLoad, nowSupplier, writeIndex);
        return Objects.requireNonNullElseGet(increaseShardsResult, () -> this.getDecreaseShardsResult(metadata, dataStream, writeIndexLoad, nowSupplier, writeIndex, this.getRemainingDecreaseShardsCooldown(metadata, dataStream)));
    }

    @Nullable
    private AutoShardingResult getIncreaseShardsResult(DataStream dataStream, double writeIndexLoad, LongSupplier nowSupplier, IndexMetadata writeIndex) {
        long optimalShardCount = DataStreamAutoShardingService.computeOptimalNumberOfShards(this.minWriteThreads, this.maxWriteThreads, writeIndexLoad);
        if (optimalShardCount > (long)writeIndex.getNumberOfShards()) {
            TimeValue timeSinceLastAutoShardingEvent = dataStream.getAutoShardingEvent() != null ? dataStream.getAutoShardingEvent().getTimeSinceLastAutoShardingEvent(nowSupplier) : TimeValue.MAX_VALUE;
            TimeValue coolDownRemaining = TimeValue.timeValueMillis((long)Math.max(0L, this.increaseShardsCooldown.millis() - timeSinceLastAutoShardingEvent.millis()));
            logger.debug("data stream autosharding service recommends increasing the number of shards from [{}] to [{}] after [{}] cooldown for data stream [{}]", (Object)writeIndex.getNumberOfShards(), (Object)optimalShardCount, (Object)coolDownRemaining, (Object)dataStream.getName());
            return new AutoShardingResult(coolDownRemaining.equals((Object)TimeValue.ZERO) ? AutoShardingType.INCREASE_SHARDS : AutoShardingType.COOLDOWN_PREVENTED_INCREASE, writeIndex.getNumberOfShards(), Math.toIntExact(optimalShardCount), coolDownRemaining, writeIndexLoad);
        }
        return null;
    }

    private TimeValue getRemainingDecreaseShardsCooldown(Metadata metadata, DataStream dataStream) {
        Index oldestBackingIndex = dataStream.getIndices().get(0);
        IndexMetadata oldestIndexMeta = metadata.getIndexSafe(oldestBackingIndex);
        return dataStream.getAutoShardingEvent() == null ? TimeValue.timeValueMillis((long)Math.max(0L, oldestIndexMeta.getCreationDate() + this.reduceShardsCooldown.millis() - this.nowSupplier.getAsLong())) : TimeValue.timeValueMillis((long)Math.max(0L, this.reduceShardsCooldown.millis() - dataStream.getAutoShardingEvent().getTimeSinceLastAutoShardingEvent(this.nowSupplier).millis()));
    }

    private AutoShardingResult getDecreaseShardsResult(Metadata metadata, DataStream dataStream, double writeIndexLoad, LongSupplier nowSupplier, IndexMetadata writeIndex, TimeValue remainingReduceShardsCooldown) {
        double maxIndexLoadWithinCoolingPeriod = DataStreamAutoShardingService.getMaxIndexLoadWithinCoolingPeriod(metadata, dataStream, writeIndexLoad, this.reduceShardsCooldown, nowSupplier);
        logger.trace("calculating the optimal number of shards for a potential decrease in number of shards for data stream [{}] with the max indexing load [{}] over the decrease shards cool down period", (Object)dataStream.getName(), (Object)maxIndexLoadWithinCoolingPeriod);
        long optimalShardCount = DataStreamAutoShardingService.computeOptimalNumberOfShards(this.minWriteThreads, this.maxWriteThreads, maxIndexLoadWithinCoolingPeriod);
        if (optimalShardCount < (long)writeIndex.getNumberOfShards()) {
            logger.debug("data stream autosharding service recommends decreasing the number of shards from [{}] to [{}] after [{}] cooldown for data stream [{}]", (Object)writeIndex.getNumberOfShards(), (Object)optimalShardCount, (Object)remainingReduceShardsCooldown, (Object)dataStream.getName());
            return new AutoShardingResult(remainingReduceShardsCooldown.equals((Object)TimeValue.ZERO) ? AutoShardingType.DECREASE_SHARDS : AutoShardingType.COOLDOWN_PREVENTED_DECREASE, writeIndex.getNumberOfShards(), Math.toIntExact(optimalShardCount), remainingReduceShardsCooldown, maxIndexLoadWithinCoolingPeriod);
        }
        logger.trace("data stream autosharding service recommends maintaining the number of shards [{}] for data stream [{}]", (Object)writeIndex.getNumberOfShards(), (Object)dataStream.getName());
        return new AutoShardingResult(AutoShardingType.NO_CHANGE_REQUIRED, writeIndex.getNumberOfShards(), writeIndex.getNumberOfShards(), TimeValue.ZERO, maxIndexLoadWithinCoolingPeriod);
    }

    static long computeOptimalNumberOfShards(int minNumberWriteThreads, int maxNumberWriteThreads, double indexingLoad) {
        return Math.max(Math.max(Math.min(DataStreamAutoShardingService.roundUp(indexingLoad / ((double)minNumberWriteThreads / 2.0)), 3L), DataStreamAutoShardingService.roundUp(indexingLoad / ((double)maxNumberWriteThreads / 2.0))), 1L);
    }

    private static long roundUp(double value) {
        return (long)Math.ceil(value);
    }

    static double getMaxIndexLoadWithinCoolingPeriod(Metadata metadata, DataStream dataStream, double writeIndexLoad, TimeValue coolingPeriod, LongSupplier nowSupplier) {
        List<IndexWriteLoad> writeLoadsWithinCoolingPeriod = DataStream.getIndicesWithinMaxAgeRange(dataStream, metadata::getIndexSafe, coolingPeriod, nowSupplier).stream().filter(index -> !index.equals(dataStream.getWriteIndex())).map(metadata::index).filter(Objects::nonNull).map(IndexMetadata::getStats).filter(Objects::nonNull).map(IndexMetadataStats::writeLoad).filter(Objects::nonNull).toList();
        double maxIndexLoadWithinCoolingPeriod = writeIndexLoad;
        for (IndexWriteLoad writeLoad : writeLoadsWithinCoolingPeriod) {
            double totalIndexLoad = 0.0;
            for (int shardId = 0; shardId < writeLoad.numberOfShards(); ++shardId) {
                OptionalDouble writeLoadForShard = writeLoad.getWriteLoadForShard(shardId);
                totalIndexLoad += writeLoadForShard.orElse(0.0);
            }
            if (!(totalIndexLoad > maxIndexLoadWithinCoolingPeriod)) continue;
            maxIndexLoadWithinCoolingPeriod = totalIndexLoad;
        }
        return maxIndexLoadWithinCoolingPeriod;
    }

    void updateIncreaseShardsCooldown(TimeValue scaleUpCooldown) {
        this.increaseShardsCooldown = scaleUpCooldown;
    }

    void updateReduceShardsCooldown(TimeValue scaleDownCooldown) {
        this.reduceShardsCooldown = scaleDownCooldown;
    }

    void updateMinWriteThreads(int minNumberWriteThreads) {
        this.minWriteThreads = minNumberWriteThreads;
    }

    void updateMaxWriteThreads(int maxNumberWriteThreads) {
        this.maxWriteThreads = maxNumberWriteThreads;
    }

    private void updateDataStreamExcludePatterns(List<String> newExcludePatterns) {
        this.dataStreamExcludePatterns = newExcludePatterns;
    }
}

