/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.index;

import java.util.Collections;
import java.util.EnumSet;
import java.util.List;
import java.util.Objects;
import java.util.function.BiFunction;
import java.util.function.Function;
import java.util.function.Supplier;
import org.apache.lucene.search.Sort;
import org.apache.lucene.search.SortField;
import org.apache.lucene.search.SortedNumericSortField;
import org.apache.lucene.search.SortedSetSortField;
import org.elasticsearch.common.logging.DeprecationCategory;
import org.elasticsearch.common.logging.DeprecationLogger;
import org.elasticsearch.common.settings.Setting;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.index.IndexMode;
import org.elasticsearch.index.IndexSettings;
import org.elasticsearch.index.IndexVersion;
import org.elasticsearch.index.IndexVersions;
import org.elasticsearch.index.fielddata.IndexFieldData;
import org.elasticsearch.index.mapper.MappedFieldType;
import org.elasticsearch.search.MultiValueMode;
import org.elasticsearch.search.lookup.SearchLookup;
import org.elasticsearch.search.sort.SortOrder;

public final class IndexSortConfig {
    private static final DeprecationLogger DEPRECATION_LOGGER = DeprecationLogger.getLogger(IndexSortConfig.class);
    public static final Setting<List<String>> INDEX_SORT_FIELD_SETTING = Setting.stringListSetting("index.sort.field", Setting.Property.IndexScope, Setting.Property.Final, Setting.Property.ServerlessPublic);
    public static final Setting<List<SortOrder>> INDEX_SORT_ORDER_SETTING = Setting.listSetting("index.sort.order", Collections.emptyList(), IndexSortConfig::parseOrderMode, Setting.Property.IndexScope, Setting.Property.Final, Setting.Property.ServerlessPublic);
    public static final Setting<List<MultiValueMode>> INDEX_SORT_MODE_SETTING = Setting.listSetting("index.sort.mode", Collections.emptyList(), IndexSortConfig::parseMultiValueMode, Setting.Property.IndexScope, Setting.Property.Final, Setting.Property.ServerlessPublic);
    public static final Setting<List<String>> INDEX_SORT_MISSING_SETTING = Setting.listSetting("index.sort.missing", Collections.emptyList(), IndexSortConfig::validateMissingValue, Setting.Property.IndexScope, Setting.Property.Final, Setting.Property.ServerlessPublic);
    public static final FieldSortSpec[] TIME_SERIES_SORT;
    public static final FieldSortSpec[] TIMESTAMP_SORT;
    public static final FieldSortSpec[] HOSTNAME_TIMESTAMP_SORT;
    public static final FieldSortSpec[] HOSTNAME_TIMESTAMP_BWC_SORT;
    final FieldSortSpec[] sortSpecs;
    private final IndexVersion indexCreatedVersion;
    private final String indexName;
    private final IndexMode indexMode;
    private static final EnumSet<SortField.Type> ALLOWED_INDEX_SORT_TYPES;

    private static String validateMissingValue(String missing) {
        if (!"_last".equals(missing) && !"_first".equals(missing)) {
            throw new IllegalArgumentException("Illegal missing value:[" + missing + "], must be one of [_last, _first]");
        }
        return missing;
    }

    private static SortOrder parseOrderMode(String value) {
        try {
            return SortOrder.fromString(value);
        }
        catch (Exception e) {
            throw new IllegalArgumentException("Illegal sort order:" + value);
        }
    }

    private static MultiValueMode parseMultiValueMode(String value) {
        MultiValueMode mode = MultiValueMode.fromString(value);
        if (mode != MultiValueMode.MAX && mode != MultiValueMode.MIN) {
            throw new IllegalArgumentException("Illegal index sort mode:[" + String.valueOf(mode) + "], must be one of [" + String.valueOf(MultiValueMode.MAX) + ", " + String.valueOf(MultiValueMode.MIN) + "]");
        }
        return mode;
    }

    public IndexSortConfig(IndexSettings indexSettings) {
        int i;
        Settings settings = indexSettings.getSettings();
        this.indexCreatedVersion = indexSettings.getIndexVersionCreated();
        this.indexName = indexSettings.getIndex().getName();
        this.indexMode = indexSettings.getMode();
        if (this.indexMode == IndexMode.TIME_SERIES) {
            this.sortSpecs = TIME_SERIES_SORT;
            return;
        }
        List<String> fields = INDEX_SORT_FIELD_SETTING.get(settings);
        if (this.indexMode == IndexMode.LOGSDB && !INDEX_SORT_FIELD_SETTING.exists(settings)) {
            if (INDEX_SORT_ORDER_SETTING.exists(settings)) {
                List<SortOrder> order = INDEX_SORT_ORDER_SETTING.get(settings);
                throw new IllegalArgumentException("index.sort.fields:" + String.valueOf(fields) + " index.sort.order:" + String.valueOf(order) + ", size mismatch");
            }
            if (INDEX_SORT_MODE_SETTING.exists(settings)) {
                List<MultiValueMode> mode = INDEX_SORT_MODE_SETTING.get(settings);
                throw new IllegalArgumentException("index.sort.fields:" + String.valueOf(fields) + " index.sort.mode:" + String.valueOf(mode) + ", size mismatch");
            }
            if (INDEX_SORT_MISSING_SETTING.exists(settings)) {
                List<String> missing = INDEX_SORT_MISSING_SETTING.get(settings);
                throw new IllegalArgumentException("index.sort.fields:" + String.valueOf(fields) + " index.sort.missing:" + String.valueOf(missing) + ", size mismatch");
            }
            IndexVersion version = indexSettings.getIndexVersionCreated();
            this.sortSpecs = version.onOrAfter(IndexVersions.LOGSB_OPTIONAL_SORTING_ON_HOST_NAME_BACKPORT) ? (IndexSettings.LOGSDB_SORT_ON_HOST_NAME.get(settings) != false ? HOSTNAME_TIMESTAMP_SORT : TIMESTAMP_SORT) : HOSTNAME_TIMESTAMP_BWC_SORT;
            return;
        }
        this.sortSpecs = (FieldSortSpec[])fields.stream().map(FieldSortSpec::new).toArray(FieldSortSpec[]::new);
        if (INDEX_SORT_ORDER_SETTING.exists(settings)) {
            List<SortOrder> orders = INDEX_SORT_ORDER_SETTING.get(settings);
            if (orders.size() != this.sortSpecs.length) {
                throw new IllegalArgumentException("index.sort.field:" + String.valueOf(fields) + " index.sort.order:" + String.valueOf(orders) + ", size mismatch");
            }
            for (i = 0; i < this.sortSpecs.length; ++i) {
                this.sortSpecs[i].order = orders.get(i);
            }
        }
        if (INDEX_SORT_MODE_SETTING.exists(settings)) {
            List<MultiValueMode> modes = INDEX_SORT_MODE_SETTING.get(settings);
            if (modes.size() != this.sortSpecs.length) {
                throw new IllegalArgumentException("index.sort.field:" + String.valueOf(fields) + " index.sort.mode:" + String.valueOf(modes) + ", size mismatch");
            }
            for (i = 0; i < this.sortSpecs.length; ++i) {
                this.sortSpecs[i].mode = modes.get(i);
            }
        }
        if (INDEX_SORT_MISSING_SETTING.exists(settings)) {
            List<String> missingValues = INDEX_SORT_MISSING_SETTING.get(settings);
            if (missingValues.size() != this.sortSpecs.length) {
                throw new IllegalArgumentException("index.sort.field:" + String.valueOf(fields) + " index.sort.missing:" + String.valueOf(missingValues) + ", size mismatch");
            }
            for (i = 0; i < this.sortSpecs.length; ++i) {
                this.sortSpecs[i].missingValue = missingValues.get(i);
            }
        }
    }

    public boolean hasIndexSort() {
        return this.sortSpecs.length > 0;
    }

    public boolean hasPrimarySortOnField(String field) {
        return this.sortSpecs.length > 0 && this.sortSpecs[0].field.equals(field);
    }

    public Sort buildIndexSort(Function<String, MappedFieldType> fieldTypeLookup, BiFunction<MappedFieldType, Supplier<SearchLookup>, IndexFieldData<?>> fieldDataLookup) {
        if (!this.hasIndexSort()) {
            return null;
        }
        SortField[] sortFields = new SortField[this.sortSpecs.length];
        for (int i = 0; i < this.sortSpecs.length; ++i) {
            IndexFieldData<?> fieldData;
            FieldSortSpec sortSpec = this.sortSpecs[i];
            MappedFieldType ft = fieldTypeLookup.apply(sortSpec.field);
            if (ft == null) {
                String err = "unknown index sort field:[" + sortSpec.field + "]";
                if (this.indexMode == IndexMode.TIME_SERIES) {
                    err = err + " required by [" + IndexSettings.MODE.getKey() + "=time_series]";
                }
                throw new IllegalArgumentException(err);
            }
            if (!Objects.equals(ft.name(), sortSpec.field)) {
                if (this.indexCreatedVersion.onOrAfter(IndexVersions.V_7_13_0)) {
                    throw new IllegalArgumentException("Cannot use alias [" + sortSpec.field + "] as an index sort field");
                }
                DEPRECATION_LOGGER.warn(DeprecationCategory.MAPPINGS, "index-sort-aliases", "Index sort for index [" + this.indexName + "] defined on field [" + sortSpec.field + "] which resolves to field [" + ft.name() + "]. You will not be able to define an index sort over aliased fields in new indexes", new Object[0]);
            }
            boolean reverse = sortSpec.order == null ? false : sortSpec.order == SortOrder.DESC;
            MultiValueMode mode = sortSpec.mode;
            if (mode == null) {
                mode = reverse ? MultiValueMode.MAX : MultiValueMode.MIN;
            }
            try {
                fieldData = fieldDataLookup.apply(ft, () -> {
                    throw new UnsupportedOperationException("index sorting not supported on runtime field [" + ft.name() + "]");
                });
            }
            catch (Exception e) {
                throw new IllegalArgumentException("docvalues not found for index sort field:[" + sortSpec.field + "]", e);
            }
            if (fieldData == null) {
                throw new IllegalArgumentException("docvalues not found for index sort field:[" + sortSpec.field + "]");
            }
            sortFields[i] = fieldData.sortField(sortSpec.missingValue, mode, null, reverse);
            IndexSortConfig.validateIndexSortField(sortFields[i]);
        }
        return new Sort(sortFields);
    }

    private static void validateIndexSortField(SortField sortField) {
        SortField.Type type = IndexSortConfig.getSortFieldType(sortField);
        if (!ALLOWED_INDEX_SORT_TYPES.contains(type)) {
            throw new IllegalArgumentException("invalid index sort field:[" + sortField.getField() + "]");
        }
    }

    public static SortField.Type getSortFieldType(SortField sortField) {
        if (sortField instanceof SortedSetSortField) {
            return SortField.Type.STRING;
        }
        if (sortField instanceof SortedNumericSortField) {
            return ((SortedNumericSortField)sortField).getNumericType();
        }
        return sortField.getType();
    }

    static {
        FieldSortSpec timeStampSpec = new FieldSortSpec("@timestamp");
        timeStampSpec.order = SortOrder.DESC;
        TIME_SERIES_SORT = new FieldSortSpec[]{new FieldSortSpec("_tsid"), timeStampSpec};
        TIMESTAMP_SORT = new FieldSortSpec[]{timeStampSpec};
        FieldSortSpec hostnameSpec = new FieldSortSpec("host.name");
        hostnameSpec.order = SortOrder.ASC;
        hostnameSpec.missingValue = "_last";
        hostnameSpec.mode = MultiValueMode.MIN;
        HOSTNAME_TIMESTAMP_SORT = new FieldSortSpec[]{hostnameSpec, timeStampSpec};
        HOSTNAME_TIMESTAMP_BWC_SORT = new FieldSortSpec[]{new FieldSortSpec("host.name"), new FieldSortSpec("@timestamp")};
        ALLOWED_INDEX_SORT_TYPES = EnumSet.of(SortField.Type.STRING, SortField.Type.LONG, SortField.Type.INT, SortField.Type.DOUBLE, SortField.Type.FLOAT);
    }

    public static class FieldSortSpec {
        final String field;
        SortOrder order;
        MultiValueMode mode;
        String missingValue;

        FieldSortSpec(String field) {
            this.field = field;
        }

        public String getField() {
            return this.field;
        }

        public SortOrder getOrder() {
            return this.order;
        }
    }
}

